/*
 * Copyright (C) 2021 Liquidaty and the zsv/lib contributors
 * All rights reserved
 *
 * This file is part of zsv/lib, distributed under the license defined at
 * https://opensource.org/licenses/MIT
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <zsv.h>
#include <zsv/utils/string.h>
#include <zsv/utils/arg.h>

static struct zsv_opts zsv_default_opts = { 0 };
char zsv_default_opts_initd = 0;

ZSV_EXPORT
struct zsv_opts zsv_get_default_opts() {
  if(!zsv_default_opts_initd) {
    zsv_default_opts_initd = 1;
    zsv_default_opts.max_row_size = ZSV_ROW_MAX_SIZE_DEFAULT;
    zsv_default_opts.max_columns = ZSV_MAX_COLS_DEFAULT;
  }
  return zsv_default_opts;
}

ZSV_EXPORT
void zsv_set_default_opts(struct zsv_opts opts) {
  zsv_default_opts = opts;
}

/**
 * str_array_index_of: return index in list, or size of list if not found
 */
static inline int str_array_index_of(const char *list[], const char *s) {
  int i;
  for(i = 0; list[i] && strcmp(list[i], s); i++) ;
  return i;
}

#ifdef ZSV_EXTRAS

ZSV_EXPORT
void zsv_set_default_progress_callback(zsv_progress_callback cb, void *ctx, size_t rows_interval, unsigned int seconds_interval) {
  struct zsv_opts opts = zsv_get_default_opts();
  opts.progress.callback = cb;
  opts.progress.ctx = ctx;
  opts.progress.rows_interval = rows_interval;
  opts.progress.seconds_interval = seconds_interval;
  zsv_set_default_opts(opts);
}

ZSV_EXPORT
void zsv_set_default_completed_callback(zsv_completed_callback cb, void *ctx) {
  struct zsv_opts opts = zsv_get_default_opts();
  opts.completed.callback = cb;
  opts.completed.ctx = ctx;
  zsv_set_default_opts(opts);
}

#endif

ZSV_EXPORT
int zsv_args_to_opts(int argc, const char *argv[],
                     int *argc_out, const char **argv_out,
                     struct zsv_opts *opts_out
                     ) {
  *opts_out = zsv_get_default_opts();
  int options_start = 1; // skip this many args before we start looking for options
  int err = 0;
  int new_argc = 0;
  for(; new_argc < options_start && new_argc < argc; new_argc++)
    argv_out[new_argc] = argv[new_argc];

#ifdef ZSV_EXTRAS
  static const char *short_args = "BcrtOqvL";
#else
  static const char *short_args = "BcrtOqv";
#endif

  static const char *long_args[] = {
    "buff-size",
    "max-column-count",
    "max-row-size",
    "tab-delim",
    "other-delim",
    "no-quote",
    "verbose",
#ifdef ZSV_EXTRAS
    "limit-rows",
#endif
    NULL
  };
  for(int i = options_start; !err && i < argc; i++) {
    char arg = 0;
    if(*argv[i] != '-') { /* pass this option through */
      argv_out[new_argc++] = argv[i];
      continue;
    }
    if(argv[i][1] != '-') {
      if(!argv[i][2] && strchr(short_args, argv[i][1]))
        arg = argv[i][1];
    } else
      arg = short_args[str_array_index_of(long_args, argv[i] + 2)];

    switch(arg) {
    case 't':
      opts_out->delimiter = '\t';
      break;
    case 'q':
      opts_out->no_quotes = 1;
      break;
    case 'v':
      opts_out->verbose = 1;
      break;
#ifdef ZSV_EXTRAS
    case 'L':
#endif
    case 'B':
    case 'c':
    case 'r':
    case 'O':
      if(++i >= argc)
        err = fprintf(stderr, "Error: option %s requires a value\n", argv[i-1]);
      else if(arg == 'O') {
        const char *val = argv[i];
        if(strlen(val) != 1 || *val == 0)
          err = fprintf(stderr, "Error: delimiter '%s' may only be a single ascii character", val);
        else if(strchr("\n\r\"", *val))
          err = fprintf(stderr, "Error: column delimiter may not be '\\n', '\\r' or '\"'\n");
        else
          opts_out->delimiter = *val;
      } else {
        const char *val = argv[i];
        /* arg = 'B', 'c', 'r' or 'L' (ZSV_EXTRAS only) */
        long n = atol(val);
#ifdef ZSV_EXTRAS
        if(arg == 'L') {
          if(n < 1)
            err = fprintf(stderr, "Error: max rows may not be less than 1 (got %s)\n", val);
          else
            opts_out->max_rows = n;
        } else
#endif
        if(arg == 'B' && n < ZSV_MIN_SCANNER_BUFFSIZE)
          err = fprintf(stderr, "Error: buff size may not be less than %u (got %s)\n",
                        ZSV_MIN_SCANNER_BUFFSIZE, val);
        else if(arg == 'c' && n < 8)
          err = fprintf(stderr, "Error: max column count may not be less than 8 (got %s)\n", val);
        else if(arg == 'r' && n < ZSV_ROW_MAX_SIZE_MIN)
          err = fprintf(stderr, "Error: max row size size may not be less than %u (got %s)\n",
                        ZSV_ROW_MAX_SIZE_MIN, val);
        else if(arg == 'B')
          opts_out->buffsize = n;
        else if(arg == 'c')
          opts_out->max_columns = n;
        else if(arg == 'r')
          opts_out->max_row_size = n;
      }
      break;
    default: /* pass this option through */
      argv_out[new_argc++] = argv[i];
      break;
    }
  }

  *argc_out = new_argc;
  return err;
}
