const std = @import("std");
const g = @import("spirv/grammar.zig");

pub fn main() !void {
    var arena = std.heap.ArenaAllocator.init(std.heap.page_allocator);
    defer arena.deinit();
    const allocator = &arena.allocator;

    const args = try std.process.argsAlloc(allocator);
    if (args.len != 2) {
        usageAndExit(std.io.getStdErr(), args[0], 1);
    }

    const spec_path = args[1];
    const spec = try std.fs.cwd().readFileAlloc(allocator, spec_path, std.math.maxInt(usize));

    var tokens = std.json.TokenStream.init(spec);
    var registry = try std.json.parse(g.Registry, &tokens, .{.allocator = allocator});

    var bw = std.io.bufferedWriter(std.io.getStdOut().writer());
    try render(bw.writer(), registry);
    try bw.flush();
}

fn render(writer: anytype, registry: g.Registry) !void {
    try writer.writeAll(
        \\//! This file is auto-generated by tools/gen_spirv_spec.zig.
        \\
        \\const Version = @import("std").builtin.Version;
        \\
    );

    switch (registry) {
        .core => |core_reg| {
            try writer.print(
                \\pub const version = Version{{ .major = {}, .minor = {}, .patch = {} }};
                \\pub const magic_number: u32 = {s};
                \\
                , .{ core_reg.major_version, core_reg.minor_version, core_reg.revision, core_reg.magic_number },
            );
            try renderOpcodes(writer, core_reg.instructions);
            try renderOperandKinds(writer, core_reg.operand_kinds);
        },
        .extension => |ext_reg| {
            try writer.print(
                \\pub const version = Version{{ .major = {}, .minor = 0, .patch = {} }};
                \\
                , .{ ext_reg.version, ext_reg.revision },
            );
            try renderOpcodes(writer, ext_reg.instructions);
            try renderOperandKinds(writer, ext_reg.operand_kinds);
        }
    }
}

fn renderOpcodes(writer: anytype, instructions: []const g.Instruction) !void {
    try writer.writeAll("pub const Opcode = extern enum(u16) {\n");
    for (instructions) |instr| {
        try writer.print("    {} = {},\n", .{ std.zig.fmtId(instr.opname), instr.opcode });
    }
    try writer.writeAll("    _,\n};\n");
}

fn renderOperandKinds(writer: anytype, kinds: []const g.OperandKind) !void {
    for (kinds) |kind| {
        switch (kind.category) {
            .ValueEnum => try renderValueEnum(writer, kind),
            .BitEnum => try renderBitEnum(writer, kind),
            else => {},
        }
    }
}

fn renderValueEnum(writer: anytype, enumeration: g.OperandKind) !void {
    try writer.print("pub const {s} = extern enum(u32) {{\n", .{ enumeration.kind });

    const enumerants = enumeration.enumerants orelse return error.InvalidRegistry;
    for (enumerants) |enumerant| {
        if (enumerant.value != .int) return error.InvalidRegistry;

        try writer.print("    {} = {},\n", .{ std.zig.fmtId(enumerant.enumerant), enumerant.value.int });
    }

    try writer.writeAll("    _,\n};\n");
}

fn renderBitEnum(writer: anytype, enumeration: g.OperandKind) !void {
    try writer.print("pub const {s} = packed struct {{\n", .{ enumeration.kind });

    var flags_by_bitpos = [_]?[]const u8{null} ** 32;
    const enumerants = enumeration.enumerants orelse return error.InvalidRegistry;
    for (enumerants) |enumerant| {
        if (enumerant.value != .bitflag) return error.InvalidRegistry;
        const value = try parseHexInt(enumerant.value.bitflag);
        if (@popCount(u32, value) != 1) {
            continue; // Skip combinations and 'none' items
        }

        var bitpos = std.math.log2_int(u32, value);
        if (flags_by_bitpos[bitpos]) |*existing|{
            // Keep the shortest
            if (enumerant.enumerant.len < existing.len)
                existing.* = enumerant.enumerant;
        } else {
            flags_by_bitpos[bitpos] = enumerant.enumerant;
        }
    }

    for (flags_by_bitpos) |maybe_flag_name, bitpos| {
        try writer.writeAll("    ");
        if (maybe_flag_name) |flag_name| {
            try writer.writeAll(flag_name);
        } else {
            try writer.print("_reserved_bit_{}", .{bitpos});
        }

        try writer.writeAll(": bool ");
        if (bitpos == 0) { // Force alignment to integer boundaries
            try writer.writeAll("align(@alignOf(u32)) ");
        }
        try writer.writeAll("= false,\n");
    }

    try writer.writeAll("};\n");
}

fn parseHexInt(text: []const u8) !u31 {
    const prefix = "0x";
    if (!std.mem.startsWith(u8, text, prefix))
        return error.InvalidHexInt;
    return try std.fmt.parseInt(u31, text[prefix.len ..], 16);
}

fn usageAndExit(file: std.fs.File, arg0: []const u8, code: u8) noreturn {
    file.writer().print(
        \\Usage: {s} <spirv json spec>
        \\
        \\Generates Zig bindings for a SPIR-V specification .json (either core or
        \\extinst versions). The result, printed to stdout, should be used to update
        \\files in src/codegen/spirv.
        \\
        \\The relevant specifications can be obtained from the SPIR-V registry:
        \\https://github.com/KhronosGroup/SPIRV-Headers/blob/master/include/spirv/unified1/
        \\
        , .{arg0}
    ) catch std.process.exit(1);
    std.process.exit(code);
}
