// Copyright (c) 2020-2021 by the Zeek Project. See LICENSE for details.

#pragma once

#include <algorithm>
#include <list>
#include <map>
#include <memory>
#include <optional>
#include <string>
#include <utility>
#include <vector>

#include <hilti/ast/function.h>
#include <hilti/base/cache.h>
#include <hilti/compiler/context.h>
#include <hilti/compiler/detail/cxx/elements.h>
#include <hilti/compiler/detail/cxx/unit.h>

namespace hilti {

namespace logging::debug {
inline const DebugStream CodeGen("codegen");
} // namespace logging::debug

class Node;
class Unit;

namespace detail {

namespace codegen {
enum class TypeUsage { Storage, CopyParameter, InParameter, InOutParameter, FunctionResult, Ctor, None };

struct CxxTypes {
    std::optional<cxx::Type> base_type;
    std::optional<cxx::Type> storage;
    std::optional<cxx::Type> result;
    std::optional<cxx::Type> param_copy;
    std::optional<cxx::Type> param_in;
    std::optional<cxx::Type> param_inout;
    std::optional<cxx::Type> ctor;
    std::optional<cxx::Expression> default_;
};

/** Structure capturing runtime type information for a specific type. */
struct CxxTypeInfo {
    bool predefined; /**< True if the type information instances is being predefined statically by the runtime library
                        (vs. generated by the codegen) */
    cxx::Expression reference;                             /**< ID to refer to this type information instance. */
    std::optional<cxx::declaration::Constant> forward;     /**< Forward declaration for type information. */
    std::optional<cxx::declaration::Constant> declaration; /**< Actual declaration for type information.  */
};

} // namespace codegen

/**
 * HILTI's code generator. This is the main internal entry point for
 * generating C++ code from HILTI source code.
 */
class CodeGen {
public:
    CodeGen(const std::shared_ptr<Context>& context) : _context(context) {}

    /** Entry point for code generation. */
    Result<cxx::Unit> compileModule(Node& root, hilti::Unit* hilti_unit,
                                    bool include_implementation); // NOLINT(google-runtime-references)

    /** Entry point for generating additional cross-unit C++ code through HILTI's linker. */
    Result<cxx::Unit> linkUnits(const std::vector<cxx::linker::MetaData>& mds);

    std::shared_ptr<Context> context() const { return _context.lock(); }
    const Options& options() const { return context()->options(); }

    // These must be called only while a module is being compiled.
    std::optional<cxx::declaration::Type> typeDeclaration(const hilti::Type& t);
    std::list<cxx::declaration::Type> typeDependencies(const hilti::Type& t);
    cxx::Type compile(const hilti::Type& t, codegen::TypeUsage usage);
    cxx::Expression compile(const hilti::Expression& e, bool lhs = false);
    cxx::Expression compile(const hilti::Ctor& c, bool lhs = false);
    cxx::Expression compile(const hilti::expression::ResolvedOperator& o, bool lhs = false);
    cxx::Block compile(const hilti::Statement& s, cxx::Block* b = nullptr);
    cxx::declaration::Function compile(const ID& id, type::Function ft, declaration::Linkage linkage,
                                       function::CallingConvention cc = function::CallingConvention::Standard,
                                       const std::optional<AttributeSet>& fattrs = {},
                                       std::optional<cxx::ID> namespace_ = {});
    std::vector<cxx::Expression> compileCallArguments(const hilti::node::Range<Expression>& args,
                                                      const hilti::node::Set<declaration::Parameter>& params);
    std::vector<cxx::Expression> compileCallArguments(const hilti::node::Range<Expression>& args,
                                                      const hilti::node::Range<declaration::Parameter>& params);
    std::optional<cxx::Expression> typeDefaultValue(const hilti::Type& t);
    codegen::TypeUsage parameterKindToTypeUsage(declaration::parameter::Kind);

    cxx::Expression typeInfo(const hilti::Type& t);
    void addTypeInfoDefinition(const hilti::Type& t);

    cxx::Expression coerce(const cxx::Expression& e, const Type& src, const Type& dst); // only for supported coercions
    cxx::Expression pack(const Expression& data, const std::vector<Expression>& args);
    cxx::Expression pack(const hilti::Type& t, const cxx::Expression& data, const std::vector<cxx::Expression>& args);
    cxx::Expression unpack(const hilti::Type& t, const Expression& data, const std::vector<Expression>& args,
                           bool throw_on_error);
    cxx::Expression unpack(const hilti::Type& t, const cxx::Expression& data, const std::vector<cxx::Expression>& args,
                           bool throw_on_error);
    void addDeclarationFor(const hilti::Type& t) { _need_decls.push_back(t); }

    cxx::Expression addTmp(const std::string& prefix, const cxx::Type& t);
    cxx::Expression addTmp(const std::string& prefix, const cxx::Expression& init);

    /**
     * Returns an ID that's unique for a given node. The ID is derived from
     * the node's location information, which must be present.
     *
     * @param prefix constant prefix that will be added to ID
     * @param n node to generate the ID for
     *
     */
    cxx::ID uniqueID(const std::string& prefix, const Node& n);

    cxx::Expression self() const { return _self.back(); }
    cxx::Expression dollardollar() const {
        return {"__dd", cxx::Side::LHS};
    } // TODO(robin): We hardcode the currently; need a stack, too?
    void pushSelf(detail::cxx::Expression e) { _self.push_back(std::move(e)); }
    void popSelf() { _self.pop_back(); }

    auto cxxBlock() const { return ! _cxx_blocks.empty() ? _cxx_blocks.back() : nullptr; }
    void pushCxxBlock(cxx::Block* b) { _cxx_blocks.push_back(b); }
    void popCxxBlock() { _cxx_blocks.pop_back(); }

    void enablePrioritizeTypes() { ++_prioritize_types; }
    void disablePrioritizeTypes() { --_prioritize_types; }
    bool prioritizeTypes() const { return _prioritize_types > 0; }

    cxx::Unit* unit() const;        // will abort if not compiling a module.
    hilti::Unit* hiltiUnit() const; // will abort if not compiling a module.

private:
    const codegen::CxxTypeInfo& _getOrCreateTypeInfo(const hilti::Type& t);

    // Adapt expression so that it can be used as a LHS. If expr is already a
    // LHS, it's returned directly. Otherwise it assigns it over into a
    // temporary, which is then returned.
    cxx::Expression _makeLhs(cxx::Expression expr, const Type& type);

    std::unique_ptr<cxx::Unit> _cxx_unit;
    hilti::Unit* _hilti_unit = nullptr;
    std::weak_ptr<Context> _context;
    std::vector<detail::cxx::Expression> _self = {{"__self", cxx::Side::LHS}};
    std::vector<detail::cxx::Block*> _cxx_blocks;
    std::vector<detail::cxx::declaration::Local> _tmps;
    std::map<std::string, int> _tmp_counters;
    std::vector<hilti::Type> _need_decls;
    hilti::util::Cache<cxx::ID, codegen::CxxTypes> _cache_types_storage;
    hilti::util::Cache<cxx::ID, codegen::CxxTypeInfo> _cache_type_info;
    hilti::util::Cache<cxx::ID, cxx::declaration::Type> _cache_types_declarations;
    int _prioritize_types = 0;
};

} // namespace detail
} // namespace hilti
