/*_
 * Copyright (c) 2005, 2006 Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: xpkg.l,v 1.4 2006/02/22 22:22:12 mww Exp $
 *
 */

%option nomain
%option noinput
%option nounput
%option nostack
%option noyywrap
%option prefix="xpkgmain"

%{
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include <unistd.h>
#include "xpkg.h"
#include "config.h"
#include "util.h"
#include "usage.txtstr.h"

#define ecerr(x,y) if( x ) { fprintf(stderr,"error: %s\n", y); goto CLEANUP; }

#define A_NOACTION 0
#define A_ERROR 1
#define A_HELP 2
#define A_VERSION 3
#define A_INSTALL 4
#define A_REGISTER 5
#define A_DELETE 6
#define A_UPGRADE 7
#define A_QUERY_NAME 8
#define A_QUERY_FILE 9
#define A_LIST 10
#define A_CREATEDB 11
#define A_TEST 12

	int action = A_NOACTION;
	char** targets = NULL;
	int targetcount = 0;

	void add_target(char* text) {
		assert(text);
		assert(strlen(text) > 0);
		if (0 == targetcount) {
			assert(NULL == targets);
			targets = calloc(1,sizeof(char*));
		} else {
			assert(targets);
			targets = realloc(targets, (targetcount+1) * sizeof(char*));
		}
		targets[targetcount] = strdup(text);
		targetcount++;
	}

	void print_version() {
		fprintf(stdout, "xpkg, version %s\n", VERSION);
	}

	void print_help() {
		const char *help = STR_STR_usage;
		fprintf(stdout, help);
	}

	void print_error() {
		fprintf(stderr, "error in arguments!\n");
	}

	void set_action(int an_action) {
		if (A_NOACTION == action)
			action = an_action;
		else
			action = A_ERROR;
	}
	
	int package_register() {
		int i;
		xpkg_registry_t reg;
		xpkg_pkg_t pkg = NULL;

		ecerr( NULL==targets, "no targets given" )
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		printf("error: %s\n", xpkg_strerror(errno));
		ecerr( XPKG_OK!=xpkg_registry_transaction_begin( reg ), "being transaction" );
		for( i=0; i<targetcount; i++ ) {
			pkg = xpkg_pkg_open(targets[i]);
			ecerr( XPKG_OK!=xpkg_registry_registerpkg( reg, pkg ), "registering package" );
			ecerr( XPKG_OK!=xpkg_pkg_close( pkg ), "closing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_transaction_commit( reg ), "commiting transaction" );
		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		xpkg_pkg_free( &pkg );
		return 0;

		CLEANUP: {
			xpkg_pkg_close( pkg );
			xpkg_registry_transaction_rollback( reg );
			xpkg_registry_close( &reg );
			xpkg_pkg_free( &pkg );
			return -1;
		}
	}

	int package_install() {
		int i;
		xpkg_registry_t reg = NULL;
		xpkg_pkg_t pkg = NULL;

		ecerr( NULL==targets, "no targets given" )
		fprintf( stdout, "installing packages... " );
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		ecerr( XPKG_OK!=xpkg_registry_transaction_begin( reg ), "being transaction" );
		for( i=0; i<targetcount; i++ ) {
			pkg = xpkg_pkg_open(targets[i]);
			ecerr( XPKG_OK!=xpkg_registry_registerpkg( reg, pkg ), "registering package" );
			ecerr( XPKG_OK!=xpkg_pkg_close( pkg ), "closing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_checkconsistence( reg ), "consistency check" );
		for( i=0; i<targetcount; i++ ) {
			pkg = xpkg_pkg_open(targets[i]);
			ecerr( XPKG_OK!=xpkg_pkg_extract( pkg, getenv("XPKG_PKGROOT")), "extracting package" );
			ecerr( XPKG_OK!=xpkg_pkg_close( pkg ), "closing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_transaction_commit( reg ), "commiting transaction" );
		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		xpkg_pkg_free( &pkg );
		fprintf( stdout, "done\n" );
		return 0;

		CLEANUP: {
			xpkg_pkg_close( pkg );
			xpkg_registry_transaction_rollback( reg );
			xpkg_registry_close( &reg );
			xpkg_pkg_free( &pkg );
			fprintf( stdout, "failed\n" );
			return -1;
		}
	}

	int package_upgrade() {
		int i;
		char *c;
		xpkg_registry_t reg = NULL;
		xpkg_pkg_t pkg = NULL;

		ecerr( NULL==targets, "no targets given" )
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		printf("error: %s\n", xpkg_strerror(errno));
		ecerr( XPKG_OK!=xpkg_registry_transaction_begin( reg ), "being transaction" );
		for (i = 0; i < targetcount; i++) {
			pkg = xpkg_pkg_open(targets[i]);
			c = xpkg_pkg_getname(pkg);
			ecerr( XPKG_OK!=xpkg_registry_removepkg( reg, c ), "removing package" );
			free( c );
			ecerr( XPKG_OK!=xpkg_registry_registerpkg( reg, pkg ), "registering package" );
			ecerr( XPKG_OK!=xpkg_pkg_close( pkg ), "closing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_checkconsistence( reg ), "consistency check" );
		fprintf( stdout, "extracting packages..." );
		for( i=0; i<targetcount; i++ ) {
			pkg = xpkg_pkg_open(targets[i]);
			ecerr( XPKG_OK!=xpkg_pkg_extract(pkg, getenv("XPKG_PKGROOT")), "extracting package" );
			ecerr( XPKG_OK!=xpkg_pkg_close( pkg ), "closing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_transaction_commit( reg ), "commiting transaction" );
		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		xpkg_pkg_free( &pkg );
		return 0;

		CLEANUP: {
			xpkg_pkg_close( pkg );
			xpkg_registry_transaction_rollback( reg );
			xpkg_registry_close( &reg );
			xpkg_pkg_free( &pkg );
			return -1;
		}
	}

	int package_delete() {
		int i;
		xpkg_registry_t reg = NULL;
		xpkg_files_t files = NULL;

		ecerr( NULL==targets, "no targets given" );
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		ecerr( XPKG_OK!=xpkg_registry_transaction_begin( reg ), "starting transaction" );

		for( i=0; i<targetcount; i++ ) {
			fprintf( stdout, "removing %s\n", targets[i] );
			ecerr( XPKG_OK!=xpkg_registry_removepkg( reg, targets[i] ), "removing package" );
		}
		ecerr( XPKG_OK!=xpkg_registry_checkconsistence( reg ), "consistency check" );
		assert(0 <= xpkg_registry_getstalefiles(reg, &files));
		assert(files);

		ecerr( XPKG_OK!=xpkg_files_delete(files, getenv("XPKG_PKGROOT")), "removing files" );

		ecerr( XPKG_OK!=xpkg_registry_removestalefiles( reg ), "removing files from registry" );
		ecerr( XPKG_OK!=xpkg_registry_transaction_commit( reg ), "commiting transaction" );

		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		return 0;

		CLEANUP: {
			xpkg_registry_transaction_rollback( reg );
			xpkg_registry_close( &reg );
			return -1;
		}
	}

	int package_query(int mode) {
		int i;
		xpkg_registry_t reg = NULL;
		xpkg_pkgs_t pkgs = NULL;

		ecerr( NULL==targets, "no targets given" )
		reg = xpkg_registry_open( getenv("XPKG_REGFILE"));
		ecerr( XPKG_OK!=xpkg_registry_transaction_begin( reg ), "starting transaction" );
		for (i = 0; i < targetcount; i++) {
			assert(0 <= xpkg_registry_querypkgs(reg, targets[i], mode, &pkgs));
			if (NULL != pkgs) {
				xpkg_pkgs_print(pkgs);
				xpkg_pkgs_free(&pkgs);
			} else {
				printf("no such package: \"%s\"\n", targets[i]);
			}
		}
		ecerr( XPKG_OK!=xpkg_registry_transaction_commit( reg ), "closing transaction" );
		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		return 0;

		CLEANUP: {
			xpkg_registry_close(&reg);
			xpkg_pkgs_free(&pkgs);
			return -1;
		}
	}

	int package_listall() {
		int n;
		xpkg_registry_t reg = NULL;
		xpkg_pkgs_t pkgs = NULL;
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		ecerr(XPKG_OK!=xpkg_registry_transaction_begin(reg), "starting transaction" );
		n = xpkg_registry_querypkgs(reg, NULL, XPKG_QUERYMODE_LISTALL, &pkgs);
		if (n > 0) {
			xpkg_pkgs_print(pkgs);
		} else {
			fprintf(stdout, "no packages installed\n");
		}
		ecerr(XPKG_OK!=xpkg_registry_transaction_commit(reg), "closing transaction");
		ecerr(XPKG_OK!=xpkg_registry_close(&reg), "closing registry");
		xpkg_pkgs_free(&pkgs);
		return 0;

		CLEANUP: {
			xpkg_registry_close(&reg);
			xpkg_pkgs_free(&pkgs);
			return -1;
		}
	}

	int createdb() {
		xpkg_registry_t reg = NULL;
		fprintf( stdout, "initializing registry...");
		//ecerr( 0==access( getenv("XPKG_REGFILE"), F_OK ), "registry already exists" );
		reg = xpkg_registry_open(getenv("XPKG_REGFILE"));
		printf("error: %s\n", xpkg_strerror(errno));
		xpkg_registry_init( reg );
		printf("error: %s\n", xpkg_strerror(errno));
		ecerr( XPKG_OK!=xpkg_registry_close( &reg ), "closing registry" );
		fprintf( stdout, " done\n" );
		return 0;

		CLEANUP: {
			fprintf( stdout, " failed\n" );
			xpkg_registry_close( &reg );
			return -1;
		}
	}

	int execute() {
		int rc = 0;
		switch( action ) {
			case A_VERSION:
				print_version();
				break;
			case A_INSTALL:
				rc = package_install();
				break;
			case A_REGISTER:
				rc = package_register();
				break;
			case A_DELETE:
				rc = package_delete();
				break;
			case A_UPGRADE:
				rc = package_upgrade();
				break;
			case A_QUERY_NAME:
				rc = package_query( XPKG_QUERYMODE_BYNAME );
				break;
			case A_QUERY_FILE:
				rc = package_query( XPKG_QUERYMODE_BYPATH );
				break;
			case A_LIST:
				rc = package_listall();
				break;
			case A_CREATEDB:
				rc = createdb();
				break;
			case A_TEST:
				//test_p();
				printf( "noop\n" );
				break;
			case A_ERROR:
				rc = -1;
				print_error();
			case A_HELP:
			default:
				print_help();
		}
		return rc;
	}
%}

%%
(-v|--verbose)					{ setenv( "XPKG_VERBOSE", "1", 0 ); };
(-d|--debug)					{ setenv( "XPKG_DEBUG", "1", 0 ); };
(-i|--install)			 		{ set_action( A_INSTALL ); };
(-r|--register)					{ set_action( A_REGISTER ); };
(-e|--erase|--uninstall)		{ set_action( A_DELETE ); };
(-u|--upgrade)					{ set_action( A_UPGRADE ); };
(-qn|--query-name)				{ set_action( A_QUERY_NAME ); };
(-qf|--query-file)				{ set_action( A_QUERY_FILE ); };
(-l|--list)						{ set_action( A_LIST ); };
(-c|--create-database)			{ set_action( A_CREATEDB ); };
(-t|--test)						{ set_action( A_TEST ); };
(-V|--version)					{ set_action( A_VERSION ); }
(-h|--help)						{ set_action( A_HELP ); }
^[^-].*$						{ add_target( yytext ); };
[[:blank:]]|\n
.								{ set_action( A_ERROR ); }
%%

int yyerror( char *c ) {
	printf( "oops: %s\n", c );
	return 0;
}

int main( int argc, char* argv[] ) {
	int i, len = 0;
	char *yybuffer = NULL;
	YY_BUFFER_STATE input_buffer = NULL;

	if( NULL==getenv("XPKG_REGFILE") ) {
		fprintf( stderr, "XPKG_REGFILE not set\n" );
		setenv( "XPKG_REGFILE", "/var/tmp/xpkg.reg", 1 );
	}
	if( NULL==getenv("XPKG_PKGROOT") ) {
		fprintf( stderr, "XPKG_PKGROOT not set\n" );
		setenv( "XPKG_PKGROOT", "/var/tmp", 1 );
	}

	yybuffer = malloc(0);
	for( i = 1; i < argc; i++ ) {
		len += 1 + strlen( argv[i] );
		yybuffer = realloc( yybuffer, len );
		strcat( yybuffer, argv[i] );
		strcat( yybuffer, "\n" );
	}

	input_buffer = yy_scan_string( yybuffer );
	i = yylex();

	return execute();
}
