/*_
 * Copyright (c) 2005, 2006 Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: pkg.c,v 1.20 2006/02/22 22:21:28 mww Exp $
 *
 */

#include <assert.h>
#include <errno.h>
#include <stdbool.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/queue.h>
#include <unistd.h>
#include <xar/xar.h>

#include "file.h"
#include "pkg.h"
#include "util.h"
#include "xml.h"
#include "xpkg.h"

/* translate xar-filetypes(char*) to xpkg-filetypes(int) */
XPKG_FILE_TYPE
translate_filetype(xar_file_t xfile) {
	const char *c;
	int rc;
	int n;

	assert(xfile);
	n = xar_prop_get( xfile, "type", &c );
	if( 0 == strcmp( "directory", c ) ) {
		rc = XPKG_FILE_DIR;
	} else if( 0 == strcmp( "file", c ) ) {
		rc = XPKG_FILE_REG;
	} else if( 0 == strcmp( "symlink", c ) ) {
		rc = XPKG_FILE_LNK;
	}
	assert(rc > 0);
	return rc;
}

char *
get_xarheader_subdoc(xar_t xarchive) {
	xar_subdoc_t sdoc;
	unsigned char *buffer;
	unsigned int len = -1;
	char *name;

	buffer = NULL;
	assert(xarchive);
	sdoc = xar_subdoc_first(xarchive);
	for (; NULL != sdoc; sdoc = xar_subdoc_next(sdoc)) {
		name = (char*)xar_subdoc_name(sdoc);
		DEBUG("found subdoc");
		if (0 == strncmp(name, "xpkg", 4)) {
			DEBUG("found xpkg-subdoc");
			xar_subdoc_copyout(sdoc, &buffer, &len);
			assert(buffer);
			break;
		}
		FREE(name);
	}
	return (char*)buffer;
}

xpkg_files_t
get_xarfileslist(xar_t xarchive) {
	char *path;
	int ftype;
	xar_iter_t xiter;
	xar_file_t xfile;
	xpkg_file_t file;
	xpkg_files_t files;

	files = files_create();
	DEBUG("getting file list");

	assert(xarchive);
	xiter = xar_iter_new(xarchive);
	xfile = xar_file_first(xarchive, xiter);
	while (NULL != xfile) {
		path = xar_get_path(xfile);
		ftype = translate_filetype(xfile);
		file = file_create(path, ftype, NULL);
		file_print(file);
		FREE(path);
		file_to_files(files, file);
		xfile = xar_file_next(xiter);
	}
	xar_iter_free(xiter);

	return files;
}

xpkg_pkg_t
xpkg_pkg_open(const char *filepath) {
	xpkg_pkg_t pkg;
	xar_t xarchive;
	char *xml;

	assert(filepath);

	xarchive = xar_open(filepath, READ);
	if (NULL != xarchive) {
		if (0 == access(filepath, READ)) {
			pkg = calloc(1, sizeof(struct xpkg_pkg_t));
			pkg->archive = xarchive;
			xml = get_xarheader_subdoc(pkg->archive);
			assert(xml);
			DEBUG(xml);
			xml_parse(xml, pkg);
			pkg->files = get_xarfileslist(xarchive);
			assert(pkg->files);
			//rc = XPKG_OK;
			//*pkg_out = pkg;
		} else {
			assert(false);
			//rc = XPKG_CANT_ACCESS_FILE;
		}
	} else {
		assert(false);
	}

	return pkg;
}

int32_t
xpkg_pkg_extract(xpkg_pkg_t pkg, const char *rootpath) {
	xar_iter_t xiter;
	xar_file_t xfile;

	DEBUG("extracting");
	assert(pkg->archive);

	if (0 != access(rootpath, W_OK)) {
		DEBUG("cant access root dir");
		return -1;
	}
	if (0 != chdir(rootpath)) {
		DEBUG("cannot chdir to root dir");
		return -1;
	}
	DEBUG("entered root dir");
	xiter = xar_iter_new(pkg->archive);
	xfile = xar_file_first(pkg->archive, xiter);
	while (NULL != xfile) {
		if (0!= xar_extract(pkg->archive, xfile))
		{
			DEBUG("extract error");
			xar_iter_free(xiter);
			return -1;
		}
		DEBUG("extracted file");
		xfile = xar_file_next(xiter);
	}
	xar_iter_free(xiter);
	DEBUG("leaving extract");
	return 0;
}

int32_t
xpkg_pkg_close(xpkg_pkg_t pkg) {
	int rc;
	if (NULL == pkg) {
		rc = XPKG_ERROR;
	} else if (NULL == pkg->archive) {
		rc = XPKG_ERROR;
	} else {
		xar_close(pkg->archive);
		pkg->archive = NULL;
		rc = XPKG_OK;
	}
	return rc;
}

void
xpkg_pkg_free(xpkg_pkg_t *pkg) {
	xpkg_pkg_t p;

	p = *pkg;
	DEBUG("freeing *pkg");
	if (NULL != p->name)
		FREE(p->name);
	if (NULL != p->version)
		FREE(p->version);
	xpkg_items_free(&(p->requires));
	xpkg_items_free(&(p->provides));
	xpkg_files_free(&(p->files));
	FREE(*pkg);
}

void
xpkg_pkgs_free(xpkg_pkgs_t *pkgs) {
	xpkg_pkg_t p;
	xpkg_pkgs_t ps;

	ps = *pkgs;
	if (NULL != ps) {
		for (;;) {
			p = (*ps).lh_first;
			if (NULL == p)
				break;
			LIST_REMOVE(p, entries);
			xpkg_pkg_free(&p);
		}
		FREE(*pkgs);
	}
}

int32_t
xpkg_pkg_print(xpkg_pkg_t pkg) {
	int rc = XPKG_OK;

	printf( "Package: %s, Version: %s, Revision %d (%d/%d)\n", pkg->name, pkg->version, pkg->revision, pkg->major, pkg->minor );
	if (NULL != getenv("XPKG_VERBOSE") && NULL != pkg->files) {
		xpkg_files_print( pkg->files );
	}
	return rc;
}

void
xpkg_pkgs_print(xpkg_pkgs_t pkgs) {
	xpkg_pkg_t pkg;

	for (pkg = (*pkgs).lh_first; NULL !=pkg; pkg = pkg->entries.le_next)
		xpkg_pkg_print(pkg);
}

xpkg_pkg_t
pkg_create(const char *name, const char* version, int32_t revision, int32_t major, int32_t minor) {
	xpkg_pkg_t pkg;

	assert(name);
	assert(version);
	pkg = calloc(1, sizeof(struct xpkg_pkg_t));
	pkg->archive = NULL;
	pkg->name = strdup(name);
	pkg->version = strdup(version);
	pkg->revision = revision;
	pkg->major = major;
	pkg->minor = minor;
	pkg->requires = NULL;
	pkg->provides = NULL;
	pkg->files = NULL;
	return pkg;
}

xpkg_pkgs_t
pkgs_create() {
	xpkg_pkgs_t p;

	p = calloc(1, sizeof(struct xpkg_pkgs_t));
	LIST_INIT(p);
	return p;
}

void
pkg_to_pkgs(xpkg_pkgs_t pkgs, xpkg_pkg_t pkg) {
	LIST_INSERT_HEAD(pkgs, pkg, entries);
}

char *
xpkg_pkg_getname(xpkg_pkg_t pkg) {
	return strdup(pkg->name);
}

