/*_
 * Copyright (c) 2005, Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: file.c,v 1.12 2006/02/22 22:21:28 mww Exp $
 *
 */

#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "file.h"
#include "util.h"
#include "xpkg.h"

xpkg_file_t
file_create(const char *path, XPKG_FILE_TYPE ftype, const char *chksum) {
	xpkg_file_t f;

	assert(path);
	assert(strlen(path)>0);
	f = calloc(1, sizeof(struct xpkg_file_t));
	assert(f);
	f->path = strdup(path);
	f->ftype = ftype;
	if (NULL != chksum)
		f->chksum = strdup(chksum);
	return f;
}

xpkg_files_t
files_create() {
	xpkg_files_t f;

	f = calloc(1, sizeof(struct xpkg_files_t));
	assert(f);
	LIST_INIT(f);
	return f;
}

void
file_to_files(xpkg_files_t files, xpkg_file_t file) {
	assert(files);
	assert(file);
	LIST_INSERT_HEAD(files, file, entries);
}

void
file_print(xpkg_file_t file) {
	assert(file);
	assert(file->path);
	printf(" %d, %s\n", file->ftype, file->path);
}

void
xpkg_files_print(xpkg_files_t files) {
	xpkg_file_t f;

	if (NULL != files)
		for (f = (*files).lh_first; NULL != f; f = f->entries.le_next) {
			assert(f);
			assert(f->path);
			assert(strlen(f->path)>0);
			printf(" %d, %s\n", f->ftype, f->path);
		}
}

void
xpkg_file_free(xpkg_file_t *file) {
	xpkg_file_t f;

	assert(*file);
	f = *file;
	assert(f->path);
	FREE(f->path);
	if (NULL != f->chksum)
		FREE(f->chksum);
	FREE(*file);
}

void
xpkg_files_free(xpkg_files_t *files) {
	xpkg_file_t f;
	xpkg_files_t n;

	if (NULL != *files) {
		n = *files;
		for (;;) {
			f = (*n).lh_first;
			if (NULL == f)
				break;
			LIST_REMOVE(f, entries);
			xpkg_file_free(&f);
		}
		FREE(*files);
	}
}

int32_t
xpkg_files_delete(xpkg_files_t files, const char *rootpath) {
	xpkg_file_t f;

	DEBUG("going to delete files");

	if (NULL == rootpath)
		rootpath = "/";
	if (NULL == files) {
		DEBUG("no files to delete");
		return 0;
	}
	if (0 != chdir(rootpath))
		/* errno set by chdir */
		return -1;

	for (f = (*files).lh_first; f != NULL; f = f->entries.le_next ) {
		DEBUG(f->path);
		switch (f->ftype) {
			case XPKG_FILE_DIR:
				DEBUG("deleting directory");
				if (0 == access(f->path, W_OK))
					rmdir(f->path);
				break;
			case XPKG_FILE_REG:
			case XPKG_FILE_LNK:
				DEBUG("deleting regular file/symlink");
				if (0 == access(f->path, W_OK))
					unlink(f->path);
				break;
			default:
				DEBUG("unknown file type");
				assert(0);
		}
	}
	return 0;
}

