/*_
 * Copyright (c) 2006, Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: xpkgcreate.l,v 1.4 2006/02/20 22:11:04 mww Exp $
 *
 */

%option noyywrap
%option stack
%option yylineno
%option nounput

%{

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <xar/xar.h>

#define FLEXERROR fprintf(stderr,"\"%s\" in line %d\n", yytext, yylineno);

char *xml;
char *filename;
char *destroot;

xar_t xarfile = NULL;		/* archive file */
xar_subdoc_t xardoc = NULL;

int add_file(char *xfilename) {
	xar_file_t xf;
	assert(xarfile);
	xf = xar_add(xarfile, xfilename);
	if (NULL == xf) {
		fprintf(stderr, "no such file \"%s\" - aborting!\n", xfilename);
		exit(1);
	}

	return 0;
}

int add_doc(char *xmlc) {
	xar_subdoc_t s = NULL;
	int i;
	s = xar_subdoc_new(xarfile, "xpkg");
	assert(s);
	i = xar_subdoc_copyin(s, (unsigned char*)xmlc, strlen(xmlc));
	assert(0 == i);
	return 0;
}

int xopen() {
	int i;
	assert(filename);
	xarfile = xar_open(filename, WRITE);
	assert(xarfile);
	xar_opt_set(xarfile, XAR_OPT_TOCCKSUM, XAR_OPT_VAL_SHA1);
	xar_opt_set(xarfile, XAR_OPT_COMPRESSION, XAR_OPT_VAL_GZIP);
	i = chdir(destroot);
	assert(0 == i);
	return 0;
}

int xclose() {
	xar_close(xarfile);
	return 0;
}

%}

%x files package provides requires num_minor num_major

%%

<*>{
%package\n		{ BEGIN package; }
%provides\n		{ asprintf(&xml, "%s<provides>\n", xml); yy_push_state(provides); }
%requires\n		{ asprintf(&xml, "%s<requires>\n", xml); yy_push_state(requires); }
%files\n		{ BEGIN files; }
}

<package>{
[[:alnum:]_]+/:[[:alnum:]_\.]+:[0-9]+:[0-9]+:[0-9]+\n	{ asprintf(&xml, "%s<name>%s</name>\n", xml, yytext); filename = strdup(yytext); }
[[:alnum:]_\.]+/:[0-9]+:[0-9]+:[0-9]+\n		{ asprintf(&xml, "%s<version>%s</version>\n", xml, yytext); asprintf(&filename, "%s-%s.xpkg", filename, yytext); xopen(); }
[0-9]+/:[0-9]+:[0-9]+\n				{ asprintf(&xml, "%s<revision>%s</revision>\n", xml, yytext); }
[0-9]+/:[0-9]\n					{ asprintf(&xml, "%s<major>%s</major>\n", xml, yytext); }
[0-9]+$						{ asprintf(&xml, "%s<minor>%s</minor>\n", xml, yytext); }
:
.						{ FLEXERROR; assert(false); }
\n
}

<provides>{
[[:alnum:]_]+/:[0-9]+:[0-9]+            { asprintf(&xml, "%s<item><name>%s</name>", xml, yytext); yy_push_state(num_major); };
\n			{ asprintf(&xml, "%s</provides>\n", xml); yy_pop_state(); }
.			{ FLEXERROR; assert(false); }
\n
}

<requires>{
[[:alnum:]_]+/:[0-9]+:[0-9]+		{ asprintf(&xml, "%s<item><name>%s</name>", xml, yytext); yy_push_state(num_major); };
\n			{ asprintf(&xml, "%s</requires>\n", xml); yy_pop_state(); }
.			{ FLEXERROR; assert(false); }
\n
}

<num_major>{
:
[0-9]+/:[0-9]+		{ asprintf(&xml, "%s<major>%s</major>", xml, yytext); yy_push_state(num_minor); };
\n			{ asprintf(&xml, "%s</item>\n", xml); yy_pop_state(); }
.			{ FLEXERROR; assert(false); }
}

<num_minor>{
:
[0-9]+/\n		{ asprintf(&xml, "%s<minor>%s</minor>", xml, yytext); yy_pop_state(); }
.			{ FLEXERROR; assert(false); }
}

<files>{
\/.+		{ add_file(yytext); };
.+		{ add_file(yytext); };
.		{ FLEXERROR assert(false); };
\n
}

%%

int main(int argc, char *argv[]) {
	int i;
	if (2 != argc && 3 != argc) {
		printf("usage: %s filename [destroot]\n", argv[0]);
		return 1;
	}
	yyin = fopen(argv[1],"r");
	if (3 == argc) {
		destroot = argv[2];
		i = access(destroot, R_OK);
		assert(0 == i);
	}
	asprintf(&xml, "<xpkg version=\"0.1\">\n");
	yylex();
	asprintf(&xml, "%s</xpkg>", xml);
	add_doc(xml);
	xclose();
	return 0;
}

