/* read the UCAC2 catalog, and probably the UCAC3. 
 * UCACSetup(): call to change options and base directories.
 * UCACFetch(): return an array of ObjF matching the given criteria.
 * stars are in 360 files each .5 deg hi, within 240 bins each .1 hour RA wide.
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "xephem.h"

#define	MAXFOV	15.0			/* max fov, degs */

typedef unsigned char UC;		/* byte */
typedef unsigned int UI;		/* unsigned integer */
typedef UC UStar[44];			/* one star in UCAC */

/* access an I*2 or I*4 at offset i in UC array a in little-endian byte order.
 * slow but ultra portable.
 */
#define	I2(a,i)		((int)((((UI)(a)[i]) | (((UI)(a)[i+1])<<8))))
#define	I4(a,i)		((int)((((UI)(a)[i]) | (((UI)(a)[i+1])<<8) | \
				(((UI)(a)[i+2])<<16) | (((UI)(a)[i+3])<<24))))

/* keep track of an array of ObjF */
typedef struct {
    ObjF *mem;				/* malloced array, or NULL if none */
    int n;				/* number actually in use */
} ObjFArray;

#define	NZW	240			/* number ra zones wide */
#define	ZW	(24./NZW)		/* zone width, hours */
#define	NZH	360			/* number dec zones hi */
#define	ZH	(180./NZH)		/* zone height, degrees */

static FILE *openIndex (char *path, char *msg);
static int addBin (ObjFArray *oap, int rz, int dz);
static int getN (int rz, int dz, int *idp);
static int readRaw (UStar buf[], int dz, int nskip, int nraw);
static void crackUStar (UStar u, int id, Obj *op);
static void jkspect (double j, double k, Obj *op);

static char *basedir;			/* full dir with zone files and index */
static FILE *indexfp;			/* index file */

/* save the path to the base dir.
 * test for some reasonable entries.
 * return 0 if looks ok, else -1 and reason in msg[].
 */
int
UCACSetup (path, msg)
char *path;
char msg[];
{
	FILE *fp;

	/* sanity check the path by looking for the index file */
	fp = openIndex (path, msg);
	if (!fp)
	    return (-1);
	fclose (fp);

	/* store persistent copy of path */
	if (basedir)
	    free (basedir);
	strcpy (basedir = malloc(strlen(path)+1), path);

	/* probably ok */
	return (0);

}

/* create or add to a malloced array of ObjF at *opp in the given region and
 *   return the new total count.
 * if opp == NULL we don't malloc anything but just do the side effects;
 * else *opp already has nopp ObjF in it (it's ok if *opp == NULL).
 * we return new total number of stars or -1 if real trouble.
 * *opp is only changed if we added any.
 * msg might contain a message regardless of the return value.
 */
int
UCACFetch (r0, d0, fov, fmag, opp, nopp, msg)
double r0;	/* center RA, rads */
double d0;	/* center Dec, rads */
double fov;	/* field of view, rads */
double fmag;	/* faintest mag */
ObjF **opp;	/* *opp will be a malloced array of the ObjF in region */
int nopp;       /* if opp: initial number of ObjF already in *opp */
char msg[];	/* filled with error message if return -1 */
{
	int d0z, drovz;			/* dec center and radius, in 0-zones */
	ObjFArray oa;			/* malloc accumulator */
	int dz;				/* scanning dec zone number */

	/* init message */
	msg[0] = '\0';

	/* insure there is a basedir set up */
	if (!basedir) {
	    strcpy (msg, "UCACFetch() called before UCACSetup()");
	    return (-1);
	}

	/* don't go crazy */
	if (fov > degrad(MAXFOV)) {
	    sprintf (msg, "UCAC FOV being clamped to %g degs", MAXFOV);
	    fov = degrad(MAXFOV);
	}

	/* open the index file */
	indexfp = openIndex (basedir, msg);
	if (indexfp == 0)
	    return (-1);

	/* init the array.
	 * mem==NULL means we just keep a count but don't build the array.
	 */
	if (opp) {
	    oa.mem = *opp;
	    if (!oa.mem)
		oa.mem = (ObjF*)malloc(1);	/* seed for realloc */
	    oa.n = nopp;
	} else {
	    oa.mem = NULL;
	    oa.n = 0;
	}

	/* convert to zones */
	d0z = (int)floor(raddeg(d0+PI/2)/ZH);
	drovz = (int)ceil(raddeg(fov/2)/ZH);

	/* scan each ra bin in each dec band */
	for (dz = d0z - drovz; dz <= d0z + drovz; dz++) {
	    int rz, rz0, rz1;			/* scanning ra zones */
	    if (dz < 0 || dz >= NZH)
		continue;			/* over a pole */
	    if (dz > 2*NZH-d0z-drovz || dz < drovz-d0z) {
		rz0 = 0;			/* pole in view */
		rz1 = NZW-1;
	    } else {
		int r0z, rrovz;			/* RA center and rad, 0-zones */
		r0z = (int)floor(radhr(r0)/ZW);
		if (fabs(d0) < PI/2) {
		    rrovz = (int)ceil(radhr(fov/2)/ZW/cos(d0));
		    if (rrovz > NZW/2)
			rrovz = NZW/2;
		} else
		    rrovz = NZW/2;
		rz0 = r0z - rrovz;		/* normal patch */
		rz1 = r0z + rrovz;
	    }
	    for (rz = rz0; rz <= rz1; rz++)
		if (addBin (&oa, rz, dz) < 0) {
		    char rstr[32], dstr[32];
		    fs_sexa (rstr, rz*ZW, 2, 60);
		    fs_sexa (dstr, dz*ZH-90, 3, 60);
		    sprintf (msg, "No UCAC data in segment at %s %s",rstr,dstr);
		    if (!opp && oa.mem)
			free ((void *)oa.mem);
		    fclose (indexfp);
		    return (-1);
		}
	}

	/* pass back to caller if interested */
	if (opp)
	    *opp = oa.mem;

	/* close index file */
	fclose (indexfp);

	/* return list */
	return (oa.n);
}

/* add the stars in the given 0-based ra/dec zone to oap.
 * return 0 if ok, else -1
 * N.B. rz may wrap 
 */
static int
addBin (ObjFArray *oap, int rz, int dz)
{
	int nthis, nprior, npriorzone, nnew;
	UStar *u;
	int r, d;
	int i;

	/* beware of ra wrap */
	if (rz < 0)
	    rz += NZW;
	if (rz >= NZW)
	    rz -= NZW;

	/* n stars up through and including this patch */
	if (getN (rz, dz, &nthis) < 0)
	    return (-1);

	/* n stars up to but not including this patch */
	if (rz > 0) {
	    r = rz-1;
	    d = dz;
	} else {
	    r = NZW-1;
	    d = dz-1;
	}
	if (d < 0)
	    nprior = 0;
	else if (getN (r, d, &nprior) < 0)
	    return (-1);

	/* n stars up through last dec zone */
	if (dz == 0)
	    npriorzone = 0;
	else if (getN (NZW-1, dz-1, &npriorzone) < 0)
	    return (-1);

	/* read the raw star records in this record */
	nnew = nthis - nprior;
	u = malloc (nnew * sizeof(UStar));
	if (!u)
	    return (-1);
	if (readRaw (u, dz, nprior-npriorzone, nnew) < 0) {
	    free ((char *)u);
	    return (-1);
	}

	/* expand obj memory and crack if interested */
	if (oap->mem) {
	    char *moreobj = realloc (oap->mem, (oap->n + nnew)*sizeof(ObjF));
	    if (!moreobj) {
		free (u);
		return (-1);
	    }
	    oap->mem = (ObjF *)moreobj;

	    for (i = 0; i < nnew; i++)
		crackUStar (u[i], nprior+i+1, (Obj *)(oap->mem + oap->n + i));
	}

	/* always count */
	oap->n += nnew;

	/* ok */
	free ((char *)u);
	return (0);
}

/* return the number of stars up through and including the given 0-based patch.
 * return 0 if ok, else -1
 */
static int
getN (int rz, int dz, int *idp)
{
	off_t offset;
	UC nat[4];

	offset = (dz*NZW + rz)*sizeof(nat);
	if (fseek (indexfp, offset, SEEK_SET) < 0)
	    return (-1);
	if (fread (nat, sizeof(nat), 1, indexfp) != 1)
	    return (-1);
	*idp = I4(nat,0);
	return (0);
}

/* read the n raw star records starting after nskip records from the given
 * 0-based ra/dec patch.
 * return 0 if ok, else -1
 */
static int
readRaw (UStar u[], int dz, int nskip, int nnew)
{
	char fn[1024];
	FILE *fp;

	sprintf (fn, "%s/z%03d", basedir, dz+1);
	fp = fopen (fn, "r");
	if (!fp)
	    return (-1);
	if (fseek (fp, nskip*sizeof(UStar), SEEK_SET) < 0) {
	    fclose (fp);
	    return (-1);
	}
	if (fread (u, sizeof(UStar), nnew, fp) != nnew) {
	    fclose (fp);
	    return (-1);
	}
	fclose (fp);
	return (0);
}

/* open index file in dir, return FILE * else NULL with message */
static FILE *
openIndex (char dir[], char msg[])
{
	static char u2[] = "u2index.da";	/* zone index file name */
	static char u3[] = "u3index.da";	/* zone index file name? */
	char full[1024];
	FILE *fp;

	sprintf (full, "%s/%s", dir, u2);
	fp = fopen (full, "r");
	if (!fp) {
	    sprintf (full, "%s/%s", dir, u3);
	    fp = fopen (full, "r");
	    if (!fp)
		sprintf (msg, "Can not find %s or %s", u2, u3);
	}
	return (fp);
}

/* convert the raw UCAC record into the ObjF portion of op */
static void
crackUStar (UStar u, int id, Obj *op)
{
#define	DPMAS	(1.0/3600000.0)		/* degrees per milliarcsecond */
	memset (op, 0, sizeof(ObjF));	/* N.B. ObjF, not Obj */

	sprintf (op->o_name, "UCAC%08d", id);
	op->o_type = FIXED;
	op->f_class = 'S';
	op->f_RA = degrad (I4(u,0)*DPMAS);
	op->f_dec = degrad (I4(u,4)*DPMAS);
	op->f_epoch = J2000;
	set_fmag (op, I2(u,8)*0.01);
	if ((char)u[30] <= -27 && (char)u[31] <= -27) {
	    /* PM only store if "goodness of fit" is less than 5 */
	    op->f_pmRA =  degrad(I4(u,20)*0.1*DPMAS)/365.0;
	    op->f_pmdec = degrad(I4(u,24)*0.1*DPMAS)/365.0;
	}
	jkspect (I2(u,36)*0.001, I2(u,40)*0.001, op);

#undef	DPMAS
}

/* given the J and K_s fields in the UCAC catalog, fill in f_spect[] in op.
 * from Brian Skiff off TASS "Stellar Catalogs" page, with artistic license.
 * http://stupendous.rit.edu/tass/catalogs/uvby.calib
 */
static void
jkspect (double j, double k, Obj *op)
{
	static struct {
	    char sp[sizeof(op->f_spect)+1];	/* +1 for \0 in init string */
	    float jminusk;			/* j - k */
	} jkmag[] = {
	    {"B2",   -0.22},
	    {"B5",   -0.20},
	    {"B0",   -0.17},
	    {"A0",    0.00},
	    {"F0",    0.15},
	    {"G2",    0.37},
	    {"G8",    0.58},
	    {"K0",    0.63},
	    {"M0",    1.01},
	    {"M2",    1.22},
	    {"M5",    1.25},
	    {"M6",    1.26},
	    {"M7",    1.27},
	    {"V4",    1.57},
	};

	double jmk = j-k;
	int i;

	for (i = 1; i < sizeof(jkmag)/sizeof(jkmag[0]); i++) {
	    if (jmk <= jkmag[i].jminusk) {
		strncpy (op->f_spect, jkmag[i-1].sp, sizeof(op->f_spect));
		return;
	    }
	}
}



/* For RCS Only -- Do Not Edit */
static char *rcsid[2] = {(char *)rcsid, "@(#) $RCSfile: ucac.c,v $ $Date: 2007/04/24 19:56:51 $ $Revision: 1.4 $ $Name:  $"};
