/* api.c API unit tests
 *
 * Copyright (C) 2006-2023 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* For AES-CBC, input lengths can optionally be validated to be a
 * multiple of the block size, by defining WOLFSSL_AES_CBC_LENGTH_CHECKS,
 * also available via the configure option --enable-aescbc-length-checks.
 */


/*----------------------------------------------------------------------------*
 | Includes
 *----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>
#undef TEST_OPENSSL_COEXIST /* can't use this option with this example */

#ifndef FOURK_BUF
    #define FOURK_BUF 4096
#endif
#ifndef TWOK_BUF
    #define TWOK_BUF 2048
#endif
#ifndef ONEK_BUF
    #define ONEK_BUF 1024
#endif
#if defined(WOLFSSL_STATIC_MEMORY)
    #include <wolfssl/wolfcrypt/memory.h>
#endif /* WOLFSSL_STATIC_MEMORY */
#ifndef HEAP_HINT
    #define HEAP_HINT NULL
#endif /* WOLFSSL_STAIC_MEMORY */
#ifdef WOLFSSL_ASNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>   /* wc_ecc_fp_free */
    #ifndef ECC_ASN963_MAX_BUF_SZ
        #define ECC_ASN963_MAX_BUF_SZ 133
    #endif
    #ifndef ECC_PRIV_KEY_BUF
        #define ECC_PRIV_KEY_BUF 66  /* For non user defined curves. */
    #endif
    /* ecc key sizes: 14, 16, 20, 24, 28, 30, 32, 40, 48, 64 */
    /* logic to choose right key ECC size */
    #if (defined(HAVE_ECC112) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 112
        #define KEY14 14
    #else
        #define KEY14 32
    #endif
    #if (defined(HAVE_ECC128) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 128
        #define KEY16 16
    #else
        #define KEY16 32
    #endif
    #if (defined(HAVE_ECC160) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 160
        #define KEY20 20
    #else
        #define KEY20 32
    #endif
    #if (defined(HAVE_ECC192) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 192
        #define KEY24 24
    #else
        #define KEY24 32
    #endif
    #if defined(HAVE_ECC224) || defined(HAVE_ALL_CURVES)
        #define KEY28 28
    #else
        #define KEY28 32
    #endif
    #if defined(HAVE_ECC239) || defined(HAVE_ALL_CURVES)
        #define KEY30 30
    #else
        #define KEY30 32
    #endif
    #define KEY32 32
    #if defined(HAVE_ECC320) || defined(HAVE_ALL_CURVES)
        #define KEY40 40
    #else
        #define KEY40 32
    #endif
    #if defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES)
        #define KEY48 48
    #else
        #define KEY48 32
    #endif
    #if defined(HAVE_ECC512) || defined(HAVE_ALL_CURVES)
        #define KEY64 64
    #else
        #define KEY64 32
    #endif

    #if !defined(HAVE_COMP_KEY)
        #if !defined(NOCOMP)
            #define NOCOMP 0
        #endif
    #else
        #if !defined(COMP)
            #define COMP 1
        #endif
    #endif
    #if !defined(DER_SZ)
        #define DER_SZ(ks) ((ks) * 2 + 1)
    #endif
#endif
#ifndef NO_ASN
    #include <wolfssl/wolfcrypt/asn_public.h>
#endif
#include <wolfssl/error-ssl.h>

#include <stdlib.h>
#include <wolfssl/ssl.h>  /* compatibility layer */
#include <wolfssl/test.h>
#include <tests/unit.h>
#include "examples/server/server.h"
     /* for testing compatibility layer callbacks */

#ifndef NO_MD5
    #include <wolfssl/wolfcrypt/md5.h>
#endif
#ifndef NO_SHA
    #include <wolfssl/wolfcrypt/sha.h>
#endif
#ifndef NO_SHA256
    #include <wolfssl/wolfcrypt/sha256.h>
#endif
#ifdef WOLFSSL_SHA512
    #include <wolfssl/wolfcrypt/sha512.h>
#endif
#ifdef WOLFSSL_SHA384
    #include <wolfssl/wolfcrypt/sha512.h>
#endif

#ifdef WOLFSSL_SHA3
    #include <wolfssl/wolfcrypt/sha3.h>
    #ifndef HEAP_HINT
        #define HEAP_HINT   NULL
    #endif
#endif

#ifndef NO_AES
    #include <wolfssl/wolfcrypt/aes.h>
    #ifdef HAVE_AES_DECRYPT
        #include <wolfssl/wolfcrypt/wc_encrypt.h>
    #endif
#endif
#ifdef WOLFSSL_RIPEMD
    #include <wolfssl/wolfcrypt/ripemd.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/wolfcrypt/des3.h>
    #include <wolfssl/wolfcrypt/wc_encrypt.h>
#endif
#ifdef WC_RC2
    #include <wolfssl/wolfcrypt/rc2.h>
#endif

#ifndef NO_HMAC
    #include <wolfssl/wolfcrypt/hmac.h>
#endif

#ifdef HAVE_CHACHA
    #include <wolfssl/wolfcrypt/chacha.h>
#endif

#ifdef HAVE_POLY1305
    #include <wolfssl/wolfcrypt/poly1305.h>
#endif

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    #include <wolfssl/wolfcrypt/chacha20_poly1305.h>
#endif

#ifdef HAVE_CAMELLIA
    #include <wolfssl/wolfcrypt/camellia.h>
#endif

#ifndef NO_RC4
    #include <wolfssl/wolfcrypt/arc4.h>
#endif

#ifdef HAVE_BLAKE2
    #include <wolfssl/wolfcrypt/blake2.h>
#endif

#include <wolfssl/wolfcrypt/hash.h>
#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>

    #define FOURK_BUF 4096
    #define GEN_BUF  294

    #ifndef USER_CRYPTO_ERROR
        #define USER_CRYPTO_ERROR (-101) /* error returned by IPP lib. */
    #endif
#endif

#ifndef NO_SIG_WRAPPER
    #include <wolfssl/wolfcrypt/signature.h>
#endif


#ifdef HAVE_AESCCM
    #include <wolfssl/wolfcrypt/aes.h>
#endif

#ifdef HAVE_PKCS7
    #include <wolfssl/wolfcrypt/pkcs7.h>
    #include <wolfssl/wolfcrypt/asn.h>
    #ifdef HAVE_LIBZ
    #include <wolfssl/wolfcrypt/compress.h>
    #endif
#endif

#ifdef WOLFSSL_SMALL_CERT_VERIFY
    #include <wolfssl/wolfcrypt/asn.h>
#endif

#ifndef NO_DSA
    #include <wolfssl/wolfcrypt/dsa.h>
    #ifndef ONEK_BUF
        #define ONEK_BUF 1024
    #endif
    #ifndef TWOK_BUF
        #define TWOK_BUF 2048
    #endif
    #ifndef FOURK_BUF
        #define FOURK_BUF 4096
    #endif
    #ifndef DSA_SIG_SIZE
        #define DSA_SIG_SIZE 40
    #endif
    #ifndef MAX_DSA_PARAM_SIZE
        #define MAX_DSA_PARAM_SIZE 256
    #endif
#endif

#ifdef WOLFSSL_CMAC
    #include <wolfssl/wolfcrypt/cmac.h>
#endif

#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif

#ifdef HAVE_PKCS12
    #include <wolfssl/wolfcrypt/pkcs12.h>
#endif

#include <wolfssl/wolfcrypt/logging.h>

#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_ALL))
    #include <wolfssl/openssl/ssl.h>
    #ifndef NO_ASN
        /* for ASN_COMMON_NAME DN_tags enum */
        #include <wolfssl/wolfcrypt/asn.h>
    #endif
    #ifdef HAVE_OCSP
        #include <wolfssl/openssl/ocsp.h>
    #endif
#endif
#ifdef OPENSSL_EXTRA
    #include <wolfssl/openssl/cmac.h>
    #include <wolfssl/openssl/x509v3.h>
    #include <wolfssl/openssl/asn1.h>
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/pkcs12.h>
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/pem.h>
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/engine.h>
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/objects.h>
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/modes.h>
    #include <wolfssl/openssl/fips_rand.h>
    #include <wolfssl/openssl/kdf.h>
#ifdef OPENSSL_ALL
    #include <wolfssl/openssl/txt_db.h>
    #include <wolfssl/openssl/lhash.h>
#endif
#ifndef NO_AES
    #include <wolfssl/openssl/aes.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/openssl/des.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/openssl/ecdsa.h>
#endif
#ifdef HAVE_PKCS7
    #include <wolfssl/openssl/pkcs7.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/openssl/ed25519.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/openssl/ed448.h>
#endif
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
    && !defined(NO_SHA256) && !defined(RC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
#endif

#if (defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)) || \
    defined(HAVE_SESSION_TICKET) || (defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)) || \
    defined(WOLFSSL_TEST_STATIC_BUILD) || defined(WOLFSSL_DTLS) || \
    defined(HAVE_ECH)
    /* for testing SSL_get_peer_cert_chain, or SESSION_TICKET_HINT_DEFAULT,
     * for setting authKeyIdSrc in WOLFSSL_X509, or testing DTLS sequence
     * number tracking */
#include "wolfssl/internal.h"
#endif

/* force enable test buffers */
#ifndef USE_CERT_BUFFERS_2048
    #define USE_CERT_BUFFERS_2048
#endif
#ifndef USE_CERT_BUFFERS_256
    #define USE_CERT_BUFFERS_256
#endif
#include <wolfssl/certs_test.h>

typedef struct testVector {
    const char* input;
    const char* output;
    size_t inLen;
    size_t outLen;

} testVector;

#if defined(HAVE_PKCS7)
    typedef struct {
        const byte* content;
        word32      contentSz;
        int         contentOID;
        int         encryptOID;
        int         keyWrapOID;
        int         keyAgreeOID;
        byte*       cert;
        size_t      certSz;
        byte*       privateKey;
        word32      privateKeySz;
    } pkcs7EnvelopedVector;

    #ifndef NO_PKCS7_ENCRYPTED_DATA
        typedef struct {
            const byte*     content;
            word32          contentSz;
            int             contentOID;
            int             encryptOID;
            byte*           encryptionKey;
            word32          encryptionKeySz;
        } pkcs7EncryptedVector;
    #endif
#endif /* HAVE_PKCS7 */


/*----------------------------------------------------------------------------*
 | Constants
 *----------------------------------------------------------------------------*/

/* Test result constants and macros. */

/* Test succeeded. */
#define TEST_SUCCESS    (1)
/* Test failed. */
#define TEST_FAIL       (0)
/* Test skipped - not run. */
#define TEST_SKIPPED    (-7777)

/* Returns the result based on whether check is true.
 *
 * @param [in] check  Condition for success.
 * @return  When condition is true: TEST_SUCCESS.
 * @return  When condition is false: TEST_FAIL.
 */
#ifdef DEBUG_WOLFSSL_VERBOSE
#define XSTRINGIFY(s) STRINGIFY(s)
#define STRINGIFY(s)  #s
#define TEST_RES_CHECK(check) ({ \
    int _ret = (check) ? TEST_SUCCESS : TEST_FAIL; \
    if (_ret == TEST_FAIL) { \
        fprintf(stderr, " check \"%s\" at %d ", \
            XSTRINGIFY(check), __LINE__); \
    } \
    _ret; })
#else
#define TEST_RES_CHECK(check) \
    ((check) ? TEST_SUCCESS : TEST_FAIL)
#endif /* DEBUG_WOLFSSL_VERBOSE */

#define TEST_STRING    "Everyone gets Friday off."
#define TEST_STRING_SZ 25

#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
#define TEST_RSA_BITS 1024
#else
#define TEST_RSA_BITS 2048
#endif
#define TEST_RSA_BYTES (TEST_RSA_BITS/8)

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    (!defined(NO_WOLFSSL_SERVER) || !defined(NO_WOLFSSL_CLIENT))
    static const char* bogusFile  =
    #ifdef _WIN32
        "NUL"
    #else
        "/dev/null"
    #endif
    ;
#endif /* !NO_FILESYSTEM && !NO_CERTS && (!NO_WOLFSSL_SERVER || !NO_WOLFSSL_CLIENT) */

enum {
    TESTING_RSA = 1,
    TESTING_ECC = 2
};

#ifdef WOLFSSL_QNX_CAAM
#include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
static int testDevId = WOLFSSL_CAAM_DEVID;
#else
static int testDevId = INVALID_DEVID;
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_RSA)        && !defined(SINGLE_THREADED) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT)
#define HAVE_IO_TESTS_DEPENDENCIES
#endif

/*----------------------------------------------------------------------------*
 | Setup
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Init(void)
{
    int result;

    result = wolfSSL_Init();

    result = TEST_RES_CHECK(result == WOLFSSL_SUCCESS);
    return result;
}


static int test_wolfSSL_Cleanup(void)
{
    int result;

    result = wolfSSL_Cleanup();

    result = TEST_RES_CHECK(result == WOLFSSL_SUCCESS);
    return result;
}


/*  Initialize the wolfCrypt state.
 *  POST: 0 success.
 */
static int test_wolfCrypt_Init(void)
{
    int result;

    result = wolfCrypt_Init();

    result = TEST_RES_CHECK(result == 0);
    return result;

} /* END test_wolfCrypt_Init */

static int test_wolfCrypt_Cleanup(void)
{
    int result;

    result = wolfCrypt_Cleanup();

    result = TEST_RES_CHECK(result == 0);
    return result;
}

/*----------------------------------------------------------------------------*
 | Platform dependent function test
 *----------------------------------------------------------------------------*/
static int test_fileAccess(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_TEST_PLATFORMDEPEND) && !defined(NO_FILESYSTEM)
    const char *fname[] = {
    svrCertFile, svrKeyFile, caCertFile,
    eccCertFile, eccKeyFile, eccRsaCertFile,
    cliCertFile, cliCertDerFile, cliKeyFile,
    dhParamFile,
    cliEccKeyFile, cliEccCertFile, caEccCertFile, edCertFile, edKeyFile,
    cliEdCertFile, cliEdKeyFile, caEdCertFile,
    NULL
    };
     const char derfile[] = "./certs/server-cert.der";
    XFILE f;
    size_t sz;
    byte *buff;
    int i;

    AssertTrue(XFOPEN("badfilename", "rb") == XBADFILE);
    for (i=0; fname[i] != NULL ; i++) {
        AssertTrue((f = XFOPEN(fname[i], "rb")) != XBADFILE);
        XFCLOSE(f);
    }

    AssertTrue((f = XFOPEN(derfile, "rb")) != XBADFILE);
    AssertTrue(XFSEEK(f, 0, XSEEK_END) == 0);
    sz = (size_t) XFTELL(f);
    XREWIND(f);
    AssertTrue(sz == sizeof_server_cert_der_2048);
    AssertTrue((buff = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE)) != NULL) ;
    AssertTrue(XFREAD(buff, 1, sz, f) == sz);
    XMEMCMP(server_cert_der_2048, buff, sz);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

/*----------------------------------------------------------------------------*
 | Method Allocators
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Method_Allocators(void)
{
    #define TEST_METHOD_ALLOCATOR(allocator, condition) \
        do {                                            \
            WOLFSSL_METHOD *method;                      \
            condition(method = allocator());            \
            XFREE(method, 0, DYNAMIC_TYPE_METHOD);      \
        } while(0)

    #define TEST_VALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, AssertNotNull)

    #define TEST_INVALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, AssertNull)

#ifndef NO_OLD_TLS
    #ifdef WOLFSSL_ALLOW_SSLV3
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_client_method);
        #endif
    #endif
    #ifdef WOLFSSL_ALLOW_TLSV10
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_client_method);
        #endif
    #endif
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_client_method);
    #endif
#endif /* !NO_OLD_TLS */

#ifndef WOLFSSL_NO_TLS12
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_client_method);
    #endif
#endif /* !WOLFSSL_NO_TLS12 */

#ifdef WOLFSSL_TLS13
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_client_method);
    #endif
#endif /* WOLFSSL_TLS13 */

#ifndef NO_WOLFSSL_SERVER
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_server_method);
#endif
#ifndef NO_WOLFSSL_CLIENT
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_client_method);
#endif

#ifdef WOLFSSL_DTLS
    #ifndef NO_OLD_TLS
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_client_method);
        #endif
    #endif
    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_client_method);
        #endif
    #endif
#endif /* WOLFSSL_DTLS */

#if !defined(NO_OLD_TLS) && defined(OPENSSL_EXTRA)
    /* Stubs */
    #ifndef NO_WOLFSSL_SERVER
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_client_method);
    #endif
#endif

    /* Test Either Method (client or server) */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_method);
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
            TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_method);
        #endif
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_method);
    #endif /* !NO_OLD_TLS */
    #ifndef WOLFSSL_NO_TLS12
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_method);
    #endif /* !WOLFSSL_NO_TLS12 */
    #ifdef WOLFSSL_TLS13
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_method);
    #endif /* WOLFSSL_TLS13 */
    #ifdef WOLFSSL_DTLS
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLS_method);
        #ifndef NO_OLD_TLS
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_method);
        #endif /* !NO_OLD_TLS */
        #ifndef WOLFSSL_NO_TLS12
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_method);
        #endif /* !WOLFSSL_NO_TLS12 */
    #endif /* WOLFSSL_DTLS */
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

    return TEST_SUCCESS;
}


/*----------------------------------------------------------------------------*
 | Context
 *----------------------------------------------------------------------------*/
#ifndef NO_WOLFSSL_SERVER
static int test_wolfSSL_CTX_new(void)
{
    WOLFSSL_CTX *ctx;
    WOLFSSL_METHOD* method;

    AssertNull(ctx = wolfSSL_CTX_new(NULL));
    AssertNotNull(method = wolfSSLv23_server_method());
    AssertNotNull(ctx = wolfSSL_CTX_new(method));

    wolfSSL_CTX_free(ctx);

    return TEST_RES_CHECK(1);
}
#endif

#if (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM)
static int test_for_double_Free(void)
{
    WOLFSSL_CTX* ctx;
    WOLFSSL*     ssl;
    int skipTest = 0;
    const char* testCertFile;
    const char* testKeyFile;
    char optionsCiphers[] = "RC4-SHA:RC4-MD5:DES-CBC3-SHA:AES128-SHA:AES256-SHA"
":NULL-SHA:NULL-SHA256:DHE-RSA-AES128-SHA:DHE-RSA-AES256-SHA:DHE-PSK-AES256-GCM"
"-SHA384:DHE-PSK-AES128-GCM-SHA256:PSK-AES256-GCM-SHA384:PSK-AES128-GCM-SHA256:"
"DHE-PSK-AES256-CBC-SHA384:DHE-PSK-AES128-CBC-SHA256:PSK-AES256-CBC-SHA384:PSK-"
"AES128-CBC-SHA256:PSK-AES128-CBC-SHA:PSK-AES256-CBC-SHA:DHE-PSK-AES128-CCM:DHE"
"-PSK-AES256-CCM:PSK-AES128-CCM:PSK-AES256-CCM:PSK-AES128-CCM-8:PSK-AES256-CCM-"
"8:DHE-PSK-NULL-SHA384:DHE-PSK-NULL-SHA256:PSK-NULL-SHA384:PSK-NULL-SHA256:PSK-"
"NULL-SHA:AES128-CCM-8:AES256-CCM-8:ECDHE-ECDSA-"
"AES128-CCM:ECDHE-ECDSA-AES128-CCM-8:ECDHE-ECDSA-AES256-CCM-8:ECDHE-RSA-AES128-"
"SHA:ECDHE-RSA-AES256-SHA:ECDHE-ECDSA-AES128-SHA:ECDHE-ECDSA-AES256-SHA:ECDHE-R"
"SA-RC4-SHA:ECDHE-RSA-DES-CBC3-SHA:ECDHE-ECDSA-RC4-SHA:ECDHE-ECDSA-DES-CBC3-SHA"
":AES128-SHA256:AES256-SHA256:DHE-RSA-AES128-SHA256:DHE-RSA-AES256-SHA256:ECDH-"
"RSA-AES128-SHA:ECDH-RSA-AES256-SHA:ECDH-ECDSA-AES128-SHA:ECDH-ECDSA-AES256-SHA"
":ECDH-RSA-RC4-SHA:ECDH-RSA-DES-CBC3-SHA:ECDH-ECDSA-RC4-SHA:ECDH-ECDSA-DES-CBC3"
"-SHA:AES128-GCM-SHA256:AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES"
"256-GCM-SHA384:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-E"
"CDSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDH-RSA-AES128-GCM-SHA25"
"6:ECDH-RSA-AES256-GCM-SHA384:ECDH-ECDSA-AES128-GCM-SHA256:ECDH-ECDSA-AES256-GC"
"M-SHA384:CAMELLIA128-SHA:DHE-RSA-CAMELLIA128-SHA:CAMELLIA256-SHA:DHE-RSA-CAMEL"
"LIA256-SHA:CAMELLIA128-SHA256:DHE-RSA-CAMELLIA128-SHA256:CAMELLIA256-SHA256:DH"
"E-RSA-CAMELLIA256-SHA256:ECDHE-RSA-AES128-SHA256:ECDHE-ECDSA-AES128-SHA256:ECD"
"H-RSA-AES128-SHA256:ECDH-ECDSA-AES128-SHA256:ECDHE-RSA-AES256-SHA384:ECDHE-ECD"
"SA-AES256-SHA384:ECDH-RSA-AES256-SHA384:ECDH-ECDSA-AES256-SHA384:ECDHE-RSA-CHA"
"CHA20-POLY1305:ECDHE-ECDSA-CHACHA20-POLY1305:DHE-RSA-CHACHA20-POLY1305:ECDHE-R"
"SA-CHACHA20-POLY1305-OLD:ECDHE-ECDSA-CHACHA20-POLY1305-OLD:DHE-RSA-CHACHA20-PO"
"LY1305-OLD:ECDHE-ECDSA-NULL-SHA:ECDHE-PSK-NULL-SHA256:ECDHE-PSK-A"
"ES128-CBC-SHA256:PSK-CHACHA20-POLY1305:ECDHE-PSK-CHACHA20-POLY1305:DHE-PSK-CHA"
"CHA20-POLY1305:EDH-RSA-DES-CBC3-SHA:TLS13-AES128-GCM-SHA256:TLS13-AES256-GCM-S"
"HA384:TLS13-CHACHA20-POLY1305-SHA256:TLS13-AES128-CCM-SHA256:TLS13-AES128-CCM-"
"8-SHA256:TLS13-SHA256-SHA256:TLS13-SHA384-SHA384";
    /* OpenVPN uses a "blacklist" method to specify which ciphers NOT to use */
#ifdef OPENSSL_EXTRA
    char openvpnCiphers[] = "DEFAULT:!EXP:!LOW:!MEDIUM:!kDH:!kECDH:!DSS:!PSK:"
                            "!SRP:!kRSA:!aNULL:!eNULL";
#endif

#ifndef NO_RSA
        testCertFile = svrCertFile;
        testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
        testCertFile = eccCertFile;
        testKeyFile = eccKeyFile;
#else
    skipTest = 1;
#endif

    if (skipTest != 1) {
#ifndef NO_WOLFSSL_SERVER
        ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
        AssertNotNull(ctx);
#else
        ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
        AssertNotNull(ctx);
#endif
        AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile, WOLFSSL_FILETYPE_PEM));
        AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile, WOLFSSL_FILETYPE_PEM));
        ssl = wolfSSL_new(ctx);
        AssertNotNull(ssl);

        /* First test freeing SSL, then CTX */
        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);

#ifndef NO_WOLFSSL_CLIENT
        ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
        AssertNotNull(ctx);
#else
        ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
        AssertNotNull(ctx);
#endif
        AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile, WOLFSSL_FILETYPE_PEM));
        AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile, WOLFSSL_FILETYPE_PEM));
        ssl = wolfSSL_new(ctx);
        AssertNotNull(ssl);

        /* Next test freeing CTX then SSL */
        wolfSSL_CTX_free(ctx);
        wolfSSL_free(ssl);

#ifndef NO_WOLFSSL_SERVER
        ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
        AssertNotNull(ctx);
#else
        ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
        AssertNotNull(ctx);
#endif
        /* Test setting ciphers at ctx level */
        AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile, WOLFSSL_FILETYPE_PEM));
        AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile, WOLFSSL_FILETYPE_PEM));
        AssertTrue(wolfSSL_CTX_set_cipher_list(ctx, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        AssertTrue(wolfSSL_CTX_set_cipher_list(ctx, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        AssertTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
#ifdef OPENSSL_EXTRA
        AssertTrue(wolfSSL_CTX_set_cipher_list(ctx, openvpnCiphers));
#endif
        AssertNotNull(ssl = wolfSSL_new(ctx));
        wolfSSL_CTX_free(ctx);
        wolfSSL_free(ssl);

#ifndef NO_WOLFSSL_CLIENT
        ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
        AssertNotNull(ctx);
#else
        ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
        AssertNotNull(ctx);
#endif
        AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile, WOLFSSL_FILETYPE_PEM));
        AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile, WOLFSSL_FILETYPE_PEM));
        ssl = wolfSSL_new(ctx);
        AssertNotNull(ssl);
        /* test setting ciphers at SSL level */
        AssertTrue(wolfSSL_set_cipher_list(ssl, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        AssertTrue(wolfSSL_set_cipher_list(ssl, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        AssertTrue(wolfSSL_set_cipher_list(ssl, "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
        wolfSSL_CTX_free(ctx);
        wolfSSL_free(ssl);
    }

    return TEST_RES_CHECK(1);
}
#endif


static int test_wolfSSL_CTX_set_cipher_list_bytes(void)
{
    int res = TEST_SKIPPED;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_SET_CIPHER_BYTES)) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC))
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX* ctx;
    WOLFSSL* ssl;

    const byte cipherList[] =
    {
        /* TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA */ 0xC0, 0x16,
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA  */ 0xC0, 0x39,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA  */ 0xC0, 0x33,
        /* TLS_DH_anon_WITH_AES_128_CBC_SHA  */ 0xC0, 0x34,
        /* TLS_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x35,
        /* TLS_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x2F,
        /* TLS_RSA_WITH_NULL_MD5             */ 0xC0, 0x01,
        /* TLS_RSA_WITH_NULL_SHA             */ 0xC0, 0x02,
        /* TLS_PSK_WITH_AES_256_CBC_SHA      */ 0xC0, 0x8d,
        /* TLS_PSK_WITH_AES_128_CBC_SHA256   */ 0xC0, 0xae,
        /* TLS_PSK_WITH_AES_256_CBC_SHA384   */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CBC_SHA      */ 0xC0, 0x8c,
        /* TLS_PSK_WITH_NULL_SHA256          */ 0xC0, 0xb0,
        /* TLS_PSK_WITH_NULL_SHA384          */ 0xC0, 0xb1,
        /* TLS_PSK_WITH_NULL_SHA             */ 0xC0, 0x2c,
        /* SSL_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x05,
        /* SSL_RSA_WITH_RC4_128_MD5          */ 0xC0, 0x04,
        /* SSL_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0A,

        /* ECC suites, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA     */ 0xC0, 0x14,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA     */ 0xC0, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA   */ 0xC0, 0x0A,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA   */ 0xC0, 0x09,
        /* TLS_ECDHE_RSA_WITH_RC4_128_SHA         */ 0xC0, 0x11,
        /* TLS_ECDHE_ECDSA_WITH_RC4_128_SHA       */ 0xC0, 0x07,
        /* TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA    */ 0xC0, 0x12,
        /* TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA  */ 0xC0, 0x08,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x27,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256*/ 0xC0, 0x23,
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384  */ 0xC0, 0x28,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384*/ 0xC0, 0x24,
        /* TLS_ECDHE_ECDSA_WITH_NULL_SHA          */ 0xC0, 0x06,
        /* TLS_ECDHE_PSK_WITH_NULL_SHA256         */ 0xC0, 0x3a,
        /* TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x37,

        /* static ECDH, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x0F,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x0E,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA    */ 0xC0, 0x05,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA    */ 0xC0, 0x04,
        /* TLS_ECDH_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x0C,
        /* TLS_ECDH_ECDSA_WITH_RC4_128_SHA        */ 0xC0, 0x02,
        /* TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0D,
        /* TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA   */ 0xC0, 0x03,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256   */ 0xC0, 0x29,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 */ 0xC0, 0x25,
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384   */ 0xC0, 0x2A,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 */ 0xC0, 0x26,

        /* WDM_WITH_NULL_SHA256 */ 0x00, 0xFE, /* wolfSSL DTLS Multicast */

        /* SHA256 */
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 */ 0x00, 0x6b,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 */ 0x00, 0x67,
        /* TLS_RSA_WITH_AES_256_CBC_SHA256     */ 0x00, 0x3d,
        /* TLS_RSA_WITH_AES_128_CBC_SHA256     */ 0x00, 0x3c,
        /* TLS_RSA_WITH_NULL_SHA256            */ 0x00, 0x3b,
        /* TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 */ 0x00, 0xb2,
        /* TLS_DHE_PSK_WITH_NULL_SHA256        */ 0x00, 0xb4,

        /* SHA384 */
        /* TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 */ 0x00, 0xb3,
        /* TLS_DHE_PSK_WITH_NULL_SHA384        */ 0x00, 0xb5,

        /* AES-GCM */
        /* TLS_RSA_WITH_AES_128_GCM_SHA256      */ 0x00, 0x9c,
        /* TLS_RSA_WITH_AES_256_GCM_SHA384      */ 0x00, 0x9d,
        /* TLS_DHE_RSA_WITH_AES_128_GCM_SHA256  */ 0x00, 0x9e,
        /* TLS_DHE_RSA_WITH_AES_256_GCM_SHA384  */ 0x00, 0x9f,
        /* TLS_DH_anon_WITH_AES_256_GCM_SHA384  */ 0x00, 0xa7,
        /* TLS_PSK_WITH_AES_128_GCM_SHA256      */ 0x00, 0xa8,
        /* TLS_PSK_WITH_AES_256_GCM_SHA384      */ 0x00, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_GCM_SHA256  */ 0x00, 0xaa,
        /* TLS_DHE_PSK_WITH_AES_256_GCM_SHA384  */ 0x00, 0xab,

        /* ECC AES-GCM, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 */ 0xC0, 0x2b,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 */ 0xC0, 0x2c,
        /* TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256  */ 0xC0, 0x2d,
        /* TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384  */ 0xC0, 0x2e,
        /* TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256   */ 0xC0, 0x2f,
        /* TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384   */ 0xC0, 0x30,
        /* TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256    */ 0xC0, 0x31,
        /* TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384    */ 0xC0, 0x32,

        /* AES-CCM, first byte is 0xC0 but isn't ECC,
         * also, in some of the other AES-CCM suites
         * there will be second byte number conflicts
         * with non-ECC AES-GCM */
        /* TLS_RSA_WITH_AES_128_CCM_8         */ 0xC0, 0xa0,
        /* TLS_RSA_WITH_AES_256_CCM_8         */ 0xC0, 0xa1,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM   */ 0xC0, 0xac,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 */ 0xC0, 0xae,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CCM           */ 0xC0, 0xa4,
        /* TLS_PSK_WITH_AES_256_CCM           */ 0xC0, 0xa5,
        /* TLS_PSK_WITH_AES_128_CCM_8         */ 0xC0, 0xa8,
        /* TLS_PSK_WITH_AES_256_CCM_8         */ 0xC0, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_CCM       */ 0xC0, 0xa6,
        /* TLS_DHE_PSK_WITH_AES_256_CCM       */ 0xC0, 0xa7,

        /* Camellia */
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA        */ 0x00, 0x41,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA        */ 0x00, 0x84,
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256     */ 0x00, 0xba,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256     */ 0x00, 0xc0,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA    */ 0x00, 0x45,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA    */ 0x00, 0x88,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 */ 0x00, 0xbe,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 */ 0x00, 0xc4,

        /* chacha20-poly1305 suites first byte is 0xCC (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xa8,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 */ 0xCC, 0xa9,
        /* TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xaa,
        /* TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xac,
        /* TLS_PSK_WITH_CHACHA20_POLY1305_SHA256         */ 0xCC, 0xab,
        /* TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xad,

        /* chacha20-poly1305 earlier version of nonce and padding (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256   */ 0xCC, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_OLD_POLY1305_SHA256 */ 0xCC, 0x14,
        /* TLS_DHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256     */ 0xCC, 0x15,

        /* ECDHE_PSK RFC8442, first byte is 0xD0 (ECDHE_PSK_BYTE) */
        /* TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 */ 0xD0, 0x01,

        /* TLS v1.3 cipher suites */
        /* TLS_AES_128_GCM_SHA256       */ 0x13, 0x01,
        /* TLS_AES_256_GCM_SHA384       */ 0x13, 0x02,
        /* TLS_CHACHA20_POLY1305_SHA256 */ 0x13, 0x03,
        /* TLS_AES_128_CCM_SHA256       */ 0x13, 0x04,
        /* TLS_AES_128_CCM_8_SHA256     */ 0x13, 0x05,

        /* TLS v1.3 Integrity only cipher suites - 0xC0 (ECC) first byte */
        /* TLS_SHA256_SHA256 */ 0xC0, 0xB4,
        /* TLS_SHA384_SHA384 */ 0xC0, 0xB5
    };

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

#ifndef NO_WOLFSSL_SERVER
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);
#endif

    AssertTrue(wolfSSL_CTX_set_cipher_list_bytes(ctx, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_CTX_free(ctx);

#ifndef NO_WOLFSSL_SERVER
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);
#endif

    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    AssertTrue(wolfSSL_set_cipher_list_bytes(ssl, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* (OPENSSL_EXTRA || WOLFSSL_SET_CIPHER_BYTES) &&
    (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) && (!NO_RSA || HAVE_ECC) */

    return res;
}


static int test_wolfSSL_CTX_use_certificate_file(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    AssertFalse(wolfSSL_CTX_use_certificate_file(NULL, svrCertFile,
                                                             WOLFSSL_FILETYPE_PEM));
    /* invalid cert file */
    AssertFalse(wolfSSL_CTX_use_certificate_file(ctx, bogusFile,
                                                             WOLFSSL_FILETYPE_PEM));
    /* invalid cert type */
    AssertFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, 9999));

#ifdef NO_RSA
    /* rsa needed */
    AssertFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO)) && !defined(NO_RSA)
static int test_wolfSSL_CTX_use_certificate_ASN1(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER) && !defined(NO_ASN)
    WOLFSSL_CTX*            ctx;
    int                     ret;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ret = SSL_CTX_use_certificate_ASN1(ctx, sizeof_server_cert_der_2048,
                server_cert_der_2048);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
#endif
    return res;
}
#endif /* (OPENSSL_ALL || WOLFSSL_ASIO) && !NO_RSA */

/*  Test function for wolfSSL_CTX_use_certificate_buffer. Load cert into
 *  context using buffer.
 *  PRE: NO_CERTS not defined; USE_CERT_BUFFERS_2048 defined; compile with
 *  --enable-testcert flag.
 */
static int test_wolfSSL_CTX_use_certificate_buffer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && defined(USE_CERT_BUFFERS_2048) && \
        !defined(NO_RSA) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx;
    int          ret;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ret = wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
#endif
    return res;

} /*END test_wolfSSL_CTX_use_certificate_buffer*/

static int test_wolfSSL_CTX_use_PrivateKey_file(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    AssertFalse(wolfSSL_CTX_use_PrivateKey_file(NULL, svrKeyFile,
                                                             WOLFSSL_FILETYPE_PEM));
    /* invalid key file */
    AssertFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, bogusFile,
                                                             WOLFSSL_FILETYPE_PEM));
    /* invalid key type */
    AssertFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, 9999));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    AssertFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}


/* test both file and buffer versions along with unloading trusted peer certs */
static int test_wolfSSL_CTX_trust_peer_cert(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && defined(WOLFSSL_TRUST_PEER_CERT) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX *ctx;
    WOLFSSL* ssl;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    AssertNotNull(ssl = wolfSSL_new(ctx));

#if !defined(NO_FILESYSTEM)
    /* invalid file */
    AssertIntNE(wolfSSL_CTX_trust_peer_cert(ctx, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    AssertIntNE(wolfSSL_CTX_trust_peer_cert(ctx, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    AssertIntNE(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    AssertIntEQ(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* unload cert */
    AssertIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    /* invalid file */
    AssertIntNE(wolfSSL_trust_peer_cert(ssl, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    AssertIntNE(wolfSSL_trust_peer_cert(ssl, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    AssertIntNE(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    AssertIntEQ(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    #ifdef WOLFSSL_LOCAL_X509_STORE
    /* unload cert */
    AssertIntNE(wolfSSL_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_Unload_trust_peers(ssl), WOLFSSL_SUCCESS);
    #endif
#endif

    /* Test of loading certs from buffers */

    /* invalid buffer */
    AssertIntNE(wolfSSL_CTX_trust_peer_buffer(ctx, NULL, -1,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
#ifdef USE_CERT_BUFFERS_1024
    AssertIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_1024,
          sizeof_client_cert_der_1024, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
#ifdef USE_CERT_BUFFERS_2048
    AssertIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_2048,
          sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* unload cert */
    AssertIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}


static int test_wolfSSL_CTX_load_verify_locations(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx;
#ifndef NO_RSA
    WOLFSSL_CERT_MANAGER* cm;
#ifdef PERSIST_CERT_CACHE
    int cacheSz;
#endif
#endif
#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    const char* load_certs_path = "./certs/external";
    const char* load_no_certs_path = "./examples";
    const char* load_expired_path = "./certs/test/expired";
#endif

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* invalid arguments */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(NULL, caCertFile, NULL), WOLFSSL_FAILURE);
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, NULL), WOLFSSL_FAILURE);

    /* invalid ca file */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, bogusFile, NULL),
     WS_RETURN_CODE(WOLFSSL_BAD_FILE,WOLFSSL_FAILURE));


#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
  (defined(WOLFSSL_QT) && \
  !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR))
    /* invalid path */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, bogusFile),
     WS_RETURN_CODE(BAD_PATH_ERROR,WOLFSSL_FAILURE));
#endif

    /* load ca cert */
#ifdef NO_RSA
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
     WS_RETURN_CODE(ASN_UNKNOWN_OID_E,WOLFSSL_FAILURE));
#else /* Skip the following test without RSA certs. */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Get cert cache size */
    cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx);
#endif
    /* Test unloading CA's */
    AssertIntEQ(wolfSSL_CTX_UnloadCAs(ctx), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    AssertIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif

    /* load ca cert again */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL), WOLFSSL_SUCCESS);

    /* Test getting CERT_MANAGER */
    AssertNotNull(cm = wolfSSL_CTX_GetCertManager(ctx));

    /* Test unloading CA's using CM */
    AssertIntEQ(wolfSSL_CertManagerUnloadCAs(cm), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    AssertIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif
#endif

#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    /* Test loading CA certificates using a path */
    #ifdef NO_RSA
    /* failure here okay since certs in external directory are RSA */
    AssertIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #else
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #endif

    /* Test loading path with no files */
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_no_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_FAILURE);

    /* Test loading expired CA certificates */
    #ifdef NO_RSA
    AssertIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #else
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #endif

    /* Test loading CA certificates and ignoring all errors */
    #ifdef NO_RSA
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_FAILURE);
    #else
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_SUCCESS);
    #endif
#endif

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_load_system_CA_certs(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SYS_CA_CERTS) && !defined(NO_WOLFSSL_CLIENT) && \
    (!defined(NO_RSA) || defined(HAVE_ECC))
    WOLFSSL_CTX* ctx;
    byte dirValid = 0;
    int ret = 0;

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    if (ctx == NULL) {
        fprintf(stderr, "wolfSSL_CTX_new failed.\n");
        ret = -1;
    }
    if (ret == 0) {
    #if defined(USE_WINDOWS_API) || defined(__APPLE__)
        dirValid = 1;
    #else
        word32 numDirs;
        const char** caDirs = wolfSSL_get_system_CA_dirs(&numDirs);

        if (caDirs == NULL || numDirs == 0) {
            fprintf(stderr, "wolfSSL_get_system_CA_dirs failed.\n");
            ret = -1;
        }
        else {
            ReadDirCtx dirCtx;
            word32 i;

            for (i = 0; i < numDirs; ++i) {
                if (wc_ReadDirFirst(&dirCtx, caDirs[i], NULL) == 0) {
                    /* Directory isn't empty. */
                    dirValid = 1;
                    wc_ReadDirClose(&dirCtx);
                    break;
                }
            }
        }
    #endif
    }
    /*
     * If the directory isn't empty, we should be able to load CA
     * certs from it. On Windows/Mac, we assume the CA cert stores are
     * usable.
     */
    if (ret == 0 && dirValid && wolfSSL_CTX_load_system_CA_certs(ctx) !=
        WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_load_system_CA_certs failed.\n");
        ret = -1;
    }
#ifdef OPENSSL_EXTRA
    if (ret == 0 &&
        wolfSSL_CTX_set_default_verify_paths(ctx) != WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_set_default_verify_paths failed.\n");
        ret = -1;
    }
#endif /* OPENSSL_EXTRA */

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SYS_CA_CERTS && !NO_WOLFSSL_CLIENT */

    return res;
}

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
static int test_cm_load_ca_buffer(const byte* cert_buf, size_t cert_sz, int file_type)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer(cm, cert_buf, cert_sz, file_type);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file(const char* ca_cert_file)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer(cert_buf, cert_sz, WOLFSSL_FILETYPE_PEM);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, cert_sz, CA_TYPE, &pDer, NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

static int test_cm_load_ca_buffer_ex(const byte* cert_buf, size_t cert_sz,
                                     int file_type, word32 flags)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer_ex(cm, cert_buf, cert_sz, file_type,
                                             0, flags);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file_ex(const char* ca_cert_file, word32 flags)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz,
                                        WOLFSSL_FILETYPE_PEM, flags);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM, flags);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, cert_sz, CA_TYPE, &pDer, NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer_ex(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1, flags);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

#endif /* !NO_FILESYSTEM && !NO_CERTS */

static int test_wolfSSL_CertManagerCheckOCSPResponse(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_OCSP) && !defined(NO_RSA)
/* Need one of these for wolfSSL_OCSP_REQUEST_new. */
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_APACHE_HTTPD) || \
    defined(HAVE_LIGHTY)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    /* Raw OCSP response bytes captured using the following setup:
     * - Run responder with
     *      openssl ocsp -port 9999 -ndays 9999
     *      -index certs/ocsp/index-intermediate1-ca-issued-certs.txt
     *      -rsigner certs/ocsp/ocsp-responder-cert.pem
     *      -rkey certs/ocsp/ocsp-responder-key.pem
     *      -CA certs/ocsp/intermediate1-ca-cert.pem
     * - Run client with
     *      openssl ocsp -host 127.0.0.1:9999 -respout resp.out
     *      -issuer certs/ocsp/intermediate1-ca-cert.pem
     *      -cert certs/ocsp/server1-cert.pem
     *      -CAfile certs/ocsp/root-ca-cert.pem -noverify
     * - Copy raw response from Wireshark.
     */
    byte response[] = {
        0x30, 0x82, 0x07, 0x40, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x07, 0x39, 0x30, 0x82, 0x07, 0x35, 0x06,
        0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x07, 0x26, 0x30, 0x82,
        0x07, 0x22, 0x30, 0x82, 0x01, 0x40, 0xa1, 0x81, 0xa1, 0x30, 0x81, 0x9e, 0x31, 0x0b, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10,
        0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65,
        0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53,
        0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67,
        0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04,
        0x03, 0x0c, 0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, 0x43, 0x53, 0x50, 0x20,
        0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a,
        0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77,
        0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31,
        0x30, 0x35, 0x30, 0x33, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30, 0x5a, 0x30, 0x64, 0x30, 0x62, 0x30,
        0x3a, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x71, 0x4d,
        0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18,
        0xda, 0x04, 0x04, 0x14, 0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2,
        0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e, 0x02, 0x01, 0x05, 0x80, 0x00, 0x18, 0x0f, 0x32,
        0x30, 0x32, 0x31, 0x30, 0x35, 0x30, 0x33, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30, 0x5a, 0xa0, 0x11,
        0x18, 0x0f, 0x32, 0x30, 0x34, 0x38, 0x30, 0x39, 0x31, 0x37, 0x32, 0x31, 0x34, 0x37, 0x31, 0x30,
        0x5a, 0xa1, 0x23, 0x30, 0x21, 0x30, 0x1f, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
        0x01, 0x02, 0x04, 0x12, 0x04, 0x10, 0x38, 0x31, 0x60, 0x99, 0xc8, 0x05, 0x09, 0x68, 0x1c, 0x33,
        0x49, 0xea, 0x45, 0x26, 0x2f, 0x6d, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
        0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x4d, 0x58, 0xcc, 0x69, 0x42, 0xe2,
        0x9e, 0x64, 0xf6, 0x57, 0xce, 0xcb, 0x5f, 0x14, 0xaf, 0x08, 0x6c, 0xc1, 0x52, 0x7a, 0x40, 0x0a,
        0xfd, 0xb6, 0xce, 0xbb, 0x40, 0xf4, 0xb9, 0xa5, 0x88, 0xc7, 0xf3, 0x42, 0x9f, 0xa9, 0x94, 0xbe,
        0x6e, 0x7e, 0x09, 0x30, 0x9d, 0x0e, 0x10, 0x6f, 0x9c, 0xd9, 0x4c, 0x71, 0x81, 0x41, 0x64, 0x95,
        0xf5, 0x85, 0x77, 0x94, 0x81, 0x61, 0x88, 0xc8, 0x0b, 0x50, 0xbb, 0x37, 0xc8, 0x86, 0x76, 0xd8,
        0xa2, 0xed, 0x66, 0x34, 0xfb, 0xe4, 0xe7, 0x09, 0x8c, 0xf5, 0xb5, 0x85, 0xd0, 0x4b, 0xb5, 0xe6,
        0x23, 0x62, 0xc3, 0xd0, 0xef, 0xf7, 0x42, 0x89, 0x02, 0x80, 0x64, 0xc9, 0xed, 0xdd, 0x7c, 0x8f,
        0x0d, 0xe7, 0x43, 0x9b, 0x88, 0x1f, 0xb0, 0xfd, 0x24, 0x01, 0xc7, 0x55, 0xc3, 0x73, 0x12, 0x84,
        0x09, 0x7c, 0x57, 0xa8, 0x5d, 0xab, 0x75, 0x29, 0x5c, 0x36, 0x97, 0x64, 0x40, 0x0b, 0x55, 0x34,
        0x0a, 0x5d, 0xb1, 0x1b, 0x61, 0x1b, 0xdc, 0xe5, 0x89, 0xdd, 0x92, 0x62, 0x57, 0xa7, 0x52, 0xb4,
        0x38, 0x9a, 0x48, 0xc8, 0x3a, 0x14, 0xde, 0x69, 0x42, 0xe9, 0x37, 0xa4, 0xe7, 0x2d, 0x00, 0xa7,
        0x0b, 0x29, 0x18, 0xd5, 0xce, 0xd9, 0x0d, 0xdd, 0xfe, 0xae, 0x86, 0xb3, 0x32, 0x1c, 0xc9, 0x33,
        0xb0, 0x2b, 0xb7, 0x3c, 0x0d, 0x43, 0xd8, 0x6c, 0xf2, 0xb7, 0xcd, 0x7b, 0xd5, 0x7d, 0xf0, 0xde,
        0x34, 0x9f, 0x6d, 0x83, 0xb9, 0xd5, 0xed, 0xe3, 0xda, 0x96, 0x40, 0x9e, 0xd6, 0xa6, 0xfd, 0x70,
        0x80, 0x70, 0x87, 0x61, 0x0f, 0xc5, 0x9f, 0x75, 0xfe, 0x11, 0x78, 0x34, 0xc9, 0x42, 0x16, 0x73,
        0x46, 0x7b, 0x05, 0x53, 0x28, 0x43, 0xbe, 0xee, 0x88, 0x67, 0x1d, 0xcc, 0x74, 0xa7, 0xb6, 0x58,
        0x7b, 0x29, 0x68, 0x40, 0xcf, 0xce, 0x7b, 0x19, 0x33, 0x68, 0xa0, 0x82, 0x04, 0xc6, 0x30, 0x82,
        0x04, 0xc2, 0x30, 0x82, 0x04, 0xbe, 0x30, 0x82, 0x03, 0xa6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
        0x01, 0x04, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
        0x00, 0x30, 0x81, 0x97, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68,
        0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c,
        0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04,
        0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03,
        0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, 0x53,
        0x53, 0x4c, 0x20, 0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09,
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40,
        0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x32,
        0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x39, 0x34, 0x39, 0x35, 0x34, 0x5a, 0x17, 0x0d, 0x32, 0x33,
        0x31, 0x31, 0x30, 0x37, 0x31, 0x39, 0x34, 0x39, 0x35, 0x34, 0x5a, 0x30, 0x81, 0x9e, 0x31, 0x0b,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06,
        0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e,
        0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74,
        0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c,
        0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45,
        0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0c, 0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, 0x43, 0x53,
        0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, 0x1d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f,
        0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
        0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xb8, 0xba, 0x23,
        0xb4, 0xf6, 0xc3, 0x7b, 0x14, 0xc3, 0xa4, 0xf5, 0x1d, 0x61, 0xa1, 0xf5, 0x1e, 0x63, 0xb9, 0x85,
        0x23, 0x34, 0x50, 0x6d, 0xf8, 0x7c, 0xa2, 0x8a, 0x04, 0x8b, 0xd5, 0x75, 0x5c, 0x2d, 0xf7, 0x63,
        0x88, 0xd1, 0x07, 0x7a, 0xea, 0x0b, 0x45, 0x35, 0x2b, 0xeb, 0x1f, 0xb1, 0x22, 0xb4, 0x94, 0x41,
        0x38, 0xe2, 0x9d, 0x74, 0xd6, 0x8b, 0x30, 0x22, 0x10, 0x51, 0xc5, 0xdb, 0xca, 0x3f, 0x46, 0x2b,
        0xfe, 0xe5, 0x5a, 0x3f, 0x41, 0x74, 0x67, 0x75, 0x95, 0xa9, 0x94, 0xd5, 0xc3, 0xee, 0x42, 0xf8,
        0x8d, 0xeb, 0x92, 0x95, 0xe1, 0xd9, 0x65, 0xb7, 0x43, 0xc4, 0x18, 0xde, 0x16, 0x80, 0x90, 0xce,
        0x24, 0x35, 0x21, 0xc4, 0x55, 0xac, 0x5a, 0x51, 0xe0, 0x2e, 0x2d, 0xb3, 0x0a, 0x5a, 0x4f, 0x4a,
        0x73, 0x31, 0x50, 0xee, 0x4a, 0x16, 0xbd, 0x39, 0x8b, 0xad, 0x05, 0x48, 0x87, 0xb1, 0x99, 0xe2,
        0x10, 0xa7, 0x06, 0x72, 0x67, 0xca, 0x5c, 0xd1, 0x97, 0xbd, 0xc8, 0xf1, 0x76, 0xf8, 0xe0, 0x4a,
        0xec, 0xbc, 0x93, 0xf4, 0x66, 0x4c, 0x28, 0x71, 0xd1, 0xd8, 0x66, 0x03, 0xb4, 0x90, 0x30, 0xbb,
        0x17, 0xb0, 0xfe, 0x97, 0xf5, 0x1e, 0xe8, 0xc7, 0x5d, 0x9b, 0x8b, 0x11, 0x19, 0x12, 0x3c, 0xab,
        0x82, 0x71, 0x78, 0xff, 0xae, 0x3f, 0x32, 0xb2, 0x08, 0x71, 0xb2, 0x1b, 0x8c, 0x27, 0xac, 0x11,
        0xb8, 0xd8, 0x43, 0x49, 0xcf, 0xb0, 0x70, 0xb1, 0xf0, 0x8c, 0xae, 0xda, 0x24, 0x87, 0x17, 0x3b,
        0xd8, 0x04, 0x65, 0x6c, 0x00, 0x76, 0x50, 0xef, 0x15, 0x08, 0xd7, 0xb4, 0x73, 0x68, 0x26, 0x14,
        0x87, 0x95, 0xc3, 0x5f, 0x6e, 0x61, 0xb8, 0x87, 0x84, 0xfa, 0x80, 0x1a, 0x0a, 0x8b, 0x98, 0xf3,
        0xe3, 0xff, 0x4e, 0x44, 0x1c, 0x65, 0x74, 0x7c, 0x71, 0x54, 0x65, 0xe5, 0x39, 0x02, 0x03, 0x01,
        0x00, 0x01, 0xa3, 0x82, 0x01, 0x0a, 0x30, 0x82, 0x01, 0x06, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d,
        0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
        0x32, 0x67, 0xe1, 0xb1, 0x79, 0xd2, 0x81, 0xfc, 0x9f, 0x23, 0x0c, 0x70, 0x40, 0x50, 0xb5, 0x46,
        0x56, 0xb8, 0x30, 0x36, 0x30, 0x81, 0xc4, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x81, 0xbc, 0x30,
        0x81, 0xb9, 0x80, 0x14, 0x73, 0xb0, 0x1c, 0xa4, 0x2f, 0x82, 0xcb, 0xcf, 0x47, 0xa5, 0x38, 0xd7,
        0xb0, 0x04, 0x82, 0x3a, 0x7e, 0x72, 0x15, 0x21, 0xa1, 0x81, 0x9d, 0xa4, 0x81, 0x9a, 0x30, 0x81,
        0x97, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67,
        0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65,
        0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07,
        0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x18, 0x30,
        0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20,
        0x72, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c,
        0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x01, 0x63, 0x30, 0x13, 0x06, 0x03, 0x55,
        0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x09,
        0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03,
        0x82, 0x01, 0x01, 0x00, 0x07, 0xca, 0xa6, 0xa1, 0x9f, 0xbf, 0xaf, 0x92, 0x41, 0x35, 0x66, 0x51,
        0xac, 0xbc, 0x2c, 0xec, 0xe7, 0x8d, 0x65, 0x7e, 0xe9, 0x40, 0xfe, 0x5a, 0xab, 0x8a, 0x1d, 0x3d,
        0x13, 0xdb, 0xb4, 0x43, 0x2c, 0x9a, 0x36, 0x98, 0x21, 0xa5, 0xe8, 0xca, 0xa9, 0x4d, 0xfc, 0xe3,
        0xf7, 0x45, 0x88, 0xcd, 0x33, 0xbf, 0x8a, 0x62, 0x10, 0x2f, 0xb2, 0xb7, 0x04, 0xef, 0x26, 0x43,
        0x51, 0x1d, 0x43, 0x62, 0x7d, 0x1e, 0x50, 0xc8, 0xd5, 0x98, 0x94, 0x71, 0x8f, 0x3b, 0x23, 0x26,
        0xf1, 0x71, 0x8e, 0x1e, 0x3d, 0x3f, 0x21, 0xfd, 0xb7, 0x2d, 0x65, 0xe4, 0x07, 0x65, 0xac, 0x3c,
        0xfc, 0xc0, 0x47, 0xa9, 0x32, 0xf6, 0xda, 0x26, 0x93, 0x10, 0xb2, 0xd1, 0x6d, 0xc8, 0x81, 0x31,
        0x7c, 0xb0, 0x6b, 0xc5, 0x22, 0x8d, 0xb3, 0xfa, 0xbe, 0x82, 0xea, 0x41, 0x42, 0xc4, 0xc0, 0xef,
        0xe3, 0x84, 0x0f, 0x6f, 0x9a, 0x03, 0x63, 0xb3, 0x30, 0xe0, 0x31, 0x81, 0x2a, 0x16, 0xb3, 0x47,
        0xd9, 0x5b, 0x38, 0x93, 0x07, 0xd0, 0x6e, 0x79, 0x52, 0x2c, 0xe5, 0x50, 0x84, 0x79, 0x10, 0xe7,
        0xf6, 0x31, 0x7a, 0x3e, 0x48, 0xa2, 0x38, 0x21, 0x90, 0x7a, 0xf2, 0x5f, 0x48, 0xa4, 0x46, 0x93,
        0x87, 0xdd, 0x5c, 0x83, 0x64, 0xea, 0xb5, 0x99, 0xa2, 0xe9, 0x01, 0x40, 0xfe, 0xf0, 0x48, 0x66,
        0x4f, 0x96, 0xf7, 0x83, 0x52, 0xf8, 0x6d, 0xf8, 0x5f, 0xed, 0x0c, 0xbb, 0xbe, 0xd0, 0x69, 0x10,
        0x4b, 0x99, 0x8f, 0xf8, 0x61, 0x53, 0x9d, 0x12, 0xca, 0x86, 0xaa, 0xb1, 0x80, 0xb4, 0xa6, 0xc1,
        0xcb, 0xb7, 0x48, 0xf7, 0x9f, 0x55, 0xb4, 0x6e, 0xab, 0xd3, 0xa1, 0xaa, 0x4b, 0xa7, 0x21, 0x6e,
        0x16, 0x7f, 0xad, 0xbb, 0xea, 0x0f, 0x41, 0x80, 0x9b, 0x7f, 0xd6, 0x46, 0xa2, 0xc0, 0x61, 0x72,
        0x59, 0x59, 0xa0, 0x07
    };
    OcspEntry entry[1];
    CertStatus status[1];
    OcspRequest* request;

    byte serial[] = {0x05};
    byte issuerHash[] = {0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18, 0xda, 0x04};
    byte issuerKeyHash[] = {0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e};


    XMEMSET(entry, 0, sizeof(OcspEntry));
    XMEMSET(status, 0, sizeof(CertStatus));

    AssertNotNull(request = wolfSSL_OCSP_REQUEST_new());
    request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                     DYNAMIC_TYPE_OCSP_REQUEST);
    AssertNotNull(request->serial);

    request->serialSz = sizeof(serial);
    XMEMCPY(request->serial, serial, sizeof(serial));
    XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
    XMEMCPY(request->issuerKeyHash, issuerKeyHash, sizeof(issuerKeyHash));

    AssertNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    AssertIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CertManagerLoadCA(cm,
        "./certs/ocsp/intermediate1-ca-cert.pem", NULL), WOLFSSL_SUCCESS);

    /* Response should be valid. */
    AssertIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

    /* Flip a byte in the request serial number, response should be invalid
     * now. */
    request->serial[0] ^= request->serial[0];
    AssertIntNE(wolfSSL_CertManagerCheckOCSPResponse(cm, response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);


    wolfSSL_OCSP_REQUEST_free(request);
    wolfSSL_CertManagerFree(cm);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_ALL || WOLFSSL_NGINX ||  WOLFSSL_HAPROXY ||
        * WOLFSSL_APACHE_HTTPD || HAVE_LIGHTY */
#endif /* HAVE_OCSP */
    return res;
}

static int test_wolfSSL_CheckOCSPResponse(void)
{
    int result = TEST_SKIPPED;
#if defined(HAVE_OCSP) && !defined(NO_RSA) && defined(OPENSSL_ALL)
    const char* responseFile = "./certs/ocsp/test-response.der";
    const char* responseMultiFile = "./certs/ocsp/test-multi-response.der";
    const char* responseNoInternFile = "./certs/ocsp/test-response-nointern.der";
    const char* caFile = "./certs/ocsp/root-ca-cert.pem";
    OcspResponse* res = NULL;
    byte data[4096];
    const unsigned char* pt;
    int  dataSz;
    XFILE f;
    WOLFSSL_OCSP_BASICRESP* bs;
    WOLFSSL_X509_STORE* st;
    WOLFSSL_X509* issuer;


    f = XFOPEN(responseFile, "rb");
    AssertTrue(f != XBADFILE);
    dataSz = (word32)XFREAD(data, 1, sizeof(data), f);
    AssertIntGT(dataSz, 0);
    XFCLOSE(f);

    pt = data;
    res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz);
    AssertNotNull(res);
    issuer = wolfSSL_X509_load_certificate_file(caFile, SSL_FILETYPE_PEM);
    AssertNotNull(issuer);
    st = wolfSSL_X509_STORE_new();
    AssertNotNull(st);
    AssertIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
    bs = wolfSSL_OCSP_response_get1_basic(res);
    AssertNotNull(bs);
    AssertIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0), WOLFSSL_SUCCESS);
    wolfSSL_OCSP_BASICRESP_free(bs);
    wolfSSL_OCSP_RESPONSE_free(res);
    wolfSSL_X509_STORE_free(st);
    wolfSSL_X509_free(issuer);

    /* check loading a response with optional certs */
    f = XFOPEN(responseNoInternFile, "rb");
    AssertTrue(f != XBADFILE);
    dataSz = (word32)XFREAD(data, 1, sizeof(data), f);
    AssertIntGT(dataSz, 0);
    XFCLOSE(f);

    pt = data;
    res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz);
    AssertNotNull(res);
    wolfSSL_OCSP_RESPONSE_free(res);

    /* check loading a response with multiple certs */
    {
        WOLFSSL_CERT_MANAGER* cm = NULL;
        OcspEntry *entry;
        CertStatus* status;
        OcspRequest* request;

        byte serial1[] = {0x01};
        byte serial[] = {0x02};

        byte issuerHash[] = {
            0x44, 0xA8, 0xDB, 0xD1, 0xBC, 0x97, 0x0A, 0x83,
            0x3B, 0x5B, 0x31, 0x9A, 0x4C, 0xB8, 0xD2, 0x52,
            0x37, 0x15, 0x8A, 0x88
        };
        byte issuerKeyHash[] = {
            0x73, 0xB0, 0x1C, 0xA4, 0x2F, 0x82, 0xCB, 0xCF,
            0x47, 0xA5, 0x38, 0xD7, 0xB0, 0x04, 0x82, 0x3A,
            0x7E, 0x72, 0x15, 0x21
        };

        entry = (OcspEntry*)XMALLOC(sizeof(OcspEntry), NULL,
             DYNAMIC_TYPE_OPENSSL);
        AssertNotNull(entry);

        status = (CertStatus*)XMALLOC(sizeof(CertStatus), NULL,
             DYNAMIC_TYPE_OPENSSL);
        AssertNotNull(status);

        XMEMSET(entry, 0, sizeof(OcspEntry));
        XMEMSET(status, 0, sizeof(CertStatus));

        AssertNotNull(request = wolfSSL_OCSP_REQUEST_new());
        request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                     DYNAMIC_TYPE_OCSP_REQUEST);
        AssertNotNull(request->serial);

        request->serialSz = sizeof(serial);
        XMEMCPY(request->serial, serial, sizeof(serial));
        XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
        XMEMCPY(request->issuerKeyHash, issuerKeyHash, sizeof(issuerKeyHash));

        AssertNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
        AssertIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
        AssertIntEQ(wolfSSL_CertManagerLoadCA(cm, caFile, NULL),
            WOLFSSL_SUCCESS);

        f = XFOPEN(responseMultiFile, "rb");
        AssertTrue(f != XBADFILE);
        dataSz = (word32)XFREAD(data, 1, sizeof(data), f);
        AssertIntGT(dataSz, 0);
        XFCLOSE(f);

        AssertIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);
        AssertIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        AssertNotNull(entry->status);

        XMEMCPY(request->serial, serial1, sizeof(serial1));
        AssertIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);

        /* store both status's in the entry to check that "next" is not
         * overwritten */
        status->next  = entry->status;
        entry->status = status;

        XMEMCPY(request->serial, serial, sizeof(serial));
        AssertIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        AssertNotNull(entry->status->next);

        /* compare the status found */
        AssertIntEQ(status->serialSz, entry->status->serialSz);
        AssertIntEQ(XMEMCMP(status->serial, entry->status->serial,
            status->serialSz), 0);

        wolfSSL_OCSP_CERTID_free(entry);
        wolfSSL_OCSP_REQUEST_free(request);
        wolfSSL_CertManagerFree(cm);
    }

#if defined(WC_RSA_PSS)
    {
        const char* responsePssFile = "./certs/ocsp/test-response-rsapss.der";

        /* check loading a response with RSA-PSS signature */
        f = XFOPEN(responsePssFile, "rb");
        AssertTrue(f != XBADFILE);
        dataSz = (word32)XFREAD(data, 1, sizeof(data), f);
        AssertIntGT(dataSz, 0);
        XFCLOSE(f);

        pt = data;
        res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz);
        AssertNotNull(res);

        /* try to verify the response */
        issuer = wolfSSL_X509_load_certificate_file(caFile, SSL_FILETYPE_PEM);
        AssertNotNull(issuer);
        st = wolfSSL_X509_STORE_new();
        AssertNotNull(st);
        AssertIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
        bs = wolfSSL_OCSP_response_get1_basic(res);
        AssertNotNull(bs);
        AssertIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0), WOLFSSL_SUCCESS);
        wolfSSL_OCSP_BASICRESP_free(bs);
        wolfSSL_OCSP_RESPONSE_free(res);
        wolfSSL_X509_STORE_free(st);
        wolfSSL_X509_free(issuer);
    }
#endif

    result = TEST_RES_CHECK(1);
#endif /* HAVE_OCSP */
    return result;
}

static int test_wolfSSL_CertManagerLoadCABuffer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ret = test_cm_load_ca_file(ca_cert);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    AssertIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    AssertIntEQ(ret, ASN_UNKNOWN_OID_E);
#else
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ret = test_cm_load_ca_file(ca_expired_cert);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    AssertIntEQ(ret, WOLFSSL_FATAL_ERROR);
    res = TEST_RES_CHECK(ret == WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    AssertIntEQ(ret, ASN_UNKNOWN_OID_E);
    res = TEST_RES_CHECK(ret == ASN_UNKNOWN_OID_E);
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
    !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    AssertIntEQ(ret, ASN_AFTER_DATE_E);
    res = TEST_RES_CHECK(ret == ASN_AFTER_DATE_E);
#else
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
    res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
#endif
#endif

    return res;
}

static int test_wolfSSL_CertManagerLoadCABuffer_ex(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ret = test_cm_load_ca_file_ex(ca_cert, WOLFSSL_LOAD_FLAG_NONE);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    AssertIntEQ(ret, WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    AssertIntEQ(ret, ASN_UNKNOWN_OID_E);
#else
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ret = test_cm_load_ca_file_ex(ca_expired_cert,
                                  WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    AssertIntEQ(ret, WOLFSSL_FATAL_ERROR);
    res = TEST_RES_CHECK(ret == WOLFSSL_FATAL_ERROR);
#elif defined(NO_RSA)
    AssertIntEQ(ret, ASN_UNKNOWN_OID_E);
    res = TEST_RES_CHECK(ret == ASN_UNKNOWN_OID_E);
#else
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
    res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
#endif
#endif

    return res;
}


static int test_wolfSSL_CertManagerGetCerts(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
     defined(WOLFSSL_SIGNER_DER_CERT)

    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_STACK* sk = NULL;
    X509* x509 = NULL;
    X509* cert1 = NULL;
    FILE* file1 = NULL;
#ifdef DEBUG_WOLFSSL_VERBOSE
    WOLFSSL_BIO* bio = NULL;
#endif
    int i = 0;
    int ret = 0;
    const byte* der;
    int derSz = 0;

    AssertNotNull(file1=fopen("./certs/ca-cert.pem", "rb"));

    AssertNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    fclose(file1);

    AssertNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    AssertNull(sk = wolfSSL_CertManagerGetCerts(cm));

    AssertNotNull(der = wolfSSL_X509_get_der(cert1, &derSz));
    ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz, WOLFSSL_FILETYPE_ASN1);
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* Check that ASN_SELF_SIGNED_E is returned for a self-signed cert for QT
     * and full OpenSSL compatibility */
    AssertIntEQ(ret, ASN_SELF_SIGNED_E);
#else
    AssertIntEQ(ret, ASN_NO_SIGNER_E);
#endif

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CertManagerLoadCA(cm,
                "./certs/ca-cert.pem", NULL));

    AssertNotNull(sk = wolfSSL_CertManagerGetCerts(cm));

    for (i = 0; i < sk_X509_num(sk); i++) {
        x509 = sk_X509_value(sk, i);
        AssertIntEQ(0, wolfSSL_X509_cmp(x509, cert1));

#ifdef DEBUG_WOLFSSL_VERBOSE
        bio = BIO_new(wolfSSL_BIO_s_file());
        if (bio != NULL) {
            BIO_set_fp(bio, stderr, BIO_NOCLOSE);
            X509_print(bio, x509);
            BIO_free(bio);
        }
#endif /* DEBUG_WOLFSSL_VERBOSE */
    }
    wolfSSL_X509_free(cert1);
    sk_X509_pop_free(sk, NULL);
    wolfSSL_CertManagerFree(cm);

    res = TEST_RES_CHECK(1);
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
          !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
          defined(WOLFSSL_SIGNER_DER_CERT) */

    return res;
}

static int test_wolfSSL_CertManagerSetVerify(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    int ret = 0;
    WOLFSSL_CERT_MANAGER* cm;
    int tmp = myVerifyAction;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* expiredCert = "./certs/test/expired/expired-cert.pem";

    cm = wolfSSL_CertManagerNew();
    AssertNotNull(cm);

    wolfSSL_CertManagerSetVerify(cm, myVerify);

    ret = wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL);
    #if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    AssertIntEQ(ret, -1);
    #else
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
    #endif
    /* Use the test CB that always accepts certs */
    myVerifyAction = VERIFY_OVERRIDE_ERROR;

    ret = wolfSSL_CertManagerVerify(cm, expiredCert, WOLFSSL_FILETYPE_PEM);
    AssertIntEQ(ret, WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ALWAYS_VERIFY_CB
    {
        const char* verifyCert = "./certs/server-cert.pem";
        /* Use the test CB that always fails certs */
        myVerifyAction = VERIFY_FORCE_FAIL;

        ret = wolfSSL_CertManagerVerify(cm, verifyCert, WOLFSSL_FILETYPE_PEM);
        AssertIntEQ(ret, VERIFY_CERT_ERROR);
    }
#endif

    wolfSSL_CertManagerFree(cm);
    myVerifyAction = tmp;

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

#if !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_UNIT_TEST_CERTS)
/* Used when debugging name constraint tests. Not static to allow use in
 * multiple locations with complex define guards. */
void DEBUG_WRITE_CERT_X509(WOLFSSL_X509* x509, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        PEM_write_bio_X509(out, x509);
        BIO_free(out);
    }
}
void DEBUG_WRITE_DER(const byte* der, int derSz, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        BIO_write(out, der, derSz);
        BIO_free(out);
    }
}
#else
#define DEBUG_WRITE_CERT_X509(x509, fileName)
#define DEBUG_WRITE_DER(der, derSz, fileName)
#endif


static int test_wolfSSL_CertManagerNameConstraint(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm;
    WOLFSSL_EVP_PKEY *priv;
    WOLFSSL_X509_NAME* name;
    const char* ca_cert = "./certs/test/cert-ext-nc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";
    int i = 0;
    static const byte extNameConsOid[] = {85, 29, 30};

    RsaKey  key;
    WC_RNG  rng;
    byte    *der;
    int     derSz;
    word32  idx = 0;
    byte    *pt;
    WOLFSSL_X509 *x509, *ca;

    wc_InitRng(&rng);

    /* load in CA private key for signing */
    AssertIntEQ(wc_InitRsaKey_ex(&key, HEAP_HINT, testDevId), 0);
    AssertIntEQ(wc_RsaPrivateKeyDecode(server_key_der_2048, &idx, &key,
                sizeof_server_key_der_2048), 0);

    /* get ca certificate then alter it */
    AssertNotNull(der =
            (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull(pt = (byte*)wolfSSL_X509_get_tbs(x509, &derSz));
    XMEMCPY(der, pt, derSz);

    /* find the name constraint extension and alter it */
    pt = der;
    for (i = 0; i < derSz - 3; i++) {
        if (XMEMCMP(pt, extNameConsOid, 3) == 0) {
            pt += 3;
            break;
        }
        pt++;
    }
    AssertIntNE(i, derSz - 3); /* did not find OID if this case is hit */

    /* go to the length value and set it to 0 */
    while (i < derSz && *pt != 0x81) {
        pt++;
        i++;
    }
    AssertIntNE(i, derSz); /* did not place to alter */
    pt++;
    *pt = 0x00;

    /* resign the altered certificate */
    AssertIntGT((derSz = wc_SignCert(derSz, CTC_SHA256wRSA, der,
                             FOURK_BUF, &key, NULL, &rng)), 0);

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_PARSE_E);
    wolfSSL_CertManagerFree(cm);

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    /* add email alt name to satisfy constraint */
    pt = (byte*)server_key_der_2048;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Good cert test with proper alt email name */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);


    /* Cert with bad alt name list */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}


static int test_wolfSSL_CertManagerNameConstraint2(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES)
    const char* ca_cert  = "./certs/test/cert-ext-ndir.der";
    const char* ca_cert2 = "./certs/test/cert-ext-ndir-exc.der";
    const char* server_cert = "./certs/server-cert.pem";
    WOLFSSL_CERT_MANAGER* cm;
    WOLFSSL_X509 *x509, *ca;

    const unsigned char *der;
    const unsigned char *pt;
    WOLFSSL_EVP_PKEY *priv;
    WOLFSSL_X509_NAME* name;
    int   derSz;

    /* C=US*/
    char altName[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53
    };

    /* C=ID */
    char altNameFail[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x44
    };

    /* C=US ST=California*/
    char altNameExc[] = {
        0x30, 0x22,
        0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
        0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x0A,
        0x43, 0x61, 0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61
    };
    /* load in CA private key for signing */
    pt = ca_key_der_2048;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &pt,
                                                sizeof_ca_key_der_2048));

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* add in matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif

    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check verify fail */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    /* add in miss matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check that it still fails if one bad altname and one good altname is in
     * the certificate */
    wolfSSL_X509_free(x509);
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check it fails with switching position of bad altname */
    wolfSSL_X509_free(x509);
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);

    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);

    /* now test with excluded name constraint */
    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert2,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    wolfSSL_X509_add_altname_ex(x509, altNameExc, sizeof(altNameExc),
            ASN_DIR_TYPE);
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    AssertNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
#else
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CertManagerNameConstraint3(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm;
    WOLFSSL_EVP_PKEY *priv;
    WOLFSSL_X509_NAME* name;
    const char* ca_cert = "./certs/test/cert-ext-mnc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509, *ca;

    pt = (byte*)server_key_der_2048;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying .wolfssl.com constraint passes */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check satisfying .random.com constraint passes */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.example.com", 24, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.example.com", ASN_RFC822_TYPE);

    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check fail case when neither constraint is matched */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.com", 16, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);

    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CertManagerNameConstraint4(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm;
    WOLFSSL_EVP_PKEY *priv;
    WOLFSSL_X509_NAME* name;
    const char* ca_cert = "./certs/test/cert-ext-ncdns.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509, *ca;

    pt = (byte*)server_key_der_2048;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check satisfying example.com constraint passes */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.example.com", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check satisfying wolfssl.com constraint passes with list of DNS's */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "extra.wolfssl.com", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-multiple-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* check fail when one DNS in the list is bad */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.nomatch.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-multiple-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);

    /* check fail case when neither constraint is matched */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"common", 6, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.random.com", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CertManagerNameConstraint5(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm;
    WOLFSSL_EVP_PKEY *priv;
    WOLFSSL_X509_NAME* name;
    const char* ca_cert = "./certs/test/cert-ext-ncmixed.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509, *ca;

    pt = (byte*)server_key_der_2048;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    AssertIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example", 7, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "good.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@into.wolfssl.com", ASN_RFC822_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    /* fail with DNS check because of common name */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@wolfssl.com", ASN_RFC822_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cn-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);

    /* fail on permitted DNS name constraint */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.wolfssl", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-1st-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);

    /* fail on permitted email name constraint */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "info@example.com", ASN_RFC822_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-2nd-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), ASN_NAME_INVALID_E);
    wolfSSL_X509_free(x509);

    /* success with empty email name */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    AssertNotNull(name = wolfSSL_X509_get_subject_name(ca));
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    AssertNotNull(name = X509_NAME_new());
    AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    AssertIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-missing-constraint-cert.pem");

    AssertNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    AssertIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_FPKI(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_FPKI) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f;
    const char* fpkiCert = "./certs/fpki-cert.der";
    DecodedCert cert;
    byte buf[4096];
    byte* uuid;
    byte* fascn;
    word32 fascnSz;
    word32 uuidSz;
    int bytes;

    f = XFOPEN(fpkiCert, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    AssertIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    AssertIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), LENGTH_ONLY_E) ;
    fascn = (byte*)XMALLOC(fascnSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    AssertNotNull(fascn);
    AssertIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    AssertIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), LENGTH_ONLY_E);
    uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    AssertNotNull(uuid);
    AssertIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    wc_FreeDecodedCert(&cert);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

/* use RID in confuncture with other names to test parsing of unknown other
 * names */
static int test_wolfSSL_OtherName(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f;
    const char* ridCert = "./certs/rid-cert.der";
    DecodedCert cert;
    byte buf[4096];
    int bytes;

    f = XFOPEN(ridCert, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    AssertIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    wc_FreeDecodedCert(&cert);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CertRsaPss(void)
{
    int res = TEST_SKIPPED;
/* FIPS v2 and below don't support long salts. */
#if !defined(NO_RSA) && defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    XFILE f;
    const char* rsaPssSha256Cert = "./certs/rsapss/ca-rsapss.der";
    const char* rsaPssRootSha256Cert = "./certs/rsapss/root-rsapss.pem";
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    const char* rsaPssSha384Cert = "./certs/rsapss/ca-3072-rsapss.der";
#endif
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    const char* rsaPssRootSha384Cert = "./certs/rsapss/root-3072-rsapss.pem";
#endif
    DecodedCert cert;
    byte buf[4096];
    int bytes;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    AssertNotNull(cm);
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha256Cert, NULL));
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha384Cert, NULL));
#endif

    f = XFOPEN(rsaPssSha256Cert, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);
    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    AssertIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);

#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    f = XFOPEN(rsaPssSha384Cert, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);
    wc_InitDecodedCert(&cert, buf, bytes, NULL);
    AssertIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);
#endif

    wolfSSL_CertManagerFree(cm);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CertManagerCRL(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(HAVE_CRL) && \
    !defined(NO_RSA)

    const char* ca_cert = "./certs/ca-cert.pem";
    const char* crl1     = "./certs/crl/crl.pem";
    const char* crl2     = "./certs/crl/crl2.pem";

    WOLFSSL_CERT_MANAGER* cm = NULL;

    AssertNotNull(cm = wolfSSL_CertManagerNew());
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl2, WOLFSSL_FILETYPE_PEM, 0));
    wolfSSL_CertManagerFreeCRL(cm);

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    wolfSSL_CertManagerFree(cm);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_load_verify_locations_ex(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX* ctx;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);

    /* test good CA */
    AssertTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_locations_ex(ctx, ca_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE));

    /* test expired CA */
#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    AssertIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    AssertIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_load_verify_buffer_ex(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(USE_CERT_BUFFERS_2048)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx;
    const char* ca_expired_cert_file = "./certs/test/expired/expired-ca.der";
    byte ca_expired_cert[TWOK_BUF];
    word32 sizeof_ca_expired_cert;
    XFILE fp;

#ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
#endif
    AssertNotNull(ctx);

    /* test good CA */
    AssertTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_cert_der_2048,
            sizeof_ca_cert_der_2048, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE));

    /* load expired CA */
    XMEMSET(ca_expired_cert, 0, sizeof(ca_expired_cert));
    fp = XFOPEN(ca_expired_cert_file, "rb");
    AssertTrue(fp != XBADFILE);
    sizeof_ca_expired_cert = (word32)XFREAD(ca_expired_cert, 1,
        sizeof(ca_expired_cert), fp);
    XFCLOSE(fp);

    /* test expired CA failure */


#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    AssertIntNE(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    AssertIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    /* test expired CA success */
    AssertIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_load_verify_chain_buffer_format(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_RSA) && defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_GEN) && defined(USE_CERT_BUFFERS_2048) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx;

  #ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    AssertTrue(WOLFSSL_SUCCESS == wolfSSL_CTX_load_verify_chain_buffer_format(
        ctx, ca_cert_chain_der, sizeof_ca_cert_chain_der,
        WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_add1_chain_cert(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(OPENSSL_EXTRA) && \
    defined(KEEP_OUR_CERT) && !defined(NO_RSA) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx;
    WOLFSSL*            ssl;
    const char *certChain[] = {
            "./certs/intermediate/client-int-cert.pem",
            "./certs/intermediate/ca-int2-cert.pem",
            "./certs/intermediate/ca-int-cert.pem",
            "./certs/ca-cert.pem",
            NULL
    };
    const char** cert;
    WOLFSSL_X509* x509;
    WOLF_STACK_OF(X509)* chain = NULL;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    AssertNotNull(ssl = wolfSSL_new(ctx));

    for (cert = certChain; *cert != NULL; cert++) {
        x509 = wolfSSL_X509_load_certificate_file(*cert, WOLFSSL_FILETYPE_PEM);
        AssertNotNull(x509);
        AssertIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 1);
        X509_free(x509);
    }
    for (cert = certChain; *cert != NULL; cert++) {
        x509 = wolfSSL_X509_load_certificate_file(*cert, WOLFSSL_FILETYPE_PEM);
        AssertNotNull(x509);
        AssertIntEQ(SSL_add1_chain_cert(ssl, x509), 1);
        X509_free(x509);
    }

    AssertIntEQ(SSL_CTX_get0_chain_certs(ctx, &chain), 1);
    AssertIntEQ(sk_X509_num(chain), 3);
    AssertIntEQ(SSL_get0_chain_certs(ssl, &chain), 1);
    AssertIntEQ(sk_X509_num(chain), 3);

    SSL_free(ssl);
    SSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_CTX_use_certificate_chain_file_format(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = "./certs/client-cert.pem";
    WOLFSSL_CTX* ctx;
    int ret = 0;

    (void)server_chain_der;
    (void)client_single_pem;
    (void)ctx;
  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);
  #endif

    AssertIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        client_single_pem, WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}

static int test_wolfSSL_CTX_SetTmpDH_file(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;
    (void)ctx;
  #ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                NULL, WOLFSSL_FILETYPE_PEM));

    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dhParamFile,
                WOLFSSL_FILETYPE_PEM));

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_CTX_SetTmpDH_buffer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;
    (void)ctx;
  #ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));

    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, dsa_key_der_2048,
                sizeof_dsa_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* success */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_CTX_SetMinMaxDhKey_Sz(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;
    (void)ctx;
  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);
  #endif

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));

    AssertIntEQ(DH_KEY_SIZE_E, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 2048));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));

    AssertIntEQ(DH_KEY_SIZE_E, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 2048));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_CTX_der_load_verify_locations(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_DER_LOAD) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    const char* derCert = "./certs/server-cert.der";
    const char* nullPath = NULL;
    const char* invalidPath = "./certs/this-cert-does-not-exist.der";
    const char* emptyPath = "";

    /* der load Case 1 ctx NULL */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);

  #ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Case 2 filePath NULL */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, nullPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
    /* Case 3 invalid format */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_FAILURE);
    /* Case 4 filePath not valid */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, invalidPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
    /* Case 5 filePath empty */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, emptyPath,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_FAILURE);
#ifndef NO_RSA
    /* Case 6 success case */
    AssertIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_CTX_enable_disable(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_CERTS
    WOLFSSL_CTX* ctx = NULL;

  #ifdef HAVE_CRL
    AssertIntEQ(wolfSSL_CTX_DisableCRL(ctx), BAD_FUNC_ARG);
    AssertIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), BAD_FUNC_ARG);
  #endif

  #ifdef HAVE_OCSP
    AssertIntEQ(wolfSSL_CTX_DisableOCSP(ctx), BAD_FUNC_ARG);
    AssertIntEQ(wolfSSL_CTX_EnableOCSP(ctx, 0), BAD_FUNC_ARG);
  #endif

  #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
      defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    AssertIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), BAD_FUNC_ARG);
    AssertIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), BAD_FUNC_ARG);
    AssertIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), BAD_FUNC_ARG);
    AssertIntEQ(wolfSSL_CTX_EnableOCSPMustStaple(ctx), BAD_FUNC_ARG);
  #endif

  #ifndef NO_WOLFSSL_CLIENT

    #ifdef HAVE_EXTENDED_MASTER
    AssertIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), BAD_FUNC_ARG);
    #endif
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    AssertNotNull(ctx);

    #ifdef HAVE_EXTENDED_MASTER
    AssertIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WOLFSSL_SUCCESS);
    #endif

  #elif !defined(NO_WOLFSSL_SERVER)
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #else
    return TEST_SUCCESS;
  #endif

  #ifdef HAVE_CRL
    AssertIntEQ(wolfSSL_CTX_DisableCRL(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WOLFSSL_SUCCESS);
  #endif

  #ifdef HAVE_OCSP
    AssertIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_URL_OVERRIDE),
                WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_NO_NONCE),
                WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_CHECKALL),
                WOLFSSL_SUCCESS);
  #endif

  #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
      defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    AssertIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
  #endif
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* NO_CERTS */

    return res;
}

static int test_wolfSSL_CTX_ticket_API(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_SESSION_TICKET) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX* ctx = NULL;
    void *userCtx = (void*)"this is my ctx";

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(ctx, userCtx));
    AssertTrue(userCtx == wolfSSL_CTX_get_TicketEncCtx(ctx));

    wolfSSL_CTX_free(ctx);

    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(NULL, userCtx));
    AssertNull(wolfSSL_CTX_get_TicketEncCtx(NULL));

    res = TEST_RES_CHECK(1);
#endif /* HAVE_SESSION_TICKET && !NO_WOLFSSL_SERVER */
    return res;
}

static int test_wolfSSL_set_minmax_proto_version(void)
{
    int res = TEST_SKIPPED;
#ifdef OPENSSL_EXTRA
    WOLFSSL_CTX *ctx;
    WOLFSSL *ssl;
    int ret;
    (void)ret;
    (void)ssl;

#ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    AssertNotNull(ssl = wolfSSL_new(ctx));

    AssertIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    AssertIntEQ(wolfSSL_set_min_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_set_min_proto_version(ssl, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_set_max_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_set_max_proto_version(ssl, 0), SSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
#ifndef NO_WOLFSSL_SERVER
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    AssertIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), SSL_FAILURE);
    AssertIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    AssertIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_TLS12) && \
    defined(OPENSSL_EXTRA) && defined(HAVE_IO_TESTS_DEPENDENCIES)
static void test_wolfSSL_CTX_set_max_proto_version_on_result(WOLFSSL* ssl)
{
    AssertStrEQ(wolfSSL_get_version(ssl), "TLSv1.2");
}

static void test_wolfSSL_CTX_set_max_proto_version_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* Set TLS 1.2 */
    AssertIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, TLS1_2_VERSION),
            WOLFSSL_SUCCESS);
}

/* Test using wolfSSL_CTX_set_max_proto_version to limit the version below
 * what was set at ctx creation. */
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    callback_functions client_cbs, server_cbs;

    XMEMSET(&client_cbs, 0, sizeof(client_cbs));
    XMEMSET(&server_cbs, 0, sizeof(server_cbs));

    client_cbs.method = wolfTLS_client_method;
    server_cbs.method = wolfTLS_server_method;

    server_cbs.ctx_ready = test_wolfSSL_CTX_set_max_proto_version_ctx_ready;

    client_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;
    server_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;

    test_wolfSSL_client_server_nofail(&client_cbs, &server_cbs);

    AssertIntEQ(client_cbs.return_code, TEST_SUCCESS);
    AssertIntEQ(server_cbs.return_code, TEST_SUCCESS);

    return TEST_RES_CHECK(1);
}
#else
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    return TEST_SKIPPED;
}
#endif

/*----------------------------------------------------------------------------*
 | SSL
 *----------------------------------------------------------------------------*/

static int test_server_wolfSSL_new(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;
    WOLFSSL_CTX *ctx_nocert;
    WOLFSSL *ssl;

    AssertNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    AssertNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));

    /* invalid context */
    AssertNull(ssl = wolfSSL_new(NULL));
#if !defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_QT) && !defined(OPENSSL_EXTRA)
    AssertNull(ssl = wolfSSL_new(ctx_nocert));
#endif

    /* success */
    AssertNotNull(ssl = wolfSSL_new(ctx));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}


static int test_client_wolfSSL_new(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx;
    WOLFSSL_CTX *ctx_nocert;
    WOLFSSL *ssl;

    AssertNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    AssertNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    AssertTrue(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));

    /* invalid context */
    AssertNull(ssl = wolfSSL_new(NULL));

    /* success */
    AssertNotNull(ssl = wolfSSL_new(ctx_nocert));
    wolfSSL_free(ssl);

    /* success */
    AssertNotNull(ssl = wolfSSL_new(ctx));
    wolfSSL_free(ssl);

    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_SetTmpDH_file(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_DH) && \
        !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;
    WOLFSSL *ssl;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#ifndef NO_RSA
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
               WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ECC)
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, eccCertFile,
               WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED25519)
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, edCertFile,
               WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED448)
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, ed448CertFile,
               WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
               WOLFSSL_FILETYPE_PEM));
#endif
    AssertNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                NULL, WOLFSSL_FILETYPE_PEM));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhParamFile,
                WOLFSSL_FILETYPE_PEM));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_SetTmpDH_buffer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_DH) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;
    WOLFSSL *ssl;

    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    AssertTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dsa_key_der_2048,
                sizeof_dsa_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* success */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

static int test_wolfSSL_SetMinMaxDhKey_Sz(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_DH) && !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx, *ctx2;
    WOLFSSL *ssl, *ssl2;
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);
    AssertTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));
    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);
    ctx2 = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx2);
    AssertTrue(wolfSSL_CTX_use_certificate_buffer(ctx2, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx2, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));
    ssl2 = wolfSSL_new(ctx2);
    AssertNotNull(ssl2);

    AssertIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 2048));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 3072));
    AssertIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 2048));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 1024));
    AssertIntEQ(DH_KEY_SIZE_E, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl2);
    wolfSSL_CTX_free(ctx2);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}


/* Test function for wolfSSL_SetMinVersion. Sets the minimum downgrade version
 * allowed.
 * POST: return 1 on success.
 */
static int test_wolfSSL_SetMinVersion(void)
{
    int                 res = TEST_SKIPPED;
#ifndef NO_WOLFSSL_CLIENT
    int                 failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*        ctx;
    WOLFSSL*            ssl;
    int                 itr;

    #ifndef NO_OLD_TLS
        const int versions[]  =  {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                   WOLFSSL_TLSV1,
                            #endif
                                   WOLFSSL_TLSV1_1,
                                   WOLFSSL_TLSV1_2};
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  =  { WOLFSSL_TLSV1_2 };
    #else
        const int versions[]  =  { WOLFSSL_TLSV1_3 };
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    ssl = wolfSSL_new(ctx);

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
       if (wolfSSL_SetMinVersion(ssl, *(versions + itr)) != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_SetMinVersion */


#ifdef OPENSSL_EXTRA
static int test_ED25519(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED25519_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED25519_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED25519_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */

    AssertIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, pub, &pubSz),
                WOLFSSL_SUCCESS);
    AssertIntEQ(privSz, ED25519_PRV_KEY_SIZE);
    AssertIntEQ(pubSz, ED25519_PUB_KEY_SIZE);

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    AssertIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz, sig,
                                      &sigSz), WOLFSSL_SUCCESS);
    AssertIntEQ(sigSz, ED25519_SIG_SIZE);

#ifdef HAVE_ED25519_VERIFY
    AssertIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
                                       sigSz), WOLFSSL_SUCCESS);
#endif /* HAVE_ED25519_VERIFY */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */

    res = TEST_RES_CHECK(1);
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return res;
}

static int test_ED448(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED448_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED448_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED448_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */

    AssertIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, pub, &pubSz),
                WOLFSSL_SUCCESS);
    AssertIntEQ(privSz, ED448_PRV_KEY_SIZE);
    AssertIntEQ(pubSz, ED448_PUB_KEY_SIZE);

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    AssertIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz, sig,
                                   &sigSz), WOLFSSL_SUCCESS);
    AssertIntEQ(sigSz, ED448_SIG_SIZE);

#ifdef HAVE_ED448_VERIFY
    AssertIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
                                     sigSz), WOLFSSL_SUCCESS);
#endif /* HAVE_ED448_VERIFY */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */

    res = TEST_RES_CHECK(1);
#endif /* HAVE_ED448 && HAVE_ED448_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return res;
}
#endif /* OPENSSL_EXTRA */

#include <wolfssl/openssl/pem.h>
/*----------------------------------------------------------------------------*
 | EVP
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_EVP_PKEY_print_public(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)
    WOLFSSL_BIO* rbio = NULL;
    WOLFSSL_BIO* wbio = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    char line[256] = { 0 };
    char line1[256] = { 0 };
    int i;

    /* test error cases */
    AssertIntEQ( EVP_PKEY_print_public(NULL,NULL,0,NULL),0L);

    /*
     *  test RSA public key print
     *  in this test, pass '3' for indent
     */
#if !defined(NO_RSA) && defined(USE_CERT_BUFFERS_1024)

    rbio = BIO_new_mem_buf( client_keypub_der_1024,
                            sizeof_client_keypub_der_1024);
    AssertNotNull(rbio);

    wolfSSL_d2i_PUBKEY_bio(rbio, &pkey);
    AssertNotNull(pkey);

    wbio = BIO_new(BIO_s_mem());
    AssertNotNull(wbio);

    AssertIntEQ(EVP_PKEY_print_public(wbio, pkey,3,NULL),1);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "   RSA Public-Key: (1024 bit)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "   Modulus:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "       00:bc:73:0e:a8:49:f3:74:a2:a9:ef:18:a5:da:55:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* skip to the end of modulus element*/
    for (i = 0; i < 8 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "   Exponent: 65537 (0x010001)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    AssertIntLE(BIO_gets(wbio, line, sizeof(line)) ,0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif  /* !NO_RSA && USE_CERT_BUFFERS_1024*/

    /*
     *  test DSA public key print
     */
#if !defined(NO_DSA) && defined(USE_CERT_BUFFERS_2048)
    rbio = BIO_new_mem_buf( dsa_pub_key_der_2048,
                            sizeof_dsa_pub_key_der_2048);
    AssertNotNull(rbio);

    wolfSSL_d2i_PUBKEY_bio(rbio, &pkey);
    AssertNotNull(pkey);

    wbio = BIO_new(BIO_s_mem());
    AssertNotNull(wbio);

    AssertIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "DSA Public-Key: (2048 bit)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "pub:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1,
        "    00:C2:35:2D:EC:83:83:6C:73:13:9E:52:7C:74:C8:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 17 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "P:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of P element*/
    for (i = 0; i < 18 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "Q:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of Q element*/
    for (i = 0; i < 3 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }
    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "G:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of G element*/
    for (i = 0; i < 18 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }
    /* should reach EOF */
    AssertIntLE(BIO_gets(wbio, line, sizeof(line)) ,0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* !NO_DSA && USE_CERT_BUFFERS_2048 */

    /*
     *  test ECC public key print
     */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)

    rbio = BIO_new_mem_buf( ecc_clikeypub_der_256,
                            sizeof_ecc_clikeypub_der_256);
    AssertNotNull(rbio);

    wolfSSL_d2i_PUBKEY_bio(rbio, &pkey);
    AssertNotNull(pkey);

    wbio = BIO_new(BIO_s_mem());
    AssertNotNull(wbio);

    AssertIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "Public-Key: (256 bit)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "pub:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1,
            "    04:55:BF:F4:0F:44:50:9A:3D:CE:9B:B7:F0:C5:4D:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 4 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "ASN1 OID: prime256v1\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "NIST CURVE: P-256\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    AssertIntLE(BIO_gets(wbio, line, sizeof(line)) ,0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* HAVE_ECC && USE_CERT_BUFFERS_256 */

    /*
     *  test DH public key print
     */
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)

    rbio = BIO_new_mem_buf( dh_pub_key_der_2048,
                            sizeof_dh_pub_key_der_2048);
    AssertNotNull(rbio);

    wolfSSL_d2i_PUBKEY_bio(rbio, &pkey);
    AssertNotNull(pkey);

    wbio = BIO_new(BIO_s_mem());
    AssertNotNull(wbio);

    AssertIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "DH Public-Key: (2048 bit)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "public-key:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1,
        "    34:41:BF:E9:F2:11:BF:05:DB:B2:72:A8:29:CC:BD:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of public-key element*/
    for (i = 0; i < 17 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "prime:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1,
        "    00:D3:B2:99:84:5C:0A:4C:E7:37:CC:FC:18:37:01:\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of prime element*/
    for (i = 0; i < 17 ;i++) {
        BIO_gets(wbio, line, sizeof(line));
    }

    BIO_gets(wbio, line, sizeof(line));
    strcpy(line1, "generator: 2 (0x02)\n");
    AssertIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* should reach EOF */
    AssertIntLE(BIO_gets(wbio, line, sizeof(line)) ,0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* WOLFSSL_DH_EXTRA && USE_CERT_BUFFERS_2048 */

    /* to prevent "unused variable" warning */
    (void)pkey;
    (void)wbio;
    (void)rbio;
    (void)line;
    (void)line1;
    (void)i;

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA */
    return res;
}
/* Test functions for base64 encode/decode */
static int test_wolfSSL_EVP_ENCODE_CTX_new(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    AssertNotNull( ctx = EVP_ENCODE_CTX_new());
    AssertIntEQ( ctx->remaining,0);
    AssertIntEQ( ctx->data[0],0);
    AssertIntEQ( ctx->data[sizeof(ctx->data) -1],0);
    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE)*/
    return res;
}
static int test_wolfSSL_EVP_ENCODE_CTX_free(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    AssertNotNull( ctx = EVP_ENCODE_CTX_new());
    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /*OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE)*/
    return res;
}

static int test_wolfSSL_EVP_EncodeInit(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    EVP_ENCODE_CTX* ctx = NULL;

    AssertNotNull( ctx = EVP_ENCODE_CTX_new());
    AssertIntEQ( ctx->remaining,0);
    AssertIntEQ( ctx->data[0],0);
    AssertIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    /* make ctx dirty */
    ctx->remaining = 10;
    XMEMSET( ctx->data, 0x77, sizeof(ctx->data));

    EVP_EncodeInit(ctx);

    AssertIntEQ( ctx->remaining,0);
    AssertIntEQ( ctx->data[0],0);
    AssertIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}
static int test_wolfSSL_EVP_EncodeUpdate(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    int outl;
    int total;

    const unsigned char plain0[] = {"Th"};
    const unsigned char plain1[] = {"This is a base64 encodeing test."};
    const unsigned char plain2[] = {"This is additional data."};

    const unsigned char enc0[]   = {"VGg=\n"};
    /* expected encoded result for the first output 64 chars plus trailing LF*/
    const unsigned char enc1[]   = {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\n"};

    const unsigned char enc2[]   =
    {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\nYWwgZGF0YS4=\n"};

    unsigned char encOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    AssertNotNull( ctx = EVP_ENCODE_CTX_new());

    EVP_EncodeInit(ctx);

    /* illegal parameter test */
    AssertIntEQ(
        EVP_EncodeUpdate(
            NULL,            /* pass NULL as ctx */
            encOutBuff,
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    AssertIntEQ(
        EVP_EncodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    AssertIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            NULL,            /* pass NULL as outl */
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    AssertIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    AssertIntEQ(EVP_EncodeBlock(NULL, NULL, 0), -1);

    /* meaningless parameter test */

    AssertIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain1,
            0),              /* pass zero input */
        1                    /* expected result code 1: success */
    );

    /* very small data encoding test */

    EVP_EncodeInit(ctx);

    AssertIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain0,
            sizeof(plain0)-1),
        1                    /* expected result code 1: success */
    );
    AssertIntEQ(outl,0);

        EVP_EncodeFinal(
            ctx,
            encOutBuff + outl,
            &outl);

    AssertIntEQ( outl, sizeof(enc0)-1);
    AssertIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    XMEMSET( encOutBuff,0, sizeof(encOutBuff));
    AssertIntEQ(EVP_EncodeBlock(encOutBuff, plain0, sizeof(plain0)-1),
                sizeof(enc0)-1);
    AssertIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    /* pass small size( < 48bytes ) input, then make sure they are not
     * encoded  and just stored in ctx
     */

    EVP_EncodeInit(ctx);

    total = 0;
    outl = 0;
    XMEMSET( encOutBuff,0, sizeof(encOutBuff));

    AssertIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff,         /* buffer for output */
        &outl,              /* size of output */
        plain1,             /* input */
        sizeof(plain1)-1),  /* size of input */
        1);                 /* expected result code 1:success */

    total += outl;

    AssertIntEQ(outl, 0);  /* no output expected */
    AssertIntEQ(ctx->remaining, sizeof(plain1) -1);
    AssertTrue(
        XSTRNCMP((const char*)(ctx->data),
                 (const char*)plain1,
                 ctx->remaining) ==0 );
    AssertTrue(encOutBuff[0] == 0);

    /* call wolfSSL_EVP_EncodeUpdate again to make it encode
     * the stored data and the new input together
     */
    AssertIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff + outl,  /* buffer for output */
        &outl,              /* size of output */
        plain2,             /* additional input */
        sizeof(plain2) -1), /* size of additional input */
        1);                 /* expected result code 1:success */

    total += outl;

    AssertIntNE(outl, 0);   /* some output is expected this time*/
    AssertIntEQ(outl, BASE64_ENCODE_RESULT_BLOCK_SIZE +1); /* 64 bytes and LF */
    AssertIntEQ(
        XSTRNCMP((const char*)encOutBuff,(const char*)enc1,sizeof(enc1) ),0);

    /* call wolfSSL_EVP_EncodeFinal to flush all the unprocessed input */
    EVP_EncodeFinal(
        ctx,
        encOutBuff + outl,
        &outl);

    total += outl;

    AssertIntNE(total,0);
    AssertIntNE(outl,0);
    AssertIntEQ(XSTRNCMP(
        (const char*)encOutBuff,(const char*)enc2,sizeof(enc2) ),0);

    /* test with illeagal parameters */
    outl = 1;
    EVP_EncodeFinal(NULL, encOutBuff + outl, &outl);
    AssertIntEQ(outl, 0);
    outl = 1;
    EVP_EncodeFinal(ctx, NULL, &outl);
    AssertIntEQ(outl, 0);
    EVP_EncodeFinal(ctx, encOutBuff + outl, NULL);
    EVP_EncodeFinal(NULL, NULL, NULL);

    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}
static int test_wolfSSL_EVP_EncodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    /* tests for wolfSSL_EVP_EncodeFinal are included in
     * test_wolfSSL_EVP_EncodeUpdate
     */
    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}


static int test_wolfSSL_EVP_DecodeInit(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    EVP_ENCODE_CTX* ctx = NULL;

    AssertNotNull( ctx = EVP_ENCODE_CTX_new());
    AssertIntEQ( ctx->remaining,0);
    AssertIntEQ( ctx->data[0],0);
    AssertIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    /* make ctx dirty */
    ctx->remaining = 10;
    XMEMSET( ctx->data, 0x77, sizeof(ctx->data));

    EVP_DecodeInit(ctx);

    AssertIntEQ( ctx->remaining,0);
    AssertIntEQ( ctx->data[0],0);
    AssertIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}
static int test_wolfSSL_EVP_DecodeUpdate(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    int outl;
    unsigned char decOutBuff[300];

    EVP_ENCODE_CTX* ctx;

    static const unsigned char enc1[]   =
            {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
/*    const unsigned char plain1[] =
    {"This is a base64 decoding test."} */

    ctx = EVP_ENCODE_CTX_new();

    EVP_DecodeInit(ctx);

    /* illegal parameter tests */

    /* pass NULL as ctx */
    AssertIntEQ(
        EVP_DecodeUpdate(
            NULL,            /* pass NULL as ctx */
            decOutBuff,
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    AssertIntEQ( outl, 0);

    /* pass NULL as output */
    AssertIntEQ(
        EVP_DecodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    AssertIntEQ( outl, 0);

    /* pass NULL as outl */
    AssertIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            NULL,            /* pass NULL as outl */
            enc1,
            sizeof(enc1)-1),
        -1                   /* expected result code -1: fail */
    );

    /* pass NULL as input */
    AssertIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    AssertIntEQ( outl, 0);

    AssertIntEQ(EVP_DecodeBlock(NULL, NULL, 0), -1);

    /* pass zero length input */

    AssertIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            enc1,
            0),              /* pass zero as input len */
        1                    /* expected result code 1: success */
    );

    /* decode correct base64 string */

    {
        static const unsigned char enc2[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
        static const unsigned char plain2[] =
        {"This is a base64 decoding test."};

        EVP_EncodeInit(ctx);

        AssertIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc2,
                sizeof(enc2)-1),
            0                    /* expected result code 0: success */
            );

        AssertIntEQ(outl,sizeof(plain2) -1);

        AssertIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );
        AssertIntEQ(outl, 0);   /* expected DecodeFinal outout no data */

        AssertIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
        AssertIntEQ(EVP_DecodeBlock(decOutBuff, enc2, sizeof(enc2)),
                    sizeof(plain2)-1);
        AssertIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
    }

    /* decode correct base64 string which does not have '\n' in its last*/

    {
        static const unsigned char enc3[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg=="}; /* 44 chars */
        static const unsigned char plain3[] =
        {"This is a base64 decoding test."}; /* 31 chars */

        EVP_EncodeInit(ctx);

        AssertIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc3,
                sizeof(enc3)-1),
            0                    /* expected result code 0: success */
            );

        AssertIntEQ(outl,sizeof(plain3)-1);   /* 31 chars should be output */

        AssertIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);

        AssertIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );

        AssertIntEQ(outl,0 );

        AssertIntEQ(EVP_DecodeBlock(decOutBuff, enc3, sizeof(enc3)-1),
                    sizeof(plain3)-1);
        AssertIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);
    }

    /* decode string which has a padding char ('=') in the illegal position*/

    {
        static const unsigned char enc4[]   =
            {"VGhpcyBpcyBhIGJhc2U2N=CBkZWNvZGluZyB0ZXN0Lg==\n"};

        EVP_EncodeInit(ctx);

        AssertIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc4,
                sizeof(enc4)-1),
            -1                    /* expected result code -1: error */
            );
        AssertIntEQ(outl,0);
        AssertIntEQ(EVP_DecodeBlock(decOutBuff, enc4, sizeof(enc4)-1), -1);
    }

    /* small data decode test */

    {
        static const unsigned char enc00[]   = {"VG"};
        static const unsigned char enc01[]   = {"g=\n"};
        static const unsigned char plain4[]  = {"Th"};

        EVP_EncodeInit(ctx);

        AssertIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc00,
                sizeof(enc00)-1),
            1                    /* expected result code 1: success */
            );
        AssertIntEQ(outl,0);

        AssertIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff + outl,
                &outl,
                enc01,
                sizeof(enc01)-1),
            0                    /* expected result code 0: success */
            );

        AssertIntEQ(outl,sizeof(plain4)-1);

        /* test with illegal parameters */
        AssertIntEQ(EVP_DecodeFinal(NULL,decOutBuff + outl,&outl), -1);
        AssertIntEQ(EVP_DecodeFinal(ctx,NULL,&outl), -1);
        AssertIntEQ(EVP_DecodeFinal(ctx,decOutBuff + outl, NULL), -1);
        AssertIntEQ(EVP_DecodeFinal(NULL,NULL, NULL), -1);

        EVP_DecodeFinal(
            ctx,
            decOutBuff + outl,
            &outl);

        AssertIntEQ( outl, 0);
        AssertIntEQ(
            XSTRNCMP(
                (const char*)decOutBuff,
                (const char*)plain4,sizeof(plain4)-1 ),
            0);
    }

    EVP_ENCODE_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}
static int test_wolfSSL_EVP_DecodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    /* tests for wolfSSL_EVP_DecodeFinal are included in
     * test_wolfSSL_EVP_DecodeUpdate
     */
    res = TEST_RES_CHECK(1);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}

/* Test function for wolfSSL_EVP_get_cipherbynid.
 */

#ifdef OPENSSL_EXTRA
static int test_wolfSSL_EVP_get_cipherbynid(void)
{
#ifndef NO_AES
    const WOLFSSL_EVP_CIPHER* c;

    c = wolfSSL_EVP_get_cipherbynid(419);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_128)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_128_CBC", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(423);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_192)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_192_CBC", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(427);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_256)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_256_CBC", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(904);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_128)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_128_CTR", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(905);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_192)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_192_CTR", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(906);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_256_CTR", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(418);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_128)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_128_ECB", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(422);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_192)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_192_ECB", c));
    #else
        AssertNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(426);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
        AssertNotNull(c);
        AssertNotNull(strcmp("EVP_AES_256_ECB", c));
    #else
        AssertNull(c);
    #endif
#endif /* !NO_AES */

#ifndef NO_DES3
    AssertNotNull(strcmp("EVP_DES_CBC", wolfSSL_EVP_get_cipherbynid(31)));
#ifdef WOLFSSL_DES_ECB
    AssertNotNull(strcmp("EVP_DES_ECB", wolfSSL_EVP_get_cipherbynid(29)));
#endif
    AssertNotNull(strcmp("EVP_DES_EDE3_CBC", wolfSSL_EVP_get_cipherbynid(44)));
#ifdef WOLFSSL_DES_ECB
    AssertNotNull(strcmp("EVP_DES_EDE3_ECB", wolfSSL_EVP_get_cipherbynid(33)));
#endif
#endif /* !NO_DES3 */

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    AssertNotNull(strcmp("EVP_CHACHA20_POLY13O5", EVP_get_cipherbynid(1018)));
#endif

    /* test for nid is out of range */
    AssertNull(wolfSSL_EVP_get_cipherbynid(1));

    return TEST_RES_CHECK(1);
}

static int test_wolfSSL_EVP_CIPHER_CTX(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
    const EVP_CIPHER *init = EVP_aes_128_cbc();
    const EVP_CIPHER *test;
    byte key[AES_BLOCK_SIZE] = {0};
    byte iv[AES_BLOCK_SIZE] = {0};

    AssertNotNull(ctx);
    wolfSSL_EVP_CIPHER_CTX_init(ctx);
    AssertIntEQ(EVP_CipherInit(ctx, init, key, iv, 1), WOLFSSL_SUCCESS);
    test = EVP_CIPHER_CTX_cipher(ctx);
    AssertTrue(init == test);
    AssertIntEQ(EVP_CIPHER_nid(test), NID_aes_128_cbc);

    AssertIntEQ(EVP_CIPHER_CTX_reset(ctx), WOLFSSL_SUCCESS);
    AssertIntEQ(EVP_CIPHER_CTX_reset(NULL), WOLFSSL_FAILURE);

    EVP_CIPHER_CTX_free(ctx);
    /* test EVP_CIPHER_CTX_cleanup with NULL */
    AssertIntEQ(EVP_CIPHER_CTX_cleanup(NULL), WOLFSSL_SUCCESS);

    res = TEST_RES_CHECK(1);
#endif /* !NO_AES && HAVE_AES_CBC && WOLFSSL_AES_128 */
    return res;
}
#endif /* OPENSSL_EXTRA */

/*----------------------------------------------------------------------------*
 | IO
 *----------------------------------------------------------------------------*/

/* helper functions */
#ifdef HAVE_IO_TESTS_DEPENDENCIES

#ifdef WOLFSSL_SESSION_EXPORT
#ifdef WOLFSSL_DTLS
/* set up function for sending session information */
static int test_export(WOLFSSL* inSsl, byte* buf, word32 sz, void* userCtx)
{
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    AssertNotNull(inSsl);
    AssertNotNull(buf);
    AssertIntNE(0, sz);

    /* Set ctx to DTLS 1.2 */
    ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method());
    AssertNotNull(ctx);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    AssertIntGE(wolfSSL_dtls_import(ssl, buf, sz), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    (void)userCtx;
    return 0;
}
#endif

/* returns negative value on fail and positive (including 0) on success */
static int nonblocking_accept_read(void* args, WOLFSSL* ssl, SOCKET_T* sockfd)
{
    int ret, err, loop_count, count, timeout = 10;
    char msg[] = "I hear you fa shizzle!";
    char input[1024];

    loop_count = ((func_args*)args)->argc;

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, 0);

        if (err == WOLFSSL_ERROR_WANT_READ ||
            err == WOLFSSL_ERROR_WANT_WRITE) {
            int select_ret;

            err = WC_PENDING_E;
            select_ret = tcp_select(*sockfd, timeout);
            if (select_ret == TEST_TIMEOUT) {
                return WOLFSSL_FATAL_ERROR;
            }
        }
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        return ret;
    }

    for (count = 0; count < loop_count; count++) {
        int select_ret;

        select_ret = tcp_select(*sockfd, timeout);
        if (select_ret == TEST_TIMEOUT) {
            ret = WOLFSSL_FATAL_ERROR;
            break;
        }

        do {
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            if (ret > 0) {
                input[ret] = '\0';
                fprintf(stderr, "Client message: %s\n", input);
            }
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);

        do {
            if ((ret = wolfSSL_write(ssl, msg, sizeof(msg))) != sizeof(msg)) {
                return WOLFSSL_FATAL_ERROR;
            }
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);
    }
    return ret;
}
#endif /* WOLFSSL_SESSION_EXPORT */

/* TODO: Expand and enable this when EVP_chacha20_poly1305 is supported */
#if defined(HAVE_SESSION_TICKET) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_AES_CBC)

    typedef struct openssl_key_ctx {
        byte name[WOLFSSL_TICKET_NAME_SZ]; /* server name */
        byte key[WOLFSSL_TICKET_KEY_SZ]; /* cipher key */
        byte hmacKey[WOLFSSL_TICKET_NAME_SZ]; /* hmac key */
        byte iv[WOLFSSL_TICKET_IV_SZ]; /* cipher iv */
    } openssl_key_ctx;

    static THREAD_LS_T openssl_key_ctx myOpenSSLKey_ctx;
    static THREAD_LS_T WC_RNG myOpenSSLKey_rng;

    static WC_INLINE int OpenSSLTicketInit(void)
    {
        int ret = wc_InitRng(&myOpenSSLKey_rng);
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.name,
                sizeof(myOpenSSLKey_ctx.name));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.key,
                sizeof(myOpenSSLKey_ctx.key));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.hmacKey,
                sizeof(myOpenSSLKey_ctx.hmacKey));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.iv,
                sizeof(myOpenSSLKey_ctx.iv));
        if (ret != 0) return ret;

        return 0;
    }

    static WC_INLINE int myTicketEncCbOpenSSL(WOLFSSL* ssl,
                             byte name[WOLFSSL_TICKET_NAME_SZ],
                             byte iv[WOLFSSL_TICKET_IV_SZ],
                             WOLFSSL_EVP_CIPHER_CTX *ectx,
                             WOLFSSL_HMAC_CTX *hctx, int enc) {
        (void)ssl;
        if (enc) {
            XMEMCPY(name, myOpenSSLKey_ctx.name, sizeof(myOpenSSLKey_ctx.name));
            XMEMCPY(iv, myOpenSSLKey_ctx.iv, sizeof(myOpenSSLKey_ctx.iv));
        }
        else if (XMEMCMP(name, myOpenSSLKey_ctx.name,
                            sizeof(myOpenSSLKey_ctx.name)) != 0 ||
                 XMEMCMP(iv, myOpenSSLKey_ctx.iv,
                            sizeof(myOpenSSLKey_ctx.iv)) != 0) {
            return 0;
        }
        HMAC_Init_ex(hctx, myOpenSSLKey_ctx.hmacKey, WOLFSSL_TICKET_NAME_SZ, EVP_sha256(), NULL);
        if (enc)
            EVP_EncryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        else
            EVP_DecryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        return 1;
    }

    static WC_INLINE void OpenSSLTicketCleanup(void)
    {
        wc_FreeRng(&myOpenSSLKey_rng);
    }
#endif

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    #ifdef WC_SHA512_DIGEST_SIZE
        #define MD_MAX_SIZE WC_SHA512_DIGEST_SIZE
    #else
        #define MD_MAX_SIZE WC_SHA256_DIGEST_SIZE
    #endif
    byte server_side_msg1[MD_MAX_SIZE] = {0};/* msg sent by server */
    byte server_side_msg2[MD_MAX_SIZE] = {0};/* msg received from client */
    byte client_side_msg1[MD_MAX_SIZE] = {0};/* msg sent by client */
    byte client_side_msg2[MD_MAX_SIZE] = {0};/* msg received from server */
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */
static THREAD_RETURN WOLFSSL_THREAD test_server_nofail(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;
    func_args* opts = (func_args*)args;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    int doUdp = 0;
    SOCKADDR_IN_T cliAddr;
    socklen_t     cliLen;
    const char* certFile = svrCertFile;
    const char* keyFile = svrKeyFile;

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len = 0;
#endif

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();

        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        goto done;
    }

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx, myTicketEncCb);
#endif
#endif
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    if (cbf != NULL)
        doUdp = cbf->doUdp;

    /* do it here to detect failure */
    tcp_accept(
        &sockfd, &clientfd, opts, port, 0, doUdp, 0, 0, 1, 0, 0);

    if (doUdp) {
        cliLen = sizeof(cliAddr);

        idx = (int)recvfrom(sockfd, input, sizeof(input), MSG_PEEK,
                  (struct sockaddr*)&cliAddr, &cliLen);

        AssertIntGT(idx, 0);
    }
    else {
        CloseSocket(sockfd);
    }

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->certPemFile != NULL)
        certFile = cbf->certPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->keyPemFile != NULL)
        keyFile = cbf->keyPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (doUdp) {
        err = wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
        if (err != WOLFSSL_SUCCESS)
            goto done;
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (args && opts->argc > 0) {
        /* set as nonblock and time out for waiting on read/write */
        tcp_set_nonblocking(&clientfd);
        wolfSSL_dtls_set_using_nonblock(ssl, 1);
    }
#endif
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
    SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (opts->argc > 0) {
        ret = nonblocking_accept_read(args, ssl, &clientfd);
        if (ret >= 0) {
            opts->return_code = TEST_SUCCESS;
        }
    #ifdef WOLFSSL_TIRTOS
        Task_yield();
    #endif
        goto done;
    }
#endif

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_accept failed");*/
        goto done;
    }

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, server_side_msg2, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_finished(ssl, server_side_msg1, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message: %s\n", input);
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
#ifdef WOLFSSL_TIRTOS
        return;
#else
        return 0;
#endif
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }

#ifndef WOLFSSL_TIRTOS
    return 0;
#endif
}

#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13)
static THREAD_RETURN WOLFSSL_THREAD test_server_loop(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    int loop_count = ((func_args*)args)->argc;
    int count = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    while (count != loop_count) {
        ssl = wolfSSL_new(ctx);
        if (ssl == NULL) {
            goto done;
        }
        if (sharedCtx && wolfSSL_use_certificate_file(ssl, svrCertFile,
                                        WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server cert chain file, "
                    "Please run from wolfSSL home dir");*/
            goto done;
        }
        if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, svrKeyFile,
                                        WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server key file, "
                    "Please run from wolfSSL home dir");*/
            goto done;
        }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif
        /* call ssl setup callback */
        if (cbf != NULL && cbf->ssl_ready != NULL) {
            cbf->ssl_ready(ssl);
        }
        /* do it here to detect failure */
        tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
        CloseSocket(sockfd);
        if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_accept(ssl);
            err = wolfSSL_get_error(ssl, 0);
        } while (err == WC_PENDING_E);
        if (ret != WOLFSSL_SUCCESS) {
            char buff[WOLFSSL_MAX_ERROR_SZ];
            fprintf(stderr, "error = %d, %s\n", err,
                wolfSSL_ERR_error_string(err, buff));
            /*err_sys("SSL_accept failed");*/
            goto done;
        }

        idx = wolfSSL_read(ssl, input, sizeof(input)-1);
        if (idx > 0) {
            input[idx] = '\0';
            fprintf(stderr, "Client message: %s\n", input);
        }

        if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
            /*err_sys("SSL_write failed");*/
    #ifdef WOLFSSL_TIRTOS
            return;
    #else
            return 0;
    #endif
        }
        /* free ssl for this connection */
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl); ssl = NULL;
        CloseSocket(clientfd);

        count++;
    }
#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    if (ssl != NULL) {
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl);
    }
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#ifndef WOLFSSL_TIRTOS
    return 0;
#endif
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13) */

typedef int (*cbType)(WOLFSSL_CTX *ctx, WOLFSSL *ssl);

static int test_client_nofail(void* args, cbType cb)
{
#if !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_CIPHER*  cipher;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  cipherSuite;
    int  sharedCtx = 0;
    int  doUdp = 0;
    const char* cipherName1, *cipherName2;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = cbf->isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

    if (cbf != NULL)
        doUdp = cbf->doUdp;

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                doUdp, 0, NULL);
    /* Connect the socket so that we don't have to set the peer later on */
    if (doUdp)
        udp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif

        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (!doUdp) {
        if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
    }
    else {
#ifdef WOLFSSL_DTLS
        if (wolfSSL_set_dtls_fd_connected(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
#else
        goto done;
#endif
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }

    /* test the various get cipher methods */
    /* Internal cipher suite names */
    cipherSuite = wolfSSL_get_current_cipher_suite(ssl);
    cipherName1 = wolfSSL_get_cipher_name(ssl);
    cipherName2 = wolfSSL_get_cipher_name_from_suite(
        (cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);

    /* IANA Cipher Suites Names */
    /* Unless WOLFSSL_CIPHER_INTERNALNAME or NO_ERROR_STRINGS,
        then it's the internal cipher suite name */
    cipher = wolfSSL_get_current_cipher(ssl);
    cipherName1 = wolfSSL_CIPHER_get_name(cipher);
    cipherName2 = wolfSSL_get_cipher(ssl);
    AssertStrEQ(cipherName1, cipherName2);
#if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
    !defined(WOLFSSL_QT)
    cipherName1 = wolfSSL_get_cipher_name_iana_from_suite(
            (cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);
#endif

    if (cb != NULL)
        (cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#else
    (void)args;
    (void)cb;
#endif /* !NO_WOLFSSL_CLIENT */
    return 0;
}

void test_wolfSSL_client_server_nofail(callback_functions* client_cb,
                                       callback_functions* server_cb)
{
    func_args client_args;
    func_args server_args;
    tcp_ready ready;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    server_args.callbacks = server_cb;
    client_args.signal = &ready;
    client_args.callbacks = client_cb;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    client_cb->return_code = client_args.return_code;
    server_cb->return_code = server_args.return_code;

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
}

#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
   !defined(WOLFSSL_TLS13) && !defined(NO_WOLFSSL_CLIENT)
static void test_client_reuse_WOLFSSLobj(void* args, void *cb, void* server_args)
{
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_SESSION* session = NULL;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  sharedCtx = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    /* keep handshakre resources for re-using WOLFSSL obj */
    wolfSSL_KeepArrays(ssl);
    if (wolfSSL_KeepHandshakeResources(ssl)) {
        /* err_sys("SSL_KeepHandshakeResources failed"); */
        goto done;
    }
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        ((cbType)cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    /* Session Resumption by re-using WOLFSSL object */
    wolfSSL_set_quiet_shutdown(ssl, 1);
    if (wolfSSL_shutdown(ssl) != WOLFSSL_SUCCESS) {
        /* err_sys ("SSL shutdown failed"); */
        goto done;
    }
    session = wolfSSL_get1_session(ssl);
    if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
        /* err_sys ("SSL_clear failed"); */
        goto done;
    }
    wolfSSL_set_session(ssl, session);
    wolfSSL_SESSION_free(session);
    session = NULL;
    /* close socket once */
    CloseSocket(sockfd);
    sockfd = 0;
    /* wait until server ready */
    wait_tcp_ready((func_args*)server_args);
    fprintf(stderr, "session resumption\n");
    /* Do re-connect  */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);
    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        ((cbType)cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    return;
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
         !defined(WOLFSSL_TLS13) && !defined(NO_WOLFSSL_CLIENT) */


static int test_client_verifyDepth(void* args)
{
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  verify_depth = ((func_args*)args)->argc;

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0)
                                                        != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);

    /* set verify depth */
    if (verify_depth == 0) {
        myVerifyAction = VERIFY_OVERRIDE_ERROR;
        SSL_CTX_set_verify_depth(ctx, verify_depth);
    }
    else if (verify_depth == -1) {
        myVerifyAction = VERIFY_USE_PREVERFIY;
        SSL_CTX_set_verify_depth(ctx, 0);
    }
    else if (verify_depth > 0) {
        myVerifyAction = VERIFY_USE_PREVERFIY;
        SSL_CTX_set_verify_depth(ctx, verify_depth);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        goto done;
    }

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sockfd);
#else
    (void)args;
#endif /* defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && !defined(NO_WOLFSSL_CLIENT) */

    return 0;
}

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
     defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)) && \
    defined(HAVE_ALPN) && defined(HAVE_SNI) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_BIO)
    #define HAVE_ALPN_PROTOS_SUPPORT
#endif

/* Generic TLS client / server with callbacks for API unit tests
 * Used by SNI / ALPN / crypto callback helper functions */
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    (defined(HAVE_SNI) || defined(HAVE_ALPN) || defined(WOLF_CRYPTO_CB) || \
     defined(HAVE_ALPN_PROTOS_SUPPORT)) || defined(WOLFSSL_STATIC_MEMORY)
    #define ENABLE_TLS_CALLBACK_TEST
#endif

#if defined(ENABLE_TLS_CALLBACK_TEST) || \
    (defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT))
/* TLS server for API unit testing - generic */
static THREAD_RETURN WOLFSSL_THREAD run_wolfssl_server(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    SOCKET_T     sfd = 0;
    SOCKET_T     cfd = 0;
    word16       port;

    char msg[] = "I hear you fa shizzle!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  idx;
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            return 0;
        }
    }
#else
    if (ctx == NULL) {
        ctx = wolfSSL_CTX_new(callbacks->method());
    }
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        return 0;
    }
#endif

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = cliCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = svrCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = svrKeyFile;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    wolfSSL_CTX_SetDevId(ctx, callbacks->devId);

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif
#if defined(WOLFSSL_SESSION_EXPORT) && defined(WOLFSSL_DTLS)
    if (callbacks->method == wolfDTLSv1_2_server_method) {
        AssertIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_CTX_dtls_set_export(ctx, test_export));
    }
#endif


    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
            WOLFSSL_FILETYPE_PEM));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
            WOLFSSL_FILETYPE_PEM));

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        AssertIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM));
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        fprintf(stderr, "SSL new failed\n");
        wolfSSL_CTX_free(ctx);
        return 0;
    }
    if (wolfSSL_dtls(ssl)) {
        SOCKADDR_IN_T cliAddr;
        socklen_t     cliLen;

        cliLen = sizeof(cliAddr);
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 1, 0, 0, 0, 0, 0);
        idx = (int)recvfrom(sfd, input, sizeof(input), MSG_PEEK,
                (struct sockaddr*)&cliAddr, &cliLen);
        AssertIntGT(idx, 0);
        wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
    }
    else {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
        CloseSocket(sfd);
    }

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_set_fd(ssl, cfd));

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM));

        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM));
    }

#ifdef NO_PSK
    #if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
    #elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
    #endif
#endif

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

#ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
#endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "accept error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_accept failed");*/
    }
    else {
    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            idx = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, idx);
        } while (err == WC_PENDING_E);
        if (idx > 0) {
            input[idx] = 0;
            fprintf(stderr, "Client message: %s\n", input);
        }

    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        AssertIntEQ(len, ret);

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(HAVE_IO_POOL) && \
        defined(WOLFSSL_DTLS)
        if (wolfSSL_dtls(ssl)) {
            byte*  import;
            word32 sz;

            wolfSSL_dtls_export(ssl, NULL, &sz);
            import = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            AssertNotNull(import);
            idx = wolfSSL_dtls_export(ssl, import, &sz);
            AssertIntGE(idx, 0);
            AssertIntGE(wolfSSL_dtls_import(ssl, import, idx), 0);
            XFREE(import, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
#endif
#ifdef WOLFSSL_TIRTOS
        Task_yield();
#endif
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);


#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#ifndef WOLFSSL_TIRTOS
    return 0;
#endif
}

/* TLS Client for API unit testing - generic */
static void run_wolfssl_client(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    SOCKET_T    sfd = 0;

    char msg[] = "hello wolfssl server!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = caCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = cliCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = cliKeyFile;

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            return;
        }
    }
#else
    if (ctx == NULL) {
        ctx = wolfSSL_CTX_new(callbacks->method());
    }
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        return;
    }
#endif

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (!callbacks->loadToSSL) {
        wolfSSL_CTX_SetDevId(ctx, callbacks->devId);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0));

    if (!callbacks->loadToSSL) {
        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM));

        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM));
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        AssertIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM));
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (wolfSSL_dtls(ssl)) {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    1, 0, ssl);
    }
    else {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    0, 0, ssl);
    }
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_set_fd(ssl, sfd));

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM));

        AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM));
    }

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
    }
    else {
        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        AssertIntEQ(len, ret);

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_PENDING_E) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_PENDING_E);
        if (ret > 0) {
            input[ret] = '\0'; /* null term */
            fprintf(stderr, "Server response: %s\n", input);
        }
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif
}

#endif /* ENABLE_TLS_CALLBACK_TEST */


static int test_wolfSSL_read_write(void)
{
    /* The unit testing for read and write shall happen simultaneously, since
     * one can't do anything with one without the other. (Except for a failure
     * test case.) This function will call all the others that will set up,
     * execute, and report their test findings.
     *
     * Set up the success case first. This function will become the template
     * for the other tests. This should eventually be renamed
     *
     * The success case isn't interesting, how can this fail?
     * - Do not give the client context a CA certificate. The connect should
     *   fail. Do not need server for this?
     * - Using NULL for the ssl object on server. Do not need client for this.
     * - Using NULL for the ssl object on client. Do not need server for this.
     * - Good ssl objects for client and server. Client write() without server
     *   read().
     * - Good ssl objects for client and server. Server write() without client
     *   read().
     * - Forgetting the password callback?
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    AssertTrue(client_args.return_code);
    AssertTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return TEST_RES_CHECK(1);
}

static int test_wolfSSL_reuse_WOLFSSLobj(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_TLS13)
    /* The unit test for session resumption by re-using WOLFSSL object.
     * WOLFSSL object is not cleared after first session. It re-use the obeject
     * for second connection.
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    /* the var is used for loop number */
    server_args.argc = 2;

    start_thread(test_server_loop, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_reuse_WOLFSSLobj(&client_args, NULL, &server_args);
    join_thread(serverThread);

    AssertTrue(client_args.return_code);
    AssertTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    res = TEST_RES_CHECK(1);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13) */
    return res;
}

static int test_wolfSSL_CTX_verifyDepth_ServerClient(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_TIRTOS) && !defined(NO_WOLFSSL_CLIENT)

    /* This unit test is to check set verify Depth */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cbf;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */

    client_args.callbacks = &client_cbf;

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    /* the var is used for loop number */
    server_args.argc = 1;

    /* test case 1 verify depth is equal to peer chain */
    {
        start_thread(test_server_nofail, &server_args, &serverThread);
        wait_tcp_ready(&server_args);

        /* the var is used for verify depth */
        client_args.argc = 2;

        test_client_verifyDepth(&client_args);
        join_thread(serverThread);
        AssertIntEQ(client_args.return_code, TEST_SUCCESS);
        AssertIntEQ(server_args.return_code, TEST_SUCCESS);
    }

    /* test case 2
     * verify depth is zero, number of peer's chain is 2.
     * verify result becomes MAX_CHAIN_ERROR, but it is overridden in
     * callback.
     */

    /* the var is used for verify depth 0 and VERIFY_OVERRIDE_ERROR   */
    {
        start_thread(test_server_nofail, &server_args, &serverThread);
        wait_tcp_ready(&server_args);

        client_args.argc = 0;
        test_client_verifyDepth(&client_args);
        join_thread(serverThread);
        AssertIntEQ(client_args.return_code, TEST_SUCCESS);
        AssertIntEQ(server_args.return_code, TEST_SUCCESS);
    }
    /* test case 3
     * verify depth is zero, number of peer's chain is 2
     * verify result becomes MAX_CHAIN_ERRO. call-back returns failure.
     * therefore, handshake becomes failure.
     */
    /* the var is used for verify depth 0 and VERIFY_USE_PREVERFIY   */
    {
        start_thread(test_server_nofail, &server_args, &serverThread);
        wait_tcp_ready(&server_args);

        client_args.argc = -1;
        test_client_verifyDepth(&client_args);
        join_thread(serverThread);
        AssertIntEQ(client_args.return_code, TEST_SUCCESS);
        AssertIntEQ(server_args.return_code, TEST_SUCCESS);
    }

    FreeTcpReady(&ready);

    res = TEST_RES_CHECK(1);
#else
    (void)test_client_verifyDepth;
#endif /* (OPENSSL_EXTRA) && !(WOLFSSL_TIRTOS) && (NO_WOLFSSL_CLIENT) */

    return res;
}


static int test_wolfSSL_CTX_set_cipher_list(void)
{
    int res = TEST_SKIPPED;

#if defined(OPENSSL_ALL) && defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_TIRTOS) && !defined(NO_AES) && !defined(WOLFSSL_NO_TLS12) \
    && !defined(NO_SHA256) && defined(HAVE_ECC)
    WOLFSSL_CTX* ctx;
    WOLFSSL_CTX* ctxClient;
    WOLFSSL*     sslClient;
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    callback_functions client_cb;
    callback_functions server_cb;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();
    InitTcpReady(&ready);

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));

    AssertNotNull((ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method())));
    AssertTrue(wolfSSL_CTX_set_cipher_list(ctx, "DEFAULT:!NULL"));
    AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    AssertIntEQ(WOLFSSL_SUCCESS,
          wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));


    AssertNotNull((ctxClient = wolfSSL_CTX_new(wolfTLSv1_2_client_method())));
    AssertTrue(wolfSSL_CTX_set_cipher_list(ctxClient, "ECDHE-RSA-AES128-SHA256"));

    client_cb.ctx = ctxClient;
    server_cb.ctx = ctx;

    /* we are responsible for free'ing WOLFSSL_CTX */
    server_cb.isSharedCtx = client_cb.isSharedCtx = 1;

    server_args.signal    = &ready;
    server_args.callbacks = &server_cb;
    client_args.signal    = &ready;
    client_args.callbacks = &client_cb;
    client_args.return_code = TEST_FAIL;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    wolfSSL_CTX_free(client_cb.ctx);
    wolfSSL_CTX_free(server_cb.ctx);

    AssertIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

    /* check with cipher string that has '+' */
    AssertNotNull((ctxClient = wolfSSL_CTX_new(wolfTLSv1_2_client_method())));
    AssertTrue(wolfSSL_CTX_set_cipher_list(ctxClient, "ECDHE+AESGCM"));
    AssertNotNull((sslClient = wolfSSL_new(ctxClient)));

    /* check for the existance of an ECDHE ECDSA cipher suite */
    {
        int i = 0;
        int found = 0;
        const char* suite;

        WOLF_STACK_OF(WOLFSSL_CIPHER)* sk;
        WOLFSSL_CIPHER* current;

        AssertNotNull((sk = wolfSSL_get_ciphers_compat(sslClient)));
        do {
            current = wolfSSL_sk_SSL_CIPHER_value(sk, i++);
            if (current) {
                suite = wolfSSL_CIPHER_get_name(current);
                if (suite && XSTRSTR(suite, "ECDSA")) {
                    found = 1;
                    break;
                }
            }
        } while (current);
        AssertIntEQ(found, 1);
    }

    wolfSSL_free(sslClient);
    wolfSSL_CTX_free(ctxClient);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_client_get_finished(void* args, cbType cb)
{
#if defined(WOLFSSL_HAVE_TLS_UNIQUE) && !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    WOLFSSL_METHOD* method  = NULL;
    size_t msg_len = 0;

    (void) args;
    (void) cb;

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

    if (cbf != NULL && cbf->method != NULL) {
        method = cbf->method();
    }
    else {
        method = wolfSSLv23_client_method();
    }
    ctx = wolfSSL_CTX_new(method);

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto done;
    }
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto done;
    }

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        goto done;
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        goto done;
    }

    /* get_finished test */
    /* 1. get own sent message */
    XMEMSET(client_side_msg1, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_finished(ssl, client_side_msg1, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);
    /* 2. get peer message */
    XMEMSET(client_side_msg2, 0, MD_MAX_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, client_side_msg2, MD_MAX_SIZE);
    AssertIntGE(msg_len, 0);

    if (cb != NULL)
        (cb)(ctx, ssl);

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_write(ssl, msg, msgSz);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_PENDING_E) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_read(ssl, reply, sizeof(reply)-1);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_PENDING_E);
    if (ret > 0) {
        reply[ret] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sockfd);

#else
    (void)args;
    (void)cb;
#endif /* WOLFSSL_HAVE_TLS_UNIQUE && !NO_WOLFSSL_CLIENT */

    return 0;
}

static int test_wolfSSL_get_finished(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_HAVE_TLS_UNIQUE)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_get_finished(&client_args, NULL);
    join_thread(serverThread);

    AssertTrue(client_args.return_code);
    AssertTrue(server_args.return_code);

    /* test received msg vs sent msg */
    AssertIntEQ(0, XMEMCMP(client_side_msg1, server_side_msg2, MD_MAX_SIZE));
    AssertIntEQ(0, XMEMCMP(client_side_msg2, server_side_msg1, MD_MAX_SIZE));

    FreeTcpReady(&ready);

    res = TEST_RES_CHECK(1);
#else
    (void)test_client_get_finished;
#endif /* !NO_RSA && WOLFSSL_HAVE_TLS_UNIQUE */

    return res;
}

#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)

/* Sessions to restore/store */
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_client_sess;
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_server_sess;
static WOLFSSL_CTX*     test_wolfSSL_CTX_add_session_server_ctx;

static void test_wolfSSL_CTX_add_session_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* Don't store sessions. Lookup is still enabled. */
    AssertIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    AssertIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
}

static void test_wolfSSL_CTX_add_session_on_result(WOLFSSL* ssl)
{
    WOLFSSL_SESSION** sess;
    if (wolfSSL_is_server(ssl))
        sess = &test_wolfSSL_CTX_add_session_server_sess;
    else
        sess = &test_wolfSSL_CTX_add_session_client_sess;
    if (*sess == NULL) {
#ifdef NO_SESSION_CACHE_REF
        AssertNotNull(*sess = wolfSSL_get1_session(ssl));
#else
        /* Test for backwards compatibility */
        if (wolfSSL_is_server(ssl)) {
            AssertNotNull(*sess = wolfSSL_get1_session(ssl));
        }
        else {
            AssertNotNull(*sess = wolfSSL_get_session(ssl));
        }
#endif
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
#endif
                )
            AssertIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        AssertIntEQ(wolfSSL_session_reused(ssl), 1);
    }
    /* Save CTX to be able to decrypt tickets */
    if (wolfSSL_is_server(ssl) &&
            test_wolfSSL_CTX_add_session_server_ctx == NULL) {
        AssertNotNull(test_wolfSSL_CTX_add_session_server_ctx
                = wolfSSL_get_SSL_CTX(ssl));
        AssertIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        AssertNotNull(peer);
        wolfSSL_X509_free(peer);
        AssertNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    #ifdef OPENSSL_EXTRA
        AssertNotNull(SSL_SESSION_get0_peer(*sess));
    #endif
    }
#endif /* SESSION_CERTS */
}

static void test_wolfSSL_CTX_add_session_ssl_ready(WOLFSSL* ssl)
{
    /* Set the session to reuse for the client */
    AssertIntEQ(wolfSSL_set_session(ssl,
            test_wolfSSL_CTX_add_session_client_sess), WOLFSSL_SUCCESS);
}
#endif

static int test_wolfSSL_CTX_add_session(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cb;
    callback_functions server_cb;
    method_provider methods[][2] = {
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
        /* Without AES there are almost no ciphersuites available. This leads
         * to no ciphersuites being available and an error. */
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method },
#endif
#ifndef WOLFSSL_NO_TLS12
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method },
#endif
        /* Needs the default ticket callback since it is tied to the
         * connection context and this makes it easy to carry over the ticket
         * crypto context between connections */
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET)
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method },
#endif
    };
    const size_t methodsLen = sizeof(methods)/sizeof(*methods);
    size_t i, j;

    for (i = 0; i < methodsLen; i++) {
        /* First run creates a connection while the second+ run will attempt
         * to resume the connection. The trick is that the internal cache
         * is turned off. wolfSSL_CTX_add_session should put the session in
         * the cache anyway. */
        test_wolfSSL_CTX_add_session_client_sess = NULL;
        test_wolfSSL_CTX_add_session_server_sess = NULL;
        test_wolfSSL_CTX_add_session_server_ctx = NULL;

        for (j = 0; j < 5; j++) {
#ifdef WOLFSSL_TIRTOS
            fdOpenSession(Task_self());
#endif

            StartTCP();
            InitTcpReady(&ready);

            XMEMSET(&client_args, 0, sizeof(func_args));
            XMEMSET(&server_args, 0, sizeof(func_args));

            XMEMSET(&client_cb, 0, sizeof(callback_functions));
            XMEMSET(&server_cb, 0, sizeof(callback_functions));
            client_cb.method  = methods[i][0];
            server_cb.method  = methods[i][1];

            server_args.signal    = &ready;
            server_args.callbacks = &server_cb;
            client_args.signal    = &ready;
            client_args.callbacks = &client_cb;

            if (test_wolfSSL_CTX_add_session_server_ctx != NULL) {
                server_cb.ctx = test_wolfSSL_CTX_add_session_server_ctx;
                server_cb.isSharedCtx = 1;
            }
            server_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            client_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            if (j != 0)
                client_cb.ssl_ready = test_wolfSSL_CTX_add_session_ssl_ready;
            server_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            client_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            server_cb.ticNoInit = 1; /* Use default builtin */

            start_thread(test_server_nofail, &server_args, &serverThread);
            wait_tcp_ready(&server_args);
            test_client_nofail(&client_args, NULL);
            join_thread(serverThread);

            AssertTrue(client_args.return_code);
            AssertTrue(server_args.return_code);

            FreeTcpReady(&ready);
        }
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_client_sess);
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_server_sess);
        wolfSSL_CTX_free(test_wolfSSL_CTX_add_session_server_ctx);
    }

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
/* canned export of a session using older version 3 */
static unsigned char version_3[] = {
    0xA5, 0xA3, 0x01, 0x88, 0x00, 0x3c, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x80, 0x0C, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x30,
    0x05, 0x09, 0x0A, 0x01, 0x01, 0x00, 0x0D, 0x05,
    0xFE, 0xFD, 0x01, 0x25, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x05, 0x00, 0x06, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x01, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x01, 0x01,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x05,
    0x12, 0xCF, 0x22, 0xA1, 0x9F, 0x1C, 0x39, 0x1D,
    0x31, 0x11, 0x12, 0x1D, 0x11, 0x18, 0x0D, 0x0B,
    0xF3, 0xE1, 0x4D, 0xDC, 0xB1, 0xF1, 0x39, 0x98,
    0x91, 0x6C, 0x48, 0xE5, 0xED, 0x11, 0x12, 0xA0,
    0x00, 0xF2, 0x25, 0x4C, 0x09, 0x26, 0xD1, 0x74,
    0xDF, 0x23, 0x40, 0x15, 0x6A, 0x42, 0x2A, 0x26,
    0xA5, 0xAC, 0x56, 0xD5, 0x4A, 0x20, 0xB7, 0xE9,
    0xEF, 0xEB, 0xAF, 0xA8, 0x1E, 0x23, 0x7C, 0x04,
    0xAA, 0xA1, 0x6D, 0x92, 0x79, 0x7B, 0xFA, 0x80,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x0C, 0x79, 0x7B, 0xFA, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xA1, 0x6D,
    0x92, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x20, 0x00, 0x04, 0x00,
    0x10, 0x00, 0x10, 0x08, 0x02, 0x05, 0x08, 0x01,
    0x30, 0x28, 0x00, 0x00, 0x0F, 0x00, 0x02, 0x00,
    0x09, 0x31, 0x32, 0x37, 0x2E, 0x30, 0x2E, 0x30,
    0x2E, 0x31, 0xED, 0x4F
};
#endif /* defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) */

static int test_wolfSSL_dtls_export(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;
    callback_functions client_cbf;
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    client_cbf.method = wolfDTLSv1_2_client_method;
    server_args.callbacks = &server_cbf;
    client_args.callbacks = &client_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    AssertTrue(client_args.return_code);
    AssertTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    {
        SOCKET_T sockfd = 0;
        WOLFSSL_CTX* ctx;
        WOLFSSL*     ssl;
        char msg[64] = "hello wolfssl!";
        char reply[1024];
        int  msgSz = (int)XSTRLEN(msg);
        byte *session, *window;
        unsigned int sessionSz, windowSz;

#ifndef TEST_IPV6
        struct sockaddr_in peerAddr;
#else
        struct sockaddr_in6 peerAddr;
#endif /* TEST_IPV6 */

        int i;


        /* Set ctx to DTLS 1.2 */
        AssertNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method()));
        AssertNotNull(ssl = wolfSSL_new(ctx));

        /* test importing version 3 */
        AssertIntGE(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);

        /* test importing bad length and bad version */
        version_3[2] += 1;
        AssertIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        version_3[2] -= 1; version_3[1] = 0XA0;
        AssertIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);


    /* check storing client state after connection and storing window only */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    server_cbf.doUdp = 1;
    server_args.callbacks = &server_cbf;
    server_args.argc = 3; /* set loop_count to 3 */


    server_args.signal = &ready;
    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* create and connect with client */
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));
    AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    AssertIntEQ(WOLFSSL_SUCCESS,
          wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    AssertIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 1, 0, NULL);
    AssertNotNull(ssl = wolfSSL_new(ctx));
    AssertIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);

    /* store server information connected too */
    XMEMSET(&peerAddr, 0, sizeof(peerAddr));
#ifndef TEST_IPV6
    peerAddr.sin_family = AF_INET;
    AssertIntEQ(XINET_PTON(AF_INET, wolfSSLIP, &peerAddr.sin_addr),1);
    peerAddr.sin_port = XHTONS(server_args.signal->port);
#else
    peerAddr.sin6_family = AF_INET6;
    AssertIntEQ(
        XINET_PTON(AF_INET6, wolfSSLIP, &peerAddr.sin6_addr),1);
    peerAddr.sin6_port = XHTONS(server_args.signal->port);
#endif

    AssertIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);

    AssertIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_dtls_export(ssl, NULL, &sessionSz), 0);
    session = (byte*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    AssertIntGT(wolfSSL_dtls_export(ssl, session, &sessionSz), 0);
    AssertIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    AssertIntGT(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
    AssertIntEQ(wolfSSL_dtls_export_state_only(ssl, NULL, &windowSz), 0);
    window = (byte*)XMALLOC(windowSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    AssertIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
    wolfSSL_free(ssl);

    for (i = 1; i < server_args.argc; i++) {
        /* restore state */
        AssertNotNull(ssl = wolfSSL_new(ctx));
        AssertIntGT(wolfSSL_dtls_import(ssl, session, sessionSz), 0);
        AssertIntGT(wolfSSL_dtls_import(ssl, window, windowSz), 0);
        AssertIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);
        AssertIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);
        AssertIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
        AssertIntGE(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
        AssertIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
        wolfSSL_free(ssl);
    }
    XFREE(session, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(window, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_CTX_free(ctx);

    fprintf(stderr, "done and waiting for server\n");
    join_thread(serverThread);
    AssertIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    }

    res = TEST_RES_CHECK(1);
#endif

    return res;
}


#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
#ifdef WOLFSSL_TLS13
static const byte canned_client_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x09, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x35, 0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0,
    0x6F, 0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A,
    0xA0, 0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};

static const byte canned_server_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x01, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x00, 0x0F,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0xD3, 0xB7, 0xEE, 0x3A, 0xA0, 0x8E, 0xA1, 0x4D,
    0xC3, 0x2E, 0x5E, 0x06, 0x35, 0x41, 0xCD, 0xF3,
    0x49, 0x31, 0x08, 0xD0, 0x6F, 0x02, 0x3D, 0xC1,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};
#endif /* WOLFSSL_TLS13 */

static const byte canned_client_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x0A, 0x0D, 0x10, 0x01, 0x01, 0x0A, 0x00,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};


static const byte canned_server_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x08, 0x0F, 0x10, 0x01, 0x01, 0x00, 0x11,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};


static THREAD_RETURN WOLFSSL_THREAD tls_export_server(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfTLSv1_2_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        goto done;
    }
    wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    /* do it here to detect failure */
    tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    CloseSocket(sockfd);

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    wolfSSL_set_fd(ssl, clientfd);

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }
    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message export/import: %s\n", input);
    }
    else {
        fprintf(stderr, "ret = %d error = %d\n", idx,
            wolfSSL_get_error(ssl, idx));
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
#ifdef WOLFSSL_TIRTOS
        return;
#else
        return 0;
#endif
    }

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AESGCM)
    OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
    TicketCleanup();
#endif
#endif

#ifndef WOLFSSL_TIRTOS
    return 0;
#endif
}


static void load_tls12_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_session,
                sizeof(canned_server_session)), sizeof(canned_server_session));
    wolfSSL_set_fd(ssl, clientfd);
}


#ifdef WOLFSSL_TLS13
static void load_tls13_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_tls13_session,
            sizeof(canned_server_tls13_session)),
            sizeof(canned_server_tls13_session));
    wolfSSL_set_fd(ssl, clientfd);
}
#endif


/* v is for version WOLFSSL_TLSV1_2 or WOLFSSL_TLSV1_3 */
static int test_wolfSSL_tls_export_run(int v)
{
    SOCKET_T sockfd = 0;
    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    char msg[64] = "hello wolfssl!";
    char reply[1024];
    word32 replySz;
    int  msgSz = (int)XSTRLEN(msg);
    const byte* clientSession = NULL;
    int   clientSessionSz = 0;

    tcp_ready ready;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    switch (v) {
        case WOLFSSL_TLSV1_2:
            server_cbf.method     = wolfTLSv1_2_server_method;
            server_cbf.ssl_ready  = load_tls12_canned_server;

            /* setup the client side */
            AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
            wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");
            clientSession   = canned_client_session;
            clientSessionSz = sizeof(canned_client_session);
            break;
    #ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            server_cbf.method     = wolfTLSv1_3_server_method;
            server_cbf.ssl_ready  = load_tls13_canned_server;

            /* setup the client side */
            AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
            clientSession   = canned_client_tls13_session;
            clientSessionSz = sizeof(canned_client_tls13_session);
            break;
    #endif
    }
    server_args.callbacks = &server_cbf;
    server_args.signal    = &ready;

    start_thread(tls_export_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);


#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    AssertNotNull(ssl = wolfSSL_new(ctx));
    tcp_connect(&sockfd, wolfSSLIP, ready.port, 0, 0, ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, clientSession, clientSessionSz),
                clientSessionSz);
    replySz = sizeof(reply);
    AssertIntGT(wolfSSL_tls_export(ssl, (byte*)reply, &replySz), 0);
#if !defined(NO_PSK) && defined(HAVE_ANON)
    /* index 20 has is setting if PSK was on and 49 is if anon is allowed */
    AssertIntEQ(XMEMCMP(reply, clientSession, replySz), 0);
#endif
    wolfSSL_set_fd(ssl, sockfd);

    AssertIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    AssertIntGT(wolfSSL_read(ssl, reply, sizeof(reply)-1), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    join_thread(serverThread);

    AssertIntEQ(server_args.return_code, TEST_SUCCESS);
    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif


    return TEST_RES_CHECK(1);
}
#endif

static int test_wolfSSL_tls_export(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_2);
    #ifdef WOLFSSL_TLS13
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_3);
    #endif
    res = TEST_RES_CHECK(1);
#endif

    return res;
}

/*----------------------------------------------------------------------------*
 | TLS extensions tests
 *----------------------------------------------------------------------------*/

#ifdef ENABLE_TLS_CALLBACK_TEST
/* Connection test runner - generic */
static void test_wolfSSL_client_server(callback_functions* client_callbacks,
                                       callback_functions* server_callbacks)
{
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();

    client_args.callbacks = client_callbacks;
    server_args.callbacks = server_callbacks;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    /* RUN Server side */
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* RUN Client side */
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    FreeTcpReady(&ready);
#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    client_callbacks->return_code = client_args.return_code;
    server_callbacks->return_code = server_args.return_code;
}
#endif /* ENABLE_TLS_CALLBACK_TEST */


#ifdef HAVE_SNI
static int test_wolfSSL_UseSNI_params(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    AssertNotNull(ctx);
    AssertNotNull(ssl);

    /* invalid [ctx|ssl] */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(NULL, 0, "ctx", 3));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    NULL, 0, "ssl", 3));
    /* invalid type */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx, -1, "ctx", 3));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl, -1, "ssl", 3));
    /* invalid data */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, NULL,  3));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, NULL,  3));
    /* success case */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, "ctx", 3));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, "ssl", 3));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT */

    return res;
}

/* BEGIN of connection tests callbacks */
static void use_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_UseSNI(ctx, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void use_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void different_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "ww2.wolfssl.com", 15));
}

static void use_SNI_WITH_CONTINUE_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                              WOLFSSL_SNI_CONTINUE_ON_MISMATCH);
}

static void use_SNI_WITH_FAKE_ANSWER_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                WOLFSSL_SNI_ANSWER_ON_MISMATCH);
}

static void use_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_MANDATORY_SNI_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_PSEUDO_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                 WOLFSSL_SNI_ANSWER_ON_MISMATCH | WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void verify_UNKNOWN_SNI_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(UNKNOWN_SNI_HOST_NAME_E, wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_ABSENT_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(SNI_ABSENT_ERROR, wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_no_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = (void*) &type; /* to be overwritten */

    AssertIntEQ(WOLFSSL_SNI_NO_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertNotNull(request);
    AssertIntEQ(0, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNull(request);
}

static void verify_SNI_real_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_REAL_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("www.wolfssl.com", (char*)request);
}

static void verify_SNI_fake_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_FAKE_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("ww2.wolfssl.com", (char*)request);
}

static void verify_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(FATAL_ERROR, wolfSSL_get_error(ssl, 0));
}
/* END of connection tests callbacks */

static int test_wolfSSL_UseSNI_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;
    size_t i;

    struct {
        method_provider client_meth;
        method_provider server_meth;
    } methods[] = {
#if defined(WOLFSSL_NO_TLS12) && !defined(WOLFSSL_TLS13)
        {wolfSSLv23_client_method, wolfSSLv23_server_method},
#endif
#ifndef WOLFSSL_NO_TLS12
        {wolfTLSv1_2_client_method, wolfTLSv1_2_server_method},
#endif
#ifdef WOLFSSL_TLS13
        {wolfTLSv1_3_client_method, wolfTLSv1_3_server_method},
#endif
    };

    for (i = 0; i < (sizeof(methods)/sizeof(*methods)); i++) {
    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = methods[i].client_meth;
    server_cb.method = methods[i].server_meth;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* success case at ctx */
    printf("success case at ctx\n");
    client_cb.ctx_ready = use_SNI_at_ctx; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
    server_cb.ctx_ready = use_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case at ssl */
    printf("success case at ssl\n");
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_SNI_at_ssl; client_cb.on_result = verify_SNI_real_matching;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_real_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* default mismatch behavior */
    printf("default mismatch behavior\n");
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = verify_FATAL_ERROR_on_client;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl;       server_cb.on_result = verify_UNKNOWN_SNI_on_server;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* continue on mismatch */
    printf("continue on mismatch\n");
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;         client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_CONTINUE_at_ssl; server_cb.on_result = verify_SNI_no_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* fake answer on mismatch */
    printf("fake answer on mismatch\n");
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;            client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_FAKE_ANSWER_at_ssl; server_cb.on_result = verify_SNI_fake_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* sni abort - success */
    printf("sni abort - success\n");
    client_cb.ctx_ready = use_SNI_at_ctx;           client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
    server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* sni abort - abort when absent (ctx) */
    printf("sni abort - abort when absent (ctx)\n");
    client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL; client_cb.on_result = verify_FATAL_ERROR_on_client;
    server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_ABSENT_on_server;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* sni abort - abort when absent (ssl) */
    printf("sni abort - abort when absent (ssl)\n");
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;                     client_cb.on_result = verify_FATAL_ERROR_on_client;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_MANDATORY_SNI_at_ssl; server_cb.on_result = verify_SNI_ABSENT_on_server;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* sni abort - success when overwritten */
    printf("sni abort - success when overwritten\n");
    client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL;           client_cb.on_result = NULL;
    server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_no_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* sni abort - success when allowing mismatches */
    printf("sni abort - success when allowing mismatches\n");
    client_cb.ctx_ready = NULL;                            client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = NULL;
    server_cb.ctx_ready = use_PSEUDO_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL;                 server_cb.on_result = verify_SNI_fake_matching;
    test_wolfSSL_client_server(&client_cb, &server_cb);
    }

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

    return res;
}

static int test_wolfSSL_SNI_GetFromBuffer(void)
{
    byte buff[] = { /* www.paypal.com */
        0x00, 0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x00, 0x60, 0x03, 0x03, 0x5c,
        0xc4, 0xb3, 0x8c, 0x87, 0xef, 0xa4, 0x09, 0xe0, 0x02, 0xab, 0x86, 0xca,
        0x76, 0xf0, 0x9e, 0x01, 0x65, 0xf6, 0xa6, 0x06, 0x13, 0x1d, 0x0f, 0xa5,
        0x79, 0xb0, 0xd4, 0x77, 0x22, 0xeb, 0x1a, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x13, 0x00, 0x11, 0x00, 0x00, 0x0e, 0x77, 0x77, 0x77,
        0x2e, 0x70, 0x61, 0x79, 0x70, 0x61, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x00,
        0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff2[] = { /* api.textmate.org */
        0x16, 0x03, 0x01, 0x00, 0xc6, 0x01, 0x00, 0x00, 0xc2, 0x03, 0x03, 0x52,
        0x8b, 0x7b, 0xca, 0x69, 0xec, 0x97, 0xd5, 0x08, 0x03, 0x50, 0xfe, 0x3b,
        0x99, 0xc3, 0x20, 0xce, 0xa5, 0xf6, 0x99, 0xa5, 0x71, 0xf9, 0x57, 0x7f,
        0x04, 0x38, 0xf6, 0x11, 0x0b, 0xb8, 0xd3, 0x00, 0x00, 0x5e, 0x00, 0xff,
        0xc0, 0x24, 0xc0, 0x23, 0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x07, 0xc0, 0x08,
        0xc0, 0x28, 0xc0, 0x27, 0xc0, 0x14, 0xc0, 0x13, 0xc0, 0x11, 0xc0, 0x12,
        0xc0, 0x26, 0xc0, 0x25, 0xc0, 0x2a, 0xc0, 0x29, 0xc0, 0x05, 0xc0, 0x04,
        0xc0, 0x02, 0xc0, 0x03, 0xc0, 0x0f, 0xc0, 0x0e, 0xc0, 0x0c, 0xc0, 0x0d,
        0x00, 0x3d, 0x00, 0x3c, 0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x67, 0x00, 0x6b, 0x00, 0x33, 0x00, 0x39, 0x00, 0x16,
        0x00, 0xaf, 0x00, 0xae, 0x00, 0x8d, 0x00, 0x8c, 0x00, 0x8a, 0x00, 0x8b,
        0x00, 0xb1, 0x00, 0xb0, 0x00, 0x2c, 0x00, 0x3b, 0x01, 0x00, 0x00, 0x3b,
        0x00, 0x00, 0x00, 0x15, 0x00, 0x13, 0x00, 0x00, 0x10, 0x61, 0x70, 0x69,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x6d, 0x61, 0x74, 0x65, 0x2e, 0x6f, 0x72,
        0x67, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00, 0x18, 0x00,
        0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x0d, 0x00, 0x0c, 0x00,
        0x0a, 0x05, 0x01, 0x04, 0x01, 0x02, 0x01, 0x04, 0x03, 0x02, 0x03
    };

    byte buff3[] = { /* no sni extension */
        0x16, 0x03, 0x03, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x49, 0x03, 0x03, 0xea,
        0xa1, 0x9f, 0x60, 0xdd, 0x52, 0x12, 0x13, 0xbd, 0x84, 0x34, 0xd5, 0x1c,
        0x38, 0x25, 0xa8, 0x97, 0xd2, 0xd5, 0xc6, 0x45, 0xaf, 0x1b, 0x08, 0xe4,
        0x1e, 0xbb, 0xdf, 0x9d, 0x39, 0xf0, 0x65, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x0a,
        0x00, 0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff4[] = { /* last extension has zero size */
        0x16, 0x03, 0x01, 0x00, 0xba, 0x01, 0x00, 0x00,
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
        0x00, 0x28, 0xcc, 0x14, 0xcc, 0x13, 0xc0, 0x2b, 0xc0, 0x2f, 0x00, 0x9e,
        0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x13, 0xc0, 0x14, 0xc0, 0x07, 0xc0, 0x11,
        0x00, 0x33, 0x00, 0x32, 0x00, 0x39, 0x00, 0x9c, 0x00, 0x2f, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00, 0x00, 0x65, 0xff, 0x01,
        0x00, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00,
        0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x23, 0x00,
        0x00, 0x33, 0x74, 0x00, 0x00, 0x00, 0x10, 0x00, 0x1b, 0x00, 0x19, 0x06,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33, 0x08, 0x73, 0x70, 0x64, 0x79, 0x2f,
        0x33, 0x2e, 0x31, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31,
        0x75, 0x50, 0x00, 0x00, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x12, 0x00, 0x10, 0x04, 0x01, 0x05, 0x01, 0x02,
        0x01, 0x04, 0x03, 0x05, 0x03, 0x02, 0x03, 0x04, 0x02, 0x02, 0x02, 0x00,
        0x12, 0x00, 0x00
    };

    byte buff5[] = { /* SSL v2.0 client hello */
        0x00, 0x2b, 0x01, 0x03, 0x01, 0x00, 0x09, 0x00, 0x00,
        /* dummy bytes bellow, just to pass size check */
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
    };

    byte result[32] = {0};
    word32 length   = 32;

    AssertIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff4, sizeof(buff4),
                                                           0, result, &length));

    AssertIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff3, sizeof(buff3),
                                                           0, result, &length));

    AssertIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           1, result, &length));

    AssertIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[0] = 0x16;

    AssertIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[1] = 0x03;

    AssertIntEQ(SNI_UNSUPPORTED, wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[2] = 0x03;

    AssertIntEQ(INCOMPLETE_DATA, wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[4] = 0x64;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    result[length] = 0;
    AssertStrEQ("www.paypal.com", (const char*) result);

    length = 32;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           0, result, &length));
    result[length] = 0;
    AssertStrEQ("api.textmate.org", (const char*) result);

    /* SSL v2.0 tests */
    AssertIntEQ(SNI_UNSUPPORTED, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x02;
    AssertIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x01; buff5[6] = 0x08;
    AssertIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[6] = 0x09; buff5[8] = 0x01;
    AssertIntEQ(BUFFER_ERROR, wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    return TEST_RES_CHECK(1);
}

#endif /* HAVE_SNI */

#endif /* HAVE_IO_TESTS_DEPENDENCIES */

static int test_wolfSSL_UseTrustedCA(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_TRUSTED_CA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx;
    WOLFSSL     *ssl;
    byte        id[20];

#ifndef NO_WOLFSSL_SERVER
    AssertNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_server_method())));
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#else
    AssertNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_client_method())));
#endif
    AssertNotNull((ssl = wolfSSL_new(ctx)));
    XMEMSET(id, 0, sizeof(id));

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(NULL, 0, NULL, 0));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1+1, NULL, 0));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, NULL, 0));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, id, 5));
#ifdef NO_SHA
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 0));

    /* success cases */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_PRE_AGREED, NULL, 0));
#ifndef NO_SHA
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 5));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* HAVE_TRUSTED_CA */
    return res;
}

static int test_wolfSSL_UseMaxFragment(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_MAX_FRAGMENT) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)

#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl;
  #ifdef OPENSSL_EXTRA
    int (*UseMaxFragment)(SSL *s, uint8_t mode);
    int (*CTX_UseMaxFragment)(SSL_CTX *c, uint8_t mode);
  #else
    int (*UseMaxFragment)(WOLFSSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(WOLFSSL_CTX *c, unsigned char mode);
  #endif

  #ifndef NO_WOLFSSL_SERVER
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    AssertNotNull(ctx);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

  #ifdef OPENSSL_EXTRA
    CTX_UseMaxFragment = SSL_CTX_set_tlsext_max_fragment_length;
    UseMaxFragment = SSL_set_tlsext_max_fragment_length;
  #else
    UseMaxFragment = wolfSSL_UseMaxFragment;
    CTX_UseMaxFragment = wolfSSL_CTX_UseMaxFragment;
  #endif

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(NULL, WOLFSSL_MFL_2_9));
    AssertIntNE(WOLFSSL_SUCCESS, UseMaxFragment(    NULL, WOLFSSL_MFL_2_9));
    AssertIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MIN-1));
    AssertIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MAX+1));
    AssertIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MIN-1));
    AssertIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MAX+1));

    /* success case */
  #ifdef OPENSSL_EXTRA
    AssertIntEQ(BAD_FUNC_ARG, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #else
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #endif
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_9));
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_10));
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_11));
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_12));
  #ifdef OPENSSL_EXTRA
    AssertIntEQ(BAD_FUNC_ARG, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_2_13));

    AssertIntEQ(BAD_FUNC_ARG, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #else
    AssertIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_13));

    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #endif
    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_9));
    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_10));
    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_11));
    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_12));

  #ifdef OPENSSL_EXTRA
    AssertIntEQ(BAD_FUNC_ARG, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #else
    AssertIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return res;
}

static int test_wolfSSL_UseTruncatedHMAC(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_TRUNCATED_HMAC) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl;

    AssertNotNull(ctx);

  #ifndef NO_WOLFSSL_SERVER
    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(NULL));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(NULL));

    /* success case */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(ctx));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return res;
}

static int test_wolfSSL_UseSupportedCurve(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_SUPPORTED_CURVES) && !defined(NO_WOLFSSL_CLIENT) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    AssertNotNull(ctx);
    AssertNotNull(ssl);

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS,
                      wolfSSL_CTX_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSupportedCurve(ctx,  0));

    AssertIntNE(WOLFSSL_SUCCESS,
                          wolfSSL_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSupportedCurve(ssl,  0));

    /* success case */
    AssertIntEQ(WOLFSSL_SUCCESS,
                       wolfSSL_CTX_UseSupportedCurve(ctx, WOLFSSL_ECC_SECP256R1));
    AssertIntEQ(WOLFSSL_SUCCESS,
                           wolfSSL_UseSupportedCurve(ssl, WOLFSSL_ECC_SECP256R1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

#if defined(HAVE_ALPN) && defined(HAVE_IO_TESTS_DEPENDENCIES)

static void verify_ALPN_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(UNKNOWN_ALPN_PROTOCOL_NAME_E, wolfSSL_get_error(ssl, 0));
}

static void use_ALPN_all(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_all_continue(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void use_ALPN_one(WOLFSSL* ssl)
{
    /* spdy/2 */
    char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown_continue(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void verify_ALPN_not_matching_spdy3(WOLFSSL* ssl)
{
    /* spdy/3 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};

    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntNE(1, sizeof(nego_proto) == protoSz);
    if (proto) {
        AssertIntNE(0, XMEMCMP(nego_proto, proto, sizeof(nego_proto)));
    }
}

static void verify_ALPN_not_matching_continue(WOLFSSL* ssl)
{
    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_ALPN_NOT_FOUND,
                wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, (0 == protoSz));
    AssertIntEQ(1, (NULL == proto));
}

static void verify_ALPN_matching_http1(WOLFSSL* ssl)
{
    /* http/1.1 */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_matching_spdy2(WOLFSSL* ssl)
{
    /* spdy/2 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_client_list(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char    *clist = NULL;
    word16  clistSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetPeerProtocol(ssl, &clist,
                                                          &clistSz));

    /* check value */
    AssertIntEQ(1, sizeof(alpn_list) == clistSz);
    AssertIntEQ(0, XMEMCMP(alpn_list, clist, clistSz));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_FreePeerProtocol(ssl, &clist));
}

static int test_wolfSSL_UseALPN_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* success case same list */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for server */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_one; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case none for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;         client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case mismatch behavior but option 'continue' set */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all_continue;     client_cb.on_result = verify_ALPN_not_matching_continue;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown_continue; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case read protocol send by client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_client_list;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* mismatch behavior with same list
        * the first and only this one must be taken */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_not_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* default mismatch behavior */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all;     client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown; server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

static int test_wolfSSL_UseALPN_params(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_WOLFSSL_CLIENT
    /* "http/1.1" */
    char http1[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    /* "spdy/1" */
    char spdy1[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x31};
    /* "spdy/2" */
    char spdy2[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    /* "spdy/3" */
    char spdy3[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char buff[256];
    word32 idx;

    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    AssertNotNull(ctx);
    AssertNotNull(ssl);

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(NULL, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, NULL, 0,
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* success case */
    /* http1 only */
    AssertIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(ssl, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy2, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* spdy3, http1, spdy2, spdy1 */
    XMEMCPY(buff, spdy3, sizeof(spdy3));
    idx = sizeof(spdy3);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, http1, sizeof(http1));
    idx += sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}
#endif /* HAVE_ALPN  */

#ifdef HAVE_ALPN_PROTOS_SUPPORT
static void CTX_set_alpn_protos(SSL_CTX *ctx)
{
    unsigned char p[] = {
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_CTX_set_alpn_protos(ctx, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif
}

static void set_alpn_protos(SSL* ssl)
{
    unsigned char p[] = {
       6, 's', 'p', 'd', 'y', '/', '3',
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_set_alpn_protos(ssl, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif

}

static void verify_alpn_matching_spdy3(WOLFSSL* ssl)
{
    /* "spdy/3" */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_alpn_matching_http1(WOLFSSL* ssl)
{
    /* "http/1.1" */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static int test_wolfSSL_set_alpn_protos(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* use CTX_alpn_protos */
    client_cb.ctx_ready = CTX_set_alpn_protos; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
    server_cb.ctx_ready = CTX_set_alpn_protos; server_cb.ssl_ready = NULL; server_cb.on_result = verify_alpn_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* use set_alpn_protos */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = set_alpn_protos; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = set_alpn_protos; server_cb.on_result = verify_alpn_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

#endif /* HAVE_ALPN_PROTOS_SUPPORT */

static int test_wolfSSL_DisableExtendedMasterSecret(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_EXTENDED_MASTER) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    AssertNotNull(ctx);
    AssertNotNull(ssl);

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_DisableExtendedMasterSecret(NULL));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_DisableExtendedMasterSecret(NULL));

    /* success cases */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_DisableExtendedMasterSecret(ctx));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_DisableExtendedMasterSecret(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_wolfSSL_UseSecureRenegotiation(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_SECURE_RENEGOTIATION) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    AssertNotNull(ctx);
    AssertNotNull(ssl);

    /* error cases */
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(NULL));
    AssertIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(NULL));

    /* success cases */
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

/* Test reconnecting with a different ciphersuite after a renegotiation. */
static int test_wolfSSL_SCR_Reconnect(void)
{
    int res = TEST_SKIPPED;

#if defined(HAVE_SECURE_RENEGOTIATION) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256)
    struct test_memio_ctx test_ctx;
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    byte data;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    test_ctx.c_ciphers = "ECDHE-RSA-AES256-GCM-SHA384";
    test_ctx.s_ciphers =
        "ECDHE-RSA-AES256-GCM-SHA384:ECDHE-RSA-CHACHA20-POLY1305";
    AssertIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_c));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_s));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_c));
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_s));
    AssertIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    /* WOLFSSL_FATAL_ERROR since it will block */
    AssertIntEQ(wolfSSL_Rehandshake(ssl_s), WOLFSSL_FATAL_ERROR);
    AssertIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
                WOLFSSL_ERROR_WANT_READ);
    AssertIntEQ(wolfSSL_read(ssl_c, &data, 1), WOLFSSL_FATAL_ERROR);
    AssertIntEQ(wolfSSL_get_error(ssl_s, WOLFSSL_FATAL_ERROR),
                WOLFSSL_ERROR_WANT_READ);
    AssertIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_c);
    ssl_c = NULL;
    wolfSSL_free(ssl_s);
    ssl_s = NULL;
    wolfSSL_CTX_free(ctx_c);
    ctx_c = NULL;
    test_ctx.c_ciphers = "ECDHE-RSA-CHACHA20-POLY1305";
    AssertIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    AssertIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_s);
    wolfSSL_free(ssl_c);
    wolfSSL_CTX_free(ctx_s);
    wolfSSL_CTX_free(ctx_c);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

#if !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_SERVER) && \
    (!defined(NO_RSA) || defined(HAVE_ECC))
/* Called when writing. */
static int DummySend(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    (void)ssl;
    (void)buf;
    (void)sz;
    (void)ctx;

    /* Force error return from wolfSSL_accept_TLSv13(). */
    return WANT_WRITE;
}
/* Called when reading. */
static int BufferInfoRecv(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    WOLFSSL_BUFFER_INFO* msg = (WOLFSSL_BUFFER_INFO*)ctx;
    int len = (int)msg->length;

    (void)ssl;
    (void)sz;

    /* Pass back as much of message as will fit in buffer. */
    if (len > sz)
        len = sz;
    XMEMCPY(buf, msg->buffer, len);
    /* Move over returned data. */
    msg->buffer += len;
    msg->length -= len;

    /* Amount actually copied. */
    return len;
}
#endif

/* Test the detection of duplicate known TLS extensions.
 * Specifically in a ClientHello.
 */
static int test_tls_ext_duplicate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_SERVER) && (!defined(NO_RSA) || defined(HAVE_ECC)) && \
    !defined(NO_FILESYSTEM)
    const unsigned char clientHelloDupTlsExt[] = {
        0x16, 0x03, 0x03, 0x00, 0x6a, 0x01, 0x00, 0x00,
        0x66, 0x03, 0x03, 0xf4, 0x65, 0xbd, 0x22, 0xfe,
        0x6e, 0xab, 0x66, 0xdd, 0xcf, 0xe9, 0x65, 0x55,
        0xe8, 0xdf, 0xc3, 0x8e, 0x4b, 0x00, 0xbc, 0xf8,
        0x23, 0x57, 0x1b, 0xa0, 0xc8, 0xa9, 0xe2, 0x8c,
        0x91, 0x6e, 0xf9, 0x20, 0xf7, 0x5c, 0xc5, 0x5b,
        0x75, 0x8c, 0x47, 0x0a, 0x0e, 0xc4, 0x1a, 0xda,
        0xef, 0x75, 0xe5, 0x21, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x13, 0x01,
        0x00, 0x9e, 0x01, 0x00,
        /* Extensions - duplicate signature algorithms. */
                                0x00, 0x19, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01,
        /* Supported Versions extension for TLS 1.3. */
                                            0x00, 0x2b,
        0x00, 0x05, 0x04, 0x03, 0x04, 0x03, 0x03
    };
    WOLFSSL_BUFFER_INFO msg;
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX *ctx;
    WOLFSSL     *ssl;

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
    AssertNotNull(ctx);

    AssertTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
        WOLFSSL_FILETYPE_PEM));
    AssertTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
        WOLFSSL_FILETYPE_PEM));

    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, BufferInfoRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, DummySend);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    msg.buffer = (unsigned char*)clientHelloDupTlsExt;
    msg.length = (unsigned int)sizeof(clientHelloDupTlsExt);
    wolfSSL_SetIOReadCtx(ssl, &msg);

    AssertIntNE(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_get_error(ssl, 0), DUPLICATE_TLS_EXT_E);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

/*----------------------------------------------------------------------------*
 | X509 Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_X509_NAME_get_entry(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_RSA)
#if defined(OPENSSL_ALL) || \
        (defined(OPENSSL_EXTRA) && \
            (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS)))
    /* use openssl like name to test mapping */
    X509_NAME_ENTRY* ne;
    X509_NAME* name;
    X509* x509;
#ifndef NO_FILESYSTEM
    ASN1_STRING* asn;
    char* subCN = NULL;
#endif
    int idx;
    ASN1_OBJECT *object = NULL;
#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    BIO* bio;
#endif
#endif

#ifndef NO_FILESYSTEM
    x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509);
    name = X509_get_subject_name(x509);
    idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1);
    AssertIntGE(idx, 0);
    ne = X509_NAME_get_entry(name, idx);
    AssertNotNull(ne);
    asn = X509_NAME_ENTRY_get_data(ne);
    AssertNotNull(asn);
    subCN = (char*)ASN1_STRING_data(asn);
    AssertNotNull(subCN);
    wolfSSL_FreeX509(x509);
#endif

    x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509);
    name = X509_get_subject_name(x509);
    idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1);
    AssertIntGE(idx, 0);

#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    AssertIntEQ(X509_NAME_print_ex(bio, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    AssertIntEQ(X509_NAME_print_ex_fp(stderr, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    BIO_free(bio);
#endif
#endif

    ne = X509_NAME_get_entry(name, idx);
    AssertNotNull(ne);
    AssertNotNull(object = X509_NAME_ENTRY_get_object(ne));
    wolfSSL_FreeX509(x509);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_ALL || (OPENSSL_EXTRA && (KEEP_PEER_CERT || SESSION_CERTS) */
#endif /* !NO_CERTS && !NO_RSA */

    return res;
}

/* Testing functions dealing with PKCS12 parsing out X509 certs */
static int test_wolfSSL_PKCS12(void)
{
    int res = TEST_SKIPPED;
    /* .p12 file is encrypted with DES3 */
#ifndef HAVE_FIPS /* Password used in cert "wolfSSL test" is only 12-bytes
                   * (96-bit) FIPS mode requires Minimum of 14-byte (112-bit)
                   * Password Key
                   */
#if defined(OPENSSL_EXTRA) && !defined(NO_DES3) && !defined(NO_FILESYSTEM) && \
    !defined(NO_ASN) && !defined(NO_PWDBASED) && !defined(NO_RSA) && \
    !defined(NO_SHA) && defined(HAVE_PKCS12) && !defined(NO_BIO)
    byte buf[6000];
    char file[] = "./certs/test-servercert.p12";
    char order[] = "./certs/ecc-rsa-server.p12";
#ifdef WC_RC2
    char rc2p12[] = "./certs/test-servercert-rc2.p12";
#endif
    char pass[] = "a password";
    const char goodPsw[] = "wolfSSL test";
    const char badPsw[] = "bad";
#ifdef HAVE_ECC
    WOLFSSL_X509_NAME* subject;
    WOLFSSL_X509     *x509;
#endif
    XFILE f;
    int  bytes, ret, goodPswLen, badPswLen;
    WOLFSSL_BIO      *bio;
    WOLFSSL_EVP_PKEY *pkey;
    WC_PKCS12        *pkcs12;
    WC_PKCS12        *pkcs12_2;
    WOLFSSL_X509     *cert;
    WOLFSSL_X509     *tmp;
    WOLF_STACK_OF(WOLFSSL_X509) *ca;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
    WOLFSSL_CTX      *ctx;
    WOLFSSL          *ssl;
    WOLF_STACK_OF(WOLFSSL_X509) *tmp_ca = NULL;
#endif

    f = XFOPEN(file, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    goodPswLen = (int)XSTRLEN(goodPsw);
    badPswLen = (int)XSTRLEN(badPsw);

    bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem());
    AssertNotNull(bio);

    AssertIntEQ(BIO_write(bio, buf, bytes), bytes); /* d2i consumes BIO */
    d2i_PKCS12_bio(bio, &pkcs12);
    AssertNotNull(pkcs12);
    BIO_free(bio);

    /* check verify MAC directly */
    ret = PKCS12_verify_mac(pkcs12, goodPsw, goodPswLen);
    AssertIntEQ(ret, 1);

    /* check verify MAC fail case directly */
    ret = PKCS12_verify_mac(pkcs12, badPsw, badPswLen);
    AssertIntEQ(ret, 0);

    /* check verify MAC fail case */
    ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL);
    AssertIntEQ(ret, 0);
    AssertNull(pkey);
    AssertNull(cert);

    /* check parse with no extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL);
    AssertIntEQ(ret, 1);
    AssertNotNull(pkey);
    AssertNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);

    /* check parse with extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca);
    AssertIntEQ(ret, 1);
    AssertNotNull(pkey);
    AssertNotNull(cert);
    AssertNotNull(ca);

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)

    /* Check that SSL_CTX_set0_chain correctly sets the certChain buffer */
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    /* Copy stack structure */
    AssertNotNull(tmp_ca = X509_chain_up_ref(ca));
    AssertIntEQ(SSL_CTX_set0_chain(ctx, tmp_ca), 1);
    /* CTX now owns the tmp_ca stack structure */
    tmp_ca = NULL;
    AssertIntEQ(wolfSSL_CTX_get_extra_chain_certs(ctx, &tmp_ca), 1);
    AssertNotNull(tmp_ca);
    AssertIntEQ(sk_X509_num(tmp_ca), sk_X509_num(ca));
    /* Check that the main cert is also set */
    AssertNotNull(SSL_CTX_get0_certificate(ctx));
    AssertNotNull(ssl = SSL_new(ctx));
    AssertNotNull(SSL_get_certificate(ssl));
    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
    /* should be 2 other certs on stack */
    tmp = sk_X509_pop(ca);
    AssertNotNull(tmp);
    X509_free(tmp);
    tmp = sk_X509_pop(ca);
    AssertNotNull(tmp);
    X509_free(tmp);
    AssertNull(sk_X509_pop(ca));

    EVP_PKEY_free(pkey);
    X509_free(cert);
    sk_X509_pop_free(ca, X509_free);

    /* check PKCS12_create */
    AssertNull(PKCS12_create(pass, NULL, NULL, NULL, NULL, -1, -1, -1, -1,0));
    AssertIntEQ(PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
            SSL_SUCCESS);
    AssertNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
                    -1, -1, 100, -1, 0)));
    EVP_PKEY_free(pkey);
    X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    AssertIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    AssertNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    /* convert to DER then back and parse */
    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    AssertIntEQ(i2d_PKCS12_bio(bio, pkcs12_2), SSL_SUCCESS);
    PKCS12_free(pkcs12_2);

    AssertNotNull(pkcs12_2 = d2i_PKCS12_bio(bio, NULL));
    BIO_free(bio);
    AssertIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

    /* should be 2 other certs on stack */
    tmp = sk_X509_pop(ca);
    AssertNotNull(tmp);
    X509_free(tmp);
    tmp = sk_X509_pop(ca);
    AssertNotNull(tmp);
    X509_free(tmp);
    AssertNull(sk_X509_pop(ca));


#ifndef NO_RC4
    PKCS12_free(pkcs12_2);
    AssertNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, NULL,
             NID_pbe_WithSHA1And128BitRC4,
             NID_pbe_WithSHA1And128BitRC4,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    AssertIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

#endif /* NO_RC4 */

    EVP_PKEY_free(pkey);
    X509_free(cert);
    PKCS12_free(pkcs12);
    PKCS12_free(pkcs12_2);
    sk_X509_pop_free(ca, NULL);

#ifdef HAVE_ECC
    /* test order of parsing */
    f = XFOPEN(order, "rb");
    AssertTrue(f != XBADFILE);
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    AssertNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    AssertNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));
    AssertIntEQ((ret = PKCS12_parse(pkcs12, "", &pkey, &cert, &ca)),
            WOLFSSL_SUCCESS);

    /* check use of pkey after parse */
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    AssertIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif

    AssertNotNull(pkey);
    AssertNotNull(cert);
    AssertNotNull(ca);

    /* compare subject lines of certificates */
    AssertNotNull(subject = wolfSSL_X509_get_subject_name(cert));
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(eccRsaCertFile,
                SSL_FILETYPE_PEM));
    AssertIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);

    /* test expected fail case */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    AssertIntNE(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    X509_free(cert);

    /* get subject line from ca stack */
    AssertNotNull(cert = sk_X509_pop(ca));
    AssertNotNull(subject = wolfSSL_X509_get_subject_name(cert));

    /* compare subject from certificate in ca to expected */
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    AssertIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);

    EVP_PKEY_free(pkey);
    X509_free(x509);
    X509_free(cert);
    BIO_free(bio);
    PKCS12_free(pkcs12);
    sk_X509_pop_free(ca, NULL); /* TEST d2i_PKCS12_fp */

    /* test order of parsing */
    f = XFOPEN(file, "rb");
    AssertTrue(f != XBADFILE);
    AssertNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    XFCLOSE(f);

    /* check verify MAC fail case */
    ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL);
    AssertIntEQ(ret, 0);
    AssertNull(pkey);
    AssertNull(cert);

    /* check parse with no extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL);
    AssertIntEQ(ret, 1);
    AssertNotNull(pkey);
    AssertNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);

    /* check parse with extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca);
    AssertIntEQ(ret, 1);
    AssertNotNull(pkey);
    AssertNotNull(cert);
    AssertNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    PKCS12_free(pkcs12);
#endif /* HAVE_ECC */

#ifdef WC_RC2
    /* test PKCS#12 with RC2 encryption */
    f = XFOPEN(rc2p12, "rb");
    AssertTrue(f != XBADFILE);
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    AssertNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    AssertNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));

    /* check verify MAC fail case */
    ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL);
    AssertIntEQ(ret, 0);
    AssertNull(pkey);
    AssertNull(cert);

    /* check parse iwth not extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL);
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
    AssertNotNull(pkey);
    AssertNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);

    /* check parse with extra certs kept */
    ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca);
    AssertIntEQ(ret, WOLFSSL_SUCCESS);
    AssertNotNull(pkey);
    AssertNotNull(cert);
    AssertNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    BIO_free(bio);
    PKCS12_free(pkcs12);
#endif /* WC_RC2 */

    /* Test i2d_PKCS12_bio */
    f = XFOPEN(file, "rb");
    AssertTrue((f != XBADFILE));
    AssertNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    XFCLOSE(f);

    bio = BIO_new(BIO_s_mem());
    AssertNotNull(bio);

    ret = i2d_PKCS12_bio(bio, pkcs12);
    AssertIntEQ(ret, 1);

    ret = i2d_PKCS12_bio(NULL, pkcs12);
    AssertIntEQ(ret, 0);

    ret = i2d_PKCS12_bio(bio, NULL);
    AssertIntEQ(ret, 0);

    PKCS12_free(pkcs12);
    BIO_free(bio);

    (void)order;

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA */
#endif /* HAVE_FIPS */
    return res;
}


#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_DES3) && !defined(NO_PWDBASED) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_MD5)
    #define TEST_PKCS8_ENC
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) \
    && defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS)

/* used to keep track if FailTestCallback was called */
static int failTestCallbackCalled = 0;

static WC_INLINE int FailTestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    (void)passwd;
    (void)sz;
    (void)rw;
    (void)userdata;

    /* mark called, test_wolfSSL_no_password_cb() will check and fail if set */
    failTestCallbackCalled = 1;

    return -1;
}
#endif

static int test_wolfSSL_no_password_cb(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) \
    && defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS)
    WOLFSSL_CTX* ctx;
    byte buff[FOURK_BUF];
    const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    XFILE f;
    int bytes;

#ifndef NO_WOLFSSL_CLIENT
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLS_client_method()));
#else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLS_server_method()));
#endif
    wolfSSL_CTX_set_default_passwd_cb(ctx, FailTestCallBack);

    AssertTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    AssertIntLE(bytes, sizeof(buff));
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    AssertTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    AssertIntLE(bytes, sizeof(buff));
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    if (failTestCallbackCalled != 0) {
        Fail(("Password callback should not be called by default"),
            ("Password callback was called without attempting "
             "to first decipher private key without password."));
    }

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

#ifdef TEST_PKCS8_ENC
/* for PKCS8 test case */
static int PKCS8TestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    int flag = 0;

    (void)rw;
    if (userdata != NULL) {
        flag = *((int*)userdata); /* user set data */
    }

    switch (flag) {
        case 1: /* flag set for specific WOLFSSL_CTX structure, note userdata
                 * can be anything the user wishes to be passed to the callback
                 * associated with the WOLFSSL_CTX */
            XSTRNCPY(passwd, "yassl123", sz);
            return 8;

        default:
            return BAD_FUNC_ARG;
    }
}
#endif /* TEST_PKCS8_ENC */

/* Testing functions dealing with PKCS8 */
static int test_wolfSSL_PKCS8(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    !defined(WOLFCRYPT_ONLY)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    byte buff[FOURK_BUF];
    byte der[FOURK_BUF];
    #ifndef NO_RSA
        const char serverKeyPkcs8PemFile[] = "./certs/server-keyPkcs8.pem";
        const char serverKeyPkcs8DerFile[] = "./certs/server-keyPkcs8.der";
    #endif
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    #ifdef HAVE_ECC
        const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    #endif
    XFILE f;
    int bytes;
    WOLFSSL_CTX* ctx;
#if defined(HAVE_ECC) && !defined(NO_CODING)
    int ret;
    ecc_key key;
    word32 x = 0;
#endif
#ifdef TEST_PKCS8_ENC
    #if !defined(NO_RSA) && !defined(NO_SHA)
        const char serverKeyPkcs8EncPemFile[] = "./certs/server-keyPkcs8Enc.pem";
        const char serverKeyPkcs8EncDerFile[] = "./certs/server-keyPkcs8Enc.der";
    #endif
    #if defined(HAVE_ECC) && !defined(NO_SHA)
        const char eccPkcs8EncPrivKeyPemFile[] = "./certs/ecc-keyPkcs8Enc.pem";
        const char eccPkcs8EncPrivKeyDerFile[] = "./certs/ecc-keyPkcs8Enc.der";
    #endif
    int flag;
#endif

    (void)der;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        AssertNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif

#ifdef TEST_PKCS8_ENC
    wolfSSL_CTX_set_default_passwd_cb(ctx, PKCS8TestCallBack);
    wolfSSL_CTX_set_default_passwd_cb_userdata(ctx, (void*)&flag);
    flag = 1; /* used by password callback as return code */

    #if !defined(NO_RSA) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted file */
    f = XFOPEN(serverKeyPkcs8EncPemFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    AssertIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    AssertIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    AssertIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading PEM PKCS8 encrypted file */
    f = XFOPEN(serverKeyPkcs8EncDerFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    flag = 1; /* used by password callback as return code */
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    AssertIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    #endif /* !NO_RSA && !NO_SHA */

    #if defined(HAVE_ECC) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted ECC Key file */
    f = XFOPEN(eccPkcs8EncPrivKeyPemFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    flag = 1; /* used by password callback as return code */
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    AssertIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    AssertIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    AssertIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading DER PKCS8 encrypted ECC Key file */
    f = XFOPEN(eccPkcs8EncPrivKeyDerFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    flag = 1; /* used by password callback as return code */
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    AssertIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* leave flag as "okay" */
    flag = 1;
    #endif /* HAVE_ECC && !NO_SHA */
#endif /* TEST_PKCS8_ENC */


#ifndef NO_RSA
    /* test loading ASN.1 (DER) PKCS8 private key file (not encrypted) */
    f = XFOPEN(serverKeyPkcs8DerFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* test loading PEM PKCS8 private key file (not encrypted) */
    f = XFOPEN(serverKeyPkcs8PemFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif /* !NO_RSA */

    /* Test PKCS8 PEM ECC key no crypt */
    f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);
#ifdef HAVE_ECC
    /* Test PKCS8 PEM ECC key no crypt */
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

#ifndef NO_CODING
    /* decrypt PKCS8 PEM to key in DER format */
    AssertIntGT((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), 0);
    ret = wc_ecc_init(&key);
    if (ret == 0) {
        ret = wc_EccPrivateKeyDecode(der, &x, &key, bytes);
        wc_ecc_free(&key);
    }
    AssertIntEQ(ret, 0);
#endif

    /* Test PKCS8 DER ECC key no crypt */
    f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buff, 1, sizeof(buff), f);
    XFCLOSE(f);

    /* Test using a PKCS8 ECC PEM */
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#else
    /* if HAVE_ECC is not defined then BEGIN EC PRIVATE KEY is not found */
    AssertIntEQ((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), ASN_NO_PEM_HEADER);
#endif /* HAVE_ECC */

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_FILESYSTEM && !NO_ASN && HAVE_PKCS8 */
    return res;
}

static int test_wolfSSL_PKCS8_ED25519(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED25519) && \
    defined(HAVE_ED25519_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGbMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAheCGLmWGh7+AICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEC4L5P6GappsTyhOOoQfvh8EQJMX\n"
    "OAdlsYKCOcFo4djg6AI1lRdeBRwVFWkha7gBdoCJOzS8wDvTbYcJMPvANu5ft3nl\n"
    "2L9W4v7swXkV+X+a1ww=\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
    WOLFSSL_CTX* ctx;
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    AssertIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#ifndef NO_WOLFSSL_SERVER
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return res;
}

static int test_wolfSSL_PKCS8_ED448(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED448) && \
    defined(HAVE_ED448_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGrMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAjSbZKnG4EPggICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEFvCFWBBHBlJBsYleBJlJWcEUNC7\n"
    "Tf5pZviT5Btar4D/MNg6BsQHSDf5KW4ix871EsgDY2Zz+euaoWspiMntz7gU+PQu\n"
    "T/JJcbD2Ly8BbE3l5WHMifAQqNLxJBfXrHkfYtAo\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
    WOLFSSL_CTX* ctx;
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    AssertIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#ifndef NO_WOLFSSL_SERVER
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    AssertNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    AssertIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return res;
}

/* Testing functions dealing with PKCS5 */
static int test_wolfSSL_PKCS5(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA) && !defined(NO_PWDBASED)
#ifdef HAVE_FIPS /* Password minimum length is 14 (112-bit) in FIPS MODE */
    const char* passwd = "myfipsPa$$W0rd";
#else
    const char *passwd = "pass1234";
#endif
    const unsigned char *salt = (unsigned char *)"salt1234";
    unsigned char *out = (unsigned char *)XMALLOC(WC_SHA_DIGEST_SIZE, NULL,
                                                  DYNAMIC_TYPE_TMP_BUFFER);
    int ret = 0;

    AssertNotNull(out);
    ret = PKCS5_PBKDF2_HMAC_SHA1(passwd,(int)XSTRLEN(passwd), salt,
                                 (int)XSTRLEN((const char *) salt), 10,
                                 WC_SHA_DIGEST_SIZE,out);
    AssertIntEQ(ret, SSL_SUCCESS);

#ifdef WOLFSSL_SHA512
    ret = PKCS5_PBKDF2_HMAC(passwd,(int)XSTRLEN(passwd), salt,
                             (int)XSTRLEN((const char *) salt), 10,
                             wolfSSL_EVP_sha512(), WC_SHA_DIGEST_SIZE, out);
    AssertIntEQ(ret, SSL_SUCCESS);
#endif

    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    res = TEST_RES_CHECK(1);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SHA) */
    return res;
}

/* test parsing URI from certificate */
static int test_wolfSSL_URI(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS) || \
    defined(OPENSSL_EXTRA))
    WOLFSSL_X509* x509;
    const char uri[] = "./certs/client-uri-cert.pem";
    const char urn[] = "./certs/client-absolute-urn.pem";
    const char badUri[] = "./certs/client-relative-uri.pem";

    x509 = wolfSSL_X509_load_certificate_file(uri, WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509);
    wolfSSL_FreeX509(x509);

    x509 = wolfSSL_X509_load_certificate_file(urn, WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509);
    wolfSSL_FreeX509(x509);

    x509 = wolfSSL_X509_load_certificate_file(badUri, WOLFSSL_FILETYPE_PEM);
#if !defined(IGNORE_NAME_CONSTRAINTS) && !defined(WOLFSSL_NO_ASN_STRICT) \
    && !defined(WOLFSSL_FPKI)
    AssertNull(x509);
#else
    AssertNotNull(x509);
    wolfSSL_FreeX509(x509);
#endif

    res = TEST_RES_CHECK(1);
#endif
    return res;
}


static int test_wolfSSL_TBS(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* x509;
    const unsigned char* tbs;
    int tbsSz;

    AssertNotNull(x509 =
          wolfSSL_X509_load_certificate_file(caCertFile, WOLFSSL_FILETYPE_PEM));

    AssertNull(tbs = wolfSSL_X509_get_tbs(NULL, &tbsSz));
    AssertNull(tbs = wolfSSL_X509_get_tbs(x509, NULL));
    AssertNotNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    AssertIntEQ(tbsSz, 1003);

    wolfSSL_FreeX509(x509);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_verify(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* ca;
    WOLFSSL_X509* serv;
    WOLFSSL_EVP_PKEY* pkey;
    unsigned char buf[2048];
    const unsigned char* pt = NULL;
    int bufSz;

    AssertNotNull(ca =
          wolfSSL_X509_load_certificate_file(caCertFile, WOLFSSL_FILETYPE_PEM));

    AssertIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, &bufSz),
            WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, NULL, &bufSz),
            WOLFSSL_SUCCESS);
    AssertIntEQ(bufSz, 294);

    bufSz = 2048;
    AssertIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
            WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_get_pubkey_type(NULL), WOLFSSL_FAILURE);
    AssertIntEQ(wolfSSL_X509_get_pubkey_type(ca), RSAk);


    AssertNotNull(serv =
          wolfSSL_X509_load_certificate_file(svrCertFile, WOLFSSL_FILETYPE_PEM));

    /* success case */
    pt = buf;
    AssertNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));

    AssertIntEQ(i2d_PUBKEY(pkey, NULL), bufSz);

    AssertIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);

    /* fail case */
    bufSz = 2048;
    AssertIntEQ(wolfSSL_X509_get_pubkey_buffer(serv, buf, &bufSz),
            WOLFSSL_SUCCESS);
    pt = buf;
    AssertNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));
    AssertIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_FAILURE);

    AssertIntEQ(wolfSSL_X509_verify(NULL, pkey), WOLFSSL_FATAL_ERROR);
    AssertIntEQ(wolfSSL_X509_verify(serv, NULL), WOLFSSL_FATAL_ERROR);
    wolfSSL_EVP_PKEY_free(pkey);

    wolfSSL_FreeX509(ca);
    wolfSSL_FreeX509(serv);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}


#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_CLIENT) && !defined(NO_DH) && !defined(NO_AES) && \
         defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(SINGLE_THREADED) && \
        defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && !defined(NO_BIO) \
        && !defined(NO_ASN_TIME)
/* create certificate with version 2 */
static void test_set_x509_badversion(WOLFSSL_CTX* ctx)
{
    WOLFSSL_X509 *x509, *x509v2;
    WOLFSSL_EVP_PKEY *priv, *pub;
    unsigned char *der = NULL, *key = NULL, *pt;
    char *header, *name;
    int derSz;
    long keySz;
    XFILE fp;
    WOLFSSL_ASN1_TIME *notBefore, *notAfter;
    time_t t;

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                WOLFSSL_FILETYPE_PEM));

    fp = XFOPEN(cliKeyFile, "rb");
    AssertIntEQ(wolfSSL_PEM_read(fp, &name, &header, &key, &keySz),
            WOLFSSL_SUCCESS);
    XFCLOSE(fp);
    pt = key;
    AssertNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, keySz));


    /* create the version 2 certificate */
    AssertNotNull(x509v2 = X509_new());
    AssertIntEQ(wolfSSL_X509_set_version(x509v2, 1), WOLFSSL_SUCCESS);

    AssertIntEQ(wolfSSL_X509_set_subject_name(x509v2,
                wolfSSL_X509_get_subject_name(x509)), WOLFSSL_SUCCESS);
    AssertIntEQ(wolfSSL_X509_set_issuer_name(x509v2,
                wolfSSL_X509_get_issuer_name(x509)), WOLFSSL_SUCCESS);
    AssertNotNull(pub = wolfSSL_X509_get_pubkey(x509));
    AssertIntEQ(X509_set_pubkey(x509v2, pub), WOLFSSL_SUCCESS);

    t = time(NULL);
    AssertNotNull(notBefore = wolfSSL_ASN1_TIME_adj(NULL, t, 0, 0));
    AssertNotNull(notAfter = wolfSSL_ASN1_TIME_adj(NULL, t, 365, 0));
    AssertTrue(wolfSSL_X509_set_notBefore(x509v2, notBefore));
    AssertTrue(wolfSSL_X509_set_notAfter(x509v2, notAfter));

    AssertIntGT(wolfSSL_X509_sign(x509v2, priv, EVP_sha256()), 0);
    derSz = wolfSSL_i2d_X509(x509v2, &der);
    AssertIntGT(derSz, 0);
    AssertIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, der, derSz,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_OPENSSL); /* TODO: Replace with API call */
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(name, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(header, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(x509v2);
    wolfSSL_EVP_PKEY_free(priv);
    wolfSSL_EVP_PKEY_free(pub);
    wolfSSL_ASN1_TIME_free(notBefore);
    wolfSSL_ASN1_TIME_free(notAfter);
}


/* override certificate version error */
static int test_override_x509(int preverify, WOLFSSL_X509_STORE_CTX* store)
{
#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    AssertIntEQ(store->error, ASN_VERSION_E);
#else
    AssertIntEQ(store->error, 0);
#endif
    AssertIntEQ((int)wolfSSL_X509_get_version(store->current_cert), 1);
    (void)preverify;
    return 1;
}


/* set verify callback that will override bad certificate version */
static void test_set_override_x509(WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, test_override_x509);
}
#endif


static int test_wolfSSL_X509_TLS_version(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        !defined(NO_WOLFSSL_CLIENT) && !defined(NO_DH) && !defined(NO_AES) && \
         defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(SINGLE_THREADED) && \
        defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && !defined(NO_BIO) \
        && !defined(NO_ASN_TIME)
    tcp_ready   ready;
    func_args   server_args;
    func_args   client_args;
    THREAD_TYPE serverThread;
    callback_functions func_cb_client;
    callback_functions func_cb_server;

    /* test server rejects a client certificate that is not version 3 */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&func_cb_client, 0, sizeof(callback_functions));
    XMEMSET(&func_cb_server, 0, sizeof(callback_functions));

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    server_args.return_code = TEST_FAIL;
    client_args.return_code = TEST_FAIL;

    func_cb_client.ctx_ready = &test_set_x509_badversion;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif
    client_args.callbacks = &func_cb_client;

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif
    server_args.callbacks = &func_cb_server;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    AssertIntEQ(client_args.return_code, TEST_FAIL);
    AssertIntEQ(server_args.return_code, TEST_FAIL);
#else
    AssertIntEQ(client_args.return_code, TEST_SUCCESS);
    AssertIntEQ(server_args.return_code, TEST_SUCCESS);
#endif

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    /* Now re run but override the bad X509 version */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&func_cb_client, 0, sizeof(callback_functions));
    XMEMSET(&func_cb_server, 0, sizeof(callback_functions));

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    server_args.return_code = TEST_FAIL;
    client_args.return_code = TEST_FAIL;

    func_cb_client.ctx_ready = &test_set_x509_badversion;
    func_cb_server.ctx_ready = &test_set_override_x509;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif
    client_args.callbacks = &func_cb_client;

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif
    server_args.callbacks = &func_cb_server;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    AssertIntEQ(client_args.return_code, TEST_SUCCESS);
    AssertIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    res = TEST_RES_CHECK(1);
#endif

    return res;
}

/* Testing function  wolfSSL_CTX_SetMinVersion; sets the minimum downgrade
 * version allowed.
 * POST: 1 on success.
 */
static int test_wolfSSL_CTX_SetMinVersion(void)
{
    int                     res = TEST_SKIPPED;
#ifndef NO_WOLFSSL_CLIENT
    int                     failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*            ctx;
    int                     itr;

    #ifndef NO_OLD_TLS
        const int versions[]  = {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                  WOLFSSL_TLSV1,
                            #endif
                                  WOLFSSL_TLSV1_1,
                                  WOLFSSL_TLSV1_2 };
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  = { WOLFSSL_TLSV1_2 };
    #elif defined(WOLFSSL_TLS13)
        const int versions[]  = { WOLFSSL_TLSV1_3 };
    #else
        const int versions[0];
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
        if (wolfSSL_CTX_SetMinVersion(ctx, *(versions + itr))
                != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_CTX_SetMinVersion */


/*----------------------------------------------------------------------------*
 | OCSP Stapling
 *----------------------------------------------------------------------------*/


/* Testing wolfSSL_UseOCSPStapling function. OCSP stapling eliminates the need
 * need to contact the CA, lowering the cost of cert revocation checking.
 * PRE: HAVE_OCSP and HAVE_CERTIFICATE_STATUS_REQUEST
 * POST: 1 returned for success.
 */
static int test_wolfSSL_UseOCSPStapling(void)
{
        int             res = TEST_SKIPPED;
    #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) && defined(HAVE_OCSP) && \
            !defined(NO_WOLFSSL_CLIENT)
        int             ret;
        WOLFSSL_CTX*    ctx;
        WOLFSSL*        ssl;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method());
    #else
        ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method());
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method());
    #else
        ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method());
    #endif
#endif
        ssl = wolfSSL_new(ctx);

        ret = wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
                                    WOLFSSL_CSR2_OCSP_USE_NONCE);

        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);

        res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
    #endif
        return res;
} /*END test_wolfSSL_UseOCSPStapling */


/* Testing OCSP stapling version 2, wolfSSL_UseOCSPStaplingV2 function. OCSP
 * stapling eliminates the need to contact the CA and lowers cert revocation
 * check.
 * PRE: HAVE_CERTIFICATE_STATUS_REQUEST_V2 and HAVE_OCSP defined.
 */
static int test_wolfSSL_UseOCSPStaplingV2(void)
{
        int                 res = TEST_SKIPPED;
    #if defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2) && defined(HAVE_OCSP) && \
            !defined(NO_WOLFSSL_CLIENT)
        int                 ret;
        WOLFSSL_CTX*        ctx;
        WOLFSSL*            ssl;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method());
    #else
        ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method());
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method());
    #else
        ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method());
    #endif
#endif
        ssl = wolfSSL_new(ctx);

        ret = wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
                                        WOLFSSL_CSR2_OCSP_USE_NONCE );

        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);

        res = TEST_RES_CHECK(ret == WOLFSSL_SUCCESS);
    #endif
        return res;

} /*END test_wolfSSL_UseOCSPStaplingV2*/

/*----------------------------------------------------------------------------*
 | Multicast Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_mcast(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_MULTICAST) && \
    (defined(WOLFSSL_TLS13) || defined(WOLFSSL_SNIFFER))
    WOLFSSL_CTX* ctx;
    WOLFSSL* ssl;
    int result;
    byte preMasterSecret[512];
    byte clientRandom[32];
    byte serverRandom[32];
    byte suite[2] = {0, 0xfe};  /* WDM_WITH_NULL_SHA256 */
    byte buf[256];
    word16 newId;

    ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method());
    AssertNotNull(ctx);

    result = wolfSSL_CTX_mcast_set_member_id(ctx, 0);
    AssertIntEQ(result, WOLFSSL_SUCCESS);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    XMEMSET(preMasterSecret, 0x23, sizeof(preMasterSecret));
    XMEMSET(clientRandom, 0xA5, sizeof(clientRandom));
    XMEMSET(serverRandom, 0x5A, sizeof(serverRandom));
    result = wolfSSL_set_secret(ssl, 23,
                preMasterSecret, sizeof(preMasterSecret),
                clientRandom, serverRandom, suite);
    AssertIntEQ(result, WOLFSSL_SUCCESS);

    result = wolfSSL_mcast_read(ssl, &newId, buf, sizeof(buf));
    AssertIntLE(result, 0);
    AssertIntLE(newId, 100);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* WOLFSSL_DTLS && WOLFSSL_MULTICAST && (WOLFSSL_TLS13 ||
        * WOLFSSL_SNIFFER) */
    return res;
}


/*----------------------------------------------------------------------------*
 |  Wolfcrypt
 *----------------------------------------------------------------------------*/

/*
 * Unit test for the wc_InitBlake2b()
 */
static int test_wc_InitBlake2b(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_BLAKE2
    Blake2b blake;
    int ret = 0;

    /* Test good arg. */
    ret = wc_InitBlake2b(&blake, 64);
    if (ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!ret) {
        ret = wc_InitBlake2b(NULL, 64);
        if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    if (!ret) {
        ret = wc_InitBlake2b(NULL, 128);
        if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    if (!ret) {
        ret = wc_InitBlake2b(&blake, 128);
        if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    if (!ret) {
        ret = wc_InitBlake2b(NULL, 0);
        if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    if (!ret) {
        ret = wc_InitBlake2b(&blake, 0);
        if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }


    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}     /*END test_wc_InitBlake2b*/

/*
 * Unit test for the wc_InitBlake2b_WithKey()
 */
static int test_wc_InitBlake2b_WithKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_BLAKE2
    Blake2b     blake;
    word32      digestSz = BLAKE2B_KEYBYTES;
    byte        key[BLAKE2B_KEYBYTES];
    word32      keylen = BLAKE2B_KEYBYTES;
    int ret = 0;

    XMEMSET(key, 0, sizeof(key));

    /* Test good arg. */
    ret = wc_InitBlake2b_WithKey(&blake, digestSz, key, keylen);
    if (ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_InitBlake2b_WithKey(NULL, digestSz, key, keylen);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_InitBlake2b_WithKey(&blake, digestSz, key, 256);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_InitBlake2b_WithKey(&blake, digestSz, NULL, keylen);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}     /*END wc_InitBlake2b_WithKey*/

/*
 * Unit test for the wc_InitBlake2s_WithKey()
 */
static int test_wc_InitBlake2s_WithKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_BLAKE2S
    Blake2s     blake;
    word32      digestSz = BLAKE2S_KEYBYTES;
    byte        *key = (byte*)"01234567890123456789012345678901";
    word32      keylen = BLAKE2S_KEYBYTES;
    int ret = 0;

    /* Test good arg. */
    ret = wc_InitBlake2s_WithKey(&blake, digestSz, key, keylen);
    if (ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_InitBlake2s_WithKey(NULL, digestSz, key, keylen);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_InitBlake2s_WithKey(&blake, digestSz, key, 256);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_InitBlake2s_WithKey(&blake, digestSz, NULL, keylen);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}     /*END wc_InitBlake2s_WithKey*/

/*
 * Unit test for the wc_InitMd5()
 */
static int test_wc_InitMd5(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_MD5
    wc_Md5 md5;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitMd5(&md5);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitMd5(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Md5Free(&md5);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
}     /* END test_wc_InitMd5 */


/*
 * Testing wc_UpdateMd5()
 */
static int test_wc_Md5Update(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_MD5
    wc_Md5 md5;
    byte hash[WC_MD5_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitMd5(&md5);
    if (ret != 0) {
        flag = ret;
    }

    /* Input */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Md5Update(&md5, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Md5Final(&md5, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\x90\x01\x50\x98\x3c\xd2\x4f\xb0\xd6\x96\x3f\x7d\x28\xe1\x7f"
                    "\x72";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Md5Update(&md5, (byte*) a.input, (word32) a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Md5Final(&md5, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /*Pass in bad values. */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_Md5Update(&md5, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_MD5_DIGEST_SIZE;

        ret = wc_Md5Update(&md5, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Md5Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Md5Free(&md5);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Md5Update()  */


/*
 *  Unit test on wc_Md5Final() in wolfcrypt/src/md5.c
 */
static int test_wc_Md5Final(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_MD5
    /* Instantiate */
    wc_Md5 md5;
    byte* hash_test[3];
    byte hash1[WC_MD5_DIGEST_SIZE];
    byte hash2[2*WC_MD5_DIGEST_SIZE];
    byte hash3[5*WC_MD5_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitMd5(&md5);
    if (ret != 0)  {
        flag = ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test)/sizeof(byte*);

    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Md5Final(&md5, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_Md5Final(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Md5Final(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Md5Final(&md5, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Md5Free(&md5);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
}

/*
 * Unit test for the wc_InitSha()
 */
static int test_wc_InitSha(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA
    wc_Sha sha;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha(&sha);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ShaFree(&sha);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_InitSha */

/*
 *  Tesing wc_ShaUpdate()
 */
static int test_wc_ShaUpdate(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA
    wc_Sha sha;
    byte hash[WC_SHA_DIGEST_SIZE];
    testVector a, b, c;
    int flag = 0;
    int ret;

    ret = wc_InitSha(&sha);
    if (ret != 0) {
        flag = ret;
    }

    /* Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_ShaUpdate(&sha, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_ShaUpdate(&sha, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_ShaUpdate(&sha, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_ShaFinal(&sha, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\xA9\x99\x3E\x36\x47\x06\x81\x6A\xBA\x3E\x25\x71\x78\x50\xC2"
                    "\x6C\x9C\xD0\xD8\x9D";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_ShaUpdate(&sha, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_ShaFinal(&sha, hash);
        if (ret !=0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try passing in bad values. */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_ShaUpdate(&sha, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA_DIGEST_SIZE;

        ret = wc_ShaUpdate(&sha, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_ShaUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ShaFree(&sha);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_ShaUpdate() */


/*
 * Unit test on wc_ShaFinal
 */
static int test_wc_ShaFinal(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA
    wc_Sha sha;
    byte* hash_test[3];
    byte hash1[WC_SHA_DIGEST_SIZE];
    byte hash2[2*WC_SHA_DIGEST_SIZE];
    byte hash3[5*WC_SHA_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /*Initialize*/
    ret = wc_InitSha(&sha);
    if (ret) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test)/sizeof(byte*);

    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_ShaFinal(&sha, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_ShaFinal(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_ShaFinal(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_ShaFinal(&sha, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ShaFree(&sha);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_ShaFinal */


/*
 * Unit test for wc_InitSha256()
 */
static int test_wc_InitSha256(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha256(&sha256);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha256(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_InitSha256 */


/*
 * Unit test for wc_Sha256Update()
 */
static int test_wc_Sha256Update(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte hash[WC_SHA256_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha256(&sha256);
    if (ret != 0) {
        flag = ret;
    }

    /*  Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha256Update(&sha256, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha256Update(&sha256, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha256Update(&sha256, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha256Final(&sha256, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
                    "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
                    "\x15\xAD";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha256Update(&sha256, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha256Final(&sha256, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA256_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try passing in bad values */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_Sha256Update(&sha256, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA256_DIGEST_SIZE;

        ret = wc_Sha256Update(&sha256, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha256Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha256Update */


/*
 * Unit test function for wc_Sha256Final()
 */
static int test_wc_Sha256Final(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte* hash_test[3];
    byte hash1[WC_SHA256_DIGEST_SIZE];
    byte hash2[2*WC_SHA256_DIGEST_SIZE];
    byte hash3[5*WC_SHA256_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha256(&sha256);
    if (ret != 0) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha256Final(&sha256, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag ) {
        ret = wc_Sha256Final(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha256Final(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha256Final(&sha256, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha256Final */
/*
 * Unit test function for wc_Sha256FinalRaw()
 */
static int test_wc_Sha256FinalRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_SHA256) && !defined(HAVE_SELFTEST) && !defined(WOLFSSL_DEVCRYPTO) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha256 sha256;
    byte* hash_test[3];
    byte hash1[WC_SHA256_DIGEST_SIZE];
    byte hash2[2*WC_SHA256_DIGEST_SIZE];
    byte hash3[5*WC_SHA256_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha256(&sha256);
    if (ret != 0) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha256FinalRaw(&sha256, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_Sha256FinalRaw(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha256FinalRaw(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha256FinalRaw(&sha256, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha256FinalRaw */
/*
 * Unit test function for wc_Sha256GetFlags()
 */
static int test_wc_Sha256GetFlags(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_SHA256) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha256 sha256;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha256(&sha256);
    if (flag == 0) {
        flag = wc_Sha256GetFlags(&sha256, &flags);
    }
    if (flag == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            flag = 0;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha256GetFlags */
/*
 * Unit test function for wc_Sha256Free()
 */
static int test_wc_Sha256Free(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
    return res;

} /* END test_wc_Sha256Free */
/*
 * Unit test function for wc_Sha256GetHash()
 */
static int test_wc_Sha256GetHash(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    byte hash1[WC_SHA256_DIGEST_SIZE];
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha256(&sha256);

    if (flag == 0) {
        flag = wc_Sha256GetHash(&sha256, hash1);
    }
    /*test bad arguments*/
    if (flag == 0) {
        flag = wc_Sha256GetHash(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }
    if (flag == 0) {
        flag = wc_Sha256GetHash(NULL, hash1);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }
    if (flag == 0) {
        flag = wc_Sha256GetHash(&sha256, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha256Free(&sha256);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha256GetHash */
/*
 * Unit test function for wc_Sha256Copy()
 */
static int test_wc_Sha256Copy(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SHA256
    wc_Sha256 sha256;
    wc_Sha256 temp;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha256(&sha256);
    if (flag == 0) {
        flag = wc_InitSha256(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha256Copy(&sha256, &temp);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha256Copy(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha256Copy(NULL, &temp);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha256Copy(&sha256, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha256Free(&sha256);
    wc_Sha256Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha256Copy */
/*
 * Testing wc_InitSha512()
 */
static int test_wc_InitSha512(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha512(&sha512);
    if (ret != 0) {
        flag  = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha512(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_InitSha512 */


/*
 *  wc_Sha512Update() test.
 */
static int test_wc_Sha512Update(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha512(&sha512);
    if (ret != 0) {
        flag = ret;
    }

    /* Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha512Update(&sha512, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512Update(&sha512,(byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512Update(&sha512, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\xdd\xaf\x35\xa1\x93\x61\x7a\xba\xcc\x41\x73\x49\xae\x20\x41"
                    "\x31\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2\x0a\x9e\xee\xe6\x4b"
                    "\x55\xd3\x9a\x21\x92\x99\x2a\x27\x4f\xc1\xa8\x36\xba\x3c"
                    "\x23\xa3\xfe\xeb\xbd\x45\x4d\x44\x23\x64\x3c\xe8\x0e\x2a"
                    "\x9a\xc9\x4f\xa5\x4c\xa4\x9f";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha512Update(&sha512, (byte*) a.input, (word32) a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha512Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA512_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try passing in bad values */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_Sha512Update(&sha512, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA512_DIGEST_SIZE;

        ret = wc_Sha512Update(&sha512, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha512Update  */

#ifdef WOLFSSL_SHA512
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
        (!defined(WOLFSSL_NOSHA512_224) || !defined(WOLFSSL_NOSHA512_256))
/* Perfoms test for
 * - wc_Sha512Final/wc_Sha512FinalRaw
 * - wc_Sha512_224Final/wc_Sha512_224Final
 * - wc_Sha512_256Final/wc_Sha512_256Final
 * parameter:
 * - type : must be one of WC_HASH_TYPE_SHA512, WC_HASH_TYPE_SHA512_224 or
 *          WC_HASH_TYPE_SHA512_256
 * - isRaw: if is non-zero, xxxFinalRaw function will be tested
 *return 0 on success
 */
static int test_Sha512_Family_Final(int type, int isRaw)
{
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i, ret;

    int(*initFp)(wc_Sha512*);
    int(*finalFp)(wc_Sha512*, byte*);
    void(*freeFp)(wc_Sha512*);

    if (type == WC_HASH_TYPE_SHA512) {
        initFp  = wc_InitSha512;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512FinalRaw : wc_Sha512Final;
#else
        finalFp = (isRaw)? NULL : wc_Sha512Final;
#endif
        freeFp  = wc_Sha512Free;
    }
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if !defined(WOLFSSL_NOSHA512_224)
    else if (type == WC_HASH_TYPE_SHA512_224) {
        initFp  = wc_InitSha512_224;
    #if !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512_224FinalRaw : wc_Sha512_224Final;
    #else
        finalFp = (isRaw)? NULL : wc_Sha512_224Final;
    #endif
        freeFp  = wc_Sha512_224Free;
    }
#endif
#if !defined(WOLFSSL_NOSHA512_256)
    else if (type == WC_HASH_TYPE_SHA512_256) {
        initFp  = wc_InitSha512_256;
    #if !defined(WOLFSSL_NO_HASH_RAW)
        finalFp = (isRaw)? wc_Sha512_256FinalRaw : wc_Sha512_256Final;
    #else
        finalFp = (isRaw)? NULL : wc_Sha512_256Final;
    #endif
        freeFp  = wc_Sha512_256Free;
    }
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    else
        return TEST_FAIL;

    /* Initialize  */
    ret = initFp(&sha512);

    if (!ret) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte *);

    /* Good test args. */
    for (i = 0; i < times && ret == 0; i++) {
        ret = finalFp(&sha512, hash_test[i]);
    }
    /* Test bad args. */
    if (!ret) {
        if (finalFp(NULL, NULL) != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (!ret) {
        if (finalFp(NULL, hash1) != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!ret) {
        if (finalFp(&sha512, NULL) != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    freeFp(&sha512);

    return ret;
}
#endif /* !HAVE_FIPS && !HAVE_SELFTEST &&
                        (!WOLFSSL_NOSHA512_224 || !WOLFSSL_NOSHA512_256) */
#endif /* WOLFSSL_SHA512 */
/*
 * Unit test function for wc_Sha512Final()
 */
static int test_wc_Sha512Final(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize  */
    ret = wc_InitSha512(&sha512);
    if (ret != 0) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte *);

    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha512Final(&sha512, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }
    /* Test bad args. */
    if (!flag) {
        ret = wc_Sha512Final(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512Final(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512Final(&sha512, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha512Final */
/*
 * Unit test function for wc_Sha512GetFlags()
 */
static int test_wc_Sha512GetFlags(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA512) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512(&sha512);
    if (flag == 0) {
        flag = wc_Sha512GetFlags(&sha512, &flags);
    }
    if (flag == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            flag = 0;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha512GetFlags */
/*
 * Unit test function for wc_Sha512FinalRaw()
 */
static int test_wc_Sha512FinalRaw(void)
{
    int res = TEST_SKIPPED;
#if (defined(WOLFSSL_SHA512) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3)))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha512 sha512;
    byte* hash_test[3];
    byte hash1[WC_SHA512_DIGEST_SIZE];
    byte hash2[2*WC_SHA512_DIGEST_SIZE];
    byte hash3[5*WC_SHA512_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha512(&sha512);
    if (ret != 0) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    /* Good test args. */
    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha512FinalRaw(&sha512, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag ) {
        ret = wc_Sha512FinalRaw(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512FinalRaw(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512FinalRaw(&sha512, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha512FinalRaw */

/*
 * Unit test function for wc_Sha512Free()
 */
static int test_wc_Sha512Free(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_Sha512Free */
#ifdef WOLFSSL_SHA512

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
        (!defined(WOLFSSL_NOSHA512_224) || !defined(WOLFSSL_NOSHA512_256))
static int test_Sha512_Family_GetHash(int type )
{
    int flag = 0;
    int(*initFp)(wc_Sha512*);
    int(*ghashFp)(wc_Sha512*, byte*);
    wc_Sha512 sha512;
    byte hash1[WC_SHA512_DIGEST_SIZE];

    if (type == WC_HASH_TYPE_SHA512) {
        initFp  = wc_InitSha512;
        ghashFp = wc_Sha512GetHash;
    }
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if !defined(WOLFSSL_NOSHA512_224)
    else if (type == WC_HASH_TYPE_SHA512_224) {
        initFp  = wc_InitSha512_224;
        ghashFp = wc_Sha512_224GetHash;
    }
#endif
#if !defined(WOLFSSL_NOSHA512_256)
    else if (type == WC_HASH_TYPE_SHA512_256) {
        initFp  = wc_InitSha512_256;
        ghashFp = wc_Sha512_256GetHash;
    }
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    else {
        initFp  = NULL;
        ghashFp = NULL;
    }

    if (initFp == NULL || ghashFp == NULL)
        return TEST_FAIL;

    if (!flag) {
        flag = initFp(&sha512);
    }

    if (!flag) {
        flag = ghashFp(&sha512, hash1);
    }

    /*test bad arguments*/
     if (!flag) {
        if (ghashFp(NULL, NULL) != BAD_FUNC_ARG )
            flag = WOLFSSL_FATAL_ERROR;
     }

     if (!flag) {
        if (ghashFp(NULL, hash1) != BAD_FUNC_ARG )
            flag = WOLFSSL_FATAL_ERROR;
     }

     if (!flag) {
        if (ghashFp(&sha512, NULL) != BAD_FUNC_ARG )
            flag = WOLFSSL_FATAL_ERROR;
    }

    wc_Sha512Free(&sha512);
    return flag;
}
#endif /* !HAVE_FIPS && !HAVE_SELFTEST &&
                        (!WOLFSSL_NOSHA512_224 || !WOLFSSL_NOSHA512_256) */
#endif /* WOLFSSL_SHA512 */
/*
 * Unit test function for wc_Sha512GetHash()
 */
static int test_wc_Sha512GetHash(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    byte hash1[WC_SHA512_DIGEST_SIZE];
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512(&sha512);

    if (flag == 0) {
        flag = wc_Sha512GetHash(&sha512, hash1);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha512GetHash(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha512GetHash(NULL, hash1);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha512GetHash(&sha512, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha512Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha512GetHash */

/*
 * Unit test function for wc_Sha512Copy()
 */
static int test_wc_Sha512Copy(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA512
    wc_Sha512 sha512;
    wc_Sha512 temp;
    int flag;

    /* Initialize */
    flag = wc_InitSha512(&sha512);
    if (flag == 0) {
        flag = wc_InitSha512(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha512Copy(&sha512, &temp);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha512Copy(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha512Copy(NULL, &temp);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha512Copy(&sha512, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha512Free(&sha512);
    wc_Sha512Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha512Copy */

static int test_wc_InitSha512_224(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha512_224(&sha512);
    if (ret != 0) {
        flag  = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha512_224(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512_224Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_224Update(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    testVector a, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha512_224(&sha512);
    if (ret != 0) {
        flag = ret;
    }

    /* Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha512_224Update(&sha512, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_224Update(&sha512,(byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_224Update(&sha512, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_224Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\x46\x34\x27\x0f\x70\x7b\x6a\x54\xda\xae\x75\x30\x46\x08"
                   "\x42\xe2\x0e\x37\xed\x26\x5c\xee\xe9\xa4\x3e\x89\x24\xaa";

        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha512_224Update(&sha512, (byte*) a.input, (word32) a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha512_224Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA512_224_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA512_224_DIGEST_SIZE;

        ret = wc_Sha512_224Update(&sha512, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512_224Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512_224Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_224Final(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    int ret = test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_224, 0);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_224GetFlags(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512, copy;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512_224(&sha512);
    if (!flag) {
        flag = wc_InitSha512_224(&copy);
    }
    if (!flag) {
        flag = wc_Sha512_224Copy(&sha512, &copy);
    }
    if (!flag) {
        flag = wc_Sha512_224GetFlags(&copy, &flags);
    }
    if (!flag) {
        if (flags & WC_HASH_FLAG_ISCOPY)
            flag = 0;
        else
            flag = WOLFSSL_FATAL_ERROR;
    }
    wc_Sha512_224Free(&copy);
    wc_Sha512_224Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_224FinalRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    defined(WOLFSSL_SHA512) &&  !defined(WOLFSSL_NOSHA512_224) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    int ret = test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_224, 1);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}

static int test_wc_Sha512_224Free(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512_224Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_224GetHash(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    int ret = test_Sha512_Family_GetHash(WC_HASH_TYPE_SHA512_224);

    res = TEST_RES_CHECK(ret == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}
static int test_wc_Sha512_224Copy(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    wc_Sha512 sha512;
    wc_Sha512 temp;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512_224(&sha512);
    if (flag == 0) {
        flag = wc_InitSha512_224(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha512_224Copy(&sha512, &temp);
    }
    /*test bad arguments*/
    if (flag == 0) {
        if (wc_Sha512_224Copy(NULL, NULL) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
    }
    if (flag == 0) {
        if (wc_Sha512_224Copy(NULL, &temp) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
    }
    if (flag == 0) {
        if (wc_Sha512_224Copy(&sha512, NULL) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
    }

    wc_Sha512_224Free(&sha512);
    wc_Sha512_224Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_InitSha512_256(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha512_256(&sha512);
    if (ret != 0) {
        flag  = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha512_256(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512_256Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_256Update(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;
    byte hash[WC_SHA512_DIGEST_SIZE];
    testVector a, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha512_256(&sha512);
    if (ret != 0) {
        flag = ret;
    }

    /* Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha512_256Update(&sha512, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_256Update(&sha512,(byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_256Update(&sha512, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha512_256Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\x53\x04\x8e\x26\x81\x94\x1e\xf9\x9b\x2e\x29\xb7\x6b\x4c"
                   "\x7d\xab\xe4\xc2\xd0\xc6\x34\xfc\x6d\x46\xe0\xe2\xf1\x31"
                   "\x07\xe7\xaf\x23";

        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha512_256Update(&sha512, (byte*) a.input, (word32) a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha512_256Final(&sha512, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA512_256_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA512_256_DIGEST_SIZE;

        ret = wc_Sha512_256Update(&sha512, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha512_256Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha512_256Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_256Final(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    int ret = test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_256, 0);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_256GetFlags(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha512 sha512, copy;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512_256(&sha512);
    if (!flag ) {
        flag = wc_InitSha512_256(&copy);
    }
    if (!flag ) {
        flag = wc_Sha512_256Copy(&sha512, &copy);
    }
    if (!flag ) {
        flag = wc_Sha512_256GetFlags(&copy, &flags);
    }
    if (!flag) {
        if (flags & WC_HASH_FLAG_ISCOPY)
            flag = 0;
        else
            flag = WOLFSSL_FATAL_ERROR;
    }

    wc_Sha512_256Free(&sha512);

    res = TEST_RES_CHECK(flag == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_256FinalRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    defined(WOLFSSL_SHA512) &&  !defined(WOLFSSL_NOSHA512_256) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    int ret = test_Sha512_Family_Final(WC_HASH_TYPE_SHA512_256, 1);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}

static int test_wc_Sha512_256Free(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512_256Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

static int test_wc_Sha512_256GetHash(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    int ret = test_Sha512_Family_GetHash(WC_HASH_TYPE_SHA512_256);

    res = TEST_RES_CHECK(ret == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;

}
static int test_wc_Sha512_256Copy(void)
{
    int res = TEST_SKIPPED;
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    wc_Sha512 sha512;
    wc_Sha512 temp;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha512_256(&sha512);
    if (flag == 0) {
        flag = wc_InitSha512_256(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha512_256Copy(&sha512, &temp);
    }
    /*test bad arguments*/
     if (flag == 0) {
        if (wc_Sha512_256Copy(NULL, NULL) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
     }
     if (flag == 0) {
        if (wc_Sha512_256Copy(NULL, &temp) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
     }
     if (flag == 0) {
        if (wc_Sha512_256Copy(&sha512, NULL) != BAD_FUNC_ARG)
            flag = WOLFSSL_FATAL_ERROR;
    }

    wc_Sha512_256Free(&sha512);
    wc_Sha512_256Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}



/*
 * Testing wc_InitSha384()
 */
static int test_wc_InitSha384(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha384(&sha384);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha384(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_InitSha384 */


/*
 * test wc_Sha384Update()
 */
static int test_wc_Sha384Update(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte hash[WC_SHA384_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha384(&sha384);
    if (ret != 0) {
        flag = ret;
    }

    /* Input */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha384Update(&sha384, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha384Update(&sha384, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha384Update(&sha384, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha384Final(&sha384, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b\xb5\xa0\x3d\x69\x9a\xc6\x50"
                   "\x07\x27\x2c\x32\xab\x0e\xde\xd1\x63\x1a\x8b\x60\x5a\x43\xff"
                   "\x5b\xed\x80\x86\x07\x2b\xa1\xe7\xcc\x23\x58\xba\xec\xa1\x34"
                   "\xc8\x25\xa7";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha384Update(&sha384, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha384Final(&sha384, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA384_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass in bad values. */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_Sha384Update(&sha384, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA384_DIGEST_SIZE;

        ret = wc_Sha384Update(&sha384, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha384Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha384Update */

/*
 * Unit test function for wc_Sha384Final();
 */
static int test_wc_Sha384Final(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte* hash_test[3];
    byte hash1[WC_SHA384_DIGEST_SIZE];
    byte hash2[2*WC_SHA384_DIGEST_SIZE];
    byte hash3[5*WC_SHA384_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha384(&sha384);
    if (ret) {
        flag = ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    /* Good test args. */
    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha384Final(&sha384, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_Sha384Final(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha384Final(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha384Final(&sha384, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha384Final */
/*
 * Unit test function for wc_Sha384GetFlags()
 */
static int test_wc_Sha384GetFlags(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha384 sha384;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha384(&sha384);
    if (flag == 0) {
        flag = wc_Sha384GetFlags(&sha384, &flags);
    }
    if (flag == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            flag = 0;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha384GetFlags */
/*
 * Unit test function for wc_Sha384FinalRaw()
 */
static int test_wc_Sha384FinalRaw(void)
{
    int res = TEST_SKIPPED;
#if (defined(WOLFSSL_SHA384) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 3)))) && \
    !defined(WOLFSSL_NO_HASH_RAW)
    wc_Sha384 sha384;
    byte* hash_test[3];
    byte hash1[WC_SHA384_DIGEST_SIZE];
    byte hash2[2*WC_SHA384_DIGEST_SIZE];
    byte hash3[5*WC_SHA384_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha384(&sha384);
    if (ret != 0) {
        flag =  ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    /* Good test args. */
    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha384FinalRaw(&sha384, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag ) {
        ret = wc_Sha384FinalRaw(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha384FinalRaw(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha384FinalRaw(&sha384, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha384FinalRaw */
/*
 * Unit test function for wc_Sha384Free()
 */
static int test_wc_Sha384Free(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
    return res;

} /* END test_wc_Sha384Free */
/*
 * Unit test function for wc_Sha384GetHash()
 */
static int test_wc_Sha384GetHash(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    byte hash1[WC_SHA384_DIGEST_SIZE];
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha384(&sha384);

    if (flag == 0) {
        flag = wc_Sha384GetHash(&sha384, hash1);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha384GetHash(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha384GetHash(NULL, hash1);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha384GetHash(&sha384, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha384Free(&sha384);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha384GetHash */
/*
 * Unit test function for wc_Sha384Copy()
 */
static int test_wc_Sha384Copy(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA384
    wc_Sha384 sha384;
    wc_Sha384 temp;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha384(&sha384);
    if (flag == 0) {
        flag = wc_InitSha384(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha384Copy(&sha384, &temp);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha384Copy(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha384Copy(NULL, &temp);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha384Copy(&sha384, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha384Free(&sha384);
    wc_Sha384Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha384Copy */

/*
 * Testing wc_InitSha224();
 */
static int test_wc_InitSha224(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitSha224(&sha224);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitSha224(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_InitSha224 */

/*
 * Unit test on wc_Sha224Update
 */
static int test_wc_Sha224Update(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte hash[WC_SHA224_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitSha224(&sha224);
    if (ret != 0) {
        flag = ret;
    }

    /* Input. */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_Sha224Update(&sha224, NULL, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha224Update(&sha224, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }

        ret = wc_Sha224Update(&sha224, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha224Final(&sha224, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\x23\x09\x7d\x22\x34\x05\xd8\x22\x86\x42\xa4\x77\xbd\xa2"
                    "\x55\xb3\x2a\xad\xbc\xe4\xbd\xa0\xb3\xf7\xe3\x6c\x9d\xa7";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_Sha224Update(&sha224, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_Sha224Final(&sha224, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA224_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass  in bad values. */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_Sha224Update(&sha224, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = WC_SHA224_DIGEST_SIZE;

        ret = wc_Sha224Update(&sha224, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha224Update(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha224Update */

/*
 * Unit test for wc_Sha224Final();
 */
static int test_wc_Sha224Final(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte* hash_test[3];
    byte hash1[WC_SHA224_DIGEST_SIZE];
    byte hash2[2*WC_SHA224_DIGEST_SIZE];
    byte hash3[5*WC_SHA224_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitSha224(&sha224);
    if (ret) {
        flag = ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    /* Good test args. */
    /* Testing oversized buffers. */
    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_Sha224Final(&sha224, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_Sha224Final(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha224Final(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_Sha224Final(&sha224, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224Final */

/*
 * Unit test function for wc_Sha224SetFlags()
 */
static int test_wc_Sha224SetFlags(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha224 sha224;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha224(&sha224);
    if (flag == 0) {
        flag = wc_Sha224SetFlags(&sha224, flags);
    }
    if (flag == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            flag = 0;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224SetFlags */

/*
 * Unit test function for wc_Sha224GetFlags()
 */
static int test_wc_Sha224GetFlags(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA224) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha224 sha224;
    word32 flags = 0;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha224(&sha224);
    if (flag == 0) {
        flag = wc_Sha224GetFlags(&sha224, &flags);
    }
    if (flag == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            flag = 0;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Sha224GetFlags */
/*
 * Unit test function for wc_Sha224Free()
 */
static int test_wc_Sha224Free(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224Free(NULL);

    res = TEST_RES_CHECK(1);
#endif
    return res;

} /* END test_wc_Sha224Free */

/*
 * Unit test function for wc_Sha224GetHash()
 */
static int test_wc_Sha224GetHash(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    byte hash1[WC_SHA224_DIGEST_SIZE];
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha224(&sha224);

    if (flag == 0) {
        flag = wc_Sha224GetHash(&sha224, hash1);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha224GetHash(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha224GetHash(NULL, hash1);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha224GetHash(&sha224, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }

    wc_Sha224Free(&sha224);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224GetHash */

/*
 * Unit test function for wc_Sha224Copy()
 */
static int test_wc_Sha224Copy(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHA224
    wc_Sha224 sha224;
    wc_Sha224 temp;
    int flag = 0;

    /* Initialize */
    flag = wc_InitSha224(&sha224);
    if (flag == 0) {
        flag = wc_InitSha224(&temp);
    }
    if (flag == 0) {
        flag = wc_Sha224Copy(&sha224, &temp);
    }
    /*test bad arguments*/
     if (flag == 0) {
        flag = wc_Sha224Copy(NULL, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha224Copy(NULL, &temp);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
     }
     if (flag == 0) {
        flag = wc_Sha224Copy(&sha224, NULL);
        if (flag == BAD_FUNC_ARG) {
            flag = 0;
        }
    }


    wc_Sha224Free(&sha224);
    wc_Sha224Free(&temp);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224Copy */


/*
 * Testing wc_InitRipeMd()
 */
static int test_wc_InitRipeMd(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;
    int ret;
    int flag = 0;

    /* Test good arg. */
    ret = wc_InitRipeMd(&ripemd);
    if (ret != 0) {
        flag = WOLFSSL_FATAL_ERROR;
    }

    /* Test bad arg. */
    if (!flag) {
        ret = wc_InitRipeMd(NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_InitRipeMd */

/*
 * Testing wc_RipeMdUpdate()
 */
static int test_wc_RipeMdUpdate(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;
    byte hash[RIPEMD_DIGEST_SIZE];
    testVector a, b, c;
    int ret;
    int flag = 0;

    ret = wc_InitRipeMd(&ripemd);
    if (ret != 0) {
        flag = ret;
    }

    /* Input */
    if (!flag) {
        a.input = "a";
        a.inLen = XSTRLEN(a.input);

        ret = wc_RipeMdUpdate(&ripemd, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_RipeMdFinal(&ripemd, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Update input. */
    if (!flag) {
        a.input = "abc";
        a.output = "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04\x4a\x8e\x98\xc6"
                   "\xb0\x87\xf1\x5a\x0b\xfc";
        a.inLen = XSTRLEN(a.input);
        a.outLen = XSTRLEN(a.output);

        ret = wc_RipeMdUpdate(&ripemd, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_RipeMdFinal(&ripemd, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, RIPEMD_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass in bad values. */
    if (!flag) {
        b.input = NULL;
        b.inLen = 0;

        ret = wc_RipeMdUpdate(&ripemd, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        c.input = NULL;
        c.inLen = RIPEMD_DIGEST_SIZE;

        ret = wc_RipeMdUpdate(&ripemd, (byte*)c.input, (word32)c.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_RipeMdUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_RipeMdUdpate */

/*
 * Unit test function for wc_RipeMdFinal()
 */
static int test_wc_RipeMdFinal(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_RIPEMD
    RipeMd ripemd;
    byte* hash_test[3];
    byte hash1[RIPEMD_DIGEST_SIZE];
    byte hash2[2*RIPEMD_DIGEST_SIZE];
    byte hash3[5*RIPEMD_DIGEST_SIZE];
    int times, i, ret;
    int flag = 0;

    /* Initialize */
    ret = wc_InitRipeMd(&ripemd);
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        hash_test[0] = hash1;
        hash_test[1] = hash2;
        hash_test[2] = hash3;
    }

    times = sizeof(hash_test) / sizeof(byte*);

    /* Testing oversized buffers. */
    for (i = 0; i < times; i++) {
        if (!flag) {
            ret = wc_RipeMdFinal(&ripemd, hash_test[i]);
            if (ret != 0) {
                flag = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_RipeMdFinal(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_RipeMdFinal(NULL, hash1);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_RipeMdFinal(&ripemd, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_RipeMdFinal */


/*
 * Testing wc_InitSha3_224, wc_InitSha3_256, wc_InitSha3_384, and
 * wc_InitSha3_512
 */
static int test_wc_InitSha3(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3)
    wc_Sha3 sha3;
    int     ret = 0;

    (void)sha3;

    #if !defined(WOLFSSL_NOSHA3_224)
        ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);

        /* Test bad args. */
        if (ret == 0) {
            ret = wc_InitSha3_224(NULL, HEAP_HINT, testDevId);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        wc_Sha3_224_Free(&sha3);
    #endif /* NOSHA3_224 */
    #if !defined(WOLFSSL_NOSHA3_256)
        if (ret == 0) {
            ret = wc_InitSha3_256(&sha3, HEAP_HINT, testDevId);
            /* Test bad args. */
            if (ret == 0) {
                ret = wc_InitSha3_256(NULL, HEAP_HINT, testDevId);
                if (ret == BAD_FUNC_ARG) {
                    ret = 0;
                }
                else if (ret == 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            wc_Sha3_256_Free(&sha3);
        } /* END sha3_256 */
    #endif /* NOSHA3_256 */
    #if !defined(WOLFSSL_NOSHA3_384)
        if (ret == 0) {
            ret = wc_InitSha3_384(&sha3, HEAP_HINT, testDevId);
            /* Test bad args. */
            if (ret == 0) {
                ret = wc_InitSha3_384(NULL, HEAP_HINT, testDevId);
                if (ret == BAD_FUNC_ARG) {
                    ret = 0;
                }
                else if (ret == 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            wc_Sha3_384_Free(&sha3);
        } /* END sha3_384 */
    #endif /* NOSHA3_384 */
    #if !defined(WOLFSSL_NOSHA3_512)
        if (ret == 0) {
            ret = wc_InitSha3_512(&sha3, HEAP_HINT, testDevId);
            /* Test bad args. */
            if (ret == 0) {
                ret = wc_InitSha3_512(NULL, HEAP_HINT, testDevId);
                if (ret == BAD_FUNC_ARG) {
                    ret = 0;
                }
                else if (ret == 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            wc_Sha3_512_Free(&sha3);
        } /* END sha3_512 */
    #endif /* NOSHA3_512 */

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_InitSha3 */


/*
 * Testing wc_Sha3_Update()
 */
static int testing_wc_Sha3_Update(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_XILINX_CRYPT) && \
   !defined(WOLFSSL_AFALG_XILINX)
    wc_Sha3 sha3;
    byte    msg[] = "Everybody's working for the weekend.";
    byte    msg2[] = "Everybody gets Friday off.";
    byte    msgCmp[] = "\x45\x76\x65\x72\x79\x62\x6f\x64\x79\x27\x73\x20"
                    "\x77\x6f\x72\x6b\x69\x6e\x67\x20\x66\x6f\x72\x20\x74"
                    "\x68\x65\x20\x77\x65\x65\x6b\x65\x6e\x64\x2e\x45\x76"
                    "\x65\x72\x79\x62\x6f\x64\x79\x20\x67\x65\x74\x73\x20"
                    "\x46\x72\x69\x64\x61\x79\x20\x6f\x66\x66\x2e";
    word32  msglen = sizeof(msg) - 1;
    word32  msg2len = sizeof(msg2);
    word32  msgCmplen = sizeof(msgCmp);
    int     ret = 0;

    #if !defined(WOLFSSL_NOSHA3_224)
        ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);
        if (ret != 0) {
            return TEST_FAIL;
        }

        ret = wc_Sha3_224_Update(&sha3, msg, msglen);
        if (XMEMCMP(msg, sha3.t, msglen) || sha3.i != msglen) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        if (ret == 0) {
            ret = wc_Sha3_224_Update(&sha3, msg2, msg2len);
            if (ret == 0 && XMEMCMP(sha3.t, msgCmp, msgCmplen) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        /* Pass bad args. */
        if (ret == 0) {
            ret = wc_Sha3_224_Update(NULL, msg2, msg2len);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_Sha3_224_Update(&sha3, NULL, 5);
            }
            if (ret == BAD_FUNC_ARG) {
                wc_Sha3_224_Free(&sha3);
                if (wc_InitSha3_224(&sha3, HEAP_HINT, testDevId)) {
                    return TEST_FAIL;
                }
                ret = wc_Sha3_224_Update(&sha3, NULL, 0);
                if (ret == 0) {
                    ret = wc_Sha3_224_Update(&sha3, msg2, msg2len);
                }
                if (ret == 0 && XMEMCMP(msg2, sha3.t, msg2len) != 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
        }
        wc_Sha3_224_Free(&sha3);
    #endif /* SHA3_224 */

    #if !defined(WOLFSSL_NOSHA3_256)
        if (ret == 0) {
            ret = wc_InitSha3_256(&sha3, HEAP_HINT, testDevId);
            if (ret != 0) {
                return TEST_FAIL;
            }
            ret = wc_Sha3_256_Update(&sha3, msg, msglen);
            if (XMEMCMP(msg, sha3.t, msglen) || sha3.i != msglen) {
                ret = WOLFSSL_FATAL_ERROR;
            }
            if (ret == 0) {
                ret = wc_Sha3_256_Update(&sha3, msg2, msg2len);
                if (XMEMCMP(sha3.t, msgCmp, msgCmplen) != 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            /* Pass bad args. */
            if (ret == 0) {
                ret = wc_Sha3_256_Update(NULL, msg2, msg2len);
                if (ret == BAD_FUNC_ARG) {
                    ret = wc_Sha3_256_Update(&sha3, NULL, 5);
                }
                if (ret == BAD_FUNC_ARG) {
                    wc_Sha3_256_Free(&sha3);
                    if (wc_InitSha3_256(&sha3, HEAP_HINT, testDevId)) {
                        return TEST_FAIL;
                    }
                    ret = wc_Sha3_256_Update(&sha3, NULL, 0);
                    if (ret == 0) {
                        ret = wc_Sha3_256_Update(&sha3, msg2, msg2len);
                    }
                    if (ret == 0 && XMEMCMP(msg2, sha3.t, msg2len) != 0) {
                        ret = WOLFSSL_FATAL_ERROR;
                    }
                }
            }
            wc_Sha3_256_Free(&sha3);
        }
    #endif /* SHA3_256 */

    #if !defined(WOLFSSL_NOSHA3_384)
        if (ret == 0) {
            ret = wc_InitSha3_384(&sha3, HEAP_HINT, testDevId);
            if (ret != 0) {
                return TEST_FAIL;
            }
            ret = wc_Sha3_384_Update(&sha3, msg, msglen);
            if (XMEMCMP(msg, sha3.t, msglen) || sha3.i != msglen) {
                ret = WOLFSSL_FATAL_ERROR;
            }
            if (ret == 0) {
                ret = wc_Sha3_384_Update(&sha3, msg2, msg2len);
                if (XMEMCMP(sha3.t, msgCmp, msgCmplen) != 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            /* Pass bad args. */
            if (ret == 0) {
                ret = wc_Sha3_384_Update(NULL, msg2, msg2len);
                if (ret == BAD_FUNC_ARG) {
                    ret = wc_Sha3_384_Update(&sha3, NULL, 5);
                }
                if (ret == BAD_FUNC_ARG) {
                    wc_Sha3_384_Free(&sha3);
                    if (wc_InitSha3_384(&sha3, HEAP_HINT, testDevId)) {
                        return TEST_FAIL;
                    }
                    ret = wc_Sha3_384_Update(&sha3, NULL, 0);
                    if (ret == 0) {
                        ret = wc_Sha3_384_Update(&sha3, msg2, msg2len);
                    }
                    if (ret == 0 && XMEMCMP(msg2, sha3.t, msg2len) != 0) {
                        ret = WOLFSSL_FATAL_ERROR;
                    }
                }
            }
            wc_Sha3_384_Free(&sha3);
        }
    #endif /* SHA3_384 */

    #if !defined(WOLFSSL_NOSHA3_512)
        if (ret == 0) {
            ret = wc_InitSha3_512(&sha3, HEAP_HINT, testDevId);
            if (ret != 0) {
                return TEST_FAIL;
            }
            ret = wc_Sha3_512_Update(&sha3, msg, msglen);
            if (XMEMCMP(msg, sha3.t, msglen) || sha3.i != msglen) {
                ret = WOLFSSL_FATAL_ERROR;
            }
            if (ret == 0) {
                ret = wc_Sha3_512_Update(&sha3, msg2, msg2len);
                if (XMEMCMP(sha3.t, msgCmp, msgCmplen) != 0) {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            }
            /* Pass bad args. */
            if (ret == 0) {
                ret = wc_Sha3_512_Update(NULL, msg2, msg2len);
                if (ret == BAD_FUNC_ARG) {
                    ret = wc_Sha3_512_Update(&sha3, NULL, 5);
                }
                if (ret == BAD_FUNC_ARG) {
                    wc_Sha3_512_Free(&sha3);
                    if (wc_InitSha3_512(&sha3, HEAP_HINT, testDevId)) {
                        return TEST_FAIL;
                    }
                    ret = wc_Sha3_512_Update(&sha3, NULL, 0);
                    if (ret == 0) {
                        ret = wc_Sha3_512_Update(&sha3, msg2, msg2len);
                    }
                    if (ret == 0 && XMEMCMP(msg2, sha3.t, msg2len) != 0) {
                        ret = WOLFSSL_FATAL_ERROR;
                    }
                }
            }
            wc_Sha3_512_Free(&sha3);
        }
    #endif /* SHA3_512 */

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SHA3 */
    return res;

} /* END testing_wc_Sha3_Update */

/*
 *  Testing wc_Sha3_224_Final()
 */
static int test_wc_Sha3_224_Final(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_224)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x8a\x24\x10\x8b\x15\x4a\xda\x21\xc9\xfd\x55"
                         "\x74\x49\x44\x79\xba\x5c\x7e\x7a\xb7\x6e\xf2"
                         "\x64\xea\xd0\xfc\xce\x33";
    byte        hash[WC_SHA3_224_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_224_DIGEST_SIZE];
    int         ret = 0;

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret= wc_Sha3_224_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
    if (ret == 0) {
        ret = wc_Sha3_224_Final(&sha3, hash);
        if (ret == 0 && XMEMCMP(expOut, hash, WC_SHA3_224_DIGEST_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_224_Final(NULL, hash);
        if (ret == 0) {
            ret = wc_Sha3_224_Final(&sha3, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Sha3_224_Free(&sha3);

    if (ret == 0) {
        ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);
        if (ret != 0) {
            return TEST_FAIL;
        }

        /* Init stack variables. */
        XMEMSET(hash, 0, sizeof(hash));
        XMEMSET(hashRet, 0, sizeof(hashRet));

        ret= wc_Sha3_224_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
        if (ret == 0) {
            ret = wc_Sha3_224_GetHash(&sha3, hashRet);
        }

        if (ret == 0) {
            ret = wc_Sha3_224_Final(&sha3, hash);
            if (ret == 0 && XMEMCMP(hash, hashRet, WC_SHA3_224_DIGEST_SIZE) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* Test bad args. */
            ret = wc_Sha3_224_GetHash(NULL, hashRet);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_Sha3_224_GetHash(&sha3, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    wc_Sha3_224_Free(&sha3);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Sha3_224_Final */


/*
 *  Testing wc_Sha3_256_Final()
 */
static int test_wc_Sha3_256_Final(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x41\xc0\xdb\xa2\xa9\xd6\x24\x08\x49\x10\x03\x76\xa8"
                        "\x23\x5e\x2c\x82\xe1\xb9\x99\x8a\x99\x9e\x21\xdb\x32"
                        "\xdd\x97\x49\x6d\x33\x76";
    byte        hash[WC_SHA3_256_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_256_DIGEST_SIZE];
    int         ret = 0;

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ret = wc_InitSha3_256(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret= wc_Sha3_256_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
    if (ret == 0) {
        ret = wc_Sha3_256_Final(&sha3, hash);
        if (ret == 0 && XMEMCMP(expOut, hash, WC_SHA3_256_DIGEST_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_256_Final(NULL, hash);
        if (ret == 0) {
            ret = wc_Sha3_256_Final(&sha3, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Sha3_256_Free(&sha3);

    if (ret == 0) {
        ret = wc_InitSha3_256(&sha3, HEAP_HINT, testDevId);
        if (ret != 0) {
            return TEST_FAIL;
        }
        /* Init stack variables. */
        XMEMSET(hash, 0, sizeof(hash));
        XMEMSET(hashRet, 0, sizeof(hashRet));

        ret= wc_Sha3_256_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
        if (ret == 0) {
            ret = wc_Sha3_256_GetHash(&sha3, hashRet);
        }
        if (ret == 0) {
            ret = wc_Sha3_256_Final(&sha3, hash);
            if (ret == 0 && XMEMCMP(hash, hashRet, WC_SHA3_256_DIGEST_SIZE) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* Test bad args. */
            ret = wc_Sha3_256_GetHash(NULL, hashRet);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_Sha3_256_GetHash(&sha3, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    wc_Sha3_256_Free(&sha3);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Sha3_256_Final */


/*
 *  Testing wc_Sha3_384_Final()
 */
static int test_wc_Sha3_384_Final(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3        sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x99\x1c\x66\x57\x55\xeb\x3a\x4b\x6b\xbd\xfb\x75\xc7"
                         "\x8a\x49\x2e\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42\x9b\xfd"
                         "\xbc\x32\xb9\xd4\xad\x5a\xa0\x4a\x1f\x07\x6e\x62\xfe"
                         "\xa1\x9e\xef\x51\xac\xd0\x65\x7c\x22";
    byte        hash[WC_SHA3_384_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_384_DIGEST_SIZE];
    int         ret = 0;

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ret = wc_InitSha3_384(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret= wc_Sha3_384_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
    if (ret == 0) {
        ret = wc_Sha3_384_Final(&sha3, hash);
        if (ret == 0 && XMEMCMP(expOut, hash, WC_SHA3_384_DIGEST_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_384_Final(NULL, hash);
        if (ret == 0) {
            ret = wc_Sha3_384_Final(&sha3, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Sha3_384_Free(&sha3);

    if (ret == 0) {
        ret = wc_InitSha3_384(&sha3, HEAP_HINT, testDevId);
        if (ret != 0) {
            return TEST_FAIL;
        }
        /* Init stack variables. */
        XMEMSET(hash, 0, sizeof(hash));
        XMEMSET(hashRet, 0, sizeof(hashRet));

        ret= wc_Sha3_384_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
        if (ret == 0) {
            ret = wc_Sha3_384_GetHash(&sha3, hashRet);
        }
        if (ret == 0) {
            ret = wc_Sha3_384_Final(&sha3, hash);
            if (ret == 0 && XMEMCMP(hash, hashRet, WC_SHA3_384_DIGEST_SIZE) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* Test bad args. */
            ret = wc_Sha3_384_GetHash(NULL, hashRet);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_Sha3_384_GetHash(&sha3, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    wc_Sha3_384_Free(&sha3);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Sha3_384_Final */



/*
 *  Testing wc_Sha3_512_Final()
 */
static int test_wc_Sha3_512_Final(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_512) && \
   !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3     sha3;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8\xb7\x7c\xb4\x86\x10"
                         "\xfc\xa8\x18\x2d\xd4\x57\xce\x6f\x32\x6a\x0f\xd3\xd7"
                         "\xec\x2f\x1e\x91\x63\x6d\xee\x69\x1f\xbe\x0c\x98\x53"
                         "\x02\xba\x1b\x0d\x8d\xc7\x8c\x08\x63\x46\xb5\x33\xb4"
                         "\x9c\x03\x0d\x99\xa2\x7d\xaf\x11\x39\xd6\xe7\x5e";
    byte        hash[WC_SHA3_512_DIGEST_SIZE];
    byte        hashRet[WC_SHA3_512_DIGEST_SIZE];
    int         ret = 0;

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ret = wc_InitSha3_512(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret= wc_Sha3_512_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
    if (ret == 0) {
        ret = wc_Sha3_512_Final(&sha3, hash);
        if (ret == 0 && XMEMCMP(expOut, hash, WC_SHA3_512_DIGEST_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_512_Final(NULL, hash);
        if (ret == 0) {
            ret = wc_Sha3_384_Final(&sha3, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Sha3_512_Free(&sha3);

    if (ret == 0) {
        ret = wc_InitSha3_512(&sha3, HEAP_HINT, testDevId);
        if (ret != 0) {
            return TEST_FAIL;
        }
        /* Init stack variables. */
        XMEMSET(hash, 0, sizeof(hash));
        XMEMSET(hashRet, 0, sizeof(hashRet));

        ret= wc_Sha3_512_Update(&sha3, (byte*)msg, (word32)XSTRLEN(msg));
        if (ret == 0) {
            ret = wc_Sha3_512_GetHash(&sha3, hashRet);
        }
        if (ret == 0) {
            ret = wc_Sha3_512_Final(&sha3, hash);
            if (ret == 0 && XMEMCMP(hash, hashRet, WC_SHA3_512_DIGEST_SIZE) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* Test bad args. */
            ret = wc_Sha3_512_GetHash(NULL, hashRet);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_Sha3_512_GetHash(&sha3, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    wc_Sha3_512_Free(&sha3);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Sha3_512_Final */


/*
 *  Testing wc_Sha3_224_Copy()
 */
static int test_wc_Sha3_224_Copy(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_224)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_224_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_224_DIGEST_SIZE];
    int         ret = 0;

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_InitSha3_224(&sha3Cpy, HEAP_HINT, testDevId);
    if (ret != 0) {
        wc_Sha3_224_Free(&sha3);
        return TEST_FAIL;
    }

    ret = wc_Sha3_224_Update(&sha3, (byte*)msg, msglen);

    if (ret == 0) {
        ret = wc_Sha3_224_Copy(&sha3Cpy, &sha3);
        if (ret == 0) {
            ret = wc_Sha3_224_Final(&sha3, hash);
            if (ret == 0) {
                ret = wc_Sha3_224_Final(&sha3Cpy, hashCpy);
            }
        }
        if (ret == 0 && XMEMCMP(hash, hashCpy, sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_224_Copy(NULL, &sha3);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Sha3_224_Copy(&sha3Cpy, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Sha3_224_Copy */



/*
 *  Testing wc_Sha3_256_Copy()
 */
static int test_wc_Sha3_256_Copy(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_256_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_256_DIGEST_SIZE];
    int         ret = 0;

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ret = wc_InitSha3_256(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_InitSha3_256(&sha3Cpy, HEAP_HINT, testDevId);
    if (ret != 0) {
        wc_Sha3_256_Free(&sha3);
        return TEST_FAIL;
    }

    ret = wc_Sha3_256_Update(&sha3, (byte*)msg, msglen);

    if (ret == 0) {
        ret = wc_Sha3_256_Copy(&sha3Cpy, &sha3);
        if (ret == 0) {
            ret = wc_Sha3_256_Final(&sha3, hash);
            if (ret == 0) {
                ret = wc_Sha3_256_Final(&sha3Cpy, hashCpy);
            }
        }
        if (ret == 0 && XMEMCMP(hash, hashCpy, sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_256_Copy(NULL, &sha3);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Sha3_256_Copy(&sha3Cpy, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Sha3_256_Copy */



/*
 *  Testing wc_Sha3_384_Copy()
 */
static int test_wc_Sha3_384_Copy(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_384)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_384_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_384_DIGEST_SIZE];
    int         ret = 0;

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ret = wc_InitSha3_384(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_InitSha3_384(&sha3Cpy, HEAP_HINT, testDevId);
    if (ret != 0) {
        wc_Sha3_384_Free(&sha3);
        return TEST_FAIL;
    }

    ret = wc_Sha3_384_Update(&sha3, (byte*)msg, msglen);

    if (ret == 0) {
        ret = wc_Sha3_384_Copy(&sha3Cpy, &sha3);
        if (ret == 0) {
            ret = wc_Sha3_384_Final(&sha3, hash);
            if (ret == 0) {
                ret = wc_Sha3_384_Final(&sha3Cpy, hashCpy);
            }
        }
        if (ret == 0 && XMEMCMP(hash, hashCpy, sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_384_Copy(NULL, &sha3);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Sha3_384_Copy(&sha3Cpy, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Sha3_384_Copy */


/*
 *  Testing wc_Sha3_512_Copy()
 */
static int test_wc_Sha3_512_Copy(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_512)
    wc_Sha3     sha3, sha3Cpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[WC_SHA3_512_DIGEST_SIZE];
    byte        hashCpy[WC_SHA3_512_DIGEST_SIZE];
    int         ret = 0;

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ret = wc_InitSha3_512(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_InitSha3_512(&sha3Cpy, HEAP_HINT, testDevId);
    if (ret != 0) {
    wc_Sha3_512_Free(&sha3);
    return TEST_FAIL;
    }

    ret = wc_Sha3_512_Update(&sha3, (byte*)msg, msglen);

    if (ret == 0) {
        ret = wc_Sha3_512_Copy(&sha3Cpy, &sha3);
        if (ret == 0) {
            ret = wc_Sha3_512_Final(&sha3, hash);
            if (ret == 0) {
                ret = wc_Sha3_512_Final(&sha3Cpy, hashCpy);
            }
        }
        if (ret == 0 && XMEMCMP(hash, hashCpy, sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Sha3_512_Copy(NULL, &sha3);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Sha3_512_Copy(&sha3Cpy, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Sha3_512_Copy */
/*
 * Unit test function for wc_Sha3_GetFlags()
 */
static int test_wc_Sha3_GetFlags(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SHA3) && defined(WOLFSSL_HASH_FLAGS)
    wc_Sha3 sha3;
    word32  flags = 0;
    int     ret = 0;

    /* Initialize */
    ret = wc_InitSha3_224(&sha3, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }
    if (ret == 0) {
        ret = wc_Sha3_GetFlags(&sha3, &flags);
    }
    if (ret == 0) {
        if (flags & WC_HASH_FLAG_ISCOPY) {
            ret = 0;
        }
    }
    wc_Sha3_224_Free(&sha3);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Sha3_GetFlags */


static int test_wc_InitShake256(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHAKE256
    wc_Shake shake;
    int      ret = 0;

    ret = wc_InitShake256(&shake, HEAP_HINT, testDevId);
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_InitShake256(NULL, HEAP_HINT, testDevId);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Shake256_Free(&shake);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_InitSha3 */


static int testing_wc_Shake256_Update(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHAKE256
    wc_Shake shake;
    byte     msg[] = "Everybody's working for the weekend.";
    byte     msg2[] = "Everybody gets Friday off.";
    byte     msgCmp[] = "\x45\x76\x65\x72\x79\x62\x6f\x64\x79\x27\x73\x20"
                        "\x77\x6f\x72\x6b\x69\x6e\x67\x20\x66\x6f\x72\x20\x74"
                        "\x68\x65\x20\x77\x65\x65\x6b\x65\x6e\x64\x2e\x45\x76"
                        "\x65\x72\x79\x62\x6f\x64\x79\x20\x67\x65\x74\x73\x20"
                        "\x46\x72\x69\x64\x61\x79\x20\x6f\x66\x66\x2e";
    word32   msglen = sizeof(msg) - 1;
    word32   msg2len = sizeof(msg2);
    word32   msgCmplen = sizeof(msgCmp);
    int      ret = 0;

    ret = wc_InitShake256(&shake, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }
    ret = wc_Shake256_Update(&shake, msg, msglen);
    if (XMEMCMP(msg, shake.t, msglen) || shake.i != msglen) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = wc_Shake256_Update(&shake, msg2, msg2len);
        if (XMEMCMP(shake.t, msgCmp, msgCmplen) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Pass bad args. */
    if (ret == 0) {
        ret = wc_Shake256_Update(NULL, msg2, msg2len);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Shake256_Update(&shake, NULL, 5);
        }
        if (ret == BAD_FUNC_ARG) {
            wc_Shake256_Free(&shake);
            if (wc_InitShake256(&shake, HEAP_HINT, testDevId)) {
                return TEST_FAIL;
            }
            ret = wc_Shake256_Update(&shake, NULL, 0);
            if (ret == 0) {
                ret = wc_Shake256_Update(&shake, msg2, msg2len);
            }
            if (ret == 0 && XMEMCMP(msg2, shake.t, msg2len) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }
    wc_Shake256_Free(&shake);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SHAKE256 */
    return res;
}

static int test_wc_Shake256_Final(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHAKE256
    wc_Shake    shake;
    const char* msg    = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnom"
                         "nopnopq";
    const char* expOut = "\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f"
                         "\x6f\x87\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b"
                         "\xe5\xd4\xfd\x2e\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59"
                         "\xaa\x80\x60\xf1\xf9\xbc\x99\x6c\x05\xac\xa3\xc6\x96"
                         "\xa8\xb6\x62\x79\xdc\x67\x2c\x74\x0b\xb2\x24\xec\x37"
                         "\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55\xf5\x1d\x97"
                         "\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a\xf2"
                         "\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67"
                         "\x60\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4";
    byte        hash[114];
    int         ret = 0;

    /* Init stack variables. */
    XMEMSET(hash, 0, sizeof(hash));

    ret = wc_InitShake256(&shake, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret= wc_Shake256_Update(&shake, (byte*)msg, (word32)XSTRLEN(msg));
    if (ret == 0) {
        ret = wc_Shake256_Final(&shake, hash, (word32)sizeof(hash));
        if (ret == 0 && XMEMCMP(expOut, hash, (word32)sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Shake256_Final(NULL, hash, (word32)sizeof(hash));
        if (ret == 0) {
            ret = wc_Shake256_Final(&shake, NULL, (word32)sizeof(hash));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Shake256_Free(&shake);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}
/*
 *  Testing wc_Shake256_Copy()
 */
static int test_wc_Shake256_Copy(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHAKE256
    wc_Shake    shake, shakeCpy;
    const char* msg = TEST_STRING;
    word32      msglen = (word32)TEST_STRING_SZ;
    byte        hash[144];
    byte        hashCpy[144];
    word32      hashLen = sizeof(hash);
    word32      hashLenCpy = sizeof(hashCpy);
    int         ret;

    XMEMSET(hash, 0, sizeof(hash));
    XMEMSET(hashCpy, 0, sizeof(hashCpy));

    ret = wc_InitShake256(&shake, HEAP_HINT, testDevId);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_InitShake256(&shakeCpy, HEAP_HINT, testDevId);
    if (ret != 0) {
        wc_Shake256_Free(&shake);
        return TEST_FAIL;
    }

    ret = wc_Shake256_Update(&shake, (byte*)msg, msglen);

    if (ret == 0) {
        ret = wc_Shake256_Copy(&shakeCpy, &shake);
        if (ret == 0) {
            ret = wc_Shake256_Final(&shake, hash, hashLen);
            if (ret == 0) {
                ret = wc_Shake256_Final(&shakeCpy, hashCpy, hashLenCpy);
            }
        }
        if (ret == 0 && XMEMCMP(hash, hashCpy, sizeof(hash)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Shake256_Copy(NULL, &shake);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Shake256_Copy(&shakeCpy, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    wc_Shake256_Free(&shake);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Shake256_Copy */
/*
 * Unit test function for wc_Shake256Hash()
 */
static int test_wc_Shake256Hash(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_SHAKE256
    const byte data[] = { /* Hello World */
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x57,0x6f,
        0x72,0x6c,0x64
    };
    word32     len = sizeof(data);
    byte       hash[144];
    word32     hashLen = sizeof(hash);
    int        ret;

    ret = wc_Shake256Hash(data, len, hash, hashLen);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}  /* END test_wc_Shake256Hash */

/*
 * Test function for wc_HmacSetKey
 */
static int test_wc_Md5HmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_MD5)
    Hmac hmac;
    int ret, times, itr;
    int flag = 0;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);
    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return TEST_FAIL;

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[itr],
                            (word32)XSTRLEN(keys[itr]));
#if defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5)
        wc_HmacFree(&hmac);
        if (ret == BAD_FUNC_ARG) {
            return TEST_SUCCESS;
        }
        else {
            return TEST_FAIL;
        }
#else
        if (ret != 0) {
            flag = ret;
        }
#endif
    }

    /* Bad args. */
    if (!flag) {
        ret = wc_HmacSetKey(NULL, WC_MD5, (byte*)keys[0],
                                        (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_MD5, NULL, (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, 20, (byte*)keys[0],
                                        (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
        if (ret != HMAC_MIN_KEYLEN_E) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#else
        if (ret != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#endif
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Md5HmacSetKey */


/*
 * testing wc_HmacSetKey() on wc_Sha hash.
 */
static int test_wc_ShaHmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    int ret, times, itr;
    int flag = 0;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);
    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[itr],
                                        (word32)XSTRLEN(keys[itr]));
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Bad args. */
    if (!flag) {
        ret = wc_HmacSetKey(NULL, WC_SHA, (byte*)keys[0],
                                        (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA, NULL, (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, 20, (byte*)keys[0],
                                        (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
        if (ret != HMAC_MIN_KEYLEN_E) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#else
        if (ret != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#endif
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_ShaHmacSetKey() */

/*
 * testing wc_HmacSetKey() on Sha224 hash.
 */
static int test_wc_Sha224HmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    int ret, times, itr;
    int flag = 0;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);
    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[itr],
                                            (word32)XSTRLEN(keys[itr]));
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Bad args. */
    if (!flag) {
        ret = wc_HmacSetKey(NULL, WC_SHA224, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA224, NULL, (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, 20, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
        if (ret != HMAC_MIN_KEYLEN_E) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#else
        if (ret != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#endif
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224HmacSetKey() */

 /*
  * testing wc_HmacSetKey() on Sha256 hash
  */
static int test_wc_Sha256HmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    int ret, times, itr;
    int flag = 0;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);
    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[itr],
                                            (word32)XSTRLEN(keys[itr]));
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Bad args. */
    if (!flag) {
        ret = wc_HmacSetKey(NULL, WC_SHA256, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA256, NULL, (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, 20, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
        if (ret != HMAC_MIN_KEYLEN_E) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#else
        if (ret != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#endif
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha256HmacSetKey() */


/*
 * testing wc_HmacSetKey on Sha384 hash.
 */
static int test_wc_Sha384HmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    int ret, times, itr;
    int flag = 0;

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
#ifndef HAVE_FIPS
        "Jefe", /* smaller than minimum FIPS key size */
#endif
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    times = sizeof(keys) / sizeof(char*);
    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    for (itr = 0; itr < times; itr++) {
        ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[itr],
                                            (word32)XSTRLEN(keys[itr]));
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Bad args. */
    if (!flag) {
        ret = wc_HmacSetKey(NULL, WC_SHA384, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA384, NULL, (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, 20, (byte*)keys[0],
                                            (word32)XSTRLEN(keys[0]));
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[0], 0);
#ifdef HAVE_FIPS
        if (ret != HMAC_MIN_KEYLEN_E) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#else
        if (ret != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
#endif
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha384HmacSetKey() */


/*
 * testing wc_HmacUpdate on wc_Md5 hash.
 */
static int test_wc_Md5HmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5))
    Hmac hmac;
    testVector a, b;
    int ret;
    int flag = 0;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);

    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys, (word32)XSTRLEN(keys));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }
    /* Update Hmac. */
    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, NULL, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Md5HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA hash.
 */
static int test_wc_ShaHmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    testVector a, b;
    int ret;
    int flag = 0;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);

    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys, (word32)XSTRLEN(keys));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }
    /* Update Hmac. */
    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, NULL, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_ShaHmacUpdate */

/*
 * testing wc_HmacUpdate on SHA224 hash.
 */
static int test_wc_Sha224HmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    testVector a, b;
    int ret;
    int flag = 0;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);

    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys, (word32)XSTRLEN(keys));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }
    /* Update Hmac. */
    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, NULL, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA256 hash.
 */
static int test_wc_Sha256HmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    testVector a, b;
    int ret;
    int flag = 0;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);

    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys, (word32)XSTRLEN(keys));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }
    /* Update Hmac. */
    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, NULL, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha256HmacUpdate */

/*
 * testing wc_HmacUpdate on SHA384  hash.
 */
static int test_wc_Sha384HmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    testVector a, b;
    int ret;
    int flag = 0;
#ifdef HAVE_FIPS
    const char* keys =
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
#else
    const char* keys = "Jefe";
#endif

    a.input = "what do ya want for nothing?";
    a.inLen  = XSTRLEN(a.input);

    b.input = "Hi There";
    b.inLen = XSTRLEN(b.input);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys, (word32)XSTRLEN(keys));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)b.input, (word32)b.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }
    /* Update Hmac. */
    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    /* Test bad args. */
    if (!flag) {
        ret = wc_HmacUpdate(NULL, (byte*)a.input, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, NULL, (word32)a.inLen);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, 0);
        if (ret != 0) {
            flag = ret;
        }
    }

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha384HmacUpdate */

/*
 * Testing wc_HmacFinal() with MD5
 */

static int test_wc_Md5HmacFinal(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 5))
    Hmac hmac;
    byte hash[WC_MD5_DIGEST_SIZE];
    testVector a;
    int ret;
    const char* key;
    int flag = 0;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\x92\x94\x72\x7a\x36\x38\xbb\x1c\x13\xf4\x8e\xf8\x15\x8b\xfc"
               "\x9d";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)key, (word32)XSTRLEN(key));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try bad parameters. */
    if (!flag) {
        ret = wc_HmacFinal(NULL, hash);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

#ifndef HAVE_FIPS
    if (!flag) {
        ret = wc_HmacFinal(&hmac, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_Md5HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA
 */
static int test_wc_ShaHmacFinal(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA)
    Hmac hmac;
    byte hash[WC_SHA_DIGEST_SIZE];
    testVector a;
    int ret;
    int flag = 0;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xb6\x17\x31\x86\x55\x05\x72\x64\xe2\x8b\xc0\xb6\xfb\x37\x8c"
               "\x8e\xf1\x46\xbe\x00";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)key, (word32)XSTRLEN(key));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try bad parameters. */
    if (!flag) {
        ret = wc_HmacFinal(NULL, hash);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

#ifndef HAVE_FIPS
    if (!flag) {
        ret = wc_HmacFinal(&hmac, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;

} /* END test_wc_ShaHmacFinal */


/*
 * Testing wc_HmacFinal() with SHA224
 */
static int test_wc_Sha224HmacFinal(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
    Hmac hmac;
    byte hash[WC_SHA224_DIGEST_SIZE];
    testVector a;
    int ret;
    int flag = 0;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19\x68\x32\x10\x7c\xd4\x9d\xf3"
               "\x3f\x47\xb4\xb1\x16\x99\x12\xba\x4f\x53\x68\x4b\x22";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)key, (word32)XSTRLEN(key));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA224_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try bad parameters. */
    if (!flag) {
        ret = wc_HmacFinal(NULL, hash);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

#ifndef HAVE_FIPS
    if (!flag) {
        ret = wc_HmacFinal(&hmac, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha224HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA256
 */
static int test_wc_Sha256HmacFinal(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && !defined(NO_SHA256)
    Hmac hmac;
    byte hash[WC_SHA256_DIGEST_SIZE];
    testVector a;
    int ret;
    int flag = 0;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1"
               "\x2b\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32"
               "\xcf\xf7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return TEST_FAIL;

    ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)key, (word32)XSTRLEN(key));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA256_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try bad parameters. */
    if (!flag) {
        ret = wc_HmacFinal(NULL, hash);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

#ifndef HAVE_FIPS
    if (!flag) {
        ret = wc_HmacFinal(&hmac, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha256HmacFinal */

/*
 * Testing wc_HmacFinal() with SHA384
 */
static int test_wc_Sha384HmacFinal(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
    Hmac hmac;
    byte hash[WC_SHA384_DIGEST_SIZE];
    testVector a;
    int ret;
    int flag = 0;
    const char* key;

    key = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b";
    a.input = "Hi There";
    a.output = "\xaf\xd0\x39\x44\xd8\x48\x95\x62\x6b\x08\x25\xf4\xab\x46\x90"
               "\x7f\x15\xf9\xda\xdb\xe4\x10\x1e\xc6\x82\xaa\x03\x4c\x7c\xeb"
               "\xc5\x9c\xfa\xea\x9e\xa9\x07\x6e\xde\x7f\x4a\xf1\x52\xe8\xb2"
               "\xfa\x9c\xb6";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = XSTRLEN(a.output);

    flag = 0;

    ret = wc_HmacInit(&hmac, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)key, (word32)XSTRLEN(key));
    if (ret != 0) {
        flag = ret;
    }

    if (!flag) {
        ret = wc_HmacUpdate(&hmac, (byte*)a.input, (word32)a.inLen);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0) {
            flag = ret;
        }
    }

    if (!flag) {
        if (XMEMCMP(hash, a.output, WC_SHA384_DIGEST_SIZE) != 0) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Try bad parameters. */
    if (!flag) {
        ret = wc_HmacFinal(NULL, hash);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#ifndef HAVE_FIPS
    if (!flag) {
        ret = wc_HmacFinal(&hmac, NULL);
        if (ret != BAD_FUNC_ARG) {
            flag = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_HmacFree(&hmac);

    res = TEST_RES_CHECK(flag == 0);
#endif
    return res;
} /* END test_wc_Sha384HmacFinal */



/*
 * Testing wc_InitCmac()
 */
static int test_wc_InitCmac(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES)
    Cmac        cmac1, cmac2, cmac3;
    /* AES 128 key. */
    byte        key1[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x10\x11\x12\x13\x14\x15\x16";
    /* AES 192 key. */
    byte        key2[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16"
                         "\x01\x02\x03\x04\x05\x06\x07\x08";

    /* AES 256 key. */
    byte        key3[] = "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16"
                         "\x01\x02\x03\x04\x05\x06\x07\x08"
                         "\x09\x01\x11\x12\x13\x14\x15\x16";

    word32      key1Sz = (word32)sizeof(key1) - 1;
    word32      key2Sz = (word32)sizeof(key2) - 1;
    word32      key3Sz = (word32)sizeof(key3) - 1;
    int         type   = WC_CMAC_AES;
    int         ret = 0;

#ifdef WOLFSSL_AES_128
    ret = wc_InitCmac(&cmac1, key1, key1Sz, type, NULL);
#endif
#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        wc_AesFree(&cmac1.aes);
        ret = wc_InitCmac(&cmac2, key2, key2Sz, type, NULL);
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        wc_AesFree(&cmac2.aes);
        ret = wc_InitCmac(&cmac3, key3, key3Sz, type, NULL);
    }
#endif

    /* Test bad args. */
    if (ret == 0) {
        wc_AesFree(&cmac3.aes);
        ret = wc_InitCmac(NULL, key3, key3Sz, type, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_InitCmac(&cmac3, NULL, key3Sz, type, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_InitCmac(&cmac3, key3, 0, type, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_InitCmac(&cmac3, key3, key3Sz, 0, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    (void)key1;
    (void)key1Sz;
    (void)key2;
    (void)key2Sz;
    (void)cmac1;
    (void)cmac2;

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_InitCmac */


/*
 * Testing wc_CmacUpdate()
 */
static int test_wc_CmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Cmac        cmac;
    byte        key[] =
    {
        0x64, 0x4c, 0xbf, 0x12, 0x85, 0x9d, 0xf0, 0x55,
        0x7e, 0xa9, 0x1f, 0x08, 0xe0, 0x51, 0xff, 0x27
    };
    byte        in[] = "\xe2\xb4\xb6\xf9\x48\x44\x02\x64"
                       "\x5c\x47\x80\x9e\xd5\xa8\x3a\x17"
                       "\xb3\x78\xcf\x85\x22\x41\x74\xd9"
                       "\xa0\x97\x39\x71\x62\xf1\x8e\x8f"
                       "\xf4";

    word32      inSz  = (word32)sizeof(in) - 1;
    word32      keySz = (word32)sizeof(key);
    int         type  = WC_CMAC_AES;
    int         ret = 0;

    ret = wc_InitCmac(&cmac, key, keySz, type, NULL);
    if (ret != 0) {
        return ret;
    }

    ret = wc_CmacUpdate(&cmac, in, inSz);

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_CmacUpdate(NULL, in, inSz);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_CmacUpdate(&cmac, NULL, 30);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        wc_AesFree(&cmac.aes);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_CmacUpdate */


/*
 * Testing wc_CmacFinal()
 */
static int test_wc_CmacFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Cmac        cmac;
    byte        key[] =
    {
        0x64, 0x4c, 0xbf, 0x12, 0x85, 0x9d, 0xf0, 0x55,
        0x7e, 0xa9, 0x1f, 0x08, 0xe0, 0x51, 0xff, 0x27
    };
    byte        msg[] =
    {
        0xe2, 0xb4, 0xb6, 0xf9, 0x48, 0x44, 0x02, 0x64,
        0x5c, 0x47, 0x80, 0x9e, 0xd5, 0xa8, 0x3a, 0x17,
        0xb3, 0x78, 0xcf, 0x85, 0x22, 0x41, 0x74, 0xd9,
        0xa0, 0x97, 0x39, 0x71, 0x62, 0xf1, 0x8e, 0x8f,
        0xf4
    };
    /* Test vectors from CMACGenAES128.rsp from
     * http://csrc.nist.gov/groups/STM/cavp/block-cipher-modes.html#cmac
     * Per RFC4493 truncation of lsb is possible.
     */
    byte        expMac[] =
    {
        0x4e, 0x6e, 0xc5, 0x6f, 0xf9, 0x5d, 0x0e, 0xae,
        0x1c, 0xf8, 0x3e, 0xfc, 0xf4, 0x4b, 0xeb
    };
    byte        mac[AES_BLOCK_SIZE];
    word32      msgSz    = (word32)sizeof(msg);
    word32      keySz    = (word32)sizeof(key);
    word32      macSz    = sizeof(mac);
    word32      badMacSz = 17;
    int         expMacSz = sizeof(expMac);
    int         type     = WC_CMAC_AES;
    int         ret = 0;

    XMEMSET(mac, 0, macSz);

    ret = wc_InitCmac(&cmac, key, keySz, type, NULL);
    if (ret != 0) {
        return ret;
    }
    ret = wc_CmacUpdate(&cmac, msg, msgSz);

    if (ret == 0) {
        ret = wc_CmacFinal(&cmac, mac, &macSz);
        if (ret == 0 && XMEMCMP(mac, expMac, expMacSz) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_CmacFinal(NULL, mac, &macSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_CmacFinal(&cmac, NULL, &macSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_CmacFinal(&cmac, mac, &badMacSz);
                if (ret == BUFFER_E) {
                    ret = 0;
                }
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_CmacFinal */


/*
 * Testing wc_AesCmacGenerate() && wc_AesCmacVerify()
 */
static int test_wc_AesCmacGenerate(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Cmac        cmac;
    byte        key[] =
    {
        0x26, 0xef, 0x8b, 0x40, 0x34, 0x11, 0x7d, 0x9e,
        0xbe, 0xc0, 0xc7, 0xfc, 0x31, 0x08, 0x54, 0x69
    };
    byte        msg[]    = "\x18\x90\x49\xef\xfd\x7c\xf9\xc8"
                           "\xf3\x59\x65\xbc\xb0\x97\x8f\xd4";
    byte        expMac[] = "\x29\x5f\x2f\x71\xfc\x58\xe6\xf6"
                           "\x3d\x32\x65\x4c\x66\x23\xc5";
    byte        mac[AES_BLOCK_SIZE];
    word32      keySz    = sizeof(key);
    word32      macSz    = sizeof(mac);
    word32      msgSz    = sizeof(msg) - 1;
    word32      expMacSz = sizeof(expMac) - 1;
    int         type     = WC_CMAC_AES;
    int         ret = 0;

    XMEMSET(mac, 0, macSz);

    ret = wc_InitCmac(&cmac, key, keySz, type, NULL);
    if (ret != 0) {
        return ret;
    }

    ret = wc_CmacUpdate(&cmac, msg, msgSz);
    if (ret != 0) {
        return ret;
    }
    else {
        wc_AesFree(&cmac.aes);
    }

    ret = wc_AesCmacGenerate(mac, &macSz, msg, msgSz, key, keySz);
    if (ret == 0 && XMEMCMP(mac, expMac, expMacSz) != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_AesCmacGenerate(NULL, &macSz, msg, msgSz, key, keySz);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCmacGenerate(mac, &macSz, msg, msgSz, NULL, keySz);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCmacGenerate(mac, &macSz, msg, msgSz, key, 0);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCmacGenerate(mac, &macSz, NULL, msgSz, key, keySz);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_AesCmacVerify(mac, macSz, msg, msgSz, key, keySz);
        /* Test bad args. */
        if (ret == 0) {
            ret = wc_AesCmacVerify(NULL, macSz, msg, msgSz, key, keySz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_AesCmacVerify(mac, 0, msg, msgSz, key, keySz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_AesCmacVerify(mac, macSz, msg, msgSz, NULL, keySz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_AesCmacVerify(mac, macSz, msg, msgSz, key, 0);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_AesCmacVerify(mac, macSz, NULL, msgSz, key, keySz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_AesCmacGenerate */


/*
 * Testing streaming AES-GCM API.
 */
static int test_wc_AesGcmStream(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(WOLFSSL_AES_128) && defined(HAVE_AESGCM) && \
    defined(WOLFSSL_AESGCM_STREAM)
    int ret = 0;
    int i;
    WC_RNG rng[1];
    Aes aesEnc[1];
    Aes aesDec[1];
    byte tag[AES_BLOCK_SIZE];
    byte in[AES_BLOCK_SIZE * 3 + 2] = { 0, };
    byte out[AES_BLOCK_SIZE * 3 + 2];
    byte plain[AES_BLOCK_SIZE * 3 + 2];
    byte aad[AES_BLOCK_SIZE * 3 + 2] = { 0, };
    byte key[AES_128_KEY_SIZE] = { 0, };
    byte iv[AES_IV_SIZE] = { 1, };
    byte ivOut[AES_IV_SIZE];
    static const byte expTagAAD1[AES_BLOCK_SIZE] = {
        0x6c, 0x35, 0xe6, 0x7f, 0x59, 0x9e, 0xa9, 0x2f,
        0x27, 0x2d, 0x5f, 0x8e, 0x7e, 0x42, 0xd3, 0x05
    };
    static const byte expTagPlain1[AES_BLOCK_SIZE] = {
        0x24, 0xba, 0x57, 0x95, 0xd0, 0x27, 0x9e, 0x78,
        0x3a, 0x88, 0x4c, 0x0a, 0x5d, 0x50, 0x23, 0xd1
    };
    static const byte expTag[AES_BLOCK_SIZE] = {
        0x22, 0x91, 0x70, 0xad, 0x42, 0xc3, 0xad, 0x96,
        0xe0, 0x31, 0x57, 0x60, 0xb7, 0x92, 0xa3, 0x6d
    };

    /* Create a random for generating IV/nonce. */
    AssertIntEQ(wc_InitRng(rng), 0);

    /* Initialize data structures. */
    AssertIntEQ(wc_AesInit(aesEnc, NULL, INVALID_DEVID), 0);
    AssertIntEQ(wc_AesInit(aesDec, NULL, INVALID_DEVID), 0);

    /* BadParameters to streaming init. */
    AssertIntEQ(wc_AesGcmEncryptInit(NULL, NULL, 0, NULL, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptInit(NULL, NULL, 0, NULL, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptInit(aesEnc, NULL, AES_128_KEY_SIZE, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptInit(aesEnc, NULL, 0, NULL, GCM_NONCE_MID_SZ),
                BAD_FUNC_ARG);

    /* Bad parameters to encrypt update. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(NULL, NULL, NULL, 0, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, in, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, NULL, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, NULL, 1),
                BAD_FUNC_ARG);
    /* Bad parameters to decrypt update. */
    AssertIntEQ(wc_AesGcmDecryptUpdate(NULL, NULL, NULL, 0, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, in, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, out, NULL, 1, NULL, 0),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, NULL, 1),
                BAD_FUNC_ARG);

    /* Bad parameters to encrypt final. */
    AssertIntEQ(wc_AesGcmEncryptFinal(NULL, NULL, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptFinal(NULL, tag, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptFinal(NULL, NULL, AES_BLOCK_SIZE),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, NULL, AES_BLOCK_SIZE),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE + 1),
                BAD_FUNC_ARG);
    /* Bad parameters to decrypt final. */
    AssertIntEQ(wc_AesGcmDecryptFinal(NULL, NULL, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptFinal(NULL, tag, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptFinal(NULL, NULL, AES_BLOCK_SIZE),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, NULL, AES_BLOCK_SIZE),
                BAD_FUNC_ARG);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE + 1),
                BAD_FUNC_ARG);

    /* Check calling final before setting key fails. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, sizeof(tag)), MISSING_KEY);
    AssertIntEQ(wc_AesGcmEncryptFinal(aesDec, tag, sizeof(tag)), MISSING_KEY);
    /* Check calling update before setting key else fails. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1),
                MISSING_KEY);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1),
                MISSING_KEY);

    /* Set key but not IV. */
    AssertIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), NULL, 0), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), NULL, 0), 0);
    /* Check calling final before setting IV fails. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, sizeof(tag)), MISSING_IV);
    AssertIntEQ(wc_AesGcmEncryptFinal(aesDec, tag, sizeof(tag)), MISSING_IV);
    /* Check calling update before setting IV else fails. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1),
                MISSING_IV);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1),
                MISSING_IV);

    /* Set IV using fixed part IV and external IV APIs. */
    AssertIntEQ(wc_AesGcmSetIV(aesEnc, GCM_NONCE_MID_SZ, iv, AES_IV_FIXED_SZ,
                               rng), 0);
    AssertIntEQ(wc_AesGcmEncryptInit_ex(aesEnc, NULL, 0, ivOut,
                                        GCM_NONCE_MID_SZ), 0);
    AssertIntEQ(wc_AesGcmSetExtIV(aesDec, ivOut, GCM_NONCE_MID_SZ), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, NULL, 0, NULL, 0), 0);
    /* Encrypt and decrypt data. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, 1, aad, 1), 0);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, 1, aad, 1), 0);
    AssertIntEQ(XMEMCMP(plain, in, 1), 0);
    /* Finalize and check tag matches. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    AssertIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encrypt/decrypt one block and AAD of one block. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, AES_BLOCK_SIZE, aad,
                                       AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, AES_BLOCK_SIZE, aad,
                                       AES_BLOCK_SIZE), 0);
    AssertIntEQ(XMEMCMP(plain, in, AES_BLOCK_SIZE), 0);
    /* Finalize and check tag matches. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    AssertIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* No data to encrypt/decrypt one byte of AAD. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad, 1), 0);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad, 1), 0);
    /* Finalize and check tag matches. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(XMEMCMP(tag, expTagAAD1, AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    AssertIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encrypt/decrypt one byte and no AAD. */
    AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out, in, 1, NULL, 0), 0);
    AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain, out, 1, NULL, 0), 0);
    AssertIntEQ(XMEMCMP(plain, in, 1), 0);
    /* Finalize and check tag matches. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(XMEMCMP(tag, expTagPlain1, AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Set key and IV through streaming init API. */
    AssertIntEQ(wc_AesGcmInit(aesEnc, key, sizeof(key), iv, AES_IV_SIZE), 0);
    AssertIntEQ(wc_AesGcmInit(aesDec, key, sizeof(key), iv, AES_IV_SIZE), 0);
    /* Encryption AES is one byte at a time */
    for (i = 0; i < (int)sizeof(aad); i++) {
        AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, NULL, NULL, 0, aad + i, 1),
            0);
    }
    for (i = 0; i < (int)sizeof(in); i++) {
        AssertIntEQ(wc_AesGcmEncryptUpdate(aesEnc, out + i, in + i, 1, NULL, 0),
                    0);
    }
    /* Decryption AES is two bytes at a time */
    for (i = 0; i < (int)sizeof(aad); i += 2) {
        AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, NULL, NULL, 0, aad + i, 2),
            0);
    }
    for (i = 0; i < (int)sizeof(aad); i += 2) {
        AssertIntEQ(wc_AesGcmDecryptUpdate(aesDec, plain + i, out + i, 2, NULL,
                                           0), 0);
    }
    AssertIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);
    /* Finalize and check tag matches. */
    AssertIntEQ(wc_AesGcmEncryptFinal(aesEnc, tag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(XMEMCMP(tag, expTag, AES_BLOCK_SIZE), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(aesDec, tag, AES_BLOCK_SIZE), 0);

    /* Check streaming encryption can be decrypted with one shot. */
    AssertIntEQ(wc_AesGcmSetKey(aesDec, key, sizeof(key)), 0);
    AssertIntEQ(wc_AesGcmDecrypt(aesDec, plain, out, sizeof(in), iv,
        AES_IV_SIZE, tag, AES_BLOCK_SIZE, aad, sizeof(aad)), 0);
    AssertIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);

    wc_AesFree(aesEnc);
    wc_AesFree(aesDec);
    wc_FreeRng(rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_AesGcmStream */


/*
 * unit test for wc_Des3_SetIV()
 */
static int test_wc_Des3_SetIV(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DES3
    Des3 des;
    int  ret = 0;
    const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };

    const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    ret = wc_Des3Init(&des, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    /* DES_ENCRYPTION or DES_DECRYPTION */
    ret = wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION);

    if (ret == 0) {
        if (XMEMCMP(iv, des.reg, DES_BLOCK_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#ifndef HAVE_FIPS /* no sanity checks with FIPS wrapper */
    /* Test explicitly wc_Des3_SetIV()  */
    if (ret == 0) {
        ret = wc_Des3_SetIV(NULL, iv);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_SetIV(&des, NULL);
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    wc_Des3Free(&des);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Des3_SetIV */

/*
 * unit test for wc_Des3_SetKey()
 */
static int test_wc_Des3_SetKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DES3
    Des3 des;
    int  ret = 0;
    const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };

    const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    ret = wc_Des3Init(&des, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    /* DES_ENCRYPTION or DES_DECRYPTION */
    ret = wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION);
    if (ret == 0) {
        if (XMEMCMP(iv, des.reg, DES_BLOCK_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Des3_SetKey(NULL, key, iv, DES_ENCRYPTION);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_SetKey(&des, NULL, iv, DES_ENCRYPTION);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_SetKey(&des, key, iv, -1);
        }
        if (ret == BAD_FUNC_ARG) {
            /* Default case. Should return 0. */
            ret = wc_Des3_SetKey(&des, key, NULL, DES_ENCRYPTION);
        }
    } /* END if ret != 0 */

    wc_Des3Free(&des);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Des3_SetKey */


/*
 * Test function for wc_Des3_CbcEncrypt and wc_Des3_CbcDecrypt
 */
static int test_wc_Des3_CbcEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DES3
    Des3 des;
    int ret = 0;
    byte cipher[24];
    byte plain[24];

    const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };

    const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    ret = wc_Des3Init(&des, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION);

    if (ret == 0) {
        ret = wc_Des3_CbcEncrypt(&des, cipher, vector, 24);

        if (ret == 0) {
            ret = wc_Des3_SetKey(&des, key, iv, DES_DECRYPTION);
        }
        if (ret == 0) {
            ret = wc_Des3_CbcDecrypt(&des, plain, cipher, 24);
        }
    }

    if (ret == 0) {
        if (XMEMCMP(plain, vector, 24) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_Des3_CbcEncrypt(NULL, cipher, vector, 24);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcEncrypt(&des, NULL, vector, 24);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcEncrypt(&des, cipher, NULL, sizeof(vector));
        }
        if (ret != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    if (ret == 0) {
        ret = wc_Des3_CbcDecrypt(NULL, plain, cipher, 24);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcDecrypt(&des, NULL, cipher, 24);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcDecrypt(&des, plain, NULL, 24);
        }
        if (ret != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    wc_Des3Free(&des);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END wc_Des3_CbcEncrypt */

/*
 *  Unit test for wc_Des3_CbcEncryptWithKey and wc_Des3_CbcDecryptWithKey
 */
static int test_wc_Des3_CbcEncryptDecryptWithKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DES3
    int ret = 0;
    word32 vectorSz, cipherSz;
    byte cipher[24];
    byte plain[24];

    byte vector[] = /* Now is the time for all w/o trailing 0 */
    {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };

    byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };


    vectorSz = sizeof(byte) * 24;
    cipherSz = sizeof(byte) * 24;

    ret = wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz, key, iv);
    if (ret == 0) {
        ret = wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, key, iv);
        if (ret == 0) {
            if (XMEMCMP(plain, vector, 24) !=  0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* pass in bad args. */
    if (ret == 0) {
        ret = wc_Des3_CbcEncryptWithKey(NULL, vector, vectorSz, key, iv);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcEncryptWithKey(cipher, NULL, vectorSz, key, iv);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz, NULL, iv);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcEncryptWithKey(cipher, vector, vectorSz,
                                                                    key, NULL);
        }
        else {
            /* Return code catch. */
            ret = WOLFSSL_FAILURE;
        }
    }

    if (ret == 0) {
        ret = wc_Des3_CbcDecryptWithKey(NULL, cipher, cipherSz, key, iv);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcDecryptWithKey(plain, NULL, cipherSz, key, iv);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, NULL, iv);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Des3_CbcDecryptWithKey(plain, cipher, cipherSz, key, NULL);
        }
        else {
            ret = WOLFSSL_FAILURE;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Des3_CbcEncryptDecryptWithKey */
/*
 *  Unit test for wc_Des3_EcbEncrypt
 */
static int test_wc_Des3_EcbEncrypt(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DES3) && defined(WOLFSSL_DES_ECB)
    Des3    des;
    int     ret = 0;
    byte    cipher[24];
    word32  cipherSz = sizeof(cipher);

    const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };

    const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81
    };

    const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    ret = wc_Des3Init(&des, NULL, INVALID_DEVID);
    if (ret != 0) {
        return ret;
    }
    if (ret == 0 ) {
        ret = wc_Des3_SetKey(&des, key, iv, DES_ENCRYPTION);
    }
    /* Bad Cases */
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(NULL, cipher, vector, cipherSz);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(&des, 0, vector, cipherSz);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(&des, cipher, NULL, cipherSz);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(&des, cipher, vector, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(NULL, 0, NULL, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    /* Good Cases */
    if (ret == 0) {
        ret = wc_Des3_EcbEncrypt(&des, cipher, vector, cipherSz);
    }
    wc_Des3Free(&des);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Des3_EcbEncrypt */

/*
 * Testing wc_Chacha_SetKey() and wc_Chacha_SetIV()
 */
static int test_wc_Chacha_SetKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CHACHA
    ChaCha      ctx;
    const byte  key[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    byte        cipher[128];
    int         ret = 0;

    ret = wc_Chacha_SetKey(&ctx, key, (word32)(sizeof(key)/sizeof(byte)));
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Chacha_SetKey(NULL, key, (word32)(sizeof(key)/sizeof(byte)));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Chacha_SetKey(&ctx, key, 18);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_Chacha_SetIV(&ctx, cipher, 0);
    }
    if (ret == 0) {
        /* Test bad args. */
        ret = wc_Chacha_SetIV(NULL, cipher, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FAILURE;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Chacha_SetKey */

/*
 * unit test for wc_Poly1305SetKey()
 */
static int test_wc_Poly1305SetKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_POLY1305
    Poly1305      ctx;
    const byte  key[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    int ret = 0;

    ret = wc_Poly1305SetKey(&ctx, key, (word32)(sizeof(key)/sizeof(byte)));
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Poly1305SetKey(NULL, key, (word32)(sizeof(key)/sizeof(byte)));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Poly1305SetKey(&ctx, NULL, (word32)(sizeof(key)/sizeof(byte)));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Poly1305SetKey(&ctx, key, 18);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Poly1305_SetKey() */

/*
 * Testing wc_Chacha_Process()
 */
static int test_wc_Chacha_Process(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CHACHA
    ChaCha      enc, dec;
    byte        cipher[128];
    byte        plain[128];
    const byte  key[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    const char* input = "Everybody gets Friday off.";
    word32      keySz = sizeof(key)/sizeof(byte);
    unsigned long int inlen = XSTRLEN(input);
    int         ret = 0;

    /*Initialize stack varialbes.*/
    XMEMSET(cipher, 0, 128);
    XMEMSET(plain, 0, 128);

    ret = wc_Chacha_SetKey(&enc, key, keySz);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_SetKey(&dec, key, keySz);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_SetIV(&enc, cipher, 0);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_SetIV(&dec, cipher, 0);
    AssertIntEQ(ret, 0);

    ret = wc_Chacha_Process(&enc, cipher, (byte*)input, (word32)inlen);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_Process(&dec, plain, cipher, (word32)inlen);
    AssertIntEQ(ret, 0);
    ret = XMEMCMP(input, plain, (int)inlen);
    AssertIntEQ(ret, 0);

#if !defined(USE_INTEL_CHACHA_SPEEDUP) && !defined(WOLFSSL_ARMASM)
    /* test checking and using leftovers, currently just in C code */
    ret = wc_Chacha_SetIV(&enc, cipher, 0);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_SetIV(&dec, cipher, 0);
    AssertIntEQ(ret, 0);

    ret = wc_Chacha_Process(&enc, cipher, (byte*)input, (word32)inlen - 2);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_Process(&enc, cipher + (inlen - 2),
            (byte*)input + (inlen - 2), 2);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_Process(&dec, plain, (byte*)cipher, (word32)inlen - 2);
    AssertIntEQ(ret, 0);
    ret = wc_Chacha_Process(&dec, cipher + (inlen - 2),
            (byte*)input + (inlen - 2), 2);
    AssertIntEQ(ret, 0);
    ret = XMEMCMP(input, plain, (int)inlen);
    AssertIntEQ(ret, 0);

    /* check edge cases with counter increment */
    {
        /* expected results collected from wolfSSL 4.3.0 encrypted in one call*/
        const byte expected[] = {
            0x54,0xB1,0xE2,0xD4,0xA2,0x4D,0x52,0x5F,
            0x42,0x04,0x89,0x7C,0x6E,0x2D,0xFC,0x2D,
            0x10,0x25,0xB6,0x92,0x71,0xD5,0xC3,0x20,
            0xE3,0x0E,0xEC,0xF4,0xD8,0x10,0x70,0x29,
            0x2D,0x4C,0x2A,0x56,0x21,0xE1,0xC7,0x37,
            0x0B,0x86,0xF5,0x02,0x8C,0xB8,0xB8,0x38,
            0x41,0xFD,0xDF,0xD9,0xC3,0xE6,0xC8,0x88,
            0x06,0x82,0xD4,0x80,0x6A,0x50,0x69,0xD5,
            0xB9,0xB0,0x2F,0x44,0x36,0x5D,0xDA,0x5E,
            0xDE,0xF6,0xF5,0xFC,0x44,0xDC,0x07,0x51,
            0xA7,0x32,0x42,0xDB,0xCC,0xBD,0xE2,0xE5,
            0x0B,0xB1,0x14,0xFF,0x12,0x80,0x16,0x43,
            0xE7,0x40,0xD5,0xEA,0xC7,0x3F,0x69,0x07,
            0x64,0xD4,0x86,0x6C,0xE2,0x1F,0x8F,0x6E,
            0x35,0x41,0xE7,0xD3,0xB5,0x5D,0xD6,0xD4,
            0x9F,0x00,0xA9,0xAE,0x3D,0x28,0xA5,0x37,
            0x80,0x3D,0x11,0x25,0xE2,0xB6,0x99,0xD9,
            0x9B,0x98,0xE9,0x37,0xB9,0xF8,0xA0,0x04,
            0xDF,0x13,0x49,0x3F,0x19,0x6A,0x45,0x06,
            0x21,0xB4,0xC7,0x3B,0x49,0x45,0xB4,0xC8,
            0x03,0x5B,0x43,0x89,0xBD,0xB3,0x96,0x4B,
            0x17,0x6F,0x85,0xC6,0xCF,0xA6,0x05,0x35,
            0x1E,0x25,0x03,0xBB,0x55,0x0A,0xD5,0x54,
            0x41,0xEA,0xEB,0x50,0x40,0x1B,0x43,0x19,
            0x59,0x1B,0x0E,0x12,0x3E,0xA2,0x71,0xC3,
            0x1A,0xA7,0x11,0x50,0x43,0x9D,0x56,0x3B,
            0x63,0x2F,0x63,0xF1,0x8D,0xAE,0xF3,0x23,
            0xFA,0x1E,0xD8,0x6A,0xE1,0xB2,0x4B,0xF3,
            0xB9,0x13,0x7A,0x72,0x2B,0x6D,0xCC,0x41,
            0x1C,0x69,0x7C,0xCD,0x43,0x6F,0xE4,0xE2,
            0x38,0x99,0xFB,0xC3,0x38,0x92,0x62,0x35,
            0xC0,0x1D,0x60,0xE4,0x4B,0xDD,0x0C,0x14
        };
        const byte iv2[] = {
            0x9D,0xED,0xE7,0x0F,0xEC,0x81,0x51,0xD9,
            0x77,0x39,0x71,0xA6,0x21,0xDF,0xB8,0x93
        };
        byte input2[256];
        int i;

        for (i = 0; i < 256; i++)
            input2[i] = i;

        ret = wc_Chacha_SetIV(&enc, iv2, 0);
        AssertIntEQ(ret, 0);

        ret = wc_Chacha_Process(&enc, cipher, input2, 64);
        AssertIntEQ(ret, 0);
        AssertIntEQ(XMEMCMP(expected, cipher, 64), 0);

        ret = wc_Chacha_Process(&enc, cipher, input2 + 64, 128);
        AssertIntEQ(ret, 0);
        AssertIntEQ(XMEMCMP(expected + 64, cipher, 128), 0);

        /* partial */
        ret = wc_Chacha_Process(&enc, cipher, input2 + 192, 32);
        AssertIntEQ(ret, 0);
        AssertIntEQ(XMEMCMP(expected + 192, cipher, 32), 0);

        ret = wc_Chacha_Process(&enc, cipher, input2 + 224, 32);
        AssertIntEQ(ret, 0);
        AssertIntEQ(XMEMCMP(expected + 224, cipher, 32), 0);
    }
#endif

    /* Test bad args. */
    ret = wc_Chacha_Process(NULL, cipher, (byte*)input, (word32)inlen);
    AssertIntEQ(ret, BAD_FUNC_ARG);
    if (ret == BAD_FUNC_ARG) {
        ret = 0;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Chacha_Process */

/*
 * Testing wc_ChaCha20Poly1305_Encrypt() and wc_ChaCha20Poly1305_Decrypt()
 */
static int test_wc_ChaCha20Poly1305_aead(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    const byte  key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    const byte  plaintext[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61,
        0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c,
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73,
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39,
        0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63,
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66,
        0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f,
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
        0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75,
        0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73,
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
        0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69,
        0x74, 0x2e
    };

    const byte  iv[] = {
        0x07, 0x00, 0x00, 0x00, 0x40, 0x41, 0x42, 0x43,
        0x44, 0x45, 0x46, 0x47
    };

    const byte  aad[] = { /* additional data */
        0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
        0xc4, 0xc5, 0xc6, 0xc7
    };
    const byte  cipher[] = { /* expected output from operation */
        0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
        0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
        0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
        0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
        0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
        0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
        0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
        0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
        0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
        0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
        0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
        0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
        0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
        0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
        0x61, 0x16
    };
    const byte  authTag[] = { /* expected output from operation */
        0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
        0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91
    };
    byte        generatedCiphertext[272];
    byte        generatedPlaintext[272];
    byte        generatedAuthTag[CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE];
    int         ret = 0;

    /* Initialize stack variables. */
    XMEMSET(generatedCiphertext, 0, 272);
    XMEMSET(generatedPlaintext, 0, 272);

    /* Test Encrypt */
    ret = wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad), plaintext,
                sizeof(plaintext), generatedCiphertext, generatedAuthTag);
    AssertIntEQ(ret, 0);
    ret = XMEMCMP(generatedCiphertext, cipher, sizeof(cipher)/sizeof(byte));
    AssertIntEQ(ret, 0);

    /* Test bad args. */
    ret = wc_ChaCha20Poly1305_Encrypt(NULL, iv, aad, sizeof(aad), plaintext,
                    sizeof(plaintext), generatedCiphertext, generatedAuthTag);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Encrypt(key, NULL, aad, sizeof(aad),
                                        plaintext, sizeof(plaintext),
                                        generatedCiphertext, generatedAuthTag);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad), NULL,
                    sizeof(plaintext), generatedCiphertext, generatedAuthTag);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
                NULL, sizeof(plaintext), generatedCiphertext, generatedAuthTag);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
                    plaintext, sizeof(plaintext), NULL, generatedAuthTag);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Encrypt(key, iv, aad, sizeof(aad),
                    plaintext, sizeof(plaintext), generatedCiphertext, NULL);
    if (ret == BAD_FUNC_ARG) {
        ret = 0;
        (void)ret; /* suppress never read */
    }

    ret = wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
                            sizeof(cipher), authTag, generatedPlaintext);
    AssertIntEQ(ret, 0);
    ret = XMEMCMP(generatedPlaintext, plaintext,
                                        sizeof(plaintext)/sizeof(byte));
    AssertIntEQ(ret, 0);

    /* Test bad args. */
    ret = wc_ChaCha20Poly1305_Decrypt(NULL, iv, aad, sizeof(aad), cipher,
                                sizeof(cipher), authTag, generatedPlaintext);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Decrypt(key, NULL, aad, sizeof(aad),
                        cipher, sizeof(cipher), authTag, generatedPlaintext);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), NULL,
                                sizeof(cipher), authTag, generatedPlaintext);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
                                    sizeof(cipher), NULL, generatedPlaintext);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), cipher,
                                                sizeof(cipher), authTag, NULL);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    ret = wc_ChaCha20Poly1305_Decrypt(key, iv, aad, sizeof(aad), NULL,
                                   sizeof(cipher), authTag, generatedPlaintext);
    AssertIntEQ(ret,  BAD_FUNC_ARG);
    if (ret == BAD_FUNC_ARG) {
        ret = 0;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test-wc_ChaCha20Poly1305_EncryptDecrypt */


/*
 * Testing function for wc_Rc2SetKey().
 */
static int test_wc_Rc2SetKey(void)
{
    int res = TEST_SKIPPED;
#ifdef WC_RC2
    Rc2     rc2;
    byte    key40[] = { 0x01, 0x02, 0x03, 0x04, 0x05 };
    byte    iv[]    = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
    int     ret = 0;

    /* valid key and IV */
    ret = wc_Rc2SetKey(&rc2, key40, (word32) sizeof(key40) / sizeof(byte),
                       iv, 40);
    if (ret == 0) {
        /* valid key, no IV */
        ret = wc_Rc2SetKey(&rc2, key40, (word32) sizeof(key40) / sizeof(byte),
                           NULL, 40);
    }

    /* bad arguments  */
    if (ret == 0) {
        /* null Rc2 struct */
        ret = wc_Rc2SetKey(NULL, key40, (word32) sizeof(key40) / sizeof(byte),
                           iv, 40);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null key */
        ret = wc_Rc2SetKey(&rc2, NULL, (word32) sizeof(key40) / sizeof(byte),
                           iv, 40);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* key size == 0 */
        ret = wc_Rc2SetKey(&rc2, key40, 0, iv, 40);
        if (ret == WC_KEY_SIZE_E) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* key size > 128 */
        ret = wc_Rc2SetKey(&rc2, key40, 129, iv, 40);
        if (ret == WC_KEY_SIZE_E) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* effective bits == 0 */
        ret = wc_Rc2SetKey(&rc2, key40, (word32)sizeof(key40) / sizeof(byte),
                           iv, 0);
        if (ret == WC_KEY_SIZE_E) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* effective bits > 1024 */
        ret = wc_Rc2SetKey(&rc2, key40, (word32)sizeof(key40) / sizeof(byte),
                           iv, 1025);
        if (ret == WC_KEY_SIZE_E) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Rc2SetKey */

/*
 * Testing function for wc_Rc2SetIV().
 */
static int test_wc_Rc2SetIV(void)
{
    int res = TEST_SKIPPED;
#ifdef WC_RC2
    Rc2     rc2;
    byte    iv[]    = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
    int     ret = 0;

    /* valid IV */
    ret = wc_Rc2SetIV(&rc2, iv);
    if (ret == 0) {
        /* valid NULL IV */
        ret = wc_Rc2SetIV(&rc2, NULL);
    }

    /* bad arguments */
    if (ret == 0) {
        ret = wc_Rc2SetIV(NULL, iv);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Rc2SetKey */

/*
 * Testing function for wc_Rc2EcbEncrypt().
 */
static int test_wc_Rc2EcbEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#ifdef WC_RC2
    Rc2     rc2;
    int     ret = 0;
    int effectiveKeyBits = 63;

    byte cipher[RC2_BLOCK_SIZE];
    byte plain[RC2_BLOCK_SIZE];

    byte key[]    = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    byte input[]  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    byte output[] = { 0xeb, 0xb7, 0x73, 0xf9, 0x93, 0x27, 0x8e, 0xff };

    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    ret = wc_Rc2SetKey(&rc2, key, (word32) sizeof(key) / sizeof(byte),
                       NULL, effectiveKeyBits);
    if (ret == 0) {
        ret = wc_Rc2EcbEncrypt(&rc2, cipher, input, RC2_BLOCK_SIZE);
        if (ret != 0 || XMEMCMP(cipher, output, RC2_BLOCK_SIZE) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }

        if (ret == 0) {
            ret = wc_Rc2EcbDecrypt(&rc2, plain, cipher, RC2_BLOCK_SIZE);
            if (ret != 0 || XMEMCMP(plain, input, RC2_BLOCK_SIZE) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Rc2EcbEncrypt bad arguments */
    if (ret == 0) {
        /* null Rc2 struct */
        ret = wc_Rc2EcbEncrypt(NULL, cipher, input, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null out buffer */
        ret = wc_Rc2EcbEncrypt(&rc2, NULL, input, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null input buffer */
        ret = wc_Rc2EcbEncrypt(&rc2, cipher, NULL, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* output buffer sz != RC2_BLOCK_SIZE (8) */
        ret = wc_Rc2EcbEncrypt(&rc2, cipher, input, 7);
        if (ret == BUFFER_E) {
            ret = 0;
        }
    }

    /* Rc2EcbDecrypt bad arguments */
    if (ret == 0) {
        /* null Rc2 struct */
        ret = wc_Rc2EcbDecrypt(NULL, plain, output, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null out buffer */
        ret = wc_Rc2EcbDecrypt(&rc2, NULL, output, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null input buffer */
        ret = wc_Rc2EcbDecrypt(&rc2, plain, NULL, RC2_BLOCK_SIZE);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* output buffer sz != RC2_BLOCK_SIZE (8) */
        ret = wc_Rc2EcbDecrypt(&rc2, plain, output, 7);
        if (ret == BUFFER_E) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Rc2SetKey */

/*
 * Testing function for wc_Rc2CbcEncrypt().
 */
static int test_wc_Rc2CbcEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#ifdef WC_RC2
    Rc2     rc2;
    int     ret = 0;
    int effectiveKeyBits = 63;

    byte cipher[RC2_BLOCK_SIZE*2];
    byte plain[RC2_BLOCK_SIZE*2];

    /* vector taken from test.c */
    byte key[]    = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte iv[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte input[]  = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    byte output[] = {
        0xeb, 0xb7, 0x73, 0xf9, 0x93, 0x27, 0x8e, 0xff,
        0xf0, 0x51, 0x77, 0x8b, 0x65, 0xdb, 0x13, 0x57
    };

    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    ret = wc_Rc2SetKey(&rc2, key, (word32) sizeof(key) / sizeof(byte),
                       iv, effectiveKeyBits);
    if (ret == 0) {
        ret = wc_Rc2CbcEncrypt(&rc2, cipher, input, sizeof(input));
        if (ret != 0 || XMEMCMP(cipher, output, sizeof(output)) != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            /* reset IV for decrypt */
            ret = wc_Rc2SetIV(&rc2, iv);
        }

        if (ret == 0) {
            ret = wc_Rc2CbcDecrypt(&rc2, plain, cipher, sizeof(cipher));
            if (ret != 0 || XMEMCMP(plain, input, sizeof(input)) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Rc2CbcEncrypt bad arguments */
    if (ret == 0) {
        /* null Rc2 struct */
        ret = wc_Rc2CbcEncrypt(NULL, cipher, input, sizeof(input));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null out buffer */
        ret = wc_Rc2CbcEncrypt(&rc2, NULL, input, sizeof(input));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null input buffer */
        ret = wc_Rc2CbcEncrypt(&rc2, cipher, NULL, sizeof(input));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    /* Rc2CbcDecrypt bad arguments */
    if (ret == 0) {
        /* in size is 0 */
        ret = wc_Rc2CbcDecrypt(&rc2, plain, output, 0);
        if (ret != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        /* null Rc2 struct */
        ret = wc_Rc2CbcDecrypt(NULL, plain, output, sizeof(output));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null out buffer */
        ret = wc_Rc2CbcDecrypt(&rc2, NULL, output, sizeof(output));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        /* null input buffer */
        ret = wc_Rc2CbcDecrypt(&rc2, plain, NULL, sizeof(output));
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_Rc2SetKey */


/*
 * Testing function for wc_AesSetIV
 */
static int test_wc_AesSetIV(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(WOLFSSL_AES_128)
    Aes     aes;
    int     ret = 0;
    byte    key16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte    iv1[]    = "1234567890abcdef";
    byte    iv2[]    = "0987654321fedcba";

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_AesSetKey(&aes, key16, (word32) sizeof(key16) / sizeof(byte),
                                                     iv1, AES_ENCRYPTION);
    if (ret == 0) {
        ret = wc_AesSetIV(&aes, iv2);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_AesSetIV(NULL, iv1);
        if (ret == BAD_FUNC_ARG) {
            /* NULL iv should return 0. */
            ret = wc_AesSetIV(&aes, NULL);
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* test_wc_AesSetIV */


/*
 * Testing function for wc_AesSetKey().
 */
static int test_wc_AesSetKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_AES
    Aes     aes;
    int     ret = 0;
    byte    key16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#ifdef WOLFSSL_AES_192
    byte    key24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte    badKey16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };
    byte    iv[]    = "1234567890abcdef";

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

#ifdef WOLFSSL_AES_128
    ret = wc_AesSetKey(&aes, key16, (word32) sizeof(key16) / sizeof(byte),
                                                        iv, AES_ENCRYPTION);
#endif
#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        ret = wc_AesSetKey (&aes, key24, (word32) sizeof(key24) / sizeof(byte),
                                                           iv, AES_ENCRYPTION);
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = wc_AesSetKey (&aes, key32, (word32) sizeof(key32) / sizeof(byte),
                                                           iv, AES_ENCRYPTION);
    }
#endif

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_AesSetKey (NULL, key16, (word32) sizeof(key16) / sizeof(byte),
                                                           iv, AES_ENCRYPTION);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_AesSetKey(&aes, badKey16,
                                    (word32) sizeof(badKey16) / sizeof(byte),
                                                         iv, AES_ENCRYPTION);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_AesSetKey */



/*
 * test function for wc_AesCbcEncrypt(), wc_AesCbcDecrypt(),
 * and wc_AesCbcDecryptWithKey()
 */
static int test_wc_AesCbcEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(HAVE_AES_DECRYPT)&& \
    defined(WOLFSSL_AES_256)
    Aes     aes;
    int     ret = 0;
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte    vector[] = /* Now is the time for all good men w/o trailing 0 */
    {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20,
        0x67,0x6f,0x6f,0x64,0x20,0x6d,0x65,0x6e
    };
    byte    iv[]    = "1234567890abcdef";
    byte    enc[sizeof(vector)];
    byte    dec[sizeof(vector)];
    int     cbcE    =   WOLFSSL_FATAL_ERROR;
    int     cbcD    =   WOLFSSL_FATAL_ERROR;
    int     cbcDWK  =   WOLFSSL_FATAL_ERROR;
    byte    dec2[sizeof(vector)];

    /* Init stack variables. */
    XMEMSET(enc, 0, sizeof(enc));
    XMEMSET(dec, 0, sizeof(vector));
    XMEMSET(dec2, 0, sizeof(vector));

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_AesSetKey(&aes, key32, AES_BLOCK_SIZE * 2, iv, AES_ENCRYPTION);
    if (ret == 0) {
        ret = wc_AesCbcEncrypt(&aes, enc, vector, sizeof(vector));
        if (ret == 0) {
            /* Re init for decrypt and set flag. */
            cbcE = 0;
            wc_AesFree(&aes);
            ret = wc_AesSetKey(&aes, key32, AES_BLOCK_SIZE * 2,
                                                    iv, AES_DECRYPTION);
        }
        if (ret == 0) {
            ret = wc_AesCbcDecrypt(&aes, dec, enc, sizeof(vector));
            if (ret != 0 || XMEMCMP(vector, dec, sizeof(vector)) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
            else {
                /* Set flag. */
                cbcD = 0;
            }
        }
    }
    /* If encrypt succeeds but cbc decrypt fails, we can still test. */
    if (ret == 0 || cbcE == 0) {
        ret = wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE,
                                     key32, sizeof(key32)/sizeof(byte), iv);
        if (ret == 0 || XMEMCMP(vector, dec2, AES_BLOCK_SIZE) == 0) {
            cbcDWK = 0;
        }
    }

    /* Pass in bad args */
    if (cbcE == 0) {
        cbcE = wc_AesCbcEncrypt(NULL, enc, vector, sizeof(vector));
        if (cbcE == BAD_FUNC_ARG) {
            cbcE = wc_AesCbcEncrypt(&aes, NULL, vector, sizeof(vector));
        }
        if (cbcE == BAD_FUNC_ARG) {
            cbcE = wc_AesCbcEncrypt(&aes, enc, NULL, sizeof(vector));
        }
        if (cbcE == BAD_FUNC_ARG) {
            cbcE = 0;
        }
        else {
            cbcE = WOLFSSL_FATAL_ERROR;
        }
#ifdef WOLFSSL_AES_CBC_LENGTH_CHECKS
        if (cbcE == 0) {
            cbcE = wc_AesCbcEncrypt(&aes, enc, vector, sizeof(vector) - 1);
        }
        if (cbcE == BAD_LENGTH_E) {
            cbcE = 0;
        }
        else {
            cbcE = WOLFSSL_FATAL_ERROR;
        }
#endif
    }
    if (cbcE == 0) {
    #if defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
        (HAVE_FIPS_VERSION == 2) && defined(WOLFSSL_AESNI)
        fprintf(stderr, "Zero length inputs not supported with AESNI in FIPS "
                        "mode (v2), skip test");
    #else
        /* Test passing in size of 0  */
        XMEMSET(enc, 0, sizeof(enc));
        cbcE = wc_AesCbcEncrypt(&aes, enc, vector, 0);
        if (cbcE == 0) {
            /* Check enc was not modified */
            int i;
            for (i = 0; i < (int)sizeof(enc); i++)
                cbcE |= enc[i];
        }
    #endif
    }
    if (cbcE != 0) {
        wc_AesFree(&aes);
        return TEST_FAIL;
    }

    if (cbcD == 0) {
        cbcD = wc_AesCbcDecrypt(NULL, dec, enc, AES_BLOCK_SIZE);
        if (cbcD == BAD_FUNC_ARG) {
            cbcD = wc_AesCbcDecrypt(&aes, NULL, enc, AES_BLOCK_SIZE);
        }
        if (cbcD == BAD_FUNC_ARG) {
            cbcD = wc_AesCbcDecrypt(&aes, dec, NULL, AES_BLOCK_SIZE);
        }
        if (cbcD == BAD_FUNC_ARG) {
            cbcD = wc_AesCbcDecrypt(&aes, dec, enc, AES_BLOCK_SIZE * 2 - 1);
        }
#ifdef WOLFSSL_AES_CBC_LENGTH_CHECKS
        if (cbcD == BAD_LENGTH_E) {
            cbcD = 0;
        }
        else {
            cbcD = WOLFSSL_FATAL_ERROR;
        }
#else
        if (cbcD == BAD_FUNC_ARG) {
            cbcD = 0;
        }
        else {
            cbcD = WOLFSSL_FATAL_ERROR;
        }
#endif
    }
    if (cbcD == 0) {
        /* Test passing in size of 0  */
        XMEMSET(dec, 0, sizeof(dec));
        cbcD = wc_AesCbcDecrypt(&aes, dec, enc, 0);
        if (cbcD == 0) {
            /* Check dec was not modified */
            int i;
            for (i = 0; i < (int)sizeof(dec); i++)
                cbcD |= dec[i];
        }
    }
    if (cbcD != 0) {
        wc_AesFree(&aes);
        return TEST_FAIL;
    }

    if (cbcDWK == 0) {
        cbcDWK = wc_AesCbcDecryptWithKey(NULL, enc, AES_BLOCK_SIZE,
                                     key32, sizeof(key32)/sizeof(byte), iv);
        if (cbcDWK == BAD_FUNC_ARG) {
            cbcDWK = wc_AesCbcDecryptWithKey(dec2, NULL, AES_BLOCK_SIZE,
                                     key32, sizeof(key32)/sizeof(byte), iv);
        }
        if (cbcDWK == BAD_FUNC_ARG) {
            cbcDWK = wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE,
                                     NULL, sizeof(key32)/sizeof(byte), iv);
        }
        if (cbcDWK == BAD_FUNC_ARG) {
            cbcDWK = wc_AesCbcDecryptWithKey(dec2, enc, AES_BLOCK_SIZE,
                                     key32, sizeof(key32)/sizeof(byte), NULL);
        }
        if (cbcDWK == BAD_FUNC_ARG) {
            cbcDWK = 0;
        }
        else {
            cbcDWK = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(cbcDWK == 0);
#endif
    return res;
} /* END test_wc_AesCbcEncryptDecrypt */

/*
 * Testing wc_AesCtrEncrypt and wc_AesCtrDecrypt
 */
static int test_wc_AesCtrEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
    Aes     aesEnc, aesDec;
    int     ret = 0;
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte    vector[] = /* Now is the time for all w/o trailing 0 */
    {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte    iv[]    = "1234567890abcdef";
    byte    enc[AES_BLOCK_SIZE * 2];
    byte    dec[AES_BLOCK_SIZE * 2];

    /* Init stack variables. */
    XMEMSET(enc, 0, AES_BLOCK_SIZE * 2);
    XMEMSET(dec, 0, AES_BLOCK_SIZE * 2);

    ret = wc_AesInit(&aesEnc, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;
    ret = wc_AesInit(&aesDec, NULL, INVALID_DEVID);
    if (ret != 0) {
        wc_AesFree(&aesEnc);
        return ret;
    }

    ret = wc_AesSetKey(&aesEnc, key32, AES_BLOCK_SIZE * 2,
                                                    iv, AES_ENCRYPTION);
    if (ret == 0) {
        ret = wc_AesCtrEncrypt(&aesEnc, enc, vector,
                                            sizeof(vector)/sizeof(byte));
        if (ret == 0) {
            /* Decrypt with wc_AesCtrEncrypt() */
            ret = wc_AesSetKey(&aesDec, key32, AES_BLOCK_SIZE * 2,
                                                    iv, AES_ENCRYPTION);
        }
        if (ret == 0) {
            ret = wc_AesCtrEncrypt(&aesDec, dec, enc, sizeof(enc)/sizeof(byte));
            if (ret != 0 || XMEMCMP(vector, dec, sizeof(vector))) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_AesCtrEncrypt(NULL, dec, enc, sizeof(enc)/sizeof(byte));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCtrEncrypt(&aesDec, NULL, enc, sizeof(enc)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCtrEncrypt(&aesDec, dec, NULL, sizeof(enc)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aesEnc);
    wc_AesFree(&aesDec);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_AesCtrEncryptDecrypt */

/*
 * test function for wc_AesGcmSetKey()
 */
static int test_wc_AesGcmSetKey(void)
{
    int res = TEST_SKIPPED;
#if  !defined(NO_AES) && defined(HAVE_AESGCM)
    Aes     aes;
    int     ret = 0;
#ifdef WOLFSSL_AES_128
    byte    key16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
#ifdef WOLFSSL_AES_192
    byte    key24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte    badKey16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };
    byte    badKey24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36
    };
    byte   badKey32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x37, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65
    };

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

#ifdef WOLFSSL_AES_128
    ret = wc_AesGcmSetKey(&aes, key16, sizeof(key16)/sizeof(byte));
#endif
#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        ret = wc_AesGcmSetKey(&aes, key24, sizeof(key24)/sizeof(byte));
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = wc_AesGcmSetKey(&aes, key32, sizeof(key32)/sizeof(byte));
    }
#endif

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_AesGcmSetKey(&aes, badKey16, sizeof(badKey16)/sizeof(byte));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesGcmSetKey(&aes, badKey24, sizeof(badKey24)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesGcmSetKey(&aes, badKey32, sizeof(badKey32)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_AesGcmSetKey */

/*
 * test function for wc_AesGcmEncrypt and wc_AesGcmDecrypt
 */
static int test_wc_AesGcmEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
    /* WOLFSSL_AFALG requires 12 byte IV */
#if !defined(NO_AES) && defined(HAVE_AESGCM) && defined(WOLFSSL_AES_256) && \
    !defined(WOLFSSL_AFALG) && !defined(WOLFSSL_DEVCRYPTO_AES)

    Aes     aes;
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    byte    vector[] = /* Now is the time for all w/o trailing 0 */
    {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    const byte a[] =
    {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };
    byte    iv[]   = "1234567890a";
    byte    longIV[]  = "1234567890abcdefghij";
    byte    enc[sizeof(vector)];
    byte    resultT[AES_BLOCK_SIZE];
    byte    dec[sizeof(vector)];
    int     gcmD     =   WOLFSSL_FATAL_ERROR;
    int     gcmE     =   WOLFSSL_FATAL_ERROR;
    int     ret = 0;

    /* Init stack variables. */
    XMEMSET(enc, 0, sizeof(vector));
    XMEMSET(dec, 0, sizeof(vector));
    XMEMSET(resultT, 0, AES_BLOCK_SIZE);

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_AesGcmSetKey(&aes, key32, sizeof(key32)/sizeof(byte));
    if (ret == 0) {
        gcmE = wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector),
                                        iv, sizeof(iv)/sizeof(byte), resultT,
                                        sizeof(resultT), a, sizeof(a));
    }
    if (gcmE == 0) { /* If encrypt fails, no decrypt. */
        gcmD = wc_AesGcmDecrypt(&aes, dec, enc, sizeof(vector),
                                        iv, sizeof(iv)/sizeof(byte), resultT,
                                        sizeof(resultT), a, sizeof(a));
        if (gcmD == 0 && (XMEMCMP(vector, dec, sizeof(vector)) != 0)) {
            gcmD = WOLFSSL_FATAL_ERROR;
        }
    }
    /*Test bad args for wc_AesGcmEncrypt and wc_AesGcmDecrypt */
    if (gcmE == 0) {
        gcmE = wc_AesGcmEncrypt(NULL, enc, vector, sizeof(vector),
                        iv, sizeof(iv)/sizeof(byte), resultT, sizeof(resultT),
                        a, sizeof(a));
        if (gcmE == BAD_FUNC_ARG) {
            gcmE = wc_AesGcmEncrypt(&aes, enc, vector,
                    sizeof(vector), iv, sizeof(iv)/sizeof(byte),
                    resultT, sizeof(resultT) + 1, a, sizeof(a));
        }
        if (gcmE == BAD_FUNC_ARG) {
            gcmE = wc_AesGcmEncrypt(&aes, enc, vector,
                    sizeof(vector), iv, sizeof(iv)/sizeof(byte),
                    resultT, sizeof(resultT) - 5, a, sizeof(a));
        }

#if (defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
        (HAVE_FIPS_VERSION == 2)) || defined(HAVE_SELFTEST) || \
        defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
        /* FIPS does not check the lower bound of ivSz */
#else
        if (gcmE == BAD_FUNC_ARG) {
            gcmE = wc_AesGcmEncrypt(&aes, enc, vector,
                    sizeof(vector), iv, 0,
                    resultT, sizeof(resultT), a, sizeof(a));
        }
#endif
        if (gcmE == BAD_FUNC_ARG) {
            gcmE = 0;
        }
        else {
            gcmE = WOLFSSL_FATAL_ERROR;
        }
    }

    /* This case is now considered good. Long IVs are now allowed.
     * Except for the original FIPS release, it still has an upper
     * bound on the IV length. */
#if (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) && \
    !defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
    if (gcmE == 0) {
        gcmE = wc_AesGcmEncrypt(&aes, enc, vector, sizeof(vector), longIV,
                        sizeof(longIV)/sizeof(byte), resultT, sizeof(resultT),
                        a, sizeof(a));
    }
#else
    (void)longIV;
#endif /* Old FIPS */
    /* END wc_AesGcmEncrypt */

    if (gcmE != 0) {
        wc_AesFree(&aes);
        return TEST_FAIL;
    }

    #ifdef HAVE_AES_DECRYPT
        if (gcmD == 0) {
            gcmD = wc_AesGcmDecrypt(NULL, dec, enc, sizeof(enc)/sizeof(byte),
                                   iv, sizeof(iv)/sizeof(byte), resultT,
                                   sizeof(resultT), a, sizeof(a));
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, NULL, enc, sizeof(enc)/sizeof(byte),
                                   iv, sizeof(iv)/sizeof(byte), resultT,
                                   sizeof(resultT), a, sizeof(a));
            }
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, dec, NULL, sizeof(enc)/sizeof(byte),
                                   iv, sizeof(iv)/sizeof(byte), resultT,
                                   sizeof(resultT), a, sizeof(a));
            }
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte),
                                   NULL, sizeof(iv)/sizeof(byte), resultT,
                                   sizeof(resultT), a, sizeof(a));
            }
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte),
                                   iv, sizeof(iv)/sizeof(byte), NULL,
                                   sizeof(resultT), a, sizeof(a));
            }
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte),
                                   iv, sizeof(iv)/sizeof(byte), resultT,
                                   sizeof(resultT) + 1, a, sizeof(a));
            }
        #if ((defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
                (HAVE_FIPS_VERSION == 2)) || defined(HAVE_SELFTEST)) && \
                !defined(WOLFSSL_AES_GCM_FIXED_IV_AAD)
                /* FIPS does not check the lower bound of ivSz */
        #else
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = wc_AesGcmDecrypt(&aes, dec, enc, sizeof(enc)/sizeof(byte),
                                   iv, 0, resultT,
                                   sizeof(resultT), a, sizeof(a));
            }
        #endif
            if (gcmD == BAD_FUNC_ARG) {
                gcmD = 0;
            }
            else {
                gcmD = WOLFSSL_FATAL_ERROR;
            }

            res = TEST_RES_CHECK(gcmD == 0);
        } /* END wc_AesGcmDecrypt */
    #endif /* HAVE_AES_DECRYPT */

    wc_AesFree(&aes);
#endif
    return res;

} /* END test_wc_AesGcmEncryptDecrypt */

/*
 * test function for mixed (one-shot encrpytion + stream decryption) AES GCM
 * using a long IV (older FIPS does NOT support long IVs).  Relates to zd15423
 */
static int test_wc_AesGcmMixedEncDecLongIV(void)
{
    int ret = TEST_SKIPPED;

#if  (!defined(HAVE_FIPS) || \
      (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) && \
     !defined(NO_AES) && defined(HAVE_AESGCM) && defined(WOLFSSL_AESGCM_STREAM)

    const byte key[] = {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };

    const byte in[] = {
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    const byte aad[] = {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };

    Aes     aesEnc, aesDec;
    byte    iv[]  = "1234567890abcdefghij";
    byte    out[sizeof(in)];
    byte    plain[sizeof(in)];
    byte    tag[AES_BLOCK_SIZE];

    XMEMSET(out, 0, sizeof(out));
    XMEMSET(plain, 0, sizeof(plain));
    XMEMSET(tag, 0, sizeof(tag));

    /* Perform one-shot encryption using long IV */
    AssertIntEQ(wc_AesInit(&aesEnc, NULL, INVALID_DEVID), 0);
    AssertIntEQ(wc_AesGcmSetKey(&aesEnc, key, sizeof(key)), 0);
    AssertIntEQ(wc_AesGcmEncrypt(&aesEnc, out, in, sizeof(in), iv, sizeof(iv),
                                 tag, sizeof(tag), aad, sizeof(aad)), 0);

    /* Perform streaming decryption using long IV */
    AssertIntEQ(wc_AesInit(&aesDec, NULL, INVALID_DEVID), 0);
    AssertIntEQ(wc_AesGcmInit(&aesDec, key, sizeof(key), iv, sizeof(iv)), 0);
    AssertIntEQ(wc_AesGcmDecryptUpdate(&aesDec, plain, out, sizeof(out), aad,
                                       sizeof(aad)), 0);
    AssertIntEQ(wc_AesGcmDecryptFinal(&aesDec, tag, sizeof(tag)), 0);
    AssertIntEQ(XMEMCMP(plain, in, sizeof(in)), 0);

    /* Free resources */
    wc_AesFree(&aesEnc);
    wc_AesFree(&aesDec);

    ret = TEST_SUCCESS;

#endif

    return ret;

} /* END wc_AesGcmMixedEncDecLongIV */

/*
 * unit test for wc_GmacSetKey()
 */
static int test_wc_GmacSetKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(HAVE_AESGCM)
    Gmac    gmac;
    byte    key16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#ifdef WOLFSSL_AES_192
    byte    key24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
#endif
#ifdef WOLFSSL_AES_256
    byte    key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
#endif
    byte    badKey16[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x66
    };
    byte    badKey24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
    byte    badKey32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };
    int     ret = 0;

    ret = wc_AesInit(&gmac.aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

#ifdef WOLFSSL_AES_128
    ret = wc_GmacSetKey(&gmac, key16, sizeof(key16)/sizeof(byte));
#endif
#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        ret = wc_GmacSetKey(&gmac, key24, sizeof(key24)/sizeof(byte));
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = wc_GmacSetKey(&gmac, key32, sizeof(key32)/sizeof(byte));
    }
#endif

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_GmacSetKey(NULL, key16, sizeof(key16)/sizeof(byte));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacSetKey(&gmac, NULL, sizeof(key16)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacSetKey(&gmac, badKey16, sizeof(badKey16)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacSetKey(&gmac, badKey24, sizeof(badKey24)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacSetKey(&gmac, badKey32, sizeof(badKey32)/sizeof(byte));
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&gmac.aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_GmacSetKey */

/*
 * unit test for wc_GmacUpdate
 */
static int test_wc_GmacUpdate(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_AES) && defined(HAVE_AESGCM)
    Gmac    gmac;
#ifdef WOLFSSL_AES_128
    const byte key16[] =
    {
        0x89, 0xc9, 0x49, 0xe9, 0xc8, 0x04, 0xaf, 0x01,
        0x4d, 0x56, 0x04, 0xb3, 0x94, 0x59, 0xf2, 0xc8
    };
#endif
#ifdef WOLFSSL_AES_192
    byte    key24[] =
    {
        0x41, 0xc5, 0xda, 0x86, 0x67, 0xef, 0x72, 0x52,
        0x20, 0xff, 0xe3, 0x9a, 0xe0, 0xac, 0x59, 0x0a,
        0xc9, 0xfc, 0xa7, 0x29, 0xab, 0x60, 0xad, 0xa0
    };
#endif
#ifdef WOLFSSL_AES_256
   byte    key32[] =
    {
        0x78, 0xdc, 0x4e, 0x0a, 0xaf, 0x52, 0xd9, 0x35,
        0xc3, 0xc0, 0x1e, 0xea, 0x57, 0x42, 0x8f, 0x00,
        0xca, 0x1f, 0xd4, 0x75, 0xf5, 0xda, 0x86, 0xa4,
        0x9c, 0x8d, 0xd7, 0x3d, 0x68, 0xc8, 0xe2, 0x23
    };
#endif
#ifdef WOLFSSL_AES_128
    const byte authIn[] =
    {
        0x82, 0xad, 0xcd, 0x63, 0x8d, 0x3f, 0xa9, 0xd9,
        0xf3, 0xe8, 0x41, 0x00, 0xd6, 0x1e, 0x07, 0x77
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte authIn2[] =
    {
       0x8b, 0x5c, 0x12, 0x4b, 0xef, 0x6e, 0x2f, 0x0f,
       0xe4, 0xd8, 0xc9, 0x5c, 0xd5, 0xfa, 0x4c, 0xf1
    };
#endif
    const byte authIn3[] =
    {
        0xb9, 0x6b, 0xaa, 0x8c, 0x1c, 0x75, 0xa6, 0x71,
        0xbf, 0xb2, 0xd0, 0x8d, 0x06, 0xbe, 0x5f, 0x36
    };
#ifdef WOLFSSL_AES_128
    const byte tag1[] = /* Known. */
    {
        0x88, 0xdb, 0x9d, 0x62, 0x17, 0x2e, 0xd0, 0x43,
        0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte tag2[] = /* Known */
    {
        0x20, 0x4b, 0xdb, 0x1b, 0xd6, 0x21, 0x54, 0xbf,
        0x08, 0x92, 0x2a, 0xaa, 0x54, 0xee, 0xd7, 0x05
    };
#endif
    const byte tag3[] = /* Known */
    {
        0x3e, 0x5d, 0x48, 0x6a, 0xa2, 0xe3, 0x0b, 0x22,
        0xe0, 0x40, 0xb8, 0x57, 0x23, 0xa0, 0x6e, 0x76
    };
#ifdef WOLFSSL_AES_128
    const byte iv[] =
    {
        0xd1, 0xb1, 0x04, 0xc8, 0x15, 0xbf, 0x1e, 0x94,
        0xe2, 0x8c, 0x8f, 0x16
    };
#endif
#ifdef WOLFSSL_AES_192
    const byte iv2[] =
    {
        0x05, 0xad, 0x13, 0xa5, 0xe2, 0xc2, 0xab, 0x66,
        0x7e, 0x1a, 0x6f, 0xbc
    };
#endif
    const byte iv3[] =
    {
        0xd7, 0x9c, 0xf2, 0x2d, 0x50, 0x4c, 0xc7, 0x93,
        0xc3, 0xfb, 0x6c, 0x8a
    };
    byte    tagOut[16];
    byte    tagOut2[24];
    byte    tagOut3[32];
    int     ret = 0;

    /* Init stack variables. */
    XMEMSET(tagOut, 0, sizeof(tagOut));
    XMEMSET(tagOut2, 0, sizeof(tagOut2));
    XMEMSET(tagOut3, 0, sizeof(tagOut3));

    ret = wc_AesInit(&gmac.aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

#ifdef WOLFSSL_AES_128
    ret = wc_GmacSetKey(&gmac, key16, sizeof(key16));
    if (ret == 0) {
        ret = wc_GmacUpdate(&gmac, iv, sizeof(iv), authIn, sizeof(authIn),
                                                    tagOut, sizeof(tag1));
        if (ret == 0) {
            ret = XMEMCMP(tag1, tagOut, sizeof(tag1));
        }
        wc_AesFree(&gmac.aes);
    }

#endif

#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        XMEMSET(&gmac, 0, sizeof(Gmac));
        ret = wc_GmacSetKey(&gmac, key24, sizeof(key24)/sizeof(byte));
    }
    if (ret == 0) {
        ret = wc_GmacUpdate(&gmac, iv2, sizeof(iv2), authIn2,
                            sizeof(authIn2), tagOut2, sizeof(tag2));
    }
    if (ret == 0) {
        ret = XMEMCMP(tagOut2, tag2, sizeof(tag2));
        wc_AesFree(&gmac.aes);
    }
#endif

#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        XMEMSET(&gmac, 0, sizeof(Gmac));
        ret = wc_GmacSetKey(&gmac, key32, sizeof(key32)/sizeof(byte));
    }
    if (ret == 0) {
        ret = wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3,
                            sizeof(authIn3), tagOut3, sizeof(tag3));
    }
    if (ret == 0) {
        ret = XMEMCMP(tag3, tagOut3, sizeof(tag3));
    }
#endif

    /*Pass bad args. */
    if (ret == 0) {
        ret = wc_GmacUpdate(NULL, iv3, sizeof(iv3), authIn3,
                                sizeof(authIn3), tagOut3, sizeof(tag3));
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3,
                                sizeof(authIn3), tagOut3, sizeof(tag3) - 5);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_GmacUpdate(&gmac, iv3, sizeof(iv3), authIn3,
                                sizeof(authIn3), tagOut3, sizeof(tag3) + 1);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_AesFree(&gmac.aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_GmacUpdate */


/*
 * testing wc_CamelliaSetKey
 */
static int test_wc_CamelliaSetKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    /*128-bit key*/
    static const byte key16[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    /* 192-bit key */
    static const byte key24[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    /* 256-bit key */
    static const byte key32[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    static const byte iv[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    int ret = 0;

    ret = wc_CamelliaSetKey(&camellia, key16, (word32)sizeof(key16), iv);
    if (ret == 0) {
        ret = wc_CamelliaSetKey(&camellia, key16,
                                        (word32)sizeof(key16), NULL);
        if (ret == 0) {
            ret = wc_CamelliaSetKey(&camellia, key24,
                                        (word32)sizeof(key24), iv);
        }
        if (ret == 0) {
            ret = wc_CamelliaSetKey(&camellia, key24,
                                        (word32)sizeof(key24), NULL);
        }
        if (ret == 0) {
            ret = wc_CamelliaSetKey(&camellia, key32,
                                        (word32)sizeof(key32), iv);
        }
        if (ret == 0) {
            ret = wc_CamelliaSetKey(&camellia, key32,
                                        (word32)sizeof(key32), NULL);
        }
    }
    /* Bad args. */
    if (ret == 0) {
        ret = wc_CamelliaSetKey(NULL, key32, (word32)sizeof(key32), iv);
        if (ret != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    } /* END bad args. */

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_CammeliaSetKey */

/*
 * Testing wc_CamelliaSetIV()
 */
static int test_wc_CamelliaSetIV(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CAMELLIA
    Camellia    camellia;
    static const byte iv[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    int ret = 0;

    ret = wc_CamelliaSetIV(&camellia, iv);
    if (ret == 0) {
        ret = wc_CamelliaSetIV(&camellia, NULL);
    }
    /* Bad args. */
    if (ret == 0) {
        ret = wc_CamelliaSetIV(NULL, NULL);
        if (ret != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_CamelliaSetIV*/

/*
 * Test wc_CamelliaEncryptDirect and wc_CamelliaDecryptDirect
 */
static int test_wc_CamelliaEncryptDecryptDirect(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    static const byte key24[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    static const byte iv[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    static const byte plainT[] =
    {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };
    byte    enc[sizeof(plainT)];
    byte    dec[sizeof(enc)];
    int     camE = WOLFSSL_FATAL_ERROR;
    int     camD = WOLFSSL_FATAL_ERROR;
    int     ret = 0;

    /*Init stack variables.*/
    XMEMSET(enc, 0, 16);
    XMEMSET(enc, 0, 16);

    ret = wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24), iv);
    if (ret == 0) {
        ret = wc_CamelliaEncryptDirect(&camellia, enc, plainT);
        if (ret == 0) {
            ret = wc_CamelliaDecryptDirect(&camellia, dec, enc);
            if (XMEMCMP(plainT, dec, CAMELLIA_BLOCK_SIZE)) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }
    /* Pass bad args. */
    if (ret == 0) {
        camE = wc_CamelliaEncryptDirect(NULL, enc, plainT);
        if (camE == BAD_FUNC_ARG) {
            camE = wc_CamelliaEncryptDirect(&camellia, NULL, plainT);
        }
        if (camE == BAD_FUNC_ARG) {
            camE = wc_CamelliaEncryptDirect(&camellia, enc, NULL);
        }
        if (camE == BAD_FUNC_ARG) {
            camE = 0;
        }
        else {
            camE = WOLFSSL_FATAL_ERROR;
        }
    }

    if (camE != 0) {
        return TEST_FAIL;
    }

    if (ret == 0) {
        camD = wc_CamelliaDecryptDirect(NULL, dec, enc);
        if (camD == BAD_FUNC_ARG) {
            camD = wc_CamelliaDecryptDirect(&camellia, NULL, enc);
        }
        if (camD == BAD_FUNC_ARG) {
            camD = wc_CamelliaDecryptDirect(&camellia, dec, NULL);
        }
        if (camD == BAD_FUNC_ARG) {
            camD = 0;
        }
        else {
            camD = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(camD == 0);
#endif
    return res;

} /* END test-wc_CamelliaEncryptDecryptDirect */

/*
 * Testing wc_CamelliaCbcEncrypt and wc_CamelliaCbcDecrypt
 */
static int test_wc_CamelliaCbcEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CAMELLIA
    Camellia camellia;
    static const byte key24[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    static const byte plainT[] =
    {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };
    byte    enc[CAMELLIA_BLOCK_SIZE];
    byte    dec[CAMELLIA_BLOCK_SIZE];
    int     camCbcE = WOLFSSL_FATAL_ERROR;
    int     camCbcD = WOLFSSL_FATAL_ERROR;
    int     ret = 0;

    /* Init stack variables. */
    XMEMSET(enc, 0, CAMELLIA_BLOCK_SIZE);
    XMEMSET(enc, 0, CAMELLIA_BLOCK_SIZE);

    ret = wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24), NULL);
    if (ret == 0) {
        ret = wc_CamelliaCbcEncrypt(&camellia, enc, plainT, CAMELLIA_BLOCK_SIZE);
        if (ret != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (ret == 0) {
        ret = wc_CamelliaSetKey(&camellia, key24, (word32)sizeof(key24), NULL);
        if (ret == 0) {
            ret = wc_CamelliaCbcDecrypt(&camellia, dec, enc, CAMELLIA_BLOCK_SIZE);
            if (XMEMCMP(plainT, dec, CAMELLIA_BLOCK_SIZE)) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Pass in bad args. */
    if (ret == 0) {
        camCbcE = wc_CamelliaCbcEncrypt(NULL, enc, plainT, CAMELLIA_BLOCK_SIZE);
        if (camCbcE == BAD_FUNC_ARG) {
            camCbcE = wc_CamelliaCbcEncrypt(&camellia, NULL, plainT,
                                                    CAMELLIA_BLOCK_SIZE);
        }
        if (camCbcE == BAD_FUNC_ARG) {
            camCbcE = wc_CamelliaCbcEncrypt(&camellia, enc, NULL,
                                                    CAMELLIA_BLOCK_SIZE);
        }
        if (camCbcE == BAD_FUNC_ARG) {
            camCbcE = 0;
        }
        else {
            camCbcE = WOLFSSL_FATAL_ERROR;
        }
    }

    if (camCbcE != 0) {
        return TEST_FAIL;
    }

    if (ret == 0) {
        camCbcD = wc_CamelliaCbcDecrypt(NULL, dec, enc, CAMELLIA_BLOCK_SIZE);
        if (camCbcD == BAD_FUNC_ARG) {
            camCbcD = wc_CamelliaCbcDecrypt(&camellia, NULL, enc,
                                                    CAMELLIA_BLOCK_SIZE);
        }
        if (camCbcD == BAD_FUNC_ARG) {
            camCbcD = wc_CamelliaCbcDecrypt(&camellia, dec, NULL,
                                                    CAMELLIA_BLOCK_SIZE);
        }
        if (camCbcD == BAD_FUNC_ARG) {
            camCbcD = 0;
        }
        else {
            camCbcD = WOLFSSL_FATAL_ERROR;
        }
    } /* END bad args. */

    res = TEST_RES_CHECK(camCbcD == 0);
#endif
    return res;

} /* END test_wc_CamelliaCbcEncryptDecrypt */


/*
 * Testing wc_Arc4SetKey()
 */
static int test_wc_Arc4SetKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_RC4
    Arc4 arc;
    const char* key = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    int keyLen = 8;
    int ret = 0;

    ret = wc_Arc4SetKey(&arc, (byte*)key, keyLen);
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_Arc4SetKey(NULL, (byte*)key, keyLen);
        if (ret == BAD_FUNC_ARG)
            ret = wc_Arc4SetKey(&arc, NULL, keyLen); /* NULL key */
        if (ret == BAD_FUNC_ARG)
            ret = wc_Arc4SetKey(&arc, (byte*)key, 0); /* length == 0 */
        if (ret == BAD_FUNC_ARG)
            ret = WOLFSSL_ERROR_NONE;
        else
            ret = WOLFSSL_FATAL_ERROR;
    } /* END test bad args. */

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_Arc4SetKey */

/*
 * Testing wc_Arc4Process for ENC/DEC.
 */
static int test_wc_Arc4Process(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_RC4
    Arc4 enc, dec;
    const char* key = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    int keyLen = 8;
    const char* input = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    byte cipher[8];
    byte plain[8];
    int ret;

    /* Init stack variables */
    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMSET(plain, 0, sizeof(plain));

    /* Use for async. */
    ret = wc_Arc4Init(&enc, NULL, INVALID_DEVID);
    if (ret == 0) {
        ret = wc_Arc4Init(&dec, NULL, INVALID_DEVID);
    }

    if (ret == 0) {
        ret = wc_Arc4SetKey(&enc, (byte*)key, keyLen);
    }
    if (ret == 0) {
        ret = wc_Arc4SetKey(&dec, (byte*)key, keyLen);
    }
    if (ret == 0) {
        ret = wc_Arc4Process(&enc, cipher, (byte*)input, keyLen);
    }
    if (ret == 0) {
        ret = wc_Arc4Process(&dec, plain, cipher, keyLen);
        if (ret != 0 || XMEMCMP(plain, input, keyLen)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    /* Bad args. */
    if (ret == 0) {
        ret = wc_Arc4Process(NULL, plain, cipher, keyLen);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Arc4Process(&dec, NULL, cipher, keyLen);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_Arc4Process(&dec, plain, NULL, keyLen);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_Arc4Free(&enc);
    wc_Arc4Free(&dec);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

}/* END test_wc_Arc4Process */


/*
 * Testing wc_Init RsaKey()
 */
static int test_wc_InitRsaKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_RSA
    RsaKey  key;
    int     ret = 0;

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_InitRsaKey(NULL, HEAP_HINT);
        #ifndef HAVE_USER_RSA
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
        #else
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
        #endif
            ret = WOLFSSL_FATAL_ERROR;
        }
    } /* end if */

    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_InitRsaKey */


/*
 * Testing wc_RsaPrivateKeyDecode()
 */
static int test_wc_RsaPrivateKeyDecode(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && (defined(USE_CERT_BUFFERS_1024)\
        || defined(USE_CERT_BUFFERS_2048)) && !defined(HAVE_FIPS)
    RsaKey  key;
    byte*   tmp;
    word32  idx = 0;
    int     bytes = 0;
    int     ret = 0;

    tmp = (byte*)XMALLOC(FOURK_BUF, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = wc_InitRsaKey(&key, HEAP_HINT);
    }
    if (ret == 0) {
        #ifdef USE_CERT_BUFFERS_1024
            XMEMCPY(tmp, client_key_der_1024, sizeof_client_key_der_1024);
            bytes = sizeof_client_key_der_1024;
        #else
            XMEMCPY(tmp, client_key_der_2048, sizeof_client_key_der_2048);
            bytes = sizeof_client_key_der_2048;
        #endif /* Use cert buffers. */

        ret = wc_RsaPrivateKeyDecode(tmp, &idx, &key, (word32)bytes);
    }
    #ifndef HAVE_USER_RSA
        /* Test bad args. */
        if (ret == 0) {
            ret = wc_RsaPrivateKeyDecode(NULL, &idx, &key, (word32)bytes);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaPrivateKeyDecode(tmp, NULL, &key, (word32)bytes);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaPrivateKeyDecode(tmp, &idx, NULL, (word32)bytes);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Test bad args. User RSA. */
        if (ret == 0) {
            ret = wc_RsaPrivateKeyDecode(NULL, &idx, &key, (word32)bytes);
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaPrivateKeyDecode(tmp, NULL, &key, (word32)bytes);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaPrivateKeyDecode(tmp, &idx, NULL, (word32)bytes);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (tmp != NULL) {
        XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaPrivateKeyDecode */

/*
 * Testing wc_RsaPublicKeyDecode()
 */
static int test_wc_RsaPublicKeyDecode(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && (defined(USE_CERT_BUFFERS_1024)\
        || defined(USE_CERT_BUFFERS_2048)) && !defined(HAVE_FIPS)
    RsaKey  keyPub;
    byte*   tmp;
    word32  idx = 0;
    int     bytes = 0;
    word32  keySz = 0;
    word32  tstKeySz = 0;
    int     ret = 0;
#if defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM)
    XFILE f;
    const char* rsaPssPubKey = "./certs/rsapss/ca-rsapss-key.der";
    const char* rsaPssPubKeyNoParams = "./certs/rsapss/ca-3072-rsapss-key.der";
    byte buf[4096];
#endif

    tmp = (byte*)XMALLOC(GEN_BUF, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = wc_InitRsaKey(&keyPub, HEAP_HINT);
    }
    if (ret == 0) {
        #ifdef USE_CERT_BUFFERS_1024
            XMEMCPY(tmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
            bytes = sizeof_client_keypub_der_1024;
            keySz = 1024;
        #else
            XMEMCPY(tmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
            bytes = sizeof_client_keypub_der_2048;
            keySz = 2048;
        #endif

        ret = wc_RsaPublicKeyDecode(tmp, &idx, &keyPub, (word32)bytes);
    }
    #ifndef HAVE_USER_RSA
        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_RsaPublicKeyDecode(NULL, &idx, &keyPub, (word32)bytes);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaPublicKeyDecode(tmp, NULL, &keyPub, (word32)bytes);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaPublicKeyDecode(tmp, &idx, NULL, (word32)bytes);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_RsaPublicKeyDecode(NULL, &idx, &keyPub, (word32)bytes);
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaPublicKeyDecode(tmp, NULL, &keyPub, (word32)bytes);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaPublicKeyDecode(tmp, &idx, NULL, (word32)bytes);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (wc_FreeRsaKey(&keyPub) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0) {
        /* Test for getting modulus key size */
        idx = 0;
        ret = wc_RsaPublicKeyDecode_ex(tmp, &idx, (word32)bytes, NULL,
            &tstKeySz, NULL, NULL);
        ret = (ret == 0 && tstKeySz == keySz/8) ? 0 : WOLFSSL_FATAL_ERROR;
    }

#if defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM)
    f = XFOPEN(rsaPssPubKey, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);
    idx = 0;
    AssertIntEQ(wc_RsaPublicKeyDecode_ex(buf, &idx, bytes, NULL, NULL, NULL,
                                         NULL), 0);
    f = XFOPEN(rsaPssPubKeyNoParams, "rb");
    AssertTrue((f != XBADFILE));
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);
    idx = 0;
    AssertIntEQ(wc_RsaPublicKeyDecode_ex(buf, &idx, bytes, NULL, NULL, NULL,
                                         NULL), 0);
#endif

    if (tmp != NULL) {
        XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}  /* END test_wc_RsaPublicKeyDecode */

/*
 * Testing wc_RsaPublicKeyDecodeRaw()
 */
static int test_wc_RsaPublicKeyDecodeRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA)
    RsaKey      key;
    const byte  n = 0x23;
    const byte  e = 0x03;
    int         nSz = sizeof(n);
    int         eSz = sizeof(e);
    int         ret;

    ret = wc_InitRsaKey(&key, HEAP_HINT);
    if (ret == 0) {
        ret = wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, &key);
    }
#ifndef HAVE_USER_RSA
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_RsaPublicKeyDecodeRaw(NULL, nSz, &e, eSz, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_RsaPublicKeyDecodeRaw(&n, nSz, NULL, eSz, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#else
    /* Pass in bad args. User RSA. */
    if (ret == 0) {
        ret = wc_RsaPublicKeyDecodeRaw(NULL, nSz, &e, eSz, &key);
        if (ret == USER_CRYPTO_ERROR) {
            ret = wc_RsaPublicKeyDecodeRaw(&n, nSz, NULL, eSz, &key);
        }
        if (ret == USER_CRYPTO_ERROR) {
            ret = wc_RsaPublicKeyDecodeRaw(&n, nSz, &e, eSz, NULL);
        }
        if (ret == USER_CRYPTO_ERROR) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaPublicKeyDecodeRaw */


#if (!defined(NO_RSA) || !defined(HAVE_FAST_RSA)) && defined(WOLFSSL_KEY_GEN)
    /* In FIPS builds, wc_MakeRsaKey() will return an error if it cannot find
     * a probable prime in 5*(modLen/2) attempts. In non-FIPS builds, it keeps
     * trying until it gets a probable prime. */
    #ifdef HAVE_FIPS
        static int MakeRsaKeyRetry(RsaKey* key, int size, long e, WC_RNG* rng)
        {
            int ret;

            for (;;) {
                ret = wc_MakeRsaKey(key, size, e, rng);
                if (ret != PRIME_GEN_E) break;
                fprintf(stderr, "MakeRsaKey couldn't find prime; "
                                "trying again.\n");
            }

            return ret;
        }
        #define MAKE_RSA_KEY(a, b, c, d) MakeRsaKeyRetry(a, b, c, d)
    #else
        #define MAKE_RSA_KEY(a, b, c, d) wc_MakeRsaKey(a, b, c, d)
    #endif
#endif


/*
 * Testing wc_MakeRsaKey()
 */
static int test_wc_MakeRsaKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)

    RsaKey  genKey;
    WC_RNG  rng;
    #if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
        (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int     bits = 1024;
    #else
    int     bits = 2048;
    #endif
    int     ret = 0;

    ret = wc_InitRsaKey(&genKey, HEAP_HINT);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
        if (ret == 0) {
            ret = MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, &rng);
            if (ret == 0 && wc_FreeRsaKey(&genKey) != 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }
    #ifndef HAVE_USER_RSA
        /* Test bad args. */
        if (ret == 0) {
            ret = MAKE_RSA_KEY(NULL, bits, WC_RSA_EXPONENT, &rng);
            if (ret == BAD_FUNC_ARG) {
                ret = MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                /* e < 3 */
                ret = MAKE_RSA_KEY(&genKey, bits, 2, &rng);
            }
            if (ret == BAD_FUNC_ARG) {
                /* e & 1 == 0 */
                ret = MAKE_RSA_KEY(&genKey, bits, 6, &rng);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Test bad args. */
        if (ret == 0) {
            ret = MAKE_RSA_KEY(NULL, bits, WC_RSA_EXPONENT, &rng);
            if (ret == USER_CRYPTO_ERROR) {
                ret = MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, NULL);
            }
            if (ret == USER_CRYPTO_ERROR) {
                /* e < 3 */
                ret = MAKE_RSA_KEY(&genKey, bits, 2, &rng);
            }
            if (ret == USER_CRYPTO_ERROR) {
                /* e & 1 == 0 */
                ret = MAKE_RSA_KEY(&genKey, bits, 6, &rng);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_MakeRsaKey */

/*
 * Test the bounds checking on the cipher text versus the key modulus.
 * 1. Make a new RSA key.
 * 2. Set c to 1.
 * 3. Decrypt c into k. (error)
 * 4. Copy the key modulus to c and sub 1 from the copy.
 * 5. Decrypt c into k. (error)
 * Valid bounds test cases are covered by all the other RSA tests.
 */
static int test_RsaDecryptBoundsCheck(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WC_RSA_NO_PADDING) && \
    (defined(USE_CERT_BUFFERS_1024) || defined(USE_CERT_BUFFERS_2048)) && \
    defined(WOLFSSL_PUBLIC_MP) && !defined(NO_RSA_BOUNDS_CHECK)
    WC_RNG rng;
    RsaKey key;
    byte flatC[256];
    word32 flatCSz;
    byte out[256];
    word32 outSz = sizeof(out);
    int ret;

    XMEMSET(&rng, 0, sizeof(rng));

    ret = wc_InitRng(&rng);

    if (ret == 0)
        ret = wc_InitRsaKey(&key, HEAP_HINT);

    if (ret == 0) {
        const byte* derKey;
        word32 derKeySz;
        word32 idx = 0;

        #ifdef USE_CERT_BUFFERS_1024
            derKey = server_key_der_1024;
            derKeySz = (word32)sizeof_server_key_der_1024;
            flatCSz = 128;
        #else
            derKey = server_key_der_2048;
            derKeySz = (word32)sizeof_server_key_der_2048;
            flatCSz = 256;
        #endif

        ret = wc_RsaPrivateKeyDecode(derKey, &idx, &key, derKeySz);
    }

    if (ret == 0) {
        XMEMSET(flatC, 0, flatCSz);
        flatC[flatCSz-1] = 1;

        ret = wc_RsaDirect(flatC, flatCSz, out, &outSz, &key,
                           RSA_PRIVATE_DECRYPT, &rng);

        if (ret == RSA_OUT_OF_RANGE_E) {
            mp_int c;
            mp_init_copy(&c, &key.n);
            mp_sub_d(&c, 1, &c);
            mp_to_unsigned_bin(&c, flatC);
            ret = wc_RsaDirect(flatC, flatCSz, out, &outSz, &key,
                               RSA_PRIVATE_DECRYPT, NULL);
            mp_clear(&c);
        }
        if (ret == RSA_OUT_OF_RANGE_E)
            ret = 0;
        else
            ret = WOLFSSL_FATAL_ERROR;
    }

    if (wc_FreeRsaKey(&key) || wc_FreeRng(&rng) || ret != 0)
        ret = WOLFSSL_FATAL_ERROR;

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_RsaDecryptBoundsCheck */

/*
 * Testing wc_SetKeyUsage()
 */
static int test_wc_SetKeyUsage(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) && !defined(HAVE_FIPS)
    Cert    myCert;
    int     ret = 0;

    ret = wc_InitCert(&myCert);

    if (ret == 0) {
        ret = wc_SetKeyUsage(&myCert, "keyEncipherment,keyAgreement");
        if (ret == 0) {
            ret = wc_SetKeyUsage(&myCert, "digitalSignature,nonRepudiation");
        }
        if (ret == 0) {
            ret = wc_SetKeyUsage(&myCert, "contentCommitment,encipherOnly");
        }
        if (ret == 0) {
            ret = wc_SetKeyUsage(&myCert, "decipherOnly");
        }
        if (ret == 0) {
            ret = wc_SetKeyUsage(&myCert, "cRLSign,keyCertSign");
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_SetKeyUsage(NULL, "decipherOnly");
        if (ret == BAD_FUNC_ARG) {
            ret = wc_SetKeyUsage(&myCert, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_SetKeyUsage(&myCert, "");
        }
        if (ret == KEYUSAGE_E) {
            ret = wc_SetKeyUsage(&myCert, ",");
        }
        if (ret == KEYUSAGE_E) {
            ret = wc_SetKeyUsage(&myCert, "digitalSignature, cRLSign");
        }
        if (ret == KEYUSAGE_E) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END  test_wc_SetKeyUsage */
/*
 * Testing wc_CheckProbablePrime()
 */
static int test_wc_CheckProbablePrime(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING)

#define CHECK_PROBABLE_PRIME_KEY_BITS 2048

    RsaKey key;
    WC_RNG rng;
    byte   e[3];
    word32 eSz = (word32)sizeof(e);
    byte   n[CHECK_PROBABLE_PRIME_KEY_BITS / 8];
    word32 nSz = (word32)sizeof(n);
    byte   d[CHECK_PROBABLE_PRIME_KEY_BITS / 8];
    word32 dSz = (word32)sizeof(d);
    byte   p[CHECK_PROBABLE_PRIME_KEY_BITS / 8 / 2];
    word32 pSz = (word32)sizeof(p);
    byte   q[CHECK_PROBABLE_PRIME_KEY_BITS / 8 / 2];
    word32 qSz = (word32)sizeof(q);
    int    nlen = CHECK_PROBABLE_PRIME_KEY_BITS;
    int    ret = 0;
    int*   isPrime;
    int    test[5];
    isPrime = test;

    ret = wc_InitRsaKey(&key, HEAP_HINT);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_RsaSetRNG(&key, &rng);
    }
    if (ret == 0) {
        ret = wc_MakeRsaKey(&key, CHECK_PROBABLE_PRIME_KEY_BITS, WC_RSA_EXPONENT, &rng);
    }
    if (ret == 0) {
        PRIVATE_KEY_UNLOCK();
        ret = wc_RsaExportKey(&key, e, &eSz, n, &nSz, d, &dSz,
                                p, &pSz, q, &qSz);
        PRIVATE_KEY_LOCK();
    }
    /* Bad cases */
    if (ret == 0) {
        ret = wc_CheckProbablePrime(NULL, pSz, q, qSz, e, eSz,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, 0, q, qSz, e, eSz,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, pSz, NULL, qSz, e, eSz,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, pSz, q, 0, e, eSz,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, pSz, q, qSz, NULL, eSz,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, pSz, q, qSz, e, 0,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_CheckProbablePrime(NULL, 0, NULL, 0, NULL, 0,
                                     nlen, isPrime);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    /* Good case */
    if (ret == 0) {
        ret = wc_CheckProbablePrime(p, pSz, q, qSz, e, eSz,
                                     nlen, isPrime);
    }
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

#undef CHECK_PROBABLE_PRIME_KEY_BITS

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END  test_wc_CheckProbablePrime */
/*
 * Testing wc_RsaPSS_Verify()
 */
static int test_wc_RsaPSS_Verify(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey        key;
    WC_RNG        rng;
    int           sz = 256;
    byte*         pt;
    const char*   szMessage = "This is the string to be signed";
    unsigned char pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    unsigned char pDecrypted[2048/8];
    word32        outLen = sizeof(pDecrypted);
    int           ret = 0;
    pt = pDecrypted;

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_RsaSetRNG(&key, &rng);
    }
    if (ret == 0) {
            ret = wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng);
    }

    if (ret == 0) {
        ret = wc_RsaPSS_Sign((byte*)szMessage, (word32)XSTRLEN(szMessage)+1,
                pSignature, sizeof(pSignature),
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng);
        if (ret > 0) {
            sz = ret;
            ret = 0;
        }
    }
    /* Bad cases */
    if (ret == 0) {
        ret = wc_RsaPSS_Verify(NULL, sz, pt, outLen,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_Verify(pSignature, 0, pt, outLen,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_Verify(pSignature, sz, NULL, outLen,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_Verify(NULL, 0, NULL, outLen,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    /* Good case */
    if (ret == 0) {
        ret = wc_RsaPSS_Verify(pSignature, sz, pt, outLen,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret > 0) {
                ret = 0;
            }
    }
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END  test_wc_RsaPSS_Verify */
/*
 * Testing wc_RsaPSS_VerifyCheck()
 */
static int test_wc_RsaPSS_VerifyCheck(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey        key;
    WC_RNG        rng;
    int           sz = 256; /* 2048/8 */
    byte*         pt;
    byte          digest[32];
    word32        digestSz = sizeof(digest);
    unsigned char pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    word32        pSignatureSz = sizeof(pSignature);
    unsigned char pDecrypted[2048/8];
    word32        outLen = sizeof(pDecrypted);
    int           ret = 0;
    pt = pDecrypted;

    XMEMSET(digest, 0, sizeof(digest));
    XMEMSET(pSignature, 0, sizeof(pSignature));

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_RsaSetRNG(&key, &rng);
    }
    if (ret == 0) {
            ret = wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng);
    }
    if (ret == 0) {
        digestSz = wc_HashGetDigestSize(WC_HASH_TYPE_SHA256);
        ret = wc_Hash(WC_HASH_TYPE_SHA256, pSignature, sz, digest, digestSz);

    }

    if (ret == 0) {
        ret = wc_RsaPSS_Sign(digest, digestSz, pSignature, pSignatureSz,
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng);
        if (ret > 0) {
            sz = ret;
            ret = 0;
        }
    }
    /* Bad cases */
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheck(NULL, sz, pt, outLen,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheck(pSignature, 0, pt, outLen,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheck(pSignature, sz, NULL, outLen,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheck(NULL, 0, NULL, outLen,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }

    /* Good case */
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheck(pSignature, sz, pt, outLen,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret > 0) {
                ret = 0;
            }
    }
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END  test_wc_RsaPSS_VerifyCheck */
/*
 * Testing wc_RsaPSS_VerifyCheckInline()
 */
static int test_wc_RsaPSS_VerifyCheckInline(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_SELFTEST) && \
 !defined(HAVE_FIPS) && defined(WC_RSA_BLINDING) && defined(WC_RSA_PSS)
    RsaKey              key;
    WC_RNG              rng;
    int                 sz = 256;
    byte*               pt;
    byte                digest[32];
    word32              digestSz = sizeof(digest);
    unsigned char       pSignature[2048/8]; /* 2048 is RSA_KEY_SIZE */
    unsigned char       pDecrypted[2048/8];
    int                 ret;
    pt = pDecrypted;

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    XMEMSET(digest, 0, sizeof(digest));
    XMEMSET(pSignature, 0, sizeof(pSignature));

    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_RsaSetRNG(&key, &rng);
    }
    if (ret == 0) {
            ret = wc_MakeRsaKey(&key, 2048, WC_RSA_EXPONENT, &rng);
    }
    if (ret == 0) {
        digestSz = wc_HashGetDigestSize(WC_HASH_TYPE_SHA256);
        ret = wc_Hash(WC_HASH_TYPE_SHA256, pSignature, sz, digest, digestSz);

    }

    if (ret == 0) {
        ret = wc_RsaPSS_Sign(digest, digestSz, pSignature, sizeof(pSignature),
                WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key, &rng);
        if (ret > 0) {
            sz = ret;
            ret = 0;
        }
    }
    /* Bad Cases */
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheckInline(NULL, sz, &pt,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheckInline(pSignature, 0, NULL,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheckInline(NULL, 0, &pt,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheckInline(pSignature, sz, &pt,
                digest, digestSz, WC_HASH_TYPE_SHA, WC_MGF1SHA256, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
    }
    /* Good case */
    if (ret == 0) {
        ret = wc_RsaPSS_VerifyCheckInline(pSignature, sz, &pt,
                digest, digestSz, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, &key);
            if (ret > 0) {
                ret = 0;
            }
    }
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END  test_wc_RsaPSS_VerifyCheckInline */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
static void sample_mutex_cb (int flag, int type, const char* file, int line)
{
    (void)flag;
    (void)type;
    (void)file;
    (void)line;
}
#endif
/*
 * Testing wc_LockMutex_ex
 */
static int test_wc_LockMutex_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int ret = 0;
    int flag = CRYPTO_LOCK;
    int type = 0;
    const char* file = "./test-LockMutex_ex.txt";
    int line = 0;

    /* without SetMutexCb */
    ret = wc_LockMutex_ex(flag, type, file, line);
    if (ret ==  BAD_STATE_E) {
        ret = 0;
    }
    /* with SetMutexCb */
    if (ret == 0) {
        ret = wc_SetMutexCb(sample_mutex_cb);
        if (ret == 0) {
             ret = wc_LockMutex_ex(flag, type, file, line);
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/*End test_wc_LockMutex_ex*/
/*
 * Testing wc_SetMutexCb
 */
static int test_wc_SetMutexCb(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int ret = wc_SetMutexCb(sample_mutex_cb);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/*End test_wc_SetMutexCb*/

/*
 * Testing wc_RsaKeyToDer()
 */
static int test_wc_RsaKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey  genKey;
    WC_RNG  rng;
    byte*   der;
    int     ret = 0;
    #if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
        (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int     bits = 1024;
    word32  derSz = 611;
    /* (2 x 128) + 2 (possible leading 00) + (5 x 64) + 5 (possible leading 00)
       + 3 (e) + 8 (ASN tag) + 10 (ASN length) + 4 seqSz + 3 version */
    #else
    int     bits = 2048;
    word32  derSz = 1196;
    /* (2 x 256) + 2 (possible leading 00) + (5 x 128) + 5 (possible leading 00)
       + 3 (e) + 8 (ASN tag) + 17 (ASN length) + 4 seqSz + 3 version */
    #endif
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&genKey, 0, sizeof(genKey));

    der = (byte*)XMALLOC(derSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    /* Init structures. */
    if (ret == 0) {
        ret = wc_InitRsaKey(&genKey, HEAP_HINT);
    }
        if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    /* Make key. */
    if (ret == 0) {
        ret = MAKE_RSA_KEY(&genKey, bits, WC_RSA_EXPONENT, &rng);
        if (ret != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_RsaKeyToDer(&genKey, der, derSz);
        if (ret > 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    #ifndef HAVE_USER_RSA
        /* Pass good/bad args. */
        if (ret == 0) {
            ret = wc_RsaKeyToDer(NULL, der, FOURK_BUF);
            if (ret == BAD_FUNC_ARG) {
                /* Get just the output length */
                ret = wc_RsaKeyToDer(&genKey, NULL, 0);
            }
            if (ret > 0) {
                /* Try Public Key. */
                genKey.type = 0;
                ret = wc_RsaKeyToDer(&genKey, der, FOURK_BUF);
            #ifdef WOLFSSL_CHECK_MEM_ZERO
                /* Put back to Private Key */
                genKey.type = 1;
            #endif
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Pass good/bad args. */
        if (ret == 0) {
            ret = wc_RsaKeyToDer(NULL, der, FOURK_BUF);
            if (ret == USER_CRYPTO_ERROR) {
                /* Get just the output length */
                ret = wc_RsaKeyToDer(&genKey, NULL, 0);
            }
            if (ret > 0) {
                /* Try Public Key. */
                genKey.type = 0;
                ret = wc_RsaKeyToDer(&genKey, der, FOURK_BUF);
            #ifdef WOLFSSL_CHECK_MEM_ZERO
                /* Put back to Private Key */
                genKey.type = 1;
            #endif
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (der != NULL) {
        XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (wc_FreeRsaKey(&genKey) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_RsaKeyToDer */

/*
 *  Testing wc_RsaKeyToPublicDer()
 */
static int test_wc_RsaKeyToPublicDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey      key;
    WC_RNG      rng;
    byte*       der;
    int         ret = 0;
    #if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
        (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int         bits = 1024;
    word32      derLen = 162;
    #else
    int         bits = 2048;
    word32      derLen = 294;
    #endif

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    der = (byte*)XMALLOC(derLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = wc_InitRsaKey(&key, HEAP_HINT);
    }
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng);
    }

    if (ret == 0) {
        /* test getting size only */
        ret = wc_RsaKeyToPublicDer(&key, NULL, derLen);
        if (ret >= 0)
            ret = 0;
    }
    if (ret == 0) {
        ret = wc_RsaKeyToPublicDer(&key, der, derLen);
        if (ret >= 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        /* test getting size only */
        ret = wc_RsaKeyToPublicDer_ex(&key, NULL, derLen, 0);
        if (ret >= 0)
            ret = 0;
    }
    if (ret == 0) {
        ret = wc_RsaKeyToPublicDer_ex(&key, der, derLen, 0);
        if (ret >= 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    #ifndef HAVE_USER_RSA
        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_RsaKeyToPublicDer(NULL, der, derLen);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaKeyToPublicDer(&key, der, -1);
            }
            if (ret == BUFFER_E || ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_RsaKeyToPublicDer(NULL, der, derLen);
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaKeyToPublicDer(&key, der, -1);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (der != NULL) {
        XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaKeyToPublicDer */

/*
 *  Testing wc_RsaPublicEncrypt() and wc_RsaPrivateDecrypt()
 */
static int test_wc_RsaPublicEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey  key;
    WC_RNG  rng;
    int     ret = 0;
    const char inStr[] = TEST_STRING;
    const word32 plainLen = (word32)TEST_STRING_SZ;
    const word32 inLen = (word32)TEST_STRING_SZ;
    int          bits = TEST_RSA_BITS;
    const word32 cipherLen = TEST_RSA_BYTES;
    word32 cipherLenResult = cipherLen;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(cipher, byte, TEST_RSA_BYTES, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || plain == NULL || cipher == NULL) {
        fprintf(stderr, "test_wc_RsaPublicEncryptDecrypt malloc failed\n");
        return MEMORY_E;
}
#endif
    XMEMCPY(in, inStr, inLen);

    ret = wc_InitRsaKey(&key, HEAP_HINT);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng);
    }
    /* Encrypt. */

    if (ret == 0) {
        ret = wc_RsaPublicEncrypt(in, inLen, cipher, cipherLen, &key, &rng);
        if (ret >= 0) {
            cipherLenResult = ret;
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass bad args. */
    /* Tests PsaPublicEncryptEx() which, is tested by another fn. No need dup.*/
    if (ret != 0) {
        return TEST_FAIL;
    }

    /* Decrypt */
    #if defined(WC_RSA_BLINDING) && !defined(HAVE_FIPS)
        /* Bind rng */
        if (ret == 0) {
            ret = wc_RsaSetRNG(&key, &rng);
        }
    #endif
    if (ret == 0) {
        ret = wc_RsaPrivateDecrypt(cipher, cipherLenResult, plain, plainLen, &key);
    }
    if (ret >= 0) {
        ret = XMEMCMP(plain, inStr, plainLen);
    }

    /* Pass in bad args. */
   /* Tests RsaPrivateDecryptEx() which, is tested by another fn. No need dup.*/

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(plain, NULL);
    WC_FREE_VAR(cipher, NULL);
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaPublicEncryptDecrypt */

/*
 * Testing wc_RsaPrivateDecrypt_ex() and wc_RsaPrivateDecryptInline_ex()
 */
static int test_wc_RsaPublicEncryptDecrypt_ex(void)
{
    int result = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN) && !defined(HAVE_FIPS)\
        && !defined(WC_NO_RSA_OAEP) && !defined(HAVE_USER_RSA)\
        && !defined(NO_SHA256)
    RsaKey  key;
    WC_RNG  rng;
    int     ret;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 plainSz = (word32)TEST_STRING_SZ;
    byte*   res = NULL;
    int     idx = 0;
    int          bits = TEST_RSA_BITS;
    const word32 cipherSz = TEST_RSA_BYTES;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(cipher, byte, TEST_RSA_BYTES, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || plain == NULL || cipher == NULL) {
        fprintf(stderr, "test_wc_RsaPublicEncryptDecrypt_exmalloc failed\n");
        return TEST_FAIL;
    }
#endif
    XMEMCPY(in, inStr, inLen);

    /* Initialize stack structures. */
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRsaKey_ex(&key, HEAP_HINT, INVALID_DEVID);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng);
    }
    /* Encrypt */
    if (ret == 0) {
        ret = wc_RsaPublicEncrypt_ex(in, inLen, cipher, cipherSz, &key, &rng,
                WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        if (ret >= 0) {
            idx = ret;
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass bad args. */
    /* Tests RsaPublicEncryptEx again. No need duplicate. */
    if (ret != 0) {
        return TEST_FAIL;
    }

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    /* Decrypt */
    #if defined(WC_RSA_BLINDING) && !defined(HAVE_FIPS)
        if (ret == 0) {
            ret = wc_RsaSetRNG(&key, &rng);
        }
    #endif
    if (ret == 0) {
        ret = wc_RsaPrivateDecrypt_ex(cipher, (word32)idx,
                plain, plainSz, &key, WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256,
                WC_MGF1SHA256, NULL, 0);
    }
    if (ret >= 0) {
        if (!XMEMCMP(plain, inStr, plainSz)) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /*Pass bad args.*/
   /* Tests RsaPrivateDecryptEx() again. No need duplicate. */
    if (ret != 0) {
        return TEST_FAIL;
    }

    if (ret == 0) {
        ret = wc_RsaPrivateDecryptInline_ex(cipher, (word32)idx,
                &res, &key, WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256,
                WC_MGF1SHA256, NULL, 0);

        if (ret >= 0) {
            if (!XMEMCMP(inStr, res, plainSz)) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }
#endif

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(plain, NULL);
    WC_FREE_VAR(cipher, NULL);
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    result = TEST_RES_CHECK(ret == 0);
#endif
    return result;

} /* END test_wc_RsaPublicEncryptDecrypt_ex */

/*
 * Tesing wc_RsaSSL_Sign() and wc_RsaSSL_Verify()
 */
static int test_wc_RsaSSL_SignVerify(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey  key;
    WC_RNG  rng;
    int     ret = 0;
    const char inStr[] = TEST_STRING;
    const word32 plainSz = (word32)TEST_STRING_SZ;
    const word32 inLen = (word32)TEST_STRING_SZ;
    word32  idx = 0;
    int          bits = TEST_RSA_BITS;
    const word32 outSz = TEST_RSA_BYTES;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, NULL);
    WC_DECLARE_VAR(out, byte, TEST_RSA_BYTES, NULL);
    WC_DECLARE_VAR(plain, byte, TEST_STRING_SZ, NULL);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || plain == NULL) {
        fprintf(stderr, "test_wc_RsaSSL_SignVerify failed\n");
        return TEST_FAIL;
    }
#endif
    XMEMCPY(in, inStr, inLen);

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }

    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng);
    }
    /* Sign. */

    if (ret == 0) {
        ret = wc_RsaSSL_Sign(in, inLen, out, outSz, &key, &rng);
        if (ret == (int)outSz) {
            idx = ret;
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#ifndef HAVE_USER_RSA
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_RsaSSL_Sign(NULL, inLen, out, outSz, &key, &rng);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_RsaSSL_Sign(in, 0, out, outSz, &key, &rng);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_RsaSSL_Sign(in, inLen, NULL, outSz, &key, &rng);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_RsaSSL_Sign(in, inLen, out, outSz, NULL, &rng);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#else
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_RsaSSL_Sign(NULL, inLen, out, outSz, &key, &rng);
        if (ret == USER_CRYPTO_ERROR) {
            ret = wc_RsaSSL_Sign(in, 0, out, outSz, &key, &rng);
        }
        if (ret == USER_CRYPTO_ERROR) {
            ret = wc_RsaSSL_Sign(in, inLen, NULL, outSz, &key, &rng);
        }
        if (ret == USER_CRYPTO_ERROR) {
           ret = wc_RsaSSL_Sign(in, inLen, out, outSz, NULL, &rng);
        }
        if (ret == USER_CRYPTO_ERROR) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif
    if (ret != 0) {
        return TEST_FAIL;
    }

    /* Verify. */
    ret = wc_RsaSSL_Verify(out, idx, plain, plainSz, &key);
    if (ret == (int)inLen) {
        ret = 0;
    }
    else {
        ret = WOLFSSL_FATAL_ERROR;
    }
    #ifndef HAVE_USER_RSA
        /* Pass bad args. */
         if (ret == 0) {
                ret = wc_RsaSSL_Verify(NULL, idx, plain, plainSz, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaSSL_Verify(out, 0, plain, plainSz, &key);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaSSL_Verify(out, idx, NULL, plainSz, &key);
            }
            if (ret == BAD_FUNC_ARG) {
               ret = wc_RsaSSL_Verify(out, idx, plain, plainSz, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Pass bad args. */
         if (ret == 0) {
                ret = wc_RsaSSL_Verify(NULL, idx, plain, plainSz, &key);
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaSSL_Verify(out, 0, plain, plainSz, &key);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaSSL_Verify(out, idx, NULL, plainSz, &key);
            }
            if (ret == USER_CRYPTO_ERROR) {
               ret = wc_RsaSSL_Verify(out, idx, plain, plainSz, NULL);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    WC_FREE_VAR(in, NULL);
    WC_FREE_VAR(out, NULL);
    WC_FREE_VAR(plain, NULL);
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaSSL_SignVerify */

/*
 * Testing wc_RsaEncryptSize()
 */
static int test_wc_RsaEncryptSize(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey  key;
    WC_RNG  rng;
    int     ret;

    ret = wc_InitRsaKey(&key, HEAP_HINT);

    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }

#if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, 1024, WC_RSA_EXPONENT, &rng);
        if (ret == 0) {
            ret = wc_RsaEncryptSize(&key);
        }
        if (ret == 128) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    else {
        ret = 0;
    }
#endif

    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, 2048, WC_RSA_EXPONENT, &rng);
        if (ret == 0) {
            ret = wc_RsaEncryptSize(&key);
        }
        if (ret == 256) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Pass in bad arg. */
    if (ret == 0) {
        ret = wc_RsaEncryptSize(NULL);
        #ifndef HAVE_USER_RSA
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        #endif
    }

    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaEncryptSize*/

/*
 * Testing wc_RsaFlattenPublicKey()
 */
static int test_wc_RsaFlattenPublicKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey  key;
    WC_RNG  rng;
    int     ret = 0;
    byte    e[256];
    byte    n[256];
    word32  eSz = sizeof(e);
    word32  nSz = sizeof(n);
    #if (!defined(WOLFSSL_SP_MATH) || defined(WOLFSSL_SP_MATH_ALL)) && \
        (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 4))
    int         bits = 1024;
    #else
    int         bits = 2048;
    #endif

    ret = wc_InitRsaKey(&key, HEAP_HINT);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }

    if (ret == 0) {
        ret = MAKE_RSA_KEY(&key, bits, WC_RSA_EXPONENT, &rng);
        if (ret >= 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_RsaFlattenPublicKey(&key, e, &eSz, n, &nSz);
    }
    #ifndef HAVE_USER_RSA
        /* Pass bad args. */
        if (ret == 0) {
            ret = wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaFlattenPublicKey(&key, NULL, &eSz, n, &nSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaFlattenPublicKey(&key, e, NULL, n, &nSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaFlattenPublicKey(&key, e, &eSz, NULL, &nSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_RsaFlattenPublicKey(&key, e, &eSz, n, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #else
        /* Pass bad args. */
        if (ret == 0) {
            ret = wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz);
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaFlattenPublicKey(&key, NULL, &eSz, n, &nSz);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaFlattenPublicKey(&key, e, NULL, n, &nSz);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaFlattenPublicKey(&key, e, &eSz, NULL, &nSz);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = wc_RsaFlattenPublicKey(&key, e, &eSz, n, NULL);
            }
            if (ret == USER_CRYPTO_ERROR) {
                ret = 0;
            }
            else {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif
    if (wc_FreeRsaKey(&key) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (wc_FreeRng(&rng) || ret != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_RsaFlattenPublicKey */



/*
 * unit test for wc_AesCcmSetKey
 */
static int test_wc_AesCcmSetKey(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_AESCCM
    Aes aes;
    int ret = 0;
    const byte  key16[] =
    {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };
    const byte  key24[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
    };
    const byte  key32[] =
    {
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66
    };

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

#ifdef WOLFSSL_AES_128
    ret = wc_AesCcmSetKey(&aes, key16, sizeof(key16));
#endif
#ifdef WOLFSSL_AES_192
    if (ret == 0) {
        ret = wc_AesCcmSetKey(&aes, key24, sizeof(key24));
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = wc_AesCcmSetKey(&aes, key32, sizeof(key32));
    }
#endif

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_AesCcmSetKey(&aes, key16, sizeof(key16) - 1);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCcmSetKey(&aes, key24, sizeof(key24) - 1);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_AesCcmSetKey(&aes, key32, sizeof(key32) - 1);
        }
        if (ret != BAD_FUNC_ARG) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    wc_AesFree(&aes);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_AesCcmSetKey */

/*
 * Unit test function for wc_AesCcmEncrypt and wc_AesCcmDecrypt
 */
static int test_wc_AesCcmEncryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_AESCCM) && defined(WOLFSSL_AES_128)
    Aes aes;
    int ret = 0;
    const byte  key16[] =
    {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };
    /* plaintext */
    const byte plainT[] =
    {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e
    };
    /* nonce */
    const byte iv[] =
    {
        0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5
    };
    const byte c[] =  /* cipher text. */
    {
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84
    };
    const byte t[] =  /* Auth tag */
    {
        0x17, 0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0
    };
    const byte authIn[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };
    byte cipherOut[sizeof(plainT)];
    byte authTag[sizeof(t)];
    int ccmE = WOLFSSL_FATAL_ERROR;
    #ifdef HAVE_AES_DECRYPT
        int ccmD = WOLFSSL_FATAL_ERROR;
        byte plainOut[sizeof(cipherOut)];
    #endif

    ret = wc_AesInit(&aes, NULL, INVALID_DEVID);
    if (ret != 0)
        return ret;

    ret = wc_AesCcmSetKey(&aes, key16, sizeof(key16));
    if (ret == 0) {
        ccmE = wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv), authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        if ((XMEMCMP(cipherOut, c, sizeof(c)) && ccmE == 0) ||
                XMEMCMP(t, authTag, sizeof(t))) {
            ccmE = WOLFSSL_FATAL_ERROR;
            ret = WOLFSSL_FATAL_ERROR;
        }
        #ifdef HAVE_AES_DECRYPT
            if (ret == 0) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, cipherOut,
                                        sizeof(plainOut), iv, sizeof(iv),
                                        authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
                if (XMEMCMP(plainOut, plainT, sizeof(plainT)) && ccmD == 0) {
                    ccmD = WOLFSSL_FATAL_ERROR;
                }
            }
        #endif
    }

    /* Pass in bad args. Encrypt*/
    if (ret == 0 && ccmE == 0) {
        ccmE = wc_AesCcmEncrypt(NULL, cipherOut, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv), authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, NULL, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv), authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, cipherOut, NULL, sizeof(cipherOut),
                                    iv, sizeof(iv), authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
                                    NULL, sizeof(iv), authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv), NULL, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv) + 1, authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }
        if (ccmE == BAD_FUNC_ARG) {
            ccmE = wc_AesCcmEncrypt(&aes, cipherOut, plainT, sizeof(cipherOut),
                                    iv, sizeof(iv) - 7, authTag, sizeof(authTag),
                                    authIn , sizeof(authIn));
        }

        if (ccmE != BAD_FUNC_ARG) {
            ccmE = WOLFSSL_FATAL_ERROR;
        }
        else {
            ccmE = 0;
        }
    } /* End Encrypt */

    if (ccmE != 0) {
        wc_AesFree(&aes);
        return TEST_FAIL;
    }
    #ifdef HAVE_AES_DECRYPT
        /* Pass in bad args. Decrypt*/
        if (ret == 0 && ccmD == 0) {
            ccmD = wc_AesCcmDecrypt(NULL, plainOut, cipherOut, sizeof(plainOut),
                                        iv, sizeof(iv), authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, NULL, cipherOut, sizeof(plainOut),
                                        iv, sizeof(iv), authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            }
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, NULL, sizeof(plainOut),
                                        iv, sizeof(iv), authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            }
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, cipherOut,
                                        sizeof(plainOut), NULL, sizeof(iv),
                                        authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            }
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, cipherOut,
                                        sizeof(plainOut), iv, sizeof(iv), NULL,
                                        sizeof(authTag), authIn, sizeof(authIn));
            }
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, cipherOut,
                                        sizeof(plainOut), iv, sizeof(iv) + 1,
                                        authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            }
            if (ccmD == BAD_FUNC_ARG) {
                ccmD = wc_AesCcmDecrypt(&aes, plainOut, cipherOut,
                                        sizeof(plainOut), iv, sizeof(iv) - 7,
                                        authTag, sizeof(authTag),
                                        authIn, sizeof(authIn));
            }
            if (ccmD != BAD_FUNC_ARG) {
                ccmD = WOLFSSL_FATAL_ERROR;
            }
            else {
                ccmD = 0;
            }
        } /* END Decrypt */

        res = TEST_RES_CHECK(ccmD == 0);
    #endif

    wc_AesFree(&aes);
#endif  /* HAVE_AESCCM */
    return res;
} /* END test_wc_AesCcmEncryptDecrypt */


/*
 * Testing wc_InitDsaKey()
 */
static int test_wc_InitDsaKey(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DSA
    DsaKey  key;
    int     ret = 0;

    ret = wc_InitDsaKey(&key);

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_InitDsaKey(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_FreeDsaKey(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_InitDsaKey */

/*
 * Testing wc_DsaSign() and wc_DsaVerify()
 */
static int test_wc_DsaSignVerify(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA)
    DsaKey  key;
    WC_RNG  rng;
    wc_Sha  sha;
    int     ret = 0;
    byte    signature[DSA_SIG_SIZE];
    byte    hash[WC_SHA_DIGEST_SIZE];
    word32  idx = 0;
    word32  bytes;
    int      answer;
#ifdef USE_CERT_BUFFERS_1024
    byte    tmp[ONEK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte    tmp[TWOK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte    tmp[TWOK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XFILE fp = XFOPEN("./certs/dsa2048.der", "rb");
    if (fp == XBADFILE) {
        return WOLFSSL_BAD_FILE;
    }
    bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp);
    XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    ret = wc_InitSha(&sha);
    if (ret == 0) {
        ret = wc_ShaUpdate(&sha, tmp, bytes);
        if (ret == 0) {
            ret = wc_ShaFinal(&sha, hash);
        }
        if (ret == 0) {
            ret = wc_InitDsaKey(&key);
        }
        if (ret == 0) {
            ret = wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes);
        }
        if (ret == 0) {
            ret = wc_InitRng(&rng);
        }
    }

    /* Sign. */
    if (ret == 0) {
        ret = wc_DsaSign(hash, signature, &key, &rng);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_DsaSign(NULL, signature, &key, &rng);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaSign(hash, NULL, &key, &rng);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaSign(hash, signature, NULL, &rng);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaSign(hash, signature, &key, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        /* Verify. */
        ret = wc_DsaVerify(hash, signature, &key, &answer);
        if (ret != 0 || answer != 1) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        else {
            ret = 0;
        }
    }

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_DsaVerify(NULL, signature, &key, &answer);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaVerify(hash, NULL, &key, &answer);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaVerify(hash, signature, NULL, &answer);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaVerify(hash, signature, &key, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#if !defined(HAVE_FIPS) && defined(WOLFSSL_PUBLIC_MP)
    /* hard set q to 0 and test fail case */
    mp_free(&key.q);
    mp_init(&key.q);
    AssertIntEQ(wc_DsaSign(hash, signature, &key, &rng), BAD_FUNC_ARG);

    mp_set(&key.q, 1);
    AssertIntEQ(wc_DsaSign(hash, signature, &key, &rng), BAD_FUNC_ARG);
#endif

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    wc_FreeDsaKey(&key);
    wc_ShaFree(&sha);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_DsaSign */

/*
 * Testing wc_DsaPrivateKeyDecode() and wc_DsaPublicKeyDecode()
 */
static int test_wc_DsaPublicPrivateKeyDecode(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA)
    DsaKey  key;
    word32  bytes;
    word32  idx  = 0;
    int     priv = 0;
    int     pub  = 0;
    int     ret = 0;

#ifdef USE_CERT_BUFFERS_1024
    byte    tmp[ONEK_BUF];
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte    tmp[TWOK_BUF];
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte    tmp[TWOK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XFILE fp = XFOPEN("./certs/dsa2048.der", "rb");
    if (fp == XBADFILE)
    {
        return WOLFSSL_BAD_FILE;
    }
    bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp);
    XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    ret = wc_InitDsaKey(&key);

    if (ret == 0) {
        priv = wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes);

        /* Test bad args. */
        if (priv == 0) {
            priv = wc_DsaPrivateKeyDecode(NULL, &idx, &key, bytes);
            if (priv == BAD_FUNC_ARG) {
                priv = wc_DsaPrivateKeyDecode(tmp, NULL, &key, bytes);
            }
            if (priv == BAD_FUNC_ARG) {
                priv = wc_DsaPrivateKeyDecode(tmp, &idx, NULL, bytes);
            }
            if (priv == BAD_FUNC_ARG) {
                priv = wc_DsaPrivateKeyDecode(tmp, &idx, &key, bytes);
            }
            if (priv == ASN_PARSE_E || priv == BUFFER_E) {
                priv = 0;
            }
            else {
                priv = WOLFSSL_FATAL_ERROR;
            }
        }

        wc_FreeDsaKey(&key);
        ret = wc_InitDsaKey(&key);
    }

    if (ret == 0) {
        idx = 0; /* Reset */
        pub = wc_DsaPublicKeyDecode(tmp, &idx, &key, bytes);

        /* Test bad args. */
        if (pub == 0) {
            pub = wc_DsaPublicKeyDecode(NULL, &idx, &key, bytes);
            if (pub == BAD_FUNC_ARG) {
                pub = wc_DsaPublicKeyDecode(tmp, NULL, &key, bytes);
            }
            if (pub == BAD_FUNC_ARG) {
                pub = wc_DsaPublicKeyDecode(tmp, &idx, NULL, bytes);
            }
            if (pub == BAD_FUNC_ARG) {
                pub = wc_DsaPublicKeyDecode(tmp, &idx, &key, bytes);
            }
            if (pub == ASN_PARSE_E || pub == BUFFER_E) {
                pub = 0;
            }
            else {
                pub = WOLFSSL_FATAL_ERROR;
            }
        }

    } /* END Public Key */

    wc_FreeDsaKey(&key);

    res = TEST_RES_CHECK(ret == 0 && pub == 0 && priv == 0);
#endif /* !NO_DSA */
    return res;

} /* END test_wc_DsaPublicPrivateKeyDecode */


/*
 * Testing wc_MakeDsaKey() and wc_MakeDsaParameters()
 */
static int test_wc_MakeDsaKey(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey  genKey;
    WC_RNG  rng;
    int     ret = 0;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&genKey, 0, sizeof(genKey));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_InitDsaKey(&genKey);
    }

    if (ret == 0) {
        ret = wc_MakeDsaParameters(&rng, ONEK_BUF, &genKey);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_MakeDsaParameters(NULL, ONEK_BUF, &genKey);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_MakeDsaParameters(&rng, ONEK_BUF, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_MakeDsaParameters(&rng, ONEK_BUF + 1, &genKey);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_MakeDsaKey(&rng, &genKey);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_MakeDsaKey(NULL, &genKey);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_MakeDsaKey(&rng, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FAILURE;
    }

    wc_FreeDsaKey(&genKey);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_MakeDsaKey */

/*
 * Testing wc_DsaKeyToDer()
 */
static int test_wc_DsaKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey  genKey;
    WC_RNG  rng;
    word32  bytes;
    word32  idx = 0;
    int     ret = 0;
#ifdef USE_CERT_BUFFERS_1024
    byte    tmp[ONEK_BUF];
    byte    der[ONEK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte    tmp[TWOK_BUF];
    byte    der[TWOK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte    tmp[TWOK_BUF];
    byte    der[TWOK_BUF];
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMSET(der, 0, sizeof(der));
    XFILE fp = XFOPEN("./certs/dsa2048.der", "rb");
    if (fp == XBADFILE) {
        return WOLFSSL_BAD_FILE;
    }
    bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp);
    XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&genKey, 0, sizeof(genKey));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_InitDsaKey(&genKey);
    }
    if (ret == 0) {
        ret = wc_MakeDsaParameters(&rng, sizeof(tmp), &genKey);
        if (ret == 0) {
            wc_FreeDsaKey(&genKey);
            ret = wc_InitDsaKey(&genKey);
        }
    }
    if (ret == 0) {
        ret = wc_DsaPrivateKeyDecode(tmp, &idx, &genKey, bytes);
    }

    if (ret == 0) {
        ret = wc_DsaKeyToDer(&genKey, der, bytes);
        if ( ret >= 0 && ( ret = XMEMCMP(der, tmp, bytes) ) == 0 ) {
            ret = 0;
        }
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_DsaKeyToDer(NULL, der, FOURK_BUF);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaKeyToDer(&genKey, NULL, FOURK_BUF);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    wc_FreeDsaKey(&genKey);

    res = TEST_RES_CHECK(ret == 0);
#endif /* !NO_DSA && WOLFSSL_KEY_GEN */
    return res;

} /* END test_wc_DsaKeyToDer */

/*
 *  Testing wc_DsaKeyToPublicDer()
 *  (indirectly testing setDsaPublicKey())
 */
static int test_wc_DsaKeyToPublicDer(void)
{
    int res = TEST_SKIPPED;
#ifndef HAVE_SELFTEST
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey  genKey;
    WC_RNG  rng;
    byte*   der;
    word32  sz;
    int     ret = 0;

    der = (byte*)XMALLOC(ONEK_BUF, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = wc_InitDsaKey(&genKey);
    }
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_MakeDsaParameters(&rng, ONEK_BUF, &genKey);
    }
    if (ret == 0) {
        ret = wc_MakeDsaKey(&rng, &genKey);
    }

    if (ret == 0) {
        ret = wc_DsaKeyToPublicDer(&genKey, der, ONEK_BUF);
        if (ret >= 0) {
            sz = ret;
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (ret == 0) {
        word32 idx = 0;
        wc_FreeDsaKey(&genKey);
        ret = wc_DsaPublicKeyDecode(der, &idx, &genKey, sz);
    }
    /* Test without the SubjectPublicKeyInfo header */
    if (ret == 0) {
        ret = wc_SetDsaPublicKey(der, &genKey, ONEK_BUF, 0);
        if (ret >= 0) {
            sz = ret;
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (ret == 0) {
        word32 idx = 0;
        wc_FreeDsaKey(&genKey);
        ret = wc_DsaPublicKeyDecode(der, &idx, &genKey, sz);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_DsaKeyToPublicDer(NULL, der, FOURK_BUF);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_DsaKeyToPublicDer(&genKey, NULL, FOURK_BUF);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    XFREE(der, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    wc_FreeDsaKey(&genKey);

    res = TEST_RES_CHECK(ret == 0);
#endif /* !NO_DSA && WOLFSSL_KEY_GEN */
#endif /* !HAVE_SELFTEST */
    return res;

} /* END test_wc_DsaKeyToPublicDer */

/*
 * Testing wc_DsaImportParamsRaw()
 */
static int test_wc_DsaImportParamsRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA)
    DsaKey  key;
    int     ret = 0;

    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";

    /* invalid p and q parameters */
    const char* invalidP = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d";
    const char* invalidQ = "96c5390a";

    ret = wc_InitDsaKey(&key);
    if (ret == 0) {
        ret = wc_DsaImportParamsRaw(&key, p, q, g);
    }

    /* test bad args */
    if (ret == 0) {
        /* null key struct */
        ret = wc_DsaImportParamsRaw(NULL, p, q, g);
        if (ret == BAD_FUNC_ARG) {
            /* null param pointers */
            ret = wc_DsaImportParamsRaw(&key, NULL, NULL, NULL);
        }

        if (ret == BAD_FUNC_ARG) {
            /* illegal p length */
            ret = wc_DsaImportParamsRaw(&key, invalidP, q, g);
        }

        if (ret == BAD_FUNC_ARG) {
            /* illegal q length */
            ret = wc_DsaImportParamsRaw(&key, p, invalidQ, g);
            if (ret == BAD_FUNC_ARG)
                ret = 0;
        }

    }

    wc_FreeDsaKey(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_DsaImportParamsRaw */

/*
 * Testing wc_DsaImportParamsRawCheck()
 */
static int test_wc_DsaImportParamsRawCheck(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    DsaKey  key;
    int     ret = 0;
    int     trusted = 0;
    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";

    /* invalid p and q parameters */
    const char* invalidP = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d";
    const char* invalidQ = "96c5390a";

    ret = wc_InitDsaKey(&key);
    if (ret == 0) {
        ret = wc_DsaImportParamsRawCheck(&key, p, q, g, trusted, NULL);
    }

    /* test bad args */
    if (ret == 0) {
        /* null key struct */
        ret = wc_DsaImportParamsRawCheck(NULL, p, q, g, trusted, NULL);
        if (ret == BAD_FUNC_ARG) {
            /* null param pointers */
            ret = wc_DsaImportParamsRawCheck(&key, NULL, NULL, NULL, trusted, NULL);
        }

        if (ret == BAD_FUNC_ARG) {
            /* illegal p length */
            ret = wc_DsaImportParamsRawCheck(&key, invalidP, q, g, trusted, NULL);
        }

        if (ret == BAD_FUNC_ARG) {
            /* illegal q length */
            ret = wc_DsaImportParamsRawCheck(&key, p, invalidQ, g, trusted, NULL);
            if (ret == BAD_FUNC_ARG)
                ret = 0;
        }

    }

    wc_FreeDsaKey(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_DsaImportParamsRawCheck */

/*
 * Testing wc_DsaExportParamsRaw()
 */
static int test_wc_DsaExportParamsRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA)
    DsaKey  key;
    int     ret = 0;

    /* [mod = L=1024, N=160], from CAVP KeyPair */
    const char* p = "d38311e2cd388c3ed698e82fdf88eb92b5a9a483dc88005d"
                    "4b725ef341eabb47cf8a7a8a41e792a156b7ce97206c4f9c"
                    "5ce6fc5ae7912102b6b502e59050b5b21ce263dddb2044b6"
                    "52236f4d42ab4b5d6aa73189cef1ace778d7845a5c1c1c71"
                    "47123188f8dc551054ee162b634d60f097f719076640e209"
                    "80a0093113a8bd73";
    const char* q = "96c5390a8b612c0e422bb2b0ea194a3ec935a281";
    const char* g = "06b7861abbd35cc89e79c52f68d20875389b127361ca66822"
                    "138ce4991d2b862259d6b4548a6495b195aa0e0b6137ca37e"
                    "b23b94074d3c3d300042bdf15762812b6333ef7b07ceba786"
                    "07610fcc9ee68491dbc1e34cd12615474e52b18bc934fb00c"
                    "61d39e7da8902291c4434a4e2224c3f4fd9f93cd6f4f17fc0"
                    "76341a7e7d9";

    const char* pCompare = "\xd3\x83\x11\xe2\xcd\x38\x8c\x3e\xd6\x98\xe8\x2f"
                           "\xdf\x88\xeb\x92\xb5\xa9\xa4\x83\xdc\x88\x00\x5d"
                           "\x4b\x72\x5e\xf3\x41\xea\xbb\x47\xcf\x8a\x7a\x8a"
                           "\x41\xe7\x92\xa1\x56\xb7\xce\x97\x20\x6c\x4f\x9c"
                           "\x5c\xe6\xfc\x5a\xe7\x91\x21\x02\xb6\xb5\x02\xe5"
                           "\x90\x50\xb5\xb2\x1c\xe2\x63\xdd\xdb\x20\x44\xb6"
                           "\x52\x23\x6f\x4d\x42\xab\x4b\x5d\x6a\xa7\x31\x89"
                           "\xce\xf1\xac\xe7\x78\xd7\x84\x5a\x5c\x1c\x1c\x71"
                           "\x47\x12\x31\x88\xf8\xdc\x55\x10\x54\xee\x16\x2b"
                           "\x63\x4d\x60\xf0\x97\xf7\x19\x07\x66\x40\xe2\x09"
                           "\x80\xa0\x09\x31\x13\xa8\xbd\x73";
    const char* qCompare = "\x96\xc5\x39\x0a\x8b\x61\x2c\x0e\x42\x2b\xb2\xb0"
                           "\xea\x19\x4a\x3e\xc9\x35\xa2\x81";
    const char* gCompare = "\x06\xb7\x86\x1a\xbb\xd3\x5c\xc8\x9e\x79\xc5\x2f"
                           "\x68\xd2\x08\x75\x38\x9b\x12\x73\x61\xca\x66\x82"
                           "\x21\x38\xce\x49\x91\xd2\xb8\x62\x25\x9d\x6b\x45"
                           "\x48\xa6\x49\x5b\x19\x5a\xa0\xe0\xb6\x13\x7c\xa3"
                           "\x7e\xb2\x3b\x94\x07\x4d\x3c\x3d\x30\x00\x42\xbd"
                           "\xf1\x57\x62\x81\x2b\x63\x33\xef\x7b\x07\xce\xba"
                           "\x78\x60\x76\x10\xfc\xc9\xee\x68\x49\x1d\xbc\x1e"
                           "\x34\xcd\x12\x61\x54\x74\xe5\x2b\x18\xbc\x93\x4f"
                           "\xb0\x0c\x61\xd3\x9e\x7d\xa8\x90\x22\x91\xc4\x43"
                           "\x4a\x4e\x22\x24\xc3\xf4\xfd\x9f\x93\xcd\x6f\x4f"
                           "\x17\xfc\x07\x63\x41\xa7\xe7\xd9";

    byte pOut[MAX_DSA_PARAM_SIZE];
    byte qOut[MAX_DSA_PARAM_SIZE];
    byte gOut[MAX_DSA_PARAM_SIZE];
    word32 pOutSz, qOutSz, gOutSz;

    ret = wc_InitDsaKey(&key);
    if (ret == 0) {
        /* first test using imported raw parameters, for expected */
        ret = wc_DsaImportParamsRaw(&key, p, q, g);
    }

    if (ret == 0) {
        pOutSz = sizeof(pOut);
        qOutSz = sizeof(qOut);
        gOutSz = sizeof(gOut);
        ret = wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz,
                                    gOut, &gOutSz);
    }

    if (ret == 0) {
        /* validate exported parameters are correct */
        if ((XMEMCMP(pOut, pCompare, pOutSz) != 0) ||
            (XMEMCMP(qOut, qCompare, qOutSz) != 0) ||
            (XMEMCMP(gOut, gCompare, gOutSz) != 0) ) {
            ret = -1;
        }
    }

    /* test bad args */
    if (ret == 0) {
        /* null key struct */
        ret = wc_DsaExportParamsRaw(NULL, pOut, &pOutSz, qOut, &qOutSz,
                                    gOut, &gOutSz);

        if (ret == BAD_FUNC_ARG) {
            /* null output pointers */
            ret = wc_DsaExportParamsRaw(&key, NULL, &pOutSz, NULL, &qOutSz,
                                        NULL, &gOutSz);
        }

        if (ret == LENGTH_ONLY_E) {
            /* null output size pointers */
            ret = wc_DsaExportParamsRaw(&key, pOut, NULL, qOut, NULL,
                                        gOut, NULL);
        }

        if (ret == BAD_FUNC_ARG) {
            /* p output buffer size too small */
            pOutSz = 1;
            ret = wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz,
                                        gOut, &gOutSz);
            pOutSz = sizeof(pOut);
        }

        if (ret == BUFFER_E) {
            /* q output buffer size too small */
            qOutSz = 1;
            ret = wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz,
                                        gOut, &gOutSz);
            qOutSz = sizeof(qOut);
        }

        if (ret == BUFFER_E) {
            /* g output buffer size too small */
            gOutSz = 1;
            ret = wc_DsaExportParamsRaw(&key, pOut, &pOutSz, qOut, &qOutSz,
                                        gOut, &gOutSz);
            if (ret == BUFFER_E)
                ret = 0;
        }
    }

    wc_FreeDsaKey(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_DsaExportParamsRaw */

/*
 * Testing wc_DsaExportKeyRaw()
 */
static int test_wc_DsaExportKeyRaw(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_DSA) && defined(WOLFSSL_KEY_GEN)
    DsaKey  key;
    WC_RNG  rng;
    int     ret = 0;

    byte xOut[MAX_DSA_PARAM_SIZE];
    byte yOut[MAX_DSA_PARAM_SIZE];
    word32 xOutSz, yOutSz;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_InitDsaKey(&key);
    }

    if (ret == 0) {
        ret = wc_MakeDsaParameters(&rng, 1024, &key);

        if (ret == 0)  {
            ret = wc_MakeDsaKey(&rng, &key);
        }
    }

    /* try successful export */
    if (ret == 0) {
        xOutSz = sizeof(xOut);
        yOutSz = sizeof(yOut);
        ret = wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz);
    }

    /* test bad args */
    if (ret == 0) {
        /* null key struct */
        ret = wc_DsaExportKeyRaw(NULL, xOut, &xOutSz, yOut, &yOutSz);

        if (ret == BAD_FUNC_ARG) {
            /* null output pointers */
            ret = wc_DsaExportKeyRaw(&key, NULL, &xOutSz, NULL, &yOutSz);
        }

        if (ret == LENGTH_ONLY_E) {
            /* null output size pointers */
            ret = wc_DsaExportKeyRaw(&key, xOut, NULL, yOut, NULL);
        }

        if (ret == BAD_FUNC_ARG) {
            /* x output buffer size too small */
            xOutSz = 1;
            ret = wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz);
            xOutSz = sizeof(xOut);
        }

        if (ret == BUFFER_E) {
            /* y output buffer size too small */
            yOutSz = 1;
            ret = wc_DsaExportKeyRaw(&key, xOut, &xOutSz, yOut, &yOutSz);

            if (ret == BUFFER_E)
                ret = 0;
        }
    }

    wc_FreeDsaKey(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_DsaExportParamsRaw */


/*
 * Testing wc_ed25519_make_key().
 */
static int test_wc_ed25519_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519)
    ed25519_key     key;
    WC_RNG          rng;
    unsigned char   pubkey[ED25519_PUB_KEY_SIZE];
    int ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed25519_init(&key);
    }
    if (ret == 0) {
        ret = wc_ed25519_make_public(&key, pubkey, sizeof(pubkey));
        if (ret == ECC_PRIV_KEY_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = -1;
        }
    }
    if (ret == 0) {
        ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed25519_make_key(NULL, ED25519_KEY_SIZE, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE - 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE + 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }


    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_make_key */


/*
 * Testing wc_ed25519_init()
 */
static int test_wc_ed25519_init(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519)
    ed25519_key    key;
    int             ret = 0;

    ret = wc_ed25519_init(&key);

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed25519_init(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_init */

/*
 * Test wc_ed25519_sign_msg() and wc_ed25519_verify_msg()
 */
static int test_wc_ed25519_sign_msg(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_SIGN)
    WC_RNG          rng;
    ed25519_key     key;
    int             ret = 0;
    byte            msg[] = "Everybody gets Friday off.\n";
    byte            sig[ED25519_SIG_SIZE];
    word32          msglen = sizeof(msg);
    word32          siglen = sizeof(sig);
    word32          badSigLen = sizeof(sig) - 1;
#ifdef HAVE_ED25519_VERIFY
    int             verify_ok = 0; /*1 = Verify success.*/
#endif

    /* Initialize stack variables. */
    XMEMSET(sig, 0, siglen);

    /* Initialize key. */
    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed25519_init(&key);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
        }
    }

    if (ret == 0) {
        ret = wc_ed25519_sign_msg(msg, msglen, sig, &siglen, &key);
    }
    /* Test bad args. */
    if (ret == 0 && siglen == ED25519_SIG_SIZE) {
        ret = wc_ed25519_sign_msg(NULL, msglen, sig, &siglen, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_sign_msg(msg, msglen, NULL, &siglen, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_sign_msg(msg, msglen, sig, NULL, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_sign_msg(msg, msglen, sig, &siglen, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_sign_msg(msg, msglen, sig, &badSigLen, &key);
        }
        if (ret == BUFFER_E && badSigLen == ED25519_SIG_SIZE) {
            badSigLen -= 1;
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    } /* END sign */

#ifdef HAVE_ED25519_VERIFY
    if (ret == 0) {

        ret = wc_ed25519_verify_msg(sig, siglen, msg, msglen, &verify_ok, &key);
        if (ret == 0  && verify_ok == 1) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }

        /* Test bad args. */
        if (ret == 0) {
            AssertIntEQ(wc_ed25519_verify_msg(sig, siglen - 1, msg,
                                              msglen, &verify_ok, &key),
                                              BAD_FUNC_ARG);
            AssertIntEQ(wc_ed25519_verify_msg(sig, siglen + 1, msg,
                                              msglen, &verify_ok, &key),
                                              BAD_FUNC_ARG);

            ret = wc_ed25519_verify_msg(NULL, siglen, msg, msglen, &verify_ok,
                                                                    &key);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_verify_msg(sig, siglen, NULL, msglen,
                                                            &verify_ok, &key);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_verify_msg(sig, siglen, msg, msglen,
                                                              NULL, &key);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_verify_msg(sig, siglen, msg, msglen,
                                                            &verify_ok, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_verify_msg(sig, badSigLen, msg, msglen,
                                                            &verify_ok, &key);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }

    } /* END verify. */
#endif /* Verify. */

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_ed25519_sign_msg */

/*
 * Testing wc_ed25519_import_public()
 */
static int test_wc_ed25519_import_public(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    WC_RNG          rng;
    ed25519_key     pubKey;
    const byte      in[] = "Ed25519PublicKeyUnitTest......\n";
    word32          inlen = sizeof(in);
    int             ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed25519_init(&pubKey);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &pubKey);
        }
    }

    if (ret == 0) {
        ret = wc_ed25519_import_public_ex(in, inlen, &pubKey, 1);

        if (ret == 0 && XMEMCMP(in, pubKey.p, inlen) == 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }

        /* Test bad args. */
        if (ret == 0) {
            ret = wc_ed25519_import_public(NULL, inlen, &pubKey);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_import_public(in, inlen, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_import_public(in, inlen - 1, &pubKey);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&pubKey);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END wc_ed25519_import_public */

/*
 * Testing wc_ed25519_import_private_key()
 */
static int test_wc_ed25519_import_private_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    WC_RNG      rng;
    ed25519_key key;
    int         ret;
    const byte  privKey[] = "Ed25519PrivateKeyUnitTest.....\n";
    const byte  pubKey[] = "Ed25519PublicKeyUnitTest......\n";
    word32      privKeySz = sizeof(privKey);
    word32      pubKeySz = sizeof(pubKey);
#ifdef HAVE_ED25519_KEY_EXPORT
    byte        bothKeys[sizeof(privKey) + sizeof(pubKey)];
    word32      bothKeysSz = sizeof(bothKeys);
#endif

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return ret;
    }
    ret = wc_ed25519_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return ret;
    }
    ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);

    if (ret == 0) {
        ret = wc_ed25519_import_private_key_ex(privKey, privKeySz, pubKey,
                                                             pubKeySz, &key, 1);
        if (ret == 0 && (XMEMCMP(pubKey, key.p, privKeySz) != 0
                                || XMEMCMP(privKey, key.k, pubKeySz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#ifdef HAVE_ED25519_KEY_EXPORT
    if (ret == 0)
        ret = wc_ed25519_export_private(&key, bothKeys, &bothKeysSz);

    if (ret == 0) {
        ret = wc_ed25519_import_private_key_ex(bothKeys, bothKeysSz, NULL, 0,
            &key, 1);
        if (ret == 0 && (XMEMCMP(pubKey, key.p, privKeySz) != 0
                                || XMEMCMP(privKey, key.k, pubKeySz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed25519_import_private_key(NULL, privKeySz, pubKey, pubKeySz,
                                                                        &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_import_private_key(privKey, privKeySz, NULL,
                                                                pubKeySz, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_import_private_key(privKey, privKeySz, pubKey,
                                                                pubKeySz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_import_private_key(privKey, privKeySz - 1, pubKey,
                                                                pubKeySz, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_import_private_key(privKey, privKeySz, pubKey,
                                                            pubKeySz - 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_import_private_key(privKey, privKeySz, NULL,
                                                            0, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_import_private_key */

/*
 * Testing wc_ed25519_export_public() and wc_ed25519_export_private_only()
 */
static int test_wc_ed25519_export(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    WC_RNG          rng;
    ed25519_key     key;
    int             ret = 0;
    byte            priv[ED25519_PRV_KEY_SIZE];
    byte            pub[ED25519_PUB_KEY_SIZE];
    word32          privSz = sizeof(priv);
    word32          pubSz = sizeof(pub);

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return ret;
    }

    ret = wc_ed25519_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return ret;
    }

    if (ret == 0) {
        ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
    }

    if (ret == 0) {
        ret = wc_ed25519_export_public(&key, pub, &pubSz);
        if (ret == 0 && (pubSz != ED25519_KEY_SIZE
                                        || XMEMCMP(key.p, pub, pubSz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        if (ret == 0) {
            ret = wc_ed25519_export_public(NULL, pub, &pubSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_public(&key, NULL, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_public(&key, pub, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (ret == 0) {
        ret = wc_ed25519_export_private_only(&key, priv, &privSz);
        if (ret == 0 && (privSz != ED25519_KEY_SIZE
                                        || XMEMCMP(key.k, priv, privSz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        if (ret == 0) {
            ret = wc_ed25519_export_private_only(NULL, priv, &privSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_private_only(&key, NULL, &privSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_private_only(&key, priv, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_export */

/*
 *  Testing wc_ed25519_size()
 */
static int test_wc_ed25519_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519)
    WC_RNG          rng;
    ed25519_key     key;
    int             ret;

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return ret;
    }
    ret = wc_ed25519_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return ret;
    }

    ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        wc_ed25519_free(&key);
        return ret;
    }

    ret = wc_ed25519_size(&key);
    /* Test bad args. */
    if (ret == ED25519_KEY_SIZE) {
        ret = wc_ed25519_size(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        ret = wc_ed25519_sig_size(&key);
        if (ret == ED25519_SIG_SIZE) {
            ret = 0;
        }
        /* Test bad args. */
        if (ret == 0) {
            ret = wc_ed25519_sig_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed25519_sig_size() */

    if (ret == 0) {
        ret = wc_ed25519_pub_size(&key);
        if (ret == ED25519_PUB_KEY_SIZE) {
            ret = 0;
        }
        if (ret == 0) {
            ret = wc_ed25519_pub_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed25519_pub_size */

    if (ret == 0) {
        ret = wc_ed25519_priv_size(&key);
        if (ret == ED25519_PRV_KEY_SIZE) {
            ret = 0;
        }
        if (ret == 0) {
            ret = wc_ed25519_priv_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed25519_pub_size */

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_size */

/*
 * Testing wc_ed25519_export_private() and wc_ed25519_export_key()
 */
static int test_wc_ed25519_exportKey(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    WC_RNG          rng;
    ed25519_key     key;
    int             ret = 0;
    byte            priv[ED25519_PRV_KEY_SIZE];
    byte            pub[ED25519_PUB_KEY_SIZE];
    byte            privOnly[ED25519_PRV_KEY_SIZE];
    word32          privSz      = sizeof(priv);
    word32          pubSz       = sizeof(pub);
    word32          privOnlySz  = sizeof(privOnly);

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return TEST_FAIL;
    }
    ret = wc_ed25519_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        wc_ed25519_free(&key);
        return TEST_FAIL;
    }

    ret = wc_ed25519_export_private(&key, privOnly, &privOnlySz);
    if (ret == 0) {
        ret = wc_ed25519_export_private(NULL, privOnly, &privOnlySz);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_export_private(&key, NULL, &privOnlySz);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed25519_export_private(&key, privOnly, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_ed25519_export_key(&key, priv, &privSz, pub, &pubSz);
        if (ret == 0) {
            ret = wc_ed25519_export_key(NULL, priv, &privSz, pub, &pubSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_key(&key, NULL, &privSz, pub, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_key(&key, priv, NULL, pub, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_key(&key, priv, &privSz, NULL, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed25519_export_key(&key, priv, &privSz, pub, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    } /* END wc_ed25519_export_key() */

    /* Cross check output. */
    if (ret == 0 && XMEMCMP(priv, privOnly, privSz) != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed25519_exportKey */

/*
 * Testing wc_Ed25519PublicKeyToDer
 */
static int test_wc_Ed25519PublicKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    int tmp;
    ed25519_key key;
    byte derBuf[1024];
    int ret = 0;

    /* Test bad args */
    tmp = wc_Ed25519PublicKeyToDer(NULL, NULL, 0, 0);
    if (tmp != BAD_FUNC_ARG) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0) {
        wc_ed25519_init(&key);
        tmp = wc_Ed25519PublicKeyToDer(&key, derBuf, 0, 0);
        if (tmp != BUFFER_E) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        wc_ed25519_free(&key);
    }

    /*  Test good args */
    if (ret == 0) {
        WC_RNG          rng;
        ret = wc_InitRng(&rng);
        if (ret != 0) {
            return TEST_FAIL;
        }
        ret = wc_ed25519_init(&key);
        if (ret != 0) {
            wc_FreeRng(&rng);
            return TEST_FAIL;
        }

        ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
        if (ret != 0) {
            wc_FreeRng(&rng);
            wc_ed25519_free(&key);
            return TEST_FAIL;
        }

        tmp = wc_Ed25519PublicKeyToDer(&key, derBuf, 1024, 1);
        if (tmp <= 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }

        wc_FreeRng(&rng);
        wc_ed25519_free(&key);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END testing wc_Ed25519PublicKeyToDer */

/*
 * Testing wc_curve25519_init and wc_curve25519_free.
 */
static int test_wc_curve25519_init(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    curve25519_key  key;
    int ret = 0;

    ret = wc_curve25519_init(&key);
    /* Test bad args for wc_curve25519_init */
    if (ret == 0) {
        ret = wc_curve25519_init(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /*  Test good args for wc_curve_25519_free */
    wc_curve25519_free(&key);

    wc_curve25519_free(NULL);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_curve25519_init and wc_curve_25519_free*/
/*
 * Testing test_wc_curve25519_size.
 */
static int test_wc_curve25519_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    curve25519_key  key;
    int ret = 0;

    ret = wc_curve25519_init(&key);

    /*  Test good args for wc_curve25519_size */
    if (ret == 0) {
        ret = wc_curve25519_size(&key);
    }

    /* Test bad args for wc_curve25519_size */
    if (ret != 0) {
        ret = wc_curve25519_size(NULL);
    }

    wc_curve25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_curve25519_size*/

/*
 * Testing test_wc_curve25519_export_key_raw().
 */
static int test_wc_curve25519_export_key_raw(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519) && defined(HAVE_CURVE25519_KEY_EXPORT)
    curve25519_key  key;
    WC_RNG          rng;
    int             ret = 0;

    byte            privateKey[CURVE25519_KEYSIZE];
    byte            publicKey[CURVE25519_KEYSIZE];
    word32          prvkSz;
    word32          pubkSz;

    byte            prik[CURVE25519_KEYSIZE];
    byte            pubk[CURVE25519_KEYSIZE];
    word32          prksz;
    word32          pbksz;

    if (0 != wc_InitRng(&rng)) {
        return TEST_FAIL;
    }
    if (0 != wc_curve25519_init(&key)) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);

    /*
        bad-argument-test cases
        target function sould return BAD_FUNC_ARG
     */
    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw(
                 NULL, privateKey, &prvkSz, publicKey, &pubkSz)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw(
                &key, NULL, &prvkSz, publicKey, &pubkSz)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw(
                &key, privateKey, NULL, publicKey, &pubkSz)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        /* prvkSz = CURVE25519_KEYSIZE; */
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw(
                &key, privateKey, &prvkSz, NULL, &pubkSz)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw(
                &key, privateKey, &prvkSz, publicKey, NULL )) {
            ret = -1;
        }
    }

    /*
        cross-testing
    */
    if (ret == 0) {
        prksz = CURVE25519_KEYSIZE;
        ret = wc_curve25519_export_private_raw(&key, prik, &prksz);
    }

    if (ret == 0) {
        pbksz = CURVE25519_KEYSIZE;
        ret = wc_curve25519_export_public(&key, pubk, &pbksz);
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        /* pubkSz = CURVE25519_KEYSIZE; */
        ret = wc_curve25519_export_key_raw(&key, privateKey, &prvkSz,
                                                     publicKey,  &pubkSz);
    }

    if (ret == 0) {
        if ((prksz  == CURVE25519_KEYSIZE) &&
            (pbksz  == CURVE25519_KEYSIZE) &&
            (prvkSz == CURVE25519_KEYSIZE) &&
            (pubkSz == CURVE25519_KEYSIZE)) {

            if (0 != XMEMCMP(privateKey, prik, CURVE25519_KEYSIZE) ||
                0 != XMEMCMP(publicKey,  pubk, CURVE25519_KEYSIZE)) {
                ret = -1;
            }
        }
    }

    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* end of test_wc_curve25519_export_key_raw */

/*
 * Testing test_wc_curve25519_export_key_raw_ex().
 */
static int test_wc_curve25519_export_key_raw_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519) && defined(HAVE_CURVE25519_KEY_EXPORT)
    curve25519_key  key;
    WC_RNG          rng;
    int             ret;

    byte            privateKey[CURVE25519_KEYSIZE];
    byte            publicKey[CURVE25519_KEYSIZE];
    word32          prvkSz;
    word32          pubkSz;

    byte            prik[CURVE25519_KEYSIZE];
    byte            pubk[CURVE25519_KEYSIZE];
    word32          prksz;
    word32          pbksz;

    if (0 != wc_InitRng(&rng)) {
        return TEST_FAIL;
    }
    if (0 != wc_curve25519_init(&key)) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);

    /*
        bad-argument-test cases
        target function sould return BAD_FUNC_ARG
    */
    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( NULL , privateKey,
                          &prvkSz, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key , NULL,
                          &prvkSz, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key,privateKey,
                             NULL, publicKey, &pubkSz, EC25519_LITTLE_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        /* prvkSz = CURVE25519_KEYSIZE; */
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, privateKey,
                               &prvkSz, NULL, &pubkSz, EC25519_LITTLE_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, privateKey,
                             &prvkSz, publicKey, NULL, EC25519_LITTLE_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        /* pubkSz = CURVE25519_KEYSIZE; */
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( NULL, privateKey,
                             &prvkSz, publicKey, &pubkSz, EC25519_BIG_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, NULL,
                             &prvkSz, publicKey, &pubkSz, EC25519_BIG_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, privateKey,
                                NULL, publicKey, &pubkSz, EC25519_BIG_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        /* prvkSz = CURVE25519_KEYSIZE; */
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, privateKey,
                                  &prvkSz, NULL, &pubkSz, EC25519_BIG_ENDIAN)) {
            ret = -1;
        }
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex( &key, privateKey,
                                &prvkSz, publicKey, NULL, EC25519_BIG_ENDIAN)) {
            ret = -1;
        }
    }

    /* illegal value for endien */

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        /* pubkSz = CURVE25519_KEYSIZE; */
        if (BAD_FUNC_ARG != wc_curve25519_export_key_raw_ex(&key, privateKey,
                           &prvkSz, publicKey, NULL, EC25519_BIG_ENDIAN + 10)) {
            ret = -1;
        }
    }

    /*
        cross-testing
    */
    if (ret == 0) {
        prksz = CURVE25519_KEYSIZE;
        ret = wc_curve25519_export_private_raw( &key, prik, &prksz);
    }

    if (ret == 0) {
        pbksz = CURVE25519_KEYSIZE;
        ret = wc_curve25519_export_public( &key, pubk, &pbksz);
    }

    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        /* pubkSz = CURVE25519_KEYSIZE; */
        ret = wc_curve25519_export_key_raw_ex( &key, privateKey, &prvkSz,
                                        publicKey, &pubkSz, EC25519_BIG_ENDIAN);
    }

    if (ret == 0 && (prksz  != CURVE25519_KEYSIZE ||
                     pbksz  != CURVE25519_KEYSIZE ||
                     prvkSz != CURVE25519_KEYSIZE ||
                     pubkSz != CURVE25519_KEYSIZE)) {
        ret = -1;
    }

    if (ret == 0 && (0 != XMEMCMP(privateKey, prik, CURVE25519_KEYSIZE) ||
                     0 != XMEMCMP(publicKey,  pubk, CURVE25519_KEYSIZE))) {
        ret = -1;
    }

    if (ret == 0) {
        ret = wc_curve25519_export_key_raw_ex(&key, privateKey, &prvkSz,
                                     publicKey, &pubkSz, EC25519_LITTLE_ENDIAN);
    }
    if (ret == 0 && (prvkSz != CURVE25519_KEYSIZE ||
                     pubkSz != CURVE25519_KEYSIZE)) {
        ret = -1;
    }

    /*
        try once with another endian
    */
    if (ret == 0) {
        prvkSz = CURVE25519_KEYSIZE;
        pubkSz = CURVE25519_KEYSIZE;
        ret = wc_curve25519_export_key_raw_ex( &key, privateKey, &prvkSz,
                                       publicKey, &pubkSz, EC25519_BIG_ENDIAN);
    }
    if (ret == 0 && (prvkSz != CURVE25519_KEYSIZE ||
                     pubkSz != CURVE25519_KEYSIZE)) {
        ret = -1;
    }

    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* end of test_wc_curve25519_export_key_raw_ex */

/*
 * Testing wc_curve25519_make_key
 */
static int test_wc_curve25519_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    WC_RNG          rng;
    curve25519_key  key;
    int             keysize;
    int             ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);
        if (ret == 0) {
            keysize = wc_curve25519_size(&key);
            if (keysize != CURVE25519_KEYSIZE) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            ret = wc_curve25519_make_key(&rng, keysize, &key);
        }
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_make_key(NULL, 0, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_key(&rng, keysize, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_key(NULL, keysize, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_key(&rng, 0, &key);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve25519_make_key*/
/*
 * Testing wc_curve25519_shared_secret_ex
 */
static int test_wc_curve25519_shared_secret_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    WC_RNG          rng;
    curve25519_key  private_key, public_key;
    byte            out[CURVE25519_KEYSIZE];
    word32          outLen = sizeof(out);
    int             endian = EC25519_BIG_ENDIAN;
    int             ret;

    ret = wc_curve25519_init(&private_key);
    if (ret == 0) {
        ret = wc_curve25519_init(&public_key);
    }
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {
        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &private_key);
    }
    if (ret == 0) {
        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &public_key);
    }
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
                                              &outLen, endian);
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(NULL, NULL, NULL,
                                              0, endian);
        if (ret == 0) {
            ret = -1;
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(NULL, &public_key, out,
                                             &outLen, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(&private_key, NULL, out,
                                              &outLen, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(&private_key, &public_key, NULL,
                                              &outLen, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
                                              NULL, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
    /*curve25519.c is checking for public_key size less than or equal to 0x7f,
     *increasing to 0x8f checks for error being returned*/
        public_key.p.point[CURVE25519_KEYSIZE-1] = 0x8F;
        ret = wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
                                              &outLen, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == ECC_BAD_ARG_E) {
           ret = 0;
        }
    }

    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve25519_shared_secret_ex(&private_key, &public_key, out,
                                              &outLen, endian);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    wc_curve25519_free(&private_key);
    wc_curve25519_free(&public_key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve25519_shared_secret_ex*/
/*
 * Testing wc_curve25519_make_pub
 */
static int test_wc_curve25519_make_pub(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_CURVE25519
    WC_RNG          rng;
    curve25519_key  key;
    byte            out[CURVE25519_KEYSIZE];
    int             ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
        if (ret == 0) {
            ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);
        }
    }
    if (ret == 0) {
      ret = wc_curve25519_make_pub((int)sizeof(out), out, (int)sizeof(key.k), key.k);
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_make_pub((int)sizeof(key.k) - 1, key.k, (int)sizeof out, out);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k), NULL);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_pub((int)sizeof out - 1, out, (int)sizeof(key.k), key.k);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_make_pub((int)sizeof out, NULL, (int)sizeof(key.k), key.k);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        /* verify clamping test */
        key.k[0] |= ~248;
        ret = wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k), key.k);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        key.k[0] &= 248;
    }
    /* repeat the expected-to-succeed test. */
    if (ret == 0) {
      ret = wc_curve25519_make_pub((int)sizeof out, out, (int)sizeof(key.k), key.k);
    }

    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve25519_make_pub */
/*
 * Testing test_wc_curve25519_export_public_ex
 */
static int test_wc_curve25519_export_public_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    WC_RNG          rng;
    curve25519_key  key;
    byte            out[CURVE25519_KEYSIZE];
    word32          outLen = sizeof(out);
    int             endian = EC25519_BIG_ENDIAN;
    int             ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);
        if (ret == 0) {
            ret = wc_curve25519_export_public(&key, out, &outLen);
            }
        if (ret == 0) {
            ret = wc_curve25519_export_public_ex(&key, out, &outLen, endian);
        }
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_export_public_ex(NULL, NULL, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_public_ex(NULL, out, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_public_ex(&key, NULL, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_public_ex(&key, out, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve25519_export_public_ex(&key, out, &outLen, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }

    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /*END test_wc_curve25519_export_public_ex*/
/*
 * Testing test_wc_curve25519_import_private_raw_ex
 */
static int test_wc_curve25519_import_private_raw_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    WC_RNG          rng;
    curve25519_key  key;
    byte            priv[CURVE25519_KEYSIZE];
    byte            pub[CURVE25519_KEYSIZE];
    word32          privSz = sizeof(priv);
    word32          pubSz = sizeof(pub);
    int             endian = EC25519_BIG_ENDIAN;
    int             ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);
        if (ret == 0) {
            ret = wc_curve25519_export_private_raw_ex(&key, priv, &privSz, endian);
        }
        if (ret == 0) {
            ret = wc_curve25519_export_public(&key, pub, &pubSz);
        }
        if (ret == 0) {
            ret = wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                     &key, endian);
        }
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(NULL, 0, NULL, 0, NULL,
                                                   endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(NULL, privSz, pub, pubSz,
                                                 &key, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(priv, privSz, NULL, pubSz,
                                                 &key, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                 NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(priv, 0, pub, pubSz,
                                                 &key, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(priv, privSz, pub, 0,
                                                 &key, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                 &key, EC25519_LITTLE_ENDIAN);

    }

    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve25519_import_private_raw_ex*/

/*
 * Testing test_wc_curve25519_import_private
 */
static int test_wc_curve25519_import_private(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    curve25519_key     key;
    WC_RNG             rng;
    byte               priv[CURVE25519_KEYSIZE];
    word32             privSz = sizeof(priv);
    int                ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve25519_make_key(&rng, CURVE25519_KEYSIZE, &key);
        if (ret == 0) {
            ret = wc_curve25519_export_private_raw(&key, priv, &privSz);
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_import_private(priv, privSz, &key);
    }
    wc_curve25519_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve25519_import*/

/*
 * Testing test_wc_curve25519_export_private_raw_ex
 */
static int test_wc_curve25519_export_private_raw_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE25519)
    curve25519_key  key;
    byte            out[CURVE25519_KEYSIZE];
    word32          outLen = sizeof(out);
    int             endian = EC25519_BIG_ENDIAN;
    int             ret;

    ret = wc_curve25519_init(&key);
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(&key, out, &outLen, endian);
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(NULL, NULL, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(NULL, out, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(&key, NULL, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(&key, out, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(&key, out, &outLen,
                                                 EC25519_LITTLE_ENDIAN);
    }
    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve25519_export_private_raw_ex(&key, out, &outLen, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }

    wc_curve25519_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/*END test_wc_curve25519_export_private_raw_ex*/

/*
 * Testing wc_ed448_make_key().
 */
static int test_wc_ed448_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448)
    ed448_key     key;
    WC_RNG        rng;
    unsigned char pubkey[ED448_PUB_KEY_SIZE];
    int           ret;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed448_init(&key);
    }
    if (ret == 0) {
        ret = wc_ed448_make_public(&key, pubkey, sizeof(pubkey));
        if (ret == ECC_PRIV_KEY_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = -1;
        }
    }
    if (ret == 0) {
        ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed448_make_key(NULL, ED448_KEY_SIZE, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE - 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE + 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_make_key */


/*
 * Testing wc_ed448_init()
 */
static int test_wc_ed448_init(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448)
    ed448_key    key;
    int          ret;

    ret = wc_ed448_init(&key);

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed448_init(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_init */

/*
 * Test wc_ed448_sign_msg() and wc_ed448_verify_msg()
 */
static int test_wc_ed448_sign_msg(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_SIGN)
    WC_RNG        rng;
    ed448_key     key;
    byte          msg[] = "Everybody gets Friday off.\n";
    byte          sig[ED448_SIG_SIZE];
    word32        msglen = sizeof(msg);
    word32        siglen = sizeof(sig);
    word32        badSigLen = sizeof(sig) - 1;
#ifdef HAVE_ED448_VERIFY
    int           verify_ok = 0; /*1 = Verify success.*/
#endif
    int           ret;

    /* Initialize stack variables. */
    XMEMSET(sig, 0, siglen);

    /* Initialize key. */
    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed448_init(&key);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
        }
    }

    if (ret == 0) {
        ret = wc_ed448_sign_msg(msg, msglen, sig, &siglen, &key, NULL, 0);
    }
    /* Test bad args. */
    if (ret == 0 && siglen == ED448_SIG_SIZE) {
        ret = wc_ed448_sign_msg(NULL, msglen, sig, &siglen, &key, NULL, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_sign_msg(msg, msglen, NULL, &siglen, &key, NULL, 0);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_sign_msg(msg, msglen, sig, NULL, &key, NULL, 0);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_sign_msg(msg, msglen, sig, &siglen, NULL, NULL, 0);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_sign_msg(msg, msglen, sig, &badSigLen, &key,
                                    NULL, 0);
        }
        if (ret == BUFFER_E && badSigLen == ED448_SIG_SIZE) {
            badSigLen -= 1;
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    } /* END sign */

#ifdef HAVE_ED448_VERIFY
    if (ret == 0) {

        ret = wc_ed448_verify_msg(sig, siglen, msg, msglen, &verify_ok,
                                  &key, NULL, 0);
        if (ret == 0  && verify_ok == 1) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }

        /* Test bad args. */
        if (ret == 0) {
            AssertIntEQ(wc_ed448_verify_msg(sig, siglen - 1, msg,
                msglen, &verify_ok, &key, NULL, 0), BAD_FUNC_ARG);
            AssertIntEQ(wc_ed448_verify_msg(sig, siglen + 1, msg,
                msglen, &verify_ok, &key, NULL, 0), BAD_FUNC_ARG);

            ret = wc_ed448_verify_msg(NULL, siglen, msg, msglen, &verify_ok,
                                      &key, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_verify_msg(sig, siglen, NULL, msglen,
                                          &verify_ok, &key, NULL, 0);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_verify_msg(sig, siglen, msg, msglen,
                                          NULL, &key, NULL, 0);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_verify_msg(sig, siglen, msg, msglen,
                                          &verify_ok, NULL, NULL, 0);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_verify_msg(sig, badSigLen, msg, msglen,
                                          &verify_ok, &key, NULL, 0);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }

    } /* END verify. */
#endif /* Verify. */

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_sign_msg */

/*
 * Testing wc_ed448_import_public()
 */
static int test_wc_ed448_import_public(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    WC_RNG          rng;
    ed448_key     pubKey;
    const byte      in[] =
                    "Ed448PublicKeyUnitTest.................................\n";
    word32          inlen = sizeof(in);
    int             ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ed448_init(&pubKey);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &pubKey);
        }
    }

    if (ret == 0) {
        ret = wc_ed448_import_public_ex(in, inlen, &pubKey, 1);

        if (ret == 0 && XMEMCMP(in, pubKey.p, inlen) == 0) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }

        /* Test bad args. */
        if (ret == 0) {
            ret = wc_ed448_import_public(NULL, inlen, &pubKey);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_import_public(in, inlen, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_import_public(in, inlen - 1, &pubKey);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&pubKey);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END wc_ed448_import_public */

/*
 * Testing wc_ed448_import_private_key()
 */
static int test_wc_ed448_import_private_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    WC_RNG      rng;
    ed448_key   key;
    const byte  privKey[] =
                    "Ed448PrivateKeyUnitTest................................\n";
    const byte  pubKey[] =
                    "Ed448PublicKeyUnitTest.................................\n";
    word32      privKeySz = sizeof(privKey);
    word32      pubKeySz = sizeof(pubKey);
#ifdef HAVE_ED448_KEY_EXPORT
    byte        bothKeys[sizeof(privKey) + sizeof(pubKey)];
    word32      bothKeysSz = sizeof(bothKeys);
#endif
    int         ret;

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return TEST_FAIL;
    }
    ret = wc_ed448_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }
    ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);

    if (ret == 0) {
        ret = wc_ed448_import_private_key_ex(privKey, privKeySz, pubKey,
                                                             pubKeySz, &key, 1);
        if (ret == 0 && (XMEMCMP(pubKey, key.p, privKeySz) != 0 ||
                                      XMEMCMP(privKey, key.k, pubKeySz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#ifdef HAVE_ED448_KEY_EXPORT
    if (ret == 0)
        ret = wc_ed448_export_private(&key, bothKeys, &bothKeysSz);

    if (ret == 0) {
        ret = wc_ed448_import_private_key_ex(bothKeys, bothKeysSz, NULL, 0,
                                                                       &key, 1);
        if (ret == 0 && (XMEMCMP(pubKey, key.p, privKeySz) != 0 ||
                                      XMEMCMP(privKey, key.k, pubKeySz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ed448_import_private_key(NULL, privKeySz, pubKey, pubKeySz,
                                                                          &key);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_import_private_key(privKey, privKeySz, NULL,
                                                                pubKeySz, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_import_private_key(privKey, privKeySz, pubKey,
                                                                pubKeySz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_import_private_key(privKey, privKeySz - 1, pubKey,
                                                                pubKeySz, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_import_private_key(privKey, privKeySz, pubKey,
                                                            pubKeySz - 1, &key);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_import_private_key(privKey, privKeySz, NULL,
                                                            0, &key);
        }

        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_import_private_key */

/*
 * Testing wc_ed448_export_public() and wc_ed448_export_private_only()
 */
static int test_wc_ed448_export(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    WC_RNG          rng;
    ed448_key       key;
    byte            priv[ED448_PRV_KEY_SIZE];
    byte            pub[ED448_PUB_KEY_SIZE];
    word32          privSz = sizeof(priv);
    word32          pubSz = sizeof(pub);
    int             ret;

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return TEST_FAIL;
    }

    ret = wc_ed448_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    if (ret == 0) {
        ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
    }

    if (ret == 0) {
        ret = wc_ed448_export_public(&key, pub, &pubSz);
        if (ret == 0 && (pubSz != ED448_KEY_SIZE ||
                                             XMEMCMP(key.p, pub, pubSz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        if (ret == 0) {
            ret = wc_ed448_export_public(NULL, pub, &pubSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_public(&key, NULL, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_public(&key, pub, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (ret == 0) {
        ret = wc_ed448_export_private_only(&key, priv, &privSz);
        if (ret == 0 && (privSz != ED448_KEY_SIZE ||
                                           XMEMCMP(key.k, priv, privSz) != 0)) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        if (ret == 0) {
            ret = wc_ed448_export_private_only(NULL, priv, &privSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_private_only(&key, NULL, &privSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_private_only(&key, priv, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_ed448_export */

/*
 *  Testing wc_ed448_size()
 */
static int test_wc_ed448_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448)
    WC_RNG          rng;
    ed448_key       key;
    int             ret = 0;

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return TEST_FAIL;
    }
    ret = wc_ed448_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        wc_ed448_free(&key);
        return TEST_FAIL;
    }

    ret = wc_ed448_size(&key);
    /* Test bad args. */
    if (ret == ED448_KEY_SIZE) {
        ret = wc_ed448_size(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    if (ret == 0) {
        ret = wc_ed448_sig_size(&key);
        if (ret == ED448_SIG_SIZE) {
            ret = 0;
        }
        /* Test bad args. */
        if (ret == 0) {
            ret = wc_ed448_sig_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed448_sig_size() */

    if (ret == 0) {
        ret = wc_ed448_pub_size(&key);
        if (ret == ED448_PUB_KEY_SIZE) {
            ret = 0;
        }
        if (ret == 0) {
            ret = wc_ed448_pub_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed448_pub_size */

    if (ret == 0) {
        ret = wc_ed448_priv_size(&key);
        if (ret == ED448_PRV_KEY_SIZE) {
            ret = 0;
        }
        if (ret == 0) {
            ret = wc_ed448_priv_size(NULL);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    } /* END wc_ed448_pub_size */

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_size */

/*
 * Testing wc_ed448_export_private() and wc_ed448_export_key()
 */
static int test_wc_ed448_exportKey(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    WC_RNG          rng;
    ed448_key       key;
    byte            priv[ED448_PRV_KEY_SIZE];
    byte            pub[ED448_PUB_KEY_SIZE];
    byte            privOnly[ED448_PRV_KEY_SIZE];
    word32          privSz      = sizeof(priv);
    word32          pubSz       = sizeof(pub);
    word32          privOnlySz  = sizeof(privOnly);
    int             ret;

    ret = wc_InitRng(&rng);
    if (ret != 0) {
        return TEST_FAIL;
    }
    ret = wc_ed448_init(&key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        return TEST_FAIL;
    }

    ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
    if (ret != 0) {
        wc_FreeRng(&rng);
        wc_ed448_free(&key);
        return TEST_FAIL;
    }

    ret = wc_ed448_export_private(&key, privOnly, &privOnlySz);
    if (ret == 0) {
        ret = wc_ed448_export_private(NULL, privOnly, &privOnlySz);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_export_private(&key, NULL, &privOnlySz);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ed448_export_private(&key, privOnly, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_ed448_export_key(&key, priv, &privSz, pub, &pubSz);
        if (ret == 0) {
            ret = wc_ed448_export_key(NULL, priv, &privSz, pub, &pubSz);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_key(&key, NULL, &privSz, pub, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_key(&key, priv, NULL, pub, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_key(&key, priv, &privSz, NULL, &pubSz);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ed448_export_key(&key, priv, &privSz, pub, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    } /* END wc_ed448_export_key() */

    /* Cross check output. */
    if (ret == 0 && XMEMCMP(priv, privOnly, privSz) != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ed448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ed448_exportKey */

/*
 * Testing wc_Ed448PublicKeyToDer
 */
static int test_wc_Ed448PublicKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    int       tmp;
    ed448_key key;
    byte      derBuf[1024];
    int       ret = 0;

    /* Test bad args */
    tmp = wc_Ed448PublicKeyToDer(NULL, NULL, 0, 0);
    if (tmp != BAD_FUNC_ARG) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0) {
        wc_ed448_init(&key);
        tmp = wc_Ed448PublicKeyToDer(&key, derBuf, 0, 0);
        if (tmp != BUFFER_E) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        wc_ed448_free(&key);
    }

    /*  Test good args */
    if (ret == 0) {
        WC_RNG          rng;
        ret = wc_InitRng(&rng);
        if (ret != 0) {
            return TEST_FAIL;
        }
        ret = wc_ed448_init(&key);
        if (ret != 0) {
            wc_FreeRng(&rng);
            return TEST_FAIL;
        }

        ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &key);
        if (ret != 0) {
            wc_FreeRng(&rng);
            wc_ed448_free(&key);
            return TEST_FAIL;
        }

        tmp = wc_Ed448PublicKeyToDer(&key, derBuf, 1024, 1);
        if (tmp <= 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }

        wc_FreeRng(&rng);
        wc_ed448_free(&key);
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END testing wc_Ed448PublicKeyToDer */

/*
 * Testing wc_curve448_init and wc_curve448_free.
 */
static int test_wc_curve448_init(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    curve448_key  key;
    int ret = 0;

    ret = wc_curve448_init(&key);

    /* Test bad args for wc_curve448_init */
    if (ret == 0) {
        ret = wc_curve448_init(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /*  Test good args for wc_curve_448_free */
    wc_curve448_free(&key);

    wc_curve448_free(NULL);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_curve448_init and wc_curve_448_free*/

/*
 * Testing wc_curve448_make_key
 */
static int test_wc_curve448_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    WC_RNG        rng;
    curve448_key  key;
    int           keysize;
    int           ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key);
        if (ret == 0) {
            keysize = wc_curve448_size(&key);
            if (keysize != CURVE448_KEY_SIZE) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            ret = wc_curve448_make_key(&rng, keysize, &key);
        }
    }
    /* test bad cases */
    if (ret == 0) {
        ret = wc_curve448_make_key(NULL, 0, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_make_key(&rng, keysize, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_make_key(NULL, keysize, &key);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_make_key(&rng, 0, &key);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }

    if (wc_FreeRng(&rng) != 0 && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    wc_curve448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve448_make_key*/
/*
 * Testing test_wc_curve448_shared_secret_ex
 */
static int test_wc_curve448_shared_secret_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    WC_RNG        rng;
    curve448_key  private_key, public_key;
    byte          out[CURVE448_KEY_SIZE];
    word32        outLen = sizeof(out);
    int           endian = EC448_BIG_ENDIAN;
    int           ret;

    ret = wc_curve448_init(&private_key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
        if (ret == 0) {
            ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &private_key);
        }
    }
    if (ret == 0) {
        ret = wc_curve448_init(&public_key);
    }
    if (ret == 0) {
        if (ret == 0) {
            ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &public_key);
        }
    }

    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(&private_key, &public_key, out,
                                            &outLen, endian);
    }
    /* test bad cases */
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(NULL, NULL, NULL, 0, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(NULL, &public_key, out,
                                            &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(&private_key, NULL, out,
                                            &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(&private_key, &public_key, NULL,
                                            &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(&private_key, &public_key, out,
                                            NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve448_shared_secret_ex(&private_key, &public_key, out,
                                            &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    wc_curve448_free(&private_key);
    wc_curve448_free(&public_key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve448_shared_secret_ex*/

/*
 * Testing test_wc_curve448_export_public_ex
 */
static int test_wc_curve448_export_public_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    WC_RNG        rng;
    curve448_key  key;
    byte          out[CURVE448_KEY_SIZE];
    word32        outLen = sizeof(out);
    int           endian = EC448_BIG_ENDIAN;
    int           ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key);
        if (ret == 0) {
            ret = wc_curve448_export_public(&key, out, &outLen);
            }
        if (ret == 0) {
            ret = wc_curve448_export_public_ex(&key, out, &outLen, endian);
        }
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve448_export_public_ex(NULL, NULL, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_public_ex(NULL, out, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_public_ex(&key, NULL, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_public_ex(&key, out, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve448_export_public_ex(&key, out, &outLen, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }

    wc_curve448_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /*END test_wc_curve448_export_public_ex*/

/*
 * Testing test_wc_curve448_export_private_raw_ex
 */
static int test_wc_curve448_export_private_raw_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    curve448_key  key;
    byte          out[CURVE448_KEY_SIZE];
    word32        outLen = sizeof(out);
    int           endian = EC448_BIG_ENDIAN;
    int           ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(&key, out, &outLen, endian);
    }
    /*test bad cases*/
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(NULL, NULL, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(NULL, out, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(&key, NULL, &outLen, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(&key, out, NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(&key, out, &outLen,
                                                 EC448_LITTLE_ENDIAN);
    }
    outLen = outLen - 2;
    if (ret == 0) {
        ret = wc_curve448_export_private_raw_ex(&key, out, &outLen, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }

    wc_curve448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/*END test_wc_curve448_export_private_raw_ex*/

/*
 * Testing test_wc_curve448_import_private_raw_ex
 */
static int test_wc_curve448_import_private_raw_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    WC_RNG             rng;
    curve448_key    key;
    byte            priv[CURVE448_KEY_SIZE];
    byte            pub[CURVE448_KEY_SIZE];
    word32          privSz = sizeof(priv);
    word32          pubSz = sizeof(pub);
    int             endian = EC448_BIG_ENDIAN;
    int             ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key);
        if (ret == 0) {
            ret = wc_curve448_export_private_raw(&key, priv, &privSz);
        }
        if (ret == 0) {
            ret = wc_curve448_export_public(&key, pub, &pubSz);
            }
        if (ret == 0) {
            ret = wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                     &key, endian);
        }
    }
    /* test bad cases */
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(NULL, 0, NULL, 0, NULL, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(NULL, privSz, pub, pubSz,
                                                 &key, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(priv, privSz, NULL, pubSz,
                                                 &key, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                 NULL, endian);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(priv, 0, pub, pubSz,
                                                 &key, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(priv, privSz, pub, 0,
                                                 &key, endian);
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private_raw_ex(priv, privSz, pub, pubSz,
                                                 &key, EC448_LITTLE_ENDIAN);

    }

    if (wc_FreeRng(&rng) != 0 && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    wc_curve448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve448_import_private_raw_ex*/
/*
 * Testing test_curve448_export_key_raw
 */
static int test_wc_curve448_export_key_raw(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    WC_RNG          rng;
    curve448_key    key;
    byte            priv[CURVE448_KEY_SIZE];
    byte            pub[CURVE448_KEY_SIZE];
    word32          privSz = sizeof(priv);
    word32          pubSz = sizeof(pub);
    int             ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key);
        if (ret == 0) {
            ret = wc_curve448_export_private_raw(&key, priv, &privSz);
        }
        if (ret == 0) {
            ret = wc_curve448_export_public(&key, pub, &pubSz);
            }
        if (ret == 0) {
            ret = wc_curve448_export_key_raw(&key, priv, &privSz, pub, &pubSz);
        }
    }

    wc_curve448_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/*END test_wc_curve448_import_private_raw_ex*/


/*
 * Testing test_wc_curve448_import_private
 */
static int test_wc_curve448_import_private(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    curve448_key       key;
    WC_RNG             rng;
    byte               priv[CURVE448_KEY_SIZE];
    word32             privSz = sizeof(priv);
    int                ret;

    ret = wc_curve448_init(&key);
    if (ret == 0) {
        ret = wc_InitRng(&rng);
    }
    if (ret == 0) {

        ret = wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &key);
        if (ret == 0) {
            ret = wc_curve448_export_private_raw(&key, priv, &privSz);
        }
    }
    if (ret == 0) {
        ret = wc_curve448_import_private(priv, privSz, &key);
    }
    wc_curve448_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /*END test_wc_curve448_import*/
/*
 * Testing test_wc_curve448_size.
 */
static int test_wc_curve448_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_CURVE448)
    curve448_key  key;
    int ret = 0;

    ret = wc_curve448_init(&key);

    /*  Test good args for wc_curve448_size */
    if (ret == 0) {
        ret = wc_curve448_size(&key);
    }

    /* Test bad args for wc_curve448_size */
    if (ret != 0) {
        ret = wc_curve448_size(NULL);
    }

    wc_curve448_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_curve448_size*/

/*
 * Testing wc_ecc_make_key.
 */
static int test_wc_ecc_make_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    WC_RNG  rng;
    ecc_key key;
    int     ret;

    ret = wc_InitRng(&rng);
    if (ret != 0)
        return TEST_FAIL;

    ret = wc_ecc_init(&key);
    if (ret == 0) {
        ret = wc_ecc_make_key(&rng, KEY14, &key);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
    #endif

        /* Pass in bad args. */
        if (ret == 0) {
            ret = wc_ecc_make_key(NULL, KEY14, &key);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_make_key(&rng, KEY14, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        wc_ecc_free(&key);
    }

    if (wc_FreeRng(&rng) != 0 && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_make_key */


/*
 * Testing wc_ecc_init()
 */
static int test_wc_ecc_init(void)
{
    int res = TEST_SKIPPED;
#ifdef HAVE_ECC
    ecc_key     key;
    int         ret;

    ret = wc_ecc_init(&key);
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_init(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ecc_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_init */

/*
 * Testing wc_ecc_check_key()
 */
static int test_wc_ecc_check_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    ecc_key     key;
    int         ret;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        ret = wc_ecc_check_key(&key);
    }

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_check_key(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_check_key */

/*
 * Testing wc_ecc_get_generator()
 */
static int test_wc_ecc_get_generator(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST) && \
    !defined(HAVE_FIPS) && defined(OPENSSL_EXTRA)
    ecc_point* pt;
    int        ret = 0;

    pt = wc_ecc_new_point();
    if (!pt) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0) {
        ret = wc_ecc_get_generator(pt, wc_ecc_get_curve_idx(ECC_SECP256R1));
    }

    /* Test bad args. */
    if (ret == MP_OKAY) {
        /* Returns Zero for bad arg. */
        ret = wc_ecc_get_generator(pt, -1);
        if (ret != MP_OKAY)
            wc_ecc_get_generator(NULL, wc_ecc_get_curve_idx(ECC_SECP256R1));
        if (ret != MP_OKAY)
            wc_ecc_get_generator(pt, 1000); /* If we ever get to 1000 curves
                                             * increase this number */
        if (ret != MP_OKAY)
            wc_ecc_get_generator(NULL, -1);
        ret = (ret == MP_OKAY) ? WOLFSSL_FATAL_ERROR : 0;
    }

    wc_ecc_del_point(pt);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_get_generator */

/*
 * Testing wc_ecc_size()
 */
static int test_wc_ecc_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    ecc_key     key;
    int         ret;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        ret = wc_ecc_size(&key);
        if (ret == KEY14) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        /* Returns Zero for bad arg. */
        ret = wc_ecc_size(NULL);
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_size */

static int test_wc_ecc_params(void)
{
    int res = TEST_SKIPPED;
    /* FIPS/CAVP self-test modules do not have `wc_ecc_get_curve_params`.
        It was added after certifications */
#if defined(HAVE_ECC) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    const ecc_set_type* ecc_set;
#if !defined(NO_ECC256) && !defined(NO_ECC_SECP)
    /* Test for SECP256R1 curve */
    int curve_id = ECC_SECP256R1;
    int curve_idx = wc_ecc_get_curve_idx(curve_id);
    AssertIntNE(curve_idx, ECC_CURVE_INVALID);
    ecc_set = wc_ecc_get_curve_params(curve_idx);
    AssertNotNull(ecc_set);
    AssertIntEQ(ecc_set->id, curve_id);
#endif
    /* Test case when SECP256R1 is not enabled */
    /* Test that we get curve params for index 0 */
    ecc_set = wc_ecc_get_curve_params(0);
    AssertNotNull(ecc_set);

    res = TEST_RES_CHECK(1);
#endif /* HAVE_ECC && !HAVE_FIPS && !HAVE_SELFTEST */
    return res;
}

/*
 * Testing wc_ecc_sign_hash() and wc_ecc_verify_hash()
 */
static int test_wc_ecc_signVerify_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_SIGN) && !defined(NO_ASN) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    ecc_key     key;
    int         ret;
    int         signH = WOLFSSL_FATAL_ERROR;
    #ifdef HAVE_ECC_VERIFY
        int     verifyH = WOLFSSL_FATAL_ERROR;
        int     verify  = 0;
    #endif
    word32      siglen = ECC_BUFSIZE;
    byte        sig[ECC_BUFSIZE];
    byte        adjustedSig[ECC_BUFSIZE+1];
    byte        digest[] = TEST_STRING;
    word32      digestlen = (word32)TEST_STRING_SZ;

    /* Init stack var */
    XMEMSET(sig, 0, siglen);
    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(adjustedSig, 0, ECC_BUFSIZE+1);

    /* Init structs. */
    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        ret = wc_ecc_sign_hash(digest, digestlen, sig, &siglen, &rng, &key);
    }

    /* Check bad args. */
    if (ret == 0) {
        signH = wc_ecc_sign_hash(NULL, digestlen, sig, &siglen, &rng, &key);
        if (signH == ECC_BAD_ARG_E) {
            signH = wc_ecc_sign_hash(digest, digestlen, NULL, &siglen,
                                                                &rng, &key);
        }
        if (signH == ECC_BAD_ARG_E) {
            signH = wc_ecc_sign_hash(digest, digestlen, sig, NULL,
                                                                &rng, &key);
        }
        if (signH == ECC_BAD_ARG_E) {
            signH = wc_ecc_sign_hash(digest, digestlen, sig, &siglen,
                                                                NULL, &key);
        }
        if (signH == ECC_BAD_ARG_E) {
            signH = wc_ecc_sign_hash(digest, digestlen, sig, &siglen,
                                                                &rng, NULL);
        }
        if (signH == ECC_BAD_ARG_E) {
            signH = 0;
        }
        else if (ret == 0) {
            signH = WOLFSSL_FATAL_ERROR;
        }
    }

#ifdef HAVE_ECC_VERIFY
    ret = wc_ecc_verify_hash(sig, siglen, digest, digestlen, &verify, &key);
    if (verify != 1 && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    /* test check on length of signature passed in */
    XMEMCPY(adjustedSig, sig, siglen);
    adjustedSig[1] = adjustedSig[1] + 1; /* add 1 to length for extra byte*/
#ifndef NO_STRICT_ECDSA_LEN
    AssertIntNE(wc_ecc_verify_hash(adjustedSig, siglen+1, digest, digestlen,
                &verify, &key), 0);
#else
    /* if NO_STRICT_ECDSA_LEN is set then extra bytes after the signature
     * is allowed */
    AssertIntEQ(wc_ecc_verify_hash(adjustedSig, siglen+1, digest, digestlen,
                &verify, &key), 0);
#endif

    /* Test bad args. */
    if (ret == 0) {
        verifyH = wc_ecc_verify_hash(NULL, siglen, digest, digestlen,
                                                        &verify, &key);
        if (verifyH == ECC_BAD_ARG_E) {
            verifyH = wc_ecc_verify_hash(sig, siglen, NULL, digestlen,
                                                        &verify, &key);
        }
        if (verifyH == ECC_BAD_ARG_E) {
            verifyH = wc_ecc_verify_hash(sig, siglen, digest, digestlen,
                                                            NULL, &key);
        }
        if (verifyH == ECC_BAD_ARG_E) {
            verifyH = wc_ecc_verify_hash(sig, siglen, digest, digestlen,
                                                        &verify, NULL);
        }
        if (verifyH == ECC_BAD_ARG_E) {
            verifyH = 0;
        }
        else if (ret == 0) {
            verifyH = WOLFSSL_FATAL_ERROR;
        }
    }

#endif /* HAVE_ECC_VERIFY */

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0 && signH == 0 && verifyH == 0);
#endif
    return res;
} /*  END test_wc_ecc_sign_hash */


/*
 * Testing wc_ecc_shared_secret()
 */
static int test_wc_ecc_shared_secret(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_DHE) && !defined(WC_NO_RNG)
    ecc_key     key, pubKey;
    WC_RNG      rng;
    int         ret;
    byte        out[KEY32];
    int         keySz = sizeof(out);
    word32      outlen = (word32)sizeof(out);

#if defined(HAVE_ECC) && !defined(NO_ECC256)
    const char* qx =
        "bb33ac4c27504ac64aa504c33cde9f36db722dce94ea2bfacb2009392c16e861";
    const char* qy =
        "02e9af4dd302939a315b9792217ff0cf18da9111023486e82058330b803489d8";
    const char* d  =
        "45b66902739c6c85a1385b72e8e8c7acc4038d533504fa6c28dc348de1a8098c";
    const char* curveName = "SECP256R1";
    const byte expected_shared_secret[] =
        {
            0x65, 0xc0, 0xd4, 0x61, 0x17, 0xe6, 0x09, 0x75,
            0xf0, 0x12, 0xa0, 0x4d, 0x0b, 0x41, 0x30, 0x7a,
            0x51, 0xf0, 0xb3, 0xaf, 0x23, 0x8f, 0x0f, 0xdf,
            0xf1, 0xff, 0x23, 0x64, 0x28, 0xca, 0xf8, 0x06
        };
#endif

    PRIVATE_KEY_UNLOCK();

    /* Initialize variables. */
    XMEMSET(out, 0, keySz);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(&pubKey, 0, sizeof(pubKey));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_init(&pubKey);
        }
    }

#if defined(HAVE_ECC) && !defined(NO_ECC256)
    if (ret == 0) {
        ret = wc_ecc_import_raw(&key, qx, qy, d, curveName);
    }
    if (ret == 0) {
        ret = wc_ecc_import_raw(&pubKey, qx, qy, NULL, curveName);
    }
#else
    if (ret == 0) {
        ret = wc_ecc_make_key(&rng, keySz, &key);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
    }
    if (ret == 0) {
        ret = wc_ecc_make_key(&rng, keySz, &pubKey);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &pubKey.asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
    }
#endif

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    if (ret == 0) {
        ret = wc_ecc_set_rng(&key, &rng);
    }
#endif

    if (ret == 0) {
        ret = wc_ecc_shared_secret(&key, &pubKey, out, &outlen);

#if defined(HAVE_ECC) && !defined(NO_ECC256)
        if (ret == 0) {
            if (0 != XMEMCMP(out, expected_shared_secret, outlen)) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
#endif

        /* Test bad args. */
        if (ret == 0) {
            ret = wc_ecc_shared_secret(NULL, &pubKey, out, &outlen);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_shared_secret(&key, NULL, out, &outlen);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_shared_secret(&key, &pubKey, NULL, &outlen);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_shared_secret(&key, &pubKey, out, NULL);
            }
            if (ret == BAD_FUNC_ARG) {
                /* Invalid length */
                outlen = 1;
                ret = wc_ecc_shared_secret(&key, &pubKey, out, &outlen);
            }

            if (ret == BUFFER_E) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);
    wc_ecc_free(&pubKey);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    PRIVATE_KEY_LOCK();

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END tests_wc_ecc_shared_secret */

/*
 * testint wc_ecc_export_x963()
 */
static int test_wc_ecc_export_x963(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    byte    out[ECC_ASN963_MAX_BUF_SZ];
    word32  outlen = sizeof(out);
    int     ret = 0;

    PRIVATE_KEY_UNLOCK();

    /* Initialize variables. */
    XMEMSET(out, 0, outlen);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY20, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }
    if (ret == 0) {
        ret = wc_ecc_export_x963(&key, out, &outlen);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_export_x963(NULL, out, &outlen);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_export_x963(&key, NULL, &outlen);
        }
        if (ret == LENGTH_ONLY_E) {
            ret = wc_ecc_export_x963(&key, out, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            key.idx = -4;
            ret = wc_ecc_export_x963(&key, out, &outlen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    PRIVATE_KEY_LOCK();

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_export_x963 */

/*
 * Testing wc_ecc_export_x963_ex()
 * compile with --enable-compkey will use compression.
 */
static int test_wc_ecc_export_x963_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    int     ret = 0;
    byte    out[ECC_ASN963_MAX_BUF_SZ];
    word32  outlen = sizeof(out);
    #ifdef HAVE_COMP_KEY
        word32  badOutLen = 5;
    #endif

    /* Init stack variables. */
    XMEMSET(out, 0, outlen);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY64, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    #ifdef HAVE_COMP_KEY
        if (ret == 0) {
            ret = wc_ecc_export_x963_ex(&key, out, &outlen, COMP);
        }
    #else
        if (ret == 0) {
            ret = wc_ecc_export_x963_ex(&key, out, &outlen, NOCOMP);
        }
    #endif

    /* Test bad args. */
    #ifdef HAVE_COMP_KEY
    if (ret == 0) {
        ret = wc_ecc_export_x963_ex(NULL, out, &outlen, COMP);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_export_x963_ex(&key, NULL, &outlen, COMP);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_export_x963_ex(&key, out, NULL, COMP);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_export_x963_ex(&key, out, &badOutLen, COMP);
        }
#if defined(HAVE_FIPS) && (!defined(FIPS_VERSION_LT) || FIPS_VERSION_LT(5,3))
        if (ret == BUFFER_E)
#else
        if (ret == LENGTH_ONLY_E)
#endif
        {
            key.idx = -4;
            ret = wc_ecc_export_x963_ex(&key, out, &outlen, COMP);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    #else
        if (ret == 0) {
            ret = wc_ecc_export_x963_ex(NULL, out, &outlen, NOCOMP);
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_export_x963_ex(&key, NULL, &outlen, NOCOMP);
            }
            if (ret == BAD_FUNC_ARG) {
                ret = wc_ecc_export_x963_ex(&key, out, &outlen, 1);
            }
            if (ret == NOT_COMPILED_IN) {
                ret = wc_ecc_export_x963_ex(&key, out, NULL, NOCOMP);
            }
            if (ret == BAD_FUNC_ARG) {
                key.idx = -4;
                ret = wc_ecc_export_x963_ex(&key, out, &outlen, NOCOMP);
            }
            if (ret == ECC_BAD_ARG_E) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    #endif

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_export_x963_ex */

/*
 * testing wc_ecc_import_x963()
 */
static int test_wc_ecc_import_x963(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_IMPORT) && \
    defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key pubKey, key;
    WC_RNG  rng;
    byte    x963[ECC_ASN963_MAX_BUF_SZ];
    word32  x963Len = (word32)sizeof(x963);
    int     ret;

    /* Init stack variables. */
    XMEMSET(x963, 0, x963Len);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(&pubKey, 0, sizeof(pubKey));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&pubKey);
        if (ret == 0) {
            ret = wc_ecc_init(&key);
        }
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY24, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            PRIVATE_KEY_UNLOCK();
            ret = wc_ecc_export_x963(&key, x963, &x963Len);
            PRIVATE_KEY_LOCK();
        }
    }

    if (ret == 0) {
        ret = wc_ecc_import_x963(x963, x963Len, &pubKey);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_import_x963(NULL, x963Len, &pubKey);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_x963(x963, x963Len, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_x963(x963, x963Len + 1, &pubKey);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);
    wc_ecc_free(&pubKey);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END wc_ecc_import_x963 */

/*
 * testing wc_ecc_import_private_key()
 */
static int ecc_import_private_key(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_IMPORT) && \
    defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key, keyImp;
    WC_RNG  rng;
    byte    privKey[ECC_PRIV_KEY_BUF]; /* Raw private key.*/
    byte    x963Key[ECC_ASN963_MAX_BUF_SZ];
    word32  privKeySz = (word32)sizeof(privKey);
    word32  x963KeySz = (word32)sizeof(x963Key);
    int     ret;

    /* Init stack variables. */
    XMEMSET(privKey, 0, privKeySz);
    XMEMSET(x963Key, 0, x963KeySz);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(&keyImp, 0, sizeof(keyImp));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_init(&keyImp);
        }
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY48, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            PRIVATE_KEY_UNLOCK();
            ret = wc_ecc_export_x963(&key, x963Key, &x963KeySz);
            PRIVATE_KEY_LOCK();
        }
        if (ret == 0) {
            ret = wc_ecc_export_private_only(&key, privKey, &privKeySz);
        }
    }

    if (ret == 0) {
        ret = wc_ecc_import_private_key(privKey, privKeySz, x963Key,
                                                x963KeySz, &keyImp);
    }
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_import_private_key(privKey, privKeySz, x963Key,
                                                x963KeySz, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_private_key(NULL, privKeySz, x963Key,
                                                x963KeySz, &keyImp);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);
    wc_ecc_free(&keyImp);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END wc_ecc_import_private_key */


/*
 * Testing wc_ecc_export_private_only()
 */
static int test_wc_ecc_export_private_only(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    ecc_key key;
    WC_RNG  rng;
    byte    out[ECC_PRIV_KEY_BUF];
    word32  outlen = sizeof(out);
    int     ret;

    /* Init stack variables. */
    XMEMSET(out, 0, outlen);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY32, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }
    if (ret == 0) {
        ret = wc_ecc_export_private_only(&key, out, &outlen);
    }
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_export_private_only(NULL, out, &outlen);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_export_private_only(&key, NULL, &outlen);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_export_private_only(&key, out, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_export_private_only */


/*
 * Testing wc_ecc_rs_to_sig()
 */
static int test_wc_ecc_rs_to_sig(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(NO_ASN)
    /* first [P-192,SHA-1] vector from FIPS 186-3 NIST vectors */
    const char*   R = "6994d962bdd0d793ffddf855ec5bf2f91a9698b46258a63e";
    const char*   S = "02ba6465a234903744ab02bc8521405b73cf5fc00e1a9f41";
    const char*   zeroStr = "0";
    byte          sig[ECC_MAX_SIG_SIZE];
    word32        siglen = (word32)sizeof(sig);
    /*R and S max size is the order of curve. 2^192.*/
    int           keySz = KEY24;
    byte          r[KEY24];
    byte          s[KEY24];
    word32        rlen = (word32)sizeof(r);
    word32        slen = (word32)sizeof(s);
    int           ret;

    /* Init stack variables. */
    XMEMSET(sig, 0, ECC_MAX_SIG_SIZE);
    XMEMSET(r, 0, keySz);
    XMEMSET(s, 0, keySz);

    ret = wc_ecc_rs_to_sig(R, S, sig, &siglen);
    if (ret == 0) {
        ret = wc_ecc_sig_to_rs(sig, siglen, r, &rlen, s, &slen);
    #if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
        if (ret == ASN_PARSE_E) {
            ret = 0;
        }
    #endif
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_rs_to_sig(NULL, S, sig, &siglen);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_rs_to_sig(R, NULL, sig, &siglen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_rs_to_sig(R, S, sig, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_rs_to_sig(R, S, NULL, &siglen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_rs_to_sig(R, zeroStr, sig, &siglen);
        }
        if (ret == MP_ZERO_E) {
            ret = wc_ecc_rs_to_sig(zeroStr, S, sig, &siglen);
        }
        if (ret == MP_ZERO_E) {
            ret = wc_ecc_sig_to_rs(NULL, siglen, r, &rlen, s, &slen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_sig_to_rs(sig, siglen, NULL, &rlen, s, &slen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_sig_to_rs(sig, siglen, r, NULL, s, &slen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_sig_to_rs(sig, siglen, r, &rlen, NULL, &slen);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_sig_to_rs(sig, siglen, r, &rlen, s, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_rs_to_sig */

static int test_wc_ecc_import_raw(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(NO_ECC256)
    ecc_key     key;
    int         ret = 0;
    const char* qx =
              "bb33ac4c27504ac64aa504c33cde9f36db722dce94ea2bfacb2009392c16e861";
    const char* qy =
              "02e9af4dd302939a315b9792217ff0cf18da9111023486e82058330b803489d8";
    const char* d  =
             "45b66902739c6c85a1385b72e8e8c7acc4038d533504fa6c28dc348de1a8098c";
    const char* curveName = "SECP256R1";
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    const char* kNullStr = "";
#endif

    ret = wc_ecc_init(&key);

    /* Test good import */
    if (ret == 0) {
        ret = wc_ecc_import_raw(&key, qx, qy, d, curveName);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_import_raw(NULL, qx, qy, d, curveName);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_raw(&key, NULL, qy, d, curveName);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_raw(&key, qx, NULL, d, curveName);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_raw(&key, qx, qy, d, NULL);
        }
    #ifdef WOLFSSL_VALIDATE_ECC_IMPORT
        if (ret == BAD_FUNC_ARG) {
        #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
            wc_ecc_free(&key);
        #endif
            ret = wc_ecc_import_raw(&key, kNullStr, kNullStr, kNullStr, curveName);
            if (ret == ECC_INF_E)
                ret = BAD_FUNC_ARG; /* This is expected by other tests */
        }
    #endif
    #if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        if (ret == BAD_FUNC_ARG) {
        #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
            wc_ecc_free(&key);
        #endif
            ret = wc_ecc_import_raw(&key, "0", qy, d, curveName);
            /* Note: SP math "is point" failure returns MP_VAL */
            if (ret == ECC_INF_E || ret == MP_VAL) {
                ret = BAD_FUNC_ARG; /* This is expected by other tests */
            }
        }
        if (ret == BAD_FUNC_ARG) {
        #if !defined(USE_FAST_MATH) && !defined(WOLFSSL_SP_MATH)
            wc_ecc_free(&key);
        #endif
            ret = wc_ecc_import_raw(&key, qx, "0", d, curveName);
            /* Note: SP math "is point" failure returns MP_VAL */
            if (ret == ECC_INF_E || ret == MP_VAL) {
                ret = BAD_FUNC_ARG; /* This is expected by other tests */
            }
        }
    #endif

        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }

    wc_ecc_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_import_raw */

static int test_wc_ecc_import_unsigned(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(NO_ECC256) && !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && HAVE_FIPS_VERSION >= 2))
    ecc_key     key;
    const byte  qx[] = {
        0xbb, 0x33, 0xac, 0x4c, 0x27, 0x50, 0x4a, 0xc6,
        0x4a, 0xa5, 0x04, 0xc3, 0x3c, 0xde, 0x9f, 0x36,
        0xdb, 0x72, 0x2d, 0xce, 0x94, 0xea, 0x2b, 0xfa,
        0xcb, 0x20, 0x09, 0x39, 0x2c, 0x16, 0xe8, 0x61
    };
    const byte  qy[] = {
        0x02, 0xe9, 0xaf, 0x4d, 0xd3, 0x02, 0x93, 0x9a,
        0x31, 0x5b, 0x97, 0x92, 0x21, 0x7f, 0xf0, 0xcf,
        0x18, 0xda, 0x91, 0x11, 0x02, 0x34, 0x86, 0xe8,
        0x20, 0x58, 0x33, 0x0b, 0x80, 0x34, 0x89, 0xd8
    };
    const byte  d[] = {
        0x45, 0xb6, 0x69, 0x02, 0x73, 0x9c, 0x6c, 0x85,
        0xa1, 0x38, 0x5b, 0x72, 0xe8, 0xe8, 0xc7, 0xac,
        0xc4, 0x03, 0x8d, 0x53, 0x35, 0x04, 0xfa, 0x6c,
        0x28, 0xdc, 0x34, 0x8d, 0xe1, 0xa8, 0x09, 0x8c
    };
#ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    const byte nullBytes[32] = {0};
#endif
    int curveId = ECC_SECP256R1;
    int ret;

    ret = wc_ecc_init(&key);

    if (ret == 0) {
        ret = wc_ecc_import_unsigned(&key, (byte*)qx, (byte*)qy, (byte*)d,
            curveId);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_import_unsigned(NULL, (byte*)qx, (byte*)qy, (byte*)d,
            curveId);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_unsigned(&key, NULL, (byte*)qy, (byte*)d,
                curveId);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_unsigned(&key, (byte*)qx, NULL, (byte*)d,
                curveId);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_unsigned(&key, (byte*)qx, (byte*)qy, (byte*)d,
                ECC_CURVE_INVALID);
        }
    #ifdef WOLFSSL_VALIDATE_ECC_IMPORT
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_import_unsigned(&key, (byte*)nullBytes,
                (byte*)nullBytes, (byte*)nullBytes, curveId);
        }
    #endif
        if (ret == BAD_FUNC_ARG || ret == ECC_INF_E) {
            ret = 0;
        }
    }

    wc_ecc_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_import_unsigned */


/*
 * Testing wc_ecc_sig_size()
 */
static int test_wc_ecc_sig_size(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key     key;
    WC_RNG      rng;
    int         keySz = KEY16;
    int         ret = 0;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));
    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        ret = wc_ecc_sig_size(&key);
        if (ret <= (2 * keySz + SIG_HEADER_SZ + ECC_MAX_PAD_SZ)) {
            ret = 0;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_sig_size */

/*
 * Testing wc_ecc_ctx_new()
 */
static int test_wc_ecc_ctx_new(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    WC_RNG      rng;
    int         ret = 0;
    ecEncCtx*   cli = NULL;
    ecEncCtx*   srv = NULL;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        cli = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng);
        srv = wc_ecc_ctx_new(REQ_RESP_SERVER, &rng);
    }
    if (ret == 0 && (cli == NULL || srv == NULL)) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    wc_ecc_ctx_free(cli);
    wc_ecc_ctx_free(srv);

    /* Test bad args. */
    if (ret == 0) {
        /* wc_ecc_ctx_new_ex() will free if returned NULL. */
        cli = wc_ecc_ctx_new(0, &rng);
        if (cli != NULL) {
            ret = WOLFSSL_FATAL_ERROR;
        }
        cli = wc_ecc_ctx_new(REQ_RESP_CLIENT, NULL);
        if (cli != NULL) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_ctx_free(cli);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_ctx_new */

/*
 * Tesing wc_ecc_reset()
 */
static int test_wc_ecc_ctx_reset(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    ecEncCtx*   ctx = NULL;
    WC_RNG      rng;
    int         ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        if ( (ctx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng)) == NULL ) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_ecc_ctx_reset(ctx, &rng);
    }

    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_ctx_reset(NULL, &rng);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_ctx_reset(ctx, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_ctx_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_ctx_reset */

/*
 * Testing wc_ecc_ctx_set_peer_salt() and wc_ecc_ctx_get_own_salt()
 */
static int test_wc_ecc_ctx_set_peer_salt(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    WC_RNG          rng;
    ecEncCtx*       cliCtx      = NULL;
    ecEncCtx*       servCtx     = NULL;
    const byte*     cliSalt     = NULL;
    const byte*     servSalt    = NULL;
    int         ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        if ( ( (cliCtx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng)) == NULL ) ||
           ( (servCtx = wc_ecc_ctx_new(REQ_RESP_SERVER, &rng)) == NULL) ) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Test bad args. */
    if (ret == 0) {
        cliSalt  = wc_ecc_ctx_get_own_salt(NULL);
        if (cliSalt != NULL) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        cliSalt  = wc_ecc_ctx_get_own_salt(cliCtx);
        servSalt = wc_ecc_ctx_get_own_salt(servCtx);
        if (cliSalt == NULL || servSalt == NULL) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        ret = wc_ecc_ctx_set_peer_salt(cliCtx, servSalt);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_ctx_set_peer_salt(NULL, servSalt);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_ctx_set_peer_salt(cliCtx, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_ctx_free(cliCtx);
    wc_ecc_ctx_free(servCtx);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;

} /* END test_wc_ecc_ctx_set_peer_salt */

/*
 * Testing wc_ecc_ctx_set_info()
 */
static int test_wc_ecc_ctx_set_info(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG)
    ecEncCtx*   ctx = NULL;
    WC_RNG      rng;
    int         ret;
    const char* optInfo = "Optional Test Info.";
    int         optInfoSz = (int)XSTRLEN(optInfo);
    const char* badOptInfo = NULL;

    ret = wc_InitRng(&rng);
    if ( (ctx = wc_ecc_ctx_new(REQ_RESP_CLIENT, &rng)) == NULL || ret != 0 ) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0) {
        ret = wc_ecc_ctx_set_info(ctx, (byte*)optInfo, optInfoSz);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_ctx_set_info(NULL, (byte*)optInfo, optInfoSz);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_ctx_set_info(ctx, (byte*)badOptInfo, optInfoSz);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_ctx_set_info(ctx, (byte*)optInfo, -1);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_ctx_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_ctx_set_info */

/*
 * Testing wc_ecc_encrypt() and wc_ecc_decrypt()
 */
static int test_wc_ecc_encryptDecrypt(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_ENCRYPT) && !defined(WC_NO_RNG) && \
    defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    ecc_key     srvKey, cliKey, tmpKey;
    WC_RNG      rng;
    int         ret;
    const char* msg   = "EccBlock Size 16";
    word32      msgSz = (word32)XSTRLEN("EccBlock Size 16");
#ifdef WOLFSSL_ECIES_OLD
    byte        out[(sizeof("EccBlock Size 16") - 1) + WC_SHA256_DIGEST_SIZE];
#elif defined(WOLFSSL_ECIES_GEN_IV)
    byte        out[KEY20 * 2 + 1 + AES_BLOCK_SIZE +
                    (sizeof("EccBlock Size 16") - 1) + WC_SHA256_DIGEST_SIZE];
#else
    byte        out[KEY20 * 2 + 1 + (sizeof("EccBlock Size 16") - 1) + WC_SHA256_DIGEST_SIZE];
#endif
    word32      outSz = (word32)sizeof(out);
    byte        plain[sizeof("EccBlock Size 16")];
    word32      plainSz = (word32)sizeof(plain);
    int         keySz = KEY20;

    /* Init stack variables. */
    XMEMSET(out, 0, outSz);
    XMEMSET(plain, 0, plainSz);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&srvKey, 0, sizeof(srvKey));
    XMEMSET(&cliKey, 0, sizeof(cliKey));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&cliKey);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &cliKey);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &cliKey.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            ret = wc_ecc_init(&srvKey);
        }
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &srvKey);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &srvKey.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            ret = wc_ecc_init(&tmpKey);
        }
    }

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    if (ret == 0) {
        ret = wc_ecc_set_rng(&srvKey, &rng);
    }
    if (ret == 0) {
        ret = wc_ecc_set_rng(&cliKey, &rng);
    }
#endif

    if (ret == 0) {
        ret = wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, out,
                                                            &outSz, NULL);
    }
    if (ret == 0) {
        ret = wc_ecc_encrypt(NULL, &srvKey, (byte*)msg, msgSz, out,
                                                            &outSz, NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_encrypt(&cliKey, NULL, (byte*)msg, msgSz, out,
                                                            &outSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_encrypt(&cliKey, &srvKey, NULL, msgSz, out,
                                                            &outSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, NULL,
                                                            &outSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_encrypt(&cliKey, &srvKey, (byte*)msg, msgSz, out,
                                                            NULL, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#ifdef WOLFSSL_ECIES_OLD
    if (ret == 0) {
        tmpKey.dp = cliKey.dp;
        ret = wc_ecc_copy_point(&cliKey.pubkey, &tmpKey.pubkey);
    }
#endif

    if (ret == 0) {
        ret = wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz, plain,
                                                        &plainSz, NULL);
    }
    if (ret == 0) {
        ret = wc_ecc_decrypt(NULL, &tmpKey, out, outSz, plain,
                                                        &plainSz, NULL);
    #ifdef WOLFSSL_ECIES_OLD
        /* NULL parameter allowed in new implementations - public key comes from
         * the message. */
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_decrypt(&srvKey, NULL, out, outSz, plain,
                                                        &plainSz, NULL);
        }
    #endif
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_decrypt(&srvKey, &tmpKey, NULL, outSz, plain,
                                                        &plainSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz, NULL,
                                                        &plainSz, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_decrypt(&srvKey, &tmpKey, out, outSz,
                                                        plain, NULL, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (XMEMCMP(msg, plain, msgSz) != 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&tmpKey);
    wc_ecc_free(&cliKey);
    wc_ecc_free(&srvKey);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_encryptDecrypt */

/*
 * Testing wc_ecc_del_point() and wc_ecc_new_point()
 */
static int test_wc_ecc_del_point(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC)
    ecc_point*   pt;

    pt = wc_ecc_new_point();
    wc_ecc_del_point(pt);

    res = TEST_RES_CHECK(pt != NULL);
#endif
    return res;

} /* END test_wc_ecc_del_point */

/*
 * Testing wc_ecc_point_is_at_infinity(), wc_ecc_export_point_der(),
 * wc_ecc_import_point_der(), wc_ecc_copy_point(), wc_ecc_point_is_on_curve(),
 * and wc_ecc_cmp_point()
 */
static int test_wc_ecc_pointFns(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
    !defined(WOLFSSL_ATECC608A)
    ecc_key     key;
    WC_RNG      rng;
    int         ret;
    ecc_point*  point = NULL;
    ecc_point*  cpypt = NULL;
    int         idx = 0;
    int         keySz = KEY32;
    byte        der[DER_SZ(KEY32)];
    word32      derlenChk = 0;
    word32      derSz = DER_SZ(KEY32);

    /* Init stack variables. */
    XMEMSET(der, 0, derSz);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        point = wc_ecc_new_point();
        if (!point) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (ret == 0) {
        cpypt = wc_ecc_new_point();
        if (!cpypt) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Export */
    if (ret == 0) {
        ret = wc_ecc_export_point_der((idx = key.idx), &key.pubkey,
                                                       NULL, &derlenChk);
        /* Check length value. */
        if (derSz == derlenChk && ret == LENGTH_ONLY_E) {
            ret = wc_ecc_export_point_der((idx = key.idx), &key.pubkey,
                                                           der, &derSz);
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_export_point_der(-2, &key.pubkey, der, &derSz);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_export_point_der((idx = key.idx), NULL, der, &derSz);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_export_point_der((idx = key.idx), &key.pubkey,
                                                                der, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Import */
    if (ret == 0) {
        ret = wc_ecc_import_point_der(der, derSz, idx, point);
        /* Condition double checks wc_ecc_cmp_point().  */
        if (ret == 0 &&
            XMEMCMP((void *)&key.pubkey, (void *)point, sizeof(key.pubkey))) {
            ret = wc_ecc_cmp_point(&key.pubkey, point);
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_import_point_der(NULL, derSz, idx, point);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_import_point_der(der, derSz, idx, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_import_point_der(der, derSz, -1, point);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_import_point_der(der, derSz + 1, idx, point);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Copy */
    if (ret == 0) {
        ret = wc_ecc_copy_point(point, cpypt);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_copy_point(NULL, cpypt);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_copy_point(point, NULL);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* Compare point */
    if (ret == 0) {
        ret = wc_ecc_cmp_point(point, cpypt);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_cmp_point(NULL, cpypt);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_cmp_point(point, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    /* At infinity if return == 1, otherwise return == 0. */
    if (ret == 0) {
        ret = wc_ecc_point_is_at_infinity(point);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_point_is_at_infinity(NULL);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

#if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2)))
#ifdef USE_ECC_B_PARAM
    /* On curve if ret == 0 */
    if (ret == 0) {
        ret = wc_ecc_point_is_on_curve(point, idx);
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_point_is_on_curve(NULL, idx);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_point_is_on_curve(point, 1000);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif /* USE_ECC_B_PARAM */
#endif /* !HAVE_SELFTEST && (!HAVE_FIPS || HAVE_FIPS_VERSION > 2) */

    /* Free */
    wc_ecc_del_point(point);
    wc_ecc_del_point(cpypt);
    wc_ecc_free(&key);
    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_pointFns */


/*
 * Testing wc_ecc_sahred_secret_ssh()
 */
static int test_wc_ecc_shared_secret_ssh(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_DHE) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
    !defined(WOLFSSL_ATECC608A)
    ecc_key     key, key2;
    WC_RNG      rng;
    int         ret;
    int         keySz = KEY32;
    int         key2Sz = KEY24;
    byte        secret[KEY32];
    word32      secretLen = keySz;

    /* Init stack variables. */
    XMEMSET(secret, 0, secretLen);
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));
    XMEMSET(&key2, 0, sizeof(key2));
    /* Make keys */
    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (wc_FreeRng(&rng) && ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (ret == 0) {
        ret = wc_InitRng(&rng);
        if (ret == 0) {
            ret = wc_ecc_init(&key2);
        }
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, key2Sz, &key2);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key2.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    if (ret == 0) {
        ret = wc_ecc_set_rng(&key, &rng);
    }
#endif

    if (ret == 0) {
        ret = wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret, &secretLen);
    }
    /* Pass in bad args. */
    if (ret == 0) {
        ret = wc_ecc_shared_secret_ssh(NULL, &key2.pubkey, secret, &secretLen);
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_shared_secret_ssh(&key, NULL, secret, &secretLen);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_shared_secret_ssh(&key, &key2.pubkey, NULL, &secretLen);
        }
        if (ret == BAD_FUNC_ARG) {
            ret = wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret, NULL);
        }
        if (ret == BAD_FUNC_ARG) {
            key.type = ECC_PUBLICKEY;
            ret = wc_ecc_shared_secret_ssh(&key, &key2.pubkey, secret, &secretLen);
            if (ret == ECC_BAD_ARG_E) {
                ret = 0;
            }
            else if (ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);
    wc_ecc_free(&key2);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_shared_secret_ssh */

/*
 * Testing wc_ecc_verify_hash_ex() and wc_ecc_verify_hash_ex()
 */
static int test_wc_ecc_verify_hash_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_SIGN) && defined(WOLFSSL_PUBLIC_MP) \
    && !defined(WC_NO_RNG) && !defined(WOLFSSL_ATECC508A) && \
       !defined(WOLFSSL_ATECC608A) && !defined(WOLFSSL_KCAPI_ECC)
    ecc_key         key;
    WC_RNG          rng;
    int             ret;
    mp_int          r;
    mp_int          s;
    mp_int          z;
    unsigned char   hash[] = "Everyone gets Friday off.EccSig";
    unsigned char   iHash[] = "Everyone gets Friday off.......";
    unsigned char   shortHash[] = TEST_STRING;
    word32          hashlen = sizeof(hash);
    word32          iHashLen = sizeof(iHash);
    word32          shortHashLen = sizeof(shortHash);
    int             keySz = KEY32;
    int             sig = WOLFSSL_FATAL_ERROR;
    int             ver = WOLFSSL_FATAL_ERROR;
    int             verify_ok = 0;

    /* Initialize r and s. */
    ret = mp_init_multi(&r, &s, &z, NULL, NULL, NULL);
    if (ret != MP_OKAY) {
        return MP_INIT_E;
    }

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, keySz, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }
    if (ret == 0) {
        ret = wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, &r, &s);
        if (ret == 0) {
            /* verify_ok should be 1. */
            ret = wc_ecc_verify_hash_ex(&r, &s, hash, hashlen, &verify_ok, &key);
            if (verify_ok != 1 && ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* verify_ok should be 0 */
            ret = wc_ecc_verify_hash_ex(&r, &s, iHash, iHashLen,
                                                    &verify_ok, &key);
            if (verify_ok != 0 && ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
        if (ret == 0) {
            /* verify_ok should be 0. */
            ret = wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen,
                                                            &verify_ok, &key);
            if (verify_ok != 0 && ret == 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

    /* Test bad args. */
    if (ret == 0) {
        if (wc_ecc_sign_hash_ex(NULL, hashlen, &rng, &key, &r, &s)
                                                == ECC_BAD_ARG_E) {
            sig = 0;
        }
        if (sig == 0 && wc_ecc_sign_hash_ex(hash, hashlen, NULL, &key, &r, &s)
                                                            != ECC_BAD_ARG_E) {
            sig = WOLFSSL_FATAL_ERROR;
        }
        if (sig == 0 && wc_ecc_sign_hash_ex(hash, hashlen, &rng, NULL, &r, &s)
                                                            != ECC_BAD_ARG_E) {
            sig = WOLFSSL_FATAL_ERROR;
        }
        if (sig == 0 && wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, NULL, &s)
                                                            != ECC_BAD_ARG_E) {
            sig = WOLFSSL_FATAL_ERROR;
        }
        if (sig == 0 && wc_ecc_sign_hash_ex(hash, hashlen, &rng, &key, &r, NULL)
                                                            != ECC_BAD_ARG_E) {
            sig = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        if (wc_ecc_verify_hash_ex(NULL, &s, shortHash, shortHashLen, &verify_ok, &key)
                                                            == ECC_BAD_ARG_E) {
            ver = 0;
        }
        if (ver == 0 && wc_ecc_verify_hash_ex(&r, NULL, shortHash, shortHashLen,
                                                &verify_ok, &key) != ECC_BAD_ARG_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (wc_ecc_verify_hash_ex(&z, &s, shortHash, shortHashLen, &verify_ok, &key)
                                                            != MP_ZERO_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (wc_ecc_verify_hash_ex(&r, &z, shortHash, shortHashLen, &verify_ok, &key)
                                                            != MP_ZERO_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (wc_ecc_verify_hash_ex(&z, &z, shortHash, shortHashLen, &verify_ok, &key)
                                                            != MP_ZERO_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (ver == 0 && wc_ecc_verify_hash_ex(&r, &s, NULL, shortHashLen, &verify_ok,
                                                       &key) != ECC_BAD_ARG_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (ver == 0 && wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen,
                                                NULL, &key) != ECC_BAD_ARG_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
        if (ver == 0 && wc_ecc_verify_hash_ex(&r, &s, shortHash, shortHashLen,
                                                &verify_ok, NULL) != ECC_BAD_ARG_E) {
            ver = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ecc_free(&key);
    mp_free(&r);
    mp_free(&s);
    if (wc_FreeRng(&rng)) {
        return WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0 && (sig != 0 || ver != 0)) {
        ret = WOLFSSL_FATAL_ERROR;
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_verify_hash_ex */

/*
 * Testing wc_ecc_mulmod()
 */

static int test_wc_ecc_mulmod(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && \
    !(defined(WOLFSSL_ATECC508A) || defined(WOLFSSL_ATECC608A) || \
      defined(WOLFSSL_VALIDATE_ECC_IMPORT))
    ecc_key     key1, key2, key3;
    WC_RNG      rng;
    int         ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key1);
        if (ret == 0) {
            ret = wc_ecc_init(&key2);
        }
        if (ret == 0) {
            ret = wc_ecc_init(&key3);
        }
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY32, &key1);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key1.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        wc_FreeRng(&rng);
    }
    if (ret == 0) {
        ret = wc_ecc_import_raw_ex(&key2, key1.dp->Gx, key1.dp->Gy, key1.dp->Af,
                                                                 ECC_SECP256R1);
        if (ret == 0) {
            ret = wc_ecc_import_raw_ex(&key3, key1.dp->Gx, key1.dp->Gy,
                                        key1.dp->prime, ECC_SECP256R1);
        }
    }

    if (ret == 0) {
        ret = wc_ecc_mulmod(&key1.k, &key2.pubkey, &key3.pubkey, &key2.k,
                                                            &key3.k, 1);
    }

    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_mulmod(NULL, &key2.pubkey, &key3.pubkey, &key2.k,
                                                            &key3.k, 1);
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_mulmod(&key1.k, NULL, &key3.pubkey, &key2.k,
                                                            &key3.k, 1);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_mulmod(&key1.k, &key2.pubkey, NULL, &key2.k,
                                                            &key3.k, 1);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = wc_ecc_mulmod(&key1.k, &key2.pubkey, &key3.pubkey,
                                                            &key2.k, NULL, 1);
        }
        if (ret == ECC_BAD_ARG_E) {
            ret = 0;
        }
        else if (ret == 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    wc_ecc_free(&key1);
    wc_ecc_free(&key2);
    wc_ecc_free(&key3);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif /* HAVE_ECC && !WOLFSSL_ATECC508A */
    return res;
} /* END test_wc_ecc_mulmod */

/*
 * Testing wc_ecc_is_valid_idx()
 */
static int test_wc_ecc_is_valid_idx(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG)
    ecc_key     key;
    WC_RNG      rng;
    int         ret;
    int         iVal = -2;
    int         iVal2 = 3000;

    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(&key, 0, sizeof(key));

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, 32, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
    }

    if (ret == 0) {
        ret = wc_ecc_is_valid_idx(key.idx);
        if (ret == 1) {
            ret = 0;
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    /* Test bad args. */
    if (ret == 0) {
        ret = wc_ecc_is_valid_idx(iVal); /* should return 0 */
        if (ret == 0) {
            ret = wc_ecc_is_valid_idx(iVal2);
        }
        if (ret != 0) {
            ret = WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_FreeRng(&rng) && ret == 0) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    wc_ecc_free(&key);
#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;


} /* END test_wc_ecc_is_valid_idx */
/*
 * Testing wc_ecc_get_curve_id_from_oid()
 */
static int test_wc_ecc_get_curve_id_from_oid(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(NO_ECC256) && !defined(HAVE_SELFTEST) && \
    !defined(HAVE_FIPS)
    const byte oid[] = {0x2A,0x86,0x48,0xCE,0x3D,0x03,0x01,0x07};
    word32 len = sizeof(oid);
    int ret;

    /* Bad Cases */
    ret = wc_ecc_get_curve_id_from_oid(NULL, len);
    if (ret == BAD_FUNC_ARG) {
        ret = 0;
    }
    if (ret == 0) {
        ret = wc_ecc_get_curve_id_from_oid(oid, 0);
        if (ret == ECC_CURVE_INVALID) {
            ret = 0;
        }
    }
    /* Good Case */
    if (ret == 0) {
        ret = wc_ecc_get_curve_id_from_oid(oid, len);
        if (ret == ECC_SECP256R1) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* END test_wc_ecc_get_curve_id_from_oid */
/*
 * Testing wc_ecc_sig_size_calc()
 */
static int test_wc_ecc_sig_size_calc(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST)
    ecc_key     key;
    WC_RNG      rng;
    int         sz = 0;
    int         ret = 0;

    ret = wc_InitRng(&rng);
    if (ret == 0) {
        ret = wc_ecc_init(&key);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, 16, &key);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        sz = key.dp->size;
    }
    if (ret == 0) {
        ret = wc_ecc_sig_size_calc(sz);
        if (ret > 0) {
            ret = 0;
        }
    }
    wc_ecc_free(&key);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
} /* END test_wc_ecc_sig_size_calc */
/*
 * Testing ToTraditional
 */
static int test_ToTraditional(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_ASN) && (defined(HAVE_PKCS8) || defined(HAVE_PKCS12)) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL))

    XFILE   f;
    byte    input[TWOK_BUF];
    word32  sz;
    int     ret;

    f = XFOPEN("./certs/server-keyPkcs8.der", "rb");
    AssertTrue((f != XBADFILE));
    sz = (word32)XFREAD(input, 1, sizeof(input), f);
    XFCLOSE(f);

    /* Good case */
    ret = ToTraditional(input, sz);
    if (ret > 0) {
        ret = 0;
    }
    /* Bad cases */
    if (ret == 0) {
        ret = ToTraditional(NULL, 0);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = ToTraditional(NULL, sz);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    if (ret == 0) {
        ret = ToTraditional(input, 0);
        if (ret == ASN_PARSE_E || ret == BUFFER_E) {
            ret = 0;
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_ToTraditional*/

/*
 * Testing wc_EccPrivateKeyToDer
 */
static int test_wc_EccPrivateKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    byte            output[ONEK_BUF];
    ecc_key         eccKey;
    WC_RNG          rng;
    word32          inLen;
    int             ret;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        ret = wc_ecc_init(&eccKey);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        inLen = (word32)sizeof(output);
        /* Bad Cases */
        if (ret == 0) {
            ret = wc_EccPrivateKeyToDer(NULL, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_EccPrivateKeyToDer(NULL, output, inLen);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_EccPrivateKeyToDer(&eccKey, NULL, inLen);
            if (ret == LENGTH_ONLY_E) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_EccPrivateKeyToDer(&eccKey, output, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        /*Good Case */
        if (ret == 0) {
            ret = wc_EccPrivateKeyToDer(&eccKey, output, inLen);
            if (ret > 0) {
        #if defined(OPENSSL_EXTRA) && defined(HAVE_ALL_CURVES)
                /* test importing private only into a PKEY struct */
                EC_KEY*   ec;
                EVP_PKEY* pkey;
                const unsigned char* der = output;

                pkey = d2i_PrivateKey(EVP_PKEY_EC, NULL, &der, ret);
                AssertNotNull(pkey);

                der = output;
                ec  = d2i_ECPrivateKey(NULL, &der, ret);
                AssertNotNull(ec);
                AssertIntEQ(EVP_PKEY_assign_EC_KEY(pkey, ec), SSL_SUCCESS);
                EVP_PKEY_free(pkey); /* EC_KEY should be free'd by free'ing pkey */
        #endif
                ret = 0;
            }
        }
        wc_ecc_free(&eccKey);
    }
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_EccPrivateKeyToDer*/

/*
 * Testing wc_DhPublicKeyDecode
 */
static int test_wc_DhPublicKeyDecode(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_DH
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)
    DhKey  key;
    word32 inOutIdx;

    AssertIntEQ(wc_InitDhKey(&key), 0);

    AssertIntEQ(wc_DhPublicKeyDecode(NULL,NULL,NULL,0),
                                                        BAD_FUNC_ARG);
    AssertIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,NULL,NULL,0),
                                                        BAD_FUNC_ARG);
    AssertIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,NULL,NULL,0),
                                                        BAD_FUNC_ARG);
    inOutIdx = 0;
    AssertIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,NULL, 0),
                                                        BAD_FUNC_ARG);
    inOutIdx = 0;
    AssertIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,&key, 0),
                                                        BAD_FUNC_ARG);
    inOutIdx = 0;
    AssertIntEQ(wc_DhPublicKeyDecode(dh_pub_key_der_2048,&inOutIdx,&key,
                                            sizeof_dh_pub_key_der_2048), 0);
    AssertTrue(key.p.used != 0 && key.g.used != 0 && key.q.used == 0 &&
              key.pub.used != 0 && key.priv.used == 0);

    wc_FreeDhKey(&key);

    res = TEST_RES_CHECK(1);
#endif
#endif /* !NO_DH */
    return res;
}

/*
 * Testing wc_Ed25519KeyToDer
 */
static int test_wc_Ed25519KeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte            output[ONEK_BUF];
    ed25519_key     ed25519Key;
    WC_RNG          rng;
    word32          inLen;
    int             ret;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        ret = wc_ed25519_init(&ed25519Key);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key);
        }
        inLen = (word32)sizeof(output);

        /* Bad Cases */
        if (ret == 0) {
            ret = wc_Ed25519KeyToDer(NULL, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519KeyToDer(NULL, output, inLen);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519KeyToDer(&ed25519Key, output, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        /* Good Cases */
        if (ret == 0) {
            /* length only */
            ret = wc_Ed25519KeyToDer(&ed25519Key, NULL, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519KeyToDer(&ed25519Key, output, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        wc_ed25519_free(&ed25519Key);
    }
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_Ed25519KeyToDer*/
/*
 * Testing wc_Ed25519PrivateKeyToDer
 */
static int test_wc_Ed25519PrivateKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte            output[ONEK_BUF];
    ed25519_key     ed25519PrivKey;
    WC_RNG          rng;
    word32          inLen;
    int             ret;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        ret = wc_ed25519_init(&ed25519PrivKey);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519PrivKey);
        }
        inLen = (word32)sizeof(output);

        /* Bad Cases */
        if (ret == 0) {
            ret = wc_Ed25519PrivateKeyToDer(NULL, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519PrivateKeyToDer(NULL, output, inLen);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, output, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        /* Good Cases */
        if (ret == 0) {
            /* length only */
            ret = wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, NULL, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed25519PrivateKeyToDer(&ed25519PrivKey, output, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        wc_ed25519_free(&ed25519PrivKey);
    }
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_Ed25519PrivateKeyToDer*/
/*
 * Testing wc_Ed448KeyToDer
 */
static int test_wc_Ed448KeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte            output[ONEK_BUF];
    ed448_key       ed448Key;
    WC_RNG          rng;
    word32          inLen;
    int             ret;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        ret = wc_ed448_init(&ed448Key);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key);
        }
        inLen = sizeof(output);

        /* Bad Cases */
        if (ret == 0) {
            ret = wc_Ed448KeyToDer(NULL, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448KeyToDer(NULL, output, inLen);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448KeyToDer(&ed448Key, output, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        /* Good Cases */
        if (ret == 0) {
            /* length only */
            ret = wc_Ed448KeyToDer(&ed448Key, NULL, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448KeyToDer(&ed448Key, output, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        wc_ed448_free(&ed448Key);
    }
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_Ed448KeyToDer*/
/*
 * Testing wc_Ed448PrivateKeyToDer
 */
static int test_wc_Ed448PrivateKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN))
    byte            output[ONEK_BUF];
    ed448_key       ed448PrivKey;
    WC_RNG          rng;
    word32          inLen;
    int             ret;

    ret = wc_InitRng(&rng);

    if (ret == 0) {
        ret = wc_ed448_init(&ed448PrivKey);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448PrivKey);
        }
        inLen = sizeof(output);

        /* Bad Cases */
        if (ret == 0) {
            ret = wc_Ed448PrivateKeyToDer(NULL, NULL, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448PrivateKeyToDer(NULL, output, inLen);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448PrivateKeyToDer(&ed448PrivKey, output, 0);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        /* Good cases */
        if (ret == 0) {
            /* length only */
            ret = wc_Ed448PrivateKeyToDer(&ed448PrivKey, NULL, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_Ed448PrivateKeyToDer(&ed448PrivKey, output, inLen);
            if (ret > 0) {
                ret = 0;
            }
        }
        wc_ed448_free(&ed448PrivKey);
    }
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_Ed448PrivateKeyToDer*/
/*
 * Testing wc_SetSubjectBuffer
 */
static int test_wc_SetSubjectBuffer(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    Cert    cert;
    FILE*   file;
    byte*   der;
    word32  derSz;
    int     ret = 0;

    derSz = FOURK_BUF;
    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ret = -1;
    }
    if (ret == 0) {
        file = XFOPEN("./certs/ca-cert.der", "rb");
        if (file != NULL) {
            derSz = (word32)XFREAD(der, 1, FOURK_BUF, file);
            XFCLOSE(file);
        }
        else {
            ret = -1;
        }
    }
    if (ret == 0) {
        ret = wc_InitCert(&cert);
    }

    if (ret == 0) {
        ret = wc_SetSubjectBuffer(&cert, der, derSz);
    }

    if (ret == 0) {
        ret = wc_SetSubjectBuffer(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG) {
            ret = 0;
        }
    }
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_SetSubjectBuffer*/

/*
 * Testing wc_SetSubjectKeyIdFromPublicKey_ex
 */
static int test_wc_SetSubjectKeyIdFromPublicKey_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG          rng;
    Cert            cert;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key     ed25519Key;
#endif
#if !defined(NO_RSA) && defined(HAVE_RSA)
    RsaKey          rsaKey;
    int             bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key         eccKey;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key       ed448Key;
#endif
    int             ret = 0;

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, testDevId);
#else
    ret = wc_InitRng(&rng);
#endif

    wc_InitCert(&cert);
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    if (ret == 0) { /*ED25519*/
        ret = wc_ed25519_init(&ed25519Key);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key);
        }
        if (ret == 0) {
            ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
                                                  &ed25519Key);
        }
        wc_ed25519_free(&ed25519Key);
    }
#endif
#if !defined(NO_RSA) && defined(HAVE_RSA) && defined(WOLFSSL_KEY_GEN)
    if (ret == 0) { /*RSA*/
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        if (ret == 0) {
            MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng);
        }
        if (ret == 0) {
            ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey);
        }
        wc_FreeRsaKey(&rsaKey);
    }
#endif
#if defined(HAVE_ECC)
    if (ret == 0) { /*ECC*/
        ret = wc_ecc_init(&eccKey);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey);
        }
        wc_ecc_free(&eccKey);
    }
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    if (ret == 0) { /*ED448*/
        ret = wc_ed448_init(&ed448Key);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key);
        }
        if (ret == 0) {
            ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED448_TYPE,
                                                  &ed448Key);
        }
        wc_ed448_free(&ed448Key);
    }
#endif

    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}/* End test_wc_SetSubjectKeyIdFromPublicKey_ex*/

/*
 * Testing wc_SetAuthKeyIdFromPublicKey_ex
 */
static int test_wc_SetAuthKeyIdFromPublicKey_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG          rng;
    Cert            cert;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key     ed25519Key;
#endif
#if !defined(NO_RSA) && defined(HAVE_RSA)
    RsaKey          rsaKey;
    int             bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key         eccKey;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key       ed448Key;
#endif
    int             ret = 0;

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, testDevId);
#else
    ret = wc_InitRng(&rng);
#endif

    wc_InitCert(&cert);
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    if (ret == 0) { /*ED25519*/
        ret = wc_ed25519_init(&ed25519Key);
        if (ret == 0) {
            ret = wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key);
        }
        if (ret == 0) {
            ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
                                                  &ed25519Key);
        }
        wc_ed25519_free(&ed25519Key);
    }
#endif
#if !defined(NO_RSA) && defined(HAVE_RSA) && defined(WOLFSSL_KEY_GEN)
    if (ret == 0) { /*RSA*/
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        if (ret == 0) {
            MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng);
        }
        if (ret == 0) {
            ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey);
        }
        wc_FreeRsaKey(&rsaKey);
    }
#endif
#if defined(HAVE_ECC)
    if (ret == 0) { /*ECC*/
        ret = wc_ecc_init(&eccKey);
        if (ret == 0) {
            ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
        }
        if (ret == 0) {
            ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey);
        }
        wc_ecc_free(&eccKey);
    }
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    if (ret == 0) { /*ED448*/
        ret = wc_ed448_init(&ed448Key);
        if (ret == 0) {
            ret = wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key);
        }
        if (ret == 0) {
            ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED448_TYPE,
                                                  &ed448Key);
        }
        wc_ed448_free(&ed448Key);
    }
#endif

    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(ret == 0);
#endif /*defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)*/
    return res;
}/* End test_wc_SetAuthKeyIdFromPublicKey_ex*/

/*
 * Testing wc_PKCS7_New()
 */
static int test_wc_PKCS7_New(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7;

    pkcs7 = wc_PKCS7_New(NULL, testDevId);
    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(pkcs7 != NULL);
#endif
    return res;
} /* END test-wc_PKCS7_New */

/*
 * Testing wc_PKCS7_Init()
 */
static int test_wc_PKCS7_Init(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7;
    void*       heap = NULL;

    pkcs7 = wc_PKCS7_New(heap, testDevId);
    AssertNotNull(pkcs7);

    AssertIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    /* Pass in bad args. */
    AssertIntEQ(wc_PKCS7_Init(NULL, heap, testDevId), BAD_FUNC_ARG);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test-wc_PKCS7_Init */


/*
 * Testing wc_PKCS7_InitWithCert()
 */
static int test_wc_PKCS7_InitWithCert(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*       pkcs7;

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char    cert[sizeof(client_cert_der_2048)];
        int              certSz = (int)sizeof(cert);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_2048, sizeof(client_cert_der_2048));
    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char    cert[sizeof(client_cert_der_1024)];
        int              certSz = (int)sizeof(cert);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_1024, sizeof_client_cert_der_1024);
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp;
        int             certSz;
        fp = XFOPEN("./certs/1024/client-cert.der", "rb");
        AssertTrue(fp != XBADFILE);

        certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024, fp);
        XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        int              certSz = (int)sizeof(cert);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof(cliecc_cert_der_256));
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp;
        int             certSz;
        fp = XFOPEN("./certs/client-ecc-cert.der", "rb");

        AssertTrue(fp != XBADFILE);

        certSz = (int)XFREAD(cert, 1, sizeof(cliecc_cert_der_256), fp);
        XFCLOSE(fp);
    #endif
#else
        #error PKCS7 requires ECC or RSA
#endif

#ifdef HAVE_ECC
    {
    /* bad test case from ZD 11011, malformed cert gives bad ECC key */
        static unsigned char certWithInvalidEccKey[] = {
        0x30, 0x82, 0x03, 0x5F, 0x30, 0x82, 0x03, 0x04, 0xA0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79,
        0x42, 0x83, 0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x31, 0xAA, 0x2C, 0x30,
        0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30,
        0x1E, 0x17, 0x0D, 0x32, 0x30, 0x30, 0x36, 0x31, 0x39, 0x31, 0x33, 0x32,
        0x33, 0x34, 0x31, 0x5A, 0x17, 0x0D, 0x32, 0x33, 0x30, 0x33, 0x31, 0x36,
        0x31, 0x33, 0x32, 0x33, 0x34, 0x31, 0x5A, 0x30, 0x81, 0x8D, 0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
        0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x06, 0x4F, 0x72,
        0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x04,
        0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D, 0x31, 0x13, 0x30, 0x11,
        0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43, 0x6C, 0x69, 0x65, 0x6E,
        0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55,
        0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74, 0x31, 0x18, 0x30, 0x26,
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
        0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x1F,
        0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C, 0x66,
        0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x59, 0x30, 0x13, 0x06,
        0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86,
        0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x02, 0x00, 0x04, 0x55, 0xBF,
        0xF4, 0x0F, 0x44, 0x50, 0x9A, 0x3D, 0xCE, 0x9B, 0xB7, 0xF0, 0xC5, 0x4D,
        0xF5, 0x70, 0x7B, 0xD4, 0xEC, 0x24, 0x8E, 0x19, 0x80, 0xEC, 0x5A, 0x4C,
        0xA2, 0x24, 0x03, 0x62, 0x2C, 0x9B, 0xDA, 0xEF, 0xA2, 0x35, 0x12, 0x43,
        0x84, 0x76, 0x16, 0xC6, 0x56, 0x95, 0x06, 0xCC, 0x01, 0xA9, 0xBD, 0xF6,
        0x75, 0x1A, 0x42, 0xF7, 0xBD, 0xA9, 0xB2, 0x36, 0x22, 0x5F, 0xC7, 0x5D,
        0x7F, 0xB4, 0xA3, 0x82, 0x01, 0x3E, 0x30, 0x82, 0x01, 0x3A, 0x30, 0x1D,
        0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x6B, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0x30, 0x81, 0xCD, 0x06, 0x03, 0x55, 0x1D,
        0x23, 0x04, 0x81, 0xC5, 0x30, 0x81, 0xC2, 0x80, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x72, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0xA1, 0x81, 0x93, 0xA4, 0x81, 0x90, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x08, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x30, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x82,
        0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79, 0x42, 0x83,
        0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x32, 0xAA, 0x2C, 0x30, 0x0C, 0x06,
        0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30,
        0x1C, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x15, 0x30, 0x13, 0x82, 0x0B,
        0x65, 0x78, 0x61, 0x6D, 0x70, 0x6C, 0x65, 0x2E, 0x63, 0x6F, 0x6D, 0x87,
        0x04, 0x23, 0x00, 0x00, 0x01, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x25,
        0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07,
        0x03, 0x01, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02,
        0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
        0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xE4, 0xA0, 0x23, 0x26,
        0x2B, 0x0B, 0x42, 0x0F, 0x97, 0x37, 0x6D, 0xCB, 0x14, 0x23, 0xC3, 0xC3,
        0xE6, 0x44, 0xCF, 0x5F, 0x4C, 0x26, 0xA3, 0x72, 0x64, 0x7A, 0x9C, 0xCB,
        0x64, 0xAB, 0xA6, 0xBE, 0x02, 0x21, 0x00, 0xAA, 0xC5, 0xA3, 0x50, 0xF6,
        0xF1, 0xA5, 0xDB, 0x05, 0xE0, 0x75, 0xD2, 0xF7, 0xBA, 0x49, 0x5F, 0x8F,
        0x7D, 0x1C, 0x44, 0xB1, 0x6E, 0xDF, 0xC8, 0xDA, 0x10, 0x48, 0x2D, 0x53,
        0x08, 0xA8, 0xB4};
#endif
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        /* If initialization is not successful, it's free'd in init func. */
        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, (word32)certSz), 0);

        wc_PKCS7_Free(pkcs7);
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));

        /* Valid initialization usage. */
        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* Pass in bad args. No need free for null checks, free at end.*/
        AssertIntEQ(wc_PKCS7_InitWithCert(NULL, (byte*)cert, (word32)certSz),
                    BAD_FUNC_ARG);
        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, (word32)certSz),
                    BAD_FUNC_ARG);

#ifdef HAVE_ECC
        AssertIntLT(wc_PKCS7_InitWithCert(pkcs7, certWithInvalidEccKey,
                                          sizeof(certWithInvalidEccKey)), 0);
    }
#endif

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_InitWithCert */


/*
 * Testing wc_PKCS7_EncodeData()
 */
static int test_wc_PKCS7_EncodeData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7;
    byte        output[FOURK_BUF];
    byte        data[] = "My encoded DER cert.";

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char cert[sizeof(client_cert_der_2048)];
        unsigned char key[sizeof(client_key_der_2048)];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);

    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char cert[sizeof(sizeof_client_cert_der_1024)];
        unsigned char key[sizeof_client_key_der_1024];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz;
        int             keySz;

        fp = XFOPEN("./certs/1024/client-cert.der", "rb");
        AssertTrue(fp != XBADFILE);
        certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/1024/client-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp);
        XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz, keySz;

        fp = XFOPEN("./certs/client-ecc-cert.der", "rb");
        AssertTrue(fp != XBADFILE);
        certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/client-ecc-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp);
        XFCLOSE(fp);
    #endif
#endif

    XMEMSET(output, 0, sizeof(output));

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, certSz), 0);

    pkcs7->content = data;
    pkcs7->contentSz = sizeof(data);
    pkcs7->privateKey = key;
    pkcs7->privateKeySz = keySz;
    AssertIntGT(wc_PKCS7_EncodeData(pkcs7, output, (word32)sizeof(output)), 0);

    /* Test bad args. */
    AssertIntEQ(wc_PKCS7_EncodeData(NULL, output, (word32)sizeof(output)),
                                                            BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeData(pkcs7, NULL, (word32)sizeof(output)),
                                                            BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeData(pkcs7, output, 5), BUFFER_E);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}  /* END test_wc_PKCS7_EncodeData */


#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
/* RSA sign raw digest callback */
static int rsaSignRawDigestCb(PKCS7* pkcs7, byte* digest, word32 digestSz,
                              byte* out, word32 outSz, byte* privateKey,
                              word32 privateKeySz, int devid, int hashOID)
{
    /* specific DigestInfo ASN.1 encoding prefix for a SHA2565 digest */
    byte digInfoEncoding[] = {
        0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86,
        0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05,
        0x00, 0x04, 0x20
    };

    int ret;
    byte digestInfo[ONEK_BUF];
    byte sig[FOURK_BUF];
    word32 digestInfoSz = 0;
    word32 idx = 0;
    RsaKey rsa;

    /* SHA-256 required only for this example callback due to above
     * digInfoEncoding[] */
    if (pkcs7 == NULL || digest == NULL || out == NULL ||
        (sizeof(digestInfo) < sizeof(digInfoEncoding) + digestSz) ||
        (hashOID != SHA256h)) {
        return -1;
    }

    /* build DigestInfo */
    XMEMCPY(digestInfo, digInfoEncoding, sizeof(digInfoEncoding));
    digestInfoSz += sizeof(digInfoEncoding);
    XMEMCPY(digestInfo + digestInfoSz, digest, digestSz);
    digestInfoSz += digestSz;

    /* set up RSA key */
    ret = wc_InitRsaKey_ex(&rsa, pkcs7->heap, devid);
    if (ret != 0) {
        return ret;
    }

    ret = wc_RsaPrivateKeyDecode(privateKey, &idx, &rsa, privateKeySz);

    /* sign DigestInfo */
    if (ret == 0) {
        ret = wc_RsaSSL_Sign(digestInfo, digestInfoSz, sig, sizeof(sig),
                             &rsa, pkcs7->rng);
        if (ret > 0) {
            if (ret > (int)outSz) {
                /* output buffer too small */
                ret = -1;
            }
            else {
                /* success, ret holds sig size */
                XMEMCPY(out, sig, ret);
            }
        }
    }

    wc_FreeRsaKey(&rsa);

    return ret;
}
#endif


/*
 * Testing wc_PKCS7_EncodeSignedData()
 */
static int test_wc_PKCS7_EncodeSignedData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7;
    WC_RNG      rng;
    byte        output[FOURK_BUF];
    byte        badOut[1];
    word32      outputSz = (word32)sizeof(output);
    word32      badOutSz = 0;
    byte        data[] = "Test data to encode.";

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz;
        int             keySz;

        fp = XFOPEN("./certs/1024/client-cert.der", "rb");
        AssertTrue(fp != XBADFILE);
        certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/1024/client-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp);
        XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, certSz);
        XMEMCPY(key, ecc_clikey_der_256, keySz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz, keySz;

        fp = XOPEN("./certs/client-ecc-cert.der", "rb");
        AssertTrue(fp != XBADFILE);
        certSz = (int)XFREAD(cert, 1, ONEK_BUF, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/client-ecc-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, ONEK_BUF, fp);
        XFCLOSE(fp);
    #endif
#endif

    XMEMSET(output, 0, outputSz);
    AssertIntEQ(wc_InitRng(&rng), 0);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    pkcs7->content = data;
    pkcs7->contentSz = (word32)sizeof(data);
    pkcs7->privateKey = key;
    pkcs7->privateKeySz = (word32)sizeof(key);
    pkcs7->encryptOID = RSAk;
#ifdef NO_SHA
    pkcs7->hashOID = SHA256h;
#else
    pkcs7->hashOID = SHAh;
#endif
    pkcs7->rng = &rng;

    AssertIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);

    wc_PKCS7_Free(pkcs7);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    /* Pass in bad args. */
    AssertIntEQ(wc_PKCS7_EncodeSignedData(NULL, output, outputSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, NULL, outputSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, badOut,
                                badOutSz), BAD_FUNC_ARG);
    pkcs7->hashOID = 0; /* bad hashOID */
    AssertIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), BAD_FUNC_ARG);

#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
    /* test RSA sign raw digest callback, if using RSA and compiled in.
     * Example callback assumes SHA-256, so only run test if compiled in. */
    wc_PKCS7_Free(pkcs7);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    pkcs7->content = data;
    pkcs7->contentSz = (word32)sizeof(data);
    pkcs7->privateKey = key;
    pkcs7->privateKeySz = (word32)sizeof(key);
    pkcs7->encryptOID = RSAk;
    pkcs7->hashOID = SHA256h;
    pkcs7->rng = &rng;

    AssertIntEQ(wc_PKCS7_SetRsaSignRawDigestCb(pkcs7, rsaSignRawDigestCb), 0);

    AssertIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);
#endif

    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_EncodeSignedData */


/*
 * Testing wc_PKCS7_EncodeSignedData_ex() and wc_PKCS7_VerifySignedData_ex()
 */
static int test_wc_PKCS7_EncodeSignedData_ex(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    int         ret, i;
    PKCS7*      pkcs7;
    WC_RNG      rng;
    byte        outputHead[FOURK_BUF/2];
    byte        outputFoot[FOURK_BUF/2];
    word32      outputHeadSz = (word32)sizeof(outputHead);
    word32      outputFootSz = (word32)sizeof(outputFoot);
    byte        data[FOURK_BUF];
    wc_HashAlg  hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = wc_HashGetDigestSize(hashType);

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz;
        int             keySz;

        fp = XFOPEN("./certs/1024/client-cert.der", "rb");
        AssertTrue((fp != XBADFILE));
        certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/1024/client-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp);
        XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp;
        int             certSz, keySz;

        fp = XFOPEN("./certs/client-ecc-cert.der", "rb");
        AssertTrue(fp != XBADFILE);
        certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256, fp);
        XFCLOSE(fp);

        fp = XFOPEN("./certs/client-ecc-key.der", "rb");
        AssertTrue(fp != XBADFILE);
        keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp);
        XFCLOSE(fp);
    #endif
#endif

    /* initialize large data with sequence */
    for (i=0; i<(int)sizeof(data); i++)
        data[i] = i & 0xff;

    XMEMSET(outputHead, 0, outputHeadSz);
    XMEMSET(outputFoot, 0, outputFootSz);
    AssertIntEQ(wc_InitRng(&rng), 0);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    pkcs7->content = NULL; /* not used for ex */
    pkcs7->contentSz = (word32)sizeof(data);
    pkcs7->privateKey = key;
    pkcs7->privateKeySz = (word32)sizeof(key);
    pkcs7->encryptOID = RSAk;
#ifdef NO_SHA
    pkcs7->hashOID = SHA256h;
#else
    pkcs7->hashOID = SHAh;
#endif
    pkcs7->rng = &rng;

    /* calculate hash for content */
    ret = wc_HashInit(&hash, hashType);
    if (ret == 0) {
        ret = wc_HashUpdate(&hash, hashType, data, sizeof(data));
        if (ret == 0) {
            ret = wc_HashFinal(&hash, hashType, hashBuf);
        }
        wc_HashFree(&hash, hashType);
    }
    AssertIntEQ(ret, 0);

    /* Perform PKCS7 sign using hash directly */
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), 0);
    AssertIntGT(outputHeadSz, 0);
    AssertIntGT(outputFootSz, 0);

    wc_PKCS7_Free(pkcs7);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* required parameter even on verify when using _ex, if using outputHead
     * and outputFoot */
    pkcs7->contentSz = (word32)sizeof(data);
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, outputFootSz), 0);

    wc_PKCS7_Free(pkcs7);

    /* assembly complete PKCS7 sign and use normal verify */
    {
        byte* output = (byte*)XMALLOC(
                outputHeadSz + sizeof(data) + outputFootSz,
                HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        word32 outputSz = 0;
        AssertNotNull(output);
        XMEMCPY(&output[outputSz], outputHead, outputHeadSz);
        outputSz += outputHeadSz;
        XMEMCPY(&output[outputSz], data, sizeof(data));
        outputSz += sizeof(data);
        XMEMCPY(&output[outputSz], outputFoot, outputFootSz);
        outputSz += outputFootSz;

        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
        AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
        XFREE(output, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }

    /* Pass in bad args. */
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(NULL, hashBuf, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, NULL, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, 0, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, NULL, outputFoot, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, NULL, &outputFootSz), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, NULL), BAD_FUNC_ARG);
    pkcs7->hashOID = 0; /* bad hashOID */
    AssertIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), BAD_FUNC_ARG);

    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(NULL, hashBuf, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);

    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, NULL, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_PKCS7_WANT_READ_E);
#else
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), BUFFER_E);
#endif
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        outputHeadSz, outputFoot, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    /* can pass in 0 buffer length with streaming API */
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), WC_PKCS7_WANT_READ_E);
#else
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), BAD_FUNC_ARG);
#endif
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, NULL, outputFootSz), BAD_FUNC_ARG);
#ifndef NO_PKCS7_STREAM
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), WC_PKCS7_WANT_READ_E);
#else
    AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), ASN_PARSE_E);
#endif

    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_EncodeSignedData_ex */


#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)

/**
 * Loads certs/keys from files or buffers into the argument buffers,
 * helper function called by CreatePKCS7SignedData().
 *
 * Returns 0 on success, negative on error.
 */
static int LoadPKCS7SignedDataCerts(
        int useIntermediateCertChain, int pkAlgoType,
        byte* intCARoot, word32* intCARootSz,
        byte* intCA1, word32* intCA1Sz,
        byte* intCA2, word32* intCA2Sz,
        byte* cert, word32* certSz,
        byte* key, word32* keySz)
{
    int ret = 0;
    FILE*  fp = NULL;

#ifndef NO_RSA
    const char* intCARootRSA   = "./certs/ca-cert.der";
    const char* intCA1RSA      = "./certs/intermediate/ca-int-cert.der";
    const char* intCA2RSA      = "./certs/intermediate/ca-int2-cert.der";
    const char* intServCertRSA = "./certs/intermediate/server-int-cert.der";
    const char* intServKeyRSA  = "./certs/server-key.der";

    #if !defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)
        const char* cli1024Cert    = "./certs/1024/client-cert.der";
        const char* cli1024Key     = "./certs/1024/client-key.der";
    #endif
#endif
#ifdef HAVE_ECC
    const char* intCARootECC   = "./certs/ca-ecc-cert.der";
    const char* intCA1ECC      = "./certs/intermediate/ca-int-ecc-cert.der";
    const char* intCA2ECC      = "./certs/intermediate/ca-int2-ecc-cert.der";
    const char* intServCertECC = "./certs/intermediate/server-int-ecc-cert.der";
    const char* intServKeyECC  = "./certs/ecc-key.der";

    #ifndef USE_CERT_BUFFERS_256
        const char* cliEccCert     = "./certs/client-ecc-cert.der";
        const char* cliEccKey      = "./certs/client-ecc-key.der";
    #endif
#endif

    if (cert == NULL || certSz == NULL || key == NULL || keySz == NULL ||
        ((useIntermediateCertChain == 1) &&
        (intCARoot == NULL || intCARootSz == NULL || intCA1 == NULL ||
         intCA1Sz == NULL || intCA2 == NULL || intCA2Sz == NULL))) {
        return BAD_FUNC_ARG;
    }

    /* Read/load certs and keys to use for signing based on PK type and chain */
    switch (pkAlgoType) {
#ifndef NO_RSA
        case RSA_TYPE:
            if (useIntermediateCertChain == 1) {
                fp = XFOPEN(intCARootRSA, "rb");
                AssertNotNull(fp);
                *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCARootSz, 0);

                fp = XFOPEN(intCA1RSA, "rb");
                AssertNotNull(fp);
                *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCA1Sz, 0);

                fp = XFOPEN(intCA2RSA, "rb");
                AssertNotNull(fp);
                *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCA2Sz, 0);

                fp = XFOPEN(intServCertRSA, "rb");
                AssertNotNull(fp);
                *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*certSz, 0);

                fp = XFOPEN(intServKeyRSA, "rb");
                AssertNotNull(fp);
                *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                XFCLOSE(fp);
                AssertIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_2048)
                *keySz  = sizeof_client_key_der_2048;
                *certSz = sizeof_client_cert_der_2048;
                XMEMCPY(key, client_key_der_2048, *keySz);
                XMEMCPY(cert, client_cert_der_2048, *certSz);
            #elif defined(USE_CERT_BUFFERS_1024)
                *keySz  = sizeof_client_key_der_1024;
                *certSz = sizeof_client_cert_der_1024;
                XMEMCPY(key, client_key_der_1024, *keySz);
                XMEMCPY(cert, client_cert_der_1024, *certSz);
            #else
                fp = XFOPEN(cli1024Key, "rb");
                AssertNotNull(fp);
                *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                XFCLOSE(fp);
                AssertIntGT(*keySz, 0);

                fp = XFOPEN(cli1024Cert, "rb");
                AssertNotNull(fp);
                *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_2048 */
            }
            break;
#endif /* !NO_RSA */
#ifdef HAVE_ECC
        case ECC_TYPE:
            if (useIntermediateCertChain == 1) {
                fp = XFOPEN(intCARootECC, "rb");
                AssertNotNull(fp);
                *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCARootSz, 0);

                fp = XFOPEN(intCA1ECC, "rb");
                AssertNotNull(fp);
                *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCA1Sz, 0);

                fp = XFOPEN(intCA2ECC, "rb");
                AssertNotNull(fp);
                *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                XFCLOSE(fp);
                AssertIntGT(*intCA2Sz, 0);

                fp = XFOPEN(intServCertECC, "rb");
                AssertNotNull(fp);
                *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*certSz, 0);

                fp = XFOPEN(intServKeyECC, "rb");
                AssertNotNull(fp);
                *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                XFCLOSE(fp);
                AssertIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_256)
                *keySz  = sizeof_ecc_clikey_der_256;
                *certSz = sizeof_cliecc_cert_der_256;
                XMEMCPY(key, ecc_clikey_der_256, *keySz);
                XMEMCPY(cert, cliecc_cert_der_256, *certSz);
            #else
                fp = XFOPEN(cliEccKey, "rb");
                AssertNotNull(fp);
                *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                XFCLOSE(fp);
                AssertIntGT(*keySz, 0);

                fp = XFOPEN(cliEccCert, "rb");
                AssertNotNull(fp);
                *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                XFCLOSE(fp);
                AssertIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_256 */
            }
            break;
#endif /* HAVE_ECC */
        default:
            WOLFSSL_MSG("Unsupported SignedData PK type");
            ret = BAD_FUNC_ARG;
            break;
    }

    return ret;
}

/**
 * Creates a PKCS7/CMS SignedData bundle to use for testing.
 *
 * output          output buffer to place SignedData
 * outputSz        size of output buffer
 * data            data buffer to be signed
 * dataSz          size of data buffer
 * withAttribs     [1/0] include attributes in SignedData message
 * detachedSig     [1/0] create detached signature, no content
 * useIntCertChain [1/0] use certificate chain and include intermediate and
 *                 root CAs in bundle
 * pkAlgoType      RSA_TYPE or ECC_TYPE, choose what key/cert type to use
 *
 * Return size of bundle created on success, negative on error */
static int CreatePKCS7SignedData(unsigned char* output, int outputSz,
                                 byte* data, word32 dataSz,
                                 int withAttribs, int detachedSig,
                                 int useIntermediateCertChain,
                                 int pkAlgoType)
{
    int ret = 0;
    WC_RNG rng;
    PKCS7* pkcs7 = NULL;

    static byte messageTypeOid[] =
               { 0x06, 0x0a, 0x60, 0x86, 0x48, 0x01, 0x86, 0xF8, 0x45, 0x01,
                 0x09, 0x02 };
    static byte messageType[] = { 0x13, 2, '1', '9' };

    PKCS7Attrib attribs[] =
    {
        { messageTypeOid, sizeof(messageTypeOid), messageType,
                                       sizeof(messageType) }
    };

    byte intCARoot[TWOK_BUF];
    byte intCA1[TWOK_BUF];
    byte intCA2[TWOK_BUF];
    byte cert[TWOK_BUF];
    byte key[TWOK_BUF];

    word32 intCARootSz = sizeof(intCARoot);
    word32 intCA1Sz    = sizeof(intCA1);
    word32 intCA2Sz    = sizeof(intCA2);
    word32 certSz      = sizeof(cert);
    word32 keySz       = sizeof(key);

    XMEMSET(intCARoot, 0, intCARootSz);
    XMEMSET(intCA1, 0, intCA1Sz);
    XMEMSET(intCA2, 0, intCA2Sz);
    XMEMSET(cert, 0, certSz);
    XMEMSET(key, 0, keySz);

    ret = LoadPKCS7SignedDataCerts(useIntermediateCertChain, pkAlgoType,
                intCARoot, &intCARootSz, intCA1, &intCA1Sz, intCA2, &intCA2Sz,
                cert, &certSz, key, &keySz);
    AssertIntEQ(ret, 0);

    XMEMSET(output, 0, outputSz);
    AssertIntEQ(wc_InitRng(&rng), 0);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (useIntermediateCertChain == 1) {
        /* Add intermediate and root CA certs into SignedData Certs SET */
        AssertIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA2, intCA2Sz), 0);
        AssertIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA1, intCA1Sz), 0);
        AssertIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCARoot, intCARootSz), 0);
    }

    pkcs7->content = data;
    pkcs7->contentSz = dataSz;
    pkcs7->privateKey = key;
    pkcs7->privateKeySz = (word32)sizeof(key);
    if (pkAlgoType == RSA_TYPE) {
        pkcs7->encryptOID = RSAk;
    }
    else {
        pkcs7->encryptOID = ECDSAk;
    }
#ifdef NO_SHA
    pkcs7->hashOID = SHA256h;
#else
    pkcs7->hashOID = SHAh;
#endif
    pkcs7->rng = &rng;
    if (withAttribs) {
        /* include a signed attribute */
        pkcs7->signedAttribs   = attribs;
        pkcs7->signedAttribsSz = (sizeof(attribs)/sizeof(PKCS7Attrib));
    }

    if (detachedSig) {
        AssertIntEQ(wc_PKCS7_SetDetached(pkcs7, 1), 0);
    }

    outputSz = wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz);
    AssertIntGT(outputSz, 0);
    wc_PKCS7_Free(pkcs7);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (detachedSig) {
        pkcs7->content = data;
        pkcs7->contentSz = dataSz;
    }
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    return outputSz;
}
#endif

/*
 * Testing wc_PKCS_VerifySignedData()
 */
static int test_wc_PKCS7_VerifySignedData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)
    PKCS7* pkcs7;
    byte   output[6000]; /* Large size needed for bundles with int CA certs */
    word32 outputSz = sizeof(output);
    byte   data[] = "Test data to encode.";
    byte   badOut[1];
    word32 badOutSz = 0;
    byte   badContent[] = "This is different content than was signed";

    int ret;
    wc_HashAlg  hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = wc_HashGetDigestSize(hashType);

#ifndef NO_RSA
    PKCS7DecodedAttrib* decodedAttrib = NULL;

    /* contentType OID (1.2.840.113549.1.9.3) */
    static const byte contentTypeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xF7, 0x0d, 0x01, 0x09, 0x03 };

    /* PKCS#7 DATA content type (contentType defaults to DATA) */
    static const byte dataType[] =
        { 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01 };

    /* messageDigest OID (1.2.840.113549.1.9.4) */
    static const byte messageDigestOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04 };

    /* signingTime OID () */
    static const byte signingTimeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x05};

#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    int dateLength = 0;
    byte dateFormat;
    const byte* datePart = NULL;
    struct tm timearg;
    time_t now;
    struct tm* nowTm = NULL;
#ifdef NEED_TMP_TIME
    struct tm tmpTimeStorage;
    struct tm* tmpTime = &tmpTimeStorage;
#endif
#endif /* !NO_ASN && !NO_ASN_TIME */

    /* Success test with RSA certs/key */
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 0, RSA_TYPE)), 0);

    /* calculate hash for content, used later */
    ret = wc_HashInit(&hash, hashType);
    if (ret == 0) {
        ret = wc_HashUpdate(&hash, hashType, data, sizeof(data));
        if (ret == 0) {
            ret = wc_HashFinal(&hash, hashType, hashBuf);
        }
        wc_HashFree(&hash, hashType);
    }
    AssertIntEQ(ret, 0);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    /* Check that decoded signed attributes are correct */

    /* messageDigest should be first */
    decodedAttrib = pkcs7->decodedAttrib;
    AssertNotNull(decodedAttrib);
    AssertIntEQ(decodedAttrib->oidSz, (word32)sizeof(messageDigestOid));
    AssertIntEQ(XMEMCMP(decodedAttrib->oid, messageDigestOid,
                        decodedAttrib->oidSz), 0);
    /* + 2 for OCTET STRING and length bytes */
    AssertIntEQ(decodedAttrib->valueSz, hashSz + 2);
    AssertNotNull(decodedAttrib->value);
    AssertIntEQ(XMEMCMP(decodedAttrib->value + 2, hashBuf, hashSz), 0);

    /* signingTime should be second */
    decodedAttrib = decodedAttrib->next;
    AssertNotNull(decodedAttrib);
    AssertIntEQ(decodedAttrib->oidSz, (word32)sizeof(signingTimeOid));
    AssertIntEQ(XMEMCMP(decodedAttrib->oid, signingTimeOid,
                        decodedAttrib->oidSz), 0);

    AssertIntGT(decodedAttrib->valueSz, 0);
    AssertNotNull(decodedAttrib->value);

    /* Verify signingTime if ASN and time are available */
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    AssertIntEQ(wc_GetDateInfo(decodedAttrib->value, decodedAttrib->valueSz,
                               &datePart, &dateFormat, &dateLength), 0);
    AssertNotNull(datePart);
    AssertIntGT(dateLength, 0);
    XMEMSET(&timearg, 0, sizeof(timearg));
    AssertIntEQ(wc_GetDateAsCalendarTime(datePart, dateLength, dateFormat,
                                         &timearg), 0);

    /* Get current time and compare year/month/day against attribute value */
    AssertIntEQ(wc_GetTime(&now, sizeof(now)), 0);
    nowTm = (struct tm*)XGMTIME((time_t*)&now, tmpTime);
    AssertNotNull(nowTm);

    AssertIntEQ(timearg.tm_year, nowTm->tm_year);
    AssertIntEQ(timearg.tm_mon, nowTm->tm_mon);
    AssertIntEQ(timearg.tm_mday, nowTm->tm_mday);
#endif /* !NO_ASN && !NO_ASN_TIME */

    /* contentType should be third */
    decodedAttrib = decodedAttrib->next;
    AssertNotNull(decodedAttrib);
    AssertIntEQ(decodedAttrib->oidSz, (word32)sizeof(contentTypeOid));
    AssertIntEQ(XMEMCMP(decodedAttrib->oid, contentTypeOid,
                        decodedAttrib->oidSz), 0);
    AssertIntEQ(decodedAttrib->valueSz, (int)sizeof(dataType) + 2);
    AssertNotNull(decodedAttrib->value);
    AssertIntEQ(XMEMCMP(decodedAttrib->value + 2, dataType,
                        sizeof(dataType)), 0);
#endif /* !NO_RSA */

#ifdef HAVE_ECC
    #ifndef NO_RSA
    wc_PKCS7_Free(pkcs7);
    #endif

    /* Success test with ECC certs/key */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 0, ECC_TYPE)), 0);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
#endif /* HAVE_ECC */

    /* Test bad args. */
#if !defined(NO_RSA) || defined(HAVE_ECC)
    AssertIntEQ(wc_PKCS7_VerifySignedData(NULL, output, outputSz),
                                          BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, NULL, outputSz),
                                          BAD_FUNC_ARG);
    #ifndef NO_PKCS7_STREAM
        /* can pass in 0 buffer length with streaming API */
        AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), WC_PKCS7_WANT_READ_E);
    #else
        AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), BAD_FUNC_ARG);
    #endif
    wc_PKCS7_Free(pkcs7);
#endif /* !NO_RSA || HAVE_ECC */

    /* Invalid content should error, use detached signature so we can
     * easily change content */
#ifndef NO_RSA
    /* Try RSA certs/key/sig first */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  1, 1, 0, RSA_TYPE)), 0);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    pkcs7->content = badContent;
    pkcs7->contentSz = sizeof(badContent);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
                SIG_VERIFY_E);
    wc_PKCS7_Free(pkcs7);

    /* Test success case with detached signature and valid content */
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    pkcs7->content = data;
    pkcs7->contentSz = sizeof(data);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);

    /* verify using pre-computed content digest only (no content) */
    {
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
                                                 output, outputSz,
                                                 NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
    }
#endif /* !NO_RSA */
#ifdef HAVE_ECC
    /* Try ECC certs/key/sig next */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  1, 1, 0, ECC_TYPE)), 0);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    pkcs7->content = badContent;
    pkcs7->contentSz = sizeof(badContent);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
                SIG_VERIFY_E);
    wc_PKCS7_Free(pkcs7);

    /* Test success case with detached signature and valid content */
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    pkcs7->content = data;
    pkcs7->contentSz = sizeof(data);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);

    /* verify using pre-computed content digest only (no content) */
    {
        /* calculate hash for content */
        ret = wc_HashInit(&hash, hashType);
        if (ret == 0) {
            ret = wc_HashUpdate(&hash, hashType, data, sizeof(data));
            if (ret == 0) {
                ret = wc_HashFinal(&hash, hashType, hashBuf);
            }
            wc_HashFree(&hash, hashType);
        }
        AssertIntEQ(ret, 0);

        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        AssertIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
                                                 output, outputSz,
                                                 NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
    }
#endif

    /* Test verify on signedData containing intermediate/root CA certs */
#ifndef NO_RSA
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 1, RSA_TYPE)), 0);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
#endif /* !NO_RSA */
#ifdef HAVE_ECC
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    AssertIntGT((outputSz = CreatePKCS7SignedData(output, outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 1, ECC_TYPE)), 0);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
#endif /* HAVE_ECC */

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_VerifySignedData() */


#if defined(HAVE_PKCS7) && !defined(NO_AES) && defined(HAVE_AES_CBC) && \
    !defined(NO_AES_256)
static const byte defKey[] = {
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
};
static byte aesHandle[32]; /* simulated hardware key handle */

/* return 0 on success */
static int myDecryptionFunc(PKCS7* pkcs7, int encryptOID, byte* iv, int ivSz,
        byte* aad, word32 aadSz, byte* authTag, word32 authTagSz,
        byte* in, int inSz, byte* out, void* usrCtx)
{
    int ret;
    Aes aes;

    if (usrCtx == NULL) {
        /* no simulated handle passed in */
        return -1;
    }

    switch (encryptOID) {
        case AES256CBCb:
            if (ivSz  != AES_BLOCK_SIZE)
                return BAD_FUNC_ARG;
            break;

        default:
            WOLFSSL_MSG("Unsupported content cipher type for test");
            return ALGO_ID_E;
    };

    /* simulate using handle to get key */
    ret = wc_AesInit(&aes, HEAP_HINT, INVALID_DEVID);
    if (ret == 0) {
        ret = wc_AesSetKey(&aes, (byte*)usrCtx, 32, iv, AES_DECRYPTION);
        if (ret == 0)
            ret = wc_AesCbcDecrypt(&aes, out, in, inSz);
        wc_AesFree(&aes);
    }

    (void)aad;
    (void)aadSz;
    (void)authTag;
    (void)authTagSz;
    (void)pkcs7;
    return ret;
}


/* returns key size on success */
static int myCEKwrapFunc(PKCS7* pkcs7, byte* cek, word32 cekSz, byte* keyId,
        word32 keyIdSz, byte* orginKey, word32 orginKeySz,
        byte* out, word32 outSz, int keyWrapAlgo, int type, int direction)
{
    int ret = -1;

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (keyId[0] != 0x00) {
        return -1;
    }

    if (type != (int)PKCS7_KEKRI) {
        return -1;
    }

    switch (keyWrapAlgo) {
        case AES256_WRAP:
            /* simulate setting a handle for later decryption but use key
             * as handle in the test case here */
            ret = wc_AesKeyUnWrap(defKey, sizeof(defKey), cek, cekSz,
                                      aesHandle, sizeof(aesHandle), NULL);
            if (ret < 0)
                return ret;

            ret = wc_PKCS7_SetDecodeEncryptedCtx(pkcs7, (void*)aesHandle);
            if (ret < 0)
                return ret;

            /* return key size on success */
            return sizeof(defKey);

        default:
            WOLFSSL_MSG("Unsupported key wrap algorithm in example");
            return BAD_KEYWRAP_ALG_E;
    };

    (void)cekSz;
    (void)cek;
    (void)outSz;
    (void)keyIdSz;
    (void)direction;
    (void)orginKey; /* used with KAKRI */
    (void)orginKeySz;
    return ret;
}
#endif /* HAVE_PKCS7 && !NO_AES && HAVE_AES_CBC && !NO_AES_256 */


/*
 * Testing wc_PKCS7_EncodeEnvelopedData()
 */
static int test_wc_PKCS7_EncodeDecodeEnvelopedData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7;
#ifdef ECC_TIMING_RESISTANT
    WC_RNG      rng;
#endif
    word32      tempWrd32   = 0;
    byte*       tmpBytePtr = NULL;
    const char  input[] = "Test data to encode.";
    int         i;
    int         testSz = 0;
    #if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) || \
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))

        byte*   rsaCert     = NULL;
        byte*   rsaPrivKey  = NULL;
        word32  rsaCertSz;
        word32  rsaPrivKeySz;
        #if !defined(NO_FILESYSTEM) && (!defined(USE_CERT_BUFFERS_1024) && \
                                           !defined(USE_CERT_BUFFERS_2048) )
            static const char* rsaClientCert = "./certs/client-cert.der";
            static const char* rsaClientKey = "./certs/client-key.der";
            rsaCertSz = (word32)sizeof(rsaClientCert);
            rsaPrivKeySz = (word32)sizeof(rsaClientKey);
        #endif
    #endif
    #if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))

        byte*   eccCert     = NULL;
        byte*   eccPrivKey  = NULL;
        word32  eccCertSz;
        word32  eccPrivKeySz;
        #if !defined(NO_FILESYSTEM) && !defined(USE_CERT_BUFFERS_256)
            static const char* eccClientCert = "./certs/client-ecc-cert.der";
            static const char* eccClientKey = "./certs/ecc-client-key.der";
        #endif
    #endif
    /* Generic buffer size. */
    byte    output[ONEK_BUF];
    byte    decoded[sizeof(input)/sizeof(char)];
    int     decodedSz = 0;
#ifndef NO_FILESYSTEM
    XFILE certFile;
    XFILE keyFile;
#endif

#if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
    /* RSA certs and keys. */
    #if defined(USE_CERT_BUFFERS_1024)
        rsaCertSz = (word32)sizeof_client_cert_der_1024;
        /* Allocate buffer space. */
        AssertNotNull(rsaCert =
                (byte*)XMALLOC(rsaCertSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        XMEMCPY(rsaCert, client_cert_der_1024, rsaCertSz);
        rsaPrivKeySz = (word32)sizeof_client_key_der_1024;
        AssertNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER));
        XMEMCPY(rsaPrivKey, client_key_der_1024, rsaPrivKeySz);

    #elif defined(USE_CERT_BUFFERS_2048)
        rsaCertSz = (word32)sizeof_client_cert_der_2048;
        /* Allocate buffer */
        AssertNotNull(rsaCert =
                (byte*)XMALLOC(rsaCertSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        XMEMCPY(rsaCert, client_cert_der_2048, rsaCertSz);
        rsaPrivKeySz = (word32)sizeof_client_key_der_2048;
        AssertNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER));
        XMEMCPY(rsaPrivKey, client_key_der_2048, rsaPrivKeySz);

    #else
        /* File system. */
        certFile = XFOPEN(rsaClientCert, "rb");
        AssertTrue(certFile != XBADFILE);
        rsaCertSz = (word32)FOURK_BUF;
        AssertNotNull(rsaCert =
                (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        rsaCertSz = (word32)XFREAD(rsaCert, 1, rsaCertSz, certFile);
        XFCLOSE(certFile);
        keyFile = XFOPEN(rsaClientKey, "rb");
        AssertTrue(keyFile != XBADFILE);
        AssertNotNull(rsaPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER));
        rsaPrivKeySz = (word32)FOURK_BUF;
        rsaPrivKeySz = (word32)XFREAD(rsaPrivKey, 1, rsaPrivKeySz, keyFile);
        XFCLOSE(keyFile);
    #endif /* USE_CERT_BUFFERS */
#endif /* NO_RSA */

/* ECC */
#if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))

    #ifdef USE_CERT_BUFFERS_256
        AssertNotNull(eccCert =
                (byte*)XMALLOC(TWOK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        eccCertSz = (word32)sizeof_cliecc_cert_der_256;
        XMEMCPY(eccCert, cliecc_cert_der_256, eccCertSz);
        AssertNotNull(eccPrivKey = (byte*)XMALLOC(TWOK_BUF, HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER));
        eccPrivKeySz = (word32)sizeof_ecc_clikey_der_256;
        XMEMCPY(eccPrivKey, ecc_clikey_der_256, eccPrivKeySz);
    #else /* File system. */
        certFile = XFOPEN(eccClientCert, "rb");
        AssertTrue(certFile != XBADFILE);
        eccCertSz = (word32)FOURK_BUF;
        AssertNotNull(eccCert =
                (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        eccCertSz = (word32)XFREAD(eccCert, 1, eccCertSz, certFile);
        XFCLOSE(certFile);
        keyFile = XFOPEN(eccClientKey, "rb");
        AssertTrue(keyFile != XBADFILE);
        eccPrivKeySz = (word32)FOURK_BUF;
        AssertNotNull(eccPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER));
        eccPrivKeySz = (word32)XFREAD(eccPrivKey, 1, eccPrivKeySz, keyFile);
        XFCLOSE(keyFile);
    #endif /* USE_CERT_BUFFERS_256 */
#endif /* END HAVE_ECC */

#ifndef NO_FILESYSTEM
    /* Silence. */
    (void)keyFile;
    (void)certFile;
#endif

    {
    const pkcs7EnvelopedVector testVectors[] = {
    /* DATA is a global variable defined in the makefile. */
#if !defined(NO_RSA)
    #ifndef NO_DES3
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, DES3b, 0, 0,
            rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
    #endif /* NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES128CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_192
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES192CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_256
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES256CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC */

#endif /* NO_RSA */
#if defined(HAVE_ECC)
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #if !defined(NO_SHA) && !defined(NO_AES_128)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES128CBCb,
                AES128_WRAP, dhSinglePass_stdDH_sha1kdf_scheme, eccCert,
                eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if !defined(NO_SHA256) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES256CBCb,
                AES256_WRAP, dhSinglePass_stdDH_sha256kdf_scheme, eccCert,
                eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if defined(WOLFSSL_SHA512) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES256CBCb,
                AES256_WRAP, dhSinglePass_stdDH_sha512kdf_scheme, eccCert,
                eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC*/
#endif /* END HAVE_ECC */
    }; /* END pkcs7EnvelopedVector */

#ifdef ECC_TIMING_RESISTANT
    AssertIntEQ(wc_InitRng(&rng), 0);
#endif

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);

    testSz = (int)sizeof(testVectors)/(int)sizeof(pkcs7EnvelopedVector);
    for (i = 0; i < testSz; i++) {
        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, (testVectors + i)->cert,
                                    (word32)(testVectors + i)->certSz), 0);
#ifdef ECC_TIMING_RESISTANT
        pkcs7->rng = &rng;
#endif

        pkcs7->content       = (byte*)(testVectors + i)->content;
        pkcs7->contentSz     = (testVectors + i)->contentSz;
        pkcs7->contentOID    = (testVectors + i)->contentOID;
        pkcs7->encryptOID    = (testVectors + i)->encryptOID;
        pkcs7->keyWrapOID    = (testVectors + i)->keyWrapOID;
        pkcs7->keyAgreeOID   = (testVectors + i)->keyAgreeOID;
        pkcs7->privateKey    = (testVectors + i)->privateKey;
        pkcs7->privateKeySz  = (testVectors + i)->privateKeySz;

        AssertIntGE(wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
                            (word32)sizeof(output)), 0);

        decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
                (word32)sizeof(output), decoded, (word32)sizeof(decoded));
        AssertIntGE(decodedSz, 0);
        /* Verify the size of each buffer. */
        AssertIntEQ((word32)sizeof(input)/sizeof(char), decodedSz);
        /* Don't free the last time through the loop. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
            AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        }
    }  /* END test loop. */
    }

    /* Test bad args. */
    AssertIntEQ(wc_PKCS7_EncodeEnvelopedData(NULL, output,
                    (word32)sizeof(output)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, NULL,
                    (word32)sizeof(output)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, output, 0), BAD_FUNC_ARG);

    /* Decode.  */
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(NULL, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), NULL, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, 0), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, NULL,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output, 0, decoded,
        (word32)sizeof(decoded)), BAD_FUNC_ARG);
    /* Should get a return of BAD_FUNC_ARG with structure data. Order matters.*/
#if defined(HAVE_ECC) && !defined(NO_AES) && defined(HAVE_AES_CBC)
    /* only a failure for KARI test cases */
    tempWrd32 = pkcs7->singleCertSz;
    pkcs7->singleCertSz = 0;
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    pkcs7->singleCertSz = tempWrd32;

    tmpBytePtr = pkcs7->singleCert;
    pkcs7->singleCert = NULL;
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    pkcs7->singleCert = tmpBytePtr;
#endif
    tempWrd32 = pkcs7->privateKeySz;
    pkcs7->privateKeySz = 0;

    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    pkcs7->privateKeySz = tempWrd32;

    tmpBytePtr = pkcs7->privateKey;
    pkcs7->privateKey = NULL;
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)), BAD_FUNC_ARG);
    pkcs7->privateKey = tmpBytePtr;

    wc_PKCS7_Free(pkcs7);

#if !defined(NO_AES) && defined(HAVE_AES_CBC) && !defined(NO_AES_256)
    /* test of decrypt callback with KEKRI enveloped data */
    {
        int envelopedSz;
        const byte keyId[] = { 0x00 };

        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        pkcs7->content      = (byte*)input;
        pkcs7->contentSz    = (word32)(sizeof(input)/sizeof(char));
        pkcs7->contentOID   = DATA;
        pkcs7->encryptOID   = AES256CBCb;
        AssertIntGT(wc_PKCS7_AddRecipient_KEKRI(pkcs7, AES256_WRAP,
                    (byte*)defKey, sizeof(defKey), (byte*)keyId,
                    sizeof(keyId), NULL, NULL, 0, NULL, 0, 0), 0);
        AssertIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, CMS_SKID), 0);
        AssertIntGT((envelopedSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
                        (word32)sizeof(output))), 0);
        wc_PKCS7_Free(pkcs7);

        /* decode envelopedData */
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_SetWrapCEKCb(pkcs7, myCEKwrapFunc), 0);
        AssertIntEQ(wc_PKCS7_SetDecodeEncryptedCb(pkcs7, myDecryptionFunc), 0);
        AssertIntGT((decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
                        envelopedSz, decoded, sizeof(decoded))), 0);
        wc_PKCS7_Free(pkcs7);
    }
#endif /* !NO_AES && !NO_AES_256 */

#ifndef NO_RSA
    if (rsaCert) {
        XFREE(rsaCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (rsaPrivKey) {
        XFREE(rsaPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /*NO_RSA */
#ifdef HAVE_ECC
    if (eccCert) {
        XFREE(eccCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (eccPrivKey) {
        XFREE(eccPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_ECC */

#ifdef ECC_TIMING_RESISTANT
    wc_FreeRng(&rng);
#endif

#if defined(USE_CERT_BUFFERS_2048) && !defined(NO_DES3) && \
    !defined(NO_RSA) && !defined(NO_SHA)
    {
        byte   out[7];
        byte   *cms;
        word32 cmsSz;
        XFILE  cmsFile;

        XMEMSET(out, 0, sizeof(out));
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        cmsFile = XFOPEN("./certs/test/ktri-keyid-cms.msg", "rb");
        AssertTrue(cmsFile != XBADFILE);
        cmsSz = (word32)FOURK_BUF;
        AssertNotNull(cms =
                (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
        cmsSz = (word32)XFREAD(cms, 1, cmsSz, cmsFile);
        XFCLOSE(cmsFile);

        AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)client_cert_der_2048,
                    sizeof_client_cert_der_2048), 0);
        pkcs7->privateKey   = (byte*)client_key_der_2048;
        pkcs7->privateKeySz = sizeof_client_key_der_2048;
        AssertIntLT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
                    2), 0);
        AssertIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
                    sizeof(out)), 0);
        XFREE(cms, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        AssertIntEQ(XMEMCMP(out, "test", 4), 0);
        wc_PKCS7_Free(pkcs7);
    }
#endif /* USE_CERT_BUFFERS_2048 && !NO_DES3 && !NO_RSA && !NO_SHA */

    res = TEST_RES_CHECK(1);
#endif /* HAVE_PKCS7 */
    return res;
} /* END test_wc_PKCS7_EncodeEnvelopedData() */


/*
 * Testing wc_PKCS7_EncodeEncryptedData()
 */
static int test_wc_PKCS7_EncodeEncryptedData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_PKCS7_ENCRYPTED_DATA)
    PKCS7*      pkcs7 = NULL;
    byte*       tmpBytePtr = NULL;
    byte        encrypted[TWOK_BUF];
    byte        decoded[TWOK_BUF];
    word32      tmpWrd32 = 0;
    int         tmpInt = 0;
    int         decodedSz;
    int         encryptedSz;
    int         testSz;
    int         i;

    const byte data[] = { /* Hello World */
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x57,0x6f,
        0x72,0x6c,0x64
    };

    #ifndef NO_DES3
        byte desKey[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
        };
        byte des3Key[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
            0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
            0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
        };
    #endif

    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        byte aes128Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_192
        byte aes192Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_256
        byte aes256Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
    #endif /* !NO_AES && HAVE_AES_CBC */
    const pkcs7EncryptedVector testVectors[] =
    {
    #ifndef NO_DES3
        {data, (word32)sizeof(data), DATA, DES3b, des3Key, sizeof(des3Key)},

        {data, (word32)sizeof(data), DATA, DESb, desKey, sizeof(desKey)},
    #endif /* !NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {data, (word32)sizeof(data), DATA, AES128CBCb, aes128Key,
         sizeof(aes128Key)},
        #endif

        #ifndef NO_AES_192
        {data, (word32)sizeof(data), DATA, AES192CBCb, aes192Key,
         sizeof(aes192Key)},
        #endif

        #ifndef NO_AES_256
        {data, (word32)sizeof(data), DATA, AES256CBCb, aes256Key,
         sizeof(aes256Key)},
        #endif

    #endif /* !NO_AES && HAVE_AES_CBC */
    };

    testSz = sizeof(testVectors) / sizeof(pkcs7EncryptedVector);

    for (i = 0; i < testSz; i++) {
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
        pkcs7->content              = (byte*)testVectors[i].content;
        pkcs7->contentSz            = testVectors[i].contentSz;
        pkcs7->contentOID           = testVectors[i].contentOID;
        pkcs7->encryptOID           = testVectors[i].encryptOID;
        pkcs7->encryptionKey        = testVectors[i].encryptionKey;
        pkcs7->encryptionKeySz      = testVectors[i].encryptionKeySz;
        pkcs7->heap                 = HEAP_HINT;

        /* encode encryptedData */
        encryptedSz = wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                                                   sizeof(encrypted));
        AssertIntGT(encryptedSz, 0);

        /* Decode encryptedData */
        decodedSz = wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
                                                    decoded, sizeof(decoded));

        AssertIntEQ(XMEMCMP(decoded, data, decodedSz), 0);
        /* Keep values for last itr. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
        }
    }
    if (pkcs7 == NULL || testSz == 0) {
        AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    }

    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(NULL, encrypted,
                     sizeof(encrypted)),BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, NULL,
                     sizeof(encrypted)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                     0), BAD_FUNC_ARG);
    /* Testing the struct. */
    tmpBytePtr = pkcs7->content;
    pkcs7->content = NULL;
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                             sizeof(encrypted)), BAD_FUNC_ARG);
    pkcs7->content = tmpBytePtr;
    tmpWrd32 = pkcs7->contentSz;
    pkcs7->contentSz = 0;
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                             sizeof(encrypted)), BAD_FUNC_ARG);
    pkcs7->contentSz = tmpWrd32;
    tmpInt = pkcs7->encryptOID;
    pkcs7->encryptOID = 0;
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                             sizeof(encrypted)), BAD_FUNC_ARG);
    pkcs7->encryptOID = tmpInt;
    tmpBytePtr = pkcs7->encryptionKey;
    pkcs7->encryptionKey = NULL;
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                             sizeof(encrypted)), BAD_FUNC_ARG);
    pkcs7->encryptionKey = tmpBytePtr;
    tmpWrd32 = pkcs7->encryptionKeySz;
    pkcs7->encryptionKeySz = 0;
    AssertIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
                             sizeof(encrypted)), BAD_FUNC_ARG);
    pkcs7->encryptionKeySz = tmpWrd32;

    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(NULL, encrypted, encryptedSz,
                decoded, sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, NULL, encryptedSz,
                decoded, sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, 0,
                decoded, sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
                NULL, sizeof(decoded)), BAD_FUNC_ARG);
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
                decoded, 0), BAD_FUNC_ARG);
    /* Test struct fields */

    tmpBytePtr = pkcs7->encryptionKey;
    pkcs7->encryptionKey = NULL;
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
                                   decoded, sizeof(decoded)), BAD_FUNC_ARG);
    pkcs7->encryptionKey = tmpBytePtr;
    pkcs7->encryptionKeySz = 0;
    AssertIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, encryptedSz,
                                   decoded, sizeof(decoded)), BAD_FUNC_ARG);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_EncodeEncryptedData() */

/*
 * Testing wc_PKCS7_Degenerate()
 */
static int test_wc_PKCS7_Degenerate(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)
    PKCS7* pkcs7;
    char   fName[] = "./certs/test-degenerate.p7b";
    XFILE  f;
    byte   der[4096];
    word32 derSz;
    int    ret;

    AssertNotNull(f = XFOPEN(fName, "rb"));
    AssertIntGT((ret = (int)fread(der, 1, sizeof(der), f)), 0);
    derSz = (word32)ret;
    XFCLOSE(f);

    /* test degenerate success */
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#else
    AssertIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif
    wc_PKCS7_Free(pkcs7);

    /* test with turning off degenerate cases */
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    wc_PKCS7_AllowDegenerate(pkcs7, 0); /* override allowing degenerate case */
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), PKCS7_NO_SIGNER_E);
    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_Degenerate() */

#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    defined(ASN_BER_TO_DER) && !defined(NO_DES3) && !defined(NO_SHA)
static byte berContent[] = {
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x03, 0xA0, 0x80, 0x30,
    0x80, 0x02, 0x01, 0x00, 0x31, 0x82, 0x01, 0x48,
    0x30, 0x82, 0x01, 0x44, 0x02, 0x01, 0x00, 0x30,
    0x81, 0xAC, 0x30, 0x81, 0x9E, 0x31, 0x0B, 0x30,
    0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
    0x55, 0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03,
    0x55, 0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E,
    0x74, 0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E,
    0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42,
    0x6F, 0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x15,
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C,
    0x0C, 0x77, 0x6F, 0x6C, 0x66, 0x53, 0x53, 0x4C,
    0x5F, 0x31, 0x30, 0x32, 0x34, 0x31, 0x19, 0x30,
    0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x10,
    0x50, 0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x6D,
    0x69, 0x6E, 0x67, 0x2D, 0x31, 0x30, 0x32, 0x34,
    0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
    0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
    0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09,
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
    0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
    0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E,
    0x63, 0x6F, 0x6D, 0x02, 0x09, 0x00, 0xBB, 0xD3,
    0x10, 0x03, 0xE6, 0x9D, 0x28, 0x03, 0x30, 0x0D,
    0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
    0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80,
    0x2F, 0xF9, 0x77, 0x4F, 0x04, 0x5C, 0x16, 0x62,
    0xF0, 0x77, 0x8D, 0x95, 0x4C, 0xB1, 0x44, 0x9A,
    0x8C, 0x3C, 0x8C, 0xE4, 0xD1, 0xC1, 0x14, 0x72,
    0xD0, 0x4A, 0x1A, 0x94, 0x27, 0x0F, 0xAA, 0xE8,
    0xD0, 0xA2, 0xE7, 0xED, 0x4C, 0x7F, 0x0F, 0xC7,
    0x1B, 0xFB, 0x81, 0x0E, 0x76, 0x8F, 0xDD, 0x32,
    0x11, 0x68, 0xA0, 0x13, 0xD2, 0x8D, 0x95, 0xEF,
    0x80, 0x53, 0x81, 0x0E, 0x1F, 0xC8, 0xD6, 0x76,
    0x5C, 0x31, 0xD3, 0x77, 0x33, 0x29, 0xA6, 0x1A,
    0xD3, 0xC6, 0x14, 0x36, 0xCA, 0x8E, 0x7D, 0x72,
    0xA0, 0x29, 0x4C, 0xC7, 0x3A, 0xAF, 0xFE, 0xF7,
    0xFC, 0xD7, 0xE2, 0x8F, 0x6A, 0x20, 0x46, 0x09,
    0x40, 0x22, 0x2D, 0x79, 0x38, 0x11, 0xB1, 0x4A,
    0xE3, 0x48, 0xE8, 0x10, 0x37, 0xA0, 0x22, 0xF7,
    0xB4, 0x79, 0xD1, 0xA9, 0x3D, 0xC2, 0xAB, 0x37,
    0xAE, 0x82, 0x68, 0x1A, 0x16, 0xEF, 0x33, 0x0C,
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x14, 0x06,
    0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x03,
    0x07, 0x04, 0x08, 0xAD, 0xD0, 0x38, 0x9B, 0x16,
    0x4B, 0x7F, 0x99, 0xA0, 0x80, 0x04, 0x82, 0x03,
    0xE8, 0x6D, 0x48, 0xFB, 0x8A, 0xBD, 0xED, 0x6C,
    0xCD, 0xC6, 0x48, 0xFD, 0xB7, 0xB0, 0x7C, 0x86,
    0x2C, 0x8D, 0xF0, 0x23, 0x12, 0xD8, 0xA3, 0x2A,
    0x21, 0x6F, 0x8B, 0x75, 0xBB, 0x47, 0x7F, 0xC9,
    0xBA, 0xBA, 0xFF, 0x91, 0x09, 0x01, 0x7A, 0x5C,
    0x96, 0x02, 0xB8, 0x8E, 0xF8, 0x67, 0x7E, 0x8F,
    0xF9, 0x51, 0x0E, 0xFF, 0x8E, 0xE2, 0x61, 0xC0,
    0xDF, 0xFA, 0xE2, 0x4C, 0x50, 0x90, 0xAE, 0xA1,
    0x15, 0x38, 0x3D, 0xBE, 0x88, 0xD7, 0x57, 0xC0,
    0x11, 0x44, 0xA2, 0x61, 0x05, 0x49, 0x6A, 0x94,
    0x04, 0x10, 0xD9, 0xC2, 0x2D, 0x15, 0x20, 0x0D,
    0xBD, 0xA2, 0xEF, 0xE4, 0x68, 0xFA, 0x39, 0x75,
    0x7E, 0xD8, 0x64, 0x44, 0xCB, 0xE0, 0x00, 0x6D,
    0x57, 0x4E, 0x8A, 0x17, 0xA9, 0x83, 0x6C, 0x7F,
    0xFE, 0x01, 0xEE, 0xDE, 0x99, 0x3A, 0xB2, 0xFF,
    0xD3, 0x72, 0x78, 0xBA, 0xF1, 0x23, 0x54, 0x48,
    0x02, 0xD8, 0x38, 0xA9, 0x54, 0xE5, 0x4A, 0x81,
    0xB9, 0xC0, 0x67, 0xB2, 0x7D, 0x3C, 0x6F, 0xCE,
    0xA4, 0xDD, 0x34, 0x5F, 0x60, 0xB1, 0xA3, 0x7A,
    0xE4, 0x43, 0xF2, 0x89, 0x64, 0x35, 0x09, 0x32,
    0x51, 0xFB, 0x5C, 0x67, 0x0C, 0x3B, 0xFC, 0x36,
    0x6B, 0x37, 0x43, 0x6C, 0x03, 0xCD, 0x44, 0xC7,
    0x2B, 0x62, 0xD6, 0xD1, 0xF4, 0x07, 0x7B, 0x19,
    0x91, 0xF0, 0xD7, 0xF5, 0x54, 0xBC, 0x0F, 0x42,
    0x6B, 0x69, 0xF7, 0xA3, 0xC8, 0xEE, 0xB9, 0x7A,
    0x9E, 0x3D, 0xDF, 0x53, 0x47, 0xF7, 0x50, 0x67,
    0x00, 0xCF, 0x2B, 0x3B, 0xE9, 0x85, 0xEE, 0xBD,
    0x4C, 0x64, 0x66, 0x0B, 0x77, 0x80, 0x9D, 0xEF,
    0x11, 0x32, 0x77, 0xA8, 0xA4, 0x5F, 0xEE, 0x2D,
    0xE0, 0x43, 0x87, 0x76, 0x87, 0x53, 0x4E, 0xD7,
    0x1A, 0x04, 0x7B, 0xE1, 0xD1, 0xE1, 0xF5, 0x87,
    0x51, 0x13, 0xE0, 0xC2, 0xAA, 0xA3, 0x4B, 0xAA,
    0x9E, 0xB4, 0xA6, 0x1D, 0x4E, 0x28, 0x57, 0x0B,
    0x80, 0x90, 0x81, 0x4E, 0x04, 0xF5, 0x30, 0x8D,
    0x51, 0xCE, 0x57, 0x2F, 0x88, 0xC5, 0x70, 0xC4,
    0x06, 0x8F, 0xDD, 0x37, 0xC1, 0x34, 0x1E, 0x0E,
    0x15, 0x32, 0x23, 0x92, 0xAB, 0x40, 0xEA, 0xF7,
    0x43, 0xE2, 0x1D, 0xE2, 0x4B, 0xC9, 0x91, 0xF4,
    0x63, 0x21, 0x34, 0xDB, 0xE9, 0x86, 0x83, 0x1A,
    0xD2, 0x52, 0xEF, 0x7A, 0xA2, 0xEE, 0xA4, 0x11,
    0x56, 0xD3, 0x6C, 0xF5, 0x6D, 0xE4, 0xA5, 0x2D,
    0x99, 0x02, 0x10, 0xDF, 0x29, 0xC5, 0xE3, 0x0B,
    0xC4, 0xA1, 0xEE, 0x5F, 0x4A, 0x10, 0xEE, 0x85,
    0x73, 0x2A, 0x92, 0x15, 0x2C, 0xC8, 0xF4, 0x8C,
    0xD7, 0x3D, 0xBC, 0xAD, 0x18, 0xE0, 0x59, 0xD3,
    0xEE, 0x75, 0x90, 0x1C, 0xCC, 0x76, 0xC6, 0x64,
    0x17, 0xD2, 0xD0, 0x91, 0xA6, 0xD0, 0xC1, 0x4A,
    0xAA, 0x58, 0x22, 0xEC, 0x45, 0x98, 0xF2, 0xCC,
    0x4C, 0xE4, 0xBF, 0xED, 0xF6, 0x44, 0x72, 0x36,
    0x65, 0x3F, 0xE3, 0xB5, 0x8B, 0x3E, 0x54, 0x9C,
    0x82, 0x86, 0x5E, 0xB0, 0xF2, 0x12, 0xE5, 0x69,
    0xFA, 0x46, 0xA2, 0x54, 0xFC, 0xF5, 0x4B, 0xE0,
    0x24, 0x3B, 0x99, 0x04, 0x1A, 0x7A, 0xF7, 0xD1,
    0xFF, 0x68, 0x97, 0xB2, 0x85, 0x82, 0x95, 0x27,
    0x2B, 0xF4, 0xE7, 0x1A, 0x74, 0x19, 0xEC, 0x8C,
    0x4E, 0xA7, 0x0F, 0xAD, 0x4F, 0x5A, 0x02, 0x80,
    0xC1, 0x6A, 0x9E, 0x54, 0xE4, 0x8E, 0xA3, 0x41,
    0x3F, 0x6F, 0x9C, 0x82, 0x9F, 0x83, 0xB0, 0x44,
    0x01, 0x5F, 0x10, 0x9D, 0xD3, 0xB6, 0x33, 0x5B,
    0xAF, 0xAC, 0x6B, 0x57, 0x2A, 0x01, 0xED, 0x0E,
    0x17, 0xB9, 0x80, 0x76, 0x12, 0x1C, 0x51, 0x56,
    0xDD, 0x6D, 0x94, 0xAB, 0xD2, 0xE5, 0x15, 0x2D,
    0x3C, 0xC5, 0xE8, 0x62, 0x05, 0x8B, 0x40, 0xB1,
    0xC2, 0x83, 0xCA, 0xAC, 0x4B, 0x8B, 0x39, 0xF7,
    0xA0, 0x08, 0x43, 0x5C, 0xF7, 0xE8, 0xED, 0x40,
    0x72, 0x73, 0xE3, 0x6B, 0x18, 0x67, 0xA0, 0xB6,
    0x0F, 0xED, 0x8F, 0x9A, 0xE4, 0x27, 0x62, 0x23,
    0xAA, 0x6D, 0x6C, 0x31, 0xC9, 0x9D, 0x6B, 0xE0,
    0xBF, 0x9D, 0x7D, 0x2E, 0x76, 0x71, 0x06, 0x39,
    0xAC, 0x96, 0x1C, 0xAF, 0x30, 0xF2, 0x62, 0x9C,
    0x84, 0x3F, 0x43, 0x5E, 0x19, 0xA8, 0xE5, 0x3C,
    0x9D, 0x43, 0x3C, 0x43, 0x41, 0xE8, 0x82, 0xE7,
    0x5B, 0xF3, 0xE2, 0x15, 0xE3, 0x52, 0x20, 0xFD,
    0x0D, 0xB2, 0x4D, 0x48, 0xAD, 0x53, 0x7E, 0x0C,
    0xF0, 0xB9, 0xBE, 0xC9, 0x58, 0x4B, 0xC8, 0xA8,
    0xA3, 0x36, 0xF1, 0x2C, 0xD2, 0xE1, 0xC8, 0xC4,
    0x3C, 0x48, 0x70, 0xC2, 0x6D, 0x6C, 0x3D, 0x99,
    0xAC, 0x43, 0x19, 0x69, 0xCA, 0x67, 0x1A, 0xC9,
    0xE1, 0x47, 0xFA, 0x0A, 0xE6, 0x5B, 0x6F, 0x61,
    0xD0, 0x03, 0xE4, 0x03, 0x4B, 0xFD, 0xE2, 0xA5,
    0x8D, 0x83, 0x01, 0x7E, 0xC0, 0x7B, 0x2E, 0x0B,
    0x29, 0xDD, 0xD6, 0xDC, 0x71, 0x46, 0xBD, 0x9A,
    0x40, 0x46, 0x1E, 0x0A, 0xB1, 0x00, 0xE7, 0x71,
    0x29, 0x77, 0xFC, 0x9A, 0x76, 0x8A, 0x5F, 0x66,
    0x9B, 0x63, 0x91, 0x12, 0x78, 0xBF, 0x67, 0xAD,
    0xA1, 0x72, 0x9E, 0xC5, 0x3E, 0xE5, 0xCB, 0xAF,
    0xD6, 0x5A, 0x0D, 0xB6, 0x9B, 0xA3, 0x78, 0xE8,
    0xB0, 0x8F, 0x69, 0xED, 0xC1, 0x73, 0xD5, 0xE5,
    0x1C, 0x18, 0xA0, 0x58, 0x4C, 0x49, 0xBD, 0x91,
    0xCE, 0x15, 0x0D, 0xAA, 0x5A, 0x07, 0xEA, 0x1C,
    0xA7, 0x4B, 0x11, 0x31, 0x80, 0xAF, 0xA1, 0x0A,
    0xED, 0x6C, 0x70, 0xE4, 0xDB, 0x75, 0x86, 0xAE,
    0xBF, 0x4A, 0x05, 0x72, 0xDE, 0x84, 0x8C, 0x7B,
    0x59, 0x81, 0x58, 0xE0, 0xC0, 0x15, 0xB5, 0xF3,
    0xD5, 0x73, 0x78, 0x83, 0x53, 0xDA, 0x92, 0xC1,
    0xE6, 0x71, 0x74, 0xC7, 0x7E, 0xAA, 0x36, 0x06,
    0xF0, 0xDF, 0xBA, 0xFB, 0xEF, 0x54, 0xE8, 0x11,
    0xB2, 0x33, 0xA3, 0x0B, 0x9E, 0x0C, 0x59, 0x75,
    0x13, 0xFA, 0x7F, 0x88, 0xB9, 0x86, 0xBD, 0x1A,
    0xDB, 0x52, 0x12, 0xFB, 0x6D, 0x1A, 0xCB, 0x49,
    0x94, 0x94, 0xC4, 0xA9, 0x99, 0xC0, 0xA4, 0xB6,
    0x60, 0x36, 0x09, 0x94, 0x2A, 0xD5, 0xC4, 0x26,
    0xF4, 0xA3, 0x6A, 0x0E, 0x57, 0x8B, 0x7C, 0xA4,
    0x1D, 0x75, 0xE8, 0x2A, 0xF3, 0xC4, 0x3C, 0x7D,
    0x45, 0x6D, 0xD8, 0x24, 0xD1, 0x3B, 0xF7, 0xCF,
    0xE4, 0x45, 0x2A, 0x55, 0xE5, 0xA9, 0x1F, 0x1C,
    0x8F, 0x55, 0x8D, 0xC1, 0xF7, 0x74, 0xCC, 0x26,
    0xC7, 0xBA, 0x2E, 0x5C, 0xC1, 0x71, 0x0A, 0xAA,
    0xD9, 0x6D, 0x76, 0xA7, 0xF9, 0xD1, 0x18, 0xCB,
    0x5A, 0x52, 0x98, 0xA8, 0x0D, 0x3F, 0x06, 0xFC,
    0x49, 0x11, 0x21, 0x5F, 0x86, 0x19, 0x33, 0x81,
    0xB5, 0x7A, 0xDA, 0xA1, 0x47, 0xBF, 0x7C, 0xD7,
    0x05, 0x96, 0xC7, 0xF5, 0xC1, 0x61, 0xE5, 0x18,
    0xA5, 0x38, 0x68, 0xED, 0xB4, 0x17, 0x62, 0x0D,
    0x01, 0x5E, 0xC3, 0x04, 0xA6, 0xBA, 0xB1, 0x01,
    0x60, 0x5C, 0xC1, 0x3A, 0x34, 0x97, 0xD6, 0xDB,
    0x67, 0x73, 0x4D, 0x33, 0x96, 0x01, 0x67, 0x44,
    0xEA, 0x47, 0x5E, 0x44, 0xB5, 0xE5, 0xD1, 0x6C,
    0x20, 0xA9, 0x6D, 0x4D, 0xBC, 0x02, 0xF0, 0x70,
    0xE4, 0xDD, 0xE9, 0xD5, 0x5C, 0x28, 0x29, 0x0B,
    0xB4, 0x60, 0x2A, 0xF1, 0xF7, 0x1A, 0xF0, 0x36,
    0xAE, 0x51, 0x3A, 0xAE, 0x6E, 0x48, 0x7D, 0xC7,
    0x5C, 0xF3, 0xDC, 0xF6, 0xED, 0x27, 0x4E, 0x8E,
    0x48, 0x18, 0x3E, 0x08, 0xF1, 0xD8, 0x3D, 0x0D,
    0xE7, 0x2F, 0x65, 0x8A, 0x6F, 0xE2, 0x1E, 0x06,
    0xC1, 0x04, 0x58, 0x7B, 0x4A, 0x75, 0x60, 0x92,
    0x13, 0xC6, 0x40, 0x2D, 0x3A, 0x8A, 0xD1, 0x03,
    0x05, 0x1F, 0x28, 0x66, 0xC2, 0x57, 0x2A, 0x4C,
    0xE1, 0xA3, 0xCB, 0xA1, 0x95, 0x30, 0x10, 0xED,
    0xDF, 0xAE, 0x70, 0x49, 0x4E, 0xF6, 0xB4, 0x5A,
    0xB6, 0x22, 0x56, 0x37, 0x05, 0xE7, 0x3E, 0xB2,
    0xE3, 0x96, 0x62, 0xEC, 0x09, 0x53, 0xC0, 0x50,
    0x3D, 0xA7, 0xBC, 0x9B, 0x39, 0x02, 0x26, 0x16,
    0xB5, 0x34, 0x17, 0xD4, 0xCA, 0xFE, 0x1D, 0xE4,
    0x5A, 0xDA, 0x4C, 0xC2, 0xCA, 0x8E, 0x79, 0xBF,
    0xD8, 0x4C, 0xBB, 0xFA, 0x30, 0x7B, 0xA9, 0x3E,
    0x52, 0x19, 0xB1, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};
#endif /* HAVE_PKCS7 && !NO_FILESYSTEM && ASN_BER_TO_DER &&
        * !NO_DES3 && !NO_SHA
        */

/*
 * Testing wc_PKCS7_BER()
 */
static int test_wc_PKCS7_BER(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    !defined(NO_SHA) && defined(ASN_BER_TO_DER)
    PKCS7* pkcs7;
    char   fName[] = "./certs/test-ber-exp02-05-2022.p7b";
    XFILE  f;
    byte   der[4096];
#ifndef NO_DES3
    byte   decoded[2048];
#endif
    word32 derSz;
    int    ret;

    AssertNotNull(f = XFOPEN(fName, "rb"));
    AssertIntGT((ret = (int)fread(der, 1, sizeof(der), f)), 0);
    derSz = (word32)ret;
    XFCLOSE(f);

    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#else
    AssertIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif
    wc_PKCS7_Free(pkcs7);

#ifndef NO_DES3
    /* decode BER content */
    AssertNotNull(f = XFOPEN("./certs/1024/client-cert.der", "rb"));
    AssertIntGT((ret = (int)fread(der, 1, sizeof(der), f)), 0);
    derSz = (word32)ret;
    XFCLOSE(f);
    AssertNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
#ifndef NO_RSA
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#else
    AssertIntNE(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#endif

    AssertNotNull(f = XFOPEN("./certs/1024/client-key.der", "rb"));
    AssertIntGT((ret = (int)fread(der, 1, sizeof(der), f)), 0);
    derSz = (word32)ret;
    XFCLOSE(f);
    pkcs7->privateKey   = der;
    pkcs7->privateKeySz = derSz;
#ifndef NO_RSA
#ifdef WOLFSSL_SP_MATH
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), WC_KEY_SIZE_E);
#else
    AssertIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), 0);
#endif
#else
    AssertIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), NOT_COMPILED_IN);
#endif
    wc_PKCS7_Free(pkcs7);
#endif /* !NO_DES3 */

    res = TEST_RES_CHECK(1);
#endif
    return res;
} /* END test_wc_PKCS7_BER() */

static int test_PKCS7_signed_enveloped(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_RSA) && !defined(NO_AES) && \
    !defined(NO_FILESYSTEM)
    XFILE  f;
    PKCS7* pkcs7;
#ifdef HAVE_AES_CBC
    PKCS7* inner;
#endif
    void*  pt;
    WC_RNG rng;
    unsigned char key[FOURK_BUF/2];
    unsigned char cert[FOURK_BUF/2];
    unsigned char env[FOURK_BUF];
    int envSz  = FOURK_BUF;
    int keySz;
    int certSz;

    unsigned char sig[FOURK_BUF * 2];
    int sigSz = FOURK_BUF * 2;
#ifdef HAVE_AES_CBC
    unsigned char decoded[FOURK_BUF];
    int decodedSz = FOURK_BUF;
#endif

    /* load cert */
    AssertNotNull(f = XFOPEN(cliCertDerFile, "rb"));
    AssertIntGT((certSz = (int)XFREAD(cert, 1, sizeof(cert), f)), 0);
    XFCLOSE(f);

    /* load key */
    AssertNotNull(f = XFOPEN(cliKeyFile, "rb"));
    AssertIntGT((keySz = (int)XFREAD(key, 1, sizeof(key), f)), 0);
    XFCLOSE(f);
    keySz = wolfSSL_KeyPemToDer(key, keySz, key, keySz, NULL);

    /* sign cert for envelope */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_InitRng(&rng), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    pkcs7->content    = cert;
    pkcs7->contentSz  = certSz;
    pkcs7->contentOID = DATA;
    pkcs7->privateKey   = key;
    pkcs7->privateKeySz = keySz;
    pkcs7->encryptOID   = RSAk;
    pkcs7->hashOID      = SHA256h;
    pkcs7->rng          = &rng;
    AssertIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

#ifdef HAVE_AES_CBC
    /* create envelope */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    pkcs7->content   = sig;
    pkcs7->contentSz = sigSz;
    pkcs7->contentOID = DATA;
    pkcs7->encryptOID = AES256CBCb;
    pkcs7->privateKey   = key;
    pkcs7->privateKeySz = keySz;
    AssertIntGT((envSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, env, envSz)), 0);
    AssertIntLT(wc_PKCS7_EncodeEnvelopedData(pkcs7, env, 2), 0);
    wc_PKCS7_Free(pkcs7);
#endif

    /* create bad signed enveloped data */
    sigSz = FOURK_BUF * 2;
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_InitRng(&rng), 0);
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    pkcs7->content    = env;
    pkcs7->contentSz  = envSz;
    pkcs7->contentOID = DATA;
    pkcs7->privateKey   = key;
    pkcs7->privateKeySz = keySz;
    pkcs7->encryptOID   = RSAk;
    pkcs7->hashOID      = SHA256h;
    pkcs7->rng = &rng;

    /* Set no certs in bundle for this test. Hang on to the pointer though to
     * free it later. */
    pt = (void*)pkcs7->certList;
    pkcs7->certList = NULL; /* no certs in bundle */
    AssertIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, sigSz)), 0);
    pkcs7->certList = (Pkcs7Cert*)pt; /* restore pointer for PKCS7 free call */
    wc_PKCS7_Free(pkcs7);

    /* check verify fails */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, sigSz),
            PKCS7_SIGNEEDS_CHECK);

    /* try verifying the signature manually */
    {
        RsaKey rKey;
        word32 idx = 0;
        byte digest[MAX_SEQ_SZ + MAX_ALGO_SZ + MAX_OCTET_STR_SZ +
            WC_MAX_DIGEST_SIZE];
        int  digestSz;

        AssertIntEQ(wc_InitRsaKey(&rKey, HEAP_HINT), 0);
        AssertIntEQ(wc_RsaPrivateKeyDecode(key, &idx, &rKey, keySz), 0);
        digestSz = wc_RsaSSL_Verify(pkcs7->signature, pkcs7->signatureSz,
                    digest, sizeof(digest), &rKey);
        AssertIntGT(digestSz, 0);
        AssertIntEQ(digestSz, pkcs7->pkcs7DigestSz);
        AssertIntEQ(XMEMCMP(digest, pkcs7->pkcs7Digest, digestSz), 0);
        AssertIntEQ(wc_FreeRsaKey(&rKey), 0);
        /* verify was success */
    }

    wc_PKCS7_Free(pkcs7);

    /* initializing the PKCS7 struct with the signing certificate should pass */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, sigSz), 0);
    wc_PKCS7_Free(pkcs7);

    /* create valid degenerate bundle */
    sigSz = FOURK_BUF * 2;
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    pkcs7->content    = env;
    pkcs7->contentSz  = envSz;
    pkcs7->contentOID = DATA;
    pkcs7->privateKey   = key;
    pkcs7->privateKeySz = keySz;
    pkcs7->encryptOID   = RSAk;
    pkcs7->hashOID      = SHA256h;
    pkcs7->rng = &rng;
    AssertIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, DEGENERATE_SID), 0);
    AssertIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    /* check verify */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, sigSz), 0);
    AssertNotNull(pkcs7->content);

#ifdef HAVE_AES_CBC
    /* check decode */
    AssertNotNull(inner = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_InitWithCert(inner, cert, certSz), 0);
    inner->privateKey   = key;
    inner->privateKeySz = keySz;
    AssertIntGT((decodedSz = wc_PKCS7_DecodeEnvelopedData(inner, pkcs7->content,
                   pkcs7->contentSz, decoded, decodedSz)), 0);
    wc_PKCS7_Free(inner);
#endif
    wc_PKCS7_Free(pkcs7);

#ifdef HAVE_AES_CBC
    /* check cert set */
    AssertNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    AssertIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    AssertIntEQ(wc_PKCS7_VerifySignedData(pkcs7, decoded, decodedSz), 0);
    AssertNotNull(pkcs7->singleCert);
    AssertIntNE(pkcs7->singleCertSz, 0);
    wc_PKCS7_Free(pkcs7);
#endif

    res = TEST_RES_CHECK(1);
#endif /* HAVE_PKCS7 && !NO_RSA && !NO_AES */
    return res;
}

static int test_wc_PKCS7_NoDefaultSignedAttribs(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7*      pkcs7;
    void*       heap = NULL;

    pkcs7 = wc_PKCS7_New(heap, testDevId);
    AssertNotNull(pkcs7);
    AssertIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    AssertIntEQ(wc_PKCS7_NoDefaultSignedAttribs(NULL), BAD_FUNC_ARG);

    AssertIntEQ(wc_PKCS7_NoDefaultSignedAttribs(pkcs7), 0);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_PKCS7_SetOriEncryptCtx(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7*      pkcs7;
    void*       heap = NULL;
    WOLFSSL_CTX* ctx;
    ctx = NULL;

    pkcs7 = wc_PKCS7_New(heap, testDevId);
    AssertNotNull(pkcs7);
    AssertIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    AssertIntEQ(wc_PKCS7_SetOriEncryptCtx(NULL, ctx), BAD_FUNC_ARG);

    AssertIntEQ(wc_PKCS7_SetOriEncryptCtx(pkcs7, ctx), 0);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_PKCS7_SetOriDecryptCtx(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7*      pkcs7;
    void*       heap = NULL;
    WOLFSSL_CTX* ctx;
    ctx = NULL;

    pkcs7 = wc_PKCS7_New(heap, testDevId);
    AssertNotNull(pkcs7);
    AssertIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    AssertIntEQ(wc_PKCS7_SetOriDecryptCtx(NULL, ctx), BAD_FUNC_ARG);

    AssertIntEQ(wc_PKCS7_SetOriDecryptCtx(pkcs7, ctx), 0);

    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_PKCS7_DecodeCompressedData(void)
{
    int res = TEST_SKIPPED;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES) && defined(HAVE_LIBZ)
    PKCS7* pkcs7;
    void*  heap = NULL;
    byte   out[4096];
    byte   *decompressed;
    int    outSz, decompressedSz;

    const char* cert = "./certs/client-cert.pem";
    byte*  cert_buf = NULL;
    size_t cert_sz = 0;

    AssertIntEQ(load_file(cert, &cert_buf, &cert_sz), 0);
    AssertNotNull((decompressed =
                (byte*)XMALLOC(cert_sz, heap, DYNAMIC_TYPE_TMP_BUFFER)));
    decompressedSz = (int)cert_sz;
    AssertNotNull((pkcs7 = wc_PKCS7_New(heap, testDevId)));

    pkcs7->content    = (byte*)cert_buf;
    pkcs7->contentSz  = (word32)cert_sz;
    pkcs7->contentOID = DATA;

    AssertIntGT((outSz = wc_PKCS7_EncodeCompressedData(pkcs7, out,
                    sizeof(out))), 0);
    wc_PKCS7_Free(pkcs7);

    /* compressed key should be smaller than when started */
    AssertIntLT(outSz, cert_sz);

    /* test decompression */
    AssertNotNull((pkcs7 = wc_PKCS7_New(heap, testDevId)));
    AssertIntEQ(pkcs7->contentOID, 0);

    /* fail case with out buffer too small */
    AssertIntLT(wc_PKCS7_DecodeCompressedData(pkcs7, out, outSz,
                decompressed, outSz), 0);

    /* success case */
    AssertIntEQ(wc_PKCS7_DecodeCompressedData(pkcs7, out, outSz,
                decompressed, decompressedSz), cert_sz);
    AssertIntEQ(pkcs7->contentOID, DATA);
    AssertIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);
    decompressed = NULL;

    /* test decompression function with different 'max' inputs */
    outSz = sizeof(out);
    AssertIntGT((outSz = wc_Compress(out, outSz, cert_buf, (word32)cert_sz, 0)),
            0);
    AssertIntLT(wc_DeCompressDynamic(&decompressed, 1, DYNAMIC_TYPE_TMP_BUFFER,
            out, outSz, 0, heap), 0);
    AssertNull(decompressed);
    AssertIntGT(wc_DeCompressDynamic(&decompressed, -1, DYNAMIC_TYPE_TMP_BUFFER,
            out, outSz, 0, heap), 0);
    AssertNotNull(decompressed);
    AssertIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);
    decompressed = NULL;

    AssertIntGT(wc_DeCompressDynamic(&decompressed, DYNAMIC_TYPE_TMP_BUFFER, 5,
            out, outSz, 0, heap), 0);
    AssertNotNull(decompressed);
    AssertIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);

    if (cert_buf)
        free(cert_buf);
    wc_PKCS7_Free(pkcs7);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_i2d_PKCS12(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_ASN) && !defined(NO_PWDBASED) && defined(HAVE_PKCS12) \
    && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES) && !defined(NO_DES3) && !defined(NO_SHA)
    WC_PKCS12* pkcs12 = NULL;
    unsigned char der[FOURK_BUF * 2];
    unsigned char* pt;
    int derSz;
    unsigned char out[FOURK_BUF * 2];
    int outSz = FOURK_BUF * 2;

    const char p12_f[] = "./certs/test-servercert.p12";
    XFILE f;

    f =  XFOPEN(p12_f, "rb");
    AssertNotNull(f);
    derSz = (int)XFREAD(der, 1, sizeof(der), f);
    AssertIntGT(derSz, 0);
    XFCLOSE(f);

    AssertNotNull(pkcs12 = wc_PKCS12_new());
    AssertIntEQ(wc_d2i_PKCS12(der, derSz, pkcs12), 0);
    AssertIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), LENGTH_ONLY_E);
    AssertIntEQ(outSz, derSz);

    outSz = derSz - 1;
    pt = out;
    AssertIntLE(wc_i2d_PKCS12(pkcs12, &pt, &outSz), 0);

    outSz = derSz;
    AssertIntEQ(wc_i2d_PKCS12(pkcs12, &pt, &outSz), derSz);
    AssertIntEQ((pt == out), 0);

    pt = NULL;
    AssertIntEQ(wc_i2d_PKCS12(pkcs12, &pt, NULL), derSz);
    XFREE(pt, NULL, DYNAMIC_TYPE_PKCS);
    wc_PKCS12_free(pkcs12);

    /* Run the same test but use wc_d2i_PKCS12_fp. */
    AssertNotNull(pkcs12 = wc_PKCS12_new());
    AssertIntEQ(wc_d2i_PKCS12_fp("./certs/test-servercert.p12", &pkcs12), 0);
    AssertIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), LENGTH_ONLY_E);
    AssertIntEQ(outSz, derSz);
    wc_PKCS12_free(pkcs12);

    /* wc_d2i_PKCS12_fp can also allocate the PKCS12 object for the caller. */
    pkcs12 = NULL;
    AssertIntEQ(wc_d2i_PKCS12_fp("./certs/test-servercert.p12", &pkcs12), 0);
    AssertIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), LENGTH_ONLY_E);
    AssertIntEQ(outSz, derSz);
    wc_PKCS12_free(pkcs12);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}


/* Testing wc_SignatureGetSize() for signature type ECC */
static int test_wc_SignatureGetSize_ecc(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SIG_WRAPPER
    int ret;
    #if defined(HAVE_ECC) && !defined(NO_ECC256)
        enum wc_SignatureType sig_type;
        word32 key_len;

        /* Initialize ECC Key */
        ecc_key ecc;
        const char* qx =
            "fa2737fb93488d19caef11ae7faf6b7f4bcd67b286e3fc54e8a65c2b74aeccb0";
        const char* qy =
            "d4ccd6dae698208aa8c3a6f39e45510d03be09b2f124bfc067856c324f9b4d09";
        const char* d =
            "be34baa8d040a3b991f9075b56ba292f755b90e4b6dc10dad36715c33cfdac25";

        ret = wc_ecc_init(&ecc);
        if (ret == 0) {
            ret = wc_ecc_import_raw(&ecc, qx, qy, d, "SECP256R1");
        }
        if (ret == 0) {
            /* Input for signature type ECC */
            sig_type = WC_SIGNATURE_TYPE_ECC;
            key_len = sizeof(ecc_key);
            ret = wc_SignatureGetSize(sig_type, &ecc, key_len);

            /* Test bad args */
            if (ret > 0) {
                sig_type = (enum wc_SignatureType) 100;
                ret = wc_SignatureGetSize(sig_type, &ecc, key_len);
                if (ret == BAD_FUNC_ARG) {
                    sig_type = WC_SIGNATURE_TYPE_ECC;
                    ret = wc_SignatureGetSize(sig_type, NULL, key_len);
                }
                if (ret >= 0) {
                    key_len = (word32) 0;
                    ret = wc_SignatureGetSize(sig_type, &ecc, key_len);
                }
                if (ret == BAD_FUNC_ARG) {
                    ret = SIG_TYPE_E;
                }
            }
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
        wc_ecc_free(&ecc);
    #else
        ret = SIG_TYPE_E;
    #endif

    res = TEST_RES_CHECK(ret == SIG_TYPE_E);
#endif /* NO_SIG_WRAPPER */
    return res;
}/* END test_wc_SignatureGetSize_ecc() */

/* Testing wc_SignatureGetSize() for signature type rsa */
static int test_wc_SignatureGetSize_rsa(void)
{
    int res = TEST_SKIPPED;
#ifndef NO_SIG_WRAPPER
    int ret = 0;
    #ifndef NO_RSA
        enum wc_SignatureType sig_type;
        word32 key_len;
        word32 idx = 0;

        /* Initialize RSA Key */
        RsaKey rsa_key;
        byte* tmp = NULL;
        size_t bytes;

        #ifdef USE_CERT_BUFFERS_1024
            bytes = (size_t)sizeof_client_key_der_1024;
            if (bytes < (size_t)sizeof_client_key_der_1024)
                bytes = (size_t)sizeof_client_cert_der_1024;
        #elif defined(USE_CERT_BUFFERS_2048)
            bytes = (size_t)sizeof_client_key_der_2048;
            if (bytes < (size_t)sizeof_client_cert_der_2048)
                bytes = (size_t)sizeof_client_cert_der_2048;
        #else
            bytes = FOURK_BUF;
        #endif

        tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (tmp != NULL) {
            #ifdef USE_CERT_BUFFERS_1024
                XMEMCPY(tmp, client_key_der_1024,
                    (size_t)sizeof_client_key_der_1024);
            #elif defined(USE_CERT_BUFFERS_2048)
                XMEMCPY(tmp, client_key_der_2048,
                    (size_t)sizeof_client_key_der_2048);
            #elif !defined(NO_FILESYSTEM)
                file = XFOPEN(clientKey, "rb");
                if (file != XBADFILE) {
                    bytes = (size_t)XFREAD(tmp, 1, FOURK_BUF, file);
                    XFCLOSE(file);
                }
                else {
                    ret = WOLFSSL_FATAL_ERROR;
                }
            #else
                ret = WOLFSSL_FATAL_ERROR;
            #endif
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }

        if (ret == 0) {
            ret = wc_InitRsaKey_ex(&rsa_key, HEAP_HINT, testDevId);
        }
        if (ret == 0) {
            ret = wc_RsaPrivateKeyDecode(tmp, &idx, &rsa_key, (word32)bytes);
        }

        if (ret == 0) {
            /* Input for signature type RSA */
            sig_type = WC_SIGNATURE_TYPE_RSA;
            key_len = sizeof(RsaKey);
            ret = wc_SignatureGetSize(sig_type, &rsa_key, key_len);

            /* Test bad args */
            if (ret > 0) {
                sig_type = (enum wc_SignatureType) 100;
                ret = wc_SignatureGetSize(sig_type, &rsa_key, key_len);
                if (ret == BAD_FUNC_ARG) {
                    sig_type = WC_SIGNATURE_TYPE_RSA;
                    ret = wc_SignatureGetSize(sig_type, NULL, key_len);
                }
            #ifndef HAVE_USER_RSA
                if (ret == BAD_FUNC_ARG)
            #else
                if (ret == 0)
            #endif
                {
                    key_len = (word32)0;
                    ret = wc_SignatureGetSize(sig_type, &rsa_key, key_len);
                }
                if (ret == BAD_FUNC_ARG) {
                    ret = SIG_TYPE_E;
                }
            }
        }
        else {
            ret = WOLFSSL_FATAL_ERROR;
        }
        wc_FreeRsaKey(&rsa_key);
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #else
        ret = SIG_TYPE_E;
    #endif

    res = TEST_RES_CHECK(ret == SIG_TYPE_E);
#endif /* NO_SIG_WRAPPER */
    return res;
}/* END test_wc_SignatureGetSize_rsa(void) */

/*----------------------------------------------------------------------------*
 | hash.h Tests
 *----------------------------------------------------------------------------*/

static int test_wc_HashInit(void)
{
    int ret = 0, i;  /* 0 indicates tests passed, 1 indicates failure */

    wc_HashAlg hash;

    /* enum for holding supported algorithms, #ifndef's restrict if disabled */
    enum wc_HashType enumArray[] = {
    #ifndef NO_MD5
            WC_HASH_TYPE_MD5,
    #endif
    #ifndef NO_SHA
            WC_HASH_TYPE_SHA,
    #endif
    #ifndef WOLFSSL_SHA224
            WC_HASH_TYPE_SHA224,
    #endif
    #ifndef NO_SHA256
            WC_HASH_TYPE_SHA256,
    #endif
    #ifndef WOLFSSL_SHA384
            WC_HASH_TYPE_SHA384,
    #endif
    #ifndef WOLFSSL_SHA512
            WC_HASH_TYPE_SHA512,
    #endif
    };
    /* dynamically finds the length */
    int enumlen = (sizeof(enumArray)/sizeof(enum wc_HashType));

    /* For loop to test various arguments... */
    for (i = 0; i < enumlen; i++) {
        /* check for bad args */
        if (wc_HashInit(&hash, enumArray[i]) == BAD_FUNC_ARG) {
            ret = 1;
            break;
        }
        wc_HashFree(&hash, enumArray[i]);

        /* check for null ptr */
        if (wc_HashInit(NULL, enumArray[i]) != BAD_FUNC_ARG) {
            ret = 1;
            break;
        }

    }  /* end of for loop */

    return TEST_RES_CHECK(ret == 0);
}  /* end of test_wc_HashInit */
/*
 * Unit test function for wc_HashSetFlags()
 */
static int test_wc_HashSetFlags(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_HASH_FLAGS
    wc_HashAlg hash;
    int ret = 0;
    word32 flags = 0;
    int i, j;
    int notSupportedLen;

    /* enum for holding supported algorithms, #ifndef's restrict if disabled */
    enum wc_HashType enumArray[] = {
    #ifndef NO_MD5
            WC_HASH_TYPE_MD5,
    #endif
    #ifndef NO_SHA
            WC_HASH_TYPE_SHA,
    #endif
    #ifdef WOLFSSL_SHA224
            WC_HASH_TYPE_SHA224,
    #endif
    #ifndef NO_SHA256
            WC_HASH_TYPE_SHA256,
    #endif
    #ifdef WOLFSSL_SHA384
            WC_HASH_TYPE_SHA384,
    #endif
    #ifdef WOLFSSL_SHA512
            WC_HASH_TYPE_SHA512,
    #endif
    #ifdef WOLFSSL_SHA3
            WC_HASH_TYPE_SHA3_224,
    #endif
    };
    enum wc_HashType notSupported[] = {
              WC_HASH_TYPE_MD5_SHA,
              WC_HASH_TYPE_MD2,
              WC_HASH_TYPE_MD4,
              WC_HASH_TYPE_BLAKE2B,
              WC_HASH_TYPE_BLAKE2S,
              WC_HASH_TYPE_NONE,
     };

    /* dynamically finds the length */
    int enumlen = (sizeof(enumArray)/sizeof(enum wc_HashType));

    /* For loop to test various arguments... */
    for (i = 0; i < enumlen; i++) {
        ret = wc_HashInit(&hash, enumArray[i]);
        if (ret == 0) {
            ret = wc_HashSetFlags(&hash, enumArray[i], flags);
        }
        if (ret == 0) {
            if (flags & WC_HASH_FLAG_ISCOPY) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_HashSetFlags(NULL, enumArray[i], flags);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }

        wc_HashFree(&hash, enumArray[i]);

    }
    /* For loop to test not supported cases */
    notSupportedLen = (sizeof(notSupported)/sizeof(enum wc_HashType));
    for (j = 0; ret == 0 && j < notSupportedLen; j++) {
        ret = wc_HashInit(&hash, notSupported[j]);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = wc_HashSetFlags(&hash, notSupported[j], flags);
            if (ret == 0) {
                ret = -1;
            }
            else if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_HashFree(&hash, notSupported[j]);
            if (ret == 0) {
                ret = -1;
            }
            else if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}  /* END test_wc_HashSetFlags */
/*
 * Unit test function for wc_HashGetFlags()
 */
static int test_wc_HashGetFlags(void)
{
    int res = TEST_SKIPPED;
#ifdef WOLFSSL_HASH_FLAGS
    wc_HashAlg hash;
    int ret = 0;
    word32 flags = 0;
    int i, j;

    /* enum for holding supported algorithms, #ifndef's restrict if disabled */
    enum wc_HashType enumArray[] = {
    #ifndef NO_MD5
            WC_HASH_TYPE_MD5,
    #endif
    #ifndef NO_SHA
            WC_HASH_TYPE_SHA,
    #endif
    #ifdef WOLFSSL_SHA224
            WC_HASH_TYPE_SHA224,
    #endif
    #ifndef NO_SHA256
            WC_HASH_TYPE_SHA256,
    #endif
    #ifdef WOLFSSL_SHA384
            WC_HASH_TYPE_SHA384,
    #endif
    #ifdef WOLFSSL_SHA512
            WC_HASH_TYPE_SHA512,
    #endif
    #ifdef WOLFSSL_SHA3
            WC_HASH_TYPE_SHA3_224,
    #endif
    };
    enum wc_HashType notSupported[] = {
              WC_HASH_TYPE_MD5_SHA,
              WC_HASH_TYPE_MD2,
              WC_HASH_TYPE_MD4,
              WC_HASH_TYPE_BLAKE2B,
              WC_HASH_TYPE_BLAKE2S,
              WC_HASH_TYPE_NONE,
    };
    int enumlen = (sizeof(enumArray)/sizeof(enum wc_HashType));
    int notSupportedLen;

    /* For loop to test various arguments... */
    for (i = 0; i < enumlen; i++) {
        ret = wc_HashInit(&hash, enumArray[i]);
        if (ret == 0) {
            ret = wc_HashGetFlags(&hash, enumArray[i], &flags);
        }
        if (ret == 0) {
            if (flags & WC_HASH_FLAG_ISCOPY) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_HashGetFlags(NULL, enumArray[i], &flags);
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        wc_HashFree(&hash, enumArray[i]);
        if (ret != 0) {
            break;
        }
    }
    /* For loop to test not supported cases */
    notSupportedLen = (sizeof(notSupported)/sizeof(enum wc_HashType));
    for (j = 0; ret == 0 && j < notSupportedLen; j++) {
        ret = wc_HashInit(&hash, notSupported[j]);
        if (ret == 0) {
            ret = -1;
        }
        else if (ret == BAD_FUNC_ARG) {
            ret = wc_HashGetFlags(&hash, notSupported[j], &flags);
            if (ret == 0) {
                ret = -1;
            }
            else if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
        if (ret == 0) {
            ret = wc_HashFree(&hash, notSupported[j]);
            if (ret == 0) {
                ret = -1;
            }
            if (ret == BAD_FUNC_ARG) {
                ret = 0;
            }
        }
    }

    res = TEST_RES_CHECK(ret == 0);
#endif
    return res;
}  /* END test_wc_HashGetFlags */

/*----------------------------------------------------------------------------*
 | Compatibility Tests
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_lhash(void)
{
    int res = TEST_SKIPPED;
#ifdef OPENSSL_ALL
    const char testStr[] = "Like a true nature's child\n"
                           "We were born\n"
                           "Born to be wild";

#ifdef NO_SHA
    AssertIntEQ(lh_strhash(testStr), 0xf9dc8a43);
#else
    AssertIntEQ(lh_strhash(testStr), 0x5b7541dc);
#endif

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_NAME(void)
{
    int res = TEST_SKIPPED;
    #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
        !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
        && !defined(NO_RSA) && defined(WOLFSSL_CERT_GEN) && \
        (defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_EXT) || \
         defined(OPENSSL_EXTRA))
    X509* x509;
    const unsigned char* c;
    unsigned char buf[4096];
    int bytes;
    XFILE f;
    const X509_NAME* a;
    const X509_NAME* b;
    X509_NAME* d2i_name = NULL;
    int sz;
    unsigned char* tmp;
    char file[] = "./certs/ca-cert.der";
#ifndef OPENSSL_EXTRA_X509_SMALL
    byte empty[] = { /* CN=empty emailAddress= */
        0x30, 0x21, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0C, 0x05, 0x65, 0x6D, 0x70,
        0x74, 0x79, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x00
    };
#endif

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test compile of deprecated function, returns 0 */
    AssertIntEQ(CRYPTO_thread_id(), 0);
#endif

    AssertNotNull(a = X509_NAME_new());
    X509_NAME_free((X509_NAME*)a);

    f = XFOPEN(file, "rb");
    AssertTrue(f != XBADFILE);
    bytes = (int)XFREAD(buf, 1, sizeof(buf), f);
    XFCLOSE(f);

    c = buf;
    AssertNotNull(x509 = wolfSSL_X509_d2i(NULL, c, bytes));

    /* test cmp function */
    AssertNotNull(a = X509_get_issuer_name(x509));
    AssertNotNull(b = X509_get_subject_name(x509));

#ifndef OPENSSL_EXTRA_X509_SMALL
    AssertIntEQ(X509_NAME_cmp(a, b), 0); /* self signed should be 0 */
#endif

    tmp = buf;
    AssertIntGT((sz = i2d_X509_NAME((X509_NAME*)a, &tmp)), 0);
    if (sz > 0 && tmp == buf) {
        fprintf(stderr, "\nERROR - %s line %d failed with:", __FILE__,
            __LINE__);
        fprintf(stderr, " Expected pointer to be incremented\n");
        abort();
    }

#ifndef OPENSSL_EXTRA_X509_SMALL
    tmp = buf;
    AssertNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
#endif

    /* if output parameter is NULL, should still return required size. */
    AssertIntGT((sz = i2d_X509_NAME((X509_NAME*)b, NULL)), 0);
    /* retry but with the function creating a buffer */
    tmp = NULL;
    AssertIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
    XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);

#ifdef WOLFSSL_CERT_NAME_ALL
    /* test for givenName and name */
    {
        WOLFSSL_X509_NAME_ENTRY* entry;
        const byte gName[] = "test-given-name";
        const byte name[] = "test-name";

        entry = wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL, NID_givenName,
            ASN_UTF8STRING, gName, sizeof(gName));
        AssertNotNull(entry);
        wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry, -1, 0);
        wolfSSL_X509_NAME_ENTRY_free(entry);

        entry = wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL, NID_name,
            ASN_UTF8STRING, name, sizeof(name));
        AssertNotNull(entry);
        wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry, -1, 0);
        wolfSSL_X509_NAME_ENTRY_free(entry);

        tmp = NULL;
        AssertIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }
#endif

    AssertNotNull(b = X509_NAME_dup((X509_NAME*)a));
#ifndef OPENSSL_EXTRA_X509_SMALL
    AssertIntEQ(X509_NAME_cmp(a, b), 0);
#endif
    X509_NAME_free((X509_NAME*)b);
    X509_NAME_free(d2i_name);
    X509_free(x509);

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test with an empty domain component */
    tmp = empty;
    sz  = sizeof(empty);
    AssertNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
    AssertIntEQ(X509_NAME_entry_count(d2i_name), 2);

    /* size of empty emailAddress will be 0 */
    tmp = buf;
    AssertIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_emailAddress,
                (char*)tmp, sizeof(buf)), 0);

    /* should contain no organization name */
    tmp = buf;
    AssertIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_organizationName,
                (char*)tmp, sizeof(buf)), -1);
    X509_NAME_free(d2i_name);
#endif

    res = TEST_RES_CHECK(1);
    #endif /* defined(OPENSSL_EXTRA) && !defined(NO_DES3) */
    return res;
}

static int test_wolfSSL_X509_NAME_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA) && !defined(NO_SHA) && !defined(NO_BIO)
    BIO* bio;
    X509* x509 = NULL;

    AssertNotNull(bio = BIO_new(BIO_s_file()));
    AssertIntGT(BIO_read_filename(bio, svrCertFile), 0);
    AssertNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    AssertIntEQ(X509_NAME_hash(X509_get_subject_name(x509)), 0x137DC03F);
    AssertIntEQ(X509_NAME_hash(X509_get_issuer_name(x509)), 0xFDB2DA4);
    X509_free(x509);
    BIO_free(bio);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_NAME_print_ex(void)
{
    int res = TEST_SKIPPED;
#if (defined(OPENSSL_ALL) || (defined(OPENSSL_EXTRA) && \
     (defined(HAVE_STUNNEL) || defined(WOLFSSL_NGINX) || \
     defined(HAVE_LIGHTY) || defined(WOLFSSL_HAPROXY) || \
     defined(WOLFSSL_OPENSSH) || defined(HAVE_SBLIM_SFCB)))) && \
    !defined(NO_BIO) && !defined(NO_RSA)
    int memSz;
    byte* mem = NULL;
    BIO* bio = NULL;
    BIO* membio = NULL;
    X509* x509 = NULL;
    X509_NAME* name = NULL;

    const char* expNormal  = "C=US, CN=wolfssl.com";
    const char* expReverse = "CN=wolfssl.com, C=US";

    const char* expNotEscaped = "C= US,+\"\\ , CN=#wolfssl.com<>;";
    const char* expNotEscapedRev = "CN=#wolfssl.com<>;, C= US,+\"\\ ";
    const char* expRFC5523 =
        "CN=\\#wolfssl.com\\<\\>\\;, C=\\ US\\,\\+\\\"\\\\\\ ";

    /* Test with real cert (svrCertFile) first */
    AssertNotNull(bio = BIO_new(BIO_s_file()));
    AssertIntGT(BIO_read_filename(bio, svrCertFile), 0);
    AssertNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    AssertNotNull(name = X509_get_subject_name(x509));

    /* Test without flags */
    AssertNotNull(membio = BIO_new(BIO_s_mem()));
    AssertIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
    BIO_free(membio);

    /* Test flag: XN_FLAG_RFC2253 */
    AssertNotNull(membio = BIO_new(BIO_s_mem()));
    AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
    BIO_free(membio);

    /* Test flag: XN_FLAG_RFC2253 | XN_FLAG_DN_REV */
    AssertNotNull(membio = BIO_new(BIO_s_mem()));
    AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253 | XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
    BIO_free(membio);

    X509_free(x509);
    BIO_free(bio);

    /* Test normal case without escaped characters */
    {
        /* Create name: "/C=US/CN=wolfssl.com" */
        AssertNotNull(name = X509_NAME_new());
        AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    MBSTRING_UTF8, (byte*)"US", 2, -1, 0),
                    WOLFSSL_SUCCESS);
        AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    MBSTRING_UTF8, (byte*)"wolfssl.com", 11, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expNormal));
        AssertIntEQ(XSTRNCMP((char*)mem, expNormal, XSTRLEN(expNormal)), 0);
        BIO_free(membio);

        /* Test flags: XN_FLAG_RFC2253 - should be reversed */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expReverse));
        BIO_free(membio);

        /* Test flags: XN_FLAG_DN_REV - reversed */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expReverse));
        AssertIntEQ(XSTRNCMP((char*)mem, expReverse, XSTRLEN(expReverse)), 0);
        BIO_free(membio);

        X509_NAME_free(name);
    }

    /* Test RFC2253 characters are escaped with backslashes */
    {
        AssertNotNull(name = X509_NAME_new());
        AssertIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    /* space at beginning and end, and: ,+"\ */
                    MBSTRING_UTF8, (byte*)" US,+\"\\ ", 8, -1, 0),
                    WOLFSSL_SUCCESS);
        AssertIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    /* # at beginning, and: <>;*/
                    MBSTRING_UTF8, (byte*)"#wolfssl.com<>;", 15, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expNotEscaped));
        AssertIntEQ(XSTRNCMP((char*)mem, expNotEscaped,
                    XSTRLEN(expNotEscaped)), 0);
        BIO_free(membio);

        /* Test flags: XN_FLAG_RFC5523 - should be reversed and escaped */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expRFC5523));
        AssertIntEQ(XSTRNCMP((char*)mem, expRFC5523, XSTRLEN(expRFC5523)), 0);
        BIO_free(membio);

        /* Test flags: XN_FLAG_DN_REV - reversed but not escaped */
        AssertNotNull(membio = BIO_new(BIO_s_mem()));
        AssertIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        AssertIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        AssertIntEQ(memSz, XSTRLEN(expNotEscapedRev));
        AssertIntEQ(XSTRNCMP((char*)mem, expNotEscapedRev,
                    XSTRLEN(expNotEscapedRev)), 0);
        BIO_free(membio);

        X509_NAME_free(name);
    }

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO_multiple_info(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack;
    X509_INFO *info;
    int len;
    int i;
    const char* files[] = {
        cliCertFile,
        cliKeyFile,
        /* This needs to be the order as svrCertFile contains the
         * intermediate cert as well. */
        svrKeyFile,
        svrCertFile,
        NULL,
    };
    const char** curFile;
    BIO *fileBIO;
    BIO *concatBIO = NULL;
    byte tmp[FOURK_BUF];

    /* concatenate the cert and the key file to force PEM_X509_INFO_read_bio
     * to group objects together. */
    AssertNotNull(concatBIO = BIO_new(BIO_s_mem()));
    for (curFile = files; *curFile != NULL; curFile++) {
        int fileLen;
        AssertNotNull(fileBIO = BIO_new_file(*curFile, "rb"));
        fileLen = wolfSSL_BIO_get_len(fileBIO);
        while ((len = BIO_read(fileBIO, tmp, sizeof(tmp))) > 0) {
            AssertIntEQ(BIO_write(concatBIO, tmp, len), len);
            fileLen -= len;
        }
        /* Make sure we read the entire file */
        AssertIntEQ(fileLen, 0);
        BIO_free(fileBIO);
    }

    AssertNotNull(info_stack = PEM_X509_INFO_read_bio(concatBIO, NULL, NULL,
                                                      NULL));
    AssertIntEQ(sk_X509_INFO_num(info_stack), 3);
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        AssertNotNull(info = sk_X509_INFO_value(info_stack, i));
        AssertNotNull(info->x509);
        AssertNull(info->crl);
        if (i != 0) {
            AssertNotNull(info->x_pkey);
            AssertIntEQ(X509_check_private_key(info->x509,
                                               info->x_pkey->dec_pkey), 1);
        }
        else {
            AssertNull(info->x_pkey);
        }
    }

    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(concatBIO);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}
#endif

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack;
    X509_INFO *info;
    BIO *cert;
    int i;
    /* PEM in hex format to avoid null terminator */
    byte data[] = {
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
        0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x63, 0x2d, 0x2d, 0x2d, 0x2d,
        0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x44, 0x4d, 0x54, 0x42, 0x75, 0x51, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x2d, 0x2d,
        0x2d, 0x2d, 0x2d
    };
    /* PEM in hex format to avoid null terminator */
    byte data2[] = {
        0x41, 0x53, 0x4e, 0x31, 0x20, 0x4f, 0x49, 0x44, 0x3a, 0x20, 0x70, 0x72,
        0x69, 0x6d, 0x65, 0x32, 0x35, 0x36, 0x76, 0x31, 0x0a, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x45, 0x43, 0x20, 0x50,
        0x41, 0x52, 0x41, 0x4d, 0x45, 0x54, 0x45, 0x52, 0x53, 0x2d, 0x2d, 0x2d,
        0x2d, 0x43, 0x65, 0x72, 0x74, 0x69, 0x2d, 0x0a, 0x42, 0x67, 0x67, 0x71,
        0x68, 0x6b, 0x6a, 0x4f, 0x50, 0x51, 0x4d, 0x42, 0x42, 0x77, 0x3d, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d
    };

    AssertNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    AssertNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        AssertNotNull(info = sk_X509_INFO_value(info_stack, i));
        AssertNotNull(info->x509);
        AssertNull(info->crl);
        AssertNull(info->x_pkey);
    }
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);

    AssertNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    AssertNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);

    /* This case should fail due to invalid input. */
    AssertNotNull(cert = BIO_new(BIO_s_mem()));
    AssertIntEQ(BIO_write(cert, data, sizeof(data)), sizeof(data));
    AssertNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);
    AssertNotNull(cert = BIO_new(BIO_s_mem()));
    AssertIntEQ(BIO_write(cert, data2, sizeof(data2)), sizeof(data2));
    AssertNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}
#endif

static int test_wolfSSL_X509_subject_name_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509;
    X509_NAME* subjectName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    AssertNotNull(subjectName = wolfSSL_X509_get_subject_name(x509));

    /* These two
     *   - X509_subject_name_hash(x509)
     *   - X509_NAME_hash(X509_get_subject_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_subject_name_hash(x509);
    AssertIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_subject_name(x509));
    AssertIntNE(ret2, 0);

    AssertIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_issuer_name_hash(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509;
    X509_NAME* issuertName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    AssertNotNull(issuertName = wolfSSL_X509_get_issuer_name(x509));

    /* These two
     *   - X509_issuer_name_hash(x509)
     *   - X509_NAME_hash(X509_get_issuer_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_issuer_name_hash(x509);
    AssertIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_issuer_name(x509));
    AssertIntNE(ret2, 0);

    AssertIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_check_host(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_SHA) && !defined(NO_RSA)

    X509* x509;
    const char altName[] = "example.com";

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));

    AssertIntEQ(X509_check_host(x509, altName, XSTRLEN(altName), 0, NULL),
            WOLFSSL_SUCCESS);

    AssertIntEQ(X509_check_host(x509, NULL, 0, 0, NULL),
            WOLFSSL_FAILURE);

    X509_free(x509);

    AssertIntEQ(X509_check_host(NULL, altName, XSTRLEN(altName), 0, NULL),
            WOLFSSL_FAILURE);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_X509_check_email(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA)
    X509* x509;
    const char goodEmail[] = "info@wolfssl.com";
    const char badEmail[] = "disinfo@wolfssl.com";

    AssertNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));

    /* Should fail on non-matching email address */
    AssertIntEQ(wolfSSL_X509_check_email(x509, badEmail, XSTRLEN(badEmail), 0),
            WOLFSSL_FAILURE);
    /* Should succeed on matching email address */
    AssertIntEQ(wolfSSL_X509_check_email(x509, goodEmail, XSTRLEN(goodEmail), 0),
            WOLFSSL_SUCCESS);
    /* Should compute length internally when not provided */
    AssertIntEQ(wolfSSL_X509_check_email(x509, goodEmail, 0, 0),
            WOLFSSL_SUCCESS);
    /* Should fail when email address is NULL */
    AssertIntEQ(wolfSSL_X509_check_email(x509, NULL, 0, 0),
            WOLFSSL_FAILURE);

    X509_free(x509);

    /* Should fail when x509 is NULL */
    AssertIntEQ(wolfSSL_X509_check_email(NULL, goodEmail, 0, 0),
            WOLFSSL_FAILURE);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && WOLFSSL_CERT_GEN */
    return res;
}

static int test_wolfSSL_DES(void)
{
    int res = TEST_SKIPPED;
    #if defined(OPENSSL_EXTRA) && !defined(NO_DES3)
    const_DES_cblock myDes;
    DES_cblock iv;
    DES_key_schedule key;
    word32 i;
    DES_LONG dl;
    unsigned char msg[] = "hello wolfssl";

    DES_check_key(1);
    DES_set_key(&myDes, &key);

    /* check, check of odd parity */
    XMEMSET(myDes, 4, sizeof(const_DES_cblock));  myDes[0] = 6; /*set even parity*/
    XMEMSET(key, 5, sizeof(DES_key_schedule));
    AssertIntEQ(DES_set_key_checked(&myDes, &key), -1);
    AssertIntNE(key[0], myDes[0]); /* should not have copied over key */

    /* set odd parity for success case */
    DES_set_odd_parity(&myDes);
    AssertIntEQ(DES_check_key_parity(&myDes), 1);
    fprintf(stderr, "%02x %02x %02x %02x", myDes[0], myDes[1], myDes[2],
        myDes[3]);
    AssertIntEQ(DES_set_key_checked(&myDes, &key), 0);
    for (i = 0; i < sizeof(DES_key_schedule); i++) {
        AssertIntEQ(key[i], myDes[i]);
    }
    AssertIntEQ(DES_is_weak_key(&myDes), 0);

    /* check weak key */
    XMEMSET(myDes, 1, sizeof(const_DES_cblock));
    XMEMSET(key, 5, sizeof(DES_key_schedule));
    AssertIntEQ(DES_set_key_checked(&myDes, &key), -2);
    AssertIntNE(key[0], myDes[0]); /* should not have copied over key */

    /* now do unchecked copy of a weak key over */
    DES_set_key_unchecked(&myDes, &key);
    /* compare arrays, should be the same */
    for (i = 0; i < sizeof(DES_key_schedule); i++) {
        AssertIntEQ(key[i], myDes[i]);
    }
    AssertIntEQ(DES_is_weak_key(&myDes), 1);

    /* check DES_key_sched API */
    XMEMSET(key, 1, sizeof(DES_key_schedule));
    AssertIntEQ(DES_key_sched(&myDes, NULL), 0);
    AssertIntEQ(DES_key_sched(NULL, &key),   0);
    AssertIntEQ(DES_key_sched(&myDes, &key), 0);
    /* compare arrays, should be the same */
    for (i = 0; i < sizeof(DES_key_schedule); i++) {
        AssertIntEQ(key[i], myDes[i]);
    }

    /* DES_cbc_cksum should return the last 4 of the last 8 bytes after
     * DES_cbc_encrypt on the input */
    XMEMSET(iv, 0, sizeof(DES_cblock));
    XMEMSET(myDes, 5, sizeof(DES_key_schedule));
    AssertIntGT((dl = DES_cbc_cksum(msg, &key, sizeof(msg), &myDes, &iv)), 0);
    AssertIntEQ(dl, 480052723);

    res = TEST_RES_CHECK(1);
    #endif /* defined(OPENSSL_EXTRA) && !defined(NO_DES3) */
    return res;
}

static int test_wc_PemToDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    int ret;
    DerBuffer* pDer = NULL;
    const char* ca_cert = "./certs/server-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    int eccKey = 0;
    EncryptedInfo info;

    XMEMSET(&info, 0, sizeof(info));

    ret = load_file(ca_cert, &cert_buf, &cert_sz);
    if (ret == 0) {
        ret = wc_PemToDer(cert_buf, cert_sz, CERT_TYPE,
            &pDer, NULL, &info, &eccKey);
        AssertIntEQ(ret, 0);

        wc_FreeDer(&pDer);
    }

    if (cert_buf)
        free(cert_buf);

#ifdef HAVE_ECC
    {
        const char* ecc_private_key = "./certs/ecc-privOnlyKey.pem";
        byte key_buf[256] = {0};

        /* Test fail of loading a key with cert type */
        AssertIntEQ(load_file(ecc_private_key, &cert_buf, &cert_sz), 0);
        key_buf[0] = '\n';
        XMEMCPY(key_buf + 1, cert_buf, cert_sz);
        AssertIntNE((ret = wc_PemToDer(key_buf, cert_sz + 1, CERT_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);

    #ifdef OPENSSL_EXTRA
        AssertIntEQ((ret = wc_PemToDer(key_buf, cert_sz + 1, PRIVATEKEY_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);
    #endif
        wc_FreeDer(&pDer);
        if (cert_buf)
            free(cert_buf);
    }
#endif
    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_AllocDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS)
    int ret;
    DerBuffer* pDer = NULL;
    word32 testSize = 1024;

    ret = wc_AllocDer(&pDer, testSize, CERT_TYPE, HEAP_HINT);
    AssertIntEQ(ret, 0);
    AssertNotNull(pDer);
    wc_FreeDer(&pDer);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_CertPemToDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    int ret;
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0, cert_dersz = 0;
    byte* cert_der = NULL;

    ret = load_file(ca_cert, &cert_buf, &cert_sz);
    if (ret == 0) {
        cert_dersz = cert_sz; /* DER will be smaller than PEM */
        cert_der = (byte*)malloc(cert_dersz);
        if (cert_der) {
            ret = wc_CertPemToDer(cert_buf, (int)cert_sz,
                cert_der, (int)cert_dersz, CERT_TYPE);
            AssertIntGE(ret, 0);
        }
    }

    if (cert_der)
        free(cert_der);
    if (cert_buf)
        free(cert_buf);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_KeyPemToDer(void)
{
    int res = TEST_SKIPPED;

#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && !defined(NO_RSA)

    int ret;
    const byte cert_buf[] = \
    "-----BEGIN PRIVATE KEY-----\n"
    "MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDMG5KgWxP002pA\n"
    "QJIdA4H5N0oM1Wf0LrHcos5RYUlrHDkC2b5p2BUpVRPmgDAFD2+8leim98x0BvcB\n"
    "k48TNzrVynuwyVEY664+iQyzEBO5v27HPRydOddprbLCvRO036XINGIjauy1jHFi\n"
    "HaDVx3bexSwgp9aefUGAszFXi4q1J4GacV7Cr2b/wBqUHqWv4ZXPu6R9/UYngTkD\n"
    "UDJL5gLlLfcLzNyyodKPHPCIAKdWn6mSVdcHk8XVpK4y9lgz4E7YDWA6ohKZgWgG\n"
    "2RDha8CMilFMDgYa0G0SiS9g3PQx0qh3AMXJJsKSVhScFCZufAE0kV6KvjP7jAqP\n"
    "XBiSkRGPAgMBAAECggEAW7hmRyY2jRX2UMJThrM9VIs6fRLnYI0dQ0tsEJj536ay\n"
    "nevQjArc05KWW0Yujg+WRDZPcry3RUqd9Djlmhp/F3Si6dpF1b+PMS3wJYVrf9Sd\n"
    "SO5W7faArU4vnyBNe0HnY1Ta5xSVI65lg1RSIs88RTZwsooJwXYDGf0shq0/21CE\n"
    "V8HOb27DDYNcEnm35lzaONjFnMqQQT2Vs9anRrPiSEXNleEvTgLVXZtGTyCGTz6v\n"
    "x86Y8eSWL9YNHvPE1I+mDPuocfSR7eRNgRu7SK3mn94W5mqd7Ns072YKX/2XN1mO\n"
    "66+ZFHO6v4dK1u7cSjuwrU1EhLHpUsgDz6Bna5InyQKBgQDv5l8RPy8UneKSADaf\n"
    "M5L/5675I/5t4nqVjvbnQje00YveLTAEjlJBNR93Biln3sYgnvNamYDCxyEuUZ/I\n"
    "S/vmBL9PoxfGZow4FcsIBOEbIn3E0SYJgCBNWthquUvGpKsYDnThJuhO+1cVmxAJ\n"
    "BUOjLFnJYHM0a+Vmk9GexT2OBwKBgQDZzkUBOK7Im3eiYytFocUJyhqMH30d49X9\n"
    "ujC7kGw4UWAqVe7YCSvlBa8nzWpRWK2kRpu3M0272RU0V4geyWqT+nr/SvRRPtNP\n"
    "F5dY8l3yR7hjtSejqqjOfBcZT6ETJxI4tiG0+Nl5BlfM5M+0nxnkWpRcHuOR3j79\n"
    "YUFERyN+OQKBgQCjlOKeUAc6d65W/+4/AFvsQ378Q57qLtSHxsR1TKHPmlNVXFqx\n"
    "wJo1/JNIBduWCEHxXHF0BdfW+RGXE/FwEt/hKLuLAhrkHmjelX2sKieU6R/5ZOQa\n"
    "9lMQbDHGFDOncAF6leD85hriQGBRSzrT69MDIOrYdfwYcroqCAGX0cb3YQKBgQC8\n"
    "iIFQylj5SyHmjcMSNjKSA8CxFDzAV8yPIdE3Oo+CvGXqn5HsrRuy1hXE9VmXapR8\n"
    "A6ackSszdHiXY0FvrNe1mfdH7wDHJwPQjdIzazCJHS3uGQxj7sDKY7226ie6pXJv\n"
    "ZrCMr2/IBAaSVGm6ppHKCeIsT4ybYm7R85KEYLPHeQKBgBeJOMBinXQfWN/1jT9b\n"
    "6Ywrutvp2zP8hVxQGSZJ0WG4iewZyFLsPUlbWRXOSYNPElHmdD0ZomdLVm+lSpAA\n"
    "XSH5FJ/IFCwqq7Eft6Gf8NFRV+NjPMUny+PnjHe4oFP8YK/Ek22K3ttNG8Hw69Aw\n"
    "AQue5o6oVfhgLiJzMdo/77gw\n"
    "-----END PRIVATE KEY-----\n";
    const int cert_sz = sizeof(cert_buf);
    const char cert_pw[] = "password";
    int cert_dersz = 0;
    byte* cert_der = NULL;

    /* Bad arg: Cert buffer is NULL */
    ret = wc_KeyPemToDer(NULL, cert_sz, cert_der, cert_dersz, "");
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* Bad arg: Cert DER buffer non-NULL but size zero (or less) */
    ret = wc_KeyPemToDer(cert_buf, cert_sz, (byte*)&cert_der, 0, "");
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* Test normal operation */
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    cert_der = (byte*)malloc(cert_dersz);
    AssertNotNull(cert_der);
    if (cert_der) {
        ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz, cert_pw);
        AssertIntGE(ret, 0);
        AssertIntLE(ret, cert_sz);
        free(cert_der);
        cert_der = NULL;
        ret = 0;
    }

    if (ret == 0) {
        /* Test NULL for DER buffer to return needed DER buffer size */
        ret = wc_KeyPemToDer(cert_buf, cert_sz, NULL, 0, "");
        AssertIntGT(ret, 0);
        AssertIntLE(ret, cert_sz);
        cert_dersz = ret;
        cert_der = (byte*)malloc(cert_dersz);
        AssertNotNull(cert_der);
        if (cert_der) {
            ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz, cert_pw);
            AssertIntGE(ret, 0);
            AssertIntLE(ret, cert_sz);
            free(cert_der);
            cert_der = NULL;
        }
    }

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_PubKeyPemToDer(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && \
   (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    int ret;
    const char* key = "./certs/ecc-client-keyPub.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0, cert_dersz = 0;
    byte* cert_der = NULL;

    ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz,
        cert_der, (int)cert_dersz);
    AssertIntGE(ret, BAD_FUNC_ARG);

    ret = load_file(key, &cert_buf, &cert_sz);
    if (ret == 0) {
        cert_dersz = cert_sz; /* DER will be smaller than PEM */
        cert_der = (byte*)malloc(cert_dersz);
        AssertNotNull(cert_der);
        if (cert_der) {
            ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
                (int)cert_dersz);
            AssertIntGE(ret, 0);
            free(cert_der);
            cert_der = NULL;
            ret = 0;
        }
    }

    if (ret == 0) {
        /* Test NULL for DER buffer to return needed DER buffer size */
        ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz, NULL, 0);
        AssertIntGT(ret, 0);
        AssertIntLE(ret, cert_sz);
        cert_dersz = ret;
        cert_der = (byte*)malloc(cert_dersz);
        AssertNotNull(cert_der);
        if (cert_der) {
            ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
                (int)cert_dersz);
            AssertIntGE(ret, 0);
            free(cert_der);
            cert_der = NULL;
        }
    }

    if (cert_buf) {
        free(cert_buf);
    }

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_PemPubKeyToDer(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    int ret;
    const char* key = "./certs/ecc-client-keyPub.pem";
    size_t cert_dersz = 1024;
    byte* cert_der = (byte*)malloc(cert_dersz);

    ret = wc_PemPubKeyToDer(NULL, cert_der, (int)cert_dersz);
    AssertIntGE(ret, BAD_FUNC_ARG);

    if (cert_der) {
        ret = wc_PemPubKeyToDer(key, cert_der, (int)cert_dersz);
        AssertIntGE(ret, 0);

        free(cert_der);
    }

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wc_GetPubKeyDerFromCert(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_RSA) || defined(HAVE_ECC)
    int ret;
    word32 idx = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    byte certBuf[6000]; /* for PEM and CSR, client-cert.pem is 5-6kB */
    word32 certBufSz = sizeof(certBuf);
#endif
#if ((!defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)) || \
     defined(WOLFSSL_CERT_REQ)) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE fp;
#endif
#ifndef NO_RSA
    RsaKey rsaKey;
    #if defined(USE_CERT_BUFFERS_2048)
        byte* rsaCertDer = (byte*)client_cert_der_2048;
        word32 rsaCertDerSz = sizeof_client_cert_der_2048;
    #elif defined(USE_CERT_BUFFERS_1024)
        byte* rsaCertDer = (byte*)client_cert_der_1024;
        word32 rsaCertDerSz = sizeof_client_cert_der_1024;
    #else
        unsigned char rsaCertDer[TWOK_BUF];
        word32 rsaCertDerSz;
    #endif
#endif
#ifdef HAVE_ECC
    ecc_key eccKey;
    #if defined(USE_CERT_BUFFERS_256)
        byte* eccCert = (byte*)cliecc_cert_der_256;
        word32 eccCertSz = sizeof_cliecc_cert_der_256;
    #else
        unsigned char eccCert[ONEK_BUF];
        word32 eccCertSz;
        XFILE fp2;
    #endif
#endif

#ifndef NO_RSA

#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    fp = XFOPEN("./certs/1024/client-cert.der", "rb");
    AssertTrue((fp != XBADFILE));
    rsaCertDerSz = (word32)XFREAD(rsaCertDer, 1, sizeof(rsaCertDer), fp);
    XFCLOSE(fp);
#endif

    /* good test case - RSA DER cert */
    wc_InitDecodedCert(&decoded, rsaCertDer, rsaCertDerSz, NULL);
    ret = wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL);
    AssertIntEQ(ret, 0);

    ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz);
    AssertIntEQ(ret, 0);
    AssertIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
    AssertIntEQ(ret, 0);
    ret = wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz);
    AssertIntEQ(ret, 0);
    wc_FreeRsaKey(&rsaKey);

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ret = wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz);
    AssertIntEQ(ret, LENGTH_ONLY_E);
    AssertIntGT(keyDerSz, 0);

    /* bad args: DecodedCert NULL */
    ret = wc_GetPubKeyDerFromCert(NULL, keyDer, &keyDerSz);
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* bad args: output key buff size */
    ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, NULL);
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* bad args: zero size output key buffer */
    keyDerSz = 0;
    ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz);
    AssertIntEQ(ret, BAD_FUNC_ARG);

    wc_FreeDecodedCert(&decoded);

    /* Certificate Request Tests */
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    {
        XMEMSET(certBuf, 0, sizeof(certBuf));
        fp = XFOPEN("./certs/csr.signed.der", "rb");
        AssertTrue((fp != XBADFILE));
        certBufSz = (word32)XFREAD(certBuf, 1, certBufSz, fp);
        XFCLOSE(fp);

        wc_InitDecodedCert(&decoded, certBuf, certBufSz, NULL);
        ret = wc_ParseCert(&decoded, CERTREQ_TYPE, VERIFY, NULL);
        AssertIntEQ(ret, 0);

        /* good test case - RSA DER certificate request */
        keyDerSz = sizeof(keyDer);
        ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz);
        AssertIntEQ(ret, 0);
        AssertIntGT(keyDerSz, 0);

        /* sanity check, verify we can import DER public key */
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        AssertIntEQ(ret, 0);
        idx = 0;
        ret = wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz);
        AssertIntEQ(ret, 0);
        wc_FreeRsaKey(&rsaKey);

        wc_FreeDecodedCert(&decoded);
    }
    #endif /* WOLFSSL_CERT_REQ */
#endif /* NO_RSA */

#ifdef HAVE_ECC
    #ifndef USE_CERT_BUFFERS_256
        fp2 = XFOPEN("./certs/client-ecc-cert.der", "rb");
        AssertTrue((fp2 != XBADFILE));
        eccCertSz = (word32)XFREAD(eccCert, 1, ONEK_BUF, fp2);
        XFCLOSE(fp2);
    #endif

    wc_InitDecodedCert(&decoded, eccCert, eccCertSz, NULL);
    ret = wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL);
    AssertIntEQ(ret, 0);

    /* good test case - ECC */
    XMEMSET(keyDer, 0, sizeof(keyDer));
    keyDerSz = sizeof(keyDer);
    ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz);
    AssertIntEQ(ret, 0);
    AssertIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_ecc_init(&eccKey);
    AssertIntEQ(ret, 0);
    idx = 0; /* reset idx to 0, used above in RSA case */
    ret = wc_EccPublicKeyDecode(keyDer, &idx, &eccKey, keyDerSz);
    AssertIntEQ(ret, 0);
    wc_ecc_free(&eccKey);

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ret = wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz);
    AssertIntEQ(ret, LENGTH_ONLY_E);
    AssertIntGT(keyDerSz, 0);

    wc_FreeDecodedCert(&decoded);
#endif

    res = TEST_RES_CHECK(1);
#endif /* !NO_RSA || HAVE_ECC */
    return res;
}

static int test_wc_CheckCertSigPubKey(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_PEM_TO_DER) && defined(HAVE_ECC)
    int ret;
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    byte* cert_der = NULL;
    word32 cert_dersz = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;

    ret = load_file(ca_cert, &cert_buf, &cert_sz);
    if (ret == 0) {
        cert_dersz = (word32)cert_sz; /* DER will be smaller than PEM */
        cert_der = (byte*)malloc(cert_dersz);
        if (cert_der) {
            ret = wc_CertPemToDer(cert_buf, (int)cert_sz,
                cert_der, (int)cert_dersz, CERT_TYPE);
            AssertIntGE(ret, 0);
        }
    }

    wc_InitDecodedCert(&decoded, cert_der, cert_dersz, NULL);
    ret = wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL);
    AssertIntEQ(ret, 0);

    ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz);
    AssertIntEQ(ret, 0);
    AssertIntGT(keyDerSz, 0);

    /* Good test case. */
    ret = wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer, keyDerSz,
                                RSAk);
    AssertIntEQ(ret, 0);

    /* No certificate. */
    ret = wc_CheckCertSigPubKey(NULL, cert_dersz, NULL, keyDer, keyDerSz,
                                ECDSAk);
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* Bad cert size. */
    ret = wc_CheckCertSigPubKey(cert_der, 0, NULL, keyDer, keyDerSz,
                                RSAk);
    AssertTrue(ret == ASN_PARSE_E || ret == BUFFER_E);

    /* No public key. */
    ret = wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, NULL, keyDerSz,
                                RSAk);
    AssertIntEQ(ret, ASN_NO_SIGNER_E);

    /* Bad public key size. */
    ret = wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer, 0,
                                RSAk);
    AssertIntEQ(ret, BAD_FUNC_ARG);

    /* Wrong aglo. */
    ret = wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer, keyDerSz,
                                ECDSAk);
    AssertIntEQ(ret, ASN_PARSE_E);

    wc_FreeDecodedCert(&decoded);
    if (cert_der)
        free(cert_der);
    if (cert_buf)
        free(cert_buf);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_certs(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA)
    X509*  x509ext;
#ifdef OPENSSL_ALL
    X509*  x509;
    WOLFSSL_X509_EXTENSION* ext;
    ASN1_OBJECT* obj;
#endif
    WOLFSSL*     ssl;
    WOLFSSL_CTX* ctx;
    STACK_OF(ASN1_OBJECT)* sk;
    ASN1_STRING* asn1_str;
    AUTHORITY_KEYID* akey;
    BASIC_CONSTRAINTS* bc;
    int crit;

#ifndef NO_WOLFSSL_SERVER
    AssertNotNull(ctx = SSL_CTX_new(SSLv23_server_method()));
#else
    AssertNotNull(ctx = SSL_CTX_new(SSLv23_client_method()));
#endif
    AssertTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile, SSL_FILETYPE_PEM));
    AssertTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    AssertTrue(SSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(SSL_CTX_check_private_key(ctx), SSL_FAILURE);
    #endif
    AssertTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(SSL_CTX_check_private_key(ctx), SSL_SUCCESS);
    #endif
    AssertNotNull(ssl = SSL_new(ctx));

    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_PK_CALLBACKS
    AssertIntEQ((int)SSL_set_tlsext_debug_arg(ssl, NULL), WOLFSSL_SUCCESS);
    #endif /* HAVE_PK_CALLBACKS */

    /* create and use x509 */
#ifdef OPENSSL_ALL
    x509 = wolfSSL_X509_load_certificate_file(cliCertFile, WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509);
#endif
    x509ext = wolfSSL_X509_load_certificate_file(cliCertFileExt, WOLFSSL_FILETYPE_PEM);
    AssertNotNull(x509ext);
    AssertIntEQ(SSL_use_certificate(ssl, x509ext), WOLFSSL_SUCCESS);

    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    /* with loading in a new cert the check on private key should now fail */
    AssertIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif


    #if defined(USE_CERT_BUFFERS_2048)
        AssertIntEQ(SSL_use_certificate_ASN1(ssl,
                                  (unsigned char*)server_cert_der_2048,
                                  sizeof_server_cert_der_2048), WOLFSSL_SUCCESS);
    #endif

    #if !defined(NO_SHA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    /************* Get Digest of Certificate ******************/
    {
        byte   digest[64]; /* max digest size */
        word32 digestSz;

        XMEMSET(digest, 0, sizeof(digest));
        AssertIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_SUCCESS);
        AssertIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha256(), digest, &digestSz),
                    WOLFSSL_SUCCESS);

        AssertIntEQ(X509_digest(NULL, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_FAILURE);
    }
    #endif /* !NO_SHA && !NO_SHA256 && !NO_PWDBASED */

    /* test and checkout X509 extensions */
    bc = (BASIC_CONSTRAINTS*)X509_get_ext_d2i(x509ext, NID_basic_constraints,
            &crit, NULL);
    AssertNotNull(bc);
    AssertIntEQ(crit, 0);

#ifdef OPENSSL_ALL
    ext = X509V3_EXT_i2d(NID_basic_constraints, crit, bc);
    AssertNotNull(ext);
    X509_EXTENSION_free(ext);

    AssertNotNull(ext = X509_EXTENSION_new());
    X509_EXTENSION_set_critical(ext, 1);
    AssertNotNull(obj = OBJ_nid2obj(NID_basic_constraints));
    AssertIntEQ(X509_EXTENSION_set_object(ext, obj), SSL_SUCCESS);
    ASN1_OBJECT_free(obj);
    X509_EXTENSION_free(ext);

    AssertNotNull(ext = X509_EXTENSION_new());
    X509_EXTENSION_set_critical(ext, 0);
    AssertIntEQ(X509_EXTENSION_set_data(ext, NULL), SSL_FAILURE);
    asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext, NID_key_usage, &crit,
            NULL);
    AssertIntEQ(X509_EXTENSION_set_data(ext, asn1_str), SSL_SUCCESS);
    ASN1_STRING_free(asn1_str); /* X509_EXTENSION_set_data has made a copy
                                 * and X509_get_ext_d2i has created new */
    X509_EXTENSION_free(ext);

#endif
    BASIC_CONSTRAINTS_free(bc);

    asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext, NID_key_usage, &crit, NULL);
    AssertNotNull(asn1_str);
    AssertIntEQ(crit, 1);
    AssertIntEQ(asn1_str->type, NID_key_usage);
#ifdef OPENSSL_ALL
    ext = X509V3_EXT_i2d(NID_key_usage, crit, asn1_str);
    AssertNotNull(ext);
    X509_EXTENSION_free(ext);
#endif
    ASN1_STRING_free(asn1_str);

#ifdef OPENSSL_ALL
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509, NID_ext_key_usage,
            &crit, NULL);
    AssertNotNull(sk);
    ext = X509V3_EXT_i2d(NID_ext_key_usage, crit, sk);
    AssertNotNull(ext);
    X509_EXTENSION_free(ext);
    sk_ASN1_OBJECT_pop_free(sk, NULL);
#else
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_ext_key_usage,
            &crit, NULL);
    AssertNull(sk);
#endif

    akey = (AUTHORITY_KEYID*)X509_get_ext_d2i(x509ext,
            NID_authority_key_identifier, &crit, NULL);
    AssertNotNull(akey);
#ifdef OPENSSL_ALL
    ext = X509V3_EXT_i2d(NID_authority_key_identifier, crit, akey);
    AssertNotNull(ext);
    X509_EXTENSION_free(ext);
#endif
    wolfSSL_AUTHORITY_KEYID_free(akey);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
            NID_private_key_usage_period, &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(GENERAL_NAME)*)X509_get_ext_d2i(x509ext, NID_subject_alt_name,
            &crit, NULL);
    {
        int i;
        for (i = 0; i < sk_GENERAL_NAME_num(sk); i++) {
            GENERAL_NAME* gen = sk_GENERAL_NAME_value(sk, i);
            AssertIntEQ(gen->type, GEN_DNS);
            AssertIntEQ(gen->d.dNSName->type, V_ASN1_IA5STRING);
        }
    }
    /* AssertNotNull(sk); no alt names set */
    sk_GENERAL_NAME_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_issuer_alt_name,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_info_access, &crit,
            NULL);
    /* AssertNotNull(sk); no auth info set */
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_sinfo_access,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_name_constraints,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
            NID_certificate_policies, &crit, NULL);
    #if !defined(WOLFSSL_SEP) && !defined(WOLFSSL_CERT_EXT)
        AssertNull(sk);
    #else
        /* AssertNotNull(sk); no cert policy set */
    #endif
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_policy_mappings,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_policy_constraints,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_inhibit_any_policy,
            &crit, NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_tlsfeature, &crit,
            NULL);
    /* AssertNotNull(sk); NID not yet supported */
    AssertIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);

    /* test invalid cases */
    crit = 0;
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, -1, &crit, NULL);
    AssertNull(sk);
    AssertIntEQ(crit, -1);
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(NULL, NID_tlsfeature,
            NULL, NULL);
    AssertNull(sk);

    AssertIntEQ(SSL_get_hit(ssl), 0);
#ifdef OPENSSL_ALL
    X509_free(x509);
#endif
    X509_free(x509ext);
    SSL_free(ssl);
    SSL_CTX_free(ctx);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && !NO_CERTS */
    return res;
}

static int test_wolfSSL_X509_check_private_key(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        defined(USE_CERT_BUFFERS_2048) && !defined(NO_CHECK_PRIVATE_KEY)
    X509*  x509;
    EVP_PKEY* pkey = NULL;
    const byte* key;

    /* Check with correct key */
    AssertNotNull((x509 = X509_load_certificate_file(cliCertFile,
                                                     SSL_FILETYPE_PEM)));
    key = client_key_der_2048;
    AssertNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &key, (long)sizeof_client_key_der_2048));
    AssertIntEQ(X509_check_private_key(x509, pkey), 1);
    EVP_PKEY_free(pkey);
    pkey = NULL;

    /* Check with wrong key */
    key = server_key_der_2048;
    AssertNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &key, (long)sizeof_server_key_der_2048));
    AssertIntEQ(X509_check_private_key(x509, pkey), 0);

    /* test for incorrect parameter */
    AssertIntEQ(X509_check_private_key(NULL, pkey), 0);
    AssertIntEQ(X509_check_private_key(x509, NULL), 0);
    AssertIntEQ(X509_check_private_key(NULL, NULL), 0);

    EVP_PKEY_free(pkey);
    X509_free(x509);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}


static int test_wolfSSL_ASN1_TIME_print(void)
{
    int res = TEST_SKIPPED;
    #if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) \
        && (defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
            defined(WOLFSSL_HAPROXY)) && defined(USE_CERT_BUFFERS_2048) && \
        !defined(NO_BIO)
    BIO*  bio;
    X509*  x509;
    const unsigned char* der = client_cert_der_2048;
    ASN1_TIME* t;
    unsigned char buf[25];

    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    AssertNotNull(x509 = wolfSSL_X509_load_certificate_buffer(der,
                sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    AssertIntEQ(ASN1_TIME_print(bio, X509_get_notBefore(x509)), 1);
    AssertIntEQ(BIO_read(bio, buf, sizeof(buf)), 24);
    AssertIntEQ(XMEMCMP(buf, "Dec 16 21:17:49 2022 GMT", sizeof(buf) - 1), 0);

    /* create a bad time and test results */
    AssertNotNull(t = X509_get_notAfter(x509));
    AssertIntEQ(ASN1_TIME_check(t), WOLFSSL_SUCCESS);
    t->data[8] = 0;
    t->data[3]  = 0;
    AssertIntNE(ASN1_TIME_print(bio, t), 1);
    AssertIntEQ(BIO_read(bio, buf, sizeof(buf)), 14);
    AssertIntEQ(XMEMCMP(buf, "Bad time value", 14), 0);
    AssertIntEQ(ASN1_TIME_check(t), WOLFSSL_FAILURE);

    BIO_free(bio);
    X509_free(x509);

    res = TEST_RES_CHECK(1);
    #endif
    return res;
}

static int test_wolfSSL_ASN1_UTCTIME_print(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && !defined(NO_BIO)
    BIO*  bio;
    ASN1_UTCTIME* utc = NULL;
    unsigned char buf[25];
    const char* validDate   = "190424111501Z"; /* UTC = YYMMDDHHMMSSZ */
    const char* invalidDate = "190424111501X"; /* UTC = YYMMDDHHMMSSZ */

    /* NULL parameter check */
    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    AssertIntEQ(ASN1_UTCTIME_print(bio, utc), 0);
    BIO_free(bio);

    /* Valid date */
    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    AssertNotNull(utc = (ASN1_UTCTIME*)XMALLOC(sizeof(ASN1_UTCTIME), NULL,
                                                           DYNAMIC_TYPE_ASN1));
    utc->type = ASN_UTC_TIME;
    utc->length = ASN_UTC_TIME_SIZE;
    XMEMCPY(utc->data, (byte*)validDate, ASN_UTC_TIME_SIZE);
    AssertIntEQ(ASN1_UTCTIME_print(bio, utc), 1);
    AssertIntEQ(BIO_read(bio, buf, sizeof(buf)), 24);
    AssertIntEQ(XMEMCMP(buf, "Apr 24 11:15:01 2019 GMT", sizeof(buf)-1), 0);

    XMEMSET(buf, 0, sizeof(buf));
    BIO_free(bio);

    /* Invalid format */
    AssertNotNull(bio = BIO_new(BIO_s_mem()));
    utc->type = ASN_UTC_TIME;
    utc->length = ASN_UTC_TIME_SIZE;
    XMEMCPY(utc->data, (byte*)invalidDate, ASN_UTC_TIME_SIZE);
    AssertIntEQ(ASN1_UTCTIME_print(bio, utc), 0);
    AssertIntEQ(BIO_read(bio, buf, sizeof(buf)), 14);
    AssertIntEQ(XMEMCMP(buf, "Bad time value", 14), 0);

    XFREE(utc, NULL, DYNAMIC_TYPE_ASN1);
    BIO_free(bio);

    res = TEST_RES_CHECK(1);
#endif /* OPENSSL_EXTRA && !NO_ASN_TIME && !NO_BIO */
    return res;
}

static int test_wolfSSL_ASN1_TIME_diff_compare(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    ASN1_TIME* fromTime;
    ASN1_TIME* toTime;
    int daysDiff;
    int secsDiff;

    AssertNotNull((fromTime = ASN1_TIME_new()));
    /* Feb 22, 2003, 21:15:15 */
    AssertIntEQ(ASN1_TIME_set_string(fromTime, "030222211515Z"), WOLFSSL_SUCCESS);
    AssertNotNull((toTime = ASN1_TIME_new()));
    /* Dec 19, 2010, 18:10:11 */
    AssertIntEQ(ASN1_TIME_set_string(toTime, "101219181011Z"), WOLFSSL_SUCCESS);
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime), WOLFSSL_SUCCESS);

    /* Error conditions. */
    AssertIntEQ(ASN1_TIME_diff(NULL, &secsDiff, fromTime, toTime),
                WOLFSSL_FAILURE);
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, NULL, fromTime, toTime),
                WOLFSSL_FAILURE);

    /* If both times are NULL, difference is 0. */
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, NULL, NULL),
                WOLFSSL_SUCCESS);
    AssertIntEQ(daysDiff, 0);
    AssertIntEQ(secsDiff, 0);

    /* If one time is NULL, it defaults to the current time. */
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, NULL, toTime),
                WOLFSSL_SUCCESS);
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, NULL),
                WOLFSSL_SUCCESS);

    /* Normal operation. Both times non-NULL. */
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime),
                WOLFSSL_SUCCESS);
    AssertIntEQ(daysDiff, 2856);
    AssertIntEQ(secsDiff, 75296);
    /* Swapping the times should return negative values. */
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, toTime, fromTime),
                WOLFSSL_SUCCESS);
    AssertIntEQ(daysDiff, -2856);
    AssertIntEQ(secsDiff, -75296);

    AssertIntEQ(ASN1_TIME_compare(fromTime, toTime), -1);
    AssertIntEQ(ASN1_TIME_compare(toTime, fromTime), 1);
    AssertIntEQ(ASN1_TIME_compare(fromTime, fromTime), 0);

    /* Compare regression test: No seconds difference, just difference in days.
     */
    ASN1_TIME_set_string(fromTime, "19700101000000Z");
    ASN1_TIME_set_string(toTime, "19800101000000Z");
    AssertIntEQ(ASN1_TIME_compare(fromTime, toTime), -1);
    AssertIntEQ(ASN1_TIME_compare(toTime, fromTime), 1);
    AssertIntEQ(ASN1_TIME_compare(fromTime, fromTime), 0);

    /* Edge case with Unix epoch. */
    AssertNotNull(ASN1_TIME_set_string(fromTime, "19700101000000Z"));
    AssertNotNull(ASN1_TIME_set_string(toTime, "19800101000000Z"));
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime),
                               WOLFSSL_SUCCESS);
    AssertIntEQ(daysDiff, 3652);
    AssertIntEQ(secsDiff, 0);

    /* Edge case with year > 2038 (year 2038 problem). */
    AssertNotNull(ASN1_TIME_set_string(toTime, "99991231235959Z"));
    AssertIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime),
                               WOLFSSL_SUCCESS);
    AssertIntEQ(daysDiff, 2932896);
    AssertIntEQ(secsDiff, 86399);

    ASN1_TIME_free(fromTime);
    ASN1_TIME_free(toTime);

    res = TEST_RES_CHECK(1);
#endif
    return res;
}

static int test_wolfSSL_ASN1_GENERALIZEDTIME_free(void)
{
    int res = TEST_SKIPPED;
    #if defined(OPENSSL_EXTRA)
    WOLFSSL_ASN1_GENERALIZEDTIME* asn1_gtime;
    unsigned char nullstr[32];

    XMEMSET(nullstr, 0, 32);
    asn1_gtime = (WOLFSSL_ASN1_GENERALIZEDTIME*)XMALLOC(
                    sizeof(WOLFSSL_ASN1_GENERALIZEDTIME), NULL,
                    DYNAMIC_TYPE_TMP_BUFFER);
    if (asn1_gtime) {
        XMEMCPY(asn1_gtime->data,"20180504123500Z",ASN_GENERALIZED_TIME_SIZE);

        wolfSSL_ASN1_GENERALIZEDTIME_free(asn1_gtime);
        AssertIntEQ(0, XMEMCMP(asn1_gtime->data, nullstr, 32));

        XFREE(asn1_gtime, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }

    res = TEST_RES_CHECK(1);
    #endif /* OPENSSL_EXTRA */
    return res;
}


static int test_wolfSSL_private_keys(void)
{
    int res = TEST_SKIPPED;
    #if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
       !defined(NO_FILESYSTEM)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL*     ssl;
    WOLFSSL_CTX* ctx;
    EVP_PKEY* pkey = NULL;

    OpenSSL_add_all_digests();
    OpenSSL_add_all_algorithms();

#ifndef NO_RSA
    #ifndef NO_WOLFSSL_SERVER
    AssertNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    AssertNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    AssertTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
    /* Have to load a cert before you can check the private key against that
     * certificates public key! */
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_FAILURE);
    #endif
    AssertTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif
    AssertNotNull(ssl = SSL_new(ctx));

    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    AssertIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

#ifdef USE_CERT_BUFFERS_2048
    {
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;
    unsigned char buf[FOURK_BUF];
    word32 bufSz;

    AssertIntEQ(SSL_use_RSAPrivateKey_ASN1(ssl,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    AssertIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    AssertIntEQ(SSL_use_PrivateKey_ASN1(0, ssl,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    AssertIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    /* test loading private key to the WOLFSSL_CTX */
    AssertIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);

    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    AssertIntNE(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    AssertIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(HAVE_USER_RSA) && !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    AssertIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    /* pkey not set yet, expecting to fail */
    AssertIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_FAILURE);

    /* set PKEY and test again */
    AssertNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &server_key, (long)sizeof_server_key_der_2048));
    AssertIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_SUCCESS);

    /* reuse PKEY structure and test
     * this should be checked with a memory management sanity checker */
    AssertFalse(server_key == (cons