/* packet-ieee80211.c
 * Routines for Wireless LAN (IEEE 802.11) dissection
 * Copyright 2000, Axis Communications AB
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Credits:
 *
 * The following people helped me by pointing out bugs etc. Thank you!
 *
 * Marco Molteni
 * Lena-Marie Nilsson
 * Magnus Hultman-Persson
 */

/*
 * 10/24/2005 - Add dissection for 802.11e
 * Zhu Yi <yi.zhu@intel.com>
 *
 * 04/21/2008 - Added dissection for 802.11p
 * Arada Systems <http://www.aradasystems.com>
 *
 * 04/20/2013 - Added dissection of 802.11ad according to the 9th draft of the standard.
 * Extended as a project in the Laboratory of Computer Communication & Networking (LCCN), Computer Science Department, Technion, Israel.
 * Majd Omari    <majd.omari@outlook.com>
 * Jalil Moraney <moraney.jalil@outlook.com>
 */

/*
 * Reference :
 * The 802.11 standard is "free", 6 month after the publication.
 *
 * IEEE Std 802.11-2012: Revision of IEEE Std 802.11-2007
 * include 10 amendments (802.11k,r,y,w,n,p,z,v,u,s) 802.11-2007
 * include 8 amendments (802.11a,b,d,e,g,h,i,j) 802.11-1999
 * http://standards.ieee.org/getieee802/download/802.11-2012.pdf
 *
 * WAPI (IE 68)
 * http://isotc.iso.org/livelink/livelink/fetch/-8913189/8913214/8913250/8913253/JTC001-N-9880.pdf?nodeid=8500308&vernum=-2
 */

#include "config.h"

#include <math.h>

#include <epan/packet.h>
#include <epan/capture_dissectors.h>
#include <epan/exceptions.h>
#include <wsutil/pint.h>
#include <wsutil/str_util.h>
#include <epan/addr_resolv.h>
#include <epan/address_types.h>
#include <epan/strutil.h>
#include <epan/prefs.h>
#include <epan/reassemble.h>
#include "packet-ipx.h"
#include "packet-llc.h"
#include "packet-ieee80211.h"
#include <epan/etypes.h>
#include <epan/oui.h>
#include <epan/crc32-tvb.h>
#include <epan/crypt/wep-wpadefs.h>
#include <epan/expert.h>
#include <epan/conversation_table.h>
#include <epan/uat.h>
#include <epan/eapol_keydes_types.h>
#include <epan/to_str.h>
#include <epan/proto_data.h>

#include "packet-wps.h"
#include "packet-e212.h"
#include "packet-sflow.h"
#include "packet-gre.h"

#include <epan/crypt/dot11decrypt_ws.h>

void proto_register_ieee80211(void);
void proto_reg_handoff_ieee80211(void);
void proto_register_wlan_rsna_eapol(void);

typedef struct {
  DOT11DECRYPT_KEY_ITEM used_key;
  guint keydata_len;
  guint8 keydata[0]; /* dynamic size */
} proto_eapol_keydata_t;

#define PROTO_EAPOL_KEYDATA_OFFSET (offsetof(proto_eapol_keydata_t, keydata))
#define PROTO_EAPOL_MAX_SIZE (DOT11DECRYPT_MAX_CAPLEN + PROTO_EAPOL_KEYDATA_OFFSET)

extern value_string_ext eap_type_vals_ext; /* from packet-eap.c */

#ifndef roundup2
#define roundup2(x, y)  (((x)+((y)-1))&(~((y)-1)))  /* if y is powers of two */
#endif

/* bitmask for bits [l..h]
 * taken from kernel's include/linux/bitops.h
 */
#define GENMASK(h, l)  (((1U << ((h) - (l) + 1)) - 1) << (l))

/* Defragment fragmented 802.11 datagrams */
static gboolean wlan_defragment = TRUE;

/* call subdissector for retransmitted frames */
static gboolean wlan_subdissector = TRUE;

/* Check for the presence of the 802.11 FCS */
static gboolean wlan_check_fcs = FALSE;

/* Check the FCS checksum */
static gboolean wlan_check_checksum = FALSE;

/* Ignore vendor-specific HT elements */
static gboolean wlan_ignore_draft_ht = FALSE;

/* Ignore the Protection bit; assume packet is decrypted */
#define WLAN_IGNORE_PROT_NO     0
#define WLAN_IGNORE_PROT_WO_IV  1
#define WLAN_IGNORE_PROT_W_IV   2
static gint wlan_ignore_prot = WLAN_IGNORE_PROT_NO;

/* The Key MIC len has been set by the user */
static gboolean wlan_key_mic_len_enable = FALSE;
static guint wlan_key_mic_len = 0;

/* Table for reassembly of fragments. */
static reassembly_table wlan_reassembly_table;

/* Statistical data */
static struct _wlan_stats wlan_stats;

/*-------------------------------------
 * UAT for WEP decoder
 *-------------------------------------
 */
static uat_wep_key_record_t *uat_wep_key_records = NULL;
static uat_t                *wep_uat             = NULL;
static guint                 num_wepkeys_uat     = 0;

static void *
uat_wep_key_record_copy_cb(void* n, const void* o, size_t siz _U_)
{
  uat_wep_key_record_t* new_key = (uat_wep_key_record_t *)n;
  const uat_wep_key_record_t* old_key = (const uat_wep_key_record_t *)o;

  new_key->string = g_strdup(old_key->string);

  return new_key;
}

static gboolean
uat_wep_key_record_update_cb(void* r, char** err)
{
  uat_wep_key_record_t* rec = (uat_wep_key_record_t *)r;
  decryption_key_t* dk;
  guint dk_type;

  if (rec->string == NULL) {
    *err = g_strdup("Key can't be blank");
    return FALSE;
  }

  g_strstrip(rec->string);
  dk = parse_key_string(rec->string, rec->key);

  if (dk != NULL) {
    dk_type = dk->type;
    free_key_string(dk);
    switch (dk_type) {
      case DOT11DECRYPT_KEY_TYPE_WEP:
      case DOT11DECRYPT_KEY_TYPE_WEP_40:
      case DOT11DECRYPT_KEY_TYPE_WEP_104:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WEP) {
          *err = g_strdup("Invalid WEP key format");
          return FALSE;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_WPA_PWD:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WPA_PWD) {
          *err = g_strdup("Invalid WPA_PWD key format");
          return FALSE;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_WPA_PSK:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WPA_PSK) {
          *err = g_strdup("Invalid WPA_PSK key format");
          return FALSE;
        }
        break;
      default:
        *err = g_strdup("Invalid key format");
        return FALSE;
        break;
    }
  } else {
    *err = g_strdup("Invalid key format");
    return FALSE;
  }
  return TRUE;
}

static void
uat_wep_key_record_free_cb(void*r)
{
  uat_wep_key_record_t* key = (uat_wep_key_record_t *)r;
  g_free(key->string);
}

UAT_VS_DEF(uat_wep_key_records, key, uat_wep_key_record_t, guint8, 0, STRING_KEY_TYPE_WEP)
UAT_CSTRING_CB_DEF(uat_wep_key_records, string, uat_wep_key_record_t)

/* Stuff for the WEP/WPA/WPA2 decoder */
static gboolean enable_decryption = TRUE;

static void
ieee_80211_add_tagged_parameters(tvbuff_t *tvb, int offset, packet_info *pinfo,
                                  proto_tree *tree, int tagged_parameters_len, int ftype,
                                  association_sanity_check_t *association_sanity_check);

static tvbuff_t *
try_decrypt(tvbuff_t *tvb, packet_info *pinfo, guint offset, guint len, gboolean scan_keys,
            guint8 *algorithm, guint32 *sec_header, guint32 *sec_trailer,
            PDOT11DECRYPT_KEY_ITEM used_key);

static int weak_iv(guchar *iv);

typedef struct mimo_control
{
  guint8 nc;
  guint8 nr;
  gboolean chan_width;
  guint8 grouping;
  guint8 coefficient_size;
  guint8 codebook_info;
  guint8 remaining_matrix_segment;
} mimo_control_t;

/* ************************************************************************* */
/*                          Miscellaneous Constants                          */
/* ************************************************************************* */
#define SHORT_STR 256
#define IS_DMG_KEY 1
#define IS_AP_KEY 2
#define IS_CTRL_GRANT_OR_GRANT_ACK_KEY 2
#define EAPOL_KEY 3
/* ************************************************************************* */
/*  Define some very useful macros that are used to analyze frame types etc. */
/* ************************************************************************* */

/*
 * Fetch the frame control field and swap it if needed.  "fcf" and "tvb"
 * must be valid variables.
 */
#define FETCH_FCF(off) ((option_flags & IEEE80211_COMMON_OPT_BROKEN_FC) ? \
  GUINT16_SWAP_LE_BE(tvb_get_letohs(tvb, off)) : \
  tvb_get_letohs(tvb, off))

/*
 * Extract the fragment number and sequence number from the sequence
 * control field.
 */
#define SEQCTL_FRAGMENT_NUMBER(x) ((x) & 0x000F)
#define SEQCTL_SEQUENCE_NUMBER(x) (((x) & 0xFFF0) >> 4)

/*
 * Extract subfields from the QoS control field.
 */
#define QOS_TID(x)            ((x) & 0x000F)
#define QOS_PRIORITY(x)       ((x) & 0x0007)
#define QOS_EOSP(x)           (((x) & 0x0010) >> 4) /* end of service period */
#define QOS_ACK_POLICY(x)     (((x) & 0x0060) >> 5)
#define QOS_AMSDU_PRESENT(x)  (((x) & 0x0080) >> 6)
#define QOS_FIELD_CONTENT(x)  (((x) & 0xFF00) >> 8)
#define QOS_MESH_CONTROL_PRESENT(x) (((x) & 0x0100) >> 8)

#define QOS_FLAG_EOSP    0x0010

/*
 * Extract subfields from the result of QOS_FIELD_CONTENT().
 */
#define QOS_PS_BUF_STATE_INDICATED(x)  (((x) & 0x02) >> 1)
#define QOS_PS_HIGHEST_PRI_BUF_AC(x)   (((x) & 0x0C) >> 2)
#define QOS_PS_QAP_BUF_LOAD(x)         (((x) & 0xF0) >> 4)

/*
 * Bits from the HT Control field.
 * 802.11-2012 and 802.11ac-2013 8.2.4.6, 32 bits.
 */
#define HTC_VHT              0x00000001
#define HTC_HE               0x00000002
#define HTC_MRQ              0x00000004
#define HTC_UNSOLICITED_MFB  0x20000000

/*
 * Extract subfields from the HT Control field.
 */
#define HTC_LAC(htc)           ((htc) & 0xFE)
#define HTC_LAC_MAI(htc)       (((htc) >> 2) & 0xF)
#define HTC_IS_ASELI(htc)      (HTC_LAC_MAI(htc) == 0xE)
#define HTC_LAC_MAI_MRQ(htc)   ((HTC_LAC_MAI(htc))  & 0x1)
#define HTC_LAC_MAI_MSI(htc)   ((HTC_LAC_MAI(htc) >> 1) & 0x7)
#define HTC_LAC_MFSI(htc)      (((htc) >> 4) & 0x7)
#define HTC_LAC_ASEL_CMD(htc)  (((htc) >> 9) & 0x7)
#define HTC_LAC_ASEL_DATA(htc) (((htc) >> 12) & 0xF)
#define HTC_LAC_MFB(htc)       (((htc) >> 9) & 0x7F)
#define HTC_CAL_POS(htc)       (((htc) >> 16) & 0x3)
#define HTC_CAL_SEQ(htc)       (((htc) >> 18) & 0x3)
#define HTC_CSI_STEERING(htc)  (((htc) >> 22) & 0x3)
#define HTC_NDP_ANN(htc)       (((htc) >> 24) & 0x1)
#define HTC_AC_CONSTRAINT(htc) (((htc) >> 30) & 0x1)
#define HTC_RDG_MORE_PPDU(htc) (((htc) >> 31) & 0x1)

#define HTC_MFB(htc)           (((htc) >> 9) & 0x7FFF)

/* VHT-MCS = 15, NUM_STS = 7 */
#define HTC_NO_FEEDBACK_PRESENT(mfb) (((mfb) & 0x7F) == 0x7F)

/*
 * Extract subfields from the key octet in WEP-encrypted frames.
 */
#define KEY_OCTET_WEP_KEY(x)   (((x) & 0xC0) >> 6)

#define KEY_EXTIV    0x20
#define EXTIV_LEN    8

/*
 * Bits from the Mesh Flags field
 */
#define MESH_FLAGS_ADDRESS_EXTENSION  0x3

/* ************************************************************************* */
/*              Constants used to identify cooked frame types                */
/* ************************************************************************* */
#define MGT_FRAME            0x00  /* Frame type is management */
#define CONTROL_FRAME        0x01  /* Frame type is control */
#define DATA_FRAME           0x02  /* Frame type is Data */
#define EXTENSION_FRAME      0x03  /* Frame type is Extension */

#define DATA_SHORT_HDR_LEN     24
#define DATA_LONG_HDR_LEN      30
#define MGT_FRAME_HDR_LEN      24  /* Length of Management frame-headers */

/* ************************************************************************* */
/*        Logical field codes (IEEE 802.11 encoding of tags)                 */
/* ************************************************************************* */
#define TAG_SSID                       0
#define TAG_SUPP_RATES                 1
#define TAG_FH_PARAMETER               2
#define TAG_DS_PARAMETER               3
#define TAG_CF_PARAMETER               4
#define TAG_TIM                        5
#define TAG_IBSS_PARAMETER             6
#define TAG_COUNTRY_INFO               7
#define TAG_FH_HOPPING_PARAMETER       8
#define TAG_FH_HOPPING_TABLE           9
#define TAG_REQUEST                   10
#define TAG_QBSS_LOAD                 11
#define TAG_EDCA_PARAM_SET            12
#define TAG_TSPEC                     13
#define TAG_TCLAS                     14
#define TAG_SCHEDULE                  15
#define TAG_CHALLENGE_TEXT            16

#define TAG_POWER_CONSTRAINT          32
#define TAG_POWER_CAPABILITY          33
#define TAG_TPC_REQUEST               34
#define TAG_TPC_REPORT                35
#define TAG_SUPPORTED_CHANNELS        36
#define TAG_CHANNEL_SWITCH_ANN        37
#define TAG_MEASURE_REQ               38
#define TAG_MEASURE_REP               39
#define TAG_QUIET                     40
#define TAG_IBSS_DFS                  41
#define TAG_ERP_INFO                  42
#define TAG_TS_DELAY                  43
#define TAG_TCLAS_PROCESS             44
#define TAG_HT_CAPABILITY             45 /* IEEE Stc 802.11n/D2.0 */
#define TAG_QOS_CAPABILITY            46
#define TAG_ERP_INFO_OLD              47 /* IEEE Std 802.11g/D4.0 */
#define TAG_RSN_IE                    48
/* Reserved 49 */
#define TAG_EXT_SUPP_RATES            50
#define TAG_AP_CHANNEL_REPORT         51
#define TAG_NEIGHBOR_REPORT           52
#define TAG_RCPI                      53
#define TAG_MOBILITY_DOMAIN           54  /* IEEE Std 802.11r-2008 */
#define TAG_FAST_BSS_TRANSITION       55  /* IEEE Std 802.11r-2008 */
#define TAG_TIMEOUT_INTERVAL          56  /* IEEE Std 802.11r-2008 */
#define TAG_RIC_DATA                  57  /* IEEE Std 802.11r-2008 */
#define TAG_DSE_REG_LOCATION          58
#define TAG_SUPPORTED_OPERATING_CLASSES             59 /* IEEE Std 802.11w-2009 */
#define TAG_EXTENDED_CHANNEL_SWITCH_ANNOUNCEMENT    60 /* IEEE Std 802.11w-2009 */
#define TAG_HT_INFO                   61  /* IEEE Stc 802.11n/D2.0 */
#define TAG_SECONDARY_CHANNEL_OFFSET  62  /* IEEE Stc 802.11n/D1.10/D2.0 */
#define TAG_BSS_AVG_ACCESS_DELAY      63
#define TAG_ANTENNA                   64
#define TAG_RSNI                      65
#define TAG_MEASURE_PILOT_TRANS       66
#define TAG_BSS_AVB_ADM_CAPACITY      67
#define TAG_IE_68_CONFLICT            68  /* Conflict: WAPI Vs. IEEE */
#define TAG_WAPI_PARAM_SET            68
#define TAG_BSS_AC_ACCESS_DELAY       68
#define TAG_TIME_ADV                  69  /* IEEE Std 802.11p-2010 */
#define TAG_RM_ENABLED_CAPABILITY     70
#define TAG_MULTIPLE_BSSID            71
#define TAG_20_40_BSS_CO_EX           72  /* IEEE P802.11n/D6.0 */
#define TAG_20_40_BSS_INTOL_CH_REP    73  /* IEEE P802.11n/D6.0 */
#define TAG_OVERLAP_BSS_SCAN_PAR      74  /* IEEE P802.11n/D6.0 */
#define TAG_RIC_DESCRIPTOR            75  /* IEEE Std 802.11r-2008 */
#define TAG_MMIE                      76  /* IEEE Std 802.11w-2009 */
#define TAG_EVENT_REQUEST             78
#define TAG_EVENT_REPORT              79
#define TAG_DIAGNOSTIC_REQUEST        80
#define TAG_DIAGNOSTIC_REPORT         81
#define TAG_LOCATION_PARAMETERS       82
#define TAG_NO_BSSID_CAPABILITY       83
#define TAG_SSID_LIST                 84
#define TAG_MULTIPLE_BSSID_INDEX      85
#define TAG_FMS_DESCRIPTOR            86
#define TAG_FMS_REQUEST               87
#define TAG_FMS_RESPONSE              88
#define TAG_QOS_TRAFFIC_CAPABILITY    89
#define TAG_BSS_MAX_IDLE_PERIOD       90
#define TAG_TFS_REQUEST               91
#define TAG_TFS_RESPONSE              92
#define TAG_WNM_SLEEP_MODE            93
#define TAG_TIM_BROADCAST_REQUEST     94
#define TAG_TIM_BROADCAST_RESPONSE    95
#define TAG_COLLOCATED_INTER_REPORT   96
#define TAG_CHANNEL_USAGE             97
#define TAG_TIME_ZONE                 98  /* IEEE Std 802.11v-2011 */
#define TAG_DMS_REQUEST               99
#define TAG_DMS_RESPONSE             100
#define TAG_LINK_IDENTIFIER          101  /* IEEE Std 802.11z-2010 */
#define TAG_WAKEUP_SCHEDULE          102  /* IEEE Std 802.11z-2010 */
#define TAG_CHANNEL_SWITCH_TIMING    104  /* IEEE Std 802.11z-2010 */
#define TAG_PTI_CONTROL              105  /* IEEE Std 802.11z-2010 */
#define TAG_PU_BUFFER_STATUS         106  /* IEEE Std 802.11z-2010 */
#define TAG_INTERWORKING             107  /* IEEE Std 802.11u-2011 */
#define TAG_ADVERTISEMENT_PROTOCOL   108  /* IEEE Std 802.11u-2011 */
#define TAG_EXPIDITED_BANDWIDTH_REQ  109  /* IEEE Std 802.11u-2011 */
#define TAG_QOS_MAP_SET              110  /* IEEE Std 802.11u-2011 */
#define TAG_ROAMING_CONSORTIUM       111  /* IEEE Std 802.11u-2011 */
#define TAG_EMERGENCY_ALERT_ID       112  /* IEEE Std 802.11u-2011 */
#define TAG_MESH_CONFIGURATION       113  /* IEEE Std 802.11s-2011 */
#define TAG_MESH_ID                  114  /* IEEE Std 802.11s-2011 */
#define TAG_MESH_LINK_METRIC_REPORT  115
#define TAG_CONGESTION_NOTIFICATION  116
#define TAG_MESH_PEERING_MGMT        117  /* IEEE Std 802.11s-2011 */
#define TAG_MESH_CHANNEL_SWITCH      118
#define TAG_MESH_AWAKE_WINDOW        119  /* IEEE Std 802.11s-2011 */
#define TAG_BEACON_TIMING            120
#define TAG_MCCAOP_SETUP_REQUEST     121
#define TAG_MCCAOP_SETUP_REPLY       122
#define TAG_MCCAOP_ADVERTISEMENT     123
#define TAG_MCCAOP_TEARDOWN          124
#define TAG_GANN                     125
#define TAG_RANN                     126  /* IEEE Std 802.11s-2011 */
#define TAG_EXTENDED_CAPABILITIES    127  /* IEEE Stc 802.11n/D1.10/D2.0 */
#define TAG_AGERE_PROPRIETARY        128
#define TAG_MESH_PREQ                130  /* IEEE Std 802.11s-2011 */
#define TAG_MESH_PREP                131  /* IEEE Std 802.11s-2011 */
#define TAG_MESH_PERR                132  /* IEEE Std 802.11s-2011 */
#define TAG_CISCO_CCX1_CKIP          133  /* Cisco Compatible eXtensions v1 */
#define TAG_CISCO_CCX2               136  /* Cisco Compatible eXtensions v2 */
#define TAG_PXU                      137
#define TAG_PXUC                     138
#define TAG_AUTH_MESH_PEERING_EXCH   139
#define TAG_MIC                      140
#define TAG_DESTINATION_URI          141
#define TAG_U_APSD_COEX              142
#define TAG_WAKEUP_SCHEDULE_AD       143  /* IEEE Std 802.11ad */
#define TAG_EXTENDED_SCHEDULE        144  /* IEEE Std 802.11ad */
#define TAG_STA_AVAILABILITY         145  /* IEEE Std 802.11ad */
#define TAG_DMG_TSPEC                146  /* IEEE Std 802.11ad */
#define TAG_NEXT_DMG_ATI             147  /* IEEE Std 802.11ad */
#define TAG_DMG_CAPABILITIES         148  /* IEEE Std 802.11ad */
#define TAG_CISCO_CCX3               149  /* Cisco Compatible eXtensions v3 */
#define TAG_CISCO_VENDOR_SPECIFIC    150  /* Cisco Compatible eXtensions */
#define TAG_DMG_OPERATION            151  /* IEEE Std 802.11ad */
#define TAG_DMG_BSS_PARAMETER_CHANGE 152  /* IEEE Std 802.11ad */
#define TAG_DMG_BEAM_REFINEMENT      153  /* IEEE Std 802.11ad */
#define TAG_CHANNEL_MEASURMENT_FB    154  /* IEEE Std 802.11ad */
#define TAG_AWAKE_WINDOW             157  /* IEEE Std 802.11ad */
#define TAG_MULTI_BAND               158  /* IEEE Std 802.11ad */
#define TAG_ADDBA_EXT                159  /* IEEE Std 802.11ad */
#define TAG_NEXTPCP_LIST             160  /* IEEE Std 802.11ad */
#define TAG_PCP_HANDOVER             161  /* IEEE Std 802.11ad */
#define TAG_DMG_LINK_MARGIN          162  /* IEEE Std 802.11ad */
#define TAG_SWITCHING_STREAM         163  /* IEEE Std 802.11ad */
#define TAG_SESSION_TRANSMISSION     164  /* IEEE Std 802.11ad */
#define TAG_DYN_TONE_PAIR_REP        165  /* IEEE Std 802.11ad */
#define TAG_CLUSTER_REP              166  /* IEEE Std 802.11ad */
#define TAG_RELAY_CAPABILITIES       167  /* IEEE Std 802.11ad */
#define TAG_RELAY_TRANSFER_PARAM     168  /* IEEE Std 802.11ad */
#define TAG_BEAMLINK_MAINTENANCE     169  /* IEEE Std 802.11ad */
#define TAG_MULTIPLE_MAC_SUBLAYERS   170  /* IEEE Std 802.11ad */
#define TAG_U_PID                    171  /* IEEE Std 802.11ad */
#define TAG_DMG_LINK_ADAPTION_ACK    172  /* IEEE Std 802.11ad */
#define TAG_SYMBOL_PROPRIETARY       173
#define TAG_MCCAOP_ADVERTISEMENT_OV  174
#define TAG_QUIET_PERIOD_REQ         175  /* IEEE Std 802.11ad */
#define TAG_QUIET_PERIOD_RES         177  /* IEEE Std 802.11ad */
#define TAG_ECAPC_POLICY             182  /* IEEE Std 802.11ad */
#define TAG_CLUSTER_TIME_OFFSET      183  /* IEEE Std 802.11ad */
#define TAG_ANTENNA_SECTOR_ID        190  /* IEEE Std 802.11ad */
#define TAG_VHT_CAPABILITY           191  /* IEEE Std 802.11ac/D3.1 */
#define TAG_VHT_OPERATION            192  /* IEEE Std 802.11ac/D3.1 */
#define TAG_EXT_BSS_LOAD             193  /* IEEE Std 802.11ac */
#define TAG_WIDE_BW_CHANNEL_SWITCH   194  /* IEEE Std 802.11ac */
#define TAG_VHT_TX_PWR_ENVELOPE      195  /* IEEE Std 802.11ac/D5.0 */
#define TAG_CHANNEL_SWITCH_WRAPPER   196  /* IEEE Std 802.11ac */
#define TAG_OPERATING_MODE_NOTIFICATION 199  /* IEEE Std 802.11ac */
#define TAG_S1G_OPEN_LOOP_LINK_MARGIN_INDEX 207 /* IEEE Std 802.11ah */
#define TAG_RPS                      208  /* IEEE Stf 802.11ah */
#define TAG_PAGE_SLICE               209  /* IEEE Stf 802.11ah */
#define TAG_AID_REQUEST              210  /* IEEE Stf 802.11ah */
#define TAG_AID_RESPONSE             211  /* IEEE Stf 802.11ah */
#define TAG_S1G_SECTOR_OPERATION     212  /* IEEE Stf 802.11ah */
#define TAG_S1G_BEACON_COMPATIBILITY 213  /* IEEE Stf 802.11ah */
#define TAG_SHORT_BEACON_INTERVAL    214  /* IEEE Stf 802.11ah */
#define TAG_CHANGE_SEQUENCE          215  /* IEEE Stf 802.11ah */
#define TAG_TWT                      216  /* IEEE Std 802.11ah */
#define TAG_S1G_CAPABILITIES         217  /* IEEE Stf 802.11ah */
#define TAG_SUBCHANNEL_SELECTIVE_TRANSMISSION 220  /* IEEE Stf 802.11ah */
#define TAG_VENDOR_SPECIFIC_IE       221
#define TAG_AUTHENTICATION_CONTROL   222  /* IEEE Stf 802.11ah */
#define TAG_TSF_TIMER_ACCURACY       223  /* IEEE Stf 802.11ah */
#define TAG_S1G_RELAY                224  /* IEEE Stf 802.11ah */
#define TAG_REACHABLE_ADDRESS        225  /* IEEE Stf 802.11ah */
#define TAG_S1G_RELAY_DISCOVERY      226  /* IEEE Stf 802.11ah */
#define TAG_AID_ANNOUNCEMENT         228  /* IEEE Stf 802.11ah */
#define TAG_PV1_PROBE_RESPONSE_OPTION 229  /* IEEE Stf 802.11ah */
#define TAG_EL_OPERATION             230  /* IEEE Stf 802.11ah */
#define TAG_SECTORIZED_GROUP_ID_LIST 231  /* IEEE Stf 802.11ah */
#define TAG_S1G_OPERATION            232  /* IEEE Stf 802.11ah */
#define TAG_HEADER_COMPRESSION       233  /* IEEE Stf 802.11ah */
#define TAG_SST_OPERATION            234  /* IEEE Stf 802.11ah */
#define TAG_MAD                      235  /* IEEE Stf 802.11ah */
#define TAG_S1G_RELAY_ACTIVATION     236  /* IEEE Stf 802.11ah */
#define TAG_CAG_NUMBER               237  /* IEEE Std 802.11ai */
#define TAG_AP_CSN                   239  /* IEEE Std 802.11ai */
#define TAG_FILS_INDICATION          240  /* IEEE Std 802.11ai */
#define TAG_DIFF_INITIAL_LINK_SETUP  241  /* IEEE Std 802.11ai */
#define TAG_FRAGMENT                 242  /* IEEE Std 802.11ai */
#define TAG_ELEMENT_ID_EXTENSION     255  /* IEEE Std 802.11ai */

static const value_string tag_num_vals[] = {
  { TAG_SSID,                                 "SSID parameter set" },
  { TAG_SUPP_RATES,                           "Supported Rates" },
  { TAG_FH_PARAMETER,                         "FH Parameter set" },
  { TAG_DS_PARAMETER,                         "DS Parameter set" },
  { TAG_CF_PARAMETER,                         "CF Parameter set" },
  { TAG_TIM,                                  "Traffic Indication Map (TIM)" },
  { TAG_IBSS_PARAMETER,                       "IBSS Parameter set" },
  { TAG_COUNTRY_INFO,                         "Country Information" },
  { TAG_FH_HOPPING_PARAMETER,                 "Hopping Pattern Parameters" },
  { TAG_FH_HOPPING_TABLE,                     "Hopping Pattern Table" },
  { TAG_REQUEST,                              "Request" },
  { TAG_QBSS_LOAD,                            "QBSS Load Element" },
  { TAG_EDCA_PARAM_SET,                       "EDCA Parameter Set" },
  { TAG_TSPEC,                                "Traffic Specification" },
  { TAG_TCLAS,                                "Traffic Classification" },
  { TAG_SCHEDULE,                             "Schedule" },
  { TAG_CHALLENGE_TEXT,                       "Challenge text" },
  { TAG_POWER_CONSTRAINT,                     "Power Constraint" },
  { TAG_POWER_CAPABILITY,                     "Power Capability" },
  { TAG_TPC_REQUEST,                          "TPC Request" },
  { TAG_TPC_REPORT,                           "TPC Report" },
  { TAG_SUPPORTED_CHANNELS,                   "Supported Channels" },
  { TAG_CHANNEL_SWITCH_ANN,                   "Channel Switch Announcement" },
  { TAG_MEASURE_REQ,                          "Measurement Request" },
  { TAG_MEASURE_REP,                          "Measurement Report" },
  { TAG_QUIET,                                "Quiet" },
  { TAG_IBSS_DFS,                             "IBSS DFS" },
  { TAG_ERP_INFO,                             "ERP Information" },
  { TAG_TS_DELAY,                             "TS Delay" },
  { TAG_TCLAS_PROCESS,                        "TCLAS Processing" },
  { TAG_HT_CAPABILITY,                        "HT Capabilities (802.11n D1.10)" },
  { TAG_QOS_CAPABILITY,                       "QoS Capability" },
  { TAG_ERP_INFO_OLD,                         "ERP Information" }, /* Reserved... */
  { TAG_RSN_IE,                               "RSN Information" },
  { TAG_EXT_SUPP_RATES,                       "Extended Supported Rates" },
  { TAG_AP_CHANNEL_REPORT,                    "AP Channel Report" },
  { TAG_NEIGHBOR_REPORT,                      "Neighbor Report" },
  { TAG_RCPI,                                 "RCPI" },
  { TAG_MOBILITY_DOMAIN,                      "Mobility Domain" },
  { TAG_FAST_BSS_TRANSITION,                  "Fast BSS Transition" },
  { TAG_TIMEOUT_INTERVAL,                     "Timeout Interval" },
  { TAG_RIC_DATA,                             "RIC Data" },
  { TAG_DSE_REG_LOCATION,                     "DSE Registered Location" },
  { TAG_SUPPORTED_OPERATING_CLASSES,          "Supported Operating Classes" },
  { TAG_EXTENDED_CHANNEL_SWITCH_ANNOUNCEMENT, "Extended Channel Switch Announcement" },
  { TAG_HT_INFO,                              "HT Information (802.11n D1.10)" },
  { TAG_SECONDARY_CHANNEL_OFFSET,             "Secondary Channel Offset (802.11n D1.10)" },
  { TAG_BSS_AVG_ACCESS_DELAY,                 "BSS Average Access Delay" },
  { TAG_ANTENNA,                              "Antenna" },
  { TAG_RSNI,                                 "RSNI" },
  { TAG_MEASURE_PILOT_TRANS,                  "Measurement Pilot Transmission" },
  { TAG_BSS_AVB_ADM_CAPACITY,                 "BSS Available Admission Capacity" },
  { TAG_IE_68_CONFLICT,                       "BSS AC Access Delay/WAPI Parameter Set" },
  { TAG_TIME_ADV,                             "Time Advertisement" },
  { TAG_RM_ENABLED_CAPABILITY,                "RM Enabled Capabilities" },
  { TAG_MULTIPLE_BSSID,                       "Multiple BSSID" },
  { TAG_20_40_BSS_CO_EX,                      "20/40 BSS Coexistence" },
  { TAG_20_40_BSS_INTOL_CH_REP,               "20/40 BSS Intolerant Channel Report" },   /* IEEE P802.11n/D6.0 */
  { TAG_OVERLAP_BSS_SCAN_PAR,                 "Overlapping BSS Scan Parameters" },       /* IEEE P802.11n/D6.0 */
  { TAG_RIC_DESCRIPTOR,                       "RIC Descriptor" },
  { TAG_MMIE,                                 "Management MIC" },
  { TAG_EVENT_REQUEST,                        "Event Request" },
  { TAG_EVENT_REPORT,                         "Event Report" },
  { TAG_DIAGNOSTIC_REQUEST,                   "Diagnostic Request" },
  { TAG_DIAGNOSTIC_REPORT,                    "Diagnostic Report" },
  { TAG_LOCATION_PARAMETERS,                  "Location Parameters" },
  { TAG_NO_BSSID_CAPABILITY,                  "Non Transmitted BSSID Capability" },
  { TAG_SSID_LIST,                            "SSID List" },
  { TAG_MULTIPLE_BSSID_INDEX,                 "Multiple BSSID Index" },
  { TAG_FMS_DESCRIPTOR,                       "FMS Descriptor" },
  { TAG_FMS_REQUEST,                          "FMS Request" },
  { TAG_FMS_RESPONSE,                         "FMS Response" },
  { TAG_QOS_TRAFFIC_CAPABILITY,               "QoS Traffic Capability" },
  { TAG_BSS_MAX_IDLE_PERIOD,                  "BSS Max Idle Period" },
  { TAG_TFS_REQUEST,                          "TFS Request" },
  { TAG_TFS_RESPONSE,                         "TFS Response" },
  { TAG_WNM_SLEEP_MODE,                       "WNM-Sleep Mode" },
  { TAG_TIM_BROADCAST_REQUEST,                "TIM Broadcast Request" },
  { TAG_TIM_BROADCAST_RESPONSE,               "TIM Broadcast Response" },
  { TAG_COLLOCATED_INTER_REPORT,              "Collocated Interference Report" },
  { TAG_CHANNEL_USAGE,                        "Channel Usage" },
  { TAG_TIME_ZONE,                            "Time Zone" },
  { TAG_DMS_REQUEST,                          "DMS Request" },
  { TAG_DMS_RESPONSE,                         "DMS Response" },
  { TAG_LINK_IDENTIFIER,                      "Link Identifier" },
  { TAG_WAKEUP_SCHEDULE,                      "Wakeup Schedule" },
  { TAG_CHANNEL_SWITCH_TIMING,                "Channel Switch Timing" },
  { TAG_PTI_CONTROL,                          "PTI Control" },
  { TAG_PU_BUFFER_STATUS,                     "PU Buffer Status" },
  { TAG_INTERWORKING,                         "Interworking" },
  { TAG_ADVERTISEMENT_PROTOCOL,               "Advertisement Protocol"},
  { TAG_EXPIDITED_BANDWIDTH_REQ,              "Expedited Bandwidth Request" },
  { TAG_QOS_MAP_SET,                          "QoS Map Set" },
  { TAG_ROAMING_CONSORTIUM,                   "Roaming Consortium" },
  { TAG_EMERGENCY_ALERT_ID,                   "Emergency Alert Identifier" },
  { TAG_MESH_CONFIGURATION,                   "Mesh Configuration" },
  { TAG_MESH_ID,                              "Mesh ID" },
  { TAG_MESH_LINK_METRIC_REPORT,              "Mesh Link Metric Report" },
  { TAG_CONGESTION_NOTIFICATION,              "Congestion Notification" },
  { TAG_MESH_PEERING_MGMT,                    "Mesh Peering Management" },
  { TAG_MESH_CHANNEL_SWITCH,                  "Mesh Channel Switch Parameters" },
  { TAG_MESH_AWAKE_WINDOW,                    "Mesh Awake Window" },
  { TAG_BEACON_TIMING,                        "Beacon Timing" },
  { TAG_MCCAOP_SETUP_REQUEST,                 "MCCAOP Setup Request" },
  { TAG_MCCAOP_SETUP_REPLY,                   "MCCAOP SETUP Reply" },
  { TAG_MCCAOP_ADVERTISEMENT,                 "MCCAOP Advertisement" },
  { TAG_MCCAOP_TEARDOWN,                      "MCCAOP Teardown" },
  { TAG_GANN,                                 "Gate Announcement" },
  { TAG_RANN,                                 "Root Announcement" },
  { TAG_EXTENDED_CAPABILITIES,                "Extended Capabilities" },
  { TAG_AGERE_PROPRIETARY,                    "Agere Proprietary" },
  { TAG_MESH_PREQ,                            "Path Request" },
  { TAG_MESH_PREP,                            "Path Reply" },
  { TAG_MESH_PERR,                            "Path Error" },
  { TAG_CISCO_CCX1_CKIP,                      "Cisco CCX1 CKIP + Device Name" },
  { TAG_CISCO_CCX2,                           "Cisco CCX2" },
  { TAG_PXU,                                  "Proxy Update" },
  { TAG_PXUC,                                 "Proxy Update Confirmation"},
  { TAG_AUTH_MESH_PEERING_EXCH,               "Auhenticated Mesh Perring Exchange" },
  { TAG_MIC,                                  "MIC (Message Integrity Code)" },
  { TAG_DESTINATION_URI,                      "Destination URI" },
  { TAG_U_APSD_COEX,                          "U-APSD Coexistence" },
  { TAG_WAKEUP_SCHEDULE_AD,                   "Wakeup Schedule 802.11ad" },
  { TAG_EXTENDED_SCHEDULE,                    "Extended Schedule" },
  { TAG_STA_AVAILABILITY,                     "STA Availability" },
  { TAG_DMG_TSPEC,                            "DMG TSPEC" },
  { TAG_NEXT_DMG_ATI,                         "Next DMG ATI" },
  { TAG_DMG_CAPABILITIES,                     "DMG Capabilities" },
  { TAG_CISCO_CCX3,                           "Cisco Unknown 95" },
  { TAG_CISCO_VENDOR_SPECIFIC,                "Vendor Specific" },
  { TAG_DMG_OPERATION,                        "DMG Operating" },
  { TAG_DMG_BSS_PARAMETER_CHANGE,             "DMG BSS Parameter Change" },
  { TAG_DMG_BEAM_REFINEMENT,                  "DMG Beam Refinement" },
  { TAG_CHANNEL_MEASURMENT_FB,                "Channel Measurement Feedback" },
  { TAG_AWAKE_WINDOW,                         "Awake Window" },
  { TAG_MULTI_BAND,                           "Multi Band" },
  { TAG_ADDBA_EXT,                            "ADDBA Extension" },
  { TAG_NEXTPCP_LIST,                         "NEXTPCP List" },
  { TAG_PCP_HANDOVER,                         "PCP Handover" },
  { TAG_DMG_LINK_MARGIN,                      "DMG Link Margin" },
  { TAG_SWITCHING_STREAM,                     "Switching Stream" },
  { TAG_SESSION_TRANSMISSION,                 "Session Transmission" },
  { TAG_DYN_TONE_PAIR_REP,                    "Dynamic Tone Pairing Report" },
  { TAG_CLUSTER_REP,                          "Cluster Report" },
  { TAG_RELAY_CAPABILITIES,                   "Relay Capabilities" },
  { TAG_RELAY_TRANSFER_PARAM,                 "Relay Transfer Parameter" },
  { TAG_BEAMLINK_MAINTENANCE,                 "Beamlink Maintenance" },
  { TAG_MULTIPLE_MAC_SUBLAYERS,               "Multiple MAC Sublayers" },
  { TAG_U_PID,                                "U-PID" },
  { TAG_DMG_LINK_ADAPTION_ACK,                "DMG Link Adaption Acknowledgment" },
  { TAG_SYMBOL_PROPRIETARY,                   "Symbol Proprietary" },
  { TAG_MCCAOP_ADVERTISEMENT_OV,              "MCCAOP Advertisement Overview" },
  { TAG_QUIET_PERIOD_REQ,                     "Quiet Period Request" },
  { TAG_QUIET_PERIOD_RES,                     "Quiet Period Response" },
  { TAG_ECAPC_POLICY,                         "ECAPC Policy" },
  { TAG_CLUSTER_TIME_OFFSET,                  "Cluster Time Offset" },
  { TAG_ANTENNA_SECTOR_ID,                    "Antenna Sector ID" },
  { TAG_VHT_CAPABILITY,                       "VHT Capabilities" },
  { TAG_VHT_OPERATION,                        "VHT Operation" },
  { TAG_EXT_BSS_LOAD,                         "Extended BSS Load" },
  { TAG_WIDE_BW_CHANNEL_SWITCH,               "Wide Bandwidth Channel Switch" },
  { TAG_VHT_TX_PWR_ENVELOPE,                  "VHT Tx Power Envelope" },
  { TAG_CHANNEL_SWITCH_WRAPPER,               "Channel Switch Wrapper" },
  { TAG_OPERATING_MODE_NOTIFICATION,          "Operating Mode Notification" },
  { TAG_S1G_OPEN_LOOP_LINK_MARGIN_INDEX,      "S1G Open-Loop Link Margin Index" },
  { TAG_RPS,                                  "RPS" },
  { TAG_PAGE_SLICE,                           "Page Slice" },
  { TAG_AID_REQUEST,                          "AID Request" },
  { TAG_AID_RESPONSE,                         "AID Response" },
  { TAG_S1G_SECTOR_OPERATION,                 "Sector Operation" },
  { TAG_S1G_BEACON_COMPATIBILITY,             "S1G Beacon Compatibility" },
  { TAG_SHORT_BEACON_INTERVAL,                "Short Beacon Interval" },
  { TAG_CHANGE_SEQUENCE,                      "Change Sequence" },
  { TAG_TWT,                                  "Target Wake Time" },
  { TAG_S1G_CAPABILITIES,                     "S1G Capabilities" },
  { TAG_SUBCHANNEL_SELECTIVE_TRANSMISSION,    "Subchannel Selective Transmission" },
  { TAG_VENDOR_SPECIFIC_IE,                   "Vendor Specific" },
  { TAG_AUTHENTICATION_CONTROL,               "Authentication Control" },
  { TAG_TSF_TIMER_ACCURACY,                   "TSF Timer Accuracy" },
  { TAG_S1G_RELAY,                            "S1G Relay" },
  { TAG_REACHABLE_ADDRESS,                    "Reachable Address" },
  { TAG_S1G_RELAY_DISCOVERY,                  "S1G Relay Discovery" },
  { TAG_AID_ANNOUNCEMENT,                     "AID Announcement" },
  { TAG_PV1_PROBE_RESPONSE_OPTION,            "PV1 Probe Response Option" },
  { TAG_EL_OPERATION,                         "EL Operation" },
  { TAG_SECTORIZED_GROUP_ID_LIST,             "Sectorized Group ID List" },
  { TAG_S1G_OPERATION,                        "S1G Operation" },
  { TAG_HEADER_COMPRESSION,                   "Header Compression" },
  { TAG_SST_OPERATION,                        "SST Operation" },
  { TAG_MAD,                                  "MAD" },
  { TAG_S1G_RELAY_ACTIVATION,                 "S1G Relay Activation" },
  { TAG_CAG_NUMBER,                           "CAG Number"},
  { TAG_AP_CSN,                               "AP-CSN"},
  { TAG_FILS_INDICATION,                      "FILS Indication"},
  { TAG_DIFF_INITIAL_LINK_SETUP,              "Differential Initial Link Setup"},
  { TAG_FRAGMENT,                             "Fragment"},
  { TAG_ELEMENT_ID_EXTENSION,                 "Element ID Extension" },
  { 0, NULL }
};
static value_string_ext tag_num_vals_ext = VALUE_STRING_EXT_INIT(tag_num_vals);

#define ETAG_ASSOC_DELAY_INFO          1
#define ETAG_FILS_REQ_PARAMS           2
#define ETAG_FILS_KEY_CONFIRM          3
#define ETAG_FILS_SESSION              4
#define ETAG_FILS_HLP_CONTAINER        5
#define ETAG_FILS_IP_ADDRESS_ASSIGN    6
#define ETAG_KEY_DELIVERY              7
#define ETAG_FILS_WRAPPED_DATA         8
#define ETAG_FTM_SYNC_INFO             9
#define ETAG_EXTENDED_REQUEST          10
#define ETAG_ESTIMATED_SERVICE_PARAM   11
#define ETAG_FILS_PUBLIC_KEY           12
#define ETAG_FILS_NONCE                13
#define ETAG_FUTURE_CHANNEL_GUIDANCE   14

/* RFC 8110 */
#define ETAG_OWE_DH_PARAMETER          32

/* 802.11AX defined tags */
#define ETAG_HE_CAPABILITIES                   35
#define ETAG_HE_OPERATION                      36
#define ETAG_UORA_PARAMETER_SET                37
#define ETAG_MU_EDCA_PARAMETER_SET             38
#define ETAG_SPATIAL_REUSE_PARAMETER_SET       39
#define ETAG_NDP_FEEDBACK_REPORT_PARAMETER_SET 41
#define ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT     42
#define ETAG_QUIET_TIME_PERIOD_SETUP           43
#define ETAG_ESS_REPORT                        44

static const value_string tag_num_vals_eid_ext[] = {
  { ETAG_ASSOC_DELAY_INFO,                    "Association Delay Info" },
  { ETAG_FILS_REQ_PARAMS,                     "FILS Request Parameters" },
  { ETAG_FILS_KEY_CONFIRM,                    "FILS Key Confirmation" },
  { ETAG_FILS_SESSION,                        "FILS Session" },
  { ETAG_FILS_HLP_CONTAINER,                  "FILS HLP Container" },
  { ETAG_FILS_IP_ADDRESS_ASSIGN,              "FILS IP Address Assignment" },
  { ETAG_KEY_DELIVERY,                        "Key Delivery" },
  { ETAG_FILS_WRAPPED_DATA,                   "FILS Wrapped Data" },
  { ETAG_FTM_SYNC_INFO,                       "FTM Synchronization Information" },
  { ETAG_EXTENDED_REQUEST,                    "Extended Request" },
  { ETAG_ESTIMATED_SERVICE_PARAM,             "Estimated Service Parameters" },
  { ETAG_FILS_PUBLIC_KEY,                     "FILS Public Key" },
  { ETAG_FILS_NONCE,                          "FILS Nonce" },
  { ETAG_FUTURE_CHANNEL_GUIDANCE,             "Future Channel Guidance" },
  { ETAG_OWE_DH_PARAMETER,                    "OWE Diffie-Hellman Parameter" },
  { ETAG_HE_CAPABILITIES,                     "HE Capabilities (IEEE Std 802.11ax/D3.0)" },
  { ETAG_HE_OPERATION,                        "HE Operation (IEEE Std 802.11ax/D3.0)" },
  { ETAG_UORA_PARAMETER_SET,                  "UORA Parameter Set" },
  { ETAG_MU_EDCA_PARAMETER_SET,               "MU EDCA Parameter Set" },
  { ETAG_SPATIAL_REUSE_PARAMETER_SET,         "Spatial Reuse Parameter Set" },
  { ETAG_NDP_FEEDBACK_REPORT_PARAMETER_SET,   "NDP Feedback Report Parameter Set" },
  { ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT,       "BSS Color Change Announcement" },
  { ETAG_QUIET_TIME_PERIOD_SETUP,             "Quiet Time Period Setup" },
  { ETAG_ESS_REPORT,                          "ESS Report" },
  { 0, NULL }
};
static value_string_ext tag_num_vals_eid_ext_ext = VALUE_STRING_EXT_INIT(tag_num_vals_eid_ext);

static const value_string wfa_subtype_vals[] = {
  { WFA_SUBTYPE_SUBSCRIPTION_REMEDIATION, "Subscription Remediation" },
  { WFA_SUBTYPE_DEAUTHENTICATION_IMMINENT, "Deauthentication Imminent" },
  { WFA_SUBTYPE_P2P, "P2P" },
  { WFA_SUBTYPE_HS20_INDICATION, "Hotspot 2.0 Indication" },
  { WFA_SUBTYPE_HS20_ANQP, "Hotspot 2.0 ANQP" },
  { WFA_SUBTYPE_OSEN, "OSU Server-only l2 Encryption Network" },
  { WFA_SUBTYPE_DPP, "Device Provisioning Protocol" },
  { WFA_SUBTYPE_IEEE1905_MULTI_AP, "IEEE1905 Multi-AP" },
  { WFA_SUBTYPE_OWE_TRANSITION_MODE, "OWE Transition Mode" },
  { 0, NULL }
};

#define DPP_CONFIGURATION_PROTOCOL 0x01

static const value_string dpp_subtype_vals[] = {
  { DPP_CONFIGURATION_PROTOCOL, "DPP Configuration Protocol" },
  { 0, NULL }
};

/* ************************************************************************* */
/*              Supported Rates (7.3.2.2)                                    */
/* ************************************************************************* */

static const value_string ieee80211_supported_rates_vals[] = {
  { 0x02, "1" },
  { 0x03, "1.5" },
  { 0x04, "2" },
  { 0x05, "2.5" },
  { 0x06, "3" },
  { 0x09, "4.5" },
  { 0x0B, "5.5" },
  { 0x0C, "6" },
  { 0x12, "9" },
  { 0x16, "11" },
  { 0x18, "12" },
  { 0x1B, "13.5" },
  { 0x24, "18" },
  { 0x2C, "22" },
  { 0x30, "24" },
  { 0x36, "27" },
  { 0x42, "33" },
  { 0x48, "36" },
  { 0x60, "48" },
  { 0x6C, "54" },
  { 0x82, "1(B)" },
  { 0x83, "1.5(B)" },
  { 0x84, "2(B)" },
  { 0x85, "2.5(B)" },
  { 0x86, "3(B)" },
  { 0x89, "4.5(B)" },
  { 0x8B, "5.5(B)" },
  { 0x8C, "6(B)" },
  { 0x92, "9(B)" },
  { 0x96, "11(B)" },
  { 0x98, "12(B)" },
  { 0x9B, "13.5(B)" },
  { 0xA4, "18(B)" },
  { 0xAC, "22(B)" },
  { 0xB0, "24(B)" },
  { 0xB6, "27(B)" },
  { 0xC2, "33(B)" },
  { 0xC8, "36(B)" },
  { 0xE0, "48(B)" },
  { 0xEC, "54(B)" },
  { 0xFF, "BSS requires support for mandatory features of HT PHY (IEEE 802.11 - Clause 20)" },
  { 0,    NULL}
};
value_string_ext ieee80211_supported_rates_vals_ext = VALUE_STRING_EXT_INIT(ieee80211_supported_rates_vals);

/* ************************************************************************* */
/*                         8.4.1.7 Reason Code field                         */
/* ************************************************************************* */
static const value_string ieee80211_reason_code[] = {
  {  1, "Unspecified reason" },
  {  2, "Previous authentication no longer valid" },
  {  3, "Deauthenticated because sending STA is leaving (or has left) IBSS or ESS" },
  {  4, "Disassociated due to inactivity" },
  {  5, "Disassociated because AP is unable to handle all currently associated STAs" },
  {  6, "Class 2 frame received from nonauthenticated STA" },
  {  7, "Class 3 frame received from nonassociated STA" },
  {  8, "Disassociated because sending STA is leaving (or has left) BSS" },
  {  9, "STA requesting (re)association is not authenticated with responding STA" },
  { 10, "Disassociated because the information in the Power Capability element is unacceptable" },
  { 11, "Disassociated because the information in the Supported Channels element is unacceptable" },
  { 12, "Disassociated due to BSS transition management" },
  { 13, "Invalid information element, i.e., an information element defined in this standard for which the content does not meet the specifications in Clause 7" },
  { 14, "Message integrity code (MIC) failure" },
  { 15, "4-Way Handshake timeout" },
  { 16, "Group Key Handshake timeout" },
  { 17, "Information element in 4-Way Handshake different from (Re)Association Request/Probe Response/Beacon frame" },
  { 18, "Invalid group cipher" },
  { 19, "Invalid pairwise cipher" },
  { 20, "Invalid AKMP" },
  { 21, "Unsupported RSN information element version" },
  { 22, "Invalid RSN information element capabilities" },
  { 23, "IEEE 802.1X authentication failed" },
  { 24, "Cipher suite rejected because of the security policy" },
  { 25, "TDLS direct-link teardown due to TDLS peer STA unreachable via the TDLS direct link" },
  { 26, "TDLS direct-link teardown for unspecified reason" },
  { 27, "Disassociated because session terminated by SSP request" },
  { 28, "Disassociated because of lack of SSP roaming agreement" },
  { 29, "Requested service rejected because of SSP cipher suite or AKM requirement " },
  { 30, "Requested service not authorized in this location" },
  { 31, "TS deleted because QoS AP lacks sufficient bandwidth for this QoS STA due to a change in BSS service characteristics or operational mode" },
  { 32, "Disassociated for unspecified, QoS-related reason" },
  { 33, "Disassociated because QoS AP lacks sufficient bandwidth for this QoS STA" },
  { 34, "Disassociated because excessive number of frames need to be acknowledged, but are not acknowledged due to AP transmissions and/or poor channel conditions" },
  { 35, "Disassociated because STA is transmitting outside the limits of its TXOPs" },
  { 36, "Requested from peer STA as the STA is leaving the BSS (or resetting)" },
  { 37, "Requested from peer STA as it does not want to use the mechanism" },
  { 38, "Requested from peer STA as the STA received frames using the mechanism for which a setup is required" },
  { 39, "Requested from peer STA due to timeout" },
  { 45, "Peer STA does not support the requested cipher suite" },
  { 46, "Disassociated because authorized access limit reached" },
  { 47, "Disassociated due to external service requirements" },
  { 48, "Invalid FT Action frame count" },
  { 49, "Invalid pairwise master key identifier (PMKI)" },
  { 50, "Invalid MDE" },
  { 51, "Invalid FTE" },
  { 52, "SME cancels the mesh peering instance with the reason other than reaching the maximum number of peer mesh STAs" },
  { 53, "The mesh STA has reached the supported maximum number of peer mesh STAs" },
  { 54, "The received information violates the Mesh Configuration policy configured in the mesh STA profile" },
  { 55, "The mesh STA has received a Mesh Peering Close message requesting to close the mesh peering" },
  { 56, "The mesh STA has re-sent dot11MeshMaxRetries Mesh Peering Open messages, without receiving a Mesh Peering Confirm message" },
  { 57, "The confirmTimer for the mesh peering instance times out" },
  { 58, "The mesh STA fails to unwrap the GTK or the values in the wrapped contents do not match" },
  { 59, "The mesh STA receives inconsistent information about the mesh parameters between Mesh Peering Management frames" },
  { 60, "The mesh STA fails the authenticated mesh peering exchange because due to failure in selecting either the pairwise ciphersuite or group ciphersuite" },
  { 61, "The mesh STA does not have proxy information for this external destination" },
  { 62, "The mesh STA does not have forwarding information for this destination" },
  { 63, "The mesh STA determines that the link to the next hop of an active path in its forwarding information is no longer usable" },
  { 64, "The Deauthentication frame was sent because the MAC address of the STA already exists in the mesh BSS. See 11.3.3 (Additional mechanisms for an AP collocated with a mesh STA)" },
  { 65, "The mesh STA performs channel switch to meet regulatory requirements" },
  { 66, "The mesh STA performs channel switch with unspecified reason" },
  { 0,    NULL}
};
static value_string_ext ieee80211_reason_code_ext = VALUE_STRING_EXT_INIT(ieee80211_reason_code);

/* ************************************************************************* */
/*                         8.4.1.9 Status Code field                         */
/* ************************************************************************* */
static const value_string ieee80211_status_code[] = {
  {  0, "Successful" },
  {  1, "Unspecified failure" },
  {  2, "TDLS wakeup schedule rejected but alternative schedule provided" },
  {  3, "TDLS wakeup schedule rejected" },
  {  4, "Reserved" },
  {  5, "Security disabled" },
  {  6, "Unacceptable lifetime" },
  {  7, "Not in same BSS" },
  {  8, "Reserved" },
  {  9, "Reserved" },
  { 10, "Cannot support all requested capabilities in the Capability Information field" },
  { 11, "Reassociation denied due to inability to confirm that association exists" },
  { 12, "Association denied due to reason outside the scope of this standard" },
  { 13, "Responding STA does not support the specified authentication algorithm" },
  { 14, "Received an Authentication frame with authentication transaction sequence number out of expected sequence" },
  { 15, "Authentication rejected because of challenge failure" },
  { 16, "Authentication rejected due to timeout waiting for next frame in sequence" },
  { 17, "Association denied because AP is unable to handle additional associated STAs" },
  { 18, "Association denied due to requesting STA not supporting all of the data rates in the BSSBasicRateSet parameter" },
  { 19, "Association denied due to requesting STA not supporting the short preamble option" },
  { 20, "Association denied due to requesting STA not supporting the PBCC modulation option" },
  { 21, "Association denied due to requesting STA not supporting the Channel Agility option" },
  { 22, "Association request rejected because Spectrum Management capability is required" },
  { 23, "Association request rejected because the information in the Power Capability element is unacceptable" },
  { 24, "Association request rejected because the information in the Supported Channels element is unacceptable" },
  { 25, "Association denied due to requesting STA not supporting the Short Slot Time option" },
  { 26, "Association denied due to requesting STA not supporting the DSSS-OFDM option" },
  { 27, "Reserved Association denied because the requesting STA does not support HT features" },
  { 28, "R0KH unreachable" },
  { 29, "Association denied because the requesting STA does not support the phased coexistence operation (PCO) transition time required by the AP" },
  { 30, "Association request rejected temporarily; try again later" },
  { 31, "Robust Management frame policy violation" },
  { 32, "Unspecified, QoS-related failure" },
  { 33, "Association denied because QoS AP or PCP has insufficient bandwidth to handle another QoS STA" },
  { 34, "Association denied due to excessive frame loss rates and/or poor conditions on current operating channel" },
  { 35, "Association (with QoS BSS) denied because the requesting STA does not support the QoS facility" },
  { 36, "Reserved" },
  { 37, "The request has been declined" },
  { 38, "The request has not been successful as one or more parameters have invalid values" },
  { 39, "The allocation or TS has not been created because the request cannot be honored; however, a suggested TSPEC/DMG TSPEC is provided so that the initiating STA may attempt to set another allocation or TS with the suggested changes to the TSPEC/DMG TSPEC" },
  { 40, "Invalid information element, i.e., an information element defined in this standard for which the content does not meet the specifications in Clause 7" },
  { 41, "Invalid group cipher" },
  { 42, "Invalid pairwise cipher" },
  { 43, "Invalid AKMP" },
  { 44, "Unsupported RSN information element version" },
  { 45, "Invalid RSN information element capabilities" },
  { 46, "Cipher suite rejected because of security policy" },
  { 47, "The TS per allocation has not been created; however, the PCP/HC may be capable of creating a TS or allocation, in response to a request, after the time indicated in the TS Delay element" },
  { 48, "Direct link is not allowed in the BSS by policy" },
  { 49, "The Destination STA is not present within this BSS" },
  { 50, "The Destination STA is not a QoS STA" },
  { 51, "Association denied because the ListenInterval is too large" },
  { 52, "Invalid FT Action frame count" },
  { 53, "Invalid pairwise master key identifier (PMKID)" },
  { 54, "Invalid MDIE" },
  { 55, "Invalid FTIE" },
  { 56, "Requested TCLAS processing is not supported by the PCP/AP" },
  { 57, "The PCP/AP has insufficient TCLAS processing resources to satisfy the request" },
  { 58, "The TS has not been created because the request cannot be honored; however, the PCP/HC suggests the STA to transition to other BSSs to setup the TS" },
  { 59, "GAS Advertisement Protocol not supported" },
  { 60, "No outstanding GAS request" },
  { 61, "GAS Response not received from the Advertisement Server" },
  { 62, "STA timed out waiting for GAS Query Response" },
  { 63, "GAS Response is larger than query response length limit" },
  { 64, "Request refused because home network does not support request" },
  { 65, "Advertisement Server in the network is not currently reachable" },
  { 66, "Reserved" },
  { 67, "Request refused due to permissions received via SSPN interface" },
  { 68, "Request refused because PCP/AP does not support unauthenticated access" },
  { 69, "Reserved" },
  { 70, "Reserved" },
  { 71, "Reserved" },
  { 72, "Invalid contents of RSNIE" },
  { 73, "U-APSD Coexistence is not supported" },
  { 74, "Requested U-APSD Coexistence mode is not supported" },
  { 75, "Requested Interval/Duration value cannot be supported with U-APSD Coexistence" },
  { 76, "Authentication is rejected because an Anti-Clogging Token is required" },
  { 77, "Authentication is rejected because the offered finite cyclic group is not supported" },
  { 78, "The TBTT adjustment request has not been successful because the STA could not find an alternative TBTT" },
  { 79, "Transmission failure" },
  { 80, "Requested TCLAS Not Supported" },
  { 81, "TCLAS Resources Exhausted" },
  { 82, "Rejected with Suggested BSS Transition" },
  { 83, "Reject with recommended schedule" },
  { 84, "Reject because no wakeup schedule specified" },
  { 85, "Success, the destination STA is in power save mode" },
  { 86, "FST pending, in process of admitting FST session" },
  { 87, "performing FST now" },
  { 88, "FST pending, gap(s) in Block Ack window" },
  { 89, "Reject because of U-PID setting" },
  { 92, "(Re)association refused for some external reason" },
  { 93, "(Re)association refused because of memory limits at the AP" },
  { 94, "(Re)association refused because emergency services are not supported at the AP" },
  { 95, "GAS query response not yet received" },
  { 96, "Reject since the request is for transition to a frequency band subject to DSE procedures and FST initiator is a dependent STA" },
  { 97, "Reserved" },
  { 98, "Reserved" },
  { 99, "The association has been denied; however, one or more Multi-band elements are included that can be used by the receiving STA to join the BSS" },
  { 100, "The request failed due to a reservation conflict" },
  { 101, "The request failed due to exceeded MAF limit" },
  { 102, "The request failed due to exceeded MCCA track limit" },
  { 103, "Association denied because the information in the Spectrum Management field is unacceptable" },
  { 104, "Association denied because the requesting STA does not support VHT features" },
  { 105, "Enablement denied" },
  { 106, "Enablement denied due to restriction from an authorized GDB" },
  { 107, "Authorization deenabled" },
  { 112, "Authentication rejected due to FILS authentication failure" },
  { 113, "Authentication rejected due to unknown Authentication Server" },
  { 0,    NULL}
};
static value_string_ext ieee80211_status_code_ext = VALUE_STRING_EXT_INIT(ieee80211_status_code);

static const value_string ieee80211_transition_reasons[] = {
  { 0, "Unspecified" },
  { 1, "Excessive frame loss rates and/or poor conditions" },
  { 2, "Excessive delay for current traffic streams" },
  { 3, "Insufficient QoS capacity for current traffic streams (TSPEC rejected)" },
  { 4, "First association to ESS (the association initiated by an Association Request frame instead of a Reassociation Request frame)" },
  { 5, "Load balancing" },
  { 6, "Better AP found" },
  { 7, "Deauthenticated or Disassociated from the previous AP" },
  { 8, "AP failed IEEE 802.1X EAP Authentication" },
  { 9, "AP failed 4-way handshake" },
  { 10, "Received too many replay counter failures" },
  { 11, "Received too many data MIC failures" },
  { 12, "Exceeded maximum number of retransmissions" },
  { 13, "Received too many broadcast disassociations" },
  { 14, "Received too many broadcast deauthentications" },
  { 15, "Previous transition failed" },
  { 16, "Low RSSI" },
  { 17, "Roam from a non-IEEE-802.11 system" },
  { 18, "Transition due to received BSS Transition Request frame" },
  { 19, "Preferred BSS transition candidate list included" },
  { 20, "Leaving ESS" },
  { 0,    NULL}
};
/* ************************************************************************* */
/*                         Frame types, and their names                      */
/* ************************************************************************* */
static const value_string frame_type_subtype_vals[] = {
  {MGT_ASSOC_REQ,             "Association Request"},
  {MGT_ASSOC_RESP,            "Association Response"},
  {MGT_REASSOC_REQ,           "Reassociation Request"},
  {MGT_REASSOC_RESP,          "Reassociation Response"},
  {MGT_PROBE_REQ,             "Probe Request"},
  {MGT_PROBE_RESP,            "Probe Response"},
  {MGT_MEASUREMENT_PILOT,     "Measurement Pilot"},
  {MGT_BEACON,                "Beacon frame"},
  {MGT_ATIM,                  "ATIM"},
  {MGT_DISASS,                "Disassociate"},
  {MGT_AUTHENTICATION,        "Authentication"},
  {MGT_DEAUTHENTICATION,      "Deauthentication"},
  {MGT_ACTION,                "Action"},
  {MGT_ACTION_NO_ACK,         "Action No Ack"},
  {MGT_ARUBA_WLAN,            "Aruba Management"},

  {CTRL_TRIGGER,              "Trigger"},
  {CTRL_BEAMFORM_RPT_POLL,    "Beamforming Report Poll"},
  {CTRL_VHT_NDP_ANNC,         "VHT/HE NDP Announcement"},
  {CTRL_CONTROL_WRAPPER,      "Control Wrapper"},
  {CTRL_BLOCK_ACK_REQ,        "802.11 Block Ack Req"},
  {CTRL_BLOCK_ACK,            "802.11 Block Ack"},
  {CTRL_PS_POLL,              "Power-Save poll"},
  {CTRL_RTS,                  "Request-to-send"},
  {CTRL_CTS,                  "Clear-to-send"},
  {CTRL_ACKNOWLEDGEMENT,      "Acknowledgement"},
  {CTRL_CFP_END,              "CF-End (Control-frame)"},
  {CTRL_CFP_ENDACK,           "CF-End + CF-Ack (Control-frame)"},

  {DATA,                      "Data"},
  {DATA_CF_ACK,               "Data + CF-Ack"},
  {DATA_CF_POLL,              "Data + CF-Poll"},
  {DATA_CF_ACK_POLL,          "Data + CF-Ack + CF-Poll"},
  {DATA_NULL_FUNCTION,        "Null function (No data)"},
  {DATA_CF_ACK_NOD,           "Acknowledgement (No data)"},
  {DATA_CF_POLL_NOD,          "CF-Poll (No data)"},
  {DATA_CF_ACK_POLL_NOD,      "CF-Ack/Poll (No data)"},
  {DATA_QOS_DATA,             "QoS Data"},
  {DATA_QOS_DATA_CF_ACK,      "QoS Data + CF-Acknowledgment"},
  {DATA_QOS_DATA_CF_POLL,     "QoS Data + CF-Poll"},
  {DATA_QOS_DATA_CF_ACK_POLL, "QoS Data + CF-Ack + CF-Poll"},
  {DATA_QOS_NULL,             "QoS Null function (No data)"},
  {DATA_QOS_CF_POLL_NOD,      "QoS CF-Poll (No Data)"},
  {DATA_QOS_CF_ACK_POLL_NOD,  "QoS CF-Ack + CF-Poll (No data)"},

  {EXTENSION_DMG_BEACON,      "DMG Beacon"},
  {EXTENSION_S1G_BEACON,      "S1G Beacon"},

  {CTRL_POLL,                 "Poll"},
  {CTRL_SPR,                  "Service Period Request"},
  {CTRL_GRANT,                "Grant"},
  {CTRL_DMG_CTS,              "DMG Clear-to-send"},
  {CTRL_DMG_DTS,              "DMG Denial-to-send"},
  {CTRL_GRANT_ACK,            "Grant Acknowledgment"},
  {CTRL_SSW,                  "Sector Sweep"},
  {CTRL_SSW_FEEDBACK,         "Sector Sweep Feedback"},
  {CTRL_SSW_ACK,              "Sector Sweep Acknowledgment"},

  {0,                         NULL}
};
static value_string_ext frame_type_subtype_vals_ext = VALUE_STRING_EXT_INIT(frame_type_subtype_vals);

/* ************************************************************************* */
/*                 802.1D Tag Name (by WME Access Category Names)            */
/* ************************************************************************* */
static const value_string ieee80211_qos_tags_acs[] = {
  { 0, "Best Effort (Best Effort)" },
  { 1, "Background (Background)" },
  { 2, "Spare (Background)" },
  { 3, "Excellent Effort (Best Effort)" },
  { 4, "Controlled Load (Video)" },
  { 5, "Video (Video)" },
  { 6, "Voice (Voice)" },
  { 7, "Network Control (Voice)" },
  { 0, NULL }
};

/* ************************************************************************* */
/*                   WME Access Category Names (by WME ACI)                  */
/* ************************************************************************* */
static const value_string wme_acs[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Video" },
  { 3, "Voice" },
  { 0, NULL }
};

/* ************************************************************************* */
/*                  Aruba Management Type                                    */
/* ************************************************************************* */
static const value_string aruba_mgt_typevals[] = {
  { 0x0001,       "Hello" },
  { 0x0002,       "Probe" },
  { 0x0003,       "MTU" },
  { 0x0004,       "Ageout" },
  { 0x0005,       "Heartbeat" },
  { 0x0006,       "Deauth" },
  { 0x0007,       "Disassoc" },
  { 0x0008,       "Probe response" },
  { 0x0009,       "Tunnel update" },
  { 0x000A,       "Laser beam active" },
  { 0x000B,       "Client IP" },
  { 0x000C,       "Laser beam active v2" },
  { 0x000D,       "AP statistics" },
  { 0,            NULL }
};
static value_string_ext aruba_mgt_typevals_ext = VALUE_STRING_EXT_INIT(aruba_mgt_typevals);

/*** Begin: Action Fixed Parameter ***/
#define CAT_SPECTRUM_MGMT          0
#define CAT_QOS                    1
#define CAT_DLS                    2
#define CAT_BLOCK_ACK              3
#define CAT_PUBLIC                 4
#define CAT_RADIO_MEASUREMENT      5
#define CAT_FAST_BSS_TRANSITION    6
#define CAT_HT                     7
#define CAT_SA_QUERY               8
#define CAT_PUBLIC_PROTECTED       9
#define CAT_WNM                   10
#define CAT_UNPROTECTED_WNM       11
#define CAT_TDLS                  12
#define CAT_MESH                  13
#define CAT_MULTIHOP              14
#define CAT_SELF_PROTECTED        15
#define CAT_DMG                   16
#define CAT_MGMT_NOTIFICATION     17
#define CAT_FAST_SESSION_TRANSFER 18
#define CAT_ROBUST_AV_STREAMING   19
#define CAT_UNPROTECTED_DMG       20
#define CAT_VHT                   21
#define CAT_S1G                   22
#define CAT_FILS                  26
#define CAT_HE                    30
#define CAT_PROTECTED_HE          31
#define CAT_VENDOR_SPECIFIC_PROTECTED 126
#define CAT_VENDOR_SPECIFIC     127

#define CAT_MESH_LINK_METRIC               31
#define CAT_MESH_PATH_SELECTION            32
#define CAT_MESH_INTERWORKING              33
#define CAT_MESH_RESOURCE_COORDINATION     34
#define CAT_MESH_SECURITY_ARCHITECTURE     35

#define SM_ACTION_MEASUREMENT_REQUEST   0
#define SM_ACTION_MEASUREMENT_REPORT    1
#define SM_ACTION_TPC_REQUEST           2
#define SM_ACTION_TPC_REPORT            3
#define SM_ACTION_CHAN_SWITCH_ANNC      4
#define SM_ACTION_EXT_CHAN_SWITCH_ANNC  5

#define QOS_ACTION_ADDTS_REQUEST     0
#define QOS_ACTION_ADDTS_RESPONSE    1
#define QOS_ACTION_DELTS             2
#define QOS_ACTION_SCHEDULE      3
#define QOS_ACTION_MAP_CONFIGURE 4

#define DLS_ACTION_REQUEST       0
#define DLS_ACTION_RESPONSE      1
#define DLS_ACTION_TEARDOWN      2

#define BA_ADD_BLOCK_ACK_REQUEST    0
#define BA_ADD_BLOCK_ACK_RESPONSE   1
#define BA_DELETE_BLOCK_ACK         2

/* Keep in sync with PPA_* defines */
#define PA_20_40_BSS_COEXISTENCE_MANAGEMENT 0
#define PA_DSE_ENABLEMENT                   1
#define PA_DSE_DEENABLEMENT                 2
#define PA_DSE_REG_LOC_ANNOUNCEMENT         3
#define PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT  4
#define PA_DSE_MEASUREMENT_REQUEST          5
#define PA_DSE_MEASUREMENT_REPORT           6
#define PA_MEASUREMENT_PILOT                7
#define PA_DSE_POWER_CONSTRAINT             8
#define PA_VENDOR_SPECIFIC                  9
#define PA_GAS_INITIAL_REQUEST             10
#define PA_GAS_INITIAL_RESPONSE            11
#define PA_GAS_COMEBACK_REQUEST            12
#define PA_GAS_COMEBACK_RESPONSE           13
#define PA_TDLS_DISCOVERY_RESPONSE         14
#define PA_LOCATION_TRACK_NOTIFICATION     15
#define PA_QAB_REQUEST                     16
#define PA_QAB_RESPONSE                    17

/* Keep in sync with PA_* defines */
#define PPA_DSE_ENABLEMENT                   1
#define PPA_DSE_DEENABLEMENT                 2
#define PPA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT  4
#define PPA_DSE_MEASUREMENT_REQUEST          5
#define PPA_DSE_MEASUREMENT_REPORT           6
#define PPA_DSE_POWER_CONSTRAINT             8
#define PPA_VENDOR_SPECIFIC                  9
#define PPA_GAS_INITIAL_REQUEST             10
#define PPA_GAS_INITIAL_RESPONSE            11
#define PPA_GAS_COMEBACK_REQUEST            12
#define PPA_GAS_COMEBACK_RESPONSE           13
#define PPA_QAB_REQUEST                     16
#define PPA_QAB_RESPONSE                    17

#define HT_ACTION_NOTIFY_CHAN_WIDTH           0
#define HT_ACTION_SM_PWR_SAVE                 1
#define HT_ACTION_PSMP_ACTION                 2
#define HT_ACTION_SET_PCO_PHASE               3
#define HT_ACTION_MIMO_CSI                    4
#define HT_ACTION_MIMO_BEAMFORMING            5
#define HT_ACTION_MIMO_COMPRESSED_BEAMFORMING 6
#define HT_ACTION_ANT_SEL_FEEDBACK            7
#define HT_ACTION_HT_INFO_EXCHANGE            8

#define DMG_ACTION_PWR_SAVE_CONFIG_REQ           0
#define DMG_ACTION_PWR_SAVE_CONFIG_RES           1
#define DMG_ACTION_INFO_REQ                      2
#define DMG_ACTION_INFO_RES                      3
#define DMG_ACTION_HANDOVER_REQ                  4
#define DMG_ACTION_HANDOVER_RES                  5
#define DMG_ACTION_DTP_REQ                       6
#define DMG_ACTION_DTP_RES                       7
#define DMG_ACTION_RELAY_SEARCH_REQ              8
#define DMG_ACTION_RELAY_SEARCH_RES              9
#define DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ 10
#define DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES 11
#define DMG_ACTION_RLS_REQ                       12
#define DMG_ACTION_RLS_RES                       13
#define DMG_ACTION_RLS_ANNOUNCE                  14
#define DMG_ACTION_RLS_TEARDOWN                  15
#define DMG_ACTION_RELAY_ACK_REQ                 16
#define DMG_ACTION_RELAY_ACK_RES                 17
#define DMG_ACTION_TPA_REQ                       18
#define DMG_ACTION_TPA_RES                       19
#define DMG_ACTION_TPA_REP                       20
#define DMG_ACTION_ROC_REQ                       21
#define DMG_ACTION_ROC_RES                       22

#define UNPROTECTED_DMG_ANNOUNCE                 0
#define UNPROTECTED_DMG_BRP                      1

#define FST_SETUP_REQUEST                        0
#define FST_SETUP_RESPONSE                       1
#define FST_TEAR_DOWN                            2
#define FST_ACK_REQUEST                          3
#define FST_ACK_RESPONSE                         4
#define FST_ON_CHANNEL_TUNNEL_REQUEST            5

/* IEEE Std 802.11r-2008, 7.4.8, Table 7-57g */
#define FT_ACTION_REQUEST               1
#define FT_ACTION_RESPONSE              2
#define FT_ACTION_CONFIRM               3
#define FT_ACTION_ACK                   4

/* SA Query Action frame codes (IEEE 802.11w-2009, 7.4.9) */
#define SA_QUERY_REQUEST                0
#define SA_QUERY_RESPONSE               1

/* IEEE Std 802.11z-2010, 7.4.11, Table 7-57v1 */
#define TDLS_SETUP_REQUEST              0
#define TDLS_SETUP_RESPONSE             1
#define TDLS_SETUP_CONFIRM              2
#define TDLS_TEARDOWN                   3
#define TDLS_PEER_TRAFFIC_INDICATION    4
#define TDLS_CHANNEL_SWITCH_REQUEST     5
#define TDLS_CHANNEL_SWITCH_RESPONSE    6
#define TDLS_PEER_PSM_REQUEST           7
#define TDLS_PEER_PSM_RESPONSE          8
#define TDLS_PEER_TRAFFIC_RESPONSE      9
#define TDLS_DISCOVERY_REQUEST          10

/* IEEE Std 802.11-2012, 8.5.7.1, Table 8-206 */
#define RM_ACTION_RADIO_MEASUREMENT_REQUEST         0
#define RM_ACTION_RADIO_MEASUREMENT_REPORT          1
#define RM_ACTION_LINK_MEASUREMENT_REQUEST          2
#define RM_ACTION_LINK_MEASUREMENT_REPORT           3
#define RM_ACTION_NEIGHBOR_REPORT_REQUEST           4
#define RM_ACTION_NEIGHBOR_REPORT_RESPONSE          5

/* 11s draft 12.0, table 7-57v30 */
#define MESH_ACTION_LINK_METRIC_REPORT              0
#define MESH_ACTION_HWMP                            1
#define MESH_ACTION_GATE_ANNOUNCE                   2
#define MESH_ACTION_CONGESTION_CTL                  3
#define MESH_ACTION_MCCA_SETUP_REQUEST              4
#define MESH_ACTION_MCCA_SETUP_REPLY                5
#define MESH_ACTION_MCCA_ADV_REQUEST                6
#define MESH_ACTION_MCCA_ADV                        7
#define MESH_ACTION_MCCA_TEARDOWN                   8
#define MESH_ACTION_TBTT_ADJ_REQUEST                9
#define MESH_ACTION_TBTT_ADJ_RESPONSE              10

/* 11s draft 12.0, table 7-57v42: Multihop Action field values */
#define MULTIHOP_ACTION_PROXY_UPDATE                0
#define MULTIHOP_ACTION_PROXY_UPDATE_CONF           1

/* 11s draft 12.0, table 7-57v24: Self-protected Action field values */
#define SELFPROT_ACTION_MESH_PEERING_OPEN           1
#define SELFPROT_ACTION_MESH_PEERING_CONFIRM        2
#define SELFPROT_ACTION_MESH_PEERING_CLOSE          3
#define SELFPROT_ACTION_MESH_GROUP_KEY_INFORM       4
#define SELFPROT_ACTION_MESH_GROUP_KEY_ACK          5

/* 11s draft 12.0, table 7-43bj6: Mesh Peering Protocol Identifier field values */
#define MESH_PEERING_PROTO_MGMT                     0
#define MESH_PEERING_PROTO_AMPE                     1
#define MESH_PEERING_PROTO_VENDOR                 255

/* Vendor actions */
/* MARVELL */
#define MRVL_ACTION_MESH_MANAGEMENT     1

#define MRVL_MESH_MGMT_ACTION_RREQ      0
#define MRVL_MESH_MGMT_ACTION_RREP      1
#define MRVL_MESH_MGMT_ACTION_RERR      2
#define MRVL_MESH_MGMT_ACTION_PLDM      3

#define ANQP_INFO_ANQP_QUERY_LIST                256
#define ANQP_INFO_ANQP_CAPAB_LIST                257
#define ANQP_INFO_VENUE_NAME_INFO                258
#define ANQP_INFO_EMERGENCY_CALL_NUMBER_INFO     259
#define ANQP_INFO_NETWORK_AUTH_TYPE_INFO         260
#define ANQP_INFO_ROAMING_CONSORTIUM_LIST        261
#define ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO 262
#define ANQP_INFO_NAI_REALM_LIST                 263
#define ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO     264
#define ANQP_INFO_AP_GEOSPATIAL_LOCATION         265
#define ANQP_INFO_AP_CIVIC_LOCATION              266
#define ANQP_INFO_AP_LOCATION_PUBLIC_ID_URI      267
#define ANQP_INFO_DOMAIN_NAME_LIST               268
#define ANQP_INFO_EMERGENCY_ALERT_ID_URI         269
#define ANQP_INFO_TDLS_CAPAB_INFO                270
#define ANQP_INFO_EMERGENCY_NAI                  271
#define ANQP_INFO_NEIGHBOR_REPORT                272
#define ANQP_INFO_QUERY_AP_LIST                  273
#define ANQP_INFO_AP_LIST_RESPONSE               274
#define ANQP_INFO_FILS_REALM_INFO                275
#define ANQP_INFO_CAG                            276
#define ANQP_INFO_VENUE_URL                      277
#define ANQP_INFO_ADVICE_OF_CHARGE               278
#define ANQP_INFO_LOCAL_CONTENT                  279
#define ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST    56797

/* ANQP information ID - IEEE Std 802.11u-2011 - Table 7-43bk */
static const value_string anqp_info_id_vals[] = {
  {ANQP_INFO_ANQP_QUERY_LIST, "ANQP Query list"},
  {ANQP_INFO_ANQP_CAPAB_LIST, "ANQP Capability list"},
  {ANQP_INFO_VENUE_NAME_INFO, "Venue Name information"},
  {ANQP_INFO_EMERGENCY_CALL_NUMBER_INFO, "Emergency Call Number information"},
  {ANQP_INFO_NETWORK_AUTH_TYPE_INFO,
   "Network Authentication Type information"},
  {ANQP_INFO_ROAMING_CONSORTIUM_LIST, "Roaming Consortium list"},
  {ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO,
   "IP Address Type Availability information"},
  {ANQP_INFO_NAI_REALM_LIST, "NAI Realm list"},
  {ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO, "3GPP Cellular Network information"},
  {ANQP_INFO_AP_GEOSPATIAL_LOCATION, "AP Geospatial Location"},
  {ANQP_INFO_AP_CIVIC_LOCATION, "AP Civic Location"},
  {ANQP_INFO_AP_LOCATION_PUBLIC_ID_URI, "AP Location Public Identifier URI"},
  {ANQP_INFO_DOMAIN_NAME_LIST, "Domain Name list"},
  {ANQP_INFO_EMERGENCY_ALERT_ID_URI, "Emergency Alert Identifier URI"},
  {ANQP_INFO_TDLS_CAPAB_INFO, "TDLS Capability information"},
  {ANQP_INFO_EMERGENCY_NAI, "Emergency NAI"},
  {ANQP_INFO_NEIGHBOR_REPORT, "Neighbor Report"},
  {ANQP_INFO_QUERY_AP_LIST, "Query AP List"},
  {ANQP_INFO_AP_LIST_RESPONSE, "AP List Response"},
  {ANQP_INFO_FILS_REALM_INFO, "FILS Realm Info"},
  {ANQP_INFO_CAG, "CAG"},
  {ANQP_INFO_VENUE_URL, "Venue URL"},
  {ANQP_INFO_ADVICE_OF_CHARGE, "Advice of Charge"},
  {ANQP_INFO_LOCAL_CONTENT, "Local Content"},
  {ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST, "ANQP vendor-specific list"},
  {0, NULL}
};
static value_string_ext anqp_info_id_vals_ext = VALUE_STRING_EXT_INIT(anqp_info_id_vals);

/* IEEE 802.11v - WNM Action field values */
enum wnm_action {
  WNM_EVENT_REQ = 0,
  WNM_EVENT_REPORT = 1,
  WNM_DIAGNOSTIC_REQ = 2,
  WNM_DIAGNOSTIC_REPORT = 3,
  WNM_LOCATION_CFG_REQ = 4,
  WNM_LOCATION_CFG_RESP = 5,
  WNM_BSS_TRANS_MGMT_QUERY = 6,
  WNM_BSS_TRANS_MGMT_REQ = 7,
  WNM_BSS_TRANS_MGMT_RESP = 8,
  WNM_FMS_REQ = 9,
  WNM_FMS_RESP = 10,
  WNM_COLLOCATED_INTERFERENCE_REQ = 11,
  WNM_COLLOCATED_INTERFERENCE_REPORT = 12,
  WNM_TFS_REQ = 13,
  WNM_TFS_RESP = 14,
  WNM_TFS_NOTIFY = 15,
  WNM_SLEEP_MODE_REQ = 16,
  WNM_SLEEP_MODE_RESP = 17,
  WNM_TIM_BROADCAST_REQ = 18,
  WNM_TIM_BROADCAST_RESP = 19,
  WNM_QOS_TRAFFIC_CAPAB_UPDATE = 20,
  WNM_CHANNEL_USAGE_REQ = 21,
  WNM_CHANNEL_USAGE_RESP = 22,
  WNM_DMS_REQ = 23,
  WNM_DMS_RESP = 24,
  WNM_TIMING_MEASUREMENT_REQ = 25,
  WNM_NOTIFICATION_REQ = 26,
  WNM_NOTIFICATION_RESP = 27
};

static const value_string wnm_action_codes[] = {
  { WNM_EVENT_REQ, "Event Request" },
  { WNM_EVENT_REPORT, "Event Report" },
  { WNM_DIAGNOSTIC_REQ, "Diagnostic Request" },
  { WNM_DIAGNOSTIC_REPORT, "Diagnostic Report" },
  { WNM_LOCATION_CFG_REQ, "Location Configuration Request" },
  { WNM_LOCATION_CFG_RESP, "Location Configuration Response" },
  { WNM_BSS_TRANS_MGMT_QUERY, "BSS Transition Management Query" },
  { WNM_BSS_TRANS_MGMT_REQ, "BSS Transition Management Request" },
  { WNM_BSS_TRANS_MGMT_RESP, "BSS Transition Management Response" },
  { WNM_FMS_REQ, "FMS Request" },
  { WNM_FMS_RESP, "FMS Response" },
  { WNM_COLLOCATED_INTERFERENCE_REQ, "Collocated Interference Request" },
  { WNM_COLLOCATED_INTERFERENCE_REPORT, "Collocated Interference Report" },
  { WNM_TFS_REQ, "TFS Request" },
  { WNM_TFS_RESP, "TFS Response" },
  { WNM_TFS_NOTIFY, "TFS Notify" },
  { WNM_SLEEP_MODE_REQ, "WNM-Sleep Mode Request" },
  { WNM_SLEEP_MODE_RESP, "WNM-Sleep Mode Response" },
  { WNM_TIM_BROADCAST_REQ, "TIM Broadcast Request" },
  { WNM_TIM_BROADCAST_RESP, "TIM Broadcast Response" },
  { WNM_QOS_TRAFFIC_CAPAB_UPDATE, "QoS Traffic Capability Update" },
  { WNM_CHANNEL_USAGE_REQ, "Channel Usage Request" },
  { WNM_CHANNEL_USAGE_RESP, "Channel Usage Response" },
  { WNM_DMS_REQ, "DMS Request" },
  { WNM_DMS_RESP, "DMS Response" },
  { WNM_TIMING_MEASUREMENT_REQ, "Timing Measurement Request" },
  { WNM_NOTIFICATION_REQ, "WNM-Notification Request" },
  { WNM_NOTIFICATION_RESP, "WNM-Notification Response" },
  { 0, NULL }
};
static value_string_ext wnm_action_codes_ext = VALUE_STRING_EXT_INIT(wnm_action_codes);

enum unprotected_wnm_action {
  UNPROTECTED_WNM_TIM = 0,
  UNPROTECTED_WNM_TIMING_MEASUREMENT = 1
};

static const value_string unprotected_wnm_action_codes[] = {
  { UNPROTECTED_WNM_TIM, "TIM" },
  { UNPROTECTED_WNM_TIMING_MEASUREMENT, "Timing Measurement" },
  { 0, NULL }
};
static value_string_ext unprotected_wnm_action_codes_ext = VALUE_STRING_EXT_INIT(unprotected_wnm_action_codes);

static const value_string wnm_notification_types[] = {
  { 0, "Firmware Update Notification" },
  { 1, "Reserved for use by WFA" },
  { 221, "Vendor Specific" },
  { 0, NULL }
};

static value_string_ext wnm_notification_types_ext =
  VALUE_STRING_EXT_INIT(wnm_notification_types);

/*** End: Action Fixed Parameter ***/

static const value_string ieee80211_tag_measure_request_type_flags[] = {
  {0x00, "Basic Request"},
  {0x01, "Clear Channel Assessment (CCA) Request"},
  {0x02, "Receive Power Indication (RPI) Histogram Request"},
  {0x03, "Channel Load Request"},
  {0x04, "Noise Histogram Request"},
  {0x05, "Beacon Request"},
  {0x06, "Frame Request"},
  {0x07, "STA Statistics Request"},
  {0x08, "Location Configuration Indication (LCI) Request"},
  {0x09, "Transmit Stream Measurement Request"},
  {0x0A, "Measurement Pause Request"},
  {0x00, NULL}
};
static value_string_ext ieee80211_tag_measure_request_type_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_request_type_flags);

static const value_string ieee80211_tag_measure_report_type_flags[] = {
  { 0x00, "Basic Report" },
  { 0x01, "Clear Channel Assessment (CCA) Report" },
  { 0x02, "Receive Power Indication (RPI) Histogram Report" },
  { 0x03, "Channel Load Report" },
  { 0x04, "Noise Histogram Report" },
  { 0x05, "Beacon Report" },
  { 0x06, "Frame Report" },
  { 0x07, "STA Statistics Report" },
  { 0x08, "Location Configuration Information (LCI) Report" },
  { 0x09, "Transmit Stream Measurement Report" },
  { 0x00, NULL }
};
static value_string_ext ieee80211_tag_measure_report_type_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_report_type_flags);

static const true_false_string ieee80211_tag_measure_report_frame_info_frame_type_flag = {
  "Measurement Pilot Frame",
  "Beacon/Probe Response Frame"
};

static const true_false_string ieee80211_tag_measure_map_field_bss_flag = {
  "At least one valid MPDU was received by another BSS or IBSS during the measurement period.",
  "No valid MPDUs were received from another BSS or IBSS during the measurement period."
};

static const value_string ieee80211_tag_measure_request_measurement_mode_flags[] = {
  { 0x00, "Passive" },
  { 0x01, "Active" },
  { 0x02, "Beacon Table" },
  { 0x00, NULL }
};

#define MEASURE_REQ_BEACON_SUB_SSID 0
#define MEASURE_REQ_BEACON_SUB_BRI 1
#define MEASURE_REQ_BEACON_SUB_RD 2
#define MEASURE_REQ_BEACON_SUB_REQUEST 10
#define MEASURE_REQ_BEACON_SUB_APCP 51
#define MEASURE_REQ_BEACON_SUB_VS 221

static const value_string ieee80211_tag_measure_request_beacon_sub_id_flags[] = {
  { MEASURE_REQ_BEACON_SUB_SSID, "SSID" },
  { MEASURE_REQ_BEACON_SUB_BRI, "Beacon Reporting Information" },
  { MEASURE_REQ_BEACON_SUB_RD, "Reporting Detail" },
  { MEASURE_REQ_BEACON_SUB_REQUEST, "Request" },
  { MEASURE_REQ_BEACON_SUB_APCP, "AP Channel Report" },
  { MEASURE_REQ_BEACON_SUB_VS, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_beacon_sub_bri_reporting_condition_flags[] = {
  { 0x00, "Report to be issued after each measurement." },
  { 0x01, "The measured RCPI level is greater than an absolute threshold." },
  { 0x02, "The measured RCPI level is less than an absolute threshold." },
  { 0x03, "The measured RSNI level is greater than an absolute threshold." },
  { 0x04, "The measured RSNI level is less than an absolute threshold." },
  { 0x05, "The measured RCPI level is greater than a threshold defined by an offset from the serving AP's reference RCPI." },
  { 0x06, "The measured RCPI level is less than a threshold defined by an offset from the serving AP's reference RCPI." },
  { 0x07, "The measured RSNI level is greater than a threshold defined by an offset from the serving AP's reference RSNI." },
  { 0x08, "The measured RSNI level is less than a threshold defined by an offset from the serving AP's reference RSNI." },
  { 0x09, "The measured RCPI level is in a range bound by the serving AP's reference RCPI and an offset from the serving AP's reference RCPI." },
  { 0x0a, "The measured RSNI level is in a range bound by the serving AP's reference RSNI and an offset from the serving AP's reference RSNI." },
  { 0xfe, "Report not required to be issued" },
  { 0x00, NULL }
};

static const value_string ieee80211_tag_measure_request_beacon_sub_reporting_detail_flags[] = {
  { 0, "No fixed length fields or elements" },
  { 1, "All fixed length fields and any requested elements in the Request information element if present" },
  { 2, "All fixed length fields and elements (default, used when Reporting Detail subelement is not included in Beacon Request" },
  { 0x00, NULL }
};

static const value_string ieee80211_tag_measure_request_group_id_flags[] = {
  { 0x00, "STA Counters from dot11CountersTable" },
  { 0x01, "STA Counters from dot11MacStatistics group" },
  { 0x02, "QoS STA Counters for UP0 from dot11QosCountersTable" },
  { 0x03, "QoS STA Counters for UP1 from dot11QosCountersTable" },
  { 0x04, "QoS STA Counters for UP2 from dot11QosCountersTable" },
  { 0x05, "QoS STA Counters for UP3 from dot11QosCountersTable" },
  { 0x06, "QoS STA Counters for UP4 from dot11QosCountersTable" },
  { 0x07, "QoS STA Counters for UP5 from dot11QosCountersTable" },
  { 0x08, "QoS STA Counters for UP6 from dot11QosCountersTable" },
  { 0x09, "QoS STA Counters for UP7 from dot11QosCountersTable" },
  { 0x0a, "BSS Average Access Delays" },
  { 0x0b, "STA Counters from dot11A-MSDU Group" },
  { 0x0c, "STA Counters from dot11A-MPDU Group" },
  { 0x0d, "STA Counters from dot11 BAR, Channel Width, PSMP Group" },
  { 0x0e, "STA Counters from dot11Protection Group" },
  { 0x0f, "STBC Group" },
  { 0x00, NULL }
};
static value_string_ext ieee80211_tag_measure_request_group_id_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_request_group_id_flags);

static const value_string ieee80211_tclas_process_flag[] = {
  {0x00, "Incoming MSDU's higher layer parameters have to match to the parameters in all associated TCLAS elements."},
  {0x01, "Incoming MSDU's higher layer parameters have to match to at least one of the associated TCLAS elements."},
  {0x02, "Incoming MSDU's that do not belong to any other TS are classified to the TS for which this TCLAS Processing element is used. In this case, there will not be any associated TCLAS elements."},
  {0, NULL}
};


#define MEASURE_REQ_CHANNEL_LOAD_SUB_REPORTING_INFO 1

static const value_string ieee80211_tag_measure_request_channel_load_sub_id_vals[] = {
  { MEASURE_REQ_CHANNEL_LOAD_SUB_REPORTING_INFO, "Channel Load Reporting Information" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_channel_load_sub_reporting_condition_vals[] = {
  { 0x00, "Report to be issued after each measurement (default, used when Channel Load Reporting Information subelement is not included in Channel Load Request)." },
  { 0x01, "Report to be issued when measured Channel Load is equal to or greater than the reference value." },
  { 0x02, "Report to be issued when measured Channel Load is equal to or less than the reference value." },
  { 0x00, NULL}
};

#define MEASURE_REQ_NOISE_HISTOGRAM_SUB_REPORTING_INFO 1

static const value_string ieee80211_tag_measure_request_noise_histogram_sub_id_vals[] = {
  { MEASURE_REQ_NOISE_HISTOGRAM_SUB_REPORTING_INFO, "Noise Histogram Reporting Information" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_noise_histogram_sub_reporting_condition_vals[] = {
  { 0x00, "Report to be issued after each measurement (default, used when Noise Histogram Reporting Information subelement is not included in Noise Histogram Request)." },
  { 0x01, "Noise Histogram Report to be issued when measured ANPI is equal to or greater than the reference value." },
  { 0x02, "Noise Histogram Report to be issued when measured ANPI is equal to or less than the reference value." },
  { 0x00, NULL}
};

#define MEASURE_REP_REPORTED_FRAME_BODY 1

static const value_string ieee80211_tag_measure_report_beacon_sub_id_vals[] = {
  { MEASURE_REP_REPORTED_FRAME_BODY, "Reported Frame Body" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string frame_type[] = {
  {MGT_FRAME,       "Management frame"},
  {CONTROL_FRAME,   "Control frame"},
  {DATA_FRAME,      "Data frame"},
  {EXTENSION_FRAME, "Extension frame"},
  {0,               NULL}
};

static const value_string tofrom_ds[] = {
  {0,                     "Not leaving DS or network is operating "
    "in AD-HOC mode (To DS: 0 From DS: 0)"},
  {FLAG_TO_DS,            "Frame from STA to DS via an AP (To DS: 1 "
    "From DS: 0)"},
  {FLAG_FROM_DS,          "Frame from DS to a STA via AP(To DS: 0 "
    "From DS: 1)"},
  {FLAG_TO_DS|FLAG_FROM_DS, "WDS (AP to AP) or Mesh (MP to MP) Frame "
    "(To DS: 1 From DS: 1)"},
  {0, NULL}
};

static const true_false_string tods_flag = {
  "Frame is entering DS",
  "Frame is not entering DS"
};

static const true_false_string fromds_flag = {
  "Frame is exiting DS",
  "Frame is not exiting DS"
};

static const true_false_string more_fragments = {
  "More fragments follow",
  "This is the last fragment"
};

static const true_false_string retry_flags = {
  "Frame is being retransmitted",
  "Frame is not being retransmitted"
};

static const true_false_string pm_flags = {
  "STA will go to sleep",
  "STA will stay up"
};

static const true_false_string md_flags = {
  "Data is buffered for STA at AP",
  "No data buffered"
};

static const true_false_string protected_flags = {
  "Data is protected",
  "Data is not protected"
};

static const true_false_string order_flags = {
  "Strictly ordered",
  "Not strictly ordered"
};

static const true_false_string cf_ess_flags = {
  "Transmitter is an AP",
  "Transmitter is a STA"
};


static const true_false_string cf_privacy_flags = {
  "AP/STA can support WEP",
  "AP/STA cannot support WEP"
};

static const true_false_string cf_ibss_flags = {
  "Transmitter belongs to an IBSS",
  "Transmitter belongs to a BSS"
};

static const true_false_string eosp_flag = {
  "End of service period",
  "Service period"
};

static const true_false_string bit4_flag = {
  "Bits 8-15 of QoS Control field are Queue Size",
  "Bits 8-15 of QoS Control field are TXOP Duration Requested"
};

static const true_false_string ieee80211_qos_amsdu_present_flag = {
  "A-MSDU",
  "MSDU"
};

static const true_false_string csa_txrestrict_flags = {
  "Tx Restrict",
  "No Tx Restrict"
};

static const true_false_string csa_initiator_flags = {
  "Initiator",
  "Non Initiator"
};

static const true_false_string mesh_config_cap_power_save_level_flags = {
   "One of the peer-specific mesh power modes is deep sleep mode",
   "No one is in deep sleep mode"
};

static const value_string sta_cf_pollable[] = {
  {0x00, "Station is not CF-Pollable"},
  {0x02, "Station is CF-Pollable, not requesting to be placed on the  CF-polling list"},
  {0x01, "Station is CF-Pollable, requesting to be placed on the CF-polling list"},
  {0x03, "Station is CF-Pollable, requesting never to be polled"},
  {0x80, "QSTA requesting association in QBSS"},
  {0x81, "Reserved"},
  {0x82, "Reserved"},
  {0x83, "Reserved"},
  {0, NULL}
};

static const value_string ap_cf_pollable[] = {
  {0x00, "No point coordinator at AP"},
  {0x02, "Point coordinator at AP for delivery only (no polling)"},
  {0x01, "Point coordinator at AP for delivery and polling"},
  {0x03, "Reserved"},
  {0x80, "QAP (HC) does not use CFP for delivery of unicast data type frames"},
  {0x82, "QAP (HC) uses CFP for delivery, but does not send CF-Polls to non-QoS STAs"},
  {0x81, "QAP (HC) uses CFP for delivery, and sends CF-Polls to non-QoS STAs"},
  {0x83, "Reserved"},
  {0, NULL}
};

#define AUTH_ALG_OPEN                   0
#define AUTH_ALG_SHARED                 1
#define AUTH_ALG_FAST_BSS_TRANS         2
#define AUTH_ALG_SAE                    3
#define AUTH_ALG_FILS_SK_WITHOUT_PFS    4
#define AUTH_ALG_FILS_SK_WITH_PFS       5
#define AUTH_ALG_FILS_PK                6
#define AUTH_ALG_NETWORK_EAP         0x80

static const value_string auth_alg[] = {
  {AUTH_ALG_OPEN,                "Open System"},
  {AUTH_ALG_SHARED,              "Shared key"},
  {AUTH_ALG_FAST_BSS_TRANS,      "Fast BSS Transition"},
  {AUTH_ALG_SAE,                 "Simultaneous Authentication of Equals (SAE)"},
  {AUTH_ALG_FILS_SK_WITHOUT_PFS, "FILS Shared Key authentication without PFS"},
  {AUTH_ALG_FILS_SK_WITH_PFS,    "FILS Shared Key authentication with PFS"},
  {AUTH_ALG_FILS_PK,             "FILS Public Key authentication"},
  {AUTH_ALG_NETWORK_EAP,         "Network EAP"},  /* Cisco proprietary? */
  {0, NULL}
};

/* IANA - Internet Key Exchange (IKE) Attributes - Group Description (https://www.iana.org/assignments/ipsec-registry/ipsec-registry.xhtml) */
static const value_string ff_finite_cyclic_group_vals[] = {
  { 1, "default 768-bit MODP group"},
  { 2, "alternate 1024-bit MODP group"},
  { 3, "EC2N group on GP[2^155]"},
  { 4, "EC2N group on GP[2^185]"},
  { 5, "1536-bit MODP group"},
  {14, "2048-bit MODP group"},
  {15, "3072-bit MODP group"},
  {16, "4096-bit MODP group"},
  {17, "6144-bit MODP group"},
  {18, "8192-bit MODP group"},
  {19, "256-bit random ECP group"},
  {20, "384-bit random ECP group"},
  {21, "521-bit random ECP group"},
  {22, "1024-bit MODP Group with 160-bit Prime Order Subgroup"},
  {23, "2048-bit MODP Group with 224-bit Prime Order Subgroup"},
  {24, "2048-bit MODP Group with 256-bit Prime Order Subgroup"},
  {25, "192-bit Random ECP Group"},
  {26, "224-bit Random ECP Group"},
  {27, "224-bit Brainpool ECP group"},
  {28, "256-bit Brainpool ECP group"},
  {29, "384-bit Brainpool ECP group"},
  {30, "512-bit Brainpool ECP group"},
  {0, NULL}
};

static const true_false_string ff_block_ack_params_amsdu_permitted_flag = {
  "Permitted in QoS Data MPDUs",
  "Not Permitted"
};

static const true_false_string ff_block_ack_params_policy_flag = {
  "Immediate Block Ack",
  "Delayed Block Ack"
};

static const value_string  ff_channel_width_vals[] = {
  {0x00, "20 MHz channel width only"},
  {0x01, "Any channel width in the STA's Supported Channel Width Set"},
  {0, NULL}
};

static const true_false_string ff_qos_info_ap_q_ack_flag = {
  "APs MIB attribute dot11QAckOptionImplemented is true",
  "APs MIB attribute dot11QAckOptionImplemented is false"
};

static const true_false_string ff_qos_info_ap_queue_req_flag = {
  "AP can process a nonzero Queue Size subfield in the QoS Control field in QoS data frames",
  "AP cannot process a nonzero Queue Size subfield in the QoS Control field in QoS data frames"
};

static const true_false_string ff_qos_info_ap_txop_request_flag = {
  "AP can process a nonzero TXOP Duration Requested subfield in the QoS Control field in QoS data frames",
  "AP cannot process a nonzero TXOP Duration Requested subfield in the QoS Control field in QoS data frames"
};

static const true_false_string ff_qos_info_sta_ac_flag = {
  "Trigger-enabled and Delivery-enabled",
  "Neither Trigger-enabled nor Delivery-enabled"
};

static const true_false_string ff_qos_info_sta_q_ack_flag = {
  "STAs MIB attribute dot11QAckOptionImplemented is true",
  "STAs MIB attribute dot11QAckOptionImplemented is false"
};

static const value_string ff_qos_info_sta_max_sp_len_flags[] = {
  {0x00, "AP may deliver all buffered MSDUs, A-MSDUs and MMPDUs"},
  {0x01, "AP may deliver a maximum of two MSDUs, A-MSDUs and MMPDUs per SP"},
  {0x02, "AP may deliver a maximum of four MSDUs, A-MSDUs and MMPDUs per SP"},
  {0x03, "AP may deliver a maximum of six MSDUs, A-MSDUs and MMPDUs per SP"},
  {0, NULL}
};

static const true_false_string ff_qos_info_sta_more_data_ack_flag = {
  "STA can process ACK frames with the More Data bit in the Frame Control field set to 1 and will remain in the wake state",
  "STA cannot process ACK frames with the More Data bit in the Frame Control field set to 1"
};

static const true_false_string ff_sm_pwr_save_sm_mode_flag = {
  "Dynamic SM Power Save mode",
  "Static SM Power Save mode"
};

static const true_false_string ff_pco_phase_cntrl_flag = {
  "40 MHz Phase",
  "20 MHz Phase"
};

static const true_false_string ff_psmp_param_set_more_psmp_flag = {
  "More PSMP Sequences Follow",
  "No PSMP Sequences Follow"
};

static const value_string ff_mimo_cntrl_nc_index_flags[] = {
  {0x00, "1 Column"},
  {0x01, "2 Columns"},
  {0x02, "3 Columns"},
  {0x03, "4 Columns"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_nr_index_flags[] = {
  {0x00, "1 Row"},
  {0x01, "2 Rows"},
  {0x02, "3 Rows"},
  {0x03, "4 Rows"},
  {0, NULL}
};

static const true_false_string ff_mimo_cntrl_channel_width_flag = {
  "40 MHz",
  "20 MHz"
};

static const true_false_string ff_ht_info_information_request_flag = {
  "Requesting HT Information Exchange management action frame",
  "Should not send an HT Information Exchange management action frame"
};

static const true_false_string ff_ht_info_40_mhz_intolerant_flag = {
  "Transmitting station is intolerant of 40 MHz operation",
  "Transmitting station permits 40 MHz operation"
};

static const true_false_string ff_ht_info_sta_chan_width_flag = {
  "40 MHz",
  "20 MHz"
};

static const value_string ff_ht_action_flags[] = {
  {HT_ACTION_NOTIFY_CHAN_WIDTH,           "Notify Channel Width"},
  {HT_ACTION_SM_PWR_SAVE,                 "Spatial Multiplexing (SM) Power Save"},
  {HT_ACTION_PSMP_ACTION,                 "Power Save Multi-Poll (PSMP) action frame"},
  {HT_ACTION_SET_PCO_PHASE,               "Set PCO Phase"},
  {HT_ACTION_MIMO_CSI,                    "MIMO CSI Matrices"},
  {HT_ACTION_MIMO_BEAMFORMING,            "MIMO Non-compressed Beamforming"},
  {HT_ACTION_MIMO_COMPRESSED_BEAMFORMING, "MIMO Compressed Beamforming"},
  {HT_ACTION_ANT_SEL_FEEDBACK,            "Antenna Selection Indices Feedback"},
  {HT_ACTION_HT_INFO_EXCHANGE,            "HT Information Exchange"},
  {0x00, NULL}
};

static const value_string ff_fst_action_flags[] = {
  {FST_SETUP_REQUEST,             "FST Setup Request"},
  {FST_SETUP_RESPONSE,            "FST Setup Response"},
  {FST_TEAR_DOWN,                 "FST Tear Down"},
  {FST_ACK_REQUEST,               "FST Ack Request"},
  {FST_ACK_RESPONSE,              "FST Ack Response"},
  {FST_ON_CHANNEL_TUNNEL_REQUEST, "FST On-channel Tunnel Request"},
  {0x00, NULL}
};

static const value_string ff_dmg_action_flags[] = {
  {DMG_ACTION_PWR_SAVE_CONFIG_REQ,           "Power Save Configuration Request"},
  {DMG_ACTION_PWR_SAVE_CONFIG_RES,           "Power Save Configuration Response"},
  {DMG_ACTION_INFO_REQ,                      "Information Request"},
  {DMG_ACTION_INFO_RES,                      "Information Response"},
  {DMG_ACTION_HANDOVER_REQ,                  "Handover Request"},
  {DMG_ACTION_HANDOVER_RES,                  "Handover Response"},
  {DMG_ACTION_DTP_REQ,                       "DTP Request"},
  {DMG_ACTION_DTP_RES,                       "DTP Response"},
  {DMG_ACTION_RELAY_SEARCH_REQ,              "Relay Search Request"},
  {DMG_ACTION_RELAY_SEARCH_RES,              "Relay Search Response"},
  {DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ, "Multi Relay Channel Measurement Request"},
  {DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES, "Multi Relay Channel Measurement Response"},
  {DMG_ACTION_RLS_REQ,                       "RLS Request"},
  {DMG_ACTION_RLS_RES,                       "RLS Response"},
  {DMG_ACTION_RLS_ANNOUNCE,                  "RLS Announcement"},
  {DMG_ACTION_RLS_TEARDOWN,                  "RLS Teardown"},
  {DMG_ACTION_RELAY_ACK_REQ,                 "Relay ACK Request"},
  {DMG_ACTION_RELAY_ACK_RES,                 "Relay ACK Response"},
  {DMG_ACTION_TPA_REQ,                       "TPA Request"},
  {DMG_ACTION_TPA_RES,                       "TPA Response"},
  {DMG_ACTION_TPA_REP,                       "TPA Report"},
  {DMG_ACTION_ROC_REQ,                       "ROC Request"},
  {DMG_ACTION_ROC_RES,                       "ROC Response"},
  {0x00, NULL}
};

static const value_string ff_unprotected_dmg_action_flags[] = {
  {UNPROTECTED_DMG_ANNOUNCE,      "Announce"},
  {UNPROTECTED_DMG_BRP,           "BRP"},
  {0x00, NULL}
};
static const value_string ff_mimo_cntrl_grouping_flags[] = {
  {0x00, "No Grouping"},
  {0x01, "Carrier Groups of 2"},
  {0x02, "Carrier Groups of 4"},
  {0x03, "Reserved"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_coefficient_size_flags[] = {
  {0x00, "4 Bits"},
  {0x01, "5 Bits"},
  {0x02, "6 Bits"},
  {0x03, "8 Bits"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_codebook_info_flags[] = {
  {0x00, "1 bit for 'Capital Psi', 3 bits for 'Small Psi'"},
  {0x01, "2 bit for 'Capital Psi', 4 bits for 'Small Psi'"},
  {0x02, "3 bit for 'Capital Psi', 5 bits for 'Small Psi'"},
  {0x03, "4 bit for 'Capital Psi', 6 bits for 'Small Psi'"},
  {0, NULL}
};

static const value_string ff_ppa_action_codes[] = {
  {PPA_DSE_ENABLEMENT,                  "Protected DSE enablement"},
  {PPA_DSE_DEENABLEMENT,                "Protected DSE deenablement"},
  {PPA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT, "Protected Extended Channel Switch Announcement"},
  {PPA_DSE_MEASUREMENT_REQUEST,         "Protected DSE measurement request"},
  {PPA_DSE_MEASUREMENT_REPORT,          "Protected DSE measurement report"},
  {PPA_DSE_POWER_CONSTRAINT,            "Protected DSE power constraint"},
  {PPA_VENDOR_SPECIFIC,                 "Protected Vendor Specific"},
  {PPA_GAS_INITIAL_REQUEST,             "Protected GAS Initial Request"},
  {PPA_GAS_INITIAL_RESPONSE,            "Protected GAS Initial Response"},
  {PPA_GAS_COMEBACK_REQUEST,            "Protected GAS Comeback Request"},
  {PPA_GAS_COMEBACK_RESPONSE,           "Protected GAS Comeback Response"},
  {PPA_QAB_REQUEST,                     "Protected QAB Request"},
  {PPA_QAB_RESPONSE,                    "Protected QAB Response"},
  {0x00, NULL}
};
static value_string_ext ff_ppa_action_codes_ext = VALUE_STRING_EXT_INIT(ff_ppa_action_codes);

static const value_string ff_pa_action_codes[] = {
  {PA_20_40_BSS_COEXISTENCE_MANAGEMENT, "20/40 BSS Coexistence Management"},
  {PA_DSE_ENABLEMENT,                  "DSE enablement"},
  {PA_DSE_DEENABLEMENT,                "DSE deenablement"},
  {PA_DSE_REG_LOC_ANNOUNCEMENT,        "DSE Registered Location Announcement"},
  {PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT, "Extended Channel Switch Announcement"},
  {PA_DSE_MEASUREMENT_REQUEST,         "DSE measurement request"},
  {PA_DSE_MEASUREMENT_REPORT,          "DSE measurement report"},
  {PA_MEASUREMENT_PILOT,               "Measurement Pilot"},
  {PA_DSE_POWER_CONSTRAINT,            "DSE power constraint"},
  {PA_VENDOR_SPECIFIC,                 "Vendor Specific"},
  {PA_GAS_INITIAL_REQUEST,             "GAS Initial Request"},
  {PA_GAS_INITIAL_RESPONSE,            "GAS Initial Response"},
  {PA_GAS_COMEBACK_REQUEST,            "GAS Comeback Request"},
  {PA_GAS_COMEBACK_RESPONSE,           "GAS Comeback Response"},
  {PA_TDLS_DISCOVERY_RESPONSE,         "TDLS Discovery Response"},
  {PA_LOCATION_TRACK_NOTIFICATION,     "Location Track Notification"},
  {PA_QAB_REQUEST,                     "QAB Request"},
  {PA_QAB_RESPONSE,                    "QAB Response"},
  {0x00, NULL}
};
static value_string_ext ff_pa_action_codes_ext = VALUE_STRING_EXT_INIT(ff_pa_action_codes);

static const value_string category_codes[] = {
  {CAT_SPECTRUM_MGMT,                    "Spectrum Management (SM)"},
  {CAT_QOS,                              "Quality of Service (QoS)"},
  {CAT_DLS,                              "Direct-Link Setup (DLS)"},
  {CAT_BLOCK_ACK,                        "Block Ack"},
  {CAT_PUBLIC,                           "Public Action"},
  {CAT_RADIO_MEASUREMENT,                "Radio Measurement"},
  {CAT_FAST_BSS_TRANSITION,              "Fast BSS Transition"},
  {CAT_HT,                               "High Throughput"},
  {CAT_SA_QUERY,                         "SA Query"},
  {CAT_PUBLIC_PROTECTED,                 "Protected Dual of Public Action"},
  {CAT_WNM,                              "WNM"},
  {CAT_UNPROTECTED_WNM,                  "Unprotected WNM"},
  {CAT_TDLS,                             "TDLS"},
  {CAT_MESH,                             "MESH"},
  {CAT_MULTIHOP,                         "Multihop"},
  {CAT_SELF_PROTECTED,                   "Self-protected"},
  {CAT_DMG,                              "DMG"},
  {CAT_MGMT_NOTIFICATION,                "Management Notification"},
  {CAT_FAST_SESSION_TRANSFER,            "Fast Session Transfer"},
  {CAT_ROBUST_AV_STREAMING,              "Robust AV Streaming"},
  {CAT_UNPROTECTED_DMG,                  "Unprotected DMG"},
  {CAT_VHT,                              "VHT"},
  {CAT_S1G,                              "S1G"},
  {CAT_FILS,                             "FILS"},
  {CAT_HE,                               "HE"},
  {CAT_PROTECTED_HE,                     "Protected HE"},
  {CAT_VENDOR_SPECIFIC_PROTECTED,        "Vendor-specific Protected"},
  {CAT_VENDOR_SPECIFIC,                  "Vendor Specific"},

  {0x80 | CAT_SPECTRUM_MGMT,             "Spectrum Management (SM) (error)"},
  {0x80 | CAT_QOS,                       "Quality of Service (QoS (error))"},
  {0x80 | CAT_DLS,                       "Direct-Link Setup (DLS) (error)"},
  {0x80 | CAT_BLOCK_ACK,                 "Block Ack (error)"},
  {0x80 | CAT_PUBLIC,                    "Public Action (error)"},
  {0x80 | CAT_RADIO_MEASUREMENT,         "Radio Measurement (error)"},
  {0x80 | CAT_FAST_BSS_TRANSITION,       "Fast BSS Transition (error)"},
  {0x80 | CAT_HT,                        "High Throughput (error)"},
  {0x80 | CAT_SA_QUERY,                  "SA Query (error)"},
  {0x80 | CAT_PUBLIC_PROTECTED,          "Protected Dual of Public Action (error)"},
  {0x80 | CAT_WNM,                       "WNM (error)"},
  {0x80 | CAT_UNPROTECTED_WNM,           "Unprotected WNM (error)"},
  {0x80 | CAT_TDLS,                      "TDLS (error)"},
  {0x80 | CAT_MESH,                      "Mesh (error)"},
  {0x80 | CAT_MULTIHOP,                  "Multihop (error)"},
  {0x80 | CAT_SELF_PROTECTED,            "Self-protected (error)"},
  {0x80 | CAT_DMG,                       "DMG (error)"},
  {0x80 | CAT_MGMT_NOTIFICATION,         "Management Notification (error)"},
  {0x80 | CAT_FAST_SESSION_TRANSFER,     "Fast Session Transfer (error)"},
  {0x80 | CAT_ROBUST_AV_STREAMING,       "Robust AV Streaming (error)"},
  {0x80 | CAT_UNPROTECTED_DMG,           "Unprotected DMG (error)"},
  {0x80 | CAT_VHT,                       "VHT (error)"},
  {0x80 | CAT_HE,                        "HE (error)"},
  {0x80 | CAT_PROTECTED_HE,              "Protected HE (error)"},
  {0x80 | CAT_VENDOR_SPECIFIC_PROTECTED, "Vendor-specific Protected (error)"},
  {0x80 | CAT_VENDOR_SPECIFIC,           "Vendor Specific (error)"},
  {0, NULL}
};
static value_string_ext category_codes_ext = VALUE_STRING_EXT_INIT(category_codes);

#define NR_SUB_ID_TSF_INFO                 1
#define NR_SUB_ID_CON_COU_STR              2
#define NR_SUB_ID_BSS_TRN_CAN_PREF         3
#define NR_SUB_ID_BSS_TER_DUR              4
#define NR_SUB_ID_BEARING                  5
#define NR_SUB_ID_WIDE_BW_CHANNEL          6

#define NR_SUB_ID_MEASUREMENT_REPORT       39
#define NR_SUB_ID_HT_CAPABILITIES          45
#define NR_SUB_ID_HT_OPERATION             61
#define NR_SUB_ID_SEC_CHANNEL_OFFSET       62
#define NR_SUB_ID_MEASUREMENT_PILOT_INFO   66
#define NR_SUB_ID_RM_ENABLE_CAP            70
#define NR_SUB_ID_HT_MULTIPLE_BSSID        71
#define NR_SUB_ID_VHT_CAPABILITIES         191
#define NR_SUB_ID_VHT_OPERATION            192
#define NR_SUB_ID_HE_CAPABILITIES          193
#define NR_SUB_ID_HE_OPERATION             194

#define NR_SUB_ID_VENDOR_SPECIFIC          221

static const value_string ieee80211_neighbor_report_subelement_id_vals[] = {
  {NR_SUB_ID_TSF_INFO, "TSF Information"},
  {NR_SUB_ID_CON_COU_STR, "Condensed Country String"},
  {NR_SUB_ID_BSS_TRN_CAN_PREF, "BSS Transition Candidate Preference"},
  {NR_SUB_ID_BSS_TER_DUR, "BSS Termination Duration"},
  {NR_SUB_ID_BEARING, "Bearing"},
  {NR_SUB_ID_WIDE_BW_CHANNEL, "Wide Bandwidth Channel"},
  {NR_SUB_ID_MEASUREMENT_REPORT, "Measurement Report"},
  {NR_SUB_ID_HT_CAPABILITIES, "HT Capabilities"},
  {NR_SUB_ID_HT_OPERATION, "HT Operation"},
  {NR_SUB_ID_SEC_CHANNEL_OFFSET, "Secondary Channel Offset"},
  {NR_SUB_ID_MEASUREMENT_PILOT_INFO, "Measurement Pilot Transmission"},
  {NR_SUB_ID_RM_ENABLE_CAP, "RM Enabled Capabilities"},
  {NR_SUB_ID_HT_MULTIPLE_BSSID, "Multiple BSSID"},
  {NR_SUB_ID_VHT_CAPABILITIES, "VHT Capabilities"},
  {NR_SUB_ID_VHT_OPERATION, "VHT Operation"},
  {NR_SUB_ID_HE_CAPABILITIES, "HE Capabilities"},
  {NR_SUB_ID_HE_OPERATION, "HE Operation"},
  {NR_SUB_ID_VENDOR_SPECIFIC, "Vendor Specific"},
  {0, NULL}
};

static const value_string ieee80211_neighbor_report_bssid_info_reachability_vals[] = {
  {0, "Reserved"},
  {1, "Not Reachable"},
  {2, "Unknown"},
  {3, "Reachable"},
  {0, NULL}
};

static const value_string action_codes[] = {
  {SM_ACTION_MEASUREMENT_REQUEST, "Measurement Request"},
  {SM_ACTION_MEASUREMENT_REPORT,  "Measurement Report"},
  {SM_ACTION_TPC_REQUEST,         "TPC Request"},
  {SM_ACTION_TPC_REPORT,          "TPC Report"},
  {SM_ACTION_CHAN_SWITCH_ANNC,    "Channel Switch Announcement"},
  {0, NULL}
};

static const value_string vendor_action_types_mrvl[] = {
  {MRVL_ACTION_MESH_MANAGEMENT, "Mesh Management"},
  {0, NULL}
};

static const value_string mesh_mgt_action_codes_mrvl[] = {
  {MRVL_MESH_MGMT_ACTION_RREQ, "Route Request"},
  {MRVL_MESH_MGMT_ACTION_RREP, "Route Response"},
  {MRVL_MESH_MGMT_ACTION_RERR, "Route Error"},
  {MRVL_MESH_MGMT_ACTION_PLDM, "Peer Link Down"},
  {0, NULL}
};

static const value_string mesh_path_selection_codes[] = {
  {0x0, "Hybrid Wireless Mesh Protocol"},
  {0, NULL}
};

static const value_string mesh_metric_codes[] = {
  {0x0, "Airtime Link Metric"},
  {0, NULL}
};

static const value_string wme_action_codes[] = {
  {0x00, "Setup request"},
  {0x01, "Setup response"},
  {0x02, "Teardown"},
  {0x00, NULL}
};

static const value_string wme_status_codes[] = {
  {0x00, "Admission accepted"},
  {0x01, "Invalid parameters"},
  {0x03, "Refused"},
  {0x00, NULL}
};

static const value_string mesh_action[] = {
  {MESH_ACTION_LINK_METRIC_REPORT, "Mesh Link Metric Report"},
  {MESH_ACTION_HWMP,               "HWMP Mesh Path Selection"},
  {MESH_ACTION_GATE_ANNOUNCE,      "Gate Announcement"},
  {MESH_ACTION_CONGESTION_CTL,     "Congestion Control Notification"},
  {MESH_ACTION_MCCA_SETUP_REQUEST, "MCCA Setup Request"},
  {MESH_ACTION_MCCA_SETUP_REPLY,   "MCCA Setup Reply"},
  {MESH_ACTION_MCCA_ADV_REQUEST,   "MCCA Advertisement Request"},
  {MESH_ACTION_MCCA_ADV,           "MCCA Advertisement"},
  {MESH_ACTION_MCCA_TEARDOWN,      "MCCA Teardown"},
  {MESH_ACTION_TBTT_ADJ_REQUEST,   "TBTT Adjustment Request"},
  {MESH_ACTION_TBTT_ADJ_RESPONSE,  "TBTT Adjustment Response"},
  {0, NULL}
};
static value_string_ext mesh_action_ext = VALUE_STRING_EXT_INIT(mesh_action);

static const value_string multihop_action[] = {
  {MULTIHOP_ACTION_PROXY_UPDATE,      "Proxy Update"},
  {MULTIHOP_ACTION_PROXY_UPDATE_CONF, "Proxy Update Confirmation"},
  {0, NULL}
};

static const value_string selfprot_action[] = {
  {SELFPROT_ACTION_MESH_PEERING_OPEN,     "Mesh Peering Open"},
  {SELFPROT_ACTION_MESH_PEERING_CONFIRM,  "Mesh Peering Confirm"},
  {SELFPROT_ACTION_MESH_PEERING_CLOSE,    "Mesh Peering Close"},
  {SELFPROT_ACTION_MESH_GROUP_KEY_INFORM, "Mesh Group Key Inform"},
  {SELFPROT_ACTION_MESH_GROUP_KEY_ACK,    "Mesh Group Key Ack"},
  {0, NULL}
};

static const value_string mesh_peering_proto_ids[] = {
  {MESH_PEERING_PROTO_MGMT,   "Mesh peering management protocol"},
  {MESH_PEERING_PROTO_AMPE,   "Authenticated mesh peering exchange protocol"},
  {MESH_PEERING_PROTO_VENDOR, "Vendor specific"},
  {0, NULL}
};

static const true_false_string hwmp_targ_usn_flags = {
  "[USN = 1] Target Sequence Number Unknown at Originator",
  "[USN = 0] Target Sequence Number Known at Originator"
};

static const true_false_string hwmp_targ_to_flags = {
  "[TO = 1] Only Target Will Respond",
  "[TO = 0] Intermediate Nodes May Respond"
};

static const value_string ack_policy[] = {
  {0x00, "Normal Ack"},
  {0x01, "No Ack"},
  {0x02, "No explicit acknowledgment"},
  {0x03, "Block Ack"},
  {0x00, NULL}
};

static const value_string qos_action_codes[] = {
  {QOS_ACTION_ADDTS_REQUEST,  "ADDTS Request"},
  {QOS_ACTION_ADDTS_RESPONSE, "ADDTS Response"},
  {QOS_ACTION_DELTS,          "DELTS"},
  {QOS_ACTION_SCHEDULE,   "Schedule"},
  {QOS_ACTION_MAP_CONFIGURE, "QoS Map Configure"},
  {0, NULL}
};

static const value_string ba_action_codes[] = {
  {BA_ADD_BLOCK_ACK_REQUEST,  "Add Block Ack Request"},
  {BA_ADD_BLOCK_ACK_RESPONSE, "Add Block Ack Response"},
  {BA_DELETE_BLOCK_ACK,       "Delete Block Ack"},
  {0x00, NULL}
};

static const value_string dls_action_codes[] = {
  {DLS_ACTION_REQUEST,  "DLS Request"},
  {DLS_ACTION_RESPONSE, "DLS Response"},
  {DLS_ACTION_TEARDOWN, "DLS Teardown"},
  {0, NULL}
};

static const value_string tsinfo_type[] = {
  {0x0, "Aperiodic or unspecified Traffic"},
  {0x1, "Periodic Traffic"},
  {0, NULL}
};

static const value_string tsinfo_direction[] = {
  {0x00, "Uplink"},
  {0x01, "Downlink"},
  {0x02, "Direct link"},
  {0x03, "Bidirectional link"},
  {0, NULL}
};

static const value_string tsinfo_access[] = {
  {0x00, "Reserved"},
  {0x01, "EDCA"},
  {0x02, "HCCA"},
  {0x03, "HEMM"},
  {0, NULL}
};

static const value_string qos_up[] = {
  {0x00, "Best Effort"},
  {0x01, "Background"},
  {0x02, "Spare"},
  {0x03, "Excellent Effort"},
  {0x04, "Controlled Load"},
  {0x05, "Video"},
  {0x06, "Voice"},
  {0x07, "Network Control"},
  {0, NULL}
};

static const value_string classifier_type[] = {
  {0x00, "Ethernet parameters"},
  {0x01, "TCP/UDP IP parameters"},
  {0x02, "IEEE 802.1D/Q parameters"},
  {0, NULL}
};

static const true_false_string ieee80211_block_ack_control_ack_policy_flag = {
    "Sender Does Not Require Immediate Acknowledgement",
    "Immediate Acknowledgement Required"
};

static const value_string ft_action_codes[] = {
  {FT_ACTION_REQUEST, "FT Request"},
  {FT_ACTION_RESPONSE, "FT Response"},
  {FT_ACTION_CONFIRM, "FT Confirm"},
  {FT_ACTION_ACK, "FT Ack"},
  {0, NULL}
};

static const value_string sa_query_action_codes[] = {
  {SA_QUERY_REQUEST, "SA Query Request"},
  {SA_QUERY_RESPONSE, "SA Query Response"},
  {0, NULL}
};

static const value_string ieee80211_data_encap_payload_types[] = {
  {1, "Remote Request/Response"},
  {2, "TDLS"},
  {0, NULL}
};

static const true_false_string rsn_preauth_flags = {
  "Transmitter supports pre-authentication",
  "Transmitter does not support pre-authentication"
};

static const true_false_string rsn_no_pairwise_flags = {
  "Transmitter cannot support WEP default key 0 simultaneously with "
  "Pairwise key",
  "Transmitter can support WEP default key 0 simultaneously with "
  "Pairwise key"
};

static const value_string rsn_cap_replay_counter[] = {
  {0x00, "1 replay counter per PTKSA/GTKSA/STAKeySA"},
  {0x01, "2 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0x02, "4 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0x03, "16 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0, NULL}
};

static const true_false_string ht_ldpc_coding_flag = {
  "Transmitter supports receiving LDPC coded packets",
  "Transmitter does not support receiving LDPC coded packets"
};

static const true_false_string ht_chan_width_flag = {
  "Transmitter supports 20MHz and 40MHz operation",
  "Transmitter only supports 20MHz operation"
};

static const value_string ht_sm_pwsave_flag[] = {
  {0x00, "Static SM Power Save mode"},
  {0x01, "Dynamic SM Power Save mode"},
  {0x02, "Reserved"},
  {0x03, "SM Power Save disabled"},
  {0x00, NULL}
};

static const true_false_string ht_green_flag = {
  "Transmitter is able to receive PPDUs with Green Field (GF) preamble",
  "Transmitter is not able to receive PPDUs with Green Field (GF) preamble"
};

static const value_string ht_rx_stbc_flag[] = {
  {0x00, "No Rx STBC support"},
  {0x01, "Rx support of one spatial stream"},
  {0x02, "Rx support of one and two spatial streams"},
  {0x03, "Rx support of one, two, and three spatial streams"},
  {0x00, NULL}
};

  /* IEEE Stc 802.11ac-2013 8.4.2.29 Extended Capabilities element*/
static const value_string vht_max_mpdu_in_amsdu[] = {
    {0x00, "No limit"},
    {0x01, "32"},
    {0x02, "16"},
    {0x03, "8"},
    {0, NULL}
};

  /* IEEE Stc 802.11ac/D3.1 */

static const value_string vht_max_mpdu_length_flag[] = {
  {0x00, "3 895"},
  {0x01, "7 991"},
  {0x02, "11 454"},
  {0x03, "Reserved"},
  {0x00, NULL}
};

static const value_string vht_supported_chan_width_set_flag[] = {
  {0x00, "Neither 160MHz nor 80+80 supported"},
  {0x01, "160MHz supported"},
  {0x02, "160MHz and 80+80 Supported"},
  {0x03, "Reserved"},
  {0x00, NULL}
};

static const value_string vht_rx_stbc_flag[] = {
  {0x00, "None"},
  {0x01, "1 Spatial Stream Supported"},
  {0x02, "1 to 2 Spatial Stream Supported"},
  {0x03, "1 to 3 Spatial Stream Supported"},
  {0x04, "1 to 4 Spatial Stream Supported"},
  {0x05, "Reserved"},
  {0x06, "Reserved"},
  {0x07, "Reserved"},
  {0x00, NULL}
};

static const value_string num_plus_one_3bit_flag[] = {
  {0x00, "1"},
  {0x01, "2"},
  {0x02, "3"},
  {0x03, "4"},
  {0x04, "5"},
  {0x05, "6"},
  {0x06, "7"},
  {0x07, "8"},
  {0x00, NULL}
};

static const value_string vht_max_ampdu_flag[] = {
  {0x00, "8 191"},
  {0x01, "16 383"},
  {0x02, "32 767"},
  {0x03, "65,535"},
  {0x04, "131 071"},
  {0x05, "262 143"},
  {0x06, "524 287"},
  {0x07, "1 048 575"},
  {0x00, NULL}
};

static const value_string vht_link_adapt_flag[] = {
  {0x00, "No Feedback"},
  {0x01, "Reserved (logically only solicited feedback)"},
  {0x02, "Unsolicited feedback only"},
  {0x03, "Both (can provide unsolicited feedback and respond to VHT MRQ)"},
  {0x00, NULL}
};

static const value_string vht_supported_mcs_flag[] = {
  {0x00, "MCS 0-7"},
  {0x01, "MCS 0-8"},
  {0x02, "MCS 0-9"},
  {0x03, "Not Supported"},
  {0x00, NULL}
};

static const value_string vht_op_channel_width_flag[] = {
  {0x00, "20 MHz or 40 MHz"},
  {0x01, "80 MHz"},
  {0x02, "160 MHz"},
  {0x03, "80+80 MHz"},
  {0x00, NULL}
};

static const value_string vht_tpe_pwr_units[] = {
  {0x00, "EIRP"},
  {0x00, NULL}
};

static const true_false_string vht_ndp_annc_sta_info_feedback_type = {
  "MU feedback requested",
  "SU feedback requested"
};

static const true_false_string ht_delayed_block_ack_flag = {
  "Transmitter supports HT-Delayed BlockAck",
  "Transmitter does not support HT-Delayed BlockAck"
};

static const true_false_string ht_max_amsdu_flag = {
  "7935 bytes",
  "3839 bytes"
};

static const true_false_string ht_dss_cck_40_flag = {
  "Will/Can use DSSS/CCK in 40 MHz",
  "Won't/Can't use of DSSS/CCK in 40 MHz"
};

static const true_false_string ht_psmp_flag = {
  "Will/Can support PSMP operation",
  "Won't/Can't support PSMP operation"
};

static const true_false_string ht_40_mhz_intolerant_flag = {
  "Use of 40 MHz transmissions restricted/disallowed",
  "Use of 40 MHz transmissions unrestricted/allowed"
};

static const value_string ampduparam_mpdu_start_spacing_flags[] = {
  {0x00, "no restriction"},
  {0x01, "1/4 [usec]"},
  {0x02, "1/2 [usec]"},
  {0x03, "1 [usec]"},
  {0x04, "2 [usec]"},
  {0x05, "4 [usec]"},
  {0x06, "8 [usec]"},
  {0x07, "16 [usec]"},
  {0x00, NULL}
};

static const true_false_string mcsset_tx_rx_mcs_set_not_equal_flag = {
  "Not Equal",
  "Equal",
};

static const value_string mcsset_tx_max_spatial_streams_flags[] = {
  {0x00, "1 spatial stream"},
  {0x01, "2 spatial streams"},
  {0x02, "3 spatial streams"},
  {0x03, "4 spatial streams"},
  {0x04, "TX MCS Set Not Defined"},
  {0x00, NULL}
};

static const value_string htex_transtime_flags[] = {
  {0x00, "No Transition"},
  {0x01, "400 usec"},
  {0x02, "1.5 msec"},
  {0x03, "5 msec"},
  {0x00, NULL}
};

static const value_string htex_mcs_flags[] = {
  {0x00, "STA does not provide MCS feedback"},
  {0x01, "Reserved"},
  {0x02, "STA provides only unsolicited MCS feedback"},
  {0x03, "STA can provide MCS feedback in response to MRQ as well as unsolicited MCS feedback"},
  {0x00, NULL}
};

static const value_string txbf_calib_flag[] = {
  {0x00, "incapable"},
  {0x01, "Limited involvement, cannot initiate"},
  {0x02, "Limited involvement, can initiate"},
  {0x03, "Fully capable"},
  {0x00, NULL}
};

static const value_string txbf_feedback_flags[] = {
  {0x00, "not supported"},
  {0x01, "delayed feedback capable"},
  {0x02, "immediate feedback capable"},
  {0x03, "delayed and immediate feedback capable"},
  {0x00, NULL}
};

static const value_string txbf_antenna_flags[] = {
  {0x00, "1 TX antenna sounding"},
  {0x01, "2 TX antenna sounding"},
  {0x02, "3 TX antenna sounding"},
  {0x03, "4 TX antenna sounding"},
  {0x00, NULL}
};

static const value_string txbf_csi_max_rows_bf_flags[] = {
  {0x00, "1 row of CSI"},
  {0x01, "2 rows of CSI"},
  {0x02, "3 rows of CSI"},
  {0x03, "4 rows of CSI"},
  {0x00, NULL}
};

static const value_string txbf_chan_est_flags[] = {
  {0x00, "1 space time stream"},
  {0x01, "2 space time streams"},
  {0x02, "3 space time streams"},
  {0x03, "4 space time streams"},
  {0x00, NULL}
};

static const value_string txbf_min_group_flags[] = {
  {0x00, "No grouping supported"},
  {0x01, "Groups of 1, 2 supported"},
  {0x02, "Groups of 1, 4 supported"},
  {0x03, "Groups of 1, 2, 4 supported"},
  {0x00, NULL}
};

static const value_string hta_ext_chan_offset_flag[] = {
  {0x00, "No Extension Channel"},
  {0x01, "Extension Channel above control channel"},
  {0x02, "Undefined"},
  {0x03, "Extension Channel below control channel"},
  {0x00, NULL}
};

static const true_false_string hta_rec_tx_width_flag = {
  "Any channel width enabled",
  "Use 20MHz channel (control)"
};

static const true_false_string hta_rifs_mode_flag = {
  "Use of RIFS permitted",
  "Use of RIFS prohibited"
};

static const true_false_string hta_controlled_access_flag = {
  "Not only PSMP",
  "PSMP only"
};

static const value_string hta_service_interval_flag[] = {
  {0x00, "5ms"},
  {0x01, "10ms"},
  {0x02, "15ms"},
  {0x03, "20ms"},
  {0x04, "25ms"},
  {0x05, "30ms"},
  {0x06, "35ms"},
  {0x07, "40ms"},
  {0x00, NULL}
};

static const value_string hta_operating_mode_flag[] = {
  {0x00, "Pure HT, no protection"},
  {0x01, "There may be non-HT devices (control & ext channel)"},
  {0x02, "No non-HT is associated, but at least 1 20MHz is. protect on"},
  {0x03, "Mixed: no non-HT is associated, protect on"},
  {0x00, NULL}
};

static const true_false_string hta_non_gf_devices_flag = {
  "All HT devices associated are GF capable",
  "One or More HT devices are not GF capable"
};

static const true_false_string hta_dual_stbc_protection_flag = {
  "Dual CTS protections is used",
  "Regular use of RTS/CTS"
};

static const true_false_string hta_secondary_beacon_flag = {
  "Secondary Beacon",
  "Primary Beacon"
};

static const true_false_string hta_lsig_txop_protection_flag = {
  "Full Support",
  "Not full support"
};

static const true_false_string hta_pco_active_flag = {
  "PCO is activated in the BSS",
  "PCO is not activated in the BSS"
};

static const true_false_string hta_pco_phase_flag = {
  "Switch to 20MHz phase/keep 20MHz",
  "Switch to 40MHz phase/keep 40MHz"
};

static const value_string ht_info_secondary_channel_offset_flags[] = {
  {0x00, "No secondary channel"},
  {0x01, "Secondary channel is above the primary channel"},
  {0x02, "Reserved"},
  {0x03, "Secondary channel is below the primary channel"},
  {0x00, NULL}
};

static const true_false_string ht_info_channel_sta_width_flag = {
  "Channel of any width supported",
  "20 MHz channel width only"
};

static const true_false_string ht_info_rifs_mode_flag = {
  "Permitted",
  "Prohibited"
};

static const value_string ht_info_operating_protection_mode_flags[] = {
  {0x00, "No protection mode"},
  {0x01, "HT non-member protection mode"},
  {0x02, "20 MHz protection mode"},
  {0x03, "non-HT mixed mode"},
  {0x00, NULL}
};

static const true_false_string ht_info_non_greenfield_sta_present_flag = {
  "One or more associated STAs are not greenfield capable",
  "All associated STAs are greenfield capable"
};

static const true_false_string ht_info_obss_non_ht_stas_present_flag = {
  "Use of protection for non-HT STAs by overlapping BSSs is needed",
  "Use of protection for non-HT STAs by overlapping BSSs is not needed"
};

static const true_false_string ht_info_dual_beacon_flag = {
  "AP transmits a secondary beacon",
  "No second beacon is transmitted"
};

static const true_false_string ht_info_secondary_beacon_flag = {
  "STBC beacon",
  "Primary beacon"
};

static const true_false_string ht_info_lsig_txop_protection_full_support_flag = {
  "All HT STAs in the BSS support L-SIG TXOP protection",
  "One or more HT STAs in the BSS do not support L-SIG TXOP protection"
};

static const true_false_string ht_info_pco_phase_flag = {
  "Switch to or continue 40 MHz phase",
  "Switch to or continue 20 MHz phase"
};

static const true_false_string htc_lac_trq_flag = {
  "Want sounding PPDU",
  "Don't want sounding PPDU"
};

static const true_false_string htc_lac_mai_mrq_flag = {
  "MCS feedback requested",
  "No MCS feedback requested"
};

static const value_string ieee80211_htc_lac_asel_command_flags[] = {
  {0x00, "Transmit Antenna Selection Sounding Indication (TXASSI)"},
  {0x01, "Transmit Antenna Selection Sounding Request (TXASSR)"},
  {0x02, "Receive Antenna Selection Sounding Indication (RXASSI)"},
  {0x03, "Receive Antenna Selection Sounding Request (RXASSR)"},
  {0x04, "Sounding Label"},
  {0x05, "No feedback, ASEL training failure"},
  {0x06, "Transmit Antenna Selection Sounding Indication (TXASSI) requesting feedback of explicit CSI"},
  {0x07, "Reserved"},
  {0x00, NULL}
};

static const value_string ieee80211_htc_cal_pos_flags[] = {
  {0x00, "Not a calibration frame"},
  {0x01, "Calibration Start"},
  {0x02, "Sounding Response"},
  {0x03, "Sounding Complete"},
  {0x00, NULL}
};

static const value_string ieee80211_htc_csi_steering_flags[] = {
  {0x00, "No feedback required"},
  {0x01, "CSI"},
  {0x02, "Non-compressed Beamforming Feedback Matrix"},
  {0x03, "Compressed Beamforming Feedback Matrix"},
  {0x00, NULL}
};

static const true_false_string ieee80211_htc_ndp_announcement_flag = {
  "NDP will follow",
  "No NDP will follow"
};

static const value_string ieee80211_htc_bw_recommended_vht_mcs_vals[] = {
  {0, "20 MHz"},
  {1, "40 MHz"},
  {2, "80 MHz"},
  {3, "160 MHz and 80+80 MHz"},
  {0, NULL}
};

static const value_string ieee80211_htc_coding_type_vals[] = {
  {0, "BCC"},
  {1, "LDPC"},
  {0, NULL}
};

static const value_string ieee80211_htc_fb_tx_type_vals[] = {
  {0, "Not beamformed"},
  {1, "Beamformed"},
  {0, NULL}
};

static const value_string ieee80211_tag_secondary_channel_offset_flags[] = {
  {0x00, "No Secondary Channel"},
  {0x01, "Above Primary Channel"},
  {0x02, "Reserved"},
  {0x03, "Below Primary Channel"},
  {0x00, NULL}
};

#define BSS_BITMASK_UP0   0x0001
#define BSS_BITMASK_UP1   0x0002
#define BSS_BITMASK_UP2   0x0004
#define BSS_BITMASK_UP3   0x0008
#define BSS_BITMASK_UP4   0x0010
#define BSS_BITMASK_UP5   0x0020
#define BSS_BITMASK_UP6   0x0040
#define BSS_BITMASK_UP7   0x0080
#define BSS_BITMASK_AC0   0x0100
#define BSS_BITMASK_AC1   0x0200
#define BSS_BITMASK_AC2   0x0400
#define BSS_BITMASK_AC3   0x0800
#define BSS_BITMASK_RSV   0xF000

static const value_string ieee80211_tag_ext_channel_switch_announcement_switch_mode_flags[] = {
  {0x00, "Frames may be transmitted before the channel switch has been completed"},
  {0x01, "No more frames are to be transmitted until the channel switch has been completed"},
  {0x00, NULL}
};

static const value_string service_interval_granularity_vals[] = {
  { 0, "5 ms" },
  { 1, "10 ms" },
  { 2, "15 ms" },
  { 3, "20 ms" },
  { 4, "25 ms" },
  { 5, "30 ms" },
  { 6, "35 ms" },
  { 7, "40 ms" },
  { 0x00, NULL }
};

static const value_string wep_type_vals[] = {
  { DOT11DECRYPT_KEY_TYPE_WEP, STRING_KEY_TYPE_WEP },
  { DOT11DECRYPT_KEY_TYPE_WPA_PWD, STRING_KEY_TYPE_WPA_PWD },
  { DOT11DECRYPT_KEY_TYPE_WPA_PSK, STRING_KEY_TYPE_WPA_PSK },
  { 0x00, NULL }
};

static const value_string ieee80211_ht_pren_type_vals[] = {
  { 51,  "HT Capabilities (802.11n D1.10)" },
  { 52,  "HT Additional Capabilities (802.11n D1.00)" },
  { 0, NULL }
};

static const true_false_string ieee80211_cf_ssw_direction = {
  "Transmitted by the beamforming responder",
  "Transmitted by the beamforming initiator"
};

static const value_string bss_type[] = {
  {0x0, "Reserved"},
  {0x1, "IBSS"},
  {0x2, "PBSS"},
  {0x3, "Infrastructure BSS"},
  {0,   NULL}
};

static const value_string band_id[] = {
  {0x0, "TV white spaces"},
  {0x1, "Sub-1 GHZ (excluding TV white spaces)"},
  {0x2, "2.4 GHZ"},
  {0x3, "3.6 GHZ"},
  {0x4, "4.9 and 5 GHZ"},
  {0x5, "60 GHZ"},
  {0,   NULL}
};

static const value_string extended_sc_mcs[] = {
  {0x0, "None"},
  {0x1, "MCS 9.1"},
  {0x2, "MCS 12.1"},
  {0x3, "MCS 12.2"},
  {0x4, "MCS 12.3"},
  {0x5, "MCS 12.4"},
  {0x6, "MCS 12.5"},
  {0x7, "MCS 12.6"},
  {0, NULL}
};

static const range_string max_basic_sf_amsdu[] = {
  {0, 0, "No Limit"},
  {1, 1, "4 Basic subframes"},
  {2, 2, "8 Basic subframes"},
  {3, 3, "16 Basic subframes"},
  {4, 4, "32 Basic subframes"},
  {5, 5, "64 Basic subframes"},
  {6, 6, "128 Basic subframes"},
  {7, 7, "256 Basic subframes"},
  {8, 255, "reserved"},
  {0, 0, NULL}
};

static const range_string max_short_sf_amsdu[] = {
  {0, 0, "No Limit"},
  {1, 1, "32 Short subframes"},
  {2, 2, "64 Short subframes"},
  {3, 3, "128 Short subframes"},
  {4, 4, "256 Short subframes"},
  {5, 5, "512 Short subframes"},
  {6, 6, "1024 Short subframes"},
  {7, 255, "reserved"},
  {0, 0, NULL}
};

static const value_string allocation_type[] = {
  {0x0, "SP Allocation"},
  {0x1, "CBAP allocation"},
  {0,   NULL}
};

static const value_string vht_operation_info_channel_width[] = {
  {0x00, "20MHz or 40MHz"},
  {0x01, "80MHZ"},
  {0x10, "160MHz"},
  {0x11, "80MHz+80MHz"},
  {0, NULL}
};
static const value_string operating_mode_field_channel_width[] = {
  {0x00, "20MHz"},
  {0x01, "40MHZ"},
  {0x02, "80MHz"},
  {0x03, "160MHz or 80MHz+80MHz"},
  {0, NULL}
};

static const value_string operat_mode_field_rxnss[] = {
  {0x0, "1Nss"},
  {0x1, "2Nss"},
  {0x2, "3Nss"},
  {0x3, "4Nss"},
  {0x4, "5Nss"},
  {0x5, "6Nss"},
  {0x6, "7Nss"},
  {0x7, "8Nss"},
  {0, NULL}
};

#define VHT_ACT_VHT_COMPRESSED_BEAMFORMING  0
#define VHT_ACT_GROUP_ID_MANAGEMENT         1
#define VHT_ACT_OPERATION_MODE_NOTIFICATION 2

static const value_string vht_action_vals[] = {
  {VHT_ACT_VHT_COMPRESSED_BEAMFORMING, "VHT Compressed Beamforming"},
  {VHT_ACT_GROUP_ID_MANAGEMENT, "Group ID Management"},
  {VHT_ACT_OPERATION_MODE_NOTIFICATION, "Operating Mode Notification"},
  {0,   NULL}
};

static const value_string ff_vht_mimo_cntrl_nc_index_vals[] = {
  {0x00, "1 Column"},
  {0x01, "2 Columns"},
  {0x02, "3 Columns"},
  {0x03, "4 Columns"},
  {0x04, "5 Columns"},
  {0x05, "6 Columns"},
  {0x06, "7 Columns"},
  {0x07, "8 Columns"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_nr_index_vals[] = {
  {0x00, "1 Row"},
  {0x01, "2 Rows"},
  {0x02, "3 Rows"},
  {0x03, "4 Rows"},
  {0x04, "5 Rows"},
  {0x05, "6 Rows"},
  {0x06, "7 Rows"},
  {0x07, "8 Rows"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_channel_width_vals[] = {
  {0x00, "20 MHz"},
  {0x01, "40 MHz"},
  {0x02, "80 MHz"},
  {0x03, "160 MHz / 80+80 MHz"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_grouping_vals[] = {
  {0x00, "1 (No Grouping)"},
  {0x01, "2"},
  {0x02, "4"},
  {0x03, "Reserved"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_feedback_vals[] = {
  {0x00, "SU"},
  {0x01, "MU"},
  {0, NULL}
};

#define S1G_ACT_AID_SWITCH_REQUEST   0
#define S1G_ACT_AID_SWITCH_RESPONSE  1
#define S1G_ACT_SYNC_CONTROL         2
#define S1G_ACT_STA_INFO_ANNOUNCE    3
#define S1G_ACT_EDCA_PARAM_SET       4
#define S1G_ACT_EL_OPERATION         5
#define S1G_ACT_TWT_SETUP            6
#define S1G_ACT_TWT_TEARDOWN         7
#define S1G_ACT_SECT_GROUP_ID_LIST   8
#define S1G_ACT_SECT_ID_FEEDBACK     9
#define S1G_ACT_RESERVED             10
#define S1G_ACT_TWT_INFORMATION      11

static const value_string s1g_action_vals[] = {
  {S1G_ACT_AID_SWITCH_REQUEST, "AID Switch Request"},
  {S1G_ACT_AID_SWITCH_RESPONSE, "AID Switch Response"},
  {S1G_ACT_SYNC_CONTROL, "Sync Control"},
  {S1G_ACT_STA_INFO_ANNOUNCE, "STA Information Announcement"},
  {S1G_ACT_EDCA_PARAM_SET, "EDCA Parameter Set"},
  {S1G_ACT_EL_OPERATION, "EL Operation"},
  {S1G_ACT_TWT_SETUP, "TWT Setup"},
  {S1G_ACT_TWT_TEARDOWN, "TWT Teardown"},
  {S1G_ACT_SECT_GROUP_ID_LIST, "Sectorized Group ID List"},
  {S1G_ACT_SECT_ID_FEEDBACK, "Sector ID Feedback"},
  {S1G_ACT_RESERVED, "Reserved"},
  {S1G_ACT_TWT_INFORMATION, "TWT Information"},
  {0,   NULL},
};

static const value_string twt_neg_type_vals[] = {
  {0x0, "Individual TWT"},
  {0x1, "Wake TBTT"},
  {0x2, "Broadcast TWT"},
  {0x3, "Broadcast TWT"},
  {0,   NULL},
};

static const true_false_string twt_responder_pm_mode_tfs = {
  "AP may doze outside the TWT",
  "AP is always awake",
};

static const true_false_string twt_requester_tfs = {
  "This STA is a TWT Requesting STA",
  "This STA is a TWT Responding STA or a TWT scheduling AP",
};

#define REQUEST_TWT   0
#define SUGGEST_TWT   1
#define DEMAND_TWT    2
#define TWT_GROUPING  3
#define ACCEPT_TWT    4
#define ALTERNATE_TWT 5
#define DICTATE_TWT   6
#define REJECT_TWT    7

static const value_string twt_setup_cmd[] = {
  {REQUEST_TWT,   "Request TWT"},
  {SUGGEST_TWT,   "Suggest TWT"},
  {DEMAND_TWT,    "Demand TWT"},
  {TWT_GROUPING,  "TWT Grouping"},
  {ACCEPT_TWT,    "Accept TWT"},
  {ALTERNATE_TWT, "Alternate TWT"},
  {DICTATE_TWT,   "Dictate TWT"},
  {REJECT_TWT,    "Reject TWT"},
  {0,   NULL},
};

static const true_false_string twt_trigger = {
  "TWT SP includes trigger frames",
  "TWT SP does not include trigger frames",
};

static const true_false_string twt_implicit = {
  "TWT is implicit",
  "TWT is explicit",
};

static const true_false_string twt_flow_type = {
  "TWT is unannounced, the TWT responding STA can send frames at any time",
  "TWT is announced, the TWT Requesting STA will send trigger frames",
};

static int proto_wlan = -1;
static int proto_centrino = -1;
static int proto_aggregate = -1;
static gboolean ieee80211_tvb_invalid = FALSE;

/* ************************************************************************* */
/*                Header field info values for FC-field                      */
/* ************************************************************************* */
static int hf_ieee80211_fc_field = -1;
static int hf_ieee80211_fc_proto_version = -1;
static int hf_ieee80211_fc_frame_type = -1;
static int hf_ieee80211_fc_frame_subtype = -1;
static int hf_ieee80211_fc_frame_extension = -1;
static int hf_ieee80211_fc_frame_type_subtype = -1;

static int hf_ieee80211_fc_flags = -1;
static int hf_ieee80211_fc_to_ds = -1;
static int hf_ieee80211_fc_from_ds = -1;
static int hf_ieee80211_fc_data_ds = -1;

static int hf_ieee80211_fc_more_frag = -1;
static int hf_ieee80211_fc_retry = -1;
static int hf_ieee80211_fc_pwr_mgt = -1;
static int hf_ieee80211_fc_more_data = -1;
static int hf_ieee80211_fc_protected = -1;
static int hf_ieee80211_fc_order = -1;

typedef struct retransmit_key {
  guint8  bssid[6];
  guint8  src[6];
  guint16 seq_control;
  guint   fnum;
} retransmit_key;

static GHashTable *fc_analyse_retransmit_table = NULL;
static GHashTable *fc_first_frame_table = NULL;

static int hf_ieee80211_fc_analysis_retransmission = -1;
static int hf_ieee80211_fc_analysis_retransmission_frame = -1;

/* ************************************************************************* */
/*                   Header values for Duration/ID field                     */
/* ************************************************************************* */
static int hf_ieee80211_did_duration = -1;
static int hf_ieee80211_assoc_id = -1;

/* ************************************************************************* */
/*         Header values for different address-fields (all 4 of them)        */
/* ************************************************************************* */
static int hf_ieee80211_addr_da = -1;  /* Destination address subfield */
static int hf_ieee80211_addr_da_resolved = -1;  /* Dst addr subfield resolved*/
static int hf_ieee80211_addr_sa = -1;  /* Source address subfield */
static int hf_ieee80211_addr_sa_resolved = -1;  /* Src addr subfield resolved*/
static int hf_ieee80211_addr_ra = -1;  /* Receiver address subfield */
static int hf_ieee80211_addr_ra_resolved = -1;  /* Rcv addr subfield resolved*/
static int hf_ieee80211_addr_ta = -1;  /* Transmitter address subfield */
static int hf_ieee80211_addr_ta_resolved = -1;  /* Txm addr subfield resolved*/
static int hf_ieee80211_addr_bssid = -1;  /* address is bssid */
static int hf_ieee80211_addr_bssid_resolved = -1;  /* bssid resolved*/
static int hf_ieee80211_addr_staa = -1;  /* address is station address */
static int hf_ieee80211_addr_staa_resolved = -1;  /* station address resolved*/

static int hf_ieee80211_addr = -1;  /* Source or destination address subfield */
static int hf_ieee80211_addr_resolved = -1;/*Src/dst address subfield resolved*/

/* ************************************************************************* */
/*                Header values for QoS control field                        */
/* ************************************************************************* */
static int hf_ieee80211_qos = -1;
static int hf_ieee80211_qos_tid = -1;
static int hf_ieee80211_qos_priority = -1;
static int hf_ieee80211_qos_ack_policy = -1;
static int hf_ieee80211_qos_amsdu_present = -1;
static int hf_ieee80211_qos_eosp = -1;
static int hf_ieee80211_qos_bit4 = -1;
static int hf_ieee80211_qos_txop_limit = -1;
static int hf_ieee80211_qos_ps_buf_state = -1;
static int hf_ieee80211_qos_buf_state_indicated = -1;
static int hf_ieee80211_qos_highest_pri_buf_ac = -1;
static int hf_ieee80211_qos_qap_buf_load = -1;
static int hf_ieee80211_qos_txop_dur_req = -1;
static int hf_ieee80211_qos_queue_size = -1;

/* ************************************************************************* */
/*                Header values for HT control field (+HTC) and HE control   */
/* ************************************************************************* */
/* 802.11-2012 and 802.11ac-2013 8.2.4.6 */
static int hf_ieee80211_htc = -1;
static int hf_ieee80211_htc_vht = -1;
static int hf_ieee80211_htc_he = -1;
static int hf_ieee80211_htc_he_ctrl_id = -1;
static int hf_ieee80211_he_a_control_padding = -1;
static int hf_ieee80211_he_trs_he_tb_ppdu_len = -1;
static int hf_ieee80211_he_trs_ru_allocation = -1;
static int hf_ieee80211_he_dl_tx_power = -1;
static int hf_ieee80211_he_ul_target_rssi = -1;
static int hf_ieee80211_he_ul_mcs = -1;
static int hf_ieee80211_he_ul_reserved = -1;
static int hf_ieee80211_he_om_rx_nss = -1;
static int hf_ieee80211_he_om_channel_width = -1;
static int hf_ieee80211_he_om_ul_mu_disable = -1;
static int hf_ieee80211_he_om_tx_nsts = -1;
static int hf_ieee80211_he_om_reserved = -1;
static int hf_ieee80211_he_hla_unsolicited_mfb = -1;
static int hf_ieee80211_he_hla_mrq = -1;
static int hf_ieee80211_he_hla_nss = -1;
static int hf_ieee80211_he_hla_he_mcs = -1;
static int hf_ieee80211_he_hla_dcm = -1;
static int hf_ieee80211_he_hla_ru = -1;
static int hf_ieee80211_he_hla_bw = -1;
static int hf_ieee80211_he_hla_msi_ppdu_type = -1;
static int hf_ieee80211_he_hla_tx_bf = -1;
static int hf_ieee80211_he_hla_reserved = -1;
static int hf_ieee80211_he_bsr_aci_bitmap = -1;
static int hf_ieee80211_he_bsr_delta_tid = -1;
static int hf_ieee80211_he_bsr_aci_high = -1;
static int hf_ieee80211_he_bsr_scaling_factor = -1;
static int hf_ieee80211_he_bsr_queue_size_high = -1;
static int hf_ieee80211_he_bsr_queue_size_all = -1;
static int hf_ieee80211_he_uph_ul_power_headroom = -1;
static int hf_ieee80211_he_uph_ul_min_transmit_power_flag = -1;
static int hf_ieee80211_he_uph_reserved = -1;
static int hf_ieee80211_he_cci_ac_constraint = -1;
static int hf_ieee80211_he_cci_rdg_more_ppdu = -1;
static int hf_ieee80211_he_cci_sr_ppdu_indic = -1;
static int hf_ieee80211_he_cci_reserved = -1;
static int hf_ieee80211_he_btc_avail_chan = -1;
static int hf_ieee80211_he_btc_reserved = -1;
static int hf_ieee80211_he_trigger_common_info = -1;
static int hf_ieee80211_he_trigger_type = -1;
static int hf_ieee80211_he_trigger_ul_length = -1;
static int hf_ieee80211_he_trigger_more_tf = -1;
static int hf_ieee80211_he_trigger_cs_required = -1;
static int hf_ieee80211_he_trigger_ul_bw = -1;
static int hf_ieee80211_he_trigger_gi_and_ltf_type = -1;
static int hf_ieee80211_he_trigger_mu_mimo_ltf_mode = -1;
static int hf_ieee80211_he_trigger_num_he_ltf_syms_etc = -1;
static int hf_ieee80211_he_trigger_ul_stbc = -1;
static int hf_ieee80211_he_trigger_ldpc_extra_sym_seg = -1;
static int hf_ieee80211_he_trigger_ap_tx_power = -1;
static int hf_ieee80211_he_trigger_ul_packet_extension = -1;
static int hf_ieee80211_he_trigger_ul_spatial_reuse = -1;
static int hf_ieee80211_he_trigger_doppler = -1;
static int hf_ieee80211_he_trigger_ul_he_sig_a_reserved = -1;
static int hf_ieee80211_he_trigger_reserved = -1;
static int hf_ieee80211_he_trigger_user_info = -1;
static int hf_ieee80211_he_trigger_bar_ctrl = -1;
static int hf_ieee80211_he_trigger_bar_ctrl_ba_ack_policy = -1;
static int hf_ieee80211_he_trigger_bar_ctrl_ba_type = -1;
static int hf_ieee80211_he_trigger_bar_ctrl_reserved = -1;
static int hf_ieee80211_he_trigger_bar_ctrl_tid_info = -1;
static int hf_ieee80211_he_trigger_bar_info = -1;
static int hf_ieee80211_he_trigger_bar_info_blk_ack_seq_ctrl = -1;
static int hf_ieee80211_he_trigger_mpdu_mu_spacing = -1;
static int hf_ieee80211_he_trigger_tid_aggregation_limit = -1;
static int hf_ieee80211_he_trigger_dependent_reserved1 = -1;
static int hf_ieee80211_he_trigger_preferred_ac = -1;
static int hf_ieee80211_he_trigger_starting_aid = -1;
static int hf_ieee80211_he_trigger_dependent_reserved2 = -1;
static int hf_ieee80211_he_trigger_feedback_type = -1;
static int hf_ieee80211_he_trigger_dependent_reserved3 = -1;
static int hf_ieee80211_he_trigger_nfrp_target_rssi = -1;
static int hf_ieee80211_he_trigger_multiplexing_flag = -1;
static int hf_ieee80211_he_trigger_dep_nfrp_user_info = -1;
static int hf_ieee80211_he_trigger_feedback_seg_retrans_bm = -1;
static int hf_ieee80211_he_trigger_aid12 = -1;
static int hf_ieee80211_he_trigger_ru_allocation = -1;
static int hf_ieee80211_he_trigger_ru_allocation_region = -1;
static int hf_ieee80211_he_trigger_ru_starting_spatial_stream = -1;
static int hf_ieee80211_he_trigger_ru_number_spatial_streams = -1;
static int hf_ieee80211_he_trigger_ru_number_ra_ru = -1;
static int hf_ieee80211_he_trigger_ru_no_more_ra_ru = -1;
static int hf_ieee80211_he_trigger_ul_fec_coding_type = -1;
static int hf_ieee80211_he_trigger_ul_mcs = -1;
static int hf_ieee80211_he_trigger_ul_dcm = -1;
static int hf_ieee80211_he_trigger_ul_target_rssi = -1;
static int hf_ieee80211_he_trigger_user_reserved = -1;
static int hf_ieee80211_he_trigger_dep_basic_user_info = -1;
static int hf_ieee80211_he_ndp_annc_token = -1;
static int hf_ieee80211_he_ndp_annc_sta = -1;
static int hf_he_ndp_sounding_dialog_token_number = -1;
static int hf_he_ndp_annc_he_subfield = -1;
static int hf_he_ndp_annc_reserved = -1;
static int hf_he_ndp_annc_aid11 = -1;
static int hf_he_ndp_annc_ru_start = -1;
static int hf_he_ndp_annc_ru_end = -1;
static int hf_he_ndp_annc_feedback_type_and_ng = -1;
static int hf_he_ndp_annc_disambiguation = -1;
static int hf_he_ndp_annc_codebook_size = -1;
static int hf_he_ndp_annc_nc = -1;
static int hf_he_qtp_control = -1;
static int hf_he_qtp_setup_quiet_period_duration = -1;
static int hf_he_qtp_setup_srv_specific_identif = -1;
static int hf_he_qtp_request_dialog_token = -1;
static int hf_he_qtp_request_quiet_period_offset = -1;
static int hf_he_qtp_request_quiet_period_duration = -1;
static int hf_he_qtp_request_quiet_period_interval = -1;
static int hf_he_qtp_request_repetition_count = -1;
static int hf_he_qtp_request_srv_specific_identif = -1;
static int hf_he_qtp_response_dialog_token = -1;
static int hf_he_qtp_response_status_code = -1;
static int hf_he_qtp_response_quiet_period_offset = -1;
static int hf_he_qtp_response_quiet_period_duration = -1;
static int hf_he_qtp_response_quiet_period_interval = -1;
static int hf_he_qtp_response_repetition_count = -1;
static int hf_he_qtp_response_srv_specific_identif = -1;
static int hf_ieee80211_htc_ht_lac = -1;
static int hf_ieee80211_htc_lac_trq = -1;
static int hf_ieee80211_htc_lac_mai_aseli = -1;
static int hf_ieee80211_htc_lac_mai_mrq = -1;
static int hf_ieee80211_htc_lac_mai_msi = -1;
static int hf_ieee80211_htc_lac_mai_reserved = -1;
static int hf_ieee80211_htc_lac_mfsi = -1;
static int hf_ieee80211_htc_lac_mfb = -1;
static int hf_ieee80211_htc_lac_asel_command = -1;
static int hf_ieee80211_htc_lac_asel_data = -1;
static int hf_ieee80211_htc_cal_pos = -1;
static int hf_ieee80211_htc_cal_seq = -1;
static int hf_ieee80211_htc_reserved1 = -1;
static int hf_ieee80211_htc_csi_steering = -1;
static int hf_ieee80211_htc_ndp_announcement = -1;
static int hf_ieee80211_htc_reserved2 = -1;
static int hf_ieee80211_htc_mrq = -1;
static int hf_ieee80211_htc_msi = -1;
static int hf_ieee80211_htc_msi_stbc_reserved = -1;
static int hf_ieee80211_htc_compressed_msi = -1;
static int hf_ieee80211_htc_ppdu_stbc_encoded = -1;
static int hf_ieee80211_htc_mfsi = -1;
static int hf_ieee80211_htc_gid_l = -1;
static int hf_ieee80211_htc_mfb = -1;
static int hf_ieee80211_htc_num_sts = -1;
static int hf_ieee80211_htc_vht_mcs = -1;
static int hf_ieee80211_htc_bw = -1;
static int hf_ieee80211_htc_snr = -1;
static int hf_ieee80211_htc_reserved3 = -1;
static int hf_ieee80211_htc_gid_h = -1;
static int hf_ieee80211_htc_coding_type = -1;
static int hf_ieee80211_htc_fb_tx_type = -1;
static int hf_ieee80211_htc_unsolicited_mfb = -1;
static int hf_ieee80211_htc_ac_constraint = -1;
static int hf_ieee80211_htc_rdg_more_ppdu = -1;

/* ************************************************************************* */
/*                Header values for sequence number field                    */
/* ************************************************************************* */
static int hf_ieee80211_frag_number = -1;
static int hf_ieee80211_seq_number = -1;

/* ************************************************************************* */
/*                   Header values for Frame Check field                     */
/* ************************************************************************* */
static int hf_ieee80211_fcs = -1;
static int hf_ieee80211_fcs_status = -1;

/* ************************************************************************* */
/*                   Header values for reassembly                            */
/* ************************************************************************* */
static int hf_ieee80211_fragments = -1;
static int hf_ieee80211_fragment = -1;
static int hf_ieee80211_fragment_overlap = -1;
static int hf_ieee80211_fragment_overlap_conflict = -1;
static int hf_ieee80211_fragment_multiple_tails = -1;
static int hf_ieee80211_fragment_too_long_fragment = -1;
static int hf_ieee80211_fragment_error = -1;
static int hf_ieee80211_fragment_count = -1;
static int hf_ieee80211_reassembled_in = -1;
static int hf_ieee80211_reassembled_length = -1;

static int proto_wlan_ext = -1;

/* ************************************************************************* */
/*                      Fixed fields found in mgt frames                     */
/* ************************************************************************* */
static int hf_ieee80211_fixed_parameters = -1;  /* Protocol payload for management frames */

static int hf_ieee80211_ff_auth_alg = -1;            /* Authentication algorithm field            */
static int hf_ieee80211_ff_auth_seq = -1;            /* Authentication transaction sequence       */
static int hf_ieee80211_ff_current_ap = -1;          /* Current AP MAC address                    */
static int hf_ieee80211_ff_listen_ival = -1;         /* Listen interval fixed field               */
static int hf_ieee80211_ff_timestamp = -1;           /* 64 bit timestamp                          */
static int hf_ieee80211_ff_beacon_interval = -1;     /* 16 bit Beacon interval                    */
static int hf_ieee80211_ff_assoc_id = -1;            /* 16 bit AID field                          */
static int hf_ieee80211_ff_reason = -1;              /* 16 bit reason code                        */
static int hf_ieee80211_ff_status_code = -1;         /* Status code                               */
static int hf_ieee80211_ff_category_code = -1;       /* 8 bit Category code */
static int hf_ieee80211_ff_action_code = -1;         /* 8 bit Action code */
static int hf_ieee80211_ff_dialog_token = -1;        /* 8 bit Dialog token */
static int hf_ieee80211_ff_followup_dialog_token = -1;
static int hf_ieee80211_ff_wme_action_code = -1;     /* Management notification action code */
static int hf_ieee80211_ff_wme_status_code = -1;     /* Management notification setup response status code */
static int hf_ieee80211_ff_qos_action_code = -1;
static int hf_ieee80211_ff_dls_action_code = -1;
static int hf_ieee80211_ff_dst_mac_addr = -1;        /* DLS destination MAC addressi */
static int hf_ieee80211_ff_src_mac_addr = -1;        /* DLS source MAC addressi */
static int hf_ieee80211_ff_req_ap_addr = -1;
static int hf_ieee80211_ff_res_ap_addr = -1;
static int hf_ieee80211_ff_check_beacon = -1;
static int hf_ieee80211_ff_dls_timeout = -1;         /* DLS timeout value */
static int hf_ieee80211_ff_ft_action_code = -1; /* 8 bit FT Action code */
static int hf_ieee80211_ff_sta_address = -1;
static int hf_ieee80211_ff_target_ap_address = -1;
static int hf_ieee80211_ff_gas_comeback_delay = -1;
static int hf_ieee80211_ff_gas_fragment_id = -1;
static int hf_ieee80211_ff_more_gas_fragments = -1;
static int hf_ieee80211_ff_query_request_length = -1;
static int hf_ieee80211_ff_query_request = -1;
static int hf_ieee80211_ff_query_response_length = -1;
static int hf_ieee80211_ff_query_response = -1;
static int hf_ieee80211_ff_anqp_info_id = -1;
static int hf_ieee80211_ff_anqp_info_length = -1;
static int hf_ieee80211_ff_anqp_info = -1;
static int hf_ieee80211_ff_anqp_query_id = -1;
static int hf_ieee80211_ff_anqp_capability = -1;
static int hf_ieee80211_ff_anqp_capability_vlen = -1;
static int hf_ieee80211_ff_anqp_capability_vendor = -1;
static int hf_ieee80211_ff_venue_info_group = -1;
static int hf_ieee80211_ff_venue_info_type = -1;
static int hf_ieee80211_ff_anqp_venue_length = -1;
static int hf_ieee80211_ff_anqp_venue_language = -1;
static int hf_ieee80211_ff_anqp_venue_name = -1;
static int hf_ieee80211_ff_anqp_nw_auth_type_indicator = -1;
static int hf_ieee80211_ff_anqp_nw_auth_type_url_len = -1;
static int hf_ieee80211_ff_anqp_nw_auth_type_url = -1;
static int hf_ieee80211_ff_anqp_roaming_consortium_oi_len = -1;
static int hf_ieee80211_ff_anqp_roaming_consortium_oi = -1;
static int hf_ieee80211_ff_anqp_ip_addr_avail_ipv6 = -1;
static int hf_ieee80211_ff_anqp_ip_addr_avail_ipv4 = -1;
static int hf_ieee80211_ff_anqp_nai_realm_count = -1;
static int hf_ieee80211_ff_anqp_nai_field_len = -1;
static int hf_ieee80211_ff_anqp_nai_realm_encoding = -1;
static int hf_ieee80211_ff_anqp_nai_realm_length = -1;
static int hf_ieee80211_ff_anqp_nai_realm = -1;
static int hf_ieee80211_ff_anqp_nai_realm_eap_count = -1;
static int hf_ieee80211_ff_anqp_nai_realm_eap_len = -1;
static int hf_ieee80211_ff_anqp_nai_realm_eap_method = -1;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_count = -1;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_id = -1;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_len = -1;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_value = -1;
static int hf_ieee80211_3gpp_gc_gud = -1;
static int hf_ieee80211_3gpp_gc_udhl = -1;
static int hf_ieee80211_3gpp_gc_iei = -1;
static int hf_ieee80211_3gpp_gc_num_plmns = -1;
static int hf_ieee80211_3gpp_gc_plmn = -1;
static int hf_ieee80211_3gpp_gc_plmn_len = -1;
static int hf_ieee80211_ff_anqp_domain_name_len = -1;
static int hf_ieee80211_ff_anqp_domain_name = -1;
static int hf_ieee80211_ff_tdls_action_code = -1;
static int hf_ieee80211_ff_target_channel = -1;
static int hf_ieee80211_ff_operating_class = -1;
static int hf_ieee80211_ff_wnm_action_code = -1;
static int hf_ieee80211_ff_unprotected_wnm_action_code = -1;
static int hf_ieee80211_ff_key_data_length = -1;
static int hf_ieee80211_ff_key_data = -1;
static int hf_ieee80211_ff_wnm_notification_type = -1;
static int hf_ieee80211_ff_rm_action_code = -1;
static int hf_ieee80211_ff_rm_dialog_token = -1;
static int hf_ieee80211_ff_rm_repetitions = -1;
static int hf_ieee80211_ff_rm_tx_power = -1;
static int hf_ieee80211_ff_rm_max_tx_power = -1;
static int hf_ieee80211_ff_tpc = -1;
static int hf_ieee80211_ff_tpc_element_id = -1;
static int hf_ieee80211_ff_tpc_length = -1;
static int hf_ieee80211_ff_tpc_tx_power = -1;
static int hf_ieee80211_ff_tpc_link_margin = -1;
static int hf_ieee80211_ff_rm_rx_antenna_id = -1;
static int hf_ieee80211_ff_rm_tx_antenna_id = -1;
static int hf_ieee80211_ff_rm_rcpi = -1;
static int hf_ieee80211_ff_rm_rsni = -1;
static int hf_ieee80211_ff_request_mode_pref_cand = -1;
static int hf_ieee80211_ff_request_mode_abridged = -1;
static int hf_ieee80211_ff_request_mode_disassoc_imminent = -1;
static int hf_ieee80211_ff_request_mode_bss_term_included = -1;
static int hf_ieee80211_ff_request_mode_ess_disassoc_imminent = -1;
static int hf_ieee80211_ff_disassoc_timer = -1;
static int hf_ieee80211_ff_validity_interval = -1;
static int hf_ieee80211_ff_bss_termination_duration = -1;
static int hf_ieee80211_ff_url_len = -1;
static int hf_ieee80211_ff_url = -1;
static int hf_ieee80211_ff_target_bss = -1;
static int hf_ieee80211_ff_bss_transition_query_reason = -1;
static int hf_ieee80211_ff_bss_transition_status_code = -1;
static int hf_ieee80211_ff_bss_termination_delay = -1;
static int hf_ieee80211_ff_bss_transition_candidate_list_entries = -1;

static int hf_ieee80211_ff_sa_query_action_code = -1;
static int hf_ieee80211_ff_transaction_id = -1;

static int hf_ieee80211_ff_send_confirm = -1;
static int hf_ieee80211_ff_anti_clogging_token = -1;
static int hf_ieee80211_ff_scalar = -1;
static int hf_ieee80211_ff_finite_field_element = -1;
static int hf_ieee80211_ff_confirm = -1;
static int hf_ieee80211_ff_finite_cyclic_group = -1;
static int hf_ieee80211_ff_sae_message_type = -1;

/* Vendor specific */
static int hf_ieee80211_ff_marvell_action_type = -1;
static int hf_ieee80211_ff_marvell_mesh_mgt_action_code = -1;
static int hf_ieee80211_ff_marvell_mesh_mgt_length = -1;     /* Mesh Management length */
static int hf_ieee80211_ff_marvell_mesh_mgt_mode = -1;       /* Mesh Management mode */
static int hf_ieee80211_ff_marvell_mesh_mgt_ttl = -1;        /* Mesh Management TTL */
static int hf_ieee80211_ff_marvell_mesh_mgt_dstcount = -1;   /* Mesh Management dst count */
static int hf_ieee80211_ff_marvell_mesh_mgt_hopcount = -1;   /* Mesh Management hop count */
static int hf_ieee80211_ff_marvell_mesh_mgt_rreqid = -1;     /* Mesh Management RREQ ID */
static int hf_ieee80211_ff_marvell_mesh_mgt_sa = -1;         /* Mesh Management src addr */
static int hf_ieee80211_ff_marvell_mesh_mgt_ssn = -1;        /* Mesh Management src sequence number */
static int hf_ieee80211_ff_marvell_mesh_mgt_metric = -1;     /* Mesh Management metric */
static int hf_ieee80211_ff_marvell_mesh_mgt_flags = -1;      /* Mesh Management RREQ flags */
static int hf_ieee80211_ff_marvell_mesh_mgt_da = -1;         /* Mesh Management dst addr */
static int hf_ieee80211_ff_marvell_mesh_mgt_dsn = -1;        /* Mesh Management dst sequence number */
static int hf_ieee80211_ff_marvell_mesh_mgt_lifetime = -1;   /* Mesh Management lifetime */


static int hf_ieee80211_ff_ba_action = -1;

static int hf_ieee80211_ff_block_ack_params = -1;
static int hf_ieee80211_ff_block_ack_params_amsdu_permitted = -1;
static int hf_ieee80211_ff_block_ack_params_policy = -1;
static int hf_ieee80211_ff_block_ack_params_tid = -1;
static int hf_ieee80211_ff_block_ack_params_buffer_size = -1;

static const int *ieee80211_ff_block_ack_params_fields[] = {
  &hf_ieee80211_ff_block_ack_params_amsdu_permitted,
  &hf_ieee80211_ff_block_ack_params_policy,
  &hf_ieee80211_ff_block_ack_params_tid,
  &hf_ieee80211_ff_block_ack_params_buffer_size,
  NULL
};

static int hf_ieee80211_ff_block_ack_timeout = -1;

static int hf_ieee80211_ff_block_ack_ssc = -1;
static int hf_ieee80211_ff_block_ack_ssc_fragment = -1;
static int hf_ieee80211_ff_block_ack_ssc_sequence = -1;

static const int *ieee80211_ff_block_ack_ssc_fields[] = {
  &hf_ieee80211_ff_block_ack_ssc_fragment,
  &hf_ieee80211_ff_block_ack_ssc_sequence,
  NULL
};

static int hf_ieee80211_ff_delba_param = -1;
static int hf_ieee80211_ff_delba_param_reserved = -1;
static int hf_ieee80211_ff_delba_param_init = -1;
static int hf_ieee80211_ff_delba_param_tid = -1;

static const int *ieee80211_ff_delba_param_fields[] = {
  &hf_ieee80211_ff_delba_param_reserved,
  &hf_ieee80211_ff_delba_param_init,
  &hf_ieee80211_ff_delba_param_tid,
  NULL
};

static int hf_ieee80211_ff_max_reg_pwr = -1;
static int hf_ieee80211_ff_measurement_pilot_int = -1;
static int hf_ieee80211_ff_country_str = -1;
static int hf_ieee80211_ff_max_tx_pwr = -1;
static int hf_ieee80211_ff_tx_pwr_used = -1;
static int hf_ieee80211_ff_transceiver_noise_floor = -1;
static int hf_ieee80211_ff_channel_width = -1;

static int hf_ieee80211_ff_qos_info_ap = -1;
static int hf_ieee80211_ff_qos_info_ap_edca_param_set_counter = -1;
static int hf_ieee80211_ff_qos_info_ap_q_ack = -1;
static int hf_ieee80211_ff_qos_info_ap_queue_req = -1;
static int hf_ieee80211_ff_qos_info_ap_txop_request = -1;
static int hf_ieee80211_ff_qos_info_ap_reserved = -1;

static const int *ieee80211_ff_qos_info_ap_fields[] = {
  &hf_ieee80211_ff_qos_info_ap_edca_param_set_counter,
  &hf_ieee80211_ff_qos_info_ap_q_ack,
  &hf_ieee80211_ff_qos_info_ap_queue_req,
  &hf_ieee80211_ff_qos_info_ap_txop_request,
  &hf_ieee80211_ff_qos_info_ap_reserved,
  NULL
};

static int hf_ieee80211_ff_qos_info_sta = -1;
static int hf_ieee80211_ff_qos_info_sta_ac_vo = -1;
static int hf_ieee80211_ff_qos_info_sta_ac_vi = -1;
static int hf_ieee80211_ff_qos_info_sta_ac_bk = -1;
static int hf_ieee80211_ff_qos_info_sta_ac_be = -1;
static int hf_ieee80211_ff_qos_info_sta_q_ack = -1;
static int hf_ieee80211_ff_qos_info_sta_max_sp_length = -1;
static int hf_ieee80211_ff_qos_info_sta_more_data_ack = -1;

static const int *ieee80211_ff_qos_info_sta_fields[] = {
  &hf_ieee80211_ff_qos_info_sta_ac_vo,
  &hf_ieee80211_ff_qos_info_sta_ac_vi,
  &hf_ieee80211_ff_qos_info_sta_ac_bk,
  &hf_ieee80211_ff_qos_info_sta_ac_be,
  &hf_ieee80211_ff_qos_info_sta_q_ack,
  &hf_ieee80211_ff_qos_info_sta_max_sp_length,
  &hf_ieee80211_ff_qos_info_sta_more_data_ack,
  NULL
};

static int hf_ieee80211_ff_sm_pwr_save = -1;
static int hf_ieee80211_ff_sm_pwr_save_enabled = -1;
static int hf_ieee80211_ff_sm_pwr_save_sm_mode = -1;
static int hf_ieee80211_ff_sm_pwr_save_reserved = -1;

static const int *ieee80211_ff_sw_pwr_save_fields[] = {
  &hf_ieee80211_ff_sm_pwr_save_enabled,
  &hf_ieee80211_ff_sm_pwr_save_sm_mode,
  &hf_ieee80211_ff_sm_pwr_save_reserved,
  NULL
};

static int hf_ieee80211_ff_pco_phase_cntrl = -1;

static int hf_ieee80211_ff_psmp_param_set = -1;
static int hf_ieee80211_ff_psmp_param_set_n_sta = -1;
static int hf_ieee80211_ff_psmp_param_set_more_psmp = -1;
static int hf_ieee80211_ff_psmp_param_set_psmp_sequence_duration = -1;

static const int *ieee80211_ff_psmp_param_set_fields[] = {
  &hf_ieee80211_ff_psmp_param_set_n_sta,
  &hf_ieee80211_ff_psmp_param_set_more_psmp,
  &hf_ieee80211_ff_psmp_param_set_psmp_sequence_duration,
  NULL
};

static int hf_ieee80211_ff_mimo_cntrl = -1;
static int hf_ieee80211_ff_mimo_cntrl_nc_index = -1;
static int hf_ieee80211_ff_mimo_cntrl_nr_index = -1;
static int hf_ieee80211_ff_mimo_cntrl_channel_width = -1;
static int hf_ieee80211_ff_mimo_cntrl_grouping = -1;
static int hf_ieee80211_ff_mimo_cntrl_coefficient_size = -1;
static int hf_ieee80211_ff_mimo_cntrl_codebook_info = -1;
static int hf_ieee80211_ff_mimo_cntrl_remaining_matrix_segment = -1;
static int hf_ieee80211_ff_mimo_cntrl_reserved = -1;
static int hf_ieee80211_ff_mimo_cntrl_sounding_timestamp = -1;

static int hf_ieee80211_ff_ant_selection = -1;
static int hf_ieee80211_ff_ant_selection_0 = -1;
static int hf_ieee80211_ff_ant_selection_1 = -1;
static int hf_ieee80211_ff_ant_selection_2 = -1;
static int hf_ieee80211_ff_ant_selection_3 = -1;
static int hf_ieee80211_ff_ant_selection_4 = -1;
static int hf_ieee80211_ff_ant_selection_5 = -1;
static int hf_ieee80211_ff_ant_selection_6 = -1;
static int hf_ieee80211_ff_ant_selection_7 = -1;

static const int *ieee80211_ff_ant_selection_fields[] = {
  &hf_ieee80211_ff_ant_selection_0,
  &hf_ieee80211_ff_ant_selection_1,
  &hf_ieee80211_ff_ant_selection_2,
  &hf_ieee80211_ff_ant_selection_3,
  &hf_ieee80211_ff_ant_selection_4,
  &hf_ieee80211_ff_ant_selection_5,
  &hf_ieee80211_ff_ant_selection_6,
  &hf_ieee80211_ff_ant_selection_7,
  NULL
};

static int hf_ieee80211_ff_ext_channel_switch_announcement = -1;
static int hf_ieee80211_ff_ext_channel_switch_announcement_switch_mode = -1;
static int hf_ieee80211_ff_ext_channel_switch_announcement_new_ope_class = -1;
static int hf_ieee80211_ff_ext_channel_switch_announcement_new_chan_number = -1;
static int hf_ieee80211_ff_ext_channel_switch_announcement_switch_count = -1;

static const int *ieee80211_ff_ext_channel_switch_announcement_fields[] = {
  &hf_ieee80211_ff_ext_channel_switch_announcement_switch_mode,
  &hf_ieee80211_ff_ext_channel_switch_announcement_new_ope_class,
  &hf_ieee80211_ff_ext_channel_switch_announcement_new_chan_number,
  &hf_ieee80211_ff_ext_channel_switch_announcement_switch_count,
  NULL
};

static int hf_ieee80211_ff_ht_info = -1;
static int hf_ieee80211_ff_ht_info_information_request = -1;
static int hf_ieee80211_ff_ht_info_40_mhz_intolerant = -1;
static int hf_ieee80211_ff_ht_info_sta_chan_width = -1;
static int hf_ieee80211_ff_ht_info_reserved = -1;

static const int *ieee80211_ff_ht_info_fields[] = {
  &hf_ieee80211_ff_ht_info_information_request,
  &hf_ieee80211_ff_ht_info_40_mhz_intolerant,
  &hf_ieee80211_ff_ht_info_sta_chan_width,
  &hf_ieee80211_ff_ht_info_reserved,
  NULL
};

static int hf_ieee80211_ff_ht_action = -1;

static int hf_ieee80211_ff_psmp_sta_info = -1;
static int hf_ieee80211_ff_psmp_sta_info_type = -1;
static int hf_ieee80211_ff_psmp_sta_info_dtt_start_offset = -1;
static int hf_ieee80211_ff_psmp_sta_info_dtt_duration = -1;
static int hf_ieee80211_ff_psmp_sta_info_sta_id = -1;
static int hf_ieee80211_ff_psmp_sta_info_utt_start_offset = -1;
static int hf_ieee80211_ff_psmp_sta_info_utt_duration = -1;
static int hf_ieee80211_ff_psmp_sta_info_reserved_small= -1;
static int hf_ieee80211_ff_psmp_sta_info_reserved_large = -1;
static int hf_ieee80211_ff_psmp_sta_info_psmp_multicast_id = -1;

static int hf_ieee80211_ff_mimo_csi_snr = -1;
static int hf_ieee80211_ff_mimo_csi_matrices = -1;
static int hf_ieee80211_ff_mimo_csi_bf_matrices = -1;
static int hf_ieee80211_ff_mimo_csi_cbf_matrices = -1;

/*** Begin: 802.11s additions ***/
static int hf_ieee80211_mesh_control_field = -1;

static int hf_ieee80211_ff_mesh_action = -1;
static int hf_ieee80211_ff_multihop_action = -1;
static int hf_ieee80211_ff_mesh_flags = -1;
static int hf_ieee80211_ff_mesh_ttl = -1;
static int hf_ieee80211_ff_mesh_sequence = -1;
static int hf_ieee80211_ff_mesh_addr4 = -1;
static int hf_ieee80211_ff_mesh_addr5 = -1;
static int hf_ieee80211_ff_mesh_addr6 = -1;
static int hf_ieee80211_ff_selfprot_action = -1;

static int hf_ieee80211_mesh_peering_proto = -1;
static int hf_ieee80211_mesh_peering_local_link_id = -1;
static int hf_ieee80211_mesh_peering_peer_link_id = -1;

static int hf_ieee80211_ff_hwmp_flags = -1;
static int hf_ieee80211_ff_hwmp_hopcount = -1;
static int hf_ieee80211_ff_hwmp_ttl = -1;
static int hf_ieee80211_ff_hwmp_pdid = -1;
static int hf_ieee80211_ff_hwmp_orig_sta = -1;
static int hf_ieee80211_ff_hwmp_orig_sn = -1;
static int hf_ieee80211_ff_hwmp_orig_ext = -1;
static int hf_ieee80211_ff_hwmp_lifetime = -1;
static int hf_ieee80211_ff_hwmp_metric = -1;
static int hf_ieee80211_ff_hwmp_targ_count = -1;
static int hf_ieee80211_ff_hwmp_targ_flags = -1;
static int hf_ieee80211_ff_hwmp_targ_to_flags = -1;
static int hf_ieee80211_ff_hwmp_targ_usn_flags = -1;
static int hf_ieee80211_ff_hwmp_targ_sta = -1;
static int hf_ieee80211_ff_hwmp_targ_sn = -1;
static int hf_ieee80211_ff_hwmp_targ_ext = -1;
static int hf_ieee80211_rann_flags = -1;
static int hf_ieee80211_rann_root_sta = -1;
static int hf_ieee80211_rann_sn = -1;
static int hf_ieee80211_rann_interval = -1;

static int hf_ieee80211_mesh_channel_switch_ttl = -1;
static int hf_ieee80211_mesh_channel_switch_flag = -1;
static int hf_ieee80211_mesh_channel_switch_reason_code = -1;
static int hf_ieee80211_mesh_channel_switch_precedence_value = -1;
static int hf_ieee80211_mesh_chswitch_flag_txrestrict = -1;
static int hf_ieee80211_mesh_chswitch_flag_initiator = -1;

static int hf_ieee80211_mesh_config_path_sel_protocol = -1;
static int hf_ieee80211_mesh_config_path_sel_metric = -1;
static int hf_ieee80211_mesh_config_congestion_control = -1;
static int hf_ieee80211_mesh_config_sync_method = -1;
static int hf_ieee80211_mesh_config_auth_protocol = -1;
static int hf_ieee80211_mesh_config_formation_info = -1;
static int hf_ieee80211_mesh_config_capability = -1;
static int hf_ieee80211_mesh_id = -1;
static int hf_ieee80211_mesh_config_cap_accepting = -1;
static int hf_ieee80211_mesh_config_cap_mcca_support = -1;
static int hf_ieee80211_mesh_config_cap_mcca_enabled = -1;
static int hf_ieee80211_mesh_config_cap_forwarding = -1;
static int hf_ieee80211_mesh_config_cap_mbca_enabled = -1;
static int hf_ieee80211_mesh_config_cap_tbtt_adjusting = -1;
static int hf_ieee80211_mesh_config_cap_power_save_level = -1;
static int hf_ieee80211_mesh_form_info_num_of_peerings = -1;
static int hf_ieee80211_mesh_awake_window = -1;

static int hf_ieee80211_ff_public_action = -1;
static int hf_ieee80211_ff_protected_public_action = -1;
static int hf_ieee80211_ff_tod = -1;
static int hf_ieee80211_ff_toa = -1;
static int hf_ieee80211_ff_max_tod_err = -1;
static int hf_ieee80211_ff_max_toa_err = -1;

/* ************************************************************************* */
/*            Flags found in the capability field (fixed field)              */
/* ************************************************************************* */
static int hf_ieee80211_ff_capture = -1;
static int hf_ieee80211_ff_cf_ess = -1;
static int hf_ieee80211_ff_cf_ibss = -1;
static int hf_ieee80211_ff_cf_sta_poll = -1; /* CF pollable status for a STA            */
static int hf_ieee80211_ff_cf_ap_poll = -1;  /* CF pollable status for an AP            */
static int hf_ieee80211_ff_cf_privacy = -1;
static int hf_ieee80211_ff_cf_preamble = -1;
static int hf_ieee80211_ff_cf_pbcc = -1;
static int hf_ieee80211_ff_cf_agility = -1;
static int hf_ieee80211_ff_short_slot_time = -1;
static int hf_ieee80211_ff_dsss_ofdm = -1;
static int hf_ieee80211_ff_cf_spec_man = -1;
static int hf_ieee80211_ff_cf_apsd = -1;
static int hf_ieee80211_ff_radio_measurement = -1;
static int hf_ieee80211_ff_cf_del_blk_ack = -1;
static int hf_ieee80211_ff_cf_imm_blk_ack = -1;

/* ************************************************************************* */
/*                       A-MSDU fields                                       */
/* ************************************************************************* */
static int hf_ieee80211_amsdu_subframe = -1;
static int hf_ieee80211_amsdu_length = -1;

/* ************************************************************************* */
/*                       Tagged value format fields                          */
/* ************************************************************************* */
static int hf_ieee80211_tagged_parameters = -1;  /* Tagged payload item */
static int hf_ieee80211_tag = -1;
static int hf_ieee80211_tag_number = -1;
static int hf_ieee80211_tag_length = -1;
static int hf_ieee80211_tag_data = -1;
static int hf_ieee80211_tag_oui = -1;
static int hf_ieee80211_tag_oui_wfa_subtype = -1;
static int hf_ieee80211_tag_ssid = -1;
static int hf_ieee80211_tag_supp_rates = -1;
static int hf_ieee80211_tag_fh_dwell_time = -1;
static int hf_ieee80211_tag_fh_hop_set = -1;
static int hf_ieee80211_tag_fh_hop_pattern = -1;
static int hf_ieee80211_tag_fh_hop_index = -1;
static int hf_ieee80211_tag_ds_param_channel = -1;
static int hf_ieee80211_tag_cfp_count = -1;
static int hf_ieee80211_tag_cfp_period = -1;
static int hf_ieee80211_tag_cfp_max_duration = -1;
static int hf_ieee80211_tag_cfp_dur_remaining = -1;
static int hf_ieee80211_tim_dtim_count = -1;
static int hf_ieee80211_tim_dtim_period = -1;
static int hf_ieee80211_tim_bmapctl = -1;
static int hf_ieee80211_tim_bmapctl_mcast = -1;
static int hf_ieee80211_tim_bmapctl_offset = -1;
static int hf_ieee80211_tim_partial_virtual_bitmap = -1;
static int hf_ieee80211_tim_aid = -1;
static int hf_ieee80211_tag_ibss_atim_window = -1;
static int hf_ieee80211_tag_country_info_code = -1;
static int hf_ieee80211_tag_country_info_env = -1;
static int hf_ieee80211_tag_country_info_pad = -1;
static int hf_ieee80211_tag_country_info_fnm = -1;
static int hf_ieee80211_tag_country_info_fnm_fcn = -1;
static int hf_ieee80211_tag_country_info_fnm_nc = -1;
static int hf_ieee80211_tag_country_info_fnm_mtpl = -1;
static int hf_ieee80211_tag_country_info_rrc = -1;
static int hf_ieee80211_tag_country_info_rrc_oei = -1;
static int hf_ieee80211_tag_country_info_rrc_oc = -1;
static int hf_ieee80211_tag_country_info_rrc_cc = -1;
static int hf_ieee80211_tag_fh_hopping_parameter_prime_radix = -1;
static int hf_ieee80211_tag_fh_hopping_parameter_nb_channels = -1;
static int hf_ieee80211_tag_fh_hopping_table_flag = -1;
static int hf_ieee80211_tag_fh_hopping_table_number_of_sets = -1;
static int hf_ieee80211_tag_fh_hopping_table_modulus = -1;
static int hf_ieee80211_tag_fh_hopping_table_offset = -1;
static int hf_ieee80211_tag_fh_hopping_random_table = -1;
static int hf_ieee80211_tag_request = -1;
static int hf_ieee80211_tag_challenge_text = -1;

static int hf_ieee80211_wep_iv = -1;
static int hf_ieee80211_wep_iv_weak = -1;
static int hf_ieee80211_tkip_extiv = -1;
static int hf_ieee80211_ccmp_extiv = -1;
static int hf_ieee80211_wep_key = -1;
static int hf_ieee80211_wep_icv = -1;
static int hf_ieee80211_fc_analysis_pmk = -1;
static int hf_ieee80211_fc_analysis_kck = -1;
static int hf_ieee80211_fc_analysis_kek = -1;
static int hf_ieee80211_fc_analysis_tk = -1;
static int hf_ieee80211_fc_analysis_gtk = -1;

static int hf_ieee80211_block_ack_control = -1;
static int hf_ieee80211_block_ack_control_ack_policy = -1;
static int hf_ieee80211_block_ack_control_type = -1;
static int hf_ieee80211_block_ack_control_reserved = -1;
static int hf_ieee80211_block_ack_control_tid_info = -1;

static int hf_ieee80211_block_ack_multi_tid_reserved = -1;
static int hf_ieee80211_block_ack_multi_tid_value = -1;
static int hf_ieee80211_block_ack_bitmap = -1;
static int hf_ieee80211_block_ack_bitmap_missing_frame = -1;
static int hf_ieee80211_block_ack_gcr_addr = -1;

static int hf_ieee80211_block_ack_multi_sta_aid11 = -1;
static int hf_ieee80211_block_ack_multi_sta_ack_type = -1;
static int hf_ieee80211_block_ack_multi_sta_tid = -1;
static int hf_ieee80211_block_ack_multi_sta_aid_tid = -1;
static int hf_ieee80211_block_ack_multi_sta_reserved = -1;
static int hf_ieee80211_block_ack_multi_sta_ra = -1;

static int hf_ieee80211_tag_measure_request_measurement_mode = -1;
static int hf_ieee80211_tag_measure_request_bssid = -1;

static int hf_ieee80211_tag_measure_request_subelement_length = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_id = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_ssid = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_bri_reporting_condition = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_bri_threshold_offset = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_reporting_detail = -1;
static int hf_ieee80211_tag_measure_request_beacon_sub_request = -1;
static int hf_ieee80211_tag_measure_request_beacon_unknown = -1;

static int hf_ieee80211_tag_measure_request_channel_load_sub_id = -1;
static int hf_ieee80211_tag_measure_request_channel_load_sub_reporting_condition = -1;
static int hf_ieee80211_tag_measure_request_channel_load_sub_reporting_ref = -1;

static int hf_ieee80211_tag_measure_request_noise_histogram_sub_id = -1;
static int hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_condition = -1;
static int hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_anpi_ref = -1;

static int hf_ieee80211_tag_measure_request_frame_request_type = -1;
static int hf_ieee80211_tag_measure_request_mac_address  = -1;
static int hf_ieee80211_tag_measure_request_peer_mac_address = -1;
static int hf_ieee80211_tag_measure_request_group_id = -1;

static int hf_ieee80211_tag_measure_request_unknown = -1;

static int hf_ieee80211_ht_pren_type = -1;
static int hf_ieee80211_ht_pren_unknown = -1;

static int hf_ieee80211_ht_cap = -1;
static int hf_ieee80211_ht_vs_cap = -1;
static int hf_ieee80211_ht_ldpc_coding = -1;
static int hf_ieee80211_ht_chan_width = -1;
static int hf_ieee80211_ht_sm_pwsave = -1;
static int hf_ieee80211_ht_green = -1;
static int hf_ieee80211_ht_short20 = -1;
static int hf_ieee80211_ht_short40 = -1;
static int hf_ieee80211_ht_tx_stbc = -1;
static int hf_ieee80211_ht_rx_stbc = -1;
static int hf_ieee80211_ht_delayed_block_ack = -1;
static int hf_ieee80211_ht_max_amsdu = -1;
static int hf_ieee80211_ht_dss_cck_40 = -1;
static int hf_ieee80211_ht_psmp = -1;
static int hf_ieee80211_ht_40_mhz_intolerant = -1;
static int hf_ieee80211_ht_l_sig = -1;

static int hf_ieee80211_ext_bss_mu_mimo_capable_sta_count = -1;
static int hf_ieee80211_ext_bss_ss_underutilization = -1;
static int hf_ieee80211_ext_bss_observable_sec_20mhz_utilization = -1;
static int hf_ieee80211_ext_bss_observable_sec_40mhz_utilization = -1;
static int hf_ieee80211_ext_bss_observable_sec_80mhz_utilization = -1;
static int hf_ieee80211_wide_bw_new_channel_width = -1;
static int hf_ieee80211_wide_bw_new_channel_center_freq_segment0 = -1;
static int hf_ieee80211_wide_bw_new_channel_center_freq_segment1 = -1;

static int hf_ieee80211_operat_notification_mode = -1;
static int hf_ieee80211_operat_mode_field_channel_width = -1;
static int hf_ieee80211_operat_mode_field_reserved = -1;
static int hf_ieee80211_operat_mode_field_rxnss = -1;
static int hf_ieee80211_operat_mode_field_rxnsstype= -1;
static int hf_ieee80211_ampduparam = -1;
static int hf_ieee80211_ampduparam_vs = -1;
static int hf_ieee80211_ampduparam_mpdu = -1;
static int hf_ieee80211_ampduparam_mpdu_start_spacing = -1;
static int hf_ieee80211_ampduparam_reserved = -1;

static int hf_ieee80211_mcsset = -1;
static int hf_ieee80211_mcsset_vs = -1;
static int hf_ieee80211_mcsset_rx_bitmask = -1;
static int hf_ieee80211_mcsset_rx_bitmask_0to7 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_8to15 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_16to23 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_24to31 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_32 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_33to38 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_39to52 = -1;
static int hf_ieee80211_mcsset_rx_bitmask_53to76 = -1;
static int hf_ieee80211_mcsset_highest_data_rate = -1;
static int hf_ieee80211_mcsset_tx_mcs_set_defined = -1;
static int hf_ieee80211_mcsset_tx_rx_mcs_set_not_equal = -1;
static int hf_ieee80211_mcsset_tx_max_spatial_streams = -1;
static int hf_ieee80211_mcsset_tx_unequal_modulation = -1;

static int hf_ieee80211_htex_cap = -1;
static int hf_ieee80211_htex_vs_cap = -1;
static int hf_ieee80211_htex_pco = -1;
static int hf_ieee80211_htex_transtime = -1;
static int hf_ieee80211_htex_mcs = -1;
static int hf_ieee80211_htex_htc_support = -1;
static int hf_ieee80211_htex_rd_responder = -1;

static int hf_ieee80211_txbf = -1;
static int hf_ieee80211_txbf_vs = -1;
static int hf_ieee80211_txbf_cap = -1;
static int hf_ieee80211_txbf_rcv_ssc = -1;
static int hf_ieee80211_txbf_tx_ssc = -1;
static int hf_ieee80211_txbf_rcv_ndp = -1;
static int hf_ieee80211_txbf_tx_ndp = -1;
static int hf_ieee80211_txbf_impl_txbf = -1;
static int hf_ieee80211_txbf_calib = -1;
static int hf_ieee80211_txbf_expl_csi = -1;
static int hf_ieee80211_txbf_expl_uncomp_fm = -1;
static int hf_ieee80211_txbf_expl_comp_fm = -1;
static int hf_ieee80211_txbf_expl_bf_csi = -1;
static int hf_ieee80211_txbf_expl_uncomp_fm_feed = -1;
static int hf_ieee80211_txbf_expl_comp_fm_feed = -1;
static int hf_ieee80211_txbf_csi_num_bf_ant = -1;
static int hf_ieee80211_txbf_min_group = -1;
static int hf_ieee80211_txbf_uncomp_sm_bf_ant = -1;
static int hf_ieee80211_txbf_comp_sm_bf_ant = -1;
static int hf_ieee80211_txbf_csi_max_rows_bf = -1;
static int hf_ieee80211_txbf_chan_est = -1;
static int hf_ieee80211_txbf_resrv = -1;

/*** Begin: 802.11n D1.10 - HT Information IE  ***/
static int hf_ieee80211_ht_info_primary_channel = -1;

static int hf_ieee80211_ht_info_delimiter1 = -1;
static int hf_ieee80211_ht_info_secondary_channel_offset = -1;
static int hf_ieee80211_ht_info_sta_channel_width = -1;
static int hf_ieee80211_ht_info_rifs_mode = -1;
static int hf_ieee80211_ht_info_reserved_b4_b7 = -1;

static int hf_ieee80211_ht_info_delimiter2 = -1;
static int hf_ieee80211_ht_info_protection = -1;
static int hf_ieee80211_ht_info_non_greenfield_sta_present = -1;
static int hf_ieee80211_ht_info_reserved_b11 = -1;
static int hf_ieee80211_ht_info_obss_non_ht_stas_present = -1;
static int hf_ieee80211_ht_info_channel_center_freq_seg_2 = -1;
static int hf_ieee80211_ht_info_reserved_b21_b23 = -1;

static int hf_ieee80211_ht_info_delimiter3 = -1;
static int hf_ieee80211_ht_info_reserved_b24_b29 = -1;
static int hf_ieee80211_ht_info_dual_beacon = -1;
static int hf_ieee80211_ht_info_dual_cts_protection = -1;
static int hf_ieee80211_ht_info_secondary_beacon = -1;
static int hf_ieee80211_ht_info_lsig_txop_protection_full_support = -1;
static int hf_ieee80211_ht_info_pco_active = -1;
static int hf_ieee80211_ht_info_pco_phase = -1;
static int hf_ieee80211_ht_info_reserved_b36_b39 = -1;
/*** End: 802.11n D1.10 - HT Information IE  ***/

static int hf_ieee80211_tag_ap_channel_report_operating_class = -1;
static int hf_ieee80211_tag_ap_channel_report_channel_list = -1;

static int hf_ieee80211_tag_secondary_channel_offset = -1;

static int hf_ieee80211_tag_bss_ap_avg_access_delay = -1;

static int hf_ieee80211_tag_antenna_id = -1;

static int hf_ieee80211_tag_rsni = -1;

static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up0 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up1 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up2 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up3 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up4 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up5 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up6 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up7 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac0 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac1 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac2 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac3 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_rsv = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up0 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up1 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up2 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up3 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up4 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up5 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up6 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up7 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac0 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac1 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac2 = -1;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac3 = -1;

static int hf_ieee80211_tag_bss_avg_ac_access_delay_be = -1;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_bk = -1;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_vi = -1;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_vo = -1;

static int hf_ieee80211_tag_rm_enabled_capabilities = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b0 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b1 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b2 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b3 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b4 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b5 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b6 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b7 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b8 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b9 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b10 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b11 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b12 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b13 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b14 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b15 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b16 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b17 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b18to20 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b21to23 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b24to26 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b27 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b28 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b29 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b30 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b31 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b32 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b33 = -1;
static int hf_ieee80211_tag_rm_enabled_capabilities_o5 = -1;

static int hf_ieee80211_tag_rcpi = -1;
static int hf_ieee80211_tag_multiple_bssid = -1;
static int hf_ieee80211_tag_multiple_bssid_subelem_id = -1;
static int hf_ieee80211_tag_multiple_bssid_subelem_len = -1;
static int hf_ieee80211_tag_multiple_bssid_subelem_reserved = -1;
static int hf_ieee80211_tag_multiple_bssid_subelem_nontrans_profile = -1;

static int hf_ieee80211_tag_20_40_bc = -1;
static int hf_ieee80211_tag_20_40_bc_information_request = -1;
static int hf_ieee80211_tag_20_40_bc_forty_mhz_intolerant = -1;
static int hf_ieee80211_tag_20_40_bc_20_mhz_bss_witdh_request = -1;
static int hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_request = -1;
static int hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_grant = -1;
static int hf_ieee80211_tag_20_40_bc_reserved = -1;

static int hf_ieee80211_tag_power_constraint_local = -1;

static int hf_ieee80211_tag_power_capability_min = -1;
static int hf_ieee80211_tag_power_capability_max = -1;

static int hf_ieee80211_tag_tpc_report_trsmt_pow = -1;
static int hf_ieee80211_tag_tpc_report_link_mrg = -1;

static int hf_ieee80211_tag_supported_channels = -1;
static int hf_ieee80211_tag_supported_channels_first = -1;
static int hf_ieee80211_tag_supported_channels_range = -1;

static int hf_ieee80211_csa_channel_switch_mode = -1;
static int hf_ieee80211_csa_new_channel_number = -1;
static int hf_ieee80211_csa_channel_switch_count = -1;

static int hf_ieee80211_tag_measure_request_token = -1;
static int hf_ieee80211_tag_measure_request_mode = -1;
static int hf_ieee80211_tag_measure_request_mode_parallel = -1;
static int hf_ieee80211_tag_measure_request_mode_enable = -1;
static int hf_ieee80211_tag_measure_request_mode_request = -1;
static int hf_ieee80211_tag_measure_request_mode_report = -1;
static int hf_ieee80211_tag_measure_request_mode_duration_mandatory = -1;
static int hf_ieee80211_tag_measure_request_mode_reserved = -1;
static int hf_ieee80211_tag_measure_request_type = -1;

static int hf_ieee80211_tag_measure_request_channel_number = -1;
static int hf_ieee80211_tag_measure_request_start_time = -1;
static int hf_ieee80211_tag_measure_request_duration = -1;

static int hf_ieee80211_tag_measure_request_operating_class = -1;
static int hf_ieee80211_tag_measure_request_randomization_interval = -1;

static int hf_ieee80211_tag_measure_report_measurement_token = -1;
static int hf_ieee80211_tag_measure_report_mode = -1;
static int hf_ieee80211_tag_measure_report_mode_late = -1;
static int hf_ieee80211_tag_measure_report_mode_incapable = -1;
static int hf_ieee80211_tag_measure_report_mode_refused = -1;
static int hf_ieee80211_tag_measure_report_mode_reserved = -1;
static int hf_ieee80211_tag_measure_report_type = -1;
static int hf_ieee80211_tag_measure_report_channel_number = -1;
static int hf_ieee80211_tag_measure_report_start_time = -1;
static int hf_ieee80211_tag_measure_report_duration = -1;

static int hf_ieee80211_tag_measure_basic_map_field = -1;
static int hf_ieee80211_tag_measure_map_field_bss = -1;
static int hf_ieee80211_tag_measure_map_field_ofdm = -1;
static int hf_ieee80211_tag_measure_map_field_unident_signal = -1;
static int hf_ieee80211_tag_measure_map_field_radar = -1;
static int hf_ieee80211_tag_measure_map_field_unmeasured = -1;
static int hf_ieee80211_tag_measure_map_field_reserved = -1;

static int hf_ieee80211_tag_measure_cca_busy_fraction = -1;

static int hf_ieee80211_tag_measure_rpi_histogram_report = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_0 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_1 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_2 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_3 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_4 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_5 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_6 = -1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_7 = -1;

static int hf_ieee80211_tag_measure_report_operating_class = -1;
static int hf_ieee80211_tag_measure_report_channel_load = -1;
static int hf_ieee80211_tag_measure_report_frame_info = -1;
static int hf_ieee80211_tag_measure_report_frame_info_phy_type = -1;
static int hf_ieee80211_tag_measure_report_frame_info_frame_type = -1;
static int hf_ieee80211_tag_measure_report_rcpi = -1;
static int hf_ieee80211_tag_measure_report_rsni = -1;
static int hf_ieee80211_tag_measure_report_bssid = -1;
static int hf_ieee80211_tag_measure_report_ant_id = -1;
static int hf_ieee80211_tag_measure_report_anpi = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_0 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_1 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_2 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_3 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_4 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_5 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_6 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_7 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_8 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_9 = -1;
static int hf_ieee80211_tag_measure_report_ipi_density_10 = -1;
static int hf_ieee80211_tag_measure_report_parent_tsf = -1;

static int hf_ieee80211_tag_measure_report_subelement_length = -1;
static int hf_ieee80211_tag_measure_report_beacon_sub_id = -1;

static int hf_ieee80211_tag_measure_report_unknown = -1;

static int hf_ieee80211_tag_quiet_count = -1;
static int hf_ieee80211_tag_quiet_period = -1;
static int hf_ieee80211_tag_quiet_duration = -1;
static int hf_ieee80211_tag_quiet_offset = -1;

static int hf_ieee80211_tag_dfs_owner = -1;
static int hf_ieee80211_tag_dfs_recovery_interval = -1;
static int hf_ieee80211_tag_dfs_channel_map = -1;
static int hf_ieee80211_tag_dfs_channel_number = -1;
static int hf_ieee80211_tag_dfs_map = -1;

static int hf_ieee80211_tag_erp_info = -1;
static int hf_ieee80211_tag_erp_info_erp_present = -1;
static int hf_ieee80211_tag_erp_info_use_protection = -1;
static int hf_ieee80211_tag_erp_info_barker_preamble_mode = -1;
static int hf_ieee80211_tag_erp_info_reserved = -1;

static int hf_ieee80211_tag_extended_capabilities = -1;
static int hf_ieee80211_tag_extended_capabilities_b0 = -1;
static int hf_ieee80211_tag_extended_capabilities_b1 = -1;
static int hf_ieee80211_tag_extended_capabilities_b2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b3 = -1;
static int hf_ieee80211_tag_extended_capabilities_b4 = -1;
static int hf_ieee80211_tag_extended_capabilities_b5 = -1;
static int hf_ieee80211_tag_extended_capabilities_b6 = -1;
static int hf_ieee80211_tag_extended_capabilities_b7 = -1;
static int hf_ieee80211_tag_extended_capabilities_b8 = -1;
static int hf_ieee80211_tag_extended_capabilities_b9 = -1;
static int hf_ieee80211_tag_extended_capabilities_b10 = -1;
static int hf_ieee80211_tag_extended_capabilities_b11 = -1;
static int hf_ieee80211_tag_extended_capabilities_b12 = -1;
static int hf_ieee80211_tag_extended_capabilities_b13 = -1;
static int hf_ieee80211_tag_extended_capabilities_b14 = -1;
static int hf_ieee80211_tag_extended_capabilities_b15 = -1;
static int hf_ieee80211_tag_extended_capabilities_b16 = -1;
static int hf_ieee80211_tag_extended_capabilities_b17 = -1;
static int hf_ieee80211_tag_extended_capabilities_b18 = -1;
static int hf_ieee80211_tag_extended_capabilities_b19 = -1;
static int hf_ieee80211_tag_extended_capabilities_b20 = -1;
static int hf_ieee80211_tag_extended_capabilities_b21 = -1;
static int hf_ieee80211_tag_extended_capabilities_b22 = -1;
static int hf_ieee80211_tag_extended_capabilities_b23 = -1;
static int hf_ieee80211_tag_extended_capabilities_b24 = -1;
static int hf_ieee80211_tag_extended_capabilities_b25 = -1;
static int hf_ieee80211_tag_extended_capabilities_b26 = -1;
static int hf_ieee80211_tag_extended_capabilities_b27 = -1;
static int hf_ieee80211_tag_extended_capabilities_b28 = -1;
static int hf_ieee80211_tag_extended_capabilities_b29 = -1;
static int hf_ieee80211_tag_extended_capabilities_b30 = -1;
static int hf_ieee80211_tag_extended_capabilities_b31 = -1;
static int hf_ieee80211_tag_extended_capabilities_b32 = -1;
static int hf_ieee80211_tag_extended_capabilities_b33 = -1;
static int hf_ieee80211_tag_extended_capabilities_b34 = -1;
static int hf_ieee80211_tag_extended_capabilities_b35 = -1;
static int hf_ieee80211_tag_extended_capabilities_b36 = -1;
static int hf_ieee80211_tag_extended_capabilities_b37 = -1;
static int hf_ieee80211_tag_extended_capabilities_b38 = -1;
static int hf_ieee80211_tag_extended_capabilities_b39 = -1;
static int hf_ieee80211_tag_extended_capabilities_b40 = -1;
static int hf_ieee80211_tag_extended_capabilities_serv_int_granularity = -1;
static int hf_ieee80211_tag_extended_capabilities_b44 = -1;
static int hf_ieee80211_tag_extended_capabilities_b45 = -1;
static int hf_ieee80211_tag_extended_capabilities_b46 = -1;
static int hf_ieee80211_tag_extended_capabilities_b47 = -1;
static int hf_ieee80211_tag_extended_capabilities_b48 = -1;
static int hf_ieee80211_tag_extended_capabilities_b49 = -1;
static int hf_ieee80211_tag_extended_capabilities_b50 = -1;
static int hf_ieee80211_tag_extended_capabilities_b51 = -1;
static int hf_ieee80211_tag_extended_capabilities_b52 = -1;
static int hf_ieee80211_tag_extended_capabilities_b53 = -1;
static int hf_ieee80211_tag_extended_capabilities_b54 = -1;
static int hf_ieee80211_tag_extended_capabilities_b55 = -1;
static int hf_ieee80211_tag_extended_capabilities_b56 = -1;
static int hf_ieee80211_tag_extended_capabilities_b57 = -1;
static int hf_ieee80211_tag_extended_capabilities_b58 = -1;
static int hf_ieee80211_tag_extended_capabilities_b59 = -1;
static int hf_ieee80211_tag_extended_capabilities_b60 = -1;
static int hf_ieee80211_tag_extended_capabilities_b61 = -1;
static int hf_ieee80211_tag_extended_capabilities_b62 = -1;
static int hf_ieee80211_tag_extended_capabilities_b63 = -1;
/* Used for the two-byte ext-cap field when present */
static int hf_ieee80211_tag_extended_capabilities_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b56_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b57_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b58_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b59_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b60_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b61_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b62_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_max_num_msdus = -1;
static int hf_ieee80211_tag_extended_capabilities_b65_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b66_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b67_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b68_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b69_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b70_2 = -1;
static int hf_ieee80211_tag_extended_capabilities_b71_2 = -1;

static int hf_ieee80211_tag_extended_capabilities_b72 = -1;
static int hf_ieee80211_tag_extended_capabilities_b73 = -1;
static int hf_ieee80211_tag_extended_capabilities_b74 = -1;
static int hf_ieee80211_tag_extended_capabilities_b75 = -1;
static int hf_ieee80211_tag_extended_capabilities_b76 = -1;
static int hf_ieee80211_tag_extended_capabilities_b77 = -1;
static int hf_ieee80211_tag_extended_capabilities_b78 = -1;
static int hf_ieee80211_tag_extended_capabilities_b79 = -1;

static int hf_ieee80211_tag_cisco_ccx1_unknown = -1;
static int hf_ieee80211_tag_cisco_ccx1_name = -1;
static int hf_ieee80211_tag_cisco_ccx1_clients = -1;
static int hf_ieee80211_tag_cisco_ccx1_unknown2 = -1;

static int hf_ieee80211_vht_cap = -1;
static int hf_ieee80211_vht_max_mpdu_length = -1;
static int hf_ieee80211_vht_supported_chan_width_set = -1;
static int hf_ieee80211_vht_rx_ldpc = -1;
static int hf_ieee80211_vht_short_gi_for_80 = -1;
static int hf_ieee80211_vht_short_gi_for_160 = -1;
static int hf_ieee80211_vht_tx_stbc = -1;
static int hf_ieee80211_vht_rx_stbc = -1;
static int hf_ieee80211_vht_su_beamformer_cap = -1;
static int hf_ieee80211_vht_su_beamformee_cap = -1;
static int hf_ieee80211_vht_beamformer_antennas = -1;
static int hf_ieee80211_vht_sounding_dimensions = -1;
static int hf_ieee80211_vht_mu_beamformer_cap = -1;
static int hf_ieee80211_vht_mu_beamformee_cap = -1;
static int hf_ieee80211_vht_txop_ps = -1;
static int hf_ieee80211_vht_var_htc_field = -1;
static int hf_ieee80211_vht_max_ampdu = -1;
static int hf_ieee80211_vht_link_adaptation_cap = -1;
static int hf_ieee80211_vht_rx_pattern = -1;
static int hf_ieee80211_vht_tx_pattern = -1;
static int hf_ieee80211_vht_ext_nss_bw_support = -1;

static int hf_ieee80211_vht_mcsset = -1;

static int hf_ieee80211_vht_mcsset_rx_mcs_map = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_1_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_2_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_3_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_4_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_5_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_6_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_7_ss = -1;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_8_ss = -1;

static int hf_ieee80211_vht_mcsset_max_nsts_total = -1;
static int hf_ieee80211_vht_mcsset_rx_highest_long_gi = -1;
static int hf_ieee80211_vht_mcsset_extended_nss_bw_capable = -1;
static int hf_ieee80211_vht_mcsset_reserved = -1;

static int hf_ieee80211_vht_mcsset_tx_mcs_map = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_1_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_2_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_3_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_4_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_5_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_6_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_7_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_8_ss = -1;

static int hf_ieee80211_vht_op = -1;
static int hf_ieee80211_vht_op_channel_width = -1;
static int hf_ieee80211_vht_op_channel_center0 = -1;
static int hf_ieee80211_vht_op_channel_center1 = -1;

static int hf_ieee80211_vht_op_basic_mcs_map = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_1_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_2_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_3_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_4_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_5_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_6_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_7_ss = -1;
static int hf_ieee80211_vht_op_max_basic_mcs_for_8_ss = -1;
static int hf_ieee80211_vht_mcsset_tx_highest_long_gi = -1;

static int hf_ieee80211_vht_tpe_pwr_info = -1;
static int hf_ieee80211_vht_tpe_pwr_info_count = -1;
static int hf_ieee80211_vht_tpe_pwr_info_unit = -1;
static int hf_ieee80211_vht_tpe_pwr_info_reserved = -1;
static int hf_ieee80211_vht_tpe_pwr_constr_20 = -1;
static int hf_ieee80211_vht_tpe_pwr_constr_40 = -1;
static int hf_ieee80211_vht_tpe_pwr_constr_80 = -1;
static int hf_ieee80211_vht_tpe_pwr_constr_160 = -1;

static int hf_ieee80211_beamform_feedback_seg_retrans_bitmap = -1;

static int hf_ieee80211_vht_ndp_annc_token = -1;
static int hf_ieee80211_vht_ndp_annc_token_number = -1;
static int hf_ieee80211_vht_ndp_annc_he_subfield = -1;
static int hf_ieee80211_vht_ndp_annc_token_reserved = -1;
static int hf_ieee80211_vht_ndp_annc_sta_info_aid12 = -1;
static int hf_ieee80211_vht_ndp_annc_sta_info_feedback_type = -1;
static int hf_ieee80211_vht_ndp_annc_sta_info_nc_index = -1;
static int hf_ieee80211_vht_ndp_annc_sta_info_reserved = -1;


static int hf_ieee80211_ff_vht_action = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_nc_index = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_nr_index = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_channel_width = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_grouping = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_codebook_info = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_feedback_type = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_remaining_feedback_seg = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_first_feedback_seg = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_reserved = -1;
static int hf_ieee80211_ff_vht_mimo_cntrl_sounding_dialog_token_number = -1;

static const int *hf_ieee80211_ff_vht_mimo_cntrl_fields[] = {
  &hf_ieee80211_ff_vht_mimo_cntrl_nc_index,
  &hf_ieee80211_ff_vht_mimo_cntrl_nr_index,
  &hf_ieee80211_ff_vht_mimo_cntrl_channel_width,
  &hf_ieee80211_ff_vht_mimo_cntrl_grouping,
  &hf_ieee80211_ff_vht_mimo_cntrl_codebook_info,
  &hf_ieee80211_ff_vht_mimo_cntrl_feedback_type,
  &hf_ieee80211_ff_vht_mimo_cntrl_remaining_feedback_seg,
  &hf_ieee80211_ff_vht_mimo_cntrl_first_feedback_seg,
  &hf_ieee80211_ff_vht_mimo_cntrl_reserved,
  &hf_ieee80211_ff_vht_mimo_cntrl_sounding_dialog_token_number,
  NULL,
};

static int hf_ieee80211_vht_compressed_beamforming_report = -1;
static int hf_ieee80211_vht_compressed_beamforming_report_snr = -1;
static int hf_ieee80211_vht_compressed_beamforming_feedback_matrix = -1;
static int hf_ieee80211_vht_group_id_management = -1;
static int hf_ieee80211_vht_membership_status_array = -1;
static int hf_ieee80211_vht_user_position_array = -1;
static int hf_ieee80211_vht_operation_mode_notification = -1;
static int hf_ieee80211_vht_membership_status_field = -1;
static int hf_ieee80211_vht_user_position_field = -1;
static int hf_ieee80211_vht_mu_exclusive_beamforming_report = -1;
static int hf_ieee80211_vht_mu_exclusive_beamforming_delta_snr = -1;
static int hf_ieee80211_vht_compressed_beamforming_phi_angle = -1;
static int hf_ieee80211_vht_compressed_beamforming_psi_angle = -1;

static int hf_ieee80211_ff_he_action = -1;
static int hf_ieee80211_ff_protected_he_action = -1;
static int hf_he_mimo_control_nc_index = -1;
static int hf_he_mimo_control_nr_index = -1;
static int hf_he_mimo_control_bw = -1;
static int hf_he_mimo_control_grouping = -1;
static int hf_he_mimo_control_codebook_info = -1;
static int hf_he_mimo_control_feedback_type = -1;
static int hf_he_mimo_control_remaining_feedback_segs = -1;
static int hf_he_mimo_control_first_feedback_seg = -1;
static int hf_he_mimo_control_ru_start_index = -1;
static int hf_he_mimo_control_ru_end_index = -1;
static int hf_he_mimo_control_sounding_dialog_token_num = -1;
static int hf_he_mimo_control_reserved = -1;
static int hf_ieee80211_he_mimo_control_field = -1;
static int hf_ieee80211_he_compressed_beamforming_report_snr = -1;
static int hf_ieee80211_he_compressed_beamform_scidx = -1;
static int hf_ieee80211_beamforming_report_len = -1;

static int hf_ieee80211_tag_neighbor_report_bssid = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_reachability = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_security = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_key_scope = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_spec_mng = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_qos = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_apsd = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_radio_msnt = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_dback = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_iback = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_mobility_domain = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_high_throughput = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_very_high_throughput = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_ftm = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_high_efficiency = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_er_bss = -1;
static int hf_ieee80211_tag_neighbor_report_bssid_info_reserved = -1;
static int hf_ieee80211_tag_neighbor_report_ope_class = -1;
static int hf_ieee80211_tag_neighbor_report_channel_number = -1;
static int hf_ieee80211_tag_neighbor_report_phy_type = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_id = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_length = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_data = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_trn_can_pref = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_ter_tsf = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_dur = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_tsf_offset = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_beacon_interval = -1;
static int hf_ieee80211_tag_neighbor_report_subelement_country_code = -1;
static int hf_ieee80211_tag_supported_ope_classes_current = -1;
static int hf_ieee80211_tag_supported_ope_classes_alternate = -1;

/* IEEE Std 802.11r-2008 7.3.2.47 */
static int hf_ieee80211_tag_mobility_domain_mdid = -1;
static int hf_ieee80211_tag_mobility_domain_ft_capab = -1;
static int hf_ieee80211_tag_mobility_domain_ft_capab_ft_over_ds = -1;
static int hf_ieee80211_tag_mobility_domain_ft_capab_resource_req = -1;

/* IEEE Std 802.11r-2008 7.3.2.48 */
static int hf_ieee80211_tag_ft_mic_control = -1;
static int hf_ieee80211_tag_ft_element_count = -1;
static int hf_ieee80211_tag_ft_mic = -1;
static int hf_ieee80211_tag_ft_anonce = -1;
static int hf_ieee80211_tag_ft_snonce = -1;
static int hf_ieee80211_tag_ft_subelem_id = -1;
static int hf_ieee80211_tag_ft_subelem_len = -1;
static int hf_ieee80211_tag_ft_subelem_data = -1;
static int hf_ieee80211_tag_ft_subelem_r1kh_id = -1;
static int hf_ieee80211_tag_ft_subelem_gtk_key_info = -1;
static int hf_ieee80211_tag_ft_subelem_gtk_key_id = -1;
static int hf_ieee80211_tag_ft_subelem_gtk_key_length = -1;
static int hf_ieee80211_tag_ft_subelem_gtk_rsc = -1;
static int hf_ieee80211_tag_ft_subelem_gtk_key = -1;
static int hf_ieee80211_tag_ft_subelem_r0kh_id = -1;
static int hf_ieee80211_tag_ft_subelem_igtk_key_id = -1;
static int hf_ieee80211_tag_ft_subelem_igtk_ipn = -1;
static int hf_ieee80211_tag_ft_subelem_igtk_key_length = -1;
static int hf_ieee80211_tag_ft_subelem_igtk_key = -1;

/* IEEE Std 802.11-2012: 11r 8.4.2.52 */
static int hf_ieee80211_tag_ric_data_id = -1;
static int hf_ieee80211_tag_ric_data_desc_cnt = -1;
static int hf_ieee80211_tag_ric_data_status_code = -1;

/* IEEE Std 802.11-2012: 11r 8.4.2.53 */
static int hf_ieee80211_tag_ric_desc_rsrc_type = -1;
static int hf_ieee80211_tag_ric_desc_var_params = -1;

/* IEEE Std 802.11w-2009 7.3.2.55 */
static int hf_ieee80211_tag_mmie_keyid = -1;
static int hf_ieee80211_tag_mmie_ipn = -1;
static int hf_ieee80211_tag_mmie_mic = -1;

/* IEEE Std 802.11-2012: 8.4.2.61 */
static int hf_ieee80211_tag_obss_spd = -1;
static int hf_ieee80211_tag_obss_sad = -1;
static int hf_ieee80211_tag_obss_cwtsi = -1;
static int hf_ieee80211_tag_obss_sptpc = -1;
static int hf_ieee80211_tag_obss_satpc = -1;
static int hf_ieee80211_tag_obss_wctdf = -1;
static int hf_ieee80211_tag_obss_sat = -1;

/* IEEE Std 802.11-2012: 8.4.2.25.1 */
static int hf_group_data_cipher_suite_oui = -1;
static int hf_group_data_cipher_suite_type = -1;
static int hf_osen_pairwise_cipher_suite_oui = -1;
static int hf_osen_pairwise_cipher_suite_type = -1;
static int hf_osen_pcs_count = -1;
static int hf_osen_akm_count = -1;
static int hf_osen_akm_cipher_suite_oui = -1;
static int hf_osen_akm_cipher_suite_type = -1;
static int hf_osen_rsn_cap_preauth = -1;
static int hf_osen_rsn_cap_no_pairwise = -1;
static int hf_osen_rsn_cap_ptksa_replay_counter = -1;
static int hf_osen_rsn_cap_gtksa_replay_counter = -1;
static int hf_osen_rsn_cap_mfpr = -1;
static int hf_osen_rsn_cap_mfpc = -1;
static int hf_osen_rsn_cap_jmr = -1;
static int hf_osen_rsn_cap_peerkey = -1;
static int hf_osen_rsn_spp_a_msdu_capable = -1;
static int hf_osen_rsn_spp_a_msdu_required = -1;
static int hf_osen_rsn_pbac = -1;
static int hf_osen_extended_key_id_iaf = -1;
static int hf_osen_reserved = -1;
static int hf_osen_rsn_cap_flags = -1;
static int hf_osen_pmkid_count = -1;
static int hf_osen_pmkid = -1;
static int hf_osen_group_management_cipher_suite_oui = -1;
static int hf_osen_group_management_cipher_suite_type = -1;

/*WAPI-Specification 7.3.2.25 : WAPI Parameter Set*/
static int hf_ieee80211_tag_wapi_param_set_version = -1;

static int hf_ieee80211_tag_wapi_param_set_akm_suite_count = -1;
static int hf_ieee80211_tag_wapi_param_set_akm_suite_oui = -1;
static int hf_ieee80211_tag_wapi_param_set_akm_suite_type = -1;

static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_count = -1;
static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_oui = -1;
static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_type = -1;

static int hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_oui = -1;
static int hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_type = -1;

static int hf_ieee80211_tag_wapi_param_set_capab = -1;
static int hf_ieee80211_tag_wapi_param_set_capab_preauth = -1;
static int hf_ieee80211_tag_wapi_param_set_capab_rsvd = -1;
static int hf_ieee80211_tag_wapi_param_set_bkid_count = -1;
static int hf_ieee80211_tag_wapi_param_set_bkid_list = -1;

/* IEEE Std 802.11v-2011 7.3.2.61 */
static int hf_ieee80211_tag_time_adv_timing_capab = -1;
static int hf_ieee80211_tag_time_adv_time_value = -1;
static int hf_ieee80211_tag_time_adv_time_value_year = -1;
static int hf_ieee80211_tag_time_adv_time_value_month = -1;
static int hf_ieee80211_tag_time_adv_time_value_day = -1;
static int hf_ieee80211_tag_time_adv_time_value_hours = -1;
static int hf_ieee80211_tag_time_adv_time_value_minutes = -1;
static int hf_ieee80211_tag_time_adv_time_value_seconds = -1;
static int hf_ieee80211_tag_time_adv_time_value_milliseconds = -1;
static int hf_ieee80211_tag_time_adv_time_value_reserved = -1;
static int hf_ieee80211_tag_time_adv_time_error = -1;
static int hf_ieee80211_tag_time_adv_time_update_counter = -1;

/* IEEE Std 802.11-2012 8.4.2.81 */
static int hf_ieee80211_tag_bss_max_idle_period = -1;
static int hf_ieee80211_tag_bss_max_idle_options_protected = -1;

/* IEEE Std 802.11-2012 8.4.2.82 */
static int hf_ieee80211_tag_tfs_request_id = -1;
static int hf_ieee80211_tag_tfs_request_ac_delete_after_match = -1;
static int hf_ieee80211_tag_tfs_request_ac_notify = -1;
static int hf_ieee80211_tag_tfs_request_subelem_id = -1;
static int hf_ieee80211_tag_tfs_request_subelem_len = -1;
static int hf_ieee80211_tag_tfs_request_subelem = -1;

/* IEEE Std 802.11-2012 8.4.2.83 */
static int hf_ieee80211_tag_tfs_response_subelem_id = -1;
static int hf_ieee80211_tag_tfs_response_subelem_len = -1;
static int hf_ieee80211_tag_tfs_response_subelem = -1;
static int hf_ieee80211_tag_tfs_response_status = -1;
static int hf_ieee80211_tag_tfs_response_id = -1;

/* IEEE Std 802.11-2012 8.4.2.84 */
static int hf_ieee80211_tag_wnm_sleep_mode_action_type = -1;
static int hf_ieee80211_tag_wnm_sleep_mode_response_status = -1;
static int hf_ieee80211_tag_wnm_sleep_mode_interval = -1;

static int hf_ieee80211_wnm_sub_elt_id = -1;
static int hf_ieee80211_wnm_sub_elt_len = -1;

/* IEEE Std 802.11v-2011 7.3.2.87 */
static int hf_ieee80211_tag_time_zone = -1;

/* IEEE Std 802.11u-2011 7.3.2.92 */
static int hf_ieee80211_tag_interworking_access_network_type = -1;
static int hf_ieee80211_tag_interworking_internet = -1;
static int hf_ieee80211_tag_interworking_asra = -1;
static int hf_ieee80211_tag_interworking_esr = -1;
static int hf_ieee80211_tag_interworking_uesa = -1;
static int hf_ieee80211_tag_interworking_hessid = -1;

/* IEEE Std 802.11-2012, 8.4.2.97 */
static int hf_ieee80211_tag_qos_map_set_dscp_exc = -1;
static int hf_ieee80211_tag_qos_map_set_dscp_exc_val = -1;
static int hf_ieee80211_tag_qos_map_set_dscp_exc_up = -1;
static int hf_ieee80211_tag_qos_map_set_range = -1;
static int hf_ieee80211_tag_qos_map_set_low = -1;
static int hf_ieee80211_tag_qos_map_set_high = -1;

/* IEEE Std 802.11u-2011 7.3.2.93 */
static int hf_ieee80211_tag_adv_proto_resp_len_limit = -1;
static int hf_ieee80211_tag_adv_proto_pame_bi = -1;
static int hf_ieee80211_tag_adv_proto_id = -1;
static int hf_ieee80211_tag_adv_vs_len = -1;
/* static int hf_ieee80211_tag_adv_proto_vs_info = -1; */

/* IEEE Std 802.11u-2011 7.3.2.96 */
static int hf_ieee80211_tag_roaming_consortium_num_anqp_oi = -1;
static int hf_ieee80211_tag_roaming_consortium_oi1_len = -1;
static int hf_ieee80211_tag_roaming_consortium_oi2_len = -1;
static int hf_ieee80211_tag_roaming_consortium_oi1 = -1;
static int hf_ieee80211_tag_roaming_consortium_oi2 = -1;
static int hf_ieee80211_tag_roaming_consortium_oi3 = -1;

/* 802.11n 7.3.2.48 */
static int hf_ieee80211_hta_cc = -1;
static int hf_ieee80211_hta_cap1 = -1;
static int hf_ieee80211_hta_cap2 = -1;
static int hf_ieee80211_hta_ext_chan_offset = -1;
static int hf_ieee80211_hta_rec_tx_width = -1;
static int hf_ieee80211_hta_rifs_mode = -1;
static int hf_ieee80211_hta_controlled_access = -1;
static int hf_ieee80211_hta_service_interval = -1;
static int hf_ieee80211_hta_operating_mode = -1;
static int hf_ieee80211_hta_non_gf_devices = -1;
static int hf_ieee80211_hta_basic_stbc_mcs = -1;
static int hf_ieee80211_hta_dual_stbc_protection = -1;
static int hf_ieee80211_hta_secondary_beacon = -1;
static int hf_ieee80211_hta_lsig_txop_protection = -1;
static int hf_ieee80211_hta_pco_active = -1;
static int hf_ieee80211_hta_pco_phase = -1;

static int hf_ieee80211_antsel = -1;
static int hf_ieee80211_antsel_vs = -1;
static int hf_ieee80211_antsel_b0 = -1;
static int hf_ieee80211_antsel_b1 = -1;
static int hf_ieee80211_antsel_b2 = -1;
static int hf_ieee80211_antsel_b3 = -1;
static int hf_ieee80211_antsel_b4 = -1;
static int hf_ieee80211_antsel_b5 = -1;
static int hf_ieee80211_antsel_b6 = -1;
static int hf_ieee80211_antsel_b7 = -1;

static int hf_ieee80211_rsn_version = -1;
static int hf_ieee80211_rsn_gcs = -1;
static int hf_ieee80211_rsn_gcs_oui = -1;
static int hf_ieee80211_rsn_gcs_type = -1;
static int hf_ieee80211_rsn_gcs_80211_type = -1;
static int hf_ieee80211_rsn_pcs_count = -1;
static int hf_ieee80211_rsn_pcs_list = -1;
static int hf_ieee80211_rsn_pcs = -1;
static int hf_ieee80211_rsn_pcs_oui = -1;
static int hf_ieee80211_rsn_pcs_80211_type = -1;
static int hf_ieee80211_rsn_pcs_type = -1;
static int hf_ieee80211_rsn_akms_count = -1;
static int hf_ieee80211_rsn_akms_list = -1;
static int hf_ieee80211_rsn_akms = -1;
static int hf_ieee80211_rsn_akms_oui = -1;
static int hf_ieee80211_rsn_akms_80211_type = -1;
static int hf_ieee80211_rsn_akms_type = -1;
static int hf_ieee80211_rsn_cap = -1;
static int hf_ieee80211_rsn_cap_preauth = -1;
static int hf_ieee80211_rsn_cap_no_pairwise = -1;
static int hf_ieee80211_rsn_cap_ptksa_replay_counter = -1;
static int hf_ieee80211_rsn_cap_gtksa_replay_counter = -1;
static int hf_ieee80211_rsn_cap_mfpr = -1;
static int hf_ieee80211_rsn_cap_mfpc = -1;
static int hf_ieee80211_rsn_cap_jmr = -1;
static int hf_ieee80211_rsn_cap_peerkey = -1;
static int hf_ieee80211_rsn_pmkid_count = -1;
static int hf_ieee80211_rsn_pmkid_list = -1;
static int hf_ieee80211_rsn_pmkid = -1;
static int hf_ieee80211_rsn_gmcs = -1;
static int hf_ieee80211_rsn_gmcs_oui = -1;
static int hf_ieee80211_rsn_gmcs_type = -1;
static int hf_ieee80211_rsn_gmcs_80211_type = -1;

static int hf_ieee80211_wfa_ie_type = -1;
static int hf_ieee80211_wfa_ie_wpa_version = -1;
static int hf_ieee80211_wfa_ie_wpa_mcs = -1;
static int hf_ieee80211_wfa_ie_wpa_mcs_oui = -1;
static int hf_ieee80211_wfa_ie_wpa_mcs_type = -1;
static int hf_ieee80211_wfa_ie_wpa_mcs_wfa_type = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs_count = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs_list = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs_oui = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs_wfa_type = -1;
static int hf_ieee80211_wfa_ie_wpa_ucs_type = -1;
static int hf_ieee80211_wfa_ie_wpa_akms_count = -1;
static int hf_ieee80211_wfa_ie_wpa_akms_list = -1;
static int hf_ieee80211_wfa_ie_wpa_akms = -1;
static int hf_ieee80211_wfa_ie_wpa_akms_oui = -1;
static int hf_ieee80211_wfa_ie_wpa_akms_wfa_type = -1;
static int hf_ieee80211_wfa_ie_wpa_akms_type = -1;
static int hf_ieee80211_wfa_ie_wme_subtype = -1;
static int hf_ieee80211_wfa_ie_wme_version = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_max_sp_length = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_be = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_bk = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vi = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vo = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_reserved = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_u_apsd = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_parameter_set_count = -1;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_reserved = -1;
static int hf_ieee80211_wfa_ie_wme_reserved = -1;
static int hf_ieee80211_wfa_ie_wme_ac_parameters = -1;
static int hf_ieee80211_wfa_ie_wme_acp_aci_aifsn = -1;
static int hf_ieee80211_wfa_ie_wme_acp_aci = -1;
static int hf_ieee80211_wfa_ie_wme_acp_acm = -1;
static int hf_ieee80211_wfa_ie_wme_acp_aifsn = -1;
static int hf_ieee80211_wfa_ie_wme_acp_reserved = -1;
static int hf_ieee80211_wfa_ie_wme_acp_ecw = -1;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_max = -1;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_min = -1;
static int hf_ieee80211_wfa_ie_wme_acp_cw_max = -1;
static int hf_ieee80211_wfa_ie_wme_acp_cw_min = -1;
static int hf_ieee80211_wfa_ie_wme_acp_txop_limit = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_tid = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_direction = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_psb = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_up = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_reserved = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_nor_msdu = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_max_msdu = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_min_srv = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_max_srv = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_inact_int = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_susp_int = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_srv_start = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_min_data = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_mean_data = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_peak_data = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_burst_size = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_delay_bound = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_min_phy = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_surplus = -1;
static int hf_ieee80211_wfa_ie_wme_tspec_medium = -1;
static int hf_ieee80211_wfa_ie_owe_bssid = -1;
static int hf_ieee80211_wfa_ie_owe_ssid_length = -1;
static int hf_ieee80211_wfa_ie_owe_ssid = -1;
static int hf_ieee80211_wfa_ie_owe_band_info = -1;
static int hf_ieee80211_wfa_ie_owe_channel_info = -1;

static int hf_ieee80211_aironet_ie_type = -1;
static int hf_ieee80211_aironet_ie_dtpc = -1;
static int hf_ieee80211_aironet_ie_dtpc_unknown = -1;
static int hf_ieee80211_aironet_ie_version = -1;
static int hf_ieee80211_aironet_ie_data = -1;
static int hf_ieee80211_aironet_ie_qos_reserved = -1;
static int hf_ieee80211_aironet_ie_qos_paramset = -1;
static int hf_ieee80211_aironet_ie_qos_val = -1;
static int hf_ieee80211_aironet_ie_clientmfp = -1;

static int hf_ieee80211_vs_nintendo_type = -1;
static int hf_ieee80211_vs_nintendo_length = -1;
static int hf_ieee80211_vs_nintendo_servicelist = -1;
static int hf_ieee80211_vs_nintendo_service = -1;
static int hf_ieee80211_vs_nintendo_consoleid = -1;
static int hf_ieee80211_vs_nintendo_unknown = -1;

static int hf_ieee80211_vs_aruba_subtype = -1;
static int hf_ieee80211_vs_aruba_apname = -1;
static int hf_ieee80211_vs_aruba_data = -1;

static int hf_ieee80211_vs_routerboard_unknown = -1;
static int hf_ieee80211_vs_routerboard_subitem = -1;
static int hf_ieee80211_vs_routerboard_subtype = -1;
static int hf_ieee80211_vs_routerboard_sublength = -1;
static int hf_ieee80211_vs_routerboard_subdata = -1;
static int hf_ieee80211_vs_routerboard_subtype1_prefix = -1;
static int hf_ieee80211_vs_routerboard_subtype1_data = -1;

static int hf_ieee80211_vs_meru_subitem = -1;
static int hf_ieee80211_vs_meru_subtype = -1;
static int hf_ieee80211_vs_meru_sublength = -1;
static int hf_ieee80211_vs_meru_subdata = -1;

static int hf_ieee80211_vs_extreme_subtype = -1;
static int hf_ieee80211_vs_extreme_subdata = -1;
static int hf_ieee80211_vs_extreme_unknown = -1;
static int hf_ieee80211_vs_extreme_ap_length = -1;
static int hf_ieee80211_vs_extreme_ap_name = -1;

static int hf_ieee80211_vs_aerohive_unknown = -1;
static int hf_ieee80211_vs_aerohive_hostname_length = -1;
static int hf_ieee80211_vs_aerohive_hostname = -1;
static int hf_ieee80211_vs_aerohive_data = -1;

static int hf_ieee80211_rsn_ie_gtk_keyid = -1;
static int hf_ieee80211_rsn_ie_gtk_tx = -1;
static int hf_ieee80211_rsn_ie_gtk_reserved1 = -1;
static int hf_ieee80211_rsn_ie_gtk_reserved2 = -1;
static int hf_ieee80211_rsn_ie_gtk_key = -1;

static int hf_ieee80211_rsn_ie_pmkid = -1;

static int hf_ieee80211_rsn_ie_igtk_keyid = -1;
static int hf_ieee80211_rsn_ie_igtk_ipn = -1;
static int hf_ieee80211_rsn_ie_igtk_key = -1;

static int hf_ieee80211_rsn_ie_unknown = -1;

static int hf_ieee80211_marvell_ie_type = -1;
static int hf_ieee80211_marvell_ie_mesh_subtype = -1;
static int hf_ieee80211_marvell_ie_mesh_version = -1;
static int hf_ieee80211_marvell_ie_mesh_active_proto_id = -1;
static int hf_ieee80211_marvell_ie_mesh_active_metric_id = -1;
static int hf_ieee80211_marvell_ie_mesh_cap = -1;
static int hf_ieee80211_marvell_ie_data = -1;

static int hf_ieee80211_atheros_ie_type = -1;
static int hf_ieee80211_atheros_ie_subtype = -1;
static int hf_ieee80211_atheros_ie_version = -1;
static int hf_ieee80211_atheros_ie_cap_f_turbop = -1;
static int hf_ieee80211_atheros_ie_cap_f_comp = -1;
static int hf_ieee80211_atheros_ie_cap_f_ff = -1;
static int hf_ieee80211_atheros_ie_cap_f_xr = -1;
static int hf_ieee80211_atheros_ie_cap_f_ar = -1;
static int hf_ieee80211_atheros_ie_cap_f_burst = -1;
static int hf_ieee80211_atheros_ie_cap_f_wme = -1;
static int hf_ieee80211_atheros_ie_cap_f_boost = -1;
static int hf_ieee80211_atheros_ie_advcap_cap = -1;
static int hf_ieee80211_atheros_ie_advcap_defkey = -1;
static int hf_ieee80211_atheros_ie_xr_info = -1;
static int hf_ieee80211_atheros_ie_xr_base_bssid = -1;
static int hf_ieee80211_atheros_ie_xr_xr_bssid = -1;
static int hf_ieee80211_atheros_ie_xr_xr_beacon = -1;
static int hf_ieee80211_atheros_ie_xr_base_cap = -1;
static int hf_ieee80211_atheros_ie_xr_xr_cap = -1;
static int hf_ieee80211_atheros_ie_data = -1;

/*QBSS - Version 1,2,802.11e*/

static int hf_ieee80211_qbss2_cal = -1;
static int hf_ieee80211_qbss2_gl = -1;
static int hf_ieee80211_qbss_cu = -1;
static int hf_ieee80211_qbss2_cu = -1;
static int hf_ieee80211_qbss_scount = -1;
static int hf_ieee80211_qbss2_scount = -1;
static int hf_ieee80211_qbss_version = -1;
static int hf_ieee80211_qbss_adc = -1;

static int hf_ieee80211_tsinfo = -1;
static int hf_ieee80211_tsinfo_type = -1;
static int hf_ieee80211_tsinfo_tsid = -1;
static int hf_ieee80211_tsinfo_dir = -1;
static int hf_ieee80211_tsinfo_access = -1;
static int hf_ieee80211_tsinfo_agg = -1;
static int hf_ieee80211_tsinfo_apsd = -1;
static int hf_ieee80211_tsinfo_up = -1;
static int hf_ieee80211_tsinfo_ack = -1;
static int hf_ieee80211_tsinfo_sched = -1;
static int hf_ieee80211_tsinfo_rsv = -1;

static const int *ieee80211_tsinfo_fields[] = {
  &hf_ieee80211_tsinfo_type,
  &hf_ieee80211_tsinfo_tsid,
  &hf_ieee80211_tsinfo_dir,
  &hf_ieee80211_tsinfo_access,
  &hf_ieee80211_tsinfo_agg,
  &hf_ieee80211_tsinfo_apsd,
  &hf_ieee80211_tsinfo_up,
  &hf_ieee80211_tsinfo_ack,
  &hf_ieee80211_tsinfo_sched,
  &hf_ieee80211_tsinfo_rsv,
  NULL
};

static int hf_ieee80211_tspec_nor_msdu = -1;
static int hf_ieee80211_tspec_max_msdu = -1;
static int hf_ieee80211_tspec_min_srv = -1;
static int hf_ieee80211_tspec_max_srv = -1;
static int hf_ieee80211_tspec_inact_int = -1;
static int hf_ieee80211_tspec_susp_int = -1;
static int hf_ieee80211_tspec_srv_start = -1;
static int hf_ieee80211_tspec_min_data = -1;
static int hf_ieee80211_tspec_mean_data = -1;
static int hf_ieee80211_tspec_peak_data = -1;
static int hf_ieee80211_tspec_burst_size = -1;
static int hf_ieee80211_tspec_delay_bound = -1;
static int hf_ieee80211_tspec_min_phy = -1;
static int hf_ieee80211_tspec_surplus = -1;
static int hf_ieee80211_tspec_medium = -1;
static int hf_ieee80211_tspec_dmg = -1;
static int hf_ieee80211_ts_delay = -1;
static int hf_ieee80211_tclas_process = -1;
static int hf_ieee80211_tag_ext_supp_rates = -1;
static int hf_ieee80211_sched_info = -1;
static int hf_ieee80211_sched_info_agg = -1;
static int hf_ieee80211_sched_info_tsid = -1;
static int hf_ieee80211_sched_info_dir = -1;
static int hf_ieee80211_sched_srv_start = -1;
static int hf_ieee80211_sched_srv_int = -1;
static int hf_ieee80211_sched_spec_int = -1;
static int hf_ieee80211_tclas_up = -1;
static int hf_ieee80211_tclas_class_type = -1;
static int hf_ieee80211_tclas_class_mask = -1;
static int hf_ieee80211_tclas_class_mask0_src_addr = -1;
static int hf_ieee80211_tclas_class_mask0_dst_addr = -1;
static int hf_ieee80211_tclas_class_mask0_type = -1;
static int hf_ieee80211_tclas_class_mask1_ver = -1;
static int hf_ieee80211_tclas_class_mask1_src_ip = -1;
static int hf_ieee80211_tclas_class_mask1_dst_ip = -1;
static int hf_ieee80211_tclas_class_mask1_src_port = -1;
static int hf_ieee80211_tclas_class_mask1_dst_port = -1;
static int hf_ieee80211_tclas_class_mask1_ipv4_dscp = -1;
static int hf_ieee80211_tclas_class_mask1_ipv4_proto = -1;
static int hf_ieee80211_tclas_class_mask1_ipv6_flow = -1;
static int hf_ieee80211_tclas_class_mask2_tci = -1;
static int hf_ieee80211_tclas_src_mac_addr = -1;
static int hf_ieee80211_tclas_dst_mac_addr = -1;
static int hf_ieee80211_tclas_ether_type = -1;
static int hf_ieee80211_tclas_version = -1;
static int hf_ieee80211_tclas_ipv4_src = -1;
static int hf_ieee80211_tclas_ipv4_dst = -1;
static int hf_ieee80211_tclas_src_port = -1;
static int hf_ieee80211_tclas_dst_port = -1;
static int hf_ieee80211_tclas_dscp = -1;
static int hf_ieee80211_tclas_protocol = -1;
static int hf_ieee80211_tclas_ipv6_src = -1;
static int hf_ieee80211_tclas_ipv6_dst = -1;
static int hf_ieee80211_tclas_flow = -1;
static int hf_ieee80211_tclas_tag_type = -1;

static int hf_ieee80211_aruba = -1;
static int hf_ieee80211_aruba_hb_seq = -1;
static int hf_ieee80211_aruba_mtu = -1;

static int hf_ieee80211_tag_vendor_oui_type = -1;
static int hf_ieee80211_tag_vendor_data = -1;

/* IEEE Std 802.11z-2010 7.3.2.62 */
static int hf_ieee80211_tag_link_id_bssid = -1;
static int hf_ieee80211_tag_link_id_init_sta = -1;
static int hf_ieee80211_tag_link_id_resp_sta = -1;

/* IEEE Std 802.11z-2010 7.3.2.63 */
static int hf_ieee80211_tag_wakeup_schedule_offset = -1;
static int hf_ieee80211_tag_wakeup_schedule_interval = -1;
static int hf_ieee80211_tag_wakeup_schedule_awake_window_slots = -1;
static int hf_ieee80211_tag_wakeup_schedule_max_awake_dur = -1;
static int hf_ieee80211_tag_wakeup_schedule_idle_count = -1;

/* IEEE Std 802.11z-2010 7.3.2.64 */
static int hf_ieee80211_tag_channel_switch_timing_switch_time = -1;
static int hf_ieee80211_tag_channel_switch_timing_switch_timeout = -1;

/* IEEE Std 802.11z-2010 7.3.2.65 */
static int hf_ieee80211_tag_pti_control_tid = -1;
static int hf_ieee80211_tag_pti_control_sequence_control = -1;

/* IEEE Std 802.11z-2010 7.3.2.66 */
static int hf_ieee80211_tag_pu_buffer_status_ac_bk = -1;
static int hf_ieee80211_tag_pu_buffer_status_ac_be = -1;
static int hf_ieee80211_tag_pu_buffer_status_ac_vi = -1;
static int hf_ieee80211_tag_pu_buffer_status_ac_vo = -1;

/* IEEE Std 802.11r-2008 7.3.2.49 */
static int hf_ieee80211_tag_timeout_int_type = -1;
static int hf_ieee80211_tag_timeout_int_value = -1;

/* Ethertype 89-0d */
static int hf_ieee80211_data_encap_payload_type = -1;

static int hf_ieee80211_anqp_wfa_subtype = -1;

static int hf_ieee80211_dpp_subtype = -1;

/* Hotspot 2.0 */
static int hf_hs20_indication_dgaf_disabled = -1;
static int hf_hs20_indication_pps_mo_id_present = -1;
static int hf_hs20_indication_anqp_domain_id_present = -1;
static int hf_hs20_reserved = -1;
static int hf_hs20_indication_release_number = -1;
static int hf_hs20_indication_pps_mo_id = -1;
static int hf_hs20_indication_anqp_domain_id = -1;

static int hf_hs20_anqp_subtype = -1;
static int hf_hs20_anqp_reserved = -1;
static int hf_hs20_anqp_payload = -1;
static int hf_hs20_anqp_hs_query_list = -1;
static int hf_hs20_anqp_hs_capability_list = -1;
static int hf_hs20_anqp_ofn_length = -1;
static int hf_hs20_anqp_ofn_language = -1;
static int hf_hs20_anqp_ofn_name = -1;
static int hf_hs20_anqp_wan_metrics_link_status = -1;
static int hf_hs20_anqp_wan_metrics_symmetric_link = -1;
static int hf_hs20_anqp_wan_metrics_at_capacity = -1;
static int hf_hs20_anqp_wan_metrics_reserved = -1;
static int hf_hs20_anqp_wan_metrics_downlink_speed = -1;
static int hf_hs20_anqp_wan_metrics_uplink_speed = -1;
static int hf_hs20_anqp_wan_metrics_downlink_load = -1;
static int hf_hs20_anqp_wan_metrics_uplink_load = -1;
static int hf_hs20_anqp_wan_metrics_lmd = -1;
static int hf_hs20_anqp_cc_proto_ip_proto = -1;
static int hf_hs20_anqp_cc_proto_port_num = -1;
static int hf_hs20_anqp_cc_proto_status = -1;
static int hf_hs20_anqp_nai_hrq_count = -1;
static int hf_hs20_anqp_nai_hrq_encoding_type = -1;
static int hf_hs20_anqp_nai_hrq_length = -1;
static int hf_hs20_anqp_nai_hrq_realm_name = -1;
static int hf_hs20_anqp_oper_class_indic = -1;
static int hf_hs20_osu_friendly_names_len = -1;
static int hf_hs20_osu_friendly_name_length = -1;
static int hf_hs20_osu_friendly_name_language = -1;
static int hf_hs20_osu_friendly_name_name = -1;
static int hf_hs20_osu_server_uri_len = -1;
static int hf_hs20_osu_server_uri = -1;
static int hf_hs20_osu_method_list_len = -1;
static int hf_hs20_osu_method_val = -1;
static int hf_hs20_icons_avail_len = -1;
static int hf_hs20_osu_providers_list_ssid_len = -1;
static int hf_hs20_osu_providers_ssid = -1;
static int hf_hs20_osu_providers_count = -1;
static int hf_hs20_osu_prov_length = -1;
static int hf_hs20_icon_request_filename = -1;
static int hf_hs20_icon_binary_file_status = -1;
static int hf_hs20_icon_type_length = -1;
static int hf_hs20_icon_type = -1;
static int hf_hs20_icon_binary_data_len = -1;
static int hf_hs20_icon_binary_data = -1;
static int hf_osu_icon_avail_width = -1;
static int hf_osu_icon_avail_height = -1;
static int hf_osu_icon_avail_lang_code = -1;
static int hf_osu_icon_avail_icon_type_len = -1;
static int hf_osu_icon_avail_icon_type = -1;
static int hf_osu_icon_avail_filename_len = -1;
static int hf_osu_icon_avail_filename = -1;
static int hf_hs20_osu_nai_len = -1;
static int hf_hs20_osu_nai = -1;
static int hf_hs20_osu_service_desc_len = -1;
static int hf_hs20_osu_service_desc_duple_len = -1;
static int hf_hs20_osu_service_desc_lang = -1;
static int hf_hs20_osu_service_desc = -1;
static int hf_hs20_anqp_venue_url_length = -1;
static int hf_hs20_anqp_venue_number = -1;
static int hf_hs20_anqp_venue_url = -1;
static int hf_hs20_anqp_advice_of_charge_length = -1;
static int hf_hs20_anqp_advice_of_charge_type = -1;
static int hf_hs20_anqp_aoc_nai_realm_encoding = -1;
static int hf_hs20_anqp_aoc_nai_realm_len = -1;
static int hf_hs20_anqp_aoc_nai_realm = -1;
static int hf_hs20_anqp_aoc_plan_len = -1;
static int hf_hs20_anqp_aoc_plan_lang = -1;
static int hf_hs20_anqp_aoc_plan_curcy = -1;
static int hf_hs20_anqp_aoc_plan_information = -1;

static int hf_hs20_subscription_remediation_url_len = -1;
static int hf_hs20_subscription_remediation_server_url = -1;
static int hf_hs20_subscription_remediation_server_method = -1;
static int hf_hs20_deauth_reason_code = -1;
static int hf_hs20_reauth_delay = -1;
static int hf_hs20_deauth_reason_url_len = -1;
static int hf_hs20_deauth_imminent_reason_url = -1;

/* IEEE Std 802.11ad */
static int hf_ieee80211_block_ack_RBUFCAP = -1;
static int hf_ieee80211_cf_response_offset = -1;
static int hf_ieee80211_grant_ack_reserved = -1;
static int hf_ieee80211_ff_dynamic_allocation = -1;
static int hf_ieee80211_ff_TID = -1;
static int hf_ieee80211_ff_alloc_type = -1;
static int hf_ieee80211_ff_src_aid = -1;
static int hf_ieee80211_ff_dest_aid = -1;
static int hf_ieee80211_ff_alloc_duration = -1;
static int hf_ieee80211_ff_b39 = -1;
static int hf_ieee80211_ff_ssw = -1;
static int hf_ieee80211_ff_ssw_direction = -1;
static int hf_ieee80211_ff_ssw_cdown = -1;
static int hf_ieee80211_ff_ssw_sector_id = -1;
static int hf_ieee80211_ff_ssw_dmg_ant_id = -1;
static int hf_ieee80211_ff_ssw_rxss_len = -1;
static int hf_ieee80211_ff_bf = -1;
static int hf_ieee80211_ff_bf_train = -1;
static int hf_ieee80211_ff_bf_is_init = -1;
static int hf_ieee80211_ff_bf_is_resp = -1;
static int hf_ieee80211_ff_bf_num_sectors = -1;
static int hf_ieee80211_ff_bf_num_rx_dmg_ants = -1;
static int hf_ieee80211_ff_bf_b12b15 = -1;
static int hf_ieee80211_ff_bf_rxss_len = -1;
static int hf_ieee80211_ff_bf_rxss_rate = -1;
static int hf_ieee80211_ff_bf_b10b15 = -1;
static int hf_ieee80211_addr_nav_da = -1;
static int hf_ieee80211_addr_nav_sa = -1;
static int hf_ieee80211_ff_sswf = -1;
static int hf_ieee80211_ff_sswf_num_rx_dmg_ants = -1;
static int hf_ieee80211_ff_sswf_poll_required = -1;
static int hf_ieee80211_ff_sswf_total_sectors = -1;
static int hf_ieee80211_ff_sswf_reserved1 = -1;
static int hf_ieee80211_ff_sswf_reserved2 = -1;
static int hf_ieee80211_ff_sswf_sector_select = -1;
static int hf_ieee80211_ff_sswf_dmg_antenna_select = -1;
static int hf_ieee80211_ff_sswf_snr_report = -1;
static int hf_ieee80211_ff_brp = -1;
static int hf_ieee80211_ff_brp_L_RX = -1;
static int hf_ieee80211_ff_brp_TX_TRN_REQ = -1;
static int hf_ieee80211_ff_brp_MID_REQ = -1;
static int hf_ieee80211_ff_brp_BC_REQ = -1;
static int hf_ieee80211_ff_brp_MID_GRANT = -1;
static int hf_ieee80211_ff_brp_BC_GRANT = -1;
static int hf_ieee80211_ff_brp_chan_FBCK_CAP = -1;
static int hf_ieee80211_ff_brp_tx_sector = -1;
static int hf_ieee80211_ff_brp_other_aid = -1;
static int hf_ieee80211_ff_brp_tx_antenna = -1;
static int hf_ieee80211_ff_brp_reserved = -1;
static int hf_ieee80211_ff_blm = -1;
static int hf_ieee80211_ff_blm_unit_index = -1;
static int hf_ieee80211_ff_blm_maint_value = -1;
static int hf_ieee80211_ff_blm_is_master = -1;
static int hf_ieee80211_ff_bic = -1;
static int hf_ieee80211_ff_bic_cc_present = -1;
static int hf_ieee80211_ff_bic_discovery_mode = -1;
static int hf_ieee80211_ff_bic_next_beacon = -1;
static int hf_ieee80211_ff_bic_ati_present = -1;
static int hf_ieee80211_ff_bic_abft_len = -1;
static int hf_ieee80211_ff_bic_fss = -1;
static int hf_ieee80211_ff_bic_is_resp = -1;
static int hf_ieee80211_ff_bic_next_abft = -1;
static int hf_ieee80211_ff_bic_frag_txss = -1;
static int hf_ieee80211_ff_bic_txss_span = -1;
static int hf_ieee80211_ff_bic_NBI_abft = -1;
static int hf_ieee80211_ff_bic_abft_count = -1;
static int hf_ieee80211_ff_bic_nabft = -1;
static int hf_ieee80211_ff_bic_pcp = -1;
static int hf_ieee80211_ff_bic_reserved = -1;
static const int *ieee80211_ff_bic_fields[] = {
  &hf_ieee80211_ff_bic_cc_present,
  &hf_ieee80211_ff_bic_discovery_mode,
  &hf_ieee80211_ff_bic_next_beacon,
  &hf_ieee80211_ff_bic_ati_present,
  &hf_ieee80211_ff_bic_abft_len,
  &hf_ieee80211_ff_bic_fss,
  &hf_ieee80211_ff_bic_is_resp,
  &hf_ieee80211_ff_bic_next_abft,
  &hf_ieee80211_ff_bic_frag_txss,
  &hf_ieee80211_ff_bic_txss_span,
  &hf_ieee80211_ff_bic_NBI_abft,
  &hf_ieee80211_ff_bic_abft_count,
  &hf_ieee80211_ff_bic_nabft,
  &hf_ieee80211_ff_bic_pcp,
  &hf_ieee80211_ff_bic_reserved,
  NULL
};
static int hf_ieee80211_ff_dmg_params = -1;
static int hf_ieee80211_ff_dmg_params_bss = -1;
static int hf_ieee80211_ff_dmg_params_cbap_only = -1;
static int hf_ieee80211_ff_dmg_params_cbap_src = -1;
static int hf_ieee80211_ff_dmg_params_privacy = -1;
static int hf_ieee80211_ff_dmg_params_policy = -1;
static int hf_ieee80211_ff_cc = -1;
static int hf_ieee80211_ff_cc_abft_resp_addr = -1;
static int hf_ieee80211_ff_cc_sp_duration = -1;
static int hf_ieee80211_ff_cc_cluster_id = -1;
static int hf_ieee80211_ff_cc_role = -1;
static int hf_ieee80211_ff_cc_max_mem = -1;
static int hf_ieee80211_ff_dmg_action_code = -1;
static int hf_ieee80211_ff_dmg_pwr_mgmt = -1;
static int hf_ieee80211_ff_subject_address = -1;
static int hf_ieee80211_ff_handover_reason = -1;
static int hf_ieee80211_ff_handover_remaining_bi = -1;
static int hf_ieee80211_ff_handover_result = -1;
static int hf_ieee80211_ff_handover_reject_reason = -1;
static int hf_ieee80211_ff_destination_reds_aid = -1;
static int hf_ieee80211_ff_destination_aid = -1;
static int hf_ieee80211_ff_realy_aid = -1;
static int hf_ieee80211_ff_source_aid = -1;
static int hf_ieee80211_ff_timing_offset = -1;
static int hf_ieee80211_ff_sampling_frequency_offset = -1;
static int hf_ieee80211_ff_relay_operation_type = -1;
static int hf_ieee80211_ff_peer_sta_aid = -1;
static int hf_ieee80211_ff_snr = -1;
static int hf_ieee80211_ff_internal_angle = -1;
static int hf_ieee80211_ff_recommend = -1;
static int hf_ieee80211_ff_unprotected_dmg_action_code = -1;
static int hf_ieee80211_ff_fst_action_code = -1;
static int hf_ieee80211_ff_llt = -1;
static int hf_ieee80211_ff_fsts_id = -1;
static int hf_ieee80211_ff_mmpdu_len = -1;
static int hf_ieee80211_ff_mmpdu_ctrl = -1;
static int hf_ieee80211_ff_oct_mmpdu = -1;
#if 0
static int hf_ieee80211_ff_rcsi = -1;
static int hf_ieee80211_ff_rcsi_aid = -1;
#endif
static int hf_ieee80211_ff_band_id = -1;
static int hf_ieee80211_tag_relay_support = -1;
static int hf_ieee80211_tag_relay_use = -1;
static int hf_ieee80211_tag_relay_permission = -1;
static int hf_ieee80211_tag_AC_power = -1;
static int hf_ieee80211_tag_relay_prefer = -1;
static int hf_ieee80211_tag_duplex = -1;
static int hf_ieee80211_tag_cooperation = -1;
static int hf_ieee80211_tag_move = -1;
static int hf_ieee80211_tag_size = -1;
static int hf_ieee80211_tag_tbtt_offset = -1;
static int hf_ieee80211_tag_bi_duration = -1;
static int hf_ieee80211_tag_dmg_capa_sta_addr = -1;
static int hf_ieee80211_tag_dmg_capa_aid = -1;
static int hf_ieee80211_tag_reverse_direction = -1;
static int hf_ieee80211_tag_hlts = -1;
static int hf_ieee80211_tag_tpc = -1;
static int hf_ieee80211_tag_spsh = -1;
static int hf_ieee80211_tag_rx_antenna = -1;
static int hf_ieee80211_tag_fast_link = -1;
static int hf_ieee80211_tag_num_sectors = -1;
static int hf_ieee80211_tag_rxss_length = -1;
static int hf_ieee80211_tag_reciprocity = -1;
static int hf_ieee80211_tag_max_ampdu_exp = -1;
static int hf_ieee80211_tag_min_mpdu_spacing = -1;
static int hf_ieee80211_tag_ba_flow_control = -1;
static int hf_ieee80211_tag_max_sc_rx_mcs = -1;
static int hf_ieee80211_tag_max_ofdm_rx_mcs = -1;
static int hf_ieee80211_tag_max_sc_tx_mcs = -1;
static int hf_ieee80211_tag_max_ofdm_tx_mcs = -1;
static int hf_ieee80211_tag_low_power_supported = -1;
static int hf_ieee80211_tag_code_rate = -1;
static int hf_ieee80211_tag_dtp = -1;
static int hf_ieee80211_tag_appdu_supp = -1;
static int hf_ieee80211_tag_heartbeat = -1;
static int hf_ieee80211_tag_other_aid = -1;
static int hf_ieee80211_tag_pattern_recip = -1;
static int hf_ieee80211_tag_heartbeat_elapsed = -1;
static int hf_ieee80211_tag_grant_ack_supp = -1;
static int hf_ieee80211_tag_RXSSTxRate_supp = -1;
static int hf_ieee80211_tag_pcp_tddti = -1;
static int hf_ieee80211_tag_pcp_PSA = -1;
static int hf_ieee80211_tag_pcp_handover = -1;
static int hf_ieee80211_tag_pcp_max_assoc = -1;
static int hf_ieee80211_tag_pcp_power_src = -1;
static int hf_ieee80211_tag_pcp_decenter = -1;
static int hf_ieee80211_tag_pcp_forwarding = -1;
static int hf_ieee80211_tag_pcp_center = -1;
static int hf_ieee80211_tag_sta_beam_track = -1;
static int hf_ieee80211_tag_ext_sc_mcs_max_tx = -1;
static int hf_ieee80211_tag_ext_sc_mcs_tx_code_7_8 = -1;
static int hf_ieee80211_tag_ext_sc_mcs_max_rx = -1;
static int hf_ieee80211_tag_ext_sc_mcs_rx_code_7_8 = -1;
static int hf_ieee80211_tag_max_basic_sf_amsdu = -1;
static int hf_ieee80211_tag_max_short_sf_amsdu = -1;
static int hf_ieee80211_tag_PSRSI = -1;
static int hf_ieee80211_tag_min_BHI_duration = -1;
static int hf_ieee80211_tag_brdct_sta_info_dur = -1;
static int hf_ieee80211_tag_assoc_resp_confirm_time = -1;
static int hf_ieee80211_tag_min_pp_duration = -1;
static int hf_ieee80211_tag_SP_idle_timeout = -1;
static int hf_ieee80211_tag_max_lost_beacons = -1;
static int hf_ieee80211_tag_type = -1;
static int hf_ieee80211_tag_tap1 = -1;
static int hf_ieee80211_tag_state1 = -1;
static int hf_ieee80211_tag_tap2 = -1;
static int hf_ieee80211_tag_state2 = -1;
static int hf_ieee80211_tag_allocation_id = -1;
static int hf_ieee80211_tag_allocation_type = -1;
static int hf_ieee80211_tag_pseudo_static = -1;
static int hf_ieee80211_tag_truncatable = -1;
static int hf_ieee80211_tag_extendable = -1;
static int hf_ieee80211_tag_pcp_active = -1;
static int hf_ieee80211_tag_lp_sc_used = -1;
static int hf_ieee80211_tag_src_aid = -1;
static int hf_ieee80211_tag_dest_aid = -1;
static int hf_ieee80211_tag_alloc_start = -1;
static int hf_ieee80211_tag_alloc_block_duration = -1;
static int hf_ieee80211_tag_num_blocks = -1;
static int hf_ieee80211_tag_alloc_block_period = -1;
static int hf_ieee80211_tag_aid = -1;
static int hf_ieee80211_tag_cbap = -1;
static int hf_ieee80211_tag_pp_avail = -1;
static int hf_ieee80211_tag_next_ati_start_time = -1;
static int hf_ieee80211_tag_next_ati_duration = -1;
static int hf_ieee80211_tag_old_bssid = -1;
static int hf_ieee80211_tag_new_pcp_addr = -1;
static int hf_ieee80211_tag_bssid = -1;
static int hf_ieee80211_tag_duplex_relay = -1;
static int hf_ieee80211_tag_cooperation_relay = -1;
static int hf_ieee80211_tag_tx_mode = -1;
static int hf_ieee80211_tag_link_change_interval = -1;
static int hf_ieee80211_tag_data_sensing_time = -1;
static int hf_ieee80211_tag_first_period = -1;
static int hf_ieee80211_tag_second_period = -1;
static int hf_ieee80211_tag_initiator = -1;
static int hf_ieee80211_tag_tx_train_res = -1;
static int hf_ieee80211_tag_rx_train_res = -1;
static int hf_ieee80211_tag_tx_trn_ok = -1;
static int hf_ieee80211_tag_txss_fbck_req = -1;
static int hf_ieee80211_tag_bs_fbck = -1;
static int hf_ieee80211_tag_bs_fbck_antenna_id = -1;
static int hf_ieee80211_tag_snr_requested = -1;
static int hf_ieee80211_tag_channel_measurement_requested = -1;
static int hf_ieee80211_tag_number_of_taps_requested = -1;
static int hf_ieee80211_tag_sector_id_order_req = -1;
static int hf_ieee80211_tag_snr_present = -1;
static int hf_ieee80211_tag_channel_measurement_present = -1;
static int hf_ieee80211_tag_tap_delay_present = -1;
static int hf_ieee80211_tag_number_of_taps_present = -1;
static int hf_ieee80211_tag_number_of_measurement = -1;
static int hf_ieee80211_tag_sector_id_order_present = -1;
static int hf_ieee80211_tag_number_of_beams = -1;
static int hf_ieee80211_tag_mid_extension = -1;
static int hf_ieee80211_tag_capability_request = -1;
static int hf_ieee80211_tag_beam_refine_reserved = -1;
static int hf_ieee80211_tag_nextpcp_list = -1;
static int hf_ieee80211_tag_nextpcp_token = -1;
static int hf_ieee80211_tag_reamaining_BI = -1;
static int hf_ieee80211_tag_request_token = -1;
static int hf_ieee80211_tag_bi_start_time = -1;
static int hf_ieee80211_tag_sleep_cycle = -1;
static int hf_ieee80211_tag_num_awake_bis = -1;
static int hf_ieee80211_tag_tspec_allocation_id = -1;
static int hf_ieee80211_tag_tspec_allocation_type = -1;
static int hf_ieee80211_tag_tspec_allocation_format = -1;
static int hf_ieee80211_tag_tspec_pseudo_static = -1;
static int hf_ieee80211_tag_tspec_truncatable = -1;
static int hf_ieee80211_tag_tspec_extendable = -1;
static int hf_ieee80211_tag_tspec_lp_sc_used = -1;
static int hf_ieee80211_tag_tspec_up = -1;
static int hf_ieee80211_tag_tspec_dest_aid = -1;
static int hf_ieee80211_tag_tspec_allocation_period = -1;
static int hf_ieee80211_tag_tspec_min_allocation = -1;
static int hf_ieee80211_tag_tspec_max_allocation = -1;
static int hf_ieee80211_tag_tspec_min_duration = -1;
static int hf_ieee80211_tag_tspec_num_of_constraints = -1;
static int hf_ieee80211_tag_tspec_tsconst_start_time = -1;
static int hf_ieee80211_tag_tspec_tsconst_duration = -1;
static int hf_ieee80211_tag_tspec_tsconst_period = -1;
static int hf_ieee80211_tag_tspec_tsconst_interferer_mac = -1;
static int hf_ieee80211_tag_channel_measurement_feedback_realtive_I = -1;
static int hf_ieee80211_tag_channel_measurement_feedback_realtive_Q = -1;
static int hf_ieee80211_tag_channel_measurement_feedback_tap_delay = -1;
static int hf_ieee80211_tag_channel_measurement_feedback_sector_id = -1;
static int hf_ieee80211_tag_channel_measurement_feedback_antenna_id = -1;
static int hf_ieee80211_tag_awake_window = -1;
static int hf_ieee80211_tag_addba_ext_no_frag = -1;
static int hf_ieee80211_tag_addba_ext_he_fragmentation_operation = -1;
static int hf_ieee80211_tag_addba_ext_reserved = -1;
static int hf_ieee80211_tag_multi_band_ctrl_sta_role = -1;
static int hf_ieee80211_tag_multi_band_ctrl_addr_present = -1;
static int hf_ieee80211_tag_multi_band_ctrl_cipher_present = -1;
static int hf_ieee80211_tag_multi_band_oper_class = -1;
static int hf_ieee80211_tag_multi_band_channel_number = -1;
static int hf_ieee80211_tag_multi_band_tsf_offset = -1;
static int hf_ieee80211_tag_multi_band_conn_ap = -1;
static int hf_ieee80211_tag_multi_band_conn_pcp = -1;
static int hf_ieee80211_tag_multi_band_conn_dls = -1;
static int hf_ieee80211_tag_multi_band_conn_tdls = -1;
static int hf_ieee80211_tag_multi_band_conn_ibss = -1;
static int hf_ieee80211_tag_multi_band_fst_timeout = -1;
static int hf_ieee80211_tag_multi_band_sta_mac = -1;
static int hf_ieee80211_tag_activity = -1;
static int hf_ieee80211_tag_dmg_link_adapt_mcs = -1;
static int hf_ieee80211_tag_dmg_link_adapt_link_margin = -1;
static int hf_ieee80211_tag_ref_timestamp = -1;
static int hf_ieee80211_tag_switching_stream_non_qos = -1;
static int hf_ieee80211_tag_switching_stream_param_num = -1;
static int hf_ieee80211_tag_switching_stream_old_tid = -1;
static int hf_ieee80211_tag_switching_stream_old_direction = -1;
static int hf_ieee80211_tag_switching_stream_new_tid = -1;
static int hf_ieee80211_tag_switching_stream_new_direction = -1;
static int hf_ieee80211_tag_switching_stream_new_valid_id = -1;
static int hf_ieee80211_tag_switching_stream_llt_type = -1;

static int hf_ieee80211_mysterious_olpc_stuff = -1;

static int hf_ieee80211_esp_access_category = -1;
static int hf_ieee80211_esp_reserved = -1;
static int hf_ieee80211_esp_data_format = -1;
static int hf_ieee80211_esp_ba_windows_size = -1;
static int hf_ieee80211_esp_est_air_time_frac = -1;
static int hf_ieee80211_esp_data_ppdu_duration_target = -1;
static int hf_ieee80211_estimated_service_params = -1;

static int hf_ieee80211_fcg_new_channel_number = -1;
static int hf_ieee80211_fcg_extra_info = -1;

static int hf_ieee80211_tag_fils_indication_info_nr_pk = -1;
static int hf_ieee80211_tag_fils_indication_info_nr_realm = -1;
static int hf_ieee80211_tag_fils_indication_info_ip_config = -1;
static int hf_ieee80211_tag_fils_indication_info_cache_id_included = -1;
static int hf_ieee80211_tag_fils_indication_info_hessid_included = -1;
static int hf_ieee80211_tag_fils_indication_info_ska_without_pfs = -1;
static int hf_ieee80211_tag_fils_indication_info_ska_with_pfs = -1;
static int hf_ieee80211_tag_fils_indication_info_pka = -1;
static int hf_ieee80211_tag_fils_indication_info_reserved = -1;
static int hf_ieee80211_tag_fils_indication_cache_identifier = -1;
static int hf_ieee80211_tag_fils_indication_hessid = -1;
static int hf_ieee80211_tag_fils_indication_realm_list = -1;
static int hf_ieee80211_tag_fils_indication_realm_identifier = -1;
static int hf_ieee80211_tag_fils_indication_public_key_list = -1;
static int hf_ieee80211_tag_fils_indication_public_key_type = -1;
static int hf_ieee80211_tag_fils_indication_public_key_length = -1;
static int hf_ieee80211_tag_fils_indication_public_key_indicator = -1;

static int hf_ieee80211_ext_tag = -1;
static int hf_ieee80211_ext_tag_number = -1;
static int hf_ieee80211_ext_tag_length = -1;
static int hf_ieee80211_fils_session = -1;
static int hf_ieee80211_fils_encrypted_data = -1;
static int hf_ieee80211_fils_wrapped_data = -1;
static int hf_ieee80211_fils_nonce = -1;

/* ************************************************************************* */
/*                              802.11AX fields                              */
/* ************************************************************************* */
static int hf_he_mac_capabilities = -1;
static int hf_he_htc_he_support = -1;
static int hf_he_twt_requester_support = -1;
static int hf_he_twt_responder_support = -1;
static int hf_he_fragmentation_support = -1;
static int hf_he_max_number_fragmented_msdus = -1;
static int hf_he_min_fragment_size = -1;
static int hf_he_trigger_frame_mac_padding_dur = -1;
static int hf_he_multi_tid_aggregation_support = -1;
static int hf_he_he_link_adaptation_support = -1;
static int hf_he_all_ack_support = -1;
static int hf_he_trs_support = -1;
static int hf_he_bsr_support = -1;
static int hf_he_broadcast_twt_support = -1;
static int hf_he_32_bit_ba_bitmap_support = -1;
static int hf_he_mu_cascading_support = -1;
static int hf_he_ack_enabled_aggregation_support = -1;
static int hf_he_reserved_b24 = -1;
static int hf_he_om_control_support = -1;
static int hf_he_ofdma_ra_support = -1;
static int hf_he_max_a_mpdu_length_exponent_ext = -1;
static int hf_he_a_msdu_fragmentation_support = -1;
static int hf_he_flexible_twt_schedule_support = -1;
static int hf_he_rx_control_frame_to_multibss = -1;
static int hf_he_bsrp_bqrp_a_mpdu_aggregation = -1;
static int hf_he_qtp_support = -1;
static int hf_he_bqr_support = -1;
static int hf_he_srp_responder = -1;
static int hf_he_ndp_feedback_report_support = -1;
static int hf_he_ops_support = -1;
static int hf_he_a_msdu_in_a_mpdu_support = -1;
static int hf_he_multi_tid_aggregation_tx_support = -1;
static int hf_he_subchannel_selective_trans_support = -1;
static int hf_he_2_996_tone_ru_support = -1;
static int hf_he_om_control_ul_mu_data_disable_rx_support = -1;
static int hf_he_reserved_bits_45_47 = -1;
static int hf_he_reserved_bit_18 = -1;
static int hf_he_reserved_bit_19 = -1;
static int hf_he_reserved_bit_25 = -1;
static int hf_he_reserved_bits_5_7 = -1;
static int hf_he_reserved_bits_8_9 = -1;
static int hf_he_reserved_bits_15_16 = -1;
static int hf_he_phy_reserved_b0 = -1;
static int hf_he_phy_cap_reserved_b0 = -1;
static int hf_he_phy_chan_width_set = -1;
static int hf_he_40mhz_channel_2_4ghz = -1;
static int hf_he_40_and_80_mhz_5ghz = -1;
static int hf_he_160_mhz_5ghz = -1;
static int hf_he_160_80_plus_80_mhz_5ghz = -1;
static int hf_he_242_tone_rus_in_2_4ghz = -1;
static int hf_he_242_tone_rus_in_5ghz = -1;
static int hf_he_chan_width_reserved = -1;
static int hf_he_mcs_max_he_mcs_80_rx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_80_rx_8_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_80_tx_8_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_rx_8_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_80p80_tx_8_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_160_rx_8_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_1_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_2_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_3_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_4_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_5_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_6_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_7_ss = -1;
static int hf_he_mcs_max_he_mcs_160_tx_8_ss = -1;
static int hf_he_rx_he_mcs_map_lte_80 = -1;
static int hf_he_tx_he_mcs_map_lte_80 = -1;
static int hf_he_rx_he_mcs_map_160 = -1;
static int hf_he_tx_he_mcs_map_160 = -1;
static int hf_he_rx_he_mcs_map_80_80 = -1;
static int hf_he_tx_he_mcs_map_80_80 = -1;
static int hf_he_ppe_thresholds_nss = -1;
static int hf_he_ppe_thresholds_ru_index_bitmask = -1;
static int hf_he_ppe_ppet16 = -1;
static int hf_he_ppe_ppet8 = -1;
static int hf_he_phy_b8_to_b23 = -1;
static int hf_he_phy_cap_punctured_preamble_rx = -1;
static int hf_he_phy_cap_device_class = -1;
static int hf_he_phy_cap_ldpc_coding_in_payload = -1;
static int hf_he_phy_cap_he_su_ppdu_1x_he_ltf_08us = -1;
static int hf_he_phy_cap_midamble_rx_max_nsts = -1;
static int hf_he_phy_cap_ndp_with_4x_he_ltf_32us = -1;
static int hf_he_phy_cap_stbc_tx_lt_80mhz = -1;
static int hf_he_phy_cap_stbc_rx_lt_80mhz = -1;
static int hf_he_phy_cap_doppler_tx = -1;
static int hf_he_phy_cap_doppler_rx = -1;
static int hf_he_phy_cap_full_bw_ul_mu_mimo = -1;
static int hf_he_phy_cap_partial_bw_ul_mu_mimo = -1;
static int hf_he_phy_b24_to_b39 = -1;
static int hf_he_phy_cap_dcm_max_constellation_tx = -1;
static int hf_he_phy_cap_dcm_max_nss_tx = -1;
static int hf_he_phy_cap_dcm_max_constellation_rx = -1;
static int hf_he_phy_cap_dcm_max_nss_rx = -1;
static int hf_he_phy_cap_rx_he_muppdu_from_non_ap = -1;
static int hf_he_phy_cap_su_beamformer = -1;
static int hf_he_phy_cap_su_beamformee = -1;
static int hf_he_phy_cap_mu_beamformer = -1;
static int hf_he_phy_cap_beamformer_sts_lte_80mhz = -1;
static int hf_he_phy_cap_beamformer_sts_gt_80mhz = -1;
static int hf_he_phy_b40_to_b55 = -1;
static int hf_he_phy_cap_number_of_sounding_dims_lte_80 = -1;
static int hf_he_phy_cap_number_of_sounding_dims_gt_80 = -1;
static int hf_he_phy_cap_ng_eq_16_su_fb = -1;
static int hf_he_phy_cap_ng_eq_16_mu_fb = -1;
static int hf_he_phy_cap_codebook_size_eq_4_2_fb = -1;
static int hf_he_phy_cap_codebook_size_eq_7_5_fb = -1;
static int hf_he_phy_cap_triggered_su_beamforming_fb = -1;
static int hf_he_phy_cap_triggered_mu_beamforming_fb = -1;
static int hf_he_phy_cap_triggered_cqi_fb = -1;
static int hf_he_phy_cap_partial_bw_extended_range = -1;
static int hf_he_phy_cap_partial_bw_dl_mu_mimo = -1;
static int hf_he_phy_cap_ppe_threshold_present = -1;
static int hf_he_phy_b56_to_b71 = -1;
static int hf_he_phy_cap_srp_based_sr_support = -1;
static int hf_he_phy_cap_power_boost_factor_ar_support = -1;
static int hf_he_phy_cap_he_su_ppdu_etc_gi = -1;
static int hf_he_phy_cap_max_nc = -1;
static int hf_he_phy_cap_stbc_tx_gt_80_mhz = -1;
static int hf_he_phy_cap_stbc_rx_gt_80_mhz = -1;
static int hf_he_phy_cap_he_er_su_ppdu_4xxx_gi = -1;
static int hf_he_phy_cap_20mhz_in_40mhz_24ghz_band = -1;
static int hf_he_phy_cap_20mhz_in_160_80p80_ppdu = -1;
static int hf_he_phy_cap_80mgz_in_160_80p80_ppdu = -1;
static int hf_he_phy_cap_he_er_su_ppdu_1xxx_gi = -1;
static int hf_he_phy_cap_midamble_rx_2x_xxx_ltf = -1;
static int hf_he_phy_b72_to_b87 = -1;
static int hf_he_phy_cap_dcm_max_bw = -1;
static int hf_he_phy_cap_longer_than_16_he_sigb_ofdm_symbol_support = -1;
static int hf_he_phy_cap_non_triggered_cqi_feedback = -1;
static int hf_he_phy_cap_tx_1024_qam_242_tone_ru_support = -1;
static int hf_he_phy_cap_rx_1024_qam_242_tone_ru_support = -1;
static int hf_rx_full_bw_su_using_he_muppdu_w_compressed_sigb = -1;
static int hf_rx_full_bw_su_using_he_muppdu_w_non_compressed_sigb = -1;
static int hf_he_phy_cap_b78_b87_reserved = -1;
static int hf_he_operation_parameter = -1;
static int hf_he_operation_default_pe_duration = -1;
static int hf_he_operation_twt_required = -1;
static int hf_he_operation_txop_duration_rts_threshold = -1;
static int hf_he_operation_vht_operation_information_present = -1;
static int hf_he_operation_co_located_bss = -1;
static int hf_he_operation_er_su_disable = -1;
static int hf_he_operation_reserved_b17_b23 = -1;
static int hf_he_bss_color_information = -1;
static int hf_he_bss_color_info_bss_color = -1;
static int hf_he_bss_color_partial_bss_color = -1;
static int hf_he_bss_color_bss_color_disabled = -1;
static int hf_he_operation_basic_mcs = -1;
static int hf_he_oper_max_he_mcs_for_1_ss = -1;
static int hf_he_oper_max_he_mcs_for_2_ss = -1;
static int hf_he_oper_max_he_mcs_for_3_ss = -1;
static int hf_he_oper_max_he_mcs_for_4_ss = -1;
static int hf_he_oper_max_he_mcs_for_5_ss = -1;
static int hf_he_oper_max_he_mcs_for_6_ss = -1;
static int hf_he_oper_max_he_mcs_for_7_ss = -1;
static int hf_he_oper_max_he_mcs_for_8_ss = -1;
static int hf_he_operation_channel_width = -1;
static int hf_he_operation_channel_center_freq_0 = -1;
static int hf_he_operation_channel_center_freq_1 = -1;
static int hf_he_operation_max_colocated_bssid_indicator = -1;
static int hf_he_muac_aci_aifsn = -1;
static int hf_he_muac_aifsn = -1;
static int hf_he_muac_acm = -1;
static int hf_he_muac_aci = -1;
static int hf_he_muac_reserved = -1;
static int hf_he_mu_edca_timer = -1;
static int hf_he_muac_ecwmin_ecwmax = -1;
static int hf_he_srp_disallowed = -1;
static int hf_he_non_srg_obss_pd_sr_disallowed = -1;
static int hf_he_non_srg_offset_present = -1;
static int hf_he_srg_information_present = -1;
static int hf_he_hesiga_spatial_reuse_value15_allowed = -1;
static int hf_he_sr_control_reserved = -1;
static int hf_he_spatial_reuse_sr_control = -1;
static int hf_he_spatial_non_srg_obss_pd_max_offset = -1;
static int hf_he_spatial_srg_obss_pd_min_offset = -1;
static int hf_he_spatial_srg_obss_pd_max_offset = -1;
static int hf_he_spatial_srg_bss_color_bitmap = -1;
static int hf_he_spatial_srg_partial_bssid_bitmap = -1;
static int hf_he_ess_report_planned_ess = -1;
static int hf_he_ess_report_edge_of_ess = -1;
static int hf_he_resource_request_buffer_thresh = -1;
static int hf_he_bss_color_change_new_color_info = -1;
static int hf_he_new_bss_color_info_color = -1;
static int hf_he_new_bss_color_info_reserved = -1;
static int hf_he_bss_color_change_switch_countdown = -1;
static int hf_he_ess_report_info_field = -1;
static int hf_he_ess_report_recommend_transition_thresh = -1;
static int hf_he_uora_field = -1;
static int hf_he_uora_eocwmin = -1;
static int hf_he_uora_owcwmax = -1;
static int hf_he_uora_reserved = -1;

static int hf_ieee80211_ff_s1g_action = -1;
static int hf_ieee80211_twt_bcast_flow = -1;
static int hf_ieee80211_twt_individual_flow = -1;
static int hf_ieee80211_twt_individual_flow_id = -1;
static int hf_ieee80211_twt_bcast_id = -1;
static int hf_ieee80211_twt_neg_type = -1;
static int hf_ieee80211_tag_twt_control_field = -1;
static int hf_ieee80211_tag_twt_ndp_paging_indicator = -1;
static int hf_ieee80211_tag_twt_responder_pm_mode = -1;
static int hf_ieee80211_tag_twt_neg_type = -1;
static int hf_ieee80211_tag_twt_ctrl_field_reserved = -1;

static int hf_ieee80211_tag_twt_req_type_field = -1;
static int hf_ieee80211_tag_twt_req_type_req = -1;
static int hf_ieee80211_tag_twt_req_type_setup_cmd = -1;
static int hf_ieee80211_tag_twt_req_type_trigger = -1;
static int hf_ieee80211_tag_twt_req_type_implicit = -1;
static int hf_ieee80211_tag_twt_req_type_flow_type = -1;
static int hf_ieee80211_tag_twt_req_type_flow_id = -1;
static int hf_ieee80211_tag_twt_req_type_wake_int_exp = -1;
static int hf_ieee80211_tag_twt_req_type_prot = -1;

static int hf_ieee80211_tag_twt_target_wake_time = -1;
static int hf_ieee80211_tag_twt_nom_min_twt_wake_dur = -1;
static int hf_ieee80211_tag_twt_wake_interval_mantissa = -1;
static int hf_ieee80211_tag_twt_channel = -1;

/* ************************************************************************* */
/*                              RFC 8110 fields                              */
/* ************************************************************************* */
static int hf_owe_dh_parameter_group = -1;
static int hf_owe_dh_parameter_public_key = -1;

/* ************************************************************************* */
/*                               Protocol trees                              */
/* ************************************************************************* */
static gint ett_80211 = -1;
static gint ett_proto_flags = -1;
static gint ett_cap_tree = -1;
static gint ett_fc_tree = -1;
static gint ett_cntrl_wrapper_fc = -1;
static gint ett_cntrl_wrapper_payload = -1;
static gint ett_fragments = -1;
static gint ett_fragment = -1;
static gint ett_block_ack = -1;
static gint ett_block_ack_tid = -1;
static gint ett_block_ack_request_control = -1;
static gint ett_block_ack_bitmap = -1;
static gint ett_block_ack_request_multi_sta_aid_tid = -1;
static gint ett_multi_sta_block_ack = -1;
static gint ett_ath_cap_tree = -1;

static gint ett_80211_mgt = -1;
static gint ett_fixed_parameters = -1;
static gint ett_tagged_parameters = -1;
static gint ett_tag_bmapctl_tree = -1;
static gint ett_tag_country_fnm_tree = -1;
static gint ett_tag_country_rcc_tree = -1;
static gint ett_qos_parameters = -1;
static gint ett_qos_ps_buf_state = -1;
static gint ett_wep_parameters = -1;
static gint ett_msh_control = -1;
static gint ett_hwmp_targ_flags_tree = -1;
static gint ett_mesh_chswitch_flag_tree = -1;
static gint ett_mesh_config_cap_tree = -1;
static gint ett_mesh_formation_info_tree = -1;

static gint ett_rsn_gcs_tree = -1;
static gint ett_rsn_pcs_tree = -1;
static gint ett_rsn_sub_pcs_tree = -1;
static gint ett_rsn_akms_tree = -1;
static gint ett_rsn_sub_akms_tree = -1;
static gint ett_rsn_cap_tree = -1;
static gint ett_rsn_pmkid_tree = -1;
static gint ett_rsn_gmcs_tree = -1;

static gint ett_wpa_mcs_tree = -1;
static gint ett_wpa_ucs_tree = -1;
static gint ett_wpa_sub_ucs_tree = -1;
static gint ett_wpa_akms_tree = -1;
static gint ett_wpa_sub_akms_tree = -1;
static gint ett_wme_ac = -1;
static gint ett_wme_aci_aifsn = -1;
static gint ett_wme_ecw = -1;
static gint ett_wme_qos_info = -1;

static gint ett_ht_cap_tree = -1;
static gint ett_ampduparam_tree = -1;
static gint ett_mcsset_tree = -1;
static gint ett_mcsbit_tree = -1;
static gint ett_htex_cap_tree = -1;
static gint ett_txbf_tree = -1;
static gint ett_antsel_tree = -1;
static gint ett_hta_cap_tree = -1;
static gint ett_hta_cap1_tree = -1;
static gint ett_hta_cap2_tree = -1;

static gint ett_htc_tree = -1;
static gint ett_htc_he_a_control = -1;
static gint ett_mfb_subtree = -1;
static gint ett_lac_subtree = -1;
static gint ett_ieee80211_buffer_status_report = -1;
static gint ett_ieee80211_a_control_padding = -1;
static gint ett_ieee80211_triggered_response_schedule = -1;
static gint ett_ieee80211_control_om = -1;
static gint ett_ieee80211_hla_control = -1;
static gint ett_ieee80211_control_uph = -1;
static gint ett_ieee80211_buffer_control_bqr = -1;
static gint ett_ieee80211_control_cci = -1;

static gint ett_vht_cap_tree = -1;
static gint ett_vht_mcsset_tree = -1;
static gint ett_vht_rx_mcsbit_tree = -1;
static gint ett_vht_tx_mcsbit_tree = -1;
static gint ett_vht_basic_mcsbit_tree = -1;
static gint ett_vht_op_tree = -1;
static gint ett_vht_tpe_info_tree = -1;

static gint ett_vht_ndp_annc = -1;
static gint ett_vht_ndp_annc_sta_info_tree = -1;
static gint ett_vht_ndp_annc_sta_list = -1;

static gint ett_ff_he_action = -1;
static gint ett_ff_protected_he_action = -1;
static gint ett_ff_he_mimo_control = -1;
static gint ett_ff_he_mimo_beamforming_report_snr = -1;
static gint ett_ff_he_mimo_feedback_matrices = -1;

static gint ett_ff_vhtmimo_cntrl = -1;
static gint ett_ff_vhtmimo_beamforming_report = -1;
static gint ett_ff_vhtmimo_beamforming_report_snr = -1;
static gint ett_ff_vhtmimo_beamforming_angle = -1;
static gint ett_ff_vhtmimo_beamforming_report_feedback_matrices = -1;
static gint ett_ff_vhtmu_exclusive_beamforming_report_matrices = -1;

static gint ett_vht_grpidmgmt = -1;
static gint ett_vht_msa = -1;
static gint ett_vht_upa = -1;

static gint ett_ht_info_delimiter1_tree = -1;
static gint ett_ht_info_delimiter2_tree = -1;
static gint ett_ht_info_delimiter3_tree = -1;

static gint ett_tag_measure_request_mode_tree = -1;
static gint ett_tag_measure_request_type_tree = -1;
static gint ett_tag_measure_report_mode_tree = -1;
static gint ett_tag_measure_report_type_tree = -1;
static gint ett_tag_measure_report_basic_map_tree = -1;
static gint ett_tag_measure_report_rpi_tree = -1;
static gint ett_tag_measure_report_frame_tree = -1;
static gint ett_tag_measure_reported_frame_tree = -1;
static gint ett_tag_bss_bitmask_tree = -1;
static gint ett_tag_dfs_map_tree = -1;
static gint ett_tag_dfs_map_flags_tree = -1;
static gint ett_tag_erp_info_tree = -1;
static gint ett_tag_ex_cap1 = -1;
static gint ett_tag_ex_cap2 = -1;
static gint ett_tag_ex_cap3 = -1;
static gint ett_tag_ex_cap4 = -1;
static gint ett_tag_ex_cap5 = -1;
static gint ett_tag_ex_cap6 = -1;
static gint ett_tag_ex_cap7 = -1;
static gint ett_tag_ex_cap8 = -1;
static gint ett_tag_ex_cap89 = -1;
static gint ett_tag_ex_cap10 = -1;

static gint ett_tag_rm_cap1 = -1;
static gint ett_tag_rm_cap2 = -1;
static gint ett_tag_rm_cap3 = -1;
static gint ett_tag_rm_cap4 = -1;
static gint ett_tag_rm_cap5 = -1;

static gint ett_tag_multiple_bssid_subelem_tree = -1;
static gint ett_tag_20_40_bc = -1;

static gint ett_tag_tclas_mask_tree = -1;

static gint ett_tag_supported_channels = -1;

static gint ett_tag_neighbor_report_bssid_info_tree = -1;
static gint ett_tag_neighbor_report_bssid_info_capability_tree = -1;
static gint ett_tag_neighbor_report_subelement_tree = -1;
static gint ett_tag_neighbor_report_sub_tag_tree = -1;

static gint ett_tag_wapi_param_set_akm_tree = -1;
static gint ett_tag_wapi_param_set_ucast_tree = -1;
static gint ett_tag_wapi_param_set_mcast_tree = -1;
static gint ett_tag_wapi_param_set_preauth_tree = -1;

static gint ett_tag_time_adv_tree = -1;

static gint ett_ff_ba_param_tree = -1;
static gint ett_ff_ba_ssc_tree = -1;
static gint ett_ff_delba_param_tree = -1;
static gint ett_ff_qos_info = -1;
static gint ett_ff_sm_pwr_save = -1;
static gint ett_ff_psmp_param_set = -1;
static gint ett_ff_mimo_cntrl = -1;
static gint ett_ff_ant_sel = -1;
static gint ett_mimo_report = -1;
static gint ett_ff_chan_switch_announce = -1;
static gint ett_ff_ht_info = -1;
static gint ett_ff_psmp_sta_info = -1;

static gint ett_tpc = -1;

static gint ett_msdu_aggregation_parent_tree = -1;
static gint ett_msdu_aggregation_subframe_tree = -1;

static gint ett_80211_mgt_ie = -1;
static gint ett_tsinfo_tree = -1;
static gint ett_sched_tree = -1;

static gint ett_fcs = -1;

static gint ett_hs20_osu_providers_list = -1;
static gint ett_hs20_osu_provider_tree = -1;
static gint ett_hs20_friendly_names_list = -1;
static gint ett_hs20_friendly_name_tree = -1;
static gint ett_hs20_osu_provider_method_list = -1;
static gint ett_osu_icons_avail_list = -1;
static gint ett_hs20_osu_icon_tree = -1;
static gint ett_hs20_osu_service_desc_list = -1;
static gint ett_hs20_osu_service_desc_tree = -1;
static gint ett_hs20_venue_url = -1;
static gint ett_hs20_advice_of_charge = -1;
static gint ett_hs20_aoc_plan = -1;

static gint ett_hs20_ofn_tree = -1;

static gint ett_adv_proto = -1;
static gint ett_adv_proto_tuple = -1;
static gint ett_gas_query = -1;
static gint ett_gas_anqp = -1;
static gint ett_nai_realm = -1;
static gint ett_nai_realm_eap = -1;
static gint ett_tag_ric_data_desc_ie = -1;
static gint ett_anqp_vendor_capab = -1;

static gint ett_osen_group_data_cipher_suite = -1;
static gint ett_osen_pairwise_cipher_suites = -1;
static gint ett_osen_pairwise_cipher_suite = -1;
static gint ett_osen_akm_cipher_suites = -1;
static gint ett_osen_akm_cipher_suite = -1;
static gint ett_osen_rsn_cap_tree = -1;
static gint ett_osen_pmkid_list = -1;
static gint ett_osen_pmkid_tree = -1;
static gint ett_osen_group_management_cipher_suite = -1;

static gint ett_hs20_cc_proto_port_tuple = -1;

static gint ett_ssid_list = -1;

static gint ett_nintendo = -1;

static gint ett_routerboard = -1;

static gint ett_meru = -1;

static gint ett_qos_map_set_exception = -1;
static gint ett_qos_map_set_range = -1;

static gint ett_wnm_notif_subelt = -1;

static gint ett_ieee80211_3gpp_plmn = -1;

static expert_field ei_ieee80211_bad_length = EI_INIT;
static expert_field ei_ieee80211_inv_val = EI_INIT;
static expert_field ei_ieee80211_vht_tpe_pwr_info_count = EI_INIT;
static expert_field ei_ieee80211_ff_query_response_length = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_nai_realm_eap_len = EI_INIT;
static expert_field ei_hs20_anqp_nai_hrq_length = EI_INIT;
static expert_field ei_ieee80211_extra_data = EI_INIT;
static expert_field ei_ieee80211_tag_data = EI_INIT;
static expert_field ei_ieee80211_tdls_setup_confirm_malformed = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_nai_field_len = EI_INIT;
static expert_field ei_ieee80211_rsn_pcs_count = EI_INIT;
static expert_field ei_ieee80211_tag_measure_request_unknown = EI_INIT;
static expert_field ei_ieee80211_tag_measure_request_beacon_unknown = EI_INIT;
static expert_field ei_ieee80211_tag_measure_report_unknown = EI_INIT;
static expert_field ei_ieee80211_tag_number = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_info_length = EI_INIT;
static expert_field ei_hs20_anqp_ofn_length = EI_INIT;
static expert_field ei_ieee80211_tdls_setup_response_malformed = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_capability = EI_INIT;
static expert_field ei_ieee80211_not_enough_room_for_anqp_header = EI_INIT;
static expert_field ei_ieee80211_ff_query_request_length = EI_INIT;
static expert_field ei_ieee80211_wfa_ie_wme_qos_info_bad_ftype = EI_INIT;
static expert_field ei_ieee80211_qos_info_bad_ftype = EI_INIT;
static expert_field ei_ieee80211_qos_bad_aifsn = EI_INIT;
static expert_field ei_ieee80211_pmkid_count_too_large = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_venue_length = EI_INIT;
static expert_field ei_ieee80211_ff_anqp_roaming_consortium_oi_len = EI_INIT;
static expert_field ei_ieee80211_tag_length = EI_INIT;
static expert_field ei_ieee80211_missing_data = EI_INIT;
static expert_field ei_ieee80211_rsn_pmkid_count = EI_INIT;
static expert_field ei_ieee80211_fc_retry = EI_INIT;
static expert_field ei_ieee80211_tag_wnm_sleep_mode_no_key_data = EI_INIT;
static expert_field ei_ieee80211_dmg_subtype = EI_INIT;
static expert_field ei_ieee80211_vht_action = EI_INIT;
static expert_field ei_ieee80211_mesh_peering_unexpected = EI_INIT;
static expert_field ei_ieee80211_fcs = EI_INIT;
static expert_field ei_ieee80211_mismatched_akm_suite = EI_INIT;
static expert_field ei_ieee80211_vs_routerboard_unexpected_len = EI_INIT;
static expert_field ei_ieee80211_twt_tear_down_bad_neg_type = EI_INIT;
static expert_field ei_ieee80211_twt_setup_not_supported_neg_type = EI_INIT;
static expert_field ei_ieee80211_twt_setup_bad_command = EI_INIT;
static expert_field ei_ieee80211_invalid_control_word = EI_INIT;

/* 802.11ad trees */
static gint ett_dynamic_alloc_tree = -1;
static gint ett_ssw_tree = -1;
static gint ett_bf_tree = -1;
static gint ett_sswf_tree = -1;
static gint ett_brp_tree = -1;
static gint ett_blm_tree = -1;
static gint ett_bic_tree = -1;
static gint ett_dmg_params_tree = -1;
static gint ett_cc_tree = -1;
static gint ett_rcsi_tree = -1;
static gint ett_80211_ext = -1;
static gint ett_allocation_tree = -1;
static gint ett_sta_info = -1;

static gint ett_ieee80211_esp = -1;

/* 802.11ah trees */
static gint ett_twt_tear_down_tree = -1;
static int ett_twt_control_field_tree = -1;
static int ett_twt_req_type_tree = -1;

/* 802.11ax trees */
static gint ett_he_mac_capabilities = -1;
static gint ett_he_phy_capabilities = -1;
static gint ett_he_phy_cap_first_byte = -1;
static gint ett_he_phy_cap_chan_width_set = -1;
static gint ett_he_phy_cap_b8_to_b23 = -1;
static gint ett_he_phy_cap_b24_to_b39 = -1;
static gint ett_he_phy_cap_b40_to_b55 = -1;
static gint ett_he_phy_cap_b56_to_b71 = -1;
static gint ett_he_phy_cap_b72_to_b87 = -1;
static gint ett_he_mcs_and_nss_set = -1;
static gint ett_he_rx_tx_he_mcs_map_lte_80 = -1;
static gint ett_he_rx_mcs_map_lte_80 = -1;
static gint ett_he_tx_mcs_map_lte_80 = -1;
static gint ett_he_rx_tx_he_mcs_map_160 = -1;
static gint ett_he_rx_mcs_map_160 = -1;
static gint ett_he_tx_mcs_map_160 = -1;
static gint ett_he_rx_tx_he_mcs_map_80_80 = -1;
static gint ett_he_rx_mcs_map_80_80 = -1;
static gint ett_he_tx_mcs_map_80_80 = -1;
static gint ett_he_ppe_threshold = -1;
static gint ett_he_ppe_nss = -1;
static gint ett_he_ppe_ru_alloc = -1;
static gint ett_he_uora_tree = -1;
static gint ett_he_aic_aifsn = -1;
static gint ett_he_spatial_reuse_control = -1;
static gint ett_he_bss_new_color_info = -1;
static gint ett_he_ess_report_info_field = -1;
static gint ett_he_operation_params = -1;
static gint ett_he_bss_color_information = -1;
static gint ett_he_oper_basic_mcs = -1;
static gint ett_he_operation_vht_op_info = -1;
static gint ett_he_mu_edca_param = -1;
static gint ett_he_trigger_common_info = -1;
static gint ett_he_trigger_base_common_info = -1;
static gint ett_he_trigger_bar_ctrl = -1;
static gint ett_he_trigger_bar_info = -1;
static gint ett_he_trigger_user_info = -1;
static gint ett_he_trigger_base_user_info = -1;
static gint ett_he_trigger_dep_basic_user_info = -1;
static gint ett_he_trigger_dep_nfrp_user_info = -1;
static gint ett_he_ndp_annc = -1;
static gint ett_he_ndp_annc_sta_list = -1;
static gint ett_he_ndp_annc_sta_item = -1;
static gint ett_he_ndp_annc_sta_info = -1;

/* 802.11ai trees */
static gint ett_fils_indication_realm_list = -1;
static gint ett_fils_indication_public_key_list = -1;

static const fragment_items frag_items = {
  &ett_fragment,
  &ett_fragments,
  &hf_ieee80211_fragments,
  &hf_ieee80211_fragment,
  &hf_ieee80211_fragment_overlap,
  &hf_ieee80211_fragment_overlap_conflict,
  &hf_ieee80211_fragment_multiple_tails,
  &hf_ieee80211_fragment_too_long_fragment,
  &hf_ieee80211_fragment_error,
  &hf_ieee80211_fragment_count,
  &hf_ieee80211_reassembled_in,
  &hf_ieee80211_reassembled_length,
  /* Reassembled data field */
  NULL,
  "fragments"
};

static const enum_val_t wlan_ignore_prot_options[] = {
  { "no",         "No",               WLAN_IGNORE_PROT_NO    },
  { "without_iv", "Yes - without IV", WLAN_IGNORE_PROT_WO_IV },
  { "with_iv",    "Yes - with IV",    WLAN_IGNORE_PROT_W_IV  },
  { NULL,         NULL,               0                     }
};

static int wlan_address_type = -1;
static int wlan_bssid_address_type = -1;

static const unsigned char bssid_broadcast_data[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
static address bssid_broadcast;
gboolean
is_broadcast_bssid(const address *bssid) {
  return addresses_equal(&bssid_broadcast, bssid);
}

static heur_dissector_list_t heur_subdissector_list;

static dissector_handle_t ieee80211_handle;
static dissector_handle_t wlan_withoutfcs_handle;
static dissector_handle_t llc_handle;
static dissector_handle_t ipx_handle;
static dissector_handle_t eth_withoutfcs_handle;

static capture_dissector_handle_t llc_cap_handle;
static capture_dissector_handle_t ipx_cap_handle;

static dissector_table_t tagged_field_table;
static dissector_table_t vendor_specific_action_table;
static dissector_table_t wifi_alliance_action_subtype_table;
static dissector_table_t vendor_specific_anqp_info_table;
static dissector_table_t wifi_alliance_anqp_info_table;
static dissector_table_t wifi_alliance_ie_table;
static dissector_table_t wifi_alliance_public_action_table;

static int wlan_tap = -1;

static const value_string access_network_type_vals[] = {
  {  0, "Private network" },
  {  1, "Private network with guest access" },
  {  2, "Chargeable public network" },
  {  3, "Free public network" },
  {  4, "Personal device network" },
  {  5, "Emergency services only network" },
  { 14, "Test or experimental" },
  { 15, "Wildcard" },
  { 0, NULL }
};

static const value_string adv_proto_id_vals[] = {
  {  0, "Access Network Query Protocol"},
  {  1, "MIH Information Service"},
  {  2, "MIH Command and Event Services Capability Discovery"},
  {  3, "Emergency Alert System (EAS)"},
  {  4, "Location-to-Service Translation Protocol"},
  {221, "Vendor Specific"},
  {0, NULL}
};

static const value_string timeout_int_types[] = {
  {1, "Reassociation deadline interval (TUs)"},
  {2, "Key lifetime interval (seconds)"},
  {3, "Association Comeback time (TUs)"},
  {4, "Time to start (TUs)"},
  {0, NULL}
};

static const value_string tdls_action_codes[] = {
  {TDLS_SETUP_REQUEST,           "TDLS Setup Request"},
  {TDLS_SETUP_RESPONSE,          "TDLS Setup Response"},
  {TDLS_SETUP_CONFIRM,           "TDLS Setup Confirm"},
  {TDLS_TEARDOWN,                "TDLS Teardown"},
  {TDLS_PEER_TRAFFIC_INDICATION, "TDLS Peer Traffic Indication"},
  {TDLS_CHANNEL_SWITCH_REQUEST,  "TDLS Channel Switch Request"},
  {TDLS_CHANNEL_SWITCH_RESPONSE, "TDLS Channel Switch Response"},
  {TDLS_PEER_PSM_REQUEST,        "TDLS Peer PSM Request"},
  {TDLS_PEER_PSM_RESPONSE,       "TDLS Peer PSM Response"},
  {TDLS_PEER_TRAFFIC_RESPONSE,   "TDLS Peer Traffic Response"},
  {TDLS_DISCOVERY_REQUEST,       "TDLS Discovery Request"},
  {0, NULL}
};
static value_string_ext tdls_action_codes_ext = VALUE_STRING_EXT_INIT(tdls_action_codes);

static const value_string rm_action_codes[] = {
  {RM_ACTION_RADIO_MEASUREMENT_REQUEST,   "Radio Measurement Request"},
  {RM_ACTION_RADIO_MEASUREMENT_REPORT,    "Radio Measurement Report"},
  {RM_ACTION_LINK_MEASUREMENT_REQUEST,    "Link Measurement Request"},
  {RM_ACTION_LINK_MEASUREMENT_REPORT,     "Link Measurement Report"},
  {RM_ACTION_NEIGHBOR_REPORT_REQUEST,     "Neighbor Report Request"},
  {RM_ACTION_NEIGHBOR_REPORT_RESPONSE,    "Neighbor Report Response"},
  {0, NULL}
};
static value_string_ext rm_action_codes_ext = VALUE_STRING_EXT_INIT(rm_action_codes);

static const val64_string number_of_taps_values[] = {
  {0x0, "1 tap"},
  {0x1, "5 taps"},
  {0x2, "15 taps"},
  {0x3, "63 taps"},
  {0, NULL}
};

DOT11DECRYPT_CONTEXT dot11decrypt_ctx;

#define PSMP_STA_INFO_BROADCAST 0
#define PSMP_STA_INFO_MULTICAST 1
#define PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED 2

#define PSMP_STA_INFO_FLAG_TYPE         0x00000003
#define PSMP_STA_INFO_FLAG_DTT_START    0x00001FFC
#define PSMP_STA_INFO_FLAG_DTT_DURATION 0x001FE000

#define PSMP_STA_INFO_FLAG_STA_ID       0x001FFFE0

#define PSMP_STA_INFO_FLAG_UTT_START    0x0000FFE0
#define PSMP_STA_INFO_FLAG_UTT_DURATION 0x03FF0000

#define PSMP_STA_INFO_FLAG_IA_RESERVED  0xFC000000

static const value_string ff_psmp_sta_info_flags[] = {
  { PSMP_STA_INFO_BROADCAST,              "Broadcast"},
  { PSMP_STA_INFO_MULTICAST,              "Multicast"},
  { PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED, "Individually Addressed"},
  {0, NULL}
};

static const char*
wlan_conv_get_filter_type(conv_item_t* conv, conv_filter_type_e filter)
{
    if ((filter == CONV_FT_SRC_ADDRESS) && (conv->src_address.type == wlan_address_type))
        return "wlan.sa";

    if ((filter == CONV_FT_DST_ADDRESS) && (conv->dst_address.type == wlan_address_type))
        return "wlan.da";

    if ((filter == CONV_FT_ANY_ADDRESS) && (conv->src_address.type == wlan_address_type))
        return "wlan.addr";

    return CONV_FILTER_INVALID;
}

static ct_dissector_info_t wlan_ct_dissector_info = {&wlan_conv_get_filter_type};

static tap_packet_status
wlan_conversation_packet(void *pct, packet_info *pinfo, epan_dissect_t *edt _U_, const void *vip)
{
  conv_hash_t *hash = (conv_hash_t*) pct;
  const wlan_hdr_t *whdr=(const wlan_hdr_t *)vip;

  add_conversation_table_data(hash, &whdr->src, &whdr->dst, 0, 0, 1, pinfo->fd->pkt_len, &pinfo->rel_ts, &pinfo->abs_ts, &wlan_ct_dissector_info, ENDPOINT_NONE);

  return TAP_PACKET_REDRAW;
}

static const char*
wlan_host_get_filter_type(hostlist_talker_t* host, conv_filter_type_e filter)
{
  if ((filter == CONV_FT_ANY_ADDRESS) && (host->myaddress.type == wlan_address_type))
    return "wlan.addr";

  return CONV_FILTER_INVALID;
}

static hostlist_dissector_info_t wlan_host_dissector_info = {&wlan_host_get_filter_type};

static tap_packet_status
wlan_hostlist_packet(void *pit, packet_info *pinfo, epan_dissect_t *edt _U_, const void *vip)
{
  conv_hash_t *hash = (conv_hash_t*) pit;
  const wlan_hdr_t *whdr=(const wlan_hdr_t *)vip;

  /* Take two "add" passes per packet, adding for each direction, ensures that all
  packets are counted properly (even if address is sending to itself)
  XXX - this could probably be done more efficiently inside hostlist_table */
  add_hostlist_table_data(hash, &whdr->src, 0, TRUE, 1, pinfo->fd->pkt_len, &wlan_host_dissector_info, ENDPOINT_NONE);
  add_hostlist_table_data(hash, &whdr->dst, 0, FALSE, 1, pinfo->fd->pkt_len, &wlan_host_dissector_info, ENDPOINT_NONE);

  return TAP_PACKET_REDRAW;
}

static const char*
wlan_col_filter_str(const address* addr _U_, gboolean is_src)
{
  if (is_src)
    return "wlan.sa";

  return "wlan.da";
}

static const char*
wlan_bssid_col_filter_str(const address* addr _U_, gboolean is_src _U_)
{
  return "wlan.bssid";
}


static void
beacon_interval_base_custom(gchar *result, guint32 beacon_interval)
{
  double temp_double;

  temp_double = (double)beacon_interval;
  g_snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (temp_double * 1024 / 1000000));
}

static void
allocation_duration_base_custom(gchar *result, guint32 allocation_duration)
{
  double temp_double;

  temp_double = (double)allocation_duration;
  g_snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (temp_double / 1000000));
}

static void
extra_one_base_custom(gchar *result, guint32 value)
{
  g_snprintf(result, ITEM_LABEL_LENGTH, "%d", value+1);
}

static void
extra_one_mul_two_base_custom(gchar *result, guint32 value)
{
  g_snprintf(result, ITEM_LABEL_LENGTH, "%d", (value+1)*2);
}

static void
rcpi_and_power_level_custom(gchar *result, guint8 value)
{
  /* 802.11-2016 section 9.4.2.38
     RCPI = |2 x (P + 110)| in steps of 0.5 dB */

  if (value == 0)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (P < -109.5 dBm)", value);
  else if (value == 220)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (P >= 0 dBm)", value);
  else if (value < 220)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (P = %.1f dBm)", value, ((double)value) / 2 - 110);
  else if (value < 255)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (Reserved)", value);
  else
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (Measurement not available)", value);
}

/*
 * We use this is displaying the ru allocation region.
 */
static guint8 global_he_trigger_bw = 0;

static void
he_ru_allocation_base_custom(gchar *result, guint32 ru_allocation)
{
  guint32 tones = 0;

  switch (global_he_trigger_bw) {
  case 0:
    if (ru_allocation <= 8) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 40) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 54) {
      tones = 106;
      break;
    }
    if (ru_allocation == 61) {
      tones = 242;
      break;
    }
    // error
    break;
  case 1:
    if (ru_allocation <= 17) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 44) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 56) {
      tones = 106;
      break;
    }
    if (ru_allocation >= 61 && ru_allocation <= 62) {
      tones = 242;
      break;
    }
    if (ru_allocation == 65) {
      tones = 484;
      break;
    }
    // error
    break;
  case 2:
    /* fall-through */
  case 3:
    if (ru_allocation <= 16) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 52) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 60) {
      tones = 106;
      break;
    }
    if (ru_allocation >= 61 && ru_allocation <= 64) {
      tones = 242;
      break;
    }
    if (ru_allocation >= 65 && ru_allocation <= 66) {
      tones = 484;
      break;
    }
    if (ru_allocation == 67) {
      tones = 996;
      break;
    }
    if (ru_allocation == 68 && global_he_trigger_bw == 3) {
      tones = 2*996;
      break;
    }
    break;
  default:
    break;
  }

  if (tones)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (%d tones)", ru_allocation, tones);
  else
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d (bogus number of tones)", ru_allocation);
}

/* ************************************************************************* */
/* Mesh Control field helper functions
 *
 * Per IEEE 802.11s Draft 12.0 section 7.2.2.1:
 *
 * The frame body consists of either:
 * The MSDU (or a fragment thereof), the Mesh Control field (if and only if the
 * frame is transmitted by a mesh STA and the Mesh Control Present subfield of
 * the QoS Control field is 1)...
 *
 * 8.2.4.5.1 "QoS Control field structure", table 8-4, in 802.11-2012,
 * seems to indicate that the bit that means "Mesh Control Present" in
 * frames sent by mesh STAs in a mesh BSS is part of the TXOP Limit field,
 * the AP PS Buffer State field, the TXOP Duration Requested field, or the
 * Queue Size field in some data frames in non-mesh BSSes.
 *
 * We need a statefull sniffer for that.  For now, use heuristics.
 *
 * Notably, only mesh data frames contain the Mesh Control field in the header.
 * Other frames that contain mesh control (i.e., multihop action frames) have
 * it deeper in the frame body where it can be definitively identified.
 * Further, mesh data frames always have to-ds and from-ds either 11 or 01.  We
 * use these facts to make our heuristics more reliable.
 * ************************************************************************* */
static int
has_mesh_control(guint16 fcf, guint16 qos_ctl, guint8 mesh_flags)
{
  /* assume mesh control present if the QOS field's Mesh Control Present bit is
   * set, all reserved bits in the mesh_flags field are zero, and the address
   * extension mode is not a reserved value.
   */
  return (((FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) || (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T2)) &&
          (QOS_MESH_CONTROL_PRESENT(qos_ctl)) &&
          ((mesh_flags & ~MESH_FLAGS_ADDRESS_EXTENSION) == 0) &&
          ((mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION) != MESH_FLAGS_ADDRESS_EXTENSION));
}

/****************************************************************************** */
/*
 * locally originated mesh frames will have a mesh control field, but no QoS header
 * detect the presence of mesh control field by checking if mesh flags are legal
 * and confirming that the next header is an 802.2 LLC header
 *
 ****************************************************************************** */
static int
has_mesh_control_local(guint16 fcf, guint8 mesh_flags, guint16 next_header)
{
  return (((FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) || (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T2)) &&
          ((mesh_flags & ~MESH_FLAGS_ADDRESS_EXTENSION) == 0) &&
          ((mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION) != MESH_FLAGS_ADDRESS_EXTENSION)) &&
          next_header == 0xaaaa;
}

static int
find_mesh_control_length(guint8 mesh_flags)
{
  return 6 + 6*(mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION);
}

static mimo_control_t
get_mimo_control(tvbuff_t *tvb, int offset)
{
  guint16        mimo;
  mimo_control_t output;

  mimo = tvb_get_letohs(tvb, offset);

  output.nc = (mimo & 0x0003) + 1;
  output.nr = ((mimo & 0x000C) >> 2) + 1;
  output.chan_width = (mimo & 0x0010) >> 4;
  output.coefficient_size = 4; /* XXX - Is this a good default? */

  switch ((mimo & 0x0060) >> 5)
    {
      case 0:
        output.grouping = 1;
        break;

      case 1:
        output.grouping = 2;
        break;

      case 2:
        output.grouping = 4;
        break;

      default:
        output.grouping = 1;
        break;
    }

  switch ((mimo & 0x0180) >> 7)
    {
      case 0:
        output.coefficient_size = 4;
        break;

      case 1:
        output.coefficient_size = 5;
        break;

      case 2:
        output.coefficient_size = 6;
        break;

      case 3:
        output.coefficient_size = 8;
        break;
    }

  output.codebook_info = (mimo & 0x0600) >> 9;
  output.remaining_matrix_segment = (mimo & 0x3800) >> 11;

  return output;
}

static int
get_mimo_na(guint8 nr, guint8 nc)
{
  if ((nr == 2) && (nc == 1)) {
    return 2;
  } else if ((nr == 2) && (nc == 2)) {
    return 2;
  } else if ((nr == 3) && (nc == 1)) {
    return 4;
  } else if ((nr == 3) && (nc == 2)) {
    return 6;
  } else if ((nr == 3) && (nc == 3)) {
    return 6;
  } else if ((nr == 4) && (nc == 1)) {
    return 6;
  } else if ((nr == 4) && (nc == 2)) {
    return 10;
  } else if ((nr == 4) && (nc == 3)) {
    return 12;
  } else if ((nr == 4) && (nc == 4)) {
    return 12;
  } else{
    return 0;
  }
}

static int
get_mimo_ns(gboolean chan_width, guint8 output_grouping)
{
  int ns = 0;

  if (chan_width)
  {
    switch (output_grouping)
      {
        case 1:
          ns = 114;
          break;

          case 2:
            ns = 58;
            break;

          case 4:
            ns = 30;
            break;

          default:
            ns = 0;
      }
  } else {
    switch (output_grouping)
      {
        case 1:
          ns = 56;
          break;

        case 2:
          ns = 30;
          break;

        case 4:
          ns = 16;
          break;

        default:
          ns = 0;
      }
  }

  return ns;
}

static int
add_mimo_csi_matrices_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc,
                        ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nr; i++)
  {
    guint8 snr;

    snr = tvb_get_guint8(tvb, offset);
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Channel %d - Signal to Noise Ratio: 0x%02X", i, snr);
    offset += 1;
  }

  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(3+(2*mimo_cntrl.nc*mimo_cntrl.nr*mimo_cntrl.coefficient_size));
  csi_matrix_size = roundup2(csi_matrix_size, 8) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static int
add_mimo_beamforming_feedback_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc, ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nc; i++)
  {
    guint8 snr;

    snr = tvb_get_guint8(tvb, offset);
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: 0x%02X", i, snr);
    offset += 1;
  }

  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(2*mimo_cntrl.nc*mimo_cntrl.nr*mimo_cntrl.coefficient_size);
  csi_matrix_size = roundup2(csi_matrix_size, 8) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_bf_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static int
add_mimo_compressed_beamforming_feedback_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, na, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc,
                        ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nc; i++)
  {
    gint8 snr;
    char edge_sign;

    snr = tvb_get_gint8(tvb, offset);

    switch(snr) {
      case -128:
        edge_sign = '<';
        break;
      case 127:
        edge_sign = '>';
        break;
      default:
        edge_sign = ' ';
        break;
    }
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: %c%3.2fdB", i, edge_sign,snr/4.0+22.0);
    offset += 1;
  }

  na = get_mimo_na(mimo_cntrl.nr, mimo_cntrl.nc);
  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(na*((mimo_cntrl.codebook_info+1)*2 + 2)/2);
  csi_matrix_size = roundup2(csi_matrix_size, 8) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_cbf_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static void
mesh_active_window_base_custom(gchar *result, guint32 mesh_active_window)
{
  g_snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (mesh_active_window * 1024.0 / 1000000));
}

/* ************************************************************************* */
/*          This is the capture function used to update packet counts        */
/* ************************************************************************* */
static gboolean
capture_ieee80211_common(const guchar * pd, int offset, int len,
                          capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_, gboolean datapad)
{
  guint16 fcf, hdr_length;

  if (!BYTES_ARE_IN_FRAME(offset, len, 2))
    return FALSE;

  fcf = pletoh16(&pd[offset]);

  if (IS_PROTECTED(FCF_FLAGS(fcf)) && (wlan_ignore_prot == WLAN_IGNORE_PROT_NO))
    return FALSE;

  switch (COMPOSE_FRAME_TYPE (fcf)) {

    case DATA:
    case DATA_CF_ACK:
    case DATA_CF_POLL:
    case DATA_CF_ACK_POLL:
    case DATA_QOS_DATA:
    case DATA_QOS_DATA_CF_ACK:
    case DATA_QOS_DATA_CF_POLL:
    case DATA_QOS_DATA_CF_ACK_POLL:
    {
      /* These are data frames that actually contain *data*. */
      hdr_length = (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) ? DATA_LONG_HDR_LEN : DATA_SHORT_HDR_LEN;

      if (DATA_FRAME_IS_QOS(COMPOSE_FRAME_TYPE(fcf))) {
        /* QoS frame, so the header includes a QoS field */
        guint16 qosoff;  /* Offset of the 2-byte QoS field */
        guint8 mesh_flags;

        qosoff = hdr_length;
        hdr_length += 2; /* Include the QoS field in the header length */

        if (HAS_HT_CONTROL(FCF_FLAGS(fcf))) {
          /* Frame has a 4-byte HT Control field */
          hdr_length += 4;
        }

        /*
         * Does it look as if we have a mesh header?
         * Look at the Mesh Control subfield of the QoS field and at the
         * purported mesh flag fields.
         */
        if (!BYTES_ARE_IN_FRAME(offset, hdr_length, 1))
          return FALSE;

        mesh_flags = pd[hdr_length];
        if (has_mesh_control(fcf, pletoh16(&pd[qosoff]), mesh_flags)) {
          /* Yes, add the length of that in as well. */
          hdr_length += find_mesh_control_length(mesh_flags);
        }

        if (datapad) {
          /*
           * Include the padding between the 802.11 header and the body,
           * as "helpfully" provided by some Atheros adapters.
           *
           * XXX - would the mesh header be part of the header or the body
           * from the point of view of the Atheros adapters that insert
           * the padding, assuming they even recognize a mesh header?
           */
          hdr_length = roundup2(hdr_length, 4);
        }
      }
      /* I guess some bridges take Netware Ethernet_802_3 frames,
         which are 802.3 frames (with a length field rather than
         a type field, but with no 802.2 header in the payload),
         and just stick the payload into an 802.11 frame.  I've seen
         captures that show frames of that sort.

         We also handle some odd form of encapsulation in which a
         complete Ethernet frame is encapsulated within an 802.11
         data frame, with no 802.2 header.  This has been seen
         from some hardware.

         On top of that, at least at some point it appeared that
         the OLPC XO sent out frames with two bytes of 0 between
         the "end" of the 802.11 header and the beginning of
         the payload.

         So, if the packet doesn't start with 0xaa 0xaa:

           we first use the same scheme that linux-wlan-ng does to detect
           those encapsulated Ethernet frames, namely looking to see whether
           the frame either starts with 6 octets that match the destination
           address from the 802.11 header or has 6 octets that match the
           source address from the 802.11 header following the first 6 octets,
           and, if so, treat it as an encapsulated Ethernet frame;

           otherwise, we use the same scheme that we use in the Ethernet
           dissector to recognize Netware 802.3 frames, namely checking
           whether the packet starts with 0xff 0xff and, if so, treat it
           as an encapsulated IPX frame, and then check whether the
           packet starts with 0x00 0x00 and, if so, treat it as an OLPC
           frame. */
      if (!BYTES_ARE_IN_FRAME(offset+hdr_length, len, 2))
        return FALSE;

      if ((pd[offset+hdr_length] != 0xaa) && (pd[offset+hdr_length+1] != 0xaa)) {
#if 0
        /* XXX - this requires us to parse the header to find the source
           and destination addresses. */
        if (BYTES_ARE_IN_FRAME(offset+hdr_length, len, 12)) {
          /* We have two MAC addresses after the header. */
          if ((memcmp(&pd[offset+hdr_length+6], pinfo->dl_src.data, 6) == 0) ||
              (memcmp(&pd[offset+hdr_length+6], pinfo->dl_dst.data, 6) == 0)) {
            return capture_eth (pd, offset + hdr_length, len, cpinfo, pseudo_header);
          }
        }
#endif
        if ((pd[offset+hdr_length] == 0xff) && (pd[offset+hdr_length+1] == 0xff))
          return call_capture_dissector (ipx_cap_handle, pd, offset+hdr_length, len, cpinfo, pseudo_header);
        else if ((pd[offset+hdr_length] == 0x00) && (pd[offset+hdr_length+1] == 0x00))
          return call_capture_dissector (llc_cap_handle, pd, offset + hdr_length + 2, len, cpinfo, pseudo_header);
      }
      else {
        return call_capture_dissector (llc_cap_handle, pd, offset + hdr_length, len, cpinfo, pseudo_header);
      }
      break;
    }
  }

  return FALSE;
}

/*
 * Handle 802.11 with a variable-length link-layer header.
 */
static gboolean
capture_ieee80211(const guchar * pd, int offset, int len, capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_)
{
  return capture_ieee80211_common (pd, offset, len, cpinfo, pseudo_header, FALSE);
}

/*
 * Handle 802.11 with a variable-length link-layer header and data padding.
 */
static gboolean
capture_ieee80211_datapad(const guchar * pd, int offset, int len,
                           capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_)
{
  return capture_ieee80211_common (pd, offset, len, cpinfo, pseudo_header, TRUE);
}


/* ************************************************************************* */
/*          Add the subtree used to store the fixed parameters               */
/* ************************************************************************* */
static proto_tree *
get_fixed_parameter_tree(proto_tree * tree, tvbuff_t *tvb, int start, int size)
{
  proto_item *fixed_fields;

  fixed_fields = proto_tree_add_item(tree, hf_ieee80211_fixed_parameters, tvb, start, size, ENC_NA);
  proto_item_append_text(fixed_fields, " (%d bytes)", size);

  return proto_item_add_subtree(fixed_fields, ett_fixed_parameters);
}


/* ************************************************************************* */
/*            Add the subtree used to store tagged parameters                */
/* ************************************************************************* */
static proto_tree *
get_tagged_parameter_tree(proto_tree * tree, tvbuff_t *tvb, int start, int size)
{
  proto_item *tagged_fields;

  tagged_fields = proto_tree_add_item(tree, hf_ieee80211_tagged_parameters, tvb, start, -1, ENC_NA);
  proto_item_append_text(tagged_fields, " (%d bytes)", size);

  return proto_item_add_subtree(tagged_fields, ett_tagged_parameters);
}


static int
dissect_vendor_action_marvell(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void *data _U_)
{
  guint8 octet;
  int offset = 0;

  octet = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_marvell_action_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  switch (octet)
    {
      case MRVL_ACTION_MESH_MANAGEMENT:
        octet = tvb_get_guint8(tvb, offset);
        proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        switch (octet)
          {
            case MRVL_MESH_MGMT_ACTION_RREQ:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_rreqid, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_sa, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ssn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dstcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            case MRVL_MESH_MGMT_ACTION_RREP:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_sa, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ssn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            case MRVL_MESH_MGMT_ACTION_RERR:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dstcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            default:
              break;
          }
        break;
      default:
        break;
    }

  return offset;
}

static int
dissect_vendor_action_wifi_alliance(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
  guint8 subtype;
  int offset = 0;
  int dissected;
  tvbuff_t *subtvb;

  subtype = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_oui_wfa_subtype, tvb, offset, 1, ENC_NA);
  offset += 1;

  subtvb = tvb_new_subset_remaining(tvb, offset);
  dissected = dissector_try_uint_new(wifi_alliance_action_subtype_table, subtype, subtvb, pinfo, tree, FALSE, NULL);
  if (dissected <= 0)
  {
      call_data_dissector(subtvb, pinfo, tree);
      dissected = tvb_reported_length(subtvb);
  }

  offset += dissected;

  return offset;
}

/*
 * This function is called from two different places. In one case it is called
 * without the tag and length. In other cases, it is called with those and
 * is asked to return the type and subtype. We know the difference because
 * type and subtype will be NULL in the first case.
 */
static guint
dissect_advertisement_protocol_common(packet_info *pinfo, proto_tree *tree,
                               tvbuff_t *tvb, int offset, guint *type,
                               guint *subtype)
{
  guint8      tag_no = 0, tag_len, left;
  proto_item *item = NULL, *adv_item;
  proto_tree *adv_tree, *adv_tuple_tree;

  if (type)
    *type = 0xff; // Last reserved value
  if (subtype)
    *subtype = 0xff;
  tag_no = tvb_get_guint8(tvb, offset);
  if (type)
    item = proto_tree_add_item(tree, hf_ieee80211_tag_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);

  /*
   * If we have the tag and len, use the len in the tvb, otherwise ask
   * for the length of the tvb.
   */
  if (type)
    tag_len = tvb_get_guint8(tvb, offset + 1);
  else
    tag_len = tvb_reported_length_remaining(tvb, 0);

  if (type && tag_no != TAG_ADVERTISEMENT_PROTOCOL) {
    expert_add_info_format(pinfo, item, &ei_ieee80211_tag_number,
                           "Unexpected IE %d (expected Advertisement "
                           "Protocol)", tag_no);
    return 2 + tag_len;
  }
  if (type)
    item = proto_tree_add_uint(tree, hf_ieee80211_tag_length, tvb, offset + 1, 1, tag_len);
  if (tag_len < 2) {
    if (!type)
      item = proto_tree_add_uint(tree, hf_ieee80211_tag_length, tvb, offset + 1, 1, tag_len);
    expert_add_info_format(pinfo, item, &ei_ieee80211_tag_length,
                           "Advertisement Protocol: IE must be at least 2 "
                           "octets long");
    return 2 + tag_len;
  }

  left = tag_len;
  if (type) /* Skip past the header if there ... */
    offset += 2;
  adv_tree = proto_tree_add_subtree(tree, tvb, offset, left,
                                 ett_adv_proto, &adv_item, "Advertisement Protocol element");

  while (left >= 2) {
    guint8 id;

    id = tvb_get_guint8(tvb, offset + 1);
    if (id == 0)
      proto_item_append_text(adv_item, ": ANQP");
    adv_tuple_tree = proto_tree_add_subtree_format(adv_tree, tvb, offset, 2, ett_adv_proto_tuple, &item,
                               "Advertisement Protocol Tuple: %s",
                               val_to_str(id, adv_proto_id_vals,
                                          "Unknown (%d)"));

    proto_tree_add_item(adv_tuple_tree,
                        hf_ieee80211_tag_adv_proto_resp_len_limit, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(adv_tuple_tree,
                        hf_ieee80211_tag_adv_proto_pame_bi, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    left--;
    proto_tree_add_item(adv_tuple_tree, hf_ieee80211_tag_adv_proto_id, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    left--;

    if ((id == 0) && type)
      *type = ADV_PROTO_ID_ANQP;

    if (id == 221) {
      /* Vendor specific */
      guint8 len = tvb_get_guint8(tvb, offset);
      guint oui;
      guint8 wfa_subtype;
      proto_tree_add_item(adv_tuple_tree, hf_ieee80211_tag_adv_vs_len, tvb, offset, 1, ENC_NA);
      offset += 1;
      left   -= 1;
      if (type)
        *type = ADV_PROTO_ID_VS;
      if (len > left) {
        expert_add_info_format(pinfo, item, &ei_ieee80211_tag_length,
                               "Vendor specific info length error");
        return 2 + tag_len;
      }
      proto_tree_add_item_ret_uint(adv_tuple_tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
      offset += 3;
      left   -= 3;
      wfa_subtype = tvb_get_guint8(tvb, offset);
      proto_tree_add_item(adv_tuple_tree, hf_ieee80211_anqp_wfa_subtype, tvb,
                        offset, 1, ENC_NA);
      offset += 1;
      left   -= 1;
      if (oui == OUI_WFA) {
        proto_tree_add_item(adv_tuple_tree, hf_ieee80211_dpp_subtype, tvb, offset, 1, ENC_NA);
        if (subtype && wfa_subtype == WFA_SUBTYPE_DPP) {
          *subtype = WFA_SUBTYPE_DPP;
          *subtype |= (tvb_get_guint8(tvb, offset) << 8);
        }
        offset++;
        left--;
      }
    }
  }

  if (left) {
    expert_add_info_format(pinfo, item, &ei_ieee80211_extra_data,
                           "Unexpected extra data in the end");
  }

  return 2 + tag_len;
}

static int
dissect_advertisement_protocol(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  return dissect_advertisement_protocol_common(pinfo, tree, tvb, 0, NULL, NULL);
}

static void
dissect_anqp_query_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  while (offset + 2 <= end) {
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_query_id,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if (offset != end) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_ff_anqp_info_length,
                           "Unexpected ANQP Query list format");
  }
}

static void
dissect_hs20_anqp_hs_capability_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_hs20_anqp_hs_capability_list,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_anqp_capab_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  guint16     id, len;
  proto_item *item;
  proto_tree *vtree;
  guint32     oui;
  guint8      subtype;

  while (offset + 2 <= end) {
    id = tvb_get_letohs(tvb, offset);
    item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_capability,
                               tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (id == ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST) {
      vtree = proto_item_add_subtree(item, ett_anqp_vendor_capab);
      len = tvb_get_letohs(tvb, offset);
      proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vlen,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if ((len < 3) || ((offset + len) > end)) {
        expert_add_info(pinfo, tree, &ei_ieee80211_ff_anqp_capability);
        return;
      }
      proto_tree_add_item_ret_uint(vtree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
      offset += 3;
      len    -= 3;

      switch (oui) {
      case OUI_WFA:
        if (len == 0)
          break;
        subtype = tvb_get_guint8(tvb, offset);
        proto_item_append_text(vtree, " - WFA - %s",
                               val_to_str(subtype, wfa_subtype_vals,
                                          "Unknown (%u)"));
        proto_tree_add_item(vtree, hf_ieee80211_anqp_wfa_subtype,
                            tvb, offset, 1, ENC_NA);
        offset++;
        len--;
        switch (subtype) {
        case WFA_SUBTYPE_HS20_ANQP:
          dissect_hs20_anqp_hs_capability_list(vtree, tvb, offset, end);
          break;
        default:
          proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vendor,
                              tvb, offset, len, ENC_NA);
          break;
        }
        break;
      default:
        proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vendor,
                            tvb, offset, len, ENC_NA);
        break;
      }

      offset += len;
    }
  }
  if (offset != end) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_ff_anqp_info_length,
                           "Unexpected ANQP Capability list format");
  }
}

static const value_string venue_group_vals[] = {
  {  0, "Unspecified" },
  {  1, "Assembly" },
  {  2, "Business" },
  {  3, "Educational" },
  {  4, "Factory and Industrial" },
  {  5, "Institutional" },
  {  6, "Mercantile" },
  {  7, "Residential" },
  {  8, "Storage" },
  {  9, "Utility and Miscellaneous" },
  { 10, "Vehicular" },
  { 11, "Outdoor" },
  { 0, NULL }
};
static value_string_ext venue_group_vals_ext = VALUE_STRING_EXT_INIT(venue_group_vals);

static void
dissect_venue_info(proto_tree *tree, tvbuff_t *tvb, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_venue_info_group,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_venue_info_type,
                      tvb, offset + 1, 1, ENC_LITTLE_ENDIAN);
}

static void
dissect_venue_name_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  proto_item *item;

  dissect_venue_info(tree, tvb, offset);
  offset += 2;
  while (offset + 4 <= end) {
    guint8 vlen = tvb_get_guint8(tvb, offset);
    item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_length,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if ((vlen > (end - offset)) || (vlen < 3)) {
      expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_venue_length);
      break;
    }
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_language,
                        tvb, offset, 3, ENC_ASCII|ENC_NA);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_name,
                        tvb, offset + 3, vlen - 3, ENC_UTF_8|ENC_NA);
    offset += vlen;
  }
}

static const value_string nw_auth_type_vals[] = {
  { 0, "Acceptance of terms and conditions" },
  { 1, "On-line enrollment supported" },
  { 2, "http/https redirection" },
  { 3, "DNS redirection" },
  { 0, NULL }
};

static void
dissect_network_auth_type(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset + 3 <= end) {
    guint16 len;
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_indicator,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    len = tvb_get_letohs(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_url_len,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (len)
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_url,
                          tvb, offset, len, ENC_ASCII|ENC_NA);
    offset += len;
  }
}

static void
add_manuf(proto_item *item, tvbuff_t *tvb, int offset)
{
  const gchar *manuf_name;

  manuf_name = tvb_get_manuf_name_if_known(tvb, offset);
  if (manuf_name == NULL)
    return;
  proto_item_append_text(item, " - %s", manuf_name);
}

static void
dissect_roaming_consortium_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                                int end)
{
  proto_item *item;
  guint8      len;

  while (offset < end) {
    len = tvb_get_guint8(tvb, offset);
    item = proto_tree_add_item(tree,
                               hf_ieee80211_ff_anqp_roaming_consortium_oi_len,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if ((len > (end - offset)) || (len < 3)) {
      expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_roaming_consortium_oi_len);
      break;
    }
    item = proto_tree_add_item(tree,
                               hf_ieee80211_ff_anqp_roaming_consortium_oi,
                               tvb, offset, len, ENC_NA);
    add_manuf(item, tvb, offset);
    offset += len;
  }
}

static const value_string ip_addr_avail_ipv6_vals[] = {
  { 0, "Address type not available" },
  { 1, "Address type available" },
  { 2, "Availability of the address type not known" },
  { 0, NULL }
};

static const value_string ip_addr_avail_ipv4_vals[] = {
  { 0, "Address type not available" },
  { 1, "Public IPv4 address available" },
  { 2, "Port-restricted IPv4 address available" },
  { 3, "Single NATed private IPv4 address available" },
  { 4, "Double NATed private IPv4 address available" },
  { 5, "Port-restricted IPv4 address and single NATed IPv4 address available" },
  { 6, "Port-restricted IPv4 address and double NATed IPv4 address available" },
  { 7, "Availability of the address type is not known" },
  { 0, NULL }
};

static void
dissect_ip_addr_type_availability_info(proto_tree *tree, tvbuff_t *tvb,
                                       int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_ip_addr_avail_ipv6,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_ip_addr_avail_ipv4,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
}

static const value_string nai_realm_encoding_vals[] = {
  { 0, "Formatted in accordance with RFC 4282" },
  { 1, "UTF-8 formatted that is not formatted in accordance with RFC 4282" },
  { 0, NULL }
};

static const value_string nai_realm_auth_param_id_vals[] = {
  {   1, "Expanded EAP Method" },
  {   2, "Non-EAP Inner Authentication Type" },
  {   3, "Inner Authentication EAP Method Type" },
  {   4, "Expanded Inner EAP Method" },
  {   5, "Credential Type" },
  {   6, "Tunneled EAP Method Credential Type" },
  { 221, "Vendor Specific" },
  { 0, NULL }
};

static void
dissect_nai_realm_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  guint16       count, len;
  proto_item   *item, *r_item;
  int           f_end, eap_end;
  guint8        nai_len, eap_count, eap_len, auth_param_count, auth_param_len;
  guint8        auth_param_id;
  proto_tree   *realm_tree, *eap_tree;
  const guint8 *realm;

  count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nai_realm_count,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  while (count > 0) {
    len = tvb_get_letohs(tvb, offset);
    realm_tree = proto_tree_add_subtree(tree, tvb, offset, 2 + len, ett_nai_realm, &r_item, "NAI Realm Data");

    item = proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_field_len,
                               tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (offset + len > end) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_ff_anqp_nai_field_len,
                             "Invalid NAI Realm List");
      break;
    }
    f_end = offset + len;
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_encoding,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    nai_len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_length,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if (offset + nai_len > f_end) {
      expert_add_info_format(pinfo, r_item, &ei_ieee80211_ff_anqp_nai_field_len,
                             "Invalid NAI Realm Data");
      break;
    }
    proto_tree_add_item_ret_string(realm_tree, hf_ieee80211_ff_anqp_nai_realm,
                        tvb, offset, nai_len, ENC_ASCII|ENC_NA, wmem_packet_scope(), &realm);
    if (realm) {
      proto_item_append_text(r_item, " (%s)", realm);
    }
    offset += nai_len;
    eap_count = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_eap_count,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    while (eap_count > 0) {
      eap_len = tvb_get_guint8(tvb, offset);
      eap_end = offset + 1 + eap_len;
      eap_tree = proto_tree_add_subtree(realm_tree, tvb, offset, 1 + eap_len,
                                 ett_nai_realm_eap, NULL, "EAP Method");

      item = proto_tree_add_item(eap_tree,
                                 hf_ieee80211_ff_anqp_nai_realm_eap_len,
                                 tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset + eap_len > f_end) {
        expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_nai_realm_eap_len);
        break;
      }

      proto_item_append_text(eap_tree, ": %s",
                             val_to_str_ext(tvb_get_guint8(tvb, offset),
                                            &eap_type_vals_ext, "Unknown (%d)"));
      proto_tree_add_item(eap_tree, hf_ieee80211_ff_anqp_nai_realm_eap_method,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      auth_param_count = tvb_get_guint8(tvb, offset);
      proto_tree_add_item(eap_tree,
                          hf_ieee80211_ff_anqp_nai_realm_auth_param_count,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;

      while (auth_param_count > 0) {
        auth_param_id = tvb_get_guint8(tvb, offset);
        proto_tree_add_item(eap_tree,
                            hf_ieee80211_ff_anqp_nai_realm_auth_param_id,
                            tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        auth_param_len = tvb_get_guint8(tvb, offset);
        proto_tree_add_item(eap_tree,
                            hf_ieee80211_ff_anqp_nai_realm_auth_param_len,
                            tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        item = proto_tree_add_item(
          eap_tree, hf_ieee80211_ff_anqp_nai_realm_auth_param_value,
          tvb, offset, auth_param_len, ENC_NA);
        if ((auth_param_id == 3) && (auth_param_len == 1)) {
          guint8 inner_method = tvb_get_guint8(tvb, offset);
          const char *str;
          str = val_to_str_ext(inner_method, &eap_type_vals_ext, "Unknown (%d)");

          proto_item_append_text(eap_tree, " / %s", str);
          proto_item_append_text(item, " - %s", str);
        }
        offset += auth_param_len;

        auth_param_count--;
      }

      offset = eap_end;
      eap_count--;
    }

    offset = f_end;
    count--;
  }
}

static void
dissect_3gpp_cellular_network_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8      iei, num, plmn_idx = 0;
  proto_item *item;

  /* See Annex A of 3GPP TS 24.234 v8.1.0 for description */
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_gud, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_udhl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  iei = tvb_get_guint8(tvb, offset);
  item = proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_iei, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  if (iei == 0)
    proto_item_append_text(item, " (PLMN List)");
  else
    return;
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_plmn_len, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  num = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_num_plmns, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  while (num > 0) {
    proto_item *plmn_item = NULL;
    proto_tree *plmn_tree = NULL;
    guint plmn_val = 0;

    if (tvb_reported_length_remaining(tvb, offset) < 3)
      break;
    plmn_val = tvb_get_letoh24(tvb, offset);
    plmn_item = proto_tree_add_uint_format(tree, hf_ieee80211_3gpp_gc_plmn,
                                tvb, offset, 3, plmn_val, "PLMN %d (0x%x)",
                                plmn_idx++, plmn_val);
    plmn_tree = proto_item_add_subtree(plmn_item, ett_ieee80211_3gpp_plmn);
    dissect_e212_mcc_mnc_wmem_packet_str(tvb, pinfo, plmn_tree, offset, E212_NONE, TRUE);
    num--;
    offset += 3;
  }
}

static void
dissect_domain_name_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  guint8 len;

  while (offset < end) {
    len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_domain_name_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_domain_name,
                        tvb, offset, len, ENC_ASCII|ENC_NA);
    offset += len;
  }
}

static int
dissect_hs20_subscription_remediation(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, void *data _U_)
{
  int offset = 0;
  guint8 url_len = tvb_get_guint8(tvb, offset);
  proto_item *pi = NULL;

  proto_tree_add_item(tree, hf_hs20_subscription_remediation_url_len, tvb, offset,
                        1, ENC_NA);
  offset++;
  if (tvb_reported_length_remaining(tvb, offset) >= url_len) {
    pi = proto_tree_add_item(tree, hf_hs20_subscription_remediation_server_url,
                        tvb, offset, url_len, ENC_ASCII|ENC_NA);
    offset += url_len;
    PROTO_ITEM_SET_URL(pi);
    proto_tree_add_item(tree, hf_hs20_subscription_remediation_server_method,
                        tvb, offset, 1, ENC_NA);
    offset++;
  }

  return offset;
}

static int
dissect_hs20_deauthentication_imminent(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, void *data _U_)
{
  int offset = 0;
  guint8 url_len = 0;
  proto_item *pi = NULL;

  proto_tree_add_item(tree, hf_hs20_deauth_reason_code, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_hs20_reauth_delay, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  url_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_hs20_deauth_reason_url_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (tvb_reported_length_remaining(tvb, offset) >= url_len) {
    pi = proto_tree_add_item(tree, hf_hs20_deauth_imminent_reason_url,
                        tvb, offset, url_len, ENC_ASCII|ENC_NA);
    offset += url_len;
    PROTO_ITEM_SET_URL(pi);
  }
  return offset;
}

#define HS20_ANQP_HS_QUERY_LIST              1
#define HS20_ANQP_HS_CAPABILITY_LIST         2
#define HS20_ANQP_OPERATOR_FRIENDLY_NAME     3
#define HS20_ANQP_WAN_METRICS                4
#define HS20_ANQP_CONNECTION_CAPABILITY      5
#define HS20_ANQP_NAI_HOME_REALM_QUERY       6
#define HS20_ANQP_OPERATING_CLASS_INDICATION 7
#define HS20_ANQP_OSU_PROVIDERS_LIST         8
/* 9 is reserved */
#define HS20_ANQP_ICON_REQUEST               10
#define HS20_ANQP_ICON_BINARY_FILE           11
#define HS20_ANQP_OPERATOR_ICON_METADATA     12
#define HS20_ANQP_ADVICE_OF_CHARGE           13

static const value_string hs20_anqp_subtype_vals[] = {
  { HS20_ANQP_HS_QUERY_LIST, "HS Query list" },
  { HS20_ANQP_HS_CAPABILITY_LIST, "HS Capability List" },
  { HS20_ANQP_OPERATOR_FRIENDLY_NAME, "Operator Friendly Name" },
  { HS20_ANQP_WAN_METRICS, "WAN Metrics" },
  { HS20_ANQP_CONNECTION_CAPABILITY, "Connection Capability" },
  { HS20_ANQP_NAI_HOME_REALM_QUERY, "NAI Home Realm Query" },
  { HS20_ANQP_OPERATING_CLASS_INDICATION, "Operating Class Indication" },
  { HS20_ANQP_OSU_PROVIDERS_LIST, "OSU Providers List" },
  { HS20_ANQP_ICON_REQUEST, "Icon Request" },
  { HS20_ANQP_ICON_BINARY_FILE, "Icon Binary File" },
  { HS20_ANQP_OPERATOR_ICON_METADATA, "Operator Icon Metadata" },
  { HS20_ANQP_ADVICE_OF_CHARGE, "Advice of Charge" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_hs_query_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_hs20_anqp_hs_query_list,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_hs20_anqp_operator_friendly_name(proto_tree *tree, tvbuff_t *tvb,
                                         packet_info *pinfo, int offset,
                                         int end, int hf_array[],
                                         gint ett_val)
{
  int ofn_index = 0;

  while (offset + 4 <= end) {
    guint8 vlen = tvb_get_guint8(tvb, offset);
    proto_tree *ofn_tree = NULL;
    proto_item *item = NULL, *pi = NULL;
    int start_offset = offset;

    ofn_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1, ett_val,
                                        &pi, "Friendly Name %d", ofn_index);

    item = proto_tree_add_item(ofn_tree, hf_array[0],
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (vlen > end - offset || vlen < 3) {
      expert_add_info(pinfo, item, &ei_hs20_anqp_ofn_length);
      break;
    }
    proto_tree_add_item(tree, hf_array[1],
                        tvb, offset, 3, ENC_ASCII|ENC_NA);
    proto_tree_add_item(tree, hf_array[2],
                        tvb, offset + 3, vlen - 3, ENC_UTF_8|ENC_NA);
    offset += vlen;

    proto_item_set_len(pi, offset - start_offset);
    ofn_index++;
  }
}

static const value_string hs20_wm_link_status_vals[] = {
  { 0, "Reserved" },
  { 1, "Link up" },
  { 2, "Link down" },
  { 3, "Link in test state" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_wan_metrics(proto_tree *tree, tvbuff_t *tvb, int offset, gboolean request)
{
  if (request)
    return;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_link_status,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_symmetric_link,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_at_capacity,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_reserved,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_downlink_speed,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_uplink_speed,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_downlink_load,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_uplink_load,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_hs20_anqp_wan_metrics_lmd,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
}

static const value_string hs20_cc_status_vals[] = {
  { 0, "Closed" },
  { 1, "Open" },
  { 2, "Unknown" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_connection_capability(proto_tree *tree, tvbuff_t *tvb,
                                        int offset, int end)
{
  proto_tree *tuple;
  while (offset + 4 <= end) {
    guint8 ip_proto, status;
    guint16 port_num;

    ip_proto = tvb_get_guint8(tvb, offset);
    port_num = tvb_get_letohs(tvb, offset + 1);
    status = tvb_get_guint8(tvb, offset + 3);

    tuple = proto_tree_add_subtree_format(tree, tvb, offset, 4, ett_hs20_cc_proto_port_tuple, NULL,
                               "ProtoPort Tuple - ip_proto=%u port_num=%u status=%s",
                               ip_proto, port_num,
                               val_to_str(status, hs20_cc_status_vals,
                                          "Reserved (%u)"));
    proto_tree_add_item(tuple, hf_hs20_anqp_cc_proto_ip_proto,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    proto_tree_add_item(tuple, hf_hs20_anqp_cc_proto_port_num,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tuple, hf_hs20_anqp_cc_proto_status,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_hs20_anqp_nai_home_realm_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                                       int offset, int end)
{
  guint8 len;
  proto_item *item;

  proto_tree_add_item(tree, hf_hs20_anqp_nai_hrq_count,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  while (offset + 2 <= end) {
    proto_tree_add_item(tree, hf_hs20_anqp_nai_hrq_encoding_type,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_guint8(tvb, offset);
    item = proto_tree_add_item(tree, hf_hs20_anqp_nai_hrq_length,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > end) {
      expert_add_info(pinfo, item, &ei_hs20_anqp_nai_hrq_length);
      break;
    }
    proto_tree_add_item(tree, hf_hs20_anqp_nai_hrq_realm_name,
                        tvb, offset, len, ENC_ASCII|ENC_NA);
    offset += len;
  }
}

static void
dissect_hs20_anqp_oper_class_indic(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_hs20_anqp_oper_class_indic,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static int
dissect_hs20_osu_friendly_names(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end _U_)
{
  int osu_fn_hf_array[3] = {hf_hs20_osu_friendly_name_length,
                            hf_hs20_osu_friendly_name_language,
                            hf_hs20_osu_friendly_name_name };
  guint16 osu_fn_count = tvb_get_letohs(tvb, offset);
  proto_tree *fn_tree = NULL;

  proto_tree_add_item(tree, hf_hs20_osu_friendly_names_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  fn_tree = proto_tree_add_subtree(tree, tvb, offset, osu_fn_count,
                        ett_hs20_friendly_names_list, NULL,
                        "Friendly Names List");

  dissect_hs20_anqp_operator_friendly_name(fn_tree, tvb, pinfo, offset,
                        offset + osu_fn_count,
                        osu_fn_hf_array, ett_hs20_friendly_name_tree);

  return offset + osu_fn_count;
}

static int
dissect_hs20_osu_icon_available(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo _U_, int offset, int end _U_, guint16 icon_index)
{
  proto_tree *icon_avail = NULL;
  proto_item *pi = NULL;
  int start_offset = offset;
  guint8 icon_type_len = 0, icon_filename_len = 0;

  icon_avail = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_osu_icon_tree, &pi,
                        "Icon Available %d", icon_index);

  proto_tree_add_item(icon_avail, hf_osu_icon_avail_width, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(icon_avail, hf_osu_icon_avail_height, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(icon_avail, hf_osu_icon_avail_lang_code, tvb, offset, 3,
                        ENC_ASCII|ENC_NA);
  offset += 3;

  icon_type_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(icon_avail, hf_osu_icon_avail_icon_type_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(icon_avail, hf_osu_icon_avail_icon_type, tvb, offset,
                        icon_type_len, ENC_ASCII|ENC_NA);
  offset += icon_type_len;

  icon_filename_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(icon_avail, hf_osu_icon_avail_filename_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(icon_avail, hf_osu_icon_avail_filename, tvb, offset,
                        icon_filename_len, ENC_ASCII|ENC_NA);
  offset += icon_filename_len;

  proto_item_set_len(pi, offset - start_offset);

  return offset;
}

static const value_string osu_method_vals[] = {
  { 0, "OMA DM" },
  { 1, "SOAP XML SPP" },
  { 0, NULL },
};

static int
dissect_hs20_osu_provider(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end, guint8 provider_index)
{
  proto_tree *prov_tree = NULL;
  proto_item *osupi = NULL, *uri_pi = NULL;
  int start_offset = offset;
  guint8 osu_server_uri_len = 0;
  guint8 osu_method_list_len = 0;
  guint16 icons_avail = 0, icons_index = 0;
  guint8 osu_nai_len = 0;
  guint16 osu_service_desc_len = 0;

  prov_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_osu_provider_tree, &osupi,
                        "OSU Provider %d", provider_index);

  proto_tree_add_item(prov_tree, hf_hs20_osu_prov_length, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  offset = dissect_hs20_osu_friendly_names(prov_tree, tvb, pinfo, offset, end);

  proto_item_set_len(osupi, offset - start_offset);

  osu_server_uri_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(prov_tree, hf_hs20_osu_server_uri_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  uri_pi = proto_tree_add_item(prov_tree, hf_hs20_osu_server_uri, tvb, offset,
                        osu_server_uri_len, ENC_ASCII|ENC_NA);
  offset += osu_server_uri_len;
  PROTO_ITEM_SET_URL(uri_pi);

  osu_method_list_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(prov_tree, hf_hs20_osu_method_list_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (osu_method_list_len > 0) {
    proto_tree *osu_method_list = NULL;
    guint8 osu_method_list_index = 0;

    osu_method_list = proto_tree_add_subtree(prov_tree, tvb, offset,
                                osu_method_list_len,
                                ett_hs20_osu_provider_method_list,
                                NULL, "OSU Method List");
    while (osu_method_list_len > osu_method_list_index) {
      proto_item *pi = NULL;
      guint8 method = tvb_get_guint8(tvb, offset);

      pi = proto_tree_add_item(osu_method_list, hf_hs20_osu_method_val, tvb,
                        offset, 1, ENC_NA);
      proto_item_append_text(pi, ": %s",
                                val_to_str(method, osu_method_vals,
                                        "Reserved"));
      offset++;
      osu_method_list_index++;
    }
  }

  icons_avail = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(prov_tree, hf_hs20_icons_avail_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  if (icons_avail > 0) {
    proto_tree *icon_list = NULL;
    proto_item *pi = NULL;

    start_offset = offset;

    icon_list = proto_tree_add_subtree(prov_tree, tvb, offset, -1,
                                ett_osu_icons_avail_list, &pi,
                                "Icons Available");

    while ((offset - start_offset) < icons_avail) {
      offset = dissect_hs20_osu_icon_available(icon_list, tvb, pinfo, offset,
                                end, icons_index);
      icons_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }

  osu_nai_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(prov_tree, hf_hs20_osu_nai_len, tvb, offset, 1, ENC_NA);
  offset++;

  if (osu_nai_len > 0) {
    proto_tree_add_item(prov_tree, hf_hs20_osu_nai, tvb, offset,
                        osu_nai_len, ENC_NA);
    offset += osu_nai_len;
  }

  osu_service_desc_len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(prov_tree, hf_hs20_osu_service_desc_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  if (osu_service_desc_len > 0) {
    proto_tree *desc_tree = NULL;
    proto_item *pi = NULL;
    guint8 service_desc_index = 0;

    start_offset = offset;
    desc_tree = proto_tree_add_subtree(prov_tree, tvb, offset, -1,
                                ett_hs20_osu_service_desc_list, &pi,
                                "OSU Service Description List");

    while ((offset - start_offset) < osu_service_desc_len) {
      proto_tree *desc_duple = NULL;
      guint8 serv_desc_len = tvb_get_guint8(tvb, offset);

      desc_duple = proto_tree_add_subtree_format(desc_tree, tvb, offset,
                                1 + serv_desc_len,
                                ett_hs20_osu_service_desc_tree, NULL,
                                "OSU Service Description Duple %d",
                                service_desc_index);

      proto_tree_add_item(desc_duple, hf_hs20_osu_service_desc_duple_len, tvb,
                                offset, 1, ENC_NA);
      offset++;

      proto_tree_add_item(desc_duple, hf_hs20_osu_service_desc_lang, tvb, offset,
                                3, ENC_ASCII|ENC_NA);
      offset += 3;

      proto_tree_add_item(desc_duple, hf_hs20_osu_service_desc, tvb, offset,
                                serv_desc_len - 3, ENC_ASCII|ENC_NA);
      offset += serv_desc_len - 3;

      service_desc_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }

  return end;
}

static void
dissect_hs20_anqp_osu_providers_list(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end)
{
  guint8 ssid_len = tvb_get_guint8(tvb, offset);
  guint8 osu_prov_count = 0, osu_prov_index = 0;

  proto_tree_add_item(tree, hf_hs20_osu_providers_list_ssid_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_hs20_osu_providers_ssid, tvb, offset, ssid_len,
                        ENC_UTF_8|ENC_NA);
  offset += ssid_len;

  osu_prov_count = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_hs20_osu_providers_count, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (osu_prov_count > 0) {
    int start_offset = offset;
    proto_item *pi = NULL;
    proto_tree *osu_prov_list = NULL;

    osu_prov_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_hs20_osu_providers_list, &pi,
                        "OSU Providers List");
    while (offset < end && osu_prov_count > osu_prov_index) {
      offset = dissect_hs20_osu_provider(osu_prov_list, tvb, pinfo, offset, end,
                        osu_prov_index);
      osu_prov_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }
}

static void
dissect_hs20_anqp_icon_request(proto_tree *tree, tvbuff_t *tvb, int offset,
  int end)
{
  proto_tree_add_item(tree, hf_hs20_icon_request_filename, tvb, offset,
                        end - offset, ENC_UTF_8|ENC_NA);
}

static const value_string hs20_icon_download_status_vals[] = {
  { 0, "Success" },
  { 1, "File not found" },
  { 2, "Unspecified file error" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_icon_binary_file(proto_tree *tree, tvbuff_t *tvb, int offset,
  int end _U_)
{
  guint8 icon_download_status = tvb_get_guint8(tvb, offset);
  proto_item *pi = NULL;
  guint8 icon_type_len = 0;
  guint16 icon_binary_data_len = 0;

  pi = proto_tree_add_item(tree, hf_hs20_icon_binary_file_status, tvb, offset, 1,
                        ENC_NA);
  offset++;
  proto_item_append_text(pi, ": %s",
                         val_to_str(icon_download_status,
                                    hs20_icon_download_status_vals,
                                    "Reserved (%u)"));

  icon_type_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_hs20_icon_type_length, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_hs20_icon_type, tvb, offset, icon_type_len,
                        ENC_UTF_8|ENC_NA);
  offset += icon_type_len;

  icon_binary_data_len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_hs20_icon_binary_data_len, tvb, offset, 2,
                        ENC_BIG_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_hs20_icon_binary_data, tvb, offset,
                        icon_binary_data_len, ENC_NA);
}

static void
dissect_hs20_anqp_operator_icon_metadata(proto_tree *tree, tvbuff_t *tvb,
  int offset, int end _U_)
{
  proto_item *pi = NULL;
  int start_offset = offset;
  guint8 icon_type_len = 0, icon_filename_len = 0;

  proto_tree_add_item(tree, hf_osu_icon_avail_width, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_osu_icon_avail_height, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_osu_icon_avail_lang_code, tvb, offset, 3,
                        ENC_ASCII|ENC_NA);
  offset += 3;

  icon_type_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_osu_icon_avail_icon_type_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_osu_icon_avail_icon_type, tvb, offset,
                        icon_type_len, ENC_ASCII|ENC_NA);
  offset += icon_type_len;

  icon_filename_len = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_osu_icon_avail_filename_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_osu_icon_avail_filename, tvb, offset,
                        icon_filename_len, ENC_ASCII|ENC_NA);
  offset += icon_filename_len;

  proto_item_set_len(pi, offset - start_offset);
}

static void
dissect_anqp_venue_url(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  guint16 url_duple_index = 0;

  while (offset < end) {
    proto_tree *venue_url = NULL;
    proto_item *url_pi = NULL;
    guint8 url_duple_len = tvb_get_guint8(tvb, offset);

    venue_url = proto_tree_add_subtree_format(tree, tvb, offset,
                        url_duple_len + 1, ett_hs20_venue_url, NULL,
                        "Venue URL Duple %d", url_duple_index);

    proto_tree_add_item(venue_url, hf_hs20_anqp_venue_url_length, tvb, offset,
                        1, ENC_NA);
    offset++;

    proto_tree_add_item(venue_url, hf_hs20_anqp_venue_number, tvb, offset, 1,
                        ENC_NA);
    offset++;

    url_pi = proto_tree_add_item(venue_url, hf_hs20_anqp_venue_url, tvb, offset,
                        url_duple_len -1, ENC_ASCII|ENC_NA);
    PROTO_ITEM_SET_URL(url_pi);

    offset += (url_duple_len - 1);

    url_duple_index++;
  }
}

static const value_string advice_of_charge_type_vals[] = {
  { 0, "Time-based" },
  { 1, "Data-volume-based" },
  { 2, "Time-and-data-volume-based" },
  { 3, "Unlimited" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_advice_of_charge(proto_tree *tree, tvbuff_t *tvb, int offset,
  int end _U_)
{
  guint16 toc_index = 0;

  while (offset < end) {
    guint16 adv_charge_len = tvb_get_letohs(tvb, offset);
    proto_tree *aoc_tree = NULL;
    proto_tree *plan_info_tree = NULL;
    proto_item *pi = NULL, *tpi = NULL;
    int start_offset = offset;
    guint8 aoc_type = 0, nai_realm_len = 0;
    guint8 plan_index = 0;
    guint16 plan_tot_len = 0;
    int plan_offset = 0;

    aoc_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_advice_of_charge, &pi,
                        "Advice of Charge Duple %d", toc_index);
    proto_tree_add_item(aoc_tree, hf_hs20_anqp_advice_of_charge_length, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    aoc_type = tvb_get_guint8(tvb, offset);
    tpi = proto_tree_add_item(aoc_tree, hf_hs20_anqp_advice_of_charge_type, tvb,
                        offset, 1, ENC_NA);
    offset++;
    proto_item_append_text(tpi, ": %s",
                                val_to_str(aoc_type,
                                        advice_of_charge_type_vals,
                                        "Reserved (%u)"));

    proto_tree_add_item(aoc_tree, hf_hs20_anqp_aoc_nai_realm_encoding, tvb,
                        offset, 1, ENC_NA);
    offset++;

    nai_realm_len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(aoc_tree, hf_hs20_anqp_aoc_nai_realm_len, tvb, offset,
                        1, ENC_NA);
    offset++;

    proto_tree_add_item(aoc_tree, hf_hs20_anqp_aoc_nai_realm, tvb, offset,
                        nai_realm_len, ENC_UTF_8|ENC_NA);
    offset += nai_realm_len;

    plan_tot_len = adv_charge_len - 3 - nai_realm_len;
    plan_offset = offset;

    while (offset < (plan_offset + plan_tot_len)) {
        guint16 plan_len = tvb_get_letohs(tvb, offset);

        plan_info_tree = proto_tree_add_subtree_format(aoc_tree, tvb, offset,
                                plan_len + 2, ett_hs20_aoc_plan, NULL,
                                "Plan #%u", plan_index);

        proto_tree_add_item(plan_info_tree, hf_hs20_anqp_aoc_plan_len, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
        offset += 2;

        proto_tree_add_item(plan_info_tree, hf_hs20_anqp_aoc_plan_lang, tvb,
                        offset, 3, ENC_ASCII|ENC_NA);
        offset += 3;

        proto_tree_add_item(plan_info_tree, hf_hs20_anqp_aoc_plan_curcy, tvb,
                        offset, 3, ENC_ASCII|ENC_NA);
        offset += 3;

        proto_tree_add_item(plan_info_tree, hf_hs20_anqp_aoc_plan_information,
                        tvb, offset, plan_len - 6, ENC_UTF_8|ENC_NA);
        offset += plan_len - 6;

        plan_index++;
    }

    proto_item_set_len(pi, offset - start_offset);

    toc_index++;
  }
}

static int
dissect_hs20_anqp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
  guint8 subtype;
  int ofn_hf_array[3] = {hf_hs20_anqp_ofn_length,
                         hf_hs20_anqp_ofn_language,
                         hf_hs20_anqp_ofn_name };

  int end = tvb_reported_length(tvb);
  int offset = 0;
  anqp_info_dissector_data_t* anqp_data = (anqp_info_dissector_data_t*)data;

  DISSECTOR_ASSERT(anqp_data);

  subtype = tvb_get_guint8(tvb, offset);
  proto_item_append_text(tree, " - HS 2.0 %s",
                         val_to_str(subtype, hs20_anqp_subtype_vals,
                                    "Reserved (%u)"));
  if (anqp_data->idx == 0) {
    col_append_fstr(pinfo->cinfo, COL_INFO, " HS 2.0 %s",
                    val_to_str(subtype, hs20_anqp_subtype_vals,
                               "Reserved (%u)"));
  } else if (anqp_data->idx == 1) {
    col_append_str(pinfo->cinfo, COL_INFO, ", ..");
  }
  proto_tree_add_item(tree, hf_hs20_anqp_subtype, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_hs20_anqp_reserved, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset++;

  switch (subtype) {
  case HS20_ANQP_HS_QUERY_LIST:
    dissect_hs20_anqp_hs_query_list(tree, tvb, offset, end);
    break;
  case HS20_ANQP_HS_CAPABILITY_LIST:
    dissect_hs20_anqp_hs_capability_list(tree, tvb, offset, end);
    break;
  case HS20_ANQP_OPERATOR_FRIENDLY_NAME:
    dissect_hs20_anqp_operator_friendly_name(tree, tvb, pinfo, offset, end,
                                ofn_hf_array, ett_hs20_ofn_tree);
    break;
  case HS20_ANQP_WAN_METRICS:
    dissect_hs20_anqp_wan_metrics(tree, tvb, offset, anqp_data->request);
    break;
  case HS20_ANQP_CONNECTION_CAPABILITY:
    dissect_hs20_anqp_connection_capability(tree, tvb, offset, end);
    break;
  case HS20_ANQP_NAI_HOME_REALM_QUERY:
    dissect_hs20_anqp_nai_home_realm_query(tree, tvb, pinfo, offset, end);
    break;
  case HS20_ANQP_OPERATING_CLASS_INDICATION:
    dissect_hs20_anqp_oper_class_indic(tree, tvb, offset, end);
    break;
  case HS20_ANQP_OSU_PROVIDERS_LIST:
    dissect_hs20_anqp_osu_providers_list(tree, tvb, pinfo, offset, end);
    break;
  case HS20_ANQP_ICON_REQUEST:
    dissect_hs20_anqp_icon_request(tree, tvb, offset, end);
    break;
  case HS20_ANQP_ICON_BINARY_FILE:
    dissect_hs20_anqp_icon_binary_file(tree, tvb, offset, end);
    break;
  case HS20_ANQP_OPERATOR_ICON_METADATA:
    dissect_hs20_anqp_operator_icon_metadata(tree, tvb, offset, end);
    break;
  case HS20_ANQP_ADVICE_OF_CHARGE:
    dissect_hs20_anqp_advice_of_charge(tree, tvb, offset, end);
    break;
  default:
    if (offset == end)
      break;
    proto_tree_add_item(tree, hf_hs20_anqp_payload, tvb, offset,
                        end - offset, ENC_NA);
    break;
  }

  return tvb_captured_length(tvb);
}

static int
dissect_vendor_wifi_alliance_anqp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
  guint8 subtype;
  int offset = 0;
  tvbuff_t *subtvb;

  subtype = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_anqp_wfa_subtype, tvb, offset, 1, ENC_NA);
  offset += 1;

  subtvb = tvb_new_subset_remaining(tvb, offset);
  if (!dissector_try_uint_new(wifi_alliance_anqp_info_table, subtype, subtvb, pinfo, tree, FALSE, data))
      call_data_dissector(subtvb, pinfo, tree);

  return tvb_captured_length(tvb);
}


static int
dissect_neighbor_report(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data);

static int
dissect_anqp_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                  gboolean request, int idx)
{
  guint16     id, len;
  guint32     oui;
  proto_item *item, *item_len;
  tvbuff_t *vendor_tvb;
  anqp_info_dissector_data_t anqp_info;

  item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info_id,
                             tvb, offset, 2, ENC_LITTLE_ENDIAN);
  id = tvb_get_letohs(tvb, offset);
  if (id != ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST) {
    if (idx == 0) {
      proto_item_append_text(tree, " - %s",
                             val_to_str_ext(id, &anqp_info_id_vals_ext, "Unknown (%u)"));
      col_append_fstr(pinfo->cinfo, COL_INFO, " %s",
                      val_to_str_ext(id, &anqp_info_id_vals_ext, "Unknown (%u)"));
    } else if (idx == 1) {
      proto_item_append_text(tree, ", ..");
      col_append_str(pinfo->cinfo, COL_INFO, ", ..");
    }
  }
  tree = proto_item_add_subtree(item, ett_gas_anqp);
  offset += 2;
  item_len = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  len = tvb_get_letohs(tvb, offset);
  offset += 2;
  if (tvb_reported_length_remaining(tvb, offset) < len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_ff_anqp_info_length);
    return 4 + len;
  }
  switch (id)
  {
  case ANQP_INFO_ANQP_QUERY_LIST:
    dissect_anqp_query_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_ANQP_CAPAB_LIST:
    dissect_anqp_capab_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_VENUE_NAME_INFO:
    dissect_venue_name_info(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_NETWORK_AUTH_TYPE_INFO:
    dissect_network_auth_type(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_ROAMING_CONSORTIUM_LIST:
    dissect_roaming_consortium_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO:
    dissect_ip_addr_type_availability_info(tree, tvb, offset);
    break;
  case ANQP_INFO_NAI_REALM_LIST:
    dissect_nai_realm_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO:
    dissect_3gpp_cellular_network_info(tree, tvb, pinfo, offset);
    break;
  case ANQP_INFO_DOMAIN_NAME_LIST:
    dissect_domain_name_list(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_NEIGHBOR_REPORT:
    {
      tvbuff_t *report_tvb;
      ieee80211_tagged_field_data_t field_data;

      report_tvb = tvb_new_subset_length(tvb, offset, len);
      field_data.item_tag = item;
      field_data.item_tag_length = item_len;
      dissect_neighbor_report(report_tvb, pinfo, tree, &field_data);
    }
    break;
  case ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST:
    proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
    offset += 3;
    vendor_tvb = tvb_new_subset_length(tvb, offset, len);

    anqp_info.request = request;
    anqp_info.idx = idx;
    if (!dissector_try_uint_new(vendor_specific_anqp_info_table, oui, vendor_tvb, pinfo, tree, FALSE, &anqp_info))
    {
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info, tvb, offset, len, ENC_NA);
    }
    break;
  case ANQP_INFO_VENUE_URL:
    dissect_anqp_venue_url(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_ADVICE_OF_CHARGE:
    dissect_hs20_anqp_advice_of_charge(tree, tvb, offset, offset + len);
    break;
  default:
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info,
                        tvb, offset, len, ENC_NA);
    break;
  }

  return 4 + len;
}

static void
dissect_anqp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, gboolean request)
{
  int idx = 0;

  proto_item_append_text(tree, ": ANQP ");
  proto_item_append_text(tree, request ? "Request" : "Response");
  if (tvb_reported_length_remaining(tvb, offset) < 4) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_not_enough_room_for_anqp_header,
                           "Not enough room for ANQP header");
    return;
  }
  col_append_fstr(pinfo->cinfo, COL_INFO, ", ANQP %s",
                  request ? "Req" : "Resp");
  while (tvb_reported_length_remaining(tvb, offset) > 0) {
    offset += dissect_anqp_info(tree, tvb, pinfo, offset, request, idx);
    idx += 1;
  }
}

static guint
dissect_gas_initial_request(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                            guint type, guint subtype)
{
  guint16     req_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Request Length (2 octets) */
  req_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + req_len, ett_gas_query, &item, "Query Request");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + req_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_request_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_request_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /*
   * Query Request (GAS query; formatted per protocol specified in the
   * Advertisement Protocol IE)
   */
  switch (type) {
  case ADV_PROTO_ID_ANQP:
    dissect_anqp(query, tvb, pinfo, offset, TRUE);
    break;
  case ADV_PROTO_ID_VS:
    if (subtype == ((DPP_CONFIGURATION_PROTOCOL << 8) | WFA_SUBTYPE_DPP)) {
       col_append_fstr(pinfo->cinfo, COL_INFO, ", DPP - %s",
                       val_to_str(subtype >> 8, dpp_subtype_vals, "Unknown (%u)"));
      dissect_wifi_dpp_config_proto(pinfo, query, tvb, offset);
    }
    break;
  default:
    proto_tree_add_item(query, hf_ieee80211_ff_query_request,
                        tvb, offset, req_len, ENC_NA);
  }
  offset += req_len;

  return offset - start;
}

static guint
dissect_gas_initial_response(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                             guint type, guint subtype)
{
  guint16     resp_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Response Length (2 octets) */
  resp_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + resp_len,
                             ett_gas_query, &item, "Query Response");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + resp_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_response_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_response_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /* Query Response (optional) */
  if (resp_len) {
    switch (type) {
    case ADV_PROTO_ID_ANQP:
      dissect_anqp(query, tvb, pinfo, offset, FALSE);
      break;
    case ADV_PROTO_ID_VS:
      if (subtype == ((DPP_CONFIGURATION_PROTOCOL << 8) | WFA_SUBTYPE_DPP)) {
         col_append_fstr(pinfo->cinfo, COL_INFO, ", DPP - %s",
                         val_to_str(subtype >> 8, dpp_subtype_vals, "Unknown (%u)"));
        dissect_wifi_dpp_config_proto(pinfo, query, tvb, offset);
      }
      break;
    default:
      proto_tree_add_item(query, hf_ieee80211_ff_query_response,
                          tvb, offset, resp_len, ENC_NA);
    }
    offset += resp_len;
  }

  return offset - start;
}

static reassembly_table gas_reassembly_table;

static gint ett_gas_resp_fragment = -1;
static gint ett_gas_resp_fragments = -1;

static int hf_ieee80211_gas_resp_fragments = -1;
static int hf_ieee80211_gas_resp_fragment = -1;
static int hf_ieee80211_gas_resp_fragment_overlap = -1;
static int hf_ieee80211_gas_resp_fragment_overlap_conflict = -1;
static int hf_ieee80211_gas_resp_fragment_multiple_tails = -1;
static int hf_ieee80211_gas_resp_fragment_too_long_fragment = -1;
static int hf_ieee80211_gas_resp_fragment_error = -1;
static int hf_ieee80211_gas_resp_fragment_count = -1;
static int hf_ieee80211_gas_resp_reassembled_in = -1;
static int hf_ieee80211_gas_resp_reassembled_length = -1;

static const fragment_items gas_resp_frag_items = {
  &ett_gas_resp_fragment,
  &ett_gas_resp_fragments,
  &hf_ieee80211_gas_resp_fragments,
  &hf_ieee80211_gas_resp_fragment,
  &hf_ieee80211_gas_resp_fragment_overlap,
  &hf_ieee80211_gas_resp_fragment_overlap_conflict,
  &hf_ieee80211_gas_resp_fragment_multiple_tails,
  &hf_ieee80211_gas_resp_fragment_too_long_fragment,
  &hf_ieee80211_gas_resp_fragment_error,
  &hf_ieee80211_gas_resp_fragment_count,
  &hf_ieee80211_gas_resp_reassembled_in,
  &hf_ieee80211_gas_resp_reassembled_length,
  /* Reassembled data field */
  NULL,
  "GAS Response fragments"
};

static guint
dissect_gas_comeback_response(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                              guint type, guint subtype _U_, guint8 frag, gboolean more,
                              guint8 dialog_token)
{
  guint16     resp_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Response Length (2 octets) */
  resp_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + resp_len,
                             ett_gas_query, &item, "Query Response");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + resp_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_response_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_response_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /* Query Response (optional) */
  if (resp_len) {
    if (type == ADV_PROTO_ID_ANQP && (frag == 0) && !more)
      dissect_anqp(query, tvb, pinfo, offset, FALSE);
    else {
      fragment_head *frag_msg;
      gboolean save_fragmented;
      tvbuff_t *new_tvb;

      save_fragmented = pinfo->fragmented;
      pinfo->fragmented = TRUE;
      frag_msg = fragment_add_seq_check(&gas_reassembly_table, tvb, offset,
                                        pinfo, dialog_token, NULL,
                                        frag, resp_len, more);
      new_tvb = process_reassembled_data(tvb, offset, pinfo,
                                         "Reassembled GAS Query Response",
                                         frag_msg, &gas_resp_frag_items,
                                         NULL, tree);
      if (new_tvb) {
        if (type == ADV_PROTO_ID_ANQP)
          dissect_anqp(query, new_tvb, pinfo, 0, FALSE);
        else
          proto_tree_add_item(query, hf_ieee80211_ff_query_response,
                              new_tvb, 0,
                              tvb_reported_length_remaining(new_tvb, 0),
                              ENC_NA);
      }

      /* The old tvb cannot be used anymore */
      ieee80211_tvb_invalid = TRUE;

      pinfo->fragmented = save_fragmented;
    }
    offset += resp_len;
  }

  return offset - start;
}

/* ************************************************************************* */
/*              Dissect and add fixed mgmt fields to protocol tree           */
/* ************************************************************************* */

static guint64 last_timestamp;

static guint
add_ff_timestamp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  last_timestamp = tvb_get_letoh64(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_timestamp, tvb, offset, 8,
                      ENC_LITTLE_ENDIAN);
  return 8;
}

static guint
add_ff_beacon_interval(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_beacon_interval, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  col_append_fstr(pinfo->cinfo, COL_INFO, ", BI=%d",
                  tvb_get_letohs(tvb, offset));
  return 2;
}

static guint
add_ff_cap_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ap_fields[] = {
    &hf_ieee80211_ff_cf_ess,
    &hf_ieee80211_ff_cf_ibss,
    &hf_ieee80211_ff_cf_ap_poll,
    &hf_ieee80211_ff_cf_privacy,
    &hf_ieee80211_ff_cf_preamble,
    &hf_ieee80211_ff_cf_pbcc,
    &hf_ieee80211_ff_cf_agility,
    &hf_ieee80211_ff_cf_spec_man,
    &hf_ieee80211_ff_short_slot_time,
    &hf_ieee80211_ff_cf_apsd,
    &hf_ieee80211_ff_radio_measurement,
    &hf_ieee80211_ff_dsss_ofdm,
    &hf_ieee80211_ff_cf_del_blk_ack,
    &hf_ieee80211_ff_cf_imm_blk_ack,
    NULL
  };

  static const int *ieee80211_sta_fields[] = {
    &hf_ieee80211_ff_cf_ess,
    &hf_ieee80211_ff_cf_ibss,
    &hf_ieee80211_ff_cf_sta_poll,
    &hf_ieee80211_ff_cf_privacy,
    &hf_ieee80211_ff_cf_preamble,
    &hf_ieee80211_ff_cf_pbcc,
    &hf_ieee80211_ff_cf_agility,
    &hf_ieee80211_ff_cf_spec_man,
    &hf_ieee80211_ff_short_slot_time,
    &hf_ieee80211_ff_cf_apsd,
    &hf_ieee80211_ff_radio_measurement,
    &hf_ieee80211_ff_dsss_ofdm,
    &hf_ieee80211_ff_cf_del_blk_ack,
    &hf_ieee80211_ff_cf_imm_blk_ack,
    NULL
  };

  if ((tvb_get_letohs(tvb, offset) & 0x0001) != 0) {
    /* This is an AP */
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_capture,
                                    ett_cap_tree, ieee80211_ap_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    p_add_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_AP_KEY, GINT_TO_POINTER(TRUE));
  } else {
    /* This is a STA */
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_capture,
                                    ett_cap_tree, ieee80211_sta_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }

  return 2;
}

static guint
add_ff_auth_alg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_auth_alg, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_auth_trans_seq(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_auth_seq, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_current_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_current_ap, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_listen_ival(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_listen_ival, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_reason_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_reason, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_assoc_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_assoc_id, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_status_code, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_category_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_category_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dialog_token, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_followup_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_followup_dialog_token, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_wme_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wme_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_wme_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wme_status_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_qos_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_qos_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_block_ack_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ba_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_block_ack_param(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_block_ack_params,
                         ett_ff_ba_param_tree,
                         ieee80211_ff_block_ack_params_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_block_ack_timeout(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_block_ack_timeout, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_block_ack_ssc(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_block_ack_ssc,
                         ett_ff_ba_ssc_tree, ieee80211_ff_block_ack_ssc_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_qos_ts_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_tsinfo,
                         ett_tsinfo_tree, ieee80211_tsinfo_fields,
                         ENC_LITTLE_ENDIAN);
  return 3;
}

static guint
add_ff_mesh_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_multihop_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_multihop_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_mesh_control(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  int    start = offset;
  guint8 flags;

  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_flags, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  flags = tvb_get_guint8(tvb, offset);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_ttl, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_sequence, tvb, offset, 4,
                      ENC_LITTLE_ENDIAN);
  offset += 4;

  switch (flags & 0x03) {
  case 1:
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr4, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    break;
  case 2:
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr5, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr6, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    break;
  case 3:
    proto_item_append_text(tree, " Unknown Address Extension Mode");
    break;
  default:
    /* no default action */
    break;
  }

  return offset - start;
}

static guint
add_ff_selfprot_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_selfprot_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_dls_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dls_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_dst_mac_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dst_mac_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_src_mac_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_src_mac_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_req_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_req_ap_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_res_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_res_ap_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_check_beacon(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_check_beacon, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static guint
add_ff_tod(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_tod, tvb, offset, 4,
                     ENC_NA);
  return 4;
}

static guint
add_ff_toa(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_toa, tvb, offset, 4,
                      ENC_NA);
  return 4;
}

static guint
add_ff_max_tod_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_tod_err, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static guint
add_ff_max_toa_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_toa_err, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static guint
add_ff_dls_timeout(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dls_timeout, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_delba_param_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_delba_param,
                         ett_ff_ba_param_tree, ieee80211_ff_delba_param_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_max_reg_pwr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_reg_pwr, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_measurement_pilot_int(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_measurement_pilot_int, tvb, offset,
                      1, ENC_NA);
  return 1;
}

static guint
add_ff_country_str(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_country_str, tvb, offset, 3,
                      ENC_ASCII|ENC_NA);
  return 3;
}

static guint
add_ff_max_tx_pwr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_tx_pwr, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_tx_pwr_used(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_tx_pwr_used, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_transceiver_noise_floor(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_transceiver_noise_floor, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_channel_width(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_channel_width, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

/* QoS Info:  802.11-2012 8.4.1.17 */
static guint
add_ff_qos_info_ap(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* From AP so decode as AP: Figure 8-51-QoS Info field when sent by a AP */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_qos_info_ap,
                                    ett_ff_qos_info, ieee80211_ff_qos_info_ap_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}


/* QoS Info:  802.11-2012 8.4.1.17 */
static guint
add_ff_qos_info_sta(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* To AP so decode as STA: Figure 8-52-QoS Info field when set by a non-AP STA */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_qos_info_sta,
                                    ett_ff_qos_info, ieee80211_ff_qos_info_sta_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static guint
add_ff_sm_pwr_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_sm_pwr_save,
                         ett_ff_sm_pwr_save, ieee80211_ff_sw_pwr_save_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_pco_phase_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_pco_phase_cntrl, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_psmp_param_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_psmp_param_set,
                         ett_ff_psmp_param_set,
                         ieee80211_ff_psmp_param_set_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_mimo_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *mimo_item;
  proto_tree *mimo_tree;
  static const int *ieee80211_mimo_fields[] = {
    &hf_ieee80211_ff_mimo_cntrl_nc_index,
    &hf_ieee80211_ff_mimo_cntrl_nr_index,
    &hf_ieee80211_ff_mimo_cntrl_channel_width,
    &hf_ieee80211_ff_mimo_cntrl_grouping,
    &hf_ieee80211_ff_mimo_cntrl_coefficient_size,
    &hf_ieee80211_ff_mimo_cntrl_codebook_info,
    &hf_ieee80211_ff_mimo_cntrl_remaining_matrix_segment,
    &hf_ieee80211_ff_mimo_cntrl_reserved,
    NULL
  };

  mimo_item = proto_tree_add_item(tree, hf_ieee80211_ff_mimo_cntrl, tvb,
                                  offset, 6, ENC_NA);
  mimo_tree = proto_item_add_subtree(mimo_item, ett_ff_mimo_cntrl);

  proto_tree_add_bitmask_list(mimo_tree, tvb, offset, 2, ieee80211_mimo_fields, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(mimo_tree, hf_ieee80211_ff_mimo_cntrl_sounding_timestamp,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);

  return 6;
}

static guint
add_ff_ant_selection(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_ant_selection,
                         ett_ff_ant_sel, ieee80211_ff_ant_selection_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_extended_channel_switch_announcement(proto_tree *tree, tvbuff_t *tvb,
                                            packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_ff_ext_channel_switch_announcement,
                         ett_ff_chan_switch_announce,
                         ieee80211_ff_ext_channel_switch_announcement_fields,
                         ENC_LITTLE_ENDIAN);
  return 4;
}

static guint
add_ff_ht_information(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_ht_info,
                         ett_ff_ht_info, ieee80211_ff_ht_info_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_ht_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ht_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_dmg_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dmg_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_dmg_pwr_mgmt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dmg_pwr_mgmt, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_psmp_sta_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *psmp_item;
  proto_tree *psmp_tree;

  psmp_item = proto_tree_add_item(tree, hf_ieee80211_ff_psmp_sta_info, tvb,
                                  offset, 8, ENC_LITTLE_ENDIAN);
  psmp_tree = proto_item_add_subtree(psmp_item, ett_ff_psmp_sta_info);

  proto_tree_add_item(psmp_item, hf_ieee80211_ff_psmp_sta_info_type, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);

  switch (tvb_get_letohl(tvb, offset) & PSMP_STA_INFO_FLAG_TYPE) {
  case PSMP_STA_INFO_BROADCAST:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    /* Missing 64 bit bitmask... */
    proto_tree_add_uint64(psmp_tree,
                          hf_ieee80211_ff_psmp_sta_info_reserved_large,
                          tvb, offset, 8,
                          (tvb_get_letoh64(tvb, offset) &
                           G_GUINT64_CONSTANT(0xFFFFFFFFFFE00000)) >> 21);
    break;
  case PSMP_STA_INFO_MULTICAST:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    /* Missing 64 bit bitmask... */
    proto_tree_add_uint64(psmp_tree,
                          hf_ieee80211_ff_psmp_sta_info_psmp_multicast_id,
                          tvb, offset, 6,
                          (tvb_get_letoh64(tvb, offset) &
                           G_GUINT64_CONSTANT(0xFFFFFFFFFFE00000)) >> 21);
    break;
  case PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_sta_id, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_utt_start_offset,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_utt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_reserved_small, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    break;
  }

  return 8;
}

static guint
add_ff_schedule_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_schedule_info_fields1[] = {
    &hf_ieee80211_sched_info_agg,
    NULL
  };
  static const int *ieee80211_schedule_info_fields2[] = {
    &hf_ieee80211_sched_info_agg,
    &hf_ieee80211_sched_info_tsid,
    &hf_ieee80211_sched_info_dir,
    NULL
  };

  if (tvb_get_letohs(tvb, offset) & 0x0001) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_sched_info,
                                    ett_sched_tree, ieee80211_schedule_info_fields2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_sched_info,
                                    ett_sched_tree, ieee80211_schedule_info_fields1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }

  return 2;
}

static guint
add_ff_pa_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_public_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_ppa_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_protected_public_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_ft_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ft_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_sta_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sta_address, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_target_ap_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_target_ap_address, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static guint
add_ff_gas_comeback_delay(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_gas_comeback_delay, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_gas_fragment_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_gas_fragment_id, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_more_gas_fragments, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_sa_query_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sa_query_action_code, tvb, offset,
                      1, ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_transaction_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_transaction_id, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_tdls_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  code = tvb_get_guint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &tdls_action_codes_ext,
                                   "Unknown TDLS Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_tdls_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_target_channel(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_target_channel, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_operating_class(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_operating_class, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_wnm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;

  code = tvb_get_guint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &wnm_action_codes_ext, "Unknown WNM Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_wnm_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_unprotected_wnm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;

  code = tvb_get_guint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &unprotected_wnm_action_codes_ext, "Unknown Unprotected WNM Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_unprotected_wnm_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_unprotected_dmg_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_unprotected_dmg_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_key_data_length(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_key_data_length, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_wnm_notification_type(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wnm_notification_type,
                      tvb, offset, 1, ENC_NA);
  return 1;
}

/* Action frame: Radio Measurement actions */
static guint
add_ff_rm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_rm_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_dialog_token, tvb, offset, 1, ENC_NA);
  return 1;
}

/* Radio Measurement Request frame, Action fields */
static guint
add_ff_rm_repetitions(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* Note: 65535 means repeated until cancelled or superseded */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_repetitions, tvb, offset, 2,
                      ENC_BIG_ENDIAN);
  return 2;
}

/* Link Measurement Request frame, Action fields*/
static guint
add_ff_rm_tx_power(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* In dBm, see 8.4.1.20 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_tx_power, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_rm_max_tx_power(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* In dBm, see 8.4.1.19 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_max_tx_power, tvb, offset, 1, ENC_NA);
  return 1;
}

/* Link Measurement Report frame, Action fields */
static guint
add_ff_rm_tpc_report(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree *tpc_tree;
  proto_item *tpc_item;

  /* 8.4.2.19 TPC Report element */
  tpc_item = proto_tree_add_item(tree, hf_ieee80211_ff_tpc, tvb, offset, 4, ENC_NA);
  tpc_tree = proto_item_add_subtree(tpc_item, ett_tpc);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_element_id, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_length, tvb, offset + 1, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_tx_power, tvb, offset + 2, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_link_margin, tvb, offset + 3, 1, ENC_NA);
  return 4;
}

static guint
add_ff_rm_rx_antenna_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.42: 0 means unknown, 1-254 is valid, 255 means multiple antennas */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rx_antenna_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_rm_tx_antenna_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.42: 0 means unknown, 1-254 is valid, 255 means multiple antennas */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_tx_antenna_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_rm_rcpi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.40: RCPI as specified for certain PHYs */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rcpi, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_rm_rsni(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.43: RSNI in steps of 0.5 dB, calculated as:
   * RSNI = (10 * log10((RCPI_{power} - ANPI_{power}) / ANPI_{power}) + 20)*2 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rsni, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_bss_transition_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_status_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_bss_termination_delay(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_bss_termination_delay, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_action_spectrum_mgmt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
    switch (tvb_get_guint8(tvb, offset + 1)) {
    case SM_ACTION_MEASUREMENT_REQUEST:
    case SM_ACTION_MEASUREMENT_REPORT:
    case SM_ACTION_TPC_REQUEST:
    case SM_ACTION_TPC_REPORT:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
      return 3;
    case SM_ACTION_CHAN_SWITCH_ANNC:
    case SM_ACTION_EXT_CHAN_SWITCH_ANNC:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      return 2;
    default:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      return 2;
    }
}

static guint
add_ff_action_qos(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  switch (tvb_get_guint8(tvb, offset + 1)) {
  case QOS_ACTION_ADDTS_REQUEST:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
    return 3;
  case QOS_ACTION_ADDTS_RESPONSE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
    add_ff_status_code(tree, tvb, pinfo, offset + 3);
    return 5;
  case QOS_ACTION_DELTS:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_qos_ts_info(tree, tvb, pinfo, offset + 2);
    add_ff_reason_code(tree, tvb, pinfo, offset + 5);
    return 7;
  case QOS_ACTION_SCHEDULE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    return 2;
  case QOS_ACTION_MAP_CONFIGURE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    return 2;
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 2;
  }
}

static guint
add_ff_action_dls(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  switch (tvb_get_guint8(tvb, offset + 1)) {
  case DLS_ACTION_REQUEST:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  2);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset +  8);
    add_ff_cap_info(tree, tvb, pinfo, offset + 14);
    add_ff_dls_timeout(tree, tvb, pinfo, offset + 16);
    return 18;
  case DLS_ACTION_RESPONSE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_status_code(tree, tvb, pinfo, offset +  2);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  4);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset + 10);
    if (!hf_ieee80211_ff_status_code) {
      add_ff_cap_info(tree, tvb, pinfo, offset + 16);
    }
    return 16;
  case DLS_ACTION_TEARDOWN:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  2);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset +  8);
    add_ff_reason_code(tree, tvb, pinfo, offset + 14);
    return 16;
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 2;
  }
}

static guint
add_ff_action_block_ack(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;

  switch (tvb_get_guint8(tvb, offset + 1)) {
  case BA_ADD_BLOCK_ACK_REQUEST:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_param(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_timeout(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_ssc(tree, tvb, pinfo, offset);
    break;
  case BA_ADD_BLOCK_ACK_RESPONSE:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_param(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_timeout(tree, tvb, pinfo, offset);
    break;
  case BA_DELETE_BLOCK_ACK:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_delba_param_set(tree, tvb, pinfo, offset);
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_public_fields(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, guint8 code)
{
  guint32  oui;
  guint    type;
  guint    subtype;
  guint8   dialog_token;
  guint8   frag;
  gboolean more;
  tvbuff_t *vendor_tvb;
  int dissected;

  guint start = offset;

  switch (code) {
  case PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT:
    offset += add_ff_extended_channel_switch_announcement(tree, tvb, pinfo, offset);
    break;
  case PA_VENDOR_SPECIFIC:
    proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
    offset += 3;
    switch (oui) {
    case OUI_WFA:
      subtype = tvb_get_guint8(tvb, offset);
      proto_tree_add_item(tree, hf_ieee80211_tag_oui_wfa_subtype, tvb, offset, 1, ENC_NA);
      offset += 1;
      vendor_tvb = tvb_new_subset_remaining(tvb, offset);
      dissected = dissector_try_uint_new(wifi_alliance_public_action_table, subtype, vendor_tvb, pinfo, tree, FALSE, NULL);
      offset += dissected;
      break;
    default:
      /* Don't know how to handle this vendor */
      break;
    }
    break;
  case PA_GAS_INITIAL_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_initial_request(tree, tvb, pinfo, offset, type,
                                          subtype);
    break;
  case PA_GAS_INITIAL_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_gas_comeback_delay(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_initial_response(tree, tvb, pinfo, offset, type,
                                           subtype);
    break;
  case PA_GAS_COMEBACK_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case PA_GAS_COMEBACK_RESPONSE:
    dialog_token = tvb_get_guint8(tvb, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    frag = tvb_get_guint8(tvb, offset) & 0x7f;
    more = (tvb_get_guint8(tvb, offset) & 0x80) != 0;
    offset += add_ff_gas_fragment_id(tree, tvb, pinfo, offset);
    offset += add_ff_gas_comeback_delay(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_comeback_response(tree, tvb, pinfo, offset, type,
                                            subtype, frag, more, dialog_token);
    break;
  case PA_TDLS_DISCOVERY_RESPONSE:
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "TDLS");
    col_set_str(pinfo->cinfo, COL_INFO, "TDLS Discovery Response");
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case PA_QAB_REQUEST:
  case PA_QAB_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_req_ap_addr(tree, tvb, pinfo, offset);
    offset += add_ff_res_ap_addr(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_public(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint start = offset;
  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_pa_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_action_public_fields(tree, tvb, pinfo, offset, code);
  return offset - start;
}

static guint
add_ff_action_protected_public(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint start = offset;
  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_ppa_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_action_public_fields(tree, tvb, pinfo, offset, code);
  return offset - start;
}

static guint
add_ff_action_radio_measurement(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint  start = offset;
  guint8 code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code = tvb_get_guint8(tvb, offset);
  offset += add_ff_rm_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case RM_ACTION_RADIO_MEASUREMENT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_repetitions(tree, tvb, pinfo, offset);
    /* Followed by Measurement Request Elements */
    break;
  case RM_ACTION_RADIO_MEASUREMENT_REPORT:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Measurement Report Elements */
    break;
  case RM_ACTION_LINK_MEASUREMENT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tx_power(tree, tvb, pinfo, offset);
    offset += add_ff_rm_max_tx_power(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_LINK_MEASUREMENT_REPORT:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tpc_report(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rx_antenna_id(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tx_antenna_id(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rcpi(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rsni(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_NEIGHBOR_REPORT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_NEIGHBOR_REPORT_RESPONSE:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Neighbor Report Elements */
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_fast_bss_transition(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint  start = offset;
  guint8 code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_ft_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case FT_ACTION_REQUEST:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    /* Followed by FT Request frame body (IEs) */
    break;
  case FT_ACTION_RESPONSE:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    /* Followed by FT Response frame body (IEs) */
    break;
  case FT_ACTION_CONFIRM:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    /* Followed by FT Confirm frame body (IEs) */
    break;
  case FT_ACTION_ACK:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    /* Followed by FT Ack frame body (IEs) */
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_sa_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint  start = offset;
  guint8 code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_sa_query_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case SA_QUERY_REQUEST:
    offset += add_ff_transaction_id(tree, tvb, pinfo, offset);
    break;
  case SA_QUERY_RESPONSE:
    offset += add_ff_transaction_id(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_mesh(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint length;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_mesh_action(tree, tvb, pinfo, offset);
  /* The only fixed fields are the category and mesh action.  The rest are IEs.
   */
  length = 2;
  if (tvb_get_guint8(tvb, 1) == MESH_ACTION_TBTT_ADJ_RESPONSE) {
    /* ..except for the TBTT Adjustment Response, which has a status code field
     */
    length += add_ff_status_code(tree, tvb, pinfo, offset);
  }
  return length;
}

static guint
add_ff_action_multihop(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_multihop_action(tree, tvb, pinfo, offset);
  offset += add_ff_mesh_control(tree, tvb, pinfo, offset);
  return offset - start;
}

static guint
add_ff_action_self_protected(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_selfprot_action(tree, tvb, pinfo, offset);

  switch (tvb_get_guint8(tvb, start + 1)) {
  case SELFPROT_ACTION_MESH_PEERING_OPEN:
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case SELFPROT_ACTION_MESH_PEERING_CONFIRM:
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    offset += add_ff_assoc_id(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;
}

static guint
add_ff_vht_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_vht_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint
add_ff_s1g_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static guint get_group_element_len(guint group) {
  switch (group) {
    /* Diffie-Hellman groups */
    case 1:
      return 96;
    case 2:
    case 22:
      return 128;
    case 5:
      return 192;
    case 14:
    case 23:
    case 24:
      return 256;
    case 15:
      return 384;
    case 16:
      return 512;
    case 17:
      return 768;
    case 18:
      return 1024;
    /* ECC groups */
    case 19:
    case 28:
      return 64;
    case 20:
    case 29:
      return 96;
    case 21:
      return 132;
    case 25:
      return 48;
    case 26:
      return 56;
    case 30:
      return 128;
    default:
      return 0;
  }
}

static guint get_scalar_len(guint group) {
  switch (group) {
    /* Diffie-Hellman groups */
    case 1:
      return 96;
    case 2:
      return 128;
    case 5:
      return 192;
    case 14:
      return 256;
    case 15:
      return 384;
    case 16:
      return 512;
    case 17:
      return 768;
    case 18:
      return 1024;
    case 22:
      return 20;
    case 23:
      return 28;
    case 24:
      return 32;
    /* ECC groups */
    case 19:
    case 28:
      return 32;
    case 20:
    case 29:
      return 48;
    case 21:
      return 66;
    case 25:
      return 24;
    case 26:
      return 28;
    case 30:
      return 64;
    default:
      return 0;
  }
}

static guint
get_ff_auth_len(tvbuff_t *tvb)
{
  guint alg, seq, status_code;
  alg = tvb_get_letohs(tvb, 0);
  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  if (alg == AUTH_ALG_SAE) {
    /* 82: Rejected with Suggested BSS Transition (cf ieee80211_status_code) */
    if ((seq == 2) && (status_code == 82))
      return 0;

    /* everything is fixed size fields */
    return tvb_reported_length_remaining(tvb, 6);
  } else if ((alg == AUTH_ALG_FILS_SK_WITH_PFS) || (alg == AUTH_ALG_FILS_PK)) {
    if ((seq ==2) && (status_code != 0))
      return 0;

    guint group = tvb_get_letohs(tvb, 6);
    guint elt_len = get_group_element_len(group);

    return 2 + elt_len;
  } else {
    return 0;
  }
}

static const value_string ff_sae_message_type_vals[] = {
  {1, "Commit" },
  {2, "Confirm" },
  {0, NULL }
};

static void
add_ff_auth_sae(proto_tree *tree, tvbuff_t *tvb)
{
  guint alg, seq, status_code, len;
  alg = tvb_get_letohs(tvb, 0);

  if (alg != AUTH_ALG_SAE)
    return;

  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  proto_tree_add_uint(tree, hf_ieee80211_ff_sae_message_type, tvb, 2, 2, seq);

  if (seq == 1)
  {
    /* 76: Authentication is rejected because an Anti-Clogging Token is required (cf ieee80211_status_code) */
    if (status_code == 76)
    {
      proto_tree_add_item(tree, hf_ieee80211_ff_finite_cyclic_group, tvb, 6, 2,
                          ENC_LITTLE_ENDIAN);
      len = tvb_reported_length_remaining(tvb, 8);
      proto_tree_add_item(tree, hf_ieee80211_ff_anti_clogging_token, tvb, 8, len,
                          ENC_NA);
    }
    else if (status_code == 0)
    {
      guint group = tvb_get_letohs(tvb, 6);
      guint sc_len, elt_len, offset;
      proto_tree_add_item(tree, hf_ieee80211_ff_finite_cyclic_group, tvb, 6, 2,
                          ENC_LITTLE_ENDIAN);
      offset = 8;
      len = tvb_reported_length_remaining(tvb, offset);
      sc_len = get_scalar_len(group);
      elt_len = get_group_element_len(group);

      if (sc_len == 0) {
        /* assume no anti-clogging token */
        if (!(len % 3))
        {
          sc_len = len / 3;
        }
        else
        {
          sc_len = len / 2;
        }
        elt_len = len - sc_len;
      }

      if ((sc_len + elt_len) < len)
      {
        len = len - (sc_len + elt_len);
        proto_tree_add_item(tree, hf_ieee80211_ff_anti_clogging_token, tvb, offset,
                            len, ENC_NA);
        offset += len;
      }
      proto_tree_add_item(tree, hf_ieee80211_ff_scalar, tvb, offset,
                          sc_len, ENC_NA);
      offset += sc_len;
      proto_tree_add_item(tree, hf_ieee80211_ff_finite_field_element, tvb, offset,
                          elt_len, ENC_NA);
    }
  }
  /* 82: Rejected with Suggested BSS Transition (cf ieee80211_status_code) */
  else if ((seq == 2) && (status_code != 82))
  {
    proto_tree_add_item(tree, hf_ieee80211_ff_send_confirm, tvb, 6, 2,
                        ENC_LITTLE_ENDIAN);
    len = tvb_reported_length_remaining(tvb, 8);
    proto_tree_add_item(tree, hf_ieee80211_ff_confirm, tvb, 8, len,
                        ENC_NA);
  };
}

static void
add_ff_auth_fils(proto_tree *tree, tvbuff_t *tvb)
{
  guint alg, seq, status_code;
  alg = tvb_get_letohs(tvb, 0);

  if ((alg != AUTH_ALG_FILS_SK_WITH_PFS) && (alg != AUTH_ALG_FILS_PK))
    return;

  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  if ((seq == 1) || (seq == 2 && status_code == 0)) {
    guint group = tvb_get_letohs(tvb, 6);
    guint elt_len, offset;
    proto_tree_add_item(tree, hf_ieee80211_ff_finite_cyclic_group, tvb, 6, 2,
                        ENC_LITTLE_ENDIAN);
    offset = 8;
    elt_len = get_group_element_len(group);

    proto_tree_add_item(tree, hf_ieee80211_ff_finite_field_element, tvb, offset,
                        elt_len, ENC_NA);
  }
}

static guint
wnm_bss_trans_mgmt_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  gint   left;
  int tmp_sublen;
  const guint8 ids[] = { TAG_NEIGHBOR_REPORT };


  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_query_reason, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  left = tvb_reported_length_remaining(tvb, offset);
  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);

    while (left > 0){
      tmp_sublen = tvb_get_guint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}


static guint
wnm_bss_trans_mgmt_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  guint8 mode;
  gint   left;
  int tmp_sublen;
  const guint8 ids[] = { TAG_NEIGHBOR_REPORT };

  static const int *ieee80211_ff_request_flags[] = {
    &hf_ieee80211_ff_request_mode_pref_cand,
    &hf_ieee80211_ff_request_mode_abridged,
    &hf_ieee80211_ff_request_mode_disassoc_imminent,
    &hf_ieee80211_ff_request_mode_bss_term_included,
    &hf_ieee80211_ff_request_mode_ess_disassoc_imminent,
    NULL
  };

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  mode = tvb_get_guint8(tvb, offset);
  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_ff_request_flags, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_ff_disassoc_timer, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_ff_validity_interval, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset += 1;

  if (mode & 0x08) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_termination_duration,
                        tvb, offset, 8, ENC_NA);
    offset += 8;
  }

  if (mode & 0x10) {
    guint8 url_len;
    url_len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_url_len, tvb, offset, 1,
                        ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_url, tvb, offset, url_len,
                        ENC_ASCII|ENC_NA);
    offset += url_len;
  }

  left = tvb_reported_length_remaining(tvb, offset);
  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);

    while (left > 0){
      tmp_sublen = tvb_get_guint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}


static guint
wnm_bss_trans_mgmt_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  guint8 code;
  gint   left;
  int tmp_sublen;
  const guint8 ids[] = { TAG_NEIGHBOR_REPORT };

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  code = tvb_get_guint8(tvb, offset);
  offset += add_ff_bss_transition_status_code(tree, tvb, pinfo, offset);
  offset += add_ff_bss_termination_delay(tree, tvb, pinfo, offset);
  if (!code) {
    proto_tree_add_item(tree, hf_ieee80211_ff_target_bss,
                        tvb, offset, 6, ENC_NA);
    offset += 6;
  }
  left = tvb_reported_length_remaining(tvb, offset);
  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);
    while (left > 0){
      tmp_sublen = tvb_get_guint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}

static guint
wnm_sleep_mode_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

static guint
wnm_sleep_mode_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  guint16 key_data_len;
  gint left;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  key_data_len = tvb_get_letohs(tvb, offset);
  offset += add_ff_key_data_length(tree, tvb, pinfo, offset);
  left = tvb_reported_length_remaining(tvb, offset);
  if (left < key_data_len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_tag_wnm_sleep_mode_no_key_data);
    return offset - start;
  }
  proto_tree_add_item(tree, hf_ieee80211_ff_key_data, tvb, offset,
                      key_data_len, ENC_NA);
  offset += key_data_len;
  return offset - start;
}

static guint
wnm_tfs_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

static guint
wnm_tfs_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

#define AP_DESCRIPTOR            0
#define FIRMWARE_VERSION_CURRENT 1
#define FIRMWARE_VERSION_NEW     2

static guint
dissect_wnm_subelements(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
  int offset) {
  guint sub_elt_id = tvb_get_guint8(tvb, offset);
  guint sub_elt_len = tvb_get_guint8(tvb, offset + 1);

  proto_tree_add_item(tree, hf_ieee80211_wnm_sub_elt_id, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_wnm_sub_elt_len, tvb, offset, 1, ENC_NA);
  offset++;

  switch (sub_elt_id) {
  case AP_DESCRIPTOR:

    break;

  case FIRMWARE_VERSION_CURRENT:

    break;

  case FIRMWARE_VERSION_NEW:

    break;
  }

  offset += sub_elt_len;
  return offset;
}

static guint
wnm_notification_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  int len = 0;
  guint8 wnm_type = 0;
  guint8 wnm_sub_elt = 0;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  wnm_type = tvb_get_guint8(tvb, offset);
  offset += add_ff_wnm_notification_type(tree, tvb, pinfo, offset);
  len = tvb_reported_length_remaining(tvb, offset);

  if (wnm_type == 0) {
    int offset_end = offset + len;
    while (offset < offset_end) {
      int start_offset = offset;
      proto_tree *wnm_list = NULL;
      proto_item *wnm_item = NULL;
      wnm_list = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_wnm_notif_subelt,
                        &wnm_item, "WNM Subelement %d", wnm_sub_elt);
      offset = dissect_wnm_subelements(wnm_list, tvb, pinfo, offset);
      proto_item_set_len(wnm_item, offset - start_offset);
    }
  }
  return offset - start;
}

static guint
add_ff_action_wnm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_wnm_action_code(tree, tvb, pinfo, offset);
  switch (code) {
  case WNM_BSS_TRANS_MGMT_QUERY:
    offset += wnm_bss_trans_mgmt_query(tree, tvb, pinfo, offset);
    break;
  case WNM_BSS_TRANS_MGMT_REQ:
    offset += wnm_bss_trans_mgmt_req(tree, tvb, pinfo, offset);
    break;
  case WNM_BSS_TRANS_MGMT_RESP:
    offset += wnm_bss_trans_mgmt_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_TFS_REQ:
    offset += wnm_tfs_req(tree, tvb, pinfo, offset);
    break;
  case WNM_TFS_RESP:
    offset += wnm_tfs_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_SLEEP_MODE_REQ:
    offset += wnm_sleep_mode_req(tree, tvb, pinfo, offset);
    break;
  case WNM_SLEEP_MODE_RESP:
    offset += wnm_sleep_mode_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_NOTIFICATION_REQ:
    offset += wnm_notification_req(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_unprotected_wnm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_unprotected_wnm_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case UNPROTECTED_WNM_TIM:
    offset += add_ff_check_beacon(tree, tvb, pinfo, offset);
    offset += add_ff_timestamp(tree, tvb, pinfo, offset);
    offset += add_ff_tod(tree, tvb, pinfo, offset);
    offset += add_ff_toa(tree, tvb, pinfo, offset);
    offset += add_ff_max_tod_err(tree, tvb, pinfo, offset);
    offset += add_ff_max_toa_err(tree, tvb, pinfo, offset);
    break;
  case UNPROTECTED_WNM_TIMING_MEASUREMENT:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_followup_dialog_token(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_tdls(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8  code;
  guint16 status;
  guint   start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code = tvb_get_guint8(tvb, offset);
  offset += add_ff_tdls_action_code(tree, tvb, pinfo, offset);
  switch (code) {
  case TDLS_SETUP_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case TDLS_SETUP_RESPONSE:
    status = tvb_get_letohs(tvb, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_reported_length_remaining(tvb, offset) < 2) {
      if (status == 0) {
        expert_add_info(pinfo, tree, &ei_ieee80211_tdls_setup_response_malformed);
      }
      break;
    }
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case TDLS_SETUP_CONFIRM:
    status = tvb_get_letohs(tvb, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    if (tvb_reported_length_remaining(tvb, offset) < 1) {
      if (status == 0) {
        expert_add_info(pinfo, tree, &ei_ieee80211_tdls_setup_confirm_malformed);
      }
      break;
    }
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_TEARDOWN:
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_TRAFFIC_INDICATION:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_CHANNEL_SWITCH_REQUEST:
    offset += add_ff_target_channel(tree, tvb, pinfo, offset);
    offset += add_ff_operating_class(tree, tvb, pinfo, offset);
    break;
  case TDLS_CHANNEL_SWITCH_RESPONSE:
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_PSM_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_PSM_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_TRAFFIC_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_DISCOVERY_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_mgmt_notification(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_wme_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  offset += add_ff_wme_status_code(tree, tvb, pinfo, offset);

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_vendor_specific(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint   start = offset;
  guint32 oui;
  tvbuff_t *vendor_tvb;
  int dissected;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
  offset += 3;

  vendor_tvb = tvb_new_subset_remaining(tvb, offset);
  dissected = dissector_try_uint_new(vendor_specific_action_table, oui, vendor_tvb, pinfo, tree, FALSE, NULL);
  if (dissected <= 0)
  {
      call_data_dissector(vendor_tvb, pinfo, tree);
      /* Skip the whole TVB because we don't know its format */
      dissected = tvb_reported_length_remaining(vendor_tvb, 0);
  }

  offset += dissected;

  return offset - start;  /* Size of fixed fields */
}

static guint
add_ff_action_ht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint  start = offset;
  guint8 n_sta, i;
  mimo_control_t mimo_cntrl;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_ht_action_code(tree, tvb, pinfo, offset);

  switch (tvb_get_guint8(tvb, offset - 1)) {
  case HT_ACTION_NOTIFY_CHAN_WIDTH:
    offset += add_ff_channel_width(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_SM_PWR_SAVE:
    offset += add_ff_sm_pwr_cntrl(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_PSMP_ACTION:
    n_sta = tvb_get_guint8(tvb, offset);
    offset += add_ff_psmp_param_set(tree, tvb, pinfo, offset);
    for (i = 0; i < (n_sta & 0x0F); i++) {
      offset += add_ff_psmp_sta_info(tree, tvb, pinfo, offset);
    }
    break;
  case HT_ACTION_SET_PCO_PHASE:
    offset += add_ff_pco_phase_cntrl(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_MIMO_CSI:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_csi_matrices_report(tree, tvb, offset, mimo_cntrl);
    break;
  case HT_ACTION_MIMO_BEAMFORMING:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_beamforming_feedback_report(tree, tvb, offset,
                                                   mimo_cntrl);
    break;
  case HT_ACTION_MIMO_COMPRESSED_BEAMFORMING:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_compressed_beamforming_feedback_report(tree, tvb,
                                                              offset,
                                                              mimo_cntrl);
    break;
  case HT_ACTION_ANT_SEL_FEEDBACK:
    offset += add_ff_ant_selection(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_HT_INFO_EXCHANGE:
    offset += add_ff_ht_information(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;
}

static guint
add_ff_beacon_interval_ctrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bic,
                                    ett_bic_tree, ieee80211_ff_bic_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  return 6;
}

static guint
add_ff_beamforming_ctrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset, gboolean isGrant)
{
  guint16 bf_field = tvb_get_letohs(tvb, offset);
  gboolean isInit = (bf_field & 0x2) >> 1;
  gboolean isResp = (bf_field & 0x4) >> 2;
  static const int *ieee80211_ff_beamforming_ctrl[] = {
    &hf_ieee80211_ff_bf_train,
    &hf_ieee80211_ff_bf_is_init,
    &hf_ieee80211_ff_bf_is_resp,
    &hf_ieee80211_ff_bf_rxss_len,
    &hf_ieee80211_ff_bf_rxss_rate,
    &hf_ieee80211_ff_bf_b10b15,
    NULL
  };

  static const int *ieee80211_ff_beamforming_ctrl_grant[] = {
    &hf_ieee80211_ff_bf_train,
    &hf_ieee80211_ff_bf_is_init,
    &hf_ieee80211_ff_bf_is_resp,
    &hf_ieee80211_ff_bf_num_sectors,
    &hf_ieee80211_ff_bf_num_rx_dmg_ants,
    &hf_ieee80211_ff_bf_b12b15,
    NULL
  };

  if((isInit==TRUE) && (isResp==TRUE) && isGrant) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bf,
                                    ett_bf_tree, ieee80211_ff_beamforming_ctrl_grant,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bf,
                                    ett_bf_tree, ieee80211_ff_beamforming_ctrl,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  return 2;
}

static guint
add_ff_dynamic_allocation(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_dynamic_allocation[] = {
    &hf_ieee80211_ff_TID,
    &hf_ieee80211_ff_alloc_type,
    &hf_ieee80211_ff_src_aid,
    &hf_ieee80211_ff_dest_aid,
    &hf_ieee80211_ff_alloc_duration,
    &hf_ieee80211_ff_b39,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_dynamic_allocation,
                                    ett_dynamic_alloc_tree, ieee80211_ff_dynamic_allocation,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  return 5;
}

static guint
add_ff_beamformed_link(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_beamformed_link[] = {
    &hf_ieee80211_ff_blm_unit_index,
    &hf_ieee80211_ff_blm_maint_value,
    &hf_ieee80211_ff_blm_is_master,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_blm,
                                    ett_blm_tree, ieee80211_ff_beamformed_link,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static guint
add_ff_BRP_request(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_BRP_request[] = {
    &hf_ieee80211_ff_brp_L_RX,
    &hf_ieee80211_ff_brp_TX_TRN_REQ,
    &hf_ieee80211_ff_brp_MID_REQ,
    &hf_ieee80211_ff_brp_BC_REQ,
    &hf_ieee80211_ff_brp_MID_GRANT,
    &hf_ieee80211_ff_brp_BC_GRANT,
    &hf_ieee80211_ff_brp_chan_FBCK_CAP,
    &hf_ieee80211_ff_brp_tx_sector,
    &hf_ieee80211_ff_brp_other_aid,
    &hf_ieee80211_ff_brp_tx_antenna,
    &hf_ieee80211_ff_brp_reserved,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_brp,
                                    ett_brp_tree, ieee80211_ff_BRP_request,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 4;
}

static guint
add_ff_sector_sweep_feedback_from_iss(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_sector_sweep_feedback_from_iss[] = {
    &hf_ieee80211_ff_sswf_total_sectors,
    &hf_ieee80211_ff_sswf_num_rx_dmg_ants,
    &hf_ieee80211_ff_sswf_reserved1,
    &hf_ieee80211_ff_sswf_poll_required,
    &hf_ieee80211_ff_sswf_reserved2,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_sswf,
                                    ett_sswf_tree, ieee80211_ff_sector_sweep_feedback_from_iss,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static guint
add_ff_sector_sweep_feedback_to_iss(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_sector_sweep_feedback_to_iss[] = {
    &hf_ieee80211_ff_sswf_sector_select,
    &hf_ieee80211_ff_sswf_dmg_antenna_select,
    &hf_ieee80211_ff_sswf_snr_report,
    &hf_ieee80211_ff_sswf_poll_required,
    &hf_ieee80211_ff_sswf_reserved2,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_sswf,
                                    ett_sswf_tree, ieee80211_ff_sector_sweep_feedback_to_iss,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static guint
add_ff_sector_sweep(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_sector_sweep[] = {
    &hf_ieee80211_ff_ssw_direction,
    &hf_ieee80211_ff_ssw_cdown,
    &hf_ieee80211_ff_ssw_sector_id,
    &hf_ieee80211_ff_ssw_dmg_ant_id,
    &hf_ieee80211_ff_ssw_rxss_len,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_ssw,
                                    ett_ssw_tree, ieee80211_ff_sector_sweep,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static guint
add_ff_dmg_params(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static const int *ieee80211_ff_dmg_params[] = {
    &hf_ieee80211_ff_dmg_params_bss,
    &hf_ieee80211_ff_dmg_params_cbap_only,
    &hf_ieee80211_ff_dmg_params_cbap_src,
    &hf_ieee80211_ff_dmg_params_privacy,
    &hf_ieee80211_ff_dmg_params_policy,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_dmg_params,
                                    ett_dmg_params_tree, ieee80211_ff_dmg_params,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static guint
add_ff_cc_field(proto_tree *tree, tvbuff_t *tvb, int offset, gboolean dis)
{
  proto_item *cc_item = proto_tree_add_item(tree, hf_ieee80211_ff_cc, tvb, offset, 8, ENC_LITTLE_ENDIAN);
  proto_tree *cc_tree = proto_item_add_subtree(cc_item, ett_cc_tree);
  guint64 cc_field;
  if(dis) {
    proto_tree_add_item(cc_tree, hf_ieee80211_ff_cc_abft_resp_addr, tvb, offset, 6, ENC_NA);
  } else {
    cc_field = tvb_get_letoh64(tvb, offset);
    /*TODO : Add support of bitmask for FT_(U)INT64 */
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_sp_duration, tvb, offset, 1, (guint32)(cc_field & 0xff));
    proto_tree_add_uint64(cc_tree, hf_ieee80211_ff_cc_cluster_id, tvb, offset+1, 6, (guint64)((cc_field & G_GUINT64_CONSTANT(0x00ffffffffffff00)) >> 8));
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_role, tvb, offset+7, 1, (guint32)((cc_field & G_GUINT64_CONSTANT(0x0300000000000000)) >> 56));
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_max_mem, tvb, offset+7, 1, (guint32)((cc_field & G_GUINT64_CONSTANT(0x7c00000000000000)) >> 58));
  }
  return 8;
}


static guint
add_ff_band_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_band_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_subject_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_subject_address, tvb, offset, 6, ENC_NA);
  return 6;
}

static guint
add_ff_handover_reason(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_reason, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_handover_remaining_bi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_remaining_bi, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_handover_result(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_result, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_handover_reject_reason(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_reject_reason, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_destination_reds_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_destination_reds_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_destination_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_destination_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_realy_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_realy_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_source_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_source_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_timing_offset(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_timing_offset, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_sampling_frequency_offset(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sampling_frequency_offset, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static guint
add_ff_relay_operation_type(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_relay_operation_type, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_fst_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_fst_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static guint
add_ff_llt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_llt, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return 4;
}

static guint
add_ff_fsts_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_fsts_id, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return 4;
}

static guint
add_ff_oct_mmpdu(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  guint start = offset;
  guint len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_mmpdu_len, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ff_mmpdu_ctrl, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ff_oct_mmpdu, tvb, offset, len, ENC_NA);
  offset += len;
  return offset - start;
}

static int
add_tag_relay_capabilities(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_tag_relay_capabilities[] = {
    &hf_ieee80211_tag_relay_support,
    &hf_ieee80211_tag_relay_use,
    &hf_ieee80211_tag_relay_permission,
    &hf_ieee80211_tag_AC_power,
    &hf_ieee80211_tag_relay_prefer,
    &hf_ieee80211_tag_duplex,
    &hf_ieee80211_tag_cooperation,
    NULL
  };

  if (tag_len < 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length must be 2");
    return 1;
  }

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_relay_capabilities, ENC_NA);
  return tvb_captured_length(tvb);
}

#if 0
static guint
add_ff_relay_capable_sta_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *rcsi_item = proto_tree_add_item(tree, hf_ieee80211_ff_rcsi, tvb, offset, 3, ENC_LITTLE_ENDIAN);
  proto_tree *rcsi_tree = proto_item_add_subtree(rcsi_item, ett_rcsi_tree);
  proto_tree_add_item(rcsi_tree, hf_ieee80211_ff_rcsi_aid, tvb, offset, 1, ENC_NA);
  offset += 1;
  add_tag_relay_capabilities(pinfo, rcsi_item, 2, rcsi_tree, tvb, &offset);
  return 3;
}
#endif

static void
dissect_ieee80211_extension(guint16 fcf, tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_item *ti;
  proto_tree *ext_tree;
  proto_tree *fixed_tree;
  proto_tree *tagged_tree;

  int offset = 0;
  int tagged_parameter_tree_len;

  ti = proto_tree_add_item(tree, proto_wlan_ext, tvb, offset, -1, ENC_NA);
  ext_tree = proto_item_add_subtree(ti, ett_80211_ext);

  switch (COMPOSE_FRAME_TYPE(fcf))
  {
    case EXTENSION_DMG_BEACON:
    {
      gboolean cc, dis;
      guint16 bic_field;
      fixed_tree = get_fixed_parameter_tree(ext_tree, tvb, offset, 20);
      offset += add_ff_timestamp(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_sector_sweep(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_beacon_interval(fixed_tree, tvb, pinfo, offset);
      bic_field = tvb_get_letohs(tvb, offset);
      cc = (bic_field & 0x1);
      dis  = (bic_field & 0x2) >> 1;
      offset += add_ff_beacon_interval_ctrl(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_dmg_params(fixed_tree, tvb, pinfo, offset);
      if(cc) {
        offset += add_ff_cc_field(fixed_tree, tvb, offset, dis);
      }
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);

      /*
       * The tagged params are optional here. See Table 8.33a of the 2012
       * version of the standard.
       */
      if (tagged_parameter_tree_len) {
        tagged_tree = get_tagged_parameter_tree(ext_tree, tvb, offset, tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree, tagged_parameter_tree_len, EXTENSION_DMG_BEACON, NULL);
      }
      break;
    }
  }
}

static guint
add_ff_action_unprotected_dmg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_unprotected_dmg_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case UNPROTECTED_DMG_ANNOUNCE:
      offset += add_ff_timestamp(tree, tvb, pinfo, offset);
      offset += add_ff_beacon_interval(tree, tvb, pinfo, offset);
      break;
    case UNPROTECTED_DMG_BRP:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_BRP_request(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

/* There is no easy way to skip all these subcarrier indices that must not
 * be displayed when showing compressed beamforming feedback matrices
 * Table 8-53g IEEE Std 802.11ac-2013 amendment.
 *
 * The irregular use of case statements in this function is to improve
 * readability in what is otherwise a large funtion that does very little.
 */
static inline int
vht_compressed_skip_scidx(guint8 nchan_width, guint8 ng, int scidx)
{
  switch(nchan_width) {
    /* 20 MHz */
    case 0:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -21: case -7: case 7: case 21:
          /* DC subcarrier */
          case 0:
            scidx++;
          default:
            break;
        }
      break;
    /* 40 MHz */
    case 1:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -53: case -25: case -11: case 11: case 25: case 53:
            scidx++;
            break;
          /* DC subcarriers */
          case -1: case 0: case 1:
            scidx = 2;
          default:
            break;
        }
      break;
    /* 80 MHz */
    case 2:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -103: case -75: case -39: case -11: case 11: case 39: case 75: case 103:
            scidx++;
            break;
          /* DC subcarriers, skip -1, 0, 1 */
          case -1:
            scidx = 2;
          default:
            break;
        }
      break;
    /* 160 MHz / 80+80 Mhz
     * Skip values here assume 160 MHz, as vht_mimo_control does not let us differentiate
     * between 160 MHz & 80-80MHz */
    case 3:
      switch (ng) {
        /* No Grouping */
        case 0:
          /* DC subcarriers, skip -5 to 5*/
          if (scidx == -5) {
            scidx = 6;
            break;
          }
          switch (scidx) {
            /* Pilot subcarriers */
            case -231: case -203: case -167: case -139: case -117: case -89: case -53: case -25:
            case 25: case 53: case 89: case 117: case 139: case 167: case 203: case 231:
              scidx++;
              break;
            /* Other subcarriers, skip -129 to -127, 127 to 129 */
            case -129:
              scidx = -126;
              break;
            case 127:
              scidx = 130;
              break;
            default:
              break;
          }
          break;
        /* Grouping of 2 */
        case 1:
          switch (scidx) {
            /* DC subcarriers */
            case -128: case -4: case -2: case 0: case 2: case 4: case 128:
              scidx++;
            default:
              break;
          }
          break;
        /* Grouping of 4 */
        case 2:
          if (scidx == -2 || scidx == 2)
            scidx++;
          break;
      }
      break;
    default:
      break;
  }

  return scidx;
}

static inline int vht_exclusive_skip_scidx(guint8 nchan_width, guint8 ng, int scidx)
{
  switch (nchan_width) {
    /* 20 MHz */
    case 0:
      switch (ng) {
        /* No Grouping */
        case 0:
          if (scidx == -2 || scidx == 1)
              scidx++;
          else
              scidx = scidx + 2;
          break;
        case 1:
          switch (scidx) {
            case -4: case 1:
              scidx = scidx + 3;
              break;
            case -1:
              scidx = 1;
              break;
            default:
              scidx = scidx + 4;
              break;
          }
          break;
        default:
          switch (scidx) {
            case -4: case 1:
              scidx = scidx + 3;
              break;
            case -1:
              scidx = 1;
              break;
            default:
              scidx = scidx + 8;
              break;
          }
          break;
      }
      break;
    /* 40 MHz */
    case 1:
    /* 80 MHz */
    case 2:
      switch (ng) {
        /* No Grouping */
        case 0:
          if (scidx == -2)
            scidx = 2;
          else
            scidx = scidx + 2;
          break;
        case 1:
          scidx = scidx + 4;
          break;
        default:
          if (scidx == -2)
            scidx = 2;
          else
            scidx = scidx + 8;
          break;
      }
      break;
    /* 160 MHz / 80+80 Mhz */
    case 3:
      switch (ng) {
        /* No Grouping */
        case 0:
          switch (scidx) {
            /* DC subcarriers, skip -4 to 4*/
            case -6:
              scidx = 6;
              break;
            /* Other subcarriers, skip -128, 128 */
            case -130:
              scidx = -126;
              break;
            case 126:
              scidx = 130;
              break;
            default:
              scidx = scidx + 2;
              break;
          }
          break;
        case 1:
          switch (scidx) {
            /* DC subcarriers, skip -4 to 4*/
            case -6:
              scidx = 6;
              break;
            default:
              scidx = scidx + 4;
              break;
          }
          break;
        default:
          switch (scidx) {
            case -6:
              scidx = 6;
              break;
            case -130:
              scidx = -126;
              break;
            case 126:
              scidx = 130;
              break;
            default:
              scidx = scidx + 8;
              break;
          }
        break;
      }
      break;
    default:
      break;
  }
  return scidx;
}

static guint
add_ff_vht_compressed_beamforming_report(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  guint32 vht_mimo;
  guint8 nc;
  guint8 nr;
  guint8 chan_width;
  guint8 grouping;
  gboolean codebook_info;
  gboolean feedback_type;
  proto_item *vht_beam_item, *vht_excl_beam_item, *phi_angle, *psi_angle;
  proto_tree *vht_beam_tree, *subtree, *vht_excl_beam_tree, *angletree;
  int i, matrix_size, len, pos, ns, scidx = 0, matrix_len;
  guint8 phi, psi, carry;
  int j, ic, off_len = 0, sscidx = 0, xnsc;
  int ir, off_pos, angle_val;
  /* Table 8-53d Order of angles in the Compressed Beamforming Feedback
   * Matrix subfield, IEEE Std 802.11ac-2013 amendment */
  static const guint8 na_arr[8][8] = { {  0,  0,  0,  0,  0,  0,  0,  0 },
                                       {  2,  2,  0,  0,  0,  0,  0,  0 },
                                       {  4,  6,  6,  0,  0,  0,  0,  0 },
                                       {  6, 10, 12, 12,  0,  0,  0,  0 },
                                       {  8, 14, 18, 20, 20,  0,  0,  0 },
                                       { 10, 18, 24, 28, 30, 30,  0,  0 },
                                       { 12, 22, 30, 36, 40, 42, 42,  0 },
                                       { 14, 26, 36, 44, 50, 54, 56, 56 }
                                     };
  /* Table 8-53g Subcarriers for which a Compressed Beamforming Feedback Matrix
   * subfield is sent back. IEEE Std 802.11ac-2013 amendment */
  static const int ns_arr[4][3] = { {  52,  30,  16 },
                                    { 108,  58,  30 },
                                    { 234, 122,  62 },
                                    { 468, 244, 124 }
                                  };

  /* Table 8-53j, no of Subcarriers for which the Delta SNR subfield is sent back to the beamformer.
   * IEEE Std 802.11ac-2013 amendment */
  static const int delta_ns_arr[4][3] = { {  30,  16,  10 },
                                          {  58,  30,  16 },
                                          { 122,  62,  32 },
                                          { 244, 124,  64 }
                                        };

  vht_mimo = tvb_get_letoh24(tvb, offset);
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_vht_mimo_cntrl,
                        ett_ff_vhtmimo_cntrl, hf_ieee80211_ff_vht_mimo_cntrl_fields, ENC_LITTLE_ENDIAN);
  offset += 3;

  /* Extract values for beamforming use */
  nc = (vht_mimo & 0x7) + 1;
  nr = ((vht_mimo & 0x38) >> 3) + 1;
  chan_width = (vht_mimo & 0xC0) >> 6;
  grouping = ((vht_mimo & 0x300) >> 8);
  codebook_info = (vht_mimo & 0x400) >> 10;
  feedback_type = (vht_mimo & 0x800) >> 11;

  if (feedback_type) {
    if (codebook_info) {
      psi = 7; phi = 9;
    } else {
      psi = 5; phi = 7;
    }
  } else {
    if (codebook_info) {
      psi = 4; phi = 6;
    } else {
      psi = 2; phi = 4;
    }
  }

  vht_beam_item = proto_tree_add_item(tree, hf_ieee80211_vht_compressed_beamforming_report, tvb,
                                  offset, -1, ENC_NA);
  vht_beam_tree = proto_item_add_subtree(vht_beam_item, ett_ff_vhtmimo_beamforming_report);

  subtree = proto_tree_add_subtree(vht_beam_tree, tvb, offset, nc,
                        ett_ff_vhtmimo_beamforming_report_snr, NULL, "Average Signal to Noise Ratio");

  for (i = 1; i <= nc; i++)
  {
    gint8 snr;
    char edge_sign;

    snr = tvb_get_gint8(tvb, offset);

    switch(snr) {
      case -128:
        edge_sign = '<';
        break;
      case 127:
        edge_sign = '>';
        break;
      default:
        edge_sign = ' ';
        break;
    }

    proto_tree_add_int_format(subtree, hf_ieee80211_vht_compressed_beamforming_report_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: %c%3.2fdB", i, edge_sign,snr/4.0+22.0);

    offset += 1;
  }

  matrix_size = na_arr[nr - 1][nc -1] * (psi + phi)/2;
  if (matrix_size % 8) {
    carry = 1;
  } else {
    carry = 0;
  }
  off_len = (matrix_size/8) + carry;
  angletree = proto_tree_add_subtree_format(vht_beam_tree, tvb, offset, off_len,
                        ett_ff_vhtmimo_beamforming_angle, NULL,"PHI and PSI Angle Decode");

  off_pos = offset*8;
  phi_angle = proto_tree_add_none_format(angletree, hf_ieee80211_vht_compressed_beamforming_phi_angle, tvb, offset, 0, "PHI(%u bits):    ", phi);
  for (ic = 1; ic <= nc; ic++) {
      for (ir = 1; ir < nr; ir++) {
          if (ir >= ic) {
              angle_val = (int) tvb_get_bits16(tvb, off_pos, phi, ENC_BIG_ENDIAN);
              if ((ir+1 < nr) || (ic+1 <= nc))
                proto_item_append_text(phi_angle, "PHI%d%d: %d, ", ir, ic, angle_val);
              else
                proto_item_append_text(phi_angle, "PHI%d%d: %d", ir, ic, angle_val);
              off_pos = off_pos + phi;
          }
      }
  }

  psi_angle = proto_tree_add_none_format(angletree, hf_ieee80211_vht_compressed_beamforming_psi_angle, tvb, offset, 0, "PSI(%u bits):    ", psi);
  for (ic = 1; ic <= nc; ic++)
      for (ir = 2; ir <= nr; ir++)
          if (ir > ic) {
              angle_val = (int) tvb_get_bits8(tvb, off_pos, psi);
              if ((ir+1 <= nr) || (ic+1 <= nc))
                proto_item_append_text(psi_angle, "PSI%d%d: %d, ", ir, ic, angle_val);
              else
                proto_item_append_text(psi_angle, "PSI%d%d: %d", ir, ic, angle_val);
              off_pos = off_pos + psi;
          }

  /* Table 8-53c Subfields of the VHT MIMO Control field (802.11ac-2013)
   * reserves value 3 of the Grouping subfield. */
  if (grouping == 3) {
    expert_add_info_format(pinfo, vht_beam_item, &ei_ieee80211_inv_val,
                           "Grouping subfield value 3 is reserved");
    return offset;
  }

  ns = ns_arr[chan_width][grouping];
  if (((matrix_size)*(ns)) % 8)
      matrix_len = (((matrix_size) * (ns)) / (8)) + 1;
  else
      matrix_len = (((matrix_size) * (ns)) / (8));
  subtree = proto_tree_add_subtree(vht_beam_tree, tvb, offset, matrix_len,
                        ett_ff_vhtmimo_beamforming_report_feedback_matrices, NULL, "Beamforming Feedback Matrix");


  switch(chan_width) {
    case 0:
      scidx = -28;
      break;
    case 1:
      scidx = -58;
      break;
    case 2:
      scidx = -122;
      break;
    case 3:
      /* This is -122 for 80+80MHz Channel Width but vht_mimo_control does not allow us
       * to differentiate between 160MHz and 80+80Mhz */
      scidx = -250;
      break;
  }

  pos = 0;
  for (i = 0; i < ns; i++) {
    if (pos % 8)
      carry = 1;
    else
      carry = 0;
    len = roundup2((pos + matrix_size), 8)/8 - roundup2(pos, 8)/8;
    scidx = vht_compressed_skip_scidx(chan_width, grouping, scidx);

    /* TODO : For certain values from na_arr, there is always going be a carry over or overflow from the previous or
       into the next octet. The largest of possible unaligned values can be 41 bytes long, and masking and shifting
       whole buffers to show correct values with padding and overflow bits is hence skipped, we only mark the bytes
       of interest instead */
    proto_tree_add_none_format(subtree, hf_ieee80211_vht_compressed_beamforming_feedback_matrix, tvb,
                                    offset - carry, len + carry, "Compressed Beamforming Feedback Matrix for subcarrier %d", scidx++);
    offset += len;
    pos += matrix_size;
  }

  if (feedback_type) {
    xnsc = delta_ns_arr[chan_width][grouping];
    if ((nc * xnsc *4) % 8)
        off_len = (nc * xnsc *4) / 8 + 1;
    else
        off_len = (nc * xnsc *4) / 8;
    switch(chan_width) {
      case 0:
        sscidx = -28;
        break;
      case 1:
        sscidx = -58;
        break;
      case 2:
        sscidx = -122;
        break;
      case 3:
        sscidx = -250;
        break;
    }
    vht_excl_beam_item = proto_tree_add_item(tree, hf_ieee80211_vht_mu_exclusive_beamforming_report, tvb, offset, off_len, ENC_NA);
    vht_excl_beam_tree = proto_item_add_subtree(vht_excl_beam_item, ett_ff_vhtmu_exclusive_beamforming_report_matrices);

    carry = 1;
    for (j = 1; j <= xnsc; j++) {
      for (ic = 1; ic <= nc; ic++) {
        if (carry % 2){
          pos = 0;
          len = 1;
        }
        else
        {
          pos = 1;
          len = 0;
        }
        proto_tree_add_none_format(vht_excl_beam_tree, hf_ieee80211_vht_mu_exclusive_beamforming_delta_snr, tvb,
                                      offset - pos, 1, "Delta SNR for space-time stream %d for subcarrier %d", ic, sscidx);
        offset += len;
        carry ++;
      }
      sscidx = vht_exclusive_skip_scidx(chan_width, grouping, sscidx);
    }
  }

  return offset;
}

static guint
add_ff_action_vht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;
  guint8 vht_action, field_val;
  guint64 msa_value;
  guint64 upa_value;
  int m;
  proto_item *ti;
  proto_tree *ti_tree;
  proto_item *msa, *upa;
  proto_tree *msa_tree, *upa_tree;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  vht_action = tvb_get_guint8(tvb, offset);
  offset += add_ff_vht_action(tree, tvb, pinfo, offset);

  switch(vht_action){
    case VHT_ACT_VHT_COMPRESSED_BEAMFORMING:{
      offset = add_ff_vht_compressed_beamforming_report(tree, tvb, pinfo, offset);
      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    case VHT_ACT_GROUP_ID_MANAGEMENT:{
      ti = proto_tree_add_item(tree, hf_ieee80211_vht_group_id_management, tvb,
                          offset, -1, ENC_NA);
      ti_tree = proto_item_add_subtree(ti, ett_vht_grpidmgmt);

      msa_value = tvb_get_letoh64(tvb, offset);
      msa = proto_tree_add_item(ti_tree, hf_ieee80211_vht_membership_status_array, tvb,
                                offset, 8, ENC_NA);
      msa_tree = proto_item_add_subtree(msa, ett_vht_msa);
      for (m = 0; m < 64; m++) {
          if (msa_value & (G_GINT64_CONSTANT(1) << m))
              proto_tree_add_uint_format(msa_tree, hf_ieee80211_vht_membership_status_field,
                                               tvb, offset + (m/8), 1, 1, "Membership Status in Group ID %d: 1", m);
      }
      offset += 8;

      upa = proto_tree_add_item(ti_tree, hf_ieee80211_vht_user_position_array, tvb,
                                      offset, 16, ENC_NA);
      upa_tree = proto_item_add_subtree(upa, ett_vht_upa);

      upa_value = tvb_get_letoh64(tvb, offset);
      for (m = 0; m < 32; m++) {
          if (msa_value & (G_GINT64_CONSTANT(1) << m)) {
              field_val = (guint8) ((upa_value >> m*2) & 0x3);
              proto_tree_add_uint_format(upa_tree, hf_ieee80211_vht_user_position_field,
                                               tvb, offset + (m/4), 1, field_val, "User Position in Group ID %d: %u", m, field_val);
          }
      }
      upa_value = tvb_get_letoh64(tvb, offset+8);
      for (m = 0; m < 32; m++) {
          if (msa_value & (G_GINT64_CONSTANT(1) << (32+m))) {
              field_val = (guint8) ((upa_value >> m*2) & 0x3);
              proto_tree_add_uint_format(upa_tree, hf_ieee80211_vht_user_position_field,
                                               tvb, (offset + 8) + (m/4), 1, field_val, "User Position in Group ID %d: %u", m, field_val);
          }
      }
      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    case VHT_ACT_OPERATION_MODE_NOTIFICATION:{
      ti = proto_tree_add_item(tree, hf_ieee80211_vht_operation_mode_notification, tvb,
                          offset, -1, ENC_NA);
      expert_add_info(pinfo, ti, &ei_ieee80211_vht_action);
      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    default:
    break;
  }


  return offset - start;
}

static guint
add_ff_s1g_twt_setup(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  const guint8 ids[] = { TAG_TWT };
  guint start = offset;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  offset += add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL);

  return offset - start;
}

static guint
add_ff_s1g_twt_teardown(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 twt_flow_id = tvb_get_guint8(tvb, offset);
  static const int *ieee80211_twt_individual_flow[] = {
    &hf_ieee80211_twt_individual_flow_id,
    &hf_ieee80211_twt_neg_type,
    NULL,
  };
  static const int *ieee80211_twt_bcast_flow[] = {
    &hf_ieee80211_twt_bcast_id,
    &hf_ieee80211_twt_neg_type,
    NULL,
  };

  // Bits 5 and 6 are the negotiation type - See ieee80211.ax/D3.0 9.6.25.9
  switch ((twt_flow_id & 0x60) >> 5) {
    case 3:
      // According to 11ax, first 5 bits are the BCAST TWT flow ID
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                        hf_ieee80211_twt_bcast_flow,
                                        ett_twt_tear_down_tree, ieee80211_twt_bcast_flow,
                                        ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
    break;
    case 0:
    case 1:
      // According to 11ah / 11ax, first 3 bits are the UCAST TWT flow ID
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                        hf_ieee80211_twt_individual_flow,
                                        ett_twt_tear_down_tree, ieee80211_twt_individual_flow,
                                        ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
      break;
    default:
      proto_tree_add_expert(tree, pinfo, &ei_ieee80211_twt_tear_down_bad_neg_type,
                            tvb, offset, tvb_reported_length_remaining(tvb, offset));
  }

  // The TWT Flow ID size
  return 1;
}

static guint
add_ff_action_s1g(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;
  guint8 s1g_action;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  s1g_action = tvb_get_guint8(tvb, offset);
  offset += add_ff_s1g_action(tree, tvb, pinfo, offset);

  switch(s1g_action) {
    case S1G_ACT_AID_SWITCH_REQUEST:
      // TODO
    break;
    case S1G_ACT_AID_SWITCH_RESPONSE:
      // TODO
    break;
    case S1G_ACT_SYNC_CONTROL:
      // TODO
    break;
    case S1G_ACT_STA_INFO_ANNOUNCE:
      // TODO
    break;
    case S1G_ACT_EDCA_PARAM_SET:
      // TODO
    break;
    case S1G_ACT_EL_OPERATION:
      // TODO
    break;
    case S1G_ACT_TWT_SETUP:
      offset += add_ff_s1g_twt_setup(tree, tvb, pinfo, offset);
    break;
    case S1G_ACT_TWT_TEARDOWN:
      offset += add_ff_s1g_twt_teardown(tree, tvb, pinfo, offset);
    break;
    case S1G_ACT_SECT_GROUP_ID_LIST:
      // TODO
    break;
    case S1G_ACT_SECT_ID_FEEDBACK:
      // TODO
    break;
    case S1G_ACT_RESERVED:
      // TODO
    break;
    case S1G_ACT_TWT_INFORMATION:
      // TODO
    break;
    default:
    break;
  }

  return offset - start;
}

#define HE_COMPRESSED_BEAMFORMING_AND_CQI 0
#define HE_QUIET_TIME_PERIOD              1

static const range_string he_action_rvals[] = {
  { HE_COMPRESSED_BEAMFORMING_AND_CQI, HE_COMPRESSED_BEAMFORMING_AND_CQI,
        "HE Compressed Beamforming And CQI" },
  { HE_QUIET_TIME_PERIOD, HE_QUIET_TIME_PERIOD,
        "Quiet Time Period" },
  { HE_QUIET_TIME_PERIOD + 1, 255, "Reserved" },
  { 0, 0, NULL }
};

#define HE_BSS_COLOR_CHANGE_ANNOUNCEMENT  0

static const range_string protected_he_action_rvals[] = {
  { HE_BSS_COLOR_CHANGE_ANNOUNCEMENT, HE_BSS_COLOR_CHANGE_ANNOUNCEMENT,
        "HE BSS Color Change Announcement" },
  { HE_BSS_COLOR_CHANGE_ANNOUNCEMENT + 1, 255, "Reserved" },
  { 0, 0, NULL }
};

/*
 * This currently only works for SU, 20MHz, 40MHz and 80MHz and grouping 4 and 16.
 */
struct scidx_start_end {
  int start;
  int end;
};

#define N_SCIDX_20MHZ_NG4 9
static const struct scidx_start_end scidx_20MHz_Ng4[N_SCIDX_20MHZ_NG4] = {
  { -122, -96 },
  { -96, -68 },
  { -68, -40 },
  { -44, -16 },
  { -16, 16 },
  { 16, 44 },
  { 40, 68 },
  { 68, 96 },
  { 96, 122 }
};

#define N_SCIDX_20MHZ_NG16 9
static const struct scidx_start_end scidx_20MHz_Ng16[9] = {
  { -122, -84 },
  { -96, -64 },
  { -80, -32 },
  { -52, -4 },
  { -20, 20 },
  { 4, 52 },
  { 32, 80 },
  { 64, 96 },
  { 84, 122 }
};

#define N_SCIDX_40MHZ_NG4 18
static const struct scidx_start_end scidx_40MHz_Ng4[N_SCIDX_40MHZ_NG4] = {
  { -500 + 256, -472 + 256 },
  { -476 + 256, -448 + 256 },
  { -488 + 256, -420 + 256 },
  { -420 + 256, -392 + 256 },
  { -392 + 256, -364 + 256 },
  { -368 + 256, -340 + 256 },
  { -340 + 256, -312 + 256 },
  { -312 + 256, -284 + 256 },
  { -288 + 256, -260 + 256 },
  {  260 - 256,  288 - 256 },
  {  284 - 256,  312 - 256 },
  {  312 - 256,  340 - 256 },
  {  340 - 256,  368 - 256 },
  {  364 - 256,  392 - 256 },
  {  392 - 256,  420 - 256 },
  {  420 - 256,  448 - 256 },
  {  448 - 256,  476 - 256 },
  {  472 - 256,  500 - 256 }
};

#define N_SCIDX_40MHZ_NG16 18
static const struct scidx_start_end scidx_40MHz_Ng16[N_SCIDX_40MHZ_NG16] = {
  { -500 + 256, -468 + 256 },
  { -484 + 256, -436 + 256 },
  { -452 + 256, -420 + 256 },
  { -420 + 256, -388 + 256 },
  { -404 + 256, -356 + 256 },
  { -372 + 256, -340 + 256 },
  { -340 + 256, -308 + 256 },
  { -324 + 256, -276 + 256 },
  { -292 + 256, -260 + 245 },
  {  260 - 256,  292 - 256 },
  {  276 - 256,  324 - 256 },
  {  308 - 256,  340 - 256 },
  {  340 - 256,  322 - 256 },
  {  356 - 256,  404 - 256 },
  {  388 - 256,  420 - 256 },
  {  420 - 256,  452 - 256 },
  {  436 - 256,  484 - 256 },
  {  468 - 256,  500 - 256 }
};

#define N_SCIDX_80MHZ_NG4 37
static const struct scidx_start_end scidx_80MHz_Ng4[N_SCIDX_80MHZ_NG4] = {
  { -500, -472 },
  { -476, -448 },
  { -448, -420 },
  { -420, -392 },
  { -392, -364 },
  { -368, -340 },
  { -340, -312 },
  { -312, -284 },
  { -288, -260 },
  { -260, -232 },
  { -232, -204 },
  { -204, -176 },
  { -180, -152 },
  { -152, -124 },
  { -124,  -96 },
  { -100,  -72 },
  {  -72,  -44 },
  {  -44,  -16 },
  {  -16,   16 },
  {   16,   44 },
  {   44,   72 },
  {   72,  100 },
  {   96,  124 },
  {  124,  152 },
  {  152,  180 },
  {  176,  204 },
  {  204,  232 },
  {  232,  260 },
  {  260,  288 },
  {  284,  312 },
  {  312,  340 },
  {  340,  368 },
  {  364,  392 },
  {  392,  420 },
  {  420,  448 },
  {  448,  476 },
  {  472,  500 }
};

#define N_SCIDX_80MHZ_NG16 37
static const struct scidx_start_end scidx_80MHz_Ng16[N_SCIDX_80MHZ_NG16] = {
  { -500, -468 },
  { -484, -436 },
  { -452, -420 },
  { -420, -388 },
  { -404, -356 },
  { -372, -340 },
  { -340, -308 },
  { -324, -276 },
  { -292, -260 },
  { -260, -228 },
  { -244, -196 },
  { -212, -164 },
  { -180, -148 },
  { -164, -116 },
  { -132,  -84 },
  { -100,  -68 },
  {  -84,  -36 },
  {  -52,   -4 },
  {  -20,   20 },
  {    4,   52 },
  {   36,   84 },
  {   68,  100 },
  {   84,  132 },
  {  116,  164 },
  {  148,  180 },
  {  164,  212 },
  {  196,  244 },
  {  228,  260 },
  {  260,  292 },
  {  276,  324 },
  {  308,  340 },
  {  340,  372 },
  {  356,  404 },
  {  388,  420 },
  {  420,  452 },
  {  436,  484 },
  {  468,  500 },
};

#define SU_FEEDBACK (0)
#define MU_FEEDBACK (1)
#define CQI_FEEDBACK (2)
#define RESERVED_FEEDBACK (3)

#define BW_20MHz (0)
#define BW_40MHz (1)
#define BW_80MHz (2)
#define BW_160MHz (3)

#define SCIDX_END_SENTINAL (0x80000000)

static int
next_he_scidx(int scidx, int bw _U_, int grouping _U_, int feedback _U_,
              int ru_start_index, int ru_end_index)
{
  int incr = 4;

  /*
   * We need to check the correct bw value to determine if we have hit
   * the end of the range of SCIDXes.
   */
  switch (bw) {
  case BW_20MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_20MHZ_NG4 ||
          scidx == scidx_20MHz_Ng4[ru_end_index].end)  /* we returned the max */
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_20MHZ_NG16 ||
          scidx == scidx_20MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_40MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_40MHZ_NG4 ||
          scidx == scidx_40MHz_Ng4[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_40MHZ_NG16 ||
          scidx == scidx_40MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_80MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_80MHZ_NG4 ||
          scidx == scidx_80MHz_Ng4[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_80MHZ_NG16 ||
          scidx == scidx_80MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_160MHz:
    return SCIDX_END_SENTINAL;
  }

  /*
   * Check if this is the first time though and figure out the starting
   * SCIDX.
   */
  if (scidx == (int)SCIDX_END_SENTINAL)
    switch (bw) {
    case BW_20MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_20MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_20MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_20MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_20MHz_Ng16[ru_start_index].start;
      }
    case BW_40MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_40MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_40MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_40MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_40MHz_Ng16[ru_start_index].start;
      }
    case BW_80MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_80MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_80MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_80MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_80MHz_Ng16[ru_start_index].start;
      }
    case BW_160MHz:
      return SCIDX_END_SENTINAL;
  }

  /*
   * for BW_20MHz it is more complex, and the start and end sets have an
   * increment of 2, and around zero they go -4, -2, 2, 4 as well.
   */
  if (bw == BW_20MHz) {
    if (grouping == 0) {
      if (scidx == -122)
        return -120;
      if (scidx == -4)
        return -2;
      if (scidx == 2)
        return 4;
      if (scidx == 120)
        return 122;
    } else {
      if (scidx == -122)
        return -116;
      if (scidx == -4)
        return -2;
      if (scidx == -2)
        return 2;
      if (scidx == 2)
        return 4;
      if (scidx == 116)
        return 122;
    }
  }

  if (grouping == 1)
    incr = 16;

  scidx += incr;

  if (scidx == 0)  /* Not sure if this is needed */
    scidx += incr;

  return scidx;
}

/*
 * This might have a problem if there are not enough bits in the TVB.
 * Will only handle a limited number of bits.
 */
static guint16
he_get_bits(tvbuff_t *tvb, int bit_offset, int bit_len)
{
  guint32 bits;
  int byte_offset = bit_offset / 8;
  int bit_start = bit_offset % 8;
  int bit_mask = (1 << bit_len) - 1;  /* Select that many bits */
  int remaining_length = tvb_reported_length_remaining(tvb, byte_offset);

  if (remaining_length >= 3)
    bits = tvb_get_letoh24(tvb, byte_offset);
  else if (remaining_length == 2)
    bits = tvb_get_letohs(tvb, byte_offset);
  else
    bits = tvb_get_guint8(tvb, byte_offset);

  bits = bits >> bit_start;

  return bits & bit_mask;
}

static int
dissect_he_feedback_matrix(proto_tree *tree, tvbuff_t *tvb, int offset,
                           int bit_offset, int scidx, int nr, int nc,
                           int phi_bits, int psi_bits)
{
  int ri, ci;
  proto_item *pi = NULL;
  int start_bit_offset = bit_offset;

  if (nc == nr)  /* If they are the same, reduce Nc by one */
    nc -= 1;

  /*
 *    * Figure out how many bits we need
 *       */
  for (ci = 1; ci <= nc; ci++) {
    for (ri = ci; ri < nr; ri++) {
      bit_offset += phi_bits;
    }
    for (ri = ci + 1; ri <= nr; ri++) {
      bit_offset += psi_bits;
    }
  }

  pi = proto_tree_add_int_format(tree, hf_ieee80211_he_compressed_beamform_scidx,
                tvb, offset, ((bit_offset + 7) / 8) - offset, scidx,
                "SCIDX: %d", scidx);

  /* Reset to the start bit offset */
  bit_offset = start_bit_offset;

  for (ci = 1; ci <= nc; ci++) {
    for (ri = ci; ri < nr; ri++) {
      int angle = he_get_bits(tvb, bit_offset, phi_bits);
      proto_item_append_text(pi, ", phi%d%d:%d", ri, ci, angle);
      bit_offset += phi_bits;
    }
    for (ri = ci + 1; ri <= nr; ri++) {
      int angle = he_get_bits(tvb, bit_offset, psi_bits);
      proto_item_append_text(pi, ", psi%d%d:%d", ri, ci, angle);
      bit_offset += psi_bits;
    }
  }

  return bit_offset;
}

static const int *he_mimo_control_headers[] = {
  &hf_he_mimo_control_nc_index,
  &hf_he_mimo_control_nr_index,
  &hf_he_mimo_control_bw,
  &hf_he_mimo_control_grouping,
  &hf_he_mimo_control_codebook_info,
  &hf_he_mimo_control_feedback_type,
  &hf_he_mimo_control_remaining_feedback_segs,
  &hf_he_mimo_control_first_feedback_seg,
  &hf_he_mimo_control_ru_start_index,
  &hf_he_mimo_control_ru_end_index,
  &hf_he_mimo_control_sounding_dialog_token_num,
  &hf_he_mimo_control_reserved,
  NULL
};

/*
 * Handle compressed beamforming matrices and CQI
 */
static guint
dissect_compressed_beamforming_and_cqi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int byte_count = 0;
  guint64 mimo_cntl = tvb_get_letoh40(tvb, offset);
  int nc = 0, nr = 0, i;
  int bw, grouping, codebook, feedback, bit_offset, scidx;
  int phi_bits = 0, psi_bits = 0;
  proto_tree *snr_tree = NULL, *feedback_tree = NULL;
  int start_offset;
  int ru_start_index, ru_end_index;

  nc = (int)((mimo_cntl & 0x07) + 1);
  nr = (int)(((mimo_cntl >> 3) & 0x07) + 1);
  bw = (int)((mimo_cntl >> 6) & 0x03);
  grouping = (int)((mimo_cntl >> 8) & 0x01);
  codebook = (int)((mimo_cntl >> 9) & 0x01);
  feedback = (int)((mimo_cntl >> 10) & 0x03);
  ru_start_index = (int)((mimo_cntl >> 16) & 0x7F);
  ru_end_index = (int)((mimo_cntl >> 23) & 0x7F);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
    hf_ieee80211_he_mimo_control_field, ett_ff_he_mimo_control,
    he_mimo_control_headers, ENC_LITTLE_ENDIAN,
    BMT_NO_APPEND);
  offset += 5;

  snr_tree = proto_tree_add_subtree(tree, tvb, offset, nc,
                ett_ff_he_mimo_beamforming_report_snr, NULL,
                "Average Signal to Noise Ratio");

  for (i = 0; i < nc; i++) {
    gint8 snr = tvb_get_guint8(tvb, offset);

    proto_tree_add_uint_format(snr_tree,
        hf_ieee80211_he_compressed_beamforming_report_snr, tvb, offset, 1,
        snr, "Stream %d: %s%0.2fdB (0x%02x)", i, (snr == 127 ? ">=" :
                                                  (snr == -128 ? "<=" : "")),
                                             (float)((float)88 + snr)/4,
                                             (guint8)snr);
    offset++;
  }

  /*
   * The rest of the data consists of the compressed beamforming matrices, one
   * for each SCIDX per group. Each matrix consists of phi and psi angles
   * encoded using the number of bits specified using the codebook field.
   *
   * The matrices contain a number entries related to Nr -1 & Nc except when
   * Nr == Nc, and then it is Nr -1 x Nc -1, with Nr - 1 phi angles, Nc - 1
   * psi angles, Nr - 2 phi angles, Nc - 2 psi angles ...
   */
  if (feedback == 0) {  /* SU */
    if (codebook == 0) {
      phi_bits = 2; psi_bits = 4;
    } else {
      phi_bits = 4; psi_bits = 6;
    }
  } else if (feedback == 1) { /* MU */
    if (grouping == 1) {
      phi_bits = 9; psi_bits = 7;
    } else {
      if (codebook == 0) {
        phi_bits = 5; psi_bits = 7;
      } else {
        phi_bits = 7; psi_bits = 9;
      }
    }
  }  /* DO something about CQI etc. */

  feedback_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_ff_he_mimo_feedback_matrices, NULL,
                        "Feedback Matrices");

  start_offset = offset;
  bit_offset = offset * 8;
  scidx = SCIDX_END_SENTINAL;
  while ((scidx = next_he_scidx(scidx, bw, grouping, feedback,
          ru_start_index, ru_end_index)) != (int)SCIDX_END_SENTINAL) {
    int prev_bit_offset = bit_offset;
    bit_offset = dissect_he_feedback_matrix(feedback_tree, tvb, offset,
                        bit_offset, scidx, nr, nc, phi_bits, psi_bits);
    if (bit_offset <= prev_bit_offset) {
      expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
      break;
    }

    offset = bit_offset / 8;
  }

  offset = (bit_offset + 7) / 8;
  proto_item_set_len(feedback_tree, offset - start_offset);

  /* Sometimes the FCS is in the buffer as well ... */
  byte_count = tvb_reported_length_remaining(tvb, offset);
  if (byte_count > 0)
    offset += byte_count; /* Should fix the real problem */

  return offset;
}

enum quiet_time_period_control {
  QTP_CONTROL_SETUP = 0,
  QTP_CONTROL_REQUEST = 1,
  QTP_CONTROL_RESPONSE = 2,
};

static const range_string quiet_time_period_control_rvals[] = {
  { QTP_CONTROL_SETUP,    QTP_CONTROL_SETUP,    "Quiet Time Period Setup" },
  { QTP_CONTROL_REQUEST,  QTP_CONTROL_REQUEST,  "Quiet Time Period Request" },
  { QTP_CONTROL_RESPONSE, QTP_CONTROL_RESPONSE, "Quiet Time Period Response" },
  { QTP_CONTROL_RESPONSE +1 , 255, "Reserved" },
  { 0, 0, NULL }
};

static void
dissect_bss_color_change(tvbuff_t *tvb, packet_info *pinfo _U_,
proto_tree *tree, int offset, int len _U_);

static guint
dissect_quiet_time_period(tvbuff_t *tvb, packet_info *pinfo _U_,
proto_tree *tree, int offset, int len _U_)
{
  guint8 control = tvb_get_guint8(tvb, offset);

  proto_tree_add_item(tree, hf_he_qtp_control, tvb, offset, 1, ENC_NA);
  offset += 1;

  switch (control)
  {
  case QTP_CONTROL_SETUP:
    proto_tree_add_item(tree, hf_he_qtp_setup_quiet_period_duration, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_setup_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  case QTP_CONTROL_REQUEST:
    proto_tree_add_item(tree, hf_he_qtp_request_dialog_token, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_he_qtp_request_quiet_period_offset, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_request_quiet_period_duration, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_he_qtp_request_quiet_period_interval, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_request_repetition_count, tvb, offset, 1,
                ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_request_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  case QTP_CONTROL_RESPONSE:
    proto_tree_add_item(tree, hf_he_qtp_response_dialog_token, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_he_qtp_response_status_code, tvb, offset, 1,
                ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_response_quiet_period_offset, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_response_quiet_period_duration, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_he_qtp_response_quiet_period_interval, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_response_repetition_count, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_he_qtp_response_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  default:
    /* Reserved */
    break;
  }

  return offset;
}

static guint
add_ff_action_he(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;
  guint8 he_action;
  guint8 length;
  //guint8 elem_id, elem_id_extension;
  proto_item *item;
  proto_tree *subtree;
  unsigned int len = tvb_reported_length_remaining(tvb, offset);

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  he_action = tvb_get_guint8(tvb, offset);

  item = proto_tree_add_item(tree, hf_ieee80211_ff_he_action, tvb, offset, 1, ENC_NA);
  offset += 1;

  subtree = proto_item_add_subtree(item, ett_ff_he_action);


  switch (he_action) {
  case HE_COMPRESSED_BEAMFORMING_AND_CQI:
    proto_tree_add_uint_format(subtree, hf_ieee80211_beamforming_report_len,
        tvb, offset, 0, len, "Total length: %u", len);
    offset = dissect_compressed_beamforming_and_cqi(subtree, tvb, pinfo, offset);
    break;

  case HE_QUIET_TIME_PERIOD:
    //elem_id = tvb_get_guint8(tvb, offset);
    length = tvb_get_guint8(tvb, offset + 1);
    //elem_id_extension = tvb_get_guint8(tvb, offset + 2);
    /* Should check following condition?
     * elem_id == TAG_ELEMENT_ID_EXTENSION && elem_id_extension == ETAG_QUIET_TIME_PERIOD
     */
    offset = dissect_quiet_time_period(tvb, pinfo, subtree, offset + 3, length);
    break;

  default:
    break;
  }
  return offset - start;
}

static guint
add_ff_action_protected_he(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint start = offset;
  guint8 protected_he_action;
  guint8 length;
  //guint8 elem_id, elem_id_extension;
  proto_item *item;
  proto_tree *subtree;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  protected_he_action = tvb_get_guint8(tvb, offset);

  item = proto_tree_add_item(tree, hf_ieee80211_ff_protected_he_action, tvb, offset,
        1, ENC_NA);
  offset += 1;

  subtree = proto_item_add_subtree(item, ett_ff_protected_he_action);


  switch (protected_he_action) {
  case HE_BSS_COLOR_CHANGE_ANNOUNCEMENT:
    //elem_id = tvb_get_guint8(tvb, offset);
    length = tvb_get_guint8(tvb, offset + 1);
    //elem_id_extension = tvb_get_guint8(tvb, offset + 2);
    /* Should check following condition?
     * elem_id == TAG_ELEMENT_ID_EXTENSION && elem_id_extension == ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT
     */
    dissect_bss_color_change(tvb, pinfo, subtree, offset + 3, length);
    offset += 5;
    break;

  default:
    break;
  }

  return offset - start;
}

static guint
add_ff_action_fst(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_fst_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case FST_SETUP_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_llt(tree, tvb, pinfo, offset);
      break;
    case FST_SETUP_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case FST_TEAR_DOWN:
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ACK_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ACK_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ON_CHANNEL_TUNNEL_REQUEST:
      offset += add_ff_oct_mmpdu(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

static guint
add_ff_action_dmg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  guint8 code;
  guint  start = offset;
  int left_offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_guint8(tvb, offset);
  offset += add_ff_dmg_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case DMG_ACTION_PWR_SAVE_CONFIG_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_dmg_pwr_mgmt(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_PWR_SAVE_CONFIG_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_INFO_REQ:
      offset += add_ff_subject_address(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_INFO_RES:
      offset += add_ff_subject_address(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_HANDOVER_REQ:
      offset += add_ff_handover_reason(tree, tvb, pinfo, offset);
      offset += add_ff_handover_remaining_bi(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_HANDOVER_RES:
      offset += add_ff_handover_result(tree, tvb, pinfo, offset);
      offset += add_ff_handover_reject_reason(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_DTP_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_DTP_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_SEARCH_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_destination_reds_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_SEARCH_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      left_offset =
          tvb_reported_length_remaining(tvb, offset);
      while(left_offset > 0) {
        proto_tree_add_item(tree, hf_ieee80211_ff_peer_sta_aid, tvb, offset, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_snr, tvb, offset+1, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_internal_angle, tvb, offset+2, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_recommend, tvb, offset+2, 1, ENC_NA);
        /* another reserved byte */
        offset += 4;
        left_offset -= 4;
      }
      break;
    case DMG_ACTION_RLS_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_realy_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_ANNOUNCE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_realy_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_TEARDOWN:
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_realy_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_ACK_REQ:
    case DMG_ACTION_RELAY_ACK_RES:
      break;
    case DMG_ACTION_TPA_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_timing_offset(tree, tvb, pinfo, offset);
      offset += add_ff_sampling_frequency_offset(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_TPA_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_TPA_REP:
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_ROC_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_relay_operation_type(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_ROC_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

static guint
add_ff_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  switch (tvb_get_guint8(tvb, offset) & 0x7f) {
  case CAT_SPECTRUM_MGMT: /* 0 */
    return add_ff_action_spectrum_mgmt(tree, tvb, pinfo, offset);
  case CAT_QOS: /* 1 */
    return add_ff_action_qos(tree, tvb, pinfo, offset);
  case CAT_DLS: /* 2 */
    return add_ff_action_dls(tree, tvb, pinfo, offset);
  case CAT_BLOCK_ACK: /* 3 */
    return add_ff_action_block_ack(tree, tvb, pinfo, offset);
  case CAT_PUBLIC: /* 4 */
    return add_ff_action_public(tree, tvb, pinfo, offset);
  case CAT_RADIO_MEASUREMENT: /* 5 */
    return add_ff_action_radio_measurement(tree, tvb, pinfo, offset);
  case CAT_FAST_BSS_TRANSITION: /* 6 */
    return add_ff_action_fast_bss_transition(tree, tvb, pinfo, offset);
  case CAT_HT: /* 7 */
    return add_ff_action_ht(tree, tvb, pinfo, offset);
  case CAT_SA_QUERY: /* 8 */
    return add_ff_action_sa_query(tree, tvb, pinfo, offset);
  case CAT_PUBLIC_PROTECTED: /* 9 */
    return add_ff_action_protected_public(tree, tvb, pinfo, offset);
  case CAT_WNM: /* 10 */
    return add_ff_action_wnm(tree, tvb, pinfo, offset);
  case CAT_UNPROTECTED_WNM: /* 11 */
    return add_ff_action_unprotected_wnm(tree, tvb, pinfo, offset);
  case CAT_TDLS: /* 12 */
    return add_ff_action_tdls(tree, tvb, pinfo, offset);
  case CAT_MESH: /* 13 */
    return add_ff_action_mesh(tree, tvb, pinfo, offset);
  case CAT_MULTIHOP: /* 14 */
    return add_ff_action_multihop(tree, tvb, pinfo, offset);
  case CAT_SELF_PROTECTED: /* 15 */
    return add_ff_action_self_protected(tree, tvb, pinfo, offset);
  case CAT_DMG: /* 16 */
    return add_ff_action_dmg(tree, tvb, pinfo, offset);
  case CAT_MGMT_NOTIFICATION:  /* Management notification frame - 17 */
    return add_ff_action_mgmt_notification(tree, tvb, pinfo, offset);
  case CAT_FAST_SESSION_TRANSFER: /* 18 */
    return add_ff_action_fst(tree, tvb, pinfo, offset);
/* case CAT_ROBUST_AV_STREAMING:  19 */
/*   return add_ff_action_robust_av_streaming(tree, tvb, pinfo, offset); */
  case CAT_UNPROTECTED_DMG: /* 20 */
    return add_ff_action_unprotected_dmg(tree, tvb, pinfo, offset);
  case CAT_VHT: /* 21 */
    return add_ff_action_vht(tree, tvb, pinfo, offset);
  case CAT_S1G: /* 22 */
    return add_ff_action_s1g(tree, tvb, pinfo, offset);
  case CAT_HE:
    return add_ff_action_he(tree, tvb, pinfo, offset);
  case CAT_PROTECTED_HE:
    return add_ff_action_protected_he(tree, tvb, pinfo, offset);
/*  case CAT_VENDOR_SPECIFIC_PROTECTED:   Vendor Specific Protected Category - 126 */
/*    return add_ff_action_vendor_specific_protected(tree, tvb, pinfo, offset);*/
  case CAT_VENDOR_SPECIFIC:  /* Vendor Specific Protected Category - 127 */
    return add_ff_action_vendor_specific(tree, tvb, pinfo, offset);
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 1;
  }
}

static const value_string ieee80211_rsn_cipher_vals[] = {
  {0, "NONE"},
  {1, "WEP (40-bit)"},
  {2, "TKIP"},
  {3, "AES (OCB)"},
  {4, "AES (CCM)"},
  {5, "WEP (104-bit)"},
  {6, "BIP (128)"},
  {7, "Group addressed traffic not allowed"},
  {8, "GCMP (128)" },
  {9, "GCMP (256)" },
  {10, "CCMP (256)" },
  {11, "BIP (GMAC-128)" },
  {12, "BIP (GMAC-256)" },
  {13, "BIP (CMAC-256)" },
  {0, NULL}
};

static const value_string ieee80211_rsn_keymgmt_vals[] = {
  {0, "NONE"},
  {1, "WPA"},
  {2, "PSK"},
  {3, "FT over IEEE 802.1X"},
  {4, "FT using PSK"},
  {5, "WPA (SHA256)"},
  {6, "PSK (SHA256)"},
  {7, "TDLS / TPK Handshake (SHA256)"},
  {8, "SAE (SHA256)" },
  {9, "FT using SAE (SHA256)" },
  {10, "APPeerKey (SHA256)" },
  {11, "WPA (SHA256-SuiteB)" },
  {12, "WPA (SHA384-SuiteB)" },
  {13, "FT over IEEE 802.1X (SHA384)" },
  {14, "FILS (SHA256 and AES-SIV-256)" },
  {15, "FILS (SHA384 and AES-SIV-512)" },
  {16, "FT over FILS (SHA256 and AES-SIV-256)" },
  {17, "FT over FILS (SHA384 and AES-SIV-512)" },
  {18, "Opportunistic Wireless Encryption"},
  {0, NULL}
};

static void
oui_base_custom(gchar *result, guint32 oui)
{
  guint8       p_oui[3];
  const gchar *manuf_name;

  p_oui[0] = oui >> 16 & 0xFF;
  p_oui[1] = oui >> 8 & 0xFF;
  p_oui[2] = oui & 0xFF;

  /* Attempt an OUI lookup. */
  manuf_name = uint_get_manuf_name_if_known(oui);
  if (manuf_name == NULL) {
    /* Could not find an OUI. */
    g_snprintf(result, ITEM_LABEL_LENGTH, "%02x:%02x:%02x", p_oui[0], p_oui[1], p_oui[2]);
  }
  else {
   /* Found an address string. */
    g_snprintf(result, ITEM_LABEL_LENGTH, "%02x:%02x:%02x (%s)", p_oui[0], p_oui[1], p_oui[2], manuf_name);
  }
}

static void
rsn_gcs_base_custom(gchar *result, guint32 gcs)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, gcs >> 8);
  tmp_str = val_to_str_wmem(NULL, gcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
rsn_pcs_base_custom(gchar *result, guint32 pcs)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, pcs >> 8);
  tmp_str = val_to_str_wmem(NULL, pcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);

}
static void
rsn_akms_base_custom(gchar *result, guint32 akms)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, akms >> 8);
  tmp_str = val_to_str_wmem(NULL, akms & 0xFF, ieee80211_rsn_keymgmt_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static gchar *
rsn_pcs_return(guint32 pcs)
{
  gchar *result;

  result = (gchar *)wmem_alloc(wmem_packet_scope(), SHORT_STR);
  result[0] = '\0';
  rsn_pcs_base_custom(result, pcs);

  return result;
}

static gchar *
rsn_akms_return(guint32 akms)
{
  gchar *result;

  result = (gchar *)wmem_alloc(wmem_packet_scope(), SHORT_STR);
  result[0] = '\0';
  rsn_akms_base_custom(result, akms);

  return result;
}

static void
rsn_gmcs_base_custom(gchar *result, guint32 gmcs)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, gmcs >> 8);
  tmp_str = val_to_str_wmem(NULL, gmcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
rsni_base_custom(gchar *result, guint32 rsni)
{
  double temp_double;

  temp_double = (double)rsni;
  g_snprintf(result, ITEM_LABEL_LENGTH, "%f dB", (temp_double / 2));
}

static void
vht_tpe_custom(gchar *result, guint8 txpwr)
{
  gint8 txpwr_db;

  txpwr_db = (gint8)(txpwr);
  g_snprintf(result, ITEM_LABEL_LENGTH, "%3.1f dBm", (txpwr_db/2.0));
}

static void
channel_number_custom(gchar *result, guint8 channel_number)
{
  switch(channel_number){
    case 0:
      g_snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on all supported channels in the specified Operating Class)", channel_number);
    break;
    case 255:
      g_snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on all supported channels listed in the AP Channel Report)", channel_number);
    break;
    default :
      g_snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on that Channel Number)", channel_number);
    break;
  }
}

/* WPA / WME */
static const value_string ieee802111_wfa_ie_type_vals[] = {
  { 1, "WPA Information Element" },
  { 2, "WMM/WME" },
  { 4, "WPS" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wpa_cipher_vals[] = {
  { 0, "NONE" },
  { 1, "WEP (40-bit)" },
  { 2, "TKIP" },
  { 3, "AES (OCB)" },
  { 4, "AES (CCM)" },
  { 5, "WEP (104-bit)" },
  { 6, "BIP" },
  { 7, "Group addressed traffic not allowed" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wpa_keymgmt_vals[] = {
  { 0, "NONE" },
  { 1, "WPA" },
  { 2, "PSK" },
  { 3, "FT over IEEE 802.1X" },
  { 4, "FT using PSK" },
  { 5, "WPA (SHA256)" },
  { 6, "PSK (SHA256)" },
  { 7, "TDLS / TPK Handshake" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_acs_vals[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Video" },
  { 3, "Voice" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_direction_vals[] = {
  { 0, "Uplink" },
  { 1, "Downlink" },
  { 2, "Direct link" },
  { 3, "Bidirectional link" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_psb_vals[] = {
  { 0, "Legacy" },
  { 1, "U-APSD" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_up_vals[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Spare" },
  { 3, "Excellent Effort" },
  { 4, "Controlled Load" },
  { 5, "Video" },
  { 6, "Voice" },
  { 7, "Network Control" },
  { 0, NULL }
};

static const value_string ieee802111_wfa_ie_wme_qos_info_sta_max_sp_length_vals[] = {
  { 0, "WMM AP may deliver all buffered frames (MSDUs and MMPDUs)" },
  { 1, "WMM AP may deliver a maximum of 2 buffered frames (MSDUs and MMPDUs) per USP" },
  { 2, "WMM AP may deliver a maximum of 4 buffered frames (MSDUs and MMPDUs) per USP" },
  { 3, "WMM AP may deliver a maximum of 6 buffered frames (MSDUs and MMPDUs) per USP" },
  { 0, NULL}
};
static const true_false_string ieee802111_wfa_ie_wme_qos_info_sta_ac_tfs = {
  "WMM delivery and trigger enabled",
  "non-WMM PS"
};

static void
wpa_mcs_base_custom(gchar *result, guint32 mcs)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, mcs >> 8);
  tmp_str = val_to_str_wmem(NULL, mcs & 0xFF, ieee80211_wfa_ie_wpa_cipher_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
wpa_ucs_base_custom(gchar *result, guint32 ucs)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, ucs >> 8);
  tmp_str = val_to_str_wmem(NULL, ucs & 0xFF, ieee80211_wfa_ie_wpa_cipher_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}
static void
wpa_akms_base_custom(gchar *result, guint32 akms)
{
  gchar oui_result[SHORT_STR];
  gchar *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, akms >> 8);
  tmp_str = val_to_str_wmem(NULL, akms & 0xFF, ieee80211_wfa_ie_wpa_keymgmt_vals, "Unknown %d");
  g_snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static gchar *
wpa_ucs_return(guint32 ucs)
{
  gchar *result;

  result = (gchar *)wmem_alloc(wmem_packet_scope(), SHORT_STR);
  result[0] = '\0';
  wpa_ucs_base_custom(result, ucs);

  return result;
}

static gchar *
wpa_akms_return(guint32 akms)
{
  gchar *result;

  result = (gchar *)wmem_alloc(wmem_packet_scope(), SHORT_STR);
  result[0] = '\0';
  wpa_akms_base_custom(result, akms);

  return result;
}

/* For each Field */
static const value_string ieee80211_wapi_suite_type[] = {
  {0, "Reserved"},
  {1, "WAI Certificate Authentication and Key Management"},
  {2, "WAI Preshared Key Authentication and Key Management"},
  {0, NULL},
};
/* For Summary Tag Information */
static const value_string ieee80211_wapi_suite_type_short[] = {
  {0, "Reserved"},
  {1, "WAI-CERT"},
  {2, "WAI-PSK"},
  {0, NULL},
};

static const value_string ieee80211_wapi_cipher_type[] = {
  {0, "Reserved"},
  {1, "WPI-SMS4"},
  {0, NULL},
};

static const value_string ieee802111_wfa_ie_wme_type[] = {
  { 0, "Information Element" },
  { 1, "Parameter Element" },
  { 2, "TSPEC Element" },
  { 0, NULL}
};

static const value_string ft_subelem_id_vals[] = {
  {0, "Reserved"},
  {1, "PMK-R1 key holder identifier (R1KH-ID)"},
  {2, "GTK subelement"},
  {3, "PMK-R0 key holder identifier (R0KH-ID)"},
  {4, "IGTK"},
  {0, NULL}
};

static int
dissect_wme_qos_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  proto_item *wme_qos_info_item;

  static const int *ieee80211_mgt_req[] = {
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_max_sp_length,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_be,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_bk,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vi,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vo,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_reserved,
    NULL
  };

  static const int *ieee80211_mgt_resp[] = {
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_u_apsd,
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_parameter_set_count,
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_reserved,
    NULL
  };

  switch (ftype) {
    case MGT_ASSOC_REQ:
    case MGT_PROBE_REQ:
    case MGT_REASSOC_REQ:
    {
      /* To AP so decode as per WMM standard Figure 7 QoS Info field when sent from WMM STA*/
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_qos_info,
                                    ett_wme_qos_info, ieee80211_mgt_req,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      break;
    }
    case MGT_BEACON:
    case MGT_PROBE_RESP:
    case MGT_ASSOC_RESP:
    case MGT_REASSOC_RESP:
    {
      /* From AP so decode as per WMM standard Figure 6 QoS Info field when sent from WMM AP */
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_qos_info,
                                    ett_wme_qos_info, ieee80211_mgt_resp,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      break;
    }
    default:
        wme_qos_info_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_qos_info, tvb, offset, 1, ENC_NA);
        expert_add_info_format(pinfo, wme_qos_info_item, &ei_ieee80211_wfa_ie_wme_qos_info_bad_ftype, "Could not deduce direction to decode correctly, ftype %u", ftype);
      break;
    }

  offset += 1;
  return offset;
}

static int
decode_qos_parameter_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  int i;
  /* WME QoS Info Field */
  offset = dissect_wme_qos_info(tree, tvb, pinfo, offset, ftype);
  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_reserved, tvb, offset, 1, ENC_NA);
  offset += 1;
  /* AC Parameters */
  for (i = 0; i < 4; i++)
  {
    proto_item *ac_item, *aci_aifsn_item, *ecw_item;
    proto_tree *ac_tree, *ecw_tree;
    guint8 aci_aifsn, ecw, ecwmin, ecwmax;
    guint16 cwmin, cwmax;
    static const int *ieee80211_wfa_ie_wme[] = {
        &hf_ieee80211_wfa_ie_wme_acp_aifsn,
        &hf_ieee80211_wfa_ie_wme_acp_acm,
        &hf_ieee80211_wfa_ie_wme_acp_aci,
        &hf_ieee80211_wfa_ie_wme_acp_reserved,
        NULL
    };

    ac_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_ac_parameters, tvb, offset, 4, ENC_NA);
    ac_tree = proto_item_add_subtree(ac_item, ett_wme_ac);

    /* ACI/AIFSN Field */
    aci_aifsn_item = proto_tree_add_bitmask_with_flags(ac_tree, tvb, offset, hf_ieee80211_wfa_ie_wme_acp_aci_aifsn,
                            ett_wme_aci_aifsn, ieee80211_wfa_ie_wme,
                            ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    aci_aifsn = tvb_get_guint8(tvb, offset);
    /* 802.11-2012, 8.4.2.31 EDCA Parameter Set element */
    if (aci_aifsn < 2) {
       expert_add_info_format(pinfo, aci_aifsn_item, &ei_ieee80211_qos_bad_aifsn,
         "The minimum value for the AIFSN subfield is 2 (found %u).", aci_aifsn);
    }
    proto_item_append_text(ac_item, " ACI %u (%s), ACM %s, AIFSN %u",
      (aci_aifsn & 0x60) >> 5, try_val_to_str((aci_aifsn & 0x60) >> 5, ieee80211_wfa_ie_wme_acs_vals),
      (aci_aifsn & 0x10) ? "yes" : "no", aci_aifsn & 0x0f);
    offset += 1;

    /* ECWmin/ECWmax field */
    ecw_item = proto_tree_add_item(ac_tree, hf_ieee80211_wfa_ie_wme_acp_ecw, tvb, offset, 1, ENC_NA);
    ecw_tree = proto_item_add_subtree(ecw_item, ett_wme_ecw);
    ecw = tvb_get_guint8(tvb, offset);
    ecwmin = ecw & 0x0f;
    ecwmax = (ecw & 0xf0) >> 4;
    cwmin= (1 << ecwmin) - 1;
    cwmax= (1 << ecwmax) - 1;
    proto_tree_add_item(ecw_tree, hf_ieee80211_wfa_ie_wme_acp_ecw_max, tvb, offset, 1, ENC_NA);
    proto_tree_add_item(ecw_tree, hf_ieee80211_wfa_ie_wme_acp_ecw_min, tvb, offset, 1, ENC_NA);
    proto_tree_add_uint(ecw_tree, hf_ieee80211_wfa_ie_wme_acp_cw_max, tvb, offset, 1, cwmax);
    proto_tree_add_uint(ecw_tree, hf_ieee80211_wfa_ie_wme_acp_cw_min, tvb, offset, 1, cwmin);
    proto_item_append_text(ac_item, ", ECWmin/max %u/%u (CWmin/max %u/%u)", ecwmin, ecwmax, cwmin, cwmax);
    offset += 1;

    /* TXOP Limit */
    proto_tree_add_item(ac_tree, hf_ieee80211_wfa_ie_wme_acp_txop_limit, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_item_append_text(ac_item, ", TXOP %u", tvb_get_letohs(tvb, offset));
    offset += 2;
  }

  return offset;
}

static int
dissect_vendor_ie_wpawme(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, guint32 tag_len, int ftype)
{
  guint8 type;

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_type, tvb, offset, 1, ENC_NA);
  type = tvb_get_guint8(tvb, offset);
  proto_item_append_text(tree, ": %s", val_to_str(type, ieee802111_wfa_ie_type_vals, "Unknown %d"));
  offset += 1;

  switch (type) {
    case 1:   /* Wi-Fi Protected Access (WPA) */
    {
      proto_item *wpa_mcs_item, *wpa_ucs_item, *wpa_akms_item;
      proto_item *wpa_sub_ucs_item, *wpa_sub_akms_item;
      proto_tree *wpa_mcs_tree, *wpa_ucs_tree, *wpa_akms_tree;
      proto_tree *wpa_sub_ucs_tree, *wpa_sub_akms_tree;
      guint16 ucs_count, akms_count;
      guint ii;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_version, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      /* Multicast Cipher Suite */
      wpa_mcs_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_mcs, tvb, offset, 4, ENC_BIG_ENDIAN);
      wpa_mcs_tree = proto_item_add_subtree(wpa_mcs_item, ett_wpa_mcs_tree);
      proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

      /* Check if OUI is 00:50:F2 (WFA) */
      if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
      {
        proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_wfa_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
      } else {
        proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
      }
      offset += 4;

      /* Unicast Cipher Suites */
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_ucs_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      ucs_count = tvb_get_letohs(tvb, offset);
      offset += 2;

      wpa_ucs_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_ucs_list, tvb, offset, ucs_count * 4, ENC_NA);
      wpa_ucs_tree = proto_item_add_subtree(wpa_ucs_item, ett_wpa_ucs_tree);
      for (ii = 0; ii < ucs_count; ii++)
      {
        wpa_sub_ucs_item = proto_tree_add_item(wpa_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs, tvb, offset, 4, ENC_BIG_ENDIAN);
        wpa_sub_ucs_tree = proto_item_add_subtree(wpa_sub_ucs_item, ett_wpa_sub_ucs_tree);
        proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

        /* Check if OUI is 00:50:F2 (WFA) */
        if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
        {
          proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_wfa_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
          proto_item_append_text(wpa_ucs_item, " %s", wpa_ucs_return(tvb_get_ntohl(tvb, offset)));
        } else {
          proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
        }
        offset += 4;
      }

      /* Authenticated Key Management Suites */
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_akms_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      akms_count = tvb_get_letohs(tvb, offset);
      offset += 2;

      wpa_akms_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_akms_list, tvb, offset, akms_count * 4, ENC_NA);
      wpa_akms_tree = proto_item_add_subtree(wpa_akms_item, ett_wpa_akms_tree);
      for (ii = 0; ii < akms_count; ii++)
      {
        wpa_sub_akms_item = proto_tree_add_item(wpa_akms_tree, hf_ieee80211_wfa_ie_wpa_akms, tvb, offset, 4, ENC_BIG_ENDIAN);
        wpa_sub_akms_tree = proto_item_add_subtree(wpa_sub_akms_item, ett_wpa_sub_akms_tree);
        proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

        /* Check if OUI is 00:50:F2 (WFA) */
        if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
        {
          proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_wfa_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
          proto_item_append_text(wpa_akms_item, " %s", wpa_akms_return(tvb_get_ntohl(tvb, offset)));
        } else {
          proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
        }
        offset += 4;
      }
      break;
    }
    case 2:   /* Wireless Multimedia Enhancements (WME) */
    {
      guint8 subtype;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_subtype, tvb, offset, 1, ENC_NA);
      subtype = tvb_get_guint8(tvb, offset);
      proto_item_append_text(tree, ": %s", val_to_str(subtype, ieee802111_wfa_ie_wme_type, "Unknown %d"));
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_version, tvb, offset, 1, ENC_NA);
      offset += 1;
      switch (subtype) {
        case 0: /* WME Information Element */
        {
          /* WME QoS Info Field */
          offset = dissect_wme_qos_info(tree, tvb, pinfo, offset, ftype);
          break;
        }
        case 1: /* WME Parameter Element */
        {
          offset = decode_qos_parameter_set(tree, tvb, pinfo, offset, ftype);
          break;
        }
        case 2:   /* WME TSPEC Element */
        {
            static const int *ieee80211_wfa_ie_wme_tspec_tsinfo[] = {
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_tid,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_direction,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_psb,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_up,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_reserved,
              NULL
            };

            proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_tspec_tsinfo,
                                    ett_tsinfo_tree, ieee80211_wfa_ie_wme_tspec_tsinfo,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset += 3;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_nor_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_max_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_max_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_inact_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_susp_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_srv_start, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_mean_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_peak_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_burst_size, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_delay_bound, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_phy, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_surplus, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_medium, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

          break;
        }
        default:
          /* No default Action */
        break;
      } /* End switch (subtype) */
      break;
    }
    case 4: /* WPS: Wifi Protected Setup */
    {
      dissect_wps_tlvs(tree, tvb, offset, tag_len-1, NULL);
    }
    break;
    default:
      /* No default Action...*/
    break;
  } /* End switch (type) */

  return offset;
}

/*
 * Dissect a group data cipher suite which consists of an OUI and a one-byte
 * selector: IEEE802.11 2012 Figure 9-256.
 *
 * Accepts a two entry array of header fields so we can use this elsewhere.
 */
static int dissect_group_data_cipher_suite(tvbuff_t *tvb, packet_info *pinfo _U_,
    proto_tree *tree, int offset, int *hf_array, gint ett_val, char *label)
{
  proto_tree *gdcs_tree = NULL;

  gdcs_tree = proto_tree_add_subtree(tree, tvb, offset, 4, ett_val, NULL,
                         label);
  proto_tree_add_item(gdcs_tree, hf_array[0], tvb, offset, 3, ENC_BIG_ENDIAN);
  offset += 3;
  proto_tree_add_item(gdcs_tree, hf_array[1], tvb, offset, 1, ENC_NA);
  offset += 1;

  return offset;
}

/*
 * Handle the HS 2.0 rev 2 OSU Server-only authenticated layer 2 Encryption
 * Network element. This is almost the same format as the RSNE so maybe some
 * common code can be used.
 */
static int
dissect_hs20_osen(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int hf_array[2] = { hf_group_data_cipher_suite_oui,
                       hf_group_data_cipher_suite_type };
  proto_tree *pwc_list = NULL;
  proto_item *pwcsi = NULL;
  guint16 pwc_count = 0, pwc_index = 0;
  guint16 akms_count = 0, akms_index = 0;
  static const int *osen_rsn_cap[] = {
    &hf_osen_rsn_cap_preauth,
    &hf_osen_rsn_cap_no_pairwise,
    &hf_osen_rsn_cap_ptksa_replay_counter,
    &hf_osen_rsn_cap_gtksa_replay_counter,
    &hf_osen_rsn_cap_mfpr,
    &hf_osen_rsn_cap_mfpc,
    &hf_osen_rsn_cap_jmr,
    &hf_osen_rsn_cap_peerkey,
    &hf_osen_rsn_spp_a_msdu_capable,
    &hf_osen_rsn_spp_a_msdu_required,
    &hf_osen_rsn_pbac,
    &hf_osen_extended_key_id_iaf,
    &hf_osen_reserved,
    NULL
  };
  guint16 pmkid_count = 0, pmkid_index = 0;
  int gmcs_array[2] = { hf_osen_group_management_cipher_suite_oui,
                        hf_osen_group_management_cipher_suite_type };

  offset = dissect_group_data_cipher_suite(tvb, pinfo, tree, offset, hf_array,
                        ett_osen_group_data_cipher_suite,
                        "OSEN Group Data Cipher Suite");

  pwc_count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_osen_pcs_count, tvb, offset,
                      2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (pwc_count > 0) {
    int start_offset = offset;
    pwc_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_osen_pairwise_cipher_suites, &pwcsi,
                        "OSEN Pairwise Cipher Suite List");

    while (pwc_count > 0) {
      if (tvb_reported_length_remaining(tvb, offset) >= 4) {
        int hf_array2[2] = { hf_osen_pairwise_cipher_suite_oui,
                            hf_osen_pairwise_cipher_suite_type };
        char label[128];

        g_snprintf(label, sizeof(label), "OSEN Pairwise Cipher Suite %d", pwc_index);
        offset = dissect_group_data_cipher_suite(tvb, pinfo, pwc_list,
                        offset, hf_array2, ett_osen_pairwise_cipher_suite,
                        label);
        pwc_index++;
        pwc_count--;
      } else {
        /* Insert the remaining? Expert Info? */
        offset += tvb_reported_length_remaining(tvb, offset);
        break;
      }
    }

    proto_item_set_len(pwcsi, offset - start_offset);
  }

  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  /* Now handle the AKM Suites */
  akms_count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_osen_akm_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (akms_count > 0) {
    int start_offset = offset;
    proto_tree *akm_list = NULL;
    proto_item *akmcsi = NULL;

    akm_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_osen_akm_cipher_suites, &akmcsi,
                        "OSEN AKM Cipher Suite List");

    while (akms_count > 0) {
      if (tvb_reported_length_remaining(tvb, offset) >= 4) {
        int hf_array3[2] = { hf_osen_akm_cipher_suite_oui,
                             hf_osen_akm_cipher_suite_type};
        char label[128];

        g_snprintf(label, sizeof(label), "OSEN AKM Cipher Suite %d", akms_index);
        offset = dissect_group_data_cipher_suite(tvb, pinfo, akm_list,
                          offset, hf_array3, ett_osen_akm_cipher_suite,
                          label);
        akms_index++;
        akms_count--;
      } else {
        /* Expert info? */
        offset += tvb_reported_length_remaining(tvb, offset);
        break;
      }
    }
    proto_item_set_len(akmcsi, offset - start_offset);
  }

  /* Any more? */
  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_osen_rsn_cap_flags,
                                    ett_osen_rsn_cap_tree, osen_rsn_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* Any more? */
  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  pmkid_count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_osen_pmkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (pmkid_count > 0) {
    proto_tree *pmkid_list = NULL;

    pmkid_list = proto_tree_add_subtree(tree, tvb, offset, pmkid_count * 16,
                                ett_osen_pmkid_list, NULL,
                                "OSEN PKMID List");

    while (pmkid_count > 0) {
      proto_tree *pmkid_tree = NULL;

      pmkid_tree = proto_tree_add_subtree_format(pmkid_list, tvb,offset, 16,
                                ett_osen_pmkid_tree, NULL,
                                "OSEN PKMID %d", pmkid_index);
      proto_tree_add_item(pmkid_tree, hf_osen_pmkid, tvb, offset, 16,
                          ENC_NA);
      offset += 16;
      pmkid_index++;
      pmkid_count--;
    }
  }

  offset = dissect_group_data_cipher_suite(tvb, pinfo, tree, offset, gmcs_array,
                        ett_osen_group_management_cipher_suite,
                        "OSEN Group Management Cipher Suite");

  return offset;
}

static const value_string hs20_indication_release_number_vals[] = {
  { 0, "Release 1" },
  { 1, "Release 2" },
  { 0, NULL }
};

static int
dissect_hs20_indication(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  static const int *ieee80211_hs20_indication[] = {
    &hf_hs20_indication_dgaf_disabled,
    &hf_hs20_indication_pps_mo_id_present,
    &hf_hs20_indication_anqp_domain_id_present,
    &hf_hs20_reserved,
    &hf_hs20_indication_release_number,
    NULL
  };
  int len = tvb_captured_length(tvb);
  int offset = 0;
  guint8 indic = tvb_get_guint8(tvb, offset);

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_hs20_indication,
                              ENC_NA);
  offset++;

  if (len >= 3 && (indic & 0x02)) { /* Contains a PPS MO ID field ... display it. */
      proto_tree_add_item(tree, hf_hs20_indication_pps_mo_id, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      offset += 2;
  }

  if ((len >= (offset + 2)) && (indic & 0x04)) {
     proto_tree_add_item(tree, hf_hs20_indication_anqp_domain_id, tvb, offset,
                         2, ENC_LITTLE_ENDIAN);
     offset += 2;
  }

  return offset;
}

static int
dissect_owe_transition_mode(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  guint8 ssid_len;

  int len = tvb_captured_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_bssid, tvb, offset, 6, ENC_NA);
  offset  += 6;
  len -= 6;

  ssid_len = tvb_get_guint8(tvb, offset);

  proto_tree_add_uint(tree, hf_ieee80211_wfa_ie_owe_ssid_length, tvb, offset, 1, ssid_len);
  offset  += 1;
  len -= 1;

  if (len < ssid_len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return offset;
  }

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_ssid, tvb, offset, ssid_len, ENC_ASCII|ENC_NA);
  offset  += len;
  len -= len;

  if (len >= 2) {
    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_band_info, tvb, offset, 1, ENC_NA);
    offset  += 1;

    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_channel_info, tvb, offset, 1, ENC_NA);
    offset  += 1;
  }

  return offset;
}

static void
dissect_vendor_ie_wfa(packet_info *pinfo, proto_item *item, tvbuff_t *tag_tvb)
{
  gint tag_len = tvb_reported_length(tag_tvb);
  guint8 subtype;
  int offset = 0;
  tvbuff_t *vendor_tvb;

  if (tag_len < 4)
    return;

  subtype = tvb_get_guint8(tag_tvb, 3);
  proto_item_append_text(item, ": %s", val_to_str_const(subtype, wfa_subtype_vals, "Unknown"));
  vendor_tvb = tvb_new_subset_length(tag_tvb, offset + 4, tag_len - 4);
  dissector_try_uint_new(wifi_alliance_ie_table, subtype, vendor_tvb, pinfo, item, FALSE, NULL);
}

static void
dissect_vendor_ie_rsn(proto_item * item, proto_tree * tree, tvbuff_t * tvb, int offset, guint32 tag_len)
{

  switch(tvb_get_guint8(tvb, offset)){
    case 1:
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=1 - GTK.
       * This is only used within EAPOL-Key frame Key Data. */
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_keyid, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_tx, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_reserved1, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_reserved2, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_key, tvb, offset, tag_len - 3, ENC_NA);
      proto_item_append_text(item, ": RSN GTK");
      break;
    }
    case 4:
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=4 - PMKID.
       * This is only used within EAPOL-Key frame Key Data. */
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_pmkid, tvb, offset, 16, ENC_NA);
      proto_item_append_text(item, ": RSN PMKID");
      break;
    }
    case 9:
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=9 - IGTK.
       * This is only used within EAPOL-Key frame Key Data. */
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_keyid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_ipn, tvb, offset, 6, ENC_LITTLE_ENDIAN);
      offset += 6;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_key, tvb, offset, tag_len - 9, ENC_NA);
      proto_item_append_text(item, ": RSN IGTK");
      break;
    }
    default:
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_unknown, tvb, offset, tag_len, ENC_NA);
      proto_item_append_text(item, ": RSN UNKNOWN");
      break;
  }
}

typedef enum {
  MARVELL_IE_MESH = 4
} marvell_ie_type_t;

static void
dissect_vendor_ie_marvell(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len)
{
  guint8 type;

  type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  switch (type) {
  case MARVELL_IE_MESH:
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_subtype, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_version, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_active_proto_id, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_active_metric_id, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_cap, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_data, tvb, offset,
      tag_len - 1, ENC_NA);
    break;
  }
}

typedef enum {
  ATHEROS_IE_ADVCAP = 1,
  ATHEROS_IE_XR = 3
} atheros_ie_type_t;

typedef enum {
  ATHEROS_IE_ADVCAP_S = 1
} atheros_ie_advcap_subtype_t;

typedef enum {
  ATHEROS_IE_XR_S = 1
} atheros_ie_xr_subtype_t;

typedef enum {
  ATHEROS_IE_CAP_TURBOP = 0x01,
  ATHEROS_IE_CAP_COMP   = 0x02,
  ATHEROS_IE_CAP_FF     = 0x04,
  ATHEROS_IE_CAP_XR     = 0x08,
  ATHEROS_IE_CAP_AR     = 0x10,
  ATHEROS_IE_CAP_BURST  = 0x20,
  ATHEROS_IE_CAP_WME    = 0x40,
  ATHEROS_IE_CAP_BOOST  = 0x80
} atheros_ie_cap_t;

static const value_string atheros_ie_type_vals[] = {
  { ATHEROS_IE_ADVCAP, "Advanced Capability"},
  { ATHEROS_IE_XR,     "eXtended Range"},
  { 0,                 NULL }
};

static const int *ieee80211_atheros_ie_cap[] = {
  &hf_ieee80211_atheros_ie_cap_f_turbop,
  &hf_ieee80211_atheros_ie_cap_f_comp,
  &hf_ieee80211_atheros_ie_cap_f_ff,
  &hf_ieee80211_atheros_ie_cap_f_xr,
  &hf_ieee80211_atheros_ie_cap_f_ar,
  &hf_ieee80211_atheros_ie_cap_f_burst,
  &hf_ieee80211_atheros_ie_cap_f_wme,
  &hf_ieee80211_atheros_ie_cap_f_boost,
  NULL
};

static void
dissect_vendor_ie_atheros(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint tag_len,
                          packet_info *pinfo, proto_item *ti_len)
{
  guint8      type;
  guint8      subtype;
  guint8      version;

  if (tag_len <= 3) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 6", tag_len+3); /* Add length of OUI to tag_length */
        return;
  }
  proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_type, tvb, offset, 1, ENC_NA);
  type = tvb_get_guint8(tvb, offset);
  proto_item_append_text(item, ": %s", val_to_str_const(type, atheros_ie_type_vals, "Unknown"));
  offset  += 1;
  tag_len -= 1;

  proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_subtype, tvb, offset, 1, ENC_NA);
  subtype  = tvb_get_guint8(tvb, offset);
  offset  += 1;
  tag_len -= 1;

  proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_version, tvb, offset, 1, ENC_NA);
  version  = tvb_get_guint8(tvb, offset);
  offset  += 1;
  tag_len -= 1;

  if (version == 0)
  {
    switch (type) {
      case ATHEROS_IE_ADVCAP:
      {
        switch (subtype) {
          case ATHEROS_IE_ADVCAP_S:
          {
            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_advcap_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_advcap_defkey, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset  += 2;
            tag_len -= 2;
            break;
          }
          default:
          /* No default Action */
          break;
        } /* End switch (subtype) */
        break;
      }
      case ATHEROS_IE_XR:
      {
        switch (subtype) {
          case ATHEROS_IE_XR_S:
          {
            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_info, tvb, offset, 1, ENC_NA);
            offset  += 1;
            tag_len -= 1;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_base_bssid, tvb, offset, 6, ENC_NA);
            offset  += 6;
            tag_len -= 6;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_xr_bssid, tvb, offset, 6, ENC_NA);
            offset  += 6;
            tag_len -= 6;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_xr_beacon, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset  += 2;
            tag_len -= 2;

            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_xr_base_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;

            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_xr_xr_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;
            break;
          }
          default:
          /* No default Action */
          break;
        } /* End switch (subtype) */
        break;
        default:
        /* No default Action */
        break;
      } /* End switch (type) */

    }
  }
  if (tag_len > 0) {
    proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_data, tvb, offset, tag_len, ENC_NA);
  }
}

typedef enum {
  AIRONET_IE_DTPC = 0,
  AIRONET_IE_UNKNOWN1 = 1,
  AIRONET_IE_VERSION = 3,
  AIRONET_IE_QOS,
  AIRONET_IE_UNKNOWN11 = 11,
  AIRONET_IE_QBSS_V2 = 14,
  AIRONET_IE_CLIENT_MFP = 20
} aironet_ie_type_t;

static const value_string aironet_ie_type_vals[] = {
  { AIRONET_IE_DTPC,      "DTPC"},
  { AIRONET_IE_UNKNOWN1,  "Unknown (1)"},
  { AIRONET_IE_VERSION,   "CCX version"},
  { AIRONET_IE_QOS,       "Qos"},
  { AIRONET_IE_UNKNOWN11, "Unknown (11)"},
  { AIRONET_IE_QBSS_V2,   "QBSS V2 - CCA"},
  { AIRONET_IE_CLIENT_MFP, "Client MFP"},
  { 0,                    NULL }
};

static const value_string aironet_mfp_vals[] = {
  { 0,      "Disabled"},
  { 1,      "Enabled"},
  { 0,      NULL }
};

static void
dissect_vendor_ie_aironet(proto_item *aironet_item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len)
{
  guint8  type;
  int i;
  gboolean dont_change = FALSE; /* Don't change the IE item text to default */

  type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  switch (type) {
  case AIRONET_IE_DTPC:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_dtpc, tvb, offset, 1, ENC_NA);
    proto_item_append_text(aironet_item, ": Aironet DTPC Powerlevel %ddBm", tvb_get_guint8(tvb, offset));
    offset += 1;
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_dtpc_unknown, tvb, offset, 1, ENC_NA);
    dont_change = TRUE;
    break;
  case AIRONET_IE_VERSION:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_version, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(aironet_item, ": Aironet CCX version = %d", tvb_get_guint8(tvb, offset));
    dont_change = TRUE;
    break;
  case AIRONET_IE_QOS:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_qos_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_qos_paramset, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    /* XXX: just copied over from WME. Maybe "Best Effort" and "Background"
     *  need to be swapped. Also, the "TXOP" may be TXOP - or not.
     */
    for (i = 0; i < 4; i++) {
      guint8 byte1, byte2;
      guint16 txop;
      byte1 = tvb_get_guint8(tvb, offset);
      byte2 = tvb_get_guint8(tvb, offset + 1);
      txop = tvb_get_letohs(tvb, offset + 2);
      proto_tree_add_bytes_format(ietree, hf_ieee80211_aironet_ie_qos_val, tvb, offset, 4, NULL,
          "CCX QoS Parameters: ACI %u (%s), Admission Control %sMandatory, AIFSN %u, ECWmin %u, ECWmax %u, TXOP %u",
        (byte1 & 0x60) >> 5, val_to_str((byte1 & 0x60) >> 5, wme_acs, "(Unknown: %d)"),
        (byte1 & 0x10) ? "" : "not ", byte1 & 0x0f,
        byte2 & 0x0f, (byte2 & 0xf0) >> 4,
        txop);
      offset += 4;
    }
    break;
  case AIRONET_IE_QBSS_V2:
    /* Extract Values */
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_scount, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_cu, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_cal, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_gl, tvb, offset + 4, 1, ENC_LITTLE_ENDIAN);
    break;
  case AIRONET_IE_CLIENT_MFP:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_clientmfp, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(aironet_item, ": Aironet Client MFP %s",
      val_to_str_const(1 & tvb_get_guint8(tvb, offset), aironet_mfp_vals, "Unknown"));
    dont_change = TRUE;
    break;
  default:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_data, tvb, offset,
      tag_len - 1, ENC_NA);
    break;
  }
  if (!dont_change) {
    proto_item_append_text(aironet_item, ": Aironet %s (%d)",
      val_to_str_const(type, aironet_ie_type_vals, "Unknown"), type);
  }
}

#define ARUBA_APNAME  3
static const value_string ieee80211_vs_aruba_subtype_vals[] = {
  { ARUBA_APNAME, "AP Name"},
  { 0,                 NULL }
};
static void
dissect_vendor_ie_aruba(proto_item *item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len)
{
  guint8 type;
  const guint8* name;

  offset += 1; /* VS OUI Type */
  tag_len -= 1;

  type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(ietree, hf_ieee80211_vs_aruba_subtype, tvb, offset, 1, ENC_NA);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_aruba_subtype_vals, "Unknown"));
  offset += 1;
  tag_len -= 1;

  switch (type) {
  case ARUBA_APNAME:
    offset += 1;
    tag_len -= 1;

    proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_aruba_apname, tvb,
                         offset, tag_len, ENC_ASCII|ENC_NA, wmem_packet_scope(), &name);
    proto_item_append_text(item, " (%s)", name);
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_vs_aruba_data, tvb, offset,
      tag_len, ENC_NA);
    proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(wmem_packet_scope(), tvb, offset, tag_len));
    break;
  }
}

static void
dissect_vendor_ie_routerboard(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len, packet_info *pinfo)
{
  guint8 type, length;
  proto_item *subitem;
  proto_tree *subtree;

  offset += 1; /* VS OUI Type */
  tag_len -= 1;
  /* FIXME: Make sure we have at least 2 bytes left */
  proto_tree_add_item(ietree, hf_ieee80211_vs_routerboard_unknown, tvb, offset, 2, ENC_NA);

  offset += 2;
  tag_len -= 2;

  while (tag_len >= 2) {
    type = tvb_get_guint8(tvb, offset);
    length = tvb_get_guint8(tvb, offset+1);
    subitem = proto_tree_add_item(ietree, hf_ieee80211_vs_routerboard_subitem, tvb, offset, length+2, ENC_NA);
    subtree = proto_item_add_subtree(subitem, ett_routerboard);
    proto_item_set_text(subitem, "Sub IE (T/L: %d/%d)", type, length);

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype, tvb, offset, 1, ENC_NA);
    offset += 1;
    tag_len -= 1;

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_sublength, tvb, offset, 1, ENC_NA);
    offset += 1;
    tag_len -= 1;

    if (tag_len < length)
      length = tag_len;
    if (length == 0) {
      expert_add_info(pinfo, subitem, &ei_ieee80211_vs_routerboard_unexpected_len);
      break;
    }

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subdata, tvb, offset, length, ENC_NA);

    if(type == 1){
      if(length == 30){
        proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype1_prefix, tvb, offset, 10, ENC_NA);
        proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype1_data, tvb, offset + 10, length - 10, ENC_ASCII|ENC_NA);
      }else{
        expert_add_info(pinfo, subitem, &ei_ieee80211_vs_routerboard_unexpected_len);
      }
    }

    offset += length;
    tag_len -= length;
  }
}

#define AEROHIVE_HOSTNAME 33
static const value_string ieee80211_vs_aerohive_type_vals[] = {
  { AEROHIVE_HOSTNAME, "Host Name"},
  { 0,                 NULL }
};
static void
dissect_vendor_ie_aerohive(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len, packet_info *pinfo)
{
  guint32 type, length;
  const guint8* hostname;
  proto_item *ti_len;

  /* VS OUI Type */
  type = tvb_get_guint8(tvb, offset);
  offset += 1;
  tag_len -= 1;

  switch(type){
    case AEROHIVE_HOSTNAME: /* Unknown (2 bytes) + Host Name Length (1 byte) + Host Name */

      proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_aerohive_type_vals, "Unknown"));

      proto_tree_add_item(ietree, hf_ieee80211_vs_aerohive_unknown, tvb, offset, 2, ENC_NA);
      offset += 2;
      tag_len -= 2;

      ti_len = proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_aerohive_hostname_length, tvb, offset, 1, ENC_NA, &length);
      offset += 1;
      tag_len -= 1;

      if (tag_len < length) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < Host Name Length");
        length = tag_len;
      }

      proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_aerohive_hostname, tvb, offset, length, ENC_ASCII|ENC_NA, wmem_packet_scope(), &hostname);
      proto_item_append_text(item, " (%s)", hostname);

    break;

    default:
      proto_tree_add_item(ietree, hf_ieee80211_vs_aerohive_data, tvb, offset, tag_len, ENC_NA);
    break;
  }
}

enum vs_nintendo_type {
  NINTENDO_SERVICES = 0x11,
  NINTENDO_CONSOLEID = 0xF0
};

static const value_string ieee80211_vs_nintendo_type_vals[] = {
  { NINTENDO_SERVICES,  "Services"},
  { NINTENDO_CONSOLEID, "ConsoleID"},
  { 0, NULL }
};

static proto_tree*
dissect_vendor_ie_nintendo_tlv(const int hfindex, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 sublen)
{
  proto_item *nintendo_item;
  proto_tree *nintendo_tree;

  nintendo_item = proto_tree_add_item(ietree, hfindex, tvb, offset, sublen, ENC_NA);
  nintendo_tree = proto_item_add_subtree(nintendo_item, ett_nintendo);

  proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_type, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_length, tvb, offset + 1,  1, ENC_NA);

  return nintendo_tree;
}

static void
dissect_vendor_ie_nintendo(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len)
{
  proto_tree *nintendo_tree;

  guint8      subtype;
  guint8      sublength;
  guint32     length = tag_len;

  /* Skip OUI type for now - the code is for type 1 (StreetPass) only */
  /* http://3dbrew.org/wiki/StreetPass */
  offset += 1;
  length -= 1;

  while(length > 0 && length < 256) { /* otherwise we are < 0 but on unsigned */
    subtype = tvb_get_guint8(tvb, offset);
    sublength = tvb_get_guint8(tvb, offset + 1);

    switch(subtype) {
    case NINTENDO_SERVICES:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_servicelist, ietree, tvb, offset, sublength + 2);
      offset += 2;
      length -= 2;

      while (sublength > 4) {

        proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_service, tvb, offset, 5, ENC_NA);
        offset += 5;
        length -= 5;
        sublength -= 5;
      }
      break;
    case NINTENDO_CONSOLEID:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_consoleid, ietree, tvb, offset, sublength + 2);
      offset += + 2;
      length -= + 2;

      proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_consoleid, tvb, offset, sublength, ENC_NA);
      offset += sublength;
      length -= sublength;
      break;
    default:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_unknown, ietree, tvb, offset, sublength + 2);
      offset += + 2;
      length -= + 2;

      proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_unknown, tvb, offset, sublength, ENC_NA);
      offset += sublength;
      length -= sublength;
      break;
    }
  }
}

static void
dissect_vendor_ie_meru(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, guint32 tag_len,
                       packet_info *pinfo)
{
  guint32 type, length;
  proto_item *subitem, *ti_len;
  proto_tree *subtree;

  while (tag_len >= 2) {
    subitem = proto_tree_add_item(ietree, hf_ieee80211_vs_meru_subitem, tvb, offset, 2, ENC_NA);
    subtree = proto_item_add_subtree(subitem, ett_meru);

    proto_tree_add_item_ret_uint(subtree, hf_ieee80211_vs_meru_subtype, tvb, offset, 1, ENC_NA, &type);
    offset += 1;
    tag_len -= 1;

    ti_len = proto_tree_add_item_ret_uint(subtree, hf_ieee80211_vs_meru_sublength, tvb, offset, 1, ENC_NA, &length);
    offset += 1;
    tag_len -= 1;

    if (tag_len < length) {
      expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < Sub Length");
      length = tag_len;
    }

    proto_item_append_text(subitem, " (t=%d, l=%d)", type, length);
    proto_item_set_len(subitem, 2+length);

    proto_tree_add_item(subtree, hf_ieee80211_vs_meru_subdata, tvb, offset, length, ENC_NA);
    offset += length;
    tag_len -= length;

  }
}

static const value_string ieee80211_vs_extreme_subtype_vals[] = {
  { 1, "AP Name"},
  { 0, NULL }
};

static void
dissect_vendor_ie_extreme(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, guint32 tag_len,
                          packet_info *pinfo)
{
  guint32 type, length;
  proto_item *ti_len;

  proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_extreme_subtype, tvb, offset, 1, ENC_NA, &type);
  offset += 1;
  tag_len -= 1;

  proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_subdata, tvb, offset, tag_len, ENC_NA);

  switch(type){
    case 1: /* Unknown (7 bytes) + AP Name Length (1 byte) + AP Name */

      proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_unknown, tvb, offset, 7, ENC_NA);
      offset += 7;
      tag_len -= 1;

      ti_len = proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_extreme_ap_length, tvb, offset, 1, ENC_NA, &length);
      offset += 1;
      tag_len -= 1;

      if (tag_len < length) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < AP Length");
        length = tag_len;
      }

    proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA);

    break;
    default:
    /* Expert info ? */
    break;
  }
}

/* 802.11-2012 8.4.2.37 QoS Capability element */
static int
dissect_qos_capability(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  switch (ftype) {
    case MGT_ASSOC_REQ:
    case MGT_PROBE_REQ:
    case MGT_REASSOC_REQ:
    {
      /* To AP so decode Qos Info as STA */
      offset += add_ff_qos_info_sta(tree, tvb, pinfo, offset);
      break;
    }

    case MGT_BEACON:
    case MGT_PROBE_RESP:
    case MGT_ASSOC_RESP:
    case MGT_REASSOC_RESP:
    {
      /* From AP so decode QoS Info as AP */
      offset += add_ff_qos_info_ap(tree, tvb, pinfo, offset);
      break;
    }

    default:
      expert_add_info_format(pinfo, proto_tree_get_parent(tree), &ei_ieee80211_qos_info_bad_ftype,
                             "Could not deduce direction to decode correctly, ftype %u", ftype);
      break;
  }

  return offset;
}

/*
 * 7.3.2.25 RSNE information element. Common format with OSEN except the
 * verison... should refactor
 */
static int
dissect_rsn_ie(packet_info *pinfo, proto_tree *tree, tvbuff_t *tvb,
               int offset, guint32 tag_len, association_sanity_check_t *association_sanity_check)
{
  proto_item *rsn_gcs_item, *rsn_pcs_item, *rsn_akms_item, *rsn_pmkid_item, *rsn_gmcs_item;
  proto_item *rsn_sub_pcs_item, *rsn_sub_akms_item;
  proto_item *rsn_pcs_count, *rsn_akms_count, *rsn_pmkid_count;
  proto_tree *rsn_gcs_tree, *rsn_pcs_tree, *rsn_akms_tree, *rsn_pmkid_tree, *rsn_gmcs_tree;
  proto_tree *rsn_sub_pcs_tree, *rsn_sub_akms_tree;
  guint16     pcs_count, akms_count, pmkid_count;
  guint       ii;
  int         tag_end = offset + tag_len;
  static const int *ieee80211_rsn_cap[] = {
    &hf_ieee80211_rsn_cap_preauth,
    &hf_ieee80211_rsn_cap_no_pairwise,
    &hf_ieee80211_rsn_cap_ptksa_replay_counter,
    &hf_ieee80211_rsn_cap_gtksa_replay_counter,
    &hf_ieee80211_rsn_cap_mfpr,
    &hf_ieee80211_rsn_cap_mfpc,
    &hf_ieee80211_rsn_cap_jmr,
    &hf_ieee80211_rsn_cap_peerkey,
    NULL
  };

  proto_tree_add_item(tree, hf_ieee80211_rsn_version, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* 7.3.2.25.1 Group Cipher suites */
  rsn_gcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_gcs, tvb, offset, 4, ENC_BIG_ENDIAN);
  rsn_gcs_tree = proto_item_add_subtree(rsn_gcs_item, ett_rsn_gcs_tree);
  proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
  if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
  {
    proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_80211_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  } else {
    proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  }
  offset += 4;

  /* 7.3.2.25.2 Pairwise Cipher suites */
  rsn_pcs_count = proto_tree_add_item(tree, hf_ieee80211_rsn_pcs_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  pcs_count = tvb_get_letohs(tvb, offset);
  offset += 2;

  if (offset + (pcs_count * 4) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_pcs_count, &ei_ieee80211_rsn_pcs_count,
        "Pairwise Cipher Suite Count too large, 4*%u > %d", pcs_count, tag_end - offset);
    pcs_count = (tag_end - offset) / 4;
  }

  rsn_pcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_pcs_list, tvb, offset, pcs_count * 4, ENC_NA);
  rsn_pcs_tree = proto_item_add_subtree(rsn_pcs_item, ett_rsn_pcs_tree);
  for (ii = 0; ii < pcs_count; ii++)
  {
    rsn_sub_pcs_item = proto_tree_add_item(rsn_pcs_tree, hf_ieee80211_rsn_pcs, tvb, offset, 4, ENC_BIG_ENDIAN);
    rsn_sub_pcs_tree = proto_item_add_subtree(rsn_sub_pcs_item, ett_rsn_sub_pcs_tree);
    proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
    if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
    {
      proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_80211_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(rsn_pcs_item, " %s", rsn_pcs_return(tvb_get_ntohl(tvb, offset)));
    } else {
      proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
    }
    offset += 4;
  }

  if (offset >= tag_end)
  {
    return offset;
  }

  /* 7.3.2.25.2 AKM suites */
  rsn_akms_count = proto_tree_add_item(tree, hf_ieee80211_rsn_akms_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  akms_count = tvb_get_letohs(tvb, offset);
  offset += 2;

  if (offset + (akms_count * 4) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_akms_count, &ei_ieee80211_rsn_pmkid_count,
        "Auth Key Management (AKM) Suite Count too large, 4*%u > %d", akms_count, tag_end - offset);
    akms_count = (tag_end - offset) / 4;
  }

  rsn_akms_item = proto_tree_add_item(tree, hf_ieee80211_rsn_akms_list, tvb, offset, akms_count * 4, ENC_NA);
  rsn_akms_tree = proto_item_add_subtree(rsn_akms_item, ett_rsn_akms_tree);
  for (ii = 0; ii < akms_count; ii++)
  {
    rsn_sub_akms_item = proto_tree_add_item(rsn_akms_tree, hf_ieee80211_rsn_akms, tvb, offset, 4, ENC_BIG_ENDIAN);
    rsn_sub_akms_tree = proto_item_add_subtree(rsn_sub_akms_item, ett_rsn_sub_akms_tree);
    proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
    if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
    {
      proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_80211_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(rsn_akms_item, " %s", rsn_akms_return(tvb_get_ntohl(tvb, offset)));

      if (association_sanity_check) {
        guint32 akm_suite = tvb_get_ntohl(tvb, offset);
        association_sanity_check->last_akm_suite = akm_suite;

        if (akm_suite == 0x000FAC03 || akm_suite == 0x000FAC04 || akm_suite == 0x000FAC09 || akm_suite == 0x000FAC10 || akm_suite == 0x000FAC11) {
          /* This is an FT AKM suite */
          association_sanity_check->has_ft_akm_suite = TRUE;
          if (association_sanity_check->rsn_first_ft_akm_suite == NULL && rsn_sub_akms_tree != NULL) {
            association_sanity_check->rsn_first_ft_akm_suite = rsn_sub_akms_tree->last_child;
          }
        } else {
          /* This is a non-FT AKM suite */
          association_sanity_check->has_non_ft_akm_suite = TRUE;
          if (association_sanity_check->rsn_first_non_ft_akm_suite == NULL && rsn_sub_akms_tree != NULL) {
            association_sanity_check->rsn_first_non_ft_akm_suite = rsn_sub_akms_tree->last_child;
          }
        }
      }
    } else {
      proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
    }
    offset += 4;
  }

  /* 7.3.2.25.3 RSN capabilities */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_rsn_cap,
                                    ett_rsn_cap_tree, ieee80211_rsn_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  if (offset >= tag_end)
  {
    return offset;
  }
  /* 7.3.2.25.4 PMKID */
  rsn_pmkid_count = proto_tree_add_item(tree, hf_ieee80211_rsn_pmkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  pmkid_count = tvb_get_letohs(tvb, offset);
  offset += 2;

  if (offset + (pmkid_count * 16) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_pmkid_count, &ei_ieee80211_pmkid_count_too_large,
        "PMKID Count too large, 16*%u > %d", pmkid_count, tag_end - offset);
    pmkid_count = (tag_end - offset) / 16;
  }

  rsn_pmkid_item = proto_tree_add_item(tree, hf_ieee80211_rsn_pmkid_list, tvb, offset, pmkid_count * 16, ENC_NA);
  rsn_pmkid_tree = proto_item_add_subtree(rsn_pmkid_item, ett_rsn_pmkid_tree);
  for (ii = 0; ii < pmkid_count; ii++)
  {
    proto_tree_add_item(rsn_pmkid_tree, hf_ieee80211_rsn_pmkid, tvb, offset, 16, ENC_NA);
    offset += 16;
  }

  if (offset >= tag_end)
  {
    return offset;
  }
  /* Group Management Cipher Suite (802.11w)*/
  rsn_gmcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_gmcs, tvb, offset, 4, ENC_BIG_ENDIAN);
  rsn_gmcs_tree = proto_item_add_subtree(rsn_gmcs_item, ett_rsn_gmcs_tree);
  proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
  /* Check if OUI is 00:0F:AC (ieee80211) */
  if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
  {
    proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_80211_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  } else {
    proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  }
  offset += 4;

  return offset;
}

/* 7.3.2.27 Extended Capabilities information element (127) */
static int
dissect_extended_capabilities_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *ti_ex_cap;
  static const int *ieee80211_tag_extended_capabilities_byte1[] = {
    &hf_ieee80211_tag_extended_capabilities_b0,
    &hf_ieee80211_tag_extended_capabilities_b1,
    &hf_ieee80211_tag_extended_capabilities_b2,
    &hf_ieee80211_tag_extended_capabilities_b3,
    &hf_ieee80211_tag_extended_capabilities_b4,
    &hf_ieee80211_tag_extended_capabilities_b5,
    &hf_ieee80211_tag_extended_capabilities_b6,
    &hf_ieee80211_tag_extended_capabilities_b7,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte2[] = {
    &hf_ieee80211_tag_extended_capabilities_b8,
    &hf_ieee80211_tag_extended_capabilities_b9,
    &hf_ieee80211_tag_extended_capabilities_b10,
    &hf_ieee80211_tag_extended_capabilities_b11,
    &hf_ieee80211_tag_extended_capabilities_b12,
    &hf_ieee80211_tag_extended_capabilities_b13,
    &hf_ieee80211_tag_extended_capabilities_b14,
    &hf_ieee80211_tag_extended_capabilities_b15,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte3[] = {
    &hf_ieee80211_tag_extended_capabilities_b16,
    &hf_ieee80211_tag_extended_capabilities_b17,
    &hf_ieee80211_tag_extended_capabilities_b18,
    &hf_ieee80211_tag_extended_capabilities_b19,
    &hf_ieee80211_tag_extended_capabilities_b20,
    &hf_ieee80211_tag_extended_capabilities_b21,
    &hf_ieee80211_tag_extended_capabilities_b22,
    &hf_ieee80211_tag_extended_capabilities_b23,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte4[] = {
    &hf_ieee80211_tag_extended_capabilities_b24,
    &hf_ieee80211_tag_extended_capabilities_b25,
    &hf_ieee80211_tag_extended_capabilities_b26,
    &hf_ieee80211_tag_extended_capabilities_b27,
    &hf_ieee80211_tag_extended_capabilities_b28,
    &hf_ieee80211_tag_extended_capabilities_b29,
    &hf_ieee80211_tag_extended_capabilities_b30,
    &hf_ieee80211_tag_extended_capabilities_b31,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte5[] = {
    &hf_ieee80211_tag_extended_capabilities_b32,
    &hf_ieee80211_tag_extended_capabilities_b33,
    &hf_ieee80211_tag_extended_capabilities_b34,
    &hf_ieee80211_tag_extended_capabilities_b35,
    &hf_ieee80211_tag_extended_capabilities_b36,
    &hf_ieee80211_tag_extended_capabilities_b37,
    &hf_ieee80211_tag_extended_capabilities_b38,
    &hf_ieee80211_tag_extended_capabilities_b39,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte6[] = {
    &hf_ieee80211_tag_extended_capabilities_b40,
    &hf_ieee80211_tag_extended_capabilities_serv_int_granularity,
    &hf_ieee80211_tag_extended_capabilities_b44,
    &hf_ieee80211_tag_extended_capabilities_b45,
    &hf_ieee80211_tag_extended_capabilities_b46,
    &hf_ieee80211_tag_extended_capabilities_b47,
    NULL
  };
  static const int *ieee80211_tag_extended_capabilities_byte7[] = {
    &hf_ieee80211_tag_extended_capabilities_b48,
    &hf_ieee80211_tag_extended_capabilities_b49,
    &hf_ieee80211_tag_extended_capabilities_b50,
    &hf_ieee80211_tag_extended_capabilities_b51,
    &hf_ieee80211_tag_extended_capabilities_b52,
    &hf_ieee80211_tag_extended_capabilities_b53,
    &hf_ieee80211_tag_extended_capabilities_b54,
    &hf_ieee80211_tag_extended_capabilities_b55,
    NULL
  };

  static const int *ieee80211_tag_extended_capabilities_byte8[] = {
    &hf_ieee80211_tag_extended_capabilities_b56,
    &hf_ieee80211_tag_extended_capabilities_b57,
    &hf_ieee80211_tag_extended_capabilities_b58,
    &hf_ieee80211_tag_extended_capabilities_b59,
    &hf_ieee80211_tag_extended_capabilities_b60,
    &hf_ieee80211_tag_extended_capabilities_b61,
    &hf_ieee80211_tag_extended_capabilities_b62,
    &hf_ieee80211_tag_extended_capabilities_b63,
    NULL
  };

  static const int *ieee80211_tag_extended_capabilities_bytes89[] = {
    &hf_ieee80211_tag_extended_capabilities_b56_2,
    &hf_ieee80211_tag_extended_capabilities_b57_2,
    &hf_ieee80211_tag_extended_capabilities_b58_2,
    &hf_ieee80211_tag_extended_capabilities_b59_2,
    &hf_ieee80211_tag_extended_capabilities_b60_2,
    &hf_ieee80211_tag_extended_capabilities_b61_2,
    &hf_ieee80211_tag_extended_capabilities_b62_2,
    &hf_ieee80211_tag_extended_capabilities_max_num_msdus,
    &hf_ieee80211_tag_extended_capabilities_b65_2,
    &hf_ieee80211_tag_extended_capabilities_b66_2,
    &hf_ieee80211_tag_extended_capabilities_b67_2,
    &hf_ieee80211_tag_extended_capabilities_b68_2,
    &hf_ieee80211_tag_extended_capabilities_b69_2,
    &hf_ieee80211_tag_extended_capabilities_b70_2,
    &hf_ieee80211_tag_extended_capabilities_b71_2,
    NULL
  };

  static const int *ieee80211_tag_extended_capabilities_byte10[] = {
    &hf_ieee80211_tag_extended_capabilities_b72,
    &hf_ieee80211_tag_extended_capabilities_b73,
    &hf_ieee80211_tag_extended_capabilities_b74,
    &hf_ieee80211_tag_extended_capabilities_b75,
    &hf_ieee80211_tag_extended_capabilities_b76,
    &hf_ieee80211_tag_extended_capabilities_b77,
    &hf_ieee80211_tag_extended_capabilities_b78,
    &hf_ieee80211_tag_extended_capabilities_b79,
    NULL
  };

  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be greater than 0", tag_len);
    return 1;
  }
  proto_item_append_text(field_data->item_tag, " (%u octet%s)", tag_len, plurality(tag_len, "", "s"));

  /* Extended Capability octet 1 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap1, ieee80211_tag_extended_capabilities_byte1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 1)");
  offset += 1;

  /* Extended Capability octet 2 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap2, ieee80211_tag_extended_capabilities_byte2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 2)");
  offset += 1;

  /* Extended Capability octet 3 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap3, ieee80211_tag_extended_capabilities_byte3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 3)");
  offset += 1;

  /* Extended Capability octet 4 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap4, ieee80211_tag_extended_capabilities_byte4,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 4)");
  offset += 1;

  /* Extended Capability octet 5 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap5, ieee80211_tag_extended_capabilities_byte5,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 5)");
  offset += 1;

  /* Extended Capability octet 6 */
  if (offset >= tag_len) {
    return offset;
  }

  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap6, ieee80211_tag_extended_capabilities_byte6,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 6)");
  offset += 1;


  /* Extended Capability octet 7 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap7, ieee80211_tag_extended_capabilities_byte7,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 7)");
  offset += 1;

  /* Extended Capability octet 8 & 9 since two bits cross the boundary */
  if (offset >= tag_len) {
    return offset;
  }

  /* If only the first of the two bytes is present, do the best we can */
  if (offset == tag_len - 1) {
    ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap8, ieee80211_tag_extended_capabilities_byte8,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    proto_item_append_text(ti_ex_cap, " (octet 8)");
    offset += 1;
  } else { /* Both bytes are there */
    ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities_2,
                                    ett_tag_ex_cap89, ieee80211_tag_extended_capabilities_bytes89,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    proto_item_append_text(ti_ex_cap, " (octets 8 & 9)");
    offset += 2;
  }

  if (offset >= tag_len) {
    return offset;
  }

  /* Extended Capability octet 10 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap10, ieee80211_tag_extended_capabilities_byte10,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 10)");
  offset += 1;

  return offset;
}
static int
dissect_vht_mcs_set(proto_tree *tree, tvbuff_t *tvb, int offset)
{
  proto_item *ti;
  proto_tree *mcs_tree;
  static const int *ieee80211_vht_mcsset_rx_max_mcs[] = {
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_1_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_2_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_3_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_4_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_5_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_6_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_7_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_8_ss,
    NULL
  };
  static const int *ieee80211_vht_mcsset_tx_max_mcs[] = {
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_1_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_2_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_3_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_4_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_5_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_6_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_7_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_8_ss,
    NULL
  };

  /* 8 byte Supported MCS set */
  ti = proto_tree_add_item(tree, hf_ieee80211_vht_mcsset, tvb, offset, 8, ENC_NA);

  mcs_tree = proto_item_add_subtree(ti, ett_vht_mcsset_tree);

  /* B0 - B15 */
  proto_tree_add_bitmask_with_flags(mcs_tree, tvb, offset, hf_ieee80211_vht_mcsset_rx_mcs_map,
                                    ett_vht_rx_mcsbit_tree, ieee80211_vht_mcsset_rx_max_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* B16 - B28 13 bits*/
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_rx_highest_long_gi, tvb, offset, 2, ENC_LITTLE_ENDIAN);

  /* B29 - B31 2 Max NSTS, total*/

  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_max_nsts_total, tvb, offset,
                                2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* B32 - B47 */
  proto_tree_add_bitmask_with_flags(mcs_tree, tvb, offset, hf_ieee80211_vht_mcsset_tx_mcs_map,
                                    ett_vht_tx_mcsbit_tree, ieee80211_vht_mcsset_tx_max_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  /* B48 - B60 13 bits */
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_tx_highest_long_gi, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  /* B61 */
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_extended_nss_bw_capable,
                                tvb, offset, 2, ENC_LITTLE_ENDIAN);

  /* B62 - B63 2 reserved bits*/
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_reserved, tvb, offset, 2,
                                ENC_LITTLE_ENDIAN);offset += 2;

  return offset;
}

static int
dissect_vht_capability_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_vht_caps[] = {
    &hf_ieee80211_vht_max_mpdu_length,
    &hf_ieee80211_vht_supported_chan_width_set,
    &hf_ieee80211_vht_rx_ldpc,
    &hf_ieee80211_vht_short_gi_for_80,
    &hf_ieee80211_vht_short_gi_for_160,
    &hf_ieee80211_vht_tx_stbc,
    /* End of first byte */
    &hf_ieee80211_vht_rx_stbc,
    &hf_ieee80211_vht_su_beamformer_cap,
    &hf_ieee80211_vht_su_beamformee_cap,
    &hf_ieee80211_vht_beamformer_antennas,
    /* End of second byte */
    &hf_ieee80211_vht_sounding_dimensions,
    &hf_ieee80211_vht_mu_beamformer_cap,
    &hf_ieee80211_vht_mu_beamformee_cap,
    &hf_ieee80211_vht_txop_ps,
    &hf_ieee80211_vht_var_htc_field,
    &hf_ieee80211_vht_max_ampdu,
    &hf_ieee80211_vht_link_adaptation_cap,
    &hf_ieee80211_vht_rx_pattern,
    &hf_ieee80211_vht_tx_pattern,
    &hf_ieee80211_vht_ext_nss_bw_support,
    NULL
  };

  if (tag_len != 12) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT Capabilities IE length %u wrong, must be = 12", tag_len);
    return 1;
  }

  /* 4 byte VHT Capabilities  Info*/
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_vht_cap,
                                    ett_vht_cap_tree, ieee80211_vht_caps,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 4;

  /* 8 byte MCS set */
  offset = dissect_vht_mcs_set(tree, tvb, offset);

  return offset;
}

static int
dissect_vht_operation_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *op_item;
  proto_tree *op_tree;
  static const int *ieee80211_vht_op_max_basic_mcs[] = {
    &hf_ieee80211_vht_op_max_basic_mcs_for_1_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_2_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_3_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_4_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_5_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_6_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_7_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_8_ss,
    NULL
  };

  if (tag_len != 5) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT Operation IE length %u wrong, must be = 5", tag_len);
    return 1;
  }

  /* 3 byte VHT Operation Info*/
  op_item = proto_tree_add_item(tree, hf_ieee80211_vht_op, tvb, offset, 3, ENC_NA);
  op_tree = proto_item_add_subtree(op_item, ett_vht_op_tree);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_width, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_center0, tvb, offset+1, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_center1, tvb, offset+2, 1, ENC_LITTLE_ENDIAN);

  offset += 3;
  /* VHT Basic MCS Set */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_vht_op_basic_mcs_map,
                                    ett_vht_basic_mcsbit_tree, ieee80211_vht_op_max_basic_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  return offset;
}

static int
dissect_vht_tx_pwr_envelope(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *tx_pwr_item, *ti;
  proto_tree *tx_pwr_info_tree;
  guint8 opt_ie_cnt=0;
  guint8 i;

  if (tag_len < 2 || tag_len > 5) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT TX PWR Envelope IE length %u wrong, must be >= 2 and <= 5", tag_len);
    return 1;
  }

  tx_pwr_item = proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_info, tvb, offset, 1, ENC_NA);
  tx_pwr_info_tree =  proto_item_add_subtree(tx_pwr_item, ett_vht_tpe_info_tree);

  ti = proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_count, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_unit, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);

  opt_ie_cnt = tvb_get_guint8(tvb, offset) & 0x07;

  offset += 1;

  /* Power Constraint info is mandatory only for 20MHz, others are optional*/
  /* Power is expressed in terms of 0.5dBm from -64 to 63 and is encoded
   * as 8-bit 2's compliment */
  for (i = 0; i <= opt_ie_cnt; i++) {
    switch(i) {
    case 0:
      proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_20, tvb, offset, 1, ENC_NA);
      offset += 1;
      break;
    case 1:
      proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_40, tvb, offset, 1, ENC_NA);
      offset += 1;
      break;
    case 2:
      proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_80, tvb, offset, 1, ENC_NA);
      offset += 1;
      break;
    case 3:
      proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_160, tvb, offset, 1, ENC_NA);
      offset += 1;
      break;
    default:
      expert_add_info(pinfo, ti, &ei_ieee80211_vht_tpe_pwr_info_count);
      offset += 1;
      break;
    }
  }

  return offset;
}

static int
dissect_mobility_domain(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (field_data->sanity_check != NULL) {
    field_data->sanity_check->association_has_mobility_domain_element = TRUE;
  }

  if (tag_len < 3) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "MDIE content length must be at least 3 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_mobility_domain_mdid,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mobility_domain_ft_capab,
                      tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mobility_domain_ft_capab_ft_over_ds,
                      tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mobility_domain_ft_capab_resource_req,
                      tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_fast_bss_transition(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len < 82) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "FTIE content length must be at least 82 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ft_mic_control,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_element_count,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_mic,
                      tvb, offset, 16, ENC_NA);
  offset += 16;
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_anonce,
                      tvb, offset, 32, ENC_NA);
  offset += 32;
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_snonce,
                      tvb, offset, 32, ENC_NA);
  offset += 32;

  while (offset + 2 <= tag_len) {
    guint8 id, len;
    int s_end;
    proto_item *ti;

    proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    id = tvb_get_guint8(tvb, offset);
    offset += 1;

    ti = proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_len,
                             tvb, offset, 1, ENC_LITTLE_ENDIAN);
    len = tvb_get_guint8(tvb, offset);
    offset += 1;

    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, ti, &ei_ieee80211_tag_length,
                            "FTIE subelement length is too large for the FTIE content length");
      return offset;
    }

    s_end = offset + len;
    switch (id) {
    case 1:
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_r1kh_id,
                          tvb, offset, len, ENC_NA);
      break;
    case 2:
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_gtk_key_info,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_gtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_gtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_gtk_rsc,
                          tvb, offset, 8, ENC_NA);
      offset += 8;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_gtk_key,
                          tvb, offset, s_end - offset, ENC_NA);
      break;
    case 3:
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_r0kh_id,
                          tvb, offset, len, ENC_NA);
      break;
    case 4:
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_igtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_igtk_ipn,
                          tvb, offset, 6, ENC_NA);
      offset += 6;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_igtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_igtk_key,
                          tvb, offset, 24, ENC_NA);
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_tag_ft_subelem_data,
                          tvb, offset, len, ENC_NA);
      break;
    }
    offset = s_end;
  }

  return tvb_captured_length(tvb);
}

static int
dissect_mmie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 16) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "MMIE content length must be at least 16 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_keyid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_ipn, tvb, offset + 2, 6,
                      ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_mic, tvb, offset + 8, 8,
                      ENC_NA);
  return tvb_captured_length(tvb);
}

static int
dissect_ssid_list(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  proto_tree *entry;
  gboolean first = TRUE;

  while (offset + 1 <= tag_len) {
    guint8 len = tvb_get_guint8(tvb, offset + 1);
    guint8 *str;

    if (offset + 2 + len > tag_len)
      break;

    str = tvb_format_text(tvb, offset + 2, len);
    proto_item_append_text(tree, "%c %s", (first ? ':' : ','), str);
    first = FALSE;
    entry = proto_tree_add_subtree_format(tree, tvb, offset, 2 + len, ett_ssid_list, NULL, "SSID: %s", str);
    proto_tree_add_item(entry, hf_ieee80211_tag_number, tvb, offset, 1,
                        ENC_LITTLE_ENDIAN);
    offset++;
    proto_tree_add_uint(entry, hf_ieee80211_tag_length, tvb, offset, 1, len);
    offset++;
    proto_tree_add_item(entry, hf_ieee80211_tag_ssid, tvb, offset, len,
                        ENC_ASCII|ENC_NA);
    offset += len;
  }

  return tvb_captured_length(tvb);
}

static int
dissect_link_identifier(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 18) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Link Identifier content length must be at least "
                          "18 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_bssid, tvb,
                      offset, 6, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_init_sta, tvb,
                      offset + 6, 6, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_resp_sta, tvb,
                      offset + 12, 6, ENC_NA);
  return tvb_captured_length(tvb);
}

static int
dissect_wakeup_schedule(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 18) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Wakeup Schedule content length must be at least "
                          "18 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_offset, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_interval, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree,
                      hf_ieee80211_tag_wakeup_schedule_awake_window_slots, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_max_awake_dur,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_idle_count, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_channel_switch_timing(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 4) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Channel Switch Timing content length must be at "
                          "least 4 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_channel_switch_timing_switch_time,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree,
                      hf_ieee80211_tag_channel_switch_timing_switch_timeout,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_pti_control(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 3) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "PTI Control content length must be at least 3 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_pti_control_tid, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_pti_control_sequence_control, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_pu_buffer_status(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_pu_buffer_status[] = {
    &hf_ieee80211_tag_pu_buffer_status_ac_bk,
    &hf_ieee80211_tag_pu_buffer_status_ac_be,
    &hf_ieee80211_tag_pu_buffer_status_ac_vi,
    &hf_ieee80211_tag_pu_buffer_status_ac_vo,
    NULL
  };

  if (tag_len < 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "PU Buffer Status content length must be at least 1 byte");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_pu_buffer_status, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_timeout_interval(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  proto_item *pi;

  pi = proto_tree_add_item(tree, hf_ieee80211_tag_timeout_int_type, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  if (tag_len < 5) {
    expert_add_info_format(pinfo, pi, &ei_ieee80211_tag_length,
                           "Timeout Interval content length must be at least "
                          "5 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_timeout_int_value, tvb,
                      offset + 1, 4, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_ric_data(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_tree  *sub_tree;
  guint8       desc_cnt = 0;
  guint32      next_ie;
  int          offset_r = 0;
  const guint8 ids[] = { TAG_RIC_DESCRIPTOR };

  if (tag_len != 4)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RIC Data Length must be 4 bytes");
    return 0;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_id, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  desc_cnt = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_desc_cnt, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_status_code, tvb,
                           offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Our Design is such that all the Resource request IE's part of the RIC
   * must be in the sub tree of RIC for better readability
   * Even omnipeek does the same way.
   */
  sub_tree = proto_item_add_subtree(tree, ett_tag_ric_data_desc_ie);

  proto_item_append_text(field_data->item_tag, " :Resource Descriptor List");
  if (desc_cnt == 0) {
    proto_item_append_text(field_data->item_tag, " :0 (Weird?)");
  }

  while ( desc_cnt != 0 ) {

    next_ie = tvb_get_guint8(tvb, offset);
    proto_item_append_text(field_data->item_tag, " :(%d:%s)", desc_cnt, val_to_str_ext(next_ie, &tag_num_vals_ext, "Reserved (%d)"));
    /* Recursive call to avoid duplication of code*/
    offset_r = add_tagged_field(pinfo, sub_tree, tvb, offset, field_data->ftype, ids, G_N_ELEMENTS(ids), NULL);
    if (offset_r == 0 )/* should never happen, returns a min of 2*/
      break;
    /* This will ensure that the IE after RIC is processed
     * only once. This gives us a good looking RIC IE :-)
     */
    tag_len += offset_r;
    desc_cnt--;
  }

  return tvb_captured_length(tvb);
}

/* Overlapping BSS Scan Parameters (74) */
static int
dissect_overlap_bss_scan_par(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int offset = 0;
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int tag_len = tvb_reported_length(tvb);

  if (tag_len != 14)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "OBSS Length must be 14 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_spd, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sad, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_cwtsi, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sptpc, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_satpc, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_wctdf, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sat, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

/* RIC Descriptor (75) */
static int
dissect_ric_descriptor(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int offset = 0;
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int tag_len = tvb_reported_length(tvb);
  guint8       rsrc_type = 0;

  if (tag_len < 1)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RIC Data Length must be at least 1 byte");
    return 1;
  }

  rsrc_type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_ric_desc_rsrc_type, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  if (rsrc_type == 1) {
    /* Block ACK params
     * 802.11-2012: 8.4.2.53 RIC Descriptor element
     * Block Ack parameter set as defined in 8.4.1.14,
     * Block Ack timeout value as defined in 8.4.1.15, and
     * Block Ack starting sequence control as defined in 8.3.1.8
     */
    /* TODO: Still figuring out how to parse these ones,
     * need a sample capture with at least HEX Dump
     */
    proto_item_append_text(field_data->item_tag, " : Block ACK Params");
    proto_tree_add_item(tree, hf_ieee80211_tag_ric_desc_var_params, tvb,
                        offset, tag_len-1, ENC_NA);
    offset += tag_len -1;
  }else {
    /* 0, 2-255 are reserved*/
    proto_item_append_text(field_data->item_tag, " :Reserved (type != 1)");
  }

  return offset;
}

static int
dissect_ext_bss_load(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_mu_mimo_capable_sta_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_ss_underutilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_20mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_40mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_80mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

static int
dissect_wide_bw_channel_switch(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_width, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_center_freq_segment0, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_center_freq_segment1, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

static int
dissect_channel_switch_wrapper(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  int tmp_sublen;
  const guint8 ids[] = { TAG_COUNTRY_INFO, TAG_WIDE_BW_CHANNEL_SWITCH,
    TAG_VHT_TX_PWR_ENVELOPE };

  /*
  Decode three subelement in IE-196(Channel Switch Wrapper element):
        (1) New Country subelement
        (2) Wide Bandwidth Channel Switch subelement
        (3) New VHT Transmit Power Envelope subelement
  */
  while (tag_len > 0){
    tmp_sublen = tvb_get_guint8(tvb, offset + 1);
    if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
      break;
    }
    tag_len -= (tmp_sublen + 2);
    offset += (tmp_sublen + 2);
  }
  return offset;
}

static int
dissect_operating_mode_notification(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  static const int *ieee80211_operat_mode_field[] = {
    &hf_ieee80211_operat_mode_field_channel_width,
    &hf_ieee80211_operat_mode_field_reserved,
    &hf_ieee80211_operat_mode_field_rxnss,
    &hf_ieee80211_operat_mode_field_rxnsstype,
    NULL
  };

  /* Operating Mode field */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_operat_notification_mode,
                                    ett_mcsbit_tree, ieee80211_operat_mode_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;
  return offset;
}

static int
dissect_mcs_set(proto_tree *tree, tvbuff_t *tvb, int offset, gboolean basic, gboolean vendorspecific)
{
  proto_item *ti;
  proto_tree *mcs_tree, *bit_tree;
  guint8 rx_nss, tx_nss; /* 0-4 for HT and 0-8 for VHT*/
  guint32 value_mcs_0_31, value_mcs_32_52, value_mcs_53_76;
  guint16 tx_mcs_set;
  rx_nss = tx_nss = 8;
  /* 16 byte Supported MCS set */
  if (vendorspecific)
  {
    ti = proto_tree_add_item(tree, hf_ieee80211_mcsset_vs, tvb, offset, 16, ENC_NA);
  } else
  {
    ti = proto_tree_add_item(tree, hf_ieee80211_mcsset, tvb, offset, 16, ENC_NA);
  }
  proto_item_append_text(ti, ": %s", basic ? "Basic MCS Set" : "MCS Set");
  mcs_tree = proto_item_add_subtree(ti, ett_mcsset_tree);

  /* Rx MCS Bitmask */
  ti = proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_rx_bitmask, tvb, offset, 10, ENC_NA);
  bit_tree = proto_item_add_subtree(ti, ett_mcsbit_tree);

  /* Bits 0 - 31 */
  value_mcs_0_31 = tvb_get_letohl(tvb, offset);

  /* Handle all zeroes/ff's case..*/
  if (value_mcs_0_31 != 0x0)
  {
    if (!(value_mcs_0_31 & (0xffffff00))) {
      /*
       * At least one MCS from 0-7 is supported, but no MCS from 8-31 are
       * supported, so only 1 spatial stream is supported.
       */
      rx_nss = 0;
    } else if (!(value_mcs_0_31 & (0xffff0000))) {
      /*
       * At least one MCS from 8-15 is supported, but no MCS from 16-31 are
       * supported, so only 2 spatial streams are supported.
       */
      rx_nss = 1;
    } else if (!(value_mcs_0_31 & (0xff000000))) {
      /*
       * At least one MCS from 16-23 is supported, but no MCS from 24-31 are
       * supported, so only 3 spatial streams are supported.
       */
      rx_nss = 2;
    } else {
      /*
       * At least one MCS from 24-31 is supported, so 4 spatial streams
       * are supported.
       */
      rx_nss = 3;
    }
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_0to7, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_8to15, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_16to23, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_24to31, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  /* Should be we check UEQM Supported?*/
  /* Bits 32 - 52 */
  value_mcs_32_52 = tvb_get_letohl(tvb, offset);
  if (!(value_mcs_32_52 & (0x1ffffe))) {
    /*
     * MCS 33-52 aren't supported, so the number of spatial streams we get
     * from whichever MCSes from 0-31 that we support is the total number
     * of spatial streams we support.
     */
    ;
  } else if (!(value_mcs_32_52 & (0x1fff80))) {
    /*
     * At least one MCS from 33-38 is supported, but no MCS from 39-52 is
     * supported, so we have at least 2 spatial streams, but none of the
     * MCSs in that range give us any more.
     */
    rx_nss = MAX(1, rx_nss);
  } else {
    /*
     * At least one MCS from 39-52 is supported, so we have at least 3
     * spatial streams.
     */
    rx_nss = MAX(2, rx_nss);
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_32, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_33to38, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_39to52, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Bits 53 - 76 */
  value_mcs_53_76 = tvb_get_letohl(tvb, offset);
  if ((value_mcs_53_76 & (0x1fffffe0))) {
    /*
     * At least one MCS from 53-76 is supported, so we have at least 4
     * spatial streams.
     */
    rx_nss = MAX(3, rx_nss);
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_53to76, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_highest_data_rate, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Follow table 8-126 from 802.11-2012 */
  tx_mcs_set = tvb_get_letohs(tvb, offset);

  if (!(tx_mcs_set & 0x0001) && !(tx_mcs_set & 0x0002))
  {
    /* TX MCS Set is not defined
     * so there is no interpretation for Max Tx Spatial Streams
     */
     tx_nss = 4; /* Not Defined*/
  }

  if ((tx_mcs_set & 0x0001) && !(tx_mcs_set & 0x0002))
  {
    /* TX MCS Set is defined to be equal to Rx MCS Set
     * So, get the Max Spatial Streams from Rx
     * MCS set
     */
     tx_nss = rx_nss;
  }
  proto_item_append_text(ti, ": %s", val_to_str(rx_nss, mcsset_tx_max_spatial_streams_flags, "Reserved:%d" ) );

  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_mcs_set_defined, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_rx_mcs_set_not_equal, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  ti = proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_max_spatial_streams, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  proto_item_append_text(ti, ", %s", val_to_str(tx_nss, mcsset_tx_max_spatial_streams_flags, "Reserved:%d" ) );
  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_unequal_modulation, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  offset += 1;

  offset += 3;
  return offset;
}

/*  802.11n D1.10 - HT Information IE  */
static int
dissect_ht_info_ie_1_1(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_ht_info1_field[] = {
    &hf_ieee80211_ht_info_secondary_channel_offset,
    &hf_ieee80211_ht_info_sta_channel_width,
    &hf_ieee80211_ht_info_rifs_mode,
    &hf_ieee80211_ht_info_reserved_b4_b7,
    NULL
  };

  static const int *ieee80211_ht_info2_field[] = {
    &hf_ieee80211_ht_info_protection,
    &hf_ieee80211_ht_info_non_greenfield_sta_present,
    &hf_ieee80211_ht_info_reserved_b11,
    &hf_ieee80211_ht_info_obss_non_ht_stas_present,
    &hf_ieee80211_ht_info_channel_center_freq_seg_2,
    &hf_ieee80211_ht_info_reserved_b21_b23,
    NULL
  };

  static const int *ieee80211_ht_info3_field[] = {
    &hf_ieee80211_ht_info_reserved_b24_b29,
    &hf_ieee80211_ht_info_dual_beacon,
    &hf_ieee80211_ht_info_dual_cts_protection,
    &hf_ieee80211_ht_info_secondary_beacon,
    &hf_ieee80211_ht_info_lsig_txop_protection_full_support,
    &hf_ieee80211_ht_info_pco_active,
    &hf_ieee80211_ht_info_pco_phase,
    &hf_ieee80211_ht_info_reserved_b36_b39,
    NULL
  };

  if (tag_len < 22) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "HT Information IE content length %u wrong, must be at least 22 bytes", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_ht_info_primary_channel, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_info_delimiter1,
                                    ett_ht_info_delimiter1_tree, ieee80211_ht_info1_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;


  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_info_delimiter2,
                                    ett_ht_info_delimiter2_tree, ieee80211_ht_info2_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_info_delimiter3,
                                    ett_ht_info_delimiter3_tree, ieee80211_ht_info3_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  offset = dissect_mcs_set(tree, tvb, offset, TRUE, FALSE);

  return offset;
}

static int
dissect_wapi_param_set(tvbuff_t *tvb, packet_info *pinfo,
                          proto_tree *tree, int offset, guint32 tag_len, proto_item *ti_len,
                          proto_item *ti, int ftype)
{
  /* Parse the WAPI Parameter Set IE Here*/
  proto_item *item;
  proto_tree *subtree;
  guint16 loop_cnt, version, akm_cnt  = 1, ucast_cnt = 1, bkid_cnt = 1;
  guint8  akm_suite_type = 0, ucast_cipher_type = 0, mcast_cipher_type = 0;
  static const int *ieee80211_tag_wapi_param_set[] = {
    &hf_ieee80211_tag_wapi_param_set_capab_preauth,
    &hf_ieee80211_tag_wapi_param_set_capab_rsvd,
    NULL
  };

  version = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_version, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /*MIN: 2 + (2+4)+ (2+4) + 4 + 2 + 0 (BKID CNT and LIST)  =20*/
  if (tag_len < 20) {
      expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                "tag_len is  %d, it's neither WAPI not BSS-AC-Access-Delay", tag_len);
    return offset;
  }

  if (version != 1) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "Version of WAPI protocol is %d, must be = 1", version);
    return offset;
  }

  /* AKM Suites: list can't be 0*/
  akm_cnt = tvb_get_letohs(tvb, offset);
  item = proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_akm_suite_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  if (akm_cnt != 0) {
    proto_item_append_text(ti, " : AKM Suite List:");
    for (loop_cnt = 0; loop_cnt < akm_cnt; loop_cnt++) {
      subtree = proto_item_add_subtree(item, ett_tag_wapi_param_set_akm_tree);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_akm_suite_oui, tvb, offset, 3, ENC_NA);
      offset += 3;
      akm_suite_type = tvb_get_guint8(tvb, offset);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_akm_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_item_append_text(ti, " (%d,%s)", loop_cnt+1, val_to_str(akm_suite_type,
      ieee80211_wapi_suite_type_short, "Reserved: %d"));
    }
    proto_item_append_text(ti, " /");
  } else {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Number of AKM suites is 0, must be min 1");
    return offset;

  }
  /* Unicast Cipher Suites: list can't be 0*/
  ucast_cnt = tvb_get_letohs(tvb, offset);
  item = proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_count,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  if (ucast_cnt != 0) {
    proto_item_append_text(ti, " Unicast Cipher List:");
    for (loop_cnt = 0; loop_cnt < ucast_cnt; loop_cnt++) {
      subtree = proto_item_add_subtree(item, ett_tag_wapi_param_set_ucast_tree);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
      offset += 3;
      ucast_cipher_type = tvb_get_guint8(tvb, offset);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_item_append_text(ti, " (%d,%s)", loop_cnt+1, val_to_str(ucast_cipher_type, ieee80211_wapi_cipher_type, "Reserved: %d"));
    }
  proto_item_append_text(ti, " /");
  } else {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Number of Unicast Cipher suites is 0, must be min 1");
    return offset;

  }

  /* Multicast Cipher Suites*/
  proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_oui, tvb, offset, 3, ENC_NA);
  offset += 3;
  mcast_cipher_type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_item_append_text(ti, " Multicast Cipher: %s", val_to_str(mcast_cipher_type, ieee80211_wapi_cipher_type, "Reserved: %d"));

  /* WAPI capability */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_wapi_param_set_capab,
                                    ett_tag_wapi_param_set_preauth_tree, ieee80211_tag_wapi_param_set,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* BKID List: The list can be 0
   * Applicable only for assoc/re-assoc
   */
  if (ftype == MGT_ASSOC_REQ || ftype == MGT_REASSOC_REQ ) {
    bkid_cnt = tvb_get_letohs(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_bkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (bkid_cnt != 0) {
      for (loop_cnt = 0; loop_cnt < bkid_cnt; loop_cnt++) {
        proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_bkid_list, tvb, offset, 16, ENC_NA);
        offset += 16;
      }
    }
  }
  return offset;
}

static int
dissect_bss_max_idle_period(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree_add_item(tree, hf_ieee80211_tag_bss_max_idle_period,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_bss_max_idle_options_protected,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  return offset;
}

enum tfs_request_subelem_id {
  TFS_REQ_SUBELEM_TFS = 1,
  TFS_REQ_SUBELEM_VENDOR_SPECIFIC = 221
};

static const value_string tfs_request_subelem_ids[] = {
  { TFS_REQ_SUBELEM_TFS, "TFS subelement" },
  { TFS_REQ_SUBELEM_VENDOR_SPECIFIC, "Vendor Specific subelement" },
  { 0, NULL }
};

static int
dissect_tfs_request(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  const guint8 ids[] = {
    1, /* TFS Subelement */
    TAG_VENDOR_SPECIFIC_IE
  };

  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_id,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_ac_delete_after_match,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_ac_notify,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  if (offset + 1 >= tag_len) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_missing_data,
                           "No TFS Request subelements in TFS Request");
    return tvb_captured_length(tvb);
  }

  while (offset + 1 < tag_len) {
    guint8 id, len;
    int s_offset, s_end;

    id = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length,
                             "Not enough data for TFS Request subelement");
      return tvb_captured_length(tvb);
    }
    switch (id) {
    case TFS_REQ_SUBELEM_TFS:
      s_offset = offset;
      s_end = offset + len;
      while (s_offset < s_end) {
        /* TODO 1 is interpreted as TAG_SUPP_RATES, fix this! */
        int tlen = add_tagged_field(pinfo, tree, tvb, s_offset, field_data->ftype, ids, G_N_ELEMENTS(ids), NULL);
        if (tlen==0)
          break;
        s_offset += tlen;
      }
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem,
                          tvb, offset, len, ENC_NA);
      break;
    }
    offset += len;
  }

  if (offset < tag_len) {
    proto_tree_add_expert_format(tree, pinfo, &ei_ieee80211_extra_data,
                           tvb, offset, tag_len - offset, "Extra data after TFS Subelements");
  }

  return tvb_captured_length(tvb);
}

enum tfs_response_subelem_id {
  TFS_RESP_SUBELEM_TFS_STATUS = 1,
  TFS_RESP_SUBELEM_TFS = 2,
  TFS_RESP_SUBELEM_VENDOR_SPECIFIC = 221
};

static const value_string tfs_response_subelem_ids[] = {
  { TFS_RESP_SUBELEM_TFS_STATUS, "TFS Status subelement" },
  { TFS_RESP_SUBELEM_TFS, "TFS subelement" },
  { TFS_RESP_SUBELEM_VENDOR_SPECIFIC, "Vendor Specific subelement" },
  { 0, NULL }
};

static int
dissect_tfs_response(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  const guint8 ids[] = {
    1, /* TFS Status subelement*/
    2, /* TFS subelement */
    TAG_VENDOR_SPECIFIC_IE
  };

  while (offset + 3 <= tag_len) {
    guint8 id, len;
    int s_offset, s_end;

    id = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_subelem_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length,
                             "Not enough data for TFS Request subelement");
      return tvb_captured_length(tvb);
    }
    switch (id) {
    case TFS_RESP_SUBELEM_TFS_STATUS:
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_status,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_id,
                          tvb, offset + 1, 1, ENC_LITTLE_ENDIAN);
      break;
    case TFS_RESP_SUBELEM_TFS:
      s_offset = offset;
      s_end = offset + len;
      while (s_offset < s_end) {
        /* TODO Element IDs 1 and 2 are misinterpreted! */
        int tlen = add_tagged_field(pinfo, tree, tvb, s_offset, field_data->ftype, ids, G_N_ELEMENTS(ids), NULL);
        if (tlen==0)
          break;
        s_offset += tlen;
      }
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_subelem,
                          tvb, offset, len, ENC_NA);
      break;
    }

    offset += len;
  }

  if (offset < tag_len) {
    proto_tree_add_expert_format(tree, pinfo, &ei_ieee80211_extra_data,
                           tvb, offset, tag_len - offset, "Extra data after TFS Subelements");
  }

  return tvb_captured_length(tvb);
}

static const value_string wnm_sleep_mode_action_types[] = {
  { 0, "Enter WNM-Sleep Mode" },
  { 1, "Exit WNM-Sleep Mode" },
  { 0, NULL }
};

static const value_string wnm_sleep_mode_response_status_vals[] = {
  { 0, "Enter/Exit WNM-Sleep Mode Accept" },
  { 1, "Exit WNM-Sleep Mode Accept, GTK/IGTK update required" },
  { 2, "Denied. The AP is unable to perform the requested action." },
  { 3, "Denied temporarily. The AP is unable to perform the requested action "
    "at the current time. The request can be submitted again at a later time."
  },
  { 4, "Denied. Due to the pending key expiration." },
  { 5, "Denied. The requested action was not granted due to other WNM services "
    "in use by the requesting STA." },
  { 0, NULL }
};

static int
dissect_wnm_sleep_mode(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree_add_item(tree, hf_ieee80211_tag_wnm_sleep_mode_action_type,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  proto_tree_add_item(tree, hf_ieee80211_tag_wnm_sleep_mode_response_status,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  proto_tree_add_item(tree, hf_ieee80211_tag_wnm_sleep_mode_interval,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  return offset;
}

static const value_string time_adv_timing_capab_vals[] = {
  { 0, "No standardized external time source" },
  { 1, "Timestamp offset based on UTC" },
  { 2, "UTC time at which the TSF timer is 0" },
  { 0, NULL }
};

static int
dissect_time_adv(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  guint8 capab;
  proto_item *item;
  proto_tree *subtree;
  struct tm tm, *now;
  time_t t;

  capab = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_timing_capab,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  switch (capab) {
  case 1:
    proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_time_value,
                        tvb, offset, 10, ENC_NA);
    offset += 10;

    proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_time_error,
                        tvb, offset, 5, ENC_NA);
    offset += 5;
    break;
  case 2:
    item = proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_time_value,
                               tvb, offset, 10, ENC_NA);
    subtree = proto_item_add_subtree(item, ett_tag_time_adv_tree);
    memset(&tm, 0, sizeof(tm));
    tm.tm_year = tvb_get_letohs(tvb, offset) - 1900;
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_year,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    tm.tm_mon = tvb_get_guint8(tvb, offset) - 1;
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_month,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    tm.tm_mday = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_day,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    tm.tm_hour = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_hours,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    tm.tm_min = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_minutes,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    tm.tm_sec = tvb_get_guint8(tvb, offset);
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_seconds,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(subtree,
                        hf_ieee80211_tag_time_adv_time_value_milliseconds,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(subtree, hf_ieee80211_tag_time_adv_time_value_reserved,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    tm.tm_isdst = -1;
    t = mktime(&tm);
    if (t != -1) {
      t += (time_t)(last_timestamp / 1000000);
      now = localtime(&t);
      if (now)
        proto_item_append_text(item,
                               ": current time=%u-%02u-%02u %02u:%02u:%02u",
                               now->tm_year + 1900, now->tm_mon + 1,
                               now->tm_mday, now->tm_hour, now->tm_min,
                               now->tm_sec);
    }

    proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_time_error,
                        tvb, offset, 5, ENC_NA);
    offset += 5;

    proto_tree_add_item(tree, hf_ieee80211_tag_time_adv_time_update_counter,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    break;
  }

  return offset;
}

static int
dissect_time_zone(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_tag_time_zone, tvb, offset, tag_len,
                      ENC_ASCII|ENC_NA);
  return tvb_captured_length(tvb);
}

static int
dissect_ap_channel_report(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "AP Channel Report length %u wrong, must be > 1", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ap_channel_report_operating_class, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": Operating Class %u, Channel List :", tvb_get_guint8(tvb, offset));
  offset += 1;

  while (offset < tag_len)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_ap_channel_report_channel_list, tvb, offset, 1, ENC_NA);
    proto_item_append_text(field_data->item_tag, " %u,", tvb_get_guint8(tvb, offset));
    offset += 1;
  }
  return tvb_captured_length(tvb);
}

static int
dissect_secondary_channel_offset_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Secondary Channel Offset length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_secondary_channel_offset, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

/* RCPI (53) */
static int
dissect_rcpi_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RCPI length %u wrong, must = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_rcpi, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

/* BSS Average Access Delay element (63) */
static int
dissect_bss_avg_access_delay_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "BSS Average Access Delay length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_bss_ap_avg_access_delay, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_antenna_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Antenna length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_antenna_id, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);

  return tvb_captured_length(tvb);
}

static int
dissect_rsni_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RSNI length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_rsni, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);

  return tvb_captured_length(tvb);
}

static int
dissect_measurement_pilot_trans_ie(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  const guint8 ids[] = { TAG_VENDOR_SPECIFIC_IE };

  /* The tag len can be 1 or more if there are sub-elements */

  proto_tree_add_item(tree, hf_ieee80211_ff_measurement_pilot_int, tvb, offset,
                      1, ENC_NA);

  tag_len--;
  offset++;

  /* Also handle the optional sub-elements */

  if (tag_len > 0) {
    while (tag_len > 0) {
      guint8 elt_len;

      elt_len = tvb_get_guint8(tvb, offset + 1);

      if(add_tagged_field(pinfo, tree, tvb, offset + 2, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        /* TODO: Add an expert info here and skip the field. */
        break;
      }

      tag_len -= elt_len + 2;
      offset += elt_len + 2;
    }
  }

  return tvb_captured_length(tvb);
}

static int
dissect_bss_available_admission_capacity_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint16 bitmask;
  static const int *ieee80211_tag_bss_avb_adm_cap_bitmask[] = {
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up0,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up1,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up2,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up3,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up4,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up5,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up6,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up7,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac0,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac1,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac2,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac3,
    &hf_ieee80211_tag_bss_avb_adm_cap_bitmask_rsv,
    NULL
  };

  if (tag_len < 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "BSS Available Admission Capacity length %u wrong, must > = 2", tag_len);
    return offset;
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_bss_avb_adm_cap_bitmask,
                                    ett_tag_bss_bitmask_tree, ieee80211_tag_bss_avb_adm_cap_bitmask,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  bitmask = tvb_get_letohs(tvb, offset);
  offset += 2;

  if(bitmask & BSS_BITMASK_UP0)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up0, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP1)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up1, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP2)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up2, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP3)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up3, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP4)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up4, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP5)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up5, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP6)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up6, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_UP7)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_up7, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_AC0)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_ac0, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_AC1)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_ac1, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_AC2)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_ac2, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if(bitmask & BSS_BITMASK_AC3)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_bss_avb_adm_cap_ac3, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  return offset;
}

static int
dissect_bss_ac_access_delay_ie(tvbuff_t *tvb, packet_info *pinfo,
                                    proto_tree *tree, int offset, guint32 tag_len, proto_item *ti_len)
{

  if (tag_len != 4) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "BSS AC Access Delay length %u wrong, must = 4", tag_len);
    return offset;
  }

  /* TODO: Display the scaled representation of the average
    medium access delay (a big (precalculed) value_string ?)
    See 8.4.2.46 BSS AC Access Delay element ... */

  proto_tree_add_item(tree, hf_ieee80211_tag_bss_avg_ac_access_delay_be, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_bss_avg_ac_access_delay_bk, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_bss_avg_ac_access_delay_vi, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_bss_avg_ac_access_delay_vo, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

/* RM Enabled Capabilities (70) */
static int
dissect_rm_enabled_capabilities_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *ti_ex_cap;
  static const int *ieee80211_tag_rm_enabled_capabilities_octet1[] = {
    &hf_ieee80211_tag_rm_enabled_capabilities_b0,
    &hf_ieee80211_tag_rm_enabled_capabilities_b1,
    &hf_ieee80211_tag_rm_enabled_capabilities_b2,
    &hf_ieee80211_tag_rm_enabled_capabilities_b3,
    &hf_ieee80211_tag_rm_enabled_capabilities_b4,
    &hf_ieee80211_tag_rm_enabled_capabilities_b5,
    &hf_ieee80211_tag_rm_enabled_capabilities_b6,
    &hf_ieee80211_tag_rm_enabled_capabilities_b7,
    NULL
  };

  static const int *ieee80211_tag_rm_enabled_capabilities_octet2[] = {
    &hf_ieee80211_tag_rm_enabled_capabilities_b8,
    &hf_ieee80211_tag_rm_enabled_capabilities_b9,
    &hf_ieee80211_tag_rm_enabled_capabilities_b10,
    &hf_ieee80211_tag_rm_enabled_capabilities_b11,
    &hf_ieee80211_tag_rm_enabled_capabilities_b12,
    &hf_ieee80211_tag_rm_enabled_capabilities_b13,
    &hf_ieee80211_tag_rm_enabled_capabilities_b14,
    &hf_ieee80211_tag_rm_enabled_capabilities_b15,
    NULL
  };

  static const int *ieee80211_tag_rm_enabled_capabilities_octet3[] = {
    &hf_ieee80211_tag_rm_enabled_capabilities_b16,
    &hf_ieee80211_tag_rm_enabled_capabilities_b17,
    &hf_ieee80211_tag_rm_enabled_capabilities_b18to20,
    &hf_ieee80211_tag_rm_enabled_capabilities_b21to23,
    NULL
  };

  static const int *ieee80211_tag_rm_enabled_capabilities_octet4[] = {
    &hf_ieee80211_tag_rm_enabled_capabilities_b24to26,
    &hf_ieee80211_tag_rm_enabled_capabilities_b27,
    &hf_ieee80211_tag_rm_enabled_capabilities_b28,
    &hf_ieee80211_tag_rm_enabled_capabilities_b29,
    &hf_ieee80211_tag_rm_enabled_capabilities_b30,
    &hf_ieee80211_tag_rm_enabled_capabilities_b31,
    NULL
  };

  static const int *ieee80211_tag_rm_enabled_capabilities_octet5[] = {
    &hf_ieee80211_tag_rm_enabled_capabilities_b32,
    &hf_ieee80211_tag_rm_enabled_capabilities_b33,
    &hf_ieee80211_tag_rm_enabled_capabilities_o5,
    NULL
  };

  if (tag_len != 5)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "RM Enabled Capabilities length %u wrong, must = 5", tag_len);
    return 1;
  }
  proto_item_append_text(field_data->item_tag, " (%d octets)", tag_len);

  /* RM Enabled Capability octet 1 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_rm_enabled_capabilities,
                                    ett_tag_rm_cap1, ieee80211_tag_rm_enabled_capabilities_octet1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 1)");
  offset += 1;

  /* RM Enabled Capability octet 2 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_rm_enabled_capabilities,
                                    ett_tag_rm_cap2, ieee80211_tag_rm_enabled_capabilities_octet2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 2)");
  offset += 1;

  /* RM Enabled Capability octet 3 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_rm_enabled_capabilities,
                                    ett_tag_rm_cap3, ieee80211_tag_rm_enabled_capabilities_octet3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 3)");
  offset += 1;

  /* RM Enabled Capability octet 4 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_rm_enabled_capabilities,
                                    ett_tag_rm_cap4, ieee80211_tag_rm_enabled_capabilities_octet4,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 4)");
  offset += 1;

  /* RM Enabled Capability octet 5 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_rm_enabled_capabilities,
                                    ett_tag_rm_cap5, ieee80211_tag_rm_enabled_capabilities_octet5,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 5)");
  offset += 1;

  return offset;
}

/* Multiple BSSID (71) */
enum multiple_bssid_subelem_id {
  MULTIPLE_BSSID_SUBELEM_NO_BSSID_PROFILE = 0,
  MULTIPLE_BSSID_SUBELEM_VENDOR_SPECIFIC = 221
};

static const value_string multiple_bssid_subelem_ids[] = {
  { MULTIPLE_BSSID_SUBELEM_NO_BSSID_PROFILE, "Nontransmitted BSSID Profile" },
  { MULTIPLE_BSSID_SUBELEM_VENDOR_SPECIFIC, "Vendor Specific" },
  { 0, NULL }
};

static int
dissect_multiple_bssid_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  guint tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  guint offset = 0;
  guint8 sub_tag_id, sub_tag_len;
  const gchar *sub_tag_name;
  proto_tree *sub_tag_tree;
  const guint8 ids[] = { TAG_VENDOR_SPECIFIC_IE };

  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Multiple BSSID length %u wrong, must be at least 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_multiple_bssid, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  /* Optional sub-elements */

  while (offset + 1 < tag_len) {
    sub_tag_id = tvb_get_guint8(tvb, offset);
    sub_tag_len = tvb_get_guint8(tvb, offset + 1);
    sub_tag_name = val_to_str(sub_tag_id, multiple_bssid_subelem_ids, "Unknown");

    sub_tag_tree = proto_tree_add_subtree_format(tree, tvb, offset, sub_tag_len + 2, ett_tag_multiple_bssid_subelem_tree, NULL, "Subelement: %s", sub_tag_name);

    proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_multiple_bssid_subelem_id, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_multiple_bssid_subelem_len, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    if (offset + sub_tag_len > tag_len) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Not enough data for subelement");
      break;
    }

    switch (sub_tag_id)
    {
    case MULTIPLE_BSSID_SUBELEM_NO_BSSID_PROFILE:
      proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_multiple_bssid_subelem_nontrans_profile, tvb, offset, sub_tag_len, ENC_NA);
      break;

    case MULTIPLE_BSSID_SUBELEM_VENDOR_SPECIFIC:
      /*
       * add_tagged_field will insert expert info if there is a problem so
       * we ignore the return value.
       */
      add_tagged_field(pinfo, sub_tag_tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL);
      break;

    default:
      /* RESERVED */
      proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_multiple_bssid_subelem_reserved, tvb, offset, sub_tag_len, ENC_NA);
      break;
    }

    offset += sub_tag_len;
  }

  if (offset < tag_len) {
    proto_tree_add_expert_format(tree, pinfo, &ei_ieee80211_extra_data,
      tvb, offset, tag_len - offset, "Extra data after subelements");
  }

  return tvb_captured_length(tvb);
}

/* 20/40 BSS Coexistence (72) */
static int
dissect_20_40_bss_coexistence(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_20_40_bss_coexistence_fields[] = {
    &hf_ieee80211_tag_20_40_bc_information_request,
    &hf_ieee80211_tag_20_40_bc_forty_mhz_intolerant,
    &hf_ieee80211_tag_20_40_bc_20_mhz_bss_witdh_request,
    &hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_request,
    &hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_grant,
    &hf_ieee80211_tag_20_40_bc_reserved,
    NULL
  };

  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "20/40 BSS Coexistence length %u wrong, must = 1", tag_len);
    return 1;
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_20_40_bc,
                                    ett_tag_20_40_bc, ieee80211_20_40_bss_coexistence_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  offset += 1;

  return offset;
}

static int
dissect_ht_capability_ie_common(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset,
                         guint32 tag_len, proto_item *ti_len, gboolean vendorspecific)
{
  proto_item *cap_item, *ti;
  proto_tree *cap_tree;
  static const int *ieee80211_ht[] = {
    &hf_ieee80211_ht_ldpc_coding,
    &hf_ieee80211_ht_chan_width,
    &hf_ieee80211_ht_sm_pwsave,
    &hf_ieee80211_ht_green,
    &hf_ieee80211_ht_short20,
    &hf_ieee80211_ht_short40,
    &hf_ieee80211_ht_tx_stbc,
    &hf_ieee80211_ht_rx_stbc,
    &hf_ieee80211_ht_delayed_block_ack,
    &hf_ieee80211_ht_max_amsdu,
    &hf_ieee80211_ht_dss_cck_40,
    &hf_ieee80211_ht_psmp,
    &hf_ieee80211_ht_40_mhz_intolerant,
    &hf_ieee80211_ht_l_sig,
    NULL
  };

  static const int *ieee80211_htex[] = {
    &hf_ieee80211_htex_pco,
    &hf_ieee80211_htex_transtime,
    &hf_ieee80211_htex_mcs,
    &hf_ieee80211_htex_htc_support,
    &hf_ieee80211_htex_rd_responder,
    NULL
  };

  static const int *ieee80211_txbf[] = {
    &hf_ieee80211_txbf_cap,
    &hf_ieee80211_txbf_rcv_ssc,
    &hf_ieee80211_txbf_tx_ssc,
    &hf_ieee80211_txbf_rcv_ndp,
    &hf_ieee80211_txbf_tx_ndp,
    &hf_ieee80211_txbf_impl_txbf,
    &hf_ieee80211_txbf_calib,
    &hf_ieee80211_txbf_expl_csi,
    &hf_ieee80211_txbf_expl_uncomp_fm,
    &hf_ieee80211_txbf_expl_comp_fm,
    &hf_ieee80211_txbf_expl_bf_csi,
    &hf_ieee80211_txbf_expl_uncomp_fm_feed,
    &hf_ieee80211_txbf_expl_comp_fm_feed,
    &hf_ieee80211_txbf_min_group,
    &hf_ieee80211_txbf_csi_num_bf_ant,
    &hf_ieee80211_txbf_uncomp_sm_bf_ant,
    &hf_ieee80211_txbf_comp_sm_bf_ant,
    &hf_ieee80211_txbf_csi_max_rows_bf,
    &hf_ieee80211_txbf_chan_est,
    &hf_ieee80211_txbf_resrv,
    NULL
  };

  static const int *ieee80211_antsel[] = {
    &hf_ieee80211_antsel_b0,
    &hf_ieee80211_antsel_b1,
    &hf_ieee80211_antsel_b2,
    &hf_ieee80211_antsel_b3,
    &hf_ieee80211_antsel_b4,
    &hf_ieee80211_antsel_b5,
    &hf_ieee80211_antsel_b6,
    &hf_ieee80211_antsel_b7,
    NULL
  };

  if (tag_len != 26) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "HT Capabilities IE length %u wrong, must be = 26", tag_len);
    return (offset > 0) ? offset : 1;
  }

  if (wlan_ignore_draft_ht && vendorspecific)
    return (offset > 0) ? offset : 1;

  /* 2 byte HT Capabilities  Info*/
  if (vendorspecific)
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_vs_cap,
                                    ett_ht_cap_tree, ieee80211_ht,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  else
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_cap,
                                    ett_ht_cap_tree, ieee80211_ht,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  offset += 2;

  /* 1 byte A-MPDU Parameters */
  if (vendorspecific)
  {
    cap_item = proto_tree_add_item(tree, hf_ieee80211_ampduparam_vs, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  } else
  {
    cap_item = proto_tree_add_item(tree, hf_ieee80211_ampduparam, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  }
  cap_tree = proto_item_add_subtree(cap_item, ett_ampduparam_tree);
  ti = proto_tree_add_item(cap_tree, hf_ieee80211_ampduparam_mpdu, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(ti, " (%04.0f[Bytes])", pow(2, 13+(tvb_get_guint8(tvb, offset) & 0x3))-1);
  proto_tree_add_item(cap_tree, hf_ieee80211_ampduparam_mpdu_start_spacing, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(cap_tree, hf_ieee80211_ampduparam_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  /* 16 byte MCS set */
  offset = dissect_mcs_set(tree, tvb, offset, FALSE, vendorspecific);


  /* 2 byte HT Extended Capabilities */
  if (vendorspecific)
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_htex_vs_cap,
                                    ett_htex_cap_tree, ieee80211_htex,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_htex_cap,
                                    ett_htex_cap_tree, ieee80211_htex,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  offset += 2;


  /* 4 byte TxBF capabilities */
  if (vendorspecific)
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_txbf_vs,
                                    ett_txbf_tree, ieee80211_txbf,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_txbf,
                                    ett_txbf_tree, ieee80211_txbf,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  offset += 4;

  /* 1 byte Antenna Selection (ASEL) capabilities */
  if (vendorspecific)
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_antsel_vs,
                                    ett_antsel_tree, ieee80211_antsel,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  else
  {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_antsel,
                                    ett_antsel_tree, ieee80211_antsel,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  offset += 1;

  return offset;
}

static int
dissect_ht_capability_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;

  return dissect_ht_capability_ie_common(tvb, pinfo, tree, 0, tvb_reported_length(tvb),
            field_data->item_tag_length, FALSE);
}

static int
dissect_ht_info_ie_1_0(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset,
                       guint32 tag_len, proto_item *ti_len)
{
  static const int *ieee80211_hta1[] = {
    &hf_ieee80211_hta_ext_chan_offset,
    &hf_ieee80211_hta_rec_tx_width,
    &hf_ieee80211_hta_rifs_mode,
    &hf_ieee80211_hta_controlled_access,
    &hf_ieee80211_hta_service_interval,
    NULL
  };

  static const int *ieee80211_hta2[] = {
    &hf_ieee80211_hta_operating_mode,
    &hf_ieee80211_hta_non_gf_devices,
    NULL
  };

  static const int *ieee80211_hta3[] = {
    &hf_ieee80211_hta_basic_stbc_mcs,
    &hf_ieee80211_hta_dual_stbc_protection,
    &hf_ieee80211_hta_secondary_beacon,
    &hf_ieee80211_hta_lsig_txop_protection,
    &hf_ieee80211_hta_pco_active,
    &hf_ieee80211_hta_pco_phase,
    NULL
  };

  if (tag_len != 22) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "Tag length %u wrong, must be = 22", tag_len);
    return offset;
  }

  if (wlan_ignore_draft_ht)
    return offset;

  /* 1 HT Control Channel */
  proto_tree_add_item(tree, hf_ieee80211_hta_cc, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  /* 1 byte HT additional capabilities */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_hta_cap1,
                                    ett_hta_cap_tree, ieee80211_hta1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  /* 2 byte HT additional capabilities */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_hta_cap2,
                                    ett_hta_cap1_tree, ieee80211_hta2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* 2 byte HT additional capabilities */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_hta_cap2,
                                    ett_hta_cap2_tree, ieee80211_hta3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* 16 byte Supported MCS set */
  offset = dissect_mcs_set(tree, tvb, offset, FALSE, TRUE);

  return offset;
}

/* 802.11n-D1.10 and 802.11n-D2.0, 7.1.3.5a */

/*
 * 8.2.4.1.10 "Order field" says:
 *
 *  The Order field is 1 bit in length. It is used for two purposes:
 *
 *    -- It is set to 1 in a non-QoS data frame transmitted by a non-QoS
 *       STA to indicate that the frame contains an MSDU, or fragment
 *       thereof, that is being transferred using the StrictlyOrdered
 *       service class.
 *
 *    -- It is set to 1 in a QoS data or management frame transmitted
 *       with a value of HT_GF or HT_MF for the FORMAT parameter of the
 *       TXVECTOR to indicate that the frame contains an HT Control field.
 *
 * 802.11ac changes the second of those clauses to say "HT_GF, HT_MF,
 * or VHT", indicates that bit B0 of the field is 0 for HT and 1 for
 * VHT (stealing a reserved bit from the Link Adaptation Control field),
 * and that everything except for "AC Constraint" and "RDG/More Cowbell^W
 * PPDU" is different for the VHT version.
 *
 *  802.11ax changes the meaning of the first two bits:
 *
 *     B0 = 0         means High Throughput
 *     B0 = 1, B1 = 0 means Very High Throughput
 *     B0 = 1, B1 = 1 means High Efficiency
 *
 * I read the second clause of 8.2.4.1.10 "Order field", as modified by
 * 802.11ac, as meaning that, for QoS data and management frames, the
 * Order field will *only* be set to 1 for HT or VHT frames, and therefore
 * that we do *not* have to determine, from radio metadata, whether the
 * frame was transmitted as an HT or VHT frame.
 *
 * (See bug 11351, in which a frame with an HT Control field, with a
 * radiotap header, lacks the MCS or VHT fields in the radiotap header,
 * so Wireshark has no clue that it's an HT or VHT field, and misdissected
 * the packet.  Omnipeek, which also appeared to have no clue that it was
 * an HT or VHT field - it called it an 802.11b frame - *did* dissect the
 * HT Control field.)
 *
 * 802.11ax changes the reserved bit to differentiate between the HE version
 * and the VHT version, and adds different types of Aggregate Control info.
 */
#define A_CONTROL_TRS 0
#define A_CONTROL_OM   1
#define A_CONTROL_HLA  2
#define A_CONTROL_BSR  3
#define A_CONTROL_UPH  4
#define A_CONTROL_BQR  5
#define A_CONTROL_CCI  6
#define A_CONTROL_BQR_REV 0x0A

static const value_string a_control_control_id_vals[] = {
  { A_CONTROL_TRS,  "Triggered response scheduling" },
  { A_CONTROL_OM,   "Operating mode" },
  { A_CONTROL_HLA,  "HE link adaptation" },
  { A_CONTROL_BSR,  "Buffer status report" },
  { A_CONTROL_UPH,  "UL power headroom" },
  { A_CONTROL_BQR,  "Bandwidth query report" },
  { A_CONTROL_CCI,  "Command Control Indication" },
  { A_CONTROL_BQR_REV, "Bandwidth Query Report (reversed bits)" },
  { 0, NULL }
};

/*
 * Print the UL target RSSI field as per the spec.
 *  0->30 map to -90 to -30 dBm.
 *  31 maps to Max ransmit power
 */
static void
ul_target_rssi_base_custom(gchar *result, guint32 target_rssi)
{
  if (target_rssi <= 30) {
    g_snprintf(result, ITEM_LABEL_LENGTH, "%ddBm", -90 + (2 * target_rssi));
  } else if (target_rssi == 31) {
    g_snprintf(result, ITEM_LABEL_LENGTH, "Max transmit power");
  }
}

static void
dissect_a_control_padding(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *trs_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x03FFFFFF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  trs_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_a_control_padding,
                                NULL, "Padding: 0x%0x", the_bits);

  proto_tree_add_uint(trs_tree, hf_ieee80211_he_a_control_padding, tvb,
                        offset, 4, the_bits);
}

static void
dissect_a_control_trs(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *trs_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x03FFFFFF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  trs_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_triggered_response_schedule,
                                NULL, "UMRS Control: 0x%08x", the_bits);

  proto_tree_add_uint(trs_tree, hf_ieee80211_he_trs_he_tb_ppdu_len, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(trs_tree, hf_ieee80211_he_trs_ru_allocation, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(trs_tree, hf_ieee80211_he_dl_tx_power, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(trs_tree, hf_ieee80211_he_ul_target_rssi, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(trs_tree, hf_ieee80211_he_ul_mcs, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(trs_tree, hf_ieee80211_he_ul_reserved, tvb,
                        offset, 4, the_bits);
}

static void
dissect_a_control_om(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *om_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x0000003FF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  om_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_control_om,
                                NULL, "OM Control: 0x%04x", the_bits);

  proto_tree_add_uint(om_tree, hf_ieee80211_he_om_rx_nss, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(om_tree, hf_ieee80211_he_om_channel_width, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(om_tree, hf_ieee80211_he_om_ul_mu_disable, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(om_tree, hf_ieee80211_he_om_tx_nsts, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(om_tree, hf_ieee80211_he_om_reserved, tvb,
                        offset, 4, the_bits);
}

static const true_false_string he_hla_tx_bf_tfs = {
  "beamformed PPDU",
  "non-beamformed PPDU"
};

static void
dissect_a_control_hla(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *hla_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x03FFFFFF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  hla_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_hla_control,
                                NULL, "HLA Control: 0x%08x", the_bits);

  proto_tree_add_boolean(hla_tree, hf_ieee80211_he_hla_unsolicited_mfb, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(hla_tree, hf_ieee80211_he_hla_mrq, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_nss, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_he_mcs, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(hla_tree, hf_ieee80211_he_hla_dcm, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_ru, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_bw, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_msi_ppdu_type, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(hla_tree, hf_ieee80211_he_hla_tx_bf, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(hla_tree, hf_ieee80211_he_hla_reserved, tvb,
                        offset, 4, the_bits);
}

static void
dissect_a_control_bsr(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *bsr_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x03FFFFFF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  bsr_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_buffer_status_report,
                                NULL, "Buffer Status Report: 0x%08x", the_bits);

  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_aci_bitmap, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_delta_tid, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_aci_high, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_scaling_factor, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_queue_size_high, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bsr_tree, hf_ieee80211_he_bsr_queue_size_all, tvb,
                        offset, 4, the_bits);
}

static void
dissect_a_control_uph(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *uph_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x000000FF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  uph_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_control_uph,
                                NULL, "UPH Control: 0x%02x", the_bits);

  proto_tree_add_uint(uph_tree, hf_ieee80211_he_uph_ul_power_headroom, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(uph_tree, hf_ieee80211_he_uph_ul_min_transmit_power_flag,
                        tvb, offset, 4, the_bits);
  proto_tree_add_uint(uph_tree, hf_ieee80211_he_uph_reserved,
                        tvb, offset, 4, the_bits);
}

static void
dissect_a_control_bqr(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *bqr_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x03FF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  bqr_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_buffer_control_bqr,
                                NULL, "BQR Control: 0x%04x", the_bits);

  proto_tree_add_uint(bqr_tree, hf_ieee80211_he_btc_avail_chan, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(bqr_tree, hf_ieee80211_he_btc_reserved, tvb,
                        offset, 4, the_bits);
}

static void
dissect_a_control_cci(proto_tree *tree, tvbuff_t *tvb, int offset,
  guint32 bits _U_, guint32 start_bit)
{
  proto_tree *cci_tree = NULL;
  guint the_bits = (tvb_get_letohl(tvb, offset) >> start_bit) & 0x000000FF;

  /*
   * We isolated the bits and moved them to the bottom ... so display them
   */
  cci_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4,
                                ett_ieee80211_control_cci,
                                NULL, "Command Control Indication: 0x%02x", the_bits);

  proto_tree_add_boolean(cci_tree, hf_ieee80211_he_cci_ac_constraint, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(cci_tree, hf_ieee80211_he_cci_rdg_more_ppdu, tvb,
                        offset, 4, the_bits);
  proto_tree_add_boolean(cci_tree, hf_ieee80211_he_cci_sr_ppdu_indic, tvb,
                        offset, 4, the_bits);
  proto_tree_add_uint(cci_tree, hf_ieee80211_he_cci_reserved, tvb,
                        offset, 4, the_bits);
}

static void
dissect_ht_control(packet_info* pinfo, proto_tree *tree, tvbuff_t *tvb, int offset)
{
  proto_item *ti;
  proto_tree *htc_tree, *lac_subtree, *mfb_subtree;
  guint32 htc;

  htc = tvb_get_letohl(tvb, offset);

  ti = proto_tree_add_item(tree, hf_ieee80211_htc, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  htc_tree = proto_item_add_subtree(ti, ett_htc_tree);

  /* Check the HT vs. VHT bit. */
  proto_tree_add_item(htc_tree, hf_ieee80211_htc_vht, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  if (htc & HTC_VHT) {
    /* VHT or HE */
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_he, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    if (htc & HTC_HE) {
      /*
       * We have a 30-bit field, of which the first is a 4-bit Control ID which
       * determines how the rest is handled. There can be multiple fields.
       */
      proto_tree *a_control_tree = NULL;
      proto_item *pi = NULL;
      guint8 start_bit_offset = 2;

      a_control_tree = proto_tree_add_subtree_format(htc_tree, tvb, offset, 4,
                                ett_htc_he_a_control, NULL,
                                "Aggregate Control: 0x%0x", htc >> 2);
      while (start_bit_offset < 32) {
        guint8 control_id = (htc >> start_bit_offset) & 0x0F;
        start_bit_offset += 4;
        if (control_id != 0 || start_bit_offset == 6) {
          pi = proto_tree_add_uint(a_control_tree, hf_ieee80211_htc_he_ctrl_id,
                        tvb, offset, 4, control_id);
          proto_item_append_text(pi, ": %s",
                        val_to_str(control_id, a_control_control_id_vals,
                                        "Reserved (%u)"));
        }
        if (start_bit_offset > 31) {
          expert_add_info(pinfo, a_control_tree, &ei_ieee80211_invalid_control_word);
          break;
        }
        switch (control_id) {
        case A_CONTROL_TRS:
          /*
           * Padding looks like TRS ... so distinguish. If there are not
           * enough bits left it must be padding
           */
          if (start_bit_offset == 6)
            dissect_a_control_trs(a_control_tree, tvb, offset, htc, start_bit_offset);
          else
            dissect_a_control_padding(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 26;
          break;
        case A_CONTROL_OM:
          dissect_a_control_om(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 12;
          break;
        case A_CONTROL_HLA:
          dissect_a_control_hla(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 26;
          break;
        case A_CONTROL_BSR:
          dissect_a_control_bsr(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 26;
          break;
        case A_CONTROL_UPH:
          dissect_a_control_uph(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 8;
          break;
        case A_CONTROL_BQR:
        case A_CONTROL_BQR_REV:
          dissect_a_control_bqr(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 10;
          break;
        case A_CONTROL_CCI:
          dissect_a_control_cci(a_control_tree, tvb, offset, htc, start_bit_offset);
          start_bit_offset += 8;
          break;
        default:
          /* Add an expert info ... */
          start_bit_offset += 32;  /* Abandon */
          break;
        }
      }
    } else {
      proto_tree_add_item(htc_tree, hf_ieee80211_htc_mrq, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      if (!(htc & HTC_UNSOLICITED_MFB)) {
        if (htc & HTC_MRQ) {
          proto_tree_add_item(htc_tree, hf_ieee80211_htc_msi, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        } else {
          proto_tree_add_item(htc_tree, hf_ieee80211_htc_msi_stbc_reserved, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        }
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_mfsi, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      } else {
        if (!HTC_NO_FEEDBACK_PRESENT(HTC_MFB(htc))) {
          proto_tree_add_item(htc_tree, hf_ieee80211_htc_compressed_msi, tvb, offset, 4, ENC_LITTLE_ENDIAN);
          proto_tree_add_item(htc_tree, hf_ieee80211_htc_ppdu_stbc_encoded, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        } else {
          proto_tree_add_item(htc_tree, hf_ieee80211_htc_msi_stbc_reserved, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        }
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_gid_l, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      }
      ti = proto_tree_add_item(htc_tree, hf_ieee80211_htc_mfb, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      mfb_subtree = proto_item_add_subtree(ti, ett_mfb_subtree);
      proto_tree_add_item(mfb_subtree, hf_ieee80211_htc_num_sts, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(mfb_subtree, hf_ieee80211_htc_vht_mcs, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(mfb_subtree, hf_ieee80211_htc_bw, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      /* This should be converted to dB by adding 22  */
      proto_tree_add_item(mfb_subtree, hf_ieee80211_htc_snr, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      if (!HTC_NO_FEEDBACK_PRESENT(HTC_MFB(htc))) {
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_gid_h, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_coding_type, tvb, offset, 4, ENC_LITTLE_ENDIAN);
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_fb_tx_type, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      } else {
        proto_tree_add_item(htc_tree, hf_ieee80211_htc_reserved3, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      }
      proto_tree_add_item(htc_tree, hf_ieee80211_htc_unsolicited_mfb, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    }
  } else {
    /* Start: Link Adaptation Control */
    ti = proto_tree_add_item(htc_tree, hf_ieee80211_htc_ht_lac, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    lac_subtree = proto_item_add_subtree(ti, ett_lac_subtree);
    proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_trq, tvb, offset, 2, ENC_LITTLE_ENDIAN);

    if (HTC_IS_ASELI(htc)) {
      proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mai_aseli, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    } else {
      proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mai_mrq, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      if (HTC_LAC_MAI_MRQ(htc)) {
        proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mai_msi, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      } else {
        proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mai_reserved, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      }
    }

    proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mfsi, tvb, offset, 2, ENC_LITTLE_ENDIAN);

    if (HTC_IS_ASELI(htc)) {
      proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_asel_command, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_asel_data, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    } else {
      proto_tree_add_item(lac_subtree, hf_ieee80211_htc_lac_mfb, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    }
    /* End: Link Adaptation Control */

    proto_tree_add_item(htc_tree, hf_ieee80211_htc_cal_pos, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_cal_seq, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_reserved1, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_csi_steering, tvb, offset, 4, ENC_LITTLE_ENDIAN);

    proto_tree_add_item(htc_tree, hf_ieee80211_htc_ndp_announcement, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_reserved2, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  }

  /*
   * These bits are part of the Aggregate Control field for 802.11ax
   */
  if (!(htc & HTC_HE)) {
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_ac_constraint, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(htc_tree, hf_ieee80211_htc_rdg_more_ppdu, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  }

  /* offset += 2; */
}

#define IEEE80211_COMMON_OPT_BROKEN_FC         0x00000001
#define IEEE80211_COMMON_OPT_IS_CENTRINO       0x00000002
#define IEEE80211_COMMON_OPT_NORMAL_QOS        0x00000004

static void
dissect_frame_control(proto_tree *tree, tvbuff_t *tvb, guint32 option_flags,
                      guint32 offset, packet_info *pinfo)
{
  guint16 fcf, flags, frame_type_subtype;
  proto_tree *fc_tree, *flag_tree;
  proto_item *fc_item, *flag_item, *hidden_item, *ti;

  fcf = FETCH_FCF(offset);

  flags = FCF_FLAGS(fcf);
  frame_type_subtype = COMPOSE_FRAME_TYPE(fcf);

  /* Swap offset... */
  if(option_flags & IEEE80211_COMMON_OPT_BROKEN_FC)
  {
    offset += 1;
  }

  proto_tree_add_uint(tree, hf_ieee80211_fc_frame_type_subtype, tvb, offset, 1, frame_type_subtype);

  fc_item = proto_tree_add_item(tree, hf_ieee80211_fc_field, tvb, offset, 2, ENC_BIG_ENDIAN);

  fc_tree = proto_item_add_subtree(fc_item, ett_fc_tree);

  proto_tree_add_item(fc_tree, hf_ieee80211_fc_proto_version, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(fc_tree, hf_ieee80211_fc_frame_type, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(fc_tree, hf_ieee80211_fc_frame_subtype, tvb, offset, 1, ENC_NA);
  /* Changing control frame extension for extension frames */
  if(IS_FRAME_EXTENSION(fcf) == 1) {
    proto_tree_add_uint(fc_tree, hf_ieee80211_fc_frame_extension, tvb, offset, 1, FCF_FRAME_EXTENSION(fcf));
  }
  offset += 1;

  /* Reswap offset...*/
  if(option_flags & IEEE80211_COMMON_OPT_BROKEN_FC)
  {
    offset -= 1;
    proto_item_append_text(fc_item, "(Swapped)");
  }

  /* Flags */
  flag_item = proto_tree_add_item(fc_tree, hf_ieee80211_fc_flags, tvb, offset, 1, ENC_NA);
  flag_tree = proto_item_add_subtree(flag_item, ett_proto_flags);
  /* Changing control frame flags for extension frames */
  if(IS_FRAME_EXTENSION(fcf) == 0) {
    proto_tree_add_item(flag_tree, hf_ieee80211_fc_data_ds, tvb, offset, 1, ENC_NA);
    hidden_item = proto_tree_add_item(flag_tree, hf_ieee80211_fc_to_ds, tvb, offset, 1, ENC_NA);
    PROTO_ITEM_SET_HIDDEN(hidden_item);
    hidden_item = proto_tree_add_item(flag_tree, hf_ieee80211_fc_from_ds, tvb, offset, 1, ENC_NA);
    PROTO_ITEM_SET_HIDDEN(hidden_item);
    proto_tree_add_item(flag_tree, hf_ieee80211_fc_more_frag, tvb, offset, 1, ENC_NA);
    ti = proto_tree_add_item(flag_tree, hf_ieee80211_fc_retry, tvb, offset, 1, ENC_NA);
    if( IS_RETRY(flags) )
    {
      expert_add_info(pinfo, ti, &ei_ieee80211_fc_retry);
      wlan_stats.fc_retry = 1;
    }
  }
  proto_tree_add_item(flag_tree, hf_ieee80211_fc_pwr_mgt, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(flag_tree, hf_ieee80211_fc_more_data, tvb, offset, 1, ENC_NA);
  /* Changing control frame flags for extension frames */
  if(IS_FRAME_EXTENSION(fcf) == 0) {
    proto_tree_add_item(flag_tree, hf_ieee80211_fc_protected, tvb, offset, 1, ENC_NA);
  }
  proto_tree_add_item(flag_tree, hf_ieee80211_fc_order, tvb, offset, 1, ENC_NA);
}

static void
dissect_durid(proto_tree *hdr_tree, tvbuff_t *tvb, guint16 fts, gint offset)
{
  guint16 durid = tvb_get_letohs(tvb, offset);

  if (durid < 0x8000) {
    proto_tree_add_uint_format_value(hdr_tree, hf_ieee80211_did_duration, tvb,
      offset, 2, durid, "%u microseconds", durid);
  } else if (((durid & 0xC000) == 0xC000) &&
             ((durid & 0x3FFF) > 0) && ((durid & 0x3FFF) <= 2007) &&
             (fts == CTRL_PS_POLL)) {
    proto_tree_add_item(hdr_tree, hf_ieee80211_assoc_id, tvb, offset, 2,
      ENC_LITTLE_ENDIAN);
  } else if (durid == 0x8000) {
    proto_tree_add_uint_format(hdr_tree, hf_ieee80211_did_duration, tvb,
      offset, 2, durid, "Duration/ID: %u", durid);
  } else {
    proto_tree_add_uint_format(hdr_tree, hf_ieee80211_did_duration, tvb,
      offset, 2, durid, "Duration/ID: %u (reserved)", durid & 0x3FFF);
  }
}


static void
dissect_vendor_ie_ht(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree,
                    guint offset, proto_item *item, proto_item *ti_len, gint tag_len)
{

  guint8 type;

  proto_tree_add_item(tree, hf_ieee80211_ht_pren_type, tvb, offset, 1, ENC_NA);
  type = tvb_get_guint8(tvb, offset);
  offset += 1;
  tag_len -= 1;


  switch(type){
    case 51:
      dissect_ht_capability_ie_common(tvb, pinfo, tree, offset, tag_len, ti_len, TRUE);
      proto_item_append_text(item, ": HT Capabilities (802.11n D1.10)");
    break;

    case 52:
      dissect_ht_info_ie_1_0(tvb, pinfo, tree, offset, tag_len, ti_len);
      proto_item_append_text(item, ": HT Additional Capabilities (802.11n D1.00)");
    break;

    default:
      proto_tree_add_item(tree, hf_ieee80211_ht_pren_unknown, tvb, offset, tag_len, ENC_NA);
    break;
  }

}

static int
dissect_interworking(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_tag_interworking[] = {
    &hf_ieee80211_tag_interworking_access_network_type,
    &hf_ieee80211_tag_interworking_internet,
    &hf_ieee80211_tag_interworking_asra,
    &hf_ieee80211_tag_interworking_esr,
    &hf_ieee80211_tag_interworking_uesa,
    NULL
  };

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_interworking, ENC_LITTLE_ENDIAN);
  offset += 1;

  if ((tag_len == (1 + 2)) || (tag_len == (1 + 2 + 6))) {
    dissect_venue_info(tree, tvb, offset);
    offset += 2;
  }

  if ((tag_len == (1 + 6)) || (tag_len == (1 + 2 + 6))) {
    proto_tree_add_item(tree, hf_ieee80211_tag_interworking_hessid,
                        tvb, offset, 6, ENC_NA);
    offset += 6;
  }

  if ((tag_len != 1) && (tag_len != (1 + 2)) && (tag_len != (1 + 6)) && (tag_len != (1 + 2 + 6))) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Invalid Interworking element length");
  }

  return offset;
}

static int
dissect_qos_map_set(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint8 left;
  guint8 val, val2;
  int i;
  proto_item *dscp_item, *item;
  proto_tree *dscp_tree;

  if (tag_len < 16 || tag_len & 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_bad_length,
                                "Truncated QoS Map Set element");
    return tvb_captured_length(tvb);
  }

  left = tag_len - 16;
  while (left >= 2) {
    dscp_item = proto_tree_add_item(tree, hf_ieee80211_tag_qos_map_set_dscp_exc,
                                    tvb, offset, 2, ENC_LITTLE_ENDIAN);
    dscp_tree = proto_item_add_subtree(dscp_item, ett_qos_map_set_exception);

    item = proto_tree_add_item(dscp_tree,
                               hf_ieee80211_tag_qos_map_set_dscp_exc_val,
                               tvb, offset, 1, ENC_NA);
    val = tvb_get_guint8(tvb, offset);
    if (val > 63 && val != 255) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                                  "Invalid DSCP Value");
    }
    offset++;

    item = proto_tree_add_item(dscp_tree,
                               hf_ieee80211_tag_qos_map_set_dscp_exc_up,
                               tvb, offset, 1, ENC_NA);
    val2 = tvb_get_guint8(tvb, offset);
    if (val2 > 7) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                                  "Invalid User Priority");
    }
    offset++;

    proto_item_append_text(dscp_item, " (0x%02x: UP %u)", val, val2);

    left -= 2;
  }

  for (i = 0; i < 8; i++) {
    dscp_item = proto_tree_add_item(tree, hf_ieee80211_tag_qos_map_set_range,
                                    tvb, offset, 2, ENC_NA);
    dscp_tree = proto_item_add_subtree(dscp_item, ett_qos_map_set_exception);

    item = proto_tree_add_item(dscp_tree, hf_ieee80211_tag_qos_map_set_low,
                               tvb, offset, 1, ENC_NA);
    val = tvb_get_guint8(tvb, offset);
    if (val > 63 && val != 255) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                                  "Invalid DSCP Value");
    }
    offset++;

    item = proto_tree_add_item(dscp_tree, hf_ieee80211_tag_qos_map_set_high,
                               tvb, offset, 1, ENC_NA);
    val2 = tvb_get_guint8(tvb, offset);
    if ((val2 > 63 && val2 != 255) || val2 < val ||
        (val == 255 && val2 != 255) || (val != 255 && val2 == 255)) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                                  "Invalid DSCP Value");
    }
    offset++;

    if (val == 255 && val2 == 255) {
      proto_item_append_text(dscp_item, " (UP %u not in use)", i);
    } else {
      proto_item_append_text(dscp_item, " (0x%02x-0x%02x: UP %u)",
                             val, val2, i);
    }
  }

  return tvb_captured_length(tvb);
}

static int
dissect_roaming_consortium(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item* item;
  guint8 oi_lens, oi1_len, oi2_len;

  proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_num_anqp_oi,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  oi_lens = tvb_get_guint8(tvb, offset);
  oi1_len = oi_lens & 0x0f;
  oi2_len = (oi_lens & 0xf0) >> 4;
  proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_oi1_len,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_oi2_len,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  if (offset + oi1_len > tag_len) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Truncated Roaming Consortium element");
    return tvb_captured_length(tvb);
  }

  item = proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_oi1,
                             tvb, offset, oi1_len, ENC_NA);
  add_manuf(item, tvb, offset);
  offset += oi1_len;

  if (offset + oi2_len > tag_len) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Truncated Roaming Consortium element");
    return tvb_captured_length(tvb);
  }

  if (oi2_len > 0) {
    item = proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_oi2,
                        tvb, offset, oi2_len, ENC_NA);
    add_manuf(item, tvb, offset);
    offset += oi2_len;
  }

  if (tag_len > offset) {
    proto_tree_add_item(tree, hf_ieee80211_tag_roaming_consortium_oi3,
                        tvb, offset, tag_len - offset, ENC_NA);
  }

  return tvb_captured_length(tvb);
}


/* ************************************************************************* */
/*           Dissect and add tagged (optional) fields to proto tree          */
/* ************************************************************************* */

static int beacon_padding = 0; /* beacon padding bug */

static int
ieee80211_tag_ssid(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.1 SSID element (0) */
  gchar *ssid; /* The SSID may consist of arbitrary bytes */
  gint ssid_len = tag_len;

  if (beacon_padding != 0) /* padding bug */
    return offset;

  if (ssid_len > MAX_SSID_LEN) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "SSID length (%u) greater than maximum (%u)",
                           ssid_len, MAX_SSID_LEN);
    ssid_len = MAX_SSID_LEN;
  }

  /*
   * XXX - the 802.11 specs aren't particularly clear on how the SSID
   * is to be interpreted.
   *
   * IEEE Std 802.11-1999, section 7.3.2.2 "Service Set Identity (SSID)
   * element" says just
   *
   *    The length of the SSID information field is between 0 and 32
   *    octets. A 0 length information field indicates the broadcast SSID.
   *
   * with no indication that those octets encode a string.
   *
   * IEEE Std 802.11-2012, section 8.4.2.2 "SSID element", says that *but*
   * says after it
   *
   *    When the UTF-8 SSID subfield of the Extended Capabilities element
   *    is equal to 1 in the frame that includes the SSID element, the
   *    SSID is interpreted using UTF-8 encoding.
   *
   *    NOTE -- This is true for Beacon and Probe Response frames when the
   *    MLME-START.request primitive was issued with the SSIDEncoding
   *    parameter equal to UTF-8.
   *
   * and the SSIDEncoding parameter can either be UNSPECIFIED or UTF-8.
   *
   * So I *guess* that means that, if the UTF-8 SSID subfield isn't
   * equal to 1, the SSID is, in theory, just a bunch of octets, but
   * in practice, *probably* ASCII as that's the typical convention,
   * and, if it is equal to 1, it's a UTF-8 string.  (Of course, a
   * host can put anything there it wants to, so we shouldn't just
   * assume that it's *valid* ASCII or *valid* UTF-8.)
   *
   * So we really should extract it as an array of ssid_len bytes,
   * pass those bytes to Dot11DecryptSetLastSSID(), and:
   *
   *    If the UTF-8 SSID subfield isn't set to 1, put the SSID in
   *    as an ENC_ASCII string;
   *
   *    If the UTF-8 SSID subfield is set to 1, put it in as an
   *    ENC_UTF_8 string;
   *
   * and rely on the libwireshark core code to somehow deal with
   * non-ASCII characters or invalid UTF-8 sequences or valid-but-
   * not-all-printable ASCII or UTF-8 strings in the protocol tree
   * display.  I'm not sure we can currently rely on it to handle
   * invalid UTF-8 or non-printable characters in UTF-8 strings,
   * however, so we just treat it as ASCII for now.
   *
   * We also need a better way of getting the display format of a
   * string value, so we can do something other than run it through
   * format_text(), which won't handle UTF-8.
   *
   * Addendum: 802.11 2012 points out that a Zero-length SSID means
   * the Wildcard SSID. Make it so. From 8.4.2.2 of 802.11 2012:
   *
   * "The length of the SSID field is between 0 and 32 octets. A SSID
   *  field of length 0 is used within Probe Request management frames to
   *  indicate the wildcard SSID. The wildcard SSID is also used in
   *  Beacon and Probe Response frames transmitted by mesh STAs."
   *
   * Also, we have to return a non-zero value here to prevent an ugly
   * undissected field warning. Since this code is only called from
   * one place and is used in call to dissector_try_uint_new, it is
   * OK to do so.
   */
  ssid = tvb_get_string_enc(wmem_packet_scope(), tvb, offset, ssid_len, ENC_ASCII);
  if (ssid_len == (gint)tag_len) {
    Dot11DecryptSetLastSSID(&dot11decrypt_ctx, (CHAR *) ssid, ssid_len);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_ssid, tvb, offset, tag_len,
                      ENC_ASCII|ENC_NA);

  if (ssid_len > 0) {
    gchar* s = format_text(wmem_packet_scope(), ssid, ssid_len);
    proto_item_append_text(field_data->item_tag, ": %s", s);

    col_append_fstr(pinfo->cinfo, COL_INFO, ", SSID=%s", s);

    /* Wlan Stats */
    memcpy(wlan_stats.ssid, ssid, MIN(ssid_len, MAX_SSID_LEN));
    wlan_stats.ssid_len = ssid_len;
  } else {
    proto_item_append_text(field_data->item_tag, ": Wildcard SSID");

    col_append_str(pinfo->cinfo, COL_INFO, ", SSID=Wildcard (Broadcast)");
    offset += 1; // Make sure we return non-zero
  }

  beacon_padding += 1; /* padding bug */

  return offset + tag_len;
}

static void
dissect_he_capabilities(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset, int len);

static void
dissect_he_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset, int len _U_);

static int
dissect_neighbor_report(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint8 sub_tag_id;
  guint32 sub_tag_len;
  const gchar *sub_tag_name;
  proto_item *parent_item;
  proto_tree *bssid_info_subtree, *bssid_info_cap_subtree, *sub_tag_tree;
  tvbuff_t *sub_tag_tvb = NULL;

  if (tag_len < 13) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Neighbor Report length %u wrong, must be > 13", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_neighbor_report_bssid, tvb, offset, 6, ENC_NA);
  offset += 6;

  /*** Begin: BSSID Information ***/

  parent_item = proto_tree_add_item(tree, hf_ieee80211_tag_neighbor_report_bssid_info, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  bssid_info_subtree = proto_item_add_subtree(parent_item, ett_tag_neighbor_report_bssid_info_tree);

  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_reachability, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_security, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_key_scope, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  parent_item = proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  bssid_info_cap_subtree = proto_item_add_subtree(parent_item, ett_tag_neighbor_report_bssid_info_capability_tree);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_spec_mng, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_qos, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_apsd, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_radio_msnt, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_dback, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_cap_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_capability_iback, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_mobility_domain, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_high_throughput, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_very_high_throughput, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_ftm, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_high_efficiency, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_er_bss, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bssid_info_subtree, hf_ieee80211_tag_neighbor_report_bssid_info_reserved, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_neighbor_report_ope_class, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_neighbor_report_channel_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_neighbor_report_phy_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  /* The Optional Subelements field format contains zero or more subelements */
  if (tag_len == 13){ /* tag_len == 13 => no Subelements */
    return tvb_captured_length(tvb);
  }

  while (offset < tag_len)
  {
    sub_tag_id = tvb_get_guint8(tvb, offset);
    sub_tag_len = tvb_get_guint8(tvb, offset + 1);
    sub_tag_name = val_to_str(sub_tag_id, ieee80211_neighbor_report_subelement_id_vals, "Unknown");

    sub_tag_tree = proto_tree_add_subtree_format(tree, tvb, offset, sub_tag_len + 2, ett_tag_neighbor_report_subelement_tree, NULL, "Subelement: %s", sub_tag_name);

    proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_id, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    sub_tag_tvb = tvb_new_subset_length(tvb, offset, sub_tag_len);

    switch (sub_tag_id) {
      case NR_SUB_ID_TSF_INFO:
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_tsf_offset, tvb, offset, 2, ENC_NA);
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_beacon_interval, tvb, offset + 2, 2, ENC_NA);
        break;
      case NR_SUB_ID_MEASUREMENT_PILOT_INFO:
        dissect_measurement_pilot_trans_ie(sub_tag_tvb, pinfo, sub_tag_tree, data);
        break;
      case NR_SUB_ID_CON_COU_STR:
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_country_code, tvb, offset, 2, ENC_ASCII | ENC_NA);
        break;
      case NR_SUB_ID_BSS_TRN_CAN_PREF:
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_bss_trn_can_pref, tvb, offset, 1, ENC_NA);
        break;
      case NR_SUB_ID_BSS_TER_DUR:
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_bss_ter_tsf, tvb, offset, 8, ENC_NA);
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_bss_dur, tvb, offset + 8, 2, ENC_NA);
        break;
      case NR_SUB_ID_HT_CAPABILITIES:
        dissect_ht_capability_ie_common(sub_tag_tvb, pinfo, sub_tag_tree, 0, sub_tag_len, field_data->item_tag_length, FALSE);
        break;
      case NR_SUB_ID_HT_OPERATION:
        dissect_ht_info_ie_1_1(sub_tag_tvb, pinfo, sub_tag_tree, data);
        break;
      case NR_SUB_ID_SEC_CHANNEL_OFFSET:
        dissect_secondary_channel_offset_ie(sub_tag_tvb, pinfo, sub_tag_tree, data);
        break;
      case NR_SUB_ID_HT_MULTIPLE_BSSID:
        dissect_multiple_bssid_ie(sub_tag_tvb, pinfo, sub_tag_tree, data);
        break;
      case NR_SUB_ID_HE_CAPABILITIES:
        dissect_he_capabilities(sub_tag_tvb, pinfo, sub_tag_tree, 0, sub_tag_len);
        break;
      case NR_SUB_ID_HE_OPERATION:
        dissect_he_operation(sub_tag_tvb, pinfo, sub_tag_tree, 0, sub_tag_len);
        break;
      case NR_SUB_ID_VENDOR_SPECIFIC:
      default:
        proto_tree_add_item(sub_tag_tree, hf_ieee80211_tag_neighbor_report_subelement_data, tvb, offset, sub_tag_len, ENC_NA);
        break;
    }

    offset += sub_tag_len;
  }

  return offset;
}

static int
ieee80211_tag_supp_rates(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.2 Supported Rates element (1) */
  if (tag_len < 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be greater than 0",
                           tag_len);
    return offset;
  }

  while (offset < tag_len) {
    proto_tree_add_item(tree, hf_ieee80211_tag_supp_rates, tvb, offset, 1,
                        ENC_LITTLE_ENDIAN);
    proto_item_append_text(field_data->item_tag, " %s,",
                           val_to_str_ext_const(tvb_get_guint8(tvb, offset),
                                                &ieee80211_supported_rates_vals_ext,
                                                "Unknown Rate"));
    offset += 1;
  }

  proto_item_append_text(field_data->item_tag, " [Mbit/sec]");

  return offset;
}

static int
ieee80211_tag_fh_parameter(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.3 FH Parameter Set element (2) */
  if (tag_len < 5) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be >= 5", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_dwell_time,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hop_set,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hop_pattern,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hop_index,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

static int
ieee80211_tag_ds_parameter(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.4 DS Parameter Set element (3) */
  if (tag_len != 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
        "Tag length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ds_param_channel,
        tvb, offset, 1, ENC_LITTLE_ENDIAN);

  proto_item_append_text(field_data->item_tag, ": Current Channel: %u",
                         tvb_get_guint8(tvb, offset));

  wlan_stats.channel = tvb_get_guint8(tvb, offset);
  offset += 1;

  return offset;
}

static int
ieee80211_tag_cf_parameter(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.5 CF Parameter Set element (4) */
  if (tag_len != 6) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u wrong, must be = 6", tag_len);
    return offset;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_cfp_count,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": CFP count %u", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_cfp_period,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": CFP Period %u", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_cfp_max_duration,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": CFP Max Duration %u",
                         tvb_get_letohs(tvb, offset));
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_cfp_dur_remaining,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": CFP Dur Remaining %u",
                         tvb_get_letohs(tvb, offset));
  offset += 1;

  return offset;
}

static int
ieee80211_tag_tim(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint aid, pvb_len, n1, i, j, byte;
  static const int *ieee80211_tim_bmapctl[] = {
    &hf_ieee80211_tim_bmapctl_mcast,
    &hf_ieee80211_tim_bmapctl_offset,
    NULL
  };

  /* 802.11-2012: 8.4.2.7 TIM element (5) */
  if (tag_len < 4) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be >= 4", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tim_dtim_count,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": DTIM %u of", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tim_dtim_period,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " %u bitmap", tvb_get_guint8(tvb, offset + 1));
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tim_bmapctl,
                                    ett_tag_bmapctl_tree, ieee80211_tim_bmapctl,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  pvb_len = tag_len - 3;
  n1 = tvb_get_guint8(tvb, offset) & 0xFE;
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tim_partial_virtual_bitmap,
                      tvb, offset, pvb_len, ENC_NA);
  /* FIXME: Handles dot11MgmtOptionMultiBSSIDActivated = false only */
  for (i = 0; i < pvb_len; i++) {
    byte = tvb_get_guint8(tvb, offset + i);
    for (j = 0; j < 8; j++) {
      if (byte & (1 << j)) {
        aid = 8*n1 + 8*i + j;
        proto_tree_add_uint(tree, hf_ieee80211_tim_aid, tvb, offset + i, 1, aid);
      }
    }
  }
  offset += pvb_len;

  return offset;
}

static int
ieee80211_tag_ibss_parameter(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.7 IBSS Parameter Set element (6) */

  if (tag_len != 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u wrong, must be = 2", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ibss_atim_window,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": ATIM window 0x%x",
                         tvb_get_letohs(tvb, offset));
  offset += 2;

  return offset;
}

static const value_string environment_vals[] = {
  { 0x20, "Any" },
  { 0x4f, "Outdoor" },
  { 0x49, "Indoor" },
  { 0,    NULL }
};

static int
ieee80211_tag_country_info(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* 7.3.2.9 Country information element (7) */
  proto_tree *sub_tree;
  proto_item *sub_item;
  const guint8* country_code;

  if (tag_len < 6) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be >= 6", tag_len);
    return 1;
  }

  proto_tree_add_item_ret_string(tree, hf_ieee80211_tag_country_info_code,
                      tvb, offset, 2, ENC_ASCII|ENC_NA, wmem_packet_scope(), &country_code);
  proto_item_append_text(field_data->item_tag, ": Country Code %s", country_code);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_country_info_env,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Environment %s",
                         val_to_str(tvb_get_guint8(tvb, offset),
                                    environment_vals, "Unknown (0x%02x)"));
  offset += 1;

  while (offset < tag_len) {
    /* Padding ? */
    if ((tag_len - offset) < 3) {
      proto_tree_add_item(tree, hf_ieee80211_tag_country_info_pad,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      continue;
    }
    if (tvb_get_guint8(tvb, offset) <= 200) { /* 802.11d */
      sub_item = proto_tree_add_item(tree, hf_ieee80211_tag_country_info_fnm,
                                     tvb, offset, 3, ENC_NA);
      sub_tree = proto_item_add_subtree(sub_item, ett_tag_country_fnm_tree);

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_fnm_fcn,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item, ": First Channel Number: %u",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_fnm_nc,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item, ", Number of Channels: %u",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_fnm_mtpl,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item,
                             ", Maximum Transmit Power Level: %d dBm",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
    } else { /* 802.11j */
      sub_item = proto_tree_add_item(tree, hf_ieee80211_tag_country_info_rrc,
                                     tvb, offset, 3, ENC_NA);
      sub_tree = proto_item_add_subtree(sub_item, ett_tag_country_rcc_tree);

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_rrc_oei,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item,
                             ": Operating Extension Identifier: %u",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_rrc_oc,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item, ", Operating Class: %u",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_country_info_rrc_cc,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(sub_item, ", Coverage Class: %u",
                             tvb_get_guint8(tvb, offset));
      offset += 1;
    }
  }

  return offset;
}

static int
ieee80211_tag_fh_hopping_parameter(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  /* 7.3.2.10 Hopping Pattern Parameters information element (8) */
  if (tag_len < 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be >= 2", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_parameter_prime_radix,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": Prime Radix: %u", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_parameter_nb_channels,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Number of Channels: %u",
                         tvb_get_guint8(tvb, offset));
  offset += 1;

  return offset;
}

static int
ieee80211_tag_fh_hopping_table(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  /* 7.3.2.11 Hopping Pattern Table information element (9) */
  if (tag_len < 4) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag length %u too short, must be >= 4", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_table_flag,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_table_number_of_sets,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_table_modulus,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_table_offset,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  while (offset < tag_len) {
    proto_tree_add_item(tree, hf_ieee80211_tag_fh_hopping_random_table,
                        tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;
  }

  return offset;
}

int
add_tagged_field(packet_info *pinfo, proto_tree *tree, tvbuff_t *tvb, int offset, int ftype,
                 const guint8 *valid_element_ids, guint valid_element_ids_count,
                 association_sanity_check_t *association_sanity_check)
{
  tvbuff_t     *tag_tvb;
  guint32       tag_no, tag_len;
  guint32       ext_tag_no, ext_tag_len;
  proto_tree   *orig_tree = tree;
  proto_item   *ti        = NULL;
  proto_item   *ti_len, *ti_tag;
  ieee80211_tagged_field_data_t field_data;
  gboolean      isDMG;

  isDMG = GPOINTER_TO_INT(p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_DMG_KEY));

  tag_no  = tvb_get_guint8(tvb, offset);
  tag_len = tvb_get_guint8(tvb, offset + 1);

  if (tree) {
    if (tag_no == TAG_ELEMENT_ID_EXTENSION) {
      ext_tag_no  = tvb_get_guint8(tvb, offset + 2);
      ti = proto_tree_add_item(orig_tree, hf_ieee80211_ext_tag, tvb, offset + 2, tag_len , ENC_NA);
      proto_item_append_text(ti, ": %s", val_to_str_ext(ext_tag_no, &tag_num_vals_eid_ext_ext, "Reserved (%d)"));
    } else {
      ti = proto_tree_add_item(orig_tree, hf_ieee80211_tag, tvb, offset, 2 + tag_len , ENC_NA);
      proto_item_append_text(ti, ": %s", val_to_str_ext(tag_no, &tag_num_vals_ext, "Reserved (%d)"));
    }

    tree = proto_item_add_subtree(ti, ett_80211_mgt_ie);

  }

  if (tag_no == TAG_ELEMENT_ID_EXTENSION) {
    ext_tag_len = tag_len - 1;
    proto_tree_add_item(tree, hf_ieee80211_tag_number, tvb, offset, 1, ENC_NA);
    ti_len = proto_tree_add_uint(tree, hf_ieee80211_ext_tag_length, tvb, offset + 1, 1, ext_tag_len);
    ti_tag = proto_tree_add_item(tree, hf_ieee80211_ext_tag_number, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
  } else {
    ti_tag = proto_tree_add_item(tree, hf_ieee80211_tag_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    ti_len = proto_tree_add_uint(tree, hf_ieee80211_tag_length, tvb, offset + 1, 1, tag_len);
  }
  if (tag_len > (guint)tvb_reported_length_remaining(tvb, offset)) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "Tag Length is longer than remaining payload");
  }

  /* If the list of valid elements is restricted, require an Element ID to be
   * present in that list. Otherwise stop decoding the value to prevent possible
   * infinite recursions due to unexpected elements. */
  if (valid_element_ids_count) {
    gboolean valid_tag_no = FALSE;
    guint i;

    for (i = 0; i < valid_element_ids_count; i++) {
      valid_tag_no = valid_element_ids[i] == tag_no;
      if (valid_tag_no)
        break;
    }
    if (!valid_tag_no) {
      expert_add_info_format(pinfo, ti_tag, &ei_ieee80211_tag_number,
          "Unexpected Element ID %d", tag_no);
        return tag_len + 1 + 1;
    }
  }

  tag_tvb = tvb_new_subset_length(tvb, offset+2, tag_len);
  field_data.sanity_check = association_sanity_check;
  field_data.ftype = ftype;
  field_data.isDMG = isDMG;
  field_data.item_tag = ti;
  field_data.item_tag_length = ti_len;
  if (!dissector_try_uint_new(tagged_field_table, tag_no, tag_tvb, pinfo, tree, FALSE, &field_data))
  {
      proto_tree_add_item(tree, hf_ieee80211_tag_data, tvb, offset + 1 + 1, tag_len, ENC_NA);
      expert_add_info_format(pinfo, ti_tag, &ei_ieee80211_tag_data,
                             "Dissector for 802.11 IE Tag"
                             " (%s) code not implemented, Contact"
                             " Wireshark developers if you want this supported", val_to_str_ext(tag_no,
                                            &tag_num_vals_ext, "(%d)"));
      proto_item_append_text(ti, ": Undecoded");
  }

  return tag_len + 1 + 1;
}

/* 7.3.2.12 Request information element (10) */
static int
ieee80211_tag_request(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  while (offset < tag_len)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_request, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
  }
  return ((tag_len > 0) ? tag_len : 1);
}

/* 7.3.2.28 BSS Load element (11) */
/* 8.4.2.30 in 802.11-2012 */
static int
ieee80211_tag_qbss_load(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if ((tag_len < 4) || (tag_len > 5))
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 4 or 5", tag_len);
    return tvb_captured_length(tvb);
  }

  if (tag_len == 4)
  {
    /* QBSS Version 1 */
    proto_item_append_text(field_data->item_tag, " Cisco QBSS Version 1 - non CCA");

    /* Extract Values */
    proto_tree_add_uint(tree, hf_ieee80211_qbss_version, tvb, offset, tag_len, 1);
    proto_tree_add_item(tree, hf_ieee80211_qbss_scount, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(tree, hf_ieee80211_qbss_cu, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(tree, hf_ieee80211_qbss_adc, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  }
  else if (tag_len == 5)
  {
    proto_item *base_item;

    /* QBSS Version 2 */
    proto_item_append_text(field_data->item_tag, " 802.11e CCA Version");

    /* Extract Values */
    proto_tree_add_uint(tree, hf_ieee80211_qbss_version, tvb, offset, tag_len, 2);
    proto_tree_add_item(tree, hf_ieee80211_qbss_scount, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    base_item = proto_tree_add_item(tree, hf_ieee80211_qbss_cu, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(base_item, " (%d%%)", 100*tvb_get_guint8(tvb, offset + 2)/255);
    base_item = proto_tree_add_item(tree, hf_ieee80211_qbss_adc, tvb, offset + 3, 2, ENC_LITTLE_ENDIAN);
    proto_item_append_text(base_item, " (%d us/s)", tvb_get_letohs(tvb, offset + 3)*32);
  }

  return tvb_captured_length(tvb);
}

/* 8.4.2.31 in 802-11-2012 */
static int
ieee80211_tag_edca_param_set(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if ((tag_len != 18))
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 18", tag_len);
    return tvb_captured_length(tvb);
  }

  decode_qos_parameter_set(tree, tvb, pinfo, offset, field_data->ftype);

  return tvb_captured_length(tvb);
}

/* TSPEC element (13) */
static int
ieee80211_tag_tspec(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (field_data->isDMG == FALSE && tag_len != 55)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 55", tag_len);
    return tvb_captured_length(tvb);
  }
  if (field_data->isDMG == TRUE && tag_len != 57)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 57", tag_len);
    return tvb_captured_length(tvb);
  }

  add_ff_qos_ts_info(tree, tvb, pinfo, offset);
  offset += 3;

  proto_tree_add_item(tree, hf_ieee80211_tspec_nor_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tspec_max_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tspec_min_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_max_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_inact_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_susp_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_srv_start, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_min_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_mean_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_peak_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_burst_size, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_delay_bound, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_min_phy, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tspec_surplus, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tspec_medium, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if(field_data->isDMG == TRUE) {
    proto_tree_add_item(tree, hf_ieee80211_tspec_dmg, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    /*offset +=2;*/
  }

  return tvb_captured_length(tvb);
}

/* 7.3.2.31 TCLAS element (14) */
static int
ieee80211_tag_tclas(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint8 type;
  guint8 version;
  static const int *ieee80211_tclas_class_mask0[] = {
    &hf_ieee80211_tclas_class_mask0_src_addr,
    &hf_ieee80211_tclas_class_mask0_dst_addr,
    &hf_ieee80211_tclas_class_mask0_type,
    NULL
  };

  static const int *ieee80211_tclas_class_mask1[] = {
    &hf_ieee80211_tclas_class_mask1_ver,
    &hf_ieee80211_tclas_class_mask1_src_ip,
    &hf_ieee80211_tclas_class_mask1_dst_ip,
    &hf_ieee80211_tclas_class_mask1_src_port,
    &hf_ieee80211_tclas_class_mask1_dst_port,
    &hf_ieee80211_tclas_class_mask1_ipv6_flow,
    NULL
  };

  static const int *ieee80211_tclas_class_mask1_4[] = {
    &hf_ieee80211_tclas_class_mask1_ver,
    &hf_ieee80211_tclas_class_mask1_src_ip,
    &hf_ieee80211_tclas_class_mask1_dst_ip,
    &hf_ieee80211_tclas_class_mask1_src_port,
    &hf_ieee80211_tclas_class_mask1_dst_port,
    &hf_ieee80211_tclas_class_mask1_ipv4_dscp,
    &hf_ieee80211_tclas_class_mask1_ipv4_proto,
    NULL
  };

  static const int *ieee80211_tclas_class_mask2[] = {
    &hf_ieee80211_tclas_class_mask2_tci,
    NULL
  };

  if (tag_len < 5)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 5", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tclas_up, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tclas_class_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  switch (type)
  {
  case 0:
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tclas_class_mask,
                                    ett_tag_tclas_mask_tree, ieee80211_tclas_class_mask0,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset++;

    proto_tree_add_item(tree, hf_ieee80211_tclas_src_mac_addr, tvb, offset, 6, ENC_NA);
    offset += 6;

    proto_tree_add_item(tree, hf_ieee80211_tclas_dst_mac_addr, tvb, offset, 6, ENC_NA);
    offset += 6;

    proto_tree_add_item(tree, hf_ieee80211_tclas_ether_type, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    /*offset += 2;*/
    break;

  case 1:
    version = tvb_get_guint8(tvb, offset+1);
    if (version == 4) {
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tclas_class_mask,
                                    ett_tag_tclas_mask_tree, ieee80211_tclas_class_mask1_4,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    } else {
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tclas_class_mask,
                                    ett_tag_tclas_mask_tree, ieee80211_tclas_class_mask1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    }
    offset += 1;

    proto_tree_add_item(tree, hf_ieee80211_tclas_version, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if (version == 4)
    {
      proto_tree_add_item(tree, hf_ieee80211_tclas_ipv4_src, tvb, offset, 4, ENC_BIG_ENDIAN);
      offset += 4;
      proto_tree_add_item(tree, hf_ieee80211_tclas_ipv4_dst, tvb, offset, 4, ENC_BIG_ENDIAN);
      offset += 4;
      proto_tree_add_item(tree, hf_ieee80211_tclas_src_port, tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_tclas_dst_port, tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_tclas_dscp, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_tclas_protocol, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      /*offset += 1;*/
    }
    else if (version == 6)
    {
      proto_tree_add_item(tree, hf_ieee80211_tclas_ipv6_src, tvb, offset, 16, ENC_NA);
      offset += 16;
      proto_tree_add_item(tree, hf_ieee80211_tclas_ipv6_dst, tvb, offset, 16, ENC_NA);
      offset += 16;
      proto_tree_add_item(tree, hf_ieee80211_tclas_src_port, tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_tclas_dst_port, tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_tclas_flow, tvb, offset, 3, ENC_BIG_ENDIAN);
      /*offset += 3;*/
    }
    break;

  case 2:
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tclas_class_mask,
                                    ett_tag_tclas_mask_tree, ieee80211_tclas_class_mask2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset++;

    proto_tree_add_item(tree, hf_ieee80211_tclas_tag_type, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    /*offset += 2;*/
    break;

  default:
    break;
  }

  return tvb_captured_length(tvb);
}

/* 7.3.2.34 Schedule element (15) */
static int
ieee80211_tag_schedule(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 14)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 14", tag_len);
    return 1;
  }

  add_ff_schedule_info(tree, tvb, pinfo, offset);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_sched_srv_start, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_sched_srv_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_sched_spec_int, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

/* 7.3.2.8 Challenge Text element (16) */
static int
ieee80211_tag_challenge_text(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_tag_challenge_text, tvb, offset, tag_len, ENC_NA);

  return ((tag_len > 0) ? tag_len : 1);
}

/* 7.3.2.15 Power Constraint element (32) */
static int
ieee80211_tag_power_constraint(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 1", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_power_constraint_local, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ": %d", tvb_get_guint8(tvb, offset));
  return tvb_captured_length(tvb);
}

/* 7.3.2.16 Power Capability element (33) */
static int
ieee80211_tag_power_capability(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 2)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 2", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_power_capability_min, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " Min: %d", tvb_get_gint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_power_capability_max, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Max: %d", tvb_get_gint8(tvb, offset));
  return tvb_captured_length(tvb);
}

/* 7.3.2.18 TPC Request element (34) */
static int
ieee80211_tag_tpc_request(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree _U_, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  if (tag_len != 0)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 0", tag_len);
    return 1; /* Even with no data, we can't return 0 */
  }

  return 1; /* Even with no data, we can't return 0 */
}

/* 7.3.2.18 TPC Report element (35) */
static int
ieee80211_tag_tpc_report(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 2)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 2", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_tpc_report_trsmt_pow, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " Transmit Power: %d", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_tpc_report_link_mrg, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Link Margin: %d", tvb_get_guint8(tvb, offset));
  return tvb_captured_length(tvb);
}

/* 7.3.2.19 Supported Channels element (36) */
static int
ieee80211_tag_supported_channels(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  proto_item *chan_item;
  proto_tree *chan_tree;
  guint       i = 1;

  if (tag_len % 2 == 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u must be even", tag_len);
    return tvb_captured_length(tvb);
  }

  while (offset < tag_len)
  {
    chan_item = proto_tree_add_item(tree, hf_ieee80211_tag_supported_channels, tvb, offset, 2, ENC_NA);
    proto_item_append_text(chan_item, " #%d", i);
    i += 1;

    chan_tree = proto_item_add_subtree(chan_item , ett_tag_supported_channels);

    proto_tree_add_item(chan_tree, hf_ieee80211_tag_supported_channels_first, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(chan_item, " First: %d", tvb_get_guint8(tvb, offset));
    offset += 1;

    proto_tree_add_item(chan_tree, hf_ieee80211_tag_supported_channels_range, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(chan_item, ", Range: %d ", tvb_get_guint8(tvb, offset));
    offset += 1;

  }
  return tvb_captured_length(tvb);
}

/* 7.3.2.20 Channel Switch Announcement element (37) */
static int
ieee80211_tag_switch_ann(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 3)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 3", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_csa_channel_switch_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " Mode: %d", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_csa_new_channel_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Number: %d ", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_csa_channel_switch_count, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, ", Count: %d ", tvb_get_guint8(tvb, offset));
  return tvb_captured_length(tvb);
}

/* 7.3.2.21 Measurement Request element (38) with update from 802.11k-2008 */
static int
ieee80211_tag_measure_req(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint8 request_type;
  proto_item *parent_item;
  proto_tree *sub_tree;
  static const int *ieee80211_tag_measure_request_mode[] = {
    &hf_ieee80211_tag_measure_request_mode_parallel,
    &hf_ieee80211_tag_measure_request_mode_enable,
    &hf_ieee80211_tag_measure_request_mode_request,
    &hf_ieee80211_tag_measure_request_mode_report,
    &hf_ieee80211_tag_measure_request_mode_duration_mandatory,
    &hf_ieee80211_tag_measure_request_mode_reserved,
    NULL
  };

  if (tag_len < 3)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 3", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_measure_request_token, tvb, offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_measure_request_mode,
                                    ett_tag_measure_request_mode_tree, ieee80211_tag_measure_request_mode,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  parent_item = proto_tree_add_item(tree, hf_ieee80211_tag_measure_request_type, tvb, offset, 1, ENC_NA);
  sub_tree = proto_item_add_subtree(parent_item, ett_tag_measure_request_type_tree);
  request_type = tvb_get_guint8(tvb, offset);
  offset += 1;

  switch (request_type) {
    case 0: /* Basic Request */
    case 1: /* Clear channel assessment (CCA) request */
    case 2: /* Receive power indication (RPI) histogram request */
    {
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_number, tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
      offset += 8;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      break;
    }
    case 3: /* Channel Load Request */
    {
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_operating_class, tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_number, tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_randomization_interval, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      while (offset < tag_len)
      {
        guint8 sub_id;
        proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_load_sub_id, tvb, offset, 1, ENC_NA);
        sub_id = tvb_get_guint8(tvb, offset);
        offset += 1;

        proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_subelement_length, tvb, offset, 1, ENC_NA);
        offset += 1;

        switch (sub_id) {
          case MEASURE_REQ_CHANNEL_LOAD_SUB_REPORTING_INFO: /* Channel Load Reporting Information (1) */
            proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_load_sub_reporting_condition, tvb, offset, 1, ENC_NA);
            offset += 1;
            proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_load_sub_reporting_ref, tvb, offset, 1, ENC_NA);
            offset += 1;
            break;
          default:
            /* no default action */
            break;
          }
     }
     break;
   }
   case 4: /* Noise Histogram Request */
   {
     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_operating_class, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_number, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_randomization_interval, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     while (offset < tag_len)
     {
       guint8 sub_id;
       proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_noise_histogram_sub_id, tvb, offset, 1, ENC_NA);
       sub_id = tvb_get_guint8(tvb, offset);
       offset += 1;

       proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_subelement_length, tvb, offset, 1, ENC_NA);
       offset += 1;

       switch (sub_id) {
         case MEASURE_REQ_NOISE_HISTOGRAM_SUB_REPORTING_INFO: /* Noise Histogram Reporting Information (1) */
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_condition, tvb, offset, 1, ENC_NA);
           offset += 1;
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_anpi_ref, tvb, offset, 1, ENC_NA);
           offset += 1;
           break;
         default:
           /* no default action */
           break;
       }
     }
     break;
   }
   case 5: /* Beacon Request */
   {
     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_operating_class, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_number, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_randomization_interval, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_measurement_mode, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_bssid, tvb, offset, 6, ENC_NA);
     offset += 6;

     while (offset < tag_len)
     {
       guint8 sub_id, sub_length, sub_tag_end;
       proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_id, tvb, offset, 1, ENC_NA);
       sub_id = tvb_get_guint8(tvb, offset);
       offset += 1;

       proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_subelement_length, tvb, offset, 1, ENC_NA);
       sub_length = tvb_get_guint8(tvb, offset);
       offset += 1;
       sub_tag_end = offset + sub_length;

       switch (sub_id) {
         case MEASURE_REQ_BEACON_SUB_SSID: /* SSID (0) */
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_ssid, tvb, offset, sub_length, ENC_ASCII|ENC_NA);
           offset += sub_length;
           break;
         case MEASURE_REQ_BEACON_SUB_BRI: /* Beacon Reporting Information (1) */
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_bri_reporting_condition, tvb, offset, 1, ENC_LITTLE_ENDIAN);
           offset += 1;
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_bri_threshold_offset, tvb, offset, 1, ENC_LITTLE_ENDIAN);
           offset += 1;
           break;
         case MEASURE_REQ_BEACON_SUB_RD: /* Reporting Detail (2) */
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_reporting_detail, tvb, offset, 1, ENC_LITTLE_ENDIAN);
           offset += 1;
           break;
         case MEASURE_REQ_BEACON_SUB_REQUEST: /* Request (10) */
           proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_sub_request, tvb, offset, 1, ENC_LITTLE_ENDIAN);
           offset += 1;
           break;
         case MEASURE_REQ_BEACON_SUB_APCP: /* AP Channel Report (51) */
           /* TODO */
           break;
         default:
           /* no default action */
           break;
       }
       if (offset < sub_tag_end)
       {
         proto_item *tix;
         tix = proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_beacon_unknown, tvb, offset, sub_tag_end - offset, ENC_NA);
         expert_add_info(pinfo, tix, &ei_ieee80211_tag_measure_request_beacon_unknown);
         offset = sub_tag_end;
       }
     }

     break;
   }
   case 6: /* Frame Request */
   {
     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_operating_class, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_channel_number, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_randomization_interval, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_frame_request_type, tvb, offset, 1, ENC_NA);
     offset += 1;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_mac_address, tvb, offset, 6, ENC_NA);
     offset += 6;

     /* TODO Add Optional Subelements */
     break;
   }
   case 7: /* BSTA Statistics Request */
   {
     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_peer_mac_address, tvb, offset, 6, ENC_NA);
     offset += 6;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_randomization_interval, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
     offset += 2;

     proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_group_id, tvb, offset, 1, ENC_NA);
     offset += 1;

     /* TODO Add Optional Subelements */
     break;
   }
   case 8: /* Location Configuration Indication (LCI) Request */
    /* TODO */
   case 9: /* Transmit Stream Measurement Request */
    /* TODO */
   case 10: /* Multicast diagnostics request */
    /* TODO */
   case 11: /* Location Civic request */
    /* TODO */
   case 12: /* Location Identifier request */
    /* TODO */
   case 13: /* Directional channel quality request */
    /* TODO */
   case 14: /* Directional measurement request */
    /* TODO */
   case 15: /* Directional statistics request */
    /* TODO */
   case 255: /* Measurement Pause Request*/
    /* TODO */
   default: /* unknown */
    break;
  }
  if (offset < tag_len)
  {
    proto_item *tix;
    tix = proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_request_unknown, tvb, offset, tag_len - offset, ENC_NA);
    expert_add_info(pinfo, tix, &ei_ieee80211_tag_measure_request_unknown);
  }

  return tvb_captured_length(tvb);
}

/* 7.3.2.22 Measurement Report element (39) with update from 802.11k-2008 */
static int
ieee80211_tag_measure_rep(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *parent_item;
  proto_tree *sub_tree;
  guint8 report_type;
  static const int *ieee80211_tag_measure_report_mode[] = {
    &hf_ieee80211_tag_measure_report_mode_late,
    &hf_ieee80211_tag_measure_report_mode_incapable,
    &hf_ieee80211_tag_measure_report_mode_refused,
    &hf_ieee80211_tag_measure_report_mode_reserved,
    NULL
  };
  static const int *ieee80211_tag_measure_map_field[] = {
    &hf_ieee80211_tag_measure_map_field_bss,
    &hf_ieee80211_tag_measure_map_field_ofdm,
    &hf_ieee80211_tag_measure_map_field_unident_signal,
    &hf_ieee80211_tag_measure_map_field_radar,
    &hf_ieee80211_tag_measure_map_field_unmeasured,
    &hf_ieee80211_tag_measure_map_field_reserved,
    NULL
  };
  static const int *ieee80211_tag_measure_report_frame_info[] = {
    &hf_ieee80211_tag_measure_report_frame_info_phy_type,
    &hf_ieee80211_tag_measure_report_frame_info_frame_type,
    NULL
  };

  if (tag_len < 3)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 3", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_measure_report_measurement_token, tvb, offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_measure_report_mode,
                                    ett_tag_measure_report_mode_tree, ieee80211_tag_measure_report_mode,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  report_type = tvb_get_guint8(tvb, offset);
  parent_item = proto_tree_add_item(tree, hf_ieee80211_tag_measure_report_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  sub_tree = proto_item_add_subtree(parent_item, ett_tag_measure_report_type_tree);
  offset += 1;

  if (tag_len == 3)
    return tvb_captured_length(tvb);

  switch (report_type) {
  case 0: /* Basic Report */
  {
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_measure_basic_map_field,
                                    ett_tag_measure_report_basic_map_tree, ieee80211_tag_measure_map_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    break;
  }
  case 1: /* Clear channel assessment (CCA) report */
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_cca_busy_fraction, tvb, offset, 1, ENC_NA);
    offset += 1;
    break;
  case 2: /* Receive power indication (RPI) histogram report */
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    parent_item = proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report, tvb, offset, 8, ENC_NA);
    sub_tree = proto_item_add_subtree(parent_item, ett_tag_measure_report_rpi_tree);

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_0, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_1, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_2, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_3, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_4, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_5, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_6, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_rpi_histogram_report_7, tvb, offset, 1, ENC_NA);
    offset += 1;
    break;
  case 3: /* Channel Load Report */
  {
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_operating_class, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_load, tvb, offset, 1, ENC_NA);
    offset += 1;

    /* TODO Add Optional Subelements */
    break;
  }
  case 4: /* Noise Histogram Report */
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_operating_class, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ant_id, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_anpi, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_0, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_1, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_2, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_3, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_4, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_5, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_6, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_7, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_8, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_9, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ipi_density_10, tvb, offset, 1, ENC_NA);
    offset += 1;

    /* TODO Add Optional Subelements */
    break;
  case 5: /* Beacon Report */
  {
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_operating_class, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_measure_report_frame_info,
                                    ett_tag_measure_report_frame_tree, ieee80211_tag_measure_report_frame_info,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_rcpi, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_rsni, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_bssid, tvb, offset, 6, ENC_NA);
    offset += 6;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_ant_id, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_parent_tsf, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;

    while (offset < tag_len)
    {
      guint8 sub_id, sub_length;
      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_beacon_sub_id, tvb, offset, 1, ENC_NA);
      sub_id = tvb_get_guint8(tvb, offset);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_subelement_length, tvb, offset, 1, ENC_NA);
      sub_length = tvb_get_guint8(tvb, offset);
      offset += 1;

      switch (sub_id) {
        case MEASURE_REP_REPORTED_FRAME_BODY: /* Reported Frame Body (1) */
        {
          proto_tree *rep_tree;

          rep_tree = proto_tree_add_subtree(sub_tree, tvb, offset, sub_length, ett_tag_measure_reported_frame_tree, NULL, "Reported Frame Body");

          add_ff_timestamp(rep_tree, tvb, pinfo, 0);
          add_ff_beacon_interval(rep_tree, tvb, pinfo, 8);
          add_ff_cap_info(rep_tree, tvb, pinfo, 10);
          offset += 12;

          ieee_80211_add_tagged_parameters(tvb, offset, pinfo, rep_tree, sub_length - 12, MGT_PROBE_RESP, NULL);
          offset += (sub_length - 12);
        }
        break;
        default:
          /* no default action */
          break;
      }
    }
    break;
  }
  case 6: /* Frame Report */
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_operating_class, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_channel_number, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_start_time, tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    /* TODO Add Optional Subelements */
    break;
  case 7: /* BSTA Statistics Report */
    /* TODO */
  case 8: /* Location Configuration Information Report element */
    /* TODO */
  case 9: /* Transmit Stream Measurement Report */
    /* TODO */
  case 10: /* Multicast diagnostics Report */
    /* TODO */
  case 11: /* Location Civic Report */
    /* TODO */
  case 12: /* Location Identifier Report */
    /* TODO */
  case 13: /* Directional channel quality Report */
    /* TODO */
  case 14: /* Directional measurement Report */
    /* TODO */
  case 15: /* Directional statistics Report */
    /* TODO */
  default: /* unknown */
    break;
  }
  if (offset < tag_len)
  {
    proto_item *tix;
    tix = proto_tree_add_item(sub_tree, hf_ieee80211_tag_measure_report_unknown, tvb, offset, tag_len - offset, ENC_NA);
    expert_add_info(pinfo, tix, &ei_ieee80211_tag_measure_report_unknown);
  }
  return tvb_captured_length(tvb);
}

/* 7.3.2.23 Quiet element (40) */
static int
ieee80211_tag_quiet(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 6)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 6", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_quiet_count, tvb, offset, 1, ENC_NA);
  proto_item_append_text(field_data->item_tag, " Count: %d", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_quiet_period, tvb, offset, 1, ENC_NA);
  proto_item_append_text(field_data->item_tag, " Period: %d", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_quiet_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " Duration: %d", tvb_get_letohs(tvb, offset));
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_quiet_offset, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " Offset: %d", tvb_get_letohs(tvb, offset));

  return tvb_captured_length(tvb);
}

/* 7.3.2.24 IBSS DFS element (41) */
static int
ieee80211_tag_ibss_dfs(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *ti_sup_map;
  proto_tree *sub_map_tree;
  if (tag_len < 7)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 7", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_dfs_owner, tvb, offset, 6, ENC_NA);
  proto_item_append_text(field_data->item_tag, " Owner: %s", tvb_ether_to_str(tvb, offset));
  offset += 6;

  proto_tree_add_item(tree, hf_ieee80211_tag_dfs_recovery_interval, tvb, offset, 1, ENC_NA);
  offset += 1;

  while (offset < tag_len)
  {
    ti_sup_map = proto_tree_add_item(tree, hf_ieee80211_tag_dfs_channel_map, tvb, offset, 2, ENC_NA);
    sub_map_tree = proto_item_add_subtree(ti_sup_map, ett_tag_dfs_map_tree);
    proto_tree_add_item(sub_map_tree, hf_ieee80211_tag_dfs_channel_number, tvb, offset, 1, ENC_NA);
    proto_tree_add_item(sub_map_tree, hf_ieee80211_tag_dfs_map, tvb, offset, 1, ENC_NA);
    offset += 2;
  }
  return tvb_captured_length(tvb);
}

/* 7.3.2.13 ERP Information element (42) */
static int
ieee80211_tag_erp_info(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int *ieee80211_tag_erp_info_flags[] = {
    &hf_ieee80211_tag_erp_info_erp_present,
    &hf_ieee80211_tag_erp_info_use_protection,
    &hf_ieee80211_tag_erp_info_barker_preamble_mode,
    &hf_ieee80211_tag_erp_info_reserved,
    NULL
  };

  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 1", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_erp_info,
                                    ett_tag_erp_info_tree, ieee80211_tag_erp_info_flags,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  return tvb_captured_length(tvb);
}

/* 7.3.2.32 TS Delay element (43) */
static int
ieee80211_tag_ts_delay(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 4)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 4", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_ts_delay, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " : %d", tvb_get_ntohl(tvb, offset));
  return tvb_captured_length(tvb);
}

/* 7.3.2.33 TCLAS Processing element (44) */
static int
ieee80211_tag_tclas_process(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 1", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tclas_process, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " : %s", val_to_str(tvb_get_guint8(tvb, offset), ieee80211_tclas_process_flag, "Unknown %d"));
  return tvb_captured_length(tvb);
}

/* 802.11-2012 8.4.2.37 QoS Capability element (46) */
static int
ieee80211_tag_qos_capability(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 1", tag_len);
    return tvb_captured_length(tvb);
  }
  dissect_qos_capability(tree, tvb, pinfo, offset, field_data->ftype);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_rsn_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len < 18)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 18", tag_len);
    return tvb_captured_length(tvb);
  }

  dissect_rsn_ie(pinfo, tree, tvb, offset, tag_len, field_data->sanity_check);
  return tvb_captured_length(tvb);
}

/* 7.3.2.14 Extended Supported Rates element (50) */
static int
ieee80211_tag_ext_supp_rates(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be greater than 0", tag_len);
    return tvb_captured_length(tvb);
  }

  while (offset < tag_len)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_ext_supp_rates, tvb, offset, 1, ENC_NA);
    proto_item_append_text(field_data->item_tag, " %s,", val_to_str_ext_const(tvb_get_guint8(tvb, offset), &ieee80211_supported_rates_vals_ext, "Unknown Rate"));
    offset += 1;
  }
  proto_item_append_text(field_data->item_tag, " [Mbit/sec]");
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_cisco_ccx1_ckip(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  /* From WCS manual:
   * If Aironet IE support is enabled, the access point sends an Aironet
   * IE 0x85 (which contains the access point name, load, number of
   * associated clients, and so on) in the beacon and probe responses of
   * this WLAN, and the controller sends Aironet IEs 0x85 and 0x95
   * (which contains the management IP address of the controller and
   * the IP address of the access point) in the reassociation response
   * if it receives Aironet IE 0x85 in the reassociation request.
   */

  if (tag_len < 26)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u too short, must be >= 26", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_cisco_ccx1_unknown, tvb, offset, 10, ENC_NA);
  offset += 10;

  /* The Name of the sending device starts at offset 10 and is up to
     15 or 16 bytes in length, \0 padded */
  proto_tree_add_item(tree, hf_ieee80211_tag_cisco_ccx1_name, tvb, offset, 16, ENC_ASCII|ENC_NA);
  offset += 16;

  /* Total number off associated clients and repeater access points */
  proto_tree_add_item(tree, hf_ieee80211_tag_cisco_ccx1_clients, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_cisco_ccx1_unknown2, tvb, offset, 3, ENC_NA);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_vendor_specific_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint32 tag_vs_len = tag_len;
  guint32       oui;

  if (tag_len < 3)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 3", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
  proto_item_append_text(field_data->item_tag, ": %s", uint_get_manuf_name_if_known(oui));

  offset += 3;
  tag_vs_len -= 3;

  if (tag_len > 0) {
    proto_tree_add_item(field_data->item_tag, hf_ieee80211_tag_vendor_oui_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  }

  switch (oui) {
    /* 802.11 specific vendor ids */
    case OUI_WPAWME:
      dissect_vendor_ie_wpawme(tree, tvb, pinfo, offset, tag_vs_len, field_data->ftype);
      break;
    case OUI_RSN:
      dissect_vendor_ie_rsn(field_data->item_tag, tree, tvb, offset, tag_vs_len);
      break;
    case OUI_PRE11N:
      dissect_vendor_ie_ht(tvb, pinfo, tree, offset, field_data->item_tag, field_data->item_tag_length, tag_vs_len);
      break;
    case OUI_WFA:
      dissect_vendor_ie_wfa(pinfo, field_data->item_tag, tvb);
      break;

    /* Normal IEEE vendor ids (from oui.h) */
    case OUI_CISCOWL:  /* Cisco Wireless (Aironet) */
      dissect_vendor_ie_aironet(field_data->item_tag, tree, tvb, offset, tag_vs_len);
      break;
    case OUI_MARVELL:
      dissect_vendor_ie_marvell(field_data->item_tag, tree, tvb, offset, tag_vs_len);
      break;
    case OUI_ATHEROS:
      dissect_vendor_ie_atheros(field_data->item_tag, tree, tvb, offset, tag_vs_len, pinfo, field_data->item_tag_length);
      break;
    case OUI_ARUBA:
      dissect_vendor_ie_aruba(field_data->item_tag, tree, tvb, offset, tag_vs_len);
      break;
    case OUI_NINTENDO:
      dissect_vendor_ie_nintendo(field_data->item_tag, tree, tvb, offset, tag_vs_len);
      break;
    case OUI_ROUTERBOARD:
      dissect_vendor_ie_routerboard(field_data->item_tag, tree, tvb, offset, tag_vs_len, pinfo);
      break;
    case OUI_MERU:
      dissect_vendor_ie_meru(field_data->item_tag, tree, tvb, offset, tag_vs_len, pinfo);
      break;
    case OUI_ZEBRA_EXTREME:
      dissect_vendor_ie_extreme(field_data->item_tag, tree, tvb, offset, tag_vs_len, pinfo);
      break;
    case OUI_AEROHIVE:
      dissect_vendor_ie_aerohive(field_data->item_tag, tree, tvb, offset, tag_vs_len, pinfo);
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_tag_vendor_data, tvb, offset, tag_vs_len, ENC_NA);
      break;
  }

  return tvb_captured_length(tvb);
}

#define HE_HTC_HE_SUPPORT                0x00000001
#define HE_FRAGMENTATION_SUPPORT         0x00000018
#define HE_ALL_ACK_SUPPORT               0x00000200
#define HE_UMRS_SUPPORT                  0x00000400
#define HE_BSR_SUPPORT                   0x00000800

static const val64_string he_fragmentation_support_vals[] = {
  { 0, "No support for dynamic fragmentation" },
  { 1, "Support for dynamic fragments in MPDUs or S-MPDUs" },
  { 2, "Support for dynamic fragments in MPDUs and S-MPDUs and up to 1 dyn frag in MSDUs..." },
  { 3, "Support for all types of dynamic fragments" },
  { 0, NULL }
};

static const val64_string he_minimum_fragmentation_size_vals[] = {
  { 0, "No restriction on minimum payload size" },
  { 1, "Minimum payload size of 128 bytes" },
  { 2, "Minimum payload size of 256 bytes" },
  { 3, "Minimum payload size of 512 bytes" },
  { 0, NULL }
};

static void
max_frag_msdus_base_custom(gchar *result, guint32 max_frag_msdus_value)
{
  if (max_frag_msdus_value == 7)
    g_snprintf(result, ITEM_LABEL_LENGTH, "No restriction");
  else
    g_snprintf(result, ITEM_LABEL_LENGTH, "%u", 1 << max_frag_msdus_value);
}

static const val64_string he_link_adaptation_support_vals[] = {
  { 0, "No feedback if the STA does not provide HE MFB" },
  { 1, "Reserved" },
  { 2, "Unsolicited if the STA can receive and provide only unsolicited HE MFB" },
  { 3, "Both" },
  { 0, NULL }
};

static const val64_string he_mimo_cntrl_nc_index_vals[] = {
  {0x00, "1 Column"},
  {0x01, "2 Columns"},
  {0x02, "3 Columns"},
  {0x03, "4 Columns"},
  {0x04, "5 Columns"},
  {0x05, "6 Columns"},
  {0x06, "7 Columns"},
  {0x07, "8 Columns"},
  {0, NULL}
};

static const val64_string he_mimo_cntrl_nr_index_vals[] = {
  {0x00, "1 Row"},
  {0x01, "2 Rows"},
  {0x02, "3 Rows"},
  {0x03, "4 Rows"},
  {0x04, "5 Rows"},
  {0x05, "6 Rows"},
  {0x06, "7 Rows"},
  {0x07, "8 Rows"},
  {0, NULL}
};

static const val64_string he_mimo_cntrl_grouping_flags[] = {
  { 0x00, "Carrier Groups of 4" },
  { 0x01, "Carrier Groups of 16" },
  { 0, NULL }
};

static const val64_string he_mimo_cntrl_feedback_vals[] = {
  { 0x00, "SU" },
  { 0x01, "MU" },
  { 0x02, "CQI feedback" },
  { 0x03, "Reserved" },
  { 0, NULL }
};

static const int *he_mac_headers[] = {
  &hf_he_htc_he_support,                           /* 0 */
  &hf_he_twt_requester_support,                    /* 1 */
  &hf_he_twt_responder_support,                    /* 2 */
  &hf_he_fragmentation_support,                    /* 3 */
  &hf_he_max_number_fragmented_msdus,              /* 4 */
  &hf_he_min_fragment_size,                        /* 5 */
  &hf_he_trigger_frame_mac_padding_dur,            /* 6 */
  &hf_he_multi_tid_aggregation_support,            /* 7 */
  &hf_he_he_link_adaptation_support,               /* 8 */
  &hf_he_all_ack_support,                          /* 9 */
  &hf_he_trs_support,                              /* 10 */
  &hf_he_bsr_support,                              /* 11 */
  &hf_he_broadcast_twt_support,                    /* 12 */
  &hf_he_32_bit_ba_bitmap_support,                 /* 13 */
  &hf_he_mu_cascading_support,                     /* 14 */
  &hf_he_ack_enabled_aggregation_support,          /* 15 */
  &hf_he_reserved_b24,                             /* 16 */
  &hf_he_om_control_support,                       /* 17 */
  &hf_he_ofdma_ra_support,                         /* 18 */
  &hf_he_max_a_mpdu_length_exponent_ext,           /* 19 */
  &hf_he_a_msdu_fragmentation_support,             /* 20 */
  &hf_he_flexible_twt_schedule_support,            /* 21 */
  &hf_he_rx_control_frame_to_multibss,             /* 22 */
  &hf_he_bsrp_bqrp_a_mpdu_aggregation,             /* 23 */
  &hf_he_qtp_support,                              /* 24 */
  &hf_he_bqr_support,                              /* 25 */
  &hf_he_srp_responder,                            /* 26 */
  &hf_he_ndp_feedback_report_support,              /* 27 */
  &hf_he_ops_support,                              /* 28 */
  &hf_he_a_msdu_in_a_mpdu_support,                 /* 29 */
  &hf_he_multi_tid_aggregation_tx_support,         /* 30 */
  &hf_he_subchannel_selective_trans_support,       /* 31 */
  &hf_he_2_996_tone_ru_support,                    /* 32 */
  &hf_he_om_control_ul_mu_data_disable_rx_support, /* 33 */
  &hf_he_reserved_bits_45_47,                      /* 34 */
  NULL
};

static const int *he_phy_first_byte_headers[] = {
  &hf_he_phy_cap_reserved_b0,
  NULL,
};

static const int *he_phy_channel_width_set_headers[] _U_ = {
  &hf_he_40mhz_channel_2_4ghz,
  &hf_he_40_and_80_mhz_5ghz,
  &hf_he_160_mhz_5ghz,
  &hf_he_160_80_plus_80_mhz_5ghz,
  &hf_he_242_tone_rus_in_2_4ghz,
  &hf_he_242_tone_rus_in_5ghz,
  &hf_he_chan_width_reserved,
  NULL
};

static const int *he_phy_b8_to_b23_headers[] = {
  &hf_he_phy_cap_punctured_preamble_rx,
  &hf_he_phy_cap_device_class,
  &hf_he_phy_cap_ldpc_coding_in_payload,
  &hf_he_phy_cap_he_su_ppdu_1x_he_ltf_08us,
  &hf_he_phy_cap_midamble_rx_max_nsts,
  &hf_he_phy_cap_ndp_with_4x_he_ltf_32us,
  &hf_he_phy_cap_stbc_tx_lt_80mhz,
  &hf_he_phy_cap_stbc_rx_lt_80mhz,
  &hf_he_phy_cap_doppler_tx,
  &hf_he_phy_cap_doppler_rx,
  &hf_he_phy_cap_full_bw_ul_mu_mimo,
  &hf_he_phy_cap_partial_bw_ul_mu_mimo,
  NULL
};

static const int *he_phy_b24_to_b39_headers[] = {
  &hf_he_phy_cap_dcm_max_constellation_tx,
  &hf_he_phy_cap_dcm_max_nss_tx,
  &hf_he_phy_cap_dcm_max_constellation_rx,
  &hf_he_phy_cap_dcm_max_nss_rx,
  &hf_he_phy_cap_rx_he_muppdu_from_non_ap,
  &hf_he_phy_cap_su_beamformer,
  &hf_he_phy_cap_su_beamformee,
  &hf_he_phy_cap_mu_beamformer,
  &hf_he_phy_cap_beamformer_sts_lte_80mhz,
  &hf_he_phy_cap_beamformer_sts_gt_80mhz,
  NULL
};

static const int *he_phy_b40_to_b55_headers[] = {
  &hf_he_phy_cap_number_of_sounding_dims_lte_80,
  &hf_he_phy_cap_number_of_sounding_dims_gt_80,
  &hf_he_phy_cap_ng_eq_16_su_fb,
  &hf_he_phy_cap_ng_eq_16_mu_fb,
  &hf_he_phy_cap_codebook_size_eq_4_2_fb,
  &hf_he_phy_cap_codebook_size_eq_7_5_fb,
  &hf_he_phy_cap_triggered_su_beamforming_fb,
  &hf_he_phy_cap_triggered_mu_beamforming_fb,
  &hf_he_phy_cap_triggered_cqi_fb,
  &hf_he_phy_cap_partial_bw_extended_range,
  &hf_he_phy_cap_partial_bw_dl_mu_mimo,
  &hf_he_phy_cap_ppe_threshold_present,
  NULL
};

static const int *he_phy_b56_to_b71_headers[] = {
  &hf_he_phy_cap_srp_based_sr_support,
  &hf_he_phy_cap_power_boost_factor_ar_support,
  &hf_he_phy_cap_he_su_ppdu_etc_gi,
  &hf_he_phy_cap_max_nc,
  &hf_he_phy_cap_stbc_tx_gt_80_mhz,
  &hf_he_phy_cap_stbc_rx_gt_80_mhz,
  &hf_he_phy_cap_he_er_su_ppdu_4xxx_gi,
  &hf_he_phy_cap_20mhz_in_40mhz_24ghz_band,
  &hf_he_phy_cap_20mhz_in_160_80p80_ppdu,
  &hf_he_phy_cap_80mgz_in_160_80p80_ppdu,
  &hf_he_phy_cap_he_er_su_ppdu_1xxx_gi,
  &hf_he_phy_cap_midamble_rx_2x_xxx_ltf,
  &hf_he_phy_cap_dcm_max_bw,
  NULL
};

static const int *he_phy_b72_to_b87_headers[] = {
  &hf_he_phy_cap_longer_than_16_he_sigb_ofdm_symbol_support,
  &hf_he_phy_cap_non_triggered_cqi_feedback,
  &hf_he_phy_cap_tx_1024_qam_242_tone_ru_support,
  &hf_he_phy_cap_rx_1024_qam_242_tone_ru_support,
  &hf_rx_full_bw_su_using_he_muppdu_w_compressed_sigb,
  &hf_rx_full_bw_su_using_he_muppdu_w_non_compressed_sigb,
  &hf_he_phy_cap_b78_b87_reserved,
  NULL
};

static const int *he_mcs_map_80_rx_headers [] = {
  &hf_he_mcs_max_he_mcs_80_rx_1_ss,
  &hf_he_mcs_max_he_mcs_80_rx_2_ss,
  &hf_he_mcs_max_he_mcs_80_rx_3_ss,
  &hf_he_mcs_max_he_mcs_80_rx_4_ss,
  &hf_he_mcs_max_he_mcs_80_rx_5_ss,
  &hf_he_mcs_max_he_mcs_80_rx_6_ss,
  &hf_he_mcs_max_he_mcs_80_rx_7_ss,
  &hf_he_mcs_max_he_mcs_80_rx_8_ss,
  NULL
};

static const int *he_mcs_map_80_tx_headers [] = {
  &hf_he_mcs_max_he_mcs_80_tx_1_ss,
  &hf_he_mcs_max_he_mcs_80_tx_2_ss,
  &hf_he_mcs_max_he_mcs_80_tx_3_ss,
  &hf_he_mcs_max_he_mcs_80_tx_4_ss,
  &hf_he_mcs_max_he_mcs_80_tx_5_ss,
  &hf_he_mcs_max_he_mcs_80_tx_6_ss,
  &hf_he_mcs_max_he_mcs_80_tx_7_ss,
  &hf_he_mcs_max_he_mcs_80_tx_8_ss,
  NULL
};

static const int *he_mcs_map_80p80_rx_headers [] = {
  &hf_he_mcs_max_he_mcs_80p80_rx_1_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_2_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_3_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_4_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_5_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_6_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_7_ss,
  &hf_he_mcs_max_he_mcs_80p80_rx_8_ss,
  NULL
};

static const int *he_mcs_map_80p80_tx_headers [] = {
  &hf_he_mcs_max_he_mcs_80p80_tx_1_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_2_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_3_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_4_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_5_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_6_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_7_ss,
  &hf_he_mcs_max_he_mcs_80p80_tx_8_ss,
  NULL
};

static const int *he_mcs_map_160_rx_headers [] = {
  &hf_he_mcs_max_he_mcs_160_rx_1_ss,
  &hf_he_mcs_max_he_mcs_160_rx_2_ss,
  &hf_he_mcs_max_he_mcs_160_rx_3_ss,
  &hf_he_mcs_max_he_mcs_160_rx_4_ss,
  &hf_he_mcs_max_he_mcs_160_rx_5_ss,
  &hf_he_mcs_max_he_mcs_160_rx_6_ss,
  &hf_he_mcs_max_he_mcs_160_rx_7_ss,
  &hf_he_mcs_max_he_mcs_160_rx_8_ss,
  NULL
};

static const int *he_mcs_map_160_tx_headers [] = {
  &hf_he_mcs_max_he_mcs_160_tx_1_ss,
  &hf_he_mcs_max_he_mcs_160_tx_2_ss,
  &hf_he_mcs_max_he_mcs_160_tx_3_ss,
  &hf_he_mcs_max_he_mcs_160_tx_4_ss,
  &hf_he_mcs_max_he_mcs_160_tx_5_ss,
  &hf_he_mcs_max_he_mcs_160_tx_6_ss,
  &hf_he_mcs_max_he_mcs_160_tx_7_ss,
  &hf_he_mcs_max_he_mcs_160_tx_8_ss,
  NULL
};

static const value_string ru_alloc_vals[] = {
  { 0, "242" },
  { 1, "484" },
  { 2, "996" },
  { 3, "2x996" },
  { 0, NULL }
};

static const value_string constellation_vals[] = {
  { 0, "BPSK" },
  { 1, "QPSK" },
  { 2, "16-QAM" },
  { 3, "64-QAM" },
  { 4, "256-QAM" },
  { 5, "1024-QAM" },
  { 6, "Reserved" },
  { 7, "None" },
  { 0, NULL }
};

#define HE_CHANNEL_WIDTH_SET_B2 0x04
#define HE_CHANNEL_WIDTH_SET_B3 0x08

static void
dissect_he_capabilities(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset, int len)
{
  guint64 he_mac_caps = tvb_get_letoh40(tvb, offset);
  guint8 phy_channel_width_set = 0;
  proto_tree *phy_cap_tree = NULL;
  guint he_mcs_and_nss_len = 4;
  proto_tree *sup_he_mcs_and_nss_tree = NULL;
  proto_tree *rx_tx_he_mcs_map_80 = NULL;
  proto_tree *rx_tx_he_mcs_map_160 = NULL;
  proto_tree *rx_tx_he_mcs_map_80_80 = NULL;

  /* Change some header fields depending on HE_HTC_HE_SUPPORT and FRAGMENTATION */
  if (!(he_mac_caps & HE_HTC_HE_SUPPORT)) {
    he_mac_headers[8] = &hf_he_reserved_bits_15_16;
    he_mac_headers[10] = &hf_he_reserved_bit_18;
    he_mac_headers[11] = &hf_he_reserved_bit_19;
    he_mac_headers[17] = &hf_he_reserved_bit_25;
  }
  if (!(he_mac_caps & HE_FRAGMENTATION_SUPPORT)) {
    he_mac_headers[4] = &hf_he_reserved_bits_5_7;
    he_mac_headers[5] = &hf_he_reserved_bits_8_9;
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_he_mac_capabilities,
                        ett_he_mac_capabilities, he_mac_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 6;

  /* Get and isolate the phy channel witdth set */
  phy_channel_width_set = tvb_get_guint8(tvb, offset) >> 1;
  phy_cap_tree = proto_tree_add_subtree(tree, tvb, offset, 9, ett_he_phy_capabilities,
                        NULL,
                        "HE Phy Capabilities Information");
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_reserved_b0, ett_he_phy_cap_first_byte,
                        he_phy_first_byte_headers, ENC_NA, BMT_NO_APPEND);
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_chan_width_set,  ett_he_phy_cap_chan_width_set,
                        he_phy_channel_width_set_headers, ENC_NA, BMT_NO_APPEND);
  offset++;
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_b8_to_b23, ett_he_phy_cap_b8_to_b23,
                        he_phy_b8_to_b23_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_b24_to_b39, ett_he_phy_cap_b24_to_b39,
                        he_phy_b24_to_b39_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_b40_to_b55, ett_he_phy_cap_b40_to_b55,
                        he_phy_b40_to_b55_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_b56_to_b71, ett_he_phy_cap_b56_to_b71,
                        he_phy_b56_to_b71_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(phy_cap_tree, tvb, offset,
                        hf_he_phy_b72_to_b87, ett_he_phy_cap_b72_to_b87,
                        he_phy_b72_to_b87_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* Need the length first */
  if (phy_channel_width_set & HE_CHANNEL_WIDTH_SET_B2)
    he_mcs_and_nss_len += 4;

  if (phy_channel_width_set & HE_CHANNEL_WIDTH_SET_B3)
    he_mcs_and_nss_len += 4;

  sup_he_mcs_and_nss_tree = proto_tree_add_subtree(tree, tvb, offset,
                        he_mcs_and_nss_len, ett_he_mcs_and_nss_set, NULL,
                        "Supported HE-MCS and NSS Set");
  rx_tx_he_mcs_map_80 = proto_tree_add_subtree(sup_he_mcs_and_nss_tree, tvb,
                        offset, 4, ett_he_rx_tx_he_mcs_map_lte_80, NULL,
                        "Rx and Tx MCS Maps <= 80 MHz");
  proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_80, tvb, offset,
                        hf_he_rx_he_mcs_map_lte_80,
                        ett_he_rx_mcs_map_lte_80, he_mcs_map_80_rx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_80, tvb, offset,
                        hf_he_tx_he_mcs_map_lte_80,
                        ett_he_tx_mcs_map_lte_80, he_mcs_map_80_tx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  if (phy_channel_width_set & HE_CHANNEL_WIDTH_SET_B2) {
    rx_tx_he_mcs_map_160 = proto_tree_add_subtree(sup_he_mcs_and_nss_tree,
                        tvb, offset, 4, ett_he_rx_tx_he_mcs_map_160, NULL,
                        "Rx and Tx MCS Maps 160 MHz");
    proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_160, tvb, offset,
                        hf_he_rx_he_mcs_map_160,
                        ett_he_rx_mcs_map_160, he_mcs_map_160_rx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 2;
    proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_160, tvb, offset,
                        hf_he_tx_he_mcs_map_160,
                        ett_he_tx_mcs_map_160, he_mcs_map_160_tx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 2;
  }

  if (phy_channel_width_set & HE_CHANNEL_WIDTH_SET_B3) {
    rx_tx_he_mcs_map_80_80 = proto_tree_add_subtree(sup_he_mcs_and_nss_tree,
                        tvb, offset, 4, ett_he_rx_tx_he_mcs_map_80_80, NULL,
                        "Rx and Tx MCS Maps 80+80 MHz");
    proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_80_80, tvb, offset,
                        hf_he_rx_he_mcs_map_80_80,
                        ett_he_rx_mcs_map_80_80, he_mcs_map_80p80_rx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 2;
    proto_tree_add_bitmask_with_flags(rx_tx_he_mcs_map_80_80, tvb, offset,
                        hf_he_tx_he_mcs_map_80_80,
                        ett_he_tx_mcs_map_80_80, he_mcs_map_80p80_tx_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 2;
  }

  if (offset < len) {
    guint8 ppe_thresholds_field = tvb_get_guint8(tvb, offset);
    guint8 nss_count = ppe_thresholds_field & 0x07, nss_index = 0;
    guint8 ru_index_bitmask = (ppe_thresholds_field >> 3) & 0x0F;
    proto_tree *ppe_tree = NULL;
    int i = 0;
    int bit_offset = 7;  /* How many bits we are into the bytes */

    ppe_tree = proto_tree_add_subtree(tree, tvb, offset, len - offset + 1,
                        ett_he_ppe_threshold, NULL,
                        "PPE Thresholds");
    proto_tree_add_item(ppe_tree, hf_he_ppe_thresholds_nss, tvb, offset,
                        1, ENC_NA);
    proto_tree_add_item(ppe_tree, hf_he_ppe_thresholds_ru_index_bitmask, tvb,
                        offset, 1, ENC_NA);

    /*
     * Now, for each of the nss values, add a sub-tree with its thresholds.
     * The actual count is one more than the number in the first three bits.
     */
    while (nss_index < nss_count + 1) {
      int start_offset = 0;
      proto_tree *nss_tree = NULL;
      proto_item *nssti = NULL;
      guint8 l_ru_bitmask = ru_index_bitmask;

      nss_tree = proto_tree_add_subtree_format(ppe_tree, tvb, offset, -1,
                        ett_he_ppe_nss, &nssti, "NSS %d", nss_index);
      start_offset = offset;

      for (i = 0; i < 4; i++) {
        if (l_ru_bitmask & 0x01) {
          int bits_avail = 8 - bit_offset, bits_needed = 6 - bits_avail;
          guint8 the_bits = 0;
          int ru_start_offset = offset;
          proto_tree *ru_alloc_tree = NULL;
          proto_item *rualti = NULL;

          ru_alloc_tree = proto_tree_add_subtree_format(nss_tree, tvb, offset,
                                        -1, ett_he_ppe_ru_alloc, &rualti,
                                        "RU allocation: %s",
                                        val_to_str(i, ru_alloc_vals, "Unk"));

          /*
           * Assemble the bits we require ... we need 6, or 2x3
           */
          if (bits_avail >= 6) { /* We can use the current byte */
            the_bits = (tvb_get_guint8(tvb, offset) >> bit_offset) & 0x3F;
          } else { /* We need two adjacent bytes */
            the_bits = (tvb_get_guint8(tvb, offset) >> bit_offset);
            offset++;
            the_bits = the_bits |
                        ((tvb_get_guint8(tvb, offset) &
                                ((1 << bits_needed) - 1)) << bits_avail);
          }
          /*
           * Now we have two three bit fields, use them.
           */
          proto_tree_add_uint(ru_alloc_tree, hf_he_ppe_ppet16, tvb, ru_start_offset,
                              offset - ru_start_offset + 1, the_bits & 0x07);
          proto_tree_add_uint(ru_alloc_tree, hf_he_ppe_ppet8, tvb, ru_start_offset,
                              offset - ru_start_offset + 1, the_bits >> 3);

          bit_offset = (bit_offset + 6) % 8;
          proto_item_set_len(rualti, offset - ru_start_offset + 1);
        }
        l_ru_bitmask = l_ru_bitmask >> 1;
      }


      proto_item_set_len(nssti, offset - start_offset);
      nss_index++;
    }
  }

  /* Add an Expert Info about extra bytes ... */

}

static const int *he_operation_headers[] = {
  &hf_he_operation_default_pe_duration,
  &hf_he_operation_twt_required,
  &hf_he_operation_txop_duration_rts_threshold,
  &hf_he_operation_vht_operation_information_present,
  &hf_he_operation_co_located_bss,
  &hf_he_operation_er_su_disable,
  &hf_he_operation_reserved_b17_b23,
  NULL
};

static const int *he_bss_color_info_headers[] = {
  &hf_he_bss_color_info_bss_color,
  &hf_he_bss_color_partial_bss_color,
  &hf_he_bss_color_bss_color_disabled,
  NULL
};

static const value_string he_mcs_map_vals[] = {
  { 0, "Support for HE-MCS 0-7" },
  { 1, "Support for HE-MCS 0-9" },
  { 2, "Support for HE-MCS 0-11" },
  { 3, "Not supported for HE PPDUs" },
  { 0, NULL }
};

static const int *he_basic_he_mcs_header[] = {
  &hf_he_oper_max_he_mcs_for_1_ss,
  &hf_he_oper_max_he_mcs_for_2_ss,
  &hf_he_oper_max_he_mcs_for_3_ss,
  &hf_he_oper_max_he_mcs_for_4_ss,
  &hf_he_oper_max_he_mcs_for_5_ss,
  &hf_he_oper_max_he_mcs_for_6_ss,
  &hf_he_oper_max_he_mcs_for_7_ss,
  &hf_he_oper_max_he_mcs_for_8_ss,
  NULL
};

#define VHT_OPERATION_INFORMATION_PRESENT 0x004000
#define CO_LOCATED_BSS                    0x008000

static const value_string channel_width_vals[] = {
  { 0, "20 MHz or 40 MHz BSS Bandwidth" },
  { 1, "80 MHz, 160 MHz or 80+80 MHz BSS Bandwidth" },
  { 2, "160 MHz BSS Bandwidth (deprecated)" },
  { 3, "Non-contiguous 80+80 MHz BSS Bandwidth (deprecated)" },
  { 0, NULL }
};

static void
dissect_he_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset, int len _U_)
{
    guint32 op_params = tvb_get_letoh24(tvb, offset);

    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_he_operation_parameter, ett_he_operation_params,
                        he_operation_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 3;

    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_he_bss_color_information, ett_he_bss_color_information,
                        he_bss_color_info_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;

    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_he_operation_basic_mcs, ett_he_oper_basic_mcs,
                        he_basic_he_mcs_header, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 2;

    if (op_params & VHT_OPERATION_INFORMATION_PRESENT) {
      proto_tree *vht_op_info = NULL;
      proto_item *pi = NULL;

      vht_op_info = proto_tree_add_subtree(tree, tvb, offset, 3,
                        ett_he_operation_vht_op_info, NULL,
                        "VHT Operation Information");
      pi = proto_tree_add_item(vht_op_info, hf_he_operation_channel_width, tvb,
                        offset, 1, ENC_NA);
      proto_item_append_text(pi, ": %s",
                        val_to_str(tvb_get_guint8(tvb, offset),
                                channel_width_vals,
                                "Reserved %u"));
      offset++;

      proto_tree_add_item(vht_op_info, hf_he_operation_channel_center_freq_0,
                        tvb, offset, 1, ENC_NA);
      offset++;

      proto_tree_add_item(vht_op_info, hf_he_operation_channel_center_freq_1,
                        tvb, offset, 1, ENC_NA);
      offset++;
    }

    if (op_params & CO_LOCATED_BSS) {
      proto_tree_add_item(tree, hf_he_operation_max_colocated_bssid_indicator,
                        tvb, offset, 1, ENC_NA);
      offset++;
    }
}

static const int *uora_headers[] = {
  &hf_he_uora_eocwmin,
  &hf_he_uora_owcwmax,
  &hf_he_uora_reserved,
  NULL
};

static void
dissect_uora_parameter_set(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_he_uora_field, ett_he_uora_tree, uora_headers,
                        ENC_NA, BMT_NO_APPEND);
  offset++;
}

static const int *muac_aci_aifsn_headers[] = {
  &hf_he_muac_aifsn,
  &hf_he_muac_acm,
  &hf_he_muac_aci,
  &hf_he_muac_reserved,
  NULL
};

static int
dissect_muac_param_record(tvbuff_t *tvb, proto_tree *tree, int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_he_muac_aci_aifsn, ett_he_aic_aifsn,
                        muac_aci_aifsn_headers, ENC_NA, BMT_NO_APPEND);
  offset++;

  proto_tree_add_item(tree, hf_he_muac_ecwmin_ecwmax, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_he_mu_edca_timer, tvb, offset, 1, ENC_NA);
  offset++;

  return offset;
}

static int
dissect_mu_edca_parameter_set(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
  int offset, int len _U_)
{
  proto_tree *param_tree = NULL;

  /* Is this from an AP or an STA? */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_qos_info_ap,
                                    ett_ff_qos_info, ieee80211_ff_qos_info_ap_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset++;

  param_tree = proto_tree_add_subtree(tree, tvb, offset, 3, ett_he_mu_edca_param,
                        NULL, "MUAC_BE Parameter Record");
  offset = dissect_muac_param_record(tvb, param_tree, offset);

  param_tree = proto_tree_add_subtree(tree, tvb, offset, 3, ett_he_mu_edca_param,
                        NULL, "MUAC_BK Parameter Record");
  offset = dissect_muac_param_record(tvb, param_tree, offset);

  param_tree = proto_tree_add_subtree(tree, tvb, offset, 3, ett_he_mu_edca_param,
                        NULL, "MUAC_VI Parameter Record");
  offset = dissect_muac_param_record(tvb, param_tree, offset);

  param_tree = proto_tree_add_subtree(tree, tvb, offset, 3, ett_he_mu_edca_param,
                        NULL, "MUAC_VO Parameter Record");
  offset = dissect_muac_param_record(tvb, param_tree, offset);

  return offset;
}

#define SRP_DISALLOWED                     0x01
#define NON_SRG_OBSS_PD_SR_DISALLOWED      0x02
#define NON_SRG_OFFSET_PRESENT             0x04
#define SRG_INFORMATION_PRESENT            0x08
#define HESIGA_SPATIAL_REUSE_VAL15_ALLOWED 0x10

static const int *sr_control_field_headers[] = {
  &hf_he_srp_disallowed,
  &hf_he_non_srg_obss_pd_sr_disallowed,
  &hf_he_non_srg_offset_present,
  &hf_he_srg_information_present,
  &hf_he_hesiga_spatial_reuse_value15_allowed,
  &hf_he_sr_control_reserved,
  NULL
};

static int
dissect_spatial_reuse_parameter_set(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_)
{
  guint8 sr_control = tvb_get_guint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_he_spatial_reuse_sr_control,
                                    ett_he_spatial_reuse_control,
                                    sr_control_field_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset++;

  if (sr_control & NON_SRG_OFFSET_PRESENT) {
    proto_tree_add_item(tree, hf_he_spatial_non_srg_obss_pd_max_offset,
                        tvb, offset, 1, ENC_NA);
    offset++;
  }

  if (sr_control & SRG_INFORMATION_PRESENT) {
    proto_tree_add_item(tree, hf_he_spatial_srg_obss_pd_min_offset, tvb,
                        offset, 1, ENC_NA);
    offset++;
    proto_tree_add_item(tree, hf_he_spatial_srg_obss_pd_max_offset, tvb,
                        offset, 1, ENC_NA);
    offset++;
    proto_tree_add_item(tree, hf_he_spatial_srg_bss_color_bitmap, tvb,
                        offset, 8, ENC_NA);
    offset += 8;
    proto_tree_add_item(tree, hf_he_spatial_srg_partial_bssid_bitmap,
                        tvb, offset, 8, ENC_NA);
    offset += 8;
  }

  return offset;
}

static void
dissect_ndp_feedback_report_set(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_)
{
  proto_tree_add_item(tree, hf_he_resource_request_buffer_thresh, tvb, offset,
                        1, ENC_NA);
}

static const int *bss_new_color_headers[] = {
  &hf_he_new_bss_color_info_color,
  &hf_he_new_bss_color_info_reserved,
  NULL
};

static void
dissect_bss_color_change(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_)
{
  proto_tree_add_item(tree, hf_he_bss_color_change_switch_countdown, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                hf_he_bss_color_change_new_color_info,
                                ett_he_bss_new_color_info, bss_new_color_headers,
                                ENC_NA, BMT_NO_APPEND);
}

static const int *ess_info_field_headers[] = {
  &hf_he_ess_report_planned_ess,
  &hf_he_ess_report_edge_of_ess,
  NULL
};

static void
dissect_ess_report(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_)
{
  guint8 bss_trans_thresh = tvb_get_guint8(tvb, offset) >> 2;


  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_he_ess_report_info_field,
                                    ett_he_ess_report_info_field,
                                    ess_info_field_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  if (bss_trans_thresh == 63)
    proto_tree_add_int_format(tree, hf_he_ess_report_recommend_transition_thresh,
                        tvb, offset, 1, bss_trans_thresh,
                        "Recommended BSS Transition Threshold: %d (No recommendation)",
                        bss_trans_thresh);
  else
    proto_tree_add_int_format(tree, hf_he_ess_report_recommend_transition_thresh, tvb,
                        offset, 1, bss_trans_thresh,
                        "Recommended BSS Transition Threshold: %d (%ddBm)",
                        bss_trans_thresh, -100 + bss_trans_thresh);
}

/*
 * There will be from 1 to 4 24-bit fields in the order of AC=BK, AC=BE,
 * AC=VI and AC=VO.
 */

static const int *esp_headers[] = {
  &hf_ieee80211_esp_access_category,
  &hf_ieee80211_esp_reserved,
  &hf_ieee80211_esp_data_format,
  &hf_ieee80211_esp_ba_windows_size,
  &hf_ieee80211_esp_est_air_time_frac,
  &hf_ieee80211_esp_data_ppdu_duration_target,
  NULL
};

static const value_string esp_access_category_vals[] = {
  { 0, "AC=BK" },
  { 1, "AC=BE" },
  { 2, "AC=VI" },
  { 3, "AC=VO" },
  { 0, NULL }
};

static const value_string esp_data_format_vals[] = {
  { 0, "No aggregation is expected to be performed" },
  { 1, "A-MSDU aggregation is expected but not A-MPDUs when type is data" },
  { 2, "A-MSDU aggregation is NOT expected but A-MPDUs aggregation is when type is data" },
  { 3, "A-MSDU aggregation is expected and A-MPDU aggregation is when type is data" },
  { 0, NULL }
};

static const value_string esp_ba_window_size_vals[] = {
  { 0, "Block Ack not expected to be used" },
  { 1, "2" },
  { 2, "4" },
  { 3, "6" },
  { 4, "8" },
  { 5, "16" },
  { 6, "32" },
  { 7, "64" },
  { 0, NULL }
};

static int
dissect_estimated_service_params(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len)
{
  while (len > 0) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_estimated_service_params, ett_ieee80211_esp,
                        esp_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 3;
    len -= 3;
  }

  return offset;
}

static int
dissect_future_channel_guidance(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_)
{
  proto_tree_add_item(tree, hf_ieee80211_fcg_new_channel_number, tvb, offset,
                        4, ENC_LITTLE_ENDIAN);
  offset += 4;

  if (len - 4 > 0) {
    proto_tree_add_item(tree, hf_ieee80211_fcg_extra_info, tvb, offset, len - 4,
                        ENC_NA);
    offset += len - 4;
  }

  return offset;
}

/* IANA, "Transform Type 4 - Diffie-Hellman Group Transform IDs" */
static const value_string owe_dh_parameter_group_vals[] = {
  { 0, "None" },
  { 1, "768-bit MODP Group" },
  { 2, "1024-bit MODP Group" },
  { 5, "1536-bit MODP Group" },
  { 14, "2048-bit MODP Group"},
  { 15, "3072-bit MODP Group"},
  { 16, "4096-bit MODP Group"},
  { 17, "6144-bit MODP Group"},
  { 18, "8192-bit MODP Group"},
  { 19, "256-bit random ECP group"},
  { 20, "384-bit random ECP group"},
  { 21, "521-bit random ECP group"},
  { 22, "1024-bit MODP Group with 160-bit Prime Order Subgroup"},
  { 23, "2048-bit MODP Group with 224-bit Prime Order Subgroup"},
  { 24, "2048-bit MODP Group with 256-bit Prime Order Subgroup"},
  { 25, "192-bit Random ECP Group"},
  { 26, "224-bit Random ECP Group"},
  { 27, "brainpoolP224r1"},
  { 28, "brainpoolP256r1"},
  { 29, "brainpoolP384r1"},
  { 30, "brainpoolP512r1"},
  { 31, "Curve25519"},
  { 32, "Curve448"},
  { 0, NULL }
};

static int
dissect_owe_dh_parameter(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, int len _U_, association_sanity_check_t* sanity_check)
{
  if (len < 2) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length,
                           "OWE: Diffie-Hellman Parameter must be at least 2 "
                           "octets long");
    return offset + len;
  }

  if (sanity_check != NULL) {
    sanity_check->owe_group = tvb_get_guint16(tvb, offset, ENC_LITTLE_ENDIAN);
  }

  proto_tree_add_item(tree, hf_owe_dh_parameter_group, tvb, offset,
                        2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_owe_dh_parameter_public_key, tvb, offset + 2,
                        len - 2, ENC_NA);
  offset += len;

  return offset;
}

static int
ieee80211_tag_twt(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *item;

  static const int *ieee80211_twt_ctrl_field[] = {
    &hf_ieee80211_tag_twt_ndp_paging_indicator,
    &hf_ieee80211_tag_twt_responder_pm_mode,
    &hf_ieee80211_tag_twt_neg_type,
    &hf_ieee80211_tag_twt_ctrl_field_reserved,
    NULL,
  };

  static const int *ieee80211_twt_req_type_field[] = {
    &hf_ieee80211_tag_twt_req_type_req,
    &hf_ieee80211_tag_twt_req_type_setup_cmd,
    &hf_ieee80211_tag_twt_req_type_trigger,
    &hf_ieee80211_tag_twt_req_type_implicit,
    &hf_ieee80211_tag_twt_req_type_flow_type,
    &hf_ieee80211_tag_twt_req_type_flow_id,
    &hf_ieee80211_tag_twt_req_type_wake_int_exp,
    &hf_ieee80211_tag_twt_req_type_prot,
    NULL,
  };

  gboolean twt_requester;
  guint8 setup_command;
  guint8 ctrl_field;
  guint16 req_type;
  guint8 neg_type;

  // 1 byte  - control
  // 2 bytes - request type
  // 8 bytes - target wake time
  // 1 byte  - nominal minimum interval TWT wake duration
  // 2 bytes - TWT wake interval mantissa
  // 1 byte  - channel
  // total: 15 bytes.
  // TODO: support other configurations that have a different size
  if (tag_len < 15) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "Tag Length %u wrong, must be >= 15", tag_len);
    return tvb_captured_length(tvb);
  }

  ctrl_field = tvb_get_guint8(tvb, offset);
  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_tag_twt_control_field,
                                    ett_twt_control_field_tree, ieee80211_twt_ctrl_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_FALSE);
  offset += 1;
  neg_type = (ctrl_field & 0xc) >> 2;

  if (neg_type != 0) {
    // This is a broadcast TWT, or uses TWT information frames and this is not supported yet
    proto_tree_add_expert(tree, pinfo, &ei_ieee80211_twt_setup_not_supported_neg_type,
                          tvb, offset, tvb_reported_length_remaining(tvb, offset));
    return tvb_captured_length(tvb);
  }

  req_type = tvb_get_guint16(tvb, offset, ENC_LITTLE_ENDIAN);
  item = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_twt_req_type_field,
                                           ett_twt_req_type_tree, ieee80211_twt_req_type_field,
                                           ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  twt_requester = req_type & 0x1;
  setup_command = (req_type & 0xe) >> 1;

  switch (setup_command) {
    case REQUEST_TWT:
    case SUGGEST_TWT:
    case DEMAND_TWT:
      // we must be TWT requester
      if (!twt_requester) {
        expert_add_info_format(pinfo, item, &ei_ieee80211_twt_setup_bad_command,
                               "Command %d is not allowed if TWT Request is not set",
                               setup_command);
        return tvb_captured_length(tvb);
      }
      break;
    case TWT_GROUPING:
      // TODO: There are more tests needed here
      //       Fall through since we can't be requester here as well.
    case ACCEPT_TWT:
    case ALTERNATE_TWT:
      // we can't be TWT requester
      if (twt_requester) {
        expert_add_info_format(pinfo, item, &ei_ieee80211_twt_setup_bad_command,
                               "Command %d is not allowed if TWT Request is set",
                               setup_command);
        return tvb_captured_length(tvb);
      }
      break;
    case DICTATE_TWT:
    case REJECT_TWT:
      // TODO: Unclear what to do here. Looks like we can't be Requester, OTOH
      //       the spec doesn't say anything
      break;
    default:
      break;
  }

  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_twt_target_wake_time, tvb, offset, 8,
                      ENC_LITTLE_ENDIAN);
  offset += 8;

  proto_tree_add_item(tree, hf_ieee80211_tag_twt_nom_min_twt_wake_dur, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_twt_wake_interval_mantissa, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_twt_channel, tvb, offset, 1, ENC_NA);

  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_fils_indication(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  guint16 info;
  guint8 nr_realm, nr_pk, i, len;
  proto_item *item;
  proto_tree *subtree;

  if (tag_len < 2)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 2", tag_len);
    return tvb_captured_length(tvb);
  }

  static const int * ieee80211_tag_fils_indication_info[] = {
    &hf_ieee80211_tag_fils_indication_info_nr_pk,
    &hf_ieee80211_tag_fils_indication_info_nr_realm,
    &hf_ieee80211_tag_fils_indication_info_ip_config,
    &hf_ieee80211_tag_fils_indication_info_cache_id_included,
    &hf_ieee80211_tag_fils_indication_info_hessid_included,
    &hf_ieee80211_tag_fils_indication_info_ska_without_pfs,
    &hf_ieee80211_tag_fils_indication_info_ska_with_pfs,
    &hf_ieee80211_tag_fils_indication_info_pka,
    &hf_ieee80211_tag_fils_indication_info_reserved,
    NULL
  };

  info = tvb_get_guint16(tvb, offset, ENC_LITTLE_ENDIAN);
  proto_tree_add_bitmask_list(tree, tvb, offset, 2, ieee80211_tag_fils_indication_info, ENC_LITTLE_ENDIAN);
  offset += 2;

  nr_pk = info & 0x07;
  nr_realm = (info >> 3) & 0x07;

  /* Cache identifier */
  if (info & (1 << 7)) {
    proto_tree_add_item(tree, hf_ieee80211_tag_fils_indication_cache_identifier, tvb, offset, 2, ENC_NA);
    offset += 2;
  }

  /* HESSID */
  if (info & (1 << 8)) {
    proto_tree_add_item(tree, hf_ieee80211_tag_fils_indication_hessid, tvb, offset, 6, ENC_NA);
    offset += 6;
  }

  /* Realm identifiers */
  if (nr_realm > 0) {
    item = proto_tree_add_item(tree, hf_ieee80211_tag_fils_indication_realm_list, tvb, offset, nr_realm * 2, ENC_NA);
    subtree = proto_item_add_subtree(item, ett_fils_indication_realm_list);
    proto_item_append_text(item, ": %u", nr_realm);

    for (i = 0; i < nr_realm; i++) {
      proto_tree_add_item(subtree, hf_ieee80211_tag_fils_indication_realm_identifier, tvb, offset, 2, ENC_NA);
      offset += 2;
    }
  }

  /* PK identifiers */
  if (nr_pk > 0) {
    item = proto_tree_add_item(tree, hf_ieee80211_tag_fils_indication_public_key_list, tvb, offset, tag_len - offset, ENC_NA);
    subtree = proto_item_add_subtree(item, ett_fils_indication_public_key_list);
    proto_item_append_text(item, ": %u", nr_pk);

    for (i = 0; i < nr_pk; i++) {
      proto_tree_add_item(subtree, hf_ieee80211_tag_fils_indication_public_key_type, tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(subtree, hf_ieee80211_tag_fils_indication_public_key_length, tvb, offset, 1, ENC_NA);
      len = tvb_get_guint8(tvb, offset);
      offset += 1;

      proto_tree_add_item(subtree, hf_ieee80211_tag_fils_indication_public_key_indicator, tvb, offset, len, ENC_NA);
      offset += len;
    }
  }

  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_element_id_extension(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  int ext_tag_len;
  guint8 ext_tag_no;

  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 1", tag_len);
    return tvb_captured_length(tvb);
  }
  ext_tag_no = tvb_get_guint8(tvb, offset++);
  ext_tag_len = tag_len - 1;

  switch (ext_tag_no) {
    case ETAG_FILS_SESSION:
      proto_tree_add_item(tree, hf_ieee80211_fils_session, tvb, offset, ext_tag_len, ENC_NA);
      if (field_data->sanity_check != NULL) {
        field_data->sanity_check->has_fils_session = TRUE;
      }
      break;
    case ETAG_FILS_WRAPPED_DATA:
      proto_tree_add_item(tree, hf_ieee80211_fils_wrapped_data, tvb, offset, ext_tag_len, ENC_NA);
      break;
    case ETAG_FILS_NONCE:
      proto_tree_add_item(tree, hf_ieee80211_fils_nonce, tvb, offset, ext_tag_len, ENC_NA);
      break;
    case ETAG_ESTIMATED_SERVICE_PARAM:
      dissect_estimated_service_params(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_FUTURE_CHANNEL_GUIDANCE:
      dissect_future_channel_guidance(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_OWE_DH_PARAMETER:
      dissect_owe_dh_parameter(tvb, pinfo, tree, offset, ext_tag_len, field_data->sanity_check);
      break;
    case ETAG_HE_CAPABILITIES:
      dissect_he_capabilities(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_HE_OPERATION:
      dissect_he_operation(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_UORA_PARAMETER_SET:
      dissect_uora_parameter_set(tvb, pinfo, tree, offset);
      break;
    case ETAG_MU_EDCA_PARAMETER_SET:
      dissect_mu_edca_parameter_set(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_SPATIAL_REUSE_PARAMETER_SET:
      dissect_spatial_reuse_parameter_set(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_NDP_FEEDBACK_REPORT_PARAMETER_SET:
      dissect_ndp_feedback_report_set(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT:
      dissect_bss_color_change(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_QUIET_TIME_PERIOD_SETUP:
      dissect_quiet_time_period(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    case ETAG_ESS_REPORT:
      dissect_ess_report(tvb, pinfo, tree, offset, ext_tag_len);
      break;
    default:
      break;
  }

  return tvb_captured_length(tvb);
}

/* Conflict: WAPI Vs. IEEE */
static int
ieee80211_tag_ie_68_conflict(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  if (tag_len >= 20) { /* It Might be WAPI*/
    dissect_wapi_param_set(tvb, pinfo, tree, 0, tag_len, field_data->item_tag_length, field_data->item_tag, field_data->ftype);
  }
  else { /* BSS AC Access Delay (68) */
     dissect_bss_ac_access_delay_ie(tvb, pinfo, tree, 0, tag_len, field_data->item_tag_length);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_mesh_peering_mgmt(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_mesh_peering_proto, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_mesh_peering_local_link_id, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  switch (tvb_get_guint8(tvb, 1))
  {                                         /* Self-protected action field */
    case SELFPROT_ACTION_MESH_PEERING_OPEN:
      break;

    case SELFPROT_ACTION_MESH_PEERING_CONFIRM:
      proto_tree_add_item(tree, hf_ieee80211_mesh_peering_peer_link_id, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;

    case SELFPROT_ACTION_MESH_PEERING_CLOSE:
      if ((tag_len == 8) || (tag_len == 24))
      {
        proto_tree_add_item(tree, hf_ieee80211_mesh_peering_peer_link_id, tvb, offset, 2, ENC_LITTLE_ENDIAN);
        offset += 2;
      }
      add_ff_reason_code(tree, tvb, pinfo, offset);
      break;

      /* unexpected values */
    default:
      proto_tree_add_expert(tree, pinfo, &ei_ieee80211_mesh_peering_unexpected , tvb, offset, tag_len);
      break;
  }
  if (tag_len - offset == 16)
  {
    proto_tree_add_item(tree, hf_ieee80211_rsn_pmkid, tvb, offset, 16, ENC_NA);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_mesh_configuration(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_item *item;
  proto_tree *subtree;
  static const int *ieee80211_mesh_config_cap[] = {
    &hf_ieee80211_mesh_config_cap_accepting,
    &hf_ieee80211_mesh_config_cap_mcca_support,
    &hf_ieee80211_mesh_config_cap_mcca_enabled,
    &hf_ieee80211_mesh_config_cap_forwarding,
    &hf_ieee80211_mesh_config_cap_mbca_enabled,
    &hf_ieee80211_mesh_config_cap_tbtt_adjusting,
    &hf_ieee80211_mesh_config_cap_power_save_level,
    NULL
  };

  proto_tree_add_item(tree, hf_ieee80211_mesh_config_path_sel_protocol, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_mesh_config_path_sel_metric, tvb, offset + 1, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_mesh_config_congestion_control, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_mesh_config_sync_method, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_mesh_config_auth_protocol, tvb, offset + 4, 1, ENC_LITTLE_ENDIAN);
  item = proto_tree_add_item(tree, hf_ieee80211_mesh_config_formation_info, tvb, offset + 5, 1, ENC_LITTLE_ENDIAN);
  subtree = proto_item_add_subtree(item, ett_mesh_formation_info_tree);
  proto_tree_add_item(subtree, hf_ieee80211_mesh_form_info_num_of_peerings, tvb, offset + 5, 1, ENC_LITTLE_ENDIAN);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset + 6, hf_ieee80211_mesh_config_capability,
                                    ett_mesh_config_cap_tree, ieee80211_mesh_config_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_mesh_id(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int offset = 0;
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int tag_len = tvb_reported_length(tvb);
  const guint8* mesh_id;

  proto_tree_add_item_ret_string(tree, hf_ieee80211_mesh_id, tvb, offset, tag_len, ENC_ASCII|ENC_NA, wmem_packet_scope(), &mesh_id);
  if (tag_len > 0) {
    gchar* s = format_text(wmem_packet_scope(), mesh_id, tag_len);
    col_append_fstr(pinfo->cinfo, COL_INFO, ", MESHID=%s", s);
    proto_item_append_text(field_data->item_tag, ": %s", s);
  }
  /* Make sure dissector is accepted */
  return ((tag_len > 0) ? tag_len : 1);
}

static int
ieee80211_tag_mesh_preq(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  guint32 flags;
  guint8 targs, i;

  proto_tree_add_item_ret_uint(tree, hf_ieee80211_ff_hwmp_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN, &flags);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_pdid, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_orig_sta, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_orig_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  if (flags & (1<<6)) {
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_orig_ext, tvb, offset, 6, ENC_NA);
    offset += 6;
  }
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_count, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  targs = tvb_get_guint8(tvb, offset);
  offset += 1;
  for (i = 0; i < targs; i++) {
    const int * targ_flags[] = {
      &hf_ieee80211_ff_hwmp_targ_to_flags,
      &hf_ieee80211_ff_hwmp_targ_usn_flags,
      NULL
    };

    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_hwmp_targ_flags,
                                   ett_hwmp_targ_flags_tree, targ_flags, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sta, tvb, offset, 6, ENC_NA);
    offset += 6;
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;
  }

  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_mesh_prep(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  guint32 flags;
  proto_tree_add_item_ret_uint(tree, hf_ieee80211_ff_hwmp_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN, &flags);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_ttl, tvb, offset , 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sta, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  if (flags & (1<<6)) {
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_ext, tvb, offset, 6, ENC_NA);
    offset += 6;
  }
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_orig_sta, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_orig_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_mesh_perr(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  guint8 targs, i;

  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_count, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  targs = tvb_get_guint8(tvb, offset);
  offset += 1;
  for (i = 0; i < targs; i++) {
    guint8 flags = tvb_get_guint8(tvb, offset);

    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sta, tvb, offset, 6, ENC_NA);
    offset += 6;
    proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;
    if (flags & (1<<6)) {
      proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_targ_ext, tvb, offset, 6, ENC_NA);
      offset += 6;
    }
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_rann(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree_add_item(tree, hf_ieee80211_rann_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_rann_root_sta, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_rann_sn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_rann_interval, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_ff_hwmp_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

/* Mesh Channel Switch Parameters (118) */
static int
ieee80211_tag_mesh_channel_switch(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_mesh_chswitch_flag[] = {
    &hf_ieee80211_mesh_chswitch_flag_initiator,
    &hf_ieee80211_mesh_chswitch_flag_txrestrict,
    NULL
  };

  if (tag_len != 6)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 6", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_mesh_channel_switch_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_item_append_text(field_data->item_tag, " TTL: %d", tvb_get_guint8(tvb, offset));
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_mesh_channel_switch_flag,
                                   ett_mesh_chswitch_flag_tree, ieee80211_mesh_chswitch_flag, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_mesh_channel_switch_reason_code, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_mesh_channel_switch_precedence_value, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

/* Mesh Awake Window Parameters (119) */
static int
ieee80211_tag_mesh_awake_window(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
        "Tag length %u wrong, must be = 2", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_mesh_awake_window, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_channel_switch_announcement(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 4)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 4", tag_len);
    return tvb_captured_length(tvb);
  }

  add_ff_extended_channel_switch_announcement(tree, tvb, pinfo, offset);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_supported_operating_classes(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item* item = NULL;
  guint8 i;
  guint8 field_len = 0;
  guint8 alt_op_class_field[256];

  if (tag_len < 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be >= 2", tag_len);
    return tvb_captured_length(tvb);
  } else if (tag_len > 255) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, uint8 <= 255", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_supported_ope_classes_current, tvb, offset++, 1, ENC_NA);

  for (i = offset; i < tag_len; i++) {
    guint8 op_class =  tvb_get_guint8(tvb, i);
    /* Field terminates immediately before OneHundredAndThirty or Zero delimiter */
    if (op_class == 130 || op_class == 0) {
      break;
    }
    alt_op_class_field[field_len++] = op_class;
  }
  if (field_len) {
    item = proto_tree_add_item(tree, hf_ieee80211_tag_supported_ope_classes_alternate, tvb, offset, field_len, ENC_NA);
  }
  for (i = 0; i < field_len; i++) {
    proto_item_append_text(item, i == 0 ? ": %d":", %d", alt_op_class_field[i]);
  }

  /* TODO parse optional Current Operating Class Extension Sequence field */
  /* TODO parse optional Operating Class Duple Sequence field */
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_bss_parameter_change(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  gboolean size;
  if (tag_len != 7)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 7", tag_len);
    return tvb_captured_length(tvb);
  }
  size = (tvb_get_guint8(tvb, offset) & 0x02) >> 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_move, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_size, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_tbtt_offset, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  if(size == TRUE) { /* if size bit is 0, the field is reserved. */
    proto_tree_add_item(tree, hf_ieee80211_tag_bi_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_capabilities(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_tag_dmg_cap1[] = {
    &hf_ieee80211_tag_reverse_direction,
    &hf_ieee80211_tag_hlts,
    &hf_ieee80211_tag_tpc,
    &hf_ieee80211_tag_spsh,
    &hf_ieee80211_tag_rx_antenna,
    &hf_ieee80211_tag_fast_link,
    &hf_ieee80211_tag_num_sectors,
    &hf_ieee80211_tag_rxss_length,
    &hf_ieee80211_tag_reciprocity,
    &hf_ieee80211_tag_max_ampdu_exp,
    NULL
  };

  static const int * ieee80211_tag_dmg_cap2[] = {
    &hf_ieee80211_tag_min_mpdu_spacing,
    &hf_ieee80211_tag_ba_flow_control,
    &hf_ieee80211_tag_max_sc_rx_mcs,
    &hf_ieee80211_tag_max_ofdm_rx_mcs,
    &hf_ieee80211_tag_max_sc_tx_mcs,
    &hf_ieee80211_tag_max_ofdm_tx_mcs,
    NULL
  };

  static const int * ieee80211_tag_dmg_cap3[] = {
    &hf_ieee80211_tag_low_power_supported,
    &hf_ieee80211_tag_code_rate,
    &hf_ieee80211_tag_dtp,
    &hf_ieee80211_tag_appdu_supp,
    &hf_ieee80211_tag_heartbeat,
    &hf_ieee80211_tag_other_aid,
    &hf_ieee80211_tag_pattern_recip,
    &hf_ieee80211_tag_heartbeat_elapsed,
    &hf_ieee80211_tag_grant_ack_supp,
    &hf_ieee80211_tag_RXSSTxRate_supp,
    NULL
  };

  static const int * ieee80211_tag_dmg_cap4[] = {
    &hf_ieee80211_tag_pcp_tddti,
    &hf_ieee80211_tag_pcp_PSA,
    &hf_ieee80211_tag_pcp_handover,
    &hf_ieee80211_tag_pcp_max_assoc,
    &hf_ieee80211_tag_pcp_power_src,
    &hf_ieee80211_tag_pcp_decenter,
    &hf_ieee80211_tag_pcp_forwarding,
    &hf_ieee80211_tag_pcp_center,
    NULL
  };

  static const int * ieee80211_tag_dmg_cap5[] = {
    &hf_ieee80211_tag_ext_sc_mcs_max_tx,
    &hf_ieee80211_tag_ext_sc_mcs_tx_code_7_8,
    &hf_ieee80211_tag_ext_sc_mcs_max_rx,
    &hf_ieee80211_tag_ext_sc_mcs_rx_code_7_8,
    NULL
  };

  /*
   * Plenty of devices still do not conform to the older version of this
   * field. So, it must be at least 17 bytes in length.
   */
  if (tag_len < 17)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must contain at least 17 bytes", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_dmg_capa_sta_addr, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_tag_dmg_capa_aid, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_bitmask_list(tree, tvb, offset, 3, ieee80211_tag_dmg_cap1, ENC_LITTLE_ENDIAN);
  offset += 3;
  proto_tree_add_bitmask_list(tree, tvb, offset, 3, ieee80211_tag_dmg_cap2, ENC_LITTLE_ENDIAN);
  offset += 3;
  proto_tree_add_bitmask_list(tree, tvb, offset, 2, ieee80211_tag_dmg_cap3, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_bitmask_list(tree, tvb, offset, 2, ieee80211_tag_dmg_cap4, ENC_LITTLE_ENDIAN);
  offset += 2;

  /*
   * There are many captures out there that do not conform to the 2016
   * version, so give them a malformed IE message now after we have dissected
   * the above
   */
  if (tag_len != 22)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u does not conform to IEEE802.11-2016, should contain 22 bytes", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_sta_beam_track, tvb, offset, 2, ENC_NA);
  offset += 2;
  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_dmg_cap5, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_max_basic_sf_amsdu, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_max_short_sf_amsdu, tvb, offset, 1, ENC_NA);

  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_operation(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_tag_dmg_operation_flags[] = {
    &hf_ieee80211_tag_pcp_tddti,
    &hf_ieee80211_tag_pcp_PSA,
    &hf_ieee80211_tag_pcp_handover,
    NULL
  };

  if (tag_len != 10)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 10", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_dmg_operation_flags, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_PSRSI, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_min_BHI_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_brdct_sta_info_dur, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_assoc_resp_confirm_time, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_min_pp_duration, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_SP_idle_timeout, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_max_lost_beacons, tvb, offset, 1, ENC_NA);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_antenna_section_id(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_tag_antenna[] = {
    &hf_ieee80211_tag_type,
    &hf_ieee80211_tag_tap1,
    &hf_ieee80211_tag_state1,
    &hf_ieee80211_tag_tap2,
    &hf_ieee80211_tag_state2,
    NULL
  };

  if (tag_len != 4)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 4", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_bitmask_list(tree, tvb, offset, 4, ieee80211_tag_antenna, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_extended_schedule(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  int i;
  gboolean isGrant;
  proto_tree * alloc_tree;
  if ((tag_len%15) != 0)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be N*15 where 0<=N<=17", tag_len);
    return tvb_captured_length(tvb);
  }
  isGrant = ((field_data->ftype==CTRL_GRANT)||(field_data->ftype==CTRL_GRANT_ACK));
  for(i=0; i < tag_len; i+=15) {
    alloc_tree = proto_tree_add_subtree_format(tree, tvb, offset, 15, ett_allocation_tree, NULL, "Allocation %d", i/15);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_allocation_id, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_allocation_type, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_pseudo_static, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_truncatable, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_extendable, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_pcp_active, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_lp_sc_used, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    offset += add_ff_beamforming_ctrl(alloc_tree, tvb, pinfo, offset, isGrant);
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_src_aid, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_dest_aid, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_alloc_start, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_alloc_block_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_num_blocks, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(alloc_tree, hf_ieee80211_tag_alloc_block_period, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_sta_availability(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  int i;
  proto_tree * sta_info_tree;
  if ((tag_len%2) != 0)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be N*2 where N>=0", tag_len);
    return tvb_captured_length(tvb);
  }
  for(i=0; i < tag_len; i+=2) {
    sta_info_tree = proto_tree_add_subtree_format(tree, tvb, offset, 2, ett_sta_info, NULL, "STA Info %d", i/2);
    proto_tree_add_item(sta_info_tree, hf_ieee80211_tag_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(sta_info_tree, hf_ieee80211_tag_cbap, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(sta_info_tree, hf_ieee80211_tag_pp_avail, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_next_dmg_ati(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 6)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be = 6", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_next_ati_start_time, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_tag_next_ati_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_nextpcp_list(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  int i;
  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be at least 1", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_nextpcp_token, tvb, offset, 1, ENC_NA);
  offset += 1;
  for(i=0; i < tag_len-1; i+=1) {
    proto_tree_add_item(tree, hf_ieee80211_tag_nextpcp_list, tvb, offset, 1, ENC_NA);
    offset += 1;
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_pcp_handover(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 13)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 13", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_old_bssid, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_tag_new_pcp_addr, tvb, offset, 6, ENC_NA);
  offset += 6;
  proto_tree_add_item(tree, hf_ieee80211_tag_reamaining_BI, tvb, offset, 1, ENC_NA);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_beamlink_maintenance(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 1", tag_len);
    return tvb_captured_length(tvb);
  }
  add_ff_beamformed_link(tree, tvb, pinfo, offset);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_quiet_period_res(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  if (tag_len != 10)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 10", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_request_token, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_bssid, tvb, offset, 6, ENC_NA);
  offset += 6;
  add_ff_sta_address(tree, tvb, pinfo, offset);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_relay_transfer_param(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 8)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 8", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_duplex_relay, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_cooperation_relay, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_tx_mode, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_link_change_interval, tvb, offset+1, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_data_sensing_time, tvb, offset+2, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_first_period, tvb, offset+3, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_second_period, tvb, offset+5, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_beam_refinement(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_dmg_beam_refinement_fields[] = {
    &hf_ieee80211_tag_initiator,
    &hf_ieee80211_tag_tx_train_res,
    &hf_ieee80211_tag_rx_train_res,
    &hf_ieee80211_tag_tx_trn_ok,
    &hf_ieee80211_tag_txss_fbck_req,
    &hf_ieee80211_tag_bs_fbck,
    &hf_ieee80211_tag_bs_fbck_antenna_id,
    &hf_ieee80211_tag_snr_requested,
    &hf_ieee80211_tag_channel_measurement_requested,
    &hf_ieee80211_tag_number_of_taps_requested,
    &hf_ieee80211_tag_sector_id_order_req,
    &hf_ieee80211_tag_snr_present,
    &hf_ieee80211_tag_channel_measurement_present,
    &hf_ieee80211_tag_tap_delay_present,
    &hf_ieee80211_tag_number_of_taps_present,
    &hf_ieee80211_tag_number_of_measurement,
    &hf_ieee80211_tag_sector_id_order_present,
    &hf_ieee80211_tag_number_of_beams,
    &hf_ieee80211_tag_mid_extension,
    &hf_ieee80211_tag_capability_request,
    &hf_ieee80211_tag_beam_refine_reserved,
    NULL
  };

  if (tag_len != 5)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 5", tag_len);
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_list(tree, tvb, offset, 5, ieee80211_dmg_beam_refinement_fields, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_wakeup_schedule_ad(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 8)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 8", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_bi_start_time, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;
  proto_tree_add_item(tree, hf_ieee80211_tag_sleep_cycle, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_num_awake_bis, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_tspec(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  gboolean isGrant;
  int num_constraints;
  if (tag_len < 14)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be at least 14", tag_len);
    return tvb_captured_length(tvb);
  }
  static const int * ieee80211_tag_tspec_flags[] = {
    &hf_ieee80211_tag_tspec_allocation_id,
    &hf_ieee80211_tag_tspec_allocation_type,
    &hf_ieee80211_tag_tspec_allocation_format,
    &hf_ieee80211_tag_tspec_pseudo_static,
    &hf_ieee80211_tag_tspec_truncatable,
    &hf_ieee80211_tag_tspec_extendable,
    &hf_ieee80211_tag_tspec_lp_sc_used,
    &hf_ieee80211_tag_tspec_up,
    &hf_ieee80211_tag_tap2,
    &hf_ieee80211_tag_tspec_dest_aid,
    NULL
  };

  proto_tree_add_bitmask_list(tree, tvb, offset, 3, ieee80211_tag_tspec_flags, ENC_LITTLE_ENDIAN);
  offset += 3;
  isGrant = ((field_data->ftype==CTRL_GRANT)||(field_data->ftype==CTRL_GRANT_ACK));
  offset += add_ff_beamforming_ctrl(tree, tvb, pinfo, 2, isGrant);
  proto_tree_add_item(tree, hf_ieee80211_tag_tspec_allocation_period, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_tspec_min_allocation, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_tspec_max_allocation, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_tag_tspec_min_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  num_constraints = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_tspec_num_of_constraints, tvb, offset, 1, ENC_NA);
  offset += 1;
  while(num_constraints > 0) {
    proto_tree_add_item(tree, hf_ieee80211_tag_tspec_tsconst_start_time, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;
    proto_tree_add_item(tree, hf_ieee80211_tag_tspec_tsconst_duration, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_tag_tspec_tsconst_period, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_tag_tspec_tsconst_interferer_mac, tvb, offset, 2, ENC_NA);
    offset += 6;
    num_constraints--;
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_channel_measurement_fb(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  int num_measurement;
  if (tag_len%5 != 0)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be multiple of 5", tag_len);
    return tvb_captured_length(tvb);
  }
  num_measurement = tvb_get_guint8(tvb, offset+1);
  offset += 2;
  while(num_measurement > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_snr, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_tag_channel_measurement_feedback_realtive_I, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_tag_channel_measurement_feedback_realtive_Q, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_tag_channel_measurement_feedback_tap_delay, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_tag_channel_measurement_feedback_sector_id, tvb, offset, 1, ENC_NA);
    proto_tree_add_item(tree, hf_ieee80211_tag_channel_measurement_feedback_antenna_id, tvb, offset, 1, ENC_NA);
    offset += 1;
    num_measurement--;
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_awake_window(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;

  if (tag_len != 2)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 2", tag_len);
  }
  else
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_awake_window, tvb, 0, 2, ENC_LITTLE_ENDIAN);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_addba_ext(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;

  if (tag_len != 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 1", tag_len);
  }
  else
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_addba_ext_no_frag, tvb, 0, 1, ENC_NA);
    proto_tree_add_item(tree, hf_ieee80211_tag_addba_ext_he_fragmentation_operation, tvb, 0, 1, ENC_NA);
    proto_tree_add_item(tree, hf_ieee80211_tag_addba_ext_reserved, tvb, 0, 1, ENC_NA);
  }
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_multi_band(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  gboolean chiper_present, addr_present;
  static const int * ieee80211_tag_multi_band_ctrl[] = {
    &hf_ieee80211_tag_multi_band_ctrl_sta_role,
    &hf_ieee80211_tag_multi_band_ctrl_addr_present,
    &hf_ieee80211_tag_multi_band_ctrl_cipher_present,
    NULL
  };
  static const int * ieee80211_tag_multi_band_conn[] = {
    &hf_ieee80211_tag_multi_band_conn_ap,
    &hf_ieee80211_tag_multi_band_conn_pcp,
    &hf_ieee80211_tag_multi_band_conn_dls,
    &hf_ieee80211_tag_multi_band_conn_tdls,
    &hf_ieee80211_tag_multi_band_conn_ibss,
    NULL
  };

  if (tag_len < 22)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be at least 22", tag_len);
    return tvb_captured_length(tvb);
  }
  chiper_present = (tvb_get_letohs(tvb, offset) & 0x08) >> 3;
  addr_present = (tvb_get_letohs(tvb, offset) & 0x10) >> 4;
  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_multi_band_ctrl, ENC_NA);
  offset += 1;
  offset += add_ff_band_id(tree, tvb, pinfo, 1);
  proto_tree_add_item(tree, hf_ieee80211_tag_multi_band_oper_class, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_multi_band_channel_number, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_bssid, tvb, offset, 6, ENC_NA);
  offset += 6;
  offset += add_ff_beacon_interval(tree, tvb, pinfo, 2);
  proto_tree_add_item(tree, hf_ieee80211_tag_multi_band_tsf_offset, tvb, offset, 8, ENC_LITTLE_ENDIAN);
  offset += 8;

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_multi_band_conn, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_multi_band_fst_timeout, tvb, offset, 1, ENC_NA);
  offset += 1;
  if(addr_present)
  {
    proto_tree_add_item(tree, hf_ieee80211_tag_multi_band_sta_mac, tvb, offset, 6, ENC_NA);
    offset += 6;
  }
  if(chiper_present)
  {
    proto_item *rsn_pcs_count, *rsn_pcs_item, *rsn_sub_pcs_item;
    proto_tree *rsn_pcs_tree, *rsn_sub_pcs_tree;
    gint ii;
    guint16     pcs_count;
    int tag_end = tvb_reported_length(tvb);
    rsn_pcs_count = proto_tree_add_item(tree, hf_ieee80211_rsn_pcs_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    pcs_count = tvb_get_letohs(tvb, offset);
    offset += 2;

    if (offset + (pcs_count * 4) > tag_end)
    {
        expert_add_info_format(pinfo, rsn_pcs_count, &ei_ieee80211_rsn_pcs_count,
        "Pairwise Cipher Suite Count too large, 4*%u > %d", pcs_count, tag_end - offset);
        pcs_count = (tag_end - offset) / 4;
    }

    rsn_pcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_pcs_list, tvb, offset, pcs_count * 4, ENC_NA);
    rsn_pcs_tree = proto_item_add_subtree(rsn_pcs_item, ett_rsn_pcs_tree);
    for (ii = 0; ii < pcs_count; ii++)
    {
      rsn_sub_pcs_item = proto_tree_add_item(rsn_pcs_tree, hf_ieee80211_rsn_pcs, tvb, offset, 4, ENC_BIG_ENDIAN);
      rsn_sub_pcs_tree = proto_item_add_subtree(rsn_sub_pcs_item, ett_rsn_sub_pcs_tree);
      proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

      /* Check if OUI is 00:0F:AC (ieee80211) */
      if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
      {
        proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_80211_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
        proto_item_append_text(rsn_pcs_item, " %s", rsn_pcs_return(tvb_get_ntohl(tvb, offset)));
      } else {
        proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
      }
      offset += 4;
    }
  }

  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_link_margin(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 8)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 8", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_activity, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_dmg_link_adapt_mcs, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_dmg_link_adapt_link_margin, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_snr, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_ref_timestamp, tvb, offset, 3, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_link_adaption_ack(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len != 5)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be 5", tag_len);
    return tvb_captured_length(tvb);
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_activity, tvb, offset, 1, ENC_NA);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_tag_ref_timestamp, tvb, offset, 3, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_switching_stream(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static const int * ieee80211_tag_switching_stream_flags[] = {
    &hf_ieee80211_tag_switching_stream_old_tid,
    &hf_ieee80211_tag_switching_stream_old_direction,
    &hf_ieee80211_tag_switching_stream_new_tid,
    &hf_ieee80211_tag_switching_stream_new_direction,
    &hf_ieee80211_tag_switching_stream_new_valid_id,
    &hf_ieee80211_tag_switching_stream_llt_type,
    NULL
  };

  int param_num;
  if (tag_len < 4)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag Length %u wrong, must be at least 4", tag_len);
    return tvb_captured_length(tvb);
  }
  offset += add_ff_band_id(tree, tvb, pinfo, 1);
  offset += add_ff_band_id(tree, tvb, pinfo, 1);
  proto_tree_add_item(tree, hf_ieee80211_tag_switching_stream_non_qos, tvb, offset, 1, ENC_NA);
  offset += 1;
  param_num = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_switching_stream_param_num, tvb, offset, 1, ENC_NA);
  offset += 1;
  while(param_num > 0)
  {
    proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_switching_stream_flags, ENC_NA);
    param_num--;
    offset += 2;
  }
  return tvb_captured_length(tvb);
}

static void
ieee_80211_add_tagged_parameters(tvbuff_t *tvb, int offset, packet_info *pinfo,
                                  proto_tree *tree, int tagged_parameters_len, int ftype,
                                  association_sanity_check_t *association_sanity_check)
{
  int next_len;
  beacon_padding = 0; /* this is for the beacon padding confused with ssid fix */
  while (tagged_parameters_len > 0) {
    /* TODO make callers optionally specify the list of valid IE IDs? */
    if ((next_len=add_tagged_field (pinfo, tree, tvb, offset, ftype, NULL, 0, association_sanity_check)) == 0)
      break;
    if (next_len > tagged_parameters_len) {
      /* XXX - flag this as an error? */
      next_len = tagged_parameters_len;
    }
    offset                += next_len;
    tagged_parameters_len -= next_len;

    /* If FILS is used, all data after the FILS Session tag in a (re)association message is encrypted */
    if (association_sanity_check != NULL && association_sanity_check->has_fils_session) {
      proto_tree_add_item(tree, hf_ieee80211_fils_encrypted_data, tvb, offset, tagged_parameters_len, ENC_NA);
      break;
    }
  }
}

static void
ieee_80211_do_association_sanity_check(packet_info *pinfo, association_sanity_check_t *sanity_check)
{
  /* Given a [re-]association request frame, consider it in its totality and
     add expert information as appropriate */

  if (sanity_check->association_has_mobility_domain_element) {
    /* This is an FT association, warn about any non-FT AKM suites */
    if (sanity_check->has_non_ft_akm_suite) {
      expert_add_info_format(pinfo, sanity_check->rsn_first_non_ft_akm_suite, &ei_ieee80211_mismatched_akm_suite,
                             "Non-FT AKM suite is prohibited for FT association request");
    }
  } else {
    /* This is a non-FT association, warn about any FT AKM suites */
    if (sanity_check->has_ft_akm_suite) {
      expert_add_info_format(pinfo, sanity_check->rsn_first_ft_akm_suite, &ei_ieee80211_mismatched_akm_suite,
                             "FT AKM suite is prohibited for non-FT association request");
    }
  }
}

static ieee80211_conversation_data_t* get_or_create_conversation_data(conversation_t *conversation) {
  ieee80211_conversation_data_t *conversation_data = (ieee80211_conversation_data_t*)conversation_get_proto_data(conversation, proto_wlan);
  if (!conversation_data) {
    conversation_data = wmem_new(wmem_file_scope(), ieee80211_conversation_data_t);
    conversation_add_proto_data(conversation, proto_wlan, conversation_data);
  }
  return conversation_data;
}

/* ************************************************************************* */
/*                     Dissect 802.11 management frame                       */
/* ************************************************************************* */
static void
dissect_ieee80211_mgt(guint16 fcf, tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_item *ti;
  proto_tree *mgt_tree;
  proto_tree *fixed_tree;
  proto_tree *tagged_tree;
  int         offset = 0;
  int         tagged_parameter_tree_len;

  conversation_t *conversation;
  ieee80211_conversation_data_t *conversation_data;

  association_sanity_check_t association_sanity_check;
  memset(&association_sanity_check, 0, sizeof(association_sanity_check));


  ieee80211_tvb_invalid = FALSE;

  ti = proto_tree_add_item(tree, proto_wlan, tvb, 0, -1, ENC_NA);
  mgt_tree = proto_item_add_subtree(ti, ett_80211_mgt);

  switch (COMPOSE_FRAME_TYPE(fcf))
  {

    case MGT_ASSOC_REQ:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 4);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 0);
      add_ff_listen_ival(fixed_tree, tvb, pinfo, 2);
      offset = 4;  /* Size of fixed fields */

      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                 tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_ASSOC_REQ, &association_sanity_check);
      ieee_80211_do_association_sanity_check(pinfo, &association_sanity_check);

      conversation = find_or_create_conversation(pinfo);
      conversation_data = get_or_create_conversation_data(conversation);
      conversation_data->last_akm_suite = association_sanity_check.last_akm_suite;
      conversation_data->owe_group = association_sanity_check.owe_group;
      break;


    case MGT_ASSOC_RESP:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 6);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 0);
      add_ff_status_code(fixed_tree, tvb, pinfo, 2);
      add_ff_assoc_id(fixed_tree, tvb, pinfo, 4);
      offset = 6;  /* Size of fixed fields */

      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                 tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_ASSOC_RESP, &association_sanity_check);
      break;


    case MGT_REASSOC_REQ:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 10);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 0);
      add_ff_listen_ival(fixed_tree, tvb, pinfo, 2);
      add_ff_current_ap_addr(fixed_tree, tvb, pinfo, 4);
      offset = 10;  /* Size of fixed fields */

      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                 tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_REASSOC_REQ, &association_sanity_check);
      ieee_80211_do_association_sanity_check(pinfo, &association_sanity_check);

      conversation = find_or_create_conversation(pinfo);
      conversation_data = get_or_create_conversation_data(conversation);
      conversation_data->last_akm_suite = association_sanity_check.last_akm_suite;
      conversation_data->owe_group = association_sanity_check.owe_group;
      break;

    case MGT_REASSOC_RESP:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 6);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 0);
      add_ff_status_code(fixed_tree, tvb, pinfo, 2);
      add_ff_assoc_id(fixed_tree, tvb, pinfo, 4);
      offset = 6;  /* Size of fixed fields */

      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                 tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_REASSOC_RESP, &association_sanity_check);
      break;


    case MGT_PROBE_REQ:
      offset = 0;
      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                 tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_PROBE_REQ, NULL);
      break;

    case MGT_PROBE_RESP:
    {
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 12);
      add_ff_timestamp(fixed_tree, tvb, pinfo, 0);
      add_ff_beacon_interval(fixed_tree, tvb, pinfo, 8);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 10);
      offset = 12;  /* Size of fixed fields */

      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset, tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree, tagged_parameter_tree_len, MGT_PROBE_RESP, NULL);
      break;
    }
    case MGT_MEASUREMENT_PILOT:
    {
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 12);
      offset += add_ff_timestamp(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_measurement_pilot_int(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_beacon_interval(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_cap_info(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_country_str(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_max_reg_pwr(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_max_tx_pwr(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_tx_pwr_used(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_transceiver_noise_floor(fixed_tree, tvb, pinfo, offset);
      /* TODO DS Parameter Set ??? */

      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset, tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree, tagged_parameter_tree_len, MGT_MEASUREMENT_PILOT, NULL);
      break;
    }
    case MGT_BEACON:    /* Dissect protocol payload fields  */
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 12);
      add_ff_timestamp(fixed_tree, tvb, pinfo, 0);
      add_ff_beacon_interval(fixed_tree, tvb, pinfo, 8);
      add_ff_cap_info(fixed_tree, tvb, pinfo, 10);
      offset = 12;  /* Size of fixed fields */

      tagged_parameter_tree_len =
          tvb_reported_length_remaining(tvb, offset);
      tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
      tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
      tagged_parameter_tree_len, MGT_BEACON, NULL);
      break;

    case MGT_ATIM:
      break;

    case MGT_DISASS:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 2);
      add_ff_reason_code(fixed_tree, tvb, pinfo, 0);
      offset = 2; /* Size of fixed fields */
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      if (tagged_parameter_tree_len > 0) {
        tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                                                tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
                                         tagged_parameter_tree_len, MGT_DISASS, NULL);
      }

      conversation = find_conversation_pinfo(pinfo, proto_wlan);
      if (conversation) {
        conversation_delete_proto_data(conversation, proto_wlan);
      }
      break;

    case MGT_AUTHENTICATION:
      offset = 6;  /* Size of fixed fields */
      offset += get_ff_auth_len(tvb);

      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, offset);
      add_ff_auth_alg(fixed_tree, tvb, pinfo, 0);
      add_ff_auth_trans_seq(fixed_tree, tvb, pinfo, 2);
      add_ff_status_code(fixed_tree, tvb, pinfo, 4);
      add_ff_auth_sae(fixed_tree, tvb);
      add_ff_auth_fils(fixed_tree, tvb);

      tagged_parameter_tree_len =
        tvb_reported_length_remaining(tvb, offset);
      if (tagged_parameter_tree_len > 0)
      {
        tagged_tree = get_tagged_parameter_tree(mgt_tree,
            tvb,
            offset,
            tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
        tagged_parameter_tree_len, MGT_AUTHENTICATION, NULL);
      }
      break;

    case MGT_DEAUTHENTICATION:
      fixed_tree = get_fixed_parameter_tree(mgt_tree, tvb, 0, 2);
      add_ff_reason_code(fixed_tree, tvb, pinfo, 0);
      offset = 2; /* Size of fixed fields */
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      if (tagged_parameter_tree_len > 0) {
        tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
                                                tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
                                         tagged_parameter_tree_len, MGT_DEAUTHENTICATION, NULL);
      }

      conversation = find_conversation_pinfo(pinfo, proto_wlan);
      if (conversation) {
        conversation_delete_proto_data(conversation, proto_wlan);
      }
      break;

    case MGT_ACTION:
    {
      proto_item *lcl_fixed_hdr;
      proto_tree *lcl_fixed_tree;
      lcl_fixed_tree = proto_tree_add_subtree(mgt_tree, tvb, 0, 0, ett_fixed_parameters, &lcl_fixed_hdr, "Fixed parameters");
      offset += add_ff_action(lcl_fixed_tree, tvb, pinfo, 0);

      proto_item_set_len(lcl_fixed_hdr, offset);
      if (ieee80211_tvb_invalid)
        break; /* Buffer not available for further processing */
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      if (tagged_parameter_tree_len > 0)
      {
        tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
          tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_ACTION, NULL);
      }
      break;
    }
    case MGT_ACTION_NO_ACK:
    {
      proto_item *lcl_fixed_hdr;
      proto_tree *lcl_fixed_tree;
      lcl_fixed_tree = proto_tree_add_subtree(mgt_tree, tvb, 0, 0, ett_fixed_parameters, &lcl_fixed_hdr, "Fixed parameters");

      offset += add_ff_action(lcl_fixed_tree, tvb, pinfo, 0);

      proto_item_set_len(lcl_fixed_hdr, offset);
      if (ieee80211_tvb_invalid)
        break; /* Buffer not available for further processing */
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
      if (tagged_parameter_tree_len > 0)
      {
        tagged_tree = get_tagged_parameter_tree(mgt_tree, tvb, offset,
          tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
          tagged_parameter_tree_len, MGT_ACTION_NO_ACK, NULL);
      }
      break;
    }
    case MGT_ARUBA_WLAN:
    {
      proto_tree *aruba_tree;
      guint16 type;
      type = tvb_get_ntohs(tvb, offset);

      aruba_tree = proto_tree_add_subtree(mgt_tree, tvb, 0, 0, ett_fixed_parameters, NULL, "Aruba Management");

      proto_tree_add_item(aruba_tree, hf_ieee80211_aruba, tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      switch(type){
        case 0x0003: /* MTU Size */
          proto_tree_add_item(aruba_tree, hf_ieee80211_aruba_mtu, tvb, offset, 2, ENC_BIG_ENDIAN);
        break;
        case 0x0005: /* HeartBeat Sequence */
          proto_tree_add_item(aruba_tree, hf_ieee80211_aruba_hb_seq, tvb, offset, 8, ENC_BIG_ENDIAN);
        break;
      }
      break;
    }
  }
}

/*
 * Dissect a Block Ack request (which is also used in Trigger frames).
 */
static const int *block_ack_control_headers[] = {
  &hf_ieee80211_block_ack_control_ack_policy,
  &hf_ieee80211_block_ack_control_type,
  &hf_ieee80211_block_ack_control_reserved,
  &hf_ieee80211_block_ack_control_tid_info,
  NULL
};

static const int *multi_sta_aid_tid_headers[] = {
  &hf_ieee80211_block_ack_multi_sta_aid11,
  &hf_ieee80211_block_ack_multi_sta_ack_type,
  &hf_ieee80211_block_ack_multi_sta_tid,
 NULL
};

/*
 * These bits are shown in reverse order in the spec.
 */
#define BASIC_BLOCK_ACK               0x0
#define EXTENDED_COMPRESSED_BLOCK_ACK 0x1
#define COMPRESSED_BLOCK_ACK          0x2
#define MULTI_TID_BLOCK_ACK           0x3
#define GCR_BLOCK_ACK                 0x6
#define GLK_GCR_BLOCK_ACK             0xA
#define MULTI_STA_BLOCK_ACK           0xB

static const value_string block_ack_type_vals[] = {
  { BASIC_BLOCK_ACK,               "Basic BlockAck" },
  { EXTENDED_COMPRESSED_BLOCK_ACK, "Extended Compressed BlockAck" },
  { COMPRESSED_BLOCK_ACK,          "Compressed BlockAck" },
  { MULTI_TID_BLOCK_ACK,           "Multi-TID BlockAck" },
  { GCR_BLOCK_ACK,                 "GCR BlockAck" },
  { GLK_GCR_BLOCK_ACK,             "GLK-GCR BlockAck" },
  { MULTI_STA_BLOCK_ACK,           "Multi-STA BlockAck" },
  { 0, NULL }
};

static int
dissect_ieee80211_block_ack_details(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, gboolean isDMG, gboolean is_req, gboolean has_fcs)
{
  proto_item     *pi;
  guint16         ba_control;
  guint8          block_ack_type;
  proto_tree     *ba_tree;
  guint8          tid_count, frag_num;
  guint           i;
  proto_tree     *ba_mtid_tree, *ba_mtid_sub_tree;
  guint16         ssn;
  guint64         bmap;
  guint           f;
  proto_item     *ba_bitmap_item;
  proto_tree     *ba_bitmap_tree;
  guint16         aid_tid;
  proto_tree     *ba_multi_sta_tree;
  int             ba_start = offset;

  ba_control = tvb_get_letohs(tvb, offset);
  block_ack_type = (ba_control & 0x001E) >> 1;
  ba_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1, ett_block_ack,
                        &pi, is_req ? "%s Request" : "%s Response",
                        val_to_str(block_ack_type, block_ack_type_vals,
                                "Reserved (%d)"));
  proto_tree_add_bitmask_with_flags(ba_tree, tvb, offset,
                        hf_ieee80211_block_ack_control,
                        ett_block_ack_request_control,
                        block_ack_control_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  switch (block_ack_type) {
  case BASIC_BLOCK_ACK:
    if (isDMG == TRUE) {
      expert_add_info_format(pinfo, ba_tree, &ei_ieee80211_dmg_subtype,
                        "DMG STAs shouldn't transmit BlockAckReq frames "
                        "with Basic BlockAckReqs");
    }

    /* Both request and response have an SSC */
    offset += add_ff_block_ack_ssc(ba_tree, tvb, pinfo, offset);

    if (!is_req) {
      proto_tree_add_item(ba_tree, hf_ieee80211_block_ack_bitmap, tvb, offset,
                        128, ENC_NA);
      offset += 128;
    }
    break;

  case COMPRESSED_BLOCK_ACK:
    /*
     * FIXME: For 802.11ax, the block ack bitmap can be 8 or 32 bytes
     * depending on the values of the fragment number subfield in the
     * SSC! All values other that 0 and 2 in bits B1 & B2 are reserved.
     */
    ssn = tvb_get_letohs(tvb, offset);
    frag_num = ssn & 0x0F;
    ssn >>= 4;
    offset += add_ff_block_ack_ssc(ba_tree, tvb, pinfo, offset);

    if (!is_req) {
      if ((frag_num & 0x06) == 0) {
        bmap = tvb_get_letoh64(tvb, offset);
        ba_bitmap_item = proto_tree_add_item(ba_tree,
                          hf_ieee80211_block_ack_bitmap,
                          tvb, offset, 8, ENC_NA);
        ba_bitmap_tree = proto_item_add_subtree(ba_bitmap_item,
                          ett_block_ack_bitmap);
        for (f = 0; f < 64; f++) {
          if (bmap & (G_GUINT64_CONSTANT(1) << f))
            continue;
          proto_tree_add_uint_format_value(ba_bitmap_tree,
                          hf_ieee80211_block_ack_bitmap_missing_frame,
                          tvb, offset + (f/8), 1, ssn + f, "%u",
                          (ssn + f) & 0x0fff);
        }
        offset += 8;
      } else if (((frag_num & 0x06) >> 1) == 2) {

        ba_bitmap_item = proto_tree_add_item(ba_tree,
                        hf_ieee80211_block_ack_bitmap, tvb,
                        offset, 32, ENC_NA);
        ba_bitmap_tree = proto_item_add_subtree(ba_bitmap_item,
                            ett_block_ack_bitmap);
        for (i = 0; i < 256; i += 64) {
          bmap = tvb_get_letoh64(tvb, offset + i/8);
          for (f = i; f <  i + 64; f++) {
            if (bmap & (G_GUINT64_CONSTANT(1) << f))
              continue;
            proto_tree_add_uint_format_value(ba_bitmap_tree,
                          hf_ieee80211_block_ack_bitmap_missing_frame,
                          tvb, offset + (f/8), 1, ssn + f, "%u",
                          (ssn + f) & 0x0fff);
          }
        }
        offset += 32;
      } else {
        /* Reserved ... */
      }
    }
    break;

  case EXTENDED_COMPRESSED_BLOCK_ACK:
    if (isDMG == FALSE) {
      expert_add_info_format(pinfo, ba_tree, &ei_ieee80211_dmg_subtype,
                        "Non-DMG STAs shouldn't transmit BlockAckReq "
                        "frames with Extended Compressed BlockAckReqs");
    }

    offset += add_ff_block_ack_ssc(ba_tree, tvb, pinfo, offset);

    if (!is_req) {
      ssn = tvb_get_letohs(tvb, offset);
      ssn >>= 4;

      bmap = tvb_get_letoh64(tvb, offset);
      ba_bitmap_item = proto_tree_add_item(ba_tree,
                        hf_ieee80211_block_ack_bitmap,
                        tvb, offset, 8, ENC_NA);
      ba_bitmap_tree = proto_item_add_subtree(ba_bitmap_item,
                        ett_block_ack_bitmap);
      for (f = 0; f < 64; f++) {
        if (bmap & (G_GUINT64_CONSTANT(1) << f))
          continue;
        proto_tree_add_uint(ba_bitmap_tree,
                        hf_ieee80211_block_ack_bitmap_missing_frame,
                        tvb, offset + (f/8), 1, ssn + f);
      }
      offset += 8;
      proto_tree_add_item(ba_tree, hf_ieee80211_block_ack_RBUFCAP, tvb, offset,
                        1, ENC_LITTLE_ENDIAN);
      offset += 1;
    }
    break;

  case MULTI_TID_BLOCK_ACK:

    if (isDMG == TRUE) {
      expert_add_info_format(pinfo, ba_tree, &ei_ieee80211_dmg_subtype,
                        "DMG STAs shouldn't transmit BlockAckReq frames "
                        "with Multi-TID BlockAckReqs");
    }

    tid_count = ((ba_control & 0xF000) >> 12) + 1;
    if (is_req) {
      ba_mtid_tree = proto_tree_add_subtree(ba_tree, tvb, offset, tid_count*4,
                ett_block_ack, NULL, "TID List");
      for (i = 0; i < tid_count; i++) {
        guint8 tid = tvb_get_guint8(tvb, offset) & 0x0F;
        ba_mtid_sub_tree = proto_tree_add_subtree_format(ba_mtid_tree, tvb,
                                offset, 4, ett_block_ack_tid, NULL,
                                "TID %u details", tid);

        proto_tree_add_item(ba_mtid_sub_tree,
                hf_ieee80211_block_ack_multi_tid_reserved, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
        proto_tree_add_item(ba_mtid_sub_tree,
                hf_ieee80211_block_ack_multi_tid_value, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
        offset += 2;

        offset += add_ff_block_ack_ssc(ba_mtid_sub_tree, tvb, pinfo, offset);
      }
    } else {
      ba_mtid_tree = proto_tree_add_subtree(ba_tree, tvb, offset, tid_count*4,
                ett_block_ack, NULL, "TID List");
      for (i = 0; i < tid_count; i++) {
        guint8 tid = tvb_get_guint8(tvb, offset) & 0x0F;
        ba_mtid_sub_tree = proto_tree_add_subtree_format(ba_mtid_tree, tvb,
                                offset, 4, ett_block_ack_tid, NULL,
                                "TID %u details", tid);

        proto_tree_add_item(ba_mtid_sub_tree,
                        hf_ieee80211_block_ack_multi_tid_reserved,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
        proto_tree_add_item(ba_mtid_sub_tree,
                        hf_ieee80211_block_ack_multi_tid_value, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
        offset += 2;

        offset += add_ff_block_ack_ssc(ba_mtid_sub_tree, tvb, pinfo, offset);
        proto_tree_add_item(ba_mtid_sub_tree,
                        hf_ieee80211_block_ack_bitmap, tvb, offset, 8, ENC_NA);
        offset += 8;
      }
    }
    break;

  case GCR_BLOCK_ACK:
    offset += add_ff_block_ack_ssc(ba_tree, tvb, pinfo, offset);

    proto_tree_add_item(ba_tree, hf_ieee80211_block_ack_gcr_addr, tvb,
                        offset, 6, ENC_NA);
    offset += 6;

    if (!is_req) {
      proto_tree_add_item(ba_tree,
                        hf_ieee80211_block_ack_bitmap, tvb, offset, 8, ENC_NA);
      offset += 8;
    }
    break;

  case MULTI_STA_BLOCK_ACK:
    while (tvb_reported_length_remaining(tvb, offset) > (has_fcs ? 4 : 0)) {
        int start = offset;
        proto_item *msta_ti = NULL;
        aid_tid = tvb_get_letohs(tvb, offset);
        ba_multi_sta_tree = proto_tree_add_subtree_format(ba_tree, tvb, offset, -1,
                                ett_multi_sta_block_ack, &msta_ti,
                                "Per AID TID Info: 0x%0x", aid_tid & 0x07ff);
        proto_tree_add_bitmask_with_flags(ba_multi_sta_tree, tvb, offset,
                        hf_ieee80211_block_ack_multi_sta_aid_tid,
                        ett_block_ack_request_multi_sta_aid_tid,
                        multi_sta_aid_tid_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
        offset += 2;

        if ((aid_tid & 0x07ff) != 2045) {
          if (((aid_tid & 0x0800) == 0) && ((aid_tid & 0xf000) >> 12) <= 7) {
            guint bitmap_size;
            ssn = tvb_get_letohs(tvb, offset);
            frag_num = ssn & 0x0F;
            ssn >>= 4;
            offset += add_ff_block_ack_ssc(ba_multi_sta_tree, tvb, pinfo, offset);
            bitmap_size = ((frag_num + 2) & 0x6) >> 1;  /* Turn into an exponent */
            bitmap_size = 4 << bitmap_size;  /* It goes 4, 8, 16, 32 */
            ba_bitmap_item = proto_tree_add_item(ba_multi_sta_tree,
                        hf_ieee80211_block_ack_bitmap, tvb,
                        offset, bitmap_size, ENC_NA);
            ba_bitmap_tree = proto_item_add_subtree(ba_bitmap_item,
                                ett_block_ack_bitmap);
            for (i = 0; i < bitmap_size * 8; i += 64) {
              bmap = tvb_get_letoh64(tvb, offset + i/8);
              for (f = i; f <  i + ((bitmap_size == 4 ? 4 : 8) * 8); f++) {
                if (bmap & (G_GUINT64_CONSTANT(1) << f))
                  continue;
                proto_tree_add_uint_format_value(ba_bitmap_tree,
                              hf_ieee80211_block_ack_bitmap_missing_frame,
                              tvb, offset + (f/8), 1, ssn + f, "%u",
                              (ssn + f) & 0x0fff);
              }
            }
            offset += bitmap_size;
          }
        } else {
          offset += add_ff_block_ack_ssc(ba_multi_sta_tree, tvb, pinfo, offset);
          proto_tree_add_item(ba_multi_sta_tree, hf_ieee80211_block_ack_multi_sta_reserved,
                                tvb, offset, 2, ENC_LITTLE_ENDIAN);
          offset += 2;
          proto_tree_add_item(ba_multi_sta_tree, hf_ieee80211_block_ack_multi_sta_ra,
                                tvb, offset, 6, ENC_NA);
          offset += 6;
        }
        proto_item_set_len(msta_ti, offset - start);
    }
    break;
  }
  proto_item_set_len(pi, offset - ba_start);

  return offset;
}

static void
dissect_ieee80211_block_ack(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, gboolean isDMG, gboolean is_req, gboolean has_fcs)
{
  const gchar *ether_name = tvb_get_ether_name(tvb, offset);
  proto_item     *hidden_item;

  proto_tree_add_item(tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_ta_resolved, tvb,
                        offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_item(tree, hf_ieee80211_addr, tvb, offset, 6,
                        ENC_NA);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_resolved, tvb,
                        offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  offset += 6;

  dissect_ieee80211_block_ack_details(tvb, pinfo, tree, offset, isDMG, is_req, has_fcs);
}

/*
 * Dissect an 802.11ax HE Trigger frame and return the actual len including
 * padding!
 */

#define TRIGGER_TYPE_BASIC      0
#define TRIGGER_TYPE_BRP        1
#define TRIGGER_TYPE_MU_BAR     2
#define TRIGGER_TYPE_MU_RTS     3
#define TRIGGER_TYPE_BSRP       4
#define TRIGGER_TYPE_GCR_MU_BAR 5
#define TRIGGER_TYPE_BQRP       6
#define TRIGGER_TYPE_NFRP       7

static const val64_string trigger_type_vals[] = {
  { 0, "Basic" },
  { 1, "Beamforming Report Poll (BRP)" },
  { 2, "MU-BAR" },
  { 3, "MU-RTS" },
  { 4, "Buffer Status Report Poll (BSRP)" },
  { 5, "GCR MU-BAR" },
  { 6, "Bandwidth Query Report Poll (BQRP)" },
  { 7, "NDP Feedback Report Poll (NFRP)" },
  { 0, NULL }
};

static const val64_string bw_subfield_vals[] = {
  { 0, "20 MHz" },
  { 1, "40 MHz" },
  { 2, "80 MHz" },
  { 3, "80+80 MHz or 160 MHz" },
  { 0, NULL }
};

static const val64_string gi_and_ltf_type_subfield_vals[] = {
  { 0, "1x LTF + 1.6 us GI" },
  { 1, "2x LTF + 1.6 us GI" },
  { 2, "4x LTF + 3.2 us GI" },
  { 3, "Reserved" },
  { 0, NULL }
};

static const true_false_string mu_mimo_ltf_mode_tfs = {
  "HE masked HE LTF sequence mode",
  "HE single stream pilot HE LTF mode"
};

static const int *he_trig_frm_bar_ctrl_fields[] = {
  &hf_ieee80211_he_trigger_bar_ctrl_ba_ack_policy,
  &hf_ieee80211_he_trigger_bar_ctrl_ba_type,
  &hf_ieee80211_he_trigger_bar_ctrl_reserved,
  &hf_ieee80211_he_trigger_bar_ctrl_tid_info,
  NULL
};

static const int *he_trig_frm_bar_info_fields[] = {
  &hf_ieee80211_he_trigger_bar_info_blk_ack_seq_ctrl,
  NULL
};

#define PRE_FEC_PADDING_FACTOR 0x3
#define PE_DISAMBIGUITY 0x4

static void
ap_tx_power_custom(gchar *result, guint32 ap_tx_power)
{
  if (ap_tx_power > 60)
    g_snprintf(result, ITEM_LABEL_LENGTH, "%s", "Reserved");
  else
    g_snprintf(result, ITEM_LABEL_LENGTH, "%d dBm", -20 + ap_tx_power);
}

static void
ul_packet_extension_base_custom(gchar *result, guint32 ul_packet_extension)
{
  int pre_fec_padding = ul_packet_extension & PRE_FEC_PADDING_FACTOR;

  if (pre_fec_padding == 0) pre_fec_padding = 4;

  if (ul_packet_extension & PE_DISAMBIGUITY) {
    g_snprintf(result, ITEM_LABEL_LENGTH,
               "PE disambiguity & pre-FEC padding factor of %d",
               pre_fec_padding);

  } else {
    g_snprintf(result, ITEM_LABEL_LENGTH,
               "no PE disambiguity & pre-FEC padding factor of %d",
               pre_fec_padding);
  }
}

static void
add_gcr_mu_bar_trigger_frame_common_info(proto_tree *tree, tvbuff_t *tvb,
  int offset)
{

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_he_trigger_bar_ctrl,
                        ett_he_trigger_bar_ctrl,
                        he_trig_frm_bar_ctrl_fields,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_he_trigger_bar_info,
                        ett_he_trigger_bar_info,
                        he_trig_frm_bar_info_fields,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
}

static const int *common_info_headers[] = {
  &hf_ieee80211_he_trigger_type,
  &hf_ieee80211_he_trigger_ul_length,
  &hf_ieee80211_he_trigger_more_tf,
  &hf_ieee80211_he_trigger_cs_required,
  &hf_ieee80211_he_trigger_ul_bw,
  &hf_ieee80211_he_trigger_gi_and_ltf_type,
  &hf_ieee80211_he_trigger_mu_mimo_ltf_mode,
  &hf_ieee80211_he_trigger_num_he_ltf_syms_etc,
  &hf_ieee80211_he_trigger_ul_stbc,
  &hf_ieee80211_he_trigger_ldpc_extra_sym_seg,
  &hf_ieee80211_he_trigger_ap_tx_power,
  &hf_ieee80211_he_trigger_ul_packet_extension,
  &hf_ieee80211_he_trigger_ul_spatial_reuse,
  &hf_ieee80211_he_trigger_doppler,
  &hf_ieee80211_he_trigger_ul_he_sig_a_reserved,
  &hf_ieee80211_he_trigger_reserved,
  NULL
};

static int
add_he_trigger_common_info(proto_tree *tree, tvbuff_t *tvb, int offset,
  packet_info *pinfo _U_, guint8 trigger_type, int *frame_len)
{
  proto_item     *pi = NULL;
  proto_tree     *common_info = NULL;
  int            length = 0;
  int            start_offset = offset;
  guint32         bw_etc = tvb_get_letoh24(tvb, offset);

  global_he_trigger_bw = (bw_etc >> 18) & 0x03;

  common_info = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_he_trigger_common_info, &pi, "Common Info");

  proto_tree_add_bitmask_with_flags(common_info, tvb, offset,
                        hf_ieee80211_he_trigger_common_info,
                        ett_he_trigger_base_common_info,
                        common_info_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 8;
  length += 8;

  /*
   * Handle the trigger dependent common info
   */
  switch (trigger_type) {
    case TRIGGER_TYPE_GCR_MU_BAR: /* Actually two uint16 fields */
      add_gcr_mu_bar_trigger_frame_common_info(common_info, tvb, offset);
      offset += 4;
      length += 4;
      break;
    default: /* No other type has a trigger dependent common info subfield */
      break;
  }
  proto_item_set_len(pi, offset - start_offset);

  *frame_len += length;
  return length;
}

static const char * he_trigger_ru_allocation_region_values[] = {
  "primary 80MHz channel for 80+80 and 160MHz",
  "secondary 80MHz channel for 80+80 and 160MHz",
};

/*
 * Format the ru allocation region bit
 */
static void
he_trigger_ru_allocation_region_custom(gchar *result, guint32 ru_alloc_val)
{
  if (global_he_trigger_bw == 3) {
    g_snprintf(result, ITEM_LABEL_LENGTH, "%s",
      he_trigger_ru_allocation_region_values[ru_alloc_val & 0x01]);
  } else {
    g_snprintf(result, ITEM_LABEL_LENGTH, "Not used for 20, 40 or 80MHz");
  }
}

static void
he_trigger_minus_one_custom(gchar *result, guint ss_alloc_value)
{
  g_snprintf(result, ITEM_LABEL_LENGTH, "%d", ss_alloc_value + 1);
}

static const true_false_string he_trigger_ul_fec_coding_type_tfs = {
  "LDPC",
  "BCC"
};

static const value_string preferred_ac_vals[] = {
  { 0, "AC_BK" },
  { 1, "AC_BE" },
  { 2, "AC_VI" },
  { 3, "AC_VO" },
  { 0, NULL }
};

static const int *basic_trigger_dependent_user_headers[] = {
  &hf_ieee80211_he_trigger_mpdu_mu_spacing,
  &hf_ieee80211_he_trigger_tid_aggregation_limit,
  &hf_ieee80211_he_trigger_dependent_reserved1,
  &hf_ieee80211_he_trigger_preferred_ac,
  NULL
};

static void
add_basic_trigger_dependent_user_info(proto_tree *tree, tvbuff_t *tvb,
  int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_he_trigger_dep_basic_user_info,
                        ett_he_trigger_dep_basic_user_info,
                        basic_trigger_dependent_user_headers,
                        ENC_NA, BMT_NO_APPEND);
}

static void
add_brp_trigger_dependent_user_info(proto_tree *tree, tvbuff_t *tvb,
  int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_he_trigger_feedback_seg_retrans_bm,
                        tvb, offset, 1, ENC_NA);
}

static int
add_mu_bar_trigger_dependent_user_info(proto_tree *tree, tvbuff_t *tvb,
  int offset, packet_info *pinfo, int *frame_len)
{
  int start_offset = offset;

  /*
   * It's a request and not DMG, I think. Also, it is only supposed to be
   * a compressed block ack or a multi-tid block ack request.
   */
  offset = dissect_ieee80211_block_ack_details(tvb, pinfo, tree, offset,
                        FALSE, TRUE, FALSE);

  *frame_len += offset - start_offset;

  return offset;
}


static const int *nfrp_trigger_dependent_user_headers[] = {
  &hf_ieee80211_he_trigger_starting_aid,
  &hf_ieee80211_he_trigger_dependent_reserved2,
  &hf_ieee80211_he_trigger_feedback_type,
  &hf_ieee80211_he_trigger_dependent_reserved3,
  &hf_ieee80211_he_trigger_nfrp_target_rssi,
  &hf_ieee80211_he_trigger_multiplexing_flag,
  NULL
};

static void
add_nfrp_trigger_dependent_user_info(proto_tree *tree, tvbuff_t *tvb,
  int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_he_trigger_dep_nfrp_user_info,
                        ett_he_trigger_dep_nfrp_user_info,
                        nfrp_trigger_dependent_user_headers,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
}

/*
 * Print the target RSSI field as per the spec.
 *  0->90 map to -110 to -20 dBm.
 *  127 maps to Max ransmit power for assigned MCS
 *  rest are reserved.
 */
static void
target_rssi_base_custom(gchar *result, guint32 target_rssi)
{
  if (target_rssi <= 90) {
    g_snprintf(result, ITEM_LABEL_LENGTH, "%ddBm", -110 + target_rssi);
  } else if (target_rssi == 127) {
    g_snprintf(result, ITEM_LABEL_LENGTH, "Max transmit power");
  } else {
    g_snprintf(result, ITEM_LABEL_LENGTH, "Reserved");
  }
}

static const int *user_info_headers_no_2045[] = {
  &hf_ieee80211_he_trigger_aid12,
  &hf_ieee80211_he_trigger_ru_allocation_region,
  &hf_ieee80211_he_trigger_ru_allocation,
  &hf_ieee80211_he_trigger_ul_fec_coding_type,
  &hf_ieee80211_he_trigger_ul_mcs,
  &hf_ieee80211_he_trigger_ul_dcm,
  &hf_ieee80211_he_trigger_ru_starting_spatial_stream,
  &hf_ieee80211_he_trigger_ru_number_spatial_streams,
  &hf_ieee80211_he_trigger_ul_target_rssi,
  &hf_ieee80211_he_trigger_user_reserved,
  NULL
};

static const int *user_info_headers_2045[] = {
  &hf_ieee80211_he_trigger_aid12,
  &hf_ieee80211_he_trigger_ru_allocation_region,
  &hf_ieee80211_he_trigger_ru_allocation,
  &hf_ieee80211_he_trigger_ul_fec_coding_type,
  &hf_ieee80211_he_trigger_ul_mcs,
  &hf_ieee80211_he_trigger_ul_dcm,
  &hf_ieee80211_he_trigger_ru_number_ra_ru,
  &hf_ieee80211_he_trigger_ru_no_more_ra_ru,
  &hf_ieee80211_he_trigger_ul_target_rssi,
  &hf_ieee80211_he_trigger_user_reserved,
  NULL
};

static int
add_he_trigger_user_info(proto_tree *tree, tvbuff_t *tvb, int offset,
  packet_info *pinfo, guint8 trigger_type, int *frame_len)
{
  proto_item     *pi = NULL;
  proto_tree     *user_info = NULL;
  int            length = 0;
  int            start_offset = offset;
  guint16         aid12_subfield = 0;

  /*
   * If the AID12 subfield has the value 4095 it indicates the start of
   * the padding field.
   */
  user_info = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_he_trigger_user_info, &pi, "User Info");
  aid12_subfield = tvb_get_letohs(tvb, offset) & 0xFFF;

  while (aid12_subfield != 4095) {

    if (aid12_subfield != 0 && aid12_subfield != 2045)
      proto_tree_add_bitmask_with_flags(user_info, tvb, offset,
                          hf_ieee80211_he_trigger_user_info,
                          ett_he_trigger_base_user_info,
                          user_info_headers_no_2045,
                          ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    else
      proto_tree_add_bitmask_with_flags(user_info, tvb, offset,
                          hf_ieee80211_he_trigger_user_info,
                          ett_he_trigger_base_user_info,
                          user_info_headers_2045,
                          ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 5;
    length += 5;

    /*
     * Handle the trigger dependent user info
     */
    switch (trigger_type) {
      case TRIGGER_TYPE_BASIC:
        add_basic_trigger_dependent_user_info(user_info, tvb, offset);
        offset++;
        length++;
        break;
      case TRIGGER_TYPE_BRP:
        add_brp_trigger_dependent_user_info(user_info, tvb, offset);
        offset++;
        length++;
        break;
      case TRIGGER_TYPE_MU_BAR:
        /* This is variable length so we need it to update the length */
        offset = add_mu_bar_trigger_dependent_user_info(user_info, tvb,
                                offset, pinfo, &length);
        break;
      case TRIGGER_TYPE_NFRP:
        add_nfrp_trigger_dependent_user_info(user_info, tvb, offset);
        offset += 5;
        length += 5;
        break;
      default:
        break;
    }

    if (tvb_reported_length_remaining(tvb, offset) < 5)
      aid12_subfield = 4095;
    else
      aid12_subfield = tvb_get_letohs(tvb, offset) & 0xFFF;
  }

  proto_item_set_len(pi, offset - start_offset);

  *frame_len += length;
  return length;
}

static int
dissect_ieee80211_he_trigger(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset)
{
  const gchar *ether_name = tvb_get_ether_name(tvb, offset);
  proto_item      *hidden_item;
  guint8          trigger_type = 0;
  int             length = 0;

  proto_tree_add_item(tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_ta_resolved,
                        tvb, offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_item(tree, hf_ieee80211_addr, tvb, offset, 6,
                        ENC_NA);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_resolved, tvb,
                        offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);

  offset += 6;
  length += 6;

  trigger_type = tvb_get_guint8(tvb, offset) & 0x0F;
  col_append_fstr(pinfo->cinfo, COL_INFO, " %s",
                val64_to_str(trigger_type, trigger_type_vals, "Reserved"));
  /*
   * Deal with the common Info and then any user info after that.
   */
  offset += add_he_trigger_common_info(tree, tvb, offset, pinfo,
                        trigger_type, &length);

  /*
   * Now the User Info field. It returns an offset not a length used.
   */
  add_he_trigger_user_info(tree, tvb, offset, pinfo,
                        trigger_type, &length);

  /*
   *  Padding should commence here ... TODO, deal with it.
   */

  return length;
}

/*
 * Dissect a VHT or an HE NDP accouncement frame. They differ past
 * the sounding dialog token with a bit in the SDT indicating VHT vs HE.
 */
#define NDP_ANNC_VHT_HE 0x02

static const true_false_string he_ndp_annc_he_subfield_vals = {
  "HE NDP Announcement frame",
  "VHT NDP Announcement frame"
};

static const int *vht_ndp_headers[] = {
  &hf_ieee80211_vht_ndp_annc_token_reserved,
  &hf_ieee80211_vht_ndp_annc_he_subfield,
  &hf_ieee80211_vht_ndp_annc_token_number,
  NULL
};

static int
dissect_ieee80211_vht_ndp_annc(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, gboolean has_fcs)
{
  guint16          sta_info;
  guint8           len_fcs = 0;
  proto_tree      *sta_list;
  proto_item      *sta_info_item, *pi;
  int              saved_offset = 0;
  int              sta_index = 0;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_vht_ndp_annc_token, ett_vht_ndp_annc,
                        vht_ndp_headers, ENC_NA, BMT_NO_APPEND);
  offset++;

  if (has_fcs){
    len_fcs = 4;
  }

  sta_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_vht_ndp_annc_sta_list, &pi, "STA list");
  saved_offset = offset;

  while (tvb_reported_length_remaining(tvb, offset) > len_fcs) {
    sta_info_item = proto_tree_add_subtree_format(sta_list, tvb, offset, 2,
                        ett_vht_ndp_annc_sta_info_tree, NULL, "STA %d",
                        sta_index++);
    proto_tree_add_item(sta_info_item, hf_ieee80211_vht_ndp_annc_sta_info_aid12,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(sta_info_item, hf_ieee80211_vht_ndp_annc_sta_info_feedback_type,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);

    sta_info = tvb_get_letohs(tvb, offset);

    if (sta_info & 0x1000)
       proto_tree_add_uint(sta_info_item,
                           hf_ieee80211_vht_ndp_annc_sta_info_nc_index,
                           tvb, offset, 2, sta_info);
    else
       proto_tree_add_item(sta_info_item, hf_ieee80211_vht_ndp_annc_sta_info_reserved,
                           tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }

  proto_item_set_len(pi, offset - saved_offset);
  return offset;
}

static const int *he_ndp_headers[] = {
  &hf_he_ndp_annc_reserved,
  &hf_he_ndp_annc_he_subfield,
  &hf_he_ndp_sounding_dialog_token_number,
  NULL
};

static const int *he_ndp_sta_headers[] = {
  &hf_he_ndp_annc_aid11,
  &hf_he_ndp_annc_ru_start,
  &hf_he_ndp_annc_ru_end,
  &hf_he_ndp_annc_feedback_type_and_ng,
  &hf_he_ndp_annc_disambiguation,
  &hf_he_ndp_annc_codebook_size,
  &hf_he_ndp_annc_nc,
  NULL
};

static int
dissect_ieee80211_he_ndp_annc(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, gboolean has_fcs)
{
  guint8           len_fcs = 0;
  proto_tree      *sta_list;
  proto_item      *pi;
  int              saved_offset;
  int              sta_index = 0;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_he_ndp_annc_token, ett_he_ndp_annc,
                        he_ndp_headers, ENC_NA, BMT_NO_APPEND);
  offset++;

  if (has_fcs){
    len_fcs = 4;
  }

  saved_offset = offset;
  sta_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_he_ndp_annc_sta_list, &pi, "STA list");

  while (tvb_reported_length_remaining(tvb, offset) > len_fcs) {
    proto_tree *sta_item;

    sta_item = proto_tree_add_subtree_format(sta_list, tvb, offset, 4,
                        ett_he_ndp_annc_sta_item, NULL, "STA %d", sta_index++);
    proto_tree_add_bitmask_with_flags(sta_item, tvb, offset,
                        hf_ieee80211_he_ndp_annc_sta, ett_he_ndp_annc_sta_info,
                        he_ndp_sta_headers, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

    offset += 4;
  }

  proto_item_set_len(pi, offset - saved_offset);
  return offset;
}

static int
dissect_ieee80211_vht_he_ndp_annc(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, int offset, gboolean has_fcs)
{
  const gchar *ether_name = tvb_get_ether_name(tvb, offset);
  proto_item      *hidden_item;
  guint8           dialog_token;

  proto_tree_add_item(tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_ta_resolved, tvb,
                        offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_item(tree, hf_ieee80211_addr, tvb, offset, 6,
                        ENC_NA);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  hidden_item = proto_tree_add_string(tree, hf_ieee80211_addr_resolved, tvb,
                        offset, 6, ether_name);
  PROTO_ITEM_SET_HIDDEN(hidden_item);
  offset += 6;

  dialog_token = tvb_get_guint8(tvb, offset);

  /*
   * Is it VHT or HE?
   */
  if (dialog_token & NDP_ANNC_VHT_HE) {
    return dissect_ieee80211_he_ndp_annc(tvb, pinfo, tree, offset, has_fcs);
  } else {
    return dissect_ieee80211_vht_ndp_annc(tvb, pinfo, tree, offset, has_fcs);
  }
}

static void
set_src_addr_cols(packet_info *pinfo, tvbuff_t *tvb, int offset, const char *type)
{
  address      ether_addr;

  set_address_tvb(&ether_addr, AT_ETHER, 6, tvb, offset);

  col_add_fstr(pinfo->cinfo, COL_RES_DL_SRC, "%s (%s)",
        address_with_resolution_to_str(wmem_packet_scope(), &ether_addr), type);
}

static void
set_dst_addr_cols(packet_info *pinfo, tvbuff_t *tvb, int offset, const char *type)
{
  address      ether_addr;

  set_address_tvb(&ether_addr, AT_ETHER, 6, tvb, offset);

  col_add_fstr(pinfo->cinfo, COL_RES_DL_DST, "%s (%s)",
        address_with_resolution_to_str(wmem_packet_scope(), &ether_addr), type);
}

static guint32
crc32_802_tvb_padded(tvbuff_t *tvb, guint hdr_len, guint hdr_size, guint len)
{
  guint32 c_crc;

  c_crc = crc32_ccitt_tvb(tvb, hdr_len);
  c_crc = crc32_ccitt_tvb_offset_seed(tvb, hdr_size, len, ~c_crc);

  return (c_crc);
}

typedef enum {
    ENCAP_802_2,
    ENCAP_IPX,
    ENCAP_ETHERNET
} encap_t;

/* ************************************************************************* */
/*                          Dissect 802.11 frame                             */
/* ************************************************************************* */
/*
 * The 802.11n specification makes some fairly significant changes to the
 * layout of the MAC header.  The first two bits of the MAC header are the
 * protocol version.  You'd think that the 802.11 committee would have
 * bumped the version to indicate a different MAC layout, but NOOOO -- we
 * have to go digging for bits in various locations instead.
 */
static int
dissect_ieee80211_common(tvbuff_t *tvb, packet_info *pinfo,
                          proto_tree *tree, guint32 option_flags,
                          struct ieee_802_11_phdr *phdr)
{
  guint16          fcf, flags, frame_type_subtype, ctrl_fcf, ctrl_type_subtype;
  guint16          cw_fcf;
  guint16          seq_control;
  guint32          seq_number, frag_number;
  gboolean         more_frags;
  proto_item      *ti          = NULL;
  proto_item      *cw_item     = NULL;
  proto_item      *hidden_item;
  proto_tree      *cw_tree     = NULL;
  guint16          hdr_len, ohdr_len;
  guint16          htc_len     = 0;
  gboolean         has_fcs;
  gint             len, reported_len, ivlen;
  gint             sta_addr_offset = 0;
  const gchar     *station_name;
  gboolean         is_amsdu    = 0;
  gboolean         save_fragmented;
  guint32          addr_type;
  guint8           octet1, octet2;
  char             out_buff[SHORT_STR];
  gint             is_iv_bad;
  guchar           iv_buff[4];
  const char      *addr1_str   = "RA";
  guint            offset;
  const gchar     *fts_str;
  gchar            flag_str[]  = "opmPRMFTC";
  gint             ii;
  guint16          qosoff      = 0;
  guint16          qos_control = 0;
  gint             meshctl_len = 0;
  guint8           mesh_flags;
  guint16          meshoff     = 0;
  static wlan_hdr_t whdrs[4];
  gboolean         retransmitted;
  gboolean         isDMG = (phdr->has_frequency ?
                                IS_80211AD(phdr->frequency) :
                                FALSE);

  encap_t     encap_type;
  proto_tree *hdr_tree = NULL;
  tvbuff_t   *next_tvb = NULL;
  wlan_hdr_t *whdr;

#define PROTECTION_ALG_WEP  DOT11DECRYPT_KEY_TYPE_WEP
#define PROTECTION_ALG_TKIP  DOT11DECRYPT_KEY_TYPE_TKIP
#define PROTECTION_ALG_CCMP  DOT11DECRYPT_KEY_TYPE_CCMP
#define PROTECTION_ALG_RSNA  PROTECTION_ALG_CCMP | PROTECTION_ALG_TKIP
#define IS_TKIP(tvb, hdr_len)  (tvb_get_guint8(tvb, hdr_len + 1) == \
  ((tvb_get_guint8(tvb, hdr_len) | 0x20) & 0x7f))
#define IS_CCMP(tvb, hdr_len)  (tvb_get_guint8(tvb, hdr_len + 2) == 0)
  guint8 algorithm=G_MAXUINT8;
  guint32 sec_header=0;
  guint32 sec_trailer=0;

  DOT11DECRYPT_KEY_ITEM  used_key;

  p_add_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_DMG_KEY, GINT_TO_POINTER(isDMG));

  whdr= &whdrs[0];

  col_set_str(pinfo->cinfo, COL_PROTOCOL, "802.11");
  col_clear(pinfo->cinfo, COL_INFO);

  fcf = FETCH_FCF(0);
  frame_type_subtype = COMPOSE_FRAME_TYPE(fcf);
  whdr->type = frame_type_subtype;
  if (frame_type_subtype == CTRL_CONTROL_WRAPPER)
    ctrl_fcf = FETCH_FCF(10);
  else
    ctrl_fcf = 0;

  fts_str = val_to_str_ext_const(frame_type_subtype, &frame_type_subtype_vals_ext,
                                 "Unrecognized (Reserved frame)");
  col_set_str(pinfo->cinfo, COL_INFO, fts_str);


# define FROM_TO_DS 3
  flags = FCF_FLAGS(fcf);
  more_frags = HAVE_FRAGMENTS(flags);

  for (ii = 0; ii < 8; ii++) {
    if (! (flags & 0x80 >> ii)) {
      flag_str[ii] = '.';
    }
  }

  switch (FCF_FRAME_TYPE (fcf)) {

  case MGT_FRAME:
    hdr_len = MGT_FRAME_HDR_LEN;
    if (HAS_HT_CONTROL(FCF_FLAGS(fcf))) {
      /*
       * Management frames with the Order bit set have an HT Control field;
       * see 8.2.4.1.10 "Order field".  If they're not HT frames, they should
       * never have the Order bit set.
       */
      hdr_len += 4;
      htc_len = 4;
    }
    break;

  case CONTROL_FRAME:
    if (frame_type_subtype == CTRL_CONTROL_WRAPPER) {
      hdr_len = 6;
      cw_fcf = ctrl_fcf;
    } else {
      hdr_len = 0;
      cw_fcf = fcf;
    }
    switch (COMPOSE_FRAME_TYPE (cw_fcf)) {

    case CTRL_TRIGGER:
      /*
       * This is a variable length frame ... we set the real length below
       * and since the common info is variable, just set the hdr len to
       * the fixed portion, 16. There can also be one or more user-info
       * sections, followed by padding.
       */
      hdr_len = 16;
      break;

    case CTRL_BEAMFORM_RPT_POLL:
      hdr_len += 17;
      break;

    case CTRL_VHT_NDP_ANNC:
      hdr_len += 17;
      /* TODO: for now we only consider a single STA, add support for more */
      hdr_len += 2;
      break;

    case CTRL_CTS:
    case CTRL_ACKNOWLEDGEMENT:
      hdr_len += 10;
      break;

    case CTRL_POLL:
      hdr_len += 18;
      break;

    case CTRL_SPR:
    case CTRL_GRANT:
    case CTRL_GRANT_ACK:
      hdr_len += 23;
      break;

    case CTRL_DMG_CTS:
      hdr_len += 16;
      break;

    case CTRL_DMG_DTS:
    case CTRL_SSW:
      hdr_len += 22;
      break;

    case CTRL_SSW_FEEDBACK:
    case CTRL_SSW_ACK:
      hdr_len += 24;
      break;

    case CTRL_RTS:
    case CTRL_PS_POLL:
    case CTRL_CFP_END:
    case CTRL_CFP_ENDACK:
    case CTRL_BLOCK_ACK_REQ:
    case CTRL_BLOCK_ACK:
      hdr_len += 16;
      break;

    default:
      hdr_len += 4;  /* XXX */
      break;
    }
    break;

  case DATA_FRAME:
    hdr_len = (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) ? DATA_LONG_HDR_LEN : DATA_SHORT_HDR_LEN;

    if ((option_flags & IEEE80211_COMMON_OPT_NORMAL_QOS) && DATA_FRAME_IS_QOS(frame_type_subtype)) {
      /* QoS frame */
      qosoff = hdr_len;
      hdr_len += 2; /* Include the QoS field in the header length */

      if (HAS_HT_CONTROL(FCF_FLAGS(fcf))) {
        /*
         * QoS data frames with the Order bit set have an HT Control field;
         * see 8.2.4.1.10 "Order field".  If they're not HT frames, they
         * should never have the Order bit set.
         */
        hdr_len += 4;
        htc_len = 4;
      }

      /*
       * Does it look as if we have a mesh header?
       * Look at the Mesh Control subfield of the QoS field and at the
       * purported mesh flag fields.
       */
      qos_control = tvb_get_letohs(tvb, qosoff);
      if (tvb_bytes_exist(tvb, hdr_len, 1)) {
        meshoff = hdr_len;
        mesh_flags = tvb_get_guint8(tvb, meshoff);
        if (has_mesh_control(fcf, qos_control, mesh_flags)) {
          /* Yes, add the length of that in as well. */
          meshctl_len = find_mesh_control_length(mesh_flags);
          hdr_len += meshctl_len;
        }
      }
    } else if ((option_flags & IEEE80211_COMMON_OPT_NORMAL_QOS) && !DATA_FRAME_IS_QOS(frame_type_subtype)) {

      if (HAS_HT_CONTROL(FCF_FLAGS(fcf))) {
        /*
         * QoS data frames with the Order bit set have an HT Control field;
         * see 8.2.4.1.10 "Order field".  If they're not HT frames, they
         * should never have the Order bit set.
         */
        hdr_len += 4;
        htc_len = 4;
      }

      /*
       * For locally originated mesh frames, the QoS header may be added by the
       * hardware, and no present in wireshark captures.  This poses a problem
       * as the QoS header indicates the presence of the mesh control header.
       *
       * Instead of QoS, we use a few heuristics to determine the presence of
       * the mesh control header, which is tricky because it can have a
       * variable length.
       *
       * Assume minimal length, and then correct if wrong.
       */
      meshctl_len = find_mesh_control_length(0);
      if (tvb_bytes_exist(tvb, hdr_len, meshctl_len)) {
        meshoff = hdr_len;
        mesh_flags = tvb_get_guint8(tvb, meshoff);
        /* now find correct length */
        meshctl_len = find_mesh_control_length(mesh_flags);
        /* ... and try to read two bytes of next header */
        if (tvb_bytes_exist(tvb, hdr_len, meshctl_len + 2)) {
          guint16 next_header = tvb_get_letohs(tvb, meshoff + meshctl_len);
          if (has_mesh_control_local(fcf, mesh_flags, next_header)) {
            /* Yes, add the length of that in as well. */
            hdr_len += meshctl_len;
            break;
          }
        }
      }
    }
    /* failed to find a mesh header */
    meshctl_len = 0;
    break;

  case EXTENSION_FRAME:
    hdr_len = 10;
    break;

  default:
    hdr_len = 4;  /* XXX */
    break;
  }

  /*
   * Some portions of this code calculate offsets relative to the end of the
   * header.  But when the header has been padded to align the data this must
   * be done relative to true header size, not the padded/aligned value.  To
   * simplify this work we stash the original header size in ohdr_len instead
   * of recalculating it every time we need it.
   */
  ohdr_len = hdr_len;
  if (phdr->datapad) {
    /*
     * Add in Atheros padding between the 802.11 header and body.
     *
     * XXX - would the mesh header be part of the header or the body
     * from the point of view of the Atheros adapters that insert
     * the padding, assuming they even recognize a mesh header?
     */
    hdr_len = roundup2(hdr_len, 4);
  }

  /* Add the FC and duration/id to the current tree */
  ti = proto_tree_add_protocol_format (tree, proto_wlan, tvb, 0, hdr_len,
                                           "IEEE 802.11 %s", fts_str);
  hdr_tree = proto_item_add_subtree(ti, ett_80211);

  dissect_frame_control(hdr_tree, tvb, option_flags, 0, pinfo);
  dissect_durid(hdr_tree, tvb, frame_type_subtype, 2);

  switch (phdr->fcs_len)
    {
      case 0: /* Definitely has no FCS */
        has_fcs = FALSE;
        break;

      case 4: /* Definitely has an FCS */
        has_fcs = TRUE;
        break;

      case -2: /* Data frames have no FCS, other frames may have an FCS */
               /* XXX: -2 currently used only in wiretap/netmon.c       */
        if (FCF_FRAME_TYPE (fcf) == DATA_FRAME)
          has_fcs = FALSE;
        else /* Management, Control, Extension */
          has_fcs = wlan_check_fcs;
        break;

      default: /* Don't know - use "wlan_check_fcs" */
        has_fcs = wlan_check_fcs;
        break;
    }

  /*
   * Decode the part of the frame header that isn't the same for all
   * frame types.
   */
  seq_control = 0;
  frag_number = 0;
  seq_number = 0;

  /* all frames have address 1 = RA */
  if (tree) {
    const gchar *ether_name = tvb_get_ether_name(tvb, 4);
    proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ra, tvb, 4, 6, ENC_NA);
    hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ra_resolved, tvb, 4, 6, ether_name);
    PROTO_ITEM_SET_HIDDEN(hidden_item);
    hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, 4, 6, ENC_NA);
    PROTO_ITEM_SET_HIDDEN(hidden_item);
    hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, 4, 6, ether_name);
    PROTO_ITEM_SET_HIDDEN(hidden_item);
  }

  switch (FCF_FRAME_TYPE (fcf))
  {

    case MGT_FRAME:
      /*
       * All management frame types have the same header.
       */
      set_address_tvb(&pinfo->dl_src, wlan_address_type, 6, tvb, 10);
      copy_address_shallow(&pinfo->src, &pinfo->dl_src);
      set_address_tvb(&pinfo->dl_dst, wlan_address_type, 6, tvb, 4);
      copy_address_shallow(&pinfo->dst, &pinfo->dl_dst);

      /* for tap */
      set_address_tvb(&whdr->bssid, wlan_bssid_address_type, 6, tvb, 16);
      copy_address_shallow(&whdr->src, &pinfo->dl_src);
      copy_address_shallow(&whdr->dst, &pinfo->dl_dst);

      seq_control = tvb_get_letohs(tvb, 22);
      frag_number = SEQCTL_FRAGMENT_NUMBER(seq_control);
      seq_number = SEQCTL_SEQUENCE_NUMBER(seq_control);

      col_append_fstr(pinfo->cinfo, COL_INFO,
            ", SN=%d", seq_number);

      col_append_fstr(pinfo->cinfo, COL_INFO,
            ", FN=%d", frag_number);

      if (tree)
      {
        const gchar *ra_da_name, *ta_sa_name, *bssid_name;

        ra_da_name = tvb_get_ether_name(tvb, 4);
        proto_tree_add_item(hdr_tree, hf_ieee80211_addr_da, tvb, 4, 6, ENC_NA);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_da_resolved, tvb, 4, 6, ra_da_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, 10, 6, ENC_NA);
        ta_sa_name = tvb_get_ether_name(tvb, 10);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, 10, 6, ta_sa_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        proto_tree_add_item(hdr_tree, hf_ieee80211_addr_sa, tvb, 10, 6, ENC_NA);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_sa_resolved, tvb, 10, 6, ta_sa_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, 16, 6, ENC_NA);
        bssid_name = tvb_get_ether_name(tvb, 16);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_bssid_resolved, tvb, 16, 6, bssid_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);

        /* FIXME: With mgmt frames FROM_TO_DS is always 0, perhaps compare address to bssid instead? */
        if ((flags & FROM_TO_DS) == FLAG_FROM_DS) { /* Receiver address */
          sta_addr_offset = 4;
        } else if ((flags & FROM_TO_DS) == FLAG_TO_DS) { /* Transmitter address */
          sta_addr_offset = 10;
        }
        if (sta_addr_offset > 0) {
          proto_tree_add_item(hdr_tree, hf_ieee80211_addr_staa, tvb, sta_addr_offset, 6, ENC_NA);
          station_name = tvb_get_ether_name(tvb, sta_addr_offset);
          hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_staa_resolved, tvb, sta_addr_offset, 6, station_name);
          PROTO_ITEM_SET_HIDDEN(hidden_item);
        }
        /* add items for wlan.addr filter */
        hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, 10, 6, ENC_NA);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, 10, 6, ta_sa_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, 16, 6, ENC_NA);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, 16, 6, bssid_name);
        PROTO_ITEM_SET_HIDDEN(hidden_item);
        proto_tree_add_item(hdr_tree, hf_ieee80211_frag_number, tvb, 22, 2, ENC_LITTLE_ENDIAN);
        proto_tree_add_item(hdr_tree, hf_ieee80211_seq_number, tvb, 22, 2, ENC_LITTLE_ENDIAN);
      }
      break;

    case CONTROL_FRAME:
    {
      /*
       * Control Wrapper frames insert themselves between address 1
       * and address 2 in a normal control frame.  Process address 1
       * first, then handle the rest of the frame in dissect_control.
       */
      if (frame_type_subtype == CTRL_CONTROL_WRAPPER) {
        offset = 10; /* FC + D/ID + Address 1 + CFC + HTC */
        ctrl_fcf = FETCH_FCF(10);
        ctrl_type_subtype = COMPOSE_FRAME_TYPE(ctrl_fcf);
      } else {
        offset = 10; /* FC + D/ID + Address 1 */
        ctrl_type_subtype = frame_type_subtype;
      }
      /* Added to disallow DMG STA to transfer packets of certain forbidden types. */
      switch (ctrl_type_subtype)
      {
        case CTRL_PS_POLL:
        case CTRL_CTS:
        case CTRL_CFP_ENDACK:
        if(isDMG == TRUE) {
          expert_add_info_format(pinfo, hdr_tree, &ei_ieee80211_dmg_subtype,
              "DMG STA shouldn't transmit control frame of type contention-free period end+ack");
        }
        break;
        default:
          break;
      }

      if (ctrl_type_subtype == CTRL_PS_POLL) {
        addr1_str = "BSSID";
        if (tree) {
          const gchar *ether_name = tvb_get_ether_name(tvb, 4);
          proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, 4, 6, ENC_NA);
          hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_bssid_resolved, tvb, 4, 6, ether_name);
          PROTO_ITEM_SET_HIDDEN(hidden_item);
        }
      }

      /* Add address 1 */
      set_dst_addr_cols(pinfo, tvb, 4, addr1_str);

      /*
       * Start shoving in other fields if needed.
       */
      if (frame_type_subtype == CTRL_CONTROL_WRAPPER) {
        /* if (tree) */
        {
          cw_tree = proto_tree_add_subtree(hdr_tree, tvb, offset, 2,
                      ett_cntrl_wrapper_fc, NULL, "Contained Frame Control");
          dissect_frame_control(cw_tree, tvb, 0, offset, pinfo);
          dissect_ht_control(pinfo, hdr_tree, tvb, offset + 2);
          offset += 6;
          hdr_tree = proto_tree_add_subtree(hdr_tree, tvb, offset, 2,
                      ett_cntrl_wrapper_fc, &cw_item, "Carried Frame");
          if (isDMG) {
            expert_add_info_format(pinfo, cw_item, &ei_ieee80211_dmg_subtype,
                                   "DMG STA shouldn't transmit Control Wrapper frame");
          }
        }
      }

      switch (ctrl_type_subtype)
      {
        case CTRL_PS_POLL:
        case CTRL_CFP_ENDACK:
        {
          set_src_addr_cols(pinfo, tvb, offset, "TA");
          /* if (tree) */
          {
            const gchar *ether_name = tvb_get_ether_name(tvb, offset);
            proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, offset, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            offset += 6;
          }
          break;
        }

        case CTRL_CFP_END:
        {
          if (isDMG)
            set_src_addr_cols(pinfo, tvb, offset, "TA");
          else
            set_src_addr_cols(pinfo, tvb, offset, "BSSID");
          /* if (tree) */
          {
            const gchar *ether_name = tvb_get_ether_name(tvb, offset);
            if (isDMG) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, offset, 6, ether_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            } else {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, offset, 6, ENC_NA);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_bssid_resolved, tvb, offset, 6, ether_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, offset, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            offset += 6;
          }
          break;
        }

        case CTRL_TRIGGER:
          set_src_addr_cols(pinfo, tvb, offset, "TA");
          /*
           * The len returned will be adjusted to include any padding required
           */
          hdr_len = dissect_ieee80211_he_trigger(tvb, pinfo, hdr_tree, offset);
          ohdr_len = hdr_len;
          has_fcs = FALSE;  /* Not sure at this stage */
          break;

        case CTRL_BEAMFORM_RPT_POLL:
        {
          set_src_addr_cols(pinfo, tvb, offset, "TA");
          /* if (tree) */
          {
            const gchar *ether_name = tvb_get_ether_name(tvb, offset);

            proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, offset, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            offset += 6;

            proto_tree_add_item(hdr_tree, hf_ieee80211_beamform_feedback_seg_retrans_bitmap, tvb, offset, 1, ENC_NA);
          }
        break;
        }

        case CTRL_VHT_NDP_ANNC:
          set_src_addr_cols(pinfo, tvb, offset, "TA");

          dissect_ieee80211_vht_he_ndp_annc(tvb, pinfo, hdr_tree, offset, has_fcs);
          break;

        case CTRL_GRANT_ACK:
        case CTRL_SSW:
        case CTRL_SSW_FEEDBACK:
        case CTRL_SSW_ACK:
        case CTRL_DMG_CTS:
        case CTRL_GRANT:
        case CTRL_SPR:
        case CTRL_POLL:
        case CTRL_RTS:
        {
          set_src_addr_cols(pinfo, tvb, offset, "TA");
          /* if (tree) */
          {
            const gchar *ether_name = tvb_get_ether_name(tvb, offset);
            proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, offset, 6, ENC_NA);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, offset, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, offset, 6, ether_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            offset += 6;
          }
          break;
        }

        case CTRL_CONTROL_WRAPPER:
          /* XXX - We shouldn't see this.  Should we throw an error? */
          break;

        case CTRL_BLOCK_ACK_REQ:
          set_src_addr_cols(pinfo, tvb, offset, "TA");

          dissect_ieee80211_block_ack(tvb, pinfo, hdr_tree, offset, isDMG, TRUE, has_fcs);
          break;

        case CTRL_BLOCK_ACK:
          set_src_addr_cols(pinfo, tvb, offset, "TA");

          dissect_ieee80211_block_ack(tvb, pinfo, hdr_tree, offset, isDMG, FALSE, has_fcs);
          break;
      }
/*
 * 802.11ad : Used for extension types.
 */
      switch (ctrl_type_subtype) {
        case CTRL_POLL: {
                proto_tree_add_item(hdr_tree, hf_ieee80211_cf_response_offset,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);

                break;
        }
        case CTRL_GRANT:
        case CTRL_GRANT_ACK:
        case CTRL_SPR: {
          gboolean isGrant;
          if(ctrl_type_subtype != CTRL_GRANT_ACK) {
            offset += add_ff_dynamic_allocation(hdr_tree, tvb, pinfo, offset);
          } else { /* CTRL_GRANT_ACK have 5 octets that are reserved*/
            proto_tree_add_item(hdr_tree, hf_ieee80211_grant_ack_reserved, tvb, offset, 5, ENC_NA);
            offset += 5;
          }
          isGrant = ((ctrl_type_subtype==CTRL_GRANT)||(ctrl_type_subtype==CTRL_GRANT_ACK));
          add_ff_beamforming_ctrl(hdr_tree, tvb, pinfo, offset, isGrant);
          /* offset += 2; */
          break;
        }
        case CTRL_SSW: {
          guint32 sector_sweep;

          sector_sweep = tvb_get_letoh24(tvb, offset);
          offset += add_ff_sector_sweep(hdr_tree, tvb, pinfo, offset);
          /* if Sector Sweep Direction = Responder, use SW Feedback field format when not transmitted as part of an ISS */
          if(sector_sweep & 0x00001) {
            add_ff_sector_sweep_feedback_to_iss(hdr_tree, tvb, pinfo, offset);
          } else {
            add_ff_sector_sweep_feedback_from_iss(hdr_tree, tvb, pinfo, offset);
          }
          /* offset += 3; */
          break;
        }
        case CTRL_SSW_ACK:
        case CTRL_SSW_FEEDBACK: {
          offset += add_ff_sector_sweep_feedback_to_iss(hdr_tree, tvb, pinfo, offset);
          offset += add_ff_BRP_request(hdr_tree, tvb, pinfo, offset);
          add_ff_beamformed_link(hdr_tree, tvb, pinfo, offset);
          /* offset += 1; */
          break;
        }
        case CTRL_DMG_DTS: {
          proto_tree_add_item(hdr_tree, hf_ieee80211_addr_nav_sa, tvb, offset, 6, ENC_NA);
          offset += 6;
          proto_tree_add_item(hdr_tree, hf_ieee80211_addr_nav_da, tvb, offset, 6, ENC_NA);
          /* offset += 6; */
          break;
        }
        default:
                break;
      }
      break;
    }

    case DATA_FRAME:
    {
      guint32 src_offset, dst_offset, da_offset, sa_offset, ta_offset = 10, bssid_offset;
      addr_type = FCF_ADDR_SELECTOR(fcf);
      if ((option_flags & IEEE80211_COMMON_OPT_NORMAL_QOS) && DATA_FRAME_IS_QOS(frame_type_subtype)) {
        if (!DATA_FRAME_IS_NULL(frame_type_subtype)) {
          is_amsdu = QOS_AMSDU_PRESENT(qos_control);
        }
      }

      /* In order to show src/dst address we must always do the following */
      switch (addr_type)
      {

        case DATA_ADDR_T1:
          da_offset = 4;
          sa_offset = 10;
          bssid_offset = 16;
          dst_offset = da_offset;
          src_offset = sa_offset;
          break;

        case DATA_ADDR_T2:
          da_offset = 4;
          sa_offset = !is_amsdu ? 16 : 0;
          bssid_offset = 10;
          dst_offset = da_offset;
          src_offset = sa_offset;
          break;

        case DATA_ADDR_T3:
          da_offset = !is_amsdu ? 16 : 0;
          sa_offset = 10;
          bssid_offset = 4;
          dst_offset = da_offset;
          src_offset = sa_offset;
          break;

        case DATA_ADDR_T4:
          if (!is_amsdu) {
            da_offset = 16;
            sa_offset = 24;
            bssid_offset = 0;
            dst_offset = da_offset;
            src_offset = sa_offset;
          } else {
            da_offset = 0;
            sa_offset = 0;
            // The second BSSID (Addr4) is handled below.
            bssid_offset = 16;
            dst_offset = 4;     // RA
            src_offset = 10;    // TA
          }
          break;

        default:
          /* All four cases are covered above. */
          DISSECTOR_ASSERT_NOT_REACHED();
      }



      set_address_tvb(&pinfo->dl_src, wlan_address_type, 6, tvb, src_offset);
      copy_address_shallow(&pinfo->src, &pinfo->dl_src);
      set_address_tvb(&pinfo->dl_dst, wlan_address_type, 6, tvb, dst_offset);
      copy_address_shallow(&pinfo->dst, &pinfo->dl_dst);

      /* for tap */
      if (bssid_offset) {
        set_address_tvb(&whdr->bssid, wlan_bssid_address_type, 6, tvb, bssid_offset);
      }

      copy_address_shallow(&whdr->src, &pinfo->dl_src);
      copy_address_shallow(&whdr->dst, &pinfo->dl_dst);

      seq_control = tvb_get_letohs(tvb, 22);
      frag_number = SEQCTL_FRAGMENT_NUMBER(seq_control);
      seq_number = SEQCTL_SEQUENCE_NUMBER(seq_control);

      col_append_fstr(pinfo->cinfo, COL_INFO,
            ", SN=%d, FN=%d", seq_number, frag_number);

      /* Now if we have a tree we start adding stuff */
      if (tree)
      {
        const gchar *ta_name, *sa_name = NULL, *da_name = NULL, *bssid_name = NULL, *bssid2_name = NULL;

        switch (addr_type)
        {
          case DATA_ADDR_T1:
          case DATA_ADDR_T2:
          case DATA_ADDR_T3:
          case DATA_ADDR_T4:
            proto_tree_add_item(hdr_tree, hf_ieee80211_addr_ta, tvb, ta_offset, 6, ENC_NA);
            ta_name = tvb_get_ether_name(tvb, ta_offset);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_ta_resolved, tvb, ta_offset, 6, ta_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            /* TA is always in the wlan.addr filter */
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, ta_offset, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
            hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, ta_offset, 6, ta_name);
            PROTO_ITEM_SET_HIDDEN(hidden_item);

            if (da_offset) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_da, tvb, da_offset, 6, ENC_NA);
              da_name = tvb_get_ether_name(tvb, da_offset);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_da_resolved, tvb, da_offset, 6, da_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }

            if (sa_offset) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_sa, tvb, sa_offset, 6, ENC_NA);
              sa_name = tvb_get_ether_name(tvb, sa_offset);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_sa_resolved, tvb, sa_offset, 6, sa_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }

            if (bssid_offset) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, bssid_offset, 6, ENC_NA);
              bssid_name = tvb_get_ether_name(tvb, bssid_offset);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_bssid_resolved, tvb, bssid_offset, 6, bssid_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }

            if (addr_type == DATA_ADDR_T4 && is_amsdu) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, 24, 6, ENC_NA);
              bssid2_name = tvb_get_ether_name(tvb, 24);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_bssid_resolved, tvb, 24, 6, bssid2_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }

            if ((flags & FROM_TO_DS) == FLAG_FROM_DS) { /* Receiver address */
              sta_addr_offset = 4;
            } else if ((flags & FROM_TO_DS) == FLAG_TO_DS) { /* Transmitter address */
              sta_addr_offset = ta_offset;
            }
            if (sta_addr_offset > 0) {
              proto_tree_add_item(hdr_tree, hf_ieee80211_addr_staa, tvb, sta_addr_offset, 6, ENC_NA);
              station_name = tvb_get_ether_name(tvb, sta_addr_offset);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_staa_resolved, tvb, sta_addr_offset, 6, station_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            proto_tree_add_item(hdr_tree, hf_ieee80211_frag_number, tvb, 22, 2, ENC_LITTLE_ENDIAN);
            proto_tree_add_item(hdr_tree, hf_ieee80211_seq_number, tvb, 22, 2, ENC_LITTLE_ENDIAN);

            /* add 3rd and 4th address for wlan.addr filter */
            if (sa_offset >= 16) {
              hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, sa_offset, 6, ENC_NA);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, sa_offset, 6, sa_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            if (da_offset >= 16 && da_offset != sa_offset) {
              hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, da_offset, 6, ENC_NA);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, da_offset, 6, da_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            if (bssid_offset >= 16 && bssid_offset != sa_offset && bssid_offset != da_offset) {
              hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, bssid_offset, 6, ENC_NA);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, bssid_offset, 6, bssid_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            if (addr_type == DATA_ADDR_T4 && is_amsdu) {
              hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, 24, 6, ENC_NA);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
              hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_resolved, tvb, 24, 6, bssid2_name);
              PROTO_ITEM_SET_HIDDEN(hidden_item);
            }
            break;
        }

      }
      break;
      }
    case EXTENSION_FRAME: {
      switch (frame_type_subtype) {
        case EXTENSION_DMG_BEACON: {
          set_dst_addr_cols(pinfo, tvb, 4, "BSSID");
          if (tree) {
            proto_tree_add_item(hdr_tree, hf_ieee80211_addr_bssid, tvb, 4, 6, ENC_NA);
            hidden_item = proto_tree_add_item(hdr_tree, hf_ieee80211_addr, tvb, 4, 6, ENC_NA);
            PROTO_ITEM_SET_HIDDEN(hidden_item);
          }
          break;
        }
      }
    }
  }

  len = tvb_captured_length_remaining(tvb, hdr_len);
  reported_len = tvb_reported_length_remaining(tvb, hdr_len);

  if (has_fcs)
    {
      /*
       * Well, this packet should, in theory, have an FCS.
       * Do we have the entire packet, and does it have enough data for
       * the FCS?
       */
      if (reported_len < 4)
      {
        /*
         * The packet is claimed not to even have enough data for a 4-byte
         * FCS.
         * Pretend it doesn't have an FCS.
         */
        ;
      }
      else if (len < reported_len)
      {
        /*
         * The packet is claimed to have enough data for a 4-byte FCS, but
         * we didn't capture all of the packet.
         * Slice off the 4-byte FCS from the reported length, and trim the
         * captured length so it's no more than the reported length; that
         * will slice off what of the FCS, if any, is in the captured
         * length.
         */
        reported_len -= 4;
        if (len > reported_len)
            len = reported_len;
      }
      else
      {
        /*
         * We have the entire packet, and it includes a 4-byte FCS.
         * Slice it off, and put it into the tree.
         */
        len          -= 4;
        reported_len -= 4;
        if (wlan_check_checksum)
        {
          guint32 sent_fcs = tvb_get_letohl(tvb, hdr_len + len);
          guint32 fcs;

          if (phdr->datapad)
            fcs = crc32_802_tvb_padded(tvb, ohdr_len, hdr_len, len);
          else
            fcs = crc32_ccitt_tvb(tvb, hdr_len + len);
          if (fcs != sent_fcs) {
            flag_str[8] = '.';
          }

          proto_tree_add_checksum(hdr_tree, tvb, hdr_len + len, hf_ieee80211_fcs, hf_ieee80211_fcs_status, &ei_ieee80211_fcs, pinfo, fcs, ENC_LITTLE_ENDIAN, PROTO_CHECKSUM_VERIFY);
        } else {
          proto_tree_add_checksum(hdr_tree, tvb, hdr_len + len, hf_ieee80211_fcs, hf_ieee80211_fcs_status, &ei_ieee80211_fcs, pinfo, 0, ENC_LITTLE_ENDIAN, PROTO_CHECKSUM_NO_FLAGS);
        }
      }
    }
  else
    {
      flag_str[8] = '\0';
    }

  proto_item_append_text(ti, ", Flags: %s", flag_str);
  col_append_fstr(pinfo->cinfo, COL_INFO, ", Flags=%s", flag_str);


  /*
   * Only management and data frames have a body, so we don't have
   * anything more to do for other types of frames.
   */
  switch (FCF_FRAME_TYPE (fcf))
    {

    case MGT_FRAME:
      if (htc_len == 4) {
        dissect_ht_control(pinfo, hdr_tree, tvb, ohdr_len - 4);
      }
      break;

    case DATA_FRAME:
      if ((option_flags & IEEE80211_COMMON_OPT_NORMAL_QOS) && tree && DATA_FRAME_IS_QOS(frame_type_subtype))
      {
        proto_item *qos_fields, *qos_ti;
        proto_tree *qos_tree;

        guint16 qos_eosp;
        guint16 qos_field_content;

        qos_fields = proto_tree_add_item(hdr_tree, hf_ieee80211_qos, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
        qos_tree = proto_item_add_subtree(qos_fields, ett_qos_parameters);

        qos_eosp = QOS_EOSP(qos_control);
        qos_field_content = QOS_FIELD_CONTENT(qos_control);

        proto_tree_add_item(qos_tree, hf_ieee80211_qos_tid, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);

        qos_ti = proto_tree_add_item(qos_tree, hf_ieee80211_qos_priority, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
        PROTO_ITEM_SET_GENERATED(qos_ti);

        if (FLAGS_DS_STATUS(flags) == (FLAG_FROM_DS|FLAG_TO_DS)) {
          /* mesh frame */
          proto_tree_add_item(qos_tree, hf_ieee80211_qos_eosp, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
        } else {
          if (flags & FLAG_TO_DS) {
            proto_tree_add_item(qos_tree, hf_ieee80211_qos_bit4, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
          } else {
            proto_tree_add_item(qos_tree, hf_ieee80211_qos_eosp, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
          }
        }

        proto_tree_add_item(qos_tree, hf_ieee80211_qos_ack_policy, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);

        if (flags & FLAG_FROM_DS) {
          if (!DATA_FRAME_IS_NULL(frame_type_subtype)) {
            proto_tree_add_item(qos_tree, hf_ieee80211_qos_amsdu_present, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
            is_amsdu = QOS_AMSDU_PRESENT(qos_control);
          }
          if (DATA_FRAME_IS_CF_POLL(frame_type_subtype)) {
            /* txop limit */
              qos_ti = proto_tree_add_item(qos_tree, hf_ieee80211_qos_txop_limit, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
            if (qos_field_content == 0) {
              proto_item_append_text(qos_ti, " (transmit one frame immediately)");
            }
          } else {
            /* qap ps buffer state */
            proto_item *qos_ps_buf_state_fields;
            proto_tree *qos_ps_buf_state_tree;

            qos_ps_buf_state_fields = proto_tree_add_item(qos_tree, hf_ieee80211_qos_ps_buf_state, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
            qos_ps_buf_state_tree = proto_item_add_subtree(qos_ps_buf_state_fields, ett_qos_ps_buf_state);

            proto_tree_add_item(qos_ps_buf_state_tree, hf_ieee80211_qos_buf_state_indicated, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);

            if (QOS_PS_BUF_STATE_INDICATED(qos_field_content)) {
              proto_tree_add_item(qos_ps_buf_state_tree, hf_ieee80211_qos_highest_pri_buf_ac, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
              qos_ti = proto_tree_add_item(qos_ps_buf_state_tree, hf_ieee80211_qos_qap_buf_load, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
              switch (QOS_PS_QAP_BUF_LOAD(qos_field_content)) {

              case 0:
                proto_item_append_text(qos_ti, " (no buffered traffic)");
                break;

              default:
                proto_item_append_text(qos_ti, " (%d octets)", QOS_PS_QAP_BUF_LOAD(qos_field_content)*4096);
                break;

              case 15:
                proto_item_append_text(qos_ti, " (greater than 57344 octets)");
                break;
              }

            }
          }
        } else {
          if (!DATA_FRAME_IS_NULL(frame_type_subtype)) {
            proto_tree_add_item(qos_tree, hf_ieee80211_qos_amsdu_present, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
            is_amsdu = QOS_AMSDU_PRESENT(qos_control);
          }
          /*
           * Only QoS Data, Qos CF-ACK and NULL frames To-DS have a Queue Size
           * field.
           */
          if ((DATA_FRAME_IS_NULL(frame_type_subtype) ||
               (frame_type_subtype & 0x7) == 0 ||
               DATA_FRAME_IS_CF_ACK(frame_type_subtype))) {
            if (qos_eosp) {
              /* queue size */
              qos_ti = proto_tree_add_item(qos_tree, hf_ieee80211_qos_queue_size, tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
              switch (qos_field_content) {
              case 0:
                proto_item_append_text(qos_ti, " (no buffered traffic in the queue)");
                break;

              default:
                proto_item_append_text(qos_ti, " (%u bytes)", qos_field_content*256);
                break;

              case 254:
                proto_item_append_text(qos_ti, " (more than 64768 octets)");
                break;

              case 255:
                proto_item_append_text(qos_ti, " (unspecified or unknown)");
                break;
              }
            } else {
              /* txop duration requested */
              qos_ti = proto_tree_add_item(qos_tree, hf_ieee80211_qos_txop_dur_req,
                                   tvb, qosoff, 2, ENC_LITTLE_ENDIAN);
              if (qos_field_content == 0) {
                proto_item_append_text(qos_ti, " (no TXOP requested)");
              }
            }
          }
        }

        /* Do we have +HTC? */
        if (htc_len == 4) {
          dissect_ht_control(pinfo, hdr_tree, tvb, ohdr_len - 4);
        }

      } /* end of qos control field */

      if (meshctl_len != 0) {
        proto_item *msh_fields;
        proto_tree *msh_tree;

        msh_fields = proto_tree_add_item(hdr_tree, hf_ieee80211_mesh_control_field, tvb, meshoff, meshctl_len, ENC_NA);
        msh_tree = proto_item_add_subtree(msh_fields, ett_msh_control);
        add_ff_mesh_control(msh_tree, tvb, pinfo, meshoff);
      }

      /*
       * No-data frames don't have a body.
       */
      if (DATA_FRAME_IS_NULL(frame_type_subtype))
        goto end_of_wlan;

      if (!wlan_subdissector) {
        guint fnum = 0;

        /* key: bssid:src
         * data: last seq_control seen and frame number
         */
        retransmitted = FALSE;
        if (!pinfo->fd->visited) {
          retransmit_key key;
          retransmit_key *result;

          memcpy(key.bssid, whdr->bssid.data, 6);
          memcpy(key.src, whdr->src.data, 6);
          key.seq_control = 0;
          result = (retransmit_key *)g_hash_table_lookup(fc_analyse_retransmit_table, &key);
          if (result && (result->seq_control == seq_control)) {
            /* keep a pointer to the first seen frame, could be done with proto data? */
            fnum = result->fnum;
            g_hash_table_insert(fc_first_frame_table, GINT_TO_POINTER(pinfo->num),
                                GINT_TO_POINTER(fnum));
            retransmitted = TRUE;
          } else {
            /* first time or new seq*/
            if (!result) {
              result = wmem_new(wmem_file_scope(), retransmit_key);
              *result = key;
              g_hash_table_insert(fc_analyse_retransmit_table, result, result);
            }
            result->seq_control = seq_control;
            result->fnum =  pinfo->num;
          }
        }
        else if ((fnum = GPOINTER_TO_UINT(g_hash_table_lookup(fc_first_frame_table, GINT_TO_POINTER(pinfo->num))))) {
          retransmitted = TRUE;
        }

        if (retransmitted) {
          col_append_str(pinfo->cinfo, COL_INFO, " [retransmitted]");
          if (tree) {
            proto_item *item;

            item=proto_tree_add_none_format(hdr_tree, hf_ieee80211_fc_analysis_retransmission, tvb, 0, 0,
                                            "Retransmitted frame");
            PROTO_ITEM_SET_GENERATED(item);
            item=proto_tree_add_uint(hdr_tree, hf_ieee80211_fc_analysis_retransmission_frame, tvb, 0, 0, fnum);
            PROTO_ITEM_SET_GENERATED(item);
          }
          next_tvb = tvb_new_subset_length_caplen(tvb, hdr_len, len, reported_len);
          call_data_dissector(next_tvb, pinfo, tree);
          goto end_of_wlan;
        }
      }

      if (enable_decryption && !pinfo->fd->visited && (len == reported_len)) {
        /* The processing will take care of 4-way handshake sessions for WPA and WPA2 decryption */
        next_tvb = try_decrypt(tvb, pinfo, hdr_len, reported_len, TRUE,
                               &algorithm, &sec_header, &sec_trailer, &used_key);
      }
      break;

    case CONTROL_FRAME:
      goto end_of_wlan;

    case EXTENSION_FRAME:
      break;

    default:
      goto end_of_wlan;
    }

  if (IS_PROTECTED(FCF_FLAGS(fcf))
      && !phdr->decrypted
      && (wlan_ignore_prot != WLAN_IGNORE_PROT_WO_IV)) {
    /*
     * It's a WEP or WPA encrypted frame, and it hasn't already been
     * decrypted; dissect the protections parameters and decrypt the data,
     * if we have a matching key. Otherwise display it as data.
     */
    gboolean    can_decrypt = FALSE;
    proto_tree *wep_tree    = NULL;
    guint32     iv;
    guint8      key, keybyte;

    if (next_tvb) {
      /* Already decrypted when searching for keys above. No need to decrypt again */
    } else if (len == reported_len) {
      next_tvb = try_decrypt(tvb, pinfo, hdr_len, reported_len, FALSE,
                             &algorithm, &sec_header, &sec_trailer, &used_key);
    }

    keybyte = tvb_get_guint8(tvb, hdr_len + 3);
    key = KEY_OCTET_WEP_KEY(keybyte);
    if ((keybyte & KEY_EXTIV) && (len >= EXTIV_LEN)) {
      /* Extended IV; this frame is likely encrypted with TKIP or CCMP */
      if (tree) {
        if (algorithm==PROTECTION_ALG_TKIP)
          wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 8,
              ett_wep_parameters, NULL, "TKIP parameters");
        else if (algorithm==PROTECTION_ALG_CCMP)
          wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 8,
            ett_wep_parameters, NULL, "CCMP parameters");
        else {
          if (IS_TKIP(tvb, hdr_len)) {
            algorithm=PROTECTION_ALG_TKIP;
            wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 8,
                ett_wep_parameters, NULL, "TKIP parameters");
          } else if (IS_CCMP(tvb, hdr_len)) {
            algorithm=PROTECTION_ALG_CCMP;
            wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 8,
                ett_wep_parameters, NULL, "CCMP parameters");
          } else
            wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 8,
                ett_wep_parameters, NULL, "TKIP/CCMP parameters");
        }
        proto_item_set_len(ti, hdr_len + 8);

        if (algorithm==PROTECTION_ALG_TKIP) {
          g_snprintf(out_buff, SHORT_STR, "0x%08X%02X%02X",
              tvb_get_letohl(tvb, hdr_len + 4),
              tvb_get_guint8(tvb, hdr_len),
              tvb_get_guint8(tvb, hdr_len + 2));
          proto_tree_add_string(wep_tree, hf_ieee80211_tkip_extiv, tvb, hdr_len,
              EXTIV_LEN, out_buff);
        } else if (algorithm==PROTECTION_ALG_CCMP) {
          g_snprintf(out_buff, SHORT_STR, "0x%08X%02X%02X",
              tvb_get_letohl(tvb, hdr_len + 4),
              tvb_get_guint8(tvb, hdr_len + 1),
              tvb_get_guint8(tvb, hdr_len));
          proto_tree_add_string(wep_tree, hf_ieee80211_ccmp_extiv, tvb, hdr_len,
              EXTIV_LEN, out_buff);
        }

        proto_tree_add_uint(wep_tree, hf_ieee80211_wep_key, tvb, hdr_len + 3, 1, key);
      }

      /* Subtract out the length of the IV. */
      len          -= EXTIV_LEN;
      reported_len -= EXTIV_LEN;
      ivlen         = EXTIV_LEN;
      /* It is unknown whether this is TKIP or CCMP, so let's not even try to
       * parse TKIP Michael MIC+ICV or CCMP MIC. */

      /* checking for the trailer                            */
      if (next_tvb!=NULL) {
        if (reported_len < (gint) sec_trailer) {
          /* There is no space for a trailer, ignore it and don't decrypt  */
          ;
        } else if (len < reported_len) {
          /* There is space for a trailer, but we haven't capture all the  */
          /* packet. Slice off the trailer, but don't try to decrypt      */
          reported_len -= sec_trailer;
          if (len > reported_len)
            len = reported_len;
        } else {
          /* Ok, we have a trailer and the whole packet. Decrypt it!      */
          /* TODO: At the moment we won't add the trailer to the tree,    */
          /* so don't remove the trailer from the packet                  */
          len          -= sec_trailer;
          reported_len -= sec_trailer;
          can_decrypt   = TRUE;

          /* Add Key information to packet */
          bytes_to_hexstr(out_buff, used_key.KeyData.Wpa.Ptk+32, DOT11DECRYPT_TK_LEN); /* TK is stored in PTK at offset 32 bytes and 16 bytes long */
          out_buff[2*DOT11DECRYPT_TK_LEN] = '\0';

          if (key == 0) { /* encrypted with pairwise key */
            ti = proto_tree_add_string(wep_tree, hf_ieee80211_fc_analysis_tk, tvb, 0, 0, out_buff);
            PROTO_ITEM_SET_GENERATED(ti);

            /* Also add the PMK used to to decrypt the packet. (PMK==PSK) */
            bytes_to_hexstr(out_buff, used_key.KeyData.Wpa.Psk, DOT11DECRYPT_WPA_PSK_LEN); /* 32 bytes */
            out_buff[2*DOT11DECRYPT_WPA_PSK_LEN] = '\0';
            ti = proto_tree_add_string(wep_tree, hf_ieee80211_fc_analysis_pmk, tvb, 0, 0, out_buff);
            PROTO_ITEM_SET_GENERATED(ti);

          } else { /* Encrypted with Group Key */
            ti = proto_tree_add_string(wep_tree, hf_ieee80211_fc_analysis_gtk, tvb, 0, 0, out_buff); /* GTK is stored in PTK at offset 32 bytes and 16 bytes long */
            PROTO_ITEM_SET_GENERATED(ti);
          }
        }
      }
    } else {
      /* No Ext. IV - WEP packet */
      /*
       * XXX - pass the IV and key to "try_decrypt_wep()", and have it pass
       * them to "wep_decrypt()", rather than having "wep_decrypt()" extract
       * them itself.
       *
       * Also, just pass the data *following* the WEP parameters as the
       * buffer to decrypt.
       */
      iv = tvb_get_ntoh24(tvb, hdr_len);
      if (tree) {
        wep_tree = proto_tree_add_subtree(hdr_tree, tvb, hdr_len, 4,
            ett_wep_parameters, NULL, "WEP parameters");

        proto_tree_add_uint(wep_tree, hf_ieee80211_wep_iv, tvb, hdr_len, 3, iv);
        tvb_memcpy(tvb, iv_buff, hdr_len, 3);
        is_iv_bad = weak_iv(iv_buff);
        if (is_iv_bad != -1) {
          proto_tree_add_boolean_format (wep_tree, hf_ieee80211_wep_iv_weak,
              tvb, 0, 0, TRUE,
              "Weak IV for key byte %d",
              is_iv_bad);
        }
      }
      if (tree)
        proto_tree_add_uint(wep_tree, hf_ieee80211_wep_key, tvb, hdr_len + 3, 1, key);

      /* Subtract out the length of the IV. */
      len          -= 4;
      reported_len -= 4;
      ivlen         = 4;

      /* Even if the decryption was not successful, set the algorithm */
      algorithm=PROTECTION_ALG_WEP;

      /*
       * Well, this packet should, in theory, have an ICV.
       * Do we have the entire packet, and does it have enough data for
       * the ICV?
       */
      if (reported_len < 4) {
        /*
         * The packet is claimed not to even have enough data for a
         * 4-byte ICV.
         * Pretend it doesn't have an ICV.
         */
        ;
      } else if (len < reported_len) {
        /*
         * The packet is claimed to have enough data for a 4-byte ICV,
         * but we didn't capture all of the packet.
         * Slice off the 4-byte ICV from the reported length, and trim
         * the captured length so it's no more than the reported length;
         * that will slice off what of the ICV, if any, is in the
         * captured length.
         */
        reported_len -= 4;
        if (len > reported_len)
          len         = reported_len;
      } else {
        /*
         * We have the entire packet, and it includes a 4-byte ICV.
         * Slice it off, and put it into the tree.
         *
         * We only support decrypting if we have the the ICV.
         *
         * XXX - the ICV is encrypted; we're putting the encrypted
         * value, not the decrypted value, into the tree.
         */
        len          -= 4;
        reported_len -= 4;
        can_decrypt   = TRUE;
      }
    }

    if (algorithm == PROTECTION_ALG_WEP) {
      g_strlcpy(wlan_stats.protection, "WEP", MAX_PROTECT_LEN);
    } else if (algorithm == PROTECTION_ALG_TKIP) {
      g_strlcpy(wlan_stats.protection, "TKIP", MAX_PROTECT_LEN);
    } else if (algorithm == PROTECTION_ALG_CCMP) {
      g_strlcpy(wlan_stats.protection, "CCMP", MAX_PROTECT_LEN);
    } else {
      g_strlcpy(wlan_stats.protection, "Unknown", MAX_PROTECT_LEN);
    }

    /* protection header                                  */
    if (!can_decrypt || (next_tvb == NULL)) {
      /*
       * WEP decode impossible or failed, treat payload as raw data
       * and don't attempt fragment reassembly or further dissection.
       */
      next_tvb = tvb_new_subset_length_caplen(tvb, hdr_len + ivlen, len, reported_len);

      if (tree) {
        if (algorithm == PROTECTION_ALG_WEP) {
          if (can_decrypt)
            proto_tree_add_uint_format_value(wep_tree, hf_ieee80211_wep_icv, tvb,
                hdr_len + ivlen + len, 4,
                tvb_get_ntohl(tvb, hdr_len + ivlen + len),
                "0x%08x (not verified)",
                tvb_get_ntohl(tvb, hdr_len + ivlen + len));
        } else if (algorithm == PROTECTION_ALG_CCMP) {
        } else if (algorithm == PROTECTION_ALG_TKIP) {
        }
      }

      if ((!(option_flags & IEEE80211_COMMON_OPT_IS_CENTRINO)) && (wlan_ignore_prot == WLAN_IGNORE_PROT_NO)) {
        /* Some wireless drivers (such as Centrino) WEP payload already decrypted */
        call_data_dissector(next_tvb, pinfo, tree);
        goto end_of_wlan;
      }
    } else {
      if (algorithm == PROTECTION_ALG_WEP) {
        if (tree)
          proto_tree_add_uint_format_value(wep_tree, hf_ieee80211_wep_icv, tvb,
              hdr_len + ivlen + len, 4,
              tvb_get_ntohl(tvb, hdr_len + ivlen + len),
              "0x%08x (correct)",
              tvb_get_ntohl(tvb, hdr_len + ivlen + len));

        add_new_data_source(pinfo, next_tvb, "Decrypted WEP data");
      } else if (algorithm==PROTECTION_ALG_CCMP) {
        add_new_data_source(pinfo, next_tvb, "Decrypted CCMP data");
      } else if (algorithm==PROTECTION_ALG_TKIP) {
        add_new_data_source(pinfo, next_tvb, "Decrypted TKIP data");
      }
#undef IS_TKIP
#undef IS_CCMP
#undef PROTECTION_ALG_CCMP
#undef PROTECTION_ALG_TKIP
#undef PROTECTION_ALG_WEP
    }

    /*
     * WEP decryption successful!
     *
     * Use the tvbuff we got back from the decryption; the data starts at
     * the beginning.  The lengths are already correct for the decoded WEP
     * payload.
     */
    hdr_len = 0;

  } else {
    /*
     * Not a WEP-encrypted frame; just use the data from the tvbuff
     * handed to us.
     *
     * The payload starts at "hdr_len" (i.e., just past the 802.11
     * MAC header), the length of data in the tvbuff following the
     * 802.11 header is "len", and the length of data in the packet
     * following the 802.11 header is "reported_len".
     */
    next_tvb = tvb;
  }

  /*
   * Do defragmentation if "wlan_defragment" is true, and we have more
   * fragments or this isn't the first fragment.
   *
   * We have to do some special handling to catch frames that
   * have the "More Fragments" indicator not set but that
   * don't show up as reassembled and don't have any other
   * fragments present.  Some networking interfaces appear
   * to do reassembly even when you're capturing raw packets
   * *and* show the reassembled packet without the "More
   * Fragments" indicator set *but* with a non-zero fragment
   * number.
   *
   * "fragment_add_seq_802_11()" handles that; we want to call it
   * even if we have a short frame, so that it does those checks - if
   * the frame is short, it doesn't do reassembly on it.
   *
   * (This could get some false positives if we really *did* only
   * capture the last fragment of a fragmented packet, but that's
   * life.)
   */
  save_fragmented = pinfo->fragmented;
  if (wlan_defragment && (more_frags || (frag_number != 0))) {
    fragment_head *fd_head;

    /*
     * If we've already seen this frame, look it up in the
     * table of reassembled packets, otherwise add it to
     * whatever reassembly is in progress, if any, and see
     * if it's done.
     */
    if (reported_len < 0)
      THROW(ReportedBoundsError);
    fd_head = fragment_add_seq_802_11(&wlan_reassembly_table,
        next_tvb, hdr_len, pinfo, seq_number, NULL,
        frag_number,
        reported_len,
        more_frags);
    next_tvb = process_reassembled_data(tvb, hdr_len, pinfo,
        "Reassembled 802.11", fd_head,
        &frag_items, NULL, hdr_tree);
   /* If we got a next_tvb, but it has an FCS, strip the FCS */
   if (next_tvb && has_fcs)
     next_tvb = tvb_new_subset_length_caplen(next_tvb, 0, len, reported_len);
  } else {
    /*
     * If this is the first fragment, dissect its contents, otherwise
     * just show it as a fragment.
     */
    if (frag_number != 0) {
      /* Not the first fragment - don't dissect it. */
      next_tvb = NULL;
    } else {
      /* First fragment, or not fragmented.  Dissect what we have here. */

      /* Get a tvbuff for the payload. */
      next_tvb = tvb_new_subset_length_caplen(next_tvb, hdr_len, len, reported_len);

      /*
       * If this is the first fragment, but not the only fragment,
       * tell the next protocol that.
       */
      if (more_frags)
        pinfo->fragmented = TRUE;
      else
        pinfo->fragmented = FALSE;
    }
  }

  if (next_tvb == NULL) {
    /* Just show this as an incomplete fragment. */
    col_set_str(pinfo->cinfo, COL_INFO, "Fragmented IEEE 802.11 frame");
    next_tvb = tvb_new_subset_length_caplen(tvb, hdr_len, len, reported_len);
    call_data_dissector(next_tvb, pinfo, tree);
    pinfo->fragmented = save_fragmented;
    goto end_of_wlan;
  }

  switch (FCF_FRAME_TYPE (fcf))
    {

    case MGT_FRAME:
      dissect_ieee80211_mgt(fcf, next_tvb, pinfo, tree);
      break;

    case DATA_FRAME:
      if (is_amsdu && (tvb_reported_length_remaining(next_tvb, 0) > 4)) {
        proto_item   *parent_item;
        proto_tree   *mpdu_tree;
        guint32       msdu_offset = 0;
        guint         i           = 1;

        parent_item = proto_tree_add_protocol_format(tree, proto_aggregate, next_tvb, 0,
                                    tvb_reported_length_remaining(next_tvb, 0), "IEEE 802.11 Aggregate MSDU");
        mpdu_tree = proto_item_add_subtree(parent_item, ett_msdu_aggregation_parent_tree);

        do {
          tvbuff_t           *msdu_tvb;
          guint16             msdu_length;
          proto_tree         *subframe_tree;
          const gchar *resolve_name;

          /*
           * IEEE Std 802.11-2012 says, in section 8.3.2.2 "A-MSDU format":
           *
           *  The A-MSDU subframe header contains three fields: DA, SA, and
           *  Length. The order of these fields and the bits within these
           *  fields are the same as the IEEE 802.3 frame format.
           *
           * which means that the length field is big-endian, not
           * little-endian.
           */
          msdu_length = tvb_get_ntohs(next_tvb, msdu_offset+12);

          parent_item = proto_tree_add_item(mpdu_tree, hf_ieee80211_amsdu_subframe, next_tvb,
                            msdu_offset, roundup2(msdu_offset+14+msdu_length, 4), ENC_NA);
          proto_item_append_text(parent_item, " #%u", i);
          subframe_tree = proto_item_add_subtree(parent_item, ett_msdu_aggregation_subframe_tree);
          i += 1;

          proto_tree_add_item(subframe_tree, hf_ieee80211_addr_da, next_tvb, msdu_offset, 6, ENC_NA);
          resolve_name = tvb_get_ether_name(next_tvb, msdu_offset);
          hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_da_resolved, next_tvb, msdu_offset, 6,
            resolve_name);
          PROTO_ITEM_SET_HIDDEN(hidden_item);
          proto_tree_add_item(subframe_tree, hf_ieee80211_addr_sa, next_tvb, msdu_offset+6, 6, ENC_NA);
          resolve_name = tvb_get_ether_name(next_tvb, msdu_offset+6);
          hidden_item = proto_tree_add_string(hdr_tree, hf_ieee80211_addr_sa_resolved, next_tvb, msdu_offset+6, 6,
            resolve_name);
          PROTO_ITEM_SET_HIDDEN(hidden_item);
          proto_tree_add_item(subframe_tree, hf_ieee80211_amsdu_length, next_tvb, msdu_offset+12, 2, ENC_BIG_ENDIAN);

          msdu_offset += 14;
          msdu_tvb = tvb_new_subset_length(next_tvb, msdu_offset, msdu_length);
          call_dissector(llc_handle, msdu_tvb, pinfo, subframe_tree);
          msdu_offset = roundup2(msdu_offset+msdu_length, 4);
        } while (tvb_reported_length_remaining(next_tvb, msdu_offset) > 14);

        break;
      }
      /* I guess some bridges take Netware Ethernet_802_3 frames,
         which are 802.3 frames (with a length field rather than
         a type field, but with no 802.2 header in the payload),
         and just stick the payload into an 802.11 frame.  I've seen
         captures that show frames of that sort.

         We also handle some odd form of encapsulation in which a
         complete Ethernet frame is encapsulated within an 802.11
         data frame, with no 802.2 header.  This has been seen
         from some hardware.

         On top of that, at least at some point it appeared that
         the OLPC XO sent out frames with two bytes of 0 between
         the "end" of the 802.11 header and the beginning of
         the payload. Something similar has also been observed
         with Atheros chipsets. There the sequence control field
         seems repeated.

         So, if the packet doesn't start with 0xaa 0xaa:

           we first use the same scheme that linux-wlan-ng does to detect
           those encapsulated Ethernet frames, namely looking to see whether
           the frame either starts with 6 octets that match the destination
           address from the 802.11 header or has 6 octets that match the
           source address from the 802.11 header following the first 6 octets,
           and, if so, treat it as an encapsulated Ethernet frame;

           otherwise, we use the same scheme that we use in the Ethernet
           dissector to recognize Netware 802.3 frames, namely checking
           whether the packet starts with 0xff 0xff and, if so, treat it
           as an encapsulated IPX frame, and then check whether the
           packet starts with 0x00 0x00 and, if so, treat it as an OLPC
           frame, or check the packet starts with the repetition of the
           sequence control field and, if so, treat it as an Atheros frame. */
      heur_dtbl_entry_t  *hdtbl_entry;
      if (dissector_try_heuristic(heur_subdissector_list, next_tvb, pinfo, tree, &hdtbl_entry, NULL)) {
        pinfo->fragmented = save_fragmented;
        goto end_of_wlan; /* heuristics dissector handled it. */
      }
      encap_type = ENCAP_802_2;
      if (tvb_bytes_exist(next_tvb, 0, 2)) {
        octet1 = tvb_get_guint8(next_tvb, 0);
        octet2 = tvb_get_guint8(next_tvb, 1);
        if ((octet1 != 0xaa) || (octet2 != 0xaa)) {
          if ((tvb_memeql(next_tvb, 6, (const guint8 *)pinfo->dl_src.data, 6) == 0) ||
              (tvb_memeql(next_tvb, 0, (const guint8 *)pinfo->dl_dst.data, 6) == 0))
            encap_type = ENCAP_ETHERNET;
          else if ((octet1 == 0xff) && (octet2 == 0xff))
            encap_type = ENCAP_IPX;
          else if (((octet1 == 0x00) && (octet2 == 0x00)) ||
                   (((octet2 << 8) | octet1) == seq_control)) {
            proto_tree_add_item(tree, hf_ieee80211_mysterious_olpc_stuff, next_tvb, 0, 2, ENC_NA);
            next_tvb = tvb_new_subset_remaining(next_tvb, 2);
          }
        }
      }

      switch (encap_type) {

      case ENCAP_802_2:
        call_dissector(llc_handle, next_tvb, pinfo, tree);
        break;

      case ENCAP_ETHERNET:
        call_dissector(eth_withoutfcs_handle, next_tvb, pinfo, tree);
        break;

      case ENCAP_IPX:
        call_dissector(ipx_handle, next_tvb, pinfo, tree);
        break;
      }
      break;

    case EXTENSION_FRAME:
    {
      dissect_ieee80211_extension(fcf, next_tvb, pinfo, tree);
      break;
    }
  }
  pinfo->fragmented = save_fragmented;

end_of_wlan:
  whdr->stats = wlan_stats;
  tap_queue_packet(wlan_tap, pinfo, whdr);
  memset(&wlan_stats, 0, sizeof wlan_stats);

  return tvb_captured_length(tvb);
}

/*
 * Dissect 802.11 with a variable-length link-layer header and with the FCS
 * presence or absence indicated by the pseudo-header, if there is one.
 */
static int
dissect_ieee80211(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
  struct ieee_802_11_phdr *phdr = (struct ieee_802_11_phdr *)data;
  struct ieee_802_11_phdr ourphdr;

  if (phdr == NULL) {
    /*
     * Fake a pseudo-header.
     * XXX - what are we supposed to do if the FCS length is unknown?
     */
    memset(&ourphdr, 0, sizeof(ourphdr));
    ourphdr.fcs_len = -1;
    ourphdr.decrypted = FALSE;
    ourphdr.datapad = FALSE;
    ourphdr.phy = PHDR_802_11_PHY_UNKNOWN;
    phdr = &ourphdr;
  }
  return dissect_ieee80211_common(tvb, pinfo, tree, IEEE80211_COMMON_OPT_NORMAL_QOS, phdr);
}

/*
 * Dissect 802.11 with a variable-length link-layer header and with an
 * FCS, but no pseudo-header.
 */
static int
dissect_ieee80211_withfcs(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  struct ieee_802_11_phdr phdr;

  /* Construct a pseudo-header to hand to the common code. */
  memset(&phdr, 0, sizeof(phdr));
  phdr.fcs_len = 4;
  phdr.decrypted = FALSE;
  phdr.datapad = FALSE;
  phdr.phy = PHDR_802_11_PHY_UNKNOWN;
  dissect_ieee80211_common(tvb, pinfo, tree, IEEE80211_COMMON_OPT_NORMAL_QOS, &phdr);
  return tvb_captured_length(tvb);
}

/*
 * Dissect 802.11 with a variable-length link-layer header and without an
 * FCS, but no pseudo-header.
 */
static int
dissect_ieee80211_withoutfcs(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  struct ieee_802_11_phdr phdr;

  /* Construct a pseudo-header to hand to the common code. */
  memset(&phdr, 0, sizeof(phdr));
  phdr.decrypted = FALSE;
  phdr.datapad = FALSE;
  phdr.phy = PHDR_802_11_PHY_UNKNOWN;
  dissect_ieee80211_common(tvb, pinfo, tree, IEEE80211_COMMON_OPT_NORMAL_QOS, &phdr);
  return tvb_captured_length(tvb);
}

/*
 * Dissect 802.11 from an Intel 2200BG adapter in a Centrino laptop
 * running Windows XP.
 *
 * From
 *
 *   https://www.wireshark.org/lists/ethereal-dev/200407/msg00184.html
 *
 * and
 *
 *   https://www.wireshark.org/lists/ethereal-dev/200407/msg00393.html:
 *
 *  I tried capturing from a Centrino laptop with the Intel 2200BG 802.11g
 *  chipset. I saw a lot of "Ethernet II" frames with 0x2452 as ethertype.
 *
 *    ...
 *
 *  This behaviour has been observed on Windows XP. In my opinion it is
 *  a "proprietary" behaviour of either the Centrino driver or the Centrino
 *  hardware. Currently I have no Linux distro installed on the machine to
 *  verify whether it is also the case.
 *
 *  These packets are seen only in a promiscuous capture:
 *    - Packets normally received by the Centrino computer have the normal
 *      structure (no 802.11/LLC header but directly IP header).
 *    - Packets that are supposed to be received by another computer have
 *      the 802.11/LLC headers. ... Also I noticed that when WEP is enabled,
 *      the 802.11 header has the flag "WEP" set to true, but the packet
 *      is already decrypted. I added a test in the code to accomodate this.
 *      For TKIP it seems to stay encrypted.
 */
static int
dissect_ieee80211_centrino(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  struct ieee_802_11_phdr phdr;

  /* Construct a pseudo-header to hand to the common code. */
  memset(&phdr, 0, sizeof(phdr));
  phdr.decrypted = FALSE;
  phdr.datapad = FALSE;
  phdr.phy = PHDR_802_11_PHY_UNKNOWN;
  dissect_ieee80211_common(tvb, pinfo, tree, IEEE80211_COMMON_OPT_IS_CENTRINO|IEEE80211_COMMON_OPT_NORMAL_QOS, &phdr);
  return tvb_captured_length(tvb);
}

/*
 * Dissect 802.11 with a variable-length link-layer header and a byte-swapped
 * control field and with no FCS (some hardware sends out LWAPP-encapsulated
 * 802.11 packets with the control field byte swapped).
 */
static int
dissect_ieee80211_bsfc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  struct ieee_802_11_phdr phdr;

  /* Construct a pseudo-header to hand to the common code. */
  memset(&phdr, 0, sizeof(phdr));
  phdr.decrypted = FALSE;
  phdr.datapad = FALSE;
  phdr.phy = PHDR_802_11_PHY_UNKNOWN;
  dissect_ieee80211_common(tvb, pinfo, tree, IEEE80211_COMMON_OPT_BROKEN_FC|IEEE80211_COMMON_OPT_NORMAL_QOS, &phdr);
  return tvb_captured_length(tvb);
}

/*
 * Dissect 802.11 with a variable-length link-layer header without qos elements
 * in data+qos frames and with no FCS (sent as WIDS frames by Cisco standalone
 * APs).
 */
static int
dissect_ieee80211_noqos(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  struct ieee_802_11_phdr phdr;

  /* Construct a pseudo-header to hand to the common code. */
  memset(&phdr, 0, sizeof(phdr));
  phdr.decrypted = FALSE;
  phdr.datapad = FALSE;
  phdr.phy = PHDR_802_11_PHY_UNKNOWN;
  dissect_ieee80211_common(tvb, pinfo, tree, 0, &phdr);
  return tvb_captured_length(tvb);
}


/* ------------- */
static guint
retransmit_hash(gconstpointer k)
{
  const retransmit_key *key = (const retransmit_key *)k;
  guint hash_val;
  int   i;

  hash_val = 0;
  for (i = 0; i < 6; i++)
    hash_val += key->bssid[i];

  for (i = 0; i < 6; i++)
    hash_val += key->src[i];

  return hash_val;
}

static gint
retransmit_equal(gconstpointer k1, gconstpointer k2)
{
  const retransmit_key *key1 = (const retransmit_key *)k1;
  const retransmit_key *key2 = (const retransmit_key *)k2;

  return ((!memcmp(key1->bssid, key2->bssid, 6) && !memcmp(key1->src, key2->src, 6)) ? TRUE:FALSE);
}

static guint
frame_hash(gconstpointer k)
{
  guint32 frame = GPOINTER_TO_UINT(k);

  return frame;
}

static gint
frame_equal(gconstpointer k1, gconstpointer k2)
{
  guint32 frame1 = GPOINTER_TO_UINT(k1);
  guint32 frame2 = GPOINTER_TO_UINT(k2);

  return frame1==frame2;
}

/*
 * EAPOL key description dissectors.
 */
#define KEY_INFO_KEYDES_VERSION_MASK        0x0007
#define KEY_INFO_KEY_TYPE_MASK              0x0008
#define KEY_INFO_KEY_INDEX_MASK             0x0030
#define KEY_INFO_INSTALL_MASK               0x0040
#define KEY_INFO_KEY_ACK_MASK               0x0080
#define KEY_INFO_KEY_MIC_MASK               0x0100
#define KEY_INFO_SECURE_MASK                0x0200
#define KEY_INFO_ERROR_MASK                 0x0400
#define KEY_INFO_REQUEST_MASK               0x0800
#define KEY_INFO_ENCRYPTED_KEY_DATA_MASK    0x1000
#define KEY_INFO_SMK_MESSAGE_MASK           0x2000

#define KEYDES_VER_TYPE1        0x01
#define KEYDES_VER_TYPE2        0x02
#define KEYDES_VER_TYPE3        0x03

static const value_string keydes_version_vals[] = {
  { KEYDES_VER_TYPE1,     "RC4 Cipher, HMAC-MD5 MIC" },
  { KEYDES_VER_TYPE2,     "AES Cipher, HMAC-SHA1 MIC" },
  { KEYDES_VER_TYPE3,     "AES Cipher, AES-128-CMAC MIC" },
  { 0, NULL }
};

static int proto_wlan_rsna_eapol = -1;

static int hf_wlan_rsna_eapol_wpa_keydes_msgnr = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_keydes_version = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_type = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_index = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_install = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_ack = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_mic = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_secure = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_error = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_request = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_encrypted_key_data = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_keyinfo_smk_message = -1;
static int hf_wlan_rsna_eapol_keydes_key_len = -1;
static int hf_wlan_rsna_eapol_keydes_replay_counter = -1;
static int hf_wlan_rsna_eapol_keydes_key_iv = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_nonce = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_rsc = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_id = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_mic = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_data_len = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_data = -1;
static int hf_wlan_rsna_eapol_wpa_keydes_padding = -1;

static gint ett_keyinfo = -1;
static gint ett_wlan_rsna_eapol_keydes_data = -1;

static const true_false_string keyinfo_key_type_tfs = { "Pairwise Key", "Group Key" };

static guint16 get_mic_len_owe(guint16 group) {
  switch(group) {
    // FFC, len(p) <= 2048
    case 1:
    case 2:
    case 5:
    case 14:
    case 22:
    case 23:
    case 24:
    // ECC, len(p) <= 256
    case 19:
    case 25:
    case 26:
    case 27:
    case 28:
    case 31:
      // HMAC-SHA-256
      return 16;

    // FFC, 2048 < len(p) <= 3072
    case 15:
    // ECC, 256 < len(p) <= 384
    case 20:
    case 29:
      // HMAC-SHA-384
      return 24;

    // FCC, 3072 < len(p)
    case 16:
    case 17:
    case 18:
    // ECC, 384 < len(p)
    case 21:
    case 30:
    case 32:
      // HMAC-SHA-512
      return 32;

    default:
      return 16;
  }
}

static guint16 get_mic_len(guint32 akm_suite) {
  switch(akm_suite) {
    case 0x000FAC0C:
    case 0x000FAC0D:
      // HMAC-SHA-384
      return 24;

    case 0x000FAC0E:
    case 0x000FAC0F:
    case 0x000FAC10:
    case 0x000FAC11:
      // AES-SIV-256 and AES-SIV-512
      return 0;

    default:
      // HMAC-SHA-1-128, AES-128-CMAC, HMAC-SHA-256
      return 16;
  }
}

static int
keydata_padding_len(tvbuff_t *tvb)
{
  int keydata_len = tvb_reported_length(tvb);
  int len_no_padding = keydata_len;
  const guint8 *keydata = tvb_get_ptr(tvb, 0, keydata_len);
  while (len_no_padding > 0 && (keydata[len_no_padding - 1] == 0x00)) {
    len_no_padding--;
  }
  if (len_no_padding > 0 && keydata[len_no_padding - 1] == 0xdd) {
    len_no_padding--;
    return keydata_len - len_no_padding;
  }
  return 0;
}

static int
dissect_wlan_rsna_eapol_wpa_or_rsn_key(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
  int         offset = 0;
  guint16     keyinfo;
  guint16     eapol_data_len;
  proto_tree *keydes_tree;
  proto_tree *ti = NULL;
  static const int * wlan_rsna_eapol_wpa_keydes_keyinfo[] = {
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_keydes_version,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_type,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_index,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_install,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_ack,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_key_mic,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_secure,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_error,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_request,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_encrypted_key_data,
    &hf_wlan_rsna_eapol_wpa_keydes_keyinfo_smk_message,
    NULL
  };
  guint16 eapol_data_offset = 76;  /* 92 - 16 */
  guint16 eapol_key_mic_len = 16; // Default MIC length

  conversation_t *conversation = find_conversation_pinfo(pinfo, 0);
  ieee80211_packet_data_t *packet_data = (ieee80211_packet_data_t*)p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, 0);

  if (wlan_key_mic_len_enable) {
    eapol_key_mic_len = wlan_key_mic_len;
  } else if (packet_data) {
    eapol_key_mic_len = packet_data->mic_len;
  } else if (conversation) {
    ieee80211_conversation_data_t *conversation_data = (ieee80211_conversation_data_t*)conversation_get_proto_data(conversation, proto_wlan);

    if (conversation_data) {
      if (conversation_data->last_akm_suite == 0x000FAC12) {
        /* For OWE the the length of MIC depends on the selected group */
        eapol_key_mic_len = get_mic_len_owe(conversation_data->owe_group);
      } else {
        eapol_key_mic_len = get_mic_len(conversation_data->last_akm_suite);
      }
    }
  }

  if (!packet_data) {
    packet_data = wmem_new(wmem_file_scope(), ieee80211_packet_data_t);
    packet_data->mic_len = eapol_key_mic_len;
    p_add_proto_data(wmem_file_scope(), pinfo, proto_wlan, 0, packet_data);
  }

  eapol_data_offset += eapol_key_mic_len;

  /*
   * RSNA key descriptors.
   */
  eapol_data_len = tvb_get_ntohs(tvb, offset+eapol_data_offset);
  keyinfo = tvb_get_ntohs(tvb, offset);
  if (keyinfo & KEY_INFO_REQUEST_MASK) {
    col_set_str(pinfo->cinfo, COL_INFO, "Key (Request)");
    if (keyinfo & KEY_INFO_ERROR_MASK)
      col_set_str(pinfo->cinfo, COL_INFO, "Key (Request, Error)");
  } else if (keyinfo & KEY_INFO_KEY_TYPE_MASK) {
    guint16 masked;
    /* Windows is setting the Secure Bit on message 2 when rekeying, so we'll ignore it */
    /* When an AEAD cipher is used no MIC is included, so we cannot rely on the MIC flag */
    masked = keyinfo &
      (KEY_INFO_INSTALL_MASK | KEY_INFO_KEY_ACK_MASK);

    switch (masked) {
    case KEY_INFO_KEY_ACK_MASK:
    {
      ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 1);

      col_set_str(pinfo->cinfo, COL_INFO, "Key (Message 1 of 4)");
      break;
    }

    case (KEY_INFO_INSTALL_MASK | KEY_INFO_KEY_ACK_MASK):
    {
      ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 3);

      col_set_str(pinfo->cinfo, COL_INFO, "Key (Message 3 of 4)");
      break;
    }

    default:
      /* We check the key length to differentiate between message 2 and 4 and just hope that
      there are no strange implementations with key data and non-zero key length in message 4.
      According to the IEEE specification, sections 11.6.6.3 and 11.6.6.5 we should
      use the Secure Bit and/or the Nonce, but there are implementations ignoring the spec.
      The Secure Bit is incorrectly set on rekeys for Windows clients for Message 2 and the Nonce is non-zero
      in Message 4 in Bug 11994 (Apple?) */
      /* When using AES-SIV without plaintext (i.e. only for integrity), the ciphertext has length 16 */
      if (((eapol_key_mic_len == 0) && (eapol_data_len > 16)) || ((eapol_key_mic_len > 0) && (eapol_data_len > 0))) {
        ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 2);

        col_set_str(pinfo->cinfo, COL_INFO, "Key (Message 2 of 4)");
      } else {
        ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 4);

        col_set_str(pinfo->cinfo, COL_INFO, "Key (Message 4 of 4)");
      }
      break;
    }
  } else {
    if (keyinfo & KEY_INFO_KEY_ACK_MASK) {
      ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 1);

      col_set_str(pinfo->cinfo, COL_INFO, "Key (Group Message 1 of 2)");
    } else {
      ti = proto_tree_add_uint(tree, hf_wlan_rsna_eapol_wpa_keydes_msgnr, tvb, offset, 0, 2);

      col_set_str(pinfo->cinfo, COL_INFO, "Key (Group Message 2 of 2)");
    }
  }

  PROTO_ITEM_SET_GENERATED(ti);

  guint16 keydes_version = tvb_get_ntohs(tvb, offset) & KEY_INFO_KEYDES_VERSION_MASK;
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_wlan_rsna_eapol_wpa_keydes_keyinfo,
                                    ett_keyinfo, wlan_rsna_eapol_wpa_keydes_keyinfo,
                                    ENC_BIG_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  proto_tree_add_item(tree, hf_wlan_rsna_eapol_keydes_key_len, tvb, offset,
                      2, ENC_BIG_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_wlan_rsna_eapol_keydes_replay_counter, tvb,
                      offset, 8, ENC_BIG_ENDIAN);
  offset += 8;
  proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_nonce, tvb, offset,
                      32, ENC_NA);
  offset += 32;
  proto_tree_add_item(tree, hf_wlan_rsna_eapol_keydes_key_iv, tvb,
                      offset, 16, ENC_NA);
  offset += 16;
  proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_rsc, tvb, offset,
                      8, ENC_NA);
  offset += 8;
  proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_id, tvb, offset, 8,
                      ENC_NA);
  offset += 8;

  if (eapol_key_mic_len > 0) {
    proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_mic, tvb, offset,
                        eapol_key_mic_len, ENC_NA);
    offset += eapol_key_mic_len;
  }

  proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_data_len, tvb,
                      offset, 2, ENC_BIG_ENDIAN);
  offset += 2;
  if (eapol_data_len != 0) {
    ti = proto_tree_add_item(tree, hf_wlan_rsna_eapol_wpa_keydes_data,
                             tvb, offset, eapol_data_len, ENC_NA);
    if ((keyinfo & KEY_INFO_ENCRYPTED_KEY_DATA_MASK) ||
        !(keyinfo & KEY_INFO_KEY_TYPE_MASK)) {
      /* RSN: EAPOL-Key Key Data is encrypted.
       * WPA: Group Keys use encrypted Key Data.
       * Decryption engine has already tried to decrypt this. If decrypted it's
       * stored in EAPOL_KEY proto data.
       * IEEE 802.11i-2004 8.5.2.
       */
      proto_eapol_keydata_t *eapol;
      eapol = (proto_eapol_keydata_t*)p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, EAPOL_KEY);

      if (eapol) {
        int keydata_len = eapol->keydata_len;
        tvbuff_t *next_tvb = tvb_new_child_real_data(tvb, eapol->keydata,
                                                     keydata_len, keydata_len);

        char out_buff[SHORT_STR];
        keydes_tree = proto_item_add_subtree(ti, ett_wlan_rsna_eapol_keydes_data);

        if (keydes_version == KEYDES_VER_TYPE1) {
          add_new_data_source(pinfo, next_tvb, "Decrypted RC4 keydata");
        } else if (keydes_version == KEYDES_VER_TYPE2) {
          add_new_data_source(pinfo, next_tvb, "Decrypted AES keydata");
          int padding_len = keydata_padding_len(next_tvb);
          ieee_80211_add_tagged_parameters(next_tvb, 0, pinfo, keydes_tree,
                                          keydata_len - padding_len,
                                          -1, NULL);
          if (padding_len) {
            proto_tree_add_item(keydes_tree, hf_wlan_rsna_eapol_wpa_keydes_padding,
                                next_tvb, keydata_len - padding_len,
                                padding_len, ENC_NA);
          }
        } else {
          /* TODO? */
        }
        /* Also add the PTK used to to decrypt and validate the keydata. */
        bytes_to_hexstr(out_buff, eapol->used_key.KeyData.Wpa.Ptk, 16); /* KCK is stored in PTK at offset 0 */
        out_buff[2*16] = '\0';
        ti = proto_tree_add_string(keydes_tree, hf_ieee80211_fc_analysis_kck, tvb, 0, 0, out_buff);
        PROTO_ITEM_SET_GENERATED(ti);

        bytes_to_hexstr(out_buff, eapol->used_key.KeyData.Wpa.Ptk+16, 16); /* KEK is stored in PTK at offset 16 */
        out_buff[2*16] = '\0';
        ti = proto_tree_add_string(keydes_tree, hf_ieee80211_fc_analysis_kek, tvb, 0, 0, out_buff);
        PROTO_ITEM_SET_GENERATED(ti);
      }
    } else {
      keydes_tree = proto_item_add_subtree(ti, ett_wlan_rsna_eapol_keydes_data);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, keydes_tree,
                                       tvb_reported_length_remaining(tvb, offset),
                                       -1, NULL);
    }
  }
  return tvb_captured_length(tvb);
}

/* It returns the algorithm used for decryption and the header and trailer lengths. */
static tvbuff_t *
try_decrypt(tvbuff_t *tvb, packet_info *pinfo, guint offset, guint len, gboolean scan_keys,
            guint8 *algorithm, guint32 *sec_header, guint32 *sec_trailer,
            PDOT11DECRYPT_KEY_ITEM used_key)
{
  const guint8      *enc_data;
  tvbuff_t          *decr_tvb = NULL;
  guint32            dec_caplen;
  guchar             dec_data[DOT11DECRYPT_MAX_CAPLEN];

  if (!enable_decryption)
    return NULL;

  /* get the entire packet                                  */
  enc_data = tvb_get_ptr(tvb, 0, len+offset);

  /*  process packet with Dot11Decrypt                              */
  gint ret = Dot11DecryptPacketProcess(&dot11decrypt_ctx, enc_data, offset, offset+len,
                                       dec_data, &dec_caplen, used_key, scan_keys);
  if (ret == DOT11DECRYPT_RET_SUCCESS) {
    guint8 *tmp;
    *algorithm=used_key->KeyType;
    switch (*algorithm) {
      case DOT11DECRYPT_KEY_TYPE_WEP:
        *sec_header=DOT11DECRYPT_WEP_HEADER;
        *sec_trailer=DOT11DECRYPT_WEP_TRAILER;
        break;
      case DOT11DECRYPT_KEY_TYPE_CCMP:
        *sec_header=DOT11DECRYPT_RSNA_HEADER;
        *sec_trailer=DOT11DECRYPT_CCMP_TRAILER;
        break;
      case DOT11DECRYPT_KEY_TYPE_TKIP:
        *sec_header=DOT11DECRYPT_RSNA_HEADER;
        *sec_trailer=DOT11DECRYPT_TKIP_TRAILER;
        break;
      default:
        return NULL;
    }
    /* allocate buffer for decrypted payload                      */
    tmp = (guint8 *)wmem_memdup(pinfo->pool, dec_data+offset, dec_caplen-offset);
    len = dec_caplen-offset;

    /* decrypt successful, let's set up a new data tvb.              */
    decr_tvb = tvb_new_child_real_data(tvb, tmp, len, len);
  } else if (ret == DOT11DECRYPT_RET_SUCCESS_HANDSHAKE && dec_caplen > 0) {
      proto_eapol_keydata_t *eapol;
      eapol = (proto_eapol_keydata_t *)wmem_alloc(wmem_file_scope(),
                                                  dec_caplen + PROTO_EAPOL_KEYDATA_OFFSET);
      memcpy(&eapol->used_key, used_key, sizeof(*used_key));
      memcpy(eapol->keydata, dec_data, dec_caplen);
      eapol->keydata_len = dec_caplen;

      /* Save decrypted eapol keydata for rsna dissector */
      p_add_proto_data(wmem_file_scope(), pinfo, proto_wlan, EAPOL_KEY, eapol);
  }

  return decr_tvb;
}

/* Collect our WEP and WPA keys */
static void
set_dot11decrypt_keys(void)
{
  guint                     i;
  DOT11DECRYPT_KEYS_COLLECTION  *keys = g_new(DOT11DECRYPT_KEYS_COLLECTION, 1);
  GByteArray                *bytes = NULL;

  keys->nKeys = 0;

  for (i = 0; (uat_wep_key_records != NULL) && (i < num_wepkeys_uat) && (i < MAX_ENCRYPTION_KEYS); i++)
  {
    decryption_key_t *dk;
    dk = parse_key_string(uat_wep_key_records[i].string, uat_wep_key_records[i].key);

    if (dk != NULL)
    {
      DOT11DECRYPT_KEY_ITEM          key;
      if (dk->type == DOT11DECRYPT_KEY_TYPE_WEP)
      {
        gboolean res;
        key.KeyType = DOT11DECRYPT_KEY_TYPE_WEP;

        bytes = g_byte_array_new();
        res = hex_str_to_bytes(dk->key->str, bytes, FALSE);

        if (dk->key->str && res && (bytes->len > 0) && (bytes->len <= DOT11DECRYPT_WEP_KEY_MAXLEN))
        {
          /*
           * WEP key is correct (well, the can be even or odd, so it is not
           * a real check, I think... is a check performed somewhere in the
           * Dot11Decrypt function???)
           */
          memcpy(key.KeyData.Wep.WepKey, bytes->data, bytes->len);
          key.KeyData.Wep.WepKeyLen = bytes->len;
          keys->Keys[keys->nKeys] = key;
          keys->nKeys += 1;
        }
      }
      else if (dk->type == DOT11DECRYPT_KEY_TYPE_WPA_PWD)
      {
        key.KeyType = DOT11DECRYPT_KEY_TYPE_WPA_PWD;

        /* XXX - This just lops the end if the key off if it's too long.
         *       Should we handle this more gracefully? */
        g_strlcpy(key.UserPwd.Passphrase, dk->key->str, DOT11DECRYPT_WPA_PASSPHRASE_MAX_LEN+1);

        key.UserPwd.SsidLen = 0;
        if ((dk->ssid != NULL) && (dk->ssid->len <= DOT11DECRYPT_WPA_SSID_MAX_LEN))
        {
          memcpy(key.UserPwd.Ssid, dk->ssid->data, dk->ssid->len);
          key.UserPwd.SsidLen = dk->ssid->len;
        }

        keys->Keys[keys->nKeys] = key;
        keys->nKeys += 1;
      }
      else if (dk->type == DOT11DECRYPT_KEY_TYPE_WPA_PSK)
      {
        key.KeyType = DOT11DECRYPT_KEY_TYPE_WPA_PSK;

        bytes = g_byte_array_new();
        hex_str_to_bytes(dk->key->str, bytes, FALSE);

        /* XXX - Pass the correct array of bytes... */
        if (bytes->len <= DOT11DECRYPT_WPA_PSK_LEN) {
          memcpy(key.KeyData.Wpa.Psk, bytes->data, bytes->len);

          keys->Keys[keys->nKeys] = key;
          keys->nKeys += 1;
        }
      }
      free_key_string(dk);
      if (bytes) {
        g_byte_array_free(bytes, TRUE);
        bytes = NULL;
      }
    }
  }

  /* Now set the keys */
  Dot11DecryptSetKeys(&dot11decrypt_ctx, keys->Keys, keys->nKeys);
  g_free(keys);
}

static void
init_wepkeys(void)
{

  /*
   * XXX - Dot11Decrypt - That God sends it to us beautiful (che dio ce la mandi bona)
   * The next lines will add a key to the Dot11Decrypt context. The keystring will be added
   * to the old WEP array too, but we don't care, because the packets will come here
   * already decrypted... One of these days we will fix this too
   */
  set_dot11decrypt_keys();
}

/*
 * This code had been taken from AirSnort crack.c function classify()
 * Permission granted by snax <at> shmoo dot com
 * weak_iv - determine which key byte an iv is useful in resolving
 * parm     - p, pointer to the first byte of an IV
 * return   -  n - this IV is weak for byte n of a WEP key
 *            -1 - this IV is not weak for any key bytes
 *
 * This function tests for IVs that are known to satisfy the criteria
 * for a weak IV as specified in FMS section 7.1
 *
 */
static int
weak_iv(guchar *iv)
{
  guchar sum, k;

  if ((iv[1] == 255) && (iv[0] > 2) && (iv[0] < 16)) {
    return iv[0] -3;
  }

  sum = iv[0] + iv[1];
  if (sum == 1) {
    if (iv[2] <= 0x0a) {
      return iv[2] +2;
    }
    else if (iv[2] == 0xff) {
      return 0;
    }
  }
  k = 0xfe - iv[2];
  if ((sum == k)  && ((iv[2] >= 0xf2) && (iv[2] <= 0xfe) && (iv[2] != 0xfd))) {
    return k;
  }
  return -1;
}

static void
wlan_retransmit_init(void)
{
  if (fc_analyse_retransmit_table) {
    g_hash_table_destroy(fc_analyse_retransmit_table);
    fc_analyse_retransmit_table = NULL;
  }

  if (fc_first_frame_table) {
    g_hash_table_destroy(fc_first_frame_table);
    fc_first_frame_table = NULL;
  }

  if (wlan_subdissector)
    return;

  fc_analyse_retransmit_table= g_hash_table_new(retransmit_hash, retransmit_equal);
  fc_first_frame_table = g_hash_table_new(frame_hash, frame_equal);

}

static int
dissect_data_encap(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int         offset = 0;
  guint8      type;
  int         tagged_parameter_tree_len;
  proto_tree *tagged_tree;

  type = tvb_get_guint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_data_encap_payload_type, tvb, offset,
                      1, ENC_LITTLE_ENDIAN);
  offset += 1;
  switch (type) {
  case 1:
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "RRB");
    /* TODO: IEEE 802.11r */
    break;
  case 2:
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "TDLS");
    col_clear(pinfo->cinfo, COL_INFO);
    offset += add_ff_action(tree, tvb, pinfo, offset);
    tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);
    if (tagged_parameter_tree_len > 0) {
      tagged_tree = get_tagged_parameter_tree(tree, tvb, offset,
                                              tagged_parameter_tree_len);
      ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree,
                                       tagged_parameter_tree_len, -1, NULL);
    }
    break;
  }
  return tvb_captured_length(tvb);
}

void
proto_register_ieee80211(void)
{

  static hf_register_info hf[] = {
    {&hf_ieee80211_fc_field,
     {"Frame Control Field", "wlan.fc",
      FT_UINT16, BASE_HEX, NULL, 0,
      "MAC Frame control", HFILL }},

    {&hf_ieee80211_fc_proto_version,
     {"Version", "wlan.fc.version",
      FT_UINT8, BASE_DEC, NULL, 0x03,
      "MAC Protocol version", HFILL }},  /* 0 */

    {&hf_ieee80211_fc_frame_type,
     {"Type", "wlan.fc.type",
      FT_UINT8, BASE_DEC, VALS(frame_type), 0x0C,
      "Frame type", HFILL }},

    {&hf_ieee80211_fc_frame_subtype,
     {"Subtype", "wlan.fc.subtype",
      FT_UINT8, BASE_DEC, NULL, 0xF0,
      "Frame subtype", HFILL }},  /* 2 */

    {&hf_ieee80211_fc_frame_type_subtype,
     {"Type/Subtype", "wlan.fc.type_subtype",
      FT_UINT16, BASE_HEX|BASE_EXT_STRING, &frame_type_subtype_vals_ext, 0x0,
      "Type and subtype combined (first byte: type, second byte: subtype)", HFILL }},

    {&hf_ieee80211_fc_frame_extension,
     {"Control Frame Extension", "wlan.fc.extension",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_flags,
     {"Flags", "wlan.flags",
      FT_UINT8, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_data_ds,
     {"DS status", "wlan.fc.ds",
      FT_UINT8, BASE_HEX, VALS(tofrom_ds), (FLAG_FROM_DS|FLAG_TO_DS),
      "Data-frame DS-traversal status", HFILL }},  /* 3 */

    {&hf_ieee80211_fc_to_ds,
     {"To DS", "wlan.fc.tods",
      FT_BOOLEAN, 8, TFS(&tods_flag), FLAG_TO_DS,
      "To DS flag", HFILL }},    /* 4 */

    {&hf_ieee80211_fc_from_ds,
     {"From DS", "wlan.fc.fromds",
      FT_BOOLEAN, 8, TFS(&fromds_flag), FLAG_FROM_DS,
      "From DS flag", HFILL }},    /* 5 */

    {&hf_ieee80211_fc_more_frag,
     {"More Fragments", "wlan.fc.frag",
      FT_BOOLEAN, 8, TFS(&more_fragments), FLAG_MORE_FRAGMENTS,
      "More Fragments flag", HFILL }},  /* 6 */

    {&hf_ieee80211_fc_retry,
     {"Retry", "wlan.fc.retry",
      FT_BOOLEAN, 8, TFS(&retry_flags), FLAG_RETRY,
      "Retransmission flag", HFILL }},

    {&hf_ieee80211_fc_analysis_retransmission,
     {"Retransmission", "wlan.analysis.retransmission",
      FT_NONE, BASE_NONE, NULL, 0x0,
      "This frame is a suspected wireless retransmission", HFILL }},

    {&hf_ieee80211_fc_analysis_retransmission_frame,
     {"Retransmission of frame", "wlan.analysis.retransmission_frame",
      FT_FRAMENUM, BASE_NONE, NULL, 0x0,
      "This is a retransmission of frame #", HFILL }},

    {&hf_ieee80211_fc_pwr_mgt,
     {"PWR MGT", "wlan.fc.pwrmgt",
      FT_BOOLEAN, 8, TFS(&pm_flags), FLAG_POWER_MGT,
      "Power management status", HFILL }},

    {&hf_ieee80211_fc_more_data,
     {"More Data", "wlan.fc.moredata",
      FT_BOOLEAN, 8, TFS(&md_flags), FLAG_MORE_DATA,
      "More data flag", HFILL }},

    {&hf_ieee80211_fc_protected,
     {"Protected flag", "wlan.fc.protected",
      FT_BOOLEAN, 8, TFS(&protected_flags), FLAG_PROTECTED,
      NULL, HFILL }},

    {&hf_ieee80211_fc_order,
     {"Order flag", "wlan.fc.order",
      FT_BOOLEAN, 8, TFS(&order_flags), FLAG_ORDER,
      "Strictly ordered flag", HFILL }},

    {&hf_ieee80211_assoc_id,
     {"Association ID", "wlan.aid",
      FT_UINT16, BASE_DEC, NULL, 0x3FFF,
      NULL, HFILL }},

    {&hf_ieee80211_did_duration,
     {"Duration", "wlan.duration",
      FT_UINT16, BASE_DEC, NULL, 0x7FFF,
      NULL, HFILL }},

    {&hf_ieee80211_addr_da,
     {"Destination address", "wlan.da",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Destination Hardware Address", HFILL }},

    {&hf_ieee80211_addr_da_resolved,
      {"Destination address (resolved)", "wlan.da_resolved", FT_STRING,
        BASE_NONE, NULL, 0x0,
        "Destination Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_addr_sa,
     {"Source address", "wlan.sa",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Source Hardware Address", HFILL }},

    {&hf_ieee80211_addr_sa_resolved,
      {"Source address (resolved)", "wlan.sa_resolved", FT_STRING,
       BASE_NONE, NULL, 0x0,
       "Source Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_addr,
      {"Hardware address", "wlan.addr",
       FT_ETHER, BASE_NONE, NULL, 0,
       "SA, DA, BSSID, RA or TA Hardware Address", HFILL }},

    {&hf_ieee80211_addr_resolved,
      { "Hardware address (resolved)", "wlan.addr_resolved", FT_STRING,
        BASE_NONE, NULL, 0x0,
        "SA, DA, BSSID, RA or TA Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_addr_ra,
     {"Receiver address", "wlan.ra",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Receiving Station Hardware Address", HFILL }},

    {&hf_ieee80211_addr_ra_resolved,
      {"Receiver address (resolved)", "wlan.ra_resolved", FT_STRING, BASE_NONE,
        NULL, 0x0, "Receiving Station Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_addr_ta,
     {"Transmitter address", "wlan.ta",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Transmitting Station Hardware Address", HFILL }},

    {&hf_ieee80211_addr_ta_resolved,
      {"Transmitter address (resolved)", "wlan.ta_resolved", FT_STRING,
        BASE_NONE, NULL, 0x0,
        "Transmitting Station Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_addr_bssid,
     {"BSS Id", "wlan.bssid",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Basic Service Set ID", HFILL }},

    {&hf_ieee80211_addr_bssid_resolved,
      {"BSS Id (resolved)", "wlan.bssid_resolved", FT_STRING, BASE_NONE, NULL,
        0x0, "Basic Service Set ID (resolved)", HFILL }},

    {&hf_ieee80211_addr_staa,
     {"STA address", "wlan.staa",
      FT_ETHER, BASE_NONE, NULL, 0,
      "Station Hardware Address", HFILL }},

    {&hf_ieee80211_addr_staa_resolved,
      {"STA address (resolved)", "wlan.staa_resolved", FT_STRING, BASE_NONE, NULL,
        0x0, "Station Hardware Address (resolved)", HFILL }},

    {&hf_ieee80211_frag_number,
     {"Fragment number", "wlan.frag",
      FT_UINT16, BASE_DEC, NULL, 0x000F,
      NULL, HFILL }},

    {&hf_ieee80211_seq_number,
     {"Sequence number", "wlan.seq",
      FT_UINT16, BASE_DEC, NULL, 0xFFF0,
      NULL, HFILL }},

    {&hf_ieee80211_mesh_control_field,
     {"Mesh Control Field", "wlan.mesh.control_field",
      FT_NONE, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_qos,
     {"Qos Control", "wlan.qos",
      FT_UINT16, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_qos_tid,
     {"TID", "wlan.qos.tid",
      FT_UINT16, BASE_DEC, NULL, 0x000F,
      NULL, HFILL }},

    {&hf_ieee80211_qos_priority,
     {"Priority", "wlan.qos.priority",
      FT_UINT16, BASE_DEC, VALS(ieee80211_qos_tags_acs), 0x0007,
      "802.1D Tag", HFILL }},

    {&hf_ieee80211_qos_eosp,
     {"EOSP", "wlan.qos.eosp",
      FT_BOOLEAN, 16, TFS(&eosp_flag), QOS_FLAG_EOSP,
      "EOSP Field", HFILL }},

    {&hf_ieee80211_qos_bit4,
     {"QoS bit 4", "wlan.qos.bit4",
      FT_BOOLEAN, 16, TFS(&bit4_flag), QOS_FLAG_EOSP,
      NULL, HFILL }},

    {&hf_ieee80211_qos_ack_policy,
     {"Ack Policy", "wlan.qos.ack",
      FT_UINT16, BASE_HEX,  VALS(ack_policy), 0x0060,
      NULL, HFILL }},

    {&hf_ieee80211_qos_amsdu_present,
     {"Payload Type", "wlan.qos.amsdupresent",
      FT_BOOLEAN, 16, TFS(&ieee80211_qos_amsdu_present_flag), 0x0080,
      NULL, HFILL }},

    {&hf_ieee80211_qos_txop_limit,
     {"TXOP Limit", "wlan.qos.txop_limit",
      FT_UINT16, BASE_DEC, NULL, 0xFF00,
      NULL, HFILL }},

    {&hf_ieee80211_qos_ps_buf_state,
     {"QAP PS Buffer State", "wlan.qos.ps_buf_state",
      FT_UINT16, BASE_HEX, NULL, 0xFF00,
      NULL, HFILL }},

    {&hf_ieee80211_qos_buf_state_indicated,
     {"Buffer State Indicated", "wlan.qos.buf_state_indicated",
      FT_BOOLEAN, 16, TFS(&tfs_yes_no), 0x0200,
      NULL, HFILL }},

    {&hf_ieee80211_qos_highest_pri_buf_ac,
     {"Highest-Priority Buffered AC", "wlan.qos.highest_pri_buf_ac",
       FT_UINT16, BASE_DEC, VALS(wme_acs), 0x0C00,
      NULL, HFILL }},

    {&hf_ieee80211_qos_qap_buf_load,
     {"QAP Buffered Load", "wlan.qos.qap_buf_load",
      FT_UINT16, BASE_DEC, NULL, 0xF000,
      NULL, HFILL }},

    {&hf_ieee80211_qos_txop_dur_req,
     {"TXOP Duration Requested", "wlan.qos.txop_dur_req",
      FT_UINT16, BASE_DEC, NULL, 0xFF00,
      NULL, HFILL }},

    {&hf_ieee80211_qos_queue_size,
     {"Queue Size", "wlan.qos.queue_size",
      FT_UINT16, BASE_DEC, NULL, 0xFF00,
      NULL, HFILL }},

    {&hf_ieee80211_fcs,
     {"Frame check sequence", "wlan.fcs",
      FT_UINT32, BASE_HEX, NULL, 0,
      "Frame Check Sequence (FCS)", HFILL }},

    {&hf_ieee80211_fcs_status,
     {"FCS Status", "wlan.fcs.status",
      FT_UINT8, BASE_NONE, VALS(proto_checksum_vals), 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_fragment_overlap,
      {"Fragment overlap", "wlan.fragment.overlap",
       FT_BOOLEAN, BASE_NONE, NULL, 0x0,
       "Fragment overlaps with other fragments", HFILL }},

    {&hf_ieee80211_fragment_overlap_conflict,
      {"Conflicting data in fragment overlap", "wlan.fragment.overlap.conflict",
       FT_BOOLEAN, BASE_NONE, NULL, 0x0,
       "Overlapping fragments contained conflicting data", HFILL }},

    {&hf_ieee80211_fragment_multiple_tails,
      {"Multiple tail fragments found", "wlan.fragment.multipletails",
       FT_BOOLEAN, BASE_NONE, NULL, 0x0,
       "Several tails were found when defragmenting the packet", HFILL }},

    {&hf_ieee80211_fragment_too_long_fragment,
      {"Fragment too long", "wlan.fragment.toolongfragment",
       FT_BOOLEAN, BASE_NONE, NULL, 0x0,
       "Fragment contained data past end of packet", HFILL }},

    {&hf_ieee80211_fragment_error,
      {"Defragmentation error", "wlan.fragment.error",
       FT_FRAMENUM, BASE_NONE, NULL, 0x0,
       "Defragmentation error due to illegal fragments", HFILL }},

    {&hf_ieee80211_fragment_count,
      {"Fragment count", "wlan.fragment.count",
       FT_UINT32, BASE_DEC, NULL, 0x0,
       NULL, HFILL }},

    {&hf_ieee80211_fragment,
      {"802.11 Fragment", "wlan.fragment",
       FT_FRAMENUM, BASE_NONE, NULL, 0x0,
       NULL, HFILL }},

    {&hf_ieee80211_fragments,
      {"802.11 Fragments", "wlan.fragments",
       FT_NONE, BASE_NONE, NULL, 0x0,
       NULL, HFILL }},

    {&hf_ieee80211_reassembled_in,
      {"Reassembled 802.11 in frame", "wlan.reassembled_in",
       FT_FRAMENUM, BASE_NONE, NULL, 0x0,
       "This 802.11 packet is reassembled in this frame", HFILL }},

    {&hf_ieee80211_reassembled_length,
      {"Reassembled 802.11 length", "wlan.reassembled.length",
       FT_UINT32, BASE_DEC, NULL, 0x0,
       "The total length of the reassembled payload", HFILL }},

    {&hf_ieee80211_wep_iv,
     {"Initialization Vector", "wlan.wep.iv",
      FT_UINT24, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_wep_iv_weak,
     {"Weak IV", "wlan.wep.weakiv",
      FT_BOOLEAN, BASE_NONE, NULL, 0x0,
       NULL, HFILL}},

    {&hf_ieee80211_tkip_extiv,
     {"TKIP Ext. Initialization Vector", "wlan.tkip.extiv",
      FT_STRING, BASE_NONE, NULL, 0,
      "TKIP Extended Initialization Vector", HFILL }},

    {&hf_ieee80211_ccmp_extiv,
     {"CCMP Ext. Initialization Vector", "wlan.ccmp.extiv",
      FT_STRING, BASE_NONE, NULL, 0,
      "CCMP Extended Initialization Vector", HFILL }},

    {&hf_ieee80211_wep_key,
     {"Key Index", "wlan.wep.key",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_wep_icv,
     {"WEP ICV", "wlan.wep.icv",
      FT_UINT32, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_analysis_pmk,
     {"PMK", "wlan.analysis.pmk",
      FT_STRING, BASE_NONE, NULL, 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_analysis_kck,
     {"KCK", "wlan.analysis.kck",
      FT_STRING, BASE_NONE, NULL, 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_analysis_kek,
     {"KEK", "wlan.analysis.kek",
      FT_STRING, BASE_NONE, NULL, 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_analysis_tk,
     {"TK", "wlan.analysis.tk",
      FT_STRING, BASE_NONE, NULL, 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_fc_analysis_gtk,
     {"GTK", "wlan.analysis.gtk",
      FT_STRING, BASE_NONE, NULL, 0x0,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_control,
     {"Block Ack Control", "wlan.ba.control",
      FT_UINT16, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_control_ack_policy,
     {"BA Ack Policy", "wlan.ba.control.ackpolicy",
      FT_BOOLEAN, 16, TFS(&ieee80211_block_ack_control_ack_policy_flag), 0x01,
      "Block Ack Request (BAR) Ack Policy", HFILL }},

    {&hf_ieee80211_block_ack_control_type,
     {"BA Type", "wlan.ba.control.ba_type",
      FT_UINT16, BASE_HEX, VALS(block_ack_type_vals), 0x001e, NULL, HFILL }},

    {&hf_ieee80211_block_ack_control_reserved,
     {"Reserved", "wlan.ba.control.reserved",
      FT_UINT16, BASE_HEX, NULL, 0x0fe0,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_control_tid_info,
     {"TID for which a Basic BlockAck frame is requested", "wlan.ba.basic.tidinfo",
      FT_UINT16, BASE_HEX, NULL, 0xf000,
      "Traffic Identifier (TID) for which a Basic BlockAck frame is requested", HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_aid11,
     {"AID11", "wlan.ba.multi_sta.aid11",
      FT_UINT16, BASE_HEX, NULL, 0x07ff, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_ack_type,
     {"Ack Type", "wlan.ba.multi_sta.ack_type",
      FT_UINT16, BASE_HEX, NULL, 0x0800, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_tid,
     {"TID", "wlan.ba.multi_sta.tid",
      FT_UINT16, BASE_HEX, NULL, 0xf000, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_aid_tid,
     {"AID TID Info", "wlan.ba.multi_sta.aid_tid_info",
      FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_reserved,
     {"Reserved", "wlan.ba.multi_sta.reserved",
      FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_sta_ra,
     {"RA", "wlan.ba.multi_sta.ra",
      FT_ETHER, BASE_NONE, NULL, 0x0, NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_tid_reserved,
     {"Reserved", "wlan.bar.mtid.tidinfo.reserved",
      FT_UINT16, BASE_HEX, 0, 0x0fff,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_multi_tid_value,
     {"Multi-TID Value", "wlan.bar.mtid.tidinfo.value",
      FT_UINT16, BASE_HEX, 0, 0xf000,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_bitmap,
     {"Block Ack Bitmap", "wlan.ba.bm",
      FT_BYTES, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    /* Used for Extended compressed BlockAck */
    {&hf_ieee80211_block_ack_RBUFCAP,
     {"Block Ack RBUFCAP", "wlan.ba.RBUFCAP",
      FT_BOOLEAN, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_bitmap_missing_frame,
     {"Missing frame", "wlan.ba.bm.missing_frame",
      FT_UINT32, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_block_ack_gcr_addr,
     {"GCR Group Address", "wlan.ba.gcr_group_addr",
      FT_ETHER, BASE_NONE, NULL, 0, NULL, HFILL }},

    {&hf_ieee80211_beamform_feedback_seg_retrans_bitmap,
     {"Feedback segment Retansmission Bitmap", "wlan.beamform.feedback_seg_retrans_bitmap",
      FT_UINT8, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_token,
     {"Sounding Dialog Token", "wlan.vht_ndp.token",
      FT_UINT8, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_token_number,
     {"Sounding Dialog Token Number", "wlan.vht_ndp.token.number",
      FT_UINT8, BASE_DEC, NULL, 0xFC,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_he_subfield,
     {"HE", "wlan.vht_ndp.token.he",
      FT_BOOLEAN, 8, TFS(&he_ndp_annc_he_subfield_vals), 0x02, NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_token_reserved,
     {"Reserved", "wlan.vht_ndp.token.reserved",
      FT_UINT8, BASE_HEX, NULL, 0x01,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_sta_info_aid12,
     {"AID12", "wlan.vht_ndp.sta_info.aid12",
      FT_UINT16, BASE_HEX, NULL, 0x0FFF,
      "12 least significant bits of the AID of the target STA", HFILL }},

    {&hf_ieee80211_vht_ndp_annc_sta_info_feedback_type,
     {"Feedback Type", "wlan.vht_ndp.sta_info.feedback_type",
      FT_BOOLEAN, 16, TFS(&vht_ndp_annc_sta_info_feedback_type), 0x1000,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_sta_info_nc_index,
     {"Nc Index", "wlan.vht_ndp.sta_info.nc_index",
      FT_UINT16, BASE_DEC, VALS(num_plus_one_3bit_flag), 0xE000,
      NULL, HFILL }},

    {&hf_ieee80211_vht_ndp_annc_sta_info_reserved,
     {"Reserved", "wlan.vht_ndp.sta_info.reserved",
      FT_UINT16, BASE_HEX, NULL, 0xE000,
      NULL, HFILL }},

    {&hf_ieee80211_data_encap_payload_type,
     {"Payload Type", "wlan.data_encap.payload_type",
      FT_UINT8, BASE_DEC, VALS(ieee80211_data_encap_payload_types), 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_tdls_action_code,
     {"Action code", "wlan.fixed.action_code",
      FT_UINT8, BASE_DEC|BASE_EXT_STRING, &tdls_action_codes_ext, 0,
      "Management action code", HFILL }},

    {&hf_ieee80211_ff_target_channel,
     {"Target Channel", "wlan.fixed.target_channel",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_operating_class,
     {"Operating Class", "wlan.fixed.operating_class",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_wnm_action_code,
     {"Action code", "wlan.fixed.action_code",
      FT_UINT8, BASE_DEC|BASE_EXT_STRING, &wnm_action_codes_ext, 0,
      "Management action code", HFILL }},

    {&hf_ieee80211_ff_unprotected_wnm_action_code,
     {"Action code", "wlan.fixed.action_code",
      FT_UINT8, BASE_DEC|BASE_EXT_STRING, &unprotected_wnm_action_codes_ext, 0,
      "Management action code", HFILL }},

    {&hf_ieee80211_ff_key_data,
     {"Key Data", "wlan.fixed.key_data",
      FT_BYTES, BASE_NONE, NULL, 0, NULL, HFILL }},

    {&hf_ieee80211_ff_key_data_length,
     {"Key Data Length", "wlan.fixed.key_data_length",
      FT_UINT8, BASE_DEC, NULL, 0, NULL, HFILL }},

    {&hf_ieee80211_ff_wnm_notification_type,
     {"WNM-Notification type", "wlan.fixed.wnm_notification_type",
      FT_UINT8, BASE_DEC|BASE_EXT_STRING, &wnm_notification_types_ext, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_action_code,
     {"Action code", "wlan.rm.action_code",
      FT_UINT8, BASE_DEC|BASE_EXT_STRING, &rm_action_codes_ext, 0,
      "Radio Measurement Action", HFILL }},

    {&hf_ieee80211_ff_rm_dialog_token,
     {"Dialog token", "wlan.rm.dialog_token",
      FT_UINT8, BASE_DEC, NULL, 0,
      "Non-zero Dialog Token identifies request/report transaction", HFILL }},

    {&hf_ieee80211_ff_rm_repetitions,
     {"Repetitions", "wlan.rm.repetitions",
      FT_UINT16, BASE_DEC, NULL, 0,
      "Numer of Repetitions, 65535 indicates repeat until cancellation", HFILL }},

    {&hf_ieee80211_ff_rm_tx_power,
     {"Transmit Power Used", "wlan.rm.tx_power",
      FT_INT8, BASE_DEC|BASE_UNIT_STRING, &units_dbm, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_max_tx_power,
     {"Max Transmit Power", "wlan.rm.max_tx_power",
      FT_INT8, BASE_DEC|BASE_UNIT_STRING, &units_dbm, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_tpc,
     {"TPC Report", "wlan.rm.tpc",
      FT_NONE, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_tpc_element_id,
     {"TPC Element ID", "wlan.rm.tpc.element_id",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_tpc_length,
     {"TPC Length", "wlan.rm.tpc.length",
      FT_UINT8, BASE_DEC, NULL, 0,
      "Length of TPC Report element (always 2)", HFILL }},

    {&hf_ieee80211_ff_tpc_tx_power,
     {"TPC Transmit Power", "wlan.rm.tpc.tx_power",
      FT_INT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_tpc_link_margin,
     {"TPC Link Margin", "wlan.rm.tpc.link_margin",
      FT_INT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_rx_antenna_id,
     {"Receive Antenna ID", "wlan.rm.rx_antenna_id",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_tx_antenna_id,
     {"Transmit Antenna ID", "wlan.rm.tx_antenna_id",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_rcpi,
     {"Received Channel Power", "wlan.rm.rcpi",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_rm_rsni,
     {"Received Signal to noise indication", "wlan.rm.rsni",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_request_mode_pref_cand,
     {"Preferred Candidate List Included", "wlan.fixed.request_mode.pref_cand",
      FT_UINT8, BASE_DEC, NULL, 0x01,
      NULL, HFILL }},

    {&hf_ieee80211_ff_request_mode_abridged,
     {"Abridged", "wlan.fixed.request_mode.abridged",
      FT_UINT8, BASE_DEC, NULL, 0x02,
      NULL, HFILL }},

    {&hf_ieee80211_ff_request_mode_disassoc_imminent,
     {"Disassociation Imminent", "wlan.fixed.request_mode.disassoc_imminent",
      FT_UINT8, BASE_DEC, NULL, 0x04,
      NULL, HFILL }},

    {&hf_ieee80211_ff_request_mode_bss_term_included,
     {"BSS Termination Included", "wlan.fixed.request_mode.bss_term_included",
      FT_UINT8, BASE_DEC, NULL, 0x08,
      NULL, HFILL }},

    {&hf_ieee80211_ff_request_mode_ess_disassoc_imminent,
     {"ESS Disassociation Imminent", "wlan.fixed.request_mode.ess_disassoc_imminent",
      FT_UINT8, BASE_DEC, NULL, 0x10,
      NULL, HFILL }},

    {&hf_ieee80211_ff_disassoc_timer,
     {"Disassociation Timer", "wlan.fixed.disassoc_timer",
      FT_UINT16, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bss_termination_delay,
     {"BSS Termination Delay", "wlan.fixed.bss_termination_delay",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bss_transition_status_code,
     {"BSS Transition Status Code", "wlan.fixed.bss_transition_status_code",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_validity_interval,
     {"Validity Interval", "wlan.fixed.validity_interval",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bss_termination_duration,
     {"BSS Termination Duration", "wlan.fixed.bss_termination_duration",
      FT_BYTES, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_url_len,
     {"Session Information URL Length",
      "wlan.fixed.session_information.url_length",
      FT_UINT8, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_url,
     {"Session Information URL", "wlan.fixed.session_information.url",
      FT_STRING, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_target_bss,
     {"BSS Transition Target BSS", "wlan.fixed.bss_transition_target_bss",
      FT_ETHER, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bss_transition_query_reason,
     {"BSS Transition Query Reason", "wlan.fixed.bss_transition_query_reason",
      FT_UINT8, BASE_DEC, VALS(ieee80211_transition_reasons), 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bss_transition_candidate_list_entries,
     {"BSS Transition Candidate List Entries", "wlan.fixed.bss_transition_candidate_list_entries",
      FT_BYTES, BASE_NONE, NULL, 0,
      NULL, HFILL }},

/* 802.11ad */
    {&hf_ieee80211_cf_response_offset,
     {"Response Offset", "wlan.res_offset",
      FT_UINT16, BASE_DEC, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_grant_ack_reserved,
     {"Reserved", "wlan.grant_ack.reserved",
      FT_BYTES, BASE_NONE, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_dynamic_allocation,
     {"Dynamic Allocation", "wlan.dynamic_allocation",
      FT_UINT40, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_TID,
     {"TID", "wlan.dynamic_allocation.tid",
      FT_UINT40, BASE_DEC, NULL, 0x000000000F,
      NULL, HFILL }},

    {&hf_ieee80211_ff_alloc_type,
     {"Allocation Type", "wlan.dynamic_allocation.alloc_type",
      FT_UINT40, BASE_DEC, NULL, 0x000000070,
      NULL, HFILL }},

    {&hf_ieee80211_ff_src_aid,
     {"Source AID", "wlan.dynamic_allocation.src_aid",
      FT_UINT40, BASE_DEC, NULL, 0x0000007F80,
      NULL, HFILL }},

    {&hf_ieee80211_ff_dest_aid,
     {"Destination AID", "wlan.dynamic_allocation.dest_aid",
      FT_UINT40, BASE_DEC, NULL, 0x00007f8000,
      NULL, HFILL }},

    {&hf_ieee80211_ff_alloc_duration,
     {"Allocation Duration", "wlan.dynamic_allocation.alloc_duration",
      FT_UINT40, BASE_CUSTOM, CF_FUNC(allocation_duration_base_custom), 0x7FFF800000,
      NULL, HFILL }},

    {&hf_ieee80211_ff_b39,
     {"Reserved (b39)", "wlan.dynamic_allocation.b39",
      FT_UINT40, BASE_HEX, NULL, 0x8000000000,
      NULL, HFILL }},

    {&hf_ieee80211_ff_ssw,
     {"Sector Sweep", "wlan.ssw",
      FT_UINT24, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_ssw_direction,
     {"Sector Sweep Direction", "wlan.ssw.direction",
      FT_BOOLEAN, 24, TFS(&ieee80211_cf_ssw_direction), 0x000001,
      NULL, HFILL}},

    {&hf_ieee80211_ff_ssw_cdown,
     {"Sector Sweep CDOWN", "wlan.ssw.cdown",
      FT_UINT24, BASE_DEC, NULL, 0x0003fe,
      NULL, HFILL }},

    {&hf_ieee80211_ff_ssw_sector_id,
     {"Sector Sweep Sector ID", "wlan.ssw.sector_id",
      FT_UINT24, BASE_DEC, NULL, 0x00fc00,
      NULL, HFILL }},

    {&hf_ieee80211_ff_ssw_dmg_ant_id,
     {"Sector Sweep DMG Antenna ID", "wlan.ssw.dmg_ant_id",
      FT_UINT24, BASE_DEC, NULL, 0x030000,
      NULL, HFILL }},

    {&hf_ieee80211_ff_ssw_rxss_len,
     {"Sector Sweep RXSS Length", "wlan.ssw.rxss_len",
      FT_UINT24, BASE_DEC, NULL, 0xfc0000,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf,
     {"Beam Forming", "wlan.bf",
      FT_UINT16, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_train,
     {"Beam Forming Training", "wlan.bf.train",
      FT_BOOLEAN, 16, NULL, 0x0001,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_is_init,
     {"Beam Forming Is InitiatorTXSS", "wlan.bf.isInit",
      FT_BOOLEAN, 16, NULL, 0x0002,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_is_resp,
     {"Beam Forming Is ResponderTXSS", "wlan.bf.isResp",
      FT_BOOLEAN, 16, NULL, 0x0004,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_rxss_len,
     {"Beam Forming RXSS Length", "wlan.bf.rxss_len",
      FT_UINT16, BASE_DEC, NULL, 0x01f8,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_rxss_rate,
     {"Beam Forming RXSS Rate", "wlan.bf.rxss_rate",
      FT_BOOLEAN, 16, NULL, 0x0200,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_b10b15,
     {"Reserved (B10-B15)", "wlan.bf.reserved",
      FT_UINT16, BASE_DEC, NULL, 0xFC00,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_num_sectors,
     {"Beam Forming Total Number of Sectors", "wlan.bf.num_sectors",
      FT_UINT16, BASE_DEC, NULL, 0x03f8,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_num_rx_dmg_ants,
     {"Beam Forming Number of DMG Antennas", "wlan.bf.num_dmg_ants",
      FT_UINT16, BASE_DEC, NULL, 0x0c00,
      NULL, HFILL }},

    {&hf_ieee80211_ff_bf_b12b15,
     {"Reserved (B12-B15)", "wlan.bf.reserved",
      FT_UINT16, BASE_DEC, NULL, 0xF000,
      NULL, HFILL }},

    {&hf_ieee80211_addr_nav_da,
     {"Destination address of STA that caused NAV update", "wlan.nav_da",
      FT_ETHER, BASE_NONE, NULL, 0,
      "DMG Destination Hardware Address", HFILL }},

    {&hf_ieee80211_addr_nav_sa,
     {"Source address of STA that caused NAV update", "wlan.nav_sa",
      FT_ETHER, BASE_NONE, NULL, 0,
      "DMG Source Hardware Address", HFILL }},

    {&hf_ieee80211_ff_sswf,
     {"Sector Sweep Feedback", "wlan.sswf",
      FT_UINT24, BASE_HEX, NULL, 0,
      NULL, HFILL }},

    {&hf_ieee80211_ff_sswf_total_sectors,
     {"Sector Sweep Feedback total number of sectors", "wlan.sswf.num_sectors",
      FT_UINT24, BASE_DEC, NULL, 0x0001ff,
