/* packet-mq-base.c
 * Routines for IBM WebSphere MQ packet dissection base
 *
 * metatech <metatech@flashmail.com>
 * robionekenobi <robionekenobi@bluewin.ch>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "config.h"
#include <string.h>
#include <glib.h>
#include <epan/packet.h>
#include "packet-mq.h"

guint64 tvb_get_guint64_endian(tvbuff_t *a_tvb, gint a_iOffset, gint a_rep)
{
    guint64 iResult;
    if (a_rep & ENC_LITTLE_ENDIAN)
        iResult = tvb_get_letoh64(a_tvb, a_iOffset);
    else
        iResult =  tvb_get_ntoh64(a_tvb, a_iOffset);
    return iResult;
}
guint32 tvb_get_guint32_endian(tvbuff_t *a_tvb, gint a_iOffset, gint a_rep)
{
    guint32 iResult;
    if (a_rep & ENC_LITTLE_ENDIAN)
        iResult = tvb_get_letohl(a_tvb, a_iOffset);
    else
        iResult =  tvb_get_ntohl(a_tvb, a_iOffset);
    return iResult;
}
guint16 tvb_get_guint16_endian(tvbuff_t *a_tvb, gint a_iOffset, gint a_rep)
{
    guint16 sResult;
    if (a_rep & ENC_LITTLE_ENDIAN)
        sResult = tvb_get_letohs(a_tvb, a_iOffset);
    else
        sResult =  tvb_get_ntohs(a_tvb, a_iOffset);
    return sResult;
}
/* This routine truncates the string at the first blank space */
gint32 strip_trailing_blanks(guint8* a_str, guint32 a_size)
{
    gint32 i = 0;
    if (a_str != NULL)
    {
        for (i = a_size-1; i >=0; i--)
        {
            if (a_str[i] != ' ' && a_str[i] != '\0')
                break;
            else
                a_str[i]='\0';
        }
    }
    return i;
}

DEF_VALRB(WaitIntv)
    DEF_VALR1(MQCGWI_DEFAULT),
    DEF_VALR1(MQWI_UNLIMITED),
    DEF_VALR3(MQWI_0        ,MQWI_7FFFFFFF, ""),
DEF_VALRE;

DEF_VALRB(OutDataLen)
    DEF_VALR1(MQCODL_AS_INPUT),
    DEF_VALR3(MQCODL_0       ,MQCODL_7FFFFFFF, ""),
DEF_VALRE;

DEF_VALRB(ccsid)
    DEF_VALR3(MQCCSI_UNDEFINED   , MQCCSI_UNDEFINED, "UNDEFINED/DEFAULT/Q_MGR"),
    DEF_VALR1(MQCCSI_EMBEDDED    ),
    DEF_VALR1(MQCCSI_INHERIT     ),
    DEF_VALR1(MQCCSI_APPL        ),
    DEF_VALR1(MQCCSI_AS_PUBLISHED),
    DEF_VALR3(MQCCSI_1           ,MQCCSI_65535, ""),
DEF_VALRE;

DEF_VALSBX(mqat)
    DEF_VALS1(MQAT_UNKNOWN          ),
    DEF_VALS1(MQAT_NO_CONTEXT       ),
    DEF_VALS1(MQAT_CICS             ),
    DEF_VALS2(MQAT_ZOS,"MQAT_MVS/MQAT_OS390/MQAT_ZOS"),
    DEF_VALS1(MQAT_IMS              ),
    DEF_VALS1(MQAT_OS2              ),
    DEF_VALS1(MQAT_DOS              ),
    DEF_VALS2(MQAT_UNIX,"MQAT_AIX/MQAT_UNIX"),
    DEF_VALS1(MQAT_QMGR             ),
    DEF_VALS1(MQAT_OS400            ),
    DEF_VALS1(MQAT_WINDOWS          ),
    DEF_VALS1(MQAT_CICS_VSE         ),
    DEF_VALS1(MQAT_WINDOWS_NT       ),
    DEF_VALS1(MQAT_VMS              ),
    DEF_VALS2(MQAT_GUARDIAN,"MQAT_NSK/MQAT_GUARDIAN"),
    DEF_VALS1(MQAT_VOS              ),
    DEF_VALS1(MQAT_OPEN_TP1         ),
    DEF_VALS1(MQAT_VM               ),
    DEF_VALS1(MQAT_IMS_BRIDGE       ),
    DEF_VALS1(MQAT_XCF              ),
    DEF_VALS1(MQAT_CICS_BRIDGE      ),
    DEF_VALS1(MQAT_NOTES_AGENT      ),
    DEF_VALS1(MQAT_TPF              ),
    DEF_VALS1(MQAT_USER             ),
    DEF_VALS2(MQAT_QMGR_PUBLISH,"MQAT_QMGR_PUBLISH/MQAT_BROKER"),
    DEF_VALS1(MQAT_JAVA             ),
    DEF_VALS1(MQAT_DQM              ),
    DEF_VALS1(MQAT_CHANNEL_INITIATOR),
    DEF_VALS1(MQAT_WLM              ),
    DEF_VALS1(MQAT_BATCH            ),
    DEF_VALS1(MQAT_RRS_BATCH        ),
    DEF_VALS1(MQAT_SIB              ),
    DEF_VALS1(MQAT_SYSTEM_EXTENSION ),
    DEF_VALS1(MQAT_MCAST_PUBLISH    ),
    DEF_VALS1(MQAT_UNKNOWN          ),
DEF_VALSE;

DEF_VALSBX(mqcc)
    DEF_VALS1(MQCC_OK     ),
    DEF_VALS1(MQCC_WARNING),
    DEF_VALS1(MQCC_FAILED ),
    DEF_VALS1(MQCC_UNKNOWN),
DEF_VALSE;

DEF_VALSBX(mqrc)
    DEF_VALS1(MQRC_NONE                      ),
    DEF_VALS1(MQRC_ALIAS_BASE_Q_TYPE_ERROR   ),
    DEF_VALS1(MQRC_ALREADY_CONNECTED         ),
    DEF_VALS1(MQRC_BACKED_OUT                ),
    DEF_VALS1(MQRC_BUFFER_ERROR              ),
    DEF_VALS1(MQRC_BUFFER_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_CHAR_ATTR_LENGTH_ERROR    ),
    DEF_VALS1(MQRC_CHAR_ATTRS_ERROR          ),
    DEF_VALS1(MQRC_CHAR_ATTRS_TOO_SHORT      ),
    DEF_VALS1(MQRC_CONNECTION_BROKEN         ),
    DEF_VALS1(MQRC_DATA_LENGTH_ERROR         ),
    DEF_VALS1(MQRC_DYNAMIC_Q_NAME_ERROR      ),
    DEF_VALS1(MQRC_ENVIRONMENT_ERROR         ),
    DEF_VALS1(MQRC_EXPIRY_ERROR              ),
    DEF_VALS1(MQRC_FEEDBACK_ERROR            ),
    DEF_VALS1(MQRC_GET_INHIBITED             ),
    DEF_VALS1(MQRC_HANDLE_NOT_AVAILABLE      ),
    DEF_VALS1(MQRC_HCONN_ERROR               ),
    DEF_VALS1(MQRC_HOBJ_ERROR                ),
    DEF_VALS1(MQRC_INHIBIT_VALUE_ERROR       ),
    DEF_VALS1(MQRC_INT_ATTR_COUNT_ERROR      ),
    DEF_VALS1(MQRC_INT_ATTR_COUNT_TOO_SMALL  ),
    DEF_VALS1(MQRC_INT_ATTRS_ARRAY_ERROR     ),
    DEF_VALS1(MQRC_SYNCPOINT_LIMIT_REACHED   ),
    DEF_VALS1(MQRC_MAX_CONNS_LIMIT_REACHED   ),
    DEF_VALS1(MQRC_MD_ERROR                  ),
    DEF_VALS1(MQRC_MISSING_REPLY_TO_Q        ),
    DEF_VALS1(MQRC_MSG_TYPE_ERROR            ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_Q         ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_Q_MGR     ),
    DEF_VALS1(MQRC_NO_MSG_AVAILABLE          ),
    DEF_VALS1(MQRC_NO_MSG_UNDER_CURSOR       ),
    DEF_VALS1(MQRC_NOT_AUTHORIZED            ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_BROWSE       ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_INPUT        ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_INQUIRE      ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_OUTPUT       ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET          ),
    DEF_VALS1(MQRC_OBJECT_CHANGED            ),
    DEF_VALS1(MQRC_OBJECT_IN_USE             ),
    DEF_VALS1(MQRC_OBJECT_TYPE_ERROR         ),
    DEF_VALS1(MQRC_OD_ERROR                  ),
    DEF_VALS1(MQRC_OPTION_NOT_VALID_FOR_TYPE ),
    DEF_VALS1(MQRC_OPTIONS_ERROR             ),
    DEF_VALS1(MQRC_PERSISTENCE_ERROR         ),
    DEF_VALS1(MQRC_PERSISTENT_NOT_ALLOWED    ),
    DEF_VALS1(MQRC_PRIORITY_EXCEEDS_MAXIMUM  ),
    DEF_VALS1(MQRC_PRIORITY_ERROR            ),
    DEF_VALS1(MQRC_PUT_INHIBITED             ),
    DEF_VALS1(MQRC_Q_DELETED                 ),
    DEF_VALS1(MQRC_Q_FULL                    ),
    DEF_VALS1(MQRC_Q_NOT_EMPTY               ),
    DEF_VALS1(MQRC_Q_SPACE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_Q_TYPE_ERROR              ),
    DEF_VALS1(MQRC_Q_MGR_NAME_ERROR          ),
    DEF_VALS1(MQRC_Q_MGR_NOT_AVAILABLE       ),
    DEF_VALS1(MQRC_REPORT_OPTIONS_ERROR      ),
    DEF_VALS1(MQRC_SECOND_MARK_NOT_ALLOWED   ),
    DEF_VALS1(MQRC_SECURITY_ERROR            ),
    DEF_VALS1(MQRC_SELECTOR_COUNT_ERROR      ),
    DEF_VALS1(MQRC_SELECTOR_LIMIT_EXCEEDED   ),
    DEF_VALS1(MQRC_SELECTOR_ERROR            ),
    DEF_VALS1(MQRC_SELECTOR_NOT_FOR_TYPE     ),
    DEF_VALS1(MQRC_SIGNAL_OUTSTANDING        ),
    DEF_VALS1(MQRC_SIGNAL_REQUEST_ACCEPTED   ),
    DEF_VALS1(MQRC_STORAGE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_SYNCPOINT_NOT_AVAILABLE   ),
    DEF_VALS1(MQRC_TRIGGER_CONTROL_ERROR     ),
    DEF_VALS1(MQRC_TRIGGER_DEPTH_ERROR       ),
    DEF_VALS1(MQRC_TRIGGER_MSG_PRIORITY_ERR  ),
    DEF_VALS1(MQRC_TRIGGER_TYPE_ERROR        ),
    DEF_VALS1(MQRC_TRUNCATED_MSG_ACCEPTED    ),
    DEF_VALS1(MQRC_TRUNCATED_MSG_FAILED      ),
    DEF_VALS1(MQRC_UNKNOWN_ALIAS_BASE_Q      ),
    DEF_VALS1(MQRC_UNKNOWN_OBJECT_NAME       ),
    DEF_VALS1(MQRC_UNKNOWN_OBJECT_Q_MGR      ),
    DEF_VALS1(MQRC_UNKNOWN_REMOTE_Q_MGR      ),
    DEF_VALS1(MQRC_WAIT_INTERVAL_ERROR       ),
    DEF_VALS1(MQRC_XMIT_Q_TYPE_ERROR         ),
    DEF_VALS1(MQRC_XMIT_Q_USAGE_ERROR        ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_PASS_ALL     ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_PASS_IDENT   ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET_ALL      ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET_IDENT    ),
    DEF_VALS1(MQRC_CONTEXT_HANDLE_ERROR      ),
    DEF_VALS1(MQRC_CONTEXT_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_SIGNAL1_ERROR             ),
    DEF_VALS1(MQRC_OBJECT_ALREADY_EXISTS     ),
    DEF_VALS1(MQRC_OBJECT_DAMAGED            ),
    DEF_VALS1(MQRC_RESOURCE_PROBLEM          ),
    DEF_VALS1(MQRC_ANOTHER_Q_MGR_CONNECTED   ),
    DEF_VALS1(MQRC_UNKNOWN_REPORT_OPTION     ),
    DEF_VALS1(MQRC_STORAGE_CLASS_ERROR       ),
    DEF_VALS1(MQRC_COD_NOT_VALID_FOR_XCF_Q   ),
    DEF_VALS1(MQRC_XWAIT_CANCELED            ),
    DEF_VALS1(MQRC_XWAIT_ERROR               ),
    DEF_VALS1(MQRC_SUPPRESSED_BY_EXIT        ),
    DEF_VALS1(MQRC_FORMAT_ERROR              ),
    DEF_VALS1(MQRC_SOURCE_CCSID_ERROR        ),
    DEF_VALS1(MQRC_SOURCE_INTEGER_ENC_ERROR  ),
    DEF_VALS1(MQRC_SOURCE_DECIMAL_ENC_ERROR  ),
    DEF_VALS1(MQRC_SOURCE_FLOAT_ENC_ERROR    ),
    DEF_VALS1(MQRC_TARGET_CCSID_ERROR        ),
    DEF_VALS1(MQRC_TARGET_INTEGER_ENC_ERROR  ),
    DEF_VALS1(MQRC_TARGET_DECIMAL_ENC_ERROR  ),
    DEF_VALS1(MQRC_TARGET_FLOAT_ENC_ERROR    ),
    DEF_VALS1(MQRC_NOT_CONVERTED             ),
    DEF_VALS1(MQRC_CONVERTED_MSG_TOO_BIG     ),
    DEF_VALS1(MQRC_TRUNCATED                 ),
    DEF_VALS1(MQRC_NO_EXTERNAL_PARTICIPANTS  ),
    DEF_VALS1(MQRC_PARTICIPANT_NOT_AVAILABLE ),
    DEF_VALS1(MQRC_OUTCOME_MIXED             ),
    DEF_VALS1(MQRC_OUTCOME_PENDING           ),
    DEF_VALS1(MQRC_BRIDGE_STARTED            ),
    DEF_VALS1(MQRC_BRIDGE_STOPPED            ),
    DEF_VALS1(MQRC_ADAPTER_STORAGE_SHORTAGE  ),
    DEF_VALS1(MQRC_UOW_IN_PROGRESS           ),
    DEF_VALS1(MQRC_ADAPTER_CONN_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_SERV_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_DEFS_ERROR        ),
    DEF_VALS1(MQRC_ADAPTER_DEFS_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_CONV_LOAD_ERROR   ),
    DEF_VALS1(MQRC_BO_ERROR                  ),
    DEF_VALS1(MQRC_DH_ERROR                  ),
    DEF_VALS1(MQRC_MULTIPLE_REASONS          ),
    DEF_VALS1(MQRC_OPEN_FAILED               ),
    DEF_VALS1(MQRC_ADAPTER_DISC_LOAD_ERROR   ),
    DEF_VALS1(MQRC_CNO_ERROR                 ),
    DEF_VALS1(MQRC_CICS_WAIT_FAILED          ),
    DEF_VALS1(MQRC_DLH_ERROR                 ),
    DEF_VALS1(MQRC_HEADER_ERROR              ),
    DEF_VALS1(MQRC_SOURCE_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_TARGET_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_SOURCE_BUFFER_ERROR       ),
    DEF_VALS1(MQRC_TARGET_BUFFER_ERROR       ),
    DEF_VALS1(MQRC_IIH_ERROR                 ),
    DEF_VALS1(MQRC_PCF_ERROR                 ),
    DEF_VALS1(MQRC_DBCS_ERROR                ),
    DEF_VALS1(MQRC_OBJECT_NAME_ERROR         ),
    DEF_VALS1(MQRC_OBJECT_Q_MGR_NAME_ERROR   ),
    DEF_VALS1(MQRC_RECS_PRESENT_ERROR        ),
    DEF_VALS1(MQRC_OBJECT_RECORDS_ERROR      ),
    DEF_VALS1(MQRC_RESPONSE_RECORDS_ERROR    ),
    DEF_VALS1(MQRC_ASID_MISMATCH             ),
    DEF_VALS1(MQRC_PMO_RECORD_FLAGS_ERROR    ),
    DEF_VALS1(MQRC_PUT_MSG_RECORDS_ERROR     ),
    DEF_VALS1(MQRC_CONN_ID_IN_USE            ),
    DEF_VALS1(MQRC_Q_MGR_QUIESCING           ),
    DEF_VALS1(MQRC_Q_MGR_STOPPING            ),
    DEF_VALS1(MQRC_DUPLICATE_RECOV_COORD     ),
    DEF_VALS1(MQRC_PMO_ERROR                 ),
    DEF_VALS1(MQRC_API_EXIT_NOT_FOUND        ),
    DEF_VALS1(MQRC_API_EXIT_LOAD_ERROR       ),
    DEF_VALS1(MQRC_REMOTE_Q_NAME_ERROR       ),
    DEF_VALS1(MQRC_INCONSISTENT_PERSISTENCE  ),
    DEF_VALS1(MQRC_GMO_ERROR                 ),
    DEF_VALS1(MQRC_CICS_BRIDGE_RESTRICTION   ),
    DEF_VALS1(MQRC_STOPPED_BY_CLUSTER_EXIT   ),
    DEF_VALS1(MQRC_CLUSTER_RESOLUTION_ERROR  ),
    DEF_VALS1(MQRC_CONVERTED_STRING_TOO_BIG  ),
    DEF_VALS1(MQRC_TMC_ERROR                 ),
    DEF_VALS1(MQRC_PAGESET_FULL              ),
    DEF_VALS1(MQRC_STORAGE_MEDIUM_FULL       ),
    DEF_VALS1(MQRC_PAGESET_ERROR             ),
    DEF_VALS1(MQRC_NAME_NOT_VALID_FOR_TYPE   ),
    DEF_VALS1(MQRC_UNEXPECTED_ERROR          ),
    DEF_VALS1(MQRC_UNKNOWN_XMIT_Q            ),
    DEF_VALS1(MQRC_UNKNOWN_DEF_XMIT_Q        ),
    DEF_VALS1(MQRC_DEF_XMIT_Q_TYPE_ERROR     ),
    DEF_VALS1(MQRC_DEF_XMIT_Q_USAGE_ERROR    ),
    DEF_VALS1(MQRC_MSG_MARKED_BROWSE_CO_OP   ),
    DEF_VALS1(MQRC_NAME_IN_USE               ),
    DEF_VALS1(MQRC_CONNECTION_QUIESCING      ),
    DEF_VALS1(MQRC_CONNECTION_STOPPING       ),
    DEF_VALS1(MQRC_ADAPTER_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_MSG_ID_ERROR              ),
    DEF_VALS1(MQRC_CORREL_ID_ERROR           ),
    DEF_VALS1(MQRC_FILE_SYSTEM_ERROR         ),
    DEF_VALS1(MQRC_NO_MSG_LOCKED             ),
    DEF_VALS1(MQRC_SOAP_DOTNET_ERROR         ),
    DEF_VALS1(MQRC_SOAP_AXIS_ERROR           ),
    DEF_VALS1(MQRC_SOAP_URL_ERROR            ),
    DEF_VALS1(MQRC_FILE_NOT_AUDITED          ),
    DEF_VALS1(MQRC_CONNECTION_NOT_AUTHORIZED ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_CHANNEL   ),
    DEF_VALS1(MQRC_CALL_IN_PROGRESS          ),
    DEF_VALS1(MQRC_RMH_ERROR                 ),
    DEF_VALS1(MQRC_Q_MGR_ACTIVE              ),
    DEF_VALS1(MQRC_Q_MGR_NOT_ACTIVE          ),
    DEF_VALS1(MQRC_Q_DEPTH_HIGH              ),
    DEF_VALS1(MQRC_Q_DEPTH_LOW               ),
    DEF_VALS1(MQRC_Q_SERVICE_INTERVAL_HIGH   ),
    DEF_VALS1(MQRC_Q_SERVICE_INTERVAL_OK     ),
    DEF_VALS1(MQRC_RFH_HEADER_FIELD_ERROR    ),
    DEF_VALS1(MQRC_RAS_PROPERTY_ERROR        ),
    DEF_VALS1(MQRC_UNIT_OF_WORK_NOT_STARTED  ),
    DEF_VALS1(MQRC_CHANNEL_AUTO_DEF_OK       ),
    DEF_VALS1(MQRC_CHANNEL_AUTO_DEF_ERROR    ),
    DEF_VALS1(MQRC_CFH_ERROR                 ),
    DEF_VALS1(MQRC_CFIL_ERROR                ),
    DEF_VALS1(MQRC_CFIN_ERROR                ),
    DEF_VALS1(MQRC_CFSL_ERROR                ),
    DEF_VALS1(MQRC_CFST_ERROR                ),
    DEF_VALS1(MQRC_INCOMPLETE_GROUP          ),
    DEF_VALS1(MQRC_INCOMPLETE_MSG            ),
    DEF_VALS1(MQRC_INCONSISTENT_CCSIDS       ),
    DEF_VALS1(MQRC_INCONSISTENT_ENCODINGS    ),
    DEF_VALS1(MQRC_INCONSISTENT_UOW          ),
    DEF_VALS1(MQRC_INVALID_MSG_UNDER_CURSOR  ),
    DEF_VALS1(MQRC_MATCH_OPTIONS_ERROR       ),
    DEF_VALS1(MQRC_MDE_ERROR                 ),
    DEF_VALS1(MQRC_MSG_FLAGS_ERROR           ),
    DEF_VALS1(MQRC_MSG_SEQ_NUMBER_ERROR      ),
    DEF_VALS1(MQRC_OFFSET_ERROR              ),
    DEF_VALS1(MQRC_ORIGINAL_LENGTH_ERROR     ),
    DEF_VALS1(MQRC_SEGMENT_LENGTH_ZERO       ),
    DEF_VALS1(MQRC_UOW_NOT_AVAILABLE         ),
    DEF_VALS1(MQRC_WRONG_GMO_VERSION         ),
    DEF_VALS1(MQRC_WRONG_MD_VERSION          ),
    DEF_VALS1(MQRC_GROUP_ID_ERROR            ),
    DEF_VALS1(MQRC_INCONSISTENT_BROWSE       ),
    DEF_VALS1(MQRC_XQH_ERROR                 ),
    DEF_VALS1(MQRC_SRC_ENV_ERROR             ),
    DEF_VALS1(MQRC_SRC_NAME_ERROR            ),
    DEF_VALS1(MQRC_DEST_ENV_ERROR            ),
    DEF_VALS1(MQRC_DEST_NAME_ERROR           ),
    DEF_VALS1(MQRC_TM_ERROR                  ),
    DEF_VALS1(MQRC_CLUSTER_EXIT_ERROR        ),
    DEF_VALS1(MQRC_CLUSTER_EXIT_LOAD_ERROR   ),
    DEF_VALS1(MQRC_CLUSTER_PUT_INHIBITED     ),
    DEF_VALS1(MQRC_CLUSTER_RESOURCE_ERROR    ),
    DEF_VALS1(MQRC_NO_DESTINATIONS_AVAILABLE ),
    DEF_VALS1(MQRC_CONN_TAG_IN_USE           ),
    DEF_VALS1(MQRC_PARTIALLY_CONVERTED       ),
    DEF_VALS1(MQRC_CONNECTION_ERROR          ),
    DEF_VALS1(MQRC_OPTION_ENVIRONMENT_ERROR  ),
    DEF_VALS1(MQRC_CD_ERROR                  ),
    DEF_VALS1(MQRC_CLIENT_CONN_ERROR         ),
    DEF_VALS1(MQRC_CHANNEL_STOPPED_BY_USER   ),
    DEF_VALS1(MQRC_HCONFIG_ERROR             ),
    DEF_VALS1(MQRC_FUNCTION_ERROR            ),
    DEF_VALS1(MQRC_CHANNEL_STARTED           ),
    DEF_VALS1(MQRC_CHANNEL_STOPPED           ),
    DEF_VALS1(MQRC_CHANNEL_CONV_ERROR        ),
    DEF_VALS1(MQRC_SERVICE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_INITIALIZATION_FAILED     ),
    DEF_VALS1(MQRC_TERMINATION_FAILED        ),
    DEF_VALS1(MQRC_UNKNOWN_Q_NAME            ),
    DEF_VALS1(MQRC_SERVICE_ERROR             ),
    DEF_VALS1(MQRC_Q_ALREADY_EXISTS          ),
    DEF_VALS1(MQRC_USER_ID_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_UNKNOWN_ENTITY            ),
    DEF_VALS1(MQRC_UNKNOWN_AUTH_ENTITY       ),
    DEF_VALS1(MQRC_UNKNOWN_REF_OBJECT        ),
    DEF_VALS1(MQRC_CHANNEL_ACTIVATED         ),
    DEF_VALS1(MQRC_CHANNEL_NOT_ACTIVATED     ),
    DEF_VALS1(MQRC_UOW_CANCELED              ),
    DEF_VALS1(MQRC_FUNCTION_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_SELECTOR_TYPE_ERROR       ),
    DEF_VALS1(MQRC_COMMAND_TYPE_ERROR        ),
    DEF_VALS1(MQRC_MULTIPLE_INSTANCE_ERROR   ),
    DEF_VALS1(MQRC_SYSTEM_ITEM_NOT_ALTERABLE ),
    DEF_VALS1(MQRC_BAG_CONVERSION_ERROR      ),
    DEF_VALS1(MQRC_SELECTOR_OUT_OF_RANGE     ),
    DEF_VALS1(MQRC_SELECTOR_NOT_UNIQUE       ),
    DEF_VALS1(MQRC_INDEX_NOT_PRESENT         ),
    DEF_VALS1(MQRC_STRING_ERROR              ),
    DEF_VALS1(MQRC_ENCODING_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_SELECTOR_NOT_PRESENT      ),
    DEF_VALS1(MQRC_OUT_SELECTOR_ERROR        ),
    DEF_VALS1(MQRC_STRING_TRUNCATED          ),
    DEF_VALS1(MQRC_SELECTOR_WRONG_TYPE       ),
    DEF_VALS1(MQRC_INCONSISTENT_ITEM_TYPE    ),
    DEF_VALS1(MQRC_INDEX_ERROR               ),
    DEF_VALS1(MQRC_SYSTEM_BAG_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_ITEM_COUNT_ERROR          ),
    DEF_VALS1(MQRC_FORMAT_NOT_SUPPORTED      ),
    DEF_VALS1(MQRC_SELECTOR_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_ITEM_VALUE_ERROR          ),
    DEF_VALS1(MQRC_HBAG_ERROR                ),
    DEF_VALS1(MQRC_PARAMETER_MISSING         ),
    DEF_VALS1(MQRC_CMD_SERVER_NOT_AVAILABLE  ),
    DEF_VALS1(MQRC_STRING_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_INQUIRY_COMMAND_ERROR     ),
    DEF_VALS1(MQRC_NESTED_BAG_NOT_SUPPORTED  ),
    DEF_VALS1(MQRC_BAG_WRONG_TYPE            ),
    DEF_VALS1(MQRC_ITEM_TYPE_ERROR           ),
    DEF_VALS1(MQRC_SYSTEM_BAG_NOT_DELETABLE  ),
    DEF_VALS1(MQRC_SYSTEM_ITEM_NOT_DELETABLE ),
    DEF_VALS1(MQRC_CODED_CHAR_SET_ID_ERROR   ),
    DEF_VALS1(MQRC_MSG_TOKEN_ERROR           ),
    DEF_VALS1(MQRC_MISSING_WIH               ),
    DEF_VALS1(MQRC_WIH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_STRING_ERROR          ),
    DEF_VALS1(MQRC_RFH_COMMAND_ERROR         ),
    DEF_VALS1(MQRC_RFH_PARM_ERROR            ),
    DEF_VALS1(MQRC_RFH_DUPLICATE_PARM        ),
    DEF_VALS1(MQRC_RFH_PARM_MISSING          ),
    DEF_VALS1(MQRC_CHAR_CONVERSION_ERROR     ),
    DEF_VALS1(MQRC_UCS2_CONVERSION_ERROR     ),
    DEF_VALS1(MQRC_DB2_NOT_AVAILABLE         ),
    DEF_VALS1(MQRC_OBJECT_NOT_UNIQUE         ),
    DEF_VALS1(MQRC_CONN_TAG_NOT_RELEASED     ),
    DEF_VALS1(MQRC_CF_NOT_AVAILABLE          ),
    DEF_VALS1(MQRC_CF_STRUC_IN_USE           ),
    DEF_VALS1(MQRC_CF_STRUC_LIST_HDR_IN_USE  ),
    DEF_VALS1(MQRC_CF_STRUC_AUTH_FAILED      ),
    DEF_VALS1(MQRC_CF_STRUC_ERROR            ),
    DEF_VALS1(MQRC_CONN_TAG_NOT_USABLE       ),
    DEF_VALS1(MQRC_GLOBAL_UOW_CONFLICT       ),
    DEF_VALS1(MQRC_LOCAL_UOW_CONFLICT        ),
    DEF_VALS1(MQRC_HANDLE_IN_USE_FOR_UOW     ),
    DEF_VALS1(MQRC_UOW_ENLISTMENT_ERROR      ),
    DEF_VALS1(MQRC_UOW_MIX_NOT_SUPPORTED     ),
    DEF_VALS1(MQRC_WXP_ERROR                 ),
    DEF_VALS1(MQRC_CURRENT_RECORD_ERROR      ),
    DEF_VALS1(MQRC_NEXT_OFFSET_ERROR         ),
    DEF_VALS1(MQRC_NO_RECORD_AVAILABLE       ),
    DEF_VALS1(MQRC_OBJECT_LEVEL_INCOMPATIBLE ),
    DEF_VALS1(MQRC_NEXT_RECORD_ERROR         ),
    DEF_VALS1(MQRC_BACKOUT_THRESHOLD_REACHED ),
    DEF_VALS1(MQRC_MSG_NOT_MATCHED           ),
    DEF_VALS1(MQRC_JMS_FORMAT_ERROR          ),
    DEF_VALS1(MQRC_SEGMENTS_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_WRONG_CF_LEVEL            ),
    DEF_VALS1(MQRC_CONFIG_CREATE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_CHANGE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_DELETE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_REFRESH_OBJECT     ),
    DEF_VALS1(MQRC_CHANNEL_SSL_ERROR         ),
    DEF_VALS1(MQRC_PARTICIPANT_NOT_DEFINED   ),
    DEF_VALS1(MQRC_CF_STRUC_FAILED           ),
    DEF_VALS1(MQRC_API_EXIT_ERROR            ),
    DEF_VALS1(MQRC_API_EXIT_INIT_ERROR       ),
    DEF_VALS1(MQRC_API_EXIT_TERM_ERROR       ),
    DEF_VALS1(MQRC_EXIT_REASON_ERROR         ),
    DEF_VALS1(MQRC_RESERVED_VALUE_ERROR      ),
    DEF_VALS1(MQRC_NO_DATA_AVAILABLE         ),
    DEF_VALS1(MQRC_SCO_ERROR                 ),
    DEF_VALS1(MQRC_KEY_REPOSITORY_ERROR      ),
    DEF_VALS1(MQRC_CRYPTO_HARDWARE_ERROR     ),
    DEF_VALS1(MQRC_AUTH_INFO_REC_COUNT_ERROR ),
    DEF_VALS1(MQRC_AUTH_INFO_REC_ERROR       ),
    DEF_VALS1(MQRC_AIR_ERROR                 ),
    DEF_VALS1(MQRC_AUTH_INFO_TYPE_ERROR      ),
    DEF_VALS1(MQRC_AUTH_INFO_CONN_NAME_ERROR ),
    DEF_VALS1(MQRC_LDAP_USER_NAME_ERROR      ),
    DEF_VALS1(MQRC_LDAP_USER_NAME_LENGTH_ERR ),
    DEF_VALS1(MQRC_LDAP_PASSWORD_ERROR       ),
    DEF_VALS1(MQRC_SSL_ALREADY_INITIALIZED   ),
    DEF_VALS1(MQRC_SSL_CONFIG_ERROR          ),
    DEF_VALS1(MQRC_SSL_INITIALIZATION_ERROR  ),
    DEF_VALS1(MQRC_Q_INDEX_TYPE_ERROR        ),
    DEF_VALS1(MQRC_CFBS_ERROR                ),
    DEF_VALS1(MQRC_SSL_NOT_ALLOWED           ),
    DEF_VALS1(MQRC_JSSE_ERROR                ),
    DEF_VALS1(MQRC_SSL_PEER_NAME_MISMATCH    ),
    DEF_VALS1(MQRC_SSL_PEER_NAME_ERROR       ),
    DEF_VALS1(MQRC_UNSUPPORTED_CIPHER_SUITE  ),
    DEF_VALS1(MQRC_SSL_CERTIFICATE_REVOKED   ),
    DEF_VALS1(MQRC_SSL_CERT_STORE_ERROR      ),
    DEF_VALS1(MQRC_CLIENT_EXIT_LOAD_ERROR    ),
    DEF_VALS1(MQRC_CLIENT_EXIT_ERROR         ),
    DEF_VALS1(MQRC_UOW_COMMITTED             ),
    DEF_VALS1(MQRC_SSL_KEY_RESET_ERROR       ),
    DEF_VALS1(MQRC_UNKNOWN_COMPONENT_NAME    ),
    DEF_VALS1(MQRC_LOGGER_STATUS             ),
    DEF_VALS1(MQRC_COMMAND_MQSC              ),
    DEF_VALS1(MQRC_COMMAND_PCF               ),
    DEF_VALS1(MQRC_CFIF_ERROR                ),
    DEF_VALS1(MQRC_CFSF_ERROR                ),
    DEF_VALS1(MQRC_CFGR_ERROR                ),
    DEF_VALS1(MQRC_MSG_NOT_ALLOWED_IN_GROUP  ),
    DEF_VALS1(MQRC_FILTER_OPERATOR_ERROR     ),
    DEF_VALS1(MQRC_NESTED_SELECTOR_ERROR     ),
    DEF_VALS1(MQRC_EPH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_FORMAT_ERROR          ),
    DEF_VALS1(MQRC_CFBF_ERROR                ),
    DEF_VALS1(MQRC_CLIENT_CHANNEL_CONFLICT   ),
    DEF_VALS1(MQRC_SD_ERROR                  ),
    DEF_VALS1(MQRC_TOPIC_STRING_ERROR        ),
    DEF_VALS1(MQRC_STS_ERROR                 ),
    DEF_VALS1(MQRC_NO_SUBSCRIPTION           ),
    DEF_VALS1(MQRC_SUBSCRIPTION_IN_USE       ),
    DEF_VALS1(MQRC_STAT_TYPE_ERROR           ),
    DEF_VALS1(MQRC_SUB_USER_DATA_ERROR       ),
    DEF_VALS1(MQRC_SUB_ALREADY_EXISTS        ),
    DEF_VALS1(MQRC_IDENTITY_MISMATCH         ),
    DEF_VALS1(MQRC_ALTER_SUB_ERROR           ),
    DEF_VALS1(MQRC_DURABILITY_NOT_ALLOWED    ),
    DEF_VALS1(MQRC_NO_RETAINED_MSG           ),
    DEF_VALS1(MQRC_SRO_ERROR                 ),
    DEF_VALS1(MQRC_SUB_NAME_ERROR            ),
    DEF_VALS1(MQRC_OBJECT_STRING_ERROR       ),
    DEF_VALS1(MQRC_PROPERTY_NAME_ERROR       ),
    DEF_VALS1(MQRC_SEGMENTATION_NOT_ALLOWED  ),
    DEF_VALS1(MQRC_CBD_ERROR                 ),
    DEF_VALS1(MQRC_CTLO_ERROR                ),
    DEF_VALS1(MQRC_NO_CALLBACKS_ACTIVE       ),
    DEF_VALS1(MQRC_CALLBACK_NOT_REGISTERED   ),
    DEF_VALS1(MQRC_OPTIONS_CHANGED           ),
    DEF_VALS1(MQRC_READ_AHEAD_MSGS           ),
    DEF_VALS1(MQRC_SELECTOR_SYNTAX_ERROR     ),
    DEF_VALS1(MQRC_HMSG_ERROR                ),
    DEF_VALS1(MQRC_CMHO_ERROR                ),
    DEF_VALS1(MQRC_DMHO_ERROR                ),
    DEF_VALS1(MQRC_SMPO_ERROR                ),
    DEF_VALS1(MQRC_IMPO_ERROR                ),
    DEF_VALS1(MQRC_PROPERTY_NAME_TOO_BIG     ),
    DEF_VALS1(MQRC_PROP_VALUE_NOT_CONVERTED  ),
    DEF_VALS1(MQRC_PROP_TYPE_NOT_SUPPORTED   ),
    DEF_VALS1(MQRC_PROPERTY_VALUE_TOO_BIG    ),
    DEF_VALS1(MQRC_PROP_CONV_NOT_SUPPORTED   ),
    DEF_VALS1(MQRC_PROPERTY_NOT_AVAILABLE    ),
    DEF_VALS1(MQRC_PROP_NUMBER_FORMAT_ERROR  ),
    DEF_VALS1(MQRC_PROPERTY_TYPE_ERROR       ),
    DEF_VALS1(MQRC_PROPERTIES_TOO_BIG        ),
    DEF_VALS1(MQRC_PUT_NOT_RETAINED          ),
    DEF_VALS1(MQRC_ALIAS_TARGTYPE_CHANGED    ),
    DEF_VALS1(MQRC_DMPO_ERROR                ),
    DEF_VALS1(MQRC_PD_ERROR                  ),
    DEF_VALS1(MQRC_CALLBACK_TYPE_ERROR       ),
    DEF_VALS1(MQRC_CBD_OPTIONS_ERROR         ),
    DEF_VALS1(MQRC_MAX_MSG_LENGTH_ERROR      ),
    DEF_VALS1(MQRC_CALLBACK_ROUTINE_ERROR    ),
    DEF_VALS1(MQRC_CALLBACK_LINK_ERROR       ),
    DEF_VALS1(MQRC_OPERATION_ERROR           ),
    DEF_VALS1(MQRC_BMHO_ERROR                ),
    DEF_VALS1(MQRC_UNSUPPORTED_PROPERTY      ),
    DEF_VALS1(MQRC_PROP_NAME_NOT_CONVERTED   ),
    DEF_VALS1(MQRC_GET_ENABLED               ),
    DEF_VALS1(MQRC_MODULE_NOT_FOUND          ),
    DEF_VALS1(MQRC_MODULE_INVALID            ),
    DEF_VALS1(MQRC_MODULE_ENTRY_NOT_FOUND    ),
    DEF_VALS1(MQRC_MIXED_CONTENT_NOT_ALLOWED ),
    DEF_VALS1(MQRC_MSG_HANDLE_IN_USE         ),
    DEF_VALS1(MQRC_HCONN_ASYNC_ACTIVE        ),
    DEF_VALS1(MQRC_MHBO_ERROR                ),
    DEF_VALS1(MQRC_PUBLICATION_FAILURE       ),
    DEF_VALS1(MQRC_SUB_INHIBITED             ),
    DEF_VALS1(MQRC_SELECTOR_ALWAYS_FALSE     ),
    DEF_VALS1(MQRC_XEPO_ERROR                ),
    DEF_VALS1(MQRC_DURABILITY_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_TOPIC_NOT_ALTERABLE       ),
    DEF_VALS1(MQRC_SUBLEVEL_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_PROPERTY_NAME_LENGTH_ERR  ),
    DEF_VALS1(MQRC_DUPLICATE_GROUP_SUB       ),
    DEF_VALS1(MQRC_GROUPING_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_SELECTOR_INVALID_FOR_TYPE ),
    DEF_VALS1(MQRC_HOBJ_QUIESCED             ),
    DEF_VALS1(MQRC_HOBJ_QUIESCED_NO_MSGS     ),
    DEF_VALS1(MQRC_SELECTION_STRING_ERROR    ),
    DEF_VALS1(MQRC_RES_OBJECT_STRING_ERROR   ),
    DEF_VALS1(MQRC_CONNECTION_SUSPENDED      ),
    DEF_VALS1(MQRC_INVALID_DESTINATION       ),
    DEF_VALS1(MQRC_INVALID_SUBSCRIPTION      ),
    DEF_VALS1(MQRC_SELECTOR_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_RETAINED_MSG_Q_ERROR      ),
    DEF_VALS1(MQRC_RETAINED_NOT_DELIVERED    ),
    DEF_VALS1(MQRC_RFH_RESTRICTED_FORMAT_ERR ),
    DEF_VALS1(MQRC_CONNECTION_STOPPED        ),
    DEF_VALS1(MQRC_ASYNC_UOW_CONFLICT        ),
    DEF_VALS1(MQRC_ASYNC_XA_CONFLICT         ),
    DEF_VALS1(MQRC_PUBSUB_INHIBITED          ),
    DEF_VALS1(MQRC_MSG_HANDLE_COPY_FAILURE   ),
    DEF_VALS1(MQRC_DEST_CLASS_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_OPERATION_NOT_ALLOWED     ),
    DEF_VALS1(MQRC_ACTION_ERROR              ),
    DEF_VALS1(MQRC_CHANNEL_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_HOST_NOT_AVAILABLE        ),
    DEF_VALS1(MQRC_CHANNEL_CONFIG_ERROR      ),
    DEF_VALS1(MQRC_UNKNOWN_CHANNEL_NAME      ),
    DEF_VALS1(MQRC_LOOPING_PUBLICATION       ),
    DEF_VALS1(MQRC_ALREADY_JOINED            ),
    DEF_VALS1(MQRC_STANDBY_Q_MGR             ),
    DEF_VALS1(MQRC_RECONNECTING              ),
    DEF_VALS1(MQRC_RECONNECTED               ),
    DEF_VALS1(MQRC_RECONNECT_QMID_MISMATCH   ),
    DEF_VALS1(MQRC_RECONNECT_INCOMPATIBLE    ),
    DEF_VALS1(MQRC_RECONNECT_FAILED          ),
    DEF_VALS1(MQRC_CALL_INTERRUPTED          ),
    DEF_VALS1(MQRC_NO_SUBS_MATCHED           ),
    DEF_VALS1(MQRC_SELECTION_NOT_AVAILABLE   ),
    DEF_VALS1(MQRC_CHANNEL_SSL_WARNING       ),
    DEF_VALS1(MQRC_OCSP_URL_ERROR            ),
    DEF_VALS1(MQRC_CONTENT_ERROR             ),
    DEF_VALS1(MQRC_RECONNECT_Q_MGR_REQD      ),
    DEF_VALS1(MQRC_RECONNECT_TIMED_OUT       ),
    DEF_VALS1(MQRC_PUBLISH_EXIT_ERROR        ),
    DEF_VALS1(MQRC_SSL_ALT_PROVIDER_REQUIRED ),
    DEF_VALS1(MQRC_PRECONN_EXIT_LOAD_ERROR   ),
    DEF_VALS1(MQRC_PRECONN_EXIT_NOT_FOUND    ),
    DEF_VALS1(MQRC_PRECONN_EXIT_ERROR        ),
    DEF_VALS1(MQRC_CD_ARRAY_ERROR            ),
    DEF_VALS1(MQRC_CHANNEL_BLOCKED           ),
    DEF_VALS1(MQRC_CHANNEL_BLOCKED_WARNING   ),
    DEF_VALS1(MQRC_SUBSCRIPTION_CREATE       ),
    DEF_VALS1(MQRC_SUBSCRIPTION_DELETE       ),
    DEF_VALS1(MQRC_SUBSCRIPTION_CHANGE       ),
    DEF_VALS1(MQRC_SUBSCRIPTION_REFRESH      ),
    DEF_VALS1(MQRC_INSTALLATION_MISMATCH     ),
    DEF_VALS1(MQRC_NOT_PRIVILEGED            ),
    DEF_VALS1(MQRC_PROPERTIES_DISABLED       ),
    DEF_VALS1(MQRC_HMSG_NOT_AVAILABLE        ),
    DEF_VALS1(MQRC_EXIT_PROPS_NOT_SUPPORTED  ),
    DEF_VALS1(MQRC_INSTALLATION_MISSING      ),
    DEF_VALS1(MQRC_FASTPATH_NOT_AVAILABLE    ),
    DEF_VALS1(MQRC_CIPHER_SPEC_NOT_SUITE_B   ),
    DEF_VALS1(MQRC_SUITE_B_ERROR             ),
    DEF_VALS1(MQRC_CERT_VAL_POLICY_ERROR     ),

    DEF_VALS1(MQRCCF_CFH_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_CFH_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_CFH_VERSION_ERROR          ),
    DEF_VALS1(MQRCCF_CFH_MSG_SEQ_NUMBER_ERR     ),
    DEF_VALS1(MQRCCF_CFH_CONTROL_ERROR          ),
    DEF_VALS1(MQRCCF_CFH_PARM_COUNT_ERROR       ),
    DEF_VALS1(MQRCCF_CFH_COMMAND_ERROR          ),
    DEF_VALS1(MQRCCF_COMMAND_FAILED             ),
    DEF_VALS1(MQRCCF_CFIN_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFST_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFST_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_FORCE_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_STRUCTURE_TYPE_ERROR       ),
    DEF_VALS1(MQRCCF_CFIN_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFST_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_MSG_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_CFIN_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFST_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_PARM_COUNT_TOO_SMALL       ),
    DEF_VALS1(MQRCCF_PARM_COUNT_TOO_BIG         ),
    DEF_VALS1(MQRCCF_Q_ALREADY_IN_CELL          ),
    DEF_VALS1(MQRCCF_Q_TYPE_ERROR               ),
    DEF_VALS1(MQRCCF_MD_FORMAT_ERROR            ),
    DEF_VALS1(MQRCCF_CFSL_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_REPLACE_VALUE_ERROR        ),
    DEF_VALS1(MQRCCF_CFIL_DUPLICATE_VALUE       ),
    DEF_VALS1(MQRCCF_CFIL_COUNT_ERROR           ),
    DEF_VALS1(MQRCCF_CFIL_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_QUIESCE_VALUE_ERROR        ),
    DEF_VALS1(MQRCCF_MODE_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_MSG_SEQ_NUMBER_ERROR       ),
    DEF_VALS1(MQRCCF_PING_DATA_COUNT_ERROR      ),
    DEF_VALS1(MQRCCF_PING_DATA_COMPARE_ERROR    ),
    DEF_VALS1(MQRCCF_CFSL_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CHANNEL_TYPE_ERROR         ),
    DEF_VALS1(MQRCCF_PARM_SEQUENCE_ERROR        ),
    DEF_VALS1(MQRCCF_XMIT_PROTOCOL_TYPE_ERR     ),
    DEF_VALS1(MQRCCF_BATCH_SIZE_ERROR           ),
    DEF_VALS1(MQRCCF_DISC_INT_ERROR             ),
    DEF_VALS1(MQRCCF_SHORT_RETRY_ERROR          ),
    DEF_VALS1(MQRCCF_SHORT_TIMER_ERROR          ),
    DEF_VALS1(MQRCCF_LONG_RETRY_ERROR           ),
    DEF_VALS1(MQRCCF_LONG_TIMER_ERROR           ),
    DEF_VALS1(MQRCCF_SEQ_NUMBER_WRAP_ERROR      ),
    DEF_VALS1(MQRCCF_MAX_MSG_LENGTH_ERROR       ),
    DEF_VALS1(MQRCCF_PUT_AUTH_ERROR             ),
    DEF_VALS1(MQRCCF_PURGE_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_CFIL_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_MSG_TRUNCATED              ),
    DEF_VALS1(MQRCCF_CCSID_ERROR                ),
    DEF_VALS1(MQRCCF_ENCODING_ERROR             ),
    DEF_VALS1(MQRCCF_QUEUES_VALUE_ERROR         ),
    DEF_VALS1(MQRCCF_DATA_CONV_VALUE_ERROR      ),
    DEF_VALS1(MQRCCF_INDOUBT_VALUE_ERROR        ),
    DEF_VALS1(MQRCCF_ESCAPE_TYPE_ERROR          ),
    DEF_VALS1(MQRCCF_REPOS_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_CHANNEL_TABLE_ERROR        ),
    DEF_VALS1(MQRCCF_MCA_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_CHL_INST_TYPE_ERROR        ),
    DEF_VALS1(MQRCCF_CHL_STATUS_NOT_FOUND       ),
    DEF_VALS1(MQRCCF_CFSL_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFSL_TOTAL_LENGTH_ERROR    ),
    DEF_VALS1(MQRCCF_CFSL_COUNT_ERROR           ),
    DEF_VALS1(MQRCCF_CFSL_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_BROKER_DELETED             ),
    DEF_VALS1(MQRCCF_STREAM_ERROR               ),
    DEF_VALS1(MQRCCF_TOPIC_ERROR                ),
    DEF_VALS1(MQRCCF_NOT_REGISTERED             ),
    DEF_VALS1(MQRCCF_Q_MGR_NAME_ERROR           ),
    DEF_VALS1(MQRCCF_INCORRECT_STREAM           ),
    DEF_VALS1(MQRCCF_Q_NAME_ERROR               ),
    DEF_VALS1(MQRCCF_NO_RETAINED_MSG            ),
    DEF_VALS1(MQRCCF_DUPLICATE_IDENTITY         ),
    DEF_VALS1(MQRCCF_INCORRECT_Q                ),
    DEF_VALS1(MQRCCF_CORREL_ID_ERROR            ),
    DEF_VALS1(MQRCCF_NOT_AUTHORIZED             ),
    DEF_VALS1(MQRCCF_UNKNOWN_STREAM             ),
    DEF_VALS1(MQRCCF_REG_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_PUB_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_UNKNOWN_BROKER             ),
    DEF_VALS1(MQRCCF_Q_MGR_CCSID_ERROR          ),
    DEF_VALS1(MQRCCF_DEL_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_CLUSTER_NAME_CONFLICT      ),
    DEF_VALS1(MQRCCF_REPOS_NAME_CONFLICT        ),
    DEF_VALS1(MQRCCF_CLUSTER_Q_USAGE_ERROR      ),
    DEF_VALS1(MQRCCF_ACTION_VALUE_ERROR         ),
    DEF_VALS1(MQRCCF_COMMS_LIBRARY_ERROR        ),
    DEF_VALS1(MQRCCF_NETBIOS_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_BROKER_COMMAND_FAILED      ),
    DEF_VALS1(MQRCCF_CFST_CONFLICTING_PARM      ),
    DEF_VALS1(MQRCCF_PATH_NOT_VALID             ),
    DEF_VALS1(MQRCCF_PARM_SYNTAX_ERROR          ),
    DEF_VALS1(MQRCCF_PWD_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_FILTER_ERROR               ),
    DEF_VALS1(MQRCCF_WRONG_USER                 ),
    DEF_VALS1(MQRCCF_DUPLICATE_SUBSCRIPTION     ),
    DEF_VALS1(MQRCCF_SUB_NAME_ERROR             ),
    DEF_VALS1(MQRCCF_SUB_IDENTITY_ERROR         ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_IN_USE        ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_LOCKED        ),
    DEF_VALS1(MQRCCF_ALREADY_JOINED             ),
    DEF_VALS1(MQRCCF_OBJECT_IN_USE              ),
    DEF_VALS1(MQRCCF_UNKNOWN_FILE_NAME          ),
    DEF_VALS1(MQRCCF_FILE_NOT_AVAILABLE         ),
    DEF_VALS1(MQRCCF_DISC_RETRY_ERROR           ),
    DEF_VALS1(MQRCCF_ALLOC_RETRY_ERROR          ),
    DEF_VALS1(MQRCCF_ALLOC_SLOW_TIMER_ERROR     ),
    DEF_VALS1(MQRCCF_ALLOC_FAST_TIMER_ERROR     ),
    DEF_VALS1(MQRCCF_PORT_NUMBER_ERROR          ),
    DEF_VALS1(MQRCCF_CHL_SYSTEM_NOT_ACTIVE      ),
    DEF_VALS1(MQRCCF_ENTITY_NAME_MISSING        ),
    DEF_VALS1(MQRCCF_PROFILE_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_AUTH_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_AUTH_VALUE_MISSING         ),
    DEF_VALS1(MQRCCF_OBJECT_TYPE_MISSING        ),
    DEF_VALS1(MQRCCF_CONNECTION_ID_ERROR        ),
    DEF_VALS1(MQRCCF_LOG_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_PROGRAM_NOT_AVAILABLE      ),
    DEF_VALS1(MQRCCF_PROGRAM_AUTH_FAILED        ),
    DEF_VALS1(MQRCCF_NONE_FOUND                 ),
    DEF_VALS1(MQRCCF_SECURITY_SWITCH_OFF        ),
    DEF_VALS1(MQRCCF_SECURITY_REFRESH_FAILED    ),
    DEF_VALS1(MQRCCF_PARM_CONFLICT              ),
    DEF_VALS1(MQRCCF_COMMAND_INHIBITED          ),
    DEF_VALS1(MQRCCF_OBJECT_BEING_DELETED       ),
    DEF_VALS1(MQRCCF_STORAGE_CLASS_IN_USE       ),
    DEF_VALS1(MQRCCF_OBJECT_NAME_RESTRICTED     ),
    DEF_VALS1(MQRCCF_OBJECT_LIMIT_EXCEEDED      ),
    DEF_VALS1(MQRCCF_OBJECT_OPEN_FORCE          ),
    DEF_VALS1(MQRCCF_DISPOSITION_CONFLICT       ),
    DEF_VALS1(MQRCCF_Q_MGR_NOT_IN_QSG           ),
    DEF_VALS1(MQRCCF_ATTR_VALUE_FIXED           ),
    DEF_VALS1(MQRCCF_NAMELIST_ERROR             ),
    DEF_VALS1(MQRCCF_NO_CHANNEL_INITIATOR       ),
    DEF_VALS1(MQRCCF_CHANNEL_INITIATOR_ERROR    ),
    DEF_VALS1(MQRCCF_COMMAND_LEVEL_CONFLICT     ),
    DEF_VALS1(MQRCCF_Q_ATTR_CONFLICT            ),
    DEF_VALS1(MQRCCF_EVENTS_DISABLED            ),
    DEF_VALS1(MQRCCF_COMMAND_SCOPE_ERROR        ),
    DEF_VALS1(MQRCCF_COMMAND_REPLY_ERROR        ),
    DEF_VALS1(MQRCCF_FUNCTION_RESTRICTED        ),
    DEF_VALS1(MQRCCF_PARM_MISSING               ),
    DEF_VALS1(MQRCCF_PARM_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_COMMAND_LENGTH_ERROR       ),
    DEF_VALS1(MQRCCF_COMMAND_ORIGIN_ERROR       ),
    DEF_VALS1(MQRCCF_LISTENER_CONFLICT          ),
    DEF_VALS1(MQRCCF_LISTENER_STARTED           ),
    DEF_VALS1(MQRCCF_LISTENER_STOPPED           ),
    DEF_VALS1(MQRCCF_CHANNEL_ERROR              ),
    DEF_VALS1(MQRCCF_CF_STRUC_ERROR             ),
    DEF_VALS1(MQRCCF_UNKNOWN_USER_ID            ),
    DEF_VALS1(MQRCCF_UNEXPECTED_ERROR           ),
    DEF_VALS1(MQRCCF_NO_XCF_PARTNER             ),
    DEF_VALS1(MQRCCF_CFGR_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFIF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFIF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFIF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFSF_FILTER_VAL_LEN_ERR    ),
    DEF_VALS1(MQRCCF_CFSF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFSF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFSF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_TOO_MANY_FILTERS           ),
    DEF_VALS1(MQRCCF_LISTENER_RUNNING           ),
    DEF_VALS1(MQRCCF_LSTR_STATUS_NOT_FOUND      ),
    DEF_VALS1(MQRCCF_SERVICE_RUNNING            ),
    DEF_VALS1(MQRCCF_SERV_STATUS_NOT_FOUND      ),
    DEF_VALS1(MQRCCF_SERVICE_STOPPED            ),
    DEF_VALS1(MQRCCF_CFBS_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFBS_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFBS_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFBS_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_CFGR_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFGR_PARM_COUNT_ERROR      ),
    DEF_VALS1(MQRCCF_CONN_NOT_STOPPED           ),
    DEF_VALS1(MQRCCF_SERVICE_REQUEST_PENDING    ),
    DEF_VALS1(MQRCCF_NO_START_CMD               ),
    DEF_VALS1(MQRCCF_NO_STOP_CMD                ),
    DEF_VALS1(MQRCCF_CFBF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFBF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFBF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFBF_FILTER_VAL_LEN_ERR    ),
    DEF_VALS1(MQRCCF_LISTENER_STILL_ACTIVE      ),
    DEF_VALS1(MQRCCF_TOPICSTR_ALREADY_EXISTS    ),
    DEF_VALS1(MQRCCF_SHARING_CONVS_ERROR        ),
    DEF_VALS1(MQRCCF_SHARING_CONVS_TYPE         ),
    DEF_VALS1(MQRCCF_SECURITY_CASE_CONFLICT     ),
    DEF_VALS1(MQRCCF_TOPIC_TYPE_ERROR           ),
    DEF_VALS1(MQRCCF_MAX_INSTANCES_ERROR        ),
    DEF_VALS1(MQRCCF_MAX_INSTS_PER_CLNT_ERR     ),
    DEF_VALS1(MQRCCF_TOPIC_STRING_NOT_FOUND     ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_POINT_ERR     ),
    DEF_VALS1(MQRCCF_SUB_ALREADY_EXISTS         ),
    DEF_VALS1(MQRCCF_UNKNOWN_OBJECT_NAME        ),
    DEF_VALS1(MQRCCF_REMOTE_Q_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_DURABILITY_NOT_ALLOWED     ),
    DEF_VALS1(MQRCCF_HOBJ_ERROR                 ),
    DEF_VALS1(MQRCCF_DEST_NAME_ERROR            ),
    DEF_VALS1(MQRCCF_INVALID_DESTINATION        ),
    DEF_VALS1(MQRCCF_PUBSUB_INHIBITED           ),
    DEF_VALS1(MQRCCF_GROUPUR_CHECKS_FAILED      ),
    DEF_VALS1(MQRCCF_SSL_ALT_PROVIDER_REQD      ),
    DEF_VALS1(MQRCCF_OBJECT_ALREADY_EXISTS      ),
    DEF_VALS1(MQRCCF_OBJECT_WRONG_TYPE          ),
    DEF_VALS1(MQRCCF_LIKE_OBJECT_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_OBJECT_OPEN                ),
    DEF_VALS1(MQRCCF_ATTR_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_UNKNOWN_Q_MGR              ),
    DEF_VALS1(MQRCCF_Q_WRONG_TYPE               ),
    DEF_VALS1(MQRCCF_OBJECT_NAME_ERROR          ),
    DEF_VALS1(MQRCCF_ALLOCATE_FAILED            ),
    DEF_VALS1(MQRCCF_HOST_NOT_AVAILABLE         ),
    DEF_VALS1(MQRCCF_CONFIGURATION_ERROR        ),
    DEF_VALS1(MQRCCF_CONNECTION_REFUSED         ),
    DEF_VALS1(MQRCCF_ENTRY_ERROR                ),
    DEF_VALS1(MQRCCF_SEND_FAILED                ),
    DEF_VALS1(MQRCCF_RECEIVED_DATA_ERROR        ),
    DEF_VALS1(MQRCCF_RECEIVE_FAILED             ),
    DEF_VALS1(MQRCCF_CONNECTION_CLOSED          ),
    DEF_VALS1(MQRCCF_NO_STORAGE                 ),
    DEF_VALS1(MQRCCF_NO_COMMS_MANAGER           ),
    DEF_VALS1(MQRCCF_LISTENER_NOT_STARTED       ),
    DEF_VALS1(MQRCCF_BIND_FAILED                ),
    DEF_VALS1(MQRCCF_CHANNEL_INDOUBT            ),
    DEF_VALS1(MQRCCF_MQCONN_FAILED              ),
    DEF_VALS1(MQRCCF_MQOPEN_FAILED              ),
    DEF_VALS1(MQRCCF_MQGET_FAILED               ),
    DEF_VALS1(MQRCCF_MQPUT_FAILED               ),
    DEF_VALS1(MQRCCF_PING_ERROR                 ),
    DEF_VALS1(MQRCCF_CHANNEL_IN_USE             ),
    DEF_VALS1(MQRCCF_CHANNEL_NOT_FOUND          ),
    DEF_VALS1(MQRCCF_UNKNOWN_REMOTE_CHANNEL     ),
    DEF_VALS1(MQRCCF_REMOTE_QM_UNAVAILABLE      ),
    DEF_VALS1(MQRCCF_REMOTE_QM_TERMINATING      ),
    DEF_VALS1(MQRCCF_MQINQ_FAILED               ),
    DEF_VALS1(MQRCCF_NOT_XMIT_Q                 ),
    DEF_VALS1(MQRCCF_CHANNEL_DISABLED           ),
    DEF_VALS1(MQRCCF_USER_EXIT_NOT_AVAILABLE    ),
    DEF_VALS1(MQRCCF_COMMIT_FAILED              ),
    DEF_VALS1(MQRCCF_WRONG_CHANNEL_TYPE         ),
    DEF_VALS1(MQRCCF_CHANNEL_ALREADY_EXISTS     ),
    DEF_VALS1(MQRCCF_DATA_TOO_LARGE             ),
    DEF_VALS1(MQRCCF_CHANNEL_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_XMIT_Q_NAME_ERROR          ),
    DEF_VALS1(MQRCCF_MCA_NAME_ERROR             ),
    DEF_VALS1(MQRCCF_SEND_EXIT_NAME_ERROR       ),
    DEF_VALS1(MQRCCF_SEC_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_MSG_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_RCV_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_XMIT_Q_NAME_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_MCA_NAME_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_DISC_INT_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_SHORT_RETRY_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_SHORT_TIMER_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_LONG_RETRY_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_LONG_TIMER_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_PUT_AUTH_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_KEEP_ALIVE_INT_ERROR       ),
    DEF_VALS1(MQRCCF_MISSING_CONN_NAME          ),
    DEF_VALS1(MQRCCF_CONN_NAME_ERROR            ),
    DEF_VALS1(MQRCCF_MQSET_FAILED               ),
    DEF_VALS1(MQRCCF_CHANNEL_NOT_ACTIVE         ),
    DEF_VALS1(MQRCCF_TERMINATED_BY_SEC_EXIT     ),
    DEF_VALS1(MQRCCF_DYNAMIC_Q_SCOPE_ERROR      ),
    DEF_VALS1(MQRCCF_CELL_DIR_NOT_AVAILABLE     ),
    DEF_VALS1(MQRCCF_MR_COUNT_ERROR             ),
    DEF_VALS1(MQRCCF_MR_COUNT_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_MR_EXIT_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_MR_EXIT_NAME_WRONG_TYPE    ),
    DEF_VALS1(MQRCCF_MR_INTERVAL_ERROR          ),
    DEF_VALS1(MQRCCF_MR_INTERVAL_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_NPM_SPEED_ERROR            ),
    DEF_VALS1(MQRCCF_NPM_SPEED_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_HB_INTERVAL_ERROR          ),
    DEF_VALS1(MQRCCF_HB_INTERVAL_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_CHAD_ERROR                 ),
    DEF_VALS1(MQRCCF_CHAD_WRONG_TYPE            ),
    DEF_VALS1(MQRCCF_CHAD_EVENT_ERROR           ),
    DEF_VALS1(MQRCCF_CHAD_EVENT_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_CHAD_EXIT_ERROR            ),
    DEF_VALS1(MQRCCF_CHAD_EXIT_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_SUPPRESSED_BY_EXIT         ),
    DEF_VALS1(MQRCCF_BATCH_INT_ERROR            ),
    DEF_VALS1(MQRCCF_BATCH_INT_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_NET_PRIORITY_ERROR         ),
    DEF_VALS1(MQRCCF_NET_PRIORITY_WRONG_TYPE    ),
    DEF_VALS1(MQRCCF_CHANNEL_CLOSED             ),
    DEF_VALS1(MQRCCF_Q_STATUS_NOT_FOUND         ),
    DEF_VALS1(MQRCCF_SSL_CIPHER_SPEC_ERROR      ),
    DEF_VALS1(MQRCCF_SSL_PEER_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_SSL_CLIENT_AUTH_ERROR      ),
    DEF_VALS1(MQRCCF_RETAINED_NOT_SUPPORTED     ),

    DEF_VALS1(MQRC_REOPEN_EXCL_INPUT_ERROR   ),
    DEF_VALS1(MQRC_REOPEN_INQUIRE_ERROR      ),
    DEF_VALS1(MQRC_REOPEN_SAVED_CONTEXT_ERR  ),
    DEF_VALS1(MQRC_REOPEN_TEMPORARY_Q_ERROR  ),
    DEF_VALS1(MQRC_ATTRIBUTE_LOCKED          ),
    DEF_VALS1(MQRC_CURSOR_NOT_VALID          ),
    DEF_VALS1(MQRC_ENCODING_ERROR            ),
    DEF_VALS1(MQRC_STRUC_ID_ERROR            ),
    DEF_VALS1(MQRC_NULL_POINTER              ),
    DEF_VALS1(MQRC_NO_CONNECTION_REFERENCE   ),
    DEF_VALS1(MQRC_NO_BUFFER                 ),
    DEF_VALS1(MQRC_BINARY_DATA_LENGTH_ERROR  ),
    DEF_VALS1(MQRC_BUFFER_NOT_AUTOMATIC      ),
    DEF_VALS1(MQRC_INSUFFICIENT_BUFFER       ),
    DEF_VALS1(MQRC_INSUFFICIENT_DATA         ),
    DEF_VALS1(MQRC_DATA_TRUNCATED            ),
    DEF_VALS1(MQRC_ZERO_LENGTH               ),
    DEF_VALS1(MQRC_NEGATIVE_LENGTH           ),
    DEF_VALS1(MQRC_NEGATIVE_OFFSET           ),
    DEF_VALS1(MQRC_INCONSISTENT_FORMAT       ),
    DEF_VALS1(MQRC_INCONSISTENT_OBJECT_STATE ),
    DEF_VALS1(MQRC_CONTEXT_OBJECT_NOT_VALID  ),
    DEF_VALS1(MQRC_CONTEXT_OPEN_ERROR        ),
    DEF_VALS1(MQRC_STRUC_LENGTH_ERROR        ),
    DEF_VALS1(MQRC_NOT_CONNECTED             ),
    DEF_VALS1(MQRC_NOT_OPEN                  ),
    DEF_VALS1(MQRC_DISTRIBUTION_LIST_EMPTY   ),
    DEF_VALS1(MQRC_INCONSISTENT_OPEN_OPTIONS ),
    DEF_VALS1(MQRC_WRONG_VERSION             ),
    DEF_VALS1(MQRC_REFERENCE_ERROR           ),
    DEF_VALS1(MQRC_XR_NOT_AVAILABLE          ),
DEF_VALSE;

DEF_VALSBX(selector)
    DEF_VALS1(MQIA_APPL_TYPE                 ),
    DEF_VALS1(MQIA_CODED_CHAR_SET_ID         ),
    DEF_VALS1(MQIA_CURRENT_Q_DEPTH           ),
    DEF_VALS1(MQIA_DEF_INPUT_OPEN_OPTION     ),
    DEF_VALS1(MQIA_DEF_PERSISTENCE           ),
    DEF_VALS1(MQIA_DEF_PRIORITY              ),
    DEF_VALS1(MQIA_DEFINITION_TYPE           ),
    DEF_VALS1(MQIA_HARDEN_GET_BACKOUT        ),
    DEF_VALS1(MQIA_INHIBIT_GET               ),
    DEF_VALS1(MQIA_INHIBIT_PUT               ),
    DEF_VALS1(MQIA_MAX_HANDLES               ),
    DEF_VALS1(MQIA_USAGE                     ),
    DEF_VALS1(MQIA_MAX_MSG_LENGTH            ),
    DEF_VALS1(MQIA_MAX_PRIORITY              ),
    DEF_VALS1(MQIA_MAX_Q_DEPTH               ),
    DEF_VALS1(MQIA_MSG_DELIVERY_SEQUENCE     ),
    DEF_VALS1(MQIA_OPEN_INPUT_COUNT          ),
    DEF_VALS1(MQIA_OPEN_OUTPUT_COUNT         ),
    DEF_VALS1(MQIA_NAME_COUNT                ),
    DEF_VALS1(MQIA_Q_TYPE                    ),
    DEF_VALS1(MQIA_RETENTION_INTERVAL        ),
    DEF_VALS1(MQIA_BACKOUT_THRESHOLD         ),
    DEF_VALS1(MQIA_SHAREABILITY              ),
    DEF_VALS1(MQIA_TRIGGER_CONTROL           ),
    DEF_VALS1(MQIA_TRIGGER_INTERVAL          ),
    DEF_VALS1(MQIA_TRIGGER_MSG_PRIORITY      ),
    DEF_VALS1(MQIA_CPI_LEVEL                 ),
    DEF_VALS1(MQIA_TRIGGER_TYPE              ),
    DEF_VALS1(MQIA_TRIGGER_DEPTH             ),
    DEF_VALS1(MQIA_SYNCPOINT                 ),
    DEF_VALS1(MQIA_COMMAND_LEVEL             ),
    DEF_VALS1(MQIA_PLATFORM                  ),
    DEF_VALS1(MQIA_MAX_UNCOMMITTED_MSGS      ),
    DEF_VALS1(MQIA_DIST_LISTS                ),
    DEF_VALS1(MQIA_TIME_SINCE_RESET          ),
    DEF_VALS1(MQIA_HIGH_Q_DEPTH              ),
    DEF_VALS1(MQIA_MSG_ENQ_COUNT             ),
    DEF_VALS1(MQIA_MSG_DEQ_COUNT             ),
    DEF_VALS1(MQIA_EXPIRY_INTERVAL           ),
    DEF_VALS1(MQIA_Q_DEPTH_HIGH_LIMIT        ),
    DEF_VALS1(MQIA_Q_DEPTH_LOW_LIMIT         ),
    DEF_VALS1(MQIA_Q_DEPTH_MAX_EVENT         ),
    DEF_VALS1(MQIA_Q_DEPTH_HIGH_EVENT        ),
    DEF_VALS1(MQIA_Q_DEPTH_LOW_EVENT         ),
    DEF_VALS1(MQIA_SCOPE                     ),
    DEF_VALS1(MQIA_Q_SERVICE_INTERVAL_EVENT  ),
    DEF_VALS1(MQIA_AUTHORITY_EVENT           ),
    DEF_VALS1(MQIA_INHIBIT_EVENT             ),
    DEF_VALS1(MQIA_LOCAL_EVENT               ),
    DEF_VALS1(MQIA_REMOTE_EVENT              ),
    DEF_VALS1(MQIA_CONFIGURATION_EVENT       ),
    DEF_VALS1(MQIA_START_STOP_EVENT          ),
    DEF_VALS1(MQIA_PERFORMANCE_EVENT         ),
    DEF_VALS1(MQIA_Q_SERVICE_INTERVAL        ),
    DEF_VALS1(MQIA_CHANNEL_AUTO_DEF          ),
    DEF_VALS1(MQIA_CHANNEL_AUTO_DEF_EVENT    ),
    DEF_VALS1(MQIA_INDEX_TYPE                ),
    DEF_VALS1(MQIA_CLUSTER_WORKLOAD_LENGTH   ),
    DEF_VALS1(MQIA_CLUSTER_Q_TYPE            ),
    DEF_VALS1(MQIA_ARCHIVE                   ),
    DEF_VALS1(MQIA_DEF_BIND                  ),
    DEF_VALS1(MQIA_PAGESET_ID                ),
    DEF_VALS1(MQIA_QSG_DISP                  ),
    DEF_VALS1(MQIA_INTRA_GROUP_QUEUING       ),
    DEF_VALS1(MQIA_IGQ_PUT_AUTHORITY         ),
    DEF_VALS1(MQIA_AUTH_INFO_TYPE            ),
    DEF_VALS1(MQIA_MSG_MARK_BROWSE_INTERVAL  ),
    DEF_VALS1(MQIA_SSL_TASKS                 ),
    DEF_VALS1(MQIA_CF_LEVEL                  ),
    DEF_VALS1(MQIA_CF_RECOVER                ),
    DEF_VALS1(MQIA_NAMELIST_TYPE             ),
    DEF_VALS1(MQIA_CHANNEL_EVENT             ),
    DEF_VALS1(MQIA_BRIDGE_EVENT              ),
    DEF_VALS1(MQIA_SSL_EVENT                 ),
    DEF_VALS1(MQIA_SSL_RESET_COUNT           ),
    DEF_VALS1(MQIA_SHARED_Q_Q_MGR_NAME       ),
    DEF_VALS1(MQIA_NPM_CLASS                 ),
    DEF_VALS1(MQIA_MAX_OPEN_Q                ),
    DEF_VALS1(MQIA_MONITOR_INTERVAL          ),
    DEF_VALS1(MQIA_Q_USERS                   ),
    DEF_VALS1(MQIA_MAX_GLOBAL_LOCKS          ),
    DEF_VALS1(MQIA_MAX_LOCAL_LOCKS           ),
    DEF_VALS1(MQIA_LISTENER_PORT_NUMBER      ),
    DEF_VALS1(MQIA_BATCH_INTERFACE_AUTO      ),
    DEF_VALS1(MQIA_CMD_SERVER_AUTO           ),
    DEF_VALS1(MQIA_CMD_SERVER_CONVERT_MSG    ),
    DEF_VALS1(MQIA_CMD_SERVER_DLQ_MSG        ),
    DEF_VALS1(MQIA_MAX_Q_TRIGGERS            ),
    DEF_VALS1(MQIA_TRIGGER_RESTART           ),
    DEF_VALS1(MQIA_SSL_FIPS_REQUIRED         ),
    DEF_VALS1(MQIA_IP_ADDRESS_VERSION        ),
    DEF_VALS1(MQIA_LOGGER_EVENT              ),
    DEF_VALS1(MQIA_CLWL_Q_RANK               ),
    DEF_VALS1(MQIA_CLWL_Q_PRIORITY           ),
    DEF_VALS1(MQIA_CLWL_MRU_CHANNELS         ),
    DEF_VALS1(MQIA_CLWL_USEQ                 ),
    DEF_VALS1(MQIA_COMMAND_EVENT             ),
    DEF_VALS1(MQIA_ACTIVE_CHANNELS           ),
    DEF_VALS1(MQIA_CHINIT_ADAPTERS           ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_CHECK         ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_TYPE          ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_INTERVAL      ),
    DEF_VALS1(MQIA_CHINIT_DISPATCHERS        ),
    DEF_VALS1(MQIA_DNS_WLM                   ),
    DEF_VALS1(MQIA_LISTENER_TIMER            ),
    DEF_VALS1(MQIA_LU62_CHANNELS             ),
    DEF_VALS1(MQIA_MAX_CHANNELS              ),
    DEF_VALS1(MQIA_OUTBOUND_PORT_MIN         ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT           ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT_TYPE      ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT_MIN       ),
    DEF_VALS1(MQIA_TCP_CHANNELS              ),
    DEF_VALS1(MQIA_TCP_KEEP_ALIVE            ),
    DEF_VALS1(MQIA_TCP_STACK_TYPE            ),
    DEF_VALS1(MQIA_CHINIT_TRACE_AUTO_START   ),
    DEF_VALS1(MQIA_CHINIT_TRACE_TABLE_SIZE   ),
    DEF_VALS1(MQIA_CHINIT_CONTROL            ),
    DEF_VALS1(MQIA_CMD_SERVER_CONTROL        ),
    DEF_VALS1(MQIA_SERVICE_TYPE              ),
    DEF_VALS1(MQIA_MONITORING_CHANNEL        ),
    DEF_VALS1(MQIA_MONITORING_Q              ),
    DEF_VALS1(MQIA_MONITORING_AUTO_CLUSSDR   ),
    DEF_VALS1(MQIA_STATISTICS_MQI            ),
    DEF_VALS1(MQIA_STATISTICS_Q              ),
    DEF_VALS1(MQIA_STATISTICS_CHANNEL        ),
    DEF_VALS1(MQIA_STATISTICS_AUTO_CLUSSDR   ),
    DEF_VALS1(MQIA_STATISTICS_INTERVAL       ),
    DEF_VALS1(MQIA_ACCOUNTING_MQI            ),
    DEF_VALS1(MQIA_ACCOUNTING_Q              ),
    DEF_VALS1(MQIA_ACCOUNTING_INTERVAL       ),
    DEF_VALS1(MQIA_ACCOUNTING_CONN_OVERRIDE  ),
    DEF_VALS1(MQIA_TRACE_ROUTE_RECORDING     ),
    DEF_VALS1(MQIA_ACTIVITY_RECORDING        ),
    DEF_VALS1(MQIA_SERVICE_CONTROL           ),
    DEF_VALS1(MQIA_OUTBOUND_PORT_MAX         ),
    DEF_VALS1(MQIA_SECURITY_CASE             ),
    DEF_VALS1(MQIA_QMOPT_CSMT_ON_ERROR       ),
    DEF_VALS1(MQIA_QMOPT_CONS_INFO_MSGS      ),
    DEF_VALS1(MQIA_QMOPT_CONS_WARNING_MSGS   ),
    DEF_VALS1(MQIA_QMOPT_CONS_ERROR_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_CONS_CRITICAL_MSGS  ),
    DEF_VALS1(MQIA_QMOPT_CONS_COMMS_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_CONS_REORG_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_CONS_SYSTEM_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_LOG_INFO_MSGS       ),
    DEF_VALS1(MQIA_QMOPT_LOG_WARNING_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_LOG_ERROR_MSGS      ),
    DEF_VALS1(MQIA_QMOPT_LOG_CRITICAL_MSGS   ),
    DEF_VALS1(MQIA_QMOPT_LOG_COMMS_MSGS      ),
    DEF_VALS1(MQIA_QMOPT_LOG_REORG_MSGS      ),
    DEF_VALS1(MQIA_QMOPT_LOG_SYSTEM_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_TRACE_MQI_CALLS     ),
    DEF_VALS1(MQIA_QMOPT_TRACE_COMMS         ),
    DEF_VALS1(MQIA_QMOPT_TRACE_REORG         ),
    DEF_VALS1(MQIA_QMOPT_TRACE_CONVERSION    ),
    DEF_VALS1(MQIA_QMOPT_TRACE_SYSTEM        ),
    DEF_VALS1(MQIA_QMOPT_INTERNAL_DUMP       ),
    DEF_VALS1(MQIA_MAX_RECOVERY_TASKS        ),
    DEF_VALS1(MQIA_MAX_CLIENTS               ),
    DEF_VALS1(MQIA_AUTO_REORGANIZATION       ),
    DEF_VALS1(MQIA_AUTO_REORG_INTERVAL       ),
    DEF_VALS1(MQIA_DURABLE_SUB               ),
    DEF_VALS1(MQIA_MULTICAST                 ),
    DEF_VALS1(MQIA_INHIBIT_PUB               ),
    DEF_VALS1(MQIA_INHIBIT_SUB               ),
    DEF_VALS1(MQIA_TREE_LIFE_TIME            ),
    DEF_VALS1(MQIA_DEF_PUT_RESPONSE_TYPE     ),
    DEF_VALS1(MQIA_TOPIC_DEF_PERSISTENCE     ),
    DEF_VALS1(MQIA_MASTER_ADMIN              ),
    DEF_VALS1(MQIA_PUBSUB_MODE               ),
    DEF_VALS1(MQIA_DEF_READ_AHEAD            ),
    DEF_VALS1(MQIA_READ_AHEAD                ),
    DEF_VALS1(MQIA_PROPERTY_CONTROL          ),
    DEF_VALS1(MQIA_MAX_PROPERTIES_LENGTH     ),
    DEF_VALS1(MQIA_BASE_TYPE                 ),
    DEF_VALS1(MQIA_PM_DELIVERY               ),
    DEF_VALS1(MQIA_NPM_DELIVERY              ),
    DEF_VALS1(MQIA_PROXY_SUB                 ),
    DEF_VALS1(MQIA_PUBSUB_NP_MSG             ),
    DEF_VALS1(MQIA_SUB_COUNT                 ),
    DEF_VALS1(MQIA_PUBSUB_NP_RESP            ),
    DEF_VALS1(MQIA_PUBSUB_MAXMSG_RETRY_COUNT ),
    DEF_VALS1(MQIA_PUBSUB_SYNC_PT            ),
    DEF_VALS1(MQIA_TOPIC_TYPE                ),
    DEF_VALS1(MQIA_PUB_COUNT                 ),
    DEF_VALS1(MQIA_WILDCARD_OPERATION        ),
    DEF_VALS1(MQIA_SUB_SCOPE                 ),
    DEF_VALS1(MQIA_PUB_SCOPE                 ),
    DEF_VALS1(MQIA_GROUP_UR                  ),
    DEF_VALS1(MQIA_UR_DISP                   ),
    DEF_VALS1(MQIA_COMM_INFO_TYPE            ),
    DEF_VALS1(MQIA_CF_OFFLOAD                ),
    DEF_VALS1(MQIA_CF_OFFLOAD_THRESHOLD1     ),
    DEF_VALS1(MQIA_CF_OFFLOAD_THRESHOLD2     ),
    DEF_VALS1(MQIA_CF_OFFLOAD_THRESHOLD3     ),
    DEF_VALS1(MQIA_CF_SMDS_BUFFERS           ),
    DEF_VALS1(MQIA_CF_OFFLDUSE               ),
    DEF_VALS1(MQIA_MAX_RESPONSES             ),
    DEF_VALS1(MQIA_RESPONSE_RESTART_POINT    ),
    DEF_VALS1(MQIA_COMM_EVENT                ),
    DEF_VALS1(MQIA_MCAST_BRIDGE              ),
    DEF_VALS1(MQIA_USE_DEAD_LETTER_Q         ),
    DEF_VALS1(MQIA_TOLERATE_UNPROTECTED      ),
    DEF_VALS1(MQIA_SIGNATURE_ALGORITHM       ),
    DEF_VALS1(MQIA_ENCRYPTION_ALGORITHM      ),
    DEF_VALS1(MQIA_POLICY_VERSION            ),
    DEF_VALS1(MQIA_ACTIVITY_CONN_OVERRIDE    ),
    DEF_VALS1(MQIA_ACTIVITY_TRACE            ),
    DEF_VALS1(MQIA_SUB_CONFIGURATION_EVENT   ),
    DEF_VALS1(MQIA_XR_CAPABILITY             ),
    DEF_VALS1(MQIA_CF_RECAUTO                ),
    DEF_VALS1(MQIA_QMGR_CFCONLOS             ),
    DEF_VALS1(MQIA_CF_CFCONLOS               ),
    DEF_VALS1(MQIA_SUITE_B_STRENGTH          ),
    DEF_VALS1(MQIA_CHLAUTH_RECORDS           ),
    DEF_VALS1(MQIA_PUBSUB_CLUSTER            ),
    DEF_VALS1(MQIA_DEF_CLUSTER_XMIT_Q_TYPE   ),
    DEF_VALS1(MQIA_PROT_POLICY_CAPABILITY    ),
    DEF_VALS1(MQIA_CERT_VAL_POLICY           ),
    DEF_VALS1(MQIA_USER_LIST                 ),
    DEF_VALS1(MQCA_APPL_ID                   ),
    DEF_VALS1(MQCA_BASE_Q_NAME               ),
    DEF_VALS1(MQCA_BASE_OBJECT_NAME          ),
    DEF_VALS1(MQCA_COMMAND_INPUT_Q_NAME      ),
    DEF_VALS1(MQCA_CREATION_DATE             ),
    DEF_VALS1(MQCA_CREATION_TIME             ),
    DEF_VALS1(MQCA_DEAD_LETTER_Q_NAME        ),
    DEF_VALS1(MQCA_ENV_DATA                  ),
    DEF_VALS1(MQCA_INITIATION_Q_NAME         ),
    DEF_VALS1(MQCA_NAMELIST_DESC             ),
    DEF_VALS1(MQCA_NAMELIST_NAME             ),
    DEF_VALS1(MQCA_PROCESS_DESC              ),
    DEF_VALS1(MQCA_PROCESS_NAME              ),
    DEF_VALS1(MQCA_Q_DESC                    ),
    DEF_VALS1(MQCA_Q_MGR_DESC                ),
    DEF_VALS1(MQCA_Q_MGR_NAME                ),
    DEF_VALS1(MQCA_Q_NAME                    ),
    DEF_VALS1(MQCA_REMOTE_Q_MGR_NAME         ),
    DEF_VALS1(MQCA_REMOTE_Q_NAME             ),
    DEF_VALS1(MQCA_BACKOUT_REQ_Q_NAME        ),
    DEF_VALS1(MQCA_NAMES                     ),
    DEF_VALS1(MQCA_USER_DATA                 ),
    DEF_VALS1(MQCA_STORAGE_CLASS             ),
    DEF_VALS1(MQCA_TRIGGER_DATA              ),
    DEF_VALS1(MQCA_XMIT_Q_NAME               ),
    DEF_VALS1(MQCA_DEF_XMIT_Q_NAME           ),
    DEF_VALS1(MQCA_CHANNEL_AUTO_DEF_EXIT     ),
    DEF_VALS1(MQCA_ALTERATION_DATE           ),
    DEF_VALS1(MQCA_ALTERATION_TIME           ),
    DEF_VALS1(MQCA_CLUSTER_NAME              ),
    DEF_VALS1(MQCA_CLUSTER_NAMELIST          ),
    DEF_VALS1(MQCA_CLUSTER_Q_MGR_NAME        ),
    DEF_VALS1(MQCA_Q_MGR_IDENTIFIER          ),
    DEF_VALS1(MQCA_CLUSTER_WORKLOAD_EXIT     ),
    DEF_VALS1(MQCA_CLUSTER_WORKLOAD_DATA     ),
    DEF_VALS1(MQCA_REPOSITORY_NAME           ),
    DEF_VALS1(MQCA_REPOSITORY_NAMELIST       ),
    DEF_VALS1(MQCA_CLUSTER_DATE              ),
    DEF_VALS1(MQCA_CLUSTER_TIME              ),
    DEF_VALS1(MQCA_CF_STRUC_NAME             ),
    DEF_VALS1(MQCA_QSG_NAME                  ),
    DEF_VALS1(MQCA_IGQ_USER_ID               ),
    DEF_VALS1(MQCA_STORAGE_CLASS_DESC        ),
    DEF_VALS1(MQCA_XCF_GROUP_NAME            ),
    DEF_VALS1(MQCA_XCF_MEMBER_NAME           ),
    DEF_VALS1(MQCA_AUTH_INFO_NAME            ),
    DEF_VALS1(MQCA_AUTH_INFO_DESC            ),
    DEF_VALS1(MQCA_LDAP_USER_NAME            ),
    DEF_VALS1(MQCA_LDAP_PASSWORD             ),
    DEF_VALS1(MQCA_SSL_KEY_REPOSITORY        ),
    DEF_VALS1(MQCA_SSL_CRL_NAMELIST          ),
    DEF_VALS1(MQCA_SSL_CRYPTO_HARDWARE       ),
    DEF_VALS1(MQCA_CF_STRUC_DESC             ),
    DEF_VALS1(MQCA_AUTH_INFO_CONN_NAME       ),
    DEF_VALS1(MQCA_CICS_FILE_NAME            ),
    DEF_VALS1(MQCA_TRIGGER_TRANS_ID          ),
    DEF_VALS1(MQCA_TRIGGER_PROGRAM_NAME      ),
    DEF_VALS1(MQCA_TRIGGER_TERM_ID           ),
    DEF_VALS1(MQCA_TRIGGER_CHANNEL_NAME      ),
    DEF_VALS1(MQCA_SYSTEM_LOG_Q_NAME         ),
    DEF_VALS1(MQCA_MONITOR_Q_NAME            ),
    DEF_VALS1(MQCA_COMMAND_REPLY_Q_NAME      ),
    DEF_VALS1(MQCA_BATCH_INTERFACE_ID        ),
    DEF_VALS1(MQCA_SSL_KEY_LIBRARY           ),
    DEF_VALS1(MQCA_SSL_KEY_MEMBER            ),
    DEF_VALS1(MQCA_DNS_GROUP                 ),
    DEF_VALS1(MQCA_LU_GROUP_NAME             ),
    DEF_VALS1(MQCA_LU_NAME                   ),
    DEF_VALS1(MQCA_LU62_ARM_SUFFIX           ),
    DEF_VALS1(MQCA_TCP_NAME                  ),
    DEF_VALS1(MQCA_CHINIT_SERVICE_PARM       ),
    DEF_VALS1(MQCA_SERVICE_NAME              ),
    DEF_VALS1(MQCA_SERVICE_DESC              ),
    DEF_VALS1(MQCA_SERVICE_START_COMMAND     ),
    DEF_VALS1(MQCA_SERVICE_START_ARGS        ),
    DEF_VALS1(MQCA_SERVICE_STOP_COMMAND      ),
    DEF_VALS1(MQCA_SERVICE_STOP_ARGS         ),
    DEF_VALS1(MQCA_STDOUT_DESTINATION        ),
    DEF_VALS1(MQCA_STDERR_DESTINATION        ),
    DEF_VALS1(MQCA_TPIPE_NAME                ),
    DEF_VALS1(MQCA_PASS_TICKET_APPL          ),
    DEF_VALS1(MQCA_AUTO_REORG_START_TIME     ),
    DEF_VALS1(MQCA_AUTO_REORG_CATALOG        ),
    DEF_VALS1(MQCA_TOPIC_NAME                ),
    DEF_VALS1(MQCA_TOPIC_DESC                ),
    DEF_VALS1(MQCA_TOPIC_STRING              ),
    DEF_VALS1(MQCA_MODEL_DURABLE_Q           ),
    DEF_VALS1(MQCA_MODEL_NON_DURABLE_Q       ),
    DEF_VALS1(MQCA_RESUME_DATE               ),
    DEF_VALS1(MQCA_RESUME_TIME               ),
    DEF_VALS1(MQCA_CHILD                     ),
    DEF_VALS1(MQCA_PARENT                    ),
    DEF_VALS1(MQCA_ADMIN_TOPIC_NAME          ),
    DEF_VALS1(MQCA_TOPIC_STRING_FILTER       ),
    DEF_VALS1(MQCA_AUTH_INFO_OCSP_URL        ),
    DEF_VALS1(MQCA_COMM_INFO_NAME            ),
    DEF_VALS1(MQCA_COMM_INFO_DESC            ),
    DEF_VALS1(MQCA_POLICY_NAME               ),
    DEF_VALS1(MQCA_SIGNER_DN                 ),
    DEF_VALS1(MQCA_RECIPIENT_DN              ),
    DEF_VALS1(MQCA_INSTALLATION_DESC         ),
    DEF_VALS1(MQCA_INSTALLATION_NAME         ),
    DEF_VALS1(MQCA_INSTALLATION_PATH         ),
    DEF_VALS1(MQCA_CHLAUTH_DESC              ),
    DEF_VALS1(MQCA_CUSTOM                    ),
    DEF_VALS1(MQCA_VERSION                   ),
    DEF_VALS1(MQCA_XR_VERSION                ),
    DEF_VALS1(MQCA_XR_SSL_CIPHER_SUITES      ),
    DEF_VALS1(MQCA_CLUS_CHL_NAME             ),
DEF_VALSE;

DEF_VALSBX(mqcmd)
    DEF_VALS1(MQCMD_NONE                     ),
    DEF_VALS1(MQCMD_CHANGE_Q_MGR             ),
    DEF_VALS1(MQCMD_INQUIRE_Q_MGR            ),
    DEF_VALS1(MQCMD_CHANGE_PROCESS           ),
    DEF_VALS1(MQCMD_COPY_PROCESS             ),
    DEF_VALS1(MQCMD_CREATE_PROCESS           ),
    DEF_VALS1(MQCMD_DELETE_PROCESS           ),
    DEF_VALS1(MQCMD_INQUIRE_PROCESS          ),
    DEF_VALS1(MQCMD_CHANGE_Q                 ),
    DEF_VALS1(MQCMD_CLEAR_Q                  ),
    DEF_VALS1(MQCMD_COPY_Q                   ),
    DEF_VALS1(MQCMD_CREATE_Q                 ),
    DEF_VALS1(MQCMD_DELETE_Q                 ),
    DEF_VALS1(MQCMD_INQUIRE_Q                ),
    DEF_VALS1(MQCMD_REFRESH_Q_MGR            ),
    DEF_VALS1(MQCMD_RESET_Q_STATS            ),
    DEF_VALS1(MQCMD_INQUIRE_Q_NAMES          ),
    DEF_VALS1(MQCMD_INQUIRE_PROCESS_NAMES    ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_NAMES    ),
    DEF_VALS1(MQCMD_CHANGE_CHANNEL           ),
    DEF_VALS1(MQCMD_COPY_CHANNEL             ),
    DEF_VALS1(MQCMD_CREATE_CHANNEL           ),
    DEF_VALS1(MQCMD_DELETE_CHANNEL           ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL          ),
    DEF_VALS1(MQCMD_PING_CHANNEL             ),
    DEF_VALS1(MQCMD_RESET_CHANNEL            ),
    DEF_VALS1(MQCMD_START_CHANNEL            ),
    DEF_VALS1(MQCMD_STOP_CHANNEL             ),
    DEF_VALS1(MQCMD_START_CHANNEL_INIT       ),
    DEF_VALS1(MQCMD_START_CHANNEL_LISTENER   ),
    DEF_VALS1(MQCMD_CHANGE_NAMELIST          ),
    DEF_VALS1(MQCMD_COPY_NAMELIST            ),
    DEF_VALS1(MQCMD_CREATE_NAMELIST          ),
    DEF_VALS1(MQCMD_DELETE_NAMELIST          ),
    DEF_VALS1(MQCMD_INQUIRE_NAMELIST         ),
    DEF_VALS1(MQCMD_INQUIRE_NAMELIST_NAMES   ),
    DEF_VALS1(MQCMD_ESCAPE                   ),
    DEF_VALS1(MQCMD_RESOLVE_CHANNEL          ),
    DEF_VALS1(MQCMD_PING_Q_MGR               ),
    DEF_VALS1(MQCMD_INQUIRE_Q_STATUS         ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_STATUS   ),
    DEF_VALS1(MQCMD_CONFIG_EVENT             ),
    DEF_VALS1(MQCMD_Q_MGR_EVENT              ),
    DEF_VALS1(MQCMD_PERFM_EVENT              ),
    DEF_VALS1(MQCMD_CHANNEL_EVENT            ),
    DEF_VALS1(MQCMD_DELETE_PUBLICATION       ),
    DEF_VALS1(MQCMD_DEREGISTER_PUBLISHER     ),
    DEF_VALS1(MQCMD_DEREGISTER_SUBSCRIBER    ),
    DEF_VALS1(MQCMD_PUBLISH                  ),
    DEF_VALS1(MQCMD_REGISTER_PUBLISHER       ),
    DEF_VALS1(MQCMD_REGISTER_SUBSCRIBER      ),
    DEF_VALS1(MQCMD_REQUEST_UPDATE           ),
    DEF_VALS1(MQCMD_BROKER_INTERNAL          ),
    DEF_VALS1(MQCMD_ACTIVITY_MSG             ),
    DEF_VALS1(MQCMD_INQUIRE_CLUSTER_Q_MGR    ),
    DEF_VALS1(MQCMD_RESUME_Q_MGR_CLUSTER     ),
    DEF_VALS1(MQCMD_SUSPEND_Q_MGR_CLUSTER    ),
    DEF_VALS1(MQCMD_REFRESH_CLUSTER          ),
    DEF_VALS1(MQCMD_RESET_CLUSTER            ),
    DEF_VALS1(MQCMD_TRACE_ROUTE              ),
    DEF_VALS1(MQCMD_REFRESH_SECURITY         ),
    DEF_VALS1(MQCMD_CHANGE_AUTH_INFO         ),
    DEF_VALS1(MQCMD_COPY_AUTH_INFO           ),
    DEF_VALS1(MQCMD_CREATE_AUTH_INFO         ),
    DEF_VALS1(MQCMD_DELETE_AUTH_INFO         ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_INFO        ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_INFO_NAMES  ),
    DEF_VALS1(MQCMD_INQUIRE_CONNECTION       ),
    DEF_VALS1(MQCMD_STOP_CONNECTION          ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_RECS        ),
    DEF_VALS1(MQCMD_INQUIRE_ENTITY_AUTH      ),
    DEF_VALS1(MQCMD_DELETE_AUTH_REC          ),
    DEF_VALS1(MQCMD_SET_AUTH_REC             ),
    DEF_VALS1(MQCMD_LOGGER_EVENT             ),
    DEF_VALS1(MQCMD_RESET_Q_MGR              ),
    DEF_VALS1(MQCMD_CHANGE_LISTENER          ),
    DEF_VALS1(MQCMD_COPY_LISTENER            ),
    DEF_VALS1(MQCMD_CREATE_LISTENER          ),
    DEF_VALS1(MQCMD_DELETE_LISTENER          ),
    DEF_VALS1(MQCMD_INQUIRE_LISTENER         ),
    DEF_VALS1(MQCMD_INQUIRE_LISTENER_STATUS  ),
    DEF_VALS1(MQCMD_COMMAND_EVENT            ),
    DEF_VALS1(MQCMD_CHANGE_SECURITY          ),
    DEF_VALS1(MQCMD_CHANGE_CF_STRUC          ),
    DEF_VALS1(MQCMD_CHANGE_STG_CLASS         ),
    DEF_VALS1(MQCMD_CHANGE_TRACE             ),
    DEF_VALS1(MQCMD_ARCHIVE_LOG              ),
    DEF_VALS1(MQCMD_BACKUP_CF_STRUC          ),
    DEF_VALS1(MQCMD_CREATE_BUFFER_POOL       ),
    DEF_VALS1(MQCMD_CREATE_PAGE_SET          ),
    DEF_VALS1(MQCMD_CREATE_CF_STRUC          ),
    DEF_VALS1(MQCMD_CREATE_STG_CLASS         ),
    DEF_VALS1(MQCMD_COPY_CF_STRUC            ),
    DEF_VALS1(MQCMD_COPY_STG_CLASS           ),
    DEF_VALS1(MQCMD_DELETE_CF_STRUC          ),
    DEF_VALS1(MQCMD_DELETE_STG_CLASS         ),
    DEF_VALS1(MQCMD_INQUIRE_ARCHIVE          ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC         ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC_STATUS  ),
    DEF_VALS1(MQCMD_INQUIRE_CMD_SERVER       ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_INIT     ),
    DEF_VALS1(MQCMD_INQUIRE_QSG              ),
    DEF_VALS1(MQCMD_INQUIRE_LOG              ),
    DEF_VALS1(MQCMD_INQUIRE_SECURITY         ),
    DEF_VALS1(MQCMD_INQUIRE_STG_CLASS        ),
    DEF_VALS1(MQCMD_INQUIRE_SYSTEM           ),
    DEF_VALS1(MQCMD_INQUIRE_THREAD           ),
    DEF_VALS1(MQCMD_INQUIRE_TRACE            ),
    DEF_VALS1(MQCMD_INQUIRE_USAGE            ),
    DEF_VALS1(MQCMD_MOVE_Q                   ),
    DEF_VALS1(MQCMD_RECOVER_BSDS             ),
    DEF_VALS1(MQCMD_RECOVER_CF_STRUC         ),
    DEF_VALS1(MQCMD_RESET_TPIPE              ),
    DEF_VALS1(MQCMD_RESOLVE_INDOUBT          ),
    DEF_VALS1(MQCMD_RESUME_Q_MGR             ),
    DEF_VALS1(MQCMD_REVERIFY_SECURITY        ),
    DEF_VALS1(MQCMD_SET_ARCHIVE              ),
    DEF_VALS1(MQCMD_SET_LOG                  ),
    DEF_VALS1(MQCMD_SET_SYSTEM               ),
    DEF_VALS1(MQCMD_START_CMD_SERVER         ),
    DEF_VALS1(MQCMD_START_Q_MGR              ),
    DEF_VALS1(MQCMD_START_TRACE              ),
    DEF_VALS1(MQCMD_STOP_CHANNEL_INIT        ),
    DEF_VALS1(MQCMD_STOP_CHANNEL_LISTENER    ),
    DEF_VALS1(MQCMD_STOP_CMD_SERVER          ),
    DEF_VALS1(MQCMD_STOP_Q_MGR               ),
    DEF_VALS1(MQCMD_STOP_TRACE               ),
    DEF_VALS1(MQCMD_SUSPEND_Q_MGR            ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC_NAMES   ),
    DEF_VALS1(MQCMD_INQUIRE_STG_CLASS_NAMES  ),
    DEF_VALS1(MQCMD_CHANGE_SERVICE           ),
    DEF_VALS1(MQCMD_COPY_SERVICE             ),
    DEF_VALS1(MQCMD_CREATE_SERVICE           ),
    DEF_VALS1(MQCMD_DELETE_SERVICE           ),
    DEF_VALS1(MQCMD_INQUIRE_SERVICE          ),
    DEF_VALS1(MQCMD_INQUIRE_SERVICE_STATUS   ),
    DEF_VALS1(MQCMD_START_SERVICE            ),
    DEF_VALS1(MQCMD_STOP_SERVICE             ),
    DEF_VALS1(MQCMD_DELETE_BUFFER_POOL       ),
    DEF_VALS1(MQCMD_DELETE_PAGE_SET          ),
    DEF_VALS1(MQCMD_CHANGE_BUFFER_POOL       ),
    DEF_VALS1(MQCMD_CHANGE_PAGE_SET          ),
    DEF_VALS1(MQCMD_INQUIRE_Q_MGR_STATUS     ),
    DEF_VALS1(MQCMD_CREATE_LOG               ),
    DEF_VALS1(MQCMD_STATISTICS_MQI           ),
    DEF_VALS1(MQCMD_STATISTICS_Q             ),
    DEF_VALS1(MQCMD_STATISTICS_CHANNEL       ),
    DEF_VALS1(MQCMD_ACCOUNTING_MQI           ),
    DEF_VALS1(MQCMD_ACCOUNTING_Q             ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_SERVICE     ),
    DEF_VALS1(MQCMD_CHANGE_TOPIC             ),
    DEF_VALS1(MQCMD_COPY_TOPIC               ),
    DEF_VALS1(MQCMD_CREATE_TOPIC             ),
    DEF_VALS1(MQCMD_DELETE_TOPIC             ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC            ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC_NAMES      ),
    DEF_VALS1(MQCMD_INQUIRE_SUBSCRIPTION     ),
    DEF_VALS1(MQCMD_CREATE_SUBSCRIPTION      ),
    DEF_VALS1(MQCMD_CHANGE_SUBSCRIPTION      ),
    DEF_VALS1(MQCMD_DELETE_SUBSCRIPTION      ),
    DEF_VALS1(MQCMD_COPY_SUBSCRIPTION        ),
    DEF_VALS1(MQCMD_INQUIRE_SUB_STATUS       ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC_STATUS     ),
    DEF_VALS1(MQCMD_CLEAR_TOPIC_STRING       ),
    DEF_VALS1(MQCMD_INQUIRE_PUBSUB_STATUS    ),
    DEF_VALS1(MQCMD_INQUIRE_SMDS             ),
    DEF_VALS1(MQCMD_CHANGE_SMDS              ),
    DEF_VALS1(MQCMD_RESET_SMDS               ),
    DEF_VALS1(MQCMD_CREATE_COMM_INFO         ),
    DEF_VALS1(MQCMD_INQUIRE_COMM_INFO        ),
    DEF_VALS1(MQCMD_CHANGE_COMM_INFO         ),
    DEF_VALS1(MQCMD_COPY_COMM_INFO           ),
    DEF_VALS1(MQCMD_DELETE_COMM_INFO         ),
    DEF_VALS1(MQCMD_PURGE_CHANNEL            ),
    DEF_VALS1(MQCMD_MQXR_DIAGNOSTICS         ),
    DEF_VALS1(MQCMD_START_SMDSCONN           ),
    DEF_VALS1(MQCMD_STOP_SMDSCONN            ),
    DEF_VALS1(MQCMD_INQUIRE_SMDSCONN         ),
    DEF_VALS1(MQCMD_INQUIRE_MQXR_STATUS      ),
    DEF_VALS1(MQCMD_START_CLIENT_TRACE       ),
    DEF_VALS1(MQCMD_STOP_CLIENT_TRACE        ),
    DEF_VALS1(MQCMD_SET_CHLAUTH_REC          ),
    DEF_VALS1(MQCMD_INQUIRE_CHLAUTH_RECS     ),
    DEF_VALS1(MQCMD_INQUIRE_PROT_POLICY      ),
    DEF_VALS1(MQCMD_CREATE_PROT_POLICY       ),
    DEF_VALS1(MQCMD_DELETE_PROT_POLICY       ),
    DEF_VALS1(MQCMD_CHANGE_PROT_POLICY       ),
    DEF_VALS1(MQCMD_ACTIVITY_TRACE           ),
    DEF_VALS1(MQCMD_RESET_CF_STRUC           ),
    DEF_VALS1(MQCMD_INQUIRE_XR_CAPABILITY    ),
DEF_VALSE;

/* Types of Structure */
DEF_VALSBX(mqcft)
    DEF_VALS1(MQCFT_NONE                     ),
    DEF_VALS1(MQCFT_COMMAND                  ),
    DEF_VALS1(MQCFT_RESPONSE                 ),
    DEF_VALS1(MQCFT_INTEGER                  ),
    DEF_VALS1(MQCFT_STRING                   ),
    DEF_VALS1(MQCFT_INTEGER_LIST             ),
    DEF_VALS1(MQCFT_STRING_LIST              ),
    DEF_VALS1(MQCFT_EVENT                    ),
    DEF_VALS1(MQCFT_USER                     ),
    DEF_VALS1(MQCFT_BYTE_STRING              ),
    DEF_VALS1(MQCFT_TRACE_ROUTE              ),
    DEF_VALS1(MQCFT_REPORT                   ),
    DEF_VALS1(MQCFT_INTEGER_FILTER           ),
    DEF_VALS1(MQCFT_STRING_FILTER            ),
    DEF_VALS1(MQCFT_BYTE_STRING_FILTER       ),
    DEF_VALS1(MQCFT_COMMAND_XR               ),
    DEF_VALS1(MQCFT_XR_MSG                   ),
    DEF_VALS1(MQCFT_XR_ITEM                  ),
    DEF_VALS1(MQCFT_XR_SUMMARY               ),
    DEF_VALS1(MQCFT_GROUP                    ),
    DEF_VALS1(MQCFT_STATISTICS               ),
    DEF_VALS1(MQCFT_ACCOUNTING               ),
    DEF_VALS1(MQCFT_INTEGER64                ),
    DEF_VALS1(MQCFT_INTEGER64_LIST           ),
    DEF_VALS1(MQCFT_APP_ACTIVITY             ),
DEF_VALSE;

DEF_VALSBX(objtype)
    DEF_VALS1(MQOT_NONE             ),
    DEF_VALS1(MQOT_Q                ),
    DEF_VALS1(MQOT_NAMELIST         ),
    DEF_VALS1(MQOT_PROCESS          ),
    DEF_VALS1(MQOT_STORAGE_CLASS    ),
    DEF_VALS1(MQOT_Q_MGR            ),
    DEF_VALS1(MQOT_CHANNEL          ),
    DEF_VALS1(MQOT_AUTH_INFO        ),
    DEF_VALS1(MQOT_TOPIC            ),
    DEF_VALS1(MQOT_COMM_INFO        ),
    DEF_VALS1(MQOT_CF_STRUC         ),
    DEF_VALS1(MQOT_LISTENER         ),
    DEF_VALS1(MQOT_SERVICE          ),
    DEF_VALS1(MQOT_RESERVED_1       ),
    DEF_VALS1(MQOT_ALL              ),
    DEF_VALS1(MQOT_ALIAS_Q          ),
    DEF_VALS1(MQOT_MODEL_Q          ),
    DEF_VALS1(MQOT_LOCAL_Q          ),
    DEF_VALS1(MQOT_REMOTE_Q         ),
    DEF_VALS1(MQOT_SENDER_CHANNEL   ),
    DEF_VALS1(MQOT_SERVER_CHANNEL   ),
    DEF_VALS1(MQOT_REQUESTER_CHANNEL),
    DEF_VALS1(MQOT_RECEIVER_CHANNEL ),
    DEF_VALS1(MQOT_CURRENT_CHANNEL  ),
    DEF_VALS1(MQOT_SAVED_CHANNEL    ),
    DEF_VALS1(MQOT_SVRCONN_CHANNEL  ),
    DEF_VALS1(MQOT_CLNTCONN_CHANNEL ),
    DEF_VALS1(MQOT_SHORT_CHANNEL    ),
    DEF_VALS1(MQOT_CHLAUTH          ),
    DEF_VALS1(MQOT_REMOTE_Q_MGR_NAME),
    DEF_VALS1(MQOT_PROT_POLICY      ),
DEF_VALSE;

DEF_VALSB(AuthInfoType)
    DEF_VALS1(MQAIT_CRL_LDAP),
    DEF_VALS1(MQAIT_OCSP),
DEF_VALSE;

DEF_VALSBX(FilterOP)
    DEF_VALS1(MQCFOP_LESS        ),
    DEF_VALS1(MQCFOP_EQUAL       ),
    DEF_VALS1(MQCFOP_GREATER     ),
    DEF_VALS1(MQCFOP_NOT_LESS    ),
    DEF_VALS1(MQCFOP_NOT_EQUAL   ),
    DEF_VALS1(MQCFOP_NOT_GREATER ),
    DEF_VALS1(MQCFOP_LIKE        ),
    DEF_VALS1(MQCFOP_NOT_LIKE    ),
    DEF_VALS1(MQCFOP_CONTAINS    ),
    DEF_VALS1(MQCFOP_EXCLUDES    ),
    DEF_VALS1(MQCFOP_CONTAINS_GEN),
    DEF_VALS1(MQCFOP_EXCLUDES_GEN),
DEF_VALSE;

DEF_VALSB(PutResponse)
    DEF_VALS1(MQPRT_ASYNC_RESPONSE    ),
    DEF_VALS1(MQPRT_RESPONSE_AS_PARENT),
    DEF_VALS1(MQPRT_SYNC_RESPONSE     ),
DEF_VALSE;

DEF_VALSB(TraceRouteRecording)
    DEF_VALS1(MQRECORDING_MSG     ),
    DEF_VALS1(MQRECORDING_Q       ),
    DEF_VALS1(MQRECORDING_DISABLED),
DEF_VALSE;

DEF_VALSB(TCPStackType)
    DEF_VALS1(MQTCPSTACK_SINGLE  ),
    DEF_VALS1(MQTCPSTACK_MULTIPLE),
DEF_VALSE;

DEF_VALSB(TCPKeepAlive)
    DEF_VALS1(MQTCPKEEP_YES),
    DEF_VALS1(MQTCPKEEP_NO ),
DEF_VALSE;

DEF_VALSB(SharedQueueQueueManagerName)
    DEF_VALS1(MQSQQM_USE   ),
    DEF_VALS1(MQSQQM_IGNORE),
DEF_VALSE;

DEF_VALSB(ReceiveTimeoutType)
    DEF_VALS1(MQRCVTIME_MULTIPLY),
    DEF_VALS1(MQRCVTIME_ADD     ),
    DEF_VALS1(MQRCVTIME_EQUAL   ),
DEF_VALSE;

DEF_VALSB(IPAddressVersion)
    DEF_VALS1(MQIPADDR_IPV4),
    DEF_VALS1(MQIPADDR_IPV6),
DEF_VALSE;

DEF_VALSB(GroupUR)
    DEF_VALS1(MQGUR_DISABLED),
    DEF_VALS1(MQGUR_ENABLED ),
DEF_VALSE;

DEF_VALSB(DNSWLM)
    DEF_VALS1(MQDNSWLM_YES),
    DEF_VALS1(MQDNSWLM_NO ),
DEF_VALSE;

DEF_VALSB(AdoptNewMCAType)
    DEF_VALS1(MQADOPT_TYPE_NO ),
    DEF_VALS1(MQADOPT_TYPE_ALL),
DEF_VALSE;

DEF_VALSB(AdoptNewMCACheck)
    DEF_VALS1(MQADOPT_CHECK_Q_MGR_NAME),
    DEF_VALS1(MQADOPT_CHECK_NET_ADDR  ),
    DEF_VALS1(MQADOPT_CHECK_ALL       ),
    DEF_VALS1(MQADOPT_CHECK_NONE      ),
DEF_VALSE;

DEF_VALSB(ActivityRecording)
    DEF_VALS1(MQRECORDING_MSG     ),
    DEF_VALS1(MQRECORDING_Q       ),
    DEF_VALS1(MQRECORDING_DISABLED),
DEF_VALSE;

DEF_VALSB(TransportType)
    DEF_VALS1(MQXPT_LU62   ),
    DEF_VALS1(MQXPT_TCP    ),
    DEF_VALS1(MQXPT_NETBIOS),
    DEF_VALS1(MQXPT_SPX    ),
DEF_VALSE;

DEF_VALSB(SSLClientAuthentication)
    DEF_VALS1(MQSCA_REQUIRED),
    DEF_VALS1(MQSCA_OPTIONAL),
DEF_VALSE;


DEF_VALSB(PutAuthority)
    DEF_VALS1(MQPA_DEFAULT         ),
    DEF_VALS1(MQPA_CONTEXT         ),
    DEF_VALS1(MQPA_ALTERNATE_OR_MCA),
    DEF_VALS1(MQPA_ONLY_MCA        ),
DEF_VALSE;

DEF_VALSB(NonPersistentMsgSpeed)
    DEF_VALS1(MQNPMS_NORMAL),
    DEF_VALS1(MQNPMS_FAST  ),
DEF_VALSE;

DEF_VALSB(MessageCompression)
    DEF_VALS1(MQCOMPRESS_NONE          ),
    DEF_VALS1(MQCOMPRESS_RLE           ),
    DEF_VALS1(MQCOMPRESS_ZLIBFAST      ),
    DEF_VALS1(MQCOMPRESS_ZLIBHIGH      ),
    DEF_VALS1(MQCOMPRESS_ANY           ),
    DEF_VALS1(MQCOMPRESS_NOT_AVAILABLE ),
DEF_VALSE;

DEF_VALSB(HeaderCompression)
    DEF_VALS1(MQCOMPRESS_NONE          ),
    DEF_VALS1(MQCOMPRESS_SYSTEM        ),
    DEF_VALS1(MQCOMPRESS_NOT_AVAILABLE ),
DEF_VALSE;

#if 0
DEF_VALSB(MCAStatus)
    DEF_VALS1(MQMCAS_STOPPED),
    DEF_VALS1(MQMCAS_RUNNING),
DEF_VALSE;
#endif

DEF_VALSB(MCAType)
    DEF_VALS1(MQMCAT_PROCESS),
    DEF_VALS1(MQMCAT_THREAD ),
DEF_VALSE;

DEF_VALSB(DataConversion)
    DEF_VALS1(MQCDC_NO_SENDER_CONVERSION),
    DEF_VALS1(MQCDC_SENDER_CONVERSION   ),
DEF_VALSE;

DEF_VALSB(Usage)
    DEF_VALS1(MQUS_NORMAL      ),
    DEF_VALS1(MQUS_TRANSMISSION),
DEF_VALSE;

DEF_VALSB(ChannelType)
    DEF_VALS1(MQCHT_SENDER   ),
    DEF_VALS1(MQCHT_SERVER   ),
    DEF_VALS1(MQCHT_RECEIVER ),
    DEF_VALS1(MQCHT_REQUESTER),
    DEF_VALS1(MQCHT_SVRCONN  ),
    DEF_VALS1(MQCHT_CLNTCONN ),
    DEF_VALS1(MQCHT_CLUSRCVR ),
    DEF_VALS1(MQCHT_CLUSSDR  ),
DEF_VALSE;

DEF_VALSB(ServiceIntervalEvent)
    DEF_VALS1(MQQSIE_NONE),
    DEF_VALS1(MQQSIE_OK  ),
    DEF_VALS1(MQQSIE_HIGH),
DEF_VALSE;

DEF_VALSB(Monitoring)
    DEF_VALS1(MQMON_NONE  ),
    DEF_VALS2(MQMON_ON    ,"ON/ENABLED"),
    DEF_VALS2(MQMON_OFF   ,"OFF/DISABLED"),
    DEF_VALS1(MQMON_LOW   ),
    DEF_VALS1(MQMON_MEDIUM),
    DEF_VALS1(MQMON_HIGH  ),
    DEF_VALS1(MQMON_Q_MGR ),
DEF_VALSE;

DEF_VALSB(QueueType)
    DEF_VALS1(MQQT_LOCAL ),
    DEF_VALS1(MQQT_MODEL ),
    DEF_VALS1(MQQT_ALIAS ),
    DEF_VALS1(MQQT_REMOTE),
    DEF_VALS1(MQQT_CLUSTER),
    DEF_VALS1(MQQT_ALL),
DEF_VALSE;

DEF_VALSB(EventRule)
    DEF_VALS1(MQEVR_ENABLED   ),
    DEF_VALS1(MQEVR_DISABLED  ),
    DEF_VALS1(MQEVR_EXCEPTION ),
    DEF_VALS1(MQEVR_NO_DISPLAY),
DEF_VALSE;

DEF_VALSB(InputOpenOption)
    DEF_VALS1(MQOO_INPUT_EXCLUSIVE),
    DEF_VALS1(MQOO_INPUT_SHARED   ),
DEF_VALSE;

DEF_VALSB(CFConLoss)
    DEF_VALS1(MQCFCONLOS_TERMINATE),
    DEF_VALS1(MQCFCONLOS_TOLERATE ),
DEF_VALSE;

DEF_VALSB(CLWLUseQ)
    DEF_VALS1(MQCLWL_USEQ_ANY     ),
    DEF_VALS1(MQCLWL_USEQ_LOCAL   ),
    DEF_VALS1(MQCLWL_USEQ_AS_Q_MGR),
DEF_VALSE;

DEF_VALSB(DefinitionType)
    DEF_VALS1(MQQDT_PREDEFINED       ),
    DEF_VALS1(MQQDT_PERMANENT_DYNAMIC),
    DEF_VALS1(MQQDT_TEMPORARY_DYNAMIC),
    DEF_VALS1(MQQDT_SHARED_DYNAMIC   ),
DEF_VALSE;

DEF_VALSB(InhibitGetVal)
    DEF_VALS1(MQQA_GET_INHIBITED),
    DEF_VALS1(MQQA_GET_ALLOWED  ),
DEF_VALSE;

DEF_VALSB(IntraGroupQueuing)
    DEF_VALS1(MQIGQ_DISABLED),
    DEF_VALS1(MQIGQ_ENABLED),
DEF_VALSE;

DEF_VALSB(InhibitPutVal)
    DEF_VALS1(MQQA_PUT_INHIBITED),
    DEF_VALS1(MQQA_PUT_ALLOWED  ),
DEF_VALSE;

DEF_VALSB(Shareability)
    DEF_VALS1(MQQA_SHAREABLE    ),
    DEF_VALS1(MQQA_NOT_SHAREABLE),
DEF_VALSE;

DEF_VALSB(BackOutHardening)
    DEF_VALS1(MQQA_BACKOUT_HARDENED    ),
    DEF_VALS1(MQQA_BACKOUT_NOT_HARDENED),
DEF_VALSE;

DEF_VALSB(MsgDelivSeq)
    DEF_VALS1(MQMDS_PRIORITY),
    DEF_VALS1(MQMDS_FIFO    ),
DEF_VALSE;

DEF_VALSB(NonpersistentMsgCls)
    DEF_VALS1(MQNPM_CLASS_NORMAL),
    DEF_VALS1(MQNPM_CLASS_HIGH  ),
DEF_VALSE;

DEF_VALSB(TriggerControl)
    DEF_VALS1(MQTC_OFF),
    DEF_VALS1(MQTC_ON ),
DEF_VALSE;

DEF_VALSB(TriggerType)
    DEF_VALS1(MQTT_NONE ),
    DEF_VALS1(MQTT_FIRST),
    DEF_VALS1(MQTT_EVERY),
    DEF_VALS1(MQTT_DEPTH),
DEF_VALSE;

#if 0
DEF_VALSB(TriggerRestart)
    DEF_VALS1(MQTRIGGER_RESTART_NO ),
    DEF_VALS1(MQTRIGGER_RESTART_YES),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueUsages)
    DEF_VALS1(MQUS_NORMAL      ),
    DEF_VALS1(MQUS_TRANSMISSION),
DEF_VALSE;
#endif

DEF_VALSB(DistributionLists)
    DEF_VALS1(MQDL_SUPPORTED    ),
    DEF_VALS1(MQDL_NOT_SUPPORTED),
DEF_VALSE;

DEF_VALSB(IndexTypes)
    DEF_VALS1(MQIT_NONE     ),
    DEF_VALS1(MQIT_MSG_ID   ),
    DEF_VALS1(MQIT_CORREL_ID),
    DEF_VALS1(MQIT_MSG_TOKEN),
    DEF_VALS1(MQIT_GROUP_ID ),
DEF_VALSE;

DEF_VALSB(Bindings)
    DEF_VALS1(MQBND_BIND_ON_OPEN  ),
    DEF_VALS1(MQBND_BIND_NOT_FIXED),
DEF_VALSE;

DEF_VALSB(QShrGrpDisp)
    DEF_VALS1(MQQSGD_ALL    ),
    DEF_VALS1(MQQSGD_Q_MGR  ),
    DEF_VALS1(MQQSGD_COPY   ),
    DEF_VALS1(MQQSGD_SHARED ),
    DEF_VALS1(MQQSGD_GROUP  ),
    DEF_VALS1(MQQSGD_PRIVATE),
    DEF_VALS1(MQQSGD_LIVE   ),
DEF_VALSE;

#if 0
DEF_VALSB(ReorgCtls)
    DEF_VALS1(MQREORG_DISABLED),
    DEF_VALS1(MQREORG_ENABLED),
DEF_VALSE;
#endif

DEF_VALSB(ReadAheadValues)
    DEF_VALS1(MQREADA_NO       ),
    DEF_VALS1(MQREADA_YES      ),
    DEF_VALS1(MQREADA_DISABLED ),
    DEF_VALS1(MQREADA_INHIBITED),
    DEF_VALS1(MQREADA_BACKLOG  ),
DEF_VALSE;

DEF_VALSB(PropertyCtlVal)
    DEF_VALS1(MQPROP_COMPATIBILITY),
    DEF_VALS1(MQPROP_NONE),
    DEF_VALS1(MQPROP_ALL),
    DEF_VALS1(MQPROP_FORCE_MQRFH2),
DEF_VALSE;

DEF_VALSB(BaseType)
    DEF_VALS1(MQOT_Q    ),
    DEF_VALS1(MQOT_TOPIC),
DEF_VALSE;

DEF_VALSB(EvtOrig)
    DEF_VALS1(MQEVO_CONSOLE ),
    DEF_VALS1(MQEVO_INIT    ),
    DEF_VALS1(MQEVO_INTERNAL),
    DEF_VALS1(MQEVO_MQSET   ),
    DEF_VALS1(MQEVO_MSG     ),
    DEF_VALS1(MQEVO_OTHER   ),
DEF_VALSE;

DEF_VALSB(ObjType)
    DEF_VALS1(MQOT_NONE             ),
    DEF_VALS1(MQOT_Q                ),
    DEF_VALS1(MQOT_NAMELIST         ),
    DEF_VALS1(MQOT_PROCESS          ),
    DEF_VALS1(MQOT_STORAGE_CLASS    ),
    DEF_VALS1(MQOT_Q_MGR            ),
    DEF_VALS1(MQOT_CHANNEL          ),
    DEF_VALS1(MQOT_AUTH_INFO        ),
    DEF_VALS1(MQOT_TOPIC            ),
    DEF_VALS1(MQOT_COMM_INFO        ),
    DEF_VALS1(MQOT_CF_STRUC         ),
    DEF_VALS1(MQOT_LISTENER         ),
    DEF_VALS1(MQOT_SERVICE          ),
    DEF_VALS1(MQOT_RESERVED_1       ),
    DEF_VALS1(MQOT_ALL              ),
    DEF_VALS1(MQOT_ALIAS_Q          ),
    DEF_VALS1(MQOT_MODEL_Q          ),
    DEF_VALS1(MQOT_LOCAL_Q          ),
    DEF_VALS1(MQOT_REMOTE_Q         ),
    DEF_VALS1(MQOT_SENDER_CHANNEL   ),
    DEF_VALS1(MQOT_SERVER_CHANNEL   ),
    DEF_VALS1(MQOT_REQUESTER_CHANNEL),
    DEF_VALS1(MQOT_RECEIVER_CHANNEL ),
    DEF_VALS1(MQOT_CURRENT_CHANNEL  ),
    DEF_VALS1(MQOT_SAVED_CHANNEL    ),
    DEF_VALS1(MQOT_SVRCONN_CHANNEL  ),
    DEF_VALS1(MQOT_CLNTCONN_CHANNEL ),
    DEF_VALS1(MQOT_SHORT_CHANNEL    ),
    DEF_VALS1(MQOT_CHLAUTH          ),
    DEF_VALS1(MQOT_REMOTE_Q_MGR_NAME),
    DEF_VALS1(MQOT_PROT_POLICY      ),
DEF_VALSE;

DEF_VALSB(Persistence)
    DEF_VALS1(MQPER_PERSISTENCE_AS_PARENT),
    DEF_VALS1(MQPER_NOT_PERSISTENT),
    DEF_VALS1(MQPER_PERSISTENT),
    DEF_VALS1(MQPER_PERSISTENCE_AS_Q_DEF),
DEF_VALSE;

#if 0
DEF_VALSB(MsgType)
    DEF_VALS1(MQMT_REQUEST),
    DEF_VALS1(MQMT_REPLY),
    DEF_VALS1(MQMT_DATAGRAM),
    DEF_VALS1(MQMT_REPORT),
DEF_VALSE;
#endif

DEF_VALSB(Platform)
    DEF_VALS2(MQPL_ZOS,"MQPL_MVS/MQPL_OS390"),
    DEF_VALS1(MQPL_OS2       ),
    DEF_VALS1(MQPL_UNIX      ),
    DEF_VALS1(MQPL_OS400     ),
    DEF_VALS1(MQPL_WINDOWS   ),
    DEF_VALS1(MQPL_WINDOWS_NT),
    DEF_VALS1(MQPL_VMS       ),
    DEF_VALS2(MQPL_NSK,"MQPL_NSS"),
    DEF_VALS1(MQPL_OPEN_TP1  ),
    DEF_VALS1(MQPL_VM        ),
    DEF_VALS1(MQPL_TPF       ),
    DEF_VALS1(MQPL_VSE       ),
DEF_VALSE;

DEF_VALSB(ApplType)
    DEF_VALS1(MQAT_UNKNOWN              ),
    DEF_VALS1(MQAT_NO_CONTEXT           ),
    DEF_VALS1(MQAT_CICS                 ),
    DEF_VALS2(MQAT_ZOS,"MQAT_MVS/MQAT_OS390"),
    DEF_VALS1(MQAT_IMS                  ),
    DEF_VALS1(MQAT_OS2                  ),
    DEF_VALS1(MQAT_DOS                  ),
    DEF_VALS2(MQAT_AIX,"MQAT_UNIX"),
    DEF_VALS1(MQAT_QMGR                 ),
    DEF_VALS1(MQAT_OS400                ),
    DEF_VALS1(MQAT_WINDOWS              ),
    DEF_VALS1(MQAT_CICS_VSE             ),
    DEF_VALS1(MQAT_WINDOWS_NT           ),
    DEF_VALS1(MQAT_VMS                  ),
    DEF_VALS2(MQAT_NSK,"MQAT_GUARDIAN"),
    DEF_VALS1(MQAT_VOS                  ),
    DEF_VALS1(MQAT_OPEN_TP1             ),
    DEF_VALS1(MQAT_VM                   ),
    DEF_VALS1(MQAT_IMS_BRIDGE           ),
    DEF_VALS1(MQAT_XCF                  ),
    DEF_VALS1(MQAT_CICS_BRIDGE          ),
    DEF_VALS1(MQAT_NOTES_AGENT          ),
    DEF_VALS1(MQAT_TPF                  ),
    DEF_VALS1(MQAT_USER                 ),
    DEF_VALS2(MQAT_QMGR_PUBLISH,"MQAT_BROKER"),
    DEF_VALS1(MQAT_JAVA                 ),
    DEF_VALS1(MQAT_DQM                  ),
    DEF_VALS1(MQAT_CHANNEL_INITIATOR    ),
    DEF_VALS1(MQAT_WLM                  ),
    DEF_VALS1(MQAT_BATCH                ),
    DEF_VALS1(MQAT_RRS_BATCH            ),
    DEF_VALS1(MQAT_SIB                  ),
    DEF_VALS1(MQAT_SYSTEM_EXTENSION     ),
DEF_VALSE;

DEF_VALSBX(PrmTyp)
    DEF_VALS1(MQCFT_NONE              ),
    DEF_VALS1(MQCFT_COMMAND           ),
    DEF_VALS1(MQCFT_RESPONSE          ),
    DEF_VALS1(MQCFT_INTEGER           ),
    DEF_VALS1(MQCFT_STRING            ),
    DEF_VALS1(MQCFT_INTEGER_LIST      ),
    DEF_VALS1(MQCFT_STRING_LIST       ),
    DEF_VALS1(MQCFT_EVENT             ),
    DEF_VALS1(MQCFT_USER              ),
    DEF_VALS1(MQCFT_BYTE_STRING       ),
    DEF_VALS1(MQCFT_TRACE_ROUTE       ),
    DEF_VALS1(MQCFT_REPORT            ),
    DEF_VALS1(MQCFT_INTEGER_FILTER    ),
    DEF_VALS1(MQCFT_STRING_FILTER     ),
    DEF_VALS1(MQCFT_BYTE_STRING_FILTER),
    DEF_VALS1(MQCFT_COMMAND_XR        ),
    DEF_VALS1(MQCFT_XR_MSG            ),
    DEF_VALS1(MQCFT_XR_ITEM           ),
    DEF_VALS1(MQCFT_XR_SUMMARY        ),
    DEF_VALS1(MQCFT_GROUP             ),
    DEF_VALS1(MQCFT_STATISTICS        ),
    DEF_VALS1(MQCFT_ACCOUNTING        ),
    DEF_VALS1(MQCFT_INTEGER64         ),
    DEF_VALS1(MQCFT_INTEGER64_LIST    ),
DEF_VALSE;

DEF_VALSB(Command)
    DEF_VALS1(MQCMD_NONE                   ),
    DEF_VALS1(MQCMD_CHANGE_Q_MGR           ),
    DEF_VALS1(MQCMD_INQUIRE_Q_MGR          ),
    DEF_VALS1(MQCMD_CHANGE_PROCESS         ),
    DEF_VALS1(MQCMD_COPY_PROCESS           ),
    DEF_VALS1(MQCMD_CREATE_PROCESS         ),
    DEF_VALS1(MQCMD_DELETE_PROCESS         ),
    DEF_VALS1(MQCMD_INQUIRE_PROCESS        ),
    DEF_VALS1(MQCMD_CHANGE_Q               ),
    DEF_VALS1(MQCMD_CLEAR_Q                ),
    DEF_VALS1(MQCMD_COPY_Q                 ),
    DEF_VALS1(MQCMD_CREATE_Q               ),
    DEF_VALS1(MQCMD_DELETE_Q               ),
    DEF_VALS1(MQCMD_INQUIRE_Q              ),
    DEF_VALS1(MQCMD_REFRESH_Q_MGR          ),
    DEF_VALS1(MQCMD_RESET_Q_STATS          ),
    DEF_VALS1(MQCMD_INQUIRE_Q_NAMES        ),
    DEF_VALS1(MQCMD_INQUIRE_PROCESS_NAMES  ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_NAMES  ),
    DEF_VALS1(MQCMD_CHANGE_CHANNEL         ),
    DEF_VALS1(MQCMD_COPY_CHANNEL           ),
    DEF_VALS1(MQCMD_CREATE_CHANNEL         ),
    DEF_VALS1(MQCMD_DELETE_CHANNEL         ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL        ),
    DEF_VALS1(MQCMD_PING_CHANNEL           ),
    DEF_VALS1(MQCMD_RESET_CHANNEL          ),
    DEF_VALS1(MQCMD_START_CHANNEL          ),
    DEF_VALS1(MQCMD_STOP_CHANNEL           ),
    DEF_VALS1(MQCMD_START_CHANNEL_INIT     ),
    DEF_VALS1(MQCMD_START_CHANNEL_LISTENER ),
    DEF_VALS1(MQCMD_CHANGE_NAMELIST        ),
    DEF_VALS1(MQCMD_COPY_NAMELIST          ),
    DEF_VALS1(MQCMD_CREATE_NAMELIST        ),
    DEF_VALS1(MQCMD_DELETE_NAMELIST        ),
    DEF_VALS1(MQCMD_INQUIRE_NAMELIST       ),
    DEF_VALS1(MQCMD_INQUIRE_NAMELIST_NAMES ),
    DEF_VALS1(MQCMD_ESCAPE                 ),
    DEF_VALS1(MQCMD_RESOLVE_CHANNEL        ),
    DEF_VALS1(MQCMD_PING_Q_MGR             ),
    DEF_VALS1(MQCMD_INQUIRE_Q_STATUS       ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_STATUS ),
    DEF_VALS1(MQCMD_CONFIG_EVENT           ),
    DEF_VALS1(MQCMD_Q_MGR_EVENT            ),
    DEF_VALS1(MQCMD_PERFM_EVENT            ),
    DEF_VALS1(MQCMD_CHANNEL_EVENT          ),
    DEF_VALS1(MQCMD_DELETE_PUBLICATION     ),
    DEF_VALS1(MQCMD_DEREGISTER_PUBLISHER   ),
    DEF_VALS1(MQCMD_DEREGISTER_SUBSCRIBER  ),
    DEF_VALS1(MQCMD_PUBLISH                ),
    DEF_VALS1(MQCMD_REGISTER_PUBLISHER     ),
    DEF_VALS1(MQCMD_REGISTER_SUBSCRIBER    ),
    DEF_VALS1(MQCMD_REQUEST_UPDATE         ),
    DEF_VALS1(MQCMD_BROKER_INTERNAL        ),
    DEF_VALS1(MQCMD_ACTIVITY_MSG           ),
    DEF_VALS1(MQCMD_INQUIRE_CLUSTER_Q_MGR  ),
    DEF_VALS1(MQCMD_RESUME_Q_MGR_CLUSTER   ),
    DEF_VALS1(MQCMD_SUSPEND_Q_MGR_CLUSTER  ),
    DEF_VALS1(MQCMD_REFRESH_CLUSTER        ),
    DEF_VALS1(MQCMD_RESET_CLUSTER          ),
    DEF_VALS1(MQCMD_TRACE_ROUTE            ),
    DEF_VALS1(MQCMD_REFRESH_SECURITY       ),
    DEF_VALS1(MQCMD_CHANGE_AUTH_INFO       ),
    DEF_VALS1(MQCMD_COPY_AUTH_INFO         ),
    DEF_VALS1(MQCMD_CREATE_AUTH_INFO       ),
    DEF_VALS1(MQCMD_DELETE_AUTH_INFO       ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_INFO      ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_INFO_NAMES),
    DEF_VALS1(MQCMD_INQUIRE_CONNECTION     ),
    DEF_VALS1(MQCMD_STOP_CONNECTION        ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_RECS      ),
    DEF_VALS1(MQCMD_INQUIRE_ENTITY_AUTH    ),
    DEF_VALS1(MQCMD_DELETE_AUTH_REC        ),
    DEF_VALS1(MQCMD_SET_AUTH_REC           ),
    DEF_VALS1(MQCMD_LOGGER_EVENT           ),
    DEF_VALS1(MQCMD_RESET_Q_MGR            ),
    DEF_VALS1(MQCMD_CHANGE_LISTENER        ),
    DEF_VALS1(MQCMD_COPY_LISTENER          ),
    DEF_VALS1(MQCMD_CREATE_LISTENER        ),
    DEF_VALS1(MQCMD_DELETE_LISTENER        ),
    DEF_VALS1(MQCMD_INQUIRE_LISTENER       ),
    DEF_VALS1(MQCMD_INQUIRE_LISTENER_STATUS),
    DEF_VALS1(MQCMD_COMMAND_EVENT          ),
    DEF_VALS1(MQCMD_CHANGE_SECURITY        ),
    DEF_VALS1(MQCMD_CHANGE_CF_STRUC        ),
    DEF_VALS1(MQCMD_CHANGE_STG_CLASS       ),
    DEF_VALS1(MQCMD_CHANGE_TRACE           ),
    DEF_VALS1(MQCMD_ARCHIVE_LOG            ),
    DEF_VALS1(MQCMD_BACKUP_CF_STRUC        ),
    DEF_VALS1(MQCMD_CREATE_BUFFER_POOL     ),
    DEF_VALS1(MQCMD_CREATE_PAGE_SET        ),
    DEF_VALS1(MQCMD_CREATE_CF_STRUC        ),
    DEF_VALS1(MQCMD_CREATE_STG_CLASS       ),
    DEF_VALS1(MQCMD_COPY_CF_STRUC          ),
    DEF_VALS1(MQCMD_COPY_STG_CLASS         ),
    DEF_VALS1(MQCMD_DELETE_CF_STRUC        ),
    DEF_VALS1(MQCMD_DELETE_STG_CLASS       ),
    DEF_VALS1(MQCMD_INQUIRE_ARCHIVE        ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC       ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC_STATUS),
    DEF_VALS1(MQCMD_INQUIRE_CMD_SERVER     ),
    DEF_VALS1(MQCMD_INQUIRE_CHANNEL_INIT   ),
    DEF_VALS1(MQCMD_INQUIRE_QSG            ),
    DEF_VALS1(MQCMD_INQUIRE_LOG            ),
    DEF_VALS1(MQCMD_INQUIRE_SECURITY       ),
    DEF_VALS1(MQCMD_INQUIRE_STG_CLASS      ),
    DEF_VALS1(MQCMD_INQUIRE_SYSTEM         ),
    DEF_VALS1(MQCMD_INQUIRE_THREAD         ),
    DEF_VALS1(MQCMD_INQUIRE_TRACE          ),
    DEF_VALS1(MQCMD_INQUIRE_USAGE          ),
    DEF_VALS1(MQCMD_MOVE_Q                 ),
    DEF_VALS1(MQCMD_RECOVER_BSDS           ),
    DEF_VALS1(MQCMD_RECOVER_CF_STRUC       ),
    DEF_VALS1(MQCMD_RESET_TPIPE            ),
    DEF_VALS1(MQCMD_RESOLVE_INDOUBT        ),
    DEF_VALS1(MQCMD_RESUME_Q_MGR           ),
    DEF_VALS1(MQCMD_REVERIFY_SECURITY      ),
    DEF_VALS1(MQCMD_SET_ARCHIVE            ),
    DEF_VALS1(MQCMD_SET_LOG                ),
    DEF_VALS1(MQCMD_SET_SYSTEM             ),
    DEF_VALS1(MQCMD_START_CMD_SERVER       ),
    DEF_VALS1(MQCMD_START_Q_MGR            ),
    DEF_VALS1(MQCMD_START_TRACE            ),
    DEF_VALS1(MQCMD_STOP_CHANNEL_INIT      ),
    DEF_VALS1(MQCMD_STOP_CHANNEL_LISTENER  ),
    DEF_VALS1(MQCMD_STOP_CMD_SERVER        ),
    DEF_VALS1(MQCMD_STOP_Q_MGR             ),
    DEF_VALS1(MQCMD_STOP_TRACE             ),
    DEF_VALS1(MQCMD_SUSPEND_Q_MGR          ),
    DEF_VALS1(MQCMD_INQUIRE_CF_STRUC_NAMES ),
    DEF_VALS1(MQCMD_INQUIRE_STG_CLASS_NAMES),
    DEF_VALS1(MQCMD_CHANGE_SERVICE         ),
    DEF_VALS1(MQCMD_COPY_SERVICE           ),
    DEF_VALS1(MQCMD_CREATE_SERVICE         ),
    DEF_VALS1(MQCMD_DELETE_SERVICE         ),
    DEF_VALS1(MQCMD_INQUIRE_SERVICE        ),
    DEF_VALS1(MQCMD_INQUIRE_SERVICE_STATUS ),
    DEF_VALS1(MQCMD_START_SERVICE          ),
    DEF_VALS1(MQCMD_STOP_SERVICE           ),
    DEF_VALS1(MQCMD_DELETE_BUFFER_POOL     ),
    DEF_VALS1(MQCMD_DELETE_PAGE_SET        ),
    DEF_VALS1(MQCMD_CHANGE_BUFFER_POOL     ),
    DEF_VALS1(MQCMD_CHANGE_PAGE_SET        ),
    DEF_VALS1(MQCMD_INQUIRE_Q_MGR_STATUS   ),
    DEF_VALS1(MQCMD_CREATE_LOG             ),
    DEF_VALS1(MQCMD_STATISTICS_MQI         ),
    DEF_VALS1(MQCMD_STATISTICS_Q           ),
    DEF_VALS1(MQCMD_STATISTICS_CHANNEL     ),
    DEF_VALS1(MQCMD_ACCOUNTING_MQI         ),
    DEF_VALS1(MQCMD_ACCOUNTING_Q           ),
    DEF_VALS1(MQCMD_INQUIRE_AUTH_SERVICE   ),
    DEF_VALS1(MQCMD_CHANGE_TOPIC           ),
    DEF_VALS1(MQCMD_COPY_TOPIC             ),
    DEF_VALS1(MQCMD_CREATE_TOPIC           ),
    DEF_VALS1(MQCMD_DELETE_TOPIC           ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC          ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC_NAMES    ),
    DEF_VALS1(MQCMD_INQUIRE_SUBSCRIPTION   ),
    DEF_VALS1(MQCMD_CREATE_SUBSCRIPTION    ),
    DEF_VALS1(MQCMD_CHANGE_SUBSCRIPTION    ),
    DEF_VALS1(MQCMD_DELETE_SUBSCRIPTION    ),
    DEF_VALS1(MQCMD_COPY_SUBSCRIPTION      ),
    DEF_VALS1(MQCMD_INQUIRE_SUB_STATUS     ),
    DEF_VALS1(MQCMD_INQUIRE_TOPIC_STATUS   ),
    DEF_VALS1(MQCMD_CLEAR_TOPIC_STRING     ),
    DEF_VALS1(MQCMD_INQUIRE_PUBSUB_STATUS  ),
DEF_VALSE;

#if 0
DEF_VALSB(CommandInformationValues)
    DEF_VALS1(MQCMDI_CMDSCOPE_ACCEPTED      ),
    DEF_VALS1(MQCMDI_CMDSCOPE_GENERATED     ),
    DEF_VALS1(MQCMDI_CMDSCOPE_COMPLETED     ),
    DEF_VALS1(MQCMDI_QSG_DISP_COMPLETED     ),
    DEF_VALS1(MQCMDI_COMMAND_ACCEPTED       ),
    DEF_VALS1(MQCMDI_CLUSTER_REQUEST_QUEUED ),
    DEF_VALS1(MQCMDI_CHANNEL_INIT_STARTED   ),
    DEF_VALS1(MQCMDI_RECOVER_STARTED        ),
    DEF_VALS1(MQCMDI_BACKUP_STARTED         ),
    DEF_VALS1(MQCMDI_RECOVER_COMPLETED      ),
    DEF_VALS1(MQCMDI_SEC_TIMER_ZERO         ),
    DEF_VALS1(MQCMDI_REFRESH_CONFIGURATION  ),
    DEF_VALS1(MQCMDI_SEC_SIGNOFF_ERROR      ),
    DEF_VALS1(MQCMDI_IMS_BRIDGE_SUSPENDED   ),
    DEF_VALS1(MQCMDI_DB2_SUSPENDED          ),
    DEF_VALS1(MQCMDI_DB2_OBSOLETE_MSGS      ),
    DEF_VALS1(MQCMDI_SEC_UPPERCASE          ),
    DEF_VALS1(MQCMDI_SEC_MIXEDCASE          ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(CtlOpt)
    DEF_VALS1(MQCFC_LAST    ),
    DEF_VALS1(MQCFC_NOT_LAST),
DEF_VALSE;
#endif

DEF_VALSBX(PrmId)
    DEF_VALS1(MQBACF_ACCOUNTING_TOKEN       ),
    DEF_VALS1(MQBACF_CF_LEID                ),
    DEF_VALS1(MQBACF_CONNECTION_ID          ),
    DEF_VALS1(MQBACF_CORREL_ID              ),
    DEF_VALS1(MQBACF_DESTINATION_CORREL_ID  ),
    DEF_VALS1(MQBACF_EVENT_ACCOUNTING_TOKEN ),
    DEF_VALS1(MQBACF_EVENT_SECURITY_ID      ),
    DEF_VALS1(MQBACF_EXTERNAL_UOW_ID        ),
    DEF_VALS1(MQBACF_GENERIC_CONNECTION_ID  ),
    DEF_VALS1(MQBACF_GROUP_ID               ),
    DEF_VALS1(MQBACF_MSG_ID                 ),
    DEF_VALS1(MQBACF_ORIGIN_UOW_ID          ),
    DEF_VALS1(MQBACF_Q_MGR_UOW_ID           ),
    DEF_VALS1(MQBACF_RESPONSE_ID            ),
    DEF_VALS1(MQBACF_RESPONSE_SET           ),
    DEF_VALS1(MQBACF_SUB_ID                 ),
    DEF_VALS1(MQCACF_ACTIVITY_DESC          ),
    DEF_VALS1(MQCACF_ADMIN_TOPIC_NAMES      ),
    DEF_VALS1(MQCACF_ADMIN_TOPIC_NAMES      ),
    DEF_VALS1(MQCACF_ALIAS_Q_NAMES          ),
    DEF_VALS1(MQCACF_APPL_DESC              ),
    DEF_VALS1(MQCACF_APPL_IDENTITY_DATA     ),
    DEF_VALS1(MQCACF_APPL_NAME              ),
    DEF_VALS1(MQCACF_APPL_ORIGIN_DATA       ),
    DEF_VALS1(MQCACF_APPL_TAG               ),
    DEF_VALS1(MQCACF_ASID                   ),
    DEF_VALS1(MQCACF_AUTH_INFO_NAMES        ),
    DEF_VALS1(MQCACF_AUTH_PROFILE_NAME      ),
    DEF_VALS1(MQCACF_AUX_ERROR_DATA_STR_1   ),
    DEF_VALS1(MQCACF_AUX_ERROR_DATA_STR_2   ),
    DEF_VALS1(MQCACF_AUX_ERROR_DATA_STR_3   ),
    DEF_VALS1(MQCACF_BACKUP_DATE            ),
    DEF_VALS1(MQCACF_BACKUP_TIME            ),
    DEF_VALS1(MQCACF_BRIDGE_NAME            ),
    DEF_VALS1(MQCACF_CF_STRUC_BACKUP_END    ),
    DEF_VALS1(MQCACF_CF_STRUC_BACKUP_START  ),
    DEF_VALS1(MQCACF_CF_STRUC_LOG_Q_MGRS    ),
    DEF_VALS1(MQCACF_CF_STRUC_NAMES         ),
    DEF_VALS1(MQCACF_CHILD_Q_MGR_NAME       ),
    DEF_VALS1(MQCACF_COMMAND_MQSC           ),
    DEF_VALS1(MQCACF_COMMAND_SCOPE          ),
    DEF_VALS1(MQCACF_CONFIGURATION_DATE     ),
    DEF_VALS1(MQCACF_CONFIGURATION_TIME     ),
    DEF_VALS1(MQCACF_CORREL_ID              ),
    DEF_VALS1(MQCACF_CURRENT_LOG_EXTENT_NAME),
    DEF_VALS1(MQCACF_DATA_SET_NAME          ),
    DEF_VALS1(MQCACF_DB2_NAME               ),
    DEF_VALS1(MQCACF_DESTINATION            ),
    DEF_VALS1(MQCACF_DESTINATION_Q_MGR      ),
    DEF_VALS1(MQCACF_DSG_NAME               ),
    DEF_VALS1(MQCACF_ENTITY_NAME            ),
    DEF_VALS1(MQCACF_ENV_INFO               ),
    DEF_VALS1(MQCACF_ESCAPE_TEXT            ),
    DEF_VALS1(MQCACF_EVENT_APPL_IDENTITY    ),
    DEF_VALS1(MQCACF_EVENT_APPL_NAME        ),
    DEF_VALS1(MQCACF_EVENT_APPL_ORIGIN      ),
    DEF_VALS1(MQCACF_EVENT_Q_MGR            ),
    DEF_VALS1(MQCACF_EVENT_USER_ID          ),
    DEF_VALS1(MQCACF_FAIL_DATE              ),
    DEF_VALS1(MQCACF_FAIL_TIME              ),
    DEF_VALS1(MQCACF_FILTER                 ),
    DEF_VALS1(MQCACF_FROM_AUTH_INFO_NAME    ),
    DEF_VALS1(MQCACF_FROM_CF_STRUC_NAME     ),
    DEF_VALS1(MQCACF_FROM_CHANNEL_NAME      ),
    DEF_VALS1(MQCACF_FROM_LISTENER_NAME     ),
    DEF_VALS1(MQCACF_FROM_NAMELIST_NAME     ),
    DEF_VALS1(MQCACF_FROM_PROCESS_NAME      ),
    DEF_VALS1(MQCACF_FROM_Q_NAME            ),
    DEF_VALS1(MQCACF_FROM_SERVICE_NAME      ),
    DEF_VALS1(MQCACF_FROM_STORAGE_CLASS     ),
    DEF_VALS1(MQCACF_FROM_SUB_NAME          ),
    DEF_VALS1(MQCACF_FROM_TOPIC_NAME        ),
    DEF_VALS1(MQCACF_GROUP_ENTITY_NAMES     ),
    DEF_VALS1(MQCACF_LAST_GET_DATE          ),
    DEF_VALS1(MQCACF_LAST_GET_TIME          ),
    DEF_VALS1(MQCACF_LAST_MSG_DATE          ),
    DEF_VALS1(MQCACF_LAST_MSG_TIME          ),
    DEF_VALS1(MQCACF_LAST_PUB_DATE          ),
    DEF_VALS1(MQCACF_LAST_PUB_TIME          ),
    DEF_VALS1(MQCACF_LAST_PUT_DATE          ),
    DEF_VALS1(MQCACF_LAST_PUT_TIME          ),
    DEF_VALS1(MQCACF_LOCAL_Q_NAMES          ),
    DEF_VALS1(MQCACF_LOG_PATH               ),
    DEF_VALS1(MQCACF_MEDIA_LOG_EXTENT_NAME  ),
    DEF_VALS1(MQCACF_MODEL_Q_NAMES          ),
    DEF_VALS1(MQCACF_NAMELIST_NAMES         ),
    DEF_VALS1(MQCACF_NONE                   ),
    DEF_VALS1(MQCACF_OBJECT_NAME            ),
    DEF_VALS1(MQCACF_OBJECT_Q_MGR_NAME      ),
    DEF_VALS1(MQCACF_OPERATION_DATE         ),
    DEF_VALS1(MQCACF_OPERATION_TIME         ),
    DEF_VALS1(MQCACF_ORIGIN_NAME            ),
    DEF_VALS1(MQCACF_PARENT_Q_MGR_NAME      ),
    DEF_VALS1(MQCACF_PRINCIPAL_ENTITY_NAMES ),
    DEF_VALS1(MQCACF_PROCESS_NAMES          ),
    DEF_VALS1(MQCACF_PSB_NAME               ),
    DEF_VALS1(MQCACF_PST_ID                 ),
    DEF_VALS1(MQCACF_PUBLISH_TIMESTAMP      ),
    DEF_VALS1(MQCACF_PUT_DATE               ),
    DEF_VALS1(MQCACF_PUT_TIME               ),
    DEF_VALS1(MQCACF_Q_MGR_CPF              ),
    DEF_VALS1(MQCACF_Q_MGR_UOW_ID           ),
    DEF_VALS1(MQCACF_Q_NAMES                ),
    DEF_VALS1(MQCACF_RECEIVER_CHANNEL_NAMES ),
    DEF_VALS1(MQCACF_REG_CORREL_ID          ),
    DEF_VALS1(MQCACF_REG_Q_MGR_NAME         ),
    DEF_VALS1(MQCACF_REG_Q_NAME             ),
    DEF_VALS1(MQCACF_REG_STREAM_NAME        ),
    DEF_VALS1(MQCACF_REG_SUB_IDENTITY       ),
    DEF_VALS1(MQCACF_REG_SUB_NAME           ),
    DEF_VALS1(MQCACF_REG_SUB_USER_DATA      ),
    DEF_VALS1(MQCACF_REG_TIME               ),
    DEF_VALS1(MQCACF_REG_TOPIC              ),
    DEF_VALS1(MQCACF_REG_USER_ID            ),
    DEF_VALS1(MQCACF_REMOTE_Q_NAMES         ),
    DEF_VALS1(MQCACF_REPLY_TO_Q             ),
    DEF_VALS1(MQCACF_REPLY_TO_Q_MGR         ),
    DEF_VALS1(MQCACF_REQUESTER_CHANNEL_NAMES),
    DEF_VALS1(MQCACF_RESOLVED_Q_NAME        ),
    DEF_VALS1(MQCACF_RESPONSE_Q_MGR_NAME    ),
    DEF_VALS1(MQCACF_RESTART_LOG_EXTENT_NAME),
    DEF_VALS1(MQCACF_ROUTING_FINGER_PRINT   ),
    DEF_VALS1(MQCACF_SECURITY_PROFILE       ),
    DEF_VALS1(MQCACF_SENDER_CHANNEL_NAMES   ),
    DEF_VALS1(MQCACF_SERVER_CHANNEL_NAMES   ),
    DEF_VALS1(MQCACF_SERVICE_COMPONENT      ),
    DEF_VALS1(MQCACF_SERVICE_START_DATE     ),
    DEF_VALS1(MQCACF_SERVICE_START_TIME     ),
    DEF_VALS1(MQCACF_STORAGE_CLASS_NAMES    ),
    DEF_VALS1(MQCACF_STREAM_NAME            ),
    DEF_VALS1(MQCACF_STRING_DATA            ),
    DEF_VALS1(MQCACF_STRUC_ID               ),
    DEF_VALS1(MQCACF_SUBSCRIPTION_IDENTITY  ),
    DEF_VALS1(MQCACF_SUBSCRIPTION_NAME      ),
    DEF_VALS1(MQCACF_SUBSCRIPTION_POINT     ),
    DEF_VALS1(MQCACF_SUBSCRIPTION_USER_DATA ),
    DEF_VALS1(MQCACF_SUB_NAME               ),
    DEF_VALS1(MQCACF_SUB_SELECTOR           ),
    DEF_VALS1(MQCACF_SUB_USER_DATA          ),
    DEF_VALS1(MQCACF_SUB_USER_ID            ),
    DEF_VALS1(MQCACF_SUPPORTED_STREAM_NAME  ),
    DEF_VALS1(MQCACF_SYSP_ARCHIVE_PFX1      ),
    DEF_VALS1(MQCACF_SYSP_ARCHIVE_PFX2      ),
    DEF_VALS1(MQCACF_SYSP_ARCHIVE_UNIT1     ),
    DEF_VALS1(MQCACF_SYSP_ARCHIVE_UNIT2     ),
    DEF_VALS1(MQCACF_SYSP_CMD_USER_ID       ),
    DEF_VALS1(MQCACF_SYSP_LOG_CORREL_ID     ),
    DEF_VALS1(MQCACF_SYSP_LOG_RBA           ),
    DEF_VALS1(MQCACF_SYSP_OFFLINE_RBA       ),
    DEF_VALS1(MQCACF_SYSP_OTMA_DRU_EXIT     ),
    DEF_VALS1(MQCACF_SYSP_OTMA_GROUP        ),
    DEF_VALS1(MQCACF_SYSP_OTMA_MEMBER       ),
    DEF_VALS1(MQCACF_SYSP_OTMA_TPIPE_PFX    ),
    DEF_VALS1(MQCACF_SYSP_Q_MGR_DATE        ),
    DEF_VALS1(MQCACF_SYSP_Q_MGR_RBA         ),
    DEF_VALS1(MQCACF_SYSP_Q_MGR_TIME        ),
    DEF_VALS1(MQCACF_SYSP_SERVICE           ),
    DEF_VALS1(MQCACF_SYSP_UNIT_VOLSER       ),
    DEF_VALS1(MQCACF_SYSTEM_NAME            ),
    DEF_VALS1(MQCACF_TASK_NUMBER            ),
    DEF_VALS1(MQCACF_TOPIC                  ),
    DEF_VALS1(MQCACF_TOPIC_NAMES            ),
    DEF_VALS1(MQCACF_TO_AUTH_INFO_NAME      ),
    DEF_VALS1(MQCACF_TO_CF_STRUC_NAME       ),
    DEF_VALS1(MQCACF_TO_CHANNEL_NAME        ),
    DEF_VALS1(MQCACF_TO_LISTENER_NAME       ),
    DEF_VALS1(MQCACF_TO_NAMELIST_NAME       ),
    DEF_VALS1(MQCACF_TO_PROCESS_NAME        ),
    DEF_VALS1(MQCACF_TO_Q_NAME              ),
    DEF_VALS1(MQCACF_TO_SERVICE_NAME        ),
    DEF_VALS1(MQCACF_TO_STORAGE_CLASS       ),
    DEF_VALS1(MQCACF_TO_SUB_NAME            ),
    DEF_VALS1(MQCACF_TO_TOPIC_NAME          ),
    DEF_VALS1(MQCACF_TRANSACTION_ID         ),
    DEF_VALS1(MQCACF_UOW_LOG_EXTENT_NAME    ),
    DEF_VALS1(MQCACF_UOW_LOG_START_DATE     ),
    DEF_VALS1(MQCACF_UOW_LOG_START_TIME     ),
    DEF_VALS1(MQCACF_UOW_START_DATE         ),
    DEF_VALS1(MQCACF_UOW_START_TIME         ),
    DEF_VALS1(MQCACF_USAGE_LOG_LRSN         ),
    DEF_VALS1(MQCACF_USAGE_LOG_RBA          ),
    DEF_VALS1(MQCACF_USER_IDENTIFIER        ),
    DEF_VALS1(MQCACF_VALUE_NAME             ),
    DEF_VALS1(MQCACF_ALTERNATE_USERID       ),
    DEF_VALS1(MQCACF_APPL_DESC              ),
    DEF_VALS1(MQCACF_APPL_FUNCTION          ),
    DEF_VALS1(MQCACF_CF_OFFLOAD_SIZE1       ),
    DEF_VALS1(MQCACF_CF_OFFLOAD_SIZE2       ),
    DEF_VALS1(MQCACF_CF_OFFLOAD_SIZE3       ),
    DEF_VALS1(MQCACF_CF_SMDS                ),
    DEF_VALS1(MQCACF_CF_SMDSCONN            ),
    DEF_VALS1(MQCACF_CF_SMDS_GENERIC_NAME   ),
    DEF_VALS1(MQCACF_CF_STRUC_NAME          ),
    DEF_VALS1(MQCACF_CHAR_ATTRS             ),
    DEF_VALS1(MQCACF_DYNAMIC_Q_NAME         ),
    DEF_VALS1(MQCACF_FROM_COMM_INFO_NAME    ),
    DEF_VALS1(MQCACF_HOST_NAME              ),
    DEF_VALS1(MQCACF_MQCB_NAME              ),
    DEF_VALS1(MQCACF_OBJECT_STRING          ),
    DEF_VALS1(MQCACF_Q_MGR_START_DATE       ),
    DEF_VALS1(MQCACF_Q_MGR_START_TIME       ),
    DEF_VALS1(MQCACF_RECOVERY_DATE          ),
    DEF_VALS1(MQCACF_RECOVERY_TIME          ),
    DEF_VALS1(MQCACF_RESOLVED_LOCAL_Q_MGR   ),
    DEF_VALS1(MQCACF_RESOLVED_LOCAL_Q_NAME  ),
    DEF_VALS1(MQCACF_RESOLVED_OBJECT_STRING ),
    DEF_VALS1(MQCACF_RESOLVED_Q_MGR         ),
    DEF_VALS1(MQCACF_ROUTING_FINGER_PRINT   ),
    DEF_VALS1(MQCACF_SELECTION_STRING       ),
    DEF_VALS1(MQCACF_TO_COMM_INFO_NAME      ),
    DEF_VALS1(MQCACF_XA_INFO                ),
    DEF_VALS1(MQCACF_XQH_PUT_DATE           ),
    DEF_VALS1(MQCACF_XQH_PUT_TIME           ),
    DEF_VALS1(MQCACF_XQH_REMOTE_Q_MGR       ),
    DEF_VALS1(MQCACF_XQH_REMOTE_Q_NAME      ),

    DEF_VALS1(MQCACH_CHANNEL_NAME           ),
    DEF_VALS1(MQCACH_CHANNEL_NAMES          ),
    DEF_VALS1(MQCACH_CHANNEL_START_DATE     ),
    DEF_VALS1(MQCACH_CHANNEL_START_TIME     ),
    DEF_VALS1(MQCACH_CLIENT_ID              ),
    DEF_VALS1(MQCACH_CLIENT_USER_ID         ),
    DEF_VALS1(MQCACH_CONNECTION_NAME        ),
    DEF_VALS1(MQCACH_CONNECTION_NAME_LIST   ),
    DEF_VALS1(MQCACH_CURRENT_LUWID          ),
    DEF_VALS1(MQCACH_DESC                   ),
    DEF_VALS1(MQCACH_FORMAT_NAME            ),
    DEF_VALS1(MQCACH_GROUP_ADDRESS          ),
    DEF_VALS1(MQCACH_IP_ADDRESS             ),
    DEF_VALS1(MQCACH_JAAS_CONFIG            ),
    DEF_VALS1(MQCACH_LAST_LUWID             ),
    DEF_VALS1(MQCACH_LAST_MSG_DATE          ),
    DEF_VALS1(MQCACH_LAST_MSG_TIME          ),
    DEF_VALS1(MQCACH_LISTENER_DESC          ),
    DEF_VALS1(MQCACH_LISTENER_NAME          ),
    DEF_VALS1(MQCACH_LISTENER_START_DATE    ),
    DEF_VALS1(MQCACH_LISTENER_START_TIME    ),
    DEF_VALS1(MQCACH_LOCAL_ADDRESS          ),
    DEF_VALS1(MQCACH_LOCAL_NAME             ),
    DEF_VALS1(MQCACH_LU_NAME                ),
    DEF_VALS1(MQCACH_MCA_JOB_NAME           ),
    DEF_VALS1(MQCACH_MCA_NAME               ),
    DEF_VALS1(MQCACH_MCA_USER_ID            ),
    DEF_VALS1(MQCACH_MCA_USER_ID_LIST       ),
    DEF_VALS1(MQCACH_MODE_NAME              ),
    DEF_VALS1(MQCACH_MR_EXIT_NAME           ),
    DEF_VALS1(MQCACH_MR_EXIT_USER_DATA      ),
    DEF_VALS1(MQCACH_MSG_EXIT_NAME          ),
    DEF_VALS1(MQCACH_MSG_EXIT_USER_DATA     ),
    DEF_VALS1(MQCACH_PASSWORD               ),
    DEF_VALS1(MQCACH_RCV_EXIT_NAME          ),
    DEF_VALS1(MQCACH_RCV_EXIT_USER_DATA     ),
    DEF_VALS1(MQCACH_REMOTE_APPL_TAG        ),
    DEF_VALS1(MQCACH_REMOTE_VERSION         ),
    DEF_VALS1(MQCACH_REMOTE_PRODUCT         ),
    DEF_VALS1(MQCACH_SEC_EXIT_NAME          ),
    DEF_VALS1(MQCACH_SEC_EXIT_USER_DATA     ),
    DEF_VALS1(MQCACH_SEND_EXIT_NAME         ),
    DEF_VALS1(MQCACH_SEND_EXIT_USER_DATA    ),
    DEF_VALS1(MQCACH_SSL_CERT_ISSUER_NAME   ),
    DEF_VALS1(MQCACH_SSL_CERT_USER_ID       ),
    DEF_VALS1(MQCACH_SSL_CIPHER_SPEC        ),
    DEF_VALS1(MQCACH_SSL_CIPHER_SUITE       ),
    DEF_VALS1(MQCACH_SSL_HANDSHAKE_STAGE    ),
    DEF_VALS1(MQCACH_SSL_KEY_RESET_DATE     ),
    DEF_VALS1(MQCACH_SSL_KEY_RESET_TIME     ),
    DEF_VALS1(MQCACH_SSL_KEY_PASSPHRASE     ),
    DEF_VALS1(MQCACH_SSL_PEER_NAME          ),
    DEF_VALS1(MQCACH_SSL_SHORT_PEER_NAME    ),
    DEF_VALS1(MQCACH_TCP_NAME               ),
    DEF_VALS1(MQCACH_TP_NAME                ),
    DEF_VALS1(MQCACH_USER_ID                ),
    DEF_VALS1(MQCACH_XMIT_Q_NAME            ),

    DEF_VALS1(MQCAMO_CLOSE_DATE             ),
    DEF_VALS1(MQCAMO_CLOSE_TIME             ),
    DEF_VALS1(MQCAMO_CONN_DATE              ),
    DEF_VALS1(MQCAMO_CONN_TIME              ),
    DEF_VALS1(MQCAMO_DISC_DATE              ),
    DEF_VALS1(MQCAMO_DISC_TIME              ),
    DEF_VALS1(MQCAMO_END_DATE               ),
    DEF_VALS1(MQCAMO_END_TIME               ),
    DEF_VALS1(MQCAMO_OPEN_DATE              ),
    DEF_VALS1(MQCAMO_OPEN_TIME              ),
    DEF_VALS1(MQCAMO_START_DATE             ),
    DEF_VALS1(MQCAMO_START_TIME             ),

    DEF_VALS1(MQCA_ADMIN_TOPIC_NAME      ),
    DEF_VALS1(MQCA_ALTERATION_DATE       ),
    DEF_VALS1(MQCA_ALTERATION_TIME       ),
    DEF_VALS1(MQCA_APPL_ID               ),
    DEF_VALS1(MQCA_AUTH_INFO_CONN_NAME   ),
    DEF_VALS1(MQCA_AUTH_INFO_DESC        ),
    DEF_VALS1(MQCA_AUTH_INFO_NAME        ),
    DEF_VALS1(MQCA_AUTH_INFO_OCSP_URL    ),
    DEF_VALS1(MQCA_AUTO_REORG_CATALOG    ),
    DEF_VALS1(MQCA_AUTO_REORG_START_TIME ),
    DEF_VALS1(MQCA_BACKOUT_REQ_Q_NAME    ),
    DEF_VALS2(MQCA_BASE_Q_NAME,"MQCA_BASE_OBJECT_NAME"),
    DEF_VALS1(MQCA_BATCH_INTERFACE_ID    ),
    DEF_VALS1(MQCA_CF_STRUC_DESC         ),
    DEF_VALS1(MQCA_CF_STRUC_NAME         ),
    DEF_VALS1(MQCA_CHANNEL_AUTO_DEF_EXIT ),
    DEF_VALS1(MQCA_CHILD                 ),
    DEF_VALS1(MQCA_CHINIT_SERVICE_PARM   ),
    DEF_VALS1(MQCA_CICS_FILE_NAME        ),
    DEF_VALS1(MQCA_CLUSTER_DATE          ),
    DEF_VALS1(MQCA_CLUSTER_NAME          ),
    DEF_VALS1(MQCA_CLUSTER_NAMELIST      ),
    DEF_VALS1(MQCA_CLUSTER_Q_MGR_NAME    ),
    DEF_VALS1(MQCA_CLUSTER_TIME          ),
    DEF_VALS1(MQCA_CLUSTER_WORKLOAD_DATA ),
    DEF_VALS1(MQCA_CLUSTER_WORKLOAD_EXIT ),
    DEF_VALS1(MQCA_COMMAND_INPUT_Q_NAME  ),
    DEF_VALS1(MQCA_COMMAND_REPLY_Q_NAME  ),
    DEF_VALS1(MQCA_CREATION_DATE         ),
    DEF_VALS1(MQCA_CREATION_TIME         ),
    DEF_VALS1(MQCA_DEAD_LETTER_Q_NAME    ),
    DEF_VALS1(MQCA_DEF_XMIT_Q_NAME       ),
    DEF_VALS1(MQCA_DNS_GROUP             ),
    DEF_VALS1(MQCA_ENV_DATA              ),
    DEF_VALS1(MQCA_IGQ_USER_ID           ),
    DEF_VALS1(MQCA_INITIATION_Q_NAME     ),
    DEF_VALS1(MQCA_LDAP_PASSWORD         ),
    DEF_VALS1(MQCA_LDAP_USER_NAME        ),
    DEF_VALS1(MQCA_LU62_ARM_SUFFIX       ),
    DEF_VALS1(MQCA_LU_GROUP_NAME         ),
    DEF_VALS1(MQCA_LU_NAME               ),
    DEF_VALS1(MQCA_MODEL_DURABLE_Q       ),
    DEF_VALS1(MQCA_MODEL_NON_DURABLE_Q   ),
    DEF_VALS1(MQCA_MONITOR_Q_NAME        ),
    DEF_VALS1(MQCA_NAMELIST_DESC         ),
    DEF_VALS1(MQCA_NAMELIST_NAME         ),
    DEF_VALS1(MQCA_NAMES                 ),
    DEF_VALS1(MQCA_PARENT                ),
    DEF_VALS1(MQCA_PASS_TICKET_APPL      ),
    DEF_VALS1(MQCA_PROCESS_DESC          ),
    DEF_VALS1(MQCA_PROCESS_NAME          ),
    DEF_VALS1(MQCA_QSG_NAME              ),
    DEF_VALS1(MQCA_Q_DESC                ),
    DEF_VALS1(MQCA_Q_MGR_DESC            ),
    DEF_VALS1(MQCA_Q_MGR_IDENTIFIER      ),
    DEF_VALS1(MQCA_Q_MGR_NAME            ),
    DEF_VALS1(MQCA_Q_NAME                ),
    DEF_VALS1(MQCA_REMOTE_Q_MGR_NAME     ),
    DEF_VALS1(MQCA_REMOTE_Q_NAME         ),
    DEF_VALS1(MQCA_REPOSITORY_NAME       ),
    DEF_VALS1(MQCA_REPOSITORY_NAMELIST   ),
    DEF_VALS1(MQCA_RESUME_DATE           ),
    DEF_VALS1(MQCA_RESUME_TIME           ),
    DEF_VALS1(MQCA_SERVICE_DESC          ),
    DEF_VALS1(MQCA_SERVICE_NAME          ),
    DEF_VALS1(MQCA_SERVICE_START_ARGS    ),
    DEF_VALS1(MQCA_SERVICE_START_COMMAND ),
    DEF_VALS1(MQCA_SERVICE_STOP_ARGS     ),
    DEF_VALS1(MQCA_SERVICE_STOP_COMMAND  ),
    DEF_VALS1(MQCA_SSL_CRL_NAMELIST      ),
    DEF_VALS1(MQCA_SSL_CRYPTO_HARDWARE   ),
    DEF_VALS1(MQCA_SSL_KEY_LIBRARY       ),
    DEF_VALS1(MQCA_SSL_KEY_MEMBER        ),
    DEF_VALS1(MQCA_SSL_KEY_REPOSITORY    ),
    DEF_VALS1(MQCA_STDERR_DESTINATION    ),
    DEF_VALS1(MQCA_STDOUT_DESTINATION    ),
    DEF_VALS1(MQCA_STORAGE_CLASS         ),
    DEF_VALS1(MQCA_STORAGE_CLASS_DESC    ),
    DEF_VALS1(MQCA_SYSTEM_LOG_Q_NAME     ),
    DEF_VALS1(MQCA_TCP_NAME              ),
    DEF_VALS1(MQCA_TOPIC_DESC            ),
    DEF_VALS1(MQCA_TOPIC_NAME            ),
    DEF_VALS1(MQCA_TOPIC_STRING          ),
    DEF_VALS1(MQCA_TOPIC_STRING_FILTER   ),
    DEF_VALS1(MQCA_TPIPE_NAME            ),
    DEF_VALS1(MQCA_TRIGGER_CHANNEL_NAME  ),
    DEF_VALS1(MQCA_TRIGGER_DATA          ),
    DEF_VALS1(MQCA_TRIGGER_PROGRAM_NAME  ),
    DEF_VALS1(MQCA_TRIGGER_TERM_ID       ),
    DEF_VALS1(MQCA_TRIGGER_TRANS_ID      ),
    DEF_VALS1(MQCA_USER_DATA             ),
    DEF_VALS1(MQCA_USER_LIST             ),
    DEF_VALS1(MQCA_XCF_GROUP_NAME        ),
    DEF_VALS1(MQCA_XCF_MEMBER_NAME       ),
    DEF_VALS1(MQCA_XR_VERSION2           ),
    DEF_VALS1(MQCA_XMIT_Q_NAME           ),
    DEF_VALS1(MQCA_CHLAUTH_DESC          ),
    DEF_VALS1(MQCA_CLUS_CHL_NAME         ),
    DEF_VALS1(MQCA_COMM_INFO_NAME        ),
    DEF_VALS1(MQCA_COMM_INFO_DESC        ),
    DEF_VALS1(MQCA_CUSTOM                ),
    DEF_VALS1(MQCA_INSTALLATION_DESC     ),
    DEF_VALS1(MQCA_INSTALLATION_NAME     ),
    DEF_VALS1(MQCA_INSTALLATION_PATH     ),
    DEF_VALS1(MQCA_POLICY_NAME           ),
    DEF_VALS1(MQCA_RECIPIENT_DN          ),
    DEF_VALS1(MQCA_SIGNER_DN             ),
    DEF_VALS1(MQCA_XR_VERSION            ),
    DEF_VALS1(MQCA_XR_SSL_CIPHER_SUITES  ),

    DEF_VALS1(MQGACF_ACTIVITY               ),
    DEF_VALS1(MQGACF_APP_DIST_LIST          ),
    DEF_VALS1(MQGACF_ACTIVITY_TRACE         ),
    DEF_VALS1(MQGACF_CHL_STATISTICS_DATA    ),
    DEF_VALS1(MQGACF_COMMAND_CONTEXT        ),
    DEF_VALS1(MQGACF_COMMAND_DATA           ),
    DEF_VALS1(MQGACF_EMBEDDED_MQMD          ),
    DEF_VALS1(MQGACF_MESSAGE                ),
    DEF_VALS1(MQGACF_MQMD                   ),
    DEF_VALS1(MQGACF_OPERATION              ),
    DEF_VALS1(MQGACF_Q_ACCOUNTING_DATA      ),
    DEF_VALS1(MQGACF_Q_STATISTICS_DATA      ),
    DEF_VALS1(MQGACF_TRACE_ROUTE            ),
    DEF_VALS1(MQGACF_VALUE_NAMING           ),

    DEF_VALS1(MQIACF_ACTION                 ),
    DEF_VALS1(MQIACF_ALL                    ),
    DEF_VALS1(MQIACF_ANONYMOUS_COUNT        ),
    DEF_VALS1(MQIACF_APPL_COUNT             ),
    DEF_VALS1(MQIACF_ASYNC_STATE            ),
    DEF_VALS1(MQIACF_AUTHORIZATION_LIST     ),
    DEF_VALS1(MQIACF_AUTH_ADD_AUTHS         ),
    DEF_VALS1(MQIACF_AUTH_INFO_ATTRS        ),
    DEF_VALS1(MQIACF_AUTH_OPTIONS           ),
    DEF_VALS1(MQIACF_AUTH_PROFILE_ATTRS     ),
    DEF_VALS1(MQIACF_AUTH_REMOVE_AUTHS      ),
    DEF_VALS1(MQIACF_AUTH_SERVICE_ATTRS     ),
    DEF_VALS1(MQIACF_AUX_ERROR_DATA_INT_1   ),
    DEF_VALS1(MQIACF_AUX_ERROR_DATA_INT_2   ),
    DEF_VALS1(MQIACF_BACKOUT_COUNT          ),
    DEF_VALS1(MQIACF_BRIDGE_TYPE            ),
    DEF_VALS1(MQIACF_BROKER_COUNT           ),
    DEF_VALS1(MQIACF_BROKER_OPTIONS         ),
    DEF_VALS1(MQIACF_BUFFER_POOL_ID         ),
    DEF_VALS1(MQIACF_CF_STATUS_BACKUP       ),
    DEF_VALS1(MQIACF_CF_STATUS_CONNECT      ),
    DEF_VALS1(MQIACF_CF_STATUS_SUMMARY      ),
    DEF_VALS1(MQIACF_CF_STATUS_TYPE         ),
    DEF_VALS1(MQIACF_CF_STRUC_ATTRS         ),
    DEF_VALS1(MQIACF_CF_STRUC_BACKUP_SIZE   ),
    DEF_VALS1(MQIACF_CF_STRUC_ENTRIES_MAX   ),
    DEF_VALS1(MQIACF_CF_STRUC_ENTRIES_USED  ),
    DEF_VALS1(MQIACF_CF_STRUC_SIZE_MAX      ),
    DEF_VALS1(MQIACF_CF_STRUC_SIZE_USED     ),
    DEF_VALS1(MQIACF_CF_STRUC_STATUS        ),
    DEF_VALS1(MQIACF_CF_STRUC_TYPE          ),
    DEF_VALS1(MQIACF_CHANNEL_ATTRS          ),
    DEF_VALS1(MQIACF_CHINIT_STATUS          ),
    DEF_VALS1(MQIACF_CHLAUTH_TYPE           ),
    DEF_VALS1(MQIACF_CLEAR_SCOPE            ),
    DEF_VALS1(MQIACF_CLEAR_TYPE             ),
    DEF_VALS1(MQIACF_CLUSTER_INFO           ),
    DEF_VALS1(MQIACF_CLUSTER_Q_MGR_ATTRS    ),
    DEF_VALS1(MQIACF_CMDSCOPE_Q_MGR_COUNT   ),
    DEF_VALS1(MQIACF_CMD_SERVER_STATUS      ),
    DEF_VALS1(MQIACF_COMMAND                ),
    DEF_VALS1(MQIACF_COMMAND_INFO           ),
    DEF_VALS1(MQIACF_COMP_CODE              ),
    DEF_VALS1(MQIACF_CONFIGURATION_EVENTS   ),
    DEF_VALS1(MQIACF_CONFIGURATION_OBJECTS  ),
    DEF_VALS1(MQIACF_CONNECTION_ATTRS       ),
    DEF_VALS1(MQIACF_CONNECTION_COUNT       ),
    DEF_VALS1(MQIACF_CONNECT_OPTIONS        ),
    DEF_VALS1(MQIACF_CONN_INFO_ALL          ),
    DEF_VALS1(MQIACF_CONN_INFO_CONN         ),
    DEF_VALS1(MQIACF_CONN_INFO_HANDLE       ),
    DEF_VALS1(MQIACF_CONN_INFO_TYPE         ),
    DEF_VALS1(MQIACF_CONV_REASON_CODE       ),
    DEF_VALS1(MQIACF_DB2_CONN_STATUS        ),
    DEF_VALS1(MQIACF_DELETE_OPTIONS         ),
    DEF_VALS1(MQIACF_DESTINATION_CLASS      ),
    DEF_VALS1(MQIACF_DISCONTINUITY_COUNT    ),
    DEF_VALS1(MQIACF_DURABLE_SUBSCRIPTION   ),
    DEF_VALS1(MQIACF_ENCODING               ),
    DEF_VALS1(MQIACF_ENTITY_TYPE            ),
    DEF_VALS2(MQIACF_ERROR_IDENTIFIER,"MQIACF_ERROR_ID"),
    DEF_VALS1(MQIACF_ERROR_OFFSET           ),
    DEF_VALS1(MQIACF_ESCAPE_TYPE            ),
    DEF_VALS1(MQIACF_EVENT_APPL_TYPE        ),
    DEF_VALS1(MQIACF_EVENT_ORIGIN           ),
    DEF_VALS1(MQIACF_EXCLUDE_INTERVAL       ),
    DEF_VALS1(MQIACF_EXPIRY                 ),
    DEF_VALS1(MQIACF_EXPIRY_Q_COUNT         ),
    DEF_VALS1(MQIACF_FEEDBACK               ),
    DEF_VALS1(MQIACF_FORCE                  ),
    DEF_VALS1(MQIACF_GROUPUR_CHECK_ID       ),
    DEF_VALS1(MQIACF_HANDLE_STATE           ),
    DEF_VALS1(MQIACF_INQUIRY                ),
    DEF_VALS1(MQIACF_INTEGER_DATA           ),
    DEF_VALS1(MQIACF_INTERFACE_VERSION      ),
    DEF_VALS1(MQIACF_LISTENER_ATTRS         ),
    DEF_VALS1(MQIACF_LISTENER_STATUS_ATTRS  ),
    DEF_VALS1(MQIACF_LOG_COMPRESSION        ),
    DEF_VALS1(MQIACF_MAX_ACTIVITIES         ),
    DEF_VALS1(MQIACF_MESSAGE_COUNT          ),
    DEF_VALS2(MQIACF_MODE,"MQIACF_QUIESCE"),
    DEF_VALS1(MQIACF_MONITORING             ),
    DEF_VALS1(MQIACF_MOVE_COUNT             ),
    DEF_VALS1(MQIACF_MOVE_TYPE              ),
    DEF_VALS1(MQIACF_MOVE_TYPE_ADD          ),
    DEF_VALS1(MQIACF_MOVE_TYPE_MOVE         ),
    DEF_VALS1(MQIACF_MSG_FLAGS              ),
    DEF_VALS1(MQIACF_MSG_LENGTH             ),
    DEF_VALS1(MQIACF_MSG_TYPE               ),
    DEF_VALS1(MQIACF_MULC_CAPTURE           ),
    DEF_VALS1(MQIACF_NAMELIST_ATTRS         ),
    DEF_VALS1(MQIACF_OBJECT_TYPE            ),
    DEF_VALS1(MQIACF_OBSOLETE_MSGS          ),
    DEF_VALS1(MQIACF_OFFSET                 ),
    DEF_VALS1(MQIACF_OLDEST_MSG_AGE         ),
    DEF_VALS1(MQIACF_OPEN_BROWSE            ),
    DEF_VALS1(MQIACF_OPEN_INPUT_TYPE        ),
    DEF_VALS1(MQIACF_OPEN_INQUIRE           ),
    DEF_VALS1(MQIACF_OPEN_OPTIONS           ),
    DEF_VALS1(MQIACF_OPEN_OUTPUT            ),
    DEF_VALS1(MQIACF_OPEN_SET               ),
    DEF_VALS1(MQIACF_OPEN_TYPE              ),
    DEF_VALS1(MQIACF_OPERATION_MODE         ),
    DEF_VALS1(MQIACF_OPERATION_TYPE         ),
    DEF_VALS1(MQIACF_OPTIONS                ),
    DEF_VALS1(MQIACF_ORIGINAL_LENGTH        ),
    DEF_VALS1(MQIACF_PAGESET_STATUS         ),
    DEF_VALS1(MQIACF_PARAMETER_ID           ),
    DEF_VALS1(MQIACF_PERMIT_STANDBY         ),
    DEF_VALS1(MQIACF_PERSISTENCE            ),
    DEF_VALS1(MQIACF_PRIORITY               ),
    DEF_VALS1(MQIACF_PROCESS_ATTRS          ),
    DEF_VALS1(MQIACF_PROCESS_ID             ),
    DEF_VALS1(MQIACF_PS_STATUS_TYPE         ),
    DEF_VALS1(MQIACF_PUBLICATION_OPTIONS    ),
    DEF_VALS1(MQIACF_PUBLISH_COUNT          ),
    DEF_VALS1(MQIACF_PUBSUB_PROPERTIES      ),
    DEF_VALS1(MQIACF_PUBSUB_STATUS          ),
    DEF_VALS1(MQIACF_PUBSUB_STATUS_ATTRS    ),
    DEF_VALS1(MQIACF_PUB_PRIORITY           ),
    DEF_VALS1(MQIACF_PURGE                  ),
    DEF_VALS1(MQIACF_QSG_DISPS              ),
    DEF_VALS1(MQIACF_Q_ATTRS                ),
    DEF_VALS1(MQIACF_Q_HANDLE               ),
    DEF_VALS1(MQIACF_Q_MGR_ATTRS            ),
    DEF_VALS1(MQIACF_Q_MGR_CLUSTER          ),
    DEF_VALS1(MQIACF_Q_MGR_DEFINITION_TYPE  ),
    DEF_VALS1(MQIACF_Q_MGR_DQM              ),
    DEF_VALS1(MQIACF_Q_MGR_EVENT            ),
    DEF_VALS1(MQIACF_Q_MGR_FACILITY         ),
    DEF_VALS1(MQIACF_Q_MGR_NUMBER           ),
    DEF_VALS1(MQIACF_Q_MGR_PUBSUB           ),
    DEF_VALS1(MQIACF_Q_MGR_STATUS           ),
    DEF_VALS1(MQIACF_Q_MGR_STATUS_ATTRS     ),
    DEF_VALS1(MQIACF_Q_MGR_SYSTEM           ),
    DEF_VALS1(MQIACF_Q_MGR_TYPE             ),
    DEF_VALS1(MQIACF_Q_MGR_VERSION          ),
    DEF_VALS1(MQIACF_Q_STATUS               ),
    DEF_VALS1(MQIACF_Q_STATUS_ATTRS         ),
    DEF_VALS1(MQIACF_Q_STATUS_TYPE          ),
    DEF_VALS1(MQIACF_Q_TIME_INDICATOR       ),
    DEF_VALS1(MQIACF_Q_TYPES                ),
    DEF_VALS1(MQIACF_REASON_CODE            ),
    DEF_VALS1(MQIACF_REASON_QUALIFIER       ),
    DEF_VALS1(MQIACF_RECORDED_ACTIVITIES    ),
    DEF_VALS1(MQIACF_REFRESH_INTERVAL       ),
    DEF_VALS1(MQIACF_REFRESH_REPOSITORY     ),
    DEF_VALS1(MQIACF_REFRESH_TYPE           ),
    DEF_VALS1(MQIACF_REGISTRATION_OPTIONS   ),
    DEF_VALS1(MQIACF_REG_REG_OPTIONS        ),
    DEF_VALS1(MQIACF_REMOVE_QUEUES          ),
    DEF_VALS1(MQIACF_REPLACE                ),
    DEF_VALS1(MQIACF_REPORT                 ),
    DEF_VALS1(MQIACF_REQUEST_ONLY           ),
    DEF_VALS1(MQIACF_RETAINED_PUBLICATION   ),
    DEF_VALS1(MQIACF_ROUTE_ACCUMULATION     ),
    DEF_VALS1(MQIACF_ROUTE_DELIVERY         ),
    DEF_VALS1(MQIACF_ROUTE_DETAIL           ),
    DEF_VALS1(MQIACF_ROUTE_FORWARDING       ),
    DEF_VALS1(MQIACF_SECURITY_ATTRS         ),
    DEF_VALS1(MQIACF_SECURITY_INTERVAL      ),
    DEF_VALS1(MQIACF_SECURITY_ITEM          ),
    DEF_VALS1(MQIACF_SECURITY_SETTING       ),
    DEF_VALS1(MQIACF_SECURITY_SWITCH        ),
    DEF_VALS1(MQIACF_SECURITY_TIMEOUT       ),
    DEF_VALS1(MQIACF_SECURITY_TYPE          ),
    DEF_VALS1(MQIACF_SELECTOR               ),
    DEF_VALS1(MQIACF_SELECTOR_TYPE          ),
    DEF_VALS1(MQIACF_SEQUENCE_NUMBER        ),
    DEF_VALS1(MQIACF_SERVICE_ATTRS          ),
    DEF_VALS1(MQIACF_SERVICE_STATUS         ),
    DEF_VALS1(MQIACF_SERVICE_STATUS_ATTRS   ),
    DEF_VALS1(MQIACF_STORAGE_CLASS_ATTRS    ),
    DEF_VALS1(MQIACF_SUBSCRIPTION_SCOPE     ),
    DEF_VALS1(MQIACF_SUB_ATTRS              ),
    DEF_VALS1(MQIACF_SUB_LEVEL              ),
    DEF_VALS1(MQIACF_SUB_OPTIONS            ),
    DEF_VALS1(MQIACF_SUB_STATUS_ATTRS       ),
    DEF_VALS1(MQIACF_SUB_SUMMARY            ),
    DEF_VALS1(MQIACF_SUB_TYPE               ),
    DEF_VALS1(MQIACF_SUSPEND                ),
    DEF_VALS1(MQIACF_SYSP_ALLOC_PRIMARY     ),
    DEF_VALS1(MQIACF_SYSP_ALLOC_SECONDARY   ),
    DEF_VALS1(MQIACF_SYSP_ALLOC_UNIT        ),
    DEF_VALS1(MQIACF_SYSP_ARCHIVE           ),
    DEF_VALS1(MQIACF_SYSP_ARCHIVE_RETAIN    ),
    DEF_VALS1(MQIACF_SYSP_ARCHIVE_WTOR      ),
    DEF_VALS1(MQIACF_SYSP_BLOCK_SIZE        ),
    DEF_VALS1(MQIACF_SYSP_CATALOG           ),
    DEF_VALS1(MQIACF_SYSP_CHKPOINT_COUNT    ),
    DEF_VALS1(MQIACF_SYSP_CLUSTER_CACHE     ),
    DEF_VALS1(MQIACF_SYSP_COMPACT           ),
    DEF_VALS1(MQIACF_SYSP_DB2_BLOB_TASKS    ),
    DEF_VALS1(MQIACF_SYSP_DB2_TASKS         ),
    DEF_VALS1(MQIACF_SYSP_DEALLOC_INTERVAL  ),
    DEF_VALS1(MQIACF_SYSP_DUAL_ACTIVE       ),
    DEF_VALS1(MQIACF_SYSP_DUAL_ARCHIVE      ),
    DEF_VALS1(MQIACF_SYSP_DUAL_BSDS         ),
    DEF_VALS1(MQIACF_SYSP_EXIT_INTERVAL     ),
    DEF_VALS1(MQIACF_SYSP_EXIT_TASKS        ),
    DEF_VALS1(MQIACF_SYSP_FULL_LOGS         ),
    DEF_VALS1(MQIACF_SYSP_IN_BUFFER_SIZE    ),
    DEF_VALS1(MQIACF_SYSP_LOG_COPY          ),
    DEF_VALS1(MQIACF_SYSP_LOG_SUSPEND       ),
    DEF_VALS1(MQIACF_SYSP_LOG_USED          ),
    DEF_VALS1(MQIACF_SYSP_MAX_ARCHIVE       ),
    DEF_VALS1(MQIACF_SYSP_MAX_CONNS         ),
    DEF_VALS1(MQIACF_SYSP_MAX_CONNS_BACK    ),
    DEF_VALS1(MQIACF_SYSP_MAX_CONNS_FORE    ),
    DEF_VALS1(MQIACF_SYSP_MAX_READ_TAPES    ),
    DEF_VALS1(MQIACF_SYSP_OFFLOAD_STATUS    ),
    DEF_VALS1(MQIACF_SYSP_OTMA_INTERVAL     ),
    DEF_VALS1(MQIACF_SYSP_OUT_BUFFER_COUNT  ),
    DEF_VALS1(MQIACF_SYSP_OUT_BUFFER_SIZE   ),
    DEF_VALS1(MQIACF_SYSP_PROTECT           ),
    DEF_VALS1(MQIACF_SYSP_QUIESCE_INTERVAL  ),
    DEF_VALS1(MQIACF_SYSP_Q_INDEX_DEFER     ),
    DEF_VALS1(MQIACF_SYSP_RESLEVEL_AUDIT    ),
    DEF_VALS1(MQIACF_SYSP_ROUTING_CODE      ),
    DEF_VALS1(MQIACF_SYSP_SMF_ACCOUNTING    ),
    DEF_VALS1(MQIACF_SYSP_SMF_INTERVAL      ),
    DEF_VALS1(MQIACF_SYSP_SMF_STATS         ),
    DEF_VALS1(MQIACF_SYSP_TIMESTAMP         ),
    DEF_VALS1(MQIACF_SYSP_TOTAL_LOGS        ),
    DEF_VALS1(MQIACF_SYSP_TRACE_CLASS       ),
    DEF_VALS1(MQIACF_SYSP_TRACE_SIZE        ),
    DEF_VALS1(MQIACF_SYSP_TYPE              ),
    DEF_VALS1(MQIACF_SYSP_UNIT_ADDRESS      ),
    DEF_VALS1(MQIACF_SYSP_UNIT_STATUS       ),
    DEF_VALS1(MQIACF_SYSP_WLM_INTERVAL      ),
    DEF_VALS1(MQIACF_SYSP_WLM_INT_UNITS     ),
    DEF_VALS1(MQIACF_THREAD_ID              ),
    DEF_VALS1(MQIACF_TOPIC_ATTRS            ),
    DEF_VALS1(MQIACF_TOPIC_PUB              ),
    DEF_VALS1(MQIACF_TOPIC_STATUS           ),
    DEF_VALS1(MQIACF_TOPIC_STATUS_ATTRS     ),
    DEF_VALS1(MQIACF_TOPIC_STATUS_TYPE      ),
    DEF_VALS1(MQIACF_TOPIC_SUB              ),
    DEF_VALS1(MQIACF_UNCOMMITTED_MSGS       ),
    DEF_VALS1(MQIACF_UNRECORDED_ACTIVITIES  ),
    DEF_VALS1(MQIACF_UOW_STATE              ),
    DEF_VALS1(MQIACF_UOW_TYPE               ),
    DEF_VALS1(MQIACF_USAGE_BUFFER_POOL      ),
    DEF_VALS1(MQIACF_USAGE_DATA_SET         ),
    DEF_VALS1(MQIACF_USAGE_DATA_SET_TYPE    ),
    DEF_VALS1(MQIACF_USAGE_EXPAND_COUNT     ),
    DEF_VALS1(MQIACF_USAGE_EXPAND_TYPE      ),
    DEF_VALS1(MQIACF_USAGE_NONPERSIST_PAGES ),
    DEF_VALS1(MQIACF_USAGE_PAGESET          ),
    DEF_VALS1(MQIACF_USAGE_PERSIST_PAGES    ),
    DEF_VALS1(MQIACF_USAGE_RESTART_EXTENTS  ),
    DEF_VALS1(MQIACF_USAGE_TOTAL_BUFFERS    ),
    DEF_VALS1(MQIACF_USAGE_TOTAL_PAGES      ),
    DEF_VALS1(MQIACF_USAGE_TYPE             ),
    DEF_VALS1(MQIACF_USAGE_UNUSED_PAGES     ),
    DEF_VALS1(MQIACF_USER_ID_SUPPORT        ),
    DEF_VALS1(MQIACF_VARIABLE_USER_ID       ),
    DEF_VALS1(MQIACF_VERSION                ),
    DEF_VALS1(MQIACF_WAIT_INTERVAL          ),
    DEF_VALS1(MQIACF_WILDCARD_SCHEMA        ),
    DEF_VALS1(MQIACF_MCAST_REL_INDICATOR    ),

    DEF_VALS1(MQIACH_ACTIVE_CHL             ),
    DEF_VALS1(MQIACH_ACTIVE_CHL_MAX         ),
    DEF_VALS1(MQIACH_ACTIVE_CHL_PAUSED      ),
    DEF_VALS1(MQIACH_ACTIVE_CHL_RETRY       ),
    DEF_VALS1(MQIACH_ACTIVE_CHL_STARTED     ),
    DEF_VALS1(MQIACH_ACTIVE_CHL_STOPPED     ),
    DEF_VALS1(MQIACH_ADAPS_MAX              ),
    DEF_VALS1(MQIACH_ADAPS_STARTED          ),
    DEF_VALS1(MQIACH_ADAPTER                ),
    DEF_VALS1(MQIACH_ALLOC_FAST_TIMER       ),
    DEF_VALS1(MQIACH_ALLOC_RETRY            ),
    DEF_VALS1(MQIACH_ALLOC_SLOW_TIMER       ),
    DEF_VALS1(MQIACH_BACKLOG                ),
    DEF_VALS1(MQIACH_BATCHES                ),
    DEF_VALS1(MQIACH_BATCH_HB               ),
    DEF_VALS1(MQIACH_BATCH_INTERVAL         ),
    DEF_VALS1(MQIACH_BATCH_SIZE             ),
    DEF_VALS1(MQIACH_BATCH_SIZE_INDICATOR   ),
    DEF_VALS1(MQIACH_BUFFERS_SENT           ),
    DEF_VALS1(MQIACH_BYTES_SENT             ),
    DEF_VALS1(MQIACH_CHANNEL_DISP           ),
    DEF_VALS1(MQIACH_CHANNEL_ERROR_DATA     ),
    DEF_VALS1(MQIACH_CHANNEL_INSTANCE_ATTRS ),
    DEF_VALS1(MQIACH_CHANNEL_INSTANCE_TYPE  ),
    DEF_VALS1(MQIACH_CHANNEL_STATUS         ),
    DEF_VALS1(MQIACH_CHANNEL_SUBSTATE       ),
    DEF_VALS1(MQIACH_CHANNEL_TABLE          ),
    DEF_VALS1(MQIACH_CHANNEL_TYPE           ),
    DEF_VALS1(MQIACH_CHANNEL_TYPES          ),
    DEF_VALS1(MQIACH_CLIENT_CHANNEL_WEIGHT  ),
    DEF_VALS1(MQIACH_CLWL_CHANNEL_PRIORITY  ),
    DEF_VALS1(MQIACH_CLWL_CHANNEL_RANK      ),
    DEF_VALS1(MQIACH_CLWL_CHANNEL_WEIGHT    ),
    DEF_VALS1(MQIACH_COMMAND_COUNT          ),
    DEF_VALS1(MQIACH_COMPRESSION_RATE       ),
    DEF_VALS1(MQIACH_COMPRESSION_TIME       ),
    DEF_VALS1(MQIACH_CONNECTION_AFFINITY    ),
    DEF_VALS1(MQIACH_CURRENT_CHL            ),
    DEF_VALS1(MQIACH_CURRENT_CHL_LU62       ),
    DEF_VALS1(MQIACH_CURRENT_CHL_MAX        ),
    DEF_VALS1(MQIACH_CURRENT_CHL_TCP        ),
    DEF_VALS1(MQIACH_CURRENT_MSGS           ),
    DEF_VALS1(MQIACH_CURRENT_SHARING_CONVS  ),
    DEF_VALS1(MQIACH_DATA_CONVERSION        ),
    DEF_VALS1(MQIACH_DATA_COUNT             ),
    DEF_VALS1(MQIACH_DEF_CHANNEL_DISP       ),
    DEF_VALS1(MQIACH_DISC_INTERVAL          ),
    DEF_VALS1(MQIACH_DISC_RETRY             ),
    DEF_VALS1(MQIACH_DISPS_MAX              ),
    DEF_VALS1(MQIACH_DISPS_STARTED          ),
    DEF_VALS1(MQIACH_EXIT_TIME_INDICATOR    ),
    DEF_VALS1(MQIACH_HB_INTERVAL            ),
    DEF_VALS1(MQIACH_HDR_COMPRESSION        ),
    DEF_VALS1(MQIACH_INBOUND_DISP           ),
    DEF_VALS1(MQIACH_INDOUBT_STATUS         ),
    DEF_VALS1(MQIACH_IN_DOUBT               ),
    DEF_VALS1(MQIACH_KEEP_ALIVE_INTERVAL    ),
    DEF_VALS1(MQIACH_LISTENER_CONTROL       ),
    DEF_VALS1(MQIACH_LISTENER_STATUS        ),
    DEF_VALS1(MQIACH_LONG_RETRIES_LEFT      ),
    DEF_VALS1(MQIACH_LONG_RETRY             ),
    DEF_VALS1(MQIACH_LONG_TIMER             ),
    DEF_VALS1(MQIACH_MAX_INSTANCES          ),
    DEF_VALS1(MQIACH_MAX_INSTS_PER_CLIENT   ),
    DEF_VALS1(MQIACH_MAX_MSG_LENGTH         ),
    DEF_VALS1(MQIACH_MAX_SHARING_CONVS      ),
    DEF_VALS1(MQIACH_MAX_XMIT_SIZE          ),
    DEF_VALS1(MQIACH_MCA_STATUS             ),
    DEF_VALS1(MQIACH_MCA_TYPE               ),
    DEF_VALS1(MQIACH_MR_COUNT               ),
    DEF_VALS1(MQIACH_MR_INTERVAL            ),
    DEF_VALS1(MQIACH_MSGS                   ),
    DEF_VALS1(MQIACH_MSG_COMPRESSION        ),
    DEF_VALS1(MQIACH_MSG_SEQUENCE_NUMBER    ),
    DEF_VALS1(MQIACH_NAME_COUNT             ),
    DEF_VALS1(MQIACH_NETWORK_PRIORITY       ),
    DEF_VALS1(MQIACH_NETWORK_TIME_INDICATOR ),
    DEF_VALS1(MQIACH_NPM_SPEED              ),
    DEF_VALS1(MQIACH_PORT                   ),
    DEF_VALS1(MQIACH_PORT_NUMBER            ),
    DEF_VALS1(MQIACH_PUT_AUTHORITY          ),
    DEF_VALS1(MQIACH_SEQUENCE_NUMBER_WRAP   ),
    DEF_VALS1(MQIACH_SESSION_COUNT          ),
    DEF_VALS1(MQIACH_SHARED_CHL_RESTART     ),
    DEF_VALS1(MQIACH_SHARING_CONVERSATIONS  ),
    DEF_VALS1(MQIACH_SHORT_RETRIES_LEFT     ),
    DEF_VALS1(MQIACH_SHORT_RETRY            ),
    DEF_VALS1(MQIACH_SHORT_TIMER            ),
    DEF_VALS1(MQIACH_SOCKET                 ),
    DEF_VALS1(MQIACH_SSLTASKS_MAX           ),
    DEF_VALS1(MQIACH_SSLTASKS_STARTED       ),
    DEF_VALS1(MQIACH_SSL_CLIENT_AUTH        ),
    DEF_VALS1(MQIACH_SSL_KEY_RESETS         ),
    DEF_VALS1(MQIACH_SSL_RETURN_CODE        ),
    DEF_VALS1(MQIACH_STOP_REQUESTED         ),
    DEF_VALS1(MQIACH_XMITQ_MSGS_AVAILABLE   ),
    DEF_VALS1(MQIACH_XMITQ_TIME_INDICATOR   ),
    DEF_VALS1(MQIACH_XMIT_PROTOCOL_TYPE     ),
    DEF_VALS2(MQIACH_BUFFERS_RECEIVED,"MQIACH_BUFFERS_RCVD"),
    DEF_VALS2(MQIACH_BYTES_RECEIVED,"MQIACH_BYTES_RCVD"),
    DEF_VALS2(MQIACH_CURRENT_SEQUENCE_NUMBER,"MQIACH_CURRENT_SEQ_NUMBER"),
    DEF_VALS2(MQIACH_LAST_SEQUENCE_NUMBER, "MQIACH_LAST_SEQ_NUMBER"),
    DEF_VALS1(MQIACH_AVAILABLE_CIPHERSPECS  ),
    DEF_VALS1(MQIACH_BATCH_DATA_LIMIT       ),
    DEF_VALS1(MQIACH_CHANNEL_SUMMARY_ATTRS  ),
    DEF_VALS1(MQIACH_DEF_RECONNECT          ),
    DEF_VALS1(MQIACH_IN_DOUBT_IN            ),
    DEF_VALS1(MQIACH_IN_DOUBT_OUT           ),
    DEF_VALS1(MQIACH_MATCH                  ),
    DEF_VALS1(MQIACH_MC_HB_INTERVAL         ),
    DEF_VALS1(MQIACH_MQTT_KEEP_ALIVE        ),
    DEF_VALS1(MQIACH_MSGS_RCVD              ),
    DEF_VALS1(MQIACH_MSGS_RECEIVED          ),
    DEF_VALS1(MQIACH_MSGS_SENT              ),
    DEF_VALS1(MQIACH_MSG_HISTORY            ),
    DEF_VALS1(MQIACH_MULTICAST_PROPERTIES   ),
    DEF_VALS1(MQIACH_NEW_SUBSCRIBER_HISTORY ),
    DEF_VALS1(MQIACH_PENDING_OUT            ),
    DEF_VALS1(MQIACH_RESET_REQUESTED        ),
    DEF_VALS1(MQIACH_USER_SOURCE            ),
    DEF_VALS1(MQIACH_USE_CLIENT_ID          ),
    DEF_VALS1(MQIACH_WARNING                ),

    DEF_VALS1(MQIAMO64_BROWSE_BYTES         ),
    DEF_VALS1(MQIAMO64_BYTES                ),
    DEF_VALS1(MQIAMO64_GET_BYTES            ),
    DEF_VALS1(MQIAMO64_PUBLISH_MSG_BYTES    ),
    DEF_VALS1(MQIAMO64_PUT_BYTES            ),
    DEF_VALS1(MQIAMO64_TOPIC_PUT_BYTES      ),
    DEF_VALS1(MQIAMO64_AVG_Q_TIME           ),
    DEF_VALS1(MQIAMO64_Q_TIME_AVG           ),
    DEF_VALS1(MQIAMO64_Q_TIME_MAX           ),
    DEF_VALS1(MQIAMO64_Q_TIME_MIN           ),
    DEF_VALS1(MQIAMO64_HIGHRES_TIME         ),

    DEF_VALS1(MQIAMO_AVG_BATCH_SIZE         ),
    DEF_VALS1(MQIAMO_BACKOUTS               ),
    DEF_VALS1(MQIAMO_BROWSES                ),
    DEF_VALS1(MQIAMO_BROWSES_FAILED         ),
    DEF_VALS1(MQIAMO_BROWSE_MAX_BYTES       ),
    DEF_VALS1(MQIAMO_BROWSE_MIN_BYTES       ),
    DEF_VALS1(MQIAMO_CBS                    ),
    DEF_VALS1(MQIAMO_CBS_FAILED             ),
    DEF_VALS1(MQIAMO_CLOSES                 ),
    DEF_VALS1(MQIAMO_CLOSES_FAILED          ),
    DEF_VALS1(MQIAMO_COMMITS                ),
    DEF_VALS1(MQIAMO_COMMITS_FAILED         ),
    DEF_VALS1(MQIAMO_CONNS                  ),
    DEF_VALS1(MQIAMO_CONNS_FAILED           ),
    DEF_VALS1(MQIAMO_CONNS_MAX              ),
    DEF_VALS1(MQIAMO_CTLS                   ),
    DEF_VALS1(MQIAMO_CTLS_FAILED            ),
    DEF_VALS1(MQIAMO_DISCS                  ),
    DEF_VALS1(MQIAMO_DISCS_IMPLICIT         ),
    DEF_VALS1(MQIAMO_DISC_TYPE              ),
    DEF_VALS1(MQIAMO_EXIT_TIME_AVG          ),
    DEF_VALS1(MQIAMO_EXIT_TIME_MAX          ),
    DEF_VALS1(MQIAMO_EXIT_TIME_MIN          ),
    DEF_VALS1(MQIAMO_FULL_BATCHES           ),
    DEF_VALS1(MQIAMO_GENERATED_MSGS         ),
    DEF_VALS1(MQIAMO_GETS                   ),
    DEF_VALS1(MQIAMO_GETS_FAILED            ),
    DEF_VALS1(MQIAMO_GET_MAX_BYTES          ),
    DEF_VALS1(MQIAMO_GET_MIN_BYTES          ),
    DEF_VALS1(MQIAMO_INCOMPLETE_BATCHES     ),
    DEF_VALS1(MQIAMO_INQS                   ),
    DEF_VALS1(MQIAMO_INQS_FAILED            ),
    DEF_VALS1(MQIAMO_MSGS                   ),
    DEF_VALS1(MQIAMO_MSGS_EXPIRED           ),
    DEF_VALS1(MQIAMO_MSGS_NOT_QUEUED        ),
    DEF_VALS1(MQIAMO_MSGS_PURGED            ),
    DEF_VALS1(MQIAMO_NET_TIME_AVG           ),
    DEF_VALS1(MQIAMO_NET_TIME_MAX           ),
    DEF_VALS1(MQIAMO_NET_TIME_MIN           ),
    DEF_VALS1(MQIAMO_OBJECT_COUNT           ),
    DEF_VALS1(MQIAMO_OPENS                  ),
    DEF_VALS1(MQIAMO_OPENS_FAILED           ),
    DEF_VALS1(MQIAMO_PUBLISH_MSG_COUNT      ),
    DEF_VALS1(MQIAMO_PUT1S                  ),
    DEF_VALS1(MQIAMO_PUT1S_FAILED           ),
    DEF_VALS1(MQIAMO_PUTS                   ),
    DEF_VALS1(MQIAMO_PUTS_FAILED            ),
    DEF_VALS1(MQIAMO_PUT_MAX_BYTES          ),
    DEF_VALS1(MQIAMO_PUT_MIN_BYTES          ),
    DEF_VALS1(MQIAMO_PUT_RETRIES            ),
    DEF_VALS1(MQIAMO_Q_MAX_DEPTH            ),
    DEF_VALS1(MQIAMO_Q_MIN_DEPTH            ),
    DEF_VALS1(MQIAMO_SETS                   ),
    DEF_VALS1(MQIAMO_SETS_FAILED            ),
    DEF_VALS1(MQIAMO_STATS                  ),
    DEF_VALS1(MQIAMO_STATS_FAILED           ),
    DEF_VALS1(MQIAMO_SUBRQS                 ),
    DEF_VALS1(MQIAMO_SUBRQS_FAILED          ),
    DEF_VALS1(MQIAMO_SUBS_DUR               ),
    DEF_VALS1(MQIAMO_SUBS_FAILED            ),
    DEF_VALS1(MQIAMO_SUBS_NDUR              ),
    DEF_VALS1(MQIAMO_SUB_DUR_HIGHWATER      ),
    DEF_VALS1(MQIAMO_SUB_DUR_LOWWATER       ),
    DEF_VALS1(MQIAMO_SUB_NDUR_HIGHWATER     ),
    DEF_VALS1(MQIAMO_SUB_NDUR_LOWWATER      ),
    DEF_VALS1(MQIAMO_TOPIC_PUT1S            ),
    DEF_VALS1(MQIAMO_TOPIC_PUT1S_FAILED     ),
    DEF_VALS1(MQIAMO_TOPIC_PUTS             ),
    DEF_VALS1(MQIAMO_TOPIC_PUTS_FAILED      ),
    DEF_VALS1(MQIAMO_UNSUBS_DUR             ),
    DEF_VALS1(MQIAMO_UNSUBS_FAILED          ),
    DEF_VALS1(MQIAMO_UNSUBS_NDUR            ),
    DEF_VALS2(MQIAMO_AVG_Q_TIME,"MQIAMO64_AVG_Q_TIME"),
    DEF_VALS2(MQIAMO_Q_TIME_AVG,"MQIAMO64_Q_TIME_AVG"),
    DEF_VALS2(MQIAMO_Q_TIME_MAX,"MQIAMO64_Q_TIME_MAX"),
    DEF_VALS2(MQIAMO_Q_TIME_MIN,"MQIAMO64_Q_TIME_MIN"),
    DEF_VALS1(MQIAMO_ACKS_RCVD              ),
    DEF_VALS1(MQIAMO_ACK_FEEDBACK           ),
    DEF_VALS1(MQIAMO_ACTIVE_ACKERS          ),
    DEF_VALS1(MQIAMO_BYTES_SENT             ),
    DEF_VALS1(MQIAMO_DEST_DATA_PORT         ),
    DEF_VALS1(MQIAMO_DEST_REPAIR_PORT       ),
    DEF_VALS1(MQIAMO_FEEDBACK_MODE          ),
    DEF_VALS1(MQIAMO_HISTORY_PKTS           ),
    DEF_VALS1(MQIAMO_INTERVAL               ),
    DEF_VALS1(MQIAMO_LAST_USED              ),
    DEF_VALS1(MQIAMO_LATE_JOIN_MARK         ),
    DEF_VALS1(MQIAMO_MCAST_BATCH_TIME       ),
    DEF_VALS1(MQIAMO_MCAST_HEARTBEAT        ),
    DEF_VALS1(MQIAMO_MCAST_XMIT_RATE        ),
    DEF_VALS1(MQIAMO_MSGS_DELIVERED         ),
    DEF_VALS1(MQIAMO_MSGS_RCVD              ),
    DEF_VALS1(MQIAMO_MSGS_SENT              ),
    DEF_VALS1(MQIAMO_MSG_BYTES_RCVD         ),
    DEF_VALS1(MQIAMO_NACKS_CREATED          ),
    DEF_VALS1(MQIAMO_NACKS_RCVD             ),
    DEF_VALS1(MQIAMO_NACK_FEEDBACK          ),
    DEF_VALS1(MQIAMO_NACK_PKTS_SENT         ),
    DEF_VALS1(MQIAMO_NUM_STREAMS            ),
    DEF_VALS1(MQIAMO_PENDING_PKTS           ),
    DEF_VALS1(MQIAMO_PKTS_DELIVERED         ),
    DEF_VALS1(MQIAMO_PKTS_DROPPED           ),
    DEF_VALS1(MQIAMO_PKTS_DUPLICATED        ),
    DEF_VALS1(MQIAMO_PKTS_LOST              ),
    DEF_VALS1(MQIAMO_PKTS_PROCESSED         ),
    DEF_VALS1(MQIAMO_PKTS_REPAIRED          ),
    DEF_VALS1(MQIAMO_PKTS_SENT              ),
    DEF_VALS1(MQIAMO_PKT_RATE               ),
    DEF_VALS1(MQIAMO_RELIABILITY_TYPE       ),
    DEF_VALS1(MQIAMO_REPAIR_BYTES           ),
    DEF_VALS1(MQIAMO_REPAIR_PKTS            ),
    DEF_VALS1(MQIAMO_REPAIR_PKTS_RCVD       ),
    DEF_VALS1(MQIAMO_REPAIR_PKTS_RQSTD      ),
    DEF_VALS1(MQIAMO_TOTAL_BYTES_SENT       ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_DELIVERED   ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_EXPIRED     ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_PROCESSED   ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_RCVD        ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_RETURNED    ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_SELECTED    ),
    DEF_VALS1(MQIAMO_TOTAL_MSGS_SENT        ),
    DEF_VALS1(MQIAMO_TOTAL_MSG_BYTES_RCVD   ),
    DEF_VALS1(MQIAMO_TOTAL_PKTS_SENT        ),
    DEF_VALS1(MQIAMO_TOTAL_REPAIR_PKTS      ),
    DEF_VALS1(MQIAMO_TOTAL_REPAIR_PKTS_RCVD ),
    DEF_VALS1(MQIAMO_TOTAL_REPAIR_PKTS_RQSTD),

    DEF_VALS1(MQIACH_AVAILABLE_CIPHERSPECS  ),
    DEF_VALS1(MQIACH_BATCH_DATA_LIMIT       ),
    DEF_VALS1(MQIACH_CHANNEL_SUMMARY_ATTRS  ),
    DEF_VALS1(MQIACH_DEF_RECONNECT          ),
    DEF_VALS1(MQIACH_MATCH                  ),
    DEF_VALS2(MQIACH_MSGS_RECEIVED,"MQIACH_MSGS_RCVD"),
    DEF_VALS1(MQIACH_MSGS_SENT              ),
    DEF_VALS1(MQIACH_PENDING_OUT            ),
    DEF_VALS1(MQCACH_REMOTE_VERSION         ),
    DEF_VALS1(MQCACH_REMOTE_PRODUCT         ),
    DEF_VALS1(MQIACH_RESET_REQUESTED        ),
    DEF_VALS1(MQIACH_USER_SOURCE            ),
    DEF_VALS1(MQIACH_WARNING                ),

    DEF_VALS1(MQIA_ACCOUNTING_CONN_OVERRIDE ),
    DEF_VALS1(MQIA_ACCOUNTING_INTERVAL      ),
    DEF_VALS1(MQIA_ACCOUNTING_MQI           ),
    DEF_VALS1(MQIA_ACCOUNTING_Q             ),
    DEF_VALS1(MQIA_ACTIVE_CHANNELS          ),
    DEF_VALS1(MQIA_ACTIVITY_RECORDING       ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_CHECK        ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_INTERVAL     ),
    DEF_VALS1(MQIA_ADOPTNEWMCA_TYPE         ),
    DEF_VALS1(MQIA_APPL_TYPE                ),
    DEF_VALS1(MQIA_ARCHIVE                  ),
    DEF_VALS1(MQIA_AUTHORITY_EVENT          ),
    DEF_VALS1(MQIA_AUTH_INFO_TYPE           ),
    DEF_VALS1(MQIA_AUTO_REORGANIZATION      ),
    DEF_VALS1(MQIA_AUTO_REORG_INTERVAL      ),
    DEF_VALS1(MQIA_BACKOUT_THRESHOLD        ),
    DEF_VALS1(MQIA_BASE_TYPE                ),
    DEF_VALS1(MQIA_BATCH_INTERFACE_AUTO     ),
    DEF_VALS1(MQIA_BRIDGE_EVENT             ),
    DEF_VALS1(MQIA_CF_LEVEL                 ),
    DEF_VALS1(MQIA_CF_RECOVER               ),
    DEF_VALS1(MQIA_CHANNEL_AUTO_DEF         ),
    DEF_VALS1(MQIA_CHANNEL_AUTO_DEF_EVENT   ),
    DEF_VALS1(MQIA_CHANNEL_EVENT            ),
    DEF_VALS1(MQIA_CHINIT_ADAPTERS          ),
    DEF_VALS1(MQIA_CHINIT_CONTROL           ),
    DEF_VALS1(MQIA_CHINIT_DISPATCHERS       ),
    DEF_VALS1(MQIA_CHINIT_TRACE_AUTO_START  ),
    DEF_VALS1(MQIA_CHINIT_TRACE_TABLE_SIZE  ),
    DEF_VALS1(MQIA_CLUSTER_Q_TYPE           ),
    DEF_VALS1(MQIA_CLUSTER_WORKLOAD_LENGTH  ),
    DEF_VALS1(MQIA_CLWL_MRU_CHANNELS        ),
    DEF_VALS1(MQIA_CLWL_Q_PRIORITY          ),
    DEF_VALS1(MQIA_CLWL_Q_RANK              ),
    DEF_VALS1(MQIA_CLWL_USEQ                ),
    DEF_VALS1(MQIA_CMD_SERVER_AUTO          ),
    DEF_VALS1(MQIA_CMD_SERVER_CONTROL       ),
    DEF_VALS1(MQIA_CMD_SERVER_CONVERT_MSG   ),
    DEF_VALS1(MQIA_CMD_SERVER_DLQ_MSG       ),
    DEF_VALS1(MQIA_CODED_CHAR_SET_ID        ),
    DEF_VALS1(MQIA_COMMAND_EVENT            ),
    DEF_VALS1(MQIA_COMMAND_LEVEL            ),
    DEF_VALS1(MQIA_CONFIGURATION_EVENT      ),
    DEF_VALS1(MQIA_CPI_LEVEL                ),
    DEF_VALS1(MQIA_CURRENT_Q_DEPTH          ),
    DEF_VALS1(MQIA_DEFINITION_TYPE          ),
    DEF_VALS1(MQIA_DEF_BIND                 ),
    DEF_VALS1(MQIA_DEF_INPUT_OPEN_OPTION    ),
    DEF_VALS1(MQIA_DEF_PERSISTENCE          ),
    DEF_VALS1(MQIA_DEF_PRIORITY             ),
    DEF_VALS1(MQIA_DEF_PUT_RESPONSE_TYPE    ),
    DEF_VALS1(MQIA_DEF_READ_AHEAD           ),
    DEF_VALS1(MQIA_DIST_LISTS               ),
    DEF_VALS1(MQIA_DNS_WLM                  ),
    DEF_VALS1(MQIA_DURABLE_SUB              ),
    DEF_VALS1(MQIA_EXPIRY_INTERVAL          ),
    DEF_VALS1(MQIA_GROUP_UR                 ),
    DEF_VALS1(MQIA_HARDEN_GET_BACKOUT       ),
    DEF_VALS1(MQIA_HIGH_Q_DEPTH             ),
    DEF_VALS1(MQIA_IGQ_PUT_AUTHORITY        ),
    DEF_VALS1(MQIA_INDEX_TYPE               ),
    DEF_VALS1(MQIA_INHIBIT_EVENT            ),
    DEF_VALS1(MQIA_INHIBIT_GET              ),
    DEF_VALS1(MQIA_INHIBIT_PUB              ),
    DEF_VALS1(MQIA_INHIBIT_PUT              ),
    DEF_VALS1(MQIA_INHIBIT_SUB              ),
    DEF_VALS1(MQIA_INTRA_GROUP_QUEUING      ),
    DEF_VALS1(MQIA_IP_ADDRESS_VERSION       ),
    DEF_VALS1(MQIA_LISTENER_PORT_NUMBER     ),
    DEF_VALS1(MQIA_LISTENER_TIMER           ),
    DEF_VALS1(MQIA_LOCAL_EVENT              ),
    DEF_VALS1(MQIA_LOGGER_EVENT             ),
    DEF_VALS1(MQIA_LU62_CHANNELS            ),
    DEF_VALS1(MQIA_MASTER_ADMIN             ),
    DEF_VALS1(MQIA_MAX_CHANNELS             ),
    DEF_VALS1(MQIA_MAX_CLIENTS              ),
    DEF_VALS1(MQIA_MAX_GLOBAL_LOCKS         ),
    DEF_VALS1(MQIA_MAX_HANDLES              ),
    DEF_VALS1(MQIA_MAX_LOCAL_LOCKS          ),
    DEF_VALS1(MQIA_MAX_MSG_LENGTH           ),
    DEF_VALS1(MQIA_MAX_OPEN_Q               ),
    DEF_VALS1(MQIA_MAX_PRIORITY             ),
    DEF_VALS1(MQIA_MAX_PROPERTIES_LENGTH    ),
    DEF_VALS1(MQIA_MAX_Q_DEPTH              ),
    DEF_VALS1(MQIA_MAX_Q_TRIGGERS           ),
    DEF_VALS1(MQIA_MAX_RECOVERY_TASKS       ),
    DEF_VALS1(MQIA_MAX_UNCOMMITTED_MSGS     ),
    DEF_VALS1(MQIA_MONITORING_AUTO_CLUSSDR  ),
    DEF_VALS1(MQIA_MONITORING_CHANNEL       ),
    DEF_VALS1(MQIA_MONITORING_Q             ),
    DEF_VALS1(MQIA_MONITOR_INTERVAL         ),
    DEF_VALS1(MQIA_MSG_DELIVERY_SEQUENCE    ),
    DEF_VALS1(MQIA_MSG_DEQ_COUNT            ),
    DEF_VALS1(MQIA_MSG_ENQ_COUNT            ),
    DEF_VALS1(MQIA_MSG_MARK_BROWSE_INTERVAL ),
    DEF_VALS1(MQIA_NAMELIST_TYPE            ),
    DEF_VALS1(MQIA_NAME_COUNT               ),
    DEF_VALS1(MQIA_NPM_CLASS                ),
    DEF_VALS1(MQIA_NPM_DELIVERY             ),
    DEF_VALS1(MQIA_OPEN_INPUT_COUNT         ),
    DEF_VALS1(MQIA_OPEN_OUTPUT_COUNT        ),
    DEF_VALS1(MQIA_OUTBOUND_PORT_MAX        ),
    DEF_VALS1(MQIA_OUTBOUND_PORT_MIN        ),
    DEF_VALS1(MQIA_PAGESET_ID               ),
    DEF_VALS1(MQIA_PERFORMANCE_EVENT        ),
    DEF_VALS1(MQIA_PLATFORM                 ),
    DEF_VALS1(MQIA_PM_DELIVERY              ),
    DEF_VALS1(MQIA_PROPERTY_CONTROL         ),
    DEF_VALS1(MQIA_PROXY_SUB                ),
    DEF_VALS1(MQIA_PUBSUB_MAXMSG_RETRY_COUNT),
    DEF_VALS1(MQIA_PUBSUB_MODE              ),
    DEF_VALS1(MQIA_PUBSUB_NP_MSG            ),
    DEF_VALS1(MQIA_PUBSUB_NP_RESP           ),
    DEF_VALS1(MQIA_PUBSUB_SYNC_PT           ),
    DEF_VALS1(MQIA_PUB_COUNT                ),
    DEF_VALS1(MQIA_PUB_SCOPE                ),
    DEF_VALS1(MQIA_QMGR_CFCONLOS            ),
    DEF_VALS1(MQIA_QMOPT_CONS_COMMS_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_CONS_CRITICAL_MSGS ),
    DEF_VALS1(MQIA_QMOPT_CONS_ERROR_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_CONS_INFO_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_CONS_REORG_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_CONS_SYSTEM_MSGS   ),
    DEF_VALS1(MQIA_QMOPT_CONS_WARNING_MSGS  ),
    DEF_VALS1(MQIA_QMOPT_CSMT_ON_ERROR      ),
    DEF_VALS1(MQIA_QMOPT_INTERNAL_DUMP      ),
    DEF_VALS1(MQIA_QMOPT_LOG_COMMS_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_LOG_CRITICAL_MSGS  ),
    DEF_VALS1(MQIA_QMOPT_LOG_ERROR_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_LOG_INFO_MSGS      ),
    DEF_VALS1(MQIA_QMOPT_LOG_REORG_MSGS     ),
    DEF_VALS1(MQIA_QMOPT_LOG_SYSTEM_MSGS    ),
    DEF_VALS1(MQIA_QMOPT_LOG_WARNING_MSGS   ),
    DEF_VALS1(MQIA_QMOPT_TRACE_COMMS        ),
    DEF_VALS1(MQIA_QMOPT_TRACE_CONVERSION   ),
    DEF_VALS1(MQIA_QMOPT_TRACE_MQI_CALLS    ),
    DEF_VALS1(MQIA_QMOPT_TRACE_REORG        ),
    DEF_VALS1(MQIA_QMOPT_TRACE_SYSTEM       ),
    DEF_VALS1(MQIA_QSG_DISP                 ),
    DEF_VALS1(MQIA_Q_DEPTH_HIGH_EVENT       ),
    DEF_VALS1(MQIA_Q_DEPTH_HIGH_LIMIT       ),
    DEF_VALS1(MQIA_Q_DEPTH_LOW_EVENT        ),
    DEF_VALS1(MQIA_Q_DEPTH_LOW_LIMIT        ),
    DEF_VALS1(MQIA_Q_DEPTH_MAX_EVENT        ),
    DEF_VALS1(MQIA_Q_SERVICE_INTERVAL       ),
    DEF_VALS1(MQIA_Q_SERVICE_INTERVAL_EVENT ),
    DEF_VALS1(MQIA_Q_TYPE                   ),
    DEF_VALS1(MQIA_Q_USERS                  ),
    DEF_VALS1(MQIA_READ_AHEAD               ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT          ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT_MIN      ),
    DEF_VALS1(MQIA_RECEIVE_TIMEOUT_TYPE     ),
    DEF_VALS1(MQIA_REMOTE_EVENT             ),
    DEF_VALS1(MQIA_RETENTION_INTERVAL       ),
    DEF_VALS1(MQIA_SCOPE                    ),
    DEF_VALS1(MQIA_SECURITY_CASE            ),
    DEF_VALS1(MQIA_SERVICE_CONTROL          ),
    DEF_VALS1(MQIA_SERVICE_TYPE             ),
    DEF_VALS1(MQIA_SHAREABILITY             ),
    DEF_VALS1(MQIA_SHARED_Q_Q_MGR_NAME      ),
    DEF_VALS1(MQIA_SSL_EVENT                ),
    DEF_VALS1(MQIA_SSL_FIPS_REQUIRED        ),
    DEF_VALS1(MQIA_SSL_RESET_COUNT          ),
    DEF_VALS1(MQIA_SSL_TASKS                ),
    DEF_VALS1(MQIA_START_STOP_EVENT         ),
    DEF_VALS1(MQIA_STATISTICS_AUTO_CLUSSDR  ),
    DEF_VALS1(MQIA_STATISTICS_CHANNEL       ),
    DEF_VALS1(MQIA_STATISTICS_INTERVAL      ),
    DEF_VALS1(MQIA_STATISTICS_MQI           ),
    DEF_VALS1(MQIA_STATISTICS_Q             ),
    DEF_VALS1(MQIA_SUB_COUNT                ),
    DEF_VALS1(MQIA_SUB_SCOPE                ),
    DEF_VALS1(MQIA_SYNCPOINT                ),
    DEF_VALS1(MQIA_TCP_CHANNELS             ),
    DEF_VALS1(MQIA_TCP_KEEP_ALIVE           ),
    DEF_VALS1(MQIA_TCP_STACK_TYPE           ),
    DEF_VALS1(MQIA_TIME_SINCE_RESET         ),
    DEF_VALS1(MQIA_TOPIC_DEF_PERSISTENCE    ),
    DEF_VALS1(MQIA_TOPIC_TYPE               ),
    DEF_VALS1(MQIA_TRACE_ROUTE_RECORDING    ),
    DEF_VALS1(MQIA_TREE_LIFE_TIME           ),
    DEF_VALS1(MQIA_TRIGGER_CONTROL          ),
    DEF_VALS1(MQIA_TRIGGER_DEPTH            ),
    DEF_VALS1(MQIA_TRIGGER_INTERVAL         ),
    DEF_VALS1(MQIA_TRIGGER_MSG_PRIORITY     ),
    DEF_VALS1(MQIA_TRIGGER_RESTART          ),
    DEF_VALS1(MQIA_TRIGGER_TYPE             ),
    DEF_VALS1(MQIA_UR_DISP                  ),
    DEF_VALS1(MQIA_USAGE                    ),
    DEF_VALS1(MQIA_USER_LIST                ),
    DEF_VALS1(MQIA_WILDCARD_OPERATION       ),
    DEF_VALS1(MQIA_ACTIVITY_CONN_OVERRIDE   ),
    DEF_VALS1(MQIA_ACTIVITY_TRACE           ),
    DEF_VALS1(MQIA_CERT_VAL_POLICY          ),
    DEF_VALS1(MQIA_CHLAUTH_RECORDS          ),
    DEF_VALS1(MQIA_COMM_EVENT               ),
    DEF_VALS1(MQIA_COMM_INFO_TYPE           ),
    DEF_VALS1(MQIA_DEF_CLUSTER_XMIT_Q_TYPE  ),
    DEF_VALS1(MQIA_ENCRYPTION_ALGORITHM     ),
    DEF_VALS1(MQIA_MCAST_BRIDGE             ),
    DEF_VALS1(MQIA_MULTICAST                ),
    DEF_VALS1(MQIA_POLICY_VERSION           ),
    DEF_VALS1(MQIA_PROT_POLICY_CAPABILITY   ),
    DEF_VALS1(MQIA_PUBSUB_CLUSTER           ),
    DEF_VALS1(MQIA_SIGNATURE_ALGORITHM      ),
    DEF_VALS1(MQIA_SUB_CONFIGURATION_EVENT  ),
    DEF_VALS1(MQIA_SUITE_B_STRENGTH         ),
    DEF_VALS1(MQIA_TOLERATE_UNPROTECTED     ),
    DEF_VALS1(MQIA_USE_DEAD_LETTER_Q        ),
    DEF_VALS1(MQIA_XR_CAPABILITY            ),
DEF_VALSE;

#if 0
DEF_VALSB(CC)
    DEF_VALS1(MQCC_OK     ),
    DEF_VALS1(MQCC_WARNING),
    DEF_VALS1(MQCC_FAILED ),
    DEF_VALS1(MQCC_UNKNOWN),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(RC)
    DEF_VALS1(MQRC_NONE                      ),
    DEF_VALS1(MQRC_ALIAS_BASE_Q_TYPE_ERROR   ),
    DEF_VALS1(MQRC_ALREADY_CONNECTED         ),
    DEF_VALS1(MQRC_BACKED_OUT                ),
    DEF_VALS1(MQRC_BUFFER_ERROR              ),
    DEF_VALS1(MQRC_BUFFER_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_CHAR_ATTR_LENGTH_ERROR    ),
    DEF_VALS1(MQRC_CHAR_ATTRS_ERROR          ),
    DEF_VALS1(MQRC_CHAR_ATTRS_TOO_SHORT      ),
    DEF_VALS1(MQRC_CONNECTION_BROKEN         ),
    DEF_VALS1(MQRC_DATA_LENGTH_ERROR         ),
    DEF_VALS1(MQRC_DYNAMIC_Q_NAME_ERROR      ),
    DEF_VALS1(MQRC_ENVIRONMENT_ERROR         ),
    DEF_VALS1(MQRC_EXPIRY_ERROR              ),
    DEF_VALS1(MQRC_FEEDBACK_ERROR            ),
    DEF_VALS1(MQRC_GET_INHIBITED             ),
    DEF_VALS1(MQRC_HANDLE_NOT_AVAILABLE      ),
    DEF_VALS1(MQRC_HCONN_ERROR               ),
    DEF_VALS1(MQRC_HOBJ_ERROR                ),
    DEF_VALS1(MQRC_INHIBIT_VALUE_ERROR       ),
    DEF_VALS1(MQRC_INT_ATTR_COUNT_ERROR      ),
    DEF_VALS1(MQRC_INT_ATTR_COUNT_TOO_SMALL  ),
    DEF_VALS1(MQRC_INT_ATTRS_ARRAY_ERROR     ),
    DEF_VALS1(MQRC_SYNCPOINT_LIMIT_REACHED   ),
    DEF_VALS1(MQRC_MAX_CONNS_LIMIT_REACHED   ),
    DEF_VALS1(MQRC_MD_ERROR                  ),
    DEF_VALS1(MQRC_MISSING_REPLY_TO_Q        ),
    DEF_VALS1(MQRC_MSG_TYPE_ERROR            ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_Q         ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_Q_MGR     ),
    DEF_VALS1(MQRC_NO_MSG_AVAILABLE          ),
    DEF_VALS1(MQRC_NO_MSG_UNDER_CURSOR       ),
    DEF_VALS1(MQRC_NOT_AUTHORIZED            ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_BROWSE       ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_INPUT        ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_INQUIRE      ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_OUTPUT       ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET          ),
    DEF_VALS1(MQRC_OBJECT_CHANGED            ),
    DEF_VALS1(MQRC_OBJECT_IN_USE             ),
    DEF_VALS1(MQRC_OBJECT_TYPE_ERROR         ),
    DEF_VALS1(MQRC_OD_ERROR                  ),
    DEF_VALS1(MQRC_OPTION_NOT_VALID_FOR_TYPE ),
    DEF_VALS1(MQRC_OPTIONS_ERROR             ),
    DEF_VALS1(MQRC_PERSISTENCE_ERROR         ),
    DEF_VALS1(MQRC_PERSISTENT_NOT_ALLOWED    ),
    DEF_VALS1(MQRC_PRIORITY_EXCEEDS_MAXIMUM  ),
    DEF_VALS1(MQRC_PRIORITY_ERROR            ),
    DEF_VALS1(MQRC_PUT_INHIBITED             ),
    DEF_VALS1(MQRC_Q_DELETED                 ),
    DEF_VALS1(MQRC_Q_FULL                    ),
    DEF_VALS1(MQRC_Q_NOT_EMPTY               ),
    DEF_VALS1(MQRC_Q_SPACE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_Q_TYPE_ERROR              ),
    DEF_VALS1(MQRC_Q_MGR_NAME_ERROR          ),
    DEF_VALS1(MQRC_Q_MGR_NOT_AVAILABLE       ),
    DEF_VALS1(MQRC_REPORT_OPTIONS_ERROR      ),
    DEF_VALS1(MQRC_SECOND_MARK_NOT_ALLOWED   ),
    DEF_VALS1(MQRC_SECURITY_ERROR            ),
    DEF_VALS1(MQRC_SELECTOR_COUNT_ERROR      ),
    DEF_VALS1(MQRC_SELECTOR_LIMIT_EXCEEDED   ),
    DEF_VALS1(MQRC_SELECTOR_ERROR            ),
    DEF_VALS1(MQRC_SELECTOR_NOT_FOR_TYPE     ),
    DEF_VALS1(MQRC_SIGNAL_OUTSTANDING        ),
    DEF_VALS1(MQRC_SIGNAL_REQUEST_ACCEPTED   ),
    DEF_VALS1(MQRC_STORAGE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_SYNCPOINT_NOT_AVAILABLE   ),
    DEF_VALS1(MQRC_TRIGGER_CONTROL_ERROR     ),
    DEF_VALS1(MQRC_TRIGGER_DEPTH_ERROR       ),
    DEF_VALS1(MQRC_TRIGGER_MSG_PRIORITY_ERR  ),
    DEF_VALS1(MQRC_TRIGGER_TYPE_ERROR        ),
    DEF_VALS1(MQRC_TRUNCATED_MSG_ACCEPTED    ),
    DEF_VALS1(MQRC_TRUNCATED_MSG_FAILED      ),
    DEF_VALS1(MQRC_UNKNOWN_ALIAS_BASE_Q      ),
    DEF_VALS1(MQRC_UNKNOWN_OBJECT_NAME       ),
    DEF_VALS1(MQRC_UNKNOWN_OBJECT_Q_MGR      ),
    DEF_VALS1(MQRC_UNKNOWN_REMOTE_Q_MGR      ),
    DEF_VALS1(MQRC_WAIT_INTERVAL_ERROR       ),
    DEF_VALS1(MQRC_XMIT_Q_TYPE_ERROR         ),
    DEF_VALS1(MQRC_XMIT_Q_USAGE_ERROR        ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_PASS_ALL     ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_PASS_IDENT   ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET_ALL      ),
    DEF_VALS1(MQRC_NOT_OPEN_FOR_SET_IDENT    ),
    DEF_VALS1(MQRC_CONTEXT_HANDLE_ERROR      ),
    DEF_VALS1(MQRC_CONTEXT_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_SIGNAL1_ERROR             ),
    DEF_VALS1(MQRC_OBJECT_ALREADY_EXISTS     ),
    DEF_VALS1(MQRC_OBJECT_DAMAGED            ),
    DEF_VALS1(MQRC_RESOURCE_PROBLEM          ),
    DEF_VALS1(MQRC_ANOTHER_Q_MGR_CONNECTED   ),
    DEF_VALS1(MQRC_UNKNOWN_REPORT_OPTION     ),
    DEF_VALS1(MQRC_STORAGE_CLASS_ERROR       ),
    DEF_VALS1(MQRC_COD_NOT_VALID_FOR_XCF_Q   ),
    DEF_VALS1(MQRC_XWAIT_CANCELED            ),
    DEF_VALS1(MQRC_XWAIT_ERROR               ),
    DEF_VALS1(MQRC_SUPPRESSED_BY_EXIT        ),
    DEF_VALS1(MQRC_FORMAT_ERROR              ),
    DEF_VALS1(MQRC_SOURCE_CCSID_ERROR        ),
    DEF_VALS1(MQRC_SOURCE_INTEGER_ENC_ERROR  ),
    DEF_VALS1(MQRC_SOURCE_DECIMAL_ENC_ERROR  ),
    DEF_VALS1(MQRC_SOURCE_FLOAT_ENC_ERROR    ),
    DEF_VALS1(MQRC_TARGET_CCSID_ERROR        ),
    DEF_VALS1(MQRC_TARGET_INTEGER_ENC_ERROR  ),
    DEF_VALS1(MQRC_TARGET_DECIMAL_ENC_ERROR  ),
    DEF_VALS1(MQRC_TARGET_FLOAT_ENC_ERROR    ),
    DEF_VALS1(MQRC_NOT_CONVERTED             ),
    DEF_VALS2(MQRC_TRUNCATED,"MQRC_CONVERTED_MSG_TOO_BIG"),
    DEF_VALS1(MQRC_NO_EXTERNAL_PARTICIPANTS  ),
    DEF_VALS1(MQRC_PARTICIPANT_NOT_AVAILABLE ),
    DEF_VALS1(MQRC_OUTCOME_MIXED             ),
    DEF_VALS1(MQRC_OUTCOME_PENDING           ),
    DEF_VALS1(MQRC_BRIDGE_STARTED            ),
    DEF_VALS1(MQRC_BRIDGE_STOPPED            ),
    DEF_VALS1(MQRC_ADAPTER_STORAGE_SHORTAGE  ),
    DEF_VALS1(MQRC_UOW_IN_PROGRESS           ),
    DEF_VALS1(MQRC_ADAPTER_CONN_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_SERV_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_DEFS_ERROR        ),
    DEF_VALS1(MQRC_ADAPTER_DEFS_LOAD_ERROR   ),
    DEF_VALS1(MQRC_ADAPTER_CONV_LOAD_ERROR   ),
    DEF_VALS1(MQRC_BO_ERROR                  ),
    DEF_VALS1(MQRC_DH_ERROR                  ),
    DEF_VALS1(MQRC_MULTIPLE_REASONS          ),
    DEF_VALS1(MQRC_OPEN_FAILED               ),
    DEF_VALS1(MQRC_ADAPTER_DISC_LOAD_ERROR   ),
    DEF_VALS1(MQRC_CNO_ERROR                 ),
    DEF_VALS1(MQRC_CICS_WAIT_FAILED          ),
    DEF_VALS1(MQRC_DLH_ERROR                 ),
    DEF_VALS1(MQRC_HEADER_ERROR              ),
    DEF_VALS1(MQRC_SOURCE_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_TARGET_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_SOURCE_BUFFER_ERROR       ),
    DEF_VALS1(MQRC_TARGET_BUFFER_ERROR       ),
    DEF_VALS1(MQRC_IIH_ERROR                 ),
    DEF_VALS1(MQRC_PCF_ERROR                 ),
    DEF_VALS1(MQRC_DBCS_ERROR                ),
    DEF_VALS1(MQRC_OBJECT_NAME_ERROR         ),
    DEF_VALS1(MQRC_OBJECT_Q_MGR_NAME_ERROR   ),
    DEF_VALS1(MQRC_RECS_PRESENT_ERROR        ),
    DEF_VALS1(MQRC_OBJECT_RECORDS_ERROR      ),
    DEF_VALS1(MQRC_RESPONSE_RECORDS_ERROR    ),
    DEF_VALS1(MQRC_ASID_MISMATCH             ),
    DEF_VALS1(MQRC_PMO_RECORD_FLAGS_ERROR    ),
    DEF_VALS1(MQRC_PUT_MSG_RECORDS_ERROR     ),
    DEF_VALS1(MQRC_CONN_ID_IN_USE            ),
    DEF_VALS1(MQRC_Q_MGR_QUIESCING           ),
    DEF_VALS1(MQRC_Q_MGR_STOPPING            ),
    DEF_VALS1(MQRC_DUPLICATE_RECOV_COORD     ),
    DEF_VALS1(MQRC_PMO_ERROR                 ),
    DEF_VALS1(MQRC_API_EXIT_NOT_FOUND        ),
    DEF_VALS1(MQRC_API_EXIT_LOAD_ERROR       ),
    DEF_VALS1(MQRC_REMOTE_Q_NAME_ERROR       ),
    DEF_VALS1(MQRC_INCONSISTENT_PERSISTENCE  ),
    DEF_VALS1(MQRC_GMO_ERROR                 ),
    DEF_VALS1(MQRC_CICS_BRIDGE_RESTRICTION   ),
    DEF_VALS1(MQRC_STOPPED_BY_CLUSTER_EXIT   ),
    DEF_VALS1(MQRC_CLUSTER_RESOLUTION_ERROR  ),
    DEF_VALS1(MQRC_CONVERTED_STRING_TOO_BIG  ),
    DEF_VALS1(MQRC_TMC_ERROR                 ),
    DEF_VALS2(MQRC_STORAGE_MEDIUM_FULL,"MQRC_PAGESET_FULL"),
    DEF_VALS1(MQRC_PAGESET_ERROR             ),
    DEF_VALS1(MQRC_NAME_NOT_VALID_FOR_TYPE   ),
    DEF_VALS1(MQRC_UNEXPECTED_ERROR          ),
    DEF_VALS1(MQRC_UNKNOWN_XMIT_Q            ),
    DEF_VALS1(MQRC_UNKNOWN_DEF_XMIT_Q        ),
    DEF_VALS1(MQRC_DEF_XMIT_Q_TYPE_ERROR     ),
    DEF_VALS1(MQRC_DEF_XMIT_Q_USAGE_ERROR    ),
    DEF_VALS1(MQRC_MSG_MARKED_BROWSE_CO_OP   ),
    DEF_VALS1(MQRC_NAME_IN_USE               ),
    DEF_VALS1(MQRC_CONNECTION_QUIESCING      ),
    DEF_VALS1(MQRC_CONNECTION_STOPPING       ),
    DEF_VALS1(MQRC_ADAPTER_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_MSG_ID_ERROR              ),
    DEF_VALS1(MQRC_CORREL_ID_ERROR           ),
    DEF_VALS1(MQRC_FILE_SYSTEM_ERROR         ),
    DEF_VALS1(MQRC_NO_MSG_LOCKED             ),
    DEF_VALS1(MQRC_SOAP_DOTNET_ERROR         ),
    DEF_VALS1(MQRC_SOAP_AXIS_ERROR           ),
    DEF_VALS1(MQRC_SOAP_URL_ERROR            ),
    DEF_VALS1(MQRC_FILE_NOT_AUDITED          ),
    DEF_VALS1(MQRC_CONNECTION_NOT_AUTHORIZED ),
    DEF_VALS1(MQRC_MSG_TOO_BIG_FOR_CHANNEL   ),
    DEF_VALS1(MQRC_CALL_IN_PROGRESS          ),
    DEF_VALS1(MQRC_RMH_ERROR                 ),
    DEF_VALS1(MQRC_Q_MGR_ACTIVE              ),
    DEF_VALS1(MQRC_Q_MGR_NOT_ACTIVE          ),
    DEF_VALS1(MQRC_Q_DEPTH_HIGH              ),
    DEF_VALS1(MQRC_Q_DEPTH_LOW               ),
    DEF_VALS1(MQRC_Q_SERVICE_INTERVAL_HIGH   ),
    DEF_VALS1(MQRC_Q_SERVICE_INTERVAL_OK     ),
    DEF_VALS1(MQRC_RFH_HEADER_FIELD_ERROR    ),
    DEF_VALS1(MQRC_RAS_PROPERTY_ERROR        ),
    DEF_VALS1(MQRC_UNIT_OF_WORK_NOT_STARTED  ),
    DEF_VALS1(MQRC_CHANNEL_AUTO_DEF_OK       ),
    DEF_VALS1(MQRC_CHANNEL_AUTO_DEF_ERROR    ),
    DEF_VALS1(MQRC_CFH_ERROR                 ),
    DEF_VALS1(MQRC_CFIL_ERROR                ),
    DEF_VALS1(MQRC_CFIN_ERROR                ),
    DEF_VALS1(MQRC_CFSL_ERROR                ),
    DEF_VALS1(MQRC_CFST_ERROR                ),
    DEF_VALS1(MQRC_INCOMPLETE_GROUP          ),
    DEF_VALS1(MQRC_INCOMPLETE_MSG            ),
    DEF_VALS1(MQRC_INCONSISTENT_CCSIDS       ),
    DEF_VALS1(MQRC_INCONSISTENT_ENCODINGS    ),
    DEF_VALS1(MQRC_INCONSISTENT_UOW          ),
    DEF_VALS1(MQRC_INVALID_MSG_UNDER_CURSOR  ),
    DEF_VALS1(MQRC_MATCH_OPTIONS_ERROR       ),
    DEF_VALS1(MQRC_MDE_ERROR                 ),
    DEF_VALS1(MQRC_MSG_FLAGS_ERROR           ),
    DEF_VALS1(MQRC_MSG_SEQ_NUMBER_ERROR      ),
    DEF_VALS1(MQRC_OFFSET_ERROR              ),
    DEF_VALS1(MQRC_ORIGINAL_LENGTH_ERROR     ),
    DEF_VALS1(MQRC_SEGMENT_LENGTH_ZERO       ),
    DEF_VALS1(MQRC_UOW_NOT_AVAILABLE         ),
    DEF_VALS1(MQRC_WRONG_GMO_VERSION         ),
    DEF_VALS1(MQRC_WRONG_MD_VERSION          ),
    DEF_VALS1(MQRC_GROUP_ID_ERROR            ),
    DEF_VALS1(MQRC_INCONSISTENT_BROWSE       ),
    DEF_VALS1(MQRC_XQH_ERROR                 ),
    DEF_VALS1(MQRC_SRC_ENV_ERROR             ),
    DEF_VALS1(MQRC_SRC_NAME_ERROR            ),
    DEF_VALS1(MQRC_DEST_ENV_ERROR            ),
    DEF_VALS1(MQRC_DEST_NAME_ERROR           ),
    DEF_VALS1(MQRC_TM_ERROR                  ),
    DEF_VALS1(MQRC_CLUSTER_EXIT_ERROR        ),
    DEF_VALS1(MQRC_CLUSTER_EXIT_LOAD_ERROR   ),
    DEF_VALS1(MQRC_CLUSTER_PUT_INHIBITED     ),
    DEF_VALS1(MQRC_CLUSTER_RESOURCE_ERROR    ),
    DEF_VALS1(MQRC_NO_DESTINATIONS_AVAILABLE ),
    DEF_VALS1(MQRC_CONN_TAG_IN_USE           ),
    DEF_VALS1(MQRC_PARTIALLY_CONVERTED       ),
    DEF_VALS1(MQRC_CONNECTION_ERROR          ),
    DEF_VALS1(MQRC_OPTION_ENVIRONMENT_ERROR  ),
    DEF_VALS1(MQRC_CD_ERROR                  ),
    DEF_VALS1(MQRC_CLIENT_CONN_ERROR         ),
    DEF_VALS1(MQRC_CHANNEL_STOPPED_BY_USER   ),
    DEF_VALS1(MQRC_HCONFIG_ERROR             ),
    DEF_VALS1(MQRC_FUNCTION_ERROR            ),
    DEF_VALS1(MQRC_CHANNEL_STARTED           ),
    DEF_VALS1(MQRC_CHANNEL_STOPPED           ),
    DEF_VALS1(MQRC_CHANNEL_CONV_ERROR        ),
    DEF_VALS1(MQRC_SERVICE_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_INITIALIZATION_FAILED     ),
    DEF_VALS1(MQRC_TERMINATION_FAILED        ),
    DEF_VALS1(MQRC_UNKNOWN_Q_NAME            ),
    DEF_VALS1(MQRC_SERVICE_ERROR             ),
    DEF_VALS1(MQRC_Q_ALREADY_EXISTS          ),
    DEF_VALS1(MQRC_USER_ID_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_UNKNOWN_ENTITY            ),
    DEF_VALS1(MQRC_UNKNOWN_AUTH_ENTITY       ),
    DEF_VALS1(MQRC_UNKNOWN_REF_OBJECT        ),
    DEF_VALS1(MQRC_CHANNEL_ACTIVATED         ),
    DEF_VALS1(MQRC_CHANNEL_NOT_ACTIVATED     ),
    DEF_VALS1(MQRC_UOW_CANCELED              ),
    DEF_VALS1(MQRC_FUNCTION_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_SELECTOR_TYPE_ERROR       ),
    DEF_VALS1(MQRC_COMMAND_TYPE_ERROR        ),
    DEF_VALS1(MQRC_MULTIPLE_INSTANCE_ERROR   ),
    DEF_VALS1(MQRC_SYSTEM_ITEM_NOT_ALTERABLE ),
    DEF_VALS1(MQRC_BAG_CONVERSION_ERROR      ),
    DEF_VALS1(MQRC_SELECTOR_OUT_OF_RANGE     ),
    DEF_VALS1(MQRC_SELECTOR_NOT_UNIQUE       ),
    DEF_VALS1(MQRC_INDEX_NOT_PRESENT         ),
    DEF_VALS1(MQRC_STRING_ERROR              ),
    DEF_VALS1(MQRC_ENCODING_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_SELECTOR_NOT_PRESENT      ),
    DEF_VALS1(MQRC_OUT_SELECTOR_ERROR        ),
    DEF_VALS1(MQRC_STRING_TRUNCATED          ),
    DEF_VALS1(MQRC_SELECTOR_WRONG_TYPE       ),
    DEF_VALS1(MQRC_INCONSISTENT_ITEM_TYPE    ),
    DEF_VALS1(MQRC_INDEX_ERROR               ),
    DEF_VALS1(MQRC_SYSTEM_BAG_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_ITEM_COUNT_ERROR          ),
    DEF_VALS1(MQRC_FORMAT_NOT_SUPPORTED      ),
    DEF_VALS1(MQRC_SELECTOR_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_ITEM_VALUE_ERROR          ),
    DEF_VALS1(MQRC_HBAG_ERROR                ),
    DEF_VALS1(MQRC_PARAMETER_MISSING         ),
    DEF_VALS1(MQRC_CMD_SERVER_NOT_AVAILABLE  ),
    DEF_VALS1(MQRC_STRING_LENGTH_ERROR       ),
    DEF_VALS1(MQRC_INQUIRY_COMMAND_ERROR     ),
    DEF_VALS1(MQRC_NESTED_BAG_NOT_SUPPORTED  ),
    DEF_VALS1(MQRC_BAG_WRONG_TYPE            ),
    DEF_VALS1(MQRC_ITEM_TYPE_ERROR           ),
    DEF_VALS1(MQRC_SYSTEM_BAG_NOT_DELETABLE  ),
    DEF_VALS1(MQRC_SYSTEM_ITEM_NOT_DELETABLE ),
    DEF_VALS1(MQRC_CODED_CHAR_SET_ID_ERROR   ),
    DEF_VALS1(MQRC_MSG_TOKEN_ERROR           ),
    DEF_VALS1(MQRC_MISSING_WIH               ),
    DEF_VALS1(MQRC_WIH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_STRING_ERROR          ),
    DEF_VALS1(MQRC_RFH_COMMAND_ERROR         ),
    DEF_VALS1(MQRC_RFH_PARM_ERROR            ),
    DEF_VALS1(MQRC_RFH_DUPLICATE_PARM        ),
    DEF_VALS1(MQRC_RFH_PARM_MISSING          ),
    DEF_VALS1(MQRC_CHAR_CONVERSION_ERROR     ),
    DEF_VALS1(MQRC_UCS2_CONVERSION_ERROR     ),
    DEF_VALS1(MQRC_DB2_NOT_AVAILABLE         ),
    DEF_VALS1(MQRC_OBJECT_NOT_UNIQUE         ),
    DEF_VALS1(MQRC_CONN_TAG_NOT_RELEASED     ),
    DEF_VALS1(MQRC_CF_NOT_AVAILABLE          ),
    DEF_VALS1(MQRC_CF_STRUC_IN_USE           ),
    DEF_VALS1(MQRC_CF_STRUC_LIST_HDR_IN_USE  ),
    DEF_VALS1(MQRC_CF_STRUC_AUTH_FAILED      ),
    DEF_VALS1(MQRC_CF_STRUC_ERROR            ),
    DEF_VALS1(MQRC_CONN_TAG_NOT_USABLE       ),
    DEF_VALS1(MQRC_GLOBAL_UOW_CONFLICT       ),
    DEF_VALS1(MQRC_LOCAL_UOW_CONFLICT        ),
    DEF_VALS1(MQRC_HANDLE_IN_USE_FOR_UOW     ),
    DEF_VALS1(MQRC_UOW_ENLISTMENT_ERROR      ),
    DEF_VALS1(MQRC_UOW_MIX_NOT_SUPPORTED     ),
    DEF_VALS1(MQRC_WXP_ERROR                 ),
    DEF_VALS1(MQRC_CURRENT_RECORD_ERROR      ),
    DEF_VALS1(MQRC_NEXT_OFFSET_ERROR         ),
    DEF_VALS1(MQRC_NO_RECORD_AVAILABLE       ),
    DEF_VALS1(MQRC_OBJECT_LEVEL_INCOMPATIBLE ),
    DEF_VALS1(MQRC_NEXT_RECORD_ERROR         ),
    DEF_VALS1(MQRC_BACKOUT_THRESHOLD_REACHED ),
    DEF_VALS1(MQRC_MSG_NOT_MATCHED           ),
    DEF_VALS1(MQRC_JMS_FORMAT_ERROR          ),
    DEF_VALS1(MQRC_SEGMENTS_NOT_SUPPORTED    ),
    DEF_VALS1(MQRC_WRONG_CF_LEVEL            ),
    DEF_VALS1(MQRC_CONFIG_CREATE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_CHANGE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_DELETE_OBJECT      ),
    DEF_VALS1(MQRC_CONFIG_REFRESH_OBJECT     ),
    DEF_VALS1(MQRC_CHANNEL_SSL_ERROR         ),
    DEF_VALS1(MQRC_PARTICIPANT_NOT_DEFINED   ),
    DEF_VALS1(MQRC_CF_STRUC_FAILED           ),
    DEF_VALS1(MQRC_API_EXIT_ERROR            ),
    DEF_VALS1(MQRC_API_EXIT_INIT_ERROR       ),
    DEF_VALS1(MQRC_API_EXIT_TERM_ERROR       ),
    DEF_VALS1(MQRC_EXIT_REASON_ERROR         ),
    DEF_VALS1(MQRC_RESERVED_VALUE_ERROR      ),
    DEF_VALS1(MQRC_NO_DATA_AVAILABLE         ),
    DEF_VALS1(MQRC_SCO_ERROR                 ),
    DEF_VALS1(MQRC_KEY_REPOSITORY_ERROR      ),
    DEF_VALS1(MQRC_CRYPTO_HARDWARE_ERROR     ),
    DEF_VALS1(MQRC_AUTH_INFO_REC_COUNT_ERROR ),
    DEF_VALS1(MQRC_AUTH_INFO_REC_ERROR       ),
    DEF_VALS1(MQRC_AIR_ERROR                 ),
    DEF_VALS1(MQRC_AUTH_INFO_TYPE_ERROR      ),
    DEF_VALS1(MQRC_AUTH_INFO_CONN_NAME_ERROR ),
    DEF_VALS1(MQRC_LDAP_USER_NAME_ERROR      ),
    DEF_VALS1(MQRC_LDAP_USER_NAME_LENGTH_ERR ),
    DEF_VALS1(MQRC_LDAP_PASSWORD_ERROR       ),
    DEF_VALS1(MQRC_SSL_ALREADY_INITIALIZED   ),
    DEF_VALS1(MQRC_SSL_CONFIG_ERROR          ),
    DEF_VALS1(MQRC_SSL_INITIALIZATION_ERROR  ),
    DEF_VALS1(MQRC_Q_INDEX_TYPE_ERROR        ),
    DEF_VALS1(MQRC_CFBS_ERROR                ),
    DEF_VALS1(MQRC_SSL_NOT_ALLOWED           ),
    DEF_VALS1(MQRC_JSSE_ERROR                ),
    DEF_VALS1(MQRC_SSL_PEER_NAME_MISMATCH    ),
    DEF_VALS1(MQRC_SSL_PEER_NAME_ERROR       ),
    DEF_VALS1(MQRC_UNSUPPORTED_CIPHER_SUITE  ),
    DEF_VALS1(MQRC_SSL_CERTIFICATE_REVOKED   ),
    DEF_VALS1(MQRC_SSL_CERT_STORE_ERROR      ),
    DEF_VALS1(MQRC_CLIENT_EXIT_LOAD_ERROR    ),
    DEF_VALS1(MQRC_CLIENT_EXIT_ERROR         ),
    DEF_VALS1(MQRC_UOW_COMMITTED             ),
    DEF_VALS1(MQRC_SSL_KEY_RESET_ERROR       ),
    DEF_VALS1(MQRC_UNKNOWN_COMPONENT_NAME    ),
    DEF_VALS1(MQRC_LOGGER_STATUS             ),
    DEF_VALS1(MQRC_COMMAND_MQSC              ),
    DEF_VALS1(MQRC_COMMAND_PCF               ),
    DEF_VALS1(MQRC_CFIF_ERROR                ),
    DEF_VALS1(MQRC_CFSF_ERROR                ),
    DEF_VALS1(MQRC_CFGR_ERROR                ),
    DEF_VALS1(MQRC_MSG_NOT_ALLOWED_IN_GROUP  ),
    DEF_VALS1(MQRC_FILTER_OPERATOR_ERROR     ),
    DEF_VALS1(MQRC_NESTED_SELECTOR_ERROR     ),
    DEF_VALS1(MQRC_EPH_ERROR                 ),
    DEF_VALS1(MQRC_RFH_FORMAT_ERROR          ),
    DEF_VALS1(MQRC_CFBF_ERROR                ),
    DEF_VALS1(MQRC_CLIENT_CHANNEL_CONFLICT   ),
    DEF_VALS1(MQRC_SD_ERROR                  ),
    DEF_VALS1(MQRC_TOPIC_STRING_ERROR        ),
    DEF_VALS1(MQRC_STS_ERROR                 ),
    DEF_VALS1(MQRC_NO_SUBSCRIPTION           ),
    DEF_VALS1(MQRC_SUBSCRIPTION_IN_USE       ),
    DEF_VALS1(MQRC_STAT_TYPE_ERROR           ),
    DEF_VALS1(MQRC_SUB_USER_DATA_ERROR       ),
    DEF_VALS1(MQRC_SUB_ALREADY_EXISTS        ),
    DEF_VALS1(MQRC_IDENTITY_MISMATCH         ),
    DEF_VALS1(MQRC_ALTER_SUB_ERROR           ),
    DEF_VALS1(MQRC_DURABILITY_NOT_ALLOWED    ),
    DEF_VALS1(MQRC_NO_RETAINED_MSG           ),
    DEF_VALS1(MQRC_SRO_ERROR                 ),
    DEF_VALS1(MQRC_SUB_NAME_ERROR            ),
    DEF_VALS1(MQRC_OBJECT_STRING_ERROR       ),
    DEF_VALS1(MQRC_PROPERTY_NAME_ERROR       ),
    DEF_VALS1(MQRC_SEGMENTATION_NOT_ALLOWED  ),
    DEF_VALS1(MQRC_CBD_ERROR                 ),
    DEF_VALS1(MQRC_CTLO_ERROR                ),
    DEF_VALS1(MQRC_NO_CALLBACKS_ACTIVE       ),
    DEF_VALS1(MQRC_CALLBACK_NOT_REGISTERED   ),
    DEF_VALS1(MQRC_OPTIONS_CHANGED           ),
    DEF_VALS1(MQRC_READ_AHEAD_MSGS           ),
    DEF_VALS1(MQRC_SELECTOR_SYNTAX_ERROR     ),
    DEF_VALS1(MQRC_HMSG_ERROR                ),
    DEF_VALS1(MQRC_CMHO_ERROR                ),
    DEF_VALS1(MQRC_DMHO_ERROR                ),
    DEF_VALS1(MQRC_SMPO_ERROR                ),
    DEF_VALS1(MQRC_IMPO_ERROR                ),
    DEF_VALS1(MQRC_PROPERTY_NAME_TOO_BIG     ),
    DEF_VALS1(MQRC_PROP_VALUE_NOT_CONVERTED  ),
    DEF_VALS1(MQRC_PROP_TYPE_NOT_SUPPORTED   ),
    DEF_VALS1(MQRC_PROPERTY_VALUE_TOO_BIG    ),
    DEF_VALS1(MQRC_PROP_CONV_NOT_SUPPORTED   ),
    DEF_VALS1(MQRC_PROPERTY_NOT_AVAILABLE    ),
    DEF_VALS1(MQRC_PROP_NUMBER_FORMAT_ERROR  ),
    DEF_VALS1(MQRC_PROPERTY_TYPE_ERROR       ),
    DEF_VALS1(MQRC_PROPERTIES_TOO_BIG        ),
    DEF_VALS1(MQRC_PUT_NOT_RETAINED          ),
    DEF_VALS1(MQRC_ALIAS_TARGTYPE_CHANGED    ),
    DEF_VALS1(MQRC_DMPO_ERROR                ),
    DEF_VALS1(MQRC_PD_ERROR                  ),
    DEF_VALS1(MQRC_CALLBACK_TYPE_ERROR       ),
    DEF_VALS1(MQRC_CBD_OPTIONS_ERROR         ),
    DEF_VALS1(MQRC_MAX_MSG_LENGTH_ERROR      ),
    DEF_VALS1(MQRC_CALLBACK_ROUTINE_ERROR    ),
    DEF_VALS1(MQRC_CALLBACK_LINK_ERROR       ),
    DEF_VALS1(MQRC_OPERATION_ERROR           ),
    DEF_VALS1(MQRC_BMHO_ERROR                ),
    DEF_VALS1(MQRC_UNSUPPORTED_PROPERTY      ),
    DEF_VALS1(MQRC_PROP_NAME_NOT_CONVERTED   ),
    DEF_VALS1(MQRC_GET_ENABLED               ),
    DEF_VALS1(MQRC_MODULE_NOT_FOUND          ),
    DEF_VALS1(MQRC_MODULE_INVALID            ),
    DEF_VALS1(MQRC_MODULE_ENTRY_NOT_FOUND    ),
    DEF_VALS1(MQRC_MIXED_CONTENT_NOT_ALLOWED ),
    DEF_VALS1(MQRC_MSG_HANDLE_IN_USE         ),
    DEF_VALS1(MQRC_HCONN_ASYNC_ACTIVE        ),
    DEF_VALS1(MQRC_MHBO_ERROR                ),
    DEF_VALS1(MQRC_PUBLICATION_FAILURE       ),
    DEF_VALS1(MQRC_SUB_INHIBITED             ),
    DEF_VALS1(MQRC_SELECTOR_ALWAYS_FALSE     ),
    DEF_VALS1(MQRC_XEPO_ERROR                ),
    DEF_VALS1(MQRC_DURABILITY_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_TOPIC_NOT_ALTERABLE       ),
    DEF_VALS1(MQRC_SUBLEVEL_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_PROPERTY_NAME_LENGTH_ERR  ),
    DEF_VALS1(MQRC_DUPLICATE_GROUP_SUB       ),
    DEF_VALS1(MQRC_GROUPING_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_SELECTOR_INVALID_FOR_TYPE ),
    DEF_VALS1(MQRC_HOBJ_QUIESCED             ),
    DEF_VALS1(MQRC_HOBJ_QUIESCED_NO_MSGS     ),
    DEF_VALS1(MQRC_SELECTION_STRING_ERROR    ),
    DEF_VALS1(MQRC_RES_OBJECT_STRING_ERROR   ),
    DEF_VALS1(MQRC_CONNECTION_SUSPENDED      ),
    DEF_VALS1(MQRC_INVALID_DESTINATION       ),
    DEF_VALS1(MQRC_INVALID_SUBSCRIPTION      ),
    DEF_VALS1(MQRC_SELECTOR_NOT_ALTERABLE    ),
    DEF_VALS1(MQRC_RETAINED_MSG_Q_ERROR      ),
    DEF_VALS1(MQRC_RETAINED_NOT_DELIVERED    ),
    DEF_VALS1(MQRC_RFH_RESTRICTED_FORMAT_ERR ),
    DEF_VALS1(MQRC_CONNECTION_STOPPED        ),
    DEF_VALS1(MQRC_ASYNC_UOW_CONFLICT        ),
    DEF_VALS1(MQRC_ASYNC_XA_CONFLICT         ),
    DEF_VALS1(MQRC_PUBSUB_INHIBITED          ),
    DEF_VALS1(MQRC_MSG_HANDLE_COPY_FAILURE   ),
    DEF_VALS1(MQRC_DEST_CLASS_NOT_ALTERABLE  ),
    DEF_VALS1(MQRC_OPERATION_NOT_ALLOWED     ),
    DEF_VALS1(MQRC_ACTION_ERROR              ),
    DEF_VALS1(MQRC_CHANNEL_NOT_AVAILABLE     ),
    DEF_VALS1(MQRC_HOST_NOT_AVAILABLE        ),
    DEF_VALS1(MQRC_CHANNEL_CONFIG_ERROR      ),
    DEF_VALS1(MQRC_UNKNOWN_CHANNEL_NAME      ),
    DEF_VALS1(MQRC_LOOPING_PUBLICATION       ),
    DEF_VALS1(MQRC_ALREADY_JOINED            ),
    DEF_VALS1(MQRC_STANDBY_Q_MGR             ),
    DEF_VALS1(MQRC_RECONNECTING              ),
    DEF_VALS1(MQRC_RECONNECTED               ),
    DEF_VALS1(MQRC_RECONNECT_QMID_MISMATCH   ),
    DEF_VALS1(MQRC_RECONNECT_INCOMPATIBLE    ),
    DEF_VALS1(MQRC_RECONNECT_FAILED          ),
    DEF_VALS1(MQRC_CALL_INTERRUPTED          ),
    DEF_VALS1(MQRC_NO_SUBS_MATCHED           ),
    DEF_VALS1(MQRC_SELECTION_NOT_AVAILABLE   ),
    DEF_VALS1(MQRC_CHANNEL_SSL_WARNING       ),
    DEF_VALS1(MQRC_OCSP_URL_ERROR            ),
    DEF_VALS1(MQRC_CONTENT_ERROR             ),
    DEF_VALS1(MQRC_RECONNECT_Q_MGR_REQD      ),
    DEF_VALS1(MQRC_RECONNECT_TIMED_OUT       ),
    DEF_VALS1(MQRC_PUBLISH_EXIT_ERROR        ),
    DEF_VALS1(MQRC_SSL_ALT_PROVIDER_REQUIRED ),
    DEF_VALS1(MQRC_PRECONN_EXIT_LOAD_ERROR   ),
    DEF_VALS1(MQRC_PRECONN_EXIT_NOT_FOUND    ),
    DEF_VALS1(MQRC_PRECONN_EXIT_ERROR        ),
    DEF_VALS1(MQRC_CD_ARRAY_ERROR            ),
    DEF_VALS1(MQRCCF_CFH_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_CFH_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_CFH_VERSION_ERROR          ),
    DEF_VALS1(MQRCCF_CFH_MSG_SEQ_NUMBER_ERR     ),
    DEF_VALS1(MQRCCF_CFH_CONTROL_ERROR          ),
    DEF_VALS1(MQRCCF_CFH_PARM_COUNT_ERROR       ),
    DEF_VALS1(MQRCCF_CFH_COMMAND_ERROR          ),
    DEF_VALS1(MQRCCF_COMMAND_FAILED             ),
    DEF_VALS1(MQRCCF_CFIN_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFST_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFST_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_FORCE_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_STRUCTURE_TYPE_ERROR       ),
    DEF_VALS1(MQRCCF_CFIN_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFST_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_MSG_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_CFIN_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFST_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_PARM_COUNT_TOO_SMALL       ),
    DEF_VALS1(MQRCCF_PARM_COUNT_TOO_BIG         ),
    DEF_VALS1(MQRCCF_Q_ALREADY_IN_CELL          ),
    DEF_VALS1(MQRCCF_Q_TYPE_ERROR               ),
    DEF_VALS1(MQRCCF_MD_FORMAT_ERROR            ),
    DEF_VALS1(MQRCCF_CFSL_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_REPLACE_VALUE_ERROR        ),
    DEF_VALS1(MQRCCF_CFIL_DUPLICATE_VALUE       ),
    DEF_VALS1(MQRCCF_CFIL_COUNT_ERROR           ),
    DEF_VALS1(MQRCCF_CFIL_LENGTH_ERROR          ),
    DEF_VALS2(MQRCCF_MODE_VALUE_ERROR,"MQRCCF_QUIESCE_VALUE_ERROR"),
    DEF_VALS1(MQRCCF_MSG_SEQ_NUMBER_ERROR       ),
    DEF_VALS1(MQRCCF_PING_DATA_COUNT_ERROR      ),
    DEF_VALS1(MQRCCF_PING_DATA_COMPARE_ERROR    ),
    DEF_VALS1(MQRCCF_CFSL_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CHANNEL_TYPE_ERROR         ),
    DEF_VALS1(MQRCCF_PARM_SEQUENCE_ERROR        ),
    DEF_VALS1(MQRCCF_XMIT_PROTOCOL_TYPE_ERR     ),
    DEF_VALS1(MQRCCF_BATCH_SIZE_ERROR           ),
    DEF_VALS1(MQRCCF_DISC_INT_ERROR             ),
    DEF_VALS1(MQRCCF_SHORT_RETRY_ERROR          ),
    DEF_VALS1(MQRCCF_SHORT_TIMER_ERROR          ),
    DEF_VALS1(MQRCCF_LONG_RETRY_ERROR           ),
    DEF_VALS1(MQRCCF_LONG_TIMER_ERROR           ),
    DEF_VALS1(MQRCCF_SEQ_NUMBER_WRAP_ERROR      ),
    DEF_VALS1(MQRCCF_MAX_MSG_LENGTH_ERROR       ),
    DEF_VALS1(MQRCCF_PUT_AUTH_ERROR             ),
    DEF_VALS1(MQRCCF_PURGE_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_CFIL_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_MSG_TRUNCATED              ),
    DEF_VALS1(MQRCCF_CCSID_ERROR                ),
    DEF_VALS1(MQRCCF_ENCODING_ERROR             ),
    DEF_VALS1(MQRCCF_QUEUES_VALUE_ERROR         ),
    DEF_VALS1(MQRCCF_DATA_CONV_VALUE_ERROR      ),
    DEF_VALS1(MQRCCF_INDOUBT_VALUE_ERROR        ),
    DEF_VALS1(MQRCCF_ESCAPE_TYPE_ERROR          ),
    DEF_VALS1(MQRCCF_REPOS_VALUE_ERROR          ),
    DEF_VALS1(MQRCCF_CHANNEL_TABLE_ERROR        ),
    DEF_VALS1(MQRCCF_MCA_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_CHL_INST_TYPE_ERROR        ),
    DEF_VALS1(MQRCCF_CHL_STATUS_NOT_FOUND       ),
    DEF_VALS1(MQRCCF_CFSL_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFSL_TOTAL_LENGTH_ERROR    ),
    DEF_VALS1(MQRCCF_CFSL_COUNT_ERROR           ),
    DEF_VALS1(MQRCCF_CFSL_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_BROKER_DELETED             ),
    DEF_VALS1(MQRCCF_STREAM_ERROR               ),
    DEF_VALS1(MQRCCF_TOPIC_ERROR                ),
    DEF_VALS1(MQRCCF_NOT_REGISTERED             ),
    DEF_VALS1(MQRCCF_Q_MGR_NAME_ERROR           ),
    DEF_VALS1(MQRCCF_INCORRECT_STREAM           ),
    DEF_VALS1(MQRCCF_Q_NAME_ERROR               ),
    DEF_VALS1(MQRCCF_NO_RETAINED_MSG            ),
    DEF_VALS1(MQRCCF_DUPLICATE_IDENTITY         ),
    DEF_VALS1(MQRCCF_INCORRECT_Q                ),
    DEF_VALS1(MQRCCF_CORREL_ID_ERROR            ),
    DEF_VALS1(MQRCCF_NOT_AUTHORIZED             ),
    DEF_VALS1(MQRCCF_UNKNOWN_STREAM             ),
    DEF_VALS1(MQRCCF_REG_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_PUB_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_UNKNOWN_BROKER             ),
    DEF_VALS1(MQRCCF_Q_MGR_CCSID_ERROR          ),
    DEF_VALS1(MQRCCF_DEL_OPTIONS_ERROR          ),
    DEF_VALS1(MQRCCF_CLUSTER_NAME_CONFLICT      ),
    DEF_VALS1(MQRCCF_REPOS_NAME_CONFLICT        ),
    DEF_VALS1(MQRCCF_CLUSTER_Q_USAGE_ERROR      ),
    DEF_VALS1(MQRCCF_ACTION_VALUE_ERROR         ),
    DEF_VALS1(MQRCCF_COMMS_LIBRARY_ERROR        ),
    DEF_VALS1(MQRCCF_NETBIOS_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_BROKER_COMMAND_FAILED      ),
    DEF_VALS1(MQRCCF_CFST_CONFLICTING_PARM      ),
    DEF_VALS1(MQRCCF_PATH_NOT_VALID             ),
    DEF_VALS1(MQRCCF_PARM_SYNTAX_ERROR          ),
    DEF_VALS1(MQRCCF_PWD_LENGTH_ERROR           ),
    DEF_VALS1(MQRCCF_FILTER_ERROR               ),
    DEF_VALS1(MQRCCF_WRONG_USER                 ),
    DEF_VALS1(MQRCCF_DUPLICATE_SUBSCRIPTION     ),
    DEF_VALS1(MQRCCF_SUB_NAME_ERROR             ),
    DEF_VALS1(MQRCCF_SUB_IDENTITY_ERROR         ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_IN_USE        ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_LOCKED        ),
    DEF_VALS1(MQRCCF_ALREADY_JOINED             ),
    DEF_VALS1(MQRCCF_OBJECT_IN_USE              ),
    DEF_VALS1(MQRCCF_UNKNOWN_FILE_NAME          ),
    DEF_VALS1(MQRCCF_FILE_NOT_AVAILABLE         ),
    DEF_VALS1(MQRCCF_DISC_RETRY_ERROR           ),
    DEF_VALS1(MQRCCF_ALLOC_RETRY_ERROR          ),
    DEF_VALS1(MQRCCF_ALLOC_SLOW_TIMER_ERROR     ),
    DEF_VALS1(MQRCCF_ALLOC_FAST_TIMER_ERROR     ),
    DEF_VALS1(MQRCCF_PORT_NUMBER_ERROR          ),
    DEF_VALS1(MQRCCF_CHL_SYSTEM_NOT_ACTIVE      ),
    DEF_VALS1(MQRCCF_ENTITY_NAME_MISSING        ),
    DEF_VALS1(MQRCCF_PROFILE_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_AUTH_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_AUTH_VALUE_MISSING         ),
    DEF_VALS1(MQRCCF_OBJECT_TYPE_MISSING        ),
    DEF_VALS1(MQRCCF_CONNECTION_ID_ERROR        ),
    DEF_VALS1(MQRCCF_LOG_TYPE_ERROR             ),
    DEF_VALS1(MQRCCF_PROGRAM_NOT_AVAILABLE      ),
    DEF_VALS1(MQRCCF_PROGRAM_AUTH_FAILED        ),
    DEF_VALS1(MQRCCF_NONE_FOUND                 ),
    DEF_VALS1(MQRCCF_SECURITY_SWITCH_OFF        ),
    DEF_VALS1(MQRCCF_SECURITY_REFRESH_FAILED    ),
    DEF_VALS1(MQRCCF_PARM_CONFLICT              ),
    DEF_VALS1(MQRCCF_COMMAND_INHIBITED          ),
    DEF_VALS1(MQRCCF_OBJECT_BEING_DELETED       ),
    DEF_VALS1(MQRCCF_STORAGE_CLASS_IN_USE       ),
    DEF_VALS1(MQRCCF_OBJECT_NAME_RESTRICTED     ),
    DEF_VALS1(MQRCCF_OBJECT_LIMIT_EXCEEDED      ),
    DEF_VALS1(MQRCCF_OBJECT_OPEN_FORCE          ),
    DEF_VALS1(MQRCCF_DISPOSITION_CONFLICT       ),
    DEF_VALS1(MQRCCF_Q_MGR_NOT_IN_QSG           ),
    DEF_VALS1(MQRCCF_ATTR_VALUE_FIXED           ),
    DEF_VALS1(MQRCCF_NAMELIST_ERROR             ),
    DEF_VALS1(MQRCCF_NO_CHANNEL_INITIATOR       ),
    DEF_VALS1(MQRCCF_CHANNEL_INITIATOR_ERROR    ),
    DEF_VALS1(MQRCCF_COMMAND_LEVEL_CONFLICT     ),
    DEF_VALS1(MQRCCF_Q_ATTR_CONFLICT            ),
    DEF_VALS1(MQRCCF_EVENTS_DISABLED            ),
    DEF_VALS1(MQRCCF_COMMAND_SCOPE_ERROR        ),
    DEF_VALS1(MQRCCF_COMMAND_REPLY_ERROR        ),
    DEF_VALS1(MQRCCF_FUNCTION_RESTRICTED        ),
    DEF_VALS1(MQRCCF_PARM_MISSING               ),
    DEF_VALS1(MQRCCF_PARM_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_COMMAND_LENGTH_ERROR       ),
    DEF_VALS1(MQRCCF_COMMAND_ORIGIN_ERROR       ),
    DEF_VALS1(MQRCCF_LISTENER_CONFLICT          ),
    DEF_VALS1(MQRCCF_LISTENER_STARTED           ),
    DEF_VALS1(MQRCCF_LISTENER_STOPPED           ),
    DEF_VALS1(MQRCCF_CHANNEL_ERROR              ),
    DEF_VALS1(MQRCCF_CF_STRUC_ERROR             ),
    DEF_VALS1(MQRCCF_UNKNOWN_USER_ID            ),
    DEF_VALS1(MQRCCF_UNEXPECTED_ERROR           ),
    DEF_VALS1(MQRCCF_NO_XCF_PARTNER             ),
    DEF_VALS1(MQRCCF_CFGR_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFIF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFIF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFIF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFSF_FILTER_VAL_LEN_ERR    ),
    DEF_VALS1(MQRCCF_CFSF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFSF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFSF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_TOO_MANY_FILTERS           ),
    DEF_VALS1(MQRCCF_LISTENER_RUNNING           ),
    DEF_VALS1(MQRCCF_LSTR_STATUS_NOT_FOUND      ),
    DEF_VALS1(MQRCCF_SERVICE_RUNNING            ),
    DEF_VALS1(MQRCCF_SERV_STATUS_NOT_FOUND      ),
    DEF_VALS1(MQRCCF_SERVICE_STOPPED            ),
    DEF_VALS1(MQRCCF_CFBS_DUPLICATE_PARM        ),
    DEF_VALS1(MQRCCF_CFBS_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFBS_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFBS_STRING_LENGTH_ERR     ),
    DEF_VALS1(MQRCCF_CFGR_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFGR_PARM_COUNT_ERROR      ),
    DEF_VALS1(MQRCCF_CONN_NOT_STOPPED           ),
    DEF_VALS1(MQRCCF_SERVICE_REQUEST_PENDING    ),
    DEF_VALS1(MQRCCF_NO_START_CMD               ),
    DEF_VALS1(MQRCCF_NO_STOP_CMD                ),
    DEF_VALS1(MQRCCF_CFBF_LENGTH_ERROR          ),
    DEF_VALS1(MQRCCF_CFBF_PARM_ID_ERROR         ),
    DEF_VALS1(MQRCCF_CFBF_OPERATOR_ERROR        ),
    DEF_VALS1(MQRCCF_CFBF_FILTER_VAL_LEN_ERR    ),
    DEF_VALS1(MQRCCF_LISTENER_STILL_ACTIVE      ),
    DEF_VALS1(MQRCCF_TOPICSTR_ALREADY_EXISTS    ),
    DEF_VALS1(MQRCCF_SHARING_CONVS_ERROR        ),
    DEF_VALS1(MQRCCF_SHARING_CONVS_TYPE         ),
    DEF_VALS1(MQRCCF_SECURITY_CASE_CONFLICT     ),
    DEF_VALS1(MQRCCF_TOPIC_TYPE_ERROR           ),
    DEF_VALS1(MQRCCF_MAX_INSTANCES_ERROR        ),
    DEF_VALS1(MQRCCF_MAX_INSTS_PER_CLNT_ERR     ),
    DEF_VALS1(MQRCCF_TOPIC_STRING_NOT_FOUND     ),
    DEF_VALS1(MQRCCF_SUBSCRIPTION_POINT_ERR     ),
    DEF_VALS1(MQRCCF_SUB_ALREADY_EXISTS         ),
    DEF_VALS1(MQRCCF_UNKNOWN_OBJECT_NAME        ),
    DEF_VALS1(MQRCCF_REMOTE_Q_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_DURABILITY_NOT_ALLOWED     ),
    DEF_VALS1(MQRCCF_HOBJ_ERROR                 ),
    DEF_VALS1(MQRCCF_DEST_NAME_ERROR            ),
    DEF_VALS1(MQRCCF_INVALID_DESTINATION        ),
    DEF_VALS1(MQRCCF_PUBSUB_INHIBITED           ),
    DEF_VALS1(MQRCCF_GROUPUR_CHECKS_FAILED      ),
    DEF_VALS1(MQRCCF_SSL_ALT_PROVIDER_REQD      ),
    DEF_VALS1(MQRCCF_OBJECT_ALREADY_EXISTS      ),
    DEF_VALS1(MQRCCF_OBJECT_WRONG_TYPE          ),
    DEF_VALS1(MQRCCF_LIKE_OBJECT_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_OBJECT_OPEN                ),
    DEF_VALS1(MQRCCF_ATTR_VALUE_ERROR           ),
    DEF_VALS1(MQRCCF_UNKNOWN_Q_MGR              ),
    DEF_VALS1(MQRCCF_Q_WRONG_TYPE               ),
    DEF_VALS1(MQRCCF_OBJECT_NAME_ERROR          ),
    DEF_VALS1(MQRCCF_ALLOCATE_FAILED            ),
    DEF_VALS1(MQRCCF_HOST_NOT_AVAILABLE         ),
    DEF_VALS1(MQRCCF_CONFIGURATION_ERROR        ),
    DEF_VALS1(MQRCCF_CONNECTION_REFUSED         ),
    DEF_VALS1(MQRCCF_ENTRY_ERROR                ),
    DEF_VALS1(MQRCCF_SEND_FAILED                ),
    DEF_VALS1(MQRCCF_RECEIVED_DATA_ERROR        ),
    DEF_VALS1(MQRCCF_RECEIVE_FAILED             ),
    DEF_VALS1(MQRCCF_CONNECTION_CLOSED          ),
    DEF_VALS1(MQRCCF_NO_STORAGE                 ),
    DEF_VALS1(MQRCCF_NO_COMMS_MANAGER           ),
    DEF_VALS1(MQRCCF_LISTENER_NOT_STARTED       ),
    DEF_VALS1(MQRCCF_BIND_FAILED                ),
    DEF_VALS1(MQRCCF_CHANNEL_INDOUBT            ),
    DEF_VALS1(MQRCCF_MQCONN_FAILED              ),
    DEF_VALS1(MQRCCF_MQOPEN_FAILED              ),
    DEF_VALS1(MQRCCF_MQGET_FAILED               ),
    DEF_VALS1(MQRCCF_MQPUT_FAILED               ),
    DEF_VALS1(MQRCCF_PING_ERROR                 ),
    DEF_VALS1(MQRCCF_CHANNEL_IN_USE             ),
    DEF_VALS1(MQRCCF_CHANNEL_NOT_FOUND          ),
    DEF_VALS1(MQRCCF_UNKNOWN_REMOTE_CHANNEL     ),
    DEF_VALS1(MQRCCF_REMOTE_QM_UNAVAILABLE      ),
    DEF_VALS1(MQRCCF_REMOTE_QM_TERMINATING      ),
    DEF_VALS1(MQRCCF_MQINQ_FAILED               ),
    DEF_VALS1(MQRCCF_NOT_XMIT_Q                 ),
    DEF_VALS1(MQRCCF_CHANNEL_DISABLED           ),
    DEF_VALS1(MQRCCF_USER_EXIT_NOT_AVAILABLE    ),
    DEF_VALS1(MQRCCF_COMMIT_FAILED              ),
    DEF_VALS1(MQRCCF_WRONG_CHANNEL_TYPE         ),
    DEF_VALS1(MQRCCF_CHANNEL_ALREADY_EXISTS     ),
    DEF_VALS1(MQRCCF_DATA_TOO_LARGE             ),
    DEF_VALS1(MQRCCF_CHANNEL_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_XMIT_Q_NAME_ERROR          ),
    DEF_VALS1(MQRCCF_MCA_NAME_ERROR             ),
    DEF_VALS1(MQRCCF_SEND_EXIT_NAME_ERROR       ),
    DEF_VALS1(MQRCCF_SEC_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_MSG_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_RCV_EXIT_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_XMIT_Q_NAME_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_MCA_NAME_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_DISC_INT_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_SHORT_RETRY_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_SHORT_TIMER_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_LONG_RETRY_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_LONG_TIMER_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_PUT_AUTH_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_KEEP_ALIVE_INT_ERROR       ),
    DEF_VALS1(MQRCCF_MISSING_CONN_NAME          ),
    DEF_VALS1(MQRCCF_CONN_NAME_ERROR            ),
    DEF_VALS1(MQRCCF_MQSET_FAILED               ),
    DEF_VALS1(MQRCCF_CHANNEL_NOT_ACTIVE         ),
    DEF_VALS1(MQRCCF_TERMINATED_BY_SEC_EXIT     ),
    DEF_VALS1(MQRCCF_DYNAMIC_Q_SCOPE_ERROR      ),
    DEF_VALS1(MQRCCF_CELL_DIR_NOT_AVAILABLE     ),
    DEF_VALS1(MQRCCF_MR_COUNT_ERROR             ),
    DEF_VALS1(MQRCCF_MR_COUNT_WRONG_TYPE        ),
    DEF_VALS1(MQRCCF_MR_EXIT_NAME_ERROR         ),
    DEF_VALS1(MQRCCF_MR_EXIT_NAME_WRONG_TYPE    ),
    DEF_VALS1(MQRCCF_MR_INTERVAL_ERROR          ),
    DEF_VALS1(MQRCCF_MR_INTERVAL_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_NPM_SPEED_ERROR            ),
    DEF_VALS1(MQRCCF_NPM_SPEED_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_HB_INTERVAL_ERROR          ),
    DEF_VALS1(MQRCCF_HB_INTERVAL_WRONG_TYPE     ),
    DEF_VALS1(MQRCCF_CHAD_ERROR                 ),
    DEF_VALS1(MQRCCF_CHAD_WRONG_TYPE            ),
    DEF_VALS1(MQRCCF_CHAD_EVENT_ERROR           ),
    DEF_VALS1(MQRCCF_CHAD_EVENT_WRONG_TYPE      ),
    DEF_VALS1(MQRCCF_CHAD_EXIT_ERROR            ),
    DEF_VALS1(MQRCCF_CHAD_EXIT_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_SUPPRESSED_BY_EXIT         ),
    DEF_VALS1(MQRCCF_BATCH_INT_ERROR            ),
    DEF_VALS1(MQRCCF_BATCH_INT_WRONG_TYPE       ),
    DEF_VALS1(MQRCCF_NET_PRIORITY_ERROR         ),
    DEF_VALS1(MQRCCF_NET_PRIORITY_WRONG_TYPE    ),
    DEF_VALS1(MQRCCF_CHANNEL_CLOSED             ),
    DEF_VALS1(MQRCCF_Q_STATUS_NOT_FOUND         ),
    DEF_VALS1(MQRCCF_SSL_CIPHER_SPEC_ERROR      ),
    DEF_VALS1(MQRCCF_SSL_PEER_NAME_ERROR        ),
    DEF_VALS1(MQRCCF_SSL_CLIENT_AUTH_ERROR      ),
    DEF_VALS1(MQRCCF_RETAINED_NOT_SUPPORTED     ),

    DEF_VALS1(MQRC_REOPEN_EXCL_INPUT_ERROR   ),
    DEF_VALS1(MQRC_REOPEN_INQUIRE_ERROR      ),
    DEF_VALS1(MQRC_REOPEN_SAVED_CONTEXT_ERR  ),
    DEF_VALS1(MQRC_REOPEN_TEMPORARY_Q_ERROR  ),
    DEF_VALS1(MQRC_ATTRIBUTE_LOCKED          ),
    DEF_VALS1(MQRC_CURSOR_NOT_VALID          ),
    DEF_VALS1(MQRC_ENCODING_ERROR            ),
    DEF_VALS1(MQRC_STRUC_ID_ERROR            ),
    DEF_VALS1(MQRC_NULL_POINTER              ),
    DEF_VALS1(MQRC_NO_CONNECTION_REFERENCE   ),
    DEF_VALS1(MQRC_NO_BUFFER                 ),
    DEF_VALS1(MQRC_BINARY_DATA_LENGTH_ERROR  ),
    DEF_VALS1(MQRC_BUFFER_NOT_AUTOMATIC      ),
    DEF_VALS1(MQRC_INSUFFICIENT_BUFFER       ),
    DEF_VALS1(MQRC_INSUFFICIENT_DATA         ),
    DEF_VALS1(MQRC_DATA_TRUNCATED            ),
    DEF_VALS1(MQRC_ZERO_LENGTH               ),
    DEF_VALS1(MQRC_NEGATIVE_LENGTH           ),
    DEF_VALS1(MQRC_NEGATIVE_OFFSET           ),
    DEF_VALS1(MQRC_INCONSISTENT_FORMAT       ),
    DEF_VALS1(MQRC_INCONSISTENT_OBJECT_STATE ),
    DEF_VALS1(MQRC_CONTEXT_OBJECT_NOT_VALID  ),
    DEF_VALS1(MQRC_CONTEXT_OPEN_ERROR        ),
    DEF_VALS1(MQRC_STRUC_LENGTH_ERROR        ),
    DEF_VALS1(MQRC_NOT_CONNECTED             ),
    DEF_VALS1(MQRC_NOT_OPEN                  ),
    DEF_VALS1(MQRC_DISTRIBUTION_LIST_EMPTY   ),
    DEF_VALS1(MQRC_INCONSISTENT_OPEN_OPTIONS ),
    DEF_VALS1(MQRC_WRONG_VERSION             ),
    DEF_VALS1(MQRC_REFERENCE_ERROR           ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ActionOptions)
    DEF_VALS1(MQACT_FORCE_REMOVE       ),
    DEF_VALS1(MQACT_ADVANCE_LOG        ),
    DEF_VALS1(MQACT_COLLECT_STATISTICS ),
    DEF_VALS1(MQACT_PUBSUB             ),
DEF_VALSE;
#endif

DEF_VALSB(AsynchronousStateValues)
    DEF_VALS1(MQAS_NONE                    ),
    DEF_VALS1(MQAS_ACTIVE                  ),
    DEF_VALS1(MQAS_INACTIVE                ),
    DEF_VALS1(MQAS_STARTED                 ),
    DEF_VALS1(MQAS_START_WAIT              ),
    DEF_VALS1(MQAS_STOPPED                 ),
    DEF_VALS1(MQAS_SUSPENDED               ),
    DEF_VALS1(MQAS_SUSPENDED_TEMPORARY     ),
DEF_VALSE;

#if 0
DEF_VALSB(AuthorityValues)
    DEF_VALS1(MQAUTH_NONE                  ),
    DEF_VALS1(MQAUTH_ALT_USER_AUTHORITY    ),
    DEF_VALS1(MQAUTH_BROWSE                ),
    DEF_VALS1(MQAUTH_CHANGE                ),
    DEF_VALS1(MQAUTH_CLEAR                 ),
    DEF_VALS1(MQAUTH_CONNECT               ),
    DEF_VALS1(MQAUTH_CREATE                ),
    DEF_VALS1(MQAUTH_DELETE                ),
    DEF_VALS1(MQAUTH_DISPLAY               ),
    DEF_VALS1(MQAUTH_INPUT                 ),
    DEF_VALS1(MQAUTH_INQUIRE               ),
    DEF_VALS1(MQAUTH_OUTPUT                ),
    DEF_VALS1(MQAUTH_PASS_ALL_CONTEXT      ),
    DEF_VALS1(MQAUTH_PASS_IDENTITY_CONTEXT ),
    DEF_VALS1(MQAUTH_SET                   ),
    DEF_VALS1(MQAUTH_SET_ALL_CONTEXT       ),
    DEF_VALS1(MQAUTH_SET_IDENTITY_CONTEXT  ),
    DEF_VALS1(MQAUTH_CONTROL               ),
    DEF_VALS1(MQAUTH_CONTROL_EXTENDED      ),
    DEF_VALS1(MQAUTH_PUBLISH               ),
    DEF_VALS1(MQAUTH_SUBSCRIBE             ),
    DEF_VALS1(MQAUTH_RESUME                ),
    DEF_VALS1(MQAUTH_SYSTEM                ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(BridgeTypes)
    DEF_VALS1(MQBT_OTMA ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(RefreshRepositoryOptions)
    DEF_VALS1(MQCFO_REFRESH_REPOSITORY_YES ),
    DEF_VALS1(MQCFO_REFRESH_REPOSITORY_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(RemoveQueuesOptions)
    DEF_VALS1(MQCFO_REMOVE_QUEUES_YES ),
    DEF_VALS1(MQCFO_REMOVE_QUEUES_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(CFRecover)
    DEF_VALS1(MQCFR_YES),
    DEF_VALS1(MQCFR_NO ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(CFStatus)
    DEF_VALS1(MQCFSTATUS_NOT_FOUND        ),
    DEF_VALS1(MQCFSTATUS_ACTIVE           ),
    DEF_VALS1(MQCFSTATUS_IN_RECOVER       ),
    DEF_VALS1(MQCFSTATUS_IN_BACKUP        ),
    DEF_VALS1(MQCFSTATUS_FAILED           ),
    DEF_VALS1(MQCFSTATUS_NONE             ),
    DEF_VALS1(MQCFSTATUS_UNKNOWN          ),
    DEF_VALS1(MQCFSTATUS_ADMIN_INCOMPLETE ),
    DEF_VALS1(MQCFSTATUS_NEVER_USED       ),
    DEF_VALS1(MQCFSTATUS_NO_BACKUP        ),
    DEF_VALS1(MQCFSTATUS_NOT_FAILED       ),
    DEF_VALS1(MQCFSTATUS_NOT_RECOVERABLE  ),
    DEF_VALS1(MQCFSTATUS_XES_ERROR        ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(CFStatusType)
    DEF_VALS1(MQIACF_CF_STATUS_SUMMARY    ),
    DEF_VALS1(MQIACF_CF_STATUS_CONNECT    ),
    DEF_VALS1(MQIACF_CF_STATUS_BACKUP     ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(CFStructType)
    DEF_VALS1(MQCFTYPE_APPL         ),
    DEF_VALS1(MQCFTYPE_ADMIN        ),
DEF_VALSE;
#endif

DEF_VALSB(IndoubtStatus)
    DEF_VALS1(MQCHIDS_NOT_INDOUBT ),
    DEF_VALS1(MQCHIDS_INDOUBT     ),
DEF_VALSE;

DEF_VALSB(ChannelDisp)
    DEF_VALS1(MQCHLD_ALL                      ),
    DEF_VALS1(MQCHLD_DEFAULT                  ),
    DEF_VALS1(MQCHLD_SHARED                   ),
    DEF_VALS1(MQCHLD_PRIVATE                  ),
    DEF_VALS1(MQCHLD_FIXSHARED                ),
DEF_VALSE;

#if 0
DEF_VALSB(ChannelInstanceType)
    DEF_VALS1(MQOT_CURRENT_CHANNEL),
    DEF_VALS1(MQOT_SAVED_CHANNEL  ),
    DEF_VALS1(MQOT_SHORT_CHANNEL  ),
DEF_VALSE;
#endif

DEF_VALSB(ChannelStatus)
    DEF_VALS1(MQCHS_INACTIVE     ),
    DEF_VALS1(MQCHS_BINDING      ),
    DEF_VALS1(MQCHS_STARTING     ),
    DEF_VALS1(MQCHS_RUNNING      ),
    DEF_VALS1(MQCHS_STOPPING     ),
    DEF_VALS1(MQCHS_RETRYING     ),
    DEF_VALS1(MQCHS_STOPPED      ),
    DEF_VALS1(MQCHS_REQUESTING   ),
    DEF_VALS1(MQCHS_PAUSED       ),
    DEF_VALS1(MQCHS_INITIALIZING ),
DEF_VALSE;

DEF_VALSB(ChannelSubStates)
    DEF_VALS1(MQCHSSTATE_COMPRESSING     ),
    DEF_VALS1(MQCHSSTATE_END_OF_BATCH    ),
    DEF_VALS1(MQCHSSTATE_HEARTBEATING    ),
    DEF_VALS1(MQCHSSTATE_IN_CHADEXIT     ),
    DEF_VALS1(MQCHSSTATE_IN_MQGET        ),
    DEF_VALS1(MQCHSSTATE_IN_MQI_CALL     ),
    DEF_VALS1(MQCHSSTATE_IN_MQPUT        ),
    DEF_VALS1(MQCHSSTATE_IN_MREXIT       ),
    DEF_VALS1(MQCHSSTATE_IN_MSGEXIT      ),
    DEF_VALS1(MQCHSSTATE_IN_SCYEXIT      ),
    DEF_VALS1(MQCHSSTATE_IN_RCVEXIT      ),
    DEF_VALS1(MQCHSSTATE_IN_SENDEXIT     ),
    DEF_VALS1(MQCHSSTATE_NAME_SERVER     ),
    DEF_VALS1(MQCHSSTATE_NET_CONNECTING  ),
    DEF_VALS1(MQCHSSTATE_OTHER           ),
    DEF_VALS1(MQCHSSTATE_RECEIVING       ),
    DEF_VALS1(MQCHSSTATE_RESYNCHING      ),
    DEF_VALS1(MQCHSSTATE_SENDING         ),
    DEF_VALS1(MQCHSSTATE_SERIALIZING     ),
    DEF_VALS1(MQCHSSTATE_SSL_HANDSHAKING ),
DEF_VALSE;

#if 0
DEF_VALSB(ChannelSharedRestartOptions)
    DEF_VALS1(MQCHSH_RESTART_NO           ),
    DEF_VALS1(MQCHSH_RESTART_YES          ),
DEF_VALSE;
#endif

DEF_VALSB(ChannelStopOptions)
    DEF_VALS1(MQCHSR_STOP_NOT_REQUESTED ),
    DEF_VALS1(MQCHSR_STOP_REQUESTED     ),
DEF_VALSE;

#if 0
DEF_VALSB(ChannelTableTypes)
    DEF_VALS1(MQCHTAB_Q_MGR    ),
    DEF_VALS1(MQCHTAB_CLNTCONN ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ChinitStatus)
    DEF_VALS1(MQSVC_STATUS_STOPPED),
    DEF_VALS1(MQSVC_STATUS_RUNNING),
    DEF_VALS1(MQSVC_STATUS_RETRYING),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ChinitInboundDisp)
    DEF_VALS1(MQINBD_Q_MGR),
    DEF_VALS1(MQINBD_GROUP),
DEF_VALSE;
#endif

DEF_VALSB(ChinitTraceAutoStart)
    DEF_VALS1(MQTRAXSTR_YES),
    DEF_VALS1(MQTRAXSTR_NO ),
DEF_VALSE;

#if 0
DEF_VALSB(ClearTopicStringScope)
    DEF_VALS1(MQCLRS_LOCAL              ),
    DEF_VALS1(MQCLRS_GLOBAL             ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ClearTopicStringType)
    DEF_VALS1(MQCLRT_RETAINED           ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(DisconnectTypes)
    DEF_VALS1(MQDISCONNECT_NORMAL      ),
    DEF_VALS1(MQDISCONNECT_IMPLICIT    ),
    DEF_VALS1(MQDISCONNECT_Q_MGR       ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(EscapeTypes)
    DEF_VALS1(MQET_MQSC ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(EventRecording)
    DEF_VALS1(MQEVR_DISABLED   ),
    DEF_VALS1(MQEVR_ENABLED    ),
    DEF_VALS1(MQEVR_EXCEPTION  ),
    DEF_VALS1(MQEVR_NO_DISPLAY ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ForceOptions)
    DEF_VALS1(MQFC_YES ),
    DEF_VALS1(MQFC_NO  ),
DEF_VALSE;
#endif

DEF_VALSB(HandleStates)
    DEF_VALS1(MQHSTATE_INACTIVE    ),
    DEF_VALS1(MQHSTATE_ACTIVE      ),
DEF_VALSE;

#if 0
DEF_VALSB(InboundDispositions)
    DEF_VALS1(MQINBD_Q_MGR     ),
    DEF_VALS1(MQINBD_GROUP     ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(IndoubtOptions)
    DEF_VALS1(MQIDO_COMMIT  ),
    DEF_VALS1(MQIDO_BACKOUT ),
DEF_VALSE;
#endif

DEF_VALSB(MessageChannelAgentStatus)
    DEF_VALS1(MQMCAS_STOPPED ),
    DEF_VALS1(MQMCAS_RUNNING ),
DEF_VALSE;

#if 0
DEF_VALSB(ModeOptions)
    DEF_VALS1(MQMODE_FORCE     ),
    DEF_VALS1(MQMODE_QUIESCE   ),
    DEF_VALS1(MQMODE_TERMINATE ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(PurgeOptions)
    DEF_VALS1(MQPO_YES ),
    DEF_VALS1(MQPO_NO  ),
DEF_VALSE;
#endif

DEF_VALSB(PubSubCluster)
    DEF_VALS1(MQPSCLUS_ENABLED ),
    DEF_VALS1(MQPSCLUS_DISABLED),
DEF_VALSE;

DEF_VALSB(PubSubMode)
    DEF_VALS1(MQPSM_COMPAT   ),
    DEF_VALS1(MQPSM_DISABLED ),
    DEF_VALS1(MQPSM_ENABLED  ),
DEF_VALSE;

DEF_VALSB(PubSubNP)
    DEF_VALS1(MQUNDELIVERED_NORMAL ),
    DEF_VALS1(MQUNDELIVERED_SAFE   ),
    DEF_VALS1(MQUNDELIVERED_DISCARD),
    DEF_VALS1(MQUNDELIVERED_KEEP   ),
DEF_VALSE;

DEF_VALSB(PubSubSync)
    DEF_VALS1(MQSYNCPOINT_YES  ),
    DEF_VALS1(MQSYNCPOINT_IFPER),
DEF_VALSE;

#if 0
DEF_VALSB(PubSubStatusType)
    DEF_VALS1(MQPSST_ALL                  ),
    DEF_VALS1(MQPSST_LOCAL                ),
    DEF_VALS1(MQPSST_PARENT               ),
    DEF_VALS1(MQPSST_CHILD                ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(PubSubStatus)
    DEF_VALS1(MQPS_STATUS_ACTIVE          ),
    DEF_VALS1(MQPS_STATUS_COMPAT          ),
    DEF_VALS1(MQPS_STATUS_ERROR           ),
    DEF_VALS1(MQPS_STATUS_INACTIVE        ),
    DEF_VALS1(MQPS_STATUS_STARTING        ),
    DEF_VALS1(MQPS_STATUS_STOPPING        ),
    DEF_VALS1(MQPS_STATUS_REFUSED         ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueManagerDefinitionTypes)
    DEF_VALS1(MQQMDT_EXPLICIT_CLUSTER_SENDER ),
    DEF_VALS1(MQQMDT_AUTO_CLUSTER_SENDER     ),
    DEF_VALS1(MQQMDT_AUTO_EXP_CLUSTER_SENDER ),
    DEF_VALS1(MQQMDT_CLUSTER_RECEIVER        ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueManagerFacility)
    DEF_VALS1(MQQMFAC_IMS_BRIDGE             ),
    DEF_VALS1(MQQMFAC_DB2                    ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueManagerStatus)
    DEF_VALS1(MQQMSTA_STARTING     ),
    DEF_VALS1(MQQMSTA_RUNNING      ),
    DEF_VALS1(MQQMSTA_QUIESCING    ),
    DEF_VALS1(MQQMSTA_STANDBY      ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueManagerTypes)
    DEF_VALS1(MQQMT_NORMAL     ),
    DEF_VALS1(MQQMT_REPOSITORY ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QuiesceOptions)
    DEF_VALS1(MQQO_YES ),
    DEF_VALS1(MQQO_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueServiceIntervalEvents)
    DEF_VALS1(MQQSIE_NONE ),
    DEF_VALS1(MQQSIE_HIGH ),
    DEF_VALS1(MQQSIE_OK   ),
DEF_VALSE;
#endif

DEF_VALSB(QueueStatusOpenTypes)
    DEF_VALS1(MQQSOT_ALL    ),
    DEF_VALS1(MQQSOT_INPUT  ),
    DEF_VALS1(MQQSOT_OUTPUT ),
DEF_VALSE;

#if 0
DEF_VALSB(NameListTypes)
    DEF_VALS1(MQNT_NONE      ),
    DEF_VALS1(MQNT_Q         ),
    DEF_VALS1(MQNT_CLUSTER   ),
    DEF_VALS1(MQNT_AUTH_INFO ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QSGStatus)
    DEF_VALS1(MQQSGS_UNKNOWN   ),
    DEF_VALS1(MQQSGS_CREATED   ),
    DEF_VALS1(MQQSGS_ACTIVE    ),
    DEF_VALS1(MQQSGS_INACTIVE  ),
    DEF_VALS1(MQQSGS_FAILED    ),
    DEF_VALS1(MQQSGS_PENDING   ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(QueueStatusType)
    DEF_VALS1(MQIACF_Q_STATUS ),
    DEF_VALS1(MQIACF_Q_HANDLE ),
DEF_VALSE;
#endif

DEF_VALSB(QueueStatusOptionsYesNo)
    DEF_VALS1(MQQSO_NO         ),
    DEF_VALS1(MQQSO_YES        ),
DEF_VALSE;

DEF_VALSB(QueueStatusOpenOptions)
    DEF_VALS1(MQQSO_NO        ),
    DEF_VALS1(MQQSO_SHARED    ),
    DEF_VALS1(MQQSO_EXCLUSIVE ),
DEF_VALSE;

#if 0
DEF_VALSB(QueueStatusUncommittedMessages)
    DEF_VALS1(MQQSUM_YES ),
    DEF_VALS1(MQQSUM_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ReplaceOptions)
    DEF_VALS1(MQRP_YES ),
    DEF_VALS1(MQRP_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ReasonQualifiers)
    DEF_VALS1(MQRQ_CONN_NOT_AUTHORIZED       ),
    DEF_VALS1(MQRQ_OPEN_NOT_AUTHORIZED       ),
    DEF_VALS1(MQRQ_CLOSE_NOT_AUTHORIZED      ),
    DEF_VALS1(MQRQ_CMD_NOT_AUTHORIZED        ),
    DEF_VALS1(MQRQ_Q_MGR_STOPPING            ),
    DEF_VALS1(MQRQ_Q_MGR_QUIESCING           ),
    DEF_VALS1(MQRQ_CHANNEL_STOPPED_OK        ),
    DEF_VALS1(MQRQ_CHANNEL_STOPPED_ERROR     ),
    DEF_VALS1(MQRQ_CHANNEL_STOPPED_RETRY     ),
    DEF_VALS1(MQRQ_CHANNEL_STOPPED_DISABLED  ),
    DEF_VALS1(MQRQ_BRIDGE_STOPPED_OK         ),
    DEF_VALS1(MQRQ_BRIDGE_STOPPED_ERROR      ),
    DEF_VALS1(MQRQ_SSL_HANDSHAKE_ERROR       ),
    DEF_VALS1(MQRQ_SSL_CIPHER_SPEC_ERROR     ),
    DEF_VALS1(MQRQ_SSL_CLIENT_AUTH_ERROR     ),
    DEF_VALS1(MQRQ_SSL_PEER_NAME_ERROR       ),
    DEF_VALS1(MQRQ_SUB_NOT_AUTHORIZED        ),
    DEF_VALS1(MQRQ_SUB_DEST_NOT_AUTHORIZED   ),
    DEF_VALS1(MQRQ_SSL_UNKNOWN_REVOCATION    ),
    DEF_VALS1(MQRQ_SYS_CONN_NOT_AUTHORIZED   ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(RefreshTypes)
    DEF_VALS1(MQRT_CONFIGURATION             ),
    DEF_VALS1(MQRT_EXPIRY                    ),
    DEF_VALS1(MQRT_NSPROC                    ),
    DEF_VALS1(MQRT_PROXYSUB                  ),
DEF_VALSE;
#endif

DEF_VALSB(QueueDefinitionScope)
    DEF_VALS1(MQSCO_Q_MGR ),
    DEF_VALS1(MQSCO_CELL  ),
DEF_VALSE;

#if 0
DEF_VALSB(SecurityItems)
    DEF_VALS1(MQSECITEM_ALL                    ),
    DEF_VALS1(MQSECITEM_MQADMIN                ),
    DEF_VALS1(MQSECITEM_MQNLIST                ),
    DEF_VALS1(MQSECITEM_MQPROC                 ),
    DEF_VALS1(MQSECITEM_MQQUEUE                ),
    DEF_VALS1(MQSECITEM_MQCONN                 ),
    DEF_VALS1(MQSECITEM_MQCMDS                 ),
    DEF_VALS1(MQSECITEM_MXADMIN                ),
    DEF_VALS1(MQSECITEM_MXNLIST                ),
    DEF_VALS1(MQSECITEM_MXPROC                 ),
    DEF_VALS1(MQSECITEM_MXQUEUE                ),
    DEF_VALS1(MQSECITEM_MXTOPIC                ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SecuritySwitches)
    DEF_VALS1(MQSECSW_PROCESS                  ),
    DEF_VALS1(MQSECSW_NAMELIST                 ),
    DEF_VALS1(MQSECSW_Q                        ),
    DEF_VALS1(MQSECSW_TOPIC                    ),
    DEF_VALS1(MQSECSW_CONTEXT                  ),
    DEF_VALS1(MQSECSW_ALTERNATE_USER           ),
    DEF_VALS1(MQSECSW_COMMAND                  ),
    DEF_VALS1(MQSECSW_CONNECTION               ),
    DEF_VALS1(MQSECSW_SUBSYSTEM                ),
    DEF_VALS1(MQSECSW_COMMAND_RESOURCES        ),
    DEF_VALS1(MQSECSW_Q_MGR                    ),
    DEF_VALS1(MQSECSW_QSG                      ),
    DEF_VALS1(MQSECSW_OFF_FOUND                ),
    DEF_VALS1(MQSECSW_ON_FOUND                 ),
    DEF_VALS1(MQSECSW_OFF_NOT_FOUND            ),
    DEF_VALS1(MQSECSW_ON_NOT_FOUND             ),
    DEF_VALS1(MQSECSW_OFF_ERROR                ),
    DEF_VALS1(MQSECSW_ON_OVERRIDDEN            ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SecurityTypes)
    DEF_VALS1(MQSECTYPE_AUTHSERV ),
    DEF_VALS1(MQSECTYPE_SSL      ),
    DEF_VALS1(MQSECTYPE_CLASSES  ),
DEF_VALSE;
#endif

DEF_VALSB(InhibitPublications)
    DEF_VALS1(MQTA_PUB_AS_PARENT ),
    DEF_VALS1(MQTA_PUB_INHIBITED ),
    DEF_VALS1(MQTA_PUB_ALLOWED   ),
DEF_VALSE;

DEF_VALSB(MsgDelivery)
    DEF_VALS1(MQDLV_AS_PARENT),
    DEF_VALS1(MQDLV_ALL),
    DEF_VALS1(MQDLV_ALL_DUR),
    DEF_VALS1(MQDLV_ALL_AVAIL),
DEF_VALSE;

#if 0
DEF_VALSB(ClusterCacheType)
    DEF_VALS1(MQCLCT_STATIC  ),
    DEF_VALS1(MQCLCT_DYNAMIC ),
DEF_VALSE;
#endif

DEF_VALSB(InhibitSubscriptions)
    DEF_VALS1(MQTA_SUB_AS_PARENT ),
    DEF_VALS1(MQTA_SUB_INHIBITED ),
    DEF_VALS1(MQTA_SUB_ALLOWED   ),
DEF_VALSE;

DEF_VALSB(ProxySubscriptions)
    DEF_VALS1(MQTA_PROXY_SUB_FORCE    ),
    DEF_VALS1(MQTA_PROXY_SUB_FIRSTUSE ),
DEF_VALSE;

DEF_VALSB(WildcardOperation)
    DEF_VALS1(MQTA_PASSTHRU ),
    DEF_VALS1(MQTA_BLOCK    ),
DEF_VALSE;

DEF_VALSB(TopicType)
    DEF_VALS1(MQTOPT_LOCAL   ),
    DEF_VALS1(MQTOPT_CLUSTER ),
DEF_VALSE;

#if 0
DEF_VALSB(WLMIntervalUnits)
    DEF_VALS1(MQTIME_UNIT_SECS ),
    DEF_VALS1(MQTIME_UNIT_MINS ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SelectorTypes)
    DEF_VALS1(MQSELTYPE_NONE     ),
    DEF_VALS1(MQSELTYPE_STANDARD ),
    DEF_VALS1(MQSELTYPE_EXTENDED ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SuspendStatus)
    DEF_VALS1(MQSUS_YES ),
    DEF_VALS1(MQSUS_NO  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SyncpointPubSub)
    DEF_VALS1(MQSYNCPOINT_YES   ),
    DEF_VALS1(MQSYNCPOINT_IFPER ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SystemParameterValues)
    DEF_VALS1(MQSYSP_NO                  ),
    DEF_VALS1(MQSYSP_YES                 ),
    DEF_VALS1(MQSYSP_EXTENDED            ),
    DEF_VALS1(MQSYSP_TYPE_INITIAL        ),
    DEF_VALS1(MQSYSP_TYPE_SET            ),
    DEF_VALS1(MQSYSP_TYPE_LOG_COPY       ),
    DEF_VALS1(MQSYSP_TYPE_LOG_STATUS     ),
    DEF_VALS1(MQSYSP_TYPE_ARCHIVE_TAPE   ),
    DEF_VALS1(MQSYSP_ALLOC_BLK           ),
    DEF_VALS1(MQSYSP_ALLOC_TRK           ),
    DEF_VALS1(MQSYSP_ALLOC_CYL           ),
    DEF_VALS1(MQSYSP_STATUS_BUSY         ),
    DEF_VALS1(MQSYSP_STATUS_PREMOUNT     ),
    DEF_VALS1(MQSYSP_STATUS_AVAILABLE    ),
    DEF_VALS1(MQSYSP_STATUS_UNKNOWN      ),
    DEF_VALS1(MQSYSP_STATUS_ALLOC_ARCHIVE),
    DEF_VALS1(MQSYSP_STATUS_COPYING_BSDS ),
    DEF_VALS1(MQSYSP_STATUS_COPYING_LOG  ),
DEF_VALSE;
#endif

DEF_VALSB(DurableSubscriptions)
    DEF_VALS1(MQSUB_DURABLE_AS_PARENT),
    DEF_VALS1(MQSUB_DURABLE_ALLOWED  ),
    DEF_VALS1(MQSUB_DURABLE_INHIBITED),
DEF_VALSE;

#if 0
DEF_VALSB(Durable)
    DEF_VALS1(MQSUB_DURABLE_ALL),
    DEF_VALS1(MQSUB_DURABLE_YES),
    DEF_VALS1(MQSUB_DURABLE_NO ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SubscriptionTypes)
    DEF_VALS1(MQSUBTYPE_API  ),
    DEF_VALS1(MQSUBTYPE_ADMIN),
    DEF_VALS1(MQSUBTYPE_PROXY),
    DEF_VALS1(MQSUBTYPE_ALL  ),
    DEF_VALS1(MQSUBTYPE_USER ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SubscriptionDestinationClass)
    DEF_VALS1(MQDC_MANAGED ),
    DEF_VALS1(MQDC_PROVIDED),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(SubscriptionRequestOnly)
    DEF_VALS1(MQRU_PUBLISH_ON_REQUEST),
    DEF_VALS1(MQRU_PUBLISH_ALL       ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(TimeUnits)
    DEF_VALS1(MQTIME_UNIT_MINS),
    DEF_VALS1(MQTIME_UNIT_SECS),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(VariableUser)
    DEF_VALS1(MQVU_ANY_USER  ),
    DEF_VALS1(MQVU_FIXED_USER),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(WildcardSchema)
    DEF_VALS1(MQWS_CHAR ),
    DEF_VALS1(MQWS_TOPIC),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(UserIDSupport)
    DEF_VALS1(MQUIDSUPP_NO ),
    DEF_VALS1(MQUIDSUPP_YES),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(UndeliveredPubSub)
    DEF_VALS1(MQUNDELIVERED_NORMAL ),
    DEF_VALS1(MQUNDELIVERED_SAFE   ),
    DEF_VALS1(MQUNDELIVERED_DISCARD),
    DEF_VALS1(MQUNDELIVERED_KEEP   ),
DEF_VALSE;
#endif

DEF_VALSB(UOWStates)
    DEF_VALS1(MQUOWST_NONE      ),
    DEF_VALS1(MQUOWST_ACTIVE    ),
    DEF_VALS1(MQUOWST_PREPARED  ),
    DEF_VALS1(MQUOWST_UNRESOLVED),
DEF_VALSE;

DEF_VALSB(UOWTypes)
    DEF_VALS1(MQUOWT_Q_MGR),
    DEF_VALS1(MQUOWT_CICS ),
    DEF_VALS1(MQUOWT_RRS  ),
    DEF_VALS1(MQUOWT_IMS  ),
    DEF_VALS1(MQUOWT_XA   ),
DEF_VALSE;

#if 0
DEF_VALSB(UsageType)
    DEF_VALS1(MQIACF_USAGE_PAGESET    ),
    DEF_VALS1(MQIACF_USAGE_BUFFER_POOL),
    DEF_VALS1(MQIACF_USAGE_DATA_SET   ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(PageSetStatus)
    DEF_VALS1(MQUSAGE_PS_AVAILABLE  ),
    DEF_VALS1(MQUSAGE_PS_DEFINED    ),
    DEF_VALS1(MQUSAGE_PS_OFFLINE    ),
    DEF_VALS1(MQUSAGE_PS_NOT_DEFINED),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ExpandType)
    DEF_VALS1(MQUSAGE_EXPAND_USER  ),
    DEF_VALS1(MQUSAGE_EXPAND_SYSTEM),
    DEF_VALS1(MQUSAGE_EXPAND_NONE  ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(DataSetType)
    DEF_VALS1(MQUSAGE_DS_OLDEST_ACTIVE_UOW ),
    DEF_VALS1(MQUSAGE_DS_OLDEST_PS_RECOVERY),
    DEF_VALS1(MQUSAGE_DS_OLDEST_CF_RECOVERY),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(ActivityOperations)
    DEF_VALS1(MQOPER_UNKNOWN          ),
    DEF_VALS1(MQOPER_BROWSE           ),
    DEF_VALS1(MQOPER_DISCARD          ),
    DEF_VALS1(MQOPER_GET              ),
    DEF_VALS1(MQOPER_PUT              ),
    DEF_VALS1(MQOPER_PUT_REPLY        ),
    DEF_VALS1(MQOPER_PUT_REPORT       ),
    DEF_VALS1(MQOPER_RECEIVE          ),
    DEF_VALS1(MQOPER_SEND             ),
    DEF_VALS1(MQOPER_TRANSFORM        ),
    DEF_VALS1(MQOPER_PUBLISH          ),
    DEF_VALS1(MQOPER_EXCLUDED_PUBLISH ),
    DEF_VALS1(MQOPER_DISCARDED_PUBLISH),
DEF_VALSE;
#endif

DEF_VALSB(ConnInfoType)
    DEF_VALS1(MQIACF_CONN_INFO_ALL   ),
    DEF_VALS1(MQIACF_CONN_INFO_CONN  ),
    DEF_VALS1(MQIACF_CONN_INFO_HANDLE),
DEF_VALSE;

#if 0
DEF_VALSB(Priority)
    DEF_VALS1(MQPRI_PRIORITY_AS_PUBLISHED),
    DEF_VALS1(MQPRI_PRIORITY_AS_PARENT),
    DEF_VALS1(MQPRI_PRIORITY_AS_Q_DEF),
    DEF_VALS1(0),
    DEF_VALS1(1),
    DEF_VALS1(2),
    DEF_VALS1(3),
    DEF_VALS1(4),
    DEF_VALS1(5),
    DEF_VALS1(6),
    DEF_VALS1(7),
    DEF_VALS1(8),
    DEF_VALS1(9),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(PublishSubscribeProperties)
    DEF_VALS1(MQPSPROP_NONE   ),
    DEF_VALS1(MQPSPROP_COMPAT ),
    DEF_VALS1(MQPSPROP_RFH2   ),
    DEF_VALS1(MQPSPROP_MSGPROP),
DEF_VALSE;
#endif

DEF_VALSB(PubSubScope)
    DEF_VALS1(MQSCOPE_ALL      ),
    DEF_VALS1(MQSCOPE_AS_PARENT),
    DEF_VALS1(MQSCOPE_QMGR     ),
DEF_VALSE;

#if 0
DEF_VALSB(EncryptionPolicySuiteB)
    DEF_VALS1(MQ_SUITE_B_NOT_AVAILABLE),
    DEF_VALS1(MQ_SUITE_B_NONE         ),
    DEF_VALS1(MQ_SUITE_B_128_BIT      ),
    DEF_VALS1(MQ_SUITE_B_192_BIT      ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(Multicast)
    DEF_VALS1(MQMC_AS_PARENT),
    DEF_VALS1(MQMC_ENABLED  ),
    DEF_VALS1(MQMC_DISABLED ),
    DEF_VALS1(MQMC_ONLY     ),
DEF_VALSE;
#endif

#if 0
DEF_VALSB(UseDLQueue)
    DEF_VALS1(MQUSEDLQ_AS_PARENT),
    DEF_VALS1(MQUSEDLQ_NO       ),
    DEF_VALS1(MQUSEDLQ_YES      ),
DEF_VALSE;
#endif

DEF_VALSB(IGQPutAuthority)
    DEF_VALS1(MQIGQPA_DEFAULT         ),
    DEF_VALS1(MQIGQPA_CONTEXT         ),
    DEF_VALS1(MQIGQPA_ONLY_IGQ        ),
    DEF_VALS1(MQIGQPA_ALTERNATE_OR_IGQ),
DEF_VALSE;

DEF_VALSB(SecurityCase)
    DEF_VALS1(MQSCYC_UPPER),
    DEF_VALS1(MQSCYC_MIXED),
DEF_VALSE;

DEF_VALSB(ChlauthType)
    DEF_VALS1(MQCAUT_BLOCKUSER ),
    DEF_VALS1(MQCAUT_BLOCKADDR ),
    DEF_VALS1(MQCAUT_SSLPEERMAP),
    DEF_VALS1(MQCAUT_ADDRESSMAP),
    DEF_VALS1(MQCAUT_USERMAP   ),
    DEF_VALS1(MQCAUT_QMGRMAP   ),
DEF_VALSE;

DEF_VALSB(UserSource)
    DEF_VALS1(MQUSRC_MAP     ),
    DEF_VALS1(MQUSRC_NOACCESS),
    DEF_VALS1(MQUSRC_CHANNEL ),
DEF_VALSE;

DEF_VALSB(Warning)
    DEF_VALS1(MQWARN_NO ),
    DEF_VALS1(MQWARN_YES),
DEF_VALSE;

DEF_VALSB(CertValPolicy)
    DEF_VALS1(MQ_CERT_VAL_POLICY_ANY    ),
    DEF_VALS1(MQ_CERT_VAL_POLICY_RFC5280),
DEF_VALSE;

DEF_VALSB(ChanAutoDef)
    DEF_VALS1(MQCHAD_DISABLED),
    DEF_VALS1(MQCHAD_ENABLED ),
DEF_VALSE;

DEF_VALSB(ChlAuthRecords)
    DEF_VALS1(MQCHLA_DISABLED),
    DEF_VALS1(MQCHLA_ENABLED ),
DEF_VALSE;

DEF_VALSB(DefClusXmitTyp)
    DEF_VALS1(MQCLXQ_SCTQ   ),
    DEF_VALS1(MQCLXQ_CHANNEL),
DEF_VALSE;

DEF_VALSB(ServiceControl)
    DEF_VALS1(MQSVC_CONTROL_Q_MGR      ),
    DEF_VALS1(MQSVC_CONTROL_Q_MGR_START),
    DEF_VALS1(MQSVC_CONTROL_MANUAL     ),
DEF_VALSE;

DEF_VALSB(Capability)
    DEF_VALS1(MQCAP_NOT_SUPPORTED),
    DEF_VALS1(MQCAP_SUPPORTED    ),
    DEF_VALS1(MQCAP_EXPIRED      ),
DEF_VALSE;

DEF_VALSB(FipsRequired)
    DEF_VALS1(MQSSL_FIPS_NO ),
    DEF_VALS1(MQSSL_FIPS_YES),
DEF_VALSE;

DEF_VALSB(SyncpointAvailability)
    DEF_VALS1(MQSP_AVAILABLE    ),
    DEF_VALS1(MQSP_NOT_AVAILABLE),
DEF_VALSE;

DEF_VALSBX(UOWControls)
    DEF_VALS1(MQCUOWC_ONLY    ),
    DEF_VALS1(MQCUOWC_CONTINUE),
    DEF_VALS1(MQCUOWC_FIRST   ),
    DEF_VALS1(MQCUOWC_MIDDLE  ),
    DEF_VALS1(MQCUOWC_LAST    ),
    DEF_VALS1(MQCUOWC_COMMIT  ),
    DEF_VALS1(MQCUOWC_BACKOUT ),
DEF_VALSE;

DEF_VALSBX(LinkType)
    DEF_VALS1(MQCLT_PROGRAM    ),
    DEF_VALS1(MQCLT_TRANSACTION),
DEF_VALSE;

DEF_VALSBX(ADSDescr)
    DEF_VALS1(MQCADSD_NONE     ),
    DEF_VALS1(MQCADSD_SEND     ),
    DEF_VALS1(MQCADSD_RECV     ),
    DEF_VALS1(MQCADSD_MSGFORMAT),
DEF_VALSE;

DEF_VALSBX(ConvTaskOpt)
    DEF_VALS1(MQCCT_YES),
    DEF_VALS1(MQCCT_NO ),
DEF_VALSE;

DEF_VALSBX(TaskEndStatus)
    DEF_VALS1(MQCTES_NOSYNC ),
    DEF_VALS1(MQCTES_COMMIT ),
    DEF_VALS1(MQCTES_BACKOUT),
    DEF_VALS1(MQCTES_ENDTASK),
DEF_VALSE;

DEF_VALSBX(MQCFINT_Parse)
    DEF_VALS2(MQIACF_ASYNC_STATE           ,GET_VALSP(AsynchronousStateValues)),
    DEF_VALS2(MQIACF_CHLAUTH_TYPE          ,GET_VALSP(ChlauthType)),
    DEF_VALS2(MQIACF_CLUSTER_Q_MGR_ATTRS   ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_COMMAND               ,GET_VALSP(Command)),
    DEF_VALS2(MQIACF_CONNECTION_ATTRS      ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_CONN_INFO_TYPE        ,GET_VALSP(ConnInfoType)),
    DEF_VALS2(MQIACF_EVENT_APPL_TYPE       ,GET_VALSP(ApplType)),
    DEF_VALS2(MQIACF_EVENT_ORIGIN          ,GET_VALSP(EvtOrig)),
    DEF_VALS2(MQIACF_HANDLE_STATE          ,GET_VALSP(HandleStates)),
    DEF_VALS2(MQIACF_OBJECT_TYPE           ,GET_VALSP(ObjType)),
    DEF_VALS2(MQIACF_OPEN_BROWSE           ,GET_VALSP(QueueStatusOptionsYesNo)),
    DEF_VALS2(MQIACF_OPEN_INPUT_TYPE       ,GET_VALSP(QueueStatusOpenOptions)),
    DEF_VALS2(MQIACF_OPEN_INQUIRE          ,GET_VALSP(QueueStatusOptionsYesNo)),
    DEF_VALS2(MQIACF_OPEN_OUTPUT           ,GET_VALSP(QueueStatusOptionsYesNo)),
    DEF_VALS2(MQIACF_OPEN_SET              ,GET_VALSP(QueueStatusOptionsYesNo)),
    DEF_VALS2(MQIACF_OPEN_TYPE             ,GET_VALSP(QueueStatusOpenTypes)),
    DEF_VALS2(MQIACF_PARAMETER_ID          ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_QSG_DISPS             ,GET_VALSP(QShrGrpDisp)),
    DEF_VALS2(MQIACF_Q_ATTRS               ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_Q_MGR_ATTRS           ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_Q_STATUS_ATTRS        ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_Q_STATUS_TYPE         ,GET_VALSP(PrmId)),
    DEF_VALS2(MQIACF_Q_TYPES               ,GET_VALSP(QueueType)),
    DEF_VALS2(MQIACF_UOW_STATE             ,GET_VALSP(UOWStates)),
    DEF_VALS2(MQIACF_UOW_TYPE              ,GET_VALSP(UOWTypes)),
    DEF_VALS2(MQIACH_CHANNEL_DISP          ,GET_VALSP(ChannelDisp)),
    DEF_VALS2(MQIACH_CHANNEL_INSTANCE_TYPE ,GET_VALSP(ObjType)),
    DEF_VALS2(MQIACH_CHANNEL_STATUS        ,GET_VALSP(ChannelStatus)),
    DEF_VALS2(MQIACH_CHANNEL_SUBSTATE      ,GET_VALSP(ChannelSubStates)),
    DEF_VALS2(MQIACH_CHANNEL_TYPE          ,GET_VALSP(ChannelType)),
    DEF_VALS2(MQIACH_DATA_CONVERSION       ,GET_VALSP(DataConversion)),
    DEF_VALS2(MQIACH_HDR_COMPRESSION       ,GET_VALSP(HeaderCompression)),
    DEF_VALS2(MQIACH_INDOUBT_STATUS        ,GET_VALSP(IndoubtStatus)),
    DEF_VALS2(MQIACH_MCA_STATUS            ,GET_VALSP(MessageChannelAgentStatus)),
    DEF_VALS2(MQIACH_MCA_TYPE              ,GET_VALSP(MCAType)),
    DEF_VALS2(MQIACH_MSG_COMPRESSION       ,GET_VALSP(MessageCompression)),
    DEF_VALS2(MQIACH_NPM_SPEED             ,GET_VALSP(NonPersistentMsgSpeed)),
    DEF_VALS2(MQIACH_PUT_AUTHORITY         ,GET_VALSP(PutAuthority)),
    DEF_VALS2(MQIACH_SSL_CLIENT_AUTH       ,GET_VALSP(SSLClientAuthentication)),
    DEF_VALS2(MQIACH_STOP_REQUESTED        ,GET_VALSP(ChannelStopOptions)),
    DEF_VALS2(MQIACH_USER_SOURCE           ,GET_VALSP(UserSource)),
    DEF_VALS2(MQIACH_WARNING               ,GET_VALSP(Warning)),
    DEF_VALS2(MQIACH_XMIT_PROTOCOL_TYPE    ,GET_VALSP(TransportType)),
    DEF_VALS2(MQIA_ACCOUNTING_CONN_OVERRIDE,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_ACCOUNTING_MQI          ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_ACCOUNTING_Q            ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_ACTIVITY_RECORDING      ,GET_VALSP(ActivityRecording)),
    DEF_VALS2(MQIA_ACTIVITY_TRACE          ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_ADOPTNEWMCA_CHECK       ,GET_VALSP(AdoptNewMCACheck)),
    DEF_VALS2(MQIA_ADOPTNEWMCA_TYPE        ,GET_VALSP(AdoptNewMCAType)),
    DEF_VALS2(MQIA_APPL_TYPE               ,GET_VALSP(ApplType)),
    DEF_VALS2(MQIA_AUTHORITY_EVENT         ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_AUTH_INFO_TYPE          ,GET_VALSP(AuthInfoType)),
    DEF_VALS2(MQIA_BASE_TYPE               ,GET_VALSP(BaseType)),
    DEF_VALS2(MQIA_BRIDGE_EVENT            ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_CERT_VAL_POLICY         ,GET_VALSP(CertValPolicy)),
    DEF_VALS2(MQIA_CHANNEL_AUTO_DEF        ,GET_VALSP(ChanAutoDef)),
    DEF_VALS2(MQIA_CHANNEL_AUTO_DEF_EVENT  ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_CHANNEL_EVENT           ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_CHINIT_CONTROL          ,GET_VALSP(ServiceControl)),
    DEF_VALS2(MQIA_CHINIT_TRACE_AUTO_START ,GET_VALSP(ChinitTraceAutoStart)),
    DEF_VALS2(MQIA_CHLAUTH_RECORDS         ,GET_VALSP(ChlAuthRecords)),
    DEF_VALS2(MQIA_CLWL_USEQ               ,GET_VALSP(CLWLUseQ)),
    DEF_VALS2(MQIA_CMD_SERVER_CONTROL      ,GET_VALSP(ServiceControl)),
    DEF_VALS2(MQIA_COMMAND_EVENT           ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_CONFIGURATION_EVENT     ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_DEFINITION_TYPE         ,GET_VALSP(DefinitionType)),
    DEF_VALS2(MQIA_DEF_BIND                ,GET_VALSP(Bindings)),
    DEF_VALS2(MQIA_DEF_CLUSTER_XMIT_Q_TYPE ,GET_VALSP(DefClusXmitTyp)),
    DEF_VALS2(MQIA_DEF_INPUT_OPEN_OPTION   ,GET_VALSP(InputOpenOption)),
    DEF_VALS2(MQIA_DEF_PERSISTENCE         ,GET_VALSP(Persistence)),
    DEF_VALS2(MQIA_DEF_PUT_RESPONSE_TYPE   ,GET_VALSP(PutResponse)),
    DEF_VALS2(MQIA_DEF_READ_AHEAD          ,GET_VALSP(ReadAheadValues)),
    DEF_VALS2(MQIA_DIST_LISTS              ,GET_VALSP(DistributionLists)),
    DEF_VALS2(MQIA_DNS_WLM                 ,GET_VALSP(DNSWLM)),
    DEF_VALS2(MQIA_DURABLE_SUB             ,GET_VALSP(DurableSubscriptions)),
    DEF_VALS2(MQIA_GROUP_UR                ,GET_VALSP(GroupUR)),
    DEF_VALS2(MQIA_HARDEN_GET_BACKOUT      ,GET_VALSP(BackOutHardening)),
    DEF_VALS2(MQIA_IGQ_PUT_AUTHORITY       ,GET_VALSP(IGQPutAuthority)),
    DEF_VALS2(MQIA_INDEX_TYPE              ,GET_VALSP(IndexTypes)),
    DEF_VALS2(MQIA_INHIBIT_EVENT           ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_INHIBIT_GET             ,GET_VALSP(InhibitGetVal)),
    DEF_VALS2(MQIA_INHIBIT_PUB             ,GET_VALSP(InhibitPublications)),
    DEF_VALS2(MQIA_INHIBIT_PUT             ,GET_VALSP(InhibitPutVal)),
    DEF_VALS2(MQIA_INHIBIT_SUB             ,GET_VALSP(InhibitSubscriptions)),
    DEF_VALS2(MQIA_INTRA_GROUP_QUEUING     ,GET_VALSP(IntraGroupQueuing)),
    DEF_VALS2(MQIA_IP_ADDRESS_VERSION      ,GET_VALSP(IPAddressVersion)),
    DEF_VALS2(MQIA_LOCAL_EVENT             ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_LOGGER_EVENT            ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_MONITORING_AUTO_CLUSSDR ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_MONITORING_AUTO_CLUSSDR ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_MONITORING_CHANNEL      ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_MONITORING_Q            ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_MSG_DELIVERY_SEQUENCE   ,GET_VALSP(MsgDelivSeq)),
    DEF_VALS2(MQIA_NPM_CLASS               ,GET_VALSP(NonpersistentMsgCls)),
    DEF_VALS2(MQIA_NPM_DELIVERY            ,GET_VALSP(MsgDelivery)),
    DEF_VALS2(MQIA_PERFORMANCE_EVENT       ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_PLATFORM                ,GET_VALSP(Platform)),
    DEF_VALS2(MQIA_PM_DELIVERY             ,GET_VALSP(MsgDelivery)),
    DEF_VALS2(MQIA_PROPERTY_CONTROL        ,GET_VALSP(PropertyCtlVal)),
    DEF_VALS2(MQIA_PROT_POLICY_CAPABILITY  ,GET_VALSP(Capability)),
    DEF_VALS2(MQIA_PROXY_SUB               ,GET_VALSP(ProxySubscriptions)),
    DEF_VALS2(MQIA_PUBSUB_CLUSTER          ,GET_VALSP(PubSubCluster)),
    DEF_VALS2(MQIA_PUBSUB_MODE             ,GET_VALSP(PubSubMode)),
    DEF_VALS2(MQIA_PUBSUB_NP_MSG           ,GET_VALSP(PubSubNP)),
    DEF_VALS2(MQIA_PUBSUB_NP_RESP          ,GET_VALSP(PubSubNP)),
    DEF_VALS2(MQIA_PUBSUB_SYNC_PT          ,GET_VALSP(PubSubSync)),
    DEF_VALS2(MQIA_PUB_SCOPE               ,GET_VALSP(PubSubScope)),
    DEF_VALS2(MQIA_QMGR_CFCONLOS           ,GET_VALSP(CFConLoss)),
    DEF_VALS2(MQIA_QSG_DISP                ,GET_VALSP(QShrGrpDisp)),
    DEF_VALS2(MQIA_Q_DEPTH_HIGH_EVENT      ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_Q_DEPTH_LOW_EVENT       ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_Q_DEPTH_MAX_EVENT       ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_Q_SERVICE_INTERVAL_EVENT,GET_VALSP(ServiceIntervalEvent)),
    DEF_VALS2(MQIA_Q_TYPE                  ,GET_VALSP(QueueType)),
    DEF_VALS2(MQIA_RECEIVE_TIMEOUT_TYPE    ,GET_VALSP(ReceiveTimeoutType)),
    DEF_VALS2(MQIA_REMOTE_EVENT            ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_SCOPE                   ,GET_VALSP(QueueDefinitionScope)),
    DEF_VALS2(MQIA_SECURITY_CASE           ,GET_VALSP(SecurityCase)),
    DEF_VALS2(MQIA_SHAREABILITY            ,GET_VALSP(Shareability)),
    DEF_VALS2(MQIA_SHARED_Q_Q_MGR_NAME     ,GET_VALSP(SharedQueueQueueManagerName)),
    DEF_VALS2(MQIA_SSL_EVENT               ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_SSL_FIPS_REQUIRED       ,GET_VALSP(FipsRequired)),
    DEF_VALS2(MQIA_START_STOP_EVENT        ,GET_VALSP(EventRule)),
    DEF_VALS2(MQIA_STATISTICS_AUTO_CLUSSDR ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_STATISTICS_CHANNEL      ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_STATISTICS_INTERVAL     ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_STATISTICS_MQI          ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_STATISTICS_Q            ,GET_VALSP(Monitoring)),
    DEF_VALS2(MQIA_SUB_SCOPE               ,GET_VALSP(PubSubScope)),
    DEF_VALS2(MQIA_SYNCPOINT               ,GET_VALSP(SyncpointAvailability)),
    DEF_VALS2(MQIA_TCP_KEEP_ALIVE          ,GET_VALSP(TCPKeepAlive)),
    DEF_VALS2(MQIA_TCP_STACK_TYPE          ,GET_VALSP(TCPStackType)),
    DEF_VALS2(MQIA_TOPIC_DEF_PERSISTENCE   ,GET_VALSP(Persistence)),
    DEF_VALS2(MQIA_TOPIC_TYPE              ,GET_VALSP(TopicType)),
    DEF_VALS2(MQIA_TRACE_ROUTE_RECORDING   ,GET_VALSP(TraceRouteRecording)),
    DEF_VALS2(MQIA_TRIGGER_CONTROL         ,GET_VALSP(TriggerControl)),
    DEF_VALS2(MQIA_TRIGGER_TYPE            ,GET_VALSP(TriggerType)),
    DEF_VALS2(MQIA_UR_DISP                 ,GET_VALSP(QShrGrpDisp)),
    DEF_VALS2(MQIA_USAGE                   ,GET_VALSP(Usage)),
    DEF_VALS2(MQIA_WILDCARD_OPERATION      ,GET_VALSP(WildcardOperation)),
    DEF_VALS2(MQIA_XR_CAPABILITY           ,GET_VALSP(Capability)),
DEF_VALSE;


/*
 * Editor modelines - http://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
