! WHIZARD 2.4.1 Mar 24 2017
!
! Copyright (C) 1999-2017 by
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!
!     with contributions from
!     Fabian Bach <fabian.bach@t-online.de>
!     Bijan Chokoufe <bijan.chokoufe@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com>
!     So Young Shim <soyoung.shim@desy.de>
!     Florian Staub <florian.staub@cern.ch>
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam,
!     Sebastian Schmidt, So-young Shim, Daniel Wiesler
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module selectors

  use kinds, only: default
  use io_units
  use diagnostics
  use rng_base

  implicit none
  private

  public :: selector_t

  type :: selector_t
     integer, dimension(:), allocatable :: map
     real(default), dimension(:), allocatable :: weight
     real(default), dimension(:), allocatable :: acc
   contains
     procedure :: write => selector_write
     procedure :: init => selector_init
     procedure :: select => selector_select
     procedure :: generate => selector_generate
     procedure :: get_weight => selector_get_weight
end type selector_t


contains

  subroutine selector_write (object, unit)
    class(selector_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u, i
    u = given_output_unit (unit)
    write (u, "(1x,A)")  "Selector: i, weight, acc. weight"
    if (allocated (object%weight)) then
       do i = 1, size (object%weight)
          write (u, "(3x,I0,1x,ES19.12,1x,ES19.12)") &
               object%map(i), object%weight(i), object%acc(i)
       end do
    else
       write (u, "(3x,A)")  "[undefined]"
    end if
  end subroutine selector_write

  subroutine selector_init (selector, weight, negative_weights)
    class(selector_t), intent(out) :: selector
    real(default), dimension(:), intent(in) :: weight
    logical, intent(in), optional :: negative_weights
    real(default) :: s
    integer :: n, i
    logical :: neg_wgt
    logical, dimension(:), allocatable :: mask
    if (size (weight) == 0) &
         call msg_bug ("Selector init: zero-size weight array")
    neg_wgt = .false.
    if (present (negative_weights))  neg_wgt = negative_weights
    if (.not. neg_wgt .and. any (weight < 0)) &
         call msg_fatal ("Selector init: negative weight encountered")
    s = sum (weight)
    allocate (mask (size (weight)), &
         source = weight /= 0)
    n = count (mask)
    if (n > 0) then
       allocate (selector%map (n), &
            source = pack ([(i, i = 1, size (weight))], mask))
       allocate (selector%weight (n), &
            source = pack (abs (weight) / s, mask))
       allocate (selector%acc (n))
       selector%acc(1) = selector%weight(1)
       do i = 2, n - 1
          selector%acc(i) = selector%acc(i-1) + selector%weight(i)
       end do
       selector%acc(n) = 1
    else
       allocate (selector%map (1), source = 1)
       allocate (selector%weight (1), source = 0._default)
       allocate (selector%acc (1), source = 1._default)
    end if
  end subroutine selector_init

  function selector_select (selector, x) result (n)
    class(selector_t), intent(in) :: selector
    real(default), intent(in) :: x
    integer :: n
    integer :: i
    if (x < 0 .or. x > 1) &
         call msg_bug ("Selector: random number out of range")
    do i = 1, size (selector%acc)
       if (x <= selector%acc(i))  exit
    end do
    n = selector%map(i)
  end function selector_select

  subroutine selector_generate (selector, rng, n)
    class(selector_t), intent(in) :: selector
    class(rng_t), intent(inout) :: rng
    integer, intent(out) :: n
    real(default) :: x
    select case (size (selector%acc))
    case (1);  n = 1
    case default
       call rng%generate (x)
       n = selector%select (x)
    end select
  end subroutine selector_generate

  function selector_get_weight (selector, n) result (weight)
    class(selector_t), intent(in) :: selector
    integer, intent(in) :: n
    real(default) :: weight
    integer :: i
    do i = 1, size (selector%weight)
       if (selector%map(i) == n) then
          weight = selector%weight(i)
          return
       end if
    end do
    weight = 0
  end function selector_get_weight


end module selectors
