! WHIZARD 2.2.6 May 02 2015
! 
! Copyright (C) 1999-2015 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Fabian Bach <fabian.bach@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com> 
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam, 
!     Sebastian Schmidt, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'
module subevt_expr

  use kinds, only: default
  use iso_varying_string, string_t => varying_string
  use io_units
  use format_utils, only: write_separator
  use diagnostics
  use lorentz
  use subevents
  use variables
  use flavors
  use quantum_numbers
  use interactions
  use particles
  use expr_base

  implicit none
  private

  public :: parton_expr_t
!  public :: interaction_to_subevt
!  public :: interaction_momenta_to_subevt
  public :: event_expr_t

  type, extends (subevt_t), abstract :: subevt_expr_t
     logical :: subevt_filled = .false.
     type(var_list_t) :: var_list
     real(default) :: sqrts_hat = 0
     integer :: n_in = 0
     integer :: n_out = 0
     integer :: n_tot = 0
     logical :: has_selection = .false.
     class(expr_t), allocatable :: selection
   contains
     procedure :: base_write => subevt_expr_write
     procedure (subevt_expr_final), deferred :: final
     procedure :: base_final => subevt_expr_final
     procedure (subevt_expr_setup_vars), deferred :: setup_vars
     procedure :: base_setup_vars => subevt_expr_setup_vars
     procedure :: setup_var_self => subevt_expr_setup_var_self
     procedure :: link_var_list => subevt_expr_link_var_list
     procedure :: setup_selection => subevt_expr_setup_selection
     procedure :: reset => subevt_expr_reset
     procedure :: base_reset => subevt_expr_reset
     procedure :: base_evaluate => subevt_expr_evaluate
  end type subevt_expr_t
  
  type, extends (subevt_expr_t) :: parton_expr_t
     integer, dimension(:), allocatable :: i_beam
     integer, dimension(:), allocatable :: i_in
     integer, dimension(:), allocatable :: i_out
     logical :: has_scale = .false.
     logical :: has_fac_scale = .false.
     logical :: has_ren_scale = .false.
     logical :: has_weight = .false.
     class(expr_t), allocatable :: scale
     class(expr_t), allocatable :: fac_scale
     class(expr_t), allocatable :: ren_scale
     class(expr_t), allocatable :: weight
   contains
     procedure :: final => parton_expr_final
     procedure :: write => parton_expr_write
     procedure :: setup_vars => parton_expr_setup_vars
     procedure :: setup_scale => parton_expr_setup_scale
     procedure :: setup_fac_scale => parton_expr_setup_fac_scale
     procedure :: setup_ren_scale => parton_expr_setup_ren_scale
     procedure :: setup_weight => parton_expr_setup_weight
     procedure :: setup_subevt => parton_expr_setup_subevt
     procedure :: fill_subevt => parton_expr_fill_subevt
     procedure :: evaluate => parton_expr_evaluate
     procedure :: get_beam_index => parton_expr_get_beam_index
     procedure :: get_in_index => parton_expr_get_in_index
  end type parton_expr_t
     
  type, extends (subevt_expr_t) :: event_expr_t
     logical :: has_reweight = .false.
     logical :: has_analysis = .false.
     class(expr_t), allocatable :: reweight
     class(expr_t), allocatable :: analysis
     logical :: has_id = .false.
     type(string_t) :: id
     logical :: has_num_id = .false.
     integer :: num_id = 0
     logical :: has_index = .false.
     integer :: index = 0
     logical :: has_sqme_ref = .false.
     real(default) :: sqme_ref = 0
     logical :: has_sqme_prc = .false.
     real(default) :: sqme_prc = 0
     logical :: has_weight_ref = .false.
     real(default) :: weight_ref = 0
     logical :: has_weight_prc = .false.
     real(default) :: weight_prc = 0
     logical :: has_excess_prc = .false.
     real(default) :: excess_prc = 0
     integer :: n_alt = 0
     logical :: has_sqme_alt = .false.
     real(default), dimension(:), allocatable :: sqme_alt
     logical :: has_weight_alt = .false.
     real(default), dimension(:), allocatable :: weight_alt
   contains
     procedure :: final => event_expr_final
     procedure :: write => event_expr_write
     procedure :: init => event_expr_init
     procedure :: setup_vars => event_expr_setup_vars
     procedure :: setup_analysis => event_expr_setup_analysis
     procedure :: setup_reweight => event_expr_setup_reweight
     procedure :: set_process_id => event_expr_set_process_id
     procedure :: set_process_num_id => event_expr_set_process_num_id
     procedure :: reset => event_expr_reset
     procedure :: set => event_expr_set
     procedure :: fill_subevt => event_expr_fill_subevt
     procedure :: evaluate => event_expr_evaluate
  end type event_expr_t
     

  interface interaction_momenta_to_subevt
     module procedure interaction_momenta_to_subevt_id
     module procedure interaction_momenta_to_subevt_tr
  end interface


contains
  
  subroutine subevt_expr_write (object, unit, pacified)
    class(subevt_expr_t), intent(in) :: object
    integer, intent(in), optional :: unit
    logical, intent(in), optional :: pacified
    integer :: u
    u = given_output_unit (unit)
    write (u, "(1x,A)")  "Local variables:"
    call write_separator (u)
    call var_list_write (object%var_list, u, follow_link=.false., &
         pacified = pacified)
    call write_separator (u)
    if (object%subevt_filled) then
       call object%subevt_t%write (u, pacified = pacified)
       if (object%has_selection) then
          call write_separator (u)
          write (u, "(1x,A)")  "Selection expression:"
          call write_separator (u)
          call object%selection%write (u)
       end if
    else
       write (u, "(1x,A)")  "subevt: [undefined]"
    end if
  end subroutine subevt_expr_write
    
  subroutine subevt_expr_final (object)
    class(subevt_expr_t), intent(inout) :: object
    call object%var_list%final ()
    if (object%has_selection) then
       call object%selection%final ()
    end if
  end subroutine subevt_expr_final
  
  subroutine subevt_expr_setup_vars (expr, sqrts)
    class(subevt_expr_t), intent(inout), target :: expr
    real(default), intent(in) :: sqrts
    call expr%var_list%final ()
    call var_list_append_real (expr%var_list, &
         var_str ("sqrts"), sqrts, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("sqrts_hat"), expr%sqrts_hat, &
         is_known = expr%subevt_filled, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_int_ptr (expr%var_list, &
         var_str ("n_in"), expr%n_in, &
         is_known = expr%subevt_filled, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_int_ptr (expr%var_list, &
         var_str ("n_out"), expr%n_out, &
         is_known = expr%subevt_filled, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_int_ptr (expr%var_list, &
         var_str ("n_tot"), expr%n_tot, &
         is_known = expr%subevt_filled, &
         locked = .true., verbose = .false., intrinsic = .true.)
  end subroutine subevt_expr_setup_vars
    
  subroutine subevt_expr_setup_var_self (expr)
    class(subevt_expr_t), intent(inout), target :: expr
    if (.not. expr%var_list%contains (var_str ("@evt"))) then
       call var_list_append_subevt_ptr &
            (expr%var_list, &
            var_str ("@evt"), expr%subevt_t, &
            is_known = expr%subevt_filled, &
            locked = .true., verbose = .false., intrinsic=.true.)
    end if
  end subroutine subevt_expr_setup_var_self
  
  subroutine subevt_expr_link_var_list (expr, var_list)
    class(subevt_expr_t), intent(inout) :: expr
    type(var_list_t), intent(in), target :: var_list
    call expr%var_list%link (var_list)
  end subroutine subevt_expr_link_var_list

  subroutine subevt_expr_setup_selection (expr, ef_cuts)
    class(subevt_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_cuts
    call ef_cuts%build (expr%selection)
    if (allocated (expr%selection)) then
       call expr%setup_var_self ()
       call expr%selection%setup_lexpr (expr%var_list)
       expr%has_selection = .true.
    end if
  end subroutine subevt_expr_setup_selection

  subroutine subevt_expr_reset (expr)
    class(subevt_expr_t), intent(inout) :: expr
    expr%subevt_filled = .false.
  end subroutine subevt_expr_reset
  
  subroutine subevt_expr_evaluate (expr, passed)
    class(subevt_expr_t), intent(inout) :: expr
    logical, intent(out) :: passed 
    if (expr%has_selection) then
       call expr%selection%evaluate ()
       if (expr%selection%is_known ()) then
          passed = expr%selection%get_log ()
       else
          call msg_error ("Evaluate selection expression: result undefined")
          passed = .false.
       end if
    else
       passed = .true.
    end if
  end subroutine subevt_expr_evaluate
  
  subroutine parton_expr_final (object)
    class(parton_expr_t), intent(inout) :: object
    call object%base_final ()
    if (object%has_scale) then
       call object%scale%final ()
    end if
    if (object%has_fac_scale) then
       call object%fac_scale%final ()
    end if
    if (object%has_ren_scale) then
       call object%ren_scale%final ()
    end if
    if (object%has_weight) then
       call object%weight%final ()
    end if
  end subroutine parton_expr_final

  subroutine parton_expr_write (object, unit, prefix, pacified)
    class(parton_expr_t), intent(in) :: object
    integer, intent(in), optional :: unit
    character(*), intent(in), optional :: prefix
    logical, intent(in), optional :: pacified
    integer :: u
    u = given_output_unit (unit)
    call object%base_write (u, pacified = pacified)
    if (object%subevt_filled) then
       if (object%has_scale) then
          call write_separator (u)
          write (u, "(1x,A)")  "Scale expression:"
          call write_separator (u)
          call object%scale%write (u)
       end if
       if (object%has_fac_scale) then
          call write_separator (u)
          write (u, "(1x,A)")  "Factorization scale expression:"
          call write_separator (u)
          call object%fac_scale%write (u)
       end if
       if (object%has_ren_scale) then
          call write_separator (u)
          write (u, "(1x,A)")  "Renormalization scale expression:"
          call write_separator (u)
          call object%ren_scale%write (u)
       end if
       if (object%has_weight) then
          call write_separator (u)
          write (u, "(1x,A)")  "Weight expression:"
          call write_separator (u)
          call object%weight%write (u)
       end if
    end if
  end subroutine parton_expr_write
    
  subroutine parton_expr_setup_vars (expr, sqrts)
    class(parton_expr_t), intent(inout), target :: expr
    real(default), intent(in) :: sqrts
    call expr%base_setup_vars (sqrts)
  end subroutine parton_expr_setup_vars

  subroutine parton_expr_setup_scale (expr, ef_scale)
    class(parton_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_scale
    call ef_scale%build (expr%scale)
    if (allocated (expr%scale)) then
       call expr%setup_var_self ()
       call expr%scale%setup_expr (expr%var_list)
       expr%has_scale = .true.
    end if
  end subroutine parton_expr_setup_scale

  subroutine parton_expr_setup_fac_scale (expr, ef_fac_scale)
    class(parton_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_fac_scale
    call ef_fac_scale%build (expr%fac_scale)
    if (allocated (expr%fac_scale)) then
       call expr%setup_var_self ()
       call expr%fac_scale%setup_expr (expr%var_list)
       expr%has_fac_scale = .true.
    end if
  end subroutine parton_expr_setup_fac_scale

  subroutine parton_expr_setup_ren_scale (expr, ef_ren_scale)
    class(parton_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_ren_scale
    call ef_ren_scale%build (expr%ren_scale)
    if (allocated (expr%ren_scale)) then
       call expr%setup_var_self ()
       call expr%ren_scale%setup_expr (expr%var_list)
       expr%has_ren_scale = .true.
    end if
  end subroutine parton_expr_setup_ren_scale

  subroutine parton_expr_setup_weight (expr, ef_weight)
    class(parton_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_weight
    call ef_weight%build (expr%weight)
    if (allocated (expr%weight)) then
       call expr%setup_var_self ()
       call expr%weight%setup_expr (expr%var_list)
       expr%has_weight = .true.
    end if
  end subroutine parton_expr_setup_weight

  subroutine parton_expr_setup_subevt (expr, int, &
       i_beam, i_in, i_out, f_beam, f_in, f_out)
    class(parton_expr_t), intent(inout) :: expr
    type(interaction_t), intent(in), target :: int
    integer, dimension(:), intent(in) :: i_beam, i_in, i_out
    type(flavor_t), dimension(:), intent(in) :: f_beam, f_in, f_out
    allocate (expr%i_beam (size (i_beam)))
    allocate (expr%i_in (size (i_in)))
    allocate (expr%i_out (size (i_out)))
    expr%i_beam = i_beam
    expr%i_in = i_in
    expr%i_out = i_out
    call interaction_to_subevt (int, &
         expr%i_beam, expr%i_in, expr%i_out, expr%subevt_t)
    call subevt_set_pdg_beam     (expr%subevt_t, f_beam%get_pdg ())
    call subevt_set_pdg_incoming (expr%subevt_t, f_in%get_pdg ())
    call subevt_set_pdg_outgoing (expr%subevt_t, f_out%get_pdg ())
    call subevt_set_p2_beam     (expr%subevt_t, f_beam%get_mass () ** 2)
    call subevt_set_p2_incoming (expr%subevt_t, f_in%get_mass ()   ** 2)
    call subevt_set_p2_outgoing (expr%subevt_t, f_out%get_mass ()  ** 2)
    expr%n_in  = size (i_in)
    expr%n_out = size (i_out)
    expr%n_tot = expr%n_in + expr%n_out
  end subroutine parton_expr_setup_subevt

  subroutine interaction_to_subevt (int, j_beam, j_in, j_out, subevt)
    type(interaction_t), intent(in), target :: int
    integer, dimension(:), intent(in) :: j_beam, j_in, j_out
    type(subevt_t), intent(out) :: subevt
    type(flavor_t), dimension(:), allocatable :: flv
    integer :: n_beam, n_in, n_out, i, j
    allocate (flv (int%get_n_tot ()))
    flv = quantum_numbers_get_flavor (int%get_quantum_numbers (1))
    n_beam = size (j_beam)
    n_in = size (j_in)
    n_out = size (j_out)
    call subevt_init (subevt, n_beam + n_in + n_out)
    do i = 1, n_beam
       j = j_beam(i)
       call subevt_set_beam (subevt, i, &
            flv(j)%get_pdg (), &
            vector4_null, &
            flv(j)%get_mass () ** 2)
    end do
    do i = 1, n_in
       j = j_in(i)
       call subevt_set_incoming (subevt, n_beam + i, &
            flv(j)%get_pdg (), &
            vector4_null, &
            flv(j)%get_mass () ** 2)
    end do
    do i = 1, n_out
       j = j_out(i)
       call subevt_set_outgoing (subevt, n_beam + n_in + i, &
            flv(j)%get_pdg (), &
            vector4_null, &
            flv(j)%get_mass () ** 2)
    end do
  end subroutine interaction_to_subevt

  subroutine interaction_momenta_to_subevt_id (int, j_beam, j_in, j_out, subevt)
    type(interaction_t), intent(in) :: int
    integer, dimension(:), intent(in) :: j_beam, j_in, j_out
    type(subevt_t), intent(inout) :: subevt
    call subevt_set_p_beam (subevt, - int%get_momenta (j_beam))
    call subevt_set_p_incoming (subevt, - int%get_momenta (j_in))
    call subevt_set_p_outgoing (subevt, int%get_momenta (j_out))
  end subroutine interaction_momenta_to_subevt_id

  subroutine interaction_momenta_to_subevt_tr &
       (int, j_beam, j_in, j_out, lt, subevt)
    type(interaction_t), intent(in) :: int
    integer, dimension(:), intent(in) :: j_beam, j_in, j_out
    type(subevt_t), intent(inout) :: subevt
    type(lorentz_transformation_t), intent(in) :: lt
    call subevt_set_p_beam &
         (subevt, - lt * int%get_momenta (j_beam))
    call subevt_set_p_incoming &
         (subevt, - lt * int%get_momenta (j_in))
    call subevt_set_p_outgoing &
         (subevt, lt * int%get_momenta (j_out))
  end subroutine interaction_momenta_to_subevt_tr

  subroutine parton_expr_fill_subevt (expr, int)
    class(parton_expr_t), intent(inout) :: expr
    type(interaction_t), intent(in), target :: int
    call interaction_momenta_to_subevt (int, &
         expr%i_beam, expr%i_in, expr%i_out, expr%subevt_t)
    expr%sqrts_hat = subevt_get_sqrts_hat (expr%subevt_t)
    expr%subevt_filled = .true.
  end subroutine parton_expr_fill_subevt
    
  subroutine parton_expr_evaluate &
       (expr, passed, scale, fac_scale, ren_scale, weight, scale_forced)
    class(parton_expr_t), intent(inout) :: expr
    logical, intent(out) :: passed
    real(default), intent(out) :: scale
    real(default), intent(out) :: fac_scale
    real(default), intent(out) :: ren_scale
    real(default), intent(out) :: weight
    real(default), intent(in), allocatable, optional :: scale_forced
    logical :: force_scale
    force_scale = .false.
    if (present (scale_forced))  force_scale = allocated (scale_forced)
    call expr%base_evaluate (passed)
    if (passed) then
       if (force_scale) then
          scale = scale_forced
       else if (expr%has_scale) then
          call expr%scale%evaluate ()
          if (expr%scale%is_known ()) then
             scale = expr%scale%get_real ()
          else
             call msg_error ("Evaluate scale expression: result undefined")
             scale = 0
          end if
       else
          scale = expr%sqrts_hat
       end if
       if (force_scale) then
          fac_scale = scale_forced
       else if (expr%has_fac_scale) then
          call expr%fac_scale%evaluate ()
          if (expr%fac_scale%is_known ()) then
             fac_scale = expr%fac_scale%get_real ()
          else
             call msg_error ("Evaluate factorization scale expression: &
                  &result undefined")
             fac_scale = 0
          end if
       else
          fac_scale = scale
       end if
       if (force_scale) then
          ren_scale = scale_forced
       else if (expr%has_ren_scale) then
          call expr%ren_scale%evaluate ()
          if (expr%ren_scale%is_known ()) then
             ren_scale = expr%ren_scale%get_real ()
          else
             call msg_error ("Evaluate renormalization scale expression: &
                  &result undefined")
             ren_scale = 0
          end if
       else
          ren_scale = scale
       end if
       if (expr%has_weight) then
          call expr%weight%evaluate ()
          if (expr%weight%is_known ()) then
             weight = expr%weight%get_real ()
          else
             call msg_error ("Evaluate weight expression: result undefined")
             weight = 0
          end if
       else
          weight = 1
       end if
    end if
  end subroutine parton_expr_evaluate
  
  subroutine parton_expr_get_beam_index (expr, i_beam)
    class(parton_expr_t), intent(in) :: expr
    integer, dimension(:), intent(out) :: i_beam
    i_beam = expr%i_beam
  end subroutine parton_expr_get_beam_index
  
  subroutine parton_expr_get_in_index (expr, i_in)
    class(parton_expr_t), intent(in) :: expr
    integer, dimension(:), intent(out) :: i_in
    i_in = expr%i_in
  end subroutine parton_expr_get_in_index
  
  subroutine event_expr_final (object)
    class(event_expr_t), intent(inout) :: object
    call object%base_final ()
    if (object%has_reweight) then
       call object%reweight%final ()
    end if
    if (object%has_analysis) then
       call object%analysis%final ()
    end if
  end subroutine event_expr_final

  subroutine event_expr_write (object, unit, prefix, pacified)
    class(event_expr_t), intent(in) :: object
    integer, intent(in), optional :: unit
    character(*), intent(in), optional :: prefix
    logical, intent(in), optional :: pacified
    integer :: u
    u = given_output_unit (unit)
    call object%base_write (u, pacified = pacified)
    if (object%subevt_filled) then
       if (object%has_reweight) then
          call write_separator (u)
          write (u, "(1x,A)")  "Reweighting expression:"
          call write_separator (u)
          call object%reweight%write (u)
       end if
       if (object%has_analysis) then
          call write_separator (u)
          write (u, "(1x,A)")  "Analysis expression:"
          call write_separator (u)
          call object%analysis%write (u)
       end if
    end if
  end subroutine event_expr_write
    
  subroutine event_expr_init (expr, n_alt)
    class(event_expr_t), intent(out) :: expr
    integer, intent(in), optional :: n_alt
    if (present (n_alt)) then
       expr%n_alt = n_alt
       allocate (expr%sqme_alt (n_alt), source = 0._default)
       allocate (expr%weight_alt (n_alt), source = 0._default)
    end if
  end subroutine event_expr_init
  
  subroutine event_expr_setup_vars (expr, sqrts)
    class(event_expr_t), intent(inout), target :: expr
    real(default), intent(in) :: sqrts
    call expr%base_setup_vars (sqrts)
    call var_list_append_string_ptr (expr%var_list, &
         var_str ("$process_id"), expr%id, &
         is_known = expr%has_id, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_int_ptr (expr%var_list, &
         var_str ("process_num_id"), expr%num_id, &
         is_known = expr%has_num_id, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("sqme"), expr%sqme_prc, &
         is_known = expr%has_sqme_prc, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("sqme_ref"), expr%sqme_ref, &
         is_known = expr%has_sqme_ref, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_int_ptr (expr%var_list, &
         var_str ("event_index"), expr%index, &
         is_known = expr%has_index, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("event_weight"), expr%weight_prc, &
         is_known = expr%has_weight_prc, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("event_weight_ref"), expr%weight_ref, &
         is_known = expr%has_weight_ref, &
         locked = .true., verbose = .false., intrinsic = .true.)
    call var_list_append_real_ptr (expr%var_list, &
         var_str ("event_excess"), expr%excess_prc, &
         is_known = expr%has_excess_prc, &
         locked = .true., verbose = .false., intrinsic = .true.)
  end subroutine event_expr_setup_vars

  subroutine event_expr_setup_analysis (expr, ef_analysis)
    class(event_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_analysis
    call ef_analysis%build (expr%analysis)
    if (allocated (expr%analysis)) then
       call expr%setup_var_self ()
       call expr%analysis%setup_lexpr (expr%var_list)
       expr%has_analysis = .true.
    end if
  end subroutine event_expr_setup_analysis

  subroutine event_expr_setup_reweight (expr, ef_reweight)
    class(event_expr_t), intent(inout), target :: expr
    class(expr_factory_t), intent(in) :: ef_reweight
    call ef_reweight%build (expr%reweight)
    if (allocated (expr%reweight)) then
       call expr%setup_var_self ()
       call expr%reweight%setup_expr (expr%var_list)
       expr%has_reweight = .true.
    end if
  end subroutine event_expr_setup_reweight

  subroutine event_expr_set_process_id (expr, id)
    class(event_expr_t), intent(inout) :: expr
    type(string_t), intent(in) :: id
    expr%id = id
    expr%has_id = .true.
  end subroutine event_expr_set_process_id
    
  subroutine event_expr_set_process_num_id (expr, num_id)
    class(event_expr_t), intent(inout) :: expr
    integer, intent(in) :: num_id
    expr%num_id = num_id
    expr%has_num_id = .true.
  end subroutine event_expr_set_process_num_id
    
  subroutine event_expr_reset (expr)
    class(event_expr_t), intent(inout) :: expr
    call expr%base_reset ()
    expr%has_sqme_ref = .false.
    expr%has_sqme_prc = .false.
    expr%has_sqme_alt = .false.
    expr%has_weight_ref = .false.
    expr%has_weight_prc = .false.
    expr%has_weight_alt = .false.
    expr%has_excess_prc = .false.
  end subroutine event_expr_reset
  
  subroutine event_expr_set (expr, &
       weight_ref, weight_prc, weight_alt, &
       excess_prc, &
       sqme_ref, sqme_prc, sqme_alt)
    class(event_expr_t), intent(inout) :: expr
    real(default), intent(in), optional :: weight_ref, weight_prc
    real(default), intent(in), optional :: excess_prc
    real(default), intent(in), optional :: sqme_ref, sqme_prc
    real(default), dimension(:), intent(in), optional :: sqme_alt, weight_alt
    if (present (sqme_ref)) then
       expr%has_sqme_ref = .true.
       expr%sqme_ref = sqme_ref
    end if
    if (present (sqme_prc)) then
       expr%has_sqme_prc = .true.
       expr%sqme_prc = sqme_prc
    end if 
    if (present (sqme_alt)) then
       expr%has_sqme_alt = .true.
       expr%sqme_alt = sqme_alt
    end if
    if (present (weight_ref)) then
       expr%has_weight_ref = .true.
       expr%weight_ref = weight_ref
    end if
    if (present (weight_prc)) then
       expr%has_weight_prc = .true.
       expr%weight_prc = weight_prc
    end if
    if (present (weight_alt)) then
       expr%has_weight_alt = .true.
       expr%weight_alt = weight_alt
    end if
    if (present (excess_prc)) then
       expr%has_excess_prc = .true.
       expr%excess_prc = excess_prc
    end if
  end subroutine event_expr_set
  
  subroutine event_expr_fill_subevt (expr, particle_set)
    class(event_expr_t), intent(inout) :: expr
    type(particle_set_t), intent(in) :: particle_set
    call particle_set%to_subevt (expr%subevt_t)
    expr%sqrts_hat = subevt_get_sqrts_hat (expr%subevt_t)
    expr%n_in  = subevt_get_n_in  (expr%subevt_t)
    expr%n_out = subevt_get_n_out (expr%subevt_t)
    expr%n_tot = expr%n_in + expr%n_out
    expr%subevt_filled = .true.
    if (expr%has_index) then
       expr%index = expr%index + 1
    else
       expr%index = 1
       expr%has_index = .true.
    end if
  end subroutine event_expr_fill_subevt
  
  subroutine event_expr_evaluate (expr, passed, reweight, analysis_flag)
    class(event_expr_t), intent(inout) :: expr
    logical, intent(out) :: passed
    real(default), intent(out) :: reweight
    logical, intent(out) :: analysis_flag
    call expr%base_evaluate (passed)
    if (passed) then
       if (expr%has_reweight) then
          call expr%reweight%evaluate ()
          if (expr%reweight%is_known ()) then
             reweight = expr%reweight%get_real ()
          else
             call msg_error ("Evaluate reweight expression: &
                  &result undefined")
             reweight = 0
          end if
       else
          reweight = 1
       end if
       if (expr%has_analysis) then
          call expr%analysis%evaluate ()
          if (expr%analysis%is_known ()) then
             analysis_flag = expr%analysis%get_log ()
          else
             call msg_error ("Evaluate analysis expression: &
                  &result undefined")
             analysis_flag = .false.
          end if
       else
          analysis_flag = .true.
       end if
    end if
  end subroutine event_expr_evaluate
  

end module subevt_expr
