! WHIZARD 2.2.6 May 02 2015
! 
! Copyright (C) 1999-2015 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Fabian Bach <fabian.bach@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com> 
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam, 
!     Sebastian Schmidt, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module powheg

  use kinds, only: default
  use constants
  use iso_varying_string, string_t => varying_string
  use, intrinsic :: iso_fortran_env
  use unit_tests
  use io_units, only: given_output_unit, free_unit
  use format_utils, only: write_separator
  use format_defs
  use os_interface
  use string_utils
  use physics_defs
  use diagnostics
  use subevents
  use rng_base
  use grids
  use solver
  use nlo_controller
  use phs_fks
  use lorentz
  use particles
  use interactions
  use colors
  use helicities
  use flavors
  use polarizations
  use pdg_arrays, only: is_quark, is_gluon

  ! For integration tests
  use rng_tao
  use os_interface
  use processes
  use process_libraries
  use models
  use model_data
  use variables
  use sm_qcd
  use prc_core
  use prc_omega
  use mci_base
  use mci_midpoint
  use phs_base
  use phs_single

  implicit none
  private

  public :: powheg_settings_t
  public :: powheg_testing_t
  public :: radiation_t
  public :: process_deps_t
  public :: event_deps_t
  public :: sudakov_t
  public :: sudakov_wrapper_t
  public :: sudakov_simple_fsr_t
  public :: sudakov_eeqq_fsr_t
  public :: powheg_t
  public :: powheg_test

  logical, parameter :: ENSURE = .true.
  logical, parameter :: DEBUG_EVENTS = .false.
  integer, parameter :: UBF_SIMPLE = 1
  integer, parameter :: UBF_EEQQ = 2
  real(default), parameter :: b0rad = (33 - 2 * 5) / (12 * pi)

  type :: powheg_settings_t
     real(default) :: pt2_min
     real(default) :: lambda
     integer :: n_init
     integer :: size_grid_xi
     integer :: size_grid_y
     integer :: upper_bound_func = UBF_SIMPLE
     logical :: rebuild_grids = .false.
  contains
     procedure :: init => powheg_settings_init
     procedure :: write => powheg_settings_write
  end type powheg_settings_t

  type :: powheg_testing_t
     integer :: n_alr, n_in, n_out_born, n_out_real
     real(default) :: sqme_born
     logical :: active
  end type powheg_testing_t

  type :: radiation_t
    real(default) :: xi, y, phi, pt2
    integer :: alr
    logical :: valid = .false.
  contains
    procedure :: write => radiation_write
  end type radiation_t

  type :: process_deps_t
     real(default) :: lambda2_gen
     integer :: n_alr
  contains
     procedure :: write => process_deps_write
  end type process_deps_t

  type :: event_deps_t
     real(default) :: cms_energy2
     type(vector4_t), dimension(:), allocatable :: p_born
     type(vector4_t), dimension(:), allocatable :: p_real
     real(default) :: sqme_born
  contains
     procedure :: write => event_deps_write
     procedure :: update => event_deps_update
     procedure :: set_cms => event_deps_set_cms
  end type event_deps_t

  type :: veto_counter_t
    integer :: n_ubf = 0
    integer :: n_first_fail = 0
    integer :: n_alpha_s = 0
    integer :: n_norm = 0
    integer :: n_sqme = 0
    integer :: veto_ubf = 0
    integer :: veto_alpha_s = 0
    integer :: veto_norm = 0
    integer :: veto_sqme = 0
    integer :: n_veto_fail = 0
  contains
    procedure :: record_ubf => veto_counter_record_ubf
    procedure :: record_first_fail => veto_counter_record_first_fail
    procedure :: record_alpha_s => veto_counter_record_alpha_s
    procedure :: record_norm => veto_counter_record_norm
    procedure :: record_sqme => veto_counter_record_sqme
    procedure :: record_fail => veto_counter_record_fail
    procedure :: write => veto_counter_write
  end type veto_counter_t

  type, abstract, extends (solver_function_t) :: sudakov_t
     type(process_deps_t), pointer :: process_deps => null()
     type(event_deps_t), pointer :: event_deps => null()
     type(powheg_settings_t), pointer :: powheg_settings => null()
     type(qcd_t), pointer :: qcd => null()
     class(rng_t), pointer :: rng => null()
     real(default) :: xi2_max = 0
     real(default) :: norm_max = 0
     real(default) :: current_pt2_max = 0
     real(default) :: last_log = 0
     real(default) :: random = 0
     type(veto_counter_t) :: veto_counter
  contains
     procedure :: write => sudakov_write
     procedure :: init => sudakov_init
     procedure :: set_normalization => sudakov_set_normalization
     procedure :: update => sudakov_update
     procedure (sudakov_upper_bound_func), deferred :: upper_bound_func
     procedure (sudakov_log_integrated_ubf), deferred :: log_integrated_ubf
     procedure (sudakov_generate_xi_and_y_and_phi), deferred :: generate_xi_and_y_and_phi
     procedure (sudakov_kt2), deferred :: kt2
     procedure (sudakov_reweight_ubf), deferred :: reweight_ubf
     procedure :: alpha_s => sudakov_alpha_s
     procedure :: generate_pt2 => sudakov_generate_pt2
     procedure :: check_solution_interval => sudakov_check_solution_interval
     procedure :: generate_emission => sudakov_generate_emission
     procedure :: evaluate => sudakov_evaluate
     procedure :: alpha_s_rad => sudakov_alpha_s_rad
     procedure :: reweight_alpha_s => sudakov_reweight_alpha_s
  end type sudakov_t

  type :: sudakov_wrapper_t
     class(sudakov_t), allocatable :: s
  end type sudakov_wrapper_t

  type, extends (sudakov_t) :: sudakov_simple_fsr_t
  contains
     procedure :: upper_bound_func => sudakov_simple_fsr_upper_bound_func
     procedure :: kt2 => sudakov_simple_fsr_kt2
     procedure :: log_integrated_ubf => sudakov_simple_fsr_log_integrated_ubf
     procedure :: reweight_ubf => sudakov_simple_fsr_reweight_ubf
     procedure :: generate_xi_and_y_and_phi => sudakov_simple_fsr_generate_xi_and_y_and_phi
     procedure :: generate_xi => sudakov_simple_fsr_generate_xi
  end type sudakov_simple_fsr_t

  type, extends (sudakov_t) :: sudakov_eeqq_fsr_t
  contains
     procedure :: kt2 => sudakov_eeqq_fsr_kt2
     procedure :: upper_bound_func => sudakov_eeqq_fsr_upper_bound_func
     procedure :: log_integrated_ubf => sudakov_eeqq_fsr_log_integrated_ubf
     procedure :: reweight_ubf => sudakov_eeqq_fsr_reweight_ubf
     procedure :: generate_xi_and_y_and_phi => sudakov_eeqq_fsr_generate_xi_and_y_and_phi
  end type sudakov_eeqq_fsr_t

  type :: powheg_t
     type(string_t) :: process_name
     class(rng_t), allocatable :: rng
     type(grid_t) :: grid
     type(phs_fks_generator_t) :: phs_fks_generator
     type(powheg_settings_t) :: settings
     type(powheg_testing_t) :: testing
     type(process_instance_t), pointer :: process_instance => null()
     type(event_deps_t) :: event_deps
     type(process_deps_t) :: process_deps
     type(sudakov_wrapper_t), dimension(:), allocatable :: sudakov
     type(qcd_t), pointer :: qcd => null()
     integer :: n_emissions = 0
  contains
     procedure :: display_grid_startup_message => &
                         powheg_display_grid_startup_message
     procedure :: write => powheg_write
     procedure :: write_statistics => powheg_write_statistics
     procedure :: connect => powheg_connect
     procedure :: setup_grids => powheg_setup_grids
     procedure :: setup_sudakovs => powheg_setup_sudakovs
     procedure :: init => powheg_init
     procedure :: import_rng => powheg_import_rng
     generic :: update => update_momenta, &
                          update_particle_set
     procedure :: update_momenta => powheg_update_momenta
     procedure :: update_particle_set => powheg_update_particle_set
     procedure :: reweight_matrix_elements => powheg_reweight_matrix_elements
     procedure :: compute_sqme_real => powheg_compute_sqme_real
     procedure :: set_scale => powheg_set_scale
     procedure :: fill_grids => powheg_fill_grids
     procedure :: generate_xi_and_y_for_grids => powheg_generate_xi_and_y_for_grids
     procedure :: prepare_momenta_for_fill_grids => powheg_prepare_momenta_for_fill_grids
     procedure :: above_pt2_min => powheg_above_pt2_min
     procedure :: set_normalizations => powheg_set_normalizations
     procedure :: save_grids => powheg_save_grids
     procedure :: load_grids => powheg_load_grids
     procedure :: requires_new_grids => powheg_requires_new_grids
     procedure :: generate_emission => powheg_generate_emission
     procedure :: build_particle_set => powheg_build_particle_set
     procedure :: reweight_norm => powheg_reweight_norm
     procedure :: norm_from_xi_and_y => powheg_norm_from_xi_and_y
     procedure :: compute_lambda2_gen => powheg_compute_lambda2_gen
     procedure :: test_sudakov => powheg_test_sudakov
  end type powheg_t


  abstract interface
     pure function sudakov_upper_bound_func (sudakov, xi, y, alpha_s) result (u)
       import
       real(default) :: u
       class(sudakov_t), intent(in) :: sudakov
       real(default), intent(in) :: xi, y, alpha_s
     end function sudakov_upper_bound_func
  end interface

  abstract interface
     pure function sudakov_log_integrated_ubf (sudakov, pt2) result (y)
       import
       real(default) :: y
       class(sudakov_t), intent(in) :: sudakov
       real(default), intent(in) :: pt2
     end function sudakov_log_integrated_ubf
  end interface

  abstract interface
     subroutine sudakov_generate_xi_and_y_and_phi (sudakov, r)
       import
       class(sudakov_t), intent(inout) :: sudakov
       type(radiation_t), intent(inout) :: r
     end subroutine sudakov_generate_xi_and_y_and_phi
  end interface

  abstract interface
     pure function sudakov_kt2 (sudakov, xi, y) result (kt2)
       import
       real(default) :: kt2
       class(sudakov_t), intent(in) :: sudakov
       real(default), intent(in) :: xi, y
     end function sudakov_kt2
  end interface

  abstract interface
     function sudakov_reweight_ubf (sudakov, pt2) result (accepted)
       import
       logical accepted
       class(sudakov_t), intent(inout) :: sudakov
       real(default), intent(in) :: pt2
    end function sudakov_reweight_ubf
  end interface


contains

  subroutine powheg_settings_init (settings, var_list)
    class(powheg_settings_t), intent(out) :: settings
    type(var_list_t), intent(in) :: var_list
    settings%size_grid_xi = &
         var_list%get_ival (var_str ("powheg_grid_size_xi"))
    settings%size_grid_y = &
         var_list%get_ival (var_str ("powheg_grid_size_y"))
    settings%n_init = &
         var_list%get_ival (var_str ("powheg_grid_sampling_points"))
    settings%pt2_min = &
         var_list%get_rval (var_str ("powheg_pt_min"))**2
    settings%lambda = var_list%get_rval (var_str ("powheg_lambda"))
    settings%rebuild_grids = &
         var_list%get_lval (var_str ("?powheg_rebuild_grids"))
  end subroutine powheg_settings_init

  subroutine powheg_settings_write (powheg_settings, unit)
    class(powheg_settings_t), intent(in) :: powheg_settings
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit);  if (u < 0)  return
    write (u, "(1X,A)") "Powheg settings:"
    write (u, "(3X,A," // FMT_16 //")") "pt2_min = ", powheg_settings%pt2_min
    write (u, "(3X,A," // FMT_16 //")") "lambda = ", powheg_settings%lambda
    write (u, "(3X,A,I12)") "n_init = ", powheg_settings%n_init
    write (u, "(3X,A,I12)") "size_grid_xi = ", powheg_settings%size_grid_xi
    write (u, "(3X,A,I12)") "size_grid_y = ", powheg_settings%size_grid_y
    write (u, "(3X,A,I12)") "upper_bound_func = ", powheg_settings%upper_bound_func
  end subroutine powheg_settings_write

  subroutine radiation_write (radiation, unit)
    class(radiation_t), intent(in) :: radiation
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit);  if (u < 0)  return
    write (u, "(1X, A)") "Radiation:"
    write (u, "(3X, A," // FMT_16 // ")") "xi = ", radiation%xi
    write (u, "(3X, A," // FMT_16 // ")") "y = ", radiation%y
    write (u, "(3X, A," // FMT_16 // ")") "phi = ", radiation%phi
    write (u, "(3X, A," // FMT_16 // ")") "pt2 = ", radiation%pt2
    write (u, "(3X, A, I12)") "alr = ", radiation%alr
  end subroutine radiation_write

  subroutine process_deps_write (process_deps, unit)
    class(process_deps_t), intent(in) :: process_deps
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit);  if (u < 0)  return
    write (u, "(1X,A)") "Process dependencies:"
    write (u, "(3X,A," // FMT_19 // ")") "lambda2_gen = ", process_deps%lambda2_gen
    write (u, "(3X,A, I12)") "n_alr = ", process_deps%n_alr
  end subroutine process_deps_write

  subroutine event_deps_write (event_deps, unit)
    class(event_deps_t), intent(in) :: event_deps
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit);  if (u < 0)  return
    write (u, "(1X,A)") "Event dependencies:"
    write (u, "(3X,A," // FMT_19 // ")") "cms_energy2 = ", event_deps%cms_energy2
    write (u, "(3X,A," // FMT_19 // ")") "sqme_born = ", event_deps%sqme_born
  end subroutine event_deps_write

  subroutine event_deps_update (event_deps, sqme_born, p_born)
    class(event_deps_t), intent(inout) :: event_deps
    real(default), intent(in) :: sqme_born
    type(vector4_t), dimension(:), intent(in) :: p_born
    event_deps%sqme_born = sqme_born
    if (ENSURE) then
       if (size (p_born) /= size (event_deps%p_born)) then
          call msg_fatal ("event_deps_update: number of born_momenta has changed")
       end if
    end if
    event_deps%p_born = p_born
    call event_deps%set_cms ()
  end subroutine event_deps_update

  pure subroutine event_deps_set_cms (event_deps)
    class(event_deps_t), intent(inout) :: event_deps
    event_deps%cms_energy2 = &
         (event_deps%p_born(1) + event_deps%p_born(2))**2
  end subroutine event_deps_set_cms

  pure subroutine veto_counter_record_ubf (counter, vetoed)
    class(veto_counter_t), intent(inout) :: counter
    logical, intent(in) :: vetoed
    counter%n_ubf = counter%n_ubf + 1
    if (vetoed) counter%veto_ubf = counter%veto_ubf + 1
  end subroutine veto_counter_record_ubf

  subroutine veto_counter_record_first_fail (counter)
    class(veto_counter_t), intent(inout) :: counter
    counter%n_first_fail = counter%n_first_fail + 1
  end subroutine veto_counter_record_first_fail

  subroutine veto_counter_record_alpha_s (counter, vetoed)
    class(veto_counter_t), intent(inout) :: counter
    logical, intent(in) :: vetoed
    counter%n_alpha_s = counter%n_alpha_s + 1
    if (vetoed) counter%veto_alpha_s = counter%veto_alpha_s + 1
  end subroutine veto_counter_record_alpha_s

  subroutine veto_counter_record_norm (counter, vetoed)
    class(veto_counter_t), intent(inout) :: counter
    logical, intent(in) :: vetoed
    counter%n_norm = counter%n_norm + 1
    if (vetoed) counter%veto_norm = counter%veto_norm + 1
  end subroutine veto_counter_record_norm

  subroutine veto_counter_record_sqme (counter, vetoed)
    class(veto_counter_t), intent(inout) :: counter
    logical, intent(in) :: vetoed
    counter%n_sqme = counter%n_sqme + 1
    if (vetoed) counter%veto_sqme = counter%veto_sqme + 1
  end subroutine veto_counter_record_sqme

  subroutine veto_counter_record_fail (counter)
    class(veto_counter_t), intent(inout) :: counter
    counter%n_veto_fail = counter%n_veto_fail + 1
  end subroutine veto_counter_record_fail

  subroutine veto_counter_write (counter, unit)
    class(veto_counter_t), intent(in) :: counter
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit); if (u < 0) return
    write (u, "(A,I12)") "Nr. of ubf-veto calls: ", counter%n_ubf
    write (u, "(A,I12)") "Nr. of ubf-vetos: ", counter%veto_ubf
    if (counter%n_ubf > 0) &
       write (u, "(A,F4.2)") "Fraction of vetoed points: ", &
                            one*counter%veto_ubf / counter%n_ubf
    call write_separator (u)
    write (u, "(A,I12)") "Nr. of alpha_s-veto calls: ", counter%n_alpha_s
    write (u, "(A,I12)") "Nr. of alpha_s-vetos: ", counter%veto_alpha_s
    if (counter%n_alpha_s > 0) &
       write (u, "(A,F4.2)") "Fraction of vetoed points: ", &
                            one*counter%veto_alpha_s / counter%n_alpha_s
    call write_separator (u)
    write (u, "(A,I0)") "Nr. of norm-veto calls: ", counter%n_norm
    write (u, "(A,I0)") "Nr. of norm-vetos: ", counter%veto_norm
    if (counter%n_norm > 0) &
       write (u, "(A,F4.2)") "Fraction of vetoed points: ", &
                            one*counter%veto_norm / counter%n_norm
    call write_separator (u)
    write (u, "(A,I0)") "Nr. of sqme-veto calls: ", counter%n_sqme
    write (u, "(A,I0)") "Nr. of sqme-vetos: ", counter%veto_sqme
    if (counter%n_sqme > 0) &
       write (u, "(A,F4.2)") "Fraction of vetoed points: ", &
                            one*counter%veto_sqme / counter%n_sqme
    call write_separator (u)
    write (u, "(A,I0)") "Nr. of upper-bound failures: ", &
                        counter%n_veto_fail
  end subroutine veto_counter_write

  subroutine sudakov_write (sudakov, unit)
    class(sudakov_t), intent(in) :: sudakov
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit);  if (u < 0)  return
    write (u, "(3X,A," // FMT_19 // ")")  "xi2_max = ", sudakov%xi2_max
    write (u, "(3X,A," // FMT_19 // ")")  "norm_max = ", sudakov%norm_max
    write (u, "(3X,A," // FMT_19 // ")")  &
         "current_pt2_max = ", sudakov%current_pt2_max
    write (u, "(3X,A," // FMT_19 // ")")  "last_log = ", sudakov%last_log
  end subroutine sudakov_write

  subroutine sudakov_init (sudakov, &
         process_deps, event_deps, powheg_settings, qcd, rng)
    class(sudakov_t), intent(out) :: sudakov
    type(process_deps_t), target, intent(in) :: process_deps
    type(event_deps_t), target, intent(in) :: event_deps
    type(powheg_settings_t), target, intent(in) :: powheg_settings
    type(qcd_t), target, intent(in) :: qcd
    class(rng_t), target, intent(in) :: rng
    sudakov%process_deps => process_deps
    sudakov%event_deps => event_deps
    sudakov%powheg_settings => powheg_settings
    sudakov%qcd => qcd
    sudakov%rng => rng
  end subroutine sudakov_init

  pure subroutine sudakov_set_normalization (sudakov, norm_max)
    class(sudakov_t), intent(inout) :: sudakov
    real(default), intent(in) :: norm_max
    sudakov%norm_max = norm_max
  end subroutine sudakov_set_normalization

  pure subroutine sudakov_update (sudakov, xi2_max)
    class(sudakov_t), intent(inout) :: sudakov
    real(default), intent(in) :: xi2_max
    sudakov%xi2_max = xi2_max
  end subroutine sudakov_update

  function sudakov_alpha_s (sudakov, kt2, use_correct) result (a)
    real(default) :: a
    class(sudakov_t), intent(in) :: sudakov
    real(default), intent(in) :: kT2
    logical, intent(in), optional :: use_correct
    logical :: yorn
    yorn = .false.; if (present (use_correct)) yorn = use_correct
    if (yorn) then
       a = get_alpha (sudakov%qcd, kT2)
    else
       a = sudakov%alpha_s_rad (kT2)
    end if
  end function sudakov_alpha_s

  function sudakov_generate_pt2 (sudakov) result (pt2)
    real(default) :: pt2
    class(sudakov_t), intent(inout) :: sudakov
    logical :: success
    success = .false.
    call sudakov%rng%generate (sudakov%random)
    pt2 = solve_interval (sudakov, &
         sudakov%powheg_settings%pt2_min, &
         sudakov%current_pt2_max, success, &
         0.001_default)
    sudakov%last_log = sudakov%last_log + &
                       sudakov%norm_max * sudakov%log_integrated_ubf (pt2)
    if (.not. success) then
       pt2 = sudakov%powheg_settings%pt2_min
    end if
  end function sudakov_generate_pt2

  subroutine sudakov_check_solution_interval (sudakov)
    class(sudakov_t), intent(inout) :: sudakov
    real(default) :: r
    real(default), parameter :: dr = 0.05
    real(default) :: pt2
    logical :: success
    r = 0._default
    do
       r = r+dr
       sudakov%random = r
       pt2  = solve_interval (sudakov, &
         sudakov%powheg_settings%pt2_min, &
         sudakov%current_pt2_max, success, &
         0.001_default)
      if (success) then
         print *, 'r: ', r, ' zero found'
      else
         print *, 'r: ', r, 'no zero found'
      end if
      if (r >= 1._default) exit
    end do
  end subroutine sudakov_check_solution_interval

  subroutine sudakov_generate_emission (sudakov, r)
    class(sudakov_t), intent(inout) :: sudakov
    type(radiation_t), intent(inout) :: r
    logical :: accepted
    sudakov%current_pt2_max = r%pt2
    sudakov%last_log = sudakov%norm_max * &
         sudakov%log_integrated_ubf (sudakov%current_pt2_max)
    if (DEBUG_EVENTS) then
       print *, 'sudakov_generate_emission'
       print *, '  sqrt (sudakov%current_pt2_max) =    ', &
            sqrt (sudakov%current_pt2_max)
       print *, '  sudakov%last_log =    ', sudakov%last_log
    end if
    LOOP_UNTIL_ACCEPTED: do
       r%valid = .false.
       r%pt2 = sudakov%generate_pt2 ()
       if (DEBUG_EVENTS) then
          print *, '  sqrt (r%pt2) =    ', sqrt (r%pt2)
          print *, '  sudakov%last_log =    ', sudakov%last_log
       end if
       if (r%pt2 <= sudakov%powheg_settings%pt2_min) then
          exit
       end if
       accepted = sudakov%reweight_ubf (r%pt2)
       call sudakov%veto_counter%record_ubf (.not. accepted)
       if (.not. accepted) then
          sudakov%current_pt2_max = r%pt2
          cycle
       end if
       accepted = sudakov%reweight_alpha_s (r%pt2)
       call sudakov%veto_counter%record_alpha_s (.not. accepted)
       if (.not. accepted) then
          sudakov%current_pt2_max = r%pt2
          cycle
       end if
       call sudakov%generate_xi_and_y_and_phi (r)
       r%valid = .true.
       exit
    end do LOOP_UNTIL_ACCEPTED
  end subroutine sudakov_generate_emission

  pure function sudakov_evaluate (solver_f, x) result (f)
    complex(default) :: f
    class(sudakov_t), intent(in) :: solver_f
    real(default), intent(in) :: x
    f = log (solver_f%random) + solver_f%norm_max * solver_f%log_integrated_ubf (x) &
         - solver_f%last_log
  end function sudakov_evaluate

  pure function sudakov_simple_fsr_upper_bound_func (sudakov, xi, y, alpha_s) result (u)
    real(default) :: u
    class(sudakov_simple_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: xi, y, alpha_s
    u = alpha_s / (xi * (1 - y))
  end function sudakov_simple_fsr_upper_bound_func

  pure function sudakov_simple_fsr_kt2 (sudakov, xi, y) result (kt2)
    real(default) :: kt2
    class(sudakov_simple_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: xi, y
    kt2 = sudakov%event_deps%cms_energy2 / 2 * xi**2 * (1 - y)
  end function sudakov_simple_fsr_kt2

  pure function sudakov_simple_fsr_log_integrated_ubf (sudakov, pt2) result (y)
    real(default) :: y
    class(sudakov_simple_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: pt2
    real(default) :: xm2s, xm2sl, pt2l
    logical :: within_boundaries
    within_boundaries = pt2 / sudakov%event_deps%cms_energy2 <= sudakov%xi2_max &
         .and. pt2 >= sudakov%powheg_settings%pt2_min
    if (within_boundaries) then
       xm2s = sudakov%xi2_max * sudakov%event_deps%cms_energy2
       xm2sl = xm2s / sudakov%process_deps%lambda2_gen
       pt2l = pt2 / sudakov%process_deps%lambda2_gen
       y = pi / b0rad * (log (xm2sl) * &
            log (log (xm2sl) / log (pt2l)) - &
            log (xm2s / pt2))
    else
       y = 0
    end if
  end function sudakov_simple_fsr_log_integrated_ubf

  function sudakov_simple_fsr_reweight_ubf (sudakov, pt2) result (accepted)
    logical :: accepted
    class(sudakov_simple_fsr_t), intent(inout) :: sudakov
    real(default), intent(in) :: pt2
    accepted = .true.
  end function sudakov_simple_fsr_reweight_ubf

  subroutine sudakov_simple_fsr_generate_xi_and_y_and_phi (sudakov, r)
    class(sudakov_simple_fsr_t), intent(inout) :: sudakov
    type(radiation_t), intent(inout) :: r
    real(default) :: s
    s = sudakov%event_deps%cms_energy2
    call sudakov%generate_xi (r)
    r%y = one - (two * r%pt2) / (s * r%xi**2)
    call sudakov%rng%generate (sudakov%random)
    r%phi = sudakov%random * twopi
    if (ENSURE) then
       call assert_equal (OUTPUT_UNIT, r%pt2, &
            s / two * r%xi**2 * (one - r%y), &
            "sudakov_generate_xi_and_y_and_phi: pt2 inconsistency")
    end if
  contains
  end subroutine sudakov_simple_fsr_generate_xi_and_y_and_phi

  subroutine sudakov_simple_fsr_generate_xi (sudakov, r)
    class(sudakov_simple_fsr_t), intent(inout) :: sudakov
    type(radiation_t), intent(inout) :: r
    real(default) :: s, xi2_max
    s = sudakov%event_deps%cms_energy2
    xi2_max = sudakov%xi2_max
    call sudakov%rng%generate (sudakov%random)
    r%xi = exp (((one - sudakov%random) * log (r%pt2 / s) + &
         sudakov%random * log (xi2_max)) / two)
  end subroutine sudakov_simple_fsr_generate_xi

  pure function sudakov_eeqq_fsr_kt2 (sudakov, xi, y) result (kt2)
    real(default) :: kt2
    class(sudakov_eeqq_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: xi, y
    kt2 = sudakov%event_deps%cms_energy2 / 2 * xi**2 * (1 - y**2) / 2
  end function sudakov_eeqq_fsr_kt2

  pure function sudakov_eeqq_fsr_upper_bound_func (sudakov, xi, y, alpha_s) result (u)
    real(default) :: u
    class(sudakov_eeqq_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: xi, y, alpha_s
    u = alpha_s / (xi * (1 - y**2))
  end function sudakov_eeqq_fsr_upper_bound_func

  pure function sudakov_eeqq_fsr_log_integrated_ubf (sudakov, pt2) result (y)
    real(default) :: y
    class(sudakov_eeqq_fsr_t), intent(in) :: sudakov
    real(default), intent(in) :: pt2
    logical :: within_boundaries
    within_boundaries = pt2 / sudakov%event_deps%cms_energy2 <= sudakov%xi2_max &
         .and. pt2 >= sudakov%powheg_settings%pt2_min
    if (within_boundaries) then
       !xm2s = sudakov%xi2_max * sudakov%event_deps%cms_energy2
       !xm2sl = xm2s / sudakov%process_deps%lambda2_gen
       !pt2l = pt2 / sudakov%process_deps%lambda2_gen
       !y = pi / b0rad * (log (xm2sl) * &
            !log (log (xm2sl) / log (pt2l)) - &
            !log (xm2s / pt2))
    else
       y = 0
    end if
  end function sudakov_eeqq_fsr_log_integrated_ubf

  function sudakov_eeqq_fsr_reweight_ubf (sudakov, pt2) result (accepted)
    logical :: accepted
    class(sudakov_eeqq_fsr_t), intent(inout) :: sudakov
    real(default), intent(in) :: pt2
    accepted = .false.
  end function sudakov_eeqq_fsr_reweight_ubf

  subroutine sudakov_eeqq_fsr_generate_xi_and_y_and_phi (sudakov, r)
    class(sudakov_eeqq_fsr_t), intent(inout) :: sudakov
    type(radiation_t), intent(inout) :: r
    real(default) :: s
    s = sudakov%event_deps%cms_energy2
    !r%xi = sudakov%generate_xi (r)
    !r%y = one - (two * r%pt2) / (s * r%xi**2)
    call sudakov%rng%generate (sudakov%random)
    r%phi = sudakov%random * twopi
    if (ENSURE) then
       call assert_equal (OUTPUT_UNIT, r%pt2, &
            s / two * r%xi**2 * (one - r%y), &
            "sudakov_generate_xi_and_y_and_phi: pt2 inconsistency")
    end if
  contains
  end subroutine sudakov_eeqq_fsr_generate_xi_and_y_and_phi

  subroutine powheg_display_grid_startup_message (powheg)
    class(powheg_t), intent(in) :: powheg
    real(default) :: points_per_cell
    write (msg_buffer, "(A,A,A)") "Generating grid for process '", &
                               char (powheg%process_name), "'"
    call msg_message
    associate (settings => powheg%settings)
       write (msg_buffer, "(A,I10)") "Number of xi-points: ", &
                                      settings%size_grid_xi
       call msg_message ()
       write (msg_buffer, "(A,I10)") "Number of y-points: ", &
                                      settings%size_grid_y
       call msg_message ()
       write (msg_buffer, "(A,I10,A)") "Using ", settings%n_init , &
                                       " sampling points"
       call msg_message ()
       points_per_cell =  settings%n_init*one / &
                          (settings%size_grid_xi * settings%size_grid_y)
       write (msg_buffer, "(A,F10.2,A)") "Average: ", points_per_cell, &
                                        " points per cell"
       call msg_message ()
       write (msg_buffer, "(A)") "Progress: "
       call msg_message ()
    end associate
  end subroutine powheg_display_grid_startup_message

  subroutine powheg_write (powheg, unit)
    class(powheg_t), intent(in) :: powheg
    integer, intent(in), optional :: unit
    integer :: u, alr
    u = given_output_unit (unit);  if (u < 0)  return
    call write_separator (u, 2)
    write (u, "(1X,A)") "POWHEG Emission Generator"
    write (u, "(1X,A)") "Process name: " // char (powheg%process_name)
    call powheg%rng%write (u)
    call powheg%settings%write (u)
    call powheg%event_deps%write (u)
    call powheg%process_deps%write (u)
    call powheg%qcd%write (u)
    do alr = 1, size(powheg%sudakov)
       call write_separator (u)
       write (u, "(1X,A,I12,A)") "sudakov (alr = ", alr, ")"
       call powheg%sudakov(alr)%s%write (u)
    end do
    call write_separator (u, 2)
  end subroutine powheg_write

  subroutine powheg_write_statistics (powheg)
    class(powheg_t), intent(in) :: powheg
    integer :: u
    integer :: alr
    type(string_t) :: filename
    u = free_unit ()
    filename = powheg%process_name // "_veto.log"
    open (file=char(filename), unit=u, action='write')
    write (u, '(A)') "Summary of Powheg veto procedure"
    do alr = 1, powheg%process_deps%n_alr
       write(u,'(A,I0)') 'alr: ', alr
       associate (veto_counter => powheg%sudakov(alr)%s%veto_counter)
          call veto_counter%write (u)
       end associate
       call write_separator (u)
    end do
    write (u,'(A,I0)') "Total number of events which radiate a gluon: ", &
                       powheg%n_emissions
  end subroutine powheg_write_statistics

  subroutine powheg_connect (powheg, process_instance, testing)
    class(powheg_t), intent(inout), target :: powheg
    type(process_instance_t), intent(in), target :: process_instance
    type(powheg_testing_t), intent(in), optional :: testing
    integer :: n_in, n_out_born, n_out_real
    if (.not. present (testing)) then
       powheg%process_instance => process_instance
       associate (nlo_controller => powheg%process_instance%nlo_controller)
          powheg%process_deps%n_alr = nlo_controller%get_n_alr ()
          n_in = nlo_controller%particle_data%n_in
          n_out_born = nlo_controller%particle_data%n_out_born
          n_out_real = nlo_controller%particle_data%n_out_real
          associate (generator => powheg%phs_fks_generator)
             call generator%setup_real_kinematics (nlo_controller%real_kinematics)
             call generator%set_emitters (nlo_controller%reg_data%emitters)
             call generator%setup_masses (n_in + n_out_born)
          end associate
       end associate
    else
       powheg%testing = testing
       powheg%process_deps%n_alr = testing%n_alr
       n_in = testing%n_in
       n_out_born = testing%n_out_born
       n_out_real = testing%n_out_real
    end if

    allocate (powheg%event_deps%p_born (n_in + n_out_born))
    allocate (powheg%event_deps%p_real (n_in + n_out_real))
    call powheg%grid%init ([powheg%settings%size_grid_xi, &
                            powheg%settings%size_grid_y, &
                            powheg%process_deps%n_alr])
    call powheg%setup_sudakovs ()
  end subroutine powheg_connect

  subroutine powheg_setup_grids (powheg)
    class(powheg_t), intent(inout) :: powheg
    if (powheg%requires_new_grids ()) then
       call powheg%fill_grids ()
       call powheg%save_grids ()
    else
       call powheg%load_grids ()
    end if
    call powheg%grid%compute_and_write_mean_and_max ()
    call powheg%set_normalizations ()
  end subroutine powheg_setup_grids

  subroutine powheg_setup_sudakovs (powheg)
    class(powheg_t), intent(inout), target :: powheg
    integer :: alr
    logical :: is_fsr
    allocate (powheg%sudakov (powheg%process_deps%n_alr))
    is_fsr = .true.
    do alr = 1, powheg%process_deps%n_alr
       if (is_fsr) then
          select case (powheg%settings%upper_bound_func)
          case (UBF_SIMPLE)
             allocate (sudakov_simple_fsr_t :: powheg%sudakov(alr)%s)
          case (UBF_EEQQ)
             allocate (sudakov_eeqq_fsr_t :: powheg%sudakov(alr)%s)
          case default
             call msg_fatal ("powheg_setup_sudakovs: Please choose upper bounding function!")
          end select
       else
          call msg_fatal ("powheg_setup_sudakovs: ISR not implemented yet!")
       end if

       call powheg%sudakov(alr)%s%init (powheg%process_deps, &
            powheg%event_deps, powheg%settings, powheg%qcd, powheg%rng)
    end do
  end subroutine powheg_setup_sudakovs

  subroutine powheg_init (powheg, settings, process_name)
    class(powheg_t), intent(out) :: powheg
    type(powheg_settings_t), intent(in) :: settings
    type(string_t), intent(in) :: process_name
    powheg%settings = settings
    powheg%process_name = process_name
  end subroutine powheg_init

  pure subroutine powheg_import_rng (powheg, rng)
    class(powheg_t), intent(inout) :: powheg
    class(rng_t), allocatable, intent(inout) :: rng
    call move_alloc (from = rng, to = powheg%rng)
  end subroutine powheg_import_rng

  subroutine powheg_update_momenta (powheg, p_born)
    class(powheg_t), intent(inout) :: powheg
    type(vector4_t), dimension(:), intent(in) :: p_born
    if (.not. powheg%testing%active) then
       call powheg%event_deps%update &
            (powheg%process_instance%sqme_collector%get_sqme_born(), p_born)
    else
       call powheg%event_deps%update &
            (powheg%testing%sqme_born, p_born)
    end if
  end subroutine powheg_update_momenta

  subroutine powheg_update_particle_set (powheg, particle_set)
    class(powheg_t), intent(inout) :: powheg
    type(particle_set_t), intent(in) :: particle_set
    call powheg%update_momenta (particle_set%get_momenta())
  end subroutine powheg_update_particle_set

  function powheg_reweight_matrix_elements (powheg, r) result (accepted)
    logical :: accepted
    class(powheg_t), intent(inout) :: powheg
    type(radiation_t), intent(in) :: r
    integer :: emitter
    real(default) :: sqme_real_x_jacobian, sqme_born
    real(default) :: norm, ubf, ubound, random, weight
    real(default) :: alpha_s
    call powheg%rng%generate (random)
    emitter = powheg%process_instance%nlo_controller%get_emitter (r%alr)
    powheg%event_deps%p_real = &
         powheg%phs_fks_generator%generate_fsr_from_xi_and_y (r%xi, &
                          r%y, r%phi, emitter, powheg%event_deps%p_born)
    norm = powheg%norm_from_xi_and_y (r)
    associate (s => powheg%sudakov(r%alr)%s)
       alpha_s = s%alpha_s (s%kt2 (r%xi, r%y), use_correct=.true.)
       ubf = s%upper_bound_func (r%xi, r%y, alpha_s)
       sqme_real_x_jacobian = powheg%compute_sqme_real (r%alr, alpha_s)
       sqme_born = powheg%event_deps%sqme_born
       ubound = sqme_born * ubf * norm
       weight = sqme_real_x_jacobian / ubound
       if (weight > 1) call s%veto_counter%record_fail()
       if (ENSURE) then
          if (weight < 0) call msg_warning ("R/B < 0!")
       end if
       accepted = random < weight
    end associate
    if (DEBUG_EVENTS) then
       print *, 'reweight_matrix_elements'
       print *, '  r%alr =    ',   r%alr
       print *, '  r%xi =    ', r%xi
       print *, '  r%y =    ', r%y
       print *, '  emitter =    ', emitter
       print *, '  random =    ', random
       print *, '  sqme_real_x_jacobian =    ', sqme_real_x_jacobian
       print *, '  sqme_born =    ', sqme_born
       print *, '  ubf =    ', ubf
       print *, '  norm =    ',   norm
       print *, '  ubound =    ', ubound
       print *, '  matrix element  accepted =    ', accepted
    end if
  end function powheg_reweight_matrix_elements

  function powheg_compute_sqme_real (powheg, alr, alpha_s) result (sqme)
    class(powheg_t), intent(inout) :: powheg
    integer, intent(in) :: alr
    real(default), intent(in) :: alpha_s
    integer :: emitter
    real(default) :: sqme
    if (.not. powheg%testing%active) then
       associate (instance => powheg%process_instance)
          emitter = instance%nlo_controller%get_emitter (alr)
          call instance%compute_sqme_real_rad (emitter, &
               powheg%event_deps%p_born, powheg%event_deps%p_real, alpha_s)
          sqme = instance%sqme_collector%sqme_real_per_emitter (emitter)
       end associate
    else
       sqme = one
    end if
  end function powheg_compute_sqme_real

  subroutine powheg_set_scale (powheg, pT2)
    class(powheg_t), intent(inout) :: powheg
    real(default), intent(in) :: pT2
    call powheg%process_instance%set_fac_scale (sqrt(pT2))
  end subroutine powheg_set_scale

  subroutine powheg_fill_grids (powheg)
    class(powheg_t), intent(inout) :: powheg
    real(default), dimension(3) :: radiation_variables
    real(default) :: f_alr, xi, y, norm, real_me, ubf
    integer :: alr
    integer :: n, n_points
    real(default) :: alpha_s
    call powheg%display_grid_startup_message()
    n_points = powheg%settings%n_init
    UNTIL_ACCEPTED: do
       EVALUATE_GRID_POINTS: do n = 1, n_points
          call powheg%prepare_momenta_for_fill_grids (radiation_variables)
          do alr = 1, powheg%process_deps%n_alr
             call powheg%generate_xi_and_y_for_grids &
                  (radiation_variables, alr, xi, y)
             associate (s => powheg%sudakov(alr)%s)
                alpha_s = s%alpha_s (s%kt2(xi, y), use_correct=.true.)
                ubf = s%upper_bound_func (xi, y, alpha_s)
                real_me = powheg%compute_sqme_real (alr, alpha_s)
             end associate
             norm = real_me / (powheg%event_deps%sqme_born * ubf)
             f_alr = (one * alr) / powheg%process_deps%n_alr - tiny_07
             call powheg%grid%update_maxima &
                  ([radiation_variables(I_XI:I_Y), f_alr], norm)
             call msg_show_progress (n, n_points)
          end do
       end do EVALUATE_GRID_POINTS
       if (powheg%grid%is_non_zero_everywhere ()) then
          return
       else
          n_points = powheg%settings%n_init / 5
          write (msg_buffer, '(A,I12,A)') 'Number of point for POWHEG grid ' // &
               'initialization was not enough. Run continues with ', &
               n_points, ' additional points to fill empty segments.'
          call msg_warning ()
       end if
    end do UNTIL_ACCEPTED

  end subroutine powheg_fill_grids

  subroutine powheg_generate_xi_and_y_for_grids (powheg, &
                                       radiation_randoms, alr, xi, y)
    class(powheg_t), intent(inout) :: powheg
    integer, intent(in) :: alr
    real(default), dimension(:), intent(in) :: radiation_randoms
    real(default), intent(out) :: xi, y
    integer :: emitter
    if (.not. powheg%testing%active) then
       associate (nlo => powheg%process_instance%nlo_controller, &
                  fks => powheg%phs_fks_generator)
          emitter = nlo%get_emitter (alr)
          powheg%event_deps%p_real = fks%generate_fsr_from_x &
                 (radiation_randoms, emitter, powheg%event_deps%p_born)
          call fks%get_radiation_variables (emitter, xi, y)
       end associate
    else
       xi = radiation_randoms (I_XI)
       y = radiation_randoms (I_Y)
    end if
  end subroutine powheg_generate_xi_and_y_for_grids

  subroutine powheg_prepare_momenta_for_fill_grids (powheg, &
                                                      radiation_randoms)
    real(default), dimension(3), intent(out) :: radiation_randoms
    class(powheg_t), intent(inout) :: powheg
    if (.not. powheg%testing%active) then
       associate (nlo => powheg%process_instance%nlo_controller, &
                  fks => powheg%phs_fks_generator, &
                  process => powheg%process_instance%process)
          do
             call process%generate_weighted_event (powheg%process_instance, 1)
             call powheg%update (nlo%int_born%get_momenta ())
             call powheg%rng%generate (radiation_randoms)
             call fks%generate_radiation_variables &
                  (radiation_randoms, powheg%event_deps%p_born)
             if (powheg%above_pt2_min ()) exit
          end do
       end associate
    else
       call powheg%rng%generate (radiation_randoms)
    end if
  end subroutine powheg_prepare_momenta_for_fill_grids

  pure function powheg_above_pt2_min (powheg) result (yorn)
    logical :: yorn
    class(powheg_t), intent(in) :: powheg
    integer :: alr, emitter
    real(default) :: xi, y
    yorn = .true.
    associate (nlo => powheg%process_instance%nlo_controller, &
                      fks => powheg%phs_fks_generator)
       do alr = 1, powheg%process_deps%n_alr
          emitter = nlo%get_emitter (alr)
          call fks%get_radiation_variables (emitter, xi, y)
          yorn = powheg%sudakov(alr)%s%kt2 (xi, y) >= powheg%settings%pt2_min
          if (.not. yorn) exit
       end do
    end associate
  end function powheg_above_pt2_min

  subroutine powheg_set_normalizations (powheg)
    class(powheg_t), intent(inout) :: powheg
    integer :: alr
    real(default) :: norm_max
    do alr = 1, powheg%process_deps%n_alr
       norm_max = powheg%grid%get_maximum_in_3d (alr)
       call powheg%sudakov(alr)%s%set_normalization (norm_max)
    end do
  end subroutine powheg_set_normalizations

  subroutine powheg_save_grids (powheg)
    class(powheg_t), intent(inout) :: powheg
    type(string_t) :: filename, n_points
    n_points = str (powheg%settings%n_init)
    filename = powheg%process_name // "_" // n_points // "_powheg_grids.dat"
    call powheg%grid%save_to_file (char (filename))
  end subroutine powheg_save_grids

  subroutine powheg_load_grids (powheg)
    class(powheg_t), intent(inout) :: powheg
    type(string_t) :: filename, n_points
    n_points = str (powheg%settings%n_init)
    filename = powheg%process_name // "_" // n_points // "_powheg_grids.dat"
    call powheg%grid%load_from_file (char (filename))
  end subroutine powheg_load_grids

  function powheg_requires_new_grids (powheg) result (requires)
    logical :: requires
    class(powheg_t), intent(in) :: powheg
    type(string_t) :: filename, n_points
    n_points = str (powheg%settings%n_init)
    filename = powheg%process_name // "_" // n_points // "_powheg_grids.dat"
    requires = .not. os_file_exist (filename) .or. powheg%settings%rebuild_grids
  end function powheg_requires_new_grids

  subroutine powheg_generate_emission (powheg, particle_set, pt2_generated)
    class(powheg_t), intent(inout) :: powheg
    type(particle_set_t), intent(inout), optional :: particle_set
    real(default), intent(out), optional :: pt2_generated
    type(radiation_t) :: r, r_max
    real(default) :: xi2_max
    integer :: alr
    logical :: accepted
    type(vector4_t), dimension(:), allocatable :: p_real_max
    r_max%pt2 = zero
    r_max%alr = 0
    associate (nlo => powheg%process_instance%nlo_controller)
      allocate (p_real_max (nlo%get_n_particles_real ()))
      do alr = 1, powheg%process_deps%n_alr
         associate (sudakov => powheg%sudakov (alr)%s)
           xi2_max = nlo%get_xi_max (alr)**2
           call sudakov%update (xi2_max)
           r%alr = alr
           r%pt2 = sudakov%xi2_max * powheg%event_deps%cms_energy2
           do
              call sudakov%generate_emission (r)
              if (r%valid) then
                 accepted = powheg%reweight_norm (r)
                 call sudakov%veto_counter%record_norm (.not. accepted)
                 if (.not. accepted) cycle
                 accepted = powheg%reweight_matrix_elements (r)
                 call sudakov%veto_counter%record_sqme (.not. accepted)
                 if (.not. accepted) cycle
              end if
              exit
           end do
           if (r%pt2 > r_max%pt2 .and. r%valid) then
              r_max = r
              p_real_max = powheg%event_deps%p_real
           end if
         end associate
      end do
      if (r_max%pt2 > powheg%settings%pt2_min) then
         powheg%n_emissions = powheg%n_emissions + 1
         call powheg%set_scale (r_max%pt2)
         if (present (particle_set)) &
              call powheg%build_particle_set (particle_set, &
              powheg%event_deps%p_born, &
              p_real_max, nlo%get_emitter (r_max%alr))
         if (present (pt2_generated)) pt2_generated = r_max%pt2
      else
         call powheg%set_scale (powheg%settings%pt2_min)
         if (present (pt2_generated)) pt2_generated = powheg%settings%pt2_min
      end if
    end associate
  end subroutine powheg_generate_emission

  subroutine powheg_build_particle_set &
       (powheg, particle_set, p_born, p_real, emitter)
    class(powheg_t), intent(inout) :: powheg
    type(particle_set_t), intent(inout) :: particle_set
    type(vector4_t), dimension(:), intent(in) :: p_born, p_real
    integer, intent(in) :: emitter
    type(particle_set_t) :: new_particle_set
    type(particle_t) :: new_particle
    integer :: i, n_particles
    integer, dimension(:), allocatable :: flv_state_real
    integer :: pdg_index_emitter, pdg_index_radiation
    integer :: n_in, n_vir, n_out, n_tot
    integer, dimension(:), allocatable :: parents, children
    type(flavor_t) :: new_flv

    n_particles = size (particle_set%prt)
    if (n_particles+1 /= size (p_real)) call msg_fatal &
                       ("Number of particles does not equal number of momenta")

    new_particle_set%n_beam = particle_set%n_beam
    new_particle_set%n_in = particle_set%n_in
    new_particle_set%n_vir = particle_set%n_vir+2
    new_particle_set%n_out = particle_set%n_out+1
    new_particle_set%n_tot = particle_set%n_tot+3
    new_particle_set%correlated_state = particle_set%correlated_state
    allocate (new_particle_set%prt (new_particle_set%n_tot))
    n_in = new_particle_set%n_in; n_vir = new_particle_set%n_vir
    n_out = new_particle_set%n_out; n_tot = new_particle_set%n_tot
    do i = 1, n_in
       new_particle_set%prt(i) = particle_set%prt(i)
       call new_particle_set%prt(i)%set_momentum (p_real(i))
       call new_particle_set%prt(i)%reset_status (PRT_INCOMING)
    end do
    do i = n_in+1, n_in+n_vir
       new_particle_set%prt(i) = particle_set%prt(i)
       call new_particle_set%prt(i)%set_momentum (p_born(i))
       call new_particle_set%prt(i)%reset_status (PRT_VIRTUAL)
    end do
    !!! Parents correct, care for children
    allocate (children (n_out))
    do i = 1, n_out
       children(i) = n_in+n_vir+i
    end do
    do i = n_in+1, n_in+n_vir
       call new_particle_set%prt(i)%set_children (children)
    end do
    do i = n_in+n_vir+1, n_tot-1
       new_particle_set%prt(i) = particle_set%prt(i-n_vir)
       call new_particle_set%prt(i)%set_momentum (p_real(i-n_vir))
       call new_particle_set%prt(i)%reset_status (PRT_OUTGOING)
    end do
    call new_particle%reset_status (PRT_OUTGOING)
    call new_particle%set_momentum (p_real (n_tot-n_vir))
    !!! Helicity and polarization handling is missing at this point
    associate (nlo_controller => powheg%process_instance%nlo_controller)
       flv_state_real = nlo_controller%get_flv_state_real (1)
    end associate
    pdg_index_emitter = flv_state_real (emitter)
    pdg_index_radiation = flv_state_real (n_tot-n_vir)
    call new_flv%init (pdg_index_radiation, &
                      powheg%process_instance%process%get_model_ptr ())
    call reassign_colors (new_particle, new_particle_set%prt(n_vir+emitter), &
                          pdg_index_radiation, pdg_index_emitter, &
                          powheg%rng)
    !!! Also, no helicities or polarizations yet
    call new_particle%set_flavor (new_flv)
    new_particle_set%prt(n_tot) = new_particle
    !!! Set proper parents for outgoing particles
    allocate (parents (n_vir))
    do i = 1, n_vir
       parents(i) = n_vir+i
    end do
    do i = n_in+n_vir+1, n_tot
       call new_particle_set%prt(i)%set_parents (parents)
    end do
    !!! Overwrite old particle set
     particle_set = new_particle_set
  end subroutine powheg_build_particle_set

  subroutine reassign_colors (prt_radiated, prt_emitter, i_rad, i_em, rng)
    type(particle_t), intent(inout) :: prt_radiated, prt_emitter
    integer, intent(in) :: i_rad, i_em
    class(rng_t), intent(inout), allocatable :: rng
    type(color_t) :: col_rad, col_em
    if (is_quark (abs (i_em)) .and. is_gluon (i_rad)) then
       call reassign_colors_qg (prt_emitter, col_rad, col_em)
    else if (is_gluon (i_em) .and. is_gluon (i_rad)) then
       call reassign_colors_gg (prt_emitter, rng, col_rad, col_em)
    else if (is_gluon (i_em) .and. is_quark (abs (i_rad))) then
       call reassign_colors_qq (prt_emitter, i_em, col_rad, col_em)
    else
       call msg_fatal ("Invalid splitting")
    end if
    call prt_emitter%set_color (col_em)
    call prt_radiated%set_color (col_rad)
  end subroutine reassign_colors

  subroutine reassign_colors_qg (prt_emitter, col_rad, col_em)
    type(particle_t), intent(in) :: prt_emitter
    type(color_t), intent(out) :: col_rad, col_em
    integer, dimension(2) :: color_rad, color_em
    integer :: i1, i2
    integer :: new_color_index
    logical :: is_anti_quark

    color_em = prt_emitter%get_color ()
    i1 = 1; i2 = 2
    is_anti_quark = color_em(2) /= 0
    if (is_anti_quark) then
       i1 = 2; i2 = 1
    end if
    new_color_index = color_em(i1)+1
    color_rad(i1) = color_em(i1)
    color_rad(i2) = new_color_index
    color_em(i1) = new_color_index
    call col_em%init_col_acl (color_em(1), color_em(2))
    call col_rad%init_col_acl (color_rad(1), color_rad(2))
  end subroutine reassign_colors_qg

  subroutine reassign_colors_gg (prt_emitter, rng, col_rad, col_em)
    !!! NOT TESTED YET
    type(particle_t), intent(in) :: prt_emitter
    class(rng_t), intent(inout), allocatable :: rng
    type(color_t), intent(out) :: col_rad, col_em
    real(default) :: random
    integer, dimension(2) :: color_rad, color_em
    integer :: i1, i2
    integer :: new_color_index
    call rng%generate (random)
    color_em = prt_emitter%get_color ()
    new_color_index = maxval (abs (color_em))
    i1 = 1; i2 = 2
    if (random < 0.5) then
       i1 = 2; i2 = 1
    end if
    color_rad(i1) = new_color_index
    color_rad(i2) = color_em(i2)
    color_em(i2) = new_color_index
    call col_em%init_col_acl (color_em(1), color_em(2))
    call col_rad%init_col_acl (color_rad(1), color_rad(2))
  end subroutine reassign_colors_gg

  subroutine reassign_colors_qq (prt_emitter, pdg_emitter, col_rad, col_em)
    !!! NOT TESTED YET
    type(particle_t), intent(in) :: prt_emitter
    integer, intent(in) :: pdg_emitter
    type(color_t), intent(out) :: col_rad, col_em
    integer, dimension(2) :: color_rad, color_em
    integer :: i1, i2
    logical :: is_anti_quark

    color_em = prt_emitter%get_color ()
    i1 = 1; i2 = 2
    is_anti_quark = pdg_emitter < 0
    if (is_anti_quark) then
       i1 = 2; i1 = 1
    end if
    color_em(i2) = 0
    color_rad(i1) = 0
    color_rad(i2) = color_em(i1)
    call col_em%init_col_acl (color_em(1), color_em(2))
    call col_rad%init_col_acl (color_rad(1), color_rad(2))
  end subroutine reassign_colors_qq

  function powheg_reweight_norm (powheg, r) result (accepted)
    logical :: accepted
    class(powheg_t), intent(inout) :: powheg
    type(radiation_t), intent(in) :: r
    real(default) :: random, norm_max, norm_true
    call powheg%rng%generate (random)
    norm_true = powheg%norm_from_xi_and_y (r)
    norm_max = powheg%sudakov(r%alr)%s%norm_max
    accepted = random < norm_true / norm_max
    if (DEBUG_EVENTS) then
       print *, 'reweight_norm'
       print *, '  r%alr =    ', r%alr
       print *, '  random =    ', random
       print *, '  norm_true =    ', norm_true
       print *, '  norm_max =    ', norm_max
       print *, '  norm accepted =    ', accepted
    end if
    if (ENSURE) then
       if (.not. (zero < r%xi .and. &
                  r%xi < sqrt(powheg%sudakov(r%alr)%s%xi2_max))) then
          call msg_bug ("powheg_reweight_norm: xi is out of bounds")
       end if
       if (norm_true > norm_max) then
          call msg_bug ("powheg_reweight_norm: norm shouldnt be larger than norm_max")
       end if
    end if
  end function powheg_reweight_norm

  function powheg_norm_from_xi_and_y (powheg, r) result (norm_true)
    real(default) :: norm_true
    class(powheg_t), intent(inout) :: powheg
    type(radiation_t), intent(in) :: r
    real(default) :: f_alr
    real(default), dimension(2) :: rands
    f_alr = (one*r%alr) / powheg%process_deps%n_alr - tiny_07
    rands(I_XI) = r%xi / sqrt (powheg%sudakov(r%alr)%s%xi2_max)
    rands(I_Y) = (one - r%y) / two
    norm_true = powheg%grid%get_value ([rands, f_alr])
  end function powheg_norm_from_xi_and_y

  subroutine powheg_compute_lambda2_gen (powheg)
    class(powheg_t), intent(inout) :: powheg
    real(default) :: alpha_s, scale_to_relate2
    scale_to_relate2 = powheg%settings%pt2_min
    alpha_s = get_alpha (powheg%qcd, scale_to_relate2)
    powheg%process_deps%lambda2_gen = exp (- one / (b0rad * alpha_s)) * &
         scale_to_relate2
  end subroutine powheg_compute_lambda2_gen

  function get_alpha (qcd, scale2) result (alpha_s)
    real(default) :: alpha_s
    class(qcd_t), intent(in) :: qcd
    real(default), intent(in) :: scale2
    integer :: nf, order
    ! TODO: (bcn 2015-01-30) implement variable flavor alpha_s
    alpha_s = qcd%alpha%get (sqrt(scale2))
    select type (alpha => qcd%alpha)
    type is (alpha_qcd_from_scale_t)
       nf = alpha%nf
       order = alpha%order
    type is (alpha_qcd_from_lambda_t)
       nf = alpha%nf
       order = alpha%order
    class default
       call msg_warning ("get_alpha: QCD type is not running!" // &
            "Assuming 5-flavors and LO (1-loop) running!")
       nf = 5
       order = 0
    end select
    if (order > 0) alpha_s = improve_nll_accuracy (alpha_s, nf)
  end function get_alpha

  pure function improve_nll_accuracy (alpha_s, n_flavors) result (alpha_s_imp)
    real(default) :: alpha_s_imp
    real(default), intent(in) :: alpha_s
    integer, intent(in) :: n_flavors
      alpha_s_imp = alpha_s * (one + alpha_s / (two*pi) * &
           ((67.0_default/18 - pi**2/6) * CA - five/9 * n_flavors))
  end function improve_nll_accuracy

  elemental function sudakov_alpha_s_rad (sudakov, scale2) result (y)
    real(default) :: y
    class(sudakov_t), intent(in) :: sudakov
    real(default), intent(in) :: scale2
    y = one / (b0rad * log (scale2 / sudakov%process_deps%lambda2_gen))
  end function sudakov_alpha_s_rad

  function sudakov_reweight_alpha_s (sudakov, pt2) result (accepted)
    logical :: accepted
    class(sudakov_t), intent(inout) :: sudakov
    real(default), intent(in) :: pt2
    real(default) :: alpha_s_true, alpha_s_rad
    logical :: alpha_s_equal
    alpha_s_true = get_alpha (sudakov%qcd, pt2)
    alpha_s_rad = sudakov%alpha_s_rad (pt2)
    call sudakov%rng%generate (sudakov%random)

    alpha_s_equal = nearly_equal (alpha_s_true, alpha_s_rad)
    accepted = alpha_s_equal .or. sudakov%random < alpha_s_true / alpha_s_rad
    if (DEBUG_EVENTS) then
       print *, 'reweight_alpha_s'
       print *, '  sudakov%random =    ', sudakov%random
       print *, '  alpha_s_true =    ', alpha_s_true
       print *, '  alpha_s_rad =    ', alpha_s_rad
       print *, '  alpha_s accepted =    ', accepted
    end if
    if (ENSURE) then
       if (alpha_s_rad < alpha_s_true .and. .not. alpha_s_equal) then
          print *, 'pt2 =    ', pt2
          print *, 'alpha_s_rad =    ', alpha_s_rad
          print *, 'alpha_s_true =    ', alpha_s_true
          print *, 'sudakov%process_deps%lambda2_gen =    ', sudakov%process_deps%lambda2_gen
          call msg_fatal ("sudakov_reweight_alpha_s: This should never happen. &
                           &Have you chosen a running alpha_s?")
       end if
    end if
  end function sudakov_reweight_alpha_s

  subroutine powheg_test (u, results)
    integer, intent(in) :: u
    type(test_results_t), intent(inout) :: results
    call test(powheg_1, "powheg_1", &
              "Initialization", u, results)
  end subroutine powheg_test

  subroutine powheg_1 (u)
    integer, intent(in) :: u
    type(powheg_t) :: powheg
    type(powheg_settings_t) :: powheg_settings
    type(powheg_testing_t) :: powheg_testing
    type(process_instance_t) :: process_instance
    type(particle_set_t) :: particle_set
    class(rng_factory_t), allocatable :: rng_factory
    class(rng_t), allocatable :: rng
    type(string_t) :: process_name
    type(vector4_t), dimension(4) :: born_momenta
    type(qcd_t), target :: qcd

    allocate (rng_test_factory_t :: rng_factory)
    call rng_factory%make (rng)
    allocate (alpha_qcd_from_lambda_t :: qcd%alpha)
    select type (alpha => qcd%alpha)
    type is (alpha_qcd_from_lambda_t)
       alpha%order = 2
    end select
    process_name = "test_powheg_1"
    powheg_settings%n_init = 1000
    powheg_settings%size_grid_xi = 2
    powheg_settings%size_grid_y = 2
    powheg_settings%pt2_min = one
    powheg_settings%lambda = LAMBDA_QCD_REF
    powheg_testing%n_alr = 3
    powheg_testing%n_in = 2
    powheg_testing%n_out_born = 2
    powheg_testing%n_out_real = 3
    powheg_testing%sqme_born = one
    powheg_testing%active = .true.
    born_momenta(1) = [50._default, zero, zero, 50._default]
    born_momenta(2) = [50._default, zero, zero, - 50._default]
    born_momenta(3) = [50._default, zero, zero, 50._default]
    born_momenta(4) = [50._default, zero, zero, - 50._default]
    particle_set%n_tot = 4
    particle_set%n_in = 2
    particle_set%n_out = 2
    call particle_set%set_momenta (born_momenta)

    write (u, "(A)")  "* Test output: powheg_1"
    write (u, "(A)")  "*   Purpose: Initialization"
    write (u, "(A)")

    call powheg%init (powheg_settings, process_name)
    powheg%qcd => qcd
    call powheg%import_rng (rng)
    call powheg%connect (process_instance, powheg_testing)
    call powheg%compute_lambda2_gen ()
    call powheg%update (particle_set)
    !!! Needs some more thought: if we just set R = 1, B = 1 the grid
    !!! setup will fail
    !!! call powheg%setup_grids ()
    !!! call powheg%generate_emission (particle_set)
    call powheg%write (u)

    write (u, "(A)")
    write (u, "(A)")  "* Test output end: powheg_1"
  end subroutine powheg_1

  subroutine powheg_test_sudakov (powheg)
    class(powheg_t), intent(inout) :: powheg
    integer :: n_calls1, n_calls2
    integer, parameter :: n_bins = 20
    real(default) :: sqme_real_x_jacobian, sqme_born
    type(vector4_t), dimension(:), allocatable :: p_born
    real(default), dimension(3) :: random
    real(default) :: xi, y, phi
    integer :: i_call, i_bin, alr, emitter
    real(default) :: alpha_s, kT2, weight
    real(default) :: pt2_min, s, random_jacobian
    real(default), dimension(n_bins) :: histo1, histo2, histo1sq
    real(default), dimension(n_bins) :: average, average_sq, error
    real(default), dimension(n_bins) :: &
         sudakov_0, sudakov_p, sudakov_m, rel_error
    integer :: u

    p_born = powheg%event_deps%p_born
    sqme_born = powheg%event_deps%sqme_born
    s = powheg%event_deps%cms_energy2
    pt2_min = powheg%settings%pt2_min
    n_calls1 = 50000; n_calls2 = 5000
    histo1 = zero; histo2 = zero; histo1sq = zero

    associate (nlo => powheg%process_instance%nlo_controller, &
               fks => powheg%phs_fks_generator)
       do i_call = 1, n_calls1
          do alr = 1, powheg%process_deps%n_alr
             call powheg%rng%generate (random)
             emitter = nlo%get_emitter (alr)
             powheg%event_deps%p_real = fks%generate_fsr_from_x (random, emitter, p_born)
             call fks%get_radiation_variables (emitter, xi, y, phi)
             kT2 = powheg%sudakov(alr)%s%kt2(xi, y)
             if (kT2 > pt2_min .and. xi < one - 1000*tiny_07) then
                alpha_s = get_alpha (powheg%qcd, kT2)
                sqme_real_x_jacobian = powheg%compute_sqme_real (alr, alpha_s)
                random_jacobian = nlo%real_kinematics%jac_rand (emitter)
                weight = sqme_real_x_jacobian * random_jacobian / sqme_born
                do i_bin = 1, n_bins
                   if (kT2 > binning(i_bin)) then
                      histo1(i_bin) = histo1(i_bin) + weight
                      histo1sq(i_bin) = histo1sq(i_bin) + weight**2
                   end if
                end do
             end if
             ! Do not cycle since there is a Heaviside in the exponent
          end do
          call msg_show_progress (i_call, n_calls1)
       end do
    end associate
    average = histo1 / n_calls1
    average_sq = histo1sq / n_calls1
    error = sqrt ((average_sq - average**2) / n_calls1)
    sudakov_0 = exp(-average)
    sudakov_p = exp(-(average + error))
    sudakov_m = exp(-(average - error))
    rel_error = (sudakov_0 - sudakov_p + sudakov_m - sudakov_0) / &
         (2 * sudakov_0) * 100

    do i_call = 1, n_calls2
       call powheg%generate_emission (pt2_generated = kT2)
       do i_bin = 1, n_bins
          if (kT2 > binning(i_bin)) then
              histo2(i_bin) = histo2(i_bin) + one
          end if
       end do
       call msg_show_progress (i_call, n_calls2)
    end do
    histo2 = histo2 / n_calls2
    histo2 = one - histo2

    u = free_unit ()
    open (file='test_sudakov_data', unit=u, action='write')
    print *, 'exp(-Integrated R/B)-distribution: '
    print *, 'pT2  sudakov_+  sudakov_0  sudakov_-  rel_err[%]: '
    do i_bin = 1, n_bins
       print *, binning (i_bin), &
            sudakov_p (i_bin), sudakov_0 (i_bin), sudakov_m (i_bin), &
            rel_error (i_bin)
       write (u, "(5(" // FMT_16 // ",2X))") binning (i_bin), &
            sudakov_p (i_bin), sudakov_0 (i_bin), sudakov_m (i_bin), &
            histo2 (i_bin)
    end do
    close (u)
    print *, '*******************************'
    print *, 'Noemission probability: '
    do i_bin = 1, n_bins
       print *, binning (i_bin), histo2 (i_bin)
    end do

  contains

    pure function binning (i) result (pt2)
      real(default) :: pt2
      integer, intent(in) :: i
      !pt2 = pt2_min + (s-pt2_min) * (i-1) / (n_bins-1)
      pt2 = pt2_min * exp (log (s / pt2_min) * (i-1) / (n_bins-1))
    end function
  end subroutine powheg_test_sudakov


end module powheg
