#ifndef INCU_LOG
#define INCU_LOG
#include <sstream>
#include <string>

namespace incu {

  class LogDevice {
  public:
    virtual void write(const std::string& msg) = 0;
  };

  enum LogGrade { l_debug, l_info, l_important, l_fatal };
  inline std::ostream& operator << (std::ostream& o, LogGrade g) {
    switch(g) {
    case l_debug: o << "DEBUG"; break;
    case l_info:  o << "INFO";  break;
    case l_important: o << "IMPORTANT"; break;
    case l_fatal: o << "FATAL"; break;
    }
    return o;
  }

  class Log : public std::ostringstream {
  public:
    Log(const LogGrade g = l_info) : grade(g) {
      using namespace std;
      if(grade < silence) setstate(ios::failbit);
      else (*this) << grade << ": ";
    }
    ~Log();

    static void addDevice(LogDevice* dev);
    static void setTreshold(LogGrade t) { silence = t; }
    /** Boring workaround ... Without this, the first thing to get appended to
     * the Log gets broken. */
    template<typename T>
    std::ostream& operator << (const T& t) {
      return *((std::ostream*)(this)) << t;
    }
  private:
    static LogGrade silence;
    class DeviceMgr;
    static DeviceMgr mgr;
    LogGrade grade;

    // Forbidden
    Log(const Log&);
    Log operator= (const Log&);
  };


  class SysLog : public LogDevice {
  public:
    SysLog(const std::string& ident, int facility = USER, int logopt = 0);
    void write(const std::string& msg);

    // Symbols to use for facility
    const static int AUTH, AUTHPRIV, CRON, DAEMON, FTP, KERN, LPR, MAIL, NEWS,
      SECURITY, SYSLOG, USER, UUCP, 
      LOCAL0, LOCAL1, LOCAL2, LOCAL3, LOCAL4, LOCAL5, LOCAL6, LOCAL7;
  };
  
  class StreamLog : public LogDevice {
  public:
    StreamLog(std::ostream& out) : output(&out) {}
    void write(const std::string& msg) { (*output) << msg << std::endl; }
  private:
    std::ostream* output;
  };

};
// Ok, this is broken ...  But I find nothing better that works with gcc
// 2.95.2.
static const incu::LogGrade l_debug = incu::l_debug;
static const incu::LogGrade l_info = incu::l_info;
static const incu::LogGrade l_important = incu::l_important;
static const incu::LogGrade l_fatal = incu::l_fatal;

#define DEBUG incu::Log(l_debug) << __FILE__ << ':' << __LINE__ << ": "

#endif // INCU_LOG
