/************************************************************************/
/* File		objectfile.h						*/
/*									*/
/* Purpose	This C++ header file defines the ObjectFile class	*/
/*		template. The ObjectFile class template provides an	*/
/*		interface to an object file. An object file can store	*/
/*		individual objects as records in the object file. The	*/
/*		records can be variable length and they can be randomly	*/
/*		accessed using record numbers. Also, the file space	*/
/*		that is freed up when a record is removed from the file	*/
/*		is reused when a new record is added to the file.	*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Thursday, May 2, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#ifndef OBJECTFILE_H			// If not defined...
#define OBJECTFILE_H			// Then process header file.

/* Headers	The following headers are required by objectfile.h.	*/

#include <stdio.h>			// sprintf function.
#include "os.h"				// Physical operating system.
#include "utilitymodule.h"		// UtilityModule class.
#include "dir.h"			// Dir class.
#include "file.h"			// File class.

/* Constants	The values of the following constants contain the	*/
/*		version information for the ObjectFile library.		*/

const int ObjectFileVersion	= 0;	// Library version number.
const int ObjectFileRevision	= 0;	// Library revision number.
const int ObjectFileUpdata	= 0;	// Library update number.

/* Constant	The following constant defines the default ObjectFile	*/
/*		file mode. This file mode is used to create a file if	*/
/*		no mode is specified.					*/

const int ObjectFileMode	= PhysicalOSOwnRead | PhysicalOSOwnWrite |
				  PhysicalOSGrpRead | PhysicalOSGrpWrite;

/* Constant	The following constant defines the default ObjectFile	*/
/*		file flag. This file flag is used to open files if no	*/
/*		flag is specified.					*/

const int ObjectFileFlag	= FileReadWrite;

/* Constants	The following constants define the valid ObjectFile	*/
/*		error codes that are used by the ObjectFile member	*/
/*		functions.						*/

const int ObjectFileErrors	= 18;	// Number of error messages.
const int ObjectFileNoError	= 0;	// No error.
const int ObjectFileNoMake	= 1;	// Could not create file.
const int ObjectFileNoOpen	= 2;	// Could not open file.
const int ObjectFileNoClose	= 3;	// Could not close file.
const int ObjectFileNoAdd	= 4;	// Could not add record.
const int ObjectFileNoRemove	= 5;	// Could not remove record.
const int ObjectFileDeleted	= 6;	// The record is deleted.
const int ObjectFileAddChain	= 7;	// Could not add del rec.
const int ObjectFileAddFront	= 8;	// Could not add front of chain.
const int ObjectFileAddEnd	= 9;	// Could not add end of chain.
const int ObjectFileAddInsert	= 10;	// Could not insert in chain.
const int ObjectFileAddJoin	= 11;	// Could not join to next.
const int ObjectFileUse		= 12;	// Could not use del rec.
const int ObjectFileNoRead	= 13;	// Could not read record.
const int ObjectFileNoWrite	= 14;	// Could not write record.
const int ObjectFileNoGet	= 15;	// Could not get next record.
const int ObjectFileHeadRead	= 16;	// Could not read header.
const int ObjectFileHeadWrite	= 17;	// Could not write header.

/* Functions	The following public functions can be used to read and	*/
/*		write data to and from an ObjectFile.			*/

status Read_String_From_ObjectFile(File, const int, String&);
status Write_String_To_ObjectFile(File, const int, String&);
status Read_Integer_From_ObjectFile(File, const int, int&);
status Write_Integer_To_ObjectFile(File, const int, int&);
status Read_Float_From_ObjectFile(File, const int, float&);
status Write_Float_To_ObjectFile(File, const int, float&);

/* Class	The following class defines the header structure that	*/
/*		is written at the beginning of every single record.	*/
/*		The ObjectFileRecHead is implemented in the file	*/
/*		objectfile.cpp.						*/

class ObjectFileRecHead
   {
      public:
         ObjectFileRecHead();
	 ObjectFileRecHead(const ObjectFileRecHead&);
	 ~ObjectFileRecHead();

	 condition Is_Deleted(void) const;
	 void Set_Deleted(const condition);
	 int Get_Record_Size(void) const;
	 void Set_Record_Size(const int);
	 FilePointer Get_Next_Deleted(void) const;
	 void Set_Next_Deleted(const FilePointer);

	 status Read_Record_Header(File&, const int, const FilePointer);
	 status Write_Record_Header(File&, const int, const FilePointer);

      private:
	 condition	itsdeleted;
	 int		itsrecordsize;
	 FilePointer	itsnextdeleted;
   };

/* Class	The following class defines the header structure that	*/
/*		is written at the beginning of every single file. The	*/
/*		ObjectFileHeader is implemented in the file		*/
/*		objectfile.cpp						*/

class ObjectFileHeader
   {
      public:
         ObjectFileHeader();
	 ~ObjectFileHeader();

	 void Get_Header_Message(char*);
	 void Set_Header_Message(const char*);
	 FilePointer Get_First_Deleted(void) const;
	 void Set_First_Deleted(const FilePointer);
	 FilePointer Get_Next_Available(void) const;
	 void Set_Next_Available(const FilePointer);

	 int Get_User_Header_Size(void) const;
	 void Set_User_Header_Size(const int);
	 status Read_ObjectFile_Header(File&, const int,
	    void* = (void*)0);
	 status Write_ObjectFile_Header(File&, const int,
	    void* = (void*)0);

      private:
	 char			itsmessage[64];
	 FilePointer		itsfirstdeleted;
	 FilePointer		itsnextavailable;
	 int			itsuserheadersize;
	 char			itsreserved[80 - (64 + 2 *
				(signed)sizeof(FilePointer) +
				(signed)sizeof(int))];
   };

/* Class	The following class defines the ObjectFile template.	*/
/*		This is the template that is responsible for		*/
/*		maintaining the ObjectFile files.			*/

template <class T>
class ObjectFile : public UtilityModule
   {
      public:
         ObjectFile();
	 ~ObjectFile();

	 status Create_ObjectFile_File(const char*,
	    const int = ObjectFileMode);
	 status Create_ObjectFile_File(const String&,
	    const int = ObjectFileMode);
	 status Create_ObjectFile_File(const char*, const int,
	    void*, const int = ObjectFileMode);
	 status Create_ObjectFile_File(const String&, const int,
	    void*, const int = ObjectFileMode);
	 status Open_ObjectFile_File(const char*,
	    const int = ObjectFileFlag);
	 status Open_ObjectFile_File(const String&,
	    const int = ObjectFileFlag);
	 status Open_ObjectFile_File(const char*, void*,
	    const int = ObjectFileFlag);
	 status Open_ObjectFile_File(const String&, void*,
	    const int = ObjectFileFlag);
	 status Close_ObjectFile_File(void);
	 status Add_ObjectFile_Record(T&, FilePointer&);
	 status Remove_ObjectFile_Record(const FilePointer);
	 status Read_ObjectFile_Record(T&, const FilePointer&);
	 status Write_ObjectFile_Record(T&, FilePointer&);
	 status Get_First_Record(T&, FilePointer&);
	 status Get_Next_Record(T&, FilePointer&);

	 status Read_ObjectFile_Header(void);
	 status Write_ObjectFile_Header(void);

      private:
         FilePointer		itsnextrecord;
         File			itsfile;
	 String			itsfilename;
	 int			itsfilehandle;
	 ObjectFileHeader	itsheader;
	 void*			itsuserheader;

	 status add_deleted_space(const FilePointer&,
	    ObjectFileRecHead&);
	 status add_to_chain(const FilePointer&,
	    ObjectFileRecHead&);
	 status add_to_front_of_chain(const FilePointer&,
	    ObjectFileRecHead&);
	 status add_to_end_of_chain(const FilePointer&,
	    ObjectFileRecHead&, const FilePointer&, ObjectFileRecHead&);
	 status insert_in_chain(const FilePointer&, ObjectFileRecHead&,
	    const FilePointer&, ObjectFileRecHead&);
	 status join_next_record(const FilePointer&, ObjectFileRecHead&);
	 status use_deleted_space(FilePointer&, int&);
   };

/************************************************************************/
/* Function	ObjectFile()						*/
/*									*/
/* Purpose	This is the ObjectFile default constructor. The default	*/
/*		constructor will prepare the object to report errors if	*/
/*		it needs to.						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The ObjectFile object will be prepared to report	*/
/*		errors.							*/
/************************************************************************/

template <class T>
ObjectFile<T>::ObjectFile()
   {
      /* The following variable is the error message array that is	*/
      /* specific to the ObjectFile class. The default UtilityModule	*/
      /* error values are overwritten by this ObjectFile object		*/
      /* constructor.							*/

      static char *errormessages[] =
         {
             "No error",				// NoError
	     "Could not create ObjectFile file",	// NoMake
	     "Could not open ObjectFile file",		// NoOpen
	     "Could not close ObjectFile file",		// NoClose
	     "Could not add ObjectFile record",		// NoAdd
	     "Could not remove ObjectFile record",	// NoRemove
	     "The ObjectFile record is deleted",	// Deleted
	     "Could not add deleted record to chain",	// AddChain
	     "Could not add to front of chain",		// AddFront
	     "Could not add to end of chain",		// AddEnd
	     "Could not insert in chain",		// AddInsert
	     "Could not join to next in chain",		// AddJoin
	     "Could not reuse deleted record",		// UseRec
	     "Could not read ObjectFile record",	// NoRead
	     "Could not write ObjectFile record",	// NoWrite
	     "Could not get next ObjectFile record",	// NoGet
	     "Could not read ObjectFile header",	// HeadRead
	     "Could not write ObjectFile header",	// HeadWrite
         };

      /* Reimplement the UtilityModule error variables .		*/

      itsmodulename		= "ObjectFile";
      itsmaxerror		= ObjectFileErrors;
      itserrormessages		= errormessages;
      itsnextrecord		= 0;
      itsuserheader		= (void*)0;
   }

/************************************************************************/
/* Function	~ObjectFile()						*/
/*									*/
/* Purpose	This is the ObjectFile default destructor. The default	*/
/*		destructor does nothing.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

template <class T>
ObjectFile<T>::~ObjectFile()
   {
   }

/************************************************************************/
/* Function	status add_deleted_space(const FilePointer& filepointer,*/
/*		   ObjectFileRecHead& recordheader)			*/
/*									*/
/* Purpose	This function decides how a record that is being	*/
/*		deleted should be added to the chain of deleted		*/
/*		records. If the record that is being deleted is the	*/
/*		first record to be deleted then this function will make	*/
/*		the record being deleted the first record in the chain	*/
/*		of deleted records. Otherwise, this function will call	*/
/*		on other functions to add the deleted record to the	*/
/*		chain of deleted records.				*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted.	*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to add the record to the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::add_deleted_space(const FilePointer& filepointer,
   ObjectFileRecHead& recordheader)
   {
      status		result		= OK;
      ObjectFileRecHead	workheader;

      recordheader.Set_Deleted(true);	// Mark the record deleted.

      /* See if this is the first record to be deleted.			*/

      if (itsheader.Get_First_Deleted() == 0)
         {
	    /* This is the first record to be deleted so make this	*/
	    /* record the first record in the chain and then write the	*/
	    /* record's header back to the file.			*/

	    itsheader.Set_First_Deleted(filepointer);
	    if (recordheader.Write_Record_Header(itsfile, itsfilehandle,
	       filepointer) == ERROR)
	       {
	          /* Could not add deleted record to chain.		*/

		  itserrorinfo	= "Attempted to add deleted record "
		  		  " to chain in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddChain;
		  result	= ERROR;
		  Report_Error();
	       }
	 }

      /* If the filepointer is less than the filepointer for the first	*/
      /* deleted record then this record should be made the first	*/
      /* record in the chain.						*/

      else if (filepointer < itsheader.Get_First_Deleted())
         {
	    /* Make this record the very first record in the chain.	*/

	    result	= add_to_front_of_chain(filepointer,
			  recordheader);
	 }

      /* If we made it here then there is at least one deleted record	*/
      /* in the chain and the record that is being deleted must come	*/
      /* after the first deleted record in the chain.			*/

      else
         {
	    /* Add this record after the first deleted record.		*/

	    result	= add_to_chain(filepointer, recordheader);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status add_to_chain(const FilePointer& filepointer,	*/
/*		   ObjectFileRecHead& recordheader)			*/
/*									*/
/* Purpose	This function will search through the chain of deleted	*/
/*		records to determine where the record that is being	*/
/*		deleted should be inserted in the chain of deleted	*/
/*		records.						*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted.	*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to add the record to the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::add_to_chain(const FilePointer& filepointer,
   ObjectFileRecHead& recordheader)
   {
      status		result		= OK;
      condition		more		= true;
      FilePointer	workpointer;
      ObjectFileRecHead	workheader;

      /* The following loop will search through the chain of deleted	*/
      /* records, looking for a place to insert the record that is	*/
      /* being deleted.							*/

      workpointer	= itsheader.Get_First_Deleted();
      do
         {
	    /* Read in the header information for the current record in	*/
	    /* the chain so that we can get the link to the next	*/
	    /* deleted record in the chain.				*/

	    if (workheader.Read_Record_Header(itsfile, itsfilehandle,
	       workpointer) == ERROR)
	       {
	          /* Could not add deleted record to chain.		*/

	          itserrorinfo	= "Attempted to add deleted record to "
				  "chain in file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileAddChain;
	          result	= ERROR;
	          Report_Error();
		  more		= false;	// End the loop.
	       }

	    /* If the pointer to the record that is being deleted is	*/
	    /* less than the pointer to the next record then this is	*/
	    /* where we need to insert the record that is being		*/
	    /* deleted.							*/

	    else if (filepointer < workheader.Get_Next_Deleted())
	       {
	          /* Insert the deleted record at this point.		*/

		  result	= insert_in_chain(filepointer,
		  		  recordheader, workpointer, workheader);
		  more		= false;	// End the loop.
	       }

	    /* If the pointer to the next deleted record is 0 then the	*/
	    /* record being deleted is the last record in the chain.	*/

	    else if (workheader.Get_Next_Deleted() == 0)
	       {
		  /* Add the deleted record to the end of the chain.	*/

		  result	= add_to_end_of_chain(filepointer,
		  		  recordheader, workpointer, workheader);
		  more		= false;	// End the loop.
	       }

	    /* Prepare to examine the next deleted record in the chain.	*/

	    workpointer	= workheader.Get_Next_Deleted();
	 }
      while (more == true);
      return(result);
   }

/************************************************************************/
/* Function	status add_to_front_of_chain(				*/
/*		   const FilePointer& filepointer,			*/
/*		   ObjectFileRecHead& recordheader)			*/
/*									*/
/* Purpose	This function is responsible for adding a record that	*/
/*		is being deleted to the front of the chain of deleted	*/
/*		records. If the record that is being deleted is		*/
/*		directly before the first deleted record in the chain	*/
/*		then the two deleted records are joined into one large	*/
/*		deleted record.						*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted.	*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to add the record to the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::add_to_front_of_chain(
   const FilePointer& filepointer, ObjectFileRecHead& recordheader)
   {
      status		result		= OK;
      ObjectFileRecHead	nextheader;

      /* Get the first deleted record's header information.		*/

      if (nextheader.Read_Record_Header(itsfile, itsfilehandle,
         itsheader.Get_First_Deleted()) == ERROR)
	 {
	    /* Could not add to front of chain.				*/

	    itserrorinfo	= "Attempted to add deleted record to "
				  "front of chain in file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileAddFront;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the pointer to the record that is being deleted + the size	*/
      /* of the record being deleted + the size of the record's		*/
      /* information header == the pointer to the first deleted record	*/
      /* then the record that is being deleted needs to be joined with	*/
      /* the first deleted record in the chain.				*/

      else if (filepointer + recordheader.Get_Record_Size() +
         sizeof(ObjectFileRecHead) == itsheader.Get_First_Deleted())
	 {
	    /* The two records need to be joined into one record. Add	*/
	    /* the size of the record that is already deleted to the	*/
	    /* size of the record being deleted then point to the next	*/
	    /* deleted record in the chain and write the record's	*/
	    /* header information back to the file.			*/

	    recordheader.Set_Record_Size(recordheader.Get_Record_Size() +
	       nextheader.Get_Record_Size() + sizeof(ObjectFileRecHead));
	    recordheader.Set_Next_Deleted(nextheader.Get_Next_Deleted());
	    itsheader.Set_First_Deleted(filepointer);
	    if (recordheader.Write_Record_Header(itsfile, itsfilehandle,
	       filepointer) == ERROR)
	       {
	          /* Could not add to front of chain.			*/

	          itserrorinfo	= "Attempted to add deleted record to "
				  "front of chain in file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileAddFront;
	          result	= ERROR;
	          Report_Error();
	       }
	 }

      /* Else, the record that is being added to the file does not need	*/
      /* to be joined with the first deleted record in the chain.	*/

      else
         {
	    /* Set the pointer to the next deleted record in the record	*/
	    /* that is being deleted and then make the record that is	*/
	    /* being deleted the first record in the chain of deleted	*/
	    /* records.							*/

	    recordheader.Set_Next_Deleted(itsheader.Get_First_Deleted());
	    itsheader.Set_First_Deleted(filepointer);
	    if (recordheader.Write_Record_Header(itsfile,
	       itsfilehandle, filepointer) == ERROR)
	       {
	          /* Could not add to front of chain.			*/

	          itserrorinfo	= "Attempted to add deleted record to "
				  "front of chain in file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileAddFront;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status add_to_end_of_chain(				*/
/*		   const FilePointer& filepointer,			*/
/*		   ObjectFileRecHead& recordheader,			*/
/*		   const FilePointer& workpointer,			*/
/*		   ObjectFileRecHead& workheader)			*/
/*									*/
/* Purpose	This function will add a record that is being deleted	*/
/*		to the end of the chain of deleted records. If the last	*/
/*		record in the chain of deleted records is immediately	*/
/*		before the record that is being deleted then the two	*/
/*		records will be joined into one large deleted record.	*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted. The	*/
/*		variable 'workpointer' must contain the pointer to the	*/
/*		last record in the chain of deleted records. The	*/
/*		variable 'workheader' must contain a reference to the	*/
/*		header information for the last record in the chain of	*/
/*		deleted records.					*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to add the record to the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::add_to_end_of_chain(const FilePointer& filepointer,
   ObjectFileRecHead& recordheader, const FilePointer& workpointer,
   ObjectFileRecHead& workheader)
   {
      status		result		= OK;

      /* If the pointer to the record that is in the chain of deleted	*/
      /* records + the size of the record that is already deleted + the	*/
      /* size of a record's header information == the pointer to the	*/
      /* record that is being deleted then the record that is being	*/
      /* deleted needs to be joined to the record that is already	*/
      /* deleted.							*/

      if (workpointer + workheader.Get_Record_Size() +
         sizeof(ObjectFileRecHead) == filepointer)
	 {
	    /* Join the two records by adding the size of the record	*/
	    /* that is being deleted + the size of a record's header	*/
	    /* information to the size of the record that is alread	*/
	    /* deleted.							*/

	    workheader.Set_Record_Size(workheader.Get_Record_Size() +
	       recordheader.Get_Record_Size() +
	       sizeof(ObjectFileRecHead));
	    if (workheader.Write_Record_Header(itsfile, itsfilehandle,
	       workpointer) == ERROR)
	       {
	          /* Could not add record to end of chain.		*/

		  itserrorinfo	= "Attempted to add record to end of "
		  		  "chain in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddEnd;
		  result	= ERROR;
		  Report_Error();
	       }
	 }

      /* Else, the two records do not need to be joined together so we	*/
      /* need to add the record that is being deleted to the end of the	*/
      /* chain of deleted records.					*/

      else
         {
	    /* Make the record that is being deleted the last record in	*/
	    /* the chain of deleted records and save both record's	*/
	    /* headers back to the file.				*/

	    workheader.Set_Next_Deleted(filepointer);
	    if (workheader.Write_Record_Header(itsfile, itsfilehandle,
	       workpointer) == ERROR)
	       {
	          /* Could not add record to end of chain.		*/

		  itserrorinfo	= "Attempted to add record to end of "
		  		  "chain in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddEnd;
		  result	= ERROR;
		  Report_Error();
	       }
	    else if (recordheader.Write_Record_Header(itsfile,
	       itsfilehandle, filepointer) == ERROR)
	       {
	          /* Could not add record to end of chain.		*/

		  itserrorinfo	= "Attempted to add record to end of "
		  		  "chain in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddEnd;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status insert_in_chain(const FilePointer& filepointer,	*/
/*		   ObjectFileRecHead& recordheader,			*/
/*		   const FilePointer& workpointer,			*/
/*		   ObjectFileRecHead& workheader)			*/
/*									*/
/* Purpose	This function will insert a record that is being	*/
/*		deleted in the chain of deleted records. If the record	*/
/*		that is being deleted is immediately before or after	*/
/*		another deleted record in the chain of deleted records	*/
/*		then the records will be joined together into one large	*/
/*		deleted record.						*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted. The	*/
/*		variable 'workpointer' must contain the pointer to the	*/
/*		record in the chain of deleted records that precedes	*/
/*		the record that is being deleted. The variable		*/
/*		'workheader' must contain a reference to the header	*/
/*		information for the record in the chain of deleted	*/
/*		records that precedes the record that is being deleted.	*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to add the record to the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::insert_in_chain(const FilePointer& filepointer,
   ObjectFileRecHead& recordheader, const FilePointer& workpointer,
   ObjectFileRecHead& workheader)
   {
      status		result		= OK;

      /* If the pointer to the current record in the chain of deleted	*/
      /* records + the size of the current record + the size of a	*/
      /* record's information header == the pointer to the record that	*/
      /* is being deleted then the two records need to be joined.	*/

      if (workpointer + workheader.Get_Record_Size() +
         sizeof(ObjectFileRecHead) == filepointer)
	 {
	    /* Add the size of the record that is already deleted + the	*/
	    /* size of the record that is being deleted + the size of a	*/
	    /* record's information header.				*/

	    workheader.Set_Record_Size(workheader.Get_Record_Size() +
	       recordheader.Get_Record_Size() +
	       sizeof(ObjectFileRecHead));
	    if (workheader.Write_Record_Header(itsfile, itsfilehandle,
	       workpointer) == ERROR)
	       {
	          /* Could not insert record in chain.			*/

		  itserrorinfo	= "Attempted to insert record in chain "
				  "in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddInsert;
		  result	= ERROR;
		  Report_Error();
	       }
	    else
	       {
		  /* Finally, call a function to see if the record that	*/
		  /* we just deleted needs to be joined with the next	*/
		  /* record in the chain of deleted records.		*/

		  result	= join_next_record(workpointer,
		  		  workheader);
	       }
	 }

      /* Else, the two records do not need to be joined. Therefore, we	*/
      /* need to add the record that is being deleted to the chain of	*/
      /* deleted records after the current record in the chain.		*/

      else
         {
	    /* Set the pointer to the next deleted record in both	*/
	    /* records and write the information headers back to the	*/
	    /* file.							*/

	    recordheader.Set_Next_Deleted(workheader.Get_Next_Deleted());
	    workheader.Set_Next_Deleted(filepointer);
	    if (recordheader.Write_Record_Header(itsfile, itsfilehandle,
	       filepointer) == ERROR)
	       {
	          /* Could not insert record in chain.			*/

		  itserrorinfo	= "Attempted to insert record in chain "
				  "in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddInsert;
		  result	= ERROR;
		  Report_Error();
	       }
	    else if (workheader.Write_Record_Header(itsfile,
	       itsfilehandle, workpointer) == ERROR)
	       {
	          /* Could not insert record in chain.			*/

		  itserrorinfo	= "Attempted to insert record in chain "
				  "in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddInsert;
		  result	= ERROR;
		  Report_Error();
	       }
	    else
	       {
		  /* Finally, call a function to see if the record that	*/
		  /* we just deleted needs to be joined with the next	*/
		  /* record in the chain of deleted records.		*/

	          result	= join_next_record(filepointer,
		  		  recordheader);
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status join_next_record(				*/
/*		   const FilePointer& recordpointer,			*/
/*		   ObjectFileRecHead& recordheader)			*/
/*									*/
/* Purpose	This function will determine if the record that is	*/
/*		being deleted needs to be joined to the next record in	*/
/*		the chain of deleted records. If they do need to be	*/
/*		joined then this function will join them.		*/
/*									*/
/* Input	This function expects the variable 'recordpointer' to	*/
/*		point to the record that is being deleted. The variable	*/
/*		'recordheader' must contain a reference to the header	*/
/*		information for the record that is being deleted.	*/
/*									*/
/* Output	If this function was able to join the records in the	*/
/*		chain of deleted records then it will return OK. If	*/
/*		this function is not able to join the records in the	*/
/*		chain of deleted records then it will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::join_next_record(const FilePointer& recordpointer,
   ObjectFileRecHead& recordheader)
   {
      status		result		= OK;
      ObjectFileRecHead	nextheader;

      /* If the record pointer + the size of the record + the size of	*/
      /* a record's information header == the pointer to the next	*/
      /* record in the chain of deleted records then join the two.	*/

      if (recordpointer + recordheader.Get_Record_Size() +
	 sizeof(ObjectFileRecHead) == recordheader.Get_Next_Deleted())
	 {
	    /* Read in the header for the next record in the chain of	*/
	    /* deleted records.						*/

	    if (nextheader.Read_Record_Header(itsfile, itsfilehandle,
	       recordheader.Get_Next_Deleted()) == ERROR)
	       {
		  /* Could not join next record in chain.		*/

		  itserrorinfo	= "Attempted to join next record "
				  "in file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileAddJoin;
		  result	= ERROR;
		  Report_Error();
		}
	     else
		{
		   /* Join the two records together by adding the size	*/
		   /* of the next record to the record being deleted.	*/

		   recordheader.Set_Record_Size(
		      recordheader.Get_Record_Size() +
		      nextheader.Get_Record_Size() +
		      sizeof(ObjectFileRecHead));
		   recordheader.Set_Next_Deleted(
		      nextheader.Get_Next_Deleted());

		   /* Write the one large record's information header	*/
		   /* back to the file.					*/

		   if (recordheader.Write_Record_Header(itsfile,
		      itsfilehandle, recordpointer) == ERROR)
		      {
			 /* Could not join next record in chain.	*/

			 itserrorinfo	= "Attempted to insert record in "
			 		  "chain in file ";
			  itserrorinfo	+= itsfilename;
			  itserror	= ObjectFileAddInsert;
			  result	= ERROR;
			  Report_Error();
		       }
		}
	 }
      return(result);
   }

/************************************************************************/
/* Function	status use_deleted_space(FilePointer& filepointer,	*/
/*		   int& recordsize)					*/
/*									*/
/* Purpose	This function will attempt to allocate space for a new	*/
/*		record from the records that have been previously	*/
/*		deleted. If a deleted record is big enough to satisfy	*/
/*		the request, then the record will be removed from the	*/
/*		chain of deleted records and used to satisfy the	*/
/*		request. If the size of the deleted record is big	*/
/*		enough to split into two records then the space for the	*/
/*		new record will come from the split record.		*/
/*									*/
/* Input	This function expects the variable 'recordsize' to	*/
/*		contain the amount of memory (in bytes) being requested	*/
/*		for a new record.					*/
/*									*/
/* Output	If this function was able to allocate space from a	*/
/*		previously deleted record then it will return OK and	*/
/*		the variable 'filepointer' will point to the record	*/
/*		that was allocated. Also, the variable 'recordsize'	*/
/*		will contain the actual size of the record that was	*/
/*		allocated. The size of the record might be slightly	*/
/*		larger than was requested. If this function was not	*/
/*		able to allocate space from a previously deleted record	*/
/*		then the variable 'filepointer' will contain 0. If this	*/
/*		function encounters any errors then it will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

template <class T>
status ObjectFile<T>::use_deleted_space(FilePointer& filepointer,
   int& recordsize)
   {
      status		result		= OK;
      int		request;
      FilePointer	prevpointer	= 0;
      ObjectFileRecHead	prevheader;
      ObjectFileRecHead	workheader;

      /* Prepare to enter the loop that will search through the chain	*/
      /* of deleted records.						*/

      filepointer	= itsheader.Get_First_Deleted();
      request		= recordsize + sizeof(ObjectFileRecHead);

      /* Search through the chain of deleted records looking for	*/
      /* a record that is big enough to satisfy the request.		*/

      while (filepointer != 0)
	 {
	    /* Read in the current record's information header so that	*/
	    /* the record size can be compared.				*/

	    if (workheader.Read_Record_Header(itsfile, itsfilehandle,
	       filepointer) == ERROR)
	       {
	          /* Could not reuse deleted record.			*/

		  itserrorinfo	= "Attempted to use deleted record in "
				  "file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= ObjectFileUse;
	          result	= ERROR;
		  Report_Error();
		  break;;
	       }

	    /* If the size of the current deleted record is big enough	*/
	    /* to satisfy the request then use this record.		*/

	    else if (workheader.Get_Record_Size() >= request)
	       {
	          /* If the size of the record is bigger than the size	*/
		  /* of the request + the size of a record's		*/
		  /* information header + the size of an object (this	*/
		  /* is the minimum space required for a record) then	*/
		  /* we can split this record into two records.		*/

	          if (workheader.Get_Record_Size() >= request +
		     (signed)sizeof(T))
		     {
		        /* Subtract the requested space from the size	*/
			/* of the deleted record. Then write the new	*/
			/* value back to the file.			*/

			workheader.Set_Record_Size(
			   workheader.Get_Record_Size() - request);
			if (workheader.Write_Record_Header(itsfile,
			   itsfilehandle, filepointer) == ERROR)
			   {
			      /* Could not reuse deleted record.	*/

			      itserrorinfo	= "Attempted to use "
						  "deleted record in "
						  "file ";
			      itserrorinfo	+= itsfilename;
			      itserror		= ObjectFileUse;
	        	      result		= ERROR;
			      Report_Error();
			      break;
			   }
			else
			   {
			      /* Point to the first byte in the space	*/
			      /* that we subtracted from the deleted	*/
			      /* record.				*/

			      filepointer	+= workheader.
			      			Get_Record_Size() +
						sizeof(ObjectFileRecHead);
			      break;
			   }
		     }

		  /* Else, the record size was not big enough to split	*/
		  /* into two records so we will use the current record	*/
		  /* even though it may actually be a few bytes bigger	*/
		  /* than it needs to be.				*/

		  else
		     {
		        recordsize	= workheader.Get_Record_Size();

			/* If prevpointer == 0 then this is the first	*/
			/* deleted record in the chain of deleted	*/
			/* records.					*/

			if (prevpointer == 0)
			   {
			      /* Remove the first deleted record from	*/
			      /* the chain of deleted records.		*/

			      itsheader.Set_First_Deleted(
			         workheader.Get_Next_Deleted());
			   }

			/* The previous record must be updated to	*/
			/* remove the current record from the chain of	*/
			/* deleted records.				*/

			else
			   {
			      /* Update the previous record to point	*/
			      /* past the current record.		*/

			      prevheader.Set_Next_Deleted(
			         workheader.Get_Next_Deleted());
			      if (prevheader.Write_Record_Header(itsfile,
			         itsfilehandle, prevpointer) == ERROR)
				 {
				    /* Could not reuse deleted record.	*/

				    itserrorinfo	= "Attempted to "
							  "use deleted "
							  "record in "
							  "file ";
				    itserrorinfo	+= itsfilename;
				    itserror		= ObjectFileUse;
	        		    result		= ERROR;
				    Report_Error();
				    break;
				 }
			   }
		     }
		  break;		// Exit loop with filepointer set.
	       }
	    else
	       {
	          /* Prepare to examine the next deleted record in the	*/
		  /* chain of deleted records.				*/

	          prevpointer	= filepointer;
	          prevheader	= workheader;
	          filepointer	= workheader.Get_Next_Deleted();
	       }
	 }				// Try the next deleted record.
      return(result);
   }

/************************************************************************/
/* Function	status Create_Data_File(const char* filename,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This ObjectFile function is responsible for creating an	*/
/*		ObjectFile file. The ObjectFile file is physically	*/
/*		created and a header structure is written to the	*/
/*		beginning of the file. The header structure contains	*/
/*		the information that is needed to store records in the	*/
/*		file. The file is closed after the header structure is	*/
/*		written.						*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectPhysicalOS is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Create_ObjectFile_File(const char* filename,
   const int mode)
   {
      status		result		= OK;
      char		message[65];

      /* Create the physical file using the File class.			*/

      if (itsfile.Create_File(filename, &itsfilehandle, mode) == ERROR)
         {
	    /* Could not create ObjectFile file.			*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to create ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoMake;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Initialize the header structure for the created file.	*/

	    sprintf(message, "Padre Software  "
			     "ObjectFile File "
			     "Version Number  "
			     "%02d.%02d.%02d        ", ObjectFileVersion,
			      ObjectFileRevision, ObjectFileUpdata);
	    itsheader.Set_Header_Message(message);
	    itsheader.Set_First_Deleted(0);
	    itsheader.Set_Next_Available(sizeof(ObjectFileHeader));
	    itsheader.Set_User_Header_Size(0);

	    /* Write the ObjectFile file's header to the created file.	*/

	    if (itsheader.Write_ObjectFile_Header(itsfile, itsfilehandle)
	       == ERROR)
	       {
	          /* Could not create ObjectFile file.			*/

		  itserrorinfo	= "Attempted to create ObjectFile file ";
		  itserrorinfo	+= filename;
		  itserror	= ObjectFileNoMake;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* Finish up by closing the created file.			*/

	    else if (itsfile.Close_File(itsfilehandle) == ERROR)
	       {
	          /* Could not create ObjectFile file.			*/

	          itsfile.Report_Error();
	          itserrorinfo	= "Attempted to create ObjectFile file ";
		  itserrorinfo	+= filename;
	          itserror	= ObjectFileNoMake;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Data_File(const String& filename,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This ObjectFile function is responsible for creating an	*/
/*		ObjectFile file. The ObjectFile file is physically	*/
/*		created and a header structure is written to the	*/
/*		beginning of the file. The header structure contains	*/
/*		the information that is needed to store records in the	*/
/*		file. The file is closed after the header structure is	*/
/*		written.						*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectPhysicalOS is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Create_ObjectFile_File(const String& filename,
   const int mode)
   {
      return(Create_ObjectFile_File(filename.Data(), mode));
   }

/************************************************************************/
/* Function	status Create_ObjectFile_File(const String& filename,	*/
/*		   const int userheadersize, const void* userheader,	*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This ObjectFile function is responsible for creating an	*/
/*		ObjectFile file with a user's header area. The user's	*/
/*		header area can be used to store additional information	*/
/*		about the file, in the file's header. The user's header	*/
/*		area is read in from the file everytime that the file's	*/
/*		header is read in from the file and the user's header	*/
/*		area is written to the file everytime that the file's	*/
/*		header is written to the file.				*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The variable		*/
/*		'userheadersize' must contain the size (in bytes) to	*/
/*		reserve at the beginning of the file for the user's	*/
/*		header area. The variable 'userheader' must contain a	*/
/*		pointer to the initial user's header that is to be	*/
/*		written to the created file.The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectFileMode is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Create_ObjectFile_File(const char* filename,
   const int userheadersize, void* userheader, const int mode)
   {
      status		result		= OK;
      char		message[65];

      /* Create the physical file using the File class.			*/

      if (itsfile.Create_File(filename, &itsfilehandle, mode) == ERROR)
         {
	    /* Could not create ObjectFile file.			*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to create ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoMake;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Initialize the header structure for the created file.	*/

	    sprintf(message, "Padre Software  "
			     "ObjectFile File "
			     "Version Number  "
			     "%02d.%02d.%02d        ", ObjectFileVersion,
			      ObjectFileRevision, ObjectFileUpdata);
	    itsheader.Set_Header_Message(message);
	    itsheader.Set_First_Deleted(0);
	    itsheader.Set_Next_Available(sizeof(ObjectFileHeader) +
	       userheadersize);
	    itsheader.Set_User_Header_Size(userheadersize);
	    itsuserheader	= userheader;

	    /* Write the ObjectFile file's header to the created file.	*/

	    if (itsheader.Write_ObjectFile_Header(itsfile, itsfilehandle,
	       userheader) == ERROR)
	       {
	          /* Could not create ObjectFile file.			*/

		  itserrorinfo	= "Attempted to create ObjectFile file ";
		  itserrorinfo	+= filename;
		  itserror	= ObjectFileNoMake;
		  result	= ERROR;
		  Report_Error();
	       }

	    /* Finish up by closing the created file.			*/

	    else if (itsfile.Close_File(itsfilehandle) == ERROR)
	       {
	          /* Could not create ObjectFile file.			*/

	          itsfile.Report_Error();
	          itserrorinfo	= "Attempted to create ObjectFile file ";
		  itserrorinfo	+= filename;
	          itserror	= ObjectFileNoMake;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_ObjectFile_File(const String& filename,	*/
/*		   const int userheadersize, const void* userheader,	*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This ObjectFile function is responsible for creating an	*/
/*		ObjectFile file with a user's header area. The user's	*/
/*		header area can be used to store additional information	*/
/*		about the file, in the file's header. The user's header	*/
/*		area is read in from the file everytime that the file's	*/
/*		header is read in from the file and the user's header	*/
/*		area is written to the file everytime that the file's	*/
/*		header is written to the file.				*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The variable		*/
/*		'userheadersize' must contain the size (in bytes) to	*/
/*		reserve at the beginning of the file for the user's	*/
/*		header area. The variable 'userheader' must contain a	*/
/*		pointer to the initial user's header that is to be	*/
/*		written to the created file.The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectFileMode is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Create_ObjectFile_File(const String& filename,
   const int userheadersize, void* userheader, const int mode)
   {
      return(Create_ObjectFile_File(filename.Data(), userheadersize,
         userheader, mode));
   }

/************************************************************************/
/* Function	status Open_ObjectFile_File(const char* filename,	*/
/*		   const int flag)					*/
/*									*/
/* Purpose	This function is responsible for opening an ObjectFile	*/
/*		object's file. To open an ObjectFile object's file,	*/
/*		this function will open the physical file that was	*/
/*		previously created with Create_ObjectFile_File and then	*/
/*		this function will read in the file's header.		*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the ObjectFile file to	*/
/*		open. The file will be opened with the permissions	*/
/*		passed to this function in the variable 'flag'. If the	*/
/*		variable 'flag' is not specified then the default value	*/
/*		of the constant ObjectFileFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Open_ObjectFile_File(const char* filename,
   const int flag)
   {
      status		result		= OK;

      /* Open the physical file using the File class.			*/

      if (itsfile.Open_File(filename, &itsfilehandle, flag) == ERROR)
         {
	    /* Could not open data file.				*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to open ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoOpen;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Read in the ObjectFile file's header information.		*/

      else if (itsheader.Read_ObjectFile_Header(itsfile, itsfilehandle)
         == ERROR)
	 {
	    /* Could not open file.					*/

	    itserrorinfo	= "Attempted to open ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Save the filename for error reporting.			*/

	    itsfilename		= filename;
	    itsheader.Set_User_Header_Size(0);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_ObjectFile_File(const String& filename,	*/
/*		   const int flag)					*/
/*									*/
/* Purpose	This function is responsible for opening an ObjectFile	*/
/*		object's file. To open an ObjectFile object's file,	*/
/*		this function will open the physical file that was	*/
/*		previously created with Create_ObjectFile_File and then	*/
/*		this function will read in the file's header.		*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the ObjectFile file to	*/
/*		open. The file will be opened with the permissions	*/
/*		passed to this function in the variable 'flag'. If the	*/
/*		variable 'flag' is not specified then the default value	*/
/*		of the constant ObjectFileFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Open_ObjectFile_File(const String& filename,
   const int flag)
   {
      return(Open_ObjectFile_File(filename.Data(), flag));
   }

/************************************************************************/
/* Function	status Open_ObjectFile_File(const char* filename,	*/
/*		   void* userheader, const int flag)			*/
/*									*/
/* Purpose	This function is responsible for opening an ObjectFile	*/
/*		file with a user's header area. The user's header area	*/
/*		must be specified when the file is created. This	*/
/*		function will open an ObjectFile file and then it will	*/
/*		read in the file's header information. The user's	*/
/*		header area will use the buffer that is pointed to by	*/
/*		the variable 'userheader'.				*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the ObjectFile file to	*/
/*		open. The variable 'userheader' must contain a pointer	*/
/*		to the buffer that holds the user's header information.	*/
/*		The file will be opened with the permissions passed to	*/
/*		this function in the variable 'flag'. If the variable	*/
/*		'flag' is not specified then the default value of the	*/
/*		constant ObjectFileFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Open_ObjectFile_File(const char* filename,
   void* userheader, const int flag)
   {
      status		result		= OK;

      /* Open the physical file using the File class.			*/

      if (itsfile.Open_File(filename, &itsfilehandle, flag) == ERROR)
         {
	    /* Could not open data file.				*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to open ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoOpen;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Read in the ObjectFile file's header information.		*/

      else if (itsheader.Read_ObjectFile_Header(itsfile, itsfilehandle,
	 userheader) == ERROR)
	 {
	    /* Could not open file.					*/

	    itserrorinfo	= "Attempted to open ObjectFile file ";
	    itserrorinfo	+= filename;
	    itserror		= ObjectFileNoOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Save the filename for error reporting.			*/

	    itsuserheader	= userheader;
	    itsfilename		= filename;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_ObjectFile_File(const String& filename,	*/
/*		   const int flag)					*/
/*									*/
/* Purpose	This function is responsible for opening an ObjectFile	*/
/*		object's file. To open an ObjectFile object's file,	*/
/*		this function will open the physical file that was	*/
/*		previously created with Create_ObjectFile_File and then	*/
/*		this function will read in the file's header.		*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the ObjectFile file to	*/
/*		open. The file will be opened with the permissions	*/
/*		passed to this function in the variable 'flag'. If the	*/
/*		variable 'flag' is not specified then the default value	*/
/*		of the constant ObjectFileFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Open_ObjectFile_File(const String& filename,
   void* userheader, const int flag)
   {
      return(Open_ObjectFile_File(filename.Data(), userheader, flag));
   }

/************************************************************************/
/* Function	status Close_ObjectFile_File(void)			*/
/*									*/
/* Purpose	This function is responsible for closing an ObjectFile	*/
/*		file. To close the ObjectFile file, this function	*/
/*		simply closes the physical file.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to close the ObjectFile file	*/
/*		then this function will return OK. If this function was	*/
/*		not able to close the ObjectFile file then this		*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Close_ObjectFile_File(void)
   {
      status		result		= OK;

      if (itsfile.Close_File(itsfilehandle) == ERROR)
         {
	    /* Could not close file.					*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to close ObjectFile file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoClose;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Add_ObjectFile_Record(T& record,			*/
/*		   FilePointer& filepointer)				*/
/*									*/
/* Purpose	This function is responsible for adding a record to an	*/
/*		ObjectFile file. If space is available from the records	*/
/*		that were previously deleted then the record will use	*/
/*		the space from a previously deleted record. If there is	*/
/*		no space available from previously deleted records then	*/
/*		the record will be appended to the end of the file.	*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to the object that is to be written to the	*/
/*		file.							*/
/*									*/
/* Output	If this function was able to add the record to the	*/
/*		ObjectFile file then it will return OK and the variable	*/
/*		'filepointer' will contain the record pointer to use	*/
/*		when you want to access the record. If this function	*/
/*		was not able to add the record to the file then it will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Add_ObjectFile_Record(T& record,
   FilePointer& filepointer)
   {
      status		result		= OK;
      int		recordsize;
      ObjectFileRecHead	recordheader;

      /* Initialize a new record's header information.			*/

      recordsize	= record.Record_Size();
      recordheader.Set_Deleted(false);
      recordheader.Set_Next_Deleted(0);

      /* See if there is deleted space available.			*/

      if (use_deleted_space(filepointer, recordsize) == ERROR)
         {
	    /* Could not add ObjectFile record.				*/

	    itserrorinfo	= "Attempted to add ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoAdd;
	    result		= ERROR;
	    Report_Error();
	  }

      /* If filepointer == 0 then there was no deleted space available.	*/

      else if (filepointer == 0)
         {
	    /* Use the next available record.				*/

	    filepointer		= itsheader.Get_Next_Available();
	    itsheader.Set_Next_Available(filepointer + recordsize +
	       sizeof(ObjectFileRecHead));
	 }

      /* Set the record size here. The size of the actual record may	*/
      /* be slightly larger than the requested size if a previously	*/
      /* deleted record is being reused.				*/

      recordheader.Set_Record_Size(recordsize);

      /* Write the new record header to the file.			*/

      if (recordheader.Write_Record_Header(itsfile, itsfilehandle,
         filepointer) == ERROR)
         {
	    /* Could not add ObjectFile record.				*/

	    itserrorinfo	= "Attempted to add ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoAdd;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Have the record write itself to the file.			*/

      else if (record.Write_To_File(itsfile, itsfilehandle) == ERROR)
         {
	    /* Could not add data record.				*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to add ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoAdd;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Finally, write the ObjectFile file's header to the file.	*/

      else if (itsheader.Write_ObjectFile_Header(itsfile, itsfilehandle,
         itsuserheader) == ERROR)
         {
	    /* Could not add data record.				*/

	    itserrorinfo	= "Attempted to add ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoAdd;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_ObjectFile_Record(			*/
/*		   const FilePointer filepointer)			*/
/*									*/
/* Purpose	This function is responsible for removing a record from	*/
/*		an ObjectFile file. The space in the file that the	*/
/*		record was stored in will be added to a linked list of	*/
/*		deleted records. This makes it possible for future	*/
/*		additions to reuse this file space.			*/
/*									*/
/* Input	This function expects the variable 'filepointer' to	*/
/*		contain a pointer to a record in an ObjectFile file.	*/
/*		The pointer is the value that is returned when a record	*/
/*		is added to the file.					*/
/*									*/
/* Output	If this function was able to remove the record from the	*/
/*		ObjectFile file then it will return OK. If this		*/
/*		function was not able to remove the record from the	*/
/*		file then it will return ERROR. All errors by this	*/
/*		function are reported to stderr.			*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Remove_ObjectFile_Record(
   const FilePointer filepointer)
   {
      status		result		= OK;
      ObjectFileRecHead	recordheader;

      /* Read in the record's header information.			*/

      if (recordheader.Read_Record_Header(itsfile, itsfilehandle,
         filepointer) == ERROR)
	 {
	    /* Could not remove data record.				*/

	    itserrorinfo	= "Attempted to remove "
				  "ObjectFile record from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoRemove;
	    result		= ERROR;
	    Report_Error();
	 }

      /* See if the record is already deleted.				*/

      else if (recordheader.Is_Deleted() == true)
         {
	    /* The record is already deleted.				*/

	    itserrorinfo	= "Attempted to remove "
				  "ObjectFile record from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileDeleted;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Add the deleted record to the deleted space.			*/

      else if (add_deleted_space(filepointer, recordheader) == ERROR)
         {
	    /* Could not remove data record.				*/

	    itserrorinfo	= "Attempted to remove "
				  "ObjectFile record from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoRemove;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Finally, write the ObjectFile file's header to the file.	*/

      else if (itsheader.Write_ObjectFile_Header(itsfile, itsfilehandle,
         itsuserheader) == ERROR)
         {
	    /* Could not remove ObjectFile record.			*/

	    itserrorinfo	= "Attempted to remove "
				  "ObjectFile record from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoRemove;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Read_ObjectFile_Record(T& record,		*/
/*		   const FilePointer filepointer)			*/
/*									*/
/* Purpose	This function is responsible for reading a record from	*/
/*		the ObjectFile file. The record is read in from the	*/
/*		file by positioning the file pointer to point to the	*/
/*		beginning of the record's data and then having the	*/
/*		object read itself in from the file.			*/
/*									*/
/* Input	This function expects the variable 'record' to		*/
/*		reference the object that will read itself in from the	*/
/*		file. The variable 'filepointer' must contain the	*/
/*		record pointer for the record that is to be read in	*/
/*		from the file. 'filepointer' is the value that is	*/
/*		is returned when a record is added to the file.		*/
/*									*/
/* Output	If this function was able to read in the record from	*/
/*		the ObjectFile file then it will return OK. If this	*/
/*		function was not able to read in the record from the	*/
/*		file then it will return ERROR. All errors by this	*/
/*		function are reported to stderr.			*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Read_ObjectFile_Record(T& record,
   const FilePointer& filepointer)
   {
      status		result		= OK;
      ObjectFileRecHead	recordheader;

      /* Read in the record's header information.			*/

      if (recordheader.Read_Record_Header(itsfile, itsfilehandle,
         filepointer) == ERROR)
	 {
	    /* Could not read data record.				*/

	    itserrorinfo	= "Attempted to read ObjectFile record "
				  "from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoRead;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that this record is not deleted.			*/

      else if (recordheader.Is_Deleted() == true)
         {
	    /* The record is deleted.					*/

	    itserrorinfo	= "Attempted to read ObjectFile record "
				  "from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileDeleted;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Have the object read itself in from the file.			*/

      else if (record.Read_From_File(itsfile, itsfilehandle) == ERROR)
         {
	    /* Could not read data record.				*/

	    itsfile.Report_Error();
	    itserrorinfo	= "Attempted to read ObjectFile record "
				  "from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoRead;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_ObjectFile_Record(T& record,		*/
/*		   FilePointer filepointer)				*/
/*									*/
/* Purpose	This function is responsible for writing a record to	*/
/*		the ObjectFile file. If the record that is being	*/
/*		written to the file is the same size as the record that	*/
/*		is being overwritten then this function will simply	*/
/*		write over the old record. However, if the record that	*/
/*		is being written to the file has changed size, this	*/
/*		function may delete the old record and add this record	*/
/*		to the file as a new record. When this function deletes	*/
/*		the old record and adds a new record, the record	*/
/*		pointer for the new record will be returned in the	*/
/*		variable 'filepointer'.					*/
/*									*/
/* Input	This function expects the variable 'record' to		*/
/*		reference the object that will write itself out to the	*/
/*		file. The variable 'filepointer' must contain the	*/
/*		record pointer for the record that is to be overwritten	*/
/*		in the file. 'filepointer' is the value that is		*/
/*		returned when a record is added to the file.		*/
/*									*/
/* Output	If this function was able to write the record to the	*/
/*		ObjectFile file then it will return OK. If the size of	*/
/*		the object being written to the file is different than	*/
/*		the size of the record that is overwritten then this	*/
/*		function may delete the old record and add a new record	*/
/*		to the file. When this function adds a new record to	*/
/*		the file then the record pointer for the new record	*/
/*		will be returned in the variable 'filepointer'. If this	*/
/*		function was not able to write the record to the file	*/
/*		then it will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Write_ObjectFile_Record(T& record,
   FilePointer& filepointer)
   {
      status		result		= OK;
      ObjectFileRecHead	recordheader;

      /* Read in the record's header information so we can test the	*/
      /* size of the record with the size of the item that is being	*/
      /* written.							*/

      if (recordheader.Read_Record_Header(itsfile, itsfilehandle,
         filepointer) == ERROR)
	 {
	    /* Could not write data record.				*/

	    itserrorinfo	= "Attempted to write ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileNoWrite;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Make sure that this record is not deleted.			*/

      else if (recordheader.Is_Deleted() == true)
         {
	    /* The record is deleted.					*/

	    itserrorinfo	= "Attempted to write ObjectFile record "
				  "to ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileDeleted;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the size of the record in the file is greater than or equal	*/
      /* to the size of the record that is being written AND the size	*/
      /* of the record in the file is less than the size of the record	*/
      /* being written + the size of a record header + the size of an	*/
      /* object (this prevents records from becoming unnecessarily	*/
      /* large) then simply write over the old record.			*/

      else if (recordheader.Get_Record_Size() >= record.Record_Size() &&
         recordheader.Get_Record_Size() < record.Record_Size() +
	 (signed)sizeof(ObjectFileRecHead) + (signed)sizeof(T))
         {
	    /* Have the object write itself to the file.		*/

	    if (record.Write_To_File(itsfile, itsfilehandle) == ERROR)
	       {
	          /* Could not write data record.			*/

	          itsfile.Report_Error();
	          itserrorinfo	= "Attempted to write ObjectFile record "
				  "to ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileNoWrite;
	          result	= ERROR;
	          Report_Error();
	       }
	 }

      /* Else, the record that is being written to the file is larger	*/
      /* than the record in the file that is being overwritten OR the	*/
      /* current record is unnecessarily large. Therefore, we need to	*/
      /* delete the old record and add the new record to the file.	*/

      else
         {
	    /* Remove the current record from the file.			*/

	    if (Remove_ObjectFile_Record(filepointer) == ERROR)
	       {
	          /* Could not write data record.			*/

	          itserrorinfo	= "Attempted to write ObjectFile record "
				  "to ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileNoWrite;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Add the record to the file as a new record.		*/

	    else if (Add_ObjectFile_Record(record, filepointer) == ERROR)
	       {
	          /* Could not write data record.			*/

	          itserrorinfo	= "Attempted to write ObjectFile record "
				  "to ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileNoWrite;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_First_Record(T& record, FilePointer& pointer)*/
/*									*/
/* Purpose	This function can be used to return the first record	*/
/*		(the record closest to the beginning of the file) in	*/
/*		the ObjectFile file.					*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a template object. This is where the	*/
/*		first record will be written. The variable 'pointer'	*/
/*		must contain a reference to a FilePointer. This is	*/
/*		where the record number for the first record will be	*/
/*		written.						*/
/*									*/
/* Ouput	If this function is able to get the first record in the	*/
/*		file, or if there are no records in the file, then this	*/
/*		function will return OK. If there were records in the	*/
/*		file then the variable 'record' will contain the data	*/
/*		for the first record and the variable 'pointer' will	*/
/*		contain the record pointer for the first record. If	*/
/*		there are no records in the file then the variable	*/
/*		'pointer' will contain 0. If this function encounters	*/
/*		an error then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Get_First_Record(T& record, FilePointer& pointer)
   {
      status		result		= OK;

      /* The size of the ObjectFile's header + the size of the user	*/
      /* defined header = the first location for a record in the file.	*/

      itsnextrecord	= sizeof(ObjectFileHeader) +
			  itsheader.Get_User_Header_Size();

      /* If the next record location is the same as the first available	*/
      /* record location then there are no records in the file.		*/

      if (itsheader.Get_Next_Available() == itsnextrecord)
         {
	    pointer	= 0;		// No first record.
	 }
      else
         {
	    /* Use the next record function to do the work.		*/

	    result	= Get_Next_Record(record, pointer);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Next_Record(T& record, FilePointer& pointer)	*/
/*									*/
/* Purpose	This function can be used to return the next record in	*/
/*		the ObjectFile file.					*/
/*									*/
/*		NOTE: You must always call the function			*/
/*		'Get_First_Record' before you use this function to list	*/
/*		the file's records.					*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to a template object. This is where the	*/
/*		next record will be written. The variable 'pointer'	*/
/*		must contain a reference to a FilePointer. This is	*/
/*		where the record number for the next record will be	*/
/*		written.						*/
/*									*/
/* Ouput	If this function is able to get the next record in the	*/
/*		file, or if there are no more records in the file then	*/
/*		this function will return OK. If there were more	*/
/*		records in the file then the variable 'record' will	*/
/*		contain the data for the next record and the variable	*/
/*		'pointer' will contain the record pointer for the next	*/
/*		record. If there are no more records in the file then	*/
/*		the variable 'pointer' will contain 0. If this function	*/
/*		encounters an error then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Get_Next_Record(T& record, FilePointer& pointer)
   {
      status		result		= OK;
      ObjectFileRecHead	recordheader;

      /* This loop will look for the next record that is not deleted.	*/

      do
         {
	    /* Point to the next record in the file.			*/

	    pointer		= itsnextrecord;

	    /* Read the record in from the file.			*/

            if (recordheader.Read_Record_Header(itsfile, itsfilehandle,
               pointer) == ERROR)
	       {
	          /* Could not get next record.				*/

	          itserrorinfo	= "Attempted to get next ObjectFile "
				  "record from ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileNoGet;
	          result	= ERROR;
	          Report_Error();
		  break;
	       }

	    /* The current record pointer + the size of the record	*/
	    /* header + the size of the current record = the next	*/
	    /* record in the file.					*/

	    itsnextrecord	+= recordheader.Get_Record_Size()
				   + sizeof(ObjectFileRecHead);
	 }

      /* Loop while there are more records in the file AND the current	*/
      /* record is deleted.						*/

      while (itsnextrecord < itsheader.Get_Next_Available() &&
         recordheader.Is_Deleted() == true);

      /* If result == OK then we either have the next record or there	*/
      /* are no more records in the file.				*/

      if (result == OK)
         {
	    /* If the next record pointer is greater than the end of	*/
	    /* the file (next available) OR if the current record is	*/
	    /* deleted then there are no more records to get.		*/

            if (itsnextrecord > itsheader.Get_Next_Available() ||
               recordheader.Is_Deleted() == true)
               {
	          pointer	= 0;		// No next record.
	       }

	    /* Else, have the object read itself in from the file.	*/

            else if (record.Read_From_File(itsfile, itsfilehandle)
	       == ERROR)
               {
	          /* Could not get next record.				*/

	          itsfile.Report_Error();
	          itserrorinfo	= "Attempted to Get next ObjectFile "
				  "record from ";
	          itserrorinfo	+= itsfilename;
	          itserror	= ObjectFileNoGet;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Read_ObjectFile_Header(void) const		*/
/*									*/
/* Purpose	This function can be called to read in an ObjectFile	*/
/*		file's header. If a user's header area was specified	*/
/*		when the file was created and opened then it is also	*/
/*		read in from the file.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function was able to read in the file's header	*/
/*		from the beginning of the file then this function will	*/
/*		return OK. If this function was not able to read in the	*/
/*		file's header then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Read_ObjectFile_Header(void)
   {
      status		result		= OK;

      if (itsheader.Read_ObjectFile_Header(itsfile, itsfilehandle,
         itsuserheader) == ERROR)
         {
	    /* Could not read header.					*/

	    itserrorinfo	= "Attempted to read header from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileHeadRead;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_ObjectFile_Header(void) const		*/
/*									*/
/* Purpose	This function can be called to write out an ObjectFile	*/
/*		file's header. If a user's header area was specified	*/
/*		when the file was created and opened then it is also	*/
/*		written out to the file.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function was able to write out the file's	*/
/*		header to the beginning of the file then this function	*/
/*		will return OK. If this function was not able to write	*/
/*		out the file's header then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

template <class T>
status ObjectFile<T>::Write_ObjectFile_Header(void)
   {
      status		result		= OK;

      if (itsheader.Write_ObjectFile_Header(itsfile, itsfilehandle,
         itsuserheader) == ERROR)
         {
	    /* Could not read header.					*/

	    itserrorinfo	= "Attempted to read header from ";
	    itserrorinfo	+= itsfilename;
	    itserror		= ObjectFileHeadWrite;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

#endif				// OBJECTFILE_H
