#!/usr/bin/perl -w
######################################################################
#
# $Id: nph-webjob.cgi,v 1.128 2012/01/07 07:56:14 mavrik Exp $
#
######################################################################
#
# Copyright 2001-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Process webjob GET/JOB/PUT requests.
#
######################################################################

use strict;
use Fcntl qw(:flock);
use FindBin qw($Bin $RealBin); use lib ("$Bin/../../lib/perl5/site_perl", "$RealBin/../../lib/perl5/site_perl", "/usr/local/webjob/lib/perl5/site_perl", "/opt/local/webjob/lib/perl5/site_perl");
use Time::HiRes qw(gettimeofday tv_interval);
use WebJob::CgiRoutines;
use WebJob::JqdRoutines 1.029;
use WebJob::KvpRoutines 1.029;
use WebJob::LogRoutines;
use WebJob::Properties 1.042;
use WebJob::TimeRoutines 1.011;
use WebJob::ValidationRoutines 1.004;

######################################################################
#
# Main Routine
#
######################################################################

  my (%hProperties, %hReturnCodes, $sLocalError);

  %hReturnCodes =
  (
    '200' => "OK",
    '204' => "No Content",
    '206' => "Partial Content",
    '251' => "Link Test OK",
    '403' => "Forbidden",
    '404' => "Not Found",
    '405' => "Method Not Allowed",
    '409' => "Conflict",
    '412' => "Precondition Failed",
    '450' => "Invalid Query",
    '451' => "File Already Exists",
    '452' => "Username Undefined",
    '453' => "Username-ClientId Mismatch",
    '454' => "Content-Length Undefined",
    '455' => "Content-Length Exceeds Limit",
    '456' => "Content-Length Mismatch",
    '457' => "File Not Available",
    '458' => "Invalid Protocol",
    '459' => "Payload Signature Not Available",
    '460' => "Queue Not Active",
    '461' => "Queue Not Available",
    '462' => "Queue Not Mapped",
    '470' => "CommonName Undefined",
    '471' => "CommonName-ClientId Mismatch",
    '490' => "User-Defined 490",
    '491' => "User-Defined 491",
    '492' => "User-Defined 492",
    '493' => "User-Defined 493",
    '494' => "User-Defined 494",
    '495' => "User-Defined 495",
    '496' => "User-Defined 496",
    '497' => "User-Defined 497",
    '498' => "User-Defined 498",
    '499' => "User-Defined 499",
    '500' => "Internal Server Error",
    '503' => "Service Unavailable",
    '550' => "Internal Server Initialization Error",
    '551' => "Internal Server Mapping Error",
    '553' => "Service Disabled",
    '590' => "User-Defined 590",
    '591' => "User-Defined 591",
    '592' => "User-Defined 592",
    '593' => "User-Defined 593",
    '594' => "User-Defined 594",
    '595' => "User-Defined 595",
    '596' => "User-Defined 596",
    '597' => "User-Defined 597",
    '598' => "User-Defined 598",
    '599' => "User-Defined 599",
  );

  ####################################################################
  #
  # Punch in and go to work.
  #
  ####################################################################

  ($hProperties{'StartTime'}, $hProperties{'StartTimeUsec'}) = gettimeofday();
  $hProperties{'Program'} = "nph-webjob.cgi";
  $hProperties{'Version'} = sprintf("%s %s", __FILE__, ('$Revision: 1.128 $' =~ /^.Revision: ([\d.]+)/));
  $hProperties{'Pid'} = $$;

  ####################################################################
  #
  # Create/Verify run time environment, and process GET/PUT requests.
  #
  ####################################################################

  if (!defined(CreateRunTimeEnvironment(\%hProperties, \$sLocalError)))
  {
    $hProperties{'ReturnStatus'} = 550;
    $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
    $hProperties{'ErrorMessage'} = $sLocalError;
  }
  else
  {
    if (Yes($hProperties{'SslRequireSsl'}) && (!defined($hProperties{'Https'}) || $hProperties{'Https'} !~ /^[Oo][Nn]$/))
    {
      $hProperties{'ReturnStatus'} = 458;
      $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
      $hProperties{'ErrorMessage'} = "HTTPS required, but client is speaking HTTP";
    }
    else
    {
      if ($hProperties{'RequestMethod'} eq "GET")
      {
        $hProperties{'ReturnStatus'} = ProcessGetRequest(\%hProperties, \$sLocalError);
        $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
        $hProperties{'ErrorMessage'} = $sLocalError;
        $hProperties{'ExpandTriggerCommandLineRoutine'} = \&ExpandGetTriggerCommandLine;
      }
      elsif ($hProperties{'RequestMethod'} eq "PUT")
      {
        $hProperties{'ReturnStatus'} = ProcessPutRequest(\%hProperties, \$sLocalError);
        $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
        $hProperties{'ErrorMessage'} = $sLocalError;
        $hProperties{'ExpandTriggerCommandLineRoutine'} = \&ExpandPutTriggerCommandLine;
      }
      else
      {
        $hProperties{'ReturnStatus'} = 405;
        $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
        $hProperties{'ErrorMessage'} = "Method ($hProperties{'RequestMethod'}) not allowed";
      }
    }
  }
  $hProperties{'ServerContentLength'} = SendResponse(\%hProperties);

  ####################################################################
  #
  # Record the official stop time.
  #
  ####################################################################

  ($hProperties{'StopTime'}, $hProperties{'StopTimeUsec'}) = gettimeofday();

  ####################################################################
  #
  # Conditionally log/spool the request.
  #
  ####################################################################

  if (Yes($hProperties{'EnableLogging'}))
  {
    NphLogMessage(\%hProperties);
  }

  if (Yes($hProperties{'EnableRequestTracking'}))
  {
    if (!TrackerSpoolRequest(\%hProperties, \$sLocalError))
    {
      $hProperties{'TrackerMessage'} = $sLocalError;
      if (Yes($hProperties{'EnableLogging'}))
      {
        TrackerLogMessage(\%hProperties);
      }
    }
  }

  ####################################################################
  #
  # Conditionally pull the GET/PUT trigger and log the result.
  #
  ####################################################################

  if
  (
    $hProperties{'ReturnStatus'} == 200 &&
    (
      ($hProperties{'RequestMethod'} eq "GET" && Yes($hProperties{'GetTriggerEnable'})) ||
      ($hProperties{'RequestMethod'} eq "PUT" && Yes($hProperties{'PutTriggerEnable'}))
    )
  )
  {
    $hProperties{'TriggerEpoch'} = time();
    $hProperties{'TriggerPidLabel'} = "parent";
    $hProperties{'TriggerPid'} = $$;
    if (!defined(TriggerExecuteCommandLine(\%hProperties, \$sLocalError)))
    {
      $hProperties{'TriggerState'} = "failed";
      $hProperties{'TriggerMessage'} = $sLocalError;
      if (Yes($hProperties{'EnableLogging'}))
      {
        TriggerLogMessage(\%hProperties);
      }
    }
  }

  ####################################################################
  #
  # Clean up and go home.
  #
  ####################################################################

  1;


######################################################################
#
# CheckHostAccessList
#
######################################################################

sub CheckHostAccessList
{
  my ($sCidrList, $sIp) = @_;

  my $sCidrRegex = qq(((?:\\d{1,3}[.]){3}(?:\\d{1,3}))\/(\\d{1,2}));

  my $sIpRegex   = qq((?:\\d{1,3}[.]){3}(?:\\d{1,3}));

  if (!defined($sCidrList) || !defined($sIp))
  {
    return 0;
  }

  if ($sIp !~ /^$sIpRegex$/)
  {
    return 0;
  }

  $sCidrList =~ s/\s+//g; # Remove whitespace.

  foreach my $sCidr (split(/,/, $sCidrList))
  {
    next if ($sCidr !~ /^$sCidrRegex$/);
    my ($sNetwork, $sMaskBits) = ($1, $2);
    my $sBits2Clear = 32 - $sMaskBits - 1;
    my $sNetMask = 0xffffffff;
    foreach my $sBit (0..$sBits2Clear)
    {
      $sNetMask ^= 1 << $sBit;
    }
    my $sBinNetwork = hex(sprintf("%02x%02x%02x%02x", split(/\./, $sNetwork)));
    my $sBinNetmask = hex(sprintf("%08x", $sNetMask));
    $sBinNetwork &= $sBinNetmask; # Cleanup the network address -- in case the user did not.
    my $sBinIp = hex(sprintf("%02x%02x%02x%02x", split(/\./, $sIp)));
    if (($sBinIp & $sBinNetmask) == $sBinNetwork)
    {
      return 1; # We have a winner!
    }
  }

  return 0;
}


######################################################################
#
# ComputeJobTime
#
######################################################################

sub ComputeJobTime
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Extract the start time from the job ID. Since 0 is a valid result,
  # a value of -1 is returned on error to maintain a numeric response.
  # Note that the regular expression used here may need to be updated
  # if the job ID format ever changes.
  #
  ####################################################################

  my ($sStartTime, $sStartTimeUsec);

  if ($$phProperties{'JobId'} !~ /^[\w-]{1,64}_(\d{10})(?:_(\d{6}))?_\d{5}$/)
  {
    return -1;
  }
  $sStartTime = $1;
  $sStartTimeUsec = $2 || 0.000000;

  my $sJobTime = tv_interval([$sStartTime, $sStartTimeUsec]);
  if ($sJobTime < 0)
  {
    return -1;
  }

  return $sJobTime;
}


######################################################################
#
# CreateJobList
#
######################################################################

sub CreateJobList
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientId',
      'JobQueuePqActiveLimit',
      'JobQueuePqAnswerLimit',
      'JobQueueSqActiveLimit',
      'JobQueueSqAnswerLimit',
      'JqdJobCount',
      'JqdJobType',
      'JqdQueueDirectory',
      'JqdQueueName',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Initialize the job list.
  #
  ####################################################################

  $$phProperties{'JqdJobList'} = "";

  ####################################################################
  #
  # Acquire the change lock. This is an exclusive lock.
  #
  ####################################################################

  my (%hQueueLockArgs);

  %hQueueLockArgs =
  (
    'LockFile' => $$phProperties{'JqdQueueDirectory'} . "/" . "change.lock",
    'LockMode' => "+<", # The file must exist or this will fail.
  );
  if (!JqdLockFile(\%hQueueLockArgs))
  {
    $$psError = $hQueueLockArgs{'Error'};
    return undef;
  }

  ####################################################################
  #
  # Check to see if the queue has been frozen or locked.
  #
  ####################################################################

  if (JqdIsQueueFrozen({ 'Directory' => $$phProperties{'JqdQueueDirectory'} }))
  {
    $$psError = "Queue is frozen";
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  if (JqdIsQueueLocked({ 'Directory' => $$phProperties{'JqdQueueDirectory'} }))
  {
    $$psError = "Queue is locked";
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  ####################################################################
  #
  # Determine the number of active (sent/open) jobs.
  #
  ####################################################################

  my ($sOpenJobCount, $sSentJobCount);

  %hLArgs =
  (
    'Directory'      => $$phProperties{'JqdQueueDirectory'},
    'JobCount'       => 0,
    'JobFiles'       => [],
    'JobState'       => "sent",
    'JobType'        => $$phProperties{'JqdJobType'},
    'ReturnJobFiles' => 0,
  );
  $sSentJobCount = JqdGetQueuedJobs(\%hLArgs);
  if (!defined($sSentJobCount))
  {
    $$psError = $hLArgs{'Error'};
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  $hLArgs{'JobState'} = "open";
  $sOpenJobCount = JqdGetQueuedJobs(\%hLArgs);
  if (!defined($sOpenJobCount))
  {
    $$psError = $hLArgs{'Error'};
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  ####################################################################
  #
  # Determine the number of jobs that we are willing to cut loose.
  #
  ####################################################################

  my ($sActiveCount, $sActiveLimit, $sAnswerCount, $sAnswerLimit);

  $sActiveCount = $sSentJobCount + $sOpenJobCount;

  if ($$phProperties{'JqdJobType'} eq "serial")
  {
    $sActiveLimit = $$phProperties{'JobQueueSqActiveLimit'};
    $sAnswerLimit = $$phProperties{'JobQueueSqAnswerLimit'};
  }
  else
  {
    $sActiveLimit = $$phProperties{'JobQueuePqActiveLimit'};
    $sAnswerLimit = $$phProperties{'JobQueuePqAnswerLimit'};
  }
  $sAnswerCount = GetJobLimit($$phProperties{'JqdJobType'}, $$phProperties{'JqdJobCount'}, $sActiveCount, $sActiveLimit, $sAnswerLimit);
  if (!defined($sAnswerCount))
  {
    $$psError = substr($$phProperties{'JqdJobType'}, 0, 1) . "-queue is blocked with $sSentJobCount sent and $sOpenJobCount open jobs"; # This is put inside ()s by the caller.
    JqdUnlockFile(\%hQueueLockArgs);
    return "0/0"; # This case is not treated as an error.
  }

  ####################################################################
  #
  # Get the current list of jobs, but don't grab more than the count
  # just computed.
  #
  ####################################################################

  my (@aTodoJobs, $sTodoJobCount);

  %hLArgs =
  (
    'Directory'      => $$phProperties{'JqdQueueDirectory'},
    'JobCount'       => $sAnswerCount,
    'JobFiles'       => \@aTodoJobs,
    'JobState'       => "todo",
    'JobType'        => $$phProperties{'JqdJobType'},
    'ReturnJobFiles' => 1,
  );
  $hLArgs{'MinPriority'} = $$phProperties{'JqdMinPriority'} if (exists($$phProperties{'JqdMinPriority'}));
  $hLArgs{'MaxPriority'} = $$phProperties{'JqdMaxPriority'} if (exists($$phProperties{'JqdMaxPriority'}));
  $sTodoJobCount = JqdGetQueuedJobs(\%hLArgs);
  if (!defined($sTodoJobCount))
  {
    $$psError = $hLArgs{'Error'};
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  if ($sTodoJobCount == 0)
  {
    $$psError = substr($$phProperties{'JqdJobType'}, 0, 1) . "-queue is empty"; # This is put inside ()s by the caller.
    JqdUnlockFile(\%hQueueLockArgs);
    return "0/0"; # This case is not treated as an error.
  }

  ####################################################################
  #
  # Prepare the job list updating the queue state along the way. If
  # there's an error in the loop, stop processing and return any jobs
  # that successfully made it through the state transition. Check to
  # ensure that the job file is writable. This is done now because
  # state transitions later on need to insert additional content, so
  # it's better to fail here. This does two things: 1) it keeps the
  # job in the todo state and 2) it causes the queue to be frozen.
  #
  ####################################################################

  my (@aJobList, $sSafeJobCount);

  $sSafeJobCount = 0;

  foreach my $sTodoJob (@aTodoJobs)
  {
    my (%hJobProperties, %hLArgs, $sNextJob, $sJobId, $sQueueTag);
    if ($sTodoJob !~ /($$phProperties{'CommonRegexes'}{'JqdQueueTag'})$/)
    {
      $sTodoJobCount--; # Ignore junk in the queue, but decrement the job count to keep the books square.
      next;
    }
    $sJobId = $1;
    if (!-f $sTodoJob || !-W _)
    {
      $$psError = "The job file ($sTodoJob) does not exist or is not writable";
      last;
    }
    %hLArgs =
    (
      'File'           => $sTodoJob,
      'Properties'     => \%hJobProperties,
      'RequiredKeys'   => ['Command', 'CommandAlias', 'CommandLine', 'CommandMd5', 'CommandPath', 'CommandSha1', 'CommandSize', 'Created', 'Creator', 'JobGroup'],
      'RequireAllKeys' => 0,
      'Template'       => $$phProperties{'CommonTemplates'}{'jqd.job'},
      'VerifyValues'   => 1,
    );
    if (!KvpGetKvps(\%hLArgs))
    {
      $$psError = $hLArgs{'Error'};
      last;
    }
    ($sNextJob = $sTodoJob) =~ s/todo\/($$phProperties{'CommonRegexes'}{'JqdQueueTag'})$/sent\/$1/;
    $sQueueTag = $1;
    if (!rename($sTodoJob, $sNextJob))
    {
      $$psError = "Unable to move the job ($sTodoJob) to the next queue state ($!)";
      last;
    }
    else
    {
      my $sPoundName = sprintf("%s.%s", $hJobProperties{'CommandMd5'}, substr($hJobProperties{'CommandSha1'}, 0, 8));
      JqdLogMessage
      (
        {
          'Command'       => $hJobProperties{'Command'},
          'CommandSize'   => $hJobProperties{'CommandSize'},
          'Creator'       => $$phProperties{'ApacheUser'},
          'JobGroup'      => $hJobProperties{'JobGroup'},
          'LogFile'       => $$phProperties{'JqdLogFile'},
          'Message'       => "Job state changed successfully.",
          'NewQueueState' => "sent",
          'OldQueueState' => "todo",
          'Pid'           => $$phProperties{'Pid'},
          'PoundName'     => $sPoundName,
          'Program'       => $$phProperties{'Program'},
          'Queue'         => $$phProperties{'ClientId'},
          'QueueTag'      => $sQueueTag,
          'Result'        => "pass",
        }
      );
    }
    push(@aJobList, ($sJobId . "=" . $hJobProperties{'CommandLine'} . "\n"));
    $sSafeJobCount++;
  }

  $$phProperties{'JqdJobList'} = join("", @aJobList);

  ####################################################################
  #
  # Perform a sanity check on the results. If there was junk in the
  # queue, it's possible to get a part/whole of 0/0 here. If that's
  # the case, just report that the queue is empty.
  #
  ####################################################################

  if ($sSafeJobCount == 0 && $sTodoJobCount == 0)
  {
    $$psError = substr($$phProperties{'JqdJobType'}, 0, 1) . "-queue is empty"; # This is put inside ()s by the caller.
  }

  ####################################################################
  #
  # Release the change lock.
  #
  ####################################################################

  JqdUnlockFile(\%hQueueLockArgs);

  return "$sSafeJobCount/$sTodoJobCount";
}


######################################################################
#
# CreateRunTimeEnvironment
#
######################################################################

sub CreateRunTimeEnvironment
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Put input/output streams in binary mode.
  #
  ####################################################################

  foreach my $sHandle (\*STDIN, \*STDOUT, \*STDERR)
  {
    binmode($sHandle);
  }

  ####################################################################
  #
  # Initialize regex variables.
  #
  ####################################################################

  $$phProperties{'CommonRegexes'} = PropertiesGetGlobalRegexes();

  ####################################################################
  #
  # Initialize environment-specific variables. Pull in SSL-related
  # variables, but only if HTTPS is defined and on.
  #
  ####################################################################

  $$phProperties{'ContentLength'}  = $ENV{'CONTENT_LENGTH'};
  $$phProperties{'Https'}          = $ENV{'HTTPS'};
  $$phProperties{'QueryString'}    = $ENV{'QUERY_STRING'};
  $$phProperties{'RemoteAddress'}  = $ENV{'REMOTE_ADDR'};
  $$phProperties{'RemoteUser'}     = $ENV{'REMOTE_USER'};
  $$phProperties{'RequestMethod'}  = $ENV{'REQUEST_METHOD'} || "";
  $$phProperties{'ServerSoftware'} = $ENV{'SERVER_SOFTWARE'};
  $$phProperties{'PropertiesFile'} = $ENV{'WEBJOB_PROPERTIES_FILE'};

  if (defined($hProperties{'Https'}) && $hProperties{'Https'} =~ /^[Oo][Nn]$/)
  {
    $$phProperties{'SslClientSDnCn'} = $ENV{'SSL_CLIENT_S_DN_CN'} || "";
  }
  else
  {
    $$phProperties{'SslClientSDnCn'} = "";
  }

  ####################################################################
  #
  # Initialize platform-specific variables.
  #
  ####################################################################

  if ($^O =~ /MSWin32/i)
  {
    $$phProperties{'OsClass'} = "WINX";
    $$phProperties{'Newline'} = "\r\n";
    $$phProperties{'NullDevice'} = "nul";
  }
  else
  {
    $$phProperties{'OsClass'} = "UNIX";
    $$phProperties{'Newline'} = "\n";
    $$phProperties{'NullDevice'} = "/dev/null";
    umask(022);
  }

#FIXME Decide how to handle an error here. This is not an essential property.
  $$phProperties{'ApacheUser'} = (getpwuid($<))[0];

  ####################################################################
  #
  # Initialize site-specific variables. Note that the properties
  # listed in the custom template are a subset of those in the
  # global template. All values in the custom template may may be
  # overridden through the use of client- and/or command-specific
  # config files. See WebJob::Properties for the exact lists.
  #
  ####################################################################

  my ($sLocalError);

  $$phProperties{'CommonTemplates'} = PropertiesGetGlobalTemplates();

  $$phProperties{'GlobalConfigTemplate'} = $$phProperties{'CommonTemplates'}{'nph-webjob.global'};

  $$phProperties{'CustomConfigTemplate'} = $$phProperties{'CommonTemplates'}{'nph-webjob.custom'};

  if (!GetGlobalConfigProperties($phProperties, $$phProperties{'GlobalConfigTemplate'}, \$sLocalError))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Initialize derived variables. If key directories weren't defined
  # in the config file, define them relative to BaseDirectory.
  #
  ####################################################################

  foreach my $sName ("Config", "Dynamic", "Incoming", "Logfiles", "Profiles", "Spool")
  {
    my $sKey = $sName . "Directory";
    if (!exists($$phProperties{$sKey}) || !defined($$phProperties{$sKey}) || length($$phProperties{$sKey}) < 1)
    {
      $$phProperties{$sKey} = $$phProperties{'BaseDirectory'} . "/" . lc($sName);
    }
  }

  if (!exists($$phProperties{'JobQueueDirectory'}) || !defined($$phProperties{'JobQueueDirectory'}) || length($$phProperties{'JobQueueDirectory'}) < 1)
  {
    $$phProperties{'JobQueueDirectory'} = $$phProperties{'SpoolDirectory'} . "/" . "jqd";
  }

  $$phProperties{'NphConfigDirectory'} = $$phProperties{'ConfigDirectory'} . "/" . "nph-webjob";
  $$phProperties{'LogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob.log";
  $$phProperties{'HookErrFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob-hook.err";
  $$phProperties{'HookLogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob-hook.log";
  $$phProperties{'HookOutFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob-hook.out";
  $$phProperties{'JqdLogFile'} = $$phProperties{'LogfilesDirectory'} . "/jqd.log";
  $$phProperties{'TriggerLogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-webjob-trigger.log";

  ####################################################################
  #
  # Conditionally initialize the host access list.
  #
  ####################################################################

  if (Yes($$phProperties{'EnableHostAccessList'}))
  {
    if (!defined(SetupHostAccessList($phProperties, \$sLocalError)))
    {
      $$psError = $sLocalError;
      return undef;
    }
  }

  ####################################################################
  #
  # Verify run time environment.
  #
  ####################################################################

  if (!defined(VerifyRunTimeEnvironment($phProperties, $$phProperties{'GlobalConfigTemplate'}, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Initialize and verify the job ID.
  #
  ####################################################################

  if ($$phProperties{'RequestMethod'} eq "GET")
  {
    $$phProperties{'JobId'} = sprintf("%s_%010u_%06d_%05d", $$phProperties{'ServerId'}, $$phProperties{'StartTime'}, $$phProperties{'StartTimeUsec'}, $$);
  }
  else
  {
    $$phProperties{'JobId'} = $ENV{'HTTP_JOB_ID'} || "NA";
  }

  if (!defined($$phProperties{'JobId'}) || $$phProperties{'JobId'} !~ /^(?:NA|$$phProperties{'CommonRegexes'}{'JobId'})$/)
  {
    $$psError = "JobId ($$phProperties{'JobId'}) is undefined or invalid";
    return undef;
  }

  ####################################################################
  #
  # Initialize the job queue tag to a default value. This is needed
  # so that hooks and triggers have something to expand even if job
  # queues are disabled.
  #
  ####################################################################

  $$phProperties{'JqdQueueTag'} = "NA"; # Not Assigned.

  1;
}


######################################################################
#
# ExpandConversionString
#
######################################################################

sub ExpandConversionString
{
  my ($sConversionString, $phConversionValues, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sConversionString) || !defined($phConversionValues))
  {
    $$psError = "Unable to proceed in ExpandConversionString() due to missing or undefined inputs";
    return undef;
  }

  ####################################################################
  #
  # Expand the provided conversion string. The TokenList must be
  # processed in reverse order (i.e., from longest to shortest).
  # Otherwise, a token such as %pid would be interpreted as the token
  # %p followed by the literal string "id". Once all regular
  # conversions are done, check for and convert any literal '%'s.
  #
  ####################################################################

  my ($sExpandedConversionString, $sTokenList);

  $sTokenList = join('|', reverse(sort(keys(%$phConversionValues))));
  $sExpandedConversionString = $sConversionString;
  $sExpandedConversionString =~ s/%($sTokenList)/$$phConversionValues{$1}/ge;
  $sExpandedConversionString =~ s/%%/%/g;

  return $sExpandedConversionString;
}


######################################################################
#
# ExpandGetHookCommandLine
#
######################################################################

sub ExpandGetHookCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientFilename',
      'ClientId',
      'ClientSystem',
      'DynamicDirectory',
      'CommonRegexes',
      'GetHookCommandLine',
      'JobId',
      'JqdQueueTag',
      'RemoteAddress',
      'ServerId',
      'HookEpoch',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings,
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'HookEpoch'}) : localtime($$phProperties{'HookEpoch'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'cmd' => $$phProperties{'ClientFilename'},
    'd'   => sprintf("%02d", $sMonthDay),
    'dynamic_cmd' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".get",
    'dynamic_dir' => $$phProperties{'DynamicDirectory'},
    'dynamic_out' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".get",
    'dynamic_sig' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".get" . $$phProperties{'DsvSignatureSuffix'},
    'H'   => sprintf("%02d", $sHour),
    'ip'  => $$phProperties{'RemoteAddress'},
    'jid' => $$phProperties{'JobId'},
    'M'   => sprintf("%02d", $sMinute),
    'm'   => sprintf("%02d", $sMonth + 1),
    'pid' => sprintf("%05d", $$),
    'jqt' => $$phProperties{'JqdQueueTag'},
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'HookEpoch'}),
    'sid' => $$phProperties{'ServerId'},
    'system_version' => $$phProperties{'ClientSystem'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'cmd' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'dynamic_cmd' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_dir' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_out' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_sig' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'jid' => $$phProperties{'CommonRegexes'}{'JobId'},
    'jqt' => $$phProperties{'CommonRegexes'}{'JqdQueueTagOrNa'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'system_version' => $$phProperties{'CommonRegexes'}{'SystemVersion'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sHookCommandLine = ExpandConversionString($$phProperties{'GetHookCommandLine'}, \%hConversionValues, \$sLocalError);
  if (!defined($sHookCommandLine))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sHookCommandLine;
}


######################################################################
#
# ExpandGetTriggerCommandLine
#
######################################################################

sub ExpandGetTriggerCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientFilename',
      'ClientId',
      'CommonRegexes',
      'GetTriggerCommandLine',
      'JobId',
      'JqdQueueTag',
      'RemoteAddress',
      'ServerId',
      'TriggerEpoch',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings,
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'TriggerEpoch'}) : localtime($$phProperties{'TriggerEpoch'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'cmd' => $$phProperties{'ClientFilename'},
    'd'   => sprintf("%02d", $sMonthDay),
    'H'   => sprintf("%02d", $sHour),
    'ip'  => $$phProperties{'RemoteAddress'},
    'jid' => $$phProperties{'JobId'},
    'jqt' => $$phProperties{'JqdQueueTag'},
    'M'   => sprintf("%02d", $sMinute),
    'm'   => sprintf("%02d", $sMonth + 1),
    'pid' => sprintf("%05d", $$),
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'TriggerEpoch'}),
    'sid' => $$phProperties{'ServerId'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'cmd' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'jid' => $$phProperties{'CommonRegexes'}{'JobId'},
    'jqt' => $$phProperties{'CommonRegexes'}{'JqdQueueTagOrNa'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sTriggerCommandLine = ExpandConversionString($$phProperties{'GetTriggerCommandLine'}, \%hConversionValues, \$sLocalError);
  if (!defined($sTriggerCommandLine))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sTriggerCommandLine;
}


######################################################################
#
# ExpandPutHookCommandLine
#
######################################################################

sub ExpandPutHookCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientFilename',
      'ClientId',
      'ClientSystem',
      'CommonRegexes',
      'DynamicDirectory',
      'EnvFile',
      'ErrFile',
      'HookEpoch',
      'LckFile',
      'PutHookCommandLine',
      'JobId',
      'JobTime',
      'JqdQueueTag',
      'OutFile',
      'RdyFile',
      'RemoteAddress',
      'ServerId',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings,
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'HookEpoch'}) : localtime($$phProperties{'HookEpoch'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'cmd' => $$phProperties{'ClientFilename'},
    'd'   => sprintf("%02d", $sMonthDay),
    'dynamic_dir' => $$phProperties{'DynamicDirectory'},
    'dynamic_out' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".put",
    'dynamic_sig' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".put" . $$phProperties{'DsvSignatureSuffix'},
    'env' => $$phProperties{'EnvFile'},
    'err' => $$phProperties{'ErrFile'},
    'H'   => sprintf("%02d", $sHour),
    'ip'  => $$phProperties{'RemoteAddress'},
    'jid' => $$phProperties{'JobId'},
    'job_time' => $$phProperties{'JobTime'},
    'jqt' => $$phProperties{'JqdQueueTag'},
    'lck' => $$phProperties{'LckFile'},
    'M'   => sprintf("%02d", $sMinute),
    'm'   => sprintf("%02d", $sMonth + 1),
    'out' => $$phProperties{'OutFile'},
    'pid' => sprintf("%05d", $$),
    'rdy' => $$phProperties{'RdyFile'},
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'HookEpoch'}),
    'sid' => $$phProperties{'ServerId'},
    'system_version' => $$phProperties{'ClientSystem'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'cmd' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'dynamic_dir' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_out' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_sig' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'env' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'err' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'jid' => $$phProperties{'CommonRegexes'}{'JobId'},
    'job_time' => $$phProperties{'CommonRegexes'}{'DecimalFloatSigned'},
    'jqt' => $$phProperties{'CommonRegexes'}{'JqdQueueTagOrNa'},
    'lck' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'out' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'rdy' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'system_version' => $$phProperties{'CommonRegexes'}{'SystemVersion'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sHookCommandLine = ExpandConversionString($$phProperties{'PutHookCommandLine'}, \%hConversionValues, \$sLocalError);
  if (!defined($sHookCommandLine))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sHookCommandLine;
}


######################################################################
#
# ExpandPutTriggerCommandLine
#
######################################################################

sub ExpandPutTriggerCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientFilename',
      'ClientId',
      'CommonRegexes',
      'EnvFile',
      'ErrFile',
      'LckFile',
      'PutTriggerCommandLine',
      'JobId',
      'JobTime',
      'JqdQueueTag',
      'OutFile',
      'RdyFile',
      'RemoteAddress',
      'ServerId',
      'TriggerEpoch',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings,
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'TriggerEpoch'}) : localtime($$phProperties{'TriggerEpoch'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'cmd' => $$phProperties{'ClientFilename'},
    'd'   => sprintf("%02d", $sMonthDay),
    'env' => $$phProperties{'EnvFile'},
    'err' => $$phProperties{'ErrFile'},
    'H'   => sprintf("%02d", $sHour),
    'ip'  => $$phProperties{'RemoteAddress'},
    'jid' => $$phProperties{'JobId'},
    'job_time' => $$phProperties{'JobTime'},
    'jqt' => $$phProperties{'JqdQueueTag'},
    'lck' => $$phProperties{'LckFile'},
    'M'   => sprintf("%02d", $sMinute),
    'm'   => sprintf("%02d", $sMonth + 1),
    'out' => $$phProperties{'OutFile'},
    'pid' => sprintf("%05d", $$),
    'rdy' => $$phProperties{'RdyFile'},
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'TriggerEpoch'}),
    'sid' => $$phProperties{'ServerId'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'cmd' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'env' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'err' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'jid' => $$phProperties{'CommonRegexes'}{'JobId'},
    'job_time' => $$phProperties{'CommonRegexes'}{'DecimalFloatSigned'},
    'jqt' => $$phProperties{'CommonRegexes'}{'JqdQueueTagOrNa'},
    'lck' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'out' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'rdy' => $$phProperties{'CommonRegexes'}{'ServerSuppliedPath'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sTriggerCommandLine = ExpandConversionString($$phProperties{'PutTriggerCommandLine'}, \%hConversionValues, \$sLocalError);
  if (!defined($sTriggerCommandLine))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sTriggerCommandLine;
}


######################################################################
#
# GetContentHandle
#
######################################################################

sub GetContentHandle
{
  my ($phProperties, $sContentFile, $sSigFile, $psError) = @_;

  my $sLocalError;
  if (-f $sSigFile && open(FH, "< $sSigFile"))
  {
    binmode(FH);
    my $sSignature = <FH>; # This file should only contain one line.
    close(FH);
    if (defined($sSignature))
    {
      $sSignature =~ s/[\r\n]*$//;
      if ($sSignature =~ /^[0-9A-Za-z+\/]{1,$$phProperties{'DsvMaxSignatureLength'}}={0,2}$/o)
      {
        $$phProperties{'DsvPayloadSignature'} = $sSignature;
      }
      else
      {
        $sLocalError = "value does not pass muster";
      }
    }
    else
    {
      $sLocalError = "value is not defined";
    }
  }
  else
  {
    $sLocalError = $!;
  }
  if (Yes($$phProperties{'DsvRequireSignatures'}) && defined($sLocalError))
  {
    $$psError = "Signature file ($sSigFile) does not exist or contains a malformed signature ($sLocalError)";
    return (undef, 459);
  }
  if (!open(FH, "< $sContentFile"))
  {
    $$psError = "Requested file ($sContentFile) could not be opened ($!)";
    return (undef, 457);
  }
  binmode(FH);

  return (\*FH, undef);
}


######################################################################
#
# GetCustomConfigProperties
#
######################################################################

sub GetCustomConfigProperties
{
  my ($phProperties, $phTemplate, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  foreach my $sInput ($phProperties, $phTemplate, $psError)
  {
    if (!defined($sInput))
    {
      $$psError = "Unable to proceed in GetCustomConfigProperties() due to missing or undefined inputs" if (defined($psError));
      return undef;
    }
  }

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientId',
      'ConfigSearchOrder',
      'CommonRegexes',
      'EnableJobQueues',
      ($$phProperties{'RequestMethod'} eq "JOB") ? 'JqdQueueName' : 'ClientFilename',
      'NphConfigDirectory',
      'RequestMethod',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Search for custom config files using the specified order. As each
  # config file is processed, its values trump those of any that came
  # before -- including any values that came from global config files.
  #
  ####################################################################

  foreach my $sFolder (split(/:/, $$phProperties{'ConfigSearchOrder'}))
  {
    ##################################################################
    #
    # The first config file defined by this loop applies globally.
    # The second applies to a particular client or command, and the
    # third applies to a particular client/command pair. Queues, on
    # the other hand, only use global and per queue config files.
    #
    ##################################################################

    my ($sFile1, $sFile2, $sFile3);

    $sFile1 = $sFile2 = $sFile3 = $$phProperties{'NphConfigDirectory'} . "/" . $sFolder . "/";

    if ($sFolder =~ /^clients$/)
    {
      $sFile1 .= "nph-webjob.cfg";
      $sFile2 .= $$phProperties{'ClientId'} . "/" . "nph-webjob.cfg";
      $sFile3 .= $$phProperties{'ClientId'} . "/" . $$phProperties{'ClientFilename'} . "/" . "nph-webjob.cfg";
    }
    elsif ($sFolder =~ /^commands$/)
    {
      $sFile1 .= "nph-webjob.cfg";
      $sFile2 .= $$phProperties{'ClientFilename'} . "/" . "nph-webjob.cfg";
      $sFile3 .= $$phProperties{'ClientFilename'} . "/" . $$phProperties{'ClientId'} . "/" . "nph-webjob.cfg";
    }
    elsif ($sFolder =~ /^queues$/)
    {
      $sFile1 .= "nph-webjob.cfg";
      $sFile2 .= $$phProperties{'JqdQueueName'} . "/" . "nph-webjob.cfg";
      $sFile3 = undef; # This is not used for queues.
    }
    else
    {
      next; # Ignore invalid directories.
    }

    foreach my $sFile ($sFile1, $sFile2, $sFile3)
    {
      next if (!defined($sFile) || !-f $sFile);

      ################################################################
      #
      # Pull in any externally defined properties according to the
      # specified template. If the template is empty, no properties
      # will be returned.
      #
      ################################################################

      my (%hLArgs, %hCustomProperties);

      %hLArgs =
      (
        'File'             => $sFile,
        'Properties'       => \%hCustomProperties,
        'RequireAllKeys'   => 0,
        'RequireKnownKeys' => 1,
        'Template'         => $phTemplate,
        'VerifyValues'     => 1,
      );
      if (!KvpGetKvps(\%hLArgs))
      {
        $$psError = $hLArgs{'Error'};
        return undef;
      }

      ################################################################
      #
      # Transfer validated properties, if any, to the main hash. This
      # is where the trump action takes place.
      #
      ################################################################

      foreach my $sProperty (keys(%hCustomProperties))
      {
        $$phProperties{$sProperty} = $hCustomProperties{$sProperty};
      }
    }
  }

  1;
}


######################################################################
#
# GetEffectiveCidrList
#
######################################################################

sub GetEffectiveCidrList
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Since ClientId and RemoteUser may be different when RequireMatch
  # is disabled, all assigned CIDR values are considered valid.
  #
  ####################################################################

  my ($sCidrList, $sTempList);

  $sCidrList = "" . $$phProperties{'AccessList'}{$$phProperties{'ClientId'}};

  if (!Yes($$phProperties{'RequireMatch'}))
  {
    $sTempList = "" . $$phProperties{'AccessList'}{$$phProperties{'RemoteUser'}};
    if (length($sCidrList) > 0 && length($sTempList) > 0)
    {
      $sCidrList .= "," . $sTempList;
    }
    else
    {
      $sCidrList .= $sTempList;
    }
  }

  return $sCidrList;
}


######################################################################
#
# GetGlobalConfigProperties
#
######################################################################

sub GetGlobalConfigProperties
{
  my ($phProperties, $phTemplate, $psError) = @_;

  ####################################################################
  #
  # Initialize regular properties with their default values. Note that
  # derived properties are initialized in CreateRunTimeEnvironment().
  #
  ####################################################################

  $$phProperties{'BaseDirectory'} = "/var/webjob";
  $$phProperties{'CapContentLength'} = "N"; # [Y|N]
  $$phProperties{'ConfigSearchOrder'} = "clients:commands";
  $$phProperties{'DsvMaxSignatureLength'} = 256;
  $$phProperties{'DsvNullSignature'} = "";
  $$phProperties{'DsvRequireSignatures'} = "N"; # [Y|N]
  $$phProperties{'DsvSignatureSuffix'} = ".sig";
  $$phProperties{'EnableGetService'} = "Y"; # [Y|N]
  $$phProperties{'EnableHostAccessList'} = "N"; # [Y|N]
  $$phProperties{'EnableConfigOverrides'} = "Y"; # [Y|N]
  $$phProperties{'EnableJobQueues'} = "Y"; # [Y|N]
  $$phProperties{'EnableLogging'} = "Y"; # [Y|N]
  $$phProperties{'EnablePutService'} = "Y"; # [Y|N]
  $$phProperties{'EnableRequestTracking'} = "N"; # [Y|N]
  $$phProperties{'FolderList'} = "common";
  $$phProperties{'GetHookCommandLine'} = "";
  $$phProperties{'GetHookLogDivertedOutput'} = "N"; # [Y|N]
  $$phProperties{'GetHookEnable'} = "N"; # [Y|N]
  $$phProperties{'GetHookStatus'} = 0;
  $$phProperties{'GetHookStatusMap'} = "";
  $$phProperties{'GetTriggerCommandLine'} = "";
  $$phProperties{'GetTriggerEnable'} = "N"; # [Y|N]
  $$phProperties{'JobQueueActive'} = "Y"; # [Y|N]
  $$phProperties{'JobQueuePqActiveLimit'} = 0;
  $$phProperties{'JobQueuePqAnswerLimit'} = 0;
  $$phProperties{'JobQueueSqActiveLimit'} = 1;
  $$phProperties{'JobQueueSqAnswerLimit'} = 0;
  $$phProperties{'MaxContentLength'} = 100000000;
  $$phProperties{'OverwriteExistingFiles'} = "N"; # [Y|N]
  $$phProperties{'PutHookCommandLine'} = "";
  $$phProperties{'PutHookLogDivertedOutput'} = "N"; # [Y|N]
  $$phProperties{'PutHookEnable'} = "N"; # [Y|N]
  $$phProperties{'PutHookStatus'} = 0;
  $$phProperties{'PutHookStatusMap'} = "";
  $$phProperties{'PutNameFormat'} = "%cid/%cmd/%Y-%m-%d/%H.%M.%S.%pid";
  $$phProperties{'PutTriggerCommandLine'} = "";
  $$phProperties{'PutTriggerEnable'} = "N"; # [Y|N]
  $$phProperties{'RequireMatch'} = "Y"; # [Y|N]
  $$phProperties{'RequireUser'} = "Y"; # [Y|N]
  $$phProperties{'ServerId'} = "server_1";
  $$phProperties{'SslRequireCn'} = "N"; # [Y|N]
  $$phProperties{'SslRequireMatch'} = "N"; # [Y|N]
  $$phProperties{'SslRequireSsl'} = "Y"; # [Y|N]
  $$phProperties{'UseGmt'} = "N"; # [Y|N]

  ####################################################################
  #
  # Pull in any externally defined properties. These properties trump
  # internally defined properties.
  #
  ####################################################################

  my (%hLArgs);

  if (!exists($$phProperties{'PropertiesFile'}) || !defined($$phProperties{'PropertiesFile'}))
  {
    $$phProperties{'PropertiesFile'} = $$phProperties{'BaseDirectory'} . "/config/nph-webjob/nph-webjob.cfg";
  }

  %hLArgs =
  (
    'File'             => $$phProperties{'PropertiesFile'},
    'Properties'       => $phProperties,
    'RequireAllKeys'   => 0,
    'RequireKnownKeys' => 1,
    'Template'         => $phTemplate,
    'VerifyValues'     => 1,
  );
  if (!KvpGetKvps(\%hLArgs))
  {
    $$psError = $hLArgs{'Error'};
    return undef;
  }

  1;
}


######################################################################
#
# GetJobLimit
#
######################################################################

sub GetJobLimit
{
  my ($sJobType, $sRequestCount, $sActiveCount, $sActiveLimit, $sAnswerLimit) = @_;

  if ($sJobType eq "serial")
  {
    if ($sRequestCount == 0 && $sActiveLimit == 0 && $sAnswerLimit == 0)
    {
      return 0;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit == 0 && $sAnswerLimit > 0)
    {
      return $sAnswerLimit;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit > 0 && $sAnswerLimit == 0)
    {
      return ($sActiveCount < $sActiveLimit) ? 0 : undef;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit > 0 && $sAnswerLimit > 0)
    {
      return ($sActiveCount < $sActiveLimit) ? $sAnswerLimit : undef;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit == 0 && $sAnswerLimit == 0)
    {
      return $sRequestCount;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit == 0 && $sAnswerLimit > 0)
    {
      return MinValue($sRequestCount, $sAnswerLimit);
    }
    elsif ($sRequestCount > 0 && $sActiveLimit > 0 && $sAnswerLimit == 0)
    {
      return ($sActiveCount < $sActiveLimit) ? $sRequestCount : undef;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit > 0 && $sAnswerLimit > 0)
    {
      return ($sActiveCount < $sActiveLimit) ? MinValue($sRequestCount, $sAnswerLimit) : undef;
    }
  }
  else
  {
    if ($sRequestCount == 0 && $sActiveLimit == 0 && $sAnswerLimit == 0)
    {
      return 0;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit == 0 && $sAnswerLimit > 0)
    {
      return $sAnswerLimit;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit > 0 && $sAnswerLimit == 0)
    {
      return ($sActiveCount < $sActiveLimit) ? $sActiveLimit - $sActiveCount : undef;
    }
    elsif ($sRequestCount == 0 && $sActiveLimit > 0 && $sAnswerLimit > 0)
    {
      return ($sActiveCount < $sActiveLimit) ? MinValue($sActiveLimit - $sActiveCount, $sAnswerLimit) : undef;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit == 0 && $sAnswerLimit == 0)
    {
      return $sRequestCount;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit == 0 && $sAnswerLimit > 0)
    {
      return MinValue($sRequestCount, $sAnswerLimit);
    }
    elsif ($sRequestCount > 0 && $sActiveLimit > 0 && $sAnswerLimit == 0)
    {
      return ($sActiveCount < $sActiveLimit) ? MinValue($sActiveLimit - $sActiveCount, $sRequestCount) : undef;
    }
    elsif ($sRequestCount > 0 && $sActiveLimit > 0 && $sAnswerLimit > 0)
    {
      return ($sActiveCount < $sActiveLimit) ? MinValue($sActiveLimit - $sActiveCount, MinValue($sRequestCount, $sAnswerLimit)) : undef;
    }
  }
}


######################################################################
#
# GetMappedValue
#
######################################################################

sub GetMappedValue
{
  my ($sStatusMap, $sMapKey) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sStatusMap) || !defined($sMapKey))
  {
    return undef;
  }

  ####################################################################
  #
  # Build a map and return the requested value.
  #
  ####################################################################

  my (%hKvps);

  foreach my $sKvp (split(/,/, $sStatusMap))
  {
    my ($sKey, $sValue) = split(/:/, $sKvp);
    $hKvps{$sKey} = $sValue;
  }

  return (exists($hKvps{$sMapKey})) ? $hKvps{$sMapKey} : undef;
}


######################################################################
#
# HookExecuteCommandLine
#
######################################################################

sub HookExecuteCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'EnableLogging',
      'ExpandHookCommandLineRoutine',
      'OsClass',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Expand the hook's command line. If the result is undefined or
  # null, abort.
  #
  ####################################################################

  my ($sLocalError);

  $$phProperties{'HookEpoch'} = time();

  $$phProperties{'HookCommandLine'} = &{$$phProperties{'ExpandHookCommandLineRoutine'}}($phProperties, \$sLocalError);
  if (!defined($$phProperties{'HookCommandLine'}))
  {
    $$psError = $sLocalError;
    return undef;
  }

  if (!length($$phProperties{'HookCommandLine'}))
  {
    $$psError = "Command line is undefined or null";
    return undef;
  }

  ####################################################################
  #
  # Create directories, as necessary, to hold the dynamic content.
  #
  ####################################################################

  my ($sMode, $sPath);

  $sMode = 0755;
  $sPath = $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'};
  if (!-d $sPath)
  {
    if (!mkdir($sPath, $sMode))
    {
      $$psError = "Directory ($sPath) could not be created ($!)";
      return undef;
    }
  }

  $sPath .= "/" . "jids";
  if (!-d $sPath)
  {
    if (!mkdir($sPath, $sMode))
    {
      $$psError = "Directory ($sPath) could not be created ($!)";
      return undef;
    }
  }

  ####################################################################
  #
  # Dup and divert stdout/stderr. This must be done to prevent the
  # response from becoming polluted by spurious hook output.
  #
  ####################################################################

  my ($sStdErrFile, $sStdOutFile);

  if (!open(STDERR_DUP, ">&STDERR"))
  {
    $$psError = "Unable to dup stderr ($!)";
    return undef;
  }
  if (!open(STDOUT_DUP, ">&STDOUT"))
  {
    $$psError = "Unable to dup stdout ($!)";
    return undef;
  }

  if
  (
    Yes($$phProperties{'EnableLogging'}) &&
    (
         ($$phProperties{'RequestMethod'} eq "GET" && Yes($$phProperties{'GetHookLogDivertedOutput'}))
      || ($$phProperties{'RequestMethod'} eq "PUT" && Yes($$phProperties{'PutHookLogDivertedOutput'}))
    )
  )
  {
    $sStdErrFile = $$phProperties{'HookErrFile'};
    $sStdOutFile = $$phProperties{'HookOutFile'};
  }
  else
  {
    $sStdErrFile = $$phProperties{'NullDevice'};
    $sStdOutFile = $$phProperties{'NullDevice'};
  }

  if (!open(STDERR, ">> $sStdErrFile"))
  {
    $$psError = "Unable to divert stderr to $sStdErrFile ($!)";
    return undef;
  }
  if (!open(STDOUT, ">> $sStdOutFile"))
  {
    $$psError = "Unable to divert stdout to $sStdOutFile ($!)";
    return undef;
  }

  ####################################################################
  #
  # Execute the hook.
  #
  ####################################################################

  $$phProperties{'HookPidLabel'} = "parent";
  $$phProperties{'HookPid'} = $$;
  $$phProperties{'HookState'} = "hooked";
  $$phProperties{'HookMessage'} = $$phProperties{'HookCommandLine'};
  if (Yes($$phProperties{'EnableLogging'}))
  {
    HookLogMessage($phProperties);
  }
  my $sHookReturn = system($$phProperties{'HookCommandLine'});
  my $sHookStatus = ($sHookReturn >> 8) & 0xff;
  my $sHookSignal = ($sHookReturn & 0x7f);
  my $sHookDumped = ($sHookReturn & 0x80) ? 1 : 0;
  if ($sHookStatus == 255)
  {
    $$phProperties{'HookState'} = "failed";
    $$phProperties{'HookMessage'} = "Unable to execute hook command ($!)";
  }
  else
  {
    $$phProperties{'HookState'} = "reaped";
    $$phProperties{'HookMessage'} = "status($sHookStatus) signal($sHookSignal) coredump($sHookDumped)";
  }
  if (Yes($$phProperties{'EnableLogging'}))
  {
    HookLogMessage($phProperties);
  }

  ####################################################################
  #
  # Restore stdout/stderr and close the duplicate handles.
  #
  ####################################################################

  if (!open(STDERR, ">&STDERR_DUP"))
  {
    $$psError = "Unable to restore stderr ($!)";
    return undef;
  }
  if (!open(STDOUT, ">&STDOUT_DUP"))
  {
    $$psError = "Unable to restore stdout ($!)";
    return undef;
  }
  close(STDERR_DUP);
  close(STDOUT_DUP);

  return $sHookStatus;
}


######################################################################
#
# HookLogMessage
#
######################################################################

sub HookLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Marshall log arguments.
  #
  ####################################################################

  my (%hLArgs);

  $hLArgs{'LogEpoch'} = $$phProperties{'HookEpoch'};

  $hLArgs{'LogFields'} =
  [
    'JobId',
    'RequestMethod',
    'ClientId',
    'ClientFilename',
    'HookPidLabel',
    'HookPid',
    'HookState',
    'Message',
  ];

  $hLArgs{'LogValues'} =
  {
    'JobId'               => $$phProperties{'JobId'},
    'RequestMethod'       => $$phProperties{'RequestMethod'},
    'ClientId'            => $$phProperties{'ClientId'},
    'ClientFilename'      => $$phProperties{'ClientFilename'},
    'HookPidLabel'        => $$phProperties{'HookPidLabel'},
    'HookPid'             => $$phProperties{'HookPid'},
    'HookState'           => $$phProperties{'HookState'},
    'Message'             => $$phProperties{'HookMessage'},
  };

  $hLArgs{'LogFile'} = $$phProperties{'HookLogFile'};

  $hLArgs{'Newline'} = $$phProperties{'Newline'};

  $hLArgs{'RevertToStderr'} = 1;

  $hLArgs{'UseGmt'} = (Yes($$phProperties{'UseGmt'})) ? 1 : 0;

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  if (!LogNf1vMessage(\%hLArgs))
  {
    print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'HookMessage'}'\n";
    print STDERR "$$phProperties{'Program'}: Error='$hLArgs{'Error'}'\n";
  }

  1;
}


######################################################################
#
# MakePutName
#
######################################################################

sub MakePutName
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientFilename',
      'ClientId',
      'CommonRegexes',
      'PutNameFormat',
      'RemoteAddress',
      'ServerId',
      'StartTime',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'StartTime'}) : localtime($$phProperties{'StartTime'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'cmd' => $$phProperties{'ClientFilename'},
    'ip'  => $$phProperties{'RemoteAddress'},
    'm'   => sprintf("%02d", $sMonth + 1),
    'd'   => sprintf("%02d", $sMonthDay),
    'H'   => sprintf("%02d", $sHour),
    'M'   => sprintf("%02d", $sMinute),
    'pid' => sprintf("%05d", $$),
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'StartTime'}),
    'sid' => $$phProperties{'ServerId'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'cmd' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sPutName = ExpandConversionString($$phProperties{'PutNameFormat'}, \%hConversionValues, \$sLocalError);
  if (!defined($sPutName))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sPutName;
}


######################################################################
#
# MakePutTree
#
######################################################################

sub MakePutTree
{
  my ($sIncomingDirectory, $sPutName, $sMode, $sPopCount, $psError) = @_;

  ####################################################################
  #
  # Pop the specified number of elements from PutName. Normally, only
  # the trailing filename is removed (i.e. PopCount = 1).
  #
  ####################################################################

  my (@aElements);

  @aElements = split(/[\/\\]/, $sPutName);

  while (defined($sPopCount) && $sPopCount-- > 0)
  {
    pop(@aElements);
  }

  ####################################################################
  #
  # Create the tree -- one element at a time. If the mkdir() fails,
  # test the directory's existence again -- another process may have
  # beat us to the punch, and that is not a fatal error.
  #
  ####################################################################

  my ($sLocalError, $sPath);

  $sPath = $sIncomingDirectory;

  foreach my $sElement (@aElements)
  {
    $sPath .= "/$sElement";
    foreach my $sTry (0..1)
    {
      if (!-d $sPath && !mkdir($sPath, $sMode))
      {
        $sLocalError = "Directory ($sPath) could not be created ($!)";
      }
      else
      {
        $sLocalError = undef;
        last;
      }
    }
    if (defined($sLocalError))
    {
      $$psError = $sLocalError;
      return undef;
    }
  }

  1;
}


######################################################################
#
# MinValue
#
######################################################################

sub MinValue
{
  return ($_[0] <= $_[1]) ? $_[0] : $_[1];
}


######################################################################
#
# NphLogMessage
#
######################################################################

sub NphLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Marshall log arguments.
  #
  ####################################################################

  my (%hLArgs, $sVariableKey);

  ($hLArgs{'LogEpoch'}, $hLArgs{'LogEpochUsec'}) = ($$phProperties{'StopTime'}, $$phProperties{'StopTimeUsec'});

  $sVariableKey = ($$phProperties{'RequestMethod'} eq "JOB") ? "JqdQueueName" : "ClientFilename";

  $hLArgs{'LogFields'} =
  [
    'JobId',
    'RemoteUser',
    'RemoteAddress',
    'RequestMethod',
    'ClientId',
    $sVariableKey,
    'ContentLength',
    'ServerContentLength',
    'Duration',
    'ReturnStatus',
    'Message'
  ];

  $hLArgs{'LogValues'} =
  {
    'JobId'               => $$phProperties{'JobId'},
    'RemoteUser'          => $$phProperties{'RemoteUser'},
    'RemoteAddress'       => $$phProperties{'RemoteAddress'},
    'RequestMethod'       => $$phProperties{'RequestMethod'},
    'ClientId'            => $$phProperties{'ClientId'},
    $sVariableKey         => $$phProperties{$sVariableKey},
    'ContentLength'       => $$phProperties{'ContentLength'},
    'ServerContentLength' => $$phProperties{'ServerContentLength'},
    'Duration'            => sprintf("%.6f", tv_interval([$$phProperties{'StartTime'}, $$phProperties{'StartTimeUsec'}], [$$phProperties{'StopTime'}, $$phProperties{'StopTimeUsec'}])),
    'ReturnStatus'        => $$phProperties{'ReturnStatus'},
    'Message'             => $$phProperties{'ErrorMessage'},
  };

  $hLArgs{'LogFile'} = $$phProperties{'LogFile'};

  $hLArgs{'Newline'} = $$phProperties{'Newline'};

  $hLArgs{'RevertToStderr'} = 1;

  $hLArgs{'UseGmt'} = (Yes($$phProperties{'UseGmt'})) ? 1 : 0;

  ####################################################################
  #
  # Deliver log message. If the operation fails, deliver the original
  # log message plus the current error message to STDERR.
  #
  ####################################################################

  if (!LogNf1vMessage(\%hLArgs))
  {
    print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'ErrorMessage'}'\n";
    print STDERR "$$phProperties{'Program'}: Error='$hLArgs{'Error'}'\n";
  }

  1;
}


######################################################################
#
# ProcessGetRequest
#
######################################################################

sub ProcessGetRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Proceed only if QueryString matches the GetQuery expression.
  #
  ####################################################################

  my %hParameters = ();
#FIXME Do we care if the same parameter is specified more than once? The last one in wins.
#FIXME Do we want to impose a limit on the number of parameters or the length of the query string?
  CgiParseQueryString($$phProperties{'QueryString'}, \%hParameters);

  if
  (
       VerifyHashKeyValue(\%hParameters, 'VERSION', $$phProperties{'CommonRegexes'}{'WebJobVersion'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'SYSTEM', $$phProperties{'CommonRegexes'}{'SystemVersion'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'CLIENTID', $$phProperties{'CommonRegexes'}{'ClientId'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'FILENAME', $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'}, 3)
  )
  {
    $$phProperties{'ClientVersion'}  = $hParameters{'VERSION'};
    $$phProperties{'ClientSystem'}   = $hParameters{'SYSTEM'};
    $$phProperties{'ClientId'}       = $hParameters{'CLIENTID'} || "nobody";
    $$phProperties{'ClientFilename'} = $hParameters{'FILENAME'};

    ##################################################################
    #
    # The QUEUETAG parameter is optional. Also, it's not supported by
    # legacy clients (i.e., versions less than 0x10800008).
    #
    ##################################################################

    if (VerifyHashKeyValue(\%hParameters, 'QUEUETAG', "(?:|$$phProperties{'CommonRegexes'}{'JqdQueueTag'})", 3))
    {
      $$phProperties{'QueueTag'} = $hParameters{'QUEUETAG'};
    }
    else
    {
      $$phProperties{'QueueTag'} = "";
    }

    ##################################################################
    #
    # Bring in any client- and/or command-specific properties.
    #
    ##################################################################

    my ($sLocalError);

    if (Yes($$phProperties{'EnableConfigOverrides'}))
    {
      if (!GetCustomConfigProperties($phProperties, \%{$$phProperties{'CustomConfigTemplate'}}, \$sLocalError))
      {
        $$psError = $sLocalError;
        return 500;
      }
    }

    ##################################################################
    #
    # Conditionally do CommonName and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'SslRequireSsl'}))
    {
      if (Yes($$phProperties{'SslRequireCn'}) && (!defined($$phProperties{'SslClientSDnCn'}) || !length($$phProperties{'SslClientSDnCn'})))
      {
        $$psError = "CommonName is undefined or null";
        return 470;
      }

      if (Yes($$phProperties{'SslRequireCn'}) && Yes($$phProperties{'SslRequireMatch'}) && $$phProperties{'SslClientSDnCn'} ne $$phProperties{'ClientId'})
      {
        $$psError = "CommonName ($$phProperties{'SslClientSDnCn'}) does not match client ID ($$phProperties{'ClientId'})";
        return 471;
      }
    }

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'RequireUser'}) && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if (Yes($$phProperties{'RequireUser'}) && Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do host access list checks.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableHostAccessList'}) && !CheckHostAccessList(GetEffectiveCidrList($phProperties), $$phProperties{'RemoteAddress'}))
    {
      if (!Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
      {
        $$psError = "Neither the ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) or the RemoteUser/Address key/value pair ($$phProperties{'RemoteUser'}=$$phProperties{'RemoteAddress'}/32) is on the host access list";
      }
      else
      {
        $$psError = "The ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) is not on the host access list";
      }
      return 403;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if (Yes($$phProperties{'CapContentLength'}) && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    ##################################################################
    #
    # Confirm that service is currently enabled.
    #
    ##################################################################

    if (!Yes($$phProperties{'EnableGetService'}))
    {
      $$psError = "GET service is disabled";
      return 553;
    }

    ##################################################################
    #
    # Conditionally pull content from the POUND and serve it up for
    # requests containing a valid queue tag.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableJobQueues'}) && $$phProperties{'QueueTag'} =~ /^($$phProperties{'CommonRegexes'}{'JqdQueueTag'})$/)
    {
      $$phProperties{'JqdQueueTag'} = $$phProperties{'JqdQueueNid'} = $1; # Assume that $1 is defined and valid.
      $$phProperties{'JqdQueueNid'} =~ s/^[ps]//;
      $$phProperties{'JqdQueueNid'} =~ s/_.*$//;
      my %hLArgs =
      (
        'ClientId'          => $$phProperties{'ClientId'},
        'JobQueueDirectory' => $$phProperties{'JobQueueDirectory'},
        'JqdQueueNid'       => $$phProperties{'JqdQueueNid'},
      );
      $$phProperties{'JqdQueueName'} = JqdGetQueueName(\%hLArgs);
      if (!defined($$phProperties{'JqdQueueName'}))
      {
        if ($hLArgs{'Error'})
        {
          $$psError = $hLArgs{'Error'};
          return 500;
        }
        $$psError = "Queue NID ($$phProperties{'JqdQueueNid'}) is not mapped.";
        return 462;
      }
      $$phProperties{'JqdQueueDirectory'} = $$phProperties{'JobQueueDirectory'} . "/" . $$phProperties{'JqdQueueName'};
      %hLArgs =
      (
        'Directory' => $$phProperties{'JqdQueueDirectory'},
      );
      if (!JqdCheckQueueTree(\%hLArgs))
      {
        $$psError = $hLArgs{'Error'};
        return 461;
      }
      my $sJobFile = $$phProperties{'JqdQueueDirectory'} . "/" . "sent" . "/" . $$phProperties{'JqdQueueTag'};
      if (!-f $sJobFile)
      {
        $$psError = "Queue file ($sJobFile) was not found in the specified queue ($$phProperties{'JqdQueueName'})";
        return 404;
      }

      ################################################################
      #
      # Pull the requested file from the POUND.
      #
      ################################################################

      my %hJobProperties = ();
      %hLArgs =
      (
        'File'           => $sJobFile,
        'Properties'     => \%hJobProperties,
        'RequiredKeys'   => ['Command', 'CommandAlias', 'CommandLine', 'CommandMd5', 'CommandPath', 'CommandSha1', 'CommandSize', 'Created', 'Creator', 'JobGroup'],
        'RequireAllKeys' => 0,
        'Template'       => $$phProperties{'CommonTemplates'}{'jqd.job'},
        'VerifyValues'   => 1,
      );
      if (!KvpGetKvps(\%hLArgs))
      {
        $$psError = $hLArgs{'Error'};
        return 500;
      }
      my $sGetName = sprintf("%s.%s", $hJobProperties{'CommandMd5'}, substr($hJobProperties{'CommandSha1'}, 0, 8));
      my $sGetFile = sprintf("%s/db/pound/commands/%s/%s/%s/%s/%s",
        $$phProperties{'BaseDirectory'},
        substr($hJobProperties{'CommandMd5'}, 0, 2),
        substr($hJobProperties{'CommandMd5'}, 2, 2),
        substr($hJobProperties{'CommandMd5'}, 4, 2),
        substr($hJobProperties{'CommandMd5'}, 6, 2),
        $sGetName
        );
      my $sSigFile = $sGetFile . $$phProperties{'DsvSignatureSuffix'};
      if (-e $sGetFile)
      {
        my ($sReturnHandle, $sReturnCode) = GetContentHandle($phProperties, $sGetFile, $sSigFile, $psError);
        if (!defined($sReturnHandle))
        {
          return $sReturnCode;
        }
        $$phProperties{'ReturnHandle'} = $sReturnHandle;
        if (!UpdateQueue($phProperties, \%hJobProperties, \$sLocalError))
        {
          $$psError = $sLocalError;
          return ($sLocalError =~ /locked/) ? 503 : 500;
        }
        $$psError = "Success ($$phProperties{'JqdQueueTag'})";
        return 200;
      }
      $$psError = "GET file ($sGetFile) was not found";
      return 404;
    }

    ##################################################################
    #
    # Conditionally create dynamic content and serve it up.
    #
    ##################################################################

    if (Yes($$phProperties{'GetHookEnable'}))
    {
      $$phProperties{'ExpandHookCommandLineRoutine'} = \&ExpandGetHookCommandLine;
      my $sHookStatus = HookExecuteCommandLine($phProperties, \$sLocalError);
      if (!defined($sHookStatus))
      {
        $$psError = $sLocalError;
        return 500;
      }
      if ($sHookStatus != $$phProperties{'GetHookStatus'})
      {
        $$psError = "Hook status mismatch ($sHookStatus != $$phProperties{'GetHookStatus'})";
        my $sReturnStatus = GetMappedValue($$phProperties{'GetHookStatusMap'}, $sHookStatus);
        return (defined($sReturnStatus) && $sReturnStatus >= 200 && $sReturnStatus < 600) ? $sReturnStatus : 551;
      }
      my $sGetName = $$phProperties{'JobId'} . ".get";
      my $sGetFile = $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $sGetName;
      my $sSigFile = $sGetFile . $$phProperties{'DsvSignatureSuffix'};
      if (-e $sGetFile)
      {
        my ($sReturnHandle, $sReturnCode) = GetContentHandle($phProperties, $sGetFile, $sSigFile, $psError);
        if (!defined($sReturnHandle))
        {
          return $sReturnCode;
        }
        $$phProperties{'ReturnHandle'} = $sReturnHandle;
        unlink($sGetFile); # This is dynamic content, so unlink it now.
        unlink($sSigFile); # This is dynamic content, so unlink it now.
        $$psError = "Success";
        return 200;
      }
      $$psError = "GET file ($sGetFile) was not found";
      return 404;
    }

    ##################################################################
    #
    # Traverse the effective folder list, locate the first occurrence
    # of the requested file, and serve it up.
    #
    ##################################################################

    my $sEffectiveFolderList = $$phProperties{'ClientId'};
    if (defined($$phProperties{'FolderList'}) && length($$phProperties{'FolderList'}) > 0)
    {
      $sEffectiveFolderList .= ":" . $$phProperties{'FolderList'};
    }

    foreach my $sFolder (split(/:/, $sEffectiveFolderList))
    {
      my $sGetFile = $$phProperties{'ProfilesDirectory'} . "/" . $sFolder . "/" . "commands" . "/" . $$phProperties{'ClientFilename'};
      my $sSigFile = $sGetFile . $$phProperties{'DsvSignatureSuffix'};
      if (-e $sGetFile)
      {
        my ($sReturnHandle, $sReturnCode) = GetContentHandle($phProperties, $sGetFile, $sSigFile, $psError);
        if (!defined($sReturnHandle))
        {
          return $sReturnCode;
        }
        $$phProperties{'ReturnHandle'} = $sReturnHandle;
        $$psError = "Success";
        return 200;
      }
    }
    $$psError = "GET file ($$phProperties{'ClientFilename'}) was not found in effective folder list ($sEffectiveFolderList)";
    return 404;
  }
  elsif
  (
       VerifyHashKeyValue(\%hParameters, 'ClientId', $$phProperties{'CommonRegexes'}{'ClientId'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'QueueName', "(?:$$phProperties{'CommonRegexes'}{'ClientId'}|$$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'})", 3)
    && VerifyHashKeyValue(\%hParameters, 'JobType', "(?:serial|parallel)", 3)
    && VerifyHashKeyValue(\%hParameters, 'JobCount', $$phProperties{'CommonRegexes'}{'Decimal16Bit'}, 3)
  )
  {
    $$phProperties{'ClientId'}     = $hParameters{'ClientId'};
    $$phProperties{'JqdQueueName'} = $hParameters{'QueueName'};
    $$phProperties{'JqdJobType'}   = $hParameters{'JobType'};
    $$phProperties{'JqdJobCount'}  = $hParameters{'JobCount'};

    ##################################################################
    #
    # The min/max priority parameters are optional.
    #
    ##################################################################

    if (exists($hParameters{'MinPriority'}))
    {
      if (VerifyHashKeyValue(\%hParameters, 'MinPriority', "\\d+", 3) && $hParameters{'MinPriority'} < 100)
      {
        $$phProperties{'JqdMinPriority'} = sprintf("%02d", $hParameters{'MinPriority'});
      }
      else
      {
        $$psError = "Invalid query string ($$phProperties{'QueryString'})";
        return 450;
      }
    }

    if (exists($hParameters{'MaxPriority'}))
    {
      if (VerifyHashKeyValue(\%hParameters, 'MaxPriority', "\\d+", 3) && $hParameters{'MaxPriority'} < 100)
      {
        $$phProperties{'JqdMaxPriority'} = sprintf("%02d", $hParameters{'MaxPriority'});
      }
      else
      {
        $$psError = "Invalid query string ($$phProperties{'QueryString'})";
        return 450;
      }
    }

    ##################################################################
    #
    # Convert this GET request to a JOB request, and initialize key
    # properties.
    #
    ##################################################################

    $$phProperties{'RequestMethod'} = "JOB";
    $$phProperties{'JqdJobList'} = ""; # Note: This is referenced in SendResponse().

    ##################################################################
    #
    # Bring in any queue-specific properties.
    #
    ##################################################################

    my ($sLocalError);

    if (Yes($$phProperties{'EnableConfigOverrides'}))
    {
      $$phProperties{'ConfigSearchOrder'} = "queues"; # Confine the search to this one path.
      GetCustomConfigProperties($phProperties, \%{$$phProperties{'CustomConfigTemplate'}}, \$sLocalError);
    }

    ##################################################################
    #
    # Conditionally do CommonName and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'SslRequireSsl'}))
    {
      if (Yes($$phProperties{'SslRequireCn'}) && (!defined($$phProperties{'SslClientSDnCn'}) || !length($$phProperties{'SslClientSDnCn'})))
      {
        $$psError = "CommonName is undefined or null";
        return 470;
      }

      if (Yes($$phProperties{'SslRequireCn'}) && Yes($$phProperties{'SslRequireMatch'}) && $$phProperties{'SslClientSDnCn'} ne $$phProperties{'ClientId'})
      {
        $$psError = "CommonName ($$phProperties{'SslClientSDnCn'}) does not match client ID ($$phProperties{'ClientId'})";
        return 471;
      }
    }

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'RequireUser'}) && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if (Yes($$phProperties{'RequireUser'}) && Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do host access list checks.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableHostAccessList'}) && !CheckHostAccessList(GetEffectiveCidrList($phProperties), $$phProperties{'RemoteAddress'}))
    {
      if (!Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
      {
        $$psError = "Neither the ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) or the RemoteUser/Address key/value pair ($$phProperties{'RemoteUser'}=$$phProperties{'RemoteAddress'}/32) is on the host access list";
      }
      else
      {
        $$psError = "The ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) is not on the host access list";
      }
      return 403;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if (Yes($$phProperties{'CapContentLength'}) && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    ##################################################################
    #
    # Confirm that service is currently enabled.
    #
    ##################################################################

    if (!Yes($$phProperties{'EnableGetService'}))
    {
      $$psError = "JOB service is disabled";
      return 553;
    }

    ##################################################################
    #
    # Conditionally retrieve the current job list and serve it up.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableJobQueues'}))
    {
      if (Yes($$phProperties{'JobQueueActive'}))
      {
        $$phProperties{'JqdQueueDirectory'} = $$phProperties{'JobQueueDirectory'} . "/" . $$phProperties{'JqdQueueName'};
        my %hLArgs =
        (
          'Directory' => $$phProperties{'JqdQueueDirectory'},
        );
        if (!JqdCheckQueueTree(\%hLArgs))
        {
          $$psError = $hLArgs{'Error'};
          return 461;
        }
        %hLArgs =
        (
          'ClientId'          => $$phProperties{'ClientId'},
          'CommonRegexes'     => $$phProperties{'CommonRegexes'},
          'JobQueueDirectory' => $$phProperties{'JobQueueDirectory'},
          'JqdQueueName'      => $$phProperties{'JqdQueueName'},
        );
        $$phProperties{'JqdQueueNid'} = JqdGetQueueNid(\%hLArgs);
        if (!defined($$phProperties{'JqdQueueNid'}))
        {
          if ($hLArgs{'Error'})
          {
            $$psError = $hLArgs{'Error'};
            return 500;
          }
          $$psError = "Queue name ($$phProperties{'JqdQueueName'}) is not mapped.";
          return 462;
        }
        my $sJobTally = CreateJobList($phProperties, \$sLocalError);
        if (!defined($sJobTally))
        {
          $$psError = $sLocalError;
          # NOTE: Errors prior to a state transition don't require a freeze.
          return ($sLocalError =~ /(?:frozen|locked)/) ? 503 : 500;
        }
        else
        {
          my ($sPart, $sWhole) = ($sJobTally =~ /(\d+)\/(\d+)/);
          if ($sPart == 0 && $sWhole == 0)
          {
            $$psError = "Success ($sLocalError)";
            return 204;
          }
          else
          {
            if ($sPart == $sWhole)
            {
              $$psError = "Success (returned $sJobTally $$phProperties{'JqdJobType'} jobs)";
              return 200;
            }
            else
            {
              $$psError = "Partial Success (returned $sJobTally $$phProperties{'JqdJobType'} jobs) Error='$sLocalError'";
              # NOTE: Errors during a state transition require a freeze.
              my %hLArgs =
              (
                'Directory' => $$phProperties{'JqdQueueDirectory'},
              );
              JqdFreezeQueue(\%hLArgs);
              return 206;
            }
          }
        }
      }
      else
      {
        $$psError = "Queue ($hProperties{'JqdQueueName'}) is not active.";
        return 460;
      }
    }
    else
    {
      $$psError = "Method ($hProperties{'RequestMethod'}) not allowed";
      return 405;
    }
  }
  else
  {
    $$psError = "Invalid query string ($$phProperties{'QueryString'})";
    return 450;
  }
}


######################################################################
#
# ProcessPutRequest
#
######################################################################

sub ProcessPutRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Proceed only if QueryString matches the PutQuery expression.
  #
  ####################################################################

  my %hParameters = ();
#FIXME Do we care if the same parameter is specified more than once? The last one in wins.
#FIXME Do we want to impose a limit on the number of parameters or the length of the query string?
  CgiParseQueryString($$phProperties{'QueryString'}, \%hParameters);

  if
  (
       VerifyHashKeyValue(\%hParameters, 'VERSION', $$phProperties{'CommonRegexes'}{'WebJobVersion'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'SYSTEM', $$phProperties{'CommonRegexes'}{'SystemVersion'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'CLIENTID', $$phProperties{'CommonRegexes'}{'ClientId'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'FILENAME', $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'RUNTYPE',  "(?:linktest|snapshot)", 3)
    && VerifyHashKeyValue(\%hParameters, 'STDOUT_LENGTH', $$phProperties{'CommonRegexes'}{'Decimal64Bit'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'STDERR_LENGTH', $$phProperties{'CommonRegexes'}{'Decimal64Bit'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'STDENV_LENGTH', $$phProperties{'CommonRegexes'}{'Decimal64Bit'}, 3)
  )
  {
    my ($sEnvLength, $sErrLength, $sOutLength);

    $$phProperties{'ClientVersion'}   = $hParameters{'VERSION'};
    $$phProperties{'ClientSystem'}    = $hParameters{'SYSTEM'};
    $$phProperties{'ClientId'}        = $hParameters{'CLIENTID'} || "nobody";
    $$phProperties{'ClientFilename'}  = $hParameters{'FILENAME'};
    $$phProperties{'ClientRunType'}   = $hParameters{'RUNTYPE'};
    $$phProperties{'ClientOutLength'} = $sOutLength = $hParameters{'STDOUT_LENGTH'};
    $$phProperties{'ClientErrLength'} = $sErrLength = $hParameters{'STDERR_LENGTH'};
    $$phProperties{'ClientEnvLength'} = $sEnvLength = $hParameters{'STDENV_LENGTH'};

    ##################################################################
    #
    # The QUEUETAG parameter is optional. Also, it's not supported by
    # legacy clients (i.e., versions less than 0x10800008).
    #
    ##################################################################

    if (VerifyHashKeyValue(\%hParameters, 'QUEUETAG', "(?:|$$phProperties{'CommonRegexes'}{'JqdQueueTag'})", 3))
    {
      $$phProperties{'QueueTag'} = $hParameters{'QUEUETAG'};
    }
    else
    {
      $$phProperties{'QueueTag'} = "";
    }

    ##################################################################
    #
    # Bring in any client- and/or command-specific properties.
    #
    ##################################################################

    my ($sLocalError);

    if (Yes($$phProperties{'EnableConfigOverrides'}))
    {
      GetCustomConfigProperties($phProperties, \%{$$phProperties{'CustomConfigTemplate'}}, \$sLocalError);
    }

    ##################################################################
    #
    # Conditionally do CommonName and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'SslRequireSsl'}))
    {
      if (Yes($$phProperties{'SslRequireCn'}) && (!defined($$phProperties{'SslClientSDnCn'}) || !length($$phProperties{'SslClientSDnCn'})))
      {
        $$psError = "CommonName is undefined or null";
        return 470;
      }

      if (Yes($$phProperties{'SslRequireCn'}) && Yes($$phProperties{'SslRequireMatch'}) && $$phProperties{'SslClientSDnCn'} ne $$phProperties{'ClientId'})
      {
        $$psError = "CommonName ($$phProperties{'SslClientSDnCn'}) does not match client ID ($$phProperties{'ClientId'})";
        return 471;
      }
    }

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'RequireUser'}) && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if (Yes($$phProperties{'RequireUser'}) && Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do host access list checks.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableHostAccessList'}) && !CheckHostAccessList(GetEffectiveCidrList($phProperties), $$phProperties{'RemoteAddress'}))
    {
      if (!Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
      {
        $$psError = "Neither the ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) or the RemoteUser/Address key/value pair ($$phProperties{'RemoteUser'}=$$phProperties{'RemoteAddress'}/32) is on the host access list";
      }
      else
      {
        $$psError = "The ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) is not on the host access list";
      }
      return 403;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if (Yes($$phProperties{'CapContentLength'}) && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    if ($$phProperties{'ContentLength'} != ($sOutLength + $sErrLength + $sEnvLength))
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) does not equal sum of individual stream lengths ($sOutLength + $sErrLength + $sEnvLength)";
      return 456;
    }

    ##################################################################
    #
    # Confirm that service is currently enabled.
    #
    ##################################################################

    if (!Yes($$phProperties{'EnablePutService'}))
    {
      $$psError = "PUT service is disabled";
      return 553;
    }

    ##################################################################
    #
    # Conditionally validate queue properties.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableJobQueues'}) && $$phProperties{'QueueTag'} =~ /^($$phProperties{'CommonRegexes'}{'JqdQueueTag'})$/)
    {
      $$phProperties{'JqdQueueTag'} = $$phProperties{'JqdQueueNid'} = $1; # Assume that $1 is defined and valid.
      $$phProperties{'JqdQueueNid'} =~ s/^[ps]//;
      $$phProperties{'JqdQueueNid'} =~ s/_.*$//;
      my %hLArgs =
      (
        'ClientId'          => $$phProperties{'ClientId'},
        'JobQueueDirectory' => $$phProperties{'JobQueueDirectory'},
        'JqdQueueNid'       => $$phProperties{'JqdQueueNid'},
      );
      $$phProperties{'JqdQueueName'} = JqdGetQueueName(\%hLArgs);
      if (!defined($$phProperties{'JqdQueueName'}))
      {
        if ($hLArgs{'Error'})
        {
          $$psError = $hLArgs{'Error'};
          return 500;
        }
        $$psError = "Queue NID ($$phProperties{'JqdQueueNid'}) is not mapped.";
        return 462;
      }
      $$phProperties{'JqdQueueDirectory'} = $$phProperties{'JobQueueDirectory'} . "/" . $$phProperties{'JqdQueueName'};
      %hLArgs =
      (
        'Directory' => $$phProperties{'JqdQueueDirectory'},
      );
      if (!JqdCheckQueueTree(\%hLArgs))
      {
        $$psError = $hLArgs{'Error'};
        return 461;
      }
      my $sJobFile = $$phProperties{'JqdQueueDirectory'} . "/" . "open" . "/" . $$phProperties{'JqdQueueTag'};
      if (!-f $sJobFile)
      {
        $$psError = "Queue file ($sJobFile) was not found in the specified queue ($$phProperties{'JqdQueueName'})";
        return 404;
      }
    }

    ##################################################################
    #
    # If this is a link test, dump the data and return success.
    #
    ##################################################################

    if ($$phProperties{'ClientRunType'} eq "linktest")
    {
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      $$psError = "Success";
      $$psError .= " ($$phProperties{'JqdQueueTag'})" if ($$phProperties{'JqdQueueTag'});
      return 251;
    }

    ##################################################################
    #
    # Make output filenames and directories.
    #
    ##################################################################

    my ($sEnvFile, $sErrFile, $sLckFile, $sOutFile, $sPutName, $sRdyFile);

    $sPutName = MakePutName($phProperties, \$sLocalError);
    if (!defined($sPutName))
    {
      $$psError = $sLocalError;
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      return 500;
    }
    $sLckFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".lck";
    $sOutFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".out";
    $sErrFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".err";
    $sEnvFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".env";
    $sRdyFile = $$phProperties{'IncomingDirectory'} . "/" . $sPutName . ".rdy";

    $$phProperties{'LckFile'} = $sLckFile;
    $$phProperties{'OutFile'} = $sOutFile;
    $$phProperties{'ErrFile'} = $sErrFile;
    $$phProperties{'EnvFile'} = $sEnvFile;
    $$phProperties{'RdyFile'} = $sRdyFile;

    if (!defined(MakePutTree($$phProperties{'IncomingDirectory'}, $sPutName, 0755, 1, \$sLocalError)))
    {
      $$psError = $sLocalError;
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      return 500;
    }

    ##################################################################
    #
    # Create a group lockfile and lock it. The purpose of the lock
    # is to prevent other instances of this script from writing to
    # any of the output files (.out, .err, .env, .rdy).
    #
    ##################################################################

    if (!open(LH, "> $sLckFile"))
    {
      $$psError = "File ($sLckFile) could not be opened ($!)";
      SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
      return 500;
    }
    flock(LH, LOCK_EX);

    ##################################################################
    #
    # Make sure that none of the output files exist.
    #
    ##################################################################

    foreach my $sPutFile ($sOutFile, $sErrFile, $sEnvFile, $sRdyFile)
    {
      if (-e $sPutFile)
      {
        if (Yes($$phProperties{'OverwriteExistingFiles'}))
        {
          unlink($sPutFile);
        }
        else
        {
          $$psError = "File ($sPutFile) already exists";
          SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
          flock(LH, LOCK_UN); close(LH); unlink($sLckFile); # Unlock, close, and remove the group lockfile.
          return 451;
        }
      }
    }

    ##################################################################
    #
    # Write the output files (.out, .err, .env, .rdy) to disk.
    #
    ##################################################################

    my (%hStreamLengths);

    $hStreamLengths{$sOutFile} = $sOutLength;
    $hStreamLengths{$sErrFile} = $sErrLength;
    $hStreamLengths{$sEnvFile} = $sEnvLength;

    foreach my $sPutFile ($sOutFile, $sErrFile, $sEnvFile, $sRdyFile)
    {
      if (!open(FH, "> $sPutFile"))
      {
        $$psError = "File ($sPutFile) could not be opened ($!)";
        SysReadWrite(\*STDIN, undef, $$phProperties{'ContentLength'}, undef); # Slurp up data to prevent a broken pipe.
        flock(LH, LOCK_UN); close(LH); unlink($sLckFile); # Unlock, close, and remove the group lockfile.
        return 500;
      }
      binmode(FH);
      flock(FH, LOCK_EX);
      if ($sPutFile eq $sRdyFile)
      {
        ##############################################################
        #
        # The job time is defined to be the time between the creation
        # of the job ID and the point at which all client data (i.e.,
        # .out, .err, and .env) are stored on disk. Since the .rdy
        # file follows the .env file (i.e., the last of the client's
        # data), this is the demarcation point for that calculation.
        #
        ##############################################################

        $$phProperties{'JobTime'} = ComputeJobTime($phProperties);

        ##############################################################
        #
        # Write key server-side information and settings to disk.
        #
        ##############################################################

        print FH "Version=", $$phProperties{'Version'}, $$phProperties{'Newline'};
        print FH "ContentLength=", $$phProperties{'ContentLength'}, $$phProperties{'Newline'};
        print FH "Https=", $$phProperties{'Https'}, $$phProperties{'Newline'};
        print FH "JobId=", $$phProperties{'JobId'}, $$phProperties{'Newline'};
        print FH "JobTime=", (($$phProperties{'JobTime'} < 0) ? "-1" : sprintf("%.6f", $$phProperties{'JobTime'})), $$phProperties{'Newline'};
        print FH "QueryString=", $$phProperties{'QueryString'}, $$phProperties{'Newline'};
        print FH "RemoteAddress=", $$phProperties{'RemoteAddress'}, $$phProperties{'Newline'};
        print FH "RemoteUser=", $$phProperties{'RemoteUser'}, $$phProperties{'Newline'};
        print FH "RequestMethod=", $$phProperties{'RequestMethod'}, $$phProperties{'Newline'};
        print FH "ServerSoftware=", $$phProperties{'ServerSoftware'}, $$phProperties{'Newline'};
        print FH "PropertiesFile=", $$phProperties{'PropertiesFile'}, $$phProperties{'Newline'};
        print FH "SslClientSDnCn=", $$phProperties{'SslClientSDnCn'}, $$phProperties{'Newline'};
        foreach my $sKey (sort(keys(%{$$phProperties{'GlobalConfigTemplate'}})))
        {
          print FH $sKey, "=", $$phProperties{$sKey}, $$phProperties{'Newline'};
        }
      }
      else
      {
        my $sByteCount = SysReadWrite(\*STDIN, \*FH, $hStreamLengths{$sPutFile}, \$sLocalError);
        if (!defined($sByteCount))
        {
          $$psError = $sLocalError;
          flock(FH, LOCK_UN); close(FH);
          flock(LH, LOCK_UN); close(LH); unlink($sLckFile); # Unlock, close, and remove the group lockfile.
          return 500;
        }
        if ($sByteCount != $hStreamLengths{$sPutFile})
        {
          $$psError = "Stream length ($hStreamLengths{$sPutFile}) does not equal number of bytes processed ($sByteCount) for output file ($sPutFile)";
          flock(FH, LOCK_UN); close(FH);
          flock(LH, LOCK_UN); close(LH); unlink($sLckFile); # Unlock, close, and remove the group lockfile.
          return 456;
        }
      }
      flock(FH, LOCK_UN); close(FH);
    }
    flock(LH, LOCK_UN); close(LH); unlink($sLckFile); # Unlock, close, and remove the group lockfile.

    ##################################################################
    #
    # Conditionally update the queue state.
    #
    ##################################################################

    my ($sJqdQueueTag);

    if (Yes($$phProperties{'EnableJobQueues'}) && $$phProperties{'JqdQueueTag'} =~ /^$$phProperties{'CommonRegexes'}{'JqdQueueTag'}$/)
    {
      if (!UpdateQueue($phProperties, undef, \$sLocalError))
      {
        $$psError = $sLocalError;
        return ($sLocalError =~ /locked/) ? 503 : 500;
      }
      $sJqdQueueTag = $$phProperties{'JqdQueueTag'};
    }

    ##################################################################
    #
    # Conditionally create dynamic content and serve it up.
    #
    ##################################################################

    if (Yes($$phProperties{'PutHookEnable'}))
    {
      $$phProperties{'ExpandHookCommandLineRoutine'} = \&ExpandPutHookCommandLine;
      my $sHookStatus = HookExecuteCommandLine($phProperties, \$sLocalError);
      if (!defined($sHookStatus))
      {
        $$psError = $sLocalError;
        return 500;
      }
      if ($sHookStatus != $$phProperties{'PutHookStatus'})
      {
        $$psError = "Hook status mismatch ($sHookStatus != $$phProperties{'PutHookStatus'})";
        my $sReturnStatus = GetMappedValue($$phProperties{'PutHookStatusMap'}, $sHookStatus);
        return (defined($sReturnStatus) && $sReturnStatus >= 200 && $sReturnStatus < 600) ? $sReturnStatus : 551;
      }
      my $sPutName = $$phProperties{'JobId'} . ".put";
      my $sPutFile = $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $sPutName;
      my $sSigFile = $sPutFile . $$phProperties{'DsvSignatureSuffix'};
      if (-e $sPutFile)
      {
        my ($sReturnHandle, $sReturnCode) = GetContentHandle($phProperties, $sPutFile, $sSigFile, $psError);
        if (!defined($sReturnHandle))
        {
          return $sReturnCode;
        }
        $$phProperties{'ReturnHandle'} = $sReturnHandle;
        unlink($sPutFile); # This is dynamic content, so unlink it now.
        unlink($sSigFile); # This is dynamic content, so unlink it now.
        $$psError = (defined($sJqdQueueTag)) ? "Success ($sJqdQueueTag)" : "Success";
        return 200;
      }
      $$psError = "PUT file ($sPutFile) was not found";
      return 404;
    }

    ##################################################################
    #
    # Traverse the effective folder list, locate the first occurrence
    # of the response file, and serve it up. Currently, response files
    # are not required. However, if DSV signatures are required, then
    # a valid signature must be returned. If no signature file exists,
    # then the null signature must be returned.
    #
    ##################################################################

    my $sEffectiveFolderList = $$phProperties{'ClientId'};
    if (defined($$phProperties{'FolderList'}) && length($$phProperties{'FolderList'}) > 0)
    {
      $sEffectiveFolderList .= ":" . $$phProperties{'FolderList'};
    }

    foreach my $sFolder (split(/:/, $sEffectiveFolderList))
    {
      my $sPutFile = $$phProperties{'ProfilesDirectory'} . "/" . $sFolder . "/" . "content" . "/" . $$phProperties{'ClientFilename'};
      my $sSigFile = $sPutFile . $$phProperties{'DsvSignatureSuffix'};
      if (-e $sPutFile)
      {
        my ($sReturnHandle, $sReturnCode) = GetContentHandle($phProperties, $sPutFile, $sSigFile, $psError);
        if (!defined($sReturnHandle))
        {
          return $sReturnCode;
        }
        $$phProperties{'ReturnHandle'} = $sReturnHandle;
        $$psError = (defined($sJqdQueueTag)) ? "Success ($sJqdQueueTag)" : "Success";
        return 200;
      }
    }
#   $$psError = "PUT file ($$phProperties{'ClientFilename'}) was not found in effective folder list ($sEffectiveFolderList)";
#   return 404;

    if (VerifyHashKeyValue($phProperties, 'DsvNullSignature', "[0-9A-Za-z+\/]{1,$$phProperties{'DsvMaxSignatureLength'}}={0,2}", 3))
    {
      $$phProperties{'DsvPayloadSignature'} = $$phProperties{'DsvNullSignature'};
    }

    if (Yes($$phProperties{'DsvRequireSignatures'}) && !exists($$phProperties{'DsvPayloadSignature'}))
    {
      $$psError = "DsvNullSignature property does not exist, is not defined, or does not pass muster";
      return (undef, 459);
    }

    ##################################################################
    #
    # Return success. Conditionally include the queue tag.
    #
    ##################################################################

    $$psError = (defined($sJqdQueueTag)) ? "Success ($sJqdQueueTag)" : "Success";
    return 200;
  }
  else
  {
    $$psError = "Invalid query string ($$phProperties{'QueryString'})";
    return 450;
  }
}


######################################################################
#
# SendResponse
#
######################################################################

sub SendResponse
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Send response header.
  #
  ####################################################################

  my ($sHandle, $sHeader, $sLength, $sReason, $sServer, $sStatus);

  $sHandle = $$phProperties{'ReturnHandle'};
  $sStatus = $$phProperties{'ReturnStatus'};
  $sReason = (defined($$phProperties{'ReturnReason'})) ? $$phProperties{'ReturnReason'} : "Undefined";
  $sServer = $$phProperties{'ServerSoftware'} || "unknown";
  $sLength = (defined($sHandle)) ? -s $sHandle : ($$phProperties{'RequestMethod'} eq 'JOB') ? length($$phProperties{'JqdJobList'}) : 0;

  $sHeader  = "HTTP/1.1 $sStatus $sReason\r\n";
  $sHeader .= "Server: $sServer\r\n";
  $sHeader .= "Content-Type: application/octet-stream\r\n";
  $sHeader .= "Content-Length: $sLength\r\n";
  if ($$phProperties{'RequestMethod'} eq 'GET')
  {
    # NOTE: The JobId is mandatory as of revision 1.90, but its value
    # could be undefined if CreateRunTimeEnvironment() errors out too
    # soon. Return a value of "NA" to handle that situation.
    if (!defined($$phProperties{'JobId'}))
    {
      $sHeader .= "Job-Id: NA\r\n";
    }
    else
    {
      $sHeader .= "Job-Id: $$phProperties{'JobId'}\r\n";
    }
  }
  if (exists($$phProperties{'DsvPayloadSignature'}) && defined($$phProperties{'DsvPayloadSignature'}))
  {
    $sHeader .= "WebJob-Payload-Signature: $$phProperties{'DsvPayloadSignature'}\r\n";
  }
  $sHeader .= "\r\n";

  syswrite(STDOUT, $sHeader, length($sHeader));

  ####################################################################
  #
  # Send content if any.
  #
  ####################################################################

  if (defined($sHandle))
  {
    SysReadWrite($sHandle, \*STDOUT, $sLength, undef);
    close($sHandle);
  }
  else
  {
    if ($$phProperties{'RequestMethod'} eq 'JOB')
    {
      syswrite(\*STDOUT, $$phProperties{'JqdJobList'}, $sLength);
    }
  }

  return $sLength;
}


######################################################################
#
# SetupHostAccessList
#
######################################################################

sub SetupHostAccessList
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Initialize the access list. This list is a hash that contains
  # client ID to IP address mappings. If the client passes all other
  # authentication tests, a lookup will be performed to determine if
  # the client's IP address is authorized. If it is, processing
  # continues as usual. Otherwise, the CGI script will return a 403
  # status code to the client, log an error message, and abort. If the
  # access list is empty or there is an error processing its file, all
  # access will be denied (i.e., this mechanism fails closed).
  #
  # The format of the list is key/value pair as follows:
  #
  #   <client_id> = <cidr>[,<cidr>]
  #
  #####################################################################

  %{$$phProperties{'AccessList'}} = (); # Initialize an empty list.

  my $sFile = $$phProperties{'NphConfigDirectory'} . "/" . "nph-webjob-hosts.access";

  if (open(AH, "< $sFile"))
  {
    while (my $sLine = <AH>)
    {
      $sLine =~ s/[\r\n]+$//; # Remove CRs and LFs.
      $sLine =~ s/#.*$//; # Remove comments.
      if ($sLine !~ /^\s*$/)
      {
        my ($sKey, $sValue) = ($sLine =~ /^([^=]*)=(.*)$/);
        $sKey =~ s/^\s+//; # Remove leading whitespace.
        $sKey =~ s/\s+$//; # Remove trailing whitespace.
        $sValue =~ s/^\s+//; # Remove leading whitespace.
        $sValue =~ s/\s+$//; # Remove trailing whitespace.
        if ($sKey =~ /^$$phProperties{'CommonRegexes'}{'ClientId'}$/)
        {
          $$phProperties{'AccessList'}{$sKey} = $sValue;
        }
      }
    }
    close(AH);
  }
  else
  {
    $$psError = "File ($sFile) could not be opened ($!)" if (defined($psError));
    return undef;
  }
}


######################################################################
#
# SysReadWrite
#
######################################################################

sub SysReadWrite
{
  my ($sReadHandle, $sWriteHandle, $sLength, $psError) = @_;

  ####################################################################
  #
  # Read/Write data, but discard data if write handle is undefined.
  #
  ####################################################################

  my ($sData, $sEOF, $sNRead, $sNProcessed, $sNWritten);

  for ($sEOF = $sNRead = $sNProcessed = 0; !$sEOF && $sLength > 0; $sLength -= $sNRead)
  {
    $sNRead = sysread($sReadHandle, $sData, ($sLength > 0x4000) ? 0x4000 : $sLength);
    if (!defined($sNRead))
    {
      $$psError = "Error reading from input stream ($!)" if (defined($psError));
      return undef;
    }
    elsif ($sNRead == 0)
    {
      $sEOF = 1;
    }
    else
    {
      if (defined($sWriteHandle))
      {
        $sNWritten = syswrite($sWriteHandle, $sData, $sNRead);
        if (!defined($sNWritten))
        {
          $$psError = "Error writing to output stream ($!)" if (defined($psError));
          return undef;
        }
      }
      else
      {
        $sNWritten = $sNRead;
      }
      $sNProcessed += $sNWritten;
    }
  }

  return $sNProcessed;
}


######################################################################
#
# TrackerLogMessage
#
######################################################################

sub TrackerLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'TrackerMessage'}'\n";

  1;
}


######################################################################
#
# TrackerSpoolRequest
#
######################################################################

sub TrackerSpoolRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure the spool directory is writable.
  #
  ####################################################################

  my ($sRidSpoolDirectory);

  $sRidSpoolDirectory = $$phProperties{'SpoolDirectory'} . "/rid";

  if (!-d $sRidSpoolDirectory || !-W _)
  {
    $$psError = "The RID spool directory ($sRidSpoolDirectory) does not exist or is not writable";
    return undef;
  }

  ####################################################################
  #
  # Marshall key/value pairs according to the request method.
  #
  ####################################################################

  my (%hKvpMap, $sDuration, $sTimestamp);

  $sTimestamp = SecondsToDateTime($$phProperties{'StopTime'}, ((Yes($$phProperties{'UseGmt'})) ? 1 : 0));

  $sDuration = sprintf("%.6f", tv_interval([$$phProperties{'StartTime'}, $$phProperties{'StartTimeUsec'}], [$$phProperties{'StopTime'}, $$phProperties{'StopTimeUsec'}]));

  $hKvpMap{'BytesRx'} = (defined($$phProperties{'ContentLength'})) ? $$phProperties{'ContentLength'} : "";
  $hKvpMap{'BytesTx'} = (defined($$phProperties{'ServerContentLength'})) ? $$phProperties{'ServerContentLength'} : "";
  $hKvpMap{'ClientId'} = (defined($$phProperties{'ClientId'})) ? $$phProperties{'ClientId'} : "";
  if ($$phProperties{'RequestMethod'} =~ /^GET|PUT$/)
  {
    $hKvpMap{'Command'} = (defined($$phProperties{'ClientFilename'})) ? $$phProperties{'ClientFilename'} : "";
  }
  $hKvpMap{'Duration'} = (defined($sDuration)) ? $sDuration : "";
  $hKvpMap{'Ip'} = (defined($$phProperties{'RemoteAddress'})) ? $$phProperties{'RemoteAddress'} : "";
  $hKvpMap{'Jid'} = (defined($$phProperties{'JobId'})) ? $$phProperties{'JobId'} : "";
  if ($$phProperties{'RequestMethod'} =~ /^GET|PUT$/)
  {
    $hKvpMap{'Jqt'} = (defined($$phProperties{'JqdQueueTag'})) ? $$phProperties{'JqdQueueTag'} : "";
  }
  $hKvpMap{'Method'} = (defined($$phProperties{'RequestMethod'})) ? $$phProperties{'RequestMethod'} : "";
  $hKvpMap{'QueueName'} = (defined($$phProperties{'JqdQueueName'})) ? $$phProperties{'JqdQueueName'} : "";
  $hKvpMap{'Reason'} = (defined($$phProperties{'ErrorMessage'})) ? $$phProperties{'ErrorMessage'} : "";
  if ($$phProperties{'RequestMethod'} =~ /^PUT$/)
  {
    $hKvpMap{'RdyFile'} = (defined($$phProperties{'RdyFile'})) ? $$phProperties{'RdyFile'} : "";
  }
  $hKvpMap{'Status'} = (defined($$phProperties{'ReturnStatus'})) ? $$phProperties{'ReturnStatus'} : "";
  $hKvpMap{'Timestamp'} = (defined($sTimestamp)) ? $sTimestamp : "";
  $hKvpMap{'Username'} = (defined($$phProperties{'RemoteUser'})) ? $$phProperties{'RemoteUser'} : "";

  ####################################################################
  #
  # Create the spool file. To avoid interference with the application
  # that processes the spool, this must be done using a temporary file
  # and an atomic move. A lock should not be required since the name
  # of each file is supposed to be unique.
  #
  ####################################################################

  my (%hLArgs, $sExtension);

  $sExtension = "." . lc($$phProperties{'RequestMethod'});

  %hLArgs =
  (
    'File'           => $sRidSpoolDirectory . "/" . $$phProperties{'JobId'} . $sExtension,
    'Properties'     => \%hKvpMap,
  );
  if (!KvpSetKvps(\%hLArgs))
  {
    $$psError = $hLArgs{'Error'};
    return undef;
  }

  1;
}


######################################################################
#
# TriggerExecuteCommandLine
#
######################################################################

sub TriggerExecuteCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'EnableLogging',
      'ExpandTriggerCommandLineRoutine',
      'OsClass',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Windows platforms are not currently supported.
  #
  ####################################################################

  if ($$phProperties{'OsClass'} eq "WINX")
  {
    $$psError = "Triggers are not currently supported on Windows platforms";
    return undef;
  }

  ####################################################################
  #
  # Expand the trigger's command line. If the result is undefined or
  # null, abort.
  #
  ####################################################################

  my ($sLocalError);

  $$phProperties{'TriggerCommandLine'} = &{$$phProperties{'ExpandTriggerCommandLineRoutine'}}($phProperties, \$sLocalError);
  if (!defined($$phProperties{'TriggerCommandLine'}))
  {
    $$psError = $sLocalError;
    return undef;
  }

  if (!length($$phProperties{'TriggerCommandLine'}))
  {
    $$psError = "Command line is undefined or null";
    return undef;
  }

  ####################################################################
  #
  # Spawn a subprocess. Set the kid's process group. This should
  # isolate the kid from signals sent to his parent or grandparent
  # (i.e., this script or the server daemon, respectively).  Close
  # STDOUT. This should prevent the kid from interfering with the
  # original CGI connection (e.g., holding the socket open). Keep
  # STDERR open open so that errors can be caught in the server's
  # error log. Change to the root directory to prevent unmounting
  # issues, which could happen if a long-running trigger process was
  # specified.
  #
  ####################################################################

  my $sKidPid = fork();

  if (!defined($sKidPid))
  {
    $$psError = "Unable to spawn process ($!)";
    return undef;
  }
  else
  {
    if ($sKidPid == 0)
    {
      setpgrp(0, 0);
      close(STDOUT);
      chdir("/");
      $$phProperties{'TriggerPidLabel'} = "kid";
      $$phProperties{'TriggerPid'} = $$;
      $$phProperties{'TriggerState'} = "pulled";
      $$phProperties{'TriggerMessage'} = $$phProperties{'TriggerCommandLine'};
      if (Yes($$phProperties{'EnableLogging'}))
      {
        TriggerLogMessage($phProperties);
      }
      my $sKidReturn = system($$phProperties{'TriggerCommandLine'});
      my $sKidStatus = ($sKidReturn >> 8) & 0xff;
      my $sKidSignal = ($sKidReturn & 0x7f);
      my $sKidDumped = ($sKidReturn & 0x80) ? 1 : 0;
      if ($sKidStatus == 255)
      {
        $$phProperties{'TriggerState'} = "failed";
        $$phProperties{'TriggerMessage'} = "Unable to execute trigger command ($!)";
      }
      else
      {
        $$phProperties{'TriggerState'} = "reaped";
        $$phProperties{'TriggerMessage'} = "status($sKidStatus) signal($sKidSignal) coredump($sKidDumped)";
      }
      if (Yes($$phProperties{'EnableLogging'}))
      {
        TriggerLogMessage($phProperties);
      }
      exit($sKidStatus);
    }
  }

  return $sKidPid;
}


######################################################################
#
# TriggerLogMessage
#
######################################################################

sub TriggerLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Marshall log arguments.
  #
  ####################################################################

  my (%hLArgs);

  $hLArgs{'LogEpoch'} = $$phProperties{'TriggerEpoch'};

  $hLArgs{'LogFields'} =
  [
    'JobId',
    'RequestMethod',
    'ClientId',
    'ClientFilename',
    'TriggerPidLabel',
    'TriggerPid',
    'TriggerState',
    'Message',
  ];

  $hLArgs{'LogValues'} =
  {
    'JobId'               => $$phProperties{'JobId'},
    'RequestMethod'       => $$phProperties{'RequestMethod'},
    'ClientId'            => $$phProperties{'ClientId'},
    'ClientFilename'      => $$phProperties{'ClientFilename'},
    'TriggerPidLabel'     => $$phProperties{'TriggerPidLabel'},
    'TriggerPid'          => $$phProperties{'TriggerPid'},
    'TriggerState'        => $$phProperties{'TriggerState'},
    'Message'             => $$phProperties{'TriggerMessage'},
  };

  $hLArgs{'LogFile'} = $$phProperties{'TriggerLogFile'};

  $hLArgs{'Newline'} = $$phProperties{'Newline'};

  $hLArgs{'RevertToStderr'} = 1;

  $hLArgs{'UseGmt'} = (Yes($$phProperties{'UseGmt'})) ? 1 : 0;

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  if (!LogNf1vMessage(\%hLArgs))
  {
    print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'TriggerMessage'}'\n";
    print STDERR "$$phProperties{'Program'}: Error='$hLArgs{'Error'}'\n";
  }

  1;
}


######################################################################
#
# UpdateQueue
#
######################################################################

sub UpdateQueue
{
  my ($phProperties, $phJobProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'JqdQueueTag',
      'JqdQueueDirectory',
      'RequestMethod',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Determine the internal state directories that will be in play,
  # and prepare the this/next queue state filenames.
  #
  ####################################################################

  my ($sNextJobFile, $sNextJobState, $sProperty, $sPropertyValue, $sThisJobFile, $sThisJobState);

  if ($$phProperties{'RequestMethod'} eq "GET")
  {
    $sThisJobState = "sent";
    $sNextJobState = "open";
    $sProperty = "JobId";
  }
  else
  {
    $sThisJobState = "open";
    $sNextJobState = "done";
    $sProperty = "RdyFile";
  }
  $sPropertyValue = $$phProperties{$sProperty} || "";
  $sThisJobFile = $$phProperties{'JqdQueueDirectory'} . "/" . $sThisJobState . "/" . $$phProperties{'JqdQueueTag'};
  $sNextJobFile = $$phProperties{'JqdQueueDirectory'} . "/" . $sNextJobState . "/" . $$phProperties{'JqdQueueTag'};

  ####################################################################
  #
  # Acquire the change lock. This is an exclusive lock.
  #
  ####################################################################

  my (%hQueueLockArgs);

  %hQueueLockArgs =
  (
    'LockFile' => $$phProperties{'JqdQueueDirectory'} . "/" . "change.lock",
    'LockMode' => "+<", # The file must exist or this will fail.
  );
  if (!JqdLockFile(\%hQueueLockArgs))
  {
    $$psError = $hQueueLockArgs{'Error'};
    return undef;
  }

  ####################################################################
  #
  # Check to see if the queue has been locked.
  #
  ####################################################################

  if (JqdIsQueueLocked({ 'Directory' => $$phProperties{'JqdQueueDirectory'} }))
  {
    $$psError = "Queue is locked";
    JqdUnlockFile(\%hQueueLockArgs);
    return undef;
  }

  ####################################################################
  #
  # Conditionally read the job file.
  #
  ####################################################################

  my (%hJobProperties);

  if (!defined($phJobProperties))
  {
    my %hLArgs =
    (
      'File'           => $sThisJobFile,
      'Properties'     => \%hJobProperties,
      'RequiredKeys'   => ['Command', 'CommandAlias', 'CommandLine', 'CommandMd5', 'CommandPath', 'CommandSha1', 'CommandSize', 'Created', 'Creator', 'JobGroup'],
      'RequireAllKeys' => 0,
      'Template'       => $$phProperties{'CommonTemplates'}{'jqd.job'},
      'VerifyValues'   => 1,
    );
    if (!KvpGetKvps(\%hLArgs))
    {
      $$psError = $hLArgs{'Error'};
      JqdUnlockFile(\%hQueueLockArgs);
      return undef;
    }
    $phJobProperties = \%hJobProperties;
  }

  ####################################################################
  #
  # Move the job file to the next queue state, and append additional
  # properties as necessary.
  #
  ####################################################################

  my ($sLocalError, $sResult);

  if (!rename($sThisJobFile, $sNextJobFile))
  {
    $sLocalError = "Unable to move the job ($sThisJobFile) to the next queue state ($!)";
    $sResult = "fail";
  }
  else
  {
    if (defined($sProperty))
    {
      my %hLArgs =
      (
        'AppendToFile'   => 1,
        'File'           => $sNextJobFile,
        'Properties'     => { $sProperty => $$phProperties{$sProperty} },
        'Template'       => { $sProperty => undef },
        'VerifyValues'   => 0,
      );
      if (!KvpSetKvps(\%hLArgs))
      {
        $sLocalError = $hLArgs{'Error'};
        $sResult = "fail";
      }
      else
      {
        $sLocalError = "Job state changed successfully.";
        $sResult = "pass";
      }
    }
    else
    {
      $sLocalError = "Job state changed successfully.";
      $sResult = "pass";
    }
  }

  ####################################################################
  #
  # Release the change lock.
  #
  ####################################################################

  JqdUnlockFile(\%hQueueLockArgs);

  ####################################################################
  #
  # Log the result of the state transition, and return to the caller.
  #
  ####################################################################

  my $sPoundName = sprintf("%s.%s", $$phJobProperties{'CommandMd5'}, substr($$phJobProperties{'CommandSha1'}, 0, 8));
  JqdLogMessage
  (
    {
      'Command'       => $$phJobProperties{'Command'},
      'CommandSize'   => $$phJobProperties{'CommandSize'},
      'Creator'       => $$phProperties{'ApacheUser'},
      'JobGroup'      => $$phJobProperties{'JobGroup'},
      'LogFile'       => $$phProperties{'JqdLogFile'},
      'Message'       => $sLocalError,
      'NewQueueState' => $sNextJobState,
      'OldQueueState' => $sThisJobState,
      'Pid'           => $$phProperties{'Pid'},
      'PoundName'     => $sPoundName,
      'Program'       => $$phProperties{'Program'},
      'Queue'         => $$phProperties{'ClientId'},
      'QueueTag'      => $$phProperties{'JqdQueueTag'},
      'Result'        => $sResult,
    }
  );

  if ($sResult eq "fail")
  {
    $$psError = $sLocalError,
    return undef;
  }

  1;
}


######################################################################
#
# VerifyConversionValues
#
######################################################################

sub VerifyConversionValues
{
  my ($phConversionValues, $phConversionChecks, $psError) = @_;

  foreach my $sKey (sort(keys(%$phConversionChecks)))
  {
    if ($$phConversionValues{$sKey} !~ /^$$phConversionChecks{$sKey}$/)
    {
      $$psError = "Conversion value ($$phConversionValues{$sKey}) for corresponding specification (%$sKey) is not valid";
      return undef;
    }
  }

  1;
}


######################################################################
#
# VerifyRunTimeEnvironment
#
######################################################################

sub VerifyRunTimeEnvironment
{
  my ($phProperties, $phRequiredProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure all required properties are defined and valid.
  #
  ####################################################################

  foreach my $sProperty (keys(%$phRequiredProperties))
  {
    my $sValue = $$phProperties{$sProperty};
    if (!defined($sValue) || $sValue !~ /^$$phRequiredProperties{$sProperty}$/)
    {
      $$psError = "$sProperty property ($sValue) is undefined or invalid";
      return undef;
    }
  }

  ####################################################################
  #
  # Make sure the config directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'NphConfigDirectory'} || !-R _)
  {
    $$psError = "Config directory ($$phProperties{'NphConfigDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the dynamic directory is writable.
  #
  ####################################################################

  if (!-d $$phProperties{'DynamicDirectory'} || !-W _)
  {
    $$psError = "Dynamic directory ($$phProperties{'DynamicDirectory'}) does not exist or is not writable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the logfiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'LogfilesDirectory'} || !-R _)
  {
    $$psError = "Logfiles directory ($$phProperties{'LogfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the profiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'ProfilesDirectory'} || !-R _)
  {
    $$psError = "Profiles directory ($$phProperties{'ProfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the incoming directory is writable.
  #
  ####################################################################

  if (!-d $$phProperties{'IncomingDirectory'} || !-W _)
  {
    $$psError = "Incoming directory ($$phProperties{'IncomingDirectory'}) does not exist or is not writable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the spool directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'SpoolDirectory'} || !-R _)
  {
    $$psError = "Spool directory ($$phProperties{'SpoolDirectory'}) does not exist or is not readable";
    return undef;
  }

  1;
}


######################################################################
#
# Yes
#
######################################################################

sub Yes
{
  return (defined($_[0]) && $_[0] =~ /^[Yy]$/) ? 1 : 0;
}

__END__

=pod

=head1 NAME

nph-webjob.cgi - Process webjob GET/JOB/PUT requests.

=head1 SYNOPSIS

    --- http.conf ---
    ...
    Include /var/webjob/config/apache/webjob.conf
    ...
    --- http.conf ---

    --- /var/webjob/config/apache/webjob.conf ---
    SetEnv WEBJOB_PROPERTIES_FILE /var/webjob/config/nph-webjob/nph-webjob.cfg

    ScriptAlias /cgi-client/ "/usr/local/webjob/cgi/cgi-client/"

    <Directory "/usr/local/webjob/cgi/cgi-client/">
      AllowOverride AuthConfig
      Options None
      Order allow,deny
      Allow from all
    </Directory>
    --- /var/webjob/config/apache/webjob.conf ---

=head1 DESCRIPTION

This utility implements a Common Gateway Interface (CGI), and its
purpose is to processes webjob GET/JOB/PUT requests.

The name of this utility must begin with the string "nph-".  This
naming convention, which stands for Non Parsed Headers, informs the
web server that the utility is responsible for generating the entire
HTTP response.  Non Parsed Headers are used by this utility because
they allow it to take full advantage of HTTP status code extensions.
This simplifies client-side error checking without giving up the
ability to generate application-specific responses.

The nph-webjob.cfg config file is used to control the CGI's runtime
behavior. To make use of it, set WEBJOB_PROPERTIES_FILE as
appropriate. Upon execution, the script looks for and processes the
file specified by this environment variable.

The cgi-client directory is where nph-webjob.cgi lives by default.
Access to this directory should be strictly controlled, and any
utilities placed in this directory should operate in the same security
context (or realm).  This utility is designed to read/write data
from/to the server's local disk on behalf of potentially hostile
clients.  Therefore, you'll want to maintain a tight rein to avoid any
unwanted results.

=head1 CONFIGURATION CONTROLS

This section describes the various controls that this program
recognizes.  In general, controls either shape runtime behavior or
provide information needed to perform a specific function.  Controls
and their values, one pair/line, are read from a file having the
following format.

    <control> = <value>

All controls are case insensitive, but, in general, their values
are not.  Comments may occur anywhere on a given line, and must
begin with a pound character (i.e., '#').  In any given line, all
text to the right of the first comment will be ignored.  White
space surrounding controls and values is ignored.

=over 5

=item B<BaseDirectory>: <path>

B<BaseDirectory> is the epicenter of activity.  The default value is
'/var/webjob'.

=item B<CapContentLength>: [Y|N]

B<CapContentLength> forces the script to abort when B<ContentLength>
exceeds B<MaxContentLength>.  The default value is 'N'.

=item B<ConfigSearchOrder>: [clients|commands|clients:commands|commands:clients]

B<ConfigSearchOrder> specifies the order in which custom config files
are sought out and processed.  Custom config files may be used to
override a predefined subset of the site-specific properties.  The
following tree enumerates the locations where global and custom config
files may exist.

    config
      |
      + nph-webjob
          |
          - nph-webjob.cfg # applies globally
          |
          + clients
          |   |
          |   - nph-webjob.cfg # applies globally
          |   |
          |   + <client-N>
          |       |
          |       - nph-webjob.cfg # applies to all commands for <client-N>
          |       |
          |       + <command-N>
          |           |
          |           - nph-webjob.cfg # applies only to <client-N>/<command-N>
          |
          + commands
              |
              - nph-webjob.cfg # applies globally
              |
              + <command-N>
                  |
                  - nph-webjob.cfg # applies to all clients for <command-N>
                  |
                  + <client-N>
                      |
                      - nph-webjob.cfg # applies only to <command-N>/<client-N>

As each config file is processed, its values trump those of any that
came before -- including any values that came from global config
files.  Supported values for this variable are 'clients', 'commands',
'clients:commands', and 'commands:clients'.  The default value is
'clients:commands'.

=item B<ConfigDirectory>: <path>

B<ConfigDirectory> is where various configuration files are
maintained.  The default value is '<BaseDirectory>/config'.

=item B<DsvMaxSignatureLength>: <integer>

B<DsvMaxSignatureLength> specifies the maximum signature length that
the script is willing to allow.  If the signature length exceeds this
limit (in bytes), the script will abort.  The default value is 256.

=item B<DsvNullSignature>: <string>

B<DsvNullSignature> specifies the signature for a null response.  The
script will abort if B<DsvRequireSignatures> is enabled and this value
is not defined.

=item B<DsvRequireSignatures>: [Y|N]

When active, B<DsvRequireSignatures> forces the script to abort if no
signature file is found, or if the signature does not meet basic
syntax checks.  A signature file must have the same basename as the
requested payload, and its suffix must match the value defined by
DsvSignatureSuffix.  The default value is 'N'.

=item B<DsvSignatureSuffix>: <suffix>

B<DsvSignatureSuffix> specifies the suffix assigned to and used by
signature files.  A signature file must have the same basename as the
requested payload, and its suffix must match the value defined by this
property.  The default value is '.sig'.

=item B<DynamicDirectory>: <path>

B<DynamicDirectory> is where dynamic content is stored.  The default
value is '<BaseDirectory>/dynamic'.

=item B<EnableGetService>: [Y|N]

When active, B<EnableGetService> allows the script to accept and
process valid GET requests.  This control can be overridden, and its
intended purpose is to administratively disable access for valid
clients (e.g., to force a failover).  The default value is 'Y'.

=item B<EnableHostAccessList>: [Y|N]

When active, B<EnableHostAccessList> causes the script to consult a
host access list to determine whether or not the client will be
granted access based on its IP address.  Note: This mechanism will
fail closed if it's enabled and the access list is missing or does not
contain the necessary client ID to IP address mappings.  The default
value is 'N'.

=item B<EnableConfigOverrides>: [Y|N]

When active, B<EnableConfigOverrides> causes the script to seek out
and process additional client- and/or command-specific config files
(see B<ConfigSearchOrder>).  The default value is 'Y'.

=item B<EnableJobQueues>: [Y|N]

When active, B<EnableJobQueues> allows the script to pull jobs from
job queues and answer JOB requests.  The default value is 'Y'.

=item B<EnableLogging>: [Y|N]

When active, B<EnableLogging> forces the script to generate a log
message for each request.  If the designated LogFile can not be
opened, the log message will be written to STDERR.  The default value
is 'Y'.

=item B<EnablePutService>: [Y|N]

When active, B<EnablePutService> allows the script to accept and
process valid PUT requests.  This control can be overridden, and its
intended purpose is to administratively disable access for valid
clients (e.g., to prevent unwanted uploads).  The default value is
'Y'.

=item B<FolderList>: <item[:item[...]]>

B<FolderList> specifies locations where shared programs can be found.
If a requested file does not exist in a given client's commands
directory, the B<FolderList> is searched according to the order given
here.  The list delimiter is a colon (i.e., ':').  An example list
would be 'common:shared'.  The default value is 'common'.

=item B<GetHookCommandLine>: <command>

B<GetHookCommandLine> is a command string consisting of zero or more
conversion specifications optionally interspersed with zero or more
plain text characters.  The following conversion specifications are
supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %cmd = Client-requested command as a string
    %d   = Day of the month as a decimal number (01-31)
    %dynamic_cmd = Alias for %dynamic_out
    %dynamic_dir = Full path of the dynamic directory
    %dynamic_out = Full path of the dynamic output file that is to be created
    %dynamic_sig = Full path of the dynamic signature file that is to be created
    %H   = Hour as a decimal number (00-23)
    %ip  = IP address as a dotted quad string
    %jid = Job ID as a string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %pid = Process ID of server-side CGI script
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %system_version = System version as given by the client
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

For example, the following command string:

    { if [ "%H"X = "23"X ] ; then cp /<path>/checker %dynamic_out ; else { echo ; echo ":" ; } > %dynamic_out ; fi ; }

will copy the checker command to the filename represented by
%dynamic_out if the current hour is 23.  Otherwise, a script that
simpy returns true is created on-the-fly -- effectively turning the
job into a NOOP.

If the specified command is an empty string, then the hook mechanism
is (effectively) disabled, and the condition is logged.  However, if
the hook is disabled (i.e., B<GetHookEnable>=N), then this control is
ignored.

=item B<GetHookLogDivertedOutput>: [Y|N]

B<GetHookLogDivertedOutput> causes the script to divert any output on
stdout/stderr to nph-webjob-hook.{err,out}.  Normally, output for
these streams is discarded since hook commands should be written to
keep them clean (e.g., by logging to a file).  The default value is
'N'.

Note: If a given hook command is not working properly, you can use
this option to capture any output that was written to stdout/stderr.
That, in turn, may help you debug the problem.

Note: If logging is disabled (i.e., B<EnableLogging>=N), this control
is ignored.

=item B<GetHookEnable>: [Y|N]

When active, B<GetHookEnable> causes the script to execute the command
line specified by B<GetHookCommandLine>.  The behavior of the hook
mechanism is to launch a subprocess and wait for it to finish.  The
purpose of the subprocess is to create dynamic content that will be
delivered to the client.  The hook mechanism is highly configurable --
config file overrides are fully supported, multiple conversion tokens
are available, and the user determines what, if any, commands are
executed when the hook is activated.  Currently, hooks are only
activated if they are enabled and a hook command has been defined.
The default value is 'N'.

=item B<GetHookStatus>: <integer>

B<GetHookStatus> determines the meaning of success for the hook.  If
the specified value does not match the exit status as returned by
system(), the hook is considered a failure and the entire job is
aborted.  Otherwise, processing continues as normal.  The default
value is '0'.

=item B<GetHookStatusMap>: <integer:integer>[,<integer:integer>[...]]

B<GetHookStatusMap> allows the user to map hook exit codes into HTTP
response codes.  No mappings are defined by default.

=item B<GetTriggerCommandLine>: <command>

B<GetTriggerCommandLine> is a command string consisting of zero or
more conversion specifications optionally interspersed with zero or
more plain text characters.  The following conversion specifications
are supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %cmd = Client-requested command as a string
    %d   = Day of the month as a decimal number (01-31)
    %H   = Hour as a decimal number (00-23)
    %jid = Job ID as a string
    %ip  = IP address as a dotted quad string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %pid = Process ID of server-side CGI script
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

For example, the following command string:

  echo "%Y-%m-%d %H:%M:%S GET %jid %cid" >> /var/log/%cid.jids

will append the current date, time, request method, job ID, and client
ID to a client-specific file in /var/log.

If the specified command is an empty string, then the trigger
mechanism is (effectively) disabled, and the condition is logged.
However, if the trigger is disabled (i.e., GetTriggerEnable=N), then
this control is ignored.

Note: Triggers are not currently supported on Windows platforms.

=item B<GetTriggerEnable>: [Y|N]

When active, B<GetTriggerEnable> causes the script to execute the
command line specified by B<GetTriggerCommandLine>.  The behavior of
the trigger mechanism is to launch a subprocess and continue with the
main line of execution.  In particular, the script will not block or
wait for the subprocess to finish, nor will it attempt check the
status or cleanup after the subprocess.  The trigger mechanism is
highly configurable -- config file overrides are fully supported,
multiple conversion tokens are available, and the user determines
what, if any, commands are executed when the trigger is pulled.
Currently, triggers are only pulled if they are enabled, a trigger
command has been defined, and the HTTP status code is 200.  The
default value is 'N'.

Note: Triggers are not currently supported on Windows platforms.

=item B<IncomingDirectory>: <path>

B<IncomingDirectory> is where various client uploads are stored.  The
default value is '<BaseDirectory>/incoming'.

=item B<JobQueueActive>: [Y|N]

When active, B<JobQueueActive> causes the script to pull jobs from the
specified job queue.  The default value is 'Y'.

=item B<JobQueueDirectory>: <path>

B<JobQueueDirectory> is where client job queues are maintained.  The
default value is '<SpoolDirectory>/jqd'.

=item B<JobQueuePqActiveLimit>: <integer>

B<JobQueuePqActiveLimit> specifies the maximum number of parallel jobs
that may be in an active state (i.e., either 'sent' or 'open').  If
B<JobQueueActive> is disabled, this control is ignored.  A value of
zero means there is no limit.  The default value is '0'.

=item B<JobQueuePqAnswerLimit>: <integer>

B<JobQueuePqAnswerLimit> specifies the maximum number of parallel jobs
that will be returned to the client.  If B<JobQueueActive> is
disabled, this control is ignored.  A value of zero means there is no
limit.  The default value is '0'.

phProperties{'JobQueuePqAnswerLimit'} = 0;

=item B<JobQueueSqActiveLimit>: <integer>

B<JobQueueSqActiveLimit> specifies the maximum number of serial jobs
that may be in an active state (i.e., either 'sent' or 'open').  If
B<JobQueueActive> is disabled, this control is ignored.  A value of
zero means there is no limit.  The default value is '1'.

=item B<JobQueueSqAnswerLimit>: <integer>

B<JobQueueSqAnswerLimit> specifies the maximum number of serial jobs
that will be returned to the client.  If B<JobQueueActive> is
disabled, this control is ignored.  A value of zero means there is no
limit.  The default value is '0'.

=item B<LogfilesDirectory>: <path>

B<LogfilesDirectory> is where log files are stored.  The default value
is '<BaseDirectory>/logfiles'.

=item B<MaxContentLength>: <integer>

B<MaxContentLength> specifies the largest upload in bytes the script
will accept.  If CapContentLength is disabled, this control has no
effect.  The default value is '100000000'.

=item B<OverwriteExistingFiles>: [Y|N]

When active, B<OverwriteExistingFiles> forces the script to unlink
existing files prior to writing the uploaded data.  The default
PutNameFormat used by this script attempts to prevent filename
collisions.  However, that behavior is user-defined, and in some
cases, it may be desirable to specify a PutNameFormat that is
guaranteed to create collisions.  In those situations, this control
must be enabled to produce the desired outcome (i.e., allow existing
files with the same name to be overwritten).  The default value is
'N'.

=item B<ProfilesDirectory>: <path>

B<ProfilesDirectory> is where client profiles and shared commands are
maintained.  The default value is '<BaseDirectory>/profiles'.

=item B<PutHookCommandLine>: <command>

B<PutHookCommandLine> is a command string consisting of zero or more
conversion specifications optionally interspersed with zero or more
plain text characters.  The following conversion specifications are
supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %cmd = Client-requested command as a string
    %d   = Day of the month as a decimal number (01-31)
    %dynamic_dir = Full path of the dynamic directory
    %dynamic_out = Full path of the dynamic output file that is to be created
    %dynamic_sig = Full path of the dynamic signature file that is to be created
    %env = Full path to .env file as a string
    %err = Full path to .err file as a string
    %H   = Hour as a decimal number (00-23)
    %ip  = IP address as a dotted quad string
    %jid = Job ID as a string
    %job_time = Time between the creation of the job ID and the .rdy file is created
    %jqt = Job Queue Tag
    %lck = Full path to .lck file as a string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %out = Full path to .out file as a string
    %pid = Process ID of server-side CGI script
    %rdy = Full path to .rdy file as a string
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %system_version = System version as given by the client
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

If the specified command is an empty string, then the hook mechanism
is (effectively) disabled, and the condition is logged.  However, if
the hook is disabled (i.e., B<PutHookEnable>=N), then this control is
ignored.

=item B<PutHookLogDivertedOutput>: [Y|N]

B<PutHookLogDivertedOutput> causes the script to divert any output on
stdout/stderr to nph-webjob-hook.{err,out}.  Normally, output for
these streams is discarded since hook commands should be written to
keep them clean (e.g., by logging to a file).  The default value is
'N'.

Note: If a given hook command is not working properly, you can use
this option to capture any output that was written to stdout/stderr.
That, in turn, may help you debug the problem.

Note: If logging is disabled (i.e., B<EnableLogging>=N), this control
is ignored.

=item B<PutHookEnable>: [Y|N]

When active, B<PutHookEnable> causes the script to execute the command
line specified by B<PutHookCommandLine>.  The behavior of the hook
mechanism is to launch a subprocess and wait for it to finish.  The
purpose of the subprocess is to create dynamic content that will be
delivered to the client.  The hook mechanism is highly configurable --
config file overrides are fully supported, multiple conversion tokens
are available, and the user determines what, if any, commands are
executed when the hook is activated.  Currently, hooks are only
activated if they are enabled and a hook command has been defined.
The default value is 'N'.

=item B<PutHookStatus>: <integer>

B<PutHookStatus> determines the meaning of success for the hook.  If
the specified value does not match the exit status as returned by
system(), the hook is considered a failure and the entire job is
aborted.  Otherwise, processing continues as normal.  The default
value is '0'.

=item B<PutHookStatusMap>: <integer:integer>[,<integer:integer>[...]]

B<PutHookStatusMap> allows the user to map hook exit codes into HTTP
response codes.  No mappings are defined by default.

=item B<PutNameFormat>: <format>

B<PutNameFormat> controls how files are named/saved in the incoming
directory.  In other words, it controls the directory's layout.
Basically, B<PutNameFormat> is a format string consisting of zero or
more conversion specifications optionally interspersed with zero or
more plain text characters.  The following conversion specifications
are supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %cmd = Client-requested command as a string
    %d   = Day of the month as a decimal number (01-31)
    %H   = Hour as a decimal number (00-23)
    %ip  = IP address as a dotted quad string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %pid = Process ID of server-side CGI script
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

For example, the following format string:

  "%cmd/%ip_%Y-%m-%d_%H.%M.%S"

will cause uploaded files to be stored in sub-directories that
correspond to the name of the command executed, and each output
filename will consist of an IP address, date, and time.

The added flexibility provided by this scheme means that it is
possible to create format strings that are problematic.  Consider the
following string:

  "%cid/%cmd"

While this is a legal format string, it is likely to cause name
collisions (e.g., the same client runs the same command two or more
times).  Therefore, it is important to create format strings that
contain enough job specific information to distinguish one set of
uploaded files from another.

The default value is '%cid/%cmd/%Y-%m-%d/%H.%M.%S.%pid'.

=item B<PutTriggerCommandLine>: <command>

B<PutTriggerCommandLine> is a command string consisting of zero or
more conversion specifications optionally interspersed with zero or
more plain text characters.  The following conversion specifications
are supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %cmd = Client-requested command as a string
    %d   = Day of the month as a decimal number (01-31)
    %env = Full path to .env file as a string
    %err = Full path to .err file as a string
    %H   = Hour as a decimal number (00-23)
    %ip  = IP address as a dotted quad string
    %jid = Job ID as a string
    %job_time = Time between the creation of the job ID and the .rdy file is created
    %jqt = Job Queue Tag
    %lck = Full path to .lck file as a string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %out = Full path to .out file as a string
    %pid = Process ID of server-side CGI script
    %rdy = Full path to .rdy file as a string
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

For example, the following command string:

  echo "%Y-%m-%d %H:%M:%S PUT %jid %cid" >> /var/log/%cid.jids

will append the current date, time, request method, job ID, and client
ID to a client-specific file in /var/log.

If the specified command is an empty string, then the trigger
mechanism is (effectively) disabled, and the condition is logged.
However, if the trigger is disabled (i.e., B<PutTriggerEnable>=N),
then this control is ignored.

Note: Triggers are not currently supported on Windows platforms.

=item B<PutTriggerEnable>: [Y|N]

When active, B<PutTriggerEnable> causes the script to execute the
command line specified by B<PutTriggerCommandLine>.  The behavior of
the trigger mechanism is to launch a subprocess and continue with the
main line of execution.  In particular, the script will not block or
wait for the subprocess to finish, nor will it attempt check the
status or cleanup after the subprocess.  The trigger mechanism is
highly configurable -- config file overrides are fully supported,
multiple conversion tokens are available, and the user determines
what, if any, commands are executed when the trigger is pulled.
Currently, triggers are only pulled if they are enabled, a trigger
command has been defined, and the HTTP status code is 200.  The
default value is 'N'.

Note: Triggers are not currently supported on Windows platforms.

=item B<RequireMatch>: [Y|N]

B<RequireMatch> forces the script to abort unless ClientId matches
RemoteUser.  When this value is disabled, any authenticated user will
be allowed to issue requests for a given client.  Disabling
B<RequireUser> implicitly disables B<RequireMatch>.  The default value
is 'Y'.

=item B<RequireUser>: [Y|N]

B<RequireUser> forces the script to abort unless RemoteUser has been
set.  The default value is 'Y'.

=item B<ServerId>: <string>

B<ServerId> specifies the identity assigned to the WebJob server.  The
default value is 'server_1'.

=item B<SpoolDirectory>: <path>

B<SpoolDirectory> is where spools and queues are maintained.  The
default value is '<BaseDirectory>/spool'.

=item B<SslRequireCn>: [Y|N]

B<SslRequireCn> forces the script to abort unless SslClientSDnCn has
been set.  If SslRequireSsl is disabled, this and all other SSL
controls are ignored.  The default value is 'N'.

=item B<SslRequireMatch>: [Y|N]

B<SslRequireMatch> forces the script to abort if ClientId does not
match SslClientSDnCn.  When this control is disabled, access will be
governed by B<RequireMatch>.  Disabling B<SslRequireCn> implicitly
disables B<SslRequireMatch>.  Also, if SslRequireSsl is disabled, this
and all other SSL controls are ignored.  The B<SslRequireMatch> check
is performed prior to (not instead of) the B<RequireMatch> check.  The
default value is 'N'.

=item B<SslRequireSsl>: [Y|N]

B<SslRequireSsl> forces the script to abort unless the client is
speaking HTTPS.  Disabling B<SslRequireSsl> implicitly disables all
SSL-related controls.  The default value is 'Y'.

=item B<UseGmt>: [Y|N]

When active, B<UseGmt> forces the script to convert all time values to
GMT.  Otherwise, time values are converted to local time.  The default
value is 'N'.

=back

=head1 FILES

=over 5

=item B<BaseDirectory>/logfiles/jqd.log

This utility writes JQD-related log messages to jqd.log.  This file
contains the following fields separated by whitespace in the given
order: Date, Time, JobId, Program, Pid, Creator, Queue, QueueTag,
OldQueueState, NewQueueState, Command, CommandSize, PoundName,
JobGroup, Result, and Message.  A single hyphen, '-', is used to
denote the fact that a particular field had an empty or undefined
value.  The Message field is set off from the other fields with a
double hyphen '--' because it uses a free form text format that can
include whitespace.

=item B<BaseDirectory>/logfiles/nph-webjob.log

This utility writes log messages to nph-webjob.log (see
nph-webjob.cfg).  This file contains the following fields separated by
whitespace in the given order: Date, Time, JobId, RemoteUser,
RemoteAddress, RequestMethod, ClientId, Filename, ContentLength,
ServerContentLength, Duration, ReturnStatus, and ErrorMessage.  A
single hyphen, '-', is used to denote the fact that a particular field
had an empty or undefined value.  The ErrorMessage field is set off
from the other fields with a double hyphen '--' because it uses a free
form text format that can include whitespace.

=item B<BaseDirectory>/logfiles/nph-webjob-hook.err

This utility captures hook-related output on stderr and writes it to
nph-webjob-hook.err.  This file is a bit bucket for dirty hook
commands.  It is also useful as a debugging aid when creating new hook
commands.

=item B<BaseDirectory>/logfiles/nph-webjob-hook.log

This utility writes hook-related log messages to nph-webjob-hook.log.
This file contains the following fields separated by whitespace in the
given order: Date, Time, JobId, RequestMethod, ClientId, Filename,
PidLabel, Pid, State, and Message.  A single hyphen, '-', is used to
denote the fact that a particular field had an empty or undefined
value.  The Message field is set off from the other fields with a
double hyphen '--' because it uses a free form text format that can
include whitespace.

=item B<BaseDirectory>/logfiles/nph-webjob-hook.out

This utility captures hook-related output on stdout and writes it to
nph-webjob-hook.out.  This file is a bit bucket for dirty hook
commands.  It is also useful as a debugging aid when creating new hook
commands.

=item B<BaseDirectory>/logfiles/nph-webjob-trigger.log

This utility writes trigger-related log messages to
nph-webjob-trigger.log.  This file contains the following fields
separated by whitespace in the given order: Date, Time, JobId,
RequestMethod, ClientId, Filename, PidLabel, Pid, State, and Message.
A single hyphen, '-', is used to denote the fact that a particular
field had an empty or undefined value.  The Message field is set off
from the other fields with a double hyphen '--' because it uses a free
form text format that can include whitespace.

=back

=head1 AUTHOR

Klayton Monroe

=head1 SEE ALSO

nph-config.cgi(1), webjob(1)

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
