#!/usr/bin/perl -w
######################################################################
#
# $Id: nph-config.cgi,v 1.36 2012/01/07 07:56:14 mavrik Exp $
#
######################################################################
#
# Copyright 2008-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Process config file requests.
#
######################################################################

use strict;
use Fcntl qw(:flock);
use FindBin qw($Bin $RealBin); use lib ("$Bin/../../lib/perl5/site_perl", "$RealBin/../../lib/perl5/site_perl", "/usr/local/webjob/lib/perl5/site_perl", "/opt/local/webjob/lib/perl5/site_perl");
use Time::HiRes qw(gettimeofday tv_interval);
use WebJob::CgiRoutines;
use WebJob::KvpRoutines 1.029;
use WebJob::LogRoutines;
use WebJob::Properties 1.035;
use WebJob::ValidationRoutines 1.004;

######################################################################
#
# Main Routine
#
######################################################################

  my (%hProperties, %hReturnCodes, $sLocalError);

  %hReturnCodes =
  (
    '200' => "OK",
    '204' => "No Content",
    '206' => "Partial Content",
    '403' => "Forbidden",
    '404' => "Not Found",
    '405' => "Method Not Allowed",
    '409' => "Conflict",
    '412' => "Precondition Failed",
    '450' => "Invalid Query",
    '452' => "Username Undefined",
    '453' => "Username-ClientId Mismatch",
    '454' => "Content-Length Undefined",
    '455' => "Content-Length Exceeds Limit",
    '456' => "Content-Length Mismatch",
    '457' => "File Not Available",
    '458' => "Invalid Protocol",
    '470' => "CommonName Undefined",
    '471' => "CommonName-ClientId Mismatch",
    '490' => "User-Defined 490",
    '491' => "User-Defined 491",
    '492' => "User-Defined 492",
    '493' => "User-Defined 493",
    '494' => "User-Defined 494",
    '495' => "User-Defined 495",
    '496' => "User-Defined 496",
    '497' => "User-Defined 497",
    '498' => "User-Defined 498",
    '499' => "User-Defined 499",
    '500' => "Internal Server Error",
    '503' => "Service Unavailable",
    '550' => "Internal Server Initialization Error",
    '551' => "Internal Server Mapping Error",
    '553' => "Service Disabled",
    '590' => "User-Defined 590",
    '591' => "User-Defined 591",
    '592' => "User-Defined 592",
    '593' => "User-Defined 593",
    '594' => "User-Defined 594",
    '595' => "User-Defined 595",
    '596' => "User-Defined 596",
    '597' => "User-Defined 597",
    '598' => "User-Defined 598",
    '599' => "User-Defined 599",
  );

  ####################################################################
  #
  # Punch in and go to work.
  #
  ####################################################################

  ($hProperties{'StartTime'}, $hProperties{'StartTimeUsec'}) = gettimeofday();
  $hProperties{'Program'} = "nph-config.cgi";
  $hProperties{'Version'} = sprintf("%s %s", __FILE__, ('$Revision: 1.36 $' =~ /^.Revision: ([\d.]+)/));

  ####################################################################
  #
  # Create/Verify run time environment, and process GET/PUT requests.
  #
  ####################################################################

  if (!defined(CreateRunTimeEnvironment(\%hProperties, \$sLocalError)))
  {
    $hProperties{'ReturnStatus'} = 550;
    $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
    $hProperties{'ErrorMessage'} = $sLocalError;
  }
  else
  {
    if (Yes($hProperties{'SslRequireSsl'}) && (!defined($hProperties{'Https'}) || $hProperties{'Https'} !~ /^[Oo][Nn]$/))
    {
      $hProperties{'ReturnStatus'} = 458;
      $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
      $hProperties{'ErrorMessage'} = "HTTPS required, but client is speaking HTTP";
    }
    else
    {
      if ($hProperties{'RequestMethod'} eq "GET")
      {
        $hProperties{'ReturnStatus'} = ProcessGetRequest(\%hProperties, \$sLocalError);
        $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
        $hProperties{'ErrorMessage'} = $sLocalError;
      }
      else
      {
        $hProperties{'ReturnStatus'} = 405;
        $hProperties{'ReturnReason'} = $hReturnCodes{$hProperties{'ReturnStatus'}};
        $hProperties{'ErrorMessage'} = "Method ($hProperties{'RequestMethod'}) not allowed";
      }
    }
  }
  $hProperties{'ServerContentLength'} = SendResponse(\%hProperties);

  ####################################################################
  #
  # Conditionally log the transaction.
  #
  ####################################################################

  if (Yes($hProperties{'EnableLogging'}))
  {
    NphLogMessage(\%hProperties);
  }

  ####################################################################
  #
  # Clean up and go home.
  #
  ####################################################################

  1;


######################################################################
#
# CheckHostAccessList
#
######################################################################

sub CheckHostAccessList
{
  my ($sCidrList, $sIp) = @_;

  my $sCidrRegex = qq(((?:\\d{1,3}[.]){3}(?:\\d{1,3}))\/(\\d{1,2}));

  my $sIpRegex   = qq((?:\\d{1,3}[.]){3}(?:\\d{1,3}));

  if (!defined($sCidrList) || !defined($sIp))
  {
    return 0;
  }

  if ($sIp !~ /^$sIpRegex$/)
  {
    return 0;
  }

  $sCidrList =~ s/\s+//g; # Remove whitespace.

  foreach my $sCidr (split(/,/, $sCidrList))
  {
    next if ($sCidr !~ /^$sCidrRegex$/);
    my ($sNetwork, $sMaskBits) = ($1, $2);
    my $sBits2Clear = 32 - $sMaskBits - 1;
    my $sNetMask = 0xffffffff;
    foreach my $sBit (0..$sBits2Clear)
    {
      $sNetMask ^= 1 << $sBit;
    }
    my $sBinNetwork = hex(sprintf("%02x%02x%02x%02x", split(/\./, $sNetwork)));
    my $sBinNetmask = hex(sprintf("%08x", $sNetMask));
    $sBinNetwork &= $sBinNetmask; # Cleanup the network address -- in case the user did not.
    my $sBinIp = hex(sprintf("%02x%02x%02x%02x", split(/\./, $sIp)));
    if (($sBinIp & $sBinNetmask) == $sBinNetwork)
    {
      return 1; # We have a winner!
    }
  }

  return 0;
}


######################################################################
#
# CreateRunTimeEnvironment
#
######################################################################

sub CreateRunTimeEnvironment
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Put input/output streams in binary mode.
  #
  ####################################################################

  foreach my $sHandle (\*STDIN, \*STDOUT, \*STDERR)
  {
    binmode($sHandle);
  }

  ####################################################################
  #
  # Initialize regex variables.
  #
  ####################################################################

  $$phProperties{'CommonRegexes'} = PropertiesGetGlobalRegexes();

  ####################################################################
  #
  # Initialize environment-specific variables. Pull in SSL-related
  # variables, but only if HTTPS is defined and on.
  #
  ####################################################################

  $$phProperties{'ContentLength'}  = $ENV{'CONTENT_LENGTH'};
  $$phProperties{'Https'}          = $ENV{'HTTPS'};
  $$phProperties{'QueryString'}    = $ENV{'QUERY_STRING'};
  $$phProperties{'RemoteAddress'}  = $ENV{'REMOTE_ADDR'};
  $$phProperties{'RemoteUser'}     = $ENV{'REMOTE_USER'};
  $$phProperties{'RequestMethod'}  = $ENV{'REQUEST_METHOD'} || "";
  $$phProperties{'ServerSoftware'} = $ENV{'SERVER_SOFTWARE'};
  $$phProperties{'PropertiesFile'} = $ENV{'CONFIG_PROPERTIES_FILE'};

  if (defined($hProperties{'Https'}) && $hProperties{'Https'} =~ /^[Oo][Nn]$/)
  {
    $$phProperties{'SslClientSDnCn'} = $ENV{'SSL_CLIENT_S_DN_CN'} || "";
  }
  else
  {
    $$phProperties{'SslClientSDnCn'} = "";
  }

  ####################################################################
  #
  # Initialize platform-specific variables.
  #
  ####################################################################

  if ($^O =~ /MSWin32/i)
  {
    $$phProperties{'OsClass'} = "WINX";
    $$phProperties{'Newline'} = "\r\n";
    $$phProperties{'NullDevice'} = "nul";
  }
  else
  {
    $$phProperties{'OsClass'} = "UNIX";
    $$phProperties{'Newline'} = "\n";
    $$phProperties{'NullDevice'} = "/dev/null";
    umask(022);
  }

  ####################################################################
  #
  # Initialize site-specific variables. Note that the properties
  # listed in the custom template are a subset of those in the
  # global template. All values in the custom template may may be
  # overridden through the use of client- and/or command-specific
  # config files. See WebJob::Properties for the exact lists.
  #
  ####################################################################

  my ($sLocalError);

  $$phProperties{'CommonTemplates'} = PropertiesGetGlobalTemplates();

  $$phProperties{'GlobalConfigTemplate'} = $$phProperties{'CommonTemplates'}{'nph-config.global'};

  $$phProperties{'CustomConfigTemplate'} = $$phProperties{'CommonTemplates'}{'nph-config.custom'};

  if (!GetGlobalConfigProperties($phProperties, $$phProperties{'GlobalConfigTemplate'}, \$sLocalError))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Initialize derived variables. If key directories weren't defined
  # in the config file, define them relative to BaseDirectory.
  #
  ####################################################################

  foreach my $sName ("Config", "Dynamic", "Logfiles", "Profiles", "Spool")
  {
    my $sKey = $sName . "Directory";
    if (!exists($$phProperties{$sKey}) || !defined($$phProperties{$sKey}) || length($$phProperties{$sKey}) < 1)
    {
      $$phProperties{$sKey} = $$phProperties{'BaseDirectory'} . "/" . lc($sName);
    }
  }

  $$phProperties{'NphConfigDirectory'} = $$phProperties{'ConfigDirectory'} . "/" . "nph-config";
  $$phProperties{'LogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-config.log";
  $$phProperties{'HookErrFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-config-hook.err";
  $$phProperties{'HookLogFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-config-hook.log";
  $$phProperties{'HookOutFile'} = $$phProperties{'LogfilesDirectory'} . "/nph-config-hook.out";

  ####################################################################
  #
  # Conditionally initialize the host access list.
  #
  ####################################################################

  if (Yes($$phProperties{'EnableHostAccessList'}))
  {
    if (!defined(SetupHostAccessList($phProperties, \$sLocalError)))
    {
      $$psError = $sLocalError;
      return undef;
    }
  }

  ####################################################################
  #
  # Verify run time environment.
  #
  ####################################################################

  if (!defined(VerifyRunTimeEnvironment($phProperties, $$phProperties{'GlobalConfigTemplate'}, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Initialize and verify the job ID.
  #
  ####################################################################

  if ($$phProperties{'RequestMethod'} eq "GET")
  {
    $$phProperties{'JobId'} = sprintf("%s_%010u_%06d_%05d", $$phProperties{'ServerId'}, $$phProperties{'StartTime'}, $$phProperties{'StartTimeUsec'}, $$);
  }
  else
  {
    $$phProperties{'JobId'} = $ENV{'HTTP_JOB_ID'} || "NA";
  }

  if (!defined($$phProperties{'JobId'}) || $$phProperties{'JobId'} !~ /^(?:NA|$$phProperties{'CommonRegexes'}{'JobId'})$/)
  {
    $$psError = "JobId ($$phProperties{'JobId'}) is undefined or invalid";
    return undef;
  }

  1;
}


######################################################################
#
# ExpandConversionString
#
######################################################################

sub ExpandConversionString
{
  my ($sConversionString, $phConversionValues, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sConversionString) || !defined($phConversionValues))
  {
    $$psError = "Unable to proceed in ExpandConversionString() due to missing or undefined inputs";
    return undef;
  }

  ####################################################################
  #
  # Expand the provided conversion string. The TokenList must be
  # processed in reverse order (i.e., from longest to shortest).
  # Otherwise, a token such as %pid would be interpreted as the token
  # %p followed by the literal string "id". Once all regular
  # conversions are done, check for and convert any literal '%'s.
  #
  ####################################################################

  my ($sExpandedConversionString, $sTokenList);

  $sTokenList = join('|', reverse(sort(keys(%$phConversionValues))));
  $sExpandedConversionString = $sConversionString;
  $sExpandedConversionString =~ s/%($sTokenList)/$$phConversionValues{$1}/ge;
  $sExpandedConversionString =~ s/%%/%/g;

  return $sExpandedConversionString;
}


######################################################################
#
# ExpandGetHookCommandLine
#
######################################################################

sub ExpandGetHookCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'Filename',
      'ClientId',
      'DynamicDirectory',
      'CommonRegexes',
      'GetHookCommandLine',
      'JobId',
      'RemoteAddress',
      'ServerId',
      'HookEpoch',
      'UseGmt',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Create conversion values.
  #
  ####################################################################

  my
  (
    $sSecond,
    $sMinute,
    $sHour,
    $sMonthDay,
    $sMonth,
    $sYear,
    $sWeekDay,
    $sYearDay,
    $sDaylightSavings,
  ) = (Yes($$phProperties{'UseGmt'})) ? gmtime($$phProperties{'HookEpoch'}) : localtime($$phProperties{'HookEpoch'});

  my %hConversionValues =
  (
    'A'   => PropertiesGetGlobalKvps()->{'WeekDays'}[$sWeekDay],
    'a'   => PropertiesGetGlobalKvps()->{'WeekDaysAbbreviated'}[$sWeekDay],
    'cid' => $$phProperties{'ClientId'},
    'd'   => sprintf("%02d", $sMonthDay),
    'dynamic_dir' => $$phProperties{'DynamicDirectory'},
    'dynamic_out' => $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $$phProperties{'JobId'} . ".get",
    'file' => $$phProperties{'Filename'},
    'H'   => sprintf("%02d", $sHour),
    'ip'  => $$phProperties{'RemoteAddress'},
    'jid' => $$phProperties{'JobId'},
    'M'   => sprintf("%02d", $sMinute),
    'm'   => sprintf("%02d", $sMonth + 1),
    'pid' => sprintf("%05d", $$),
    'S'   => sprintf("%02d", $sSecond),
    's'   => sprintf("%010u", $$phProperties{'HookEpoch'}),
    'sid' => $$phProperties{'ServerId'},
    'u'   => sprintf("%d", $sWeekDay + 1),
    'w'   => sprintf("%d", $sWeekDay),
    'Y'   => sprintf("%04d", $sYear + 1900),
  );

  ####################################################################
  #
  # Verify conversion values.
  #
  ####################################################################

  my ($sLocalError);

  my %hConversionChecks =
  (
    'A'   => $$phProperties{'CommonRegexes'}{'strftime_A'},
    'a'   => $$phProperties{'CommonRegexes'}{'strftime_a'},
    'cid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'd'   => $$phProperties{'CommonRegexes'}{'strftime_d'},
    'dynamic_dir' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'dynamic_out' => $$phProperties{'CommonRegexes'}{'FilePath'},
    'file' => $$phProperties{'CommonRegexes'}{'ClientSuppliedFilename'},
    'H'   => $$phProperties{'CommonRegexes'}{'strftime_H'},
    'ip'  => $$phProperties{'CommonRegexes'}{'Ip'},
    'jid' => $$phProperties{'CommonRegexes'}{'JobId'},
    'M'   => $$phProperties{'CommonRegexes'}{'strftime_M'},
    'm'   => $$phProperties{'CommonRegexes'}{'strftime_m'},
    'pid' => $$phProperties{'CommonRegexes'}{'ProcessId'},
    'S'   => $$phProperties{'CommonRegexes'}{'strftime_S'},
    's'   => $$phProperties{'CommonRegexes'}{'strftime_s'},
    'sid' => $$phProperties{'CommonRegexes'}{'ClientId'},
    'u'   => $$phProperties{'CommonRegexes'}{'strftime_u'},
    'w'   => $$phProperties{'CommonRegexes'}{'strftime_w'},
    'Y'   => $$phProperties{'CommonRegexes'}{'strftime_Y'},
  );

  if (!defined(VerifyConversionValues(\%hConversionValues, \%hConversionChecks, \$sLocalError)))
  {
    $$psError = $sLocalError;
    return undef;
  }

  ####################################################################
  #
  # Expand conversion values.
  #
  ####################################################################

  my $sHookCommandLine = ExpandConversionString($$phProperties{'GetHookCommandLine'}, \%hConversionValues, \$sLocalError);
  if (!defined($sHookCommandLine))
  {
    $$psError = $sLocalError;
    return undef;
  }

  return $sHookCommandLine;
}


######################################################################
#
# GetCustomConfigProperties
#
######################################################################

sub GetCustomConfigProperties
{
  my ($phProperties, $phTemplate, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  foreach my $sInput ($phProperties, $phTemplate, $psError)
  {
    if (!defined($sInput))
    {
      $$psError = "Unable to proceed in GetCustomConfigProperties() due to missing or undefined inputs" if (defined($psError));
      return undef;
    }
  }

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'ClientId',
      'ConfigSearchOrder',
      'CommonRegexes',
      'Filename',
      'NphConfigDirectory',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Search for custom config files using the specified order. As each
  # config file is processed, its values trump those of any that came
  # before -- including any values that came from global config files.
  #
  ####################################################################

  foreach my $sFolder (split(/:/, $$phProperties{'ConfigSearchOrder'}))
  {
    ##################################################################
    #
    # The first config file defined by this loop applies globally.
    # The second applies to a particular client or config, and the
    # third applies to a particular client/config pair.
    #
    ##################################################################

    my ($sFile1, $sFile2, $sFile3);

    $sFile1 = $sFile2 = $sFile3 = $$phProperties{'NphConfigDirectory'} . "/" . $sFolder . "/";

    if ($sFolder =~ /^clients$/)
    {
      $sFile1 .= "nph-config.cfg";
      $sFile2 .= $$phProperties{'ClientId'} . "/" . "nph-config.cfg";
      $sFile3 .= $$phProperties{'ClientId'} . "/" . $$phProperties{'Filename'} . "/" . "nph-config.cfg";
    }
    elsif ($sFolder =~ /^configs$/)
    {
      $sFile1 .= "nph-config.cfg";
      $sFile2 .= $$phProperties{'Filename'} . "/" . "nph-config.cfg";
      $sFile3 .= $$phProperties{'Filename'} . "/" . $$phProperties{'ClientId'} . "/" . "nph-config.cfg";
    }
    else
    {
      next; # Ignore invalid directories.
    }

    foreach my $sFile ($sFile1, $sFile2, $sFile3)
    {
      next if (!defined($sFile) || !-f $sFile);

      ################################################################
      #
      # Pull in any externally defined properties according to the
      # specified template. If the template is empty, no properties
      # will be returned.
      #
      ################################################################

      my (%hLArgs, %hCustomProperties);

      %hLArgs =
      (
        'File'             => $sFile,
        'Properties'       => \%hCustomProperties,
        'RequireAllKeys'   => 0,
        'RequireKnownKeys' => 1,
        'Template'         => $phTemplate,
        'VerifyValues'     => 1,
      );
      if (!KvpGetKvps(\%hLArgs))
      {
        $$psError = $hLArgs{'Error'};
        return undef;
      }

      ################################################################
      #
      # Transfer validated properties, if any, to the main hash. This
      # is where the trump action takes place.
      #
      ################################################################

      foreach my $sProperty (keys(%hCustomProperties))
      {
        $$phProperties{$sProperty} = $hCustomProperties{$sProperty};
      }
    }
  }

  1;
}


######################################################################
#
# GetGlobalConfigProperties
#
######################################################################

sub GetGlobalConfigProperties
{
  my ($phProperties, $phTemplate, $psError) = @_;

  ####################################################################
  #
  # Initialize regular properties with their default values. Note that
  # derived properties are initialized in CreateRunTimeEnvironment().
  #
  ####################################################################

  $$phProperties{'BaseDirectory'} = "/var/webjob";
  $$phProperties{'CapContentLength'} = "N"; # [Y|N]
  $$phProperties{'ConfigSearchOrder'} = "clients:configs";
  $$phProperties{'EnableGetService'} = "Y"; # [Y|N]
  $$phProperties{'EnableConfigOverrides'} = "Y"; # [Y|N]
  $$phProperties{'EnableHostAccessList'} = "N"; # [Y|N]
  $$phProperties{'EnableLogging'} = "Y"; # [Y|N]
  $$phProperties{'FolderList'} = "common";
  $$phProperties{'GetHookCommandLine'} = "";
  $$phProperties{'GetHookLogDivertedOutput'} = "N"; # [Y|N]
  $$phProperties{'GetHookEnable'} = "N"; # [Y|N]
  $$phProperties{'GetHookStatus'} = 0;
  $$phProperties{'GetHookStatusMap'} = "";
  $$phProperties{'MaxContentLength'} = 100000000; # 100 MB
  $$phProperties{'RequireMatch'} = "Y"; # [Y|N]
  $$phProperties{'RequireUser'} = "Y"; # [Y|N]
  $$phProperties{'ServerId'} = "server_1";
  $$phProperties{'SslRequireCn'} = "N"; # [Y|N]
  $$phProperties{'SslRequireMatch'} = "N"; # [Y|N]
  $$phProperties{'SslRequireSsl'} = "Y"; # [Y|N]
  $$phProperties{'UseGmt'} = "N"; # [Y|N]

  ####################################################################
  #
  # Pull in any externally defined properties. These properties trump
  # internally defined properties.
  #
  ####################################################################

  my (%hLArgs);

  if (!exists($$phProperties{'PropertiesFile'}) || !defined($$phProperties{'PropertiesFile'}))
  {
    $$phProperties{'PropertiesFile'} = $$phProperties{'BaseDirectory'} . "/config/nph-config/nph-config.cfg";
  }

  %hLArgs =
  (
    'File'             => $$phProperties{'PropertiesFile'},
    'Properties'       => $phProperties,
    'RequireAllKeys'   => 0,
    'RequireKnownKeys' => 1,
    'Template'         => $phTemplate,
    'VerifyValues'     => 1,
  );
  if (!KvpGetKvps(\%hLArgs))
  {
    $$psError = $hLArgs{'Error'};
    return undef;
  }

  1;
}


######################################################################
#
# GetMappedValue
#
######################################################################

sub GetMappedValue
{
  my ($sStatusMap, $sMapKey) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sStatusMap) || !defined($sMapKey))
  {
    return undef;
  }

  ####################################################################
  #
  # Build a map and return the requested value.
  #
  ####################################################################

  my (%hKvps);

  foreach my $sKvp (split(/,/, $sStatusMap))
  {
    my ($sKey, $sValue) = split(/:/, $sKvp);
    $hKvps{$sKey} = $sValue;
  }

  return (exists($hKvps{$sMapKey})) ? $hKvps{$sMapKey} : undef;
}


######################################################################
#
# HookExecuteCommandLine
#
######################################################################

sub HookExecuteCommandLine
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phProperties,
    'Keys' =>
    [
      'EnableLogging',
      'ExpandHookCommandLineRoutine',
      'OsClass',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($phProperties));
    return undef;
  }

  ####################################################################
  #
  # Expand the hook's command line. If the result is undefined or
  # null, abort.
  #
  ####################################################################

  my ($sLocalError);

  $$phProperties{'HookEpoch'} = time();

  $$phProperties{'HookCommandLine'} = &{$$phProperties{'ExpandHookCommandLineRoutine'}}($phProperties, \$sLocalError);
  if (!defined($$phProperties{'HookCommandLine'}))
  {
    $$psError = $sLocalError;
    return undef;
  }

  if (!length($$phProperties{'HookCommandLine'}))
  {
    $$psError = "Command line is undefined or null";
    return undef;
  }

  ####################################################################
  #
  # Create directories, as necessary, to hold the dynamic content.
  #
  ####################################################################

  my ($sMode, $sPath);

  $sMode = 0755;
  $sPath = $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'};
  if (!-d $sPath)
  {
    if (!mkdir($sPath, $sMode))
    {
      $$psError = "Directory ($sPath) could not be created ($!)";
      return undef;
    }
  }

  $sPath .= "/" . "jids";
  if (!-d $sPath)
  {
    if (!mkdir($sPath, $sMode))
    {
      $$psError = "Directory ($sPath) could not be created ($!)";
      return undef;
    }
  }

  ####################################################################
  #
  # Dup and divert stdout/stderr. This must be done to prevent the
  # response from becoming polluted by spurious hook output.
  #
  ####################################################################

  my ($sStdErrFile, $sStdOutFile);

  if (!open(STDERR_DUP, ">&STDERR"))
  {
    $$psError = "Unable to dup stderr ($!)";
    return undef;
  }
  if (!open(STDOUT_DUP, ">&STDOUT"))
  {
    $$psError = "Unable to dup stdout ($!)";
    return undef;
  }

  if
  (
    Yes($$phProperties{'EnableLogging'}) &&
    (
      ($$phProperties{'RequestMethod'} eq "GET" && Yes($$phProperties{'GetHookLogDivertedOutput'}))
    )
  )
  {
    $sStdErrFile = $$phProperties{'HookErrFile'};
    $sStdOutFile = $$phProperties{'HookOutFile'};
  }
  else
  {
    $sStdErrFile = $$phProperties{'NullDevice'};
    $sStdOutFile = $$phProperties{'NullDevice'};
  }

  if (!open(STDERR, ">> $sStdErrFile"))
  {
    $$psError = "Unable to divert stderr to $sStdErrFile ($!)";
    return undef;
  }
  if (!open(STDOUT, ">> $sStdOutFile"))
  {
    $$psError = "Unable to divert stdout to $sStdOutFile ($!)";
    return undef;
  }

  ####################################################################
  #
  # Execute the hook.
  #
  ####################################################################

  $$phProperties{'HookPidLabel'} = "parent";
  $$phProperties{'HookPid'} = $$;
  $$phProperties{'HookState'} = "hooked";
  $$phProperties{'HookMessage'} = $$phProperties{'HookCommandLine'};
  if (Yes($$phProperties{'EnableLogging'}))
  {
    HookLogMessage($phProperties);
  }
  my $sHookReturn = system($$phProperties{'HookCommandLine'});
  my $sHookStatus = ($sHookReturn >> 8) & 0xff;
  my $sHookSignal = ($sHookReturn & 0x7f);
  my $sHookDumped = ($sHookReturn & 0x80) ? 1 : 0;
  if ($sHookStatus == 255)
  {
    $$phProperties{'HookState'} = "failed";
    $$phProperties{'HookMessage'} = "Unable to execute hook command ($!)";
  }
  else
  {
    $$phProperties{'HookState'} = "reaped";
    $$phProperties{'HookMessage'} = "status($sHookStatus) signal($sHookSignal) coredump($sHookDumped)";
  }
  if (Yes($$phProperties{'EnableLogging'}))
  {
    HookLogMessage($phProperties);
  }

  ####################################################################
  #
  # Restore stdout/stderr and close the duplicate handles.
  #
  ####################################################################

  if (!open(STDERR, ">&STDERR_DUP"))
  {
    $$psError = "Unable to restore stderr ($!)";
    return undef;
  }
  if (!open(STDOUT, ">&STDOUT_DUP"))
  {
    $$psError = "Unable to restore stdout ($!)";
    return undef;
  }
  close(STDERR_DUP);
  close(STDOUT_DUP);

  return $sHookStatus;
}


######################################################################
#
# HookLogMessage
#
######################################################################

sub HookLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Marshall log arguments.
  #
  ####################################################################

  my (%hLArgs);

  $hLArgs{'LogEpoch'} = $$phProperties{'HookEpoch'};

  $hLArgs{'LogFields'} =
  [
    'JobId',
    'RequestMethod',
    'ClientId',
    'Filename',
    'HookPidLabel',
    'HookPid',
    'HookState',
    'Message',
  ];

  $hLArgs{'LogValues'} =
  {
    'JobId'               => $$phProperties{'JobId'},
    'RequestMethod'       => $$phProperties{'RequestMethod'},
    'ClientId'            => $$phProperties{'ClientId'},
    'Filename'            => $$phProperties{'Filename'},
    'HookPidLabel'        => $$phProperties{'HookPidLabel'},
    'HookPid'             => $$phProperties{'HookPid'},
    'HookState'           => $$phProperties{'HookState'},
    'Message'             => $$phProperties{'HookMessage'},
  };

  $hLArgs{'LogFile'} = $$phProperties{'HookLogFile'};

  $hLArgs{'Newline'} = $$phProperties{'Newline'};

  $hLArgs{'RevertToStderr'} = 1;

  $hLArgs{'UseGmt'} = (Yes($$phProperties{'UseGmt'})) ? 1 : 0;

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  if (!LogNf1vMessage(\%hLArgs))
  {
    print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'HookMessage'}'\n";
    print STDERR "$$phProperties{'Program'}: Error='$hLArgs{'Error'}'\n";
  }

  1;
}


######################################################################
#
# NphLogMessage
#
######################################################################

sub NphLogMessage
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Marshall log arguments.
  #
  ####################################################################

  my (%hLArgs);

  ($hLArgs{'LogEpoch'}, $hLArgs{'LogEpochUsec'}) = gettimeofday();

  $hLArgs{'LogFields'} =
  [
    'JobId',
    'RemoteUser',
    'RemoteAddress',
    'RequestMethod',
    'ClientId',
    'Filename',
    'ContentLength',
    'ServerContentLength',
    'Duration',
    'ReturnStatus',
    'Message',
  ];

  $hLArgs{'LogValues'} =
  {
    'JobId'               => $$phProperties{'JobId'},
    'RemoteUser'          => $$phProperties{'RemoteUser'},
    'RemoteAddress'       => $$phProperties{'RemoteAddress'},
    'RequestMethod'       => $$phProperties{'RequestMethod'},
    'ClientId'            => $$phProperties{'ClientId'},
    'Filename'            => $$phProperties{'Filename'},
    'ContentLength'       => $$phProperties{'ContentLength'},
    'ServerContentLength' => $$phProperties{'ServerContentLength'},
    'Duration'            => sprintf("%.6f", tv_interval([$$phProperties{'StartTime'}, $$phProperties{'StartTimeUsec'}], [$hLArgs{'LogEpoch'}, $hLArgs{'LogEpochUsec'}])),
    'ReturnStatus'        => $$phProperties{'ReturnStatus'},
    'Message'             => $$phProperties{'ErrorMessage'},
  };

  $hLArgs{'LogFile'} = $$phProperties{'LogFile'};

  $hLArgs{'Newline'} = $$phProperties{'Newline'};

  $hLArgs{'RevertToStderr'} = 1;

  $hLArgs{'UseGmt'} = (Yes($$phProperties{'UseGmt'})) ? 1 : 0;

  ####################################################################
  #
  # Deliver log message.
  #
  ####################################################################

  if (!LogNf1vMessage(\%hLArgs))
  {
    print STDERR "$$phProperties{'Program'}: Error='$$phProperties{'ErrorMessage'}'\n";
    print STDERR "$$phProperties{'Program'}: Error='$hLArgs{'Error'}'\n";
  }

  1;
}


######################################################################
#
# ProcessGetRequest
#
######################################################################

sub ProcessGetRequest
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Proceed only if QueryString matches the GetQuery expression.
  #
  ####################################################################

  my %hParameters = ();
#FIXME Do we care if the same parameter is specified more than once? The last one in wins.
#FIXME Do we want to impose a limit on the number of parameters or the length of the query string?
  CgiParseQueryString($$phProperties{'QueryString'}, \%hParameters);

  if
  (
       VerifyHashKeyValue(\%hParameters, 'ClientId', $$phProperties{'CommonRegexes'}{'ClientId'}, 3)
    && VerifyHashKeyValue(\%hParameters, 'Filename', $$phProperties{'CommonRegexes'}{'Filename'}, 3)
  )
  {
    $$phProperties{'ClientId'} = $hParameters{'ClientId'} || "nobody";
    $$phProperties{'Filename'} = $hParameters{'Filename'};

    ##################################################################
    #
    # Bring in any client-specific properties.
    #
    ##################################################################

    my ($sLocalError);

    if (Yes($$phProperties{'EnableConfigOverrides'}))
    {
      if (!GetCustomConfigProperties($phProperties, \%{$$phProperties{'CustomConfigTemplate'}}, \$sLocalError))
      {
        $$psError = $sLocalError;
        return 500;
      }
    }

    ##################################################################
    #
    # Conditionally do CommonName and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'SslRequireSsl'}))
    {
      if (Yes($$phProperties{'SslRequireCn'}) && (!defined($$phProperties{'SslClientSDnCn'}) || !length($$phProperties{'SslClientSDnCn'})))
      {
        $$psError = "CommonName is undefined or null";
        return 470;
      }

      if (Yes($$phProperties{'SslRequireCn'}) && Yes($$phProperties{'SslRequireMatch'}) && $$phProperties{'SslClientSDnCn'} ne $$phProperties{'ClientId'})
      {
        $$psError = "CommonName ($$phProperties{'SslClientSDnCn'}) does not match client ID ($$phProperties{'ClientId'})";
        return 471;
      }
    }

    ##################################################################
    #
    # Do username and client ID checks.
    #
    ##################################################################

    if (Yes($$phProperties{'RequireUser'}) && (!defined($$phProperties{'RemoteUser'}) || !length($$phProperties{'RemoteUser'})))
    {
      $$psError = "Remote user is undefined or null";
      return 452;
    }

    if (Yes($$phProperties{'RequireUser'}) && Yes($$phProperties{'RequireMatch'}) && $$phProperties{'RemoteUser'} ne $$phProperties{'ClientId'})
    {
      $$psError = "Remote user ($$phProperties{'RemoteUser'}) does not match client ID ($$phProperties{'ClientId'})";
      return 453;
    }

    ##################################################################
    #
    # Do host access list checks.
    #
    ##################################################################

    if (Yes($$phProperties{'EnableHostAccessList'}) && !CheckHostAccessList($$phProperties{'AccessList'}{$$phProperties{'ClientId'}}, $$phProperties{'RemoteAddress'}))
    {
      $$psError = "The ClientId/Address key/value pair ($$phProperties{'ClientId'}=$$phProperties{'RemoteAddress'}/32) is not on the host access list";
      return 403;
    }

    ##################################################################
    #
    # Do content length checks.
    #
    ##################################################################

    if (!defined($$phProperties{'ContentLength'}) || !length($$phProperties{'ContentLength'}))
    {
      $$psError = "Content length is undefined or null";
      return 454;
    }

    if (Yes($$phProperties{'CapContentLength'}) && $$phProperties{'ContentLength'} > $$phProperties{'MaxContentLength'})
    {
      $$psError = "Content length ($$phProperties{'ContentLength'}) exceeds maximum allowed length ($$phProperties{'MaxContentLength'})";
      return 455;
    }

    ##################################################################
    #
    # Confirm that service is currently enabled.
    #
    ##################################################################

    if (!Yes($$phProperties{'EnableGetService'}))
    {
      $$psError = "GET service is disabled";
      return 553;
    }

    ##################################################################
    #
    # Conditionally create dynamic content and serve it up.
    #
    ##################################################################

    if (Yes($$phProperties{'GetHookEnable'}))
    {
      $$phProperties{'ExpandHookCommandLineRoutine'} = \&ExpandGetHookCommandLine;
      my $sHookStatus = HookExecuteCommandLine($phProperties, \$sLocalError);
      if (!defined($sHookStatus))
      {
        $$psError = $sLocalError;
        return 500;
      }
      if ($sHookStatus != $$phProperties{'GetHookStatus'})
      {
        $$psError = "Hook status mismatch ($sHookStatus != $$phProperties{'GetHookStatus'})";
        my $sReturnStatus = GetMappedValue($$phProperties{'GetHookStatusMap'}, $sHookStatus);
        return (defined($sReturnStatus) && $sReturnStatus >= 200 && $sReturnStatus < 600) ? $sReturnStatus : 551;
      }
      my $sGetName = $$phProperties{'JobId'} . ".get";
      my $sGetFile = $$phProperties{'DynamicDirectory'} . "/" . $$phProperties{'ClientId'} . "/" . "jids" . "/" . $sGetName;
      if (-e $sGetFile)
      {
        if (!open(FH, "< $sGetFile"))
        {
          $$psError = "Requested file ($sGetFile) could not be opened ($!)";
          return 457;
        }
        binmode(FH);
        $$phProperties{'ReturnHandle'} = \*FH;
        unlink($sGetFile); # This is dynamic content, so unlink it now.
        $$psError = "Success";
        return 200;
      }
      $$psError = "GET file ($sGetFile) was not found";
      return 404;
    }

    ##################################################################
    #
    # Traverse the effective folder list, locate the first occurrence
    # of the requested file, and serve it up.
    #
    ##################################################################

    my $sEffectiveFolderList = $$phProperties{'ClientId'};
    if (defined($$phProperties{'FolderList'}) && length($$phProperties{'FolderList'}) > 0)
    {
      $sEffectiveFolderList .= ":" . $$phProperties{'FolderList'};
    }

    foreach my $sFolder (split(/:/, $sEffectiveFolderList))
    {
      my $sGetFile = $$phProperties{'ProfilesDirectory'} . "/" . $sFolder . "/" . "configs" . "/" . $$phProperties{'Filename'};
      if (-e $sGetFile)
      {
        if (!open(FH, "< $sGetFile"))
        {
          $$psError = "Requested file ($sGetFile) could not be opened ($!)";
          return 457;
        }
        binmode(FH);
        $$phProperties{'ReturnHandle'} = \*FH;
        $$psError = "Success";
        return 200;
      }
    }
    $$psError = "Requested file ($$phProperties{'Filename'}) was not found in effective folder list ($sEffectiveFolderList)";
    return 404;
  }
  else
  {
    $$psError = "Invalid query string ($$phProperties{'QueryString'})";
    return 450;
  }
}


######################################################################
#
# SendResponse
#
######################################################################

sub SendResponse
{
  my ($phProperties) = @_;

  ####################################################################
  #
  # Send response header.
  #
  ####################################################################

  my ($sHandle, $sHeader, $sLength, $sReason, $sServer, $sStatus);

  $sHandle = $$phProperties{'ReturnHandle'};
  $sStatus = $$phProperties{'ReturnStatus'};
  $sReason = $$phProperties{'ReturnReason'};
  $sServer = $$phProperties{'ServerSoftware'} || "unknown";
  $sLength = (defined($sHandle)) ? -s $sHandle : 0;

  $sHeader  = "HTTP/1.1 $sStatus $sReason\r\n";
  $sHeader .= "Server: $sServer\r\n";
  $sHeader .= "Content-Type: application/octet-stream\r\n";
  $sHeader .= "Content-Length: $sLength\r\n";
  if ($$phProperties{'RequestMethod'} eq 'GET')
  {
    # NOTE: The JobId value could be undefined if
    # CreateRunTimeEnvironment() errors out too soon. Return a value
    # of "NA" to handle that situation.
    if (!defined($$phProperties{'JobId'}))
    {
      $sHeader .= "Job-Id: NA\r\n";
    }
    else
    {
      $sHeader .= "Job-Id: $$phProperties{'JobId'}\r\n";
    }
  }
  $sHeader .= "\r\n";

  syswrite(STDOUT, $sHeader, length($sHeader));

  ####################################################################
  #
  # Send content if any.
  #
  ####################################################################

  if (defined($sHandle))
  {
    SysReadWrite($sHandle, \*STDOUT, $sLength, undef);
    close($sHandle);
  }

  return $sLength;
}


######################################################################
#
# SetupHostAccessList
#
######################################################################

sub SetupHostAccessList
{
  my ($phProperties, $psError) = @_;

  ####################################################################
  #
  # Initialize the access list. This list is a hash that contains
  # client ID to IP address mappings. If the client passes all other
  # authentication tests, a lookup will be performed to determine if
  # the client's IP address is authorized. If it is, processing
  # continues as usual. Otherwise, the CGI script will return a 403
  # status code to the client, log an error message, and abort. If the
  # access list is empty or there is an error processing its file, all
  # access will be denied (i.e., this mechanism fails closed).
  #
  # The format of the list is key/value pair as follows:
  #
  #   <client_id> = <cidr>[,<cidr>]
  #
  #####################################################################

  %{$$phProperties{'AccessList'}} = (); # Initialize an empty list.

  my $sFile = $$phProperties{'NphConfigDirectory'} . "/" . "nph-config-hosts.access";

  if (open(AH, "< $sFile"))
  {
    while (my $sLine = <AH>)
    {
      $sLine =~ s/[\r\n]+$//; # Remove CRs and LFs.
      $sLine =~ s/#.*$//; # Remove comments.
      if ($sLine !~ /^\s*$/)
      {
        my ($sKey, $sValue) = ($sLine =~ /^([^=]*)=(.*)$/);
        $sKey =~ s/^\s+//; # Remove leading whitespace.
        $sKey =~ s/\s+$//; # Remove trailing whitespace.
        $sValue =~ s/^\s+//; # Remove leading whitespace.
        $sValue =~ s/\s+$//; # Remove trailing whitespace.
        if ($sKey =~ /^$$phProperties{'CommonRegexes'}{'ClientId'}$/)
        {
          $$phProperties{'AccessList'}{$sKey} = $sValue;
        }
      }
    }
    close(AH);
  }
  else
  {
    $$psError = "File ($sFile) could not be opened ($!)" if (defined($psError));
    return undef;
  }
}


######################################################################
#
# SysReadWrite
#
######################################################################

sub SysReadWrite
{
  my ($sReadHandle, $sWriteHandle, $sLength, $psError) = @_;

  ####################################################################
  #
  # Read/Write data, but discard data if write handle is undefined.
  #
  ####################################################################

  my ($sData, $sEOF, $sNRead, $sNProcessed, $sNWritten);

  for ($sEOF = $sNRead = $sNProcessed = 0; !$sEOF && $sLength > 0; $sLength -= $sNRead)
  {
    $sNRead = sysread($sReadHandle, $sData, ($sLength > 0x4000) ? 0x4000 : $sLength);
    if (!defined($sNRead))
    {
      $$psError = "Error reading from input stream ($!)" if (defined($psError));
      return undef;
    }
    elsif ($sNRead == 0)
    {
      $sEOF = 1;
    }
    else
    {
      if (defined($sWriteHandle))
      {
        $sNWritten = syswrite($sWriteHandle, $sData, $sNRead);
        if (!defined($sNWritten))
        {
          $$psError = "Error writing to output stream ($!)" if (defined($psError));
          return undef;
        }
      }
      else
      {
        $sNWritten = $sNRead;
      }
      $sNProcessed += $sNWritten;
    }
  }

  return $sNProcessed;
}


######################################################################
#
# VerifyConversionValues
#
######################################################################

sub VerifyConversionValues
{
  my ($phConversionValues, $phConversionChecks, $psError) = @_;

  foreach my $sKey (sort(keys(%$phConversionChecks)))
  {
    if ($$phConversionValues{$sKey} !~ /^$$phConversionChecks{$sKey}$/)
    {
      $$psError = "Conversion value ($$phConversionValues{$sKey}) for corresponding specification (%$sKey) is not valid";
      return undef;
    }
  }

  1;
}


######################################################################
#
# VerifyRunTimeEnvironment
#
######################################################################

sub VerifyRunTimeEnvironment
{
  my ($phProperties, $phRequiredProperties, $psError) = @_;

  ####################################################################
  #
  # Make sure all required properties are defined and valid.
  #
  ####################################################################

  foreach my $sProperty (keys(%$phRequiredProperties))
  {
    my $sValue = $$phProperties{$sProperty};
    if (!defined($sValue) || $sValue !~ /^$$phRequiredProperties{$sProperty}$/)
    {
      $$psError = "$sProperty property ($sValue) is undefined or invalid";
      return undef;
    }
  }

  ####################################################################
  #
  # Make sure the config directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'NphConfigDirectory'} || !-R _)
  {
    $$psError = "Config directory ($$phProperties{'NphConfigDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the dynamic directory is writable.
  #
  ####################################################################

  if (!-d $$phProperties{'DynamicDirectory'} || !-W _)
  {
    $$psError = "Dynamic directory ($$phProperties{'DynamicDirectory'}) does not exist or is not writable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the logfiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'LogfilesDirectory'} || !-R _)
  {
    $$psError = "Logfiles directory ($$phProperties{'LogfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the profiles directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'ProfilesDirectory'} || !-R _)
  {
    $$psError = "Profiles directory ($$phProperties{'ProfilesDirectory'}) does not exist or is not readable";
    return undef;
  }

  ####################################################################
  #
  # Make sure the spool directory is readable.
  #
  ####################################################################

  if (!-d $$phProperties{'SpoolDirectory'} || !-R _)
  {
    $$psError = "Spool directory ($$phProperties{'SpoolDirectory'}) does not exist or is not readable";
    return undef;
  }

  1;
}


######################################################################
#
# Yes
#
######################################################################

sub Yes
{
  return (defined($_[0]) && $_[0] =~ /^[Yy]$/) ? 1 : 0;
}

__END__

=pod

=head1 NAME

nph-config.cgi - Process config file requests.

=head1 SYNOPSIS

    --- http.conf ---
    ...
    Include /var/webjob/config/apache/webjob.conf
    ...
    --- http.conf ---

    --- /var/webjob/config/apache/webjob.conf ---
    SetEnv CONFIG_PROPERTIES_FILE /var/webjob/config/nph-config/nph-config.cfg

    ScriptAlias /cgi-client/ "/usr/local/webjob/cgi/cgi-client/"

    <Directory "/usr/local/webjob/cgi/cgi-client/">
      AllowOverride AuthConfig
      Options None
      Order allow,deny
      Allow from all
    </Directory>
    --- /var/webjob/config/apache/webjob.conf ---

=head1 DESCRIPTION

This utility implements a Common Gateway Interface (CGI), and its
purpose is to process requests for various configuration files.

The name of this utility must begin with the string "nph-".  This
naming convention, which stands for Non Parsed Headers, informs the
web server that the utility is responsible for generating the entire
HTTP response.  Non Parsed Headers are used by this utility because
they allow it to take full advantage of HTTP status code extensions.
This simplifies client-side error checking without giving up the
ability to generate application-specific responses.

The nph-config.cfg config file is used to control the CGI's runtime
behavior. To make use of it, set CONFIG_PROPERTIES_FILE as
appropriate. Upon execution, the script looks for and processes the
file specified by this environment variable.

The cgi-client directory is where nph-config.cgi lives by default.
Access to this directory should be strictly controlled, and any
utilities placed in this directory should operate in the same security
context (or realm).  This utility is designed to read/write data
from/to the server's local disk on behalf of potentially hostile
clients.  Therefore, you'll want to maintain a tight rein to avoid any
unwanted results.

=head1 CONFIGURATION CONTROLS

This section describes the various controls that this program
recognizes.  In general, controls either shape runtime behavior or
provide information needed to perform a specific function.  Controls
and their values, one pair/line, are read from a file having the
following format.

    <control> = <value>

All controls are case insensitive, but, in general, their values
are not.  Comments may occur anywhere on a given line, and must
begin with a pound character (i.e., '#').  In any given line, all
text to the right of the first comment will be ignored.  White
space surrounding controls and values is ignored.

=over 5

=item B<BaseDirectory>: <path>

B<BaseDirectory> is the epicenter of activity.  The default value is
'/var/webjob'.

=item B<CapContentLength>: [Y|N]

B<CapContentLength> forces the script to abort when B<ContentLength>
exceeds B<MaxContentLength>.  The default value is 'N'.

=item B<ConfigDirectory>: <path>

B<ConfigDirectory> is where various configuration files are
maintained.  The default value is '<BaseDirectory>/config'.

=item B<ConfigSearchOrder>: [clients|configs|clients:configs|configs:clients]

B<ConfigSearchOrder> specifies the order in which custom config files
are sought out and processed.  Custom config files may be used to
override a predefined subset of the site-specific properties.  The
following tree enumerates the locations where global and custom config
files may exist.

    config
      |
      + nph-config
          |
          - nph-config.cfg # applies globally
          |
          + clients
          |   |
          |   - nph-config.cfg # applies globally
          |   |
          |   + <client-N>
          |       |
          |       - nph-config.cfg # applies to all configs for <client-N>
          |       |
          |       + <config-N>
          |           |
          |           - nph-config.cfg # applies only to <client-N>/<config-N>
          |
          + configs
              |
              - nph-config.cfg # applies globally
              |
              + <config-N>
                  |
                  - nph-config.cfg # applies to all clients for <config-N>
                  |
                  + <client-N>
                      |
                      - nph-config.cfg # applies only to <config-N>/<client-N>

As each config file is processed, its values trump those of any that
came before -- including any values that came from global config
files.  Supported values for this variable are 'clients', 'configs',
'clients:configs', and 'configs:clients'.  The default value is
'clients:configs'.

=item B<DynamicDirectory>: <path>

B<DynamicDirectory> is where dynamic content is stored.  The default
value is '<BaseDirectory>/dynamic'.

=item B<EnableConfigOverrides>: [Y|N]

When active, B<EnableConfigOverrides> causes the script to seek out
and process additional client- and/or command-specific config files
(see B<ConfigSearchOrder>).  The default value is 'Y'.

=item B<EnableGetService>: [Y|N]

When active, B<EnableGetService> allows the script to accept and
process valid GET requests.  This control can be overridden, and its
intended purpose is to administratively disable access for valid
clients (e.g., to force a failover).  The default value is 'Y'.

=item B<EnableHostAccessList>: [Y|N]

When active, B<EnableHostAccessList> causes the script to consult a
host access list to determine whether or not the client will be
granted access based on its IP address.  Note: This mechanism will
fail closed if it's enabled and the access list is missing or does not
contain the necessary client ID to IP address mappings.  The default
value is 'N'.

=item B<EnableLogging>: [Y|N]

When active, B<EnableLogging> forces the script to generate a log
message for each request.  If the designated LogFile can not be
opened, the log message will be written to STDERR.  The default value
is 'Y'.

=item B<FolderList>: <item[:item[...]]>

B<FolderList> specifies locations where shared programs can be found.
If a requested file does not exist in a given client's configs
directory, the B<FolderList> is searched according to the order given
here.  The list delimiter is a colon (i.e., ':').  An example list
would be 'common:shared'.  The default value is 'common'.

=item B<GetHookCommandLine>: <command>

B<GetHookCommandLine> is a command string consisting of zero or more
conversion specifications optionally interspersed with zero or more
plain text characters.  The following conversion specifications are
supported:

    %A   = Weekday as a full name
    %a   = Weekday as an abbreviated name
    %cid = Client ID as a string
    %d   = Day of the month as a decimal number (01-31)
    %dynamic_dir = Full path of the dynamic directory
    %dynamic_out = Full path of the dynamic output file that is to be created
    %file = Client-requested filename as a string
    %H   = Hour as a decimal number (00-23)
    %ip  = IP address as a dotted quad string
    %jid = Job ID as a string
    %M   = Minute as a decimal number (00-59)
    %m   = Month as a decimal number (01-12)
    %pid = Process ID of server-side CGI script
    %S   = Second as a decimal number (00-60)
    %s   = Number of seconds since the Epoch
    %sid = Server ID as a string
    %u   = Weekday as a decimal number (1-7)
    %w   = Weekday as a decimal number (0-6)
    %Y   = Year with century as a decimal number

If the specified command is an empty string, then the hook mechanism
is (effectively) disabled, and the condition is logged.  However, if
the hook is disabled (i.e., B<GetHookEnable>=N), then this control is
ignored.

=item B<GetHookLogDivertedOutput>: [Y|N]

B<GetHookLogDivertedOutput> causes the script to divert any output on
stdout/stderr to nph-webjob-hook.{err,out}.  Normally, output for
these streams is discarded since hook commands should be written to
keep them clean (e.g., by logging to a file).  The default value is
'N'.

Note: If a given hook command is not working properly, you can use
this option to capture any output that was written to stdout/stderr.
That, in turn, may help you debug the problem.

Note: If logging is disabled (i.e., B<EnableLogging>=N), this control
is ignored.

=item B<GetHookEnable>: [Y|N]

When active, B<GetHookEnable> causes the script to execute the command
line specified by B<GetHookCommandLine>.  The behavior of the hook
mechanism is to launch a subprocess and wait for it to finish.  The
purpose of the subprocess is to create dynamic content that will be
delivered to the client.  The hook mechanism is highly configurable --
config file overrides are fully supported, multiple conversion tokens
are available, and the user determines what, if any, commands are
executed when the hook is activated.  Currently, hooks are only
activated if they are enabled and a hook command has been defined.
The default value is 'N'.

=item B<GetHookStatus>: <integer>

B<GetHookStatus> determines the meaning of success for the hook.  If
the specified value does not match the exit status as returned by
system(), the hook is considered a failure and the entire job is
aborted.  Otherwise, processing continues as normal.  The default
value is '0'.

=item B<GetHookStatusMap>: <integer:integer>[,<integer:integer>[...]]

B<GetHookStatusMap> allows the user to map hook exit codes into HTTP
response codes.  No mappings are defined by default.

=item B<LogfilesDirectory>: <path>

B<LogfilesDirectory> is where log files are stored.  The default value
is '<BaseDirectory>/logfiles'.

=item B<MaxContentLength>: <integer>

B<MaxContentLength> specifies the largest upload in bytes the script
will accept.  If CapContentLength is disabled, this control has no
effect.  The default value is '100000000'.

=item B<ProfilesDirectory>: <path>

B<ProfilesDirectory> is where client profiles and shared commands are
maintained.  The default value is '<BaseDirectory>/profiles'.

=item B<RequireMatch>: [Y|N]

B<RequireMatch> forces the script to abort unless ClientId matches
RemoteUser.  When this value is disabled, any authenticated user will
be allowed to issue requests for a given client.  Disabling
B<RequireUser> implicitly disables B<RequireMatch>.  The default value
is 'Y'.

=item B<RequireUser>: [Y|N]

B<RequireUser> forces the script to abort unless RemoteUser has been
set.  The default value is 'Y'.

=item B<ServerId>: <string>

B<ServerId> specifies the identity assigned to the WebJob server.  The
default value is 'server_1'.

=item B<SpoolDirectory>: <path>

B<SpoolDirectory> is where spools and queues are maintained.  The
default value is '<BaseDirectory>/spool'.

=item B<SslRequireCn>: [Y|N]

B<SslRequireCn> forces the script to abort unless SslClientSDnCn has
been set.  If SslRequireSsl is disabled, this and all other SSL
controls are ignored.  The default value is 'N'.

=item B<SslRequireMatch>: [Y|N]

B<SslRequireMatch> forces the script to abort if ClientId does not
match SslClientSDnCn.  When this control is disabled, access will be
governed by B<RequireMatch>.  Disabling B<SslRequireCn> implicitly
disables B<SslRequireMatch>.  Also, if SslRequireSsl is disabled, this
and all other SSL controls are ignored.  The B<SslRequireMatch> check
is performed prior to (not instead of) the B<RequireMatch> check.  The
default value is 'N'.

=item B<SslRequireSsl>: [Y|N]

B<SslRequireSsl> forces the script to abort unless the client is
speaking HTTPS.  Disabling B<SslRequireSsl> implicitly disables all
SSL-related controls.  The default value is 'Y'.

=item B<UseGmt>: [Y|N]

When active, B<UseGmt> forces the script to convert all time values to
GMT.  Otherwise, time values are converted to local time.  The default
value is 'N'.

=back

=head1 FILES

=over 5

=item B<BaseDirectory>/logfiles/nph-config.log

This utility writes log messages to nph-config.log (see
nph-config.cfg).  This file contains the following fields separated by
whitespace in the given order: Date, Time, JobId, RemoteUser,
RemoteAddress, RequestMethod, ClientId, Filename, ContentLength,
ServerContentLength, Duration, ReturnStatus, and ErrorMessage.  A
single hyphen, '-', is used to denote the fact that a particular field
had an empty or undefined value.  The ErrorMessage field is set off
from the other fields with a double hyphen '--' because it uses a free
form text format that can include whitespace.

=item B<BaseDirectory>/logfiles/nph-config-hook.err

This utility captures hook-related output on stderr and writes it to
nph-config-hook.err.  This file is a bit bucket for dirty hook
commands.  It is also useful as a debugging aid when creating new hook
commands.

=item B<BaseDirectory>/logfiles/nph-config-hook.log

This utility writes hook-related log messages to nph-config-hook.log.
This file contains the following fields separated by whitespace in the
given order: Date, Time, JobId, RequestMethod, ClientId, Filename,
PidLabel, Pid, State, and Message.  A single hyphen, '-', is used to
denote the fact that a particular field had an empty or undefined
value.  The Message field is set off from the other fields with a
double hyphen '--' because it uses a free form text format that can
include whitespace.

=item B<BaseDirectory>/logfiles/nph-config-hook.out

This utility captures hook-related output on stdout and writes it to
nph-config-hook.out.  This file is a bit bucket for dirty hook
commands.  It is also useful as a debugging aid when creating new hook
commands.

=back

=head1 AUTHOR

Klayton Monroe

=head1 SEE ALSO

nph-webjob.cgi(1), webjob(1)

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
