#!/usr/bin/perl -w
######################################################################
#
# $Id: WebJob-VersionRoutines.t,v 1.3 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2010-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Tests for WebJob::VersionRoutines.
#
######################################################################

use 5.008;
use strict;
use Test;

BEGIN
{
  my %hPlan =
  (
    'tests' => 1 + 216 + 18,
  );
  plan(%hPlan);
};

######################################################################
#
# Tests
#
######################################################################

  my (@aTuples);

  ####################################################################
  #
  # Test: The target module must load without error.
  #
  ####################################################################

  use WebJob::VersionRoutines;
  ok(1);

  ####################################################################
  #
  # Test: Convert regular version numbers to strings and back.
  #
  ####################################################################

  @aTuples =
  (
    ##################################################################
    #
    # 12 x 3 x 2 regular version numbers in 1.0.0 (CVS/EXE/TAR).
    #
    ##################################################################

    "0x10000000" . ":" . "V1_0_0_DS0"    . ":" . "tool 1.0.0 (ds0)"    . ":" . "tool-1.0.0.ds0.tgz",
    "0x10000001" . ":" . "V1_0_0_DS1"    . ":" . "tool 1.0.0 (ds1)"    . ":" . "tool-1.0.0.ds1.tgz",
    "0x100003ff" . ":" . "V1_0_0_DS1023" . ":" . "tool 1.0.0 (ds1023)" . ":" . "tool-1.0.0.ds1023.tgz",

    "0x10000400" . ":" . "V1_0_0_RC0"    . ":" . "tool 1.0.0 (rc0)"    . ":" . "tool-1.0.0.rc0.tgz",
    "0x10000401" . ":" . "V1_0_0_RC1"    . ":" . "tool 1.0.0 (rc1)"    . ":" . "tool-1.0.0.rc1.tgz",
    "0x100007ff" . ":" . "V1_0_0_RC1023" . ":" . "tool 1.0.0 (rc1023)" . ":" . "tool-1.0.0.rc1023.tgz",

    "0x10000800" . ":" . "V1_0_0_SR0"    . ":" . "tool 1.0.0 (sr0)"    . ":" . "tool-1.0.0.sr0.tgz",
    "0x10000801" . ":" . "V1_0_0_SR1"    . ":" . "tool 1.0.0 (sr1)"    . ":" . "tool-1.0.0.sr1.tgz",
    "0x10000bff" . ":" . "V1_0_0_SR1023" . ":" . "tool 1.0.0 (sr1023)" . ":" . "tool-1.0.0.sr1023.tgz",

    "0x10000c00" . ":" . "V1_0_0_XS0"    . ":" . "tool 1.0.0 (xs0)"    . ":" . "tool-1.0.0.xs0.tgz",
    "0x10000c01" . ":" . "V1_0_0_XS1"    . ":" . "tool 1.0.0 (xs1)"    . ":" . "tool-1.0.0.xs1.tgz",
    "0x10000fff" . ":" . "V1_0_0_XS1023" . ":" . "tool 1.0.0 (xs1023)" . ":" . "tool-1.0.0.xs1023.tgz",
#   ^^^^^^^^^^^^         ^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^^^
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             +---------------------> TAR
#   |                    |                       +---------------------------------------------------> EXE
#   |                    +---------------------------------------------------------------------------> CVS
#   +------------------------------------------------------------------------------------------------> Version Number

    ##################################################################
    #
    # 12 x 3 x 2 regular version numbers in 1.1.0 (CVS/EXE/TAR).
    #
    ##################################################################

    "0x10100000" . ":" . "V1_1_0_DS0"    . ":" . "tool 1.1.0 (ds0)"    . ":" . "tool-1.1.0.ds0.tgz",
    "0x10100001" . ":" . "V1_1_0_DS1"    . ":" . "tool 1.1.0 (ds1)"    . ":" . "tool-1.1.0.ds1.tgz",
    "0x101003ff" . ":" . "V1_1_0_DS1023" . ":" . "tool 1.1.0 (ds1023)" . ":" . "tool-1.1.0.ds1023.tgz",

    "0x10100400" . ":" . "V1_1_0_RC0"    . ":" . "tool 1.1.0 (rc0)"    . ":" . "tool-1.1.0.rc0.tgz",
    "0x10100401" . ":" . "V1_1_0_RC1"    . ":" . "tool 1.1.0 (rc1)"    . ":" . "tool-1.1.0.rc1.tgz",
    "0x101007ff" . ":" . "V1_1_0_RC1023" . ":" . "tool 1.1.0 (rc1023)" . ":" . "tool-1.1.0.rc1023.tgz",

    "0x10100800" . ":" . "V1_1_0_SR0"    . ":" . "tool 1.1.0 (sr0)"    . ":" . "tool-1.1.0.sr0.tgz",
    "0x10100801" . ":" . "V1_1_0_SR1"    . ":" . "tool 1.1.0 (sr1)"    . ":" . "tool-1.1.0.sr1.tgz",
    "0x10100bff" . ":" . "V1_1_0_SR1023" . ":" . "tool 1.1.0 (sr1023)" . ":" . "tool-1.1.0.sr1023.tgz",

    "0x10100c00" . ":" . "V1_1_0_XS0"    . ":" . "tool 1.1.0 (xs0)"    . ":" . "tool-1.1.0.xs0.tgz",
    "0x10100c01" . ":" . "V1_1_0_XS1"    . ":" . "tool 1.1.0 (xs1)"    . ":" . "tool-1.1.0.xs1.tgz",
    "0x10100fff" . ":" . "V1_1_0_XS1023" . ":" . "tool 1.1.0 (xs1023)" . ":" . "tool-1.1.0.xs1023.tgz",
#   ^^^^^^^^^^^^         ^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^^^
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             +---------------------> TAR
#   |                    |                       +---------------------------------------------------> EXE
#   |                    +---------------------------------------------------------------------------> CVS
#   +------------------------------------------------------------------------------------------------> Version Number

    ##################################################################
    #
    # 12 x 3 x 2 regular version numbers in 1.0.1 (CVS/EXE/TAR).
    #
    ##################################################################

    "0x10001000" . ":" . "V1_0_1_DS0"    . ":" . "tool 1.0.1 (ds0)"    . ":" . "tool-1.0.1.ds0.tgz",
    "0x10001001" . ":" . "V1_0_1_DS1"    . ":" . "tool 1.0.1 (ds1)"    . ":" . "tool-1.0.1.ds1.tgz",
    "0x100013ff" . ":" . "V1_0_1_DS1023" . ":" . "tool 1.0.1 (ds1023)" . ":" . "tool-1.0.1.ds1023.tgz",

    "0x10001400" . ":" . "V1_0_1_RC0"    . ":" . "tool 1.0.1 (rc0)"    . ":" . "tool-1.0.1.rc0.tgz",
    "0x10001401" . ":" . "V1_0_1_RC1"    . ":" . "tool 1.0.1 (rc1)"    . ":" . "tool-1.0.1.rc1.tgz",
    "0x100017ff" . ":" . "V1_0_1_RC1023" . ":" . "tool 1.0.1 (rc1023)" . ":" . "tool-1.0.1.rc1023.tgz",

    "0x10001800" . ":" . "V1_0_1_SR0"    . ":" . "tool 1.0.1 (sr0)"    . ":" . "tool-1.0.1.sr0.tgz",
    "0x10001801" . ":" . "V1_0_1_SR1"    . ":" . "tool 1.0.1 (sr1)"    . ":" . "tool-1.0.1.sr1.tgz",
    "0x10001bff" . ":" . "V1_0_1_SR1023" . ":" . "tool 1.0.1 (sr1023)" . ":" . "tool-1.0.1.sr1023.tgz",

    "0x10001c00" . ":" . "V1_0_1_XS0"    . ":" . "tool 1.0.1 (xs0)"    . ":" . "tool-1.0.1.xs0.tgz",
    "0x10001c01" . ":" . "V1_0_1_XS1"    . ":" . "tool 1.0.1 (xs1)"    . ":" . "tool-1.0.1.xs1.tgz",
    "0x10001fff" . ":" . "V1_0_1_XS1023" . ":" . "tool 1.0.1 (xs1023)" . ":" . "tool-1.0.1.xs1023.tgz",
#   ^^^^^^^^^^^^         ^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^         ^^^^^^^^^^^^^^^^^^^^^^^
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             +---------------------> TAR
#   |                    |                       +---------------------------------------------------> EXE
#   |                    +---------------------------------------------------------------------------> CVS
#   +------------------------------------------------------------------------------------------------> Version Number
  );

  @aTuples = map("0:$_", @aTuples);

  TestLoop(\@aTuples);

  ####################################################################
  #
  # Test: Convert release version numbers to strings and back.
  #
  ####################################################################

  @aTuples =
  (
    ##################################################################
    #
    # 3 x 3 x 2 release version numbers (CVS/EXE/TAR).
    #
    ##################################################################

    "0x10000800" . ":" . "V1_0_0"        . ":" . "tool 1.0.0"          . ":" . "tool-1.0.0.tgz",
    "0x10100800" . ":" . "V1_1_0"        . ":" . "tool 1.1.0"          . ":" . "tool-1.1.0.tgz",
    "0x10001800" . ":" . "V1_0_1"        . ":" . "tool 1.0.1"          . ":" . "tool-1.0.1.tgz",
#   ^^^^^^^^^^^^         ^^^^^^^^                ^^^^^^^^^^^^                  ^^^^^^^^^^^^^^^^
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             |
#   |                    |                       |                             +---------------------> TAR
#   |                    |                       +---------------------------------------------------> EXE
#   |                    +---------------------------------------------------------------------------> CVS
#   +------------------------------------------------------------------------------------------------> Version Number
  );

  @aTuples = map("1:$_", @aTuples);

  TestLoop(\@aTuples, 1);

######################################################################
#
# TestLoop
#
######################################################################

sub TestLoop
{
  my ($paTuples) = @_;

  my @aRegularTemplates =
  (
    "V%major_%minor_%patch_%state%build",
    "tool %major.%minor.%patch (%state%build)",
    "tool-%major.%minor.%patch.%state%build.tgz",
  );

  my @aReleaseTemplates =
  (
    "V%major_%minor_%patch",
    "tool %major.%minor.%patch",
    "tool-%major.%minor.%patch.tgz",
  );

  foreach my $sTuple (@$paTuples)
  {
    my ($sRelease, $sHexNumber, $sCvsString, $sExeString, $sTarString) = split(/:/, $sTuple);
    my $sIndex = 0;
    my $sVersionNumber = hex($sHexNumber);
    foreach my $sVersionString ($sCvsString, $sExeString, $sTarString)
    {
      my %hLArgs =
      (
        'StateFormat' => ($sIndex == 0) ? "upper" : "lower",
        'Template' => ($sRelease) ? $aReleaseTemplates[$sIndex++] : $aRegularTemplates[$sIndex++],
        'VersionNumber' => $sVersionNumber,
      );
      my $sResult = VersionNumberToString(\%hLArgs);
      ok(defined($sResult) && $sResult eq $sVersionString);
    }
    foreach my $sVersionString ($sCvsString, $sExeString, $sTarString)
    {
      my %hLArgs =
      (
        'VersionString' => $sVersionString,
      );
      my $sResult = VersionStringToNumber(\%hLArgs);
      ok(defined($sResult) && $sResult == $sVersionNumber);
    }
  }

  1;
}
