#!/usr/bin/perl -w
######################################################################
#
# $Id: WebJob-Properties.t,v 1.12 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2007-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Tests for WebJob::Properties.
#
######################################################################

use 5.008;
use strict;
use Test;

BEGIN
{
  my %hPlan =
  (
    'tests' => 1 + (2 * 3) + 51 + 24 + 15 + 15 + 39 + 40,
  );
  plan(%hPlan);
};

######################################################################
#
# Tests
#
######################################################################

  my (@aTuples);

  ####################################################################
  #
  # Test: The target module must load without error.
  #
  ####################################################################

  use WebJob::Properties;
  ok(1);

  ####################################################################
  #
  # Test: The global exit codes hash must be defined.
  #
  ####################################################################

  my $phGlobalExitCodes = PropertiesGetGlobalExitCodes();
  ok(defined($phGlobalExitCodes));

  ####################################################################
  #
  # Test: The global exit codes hash must not be empty.
  #
  ####################################################################

  ok(scalar(keys(%$phGlobalExitCodes)));

  ####################################################################
  #
  # Test: The global regexes hash must be defined.
  #
  ####################################################################

  my $phGlobalRegexes = PropertiesGetGlobalRegexes();
  ok(defined($phGlobalRegexes));

  ####################################################################
  #
  # Test: The global regexes hash must not be empty.
  #
  ####################################################################

  ok(scalar(keys(%$phGlobalRegexes)));

  ####################################################################
  #
  # Test: The global templates hash must be defined.
  #
  ####################################################################

  my $phGlobalTemplates = PropertiesGetGlobalTemplates();
  ok(defined($phGlobalTemplates));

  ####################################################################
  #
  # Test: The global templates hash must not be empty.
  #
  ####################################################################

  ok(scalar(keys(%$phGlobalTemplates)));

  ####################################################################
  #
  # Test: Send AbsolutePath through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "fail,c:",                 # Do not allow relative paths.
    "fail,c:foo",              # Do not allow relative paths.
    "fail,c:foo\\",            # Do not allow relative paths.
    "fail,c:foo\\bar",         # Do not allow relative paths.
    "fail,c:.",                # Do not allow relative paths.
    "fail,c:.\\",              # Do not allow relative paths.
    "fail,c:..",               # Do not allow relative paths.
    "fail,c:..\\",             # Do not allow relative paths.
    "pass,c:\\",
    "pass,c:\\\\",
    "pass,c:\\\\\\",
    "pass,c:\\foo",
    "pass,c:\\foo\\",
    "pass,c:\\foo\\\\",
    "pass,c:\\foo\\bar",
    "pass,c:\\foo\\\\bar",
    "pass,c:\\foo\\.\\bar",
    "pass,c:\\foo\\..\\bar",
    "fail,c:\\foo/bar",        # Do not allow mixed path separators.
    "pass,/",
    "pass,//",
    "pass,///",
    "pass,/foo",
    "pass,/foo/",
    "pass,/foo//",
    "pass,/foo///",
    "pass,/foo/bar",
    "pass,/foo//bar",
    "pass,/foo///bar",
    "pass,/foo/./bar",
    "pass,/foo/../bar",
    "fail,/foo\\bar",          # Do not allow mixed path separators.
    "fail,/\\",                # Do not allow mixed path separators.
    "pass,/.",
    "pass,/..",
    "fail,.",                  # Do not allow relative paths.
    "fail,./",                 # Do not allow relative paths.
    "fail,./foo",              # Do not allow relative paths.
    "fail,./foo/.",            # Do not allow relative paths.
    "fail,./foo/./bar",        # Do not allow relative paths.
    "fail,..",                 # Do not allow relative paths.
    "fail,../",                # Do not allow relative paths.
    "fail,../foo",             # Do not allow relative paths.
    "fail,../foo/.",           # Do not allow relative paths.
    "fail,../foo/./bar",       # Do not allow relative paths.

    "pass,c:\\ ",              # Allow embedded spaces.
    "pass,c:\\foo bar",        # Allow embedded spaces.
    "pass,c:\\foo bar\\com",   # Allow embedded spaces.
    "pass,/ ",                 # Allow embedded spaces.
    "pass,/foo bar",           # Allow embedded spaces.
    "pass,/foo bar/com",       # Allow embedded spaces.
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'AbsolutePath'});

  ####################################################################
  #
  # Test: Send Fingerprint128Bit through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "fail,",
    "fail,5a",
    "fail,5A",
    "fail,5a:",
    "fail,5A:",
    "fail,5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20",
    "fail,5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20",
    "fail,5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:",
    "fail,5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:",
    "pass,5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46",
    "pass,5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46",
    "fail,5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46:",
    "fail,5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46:",
    "fail,5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46:00",
    "fail,5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46:00",
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'Fingerprint128Bit'});

  ####################################################################
  #
  # Test: Send Fingerprint160Bit through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "fail,",
    "fail,5a",
    "fail,5A",
    "fail,5a:",
    "fail,5A:",
    "fail,00:01:02:03:5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20",
    "fail,00:01:02:03:5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20",
    "fail,00:01:02:03:5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:",
    "fail,00:01:02:03:5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:",
    "pass,00:01:02:03:5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46",
    "pass,00:01:02:03:5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46",
    "fail,00:01:02:03:5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46:",
    "fail,00:01:02:03:5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46:",
    "fail,00:01:02:03:5a:5d:56:14:e7:61:65:3c:7a:43:4b:6e:81:5f:20:46:00",
    "fail,00:01:02:03:5A:5D:56:14:E7:61:65:3C:7A:43:4B:6E:81:5F:20:46:00",
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'Fingerprint160Bit'});

  ####################################################################
  #
  # Test: Send FolderList through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "pass,",
    "fail,:",
    "fail,::",
    "fail,:foo",
    "fail,:foo:",
    "fail,foo:",
    "fail,foo::",
    "fail,foo::bar",
    "pass,foo",
    "pass,foo:bar",
    "pass,.",
    "pass,..",
    "fail,/",
    "fail,/foo",
    "fail,\\",
    "fail,\\foo",
    "fail,./",
    "fail,./foo",
    "fail,../",
    "fail,../foo",
    "fail,.\\",
    "fail,.\\foo",
    "fail,..\\",
    "fail,..\\foo",
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'FolderList'});

  ####################################################################
  #
  # Test: Send TrueFalse through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "fail,",
    "pass,1",
    "pass,0",
    "pass,T",
    "pass,F",
    "pass,t",
    "pass,f",
    "pass,TRUE",
    "pass,FALSE",
    "pass,true",
    "pass,false",
    "pass,TrUe",
    "pass,FaLsE",
    "fail,1_junk",
    "fail,0_junk",
    "fail,t_junk",
    "fail,f_junk",
    "fail,true_junk",
    "fail,false_junk",
    "fail,junk_1",
    "fail,junk_0",
    "fail,junk_t",
    "fail,junk_f",
    "fail,junk_true",
    "fail,junk_false",
    "fail,junk_1_junk",
    "fail,junk_0_junk",
    "fail,junk_t_junk",
    "fail,junk_f_junk",
    "fail,junk_true_junk",
    "fail,junk_false_junk",
    "fail,rue",
    "fail,RUE",
    "fail,alse",
    "fail,ALSE",
    "fail,tru",
    "fail,TRU",
    "fail,fals",
    "fail,FALS",
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'TrueFalse'});

  ####################################################################
  #
  # Test: Send IntegerMap through the gauntlet.
  #
  ####################################################################

  @aTuples =
  (
    "pass,",

    "fail,1",
    "fail,A",
    "fail,1:",
    "fail,A:",
    "pass,1:100",
    "fail,A:AAA",
    "fail,1:100%2c2",
    "fail,A:AAA%2cB",
    "fail,1:100%2c2:",
    "fail,A:AAA%2cB:",
    "pass,1:100%2c2:200",
    "fail,A:AAA%2cB:BBB",
    "pass,1:100%2c2:200%2c3:300",

    "fail,%2c1",
    "fail,%2cA",
    "fail,%2c1:",
    "fail,%2cA:",
    "fail,%2c1:100",
    "fail,%2cA:AAA",
    "fail,%2c1:100%2c2",
    "fail,%2cA:AAA%2cB",
    "fail,%2c1:100%2c2:",
    "fail,%2cA:AAA%2cB:",
    "fail,%2c1:100%2c2:200",
    "fail,%2cA:AAA%2cB:BBB",
    "fail,%2c1:100%2c2:200%2c3:300",

    "fail,1%2c",
    "fail,A%2c",
    "fail,1:%2c",
    "fail,A:%2c",
    "fail,1:100%2c",
    "fail,A:AAA%2c",
    "fail,1:100%2c2%2c",
    "fail,A:AAA%2cB%2c",
    "fail,1:100%2c2:%2c",
    "fail,A:AAA%2cB:%2c",
    "fail,1:100%2c2:200%2c",
    "fail,A:AAA%2cB:BBB%2c",
    "fail,1:100%2c2:200%2c3:300%2c",
  );

  TestLoop(\@aTuples, $$phGlobalRegexes{'IntegerMap'});


######################################################################
#
# TestLoop
#
######################################################################

sub TestLoop
{
  my ($paTuples, $sRegex) = @_;

  foreach my $sTuple (@$paTuples)
  {
    my ($sExpectedResult, $sPath) = split(/,/, $sTuple);
    $sPath =~ s/%2c/,/g;
    my $sResult = ($sPath =~ /^$sRegex$/) ? "pass" : "fail";
    ok($sResult eq $sExpectedResult);
  }

  1;
}
