######################################################################
#
# $Id: ValidationRoutines.pm,v 1.7 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2007-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Validation routines
#
######################################################################

package WebJob::ValidationRoutines;

require Exporter;

use 5.008;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

@EXPORT = qw(VerifyHashKeys VerifyHashKeyValue);

@EXPORT_OK = ();
@ISA = qw(Exporter);
$VERSION = do { my @r = (q$Revision: 1.7 $ =~ /(\d+)/g); sprintf("%d."."%03d" x $#r, @r); };

######################################################################
#
# VerifyHashKeys
#
######################################################################

sub VerifyHashKeys
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # The input hash must be defined.
  #
  ####################################################################

  if (!defined($phPArgs))
  {
    return undef;
  }

  ####################################################################
  #
  # All keys must exist and be defined in the sub-hash. Test all keys
  # before returning. In other words, don't stop at the first sign of
  # trouble. Report all missing/undefined keys.
  #
  ####################################################################

  my @aKeys = ();

  foreach my $sKey (@{$$phPArgs{'Keys'}})
  {
    if (!exists($$phPArgs{'Hash'}{$sKey}) || !defined($$phPArgs{'Hash'}{$sKey}))
    {
      push(@aKeys, $sKey);
    }
  }

  if (scalar(@aKeys) > 0)
  {
    my $sCaller = (reverse(split(/::/, (caller(1))[3])))[0] . "()";
    my $sKeys = join(",", @aKeys);
    $$phPArgs{'Error'} = "Unable to proceed in $sCaller due to missing or undefined inputs ($sKeys).";
    return undef;
  }

  1;
}


######################################################################
#
# VerifyHashKeyValue
#
######################################################################

sub VerifyHashKeyValue
{
  my ($phKvps, $sKey, $sValueRegex, $sAnchorFlag) = @_;

  ####################################################################
  #
  # The key/value pair hash, key name, and value regular expression
  # must be defined. Otherwise, it's an error.
  #
  ####################################################################

  if (!defined($phKvps) || !defined($sKey) || !defined($sValueRegex))
  {
    return undef;
  }

  ####################################################################
  #
  # Use both anchors unless the anchor flag is in the range [0-2].
  #
  ####################################################################

  $sAnchorFlag = 3 unless (defined($sAnchorFlag) && $sAnchorFlag =~ /^[0-3]$/o);

  ####################################################################
  #
  # To pass muster, the key must exist and its value must be defined
  # and match the specified regular expression.
  #
  ####################################################################

  if ($sAnchorFlag == 0) # No anchors.
  {
    return (exists($$phKvps{$sKey}) && defined($$phKvps{$sKey}) && $$phKvps{$sKey} =~ /$sValueRegex/) ? 1 : 0;
  }
  elsif ($sAnchorFlag == 1) # Left anchor only.
  {
    return (exists($$phKvps{$sKey}) && defined($$phKvps{$sKey}) && $$phKvps{$sKey} =~ /^$sValueRegex/) ? 1 : 0;
  }
  elsif ($sAnchorFlag == 2) # Right anchor only.
  {
    return (exists($$phKvps{$sKey}) && defined($$phKvps{$sKey}) && $$phKvps{$sKey} =~ /$sValueRegex$/) ? 1 : 0;
  }
  else # Both anchors.
  {
    return (exists($$phKvps{$sKey}) && defined($$phKvps{$sKey}) && $$phKvps{$sKey} =~ /^$sValueRegex$/) ? 1 : 0;
  }
}

1;

__END__

=pod

=head1 NAME

WebJob::ValidationRoutines - Validation routines

=head1 SYNOPSIS

    use WebJob::ValidationRoutines;

=head1 DESCRIPTION

This module is a collection of validation routines designed to support
various WebJob server-side utilities. As such, minimal effort was put
into supporting this code for general consumption. In other words, use
at your own risk and don't expect the interface to remain the same or
backwards compatible from release to release. This module does not
provide an OO interface, nor will it do so anytime soon.

=head1 AUTHOR

Klayton Monroe

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
