######################################################################
#
# $Id: TimeRoutines.pm,v 1.14 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2007-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Time-related routines
#
######################################################################

package WebJob::TimeRoutines;

require Exporter;

use 5.008;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use Time::Local;

@EXPORT = qw(DateTimeToSeconds DhmsToSeconds DurationToSeconds SecondsToDateTime SecondsToDhms SecondsToDuration);
@EXPORT_OK = ();
@ISA = qw(Exporter);
$VERSION = do { my @r = (q$Revision: 1.14 $ =~ /(\d+)/g); sprintf("%d."."%03d" x $#r, @r); };

######################################################################
#
# DateTimeToSeconds
#
######################################################################

sub DateTimeToSeconds
{
  my ($sDateTime, $sUseGmt) = @_;

  ####################################################################
  #
  # Break the date/time value into pieces. Then, convert and return
  # the number of seconds since the UNIX epoch.
  #
  #     Year = $1
  #    Month = $2
  # MonthDay = $3
  #    Hours = $4
  #  Minutes = $5
  #  Seconds = $6
  #
  ####################################################################

  if (defined($sDateTime) && $sDateTime =~ /^(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})$/)
  {
    if ($sUseGmt)
    {
      return timegm($6, $5, $4, $3, $2 - 1, $1 - 1900);
    }
    else
    {
      return timelocal($6, $5, $4, $3, $2 - 1, $1 - 1900);
    }
  }

  return undef;
}


######################################################################
#
# DhmsToSeconds
#
######################################################################

sub DhmsToSeconds
{
  my ($sDhms) = @_;

  if ($sDhms =~ /^(?:(?:(?:(\d+)[.:d])?(\d{1,2})[.:h])?(\d{1,2})[.:m])?(\d{1,2})s?$/)
  {
    my $sDays = (defined($1)) ? $1 : 0;
    my $sHours = (defined($2)) ? $2 : 0;
    my $sMinutes = (defined($3)) ? $3 : 0;
    my $sSeconds = (defined($4)) ? $4 : 0;
    return (($sDays * 86400) + ($sHours * 3600) + ($sMinutes * 60) + $sSeconds);
  }

  return undef;
}


######################################################################
#
# DurationToSeconds
#
######################################################################

sub DurationToSeconds
{
  my ($sDuration) = @_;

  ####################################################################
  #
  # Convert the specified duration (in ISO 8601:2004(E) duration
  # format) to the number of seconds since the UNIX epoch. Assume that
  # one month is equivalent to 30 days. Allow any time components to
  # be omitted, not just the lowest order components. However, at
  # least one time component must be specified (i.e., 'P' and 'PT' are
  # not valid durations).
  #
  ####################################################################

  if ($sDuration =~ /^[Pp][Tt]?$/)
  {
    return undef;
  }
  elsif ($sDuration =~ /^[Pp](?:(?:(\d+)[Yy])?(?:(\d+)[Mm])?(?:(\d+)[Dd])?)?(?:[Tt](?:(\d+)[Hh])?(?:(\d+)[Mm])?(?:(\d+)[Ss])?)?$/)
  {
    my $sYears = (defined($1)) ? $1 : 0;
    my $sMonths = (defined($2)) ? $2 : 0;
    my $sDays = (defined($3)) ? $3 : 0;
    my $sHours = (defined($4)) ? $4 : 0;
    my $sMinutes = (defined($5)) ? $5 : 0;
    my $sSeconds = (defined($6)) ? $6 : 0;
    return (($sYears * (365 * 24 * 3600)) + ($sMonths * (30 * 24 * 3600)) + ($sDays * (24 * 3600)) + ($sHours * 3600) + ($sMinutes * 60) + $sSeconds);
  }
  elsif ($sDuration =~ /^[Pp](?:(\d+)[Ww])$/)
  {
    my $sWeeks = (defined($1)) ? $1 : 0;
    return ($sWeeks * (7 * 24 * 3600));
  }
  else
  {
    return undef;
  }
}


######################################################################
#
# SecondsToDateTime
#
######################################################################

sub SecondsToDateTime
{
  my ($sSeconds, $sUseGmt) = @_;

  if (defined($sSeconds) && $sSeconds =~ /^\d+$/)
  {
    my
    (
      $sSecond,
      $sMinute,
      $sHour,
      $sMonthDay,
      $sMonth,
      $sYear,
      $sWeekDay,
      $sYearDay,
      $sDaylightSavings
    ) = ($sUseGmt) ? gmtime($sSeconds) : localtime($sSeconds);

    my $sDateTime = sprintf("%04s-%02s-%02s %02s:%02s:%02s",
      $sYear + 1900,
      $sMonth + 1,
      $sMonthDay,
      $sHour,
      $sMinute,
      $sSecond
      );

    return $sDateTime;
  }

  return undef;
}


######################################################################
#
# SecondsToDhms
#
######################################################################

sub SecondsToDhms
{
  my ($sSeconds) = @_;

  ####################################################################
  #
  # A value of -1 indicates that seconds is undefined.
  #
  ####################################################################

  my $sDhms = "";

  if (!defined($sSeconds) || $sSeconds < 0)
  {
    $sDhms = "?";
  }
  elsif ($sSeconds == 0)
  {
    $sDhms = "0s";
  }
  else
  {
    my $sDays     = $sSeconds / 86400;
       $sSeconds -= int($sDays) * 86400;
    my $sHours    = $sSeconds / 3600;
       $sSeconds -= int($sHours) * 3600;
    my $sMinutes  = $sSeconds / 60;
       $sSeconds -= int($sMinutes) * 60;

    $sDhms  = (int($sDays)) ? sprintf("%d%s", int($sDays), "d") : "";
    $sDhms .= (int($sDays) || int($sHours)) ? sprintf("%d%s", int($sHours), "h") : "";
    $sDhms .= (int($sDays) || int($sHours) || int($sMinutes)) ? sprintf("%d%s", int($sMinutes), "m") : "";
    $sDhms .= (int($sDays) || int($sHours) || int($sMinutes) || int($sSeconds)) ? sprintf("%d%s", int($sSeconds), "s") : "";
  }

  return $sDhms;
}


######################################################################
#
# SecondsToDuration
#
######################################################################

sub SecondsToDuration
{
  my ($sSeconds) = @_;

  ####################################################################
  #
  # Convert the specified number of seconds to an ISO 8601:2004(E)
  # duration format.
  #
  ####################################################################

  my $sDuration = "";

  if (!defined($sSeconds) || $sSeconds < 0)
  {
    $sDuration = "?";
  }
  elsif ($sSeconds == 0)
  {
    $sDuration = "PT0S";
  }
  else
  {
    my $sYears    = $sSeconds / (365 * 24 * 3600);
       $sSeconds -= int($sYears) * (365 * 24 * 3600);
    my $sMonths   = $sSeconds / (30 * 24 * 3600);
       $sSeconds -= int($sMonths) * (30 * 24 * 3600);
    my $sDays     = $sSeconds / (24 * 3600);
       $sSeconds -= int($sDays) * (24 * 3600);
    my $sHours    = $sSeconds / 3600;
       $sSeconds -= int($sHours) * 3600;
    my $sMinutes  = $sSeconds / 60;
       $sSeconds -= int($sMinutes) * 60;

    $sDuration .= "P";
    $sDuration .= (int($sYears)) ? sprintf("%d%s", int($sYears), "Y") : "";
    $sDuration .= (int($sMonths)) ? sprintf("%d%s", int($sMonths), "M") : "";
    $sDuration .= (int($sDays)) ? sprintf("%d%s", int($sDays), "D") : "";
    $sDuration .= "T" if (int($sHours) || int($sMinutes) || int($sSeconds));
    $sDuration .= (int($sHours)) ? sprintf("%d%s", int($sHours), "H") : "";
    $sDuration .= (int($sMinutes)) ? sprintf("%d%s", int($sMinutes), "M") : "";
    $sDuration .= (int($sSeconds)) ? sprintf("%d%s", int($sSeconds), "S") : "";
  }

  return $sDuration;
}

1;

__END__

=pod

=head1 NAME

WebJob::TimeRoutines - Time-related routines

=head1 SYNOPSIS

    use WebJob::TimeRoutines;

=head1 DESCRIPTION

This module is a collection of time-related routines designed to
support various WebJob server-side utilities. As such, minimal effort
was put into supporting this code for general consumption. In other
words, use at your own risk and don't expect the interface to remain
the same or backwards compatible from release to release. This module
does not provide an OO interface, nor will it do so anytime soon.

=head1 AUTHOR

Klayton Monroe

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
