######################################################################
#
# $Id: MldbmRoutines.pm,v 1.13 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2007-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: Common MLDBM-specific routines
#
######################################################################

package WebJob::MldbmRoutines;

require Exporter;

use 5.008;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use DB_File;
use Fcntl qw(:DEFAULT :flock);
use FileHandle;
use MLDBM qw(DB_File Storable);
use WebJob::Properties;
use WebJob::ValidationRoutines;

@EXPORT = qw(MldbmConnect MldbmCreateDb MldbmDeleteDb MldbmDisconnect MldbmGetConfigKvps MldbmGetJobKvps MldbmNewContext MldbmSetConfigKvps);
@EXPORT_OK = ();
@ISA = qw(Exporter);
$VERSION = do { my @r = (q$Revision: 1.13 $ =~ /(\d+)/g); sprintf("%d."."%03d" x $#r, @r); };

######################################################################
#
# MldbmConnect
#
######################################################################

sub MldbmConnect
{
  my ($phDbContext, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phDbContext,
    'Keys' =>
    [
      'DbClass',
      'DbFile',
      'DbFlags',
      'DbMode',
      'LockFlags',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($psError));
    return undef;
  }

  ####################################################################
  #
  # Create/Open and lock the specified lock file.
  #
  ####################################################################

  my ($oFileHandle, $sLockMode, $sLockFile, $sLockFlags);

  $sLockFile = (defined($$phDbContext{'LockFile'})) ? $$phDbContext{'LockFile'} : $$phDbContext{'DbFile'} . ".lock";

  $sLockMode = (defined($$phDbContext{'LockMode'})) ? $$phDbContext{'LockMode'} : "+<";

  $sLockFlags = (exists($$phDbContext{'LockFlags'})) ? $$phDbContext{'LockFlags'} : LOCK_SH;

  $oFileHandle = new FileHandle;
  if (!defined($oFileHandle))
  {
    $$psError = "Unable to obtain a lock handle.";
    return undef;
  }

  if (!$oFileHandle->open("$sLockMode $sLockFile"))
  {
    $$psError = "Lock file ($sLockFile) could not be created/opened ($!).";
    return undef;
  }

  if (!flock($oFileHandle, $sLockFlags))
  {
    $$psError = "Lock file ($sLockFile) could not be locked ($!).";
    return undef;
  }
  $$phDbContext{'LockHandle'} = $oFileHandle;

  ####################################################################
  #
  # Tie the specified DB.
  #
  ####################################################################

  if
  (
    !tie
    (
      %{$$phDbContext{'DbHandle'}},
      $$phDbContext{'DbClass'},
      $$phDbContext{'DbFile'},
      $$phDbContext{'DbFlags'},
      $$phDbContext{'DbMode'},
      $DB_BTREE
    )
  )
  {
    $$psError = "DB ($$phDbContext{'DbFile'}) could not be tied ($!).";
    return undef;
  }

  1;
}


######################################################################
#
# MldbmCreateDb
#
######################################################################

sub MldbmCreateDb
{
  my ($sDbFile, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sDbFile))
  {
    $$psError = "Unable to proceed due to missing or undefined inputs.";
    return undef;
  }

  ####################################################################
  #
  # If the specified DB doesn't exist, create it.
  #
  ####################################################################

  my ($phDbContext, $sLocalError);

  if (!-f $sDbFile)
  {
    $phDbContext = MldbmNewContext
    (
      {
        'DbFile'     => $sDbFile,
        'DbFlags'    => O_CREAT|O_TRUNC|O_RDWR,
        'LockFlags'  => LOCK_EX,
        'LockMode'   => ">",
      }
    );
    if (!MldbmConnect($phDbContext, \$sLocalError))
    {
      $$psError = $sLocalError;
      return undef;
    }
    $$phDbContext{'DbHandle'}{''} = undef; # Insert seed key. This forces initialization.
    delete($$phDbContext{'DbHandle'}{''}); # Delete seed key.
    MldbmDisconnect($phDbContext);
  }
  else
  {
    $$psError = "DB ($sDbFile) already exists.";
    return undef;
  }

  1;
}


######################################################################
#
# MldbmDeleteDb
#
######################################################################

sub MldbmDeleteDb
{
  my ($sDbFile, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  if (!defined($sDbFile))
  {
    $$psError = "Unable to proceed due to missing or undefined inputs.";
    return undef;
  }

  ####################################################################
  #
  # If the specified DB exists and can be tie()'d, unlink it.
  #
  ####################################################################

  my ($phDbContext, $sLocalError);

  if (-f $sDbFile && -s _)
  {
    $phDbContext = MldbmNewContext
    (
      {
        'DbFile'     => $sDbFile,
        'DbFlags'    => O_RDONLY,
        'LockFlags'  => LOCK_EX,
        'LockMode'   => "+<",
        'LockRemove' => 1,
      }
    );

    if (!MldbmConnect($phDbContext, \$sLocalError))
    {
      $$psError = $sLocalError;
      return undef;
    }
    if (!MldbmDisconnect($phDbContext))
    {
      $$psError = $sLocalError;
      return undef;
    }
    if (!unlink($sDbFile))
    {
      $$psError = "DB ($sDbFile) could not be removed ($!).";
      return undef;
    }
  }
  else
  {
    $$psError = "DB ($sDbFile) does not exist, is not a regular file, or is empty.";
    return undef;
  }

  1;
}


######################################################################
#
# MldbmDisconnect
#
######################################################################

sub MldbmDisconnect
{
  my ($phDbContext, $psError) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phDbContext,
    'Keys' =>
    [
      'DbFile',
      'LockHandle',
      'LockRemove',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$psError = $hLArgs{'Error'} if (defined($psError));
    return undef;
  }

  ####################################################################
  #
  # Untie and unlock the specified DB and lock files.
  #
  ####################################################################

  my ($sLockFile);

  if (!untie(%{$$phDbContext{'DbHandle'}}))
  {
    $$psError = "DB ($$phDbContext{'DbFile'}) could not be untied ($!).";
    return undef;
  }

  $sLockFile = (defined($$phDbContext{'LockFile'})) ? $$phDbContext{'LockFile'} : $$phDbContext{'DbFile'} . ".lock";

  if (!flock($$phDbContext{'LockHandle'}, LOCK_UN))
  {
    $$psError = "Lock file ($sLockFile) could not be unlocked ($!).";
    return undef;
  }
#  $$phDbContext{'LockHandle'}->close();
  undef($$phDbContext{'LockHandle'}); # According to the documentation, this will automatically close the file.

  if ($$phDbContext{'LockRemove'})
  {
    if (!unlink($sLockFile))
    {
      $$psError = "Lock file ($sLockFile) could not be unlinked ($!).";
      return undef;
    }
  }

  1;
}


######################################################################
#
# MldbmGetConfigKvps
#
######################################################################

sub MldbmGetConfigKvps
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
      'ClientId',
      'DbFile',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my $phGlobalRegexes = PropertiesGetGlobalRegexes();

  $$phPArgs{'ClientExists'} = 0;

  $$phPArgs{'ConfigExists'} = 0;

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  my $sClientId = $$phPArgs{'ClientId'};

  if ($sClientId !~ /^$$phGlobalRegexes{'ClientId'}$/)
  {
    $$phPArgs{'Error'} = "The client ID ($sClientId) does not pass muster.";
    return undef;
  }

  if (!-f $$phPArgs{'DbFile'})
  {
    $$phPArgs{'Error'} = "File ($$phPArgs{'DbFile'}) does not exist or is not regular.";
    return undef;
  }

  ####################################################################
  #
  # Connect to the database.
  #
  ####################################################################

  my ($sError);

  my $phDbContext = MldbmNewContext
  (
    {
      'DbFile'     => $$phPArgs{'DbFile'},
      'DbFlags'    => O_RDONLY,
      'LockFlags'  => LOCK_SH,
      'LockMode'   => "<",
    }
  );
  if (!MldbmConnect($phDbContext, \$sError))
  {
    $$phPArgs{'Error'} = $sError;
    return undef;
  }

  ####################################################################
  #
  # Get the specified key/value pairs.
  #
  ####################################################################

  my $phDb = $$phDbContext{'DbHandle'};

  if (exists($$phDb{$sClientId}))
  {
    $$phPArgs{'ClientExists'} = 1;
    if (exists($$phDb{$sClientId}{'Config'}))
    {
      $$phPArgs{'ConfigExists'} = 1;
      %{$$phPArgs{'KvpMap'}} = %{$$phDb{$sClientId}{'Config'}};
    }
  }

  ####################################################################
  #
  # Disconnect from the database.
  #
  ####################################################################

  MldbmDisconnect($phDbContext, \$sError);

  1;
}


######################################################################
#
# MldbmGetJobKvps
#
######################################################################

sub MldbmGetJobKvps
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
      'ClientId',
      'DbFile',
      'Job',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my $phGlobalRegexes = PropertiesGetGlobalRegexes();

  $$phPArgs{'ClientExists'} = 0;

  $$phPArgs{'JobExists'} = 0;

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  my $sClientId = $$phPArgs{'ClientId'};

  if ($sClientId !~ /^$$phGlobalRegexes{'ClientId'}$/)
  {
    $$phPArgs{'Error'} = "The client ID ($sClientId) does not pass muster.";
    return undef;
  }

  if (!-f $$phPArgs{'DbFile'})
  {
    $$phPArgs{'Error'} = "File ($$phPArgs{'DbFile'}) does not exist or is not regular.";
    return undef;
  }

  ####################################################################
  #
  # Connect to the database.
  #
  ####################################################################

  my ($sError);

  my $phDbContext = MldbmNewContext
  (
    {
      'DbFile'     => $$phPArgs{'DbFile'},
      'DbFlags'    => O_RDONLY,
      'LockFlags'  => LOCK_SH,
      'LockMode'   => "<",
    }
  );
  if (!MldbmConnect($phDbContext, \$sError))
  {
    $$phPArgs{'Error'} = $sError;
    return undef;
  }

  ####################################################################
  #
  # Get the specified key/value pairs.
  #
  ####################################################################

  my $phDb = $$phDbContext{'DbHandle'};

  my $sJob = $$phPArgs{'Job'};

  if (exists($$phDb{$sClientId}))
  {
    $$phPArgs{'ClientExists'} = 1;
    if (exists($$phDb{$sClientId}{'Jobs'}{$sJob}))
    {
      $$phPArgs{'JobExists'} = 1;
      %{$$phPArgs{'KvpMap'}} = %{$$phDb{$sClientId}{'Jobs'}{$sJob}};
    }
  }

  ####################################################################
  #
  # Disconnect from the database.
  #
  ####################################################################

  MldbmDisconnect($phDbContext, \$sError);

  1;
}


######################################################################
#
# MldbmNewContext
#
######################################################################

sub MldbmNewContext
{
  my ($phOptions) = @_;

  my %hDbContext =
  (
    'DbClass'    => "MLDBM",
    'DbFile'     => undef,
    'DbFlags'    => O_RDONLY,
    'DbMode'     => 0644,
    'LockFile'   => undef,
    'LockFlags'  => LOCK_SH,
    'LockMode'   => "<",
    'LockRemove' => 0,
  );

  if (defined($phOptions))
  {
    foreach my $sKey (keys(%hDbContext))
    {
      if (exists($$phOptions{$sKey}))
      {
        $hDbContext{$sKey} = $$phOptions{$sKey};
      }
    }
  }

  return \%hDbContext;
}


######################################################################
#
# MldbmSetConfigKvps
#
######################################################################

sub MldbmSetConfigKvps
{
  my ($phPArgs) = @_;

  ####################################################################
  #
  # Make sure that required inputs are defined.
  #
  ####################################################################

  my %hLArgs =
  (
    'Hash' => $phPArgs,
    'Keys' =>
    [
      'ClientId',
      'DbFile',
      'KvpMap',
    ],
  );
  if (!defined(VerifyHashKeys(\%hLArgs)))
  {
    $$phPArgs{'Error'} = $hLArgs{'Error'} if (defined($phPArgs));
    return undef;
  }

  ####################################################################
  #
  # Assign default values where necessary.
  #
  ####################################################################

  my (%hClient, $phGlobalRegexes, $sClientId);

  $phGlobalRegexes = PropertiesGetGlobalRegexes();

  $sClientId = $$phPArgs{'ClientId'};

  %hClient = ( 'Config' => $$phPArgs{'KvpMap'} );

  ####################################################################
  #
  # Make sure we have the necessary prerequisites.
  #
  ####################################################################

  if ($sClientId !~ /^$$phGlobalRegexes{'ClientId'}$/)
  {
    $$phPArgs{'Error'} = "The client ID ($sClientId) does not pass muster.";
    return undef;
  }

  if (!-f $$phPArgs{'DbFile'})
  {
    $$phPArgs{'Error'} = "File ($$phPArgs{'DbFile'}) does not exist or is not regular.";
    return undef;
  }

  ####################################################################
  #
  # Connect to the database.
  #
  ####################################################################

  my ($phDb, $phDbContext, $sError);

  $phDbContext = MldbmNewContext
  (
    {
      'DbFile'     => $$phPArgs{'DbFile'},
      'DbFlags'    => O_RDWR,
      'LockFlags'  => LOCK_EX,
      'LockMode'   => "+<",
    }
  );
  if (!MldbmConnect($phDbContext, \$sError))
  {
    $$phPArgs{'Error'} = $sError;
    return undef;
  }
  $phDb = $$phDbContext{'DbHandle'};

  ####################################################################
  #
  # Set the key/value pair(s). Add intermediate layers as required.
  #
  ####################################################################

  if (exists($$phDb{$sClientId}))
  {
    my $phClient = $$phDb{$sClientId};
    if (exists($$phClient{'Config'}))
    {
      foreach my $sKey (keys(%{$$phPArgs{'KvpMap'}}))
      {
        $$phClient{'Config'}{$sKey} = $$phPArgs{'KvpMap'}{$sKey};
      }
    }
    else
    {
      $$phClient{'Config'} = $$phPArgs{'KvpMap'};
    }
    $$phDb{$sClientId} = $phClient;
  }
  else
  {
    $$phDb{$sClientId} = \%hClient;
  }

  ####################################################################
  #
  # Disconnect from the database.
  #
  ####################################################################

  MldbmDisconnect($phDbContext, \$sError);

  1;
}

1;

__END__

=pod

=head1 NAME

WebJob::MldbmRoutines - Common MLDBM-specific routines

=head1 SYNOPSIS

    use Fcntl qw(:DEFAULT :flock);
    use WebJob::MldbmRoutines;

=head1 DESCRIPTION

This module is a collection of common routines designed to support
various WebJob server-side utilities. As such, minimal effort was put
into supporting this code for general consumption. In other words, use
at your own risk and don't expect the interface to remain the same or
backwards compatible from release to release. This module does not
provide an OO interface, nor will it do so anytime soon.

=head1 SEE ALSO

webjob-mldbm-create-client(1), webjob-mldbm-create-db(1), webjob-mldbm-create-job(1), webjob-mldbm-delete-client(1), webjob-mldbm-delete-db(1), webjob-mldbm-delete-job(1), webjob-mldbm-dump-db(1), webjob-mldbm-get-config-kvps(1), webjob-mldbm-get-job-kvps(1), webjob-mldbm-get-status(1), webjob-mldbm-list-clients(1), webjob-mldbm-list-jobs(1), webjob-mldbm-load-db(1), webjob-mldbm-set-config-kvps(1), webjob-mldbm-set-job-kvps(1)

=head1 AUTHOR

Klayton Monroe

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
