######################################################################
#
# $Id: CgiRoutines.pm,v 1.4 2012/01/07 07:56:13 mavrik Exp $
#
######################################################################
#
# Copyright 2010-2012 The WebJob Project, All Rights Reserved.
#
######################################################################
#
# Purpose: CGI routines
#
######################################################################

package WebJob::CgiRoutines;

require Exporter;

use 5.008;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

@EXPORT = qw(CgiGetValue CgiParseQueryString CgiUrlDecode);

@EXPORT_OK = ();
@ISA = qw(Exporter);
$VERSION = do { my @r = (q$Revision: 1.4 $ =~ /(\d+)/g); sprintf("%d."."%03d" x $#r, @r); };

######################################################################
#
# CgiGetValue
#
######################################################################

sub CgiGetValue
{
  my ($phParameters, $sKey) = @_;

  if (defined($sKey) && defined($$phParameters{$sKey}))
  {
    if (ref($$phParameters{$sKey}) eq "ARRAY")
    {
      return wantarray() ? @{$$phParameters{$sKey}} : @{$$phParameters{$sKey}}[0];
    }
    else
    {
      return wantarray() ? ($$phParameters{$sKey}) : $$phParameters{$sKey};
    }
  }

  return; # Return an empty list or undefined depending on the context.
}


######################################################################
#
# CgiParseQueryString
#
######################################################################

sub CgiParseQueryString
{
  my ($sQueryString, $phParameters, $phOptions) = @_;

  ####################################################################
  #
  # Split the query string into pieces, decode the key/value pairs,
  # and populate the parameters hash.
  #
  ####################################################################

  my (@aKvps) = split(/&/, $sQueryString);
  foreach my $sKvp (@aKvps)
  {
    my ($sKey, $sValue) = split(/=/, $sKvp, 2);
    next unless (defined($sKey));
    $sValue = "" unless (defined($sValue));
    $sKey = CgiUrlDecode($sKey);
    $sValue = CgiUrlDecode($sValue);
    if ($$phOptions{'AcceptMultipleValues'})
    {
      push(@{$$phParameters{$sKey}}, $sValue);
    }
    else
    {
      $$phParameters{$sKey} = $sValue;
    }
  }

  1;
}


######################################################################
#
# CgiUrlDecode
#
######################################################################

sub CgiUrlDecode
{
  my ($sData) = @_;

  $sData =~ s/\+/ /sg;
  $sData =~ s/%([0-9a-fA-F]{2})/pack('C', hex($1))/seg;

  return $sData;
}

1;

__END__

=pod

=head1 NAME

WebJob::CgiRoutines - CGI routines

=head1 SYNOPSIS

    use WebJob::CgiRoutines;

=head1 DESCRIPTION

This module is a collection of CGI routines designed to support WebJob
CGI utilities. As such, minimal effort was put into supporting this
code for general consumption. In other words, use at your own risk and
don't expect the interface to remain the same or backwards compatible
from release to release. This module does not provide an OO interface,
nor will it do so anytime soon.

=head1 AUTHOR

Klayton Monroe

=head1 LICENSE

All documentation and code are distributed under same terms and
conditions as B<WebJob>.

=cut
