/***************************************************************************
                          wap11config.h  -  description
                             -------------------
    begin                : Fri Dec 14 2001
    copyright            : (C) 2001, 2002, 2003 by Ori Pessach
    email                : mail@oripessach.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef WAP11CONFIG_H
#define WAP11CONFIG_H

#include <netinet/in.h>
#if NET_SNMP
#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/library/system.h>
#else
#include <ucd-snmp/ucd-snmp-config.h>
#include <ucd-snmp/ucd-snmp-includes.h>
#include <ucd-snmp/system.h>
#endif
#include <map>
#include <string>
#include <vector>

/**This class encapsulates the WAP11 access point's configuration, and hides
	 the nasty SNMP hackery that lets you get at it.

  *@author Ori Pessach
  */

class WAP11Config {
public: 
	WAP11Config();
	~WAP11Config();
  /** Sets the password (community, actually). You must call this before calling Get(). */
  void SetPassword(const std::string &password);
  /** Sets the IP address or name of the AP. You must call this before calling Get(). */
  void SetIPAddress(const std::string &ip_address);

  /** Tries to connect to the AP and retrieve its configuration. Returns true on success,
false otherwise. */
  bool Get();

  /** Returns the firmware version from the system description variable. */
  std::string GetFirmwareVersion();
  /** Returns an index between 1 and 4, showing which WEP key is used. */
  int GetDefaultWEPKey();
  /** Returns the first WEP key. All 40 bits of it. */
  std::string GetWEPKey(int i);
  /** Sets the passphrase, and computes the WEP keys from it. */
  void SetPassphrase(const char *pp, int wep_enable);
  /** Set the WAP11's configuration via SNMP. */
  void Upload();
  /** Adds an integer variable to the PDU */
  void AddVar(struct snmp_pdu *pdu, const char *var, int i);
  /** Extract the variables from the response structure, and store them in valuemap. */
  void ExtractVariables(const snmp_pdu *response);
  /** Sets value to the variable's value */
  void GetVariable(const std::string &key, int &value);
  /** Retrieve an IP Address value */
  void GetVariable(const std::string &key, uint32_t &value);
  /** Retrieve a string value */
  void GetVariable(const std::string &key, std::string &value);

  typedef std::vector< unsigned char > bytevector_t;

  /** Retrieves an Octet String value */
  void GetVariable(const std::string &key, bytevector_t &value);
  /** Converts a 32 bit number to an IP Address in dot notation. */
  std::string DotNotation(uint32_t ipa);

  /** Sets a value in the value map, with key 'key'. */
  void SetVariable(const std::string &key, int value);
  /** Sets a string value in the value map, with key 'key'. */
  void SetVariable(const std::string &key, const std::string &alue);
  /** Sets an IP Address value in the value map, with key 'key' */
  void SetVariable(const std::string &key, uint32_t ipa);
  /** Adds an octet stream value to valuemap, with key 'key' */
  void SetVariable(const std::string &key, const bytevector_t &value);
  /** Adds a string value to the PDU. */
  void AddVar(struct snmp_pdu *pdu, const char *var, std::string s);
  /** Adds an octet string variable to a pdu */
  void AddVar(struct snmp_pdu *pdu, const char *var, bytevector_t &);
  /** Convert a number between 0 & 15 to a hex digit. */
  char HexDigit(int n);
  /** Issues a soft reset, by setting the sysReset node to 1 (enable). */
  void Reset();
  /** Retrieves the ethernet and wireless stats fields from the AP */
  void GetStats();
  /** Returns the community string */
  const std::string & GetPassword();
  /** Updates the authorization password (community string), and uploads it to the AP. */
  void UpdatePassword(const std::string &password);
  /** Restores the access point to factory defaults. */
  void RestoreDefaults();
  /** Extracts and returns the regulatory domain */
  long GetRegDomain();
  /** Decodes the regulatory domain number passed as an integer, and return a string describing the domain. */
  std::string DecodeDomain(int domain);
  /** Returns the list of authorized MAC addresses */
  std::vector<std::string> GetMACList();
  /** Sets the authorized MAC list. */
  void SetMACList(const std::vector<std::string> &macs);

private:

  struct Value
  {
    enum ValueType {Unknown, Integer, String, OctetString, IPAddress};
    ValueType type;

    // No constructors in unions. Bummer.
    int integer;
    std::string str;
    bytevector_t octstr;
    uint32_t ipa;
  };

  struct vardesc_t
  {
    const char *var;
    Value::ValueType type;
  };

  typedef std::map< std::string, Value, std::less< std::string > > valuemap_t;

  valuemap_t valuemap;
  typedef std::map< std::string, std::string, std::less< std::string > > mib_t;
	typedef std::map< std::string, Value::ValueType, std::less< std::string > > mib_type_t;
  mib_t mini_mib;
	mib_type_t mini_mib_type;
  struct snmp_session session;
  struct snmp_session *ss;
  std::string password;
  std::string access_point_address;

private: // Private methods
  struct AuthorizedMacTableString
  {
    unsigned short action;
    unsigned short num_of_all_table_addresses;
    unsigned short num_of_current_address;
    unsigned char mac_address[6];
  };
  /** Small helper function to add a variable to a pdu. */
  void AddVar(struct snmp_pdu *pdu, const char *var, Value::ValueType type=Value::Unknown);
  /** Construct a GET pdu and retreieve a synchronous response. */
  void SNMPGet(vardesc_t *vars);
  /** Stuffs the variable's new value in the pdu. */
  void UpdateVar(struct snmp_pdu *pdu, const char *var);
  /** Returns a string with the formatted key. The length depends on wep_enable - 
1 - 5 octets
2 - empty string
3 - 13 octets */
  std::string FormatKey(const bytevector_t &v, int wep_enable);
  /** Uploads the variables in 'vars' to the AP. Throws an exception if the upload fails. */
  void UploadVariables(const vardesc_t *vars);
  /** Updates our reverse mapping from variable names to OID */
  void UpdateMiniMIB(const char *var_name, Value::ValueType);
  /** Converts a vector containing 6 octets into a string with a hex representation of the MAC address. */
  std::string MacToHex(const bytevector_t &v);
  /** Convert a string containing 6 hex octets into a byte vector. */
  bytevector_t HexToMac(const std::string &s);

protected: // Protected methods
  /** Sets the variables with an SNMP request, and retrieves the results. */
  void SNMPSet(vardesc_t *vars);
};

#endif
