/***************************************************************************
                          wap11config.cpp  -  description
                             -------------------
    begin                : Fri Dec 14 2001
    copyright            : (C) 2001, 2002, 2003 by Ori Pessach
    email                : mail@oripessach.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include "wap11config.h"
#include "wap11exception.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#if NET_SNMP
#include <net-snmp/library/asn1.h>
#include <net-snmp/library/mib.h>
#include <net-snmp/library/parse.h>
#else // NET_SNMP
#include <ucd-snmp/asn1.h>
#include <ucd-snmp/mib.h>
#include <ucd-snmp/parse.h>
#endif // NET_SNMP
#include "sstream"

WAP11Config::WAP11Config()
{
  init_snmp("WAP11-SNMP");
  add_mibdir(MIBSDIR);
  init_mib_internals();
  snmp_sess_init(&session);
  if(read_module("ATMEL-MIB")==0)
  {
    THROW_WAP11_EXCEPTION( WAP11SNMPAuthError );
  }
}

WAP11Config::~WAP11Config()
{
  snmp_close(&session);
}

/** Tries to connect to the AP and retrieve its configuration. Returns true on success,
false otherwise. */
bool WAP11Config::Get()
{
  snmp_sess_init(&session);
  // Silly UCD-SNMP doesn't use const!
  session.peername=(char*)access_point_address.c_str();
  session.version = SNMP_VERSION_1;
  // set the SNMPv1 community name used for authentication:
  session.community = (u_char*)password.c_str();
  session.community_len = strlen((char*)session.community);

  SOCK_STARTUP;
  ss = snmp_open(&session);
  if (!ss) {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, "Unable to establish SNMP session." );
  }

  vardesc_t vars0[] = {
    {"sysCtrlGRP.sysDescr.0", Value::String},
    {"", Value::Unknown}
  };
  SNMPGet(vars0);

  vardesc_t vars1[]= {
    {"OperationalSettingsGRP.operChannelID.0", Value::Integer},
    {"OperationalSettingsGRP.operESSID.0", Value::String},
    {"OperationalSettingsGRP.operESSIDLength.0", Value::Unknown},
    {"OperationalSettingsGRP.operRTSThreshold.0", Value::Integer},
    {"OperationalSettingsGRP.operFragmentationThreshold.0", Value::Integer},
    {"OperationalSettingsGRP.operPreambleType.0", Value::Integer},
    {"OperationalSettingsGRP.operAuthenticationType.0", Value::Integer},
    {"OperationalSettingsGRP.operBasicRates.0", Value::OctetString},
    {"OperationalSettingsGRP.operAutoRateFallBack.0", Value::Integer},
    {"OperationalSettingsGRP.operAccessPointName.0", Value::String},
    {"AuthorizedMacAddressesGRP.AuthorizationMacEnable.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars1);

  vardesc_t vars2[]= {
    {"operBridgingLevel.operIPAddress.0", Value::IPAddress},
    {"operBridgingLevel.operIPMask.0", Value::IPAddress},
    {"operBridgingLevel.operEthernetAddress.0", Value::OctetString},
    {"operBridgingLevel.operGateway.0", Value::IPAddress},
    {"operBridgingLevel.operDHCP.0", Value::Integer},
    {"operBridgingLevel.operPrimaryPort.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars2);

  vardesc_t vars3[]= {
    {"privacyGRP.defaultWEPKey1.0", Value::OctetString},
    {"privacyGRP.defaultWEPKey2.0", Value::OctetString},
    {"privacyGRP.defaultWEPKey3.0", Value::OctetString},
    {"privacyGRP.defaultWEPKey4.0", Value::OctetString},
    {"privacyGRP.privacyDefaultWEPKeyID.0", Value::Integer},
    {"privacyGRP.privacyWEPEnable.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars3);

  vardesc_t vars4[]= {
    {"operBridgeOperationalMode.bridgeOperationalMode.0", Value::Integer},
    {"operBridgeOperationalMode.bridgeRemoteBridgeBSSID.0", Value::OctetString},
    {"", Value::Unknown} };
  SNMPGet(vars4);

  vardesc_t vars5[]= {
    {"EthStatisticsGRP.EthRxStatistics.0", Value::OctetString},
    {"EthStatisticsGRP.EthTxStatistics.0", Value::OctetString},
    {"", Value::Unknown} };
  SNMPGet(vars5);

  vardesc_t vars6[]= {
    {"WLstatisticsGRP.wirelessStatistics.0", Value::OctetString},
    {"", Value::Unknown} };
  SNMPGet(vars6);

  vardesc_t vars7[]= {
    {"TestModeSettingsGRP.TestModeRadioConfiguration.0", Value::OctetString},
    {"", Value::Unknown} };
  SNMPGet(vars7);

  vardesc_t vars8[]= {
    {"sysCtrlGRP.sysDeviceInfo.0", Value::OctetString},
    {"sysCtrlGRP.sysTrapSwitch.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars8);

  vardesc_t vars9[]= {
    {"AssociatedStationsGRP.AssociatedSTAsNum.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars9);

  vardesc_t vars10[]= {
    {"sysCtrlGRP.sysDeviceInfo.0", Value::OctetString},
    {"", Value::Unknown} };
  SNMPGet(vars10);

  vardesc_t vars11[]= {
    {"sysCtrlGRP.sysTrapSwitch.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars11);

  vardesc_t vars12[]= {
    {"operFiltering.operIPFilter.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars12);

  vardesc_t vars13[]= {
    {"OperationalSettingsGRP.operSSIDBroadcasting.0", Value::Integer},
    {"OperationalSettingsGRP.operAntennaSettings.0", Value::Integer},
    {"", Value::Unknown} };
  SNMPGet(vars13);

  return true;
}

/** Updates our reverse mapping from variable names to OID */
void WAP11Config::UpdateMiniMIB(const char *var_name, Value::ValueType type)
{

  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;

  if(get_node(var_name, anOID, &anOID_len)==0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, std::string(var_name)+" "+snmp_errstring(snmp_errno) );
  }
  std::string name;
  for(int i=0; i<(int)anOID_len; i++)
  {
    char t[16];
    sprintf(t, ".%d", (int)anOID[i]);
    name+=t;
  }
  mini_mib[name]=var_name;
  mini_mib_type[name]=type;
}

/** Small helper function to add a variable to a pdu. */
void WAP11Config::AddVar(struct snmp_pdu *pdu, const char *var, Value::ValueType type)
{
  UpdateMiniMIB(var, type);

  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;

  if(get_node(var, anOID, &anOID_len)==0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, std::string(var)+" "+snmp_errstring(snmp_errno) );
  }
  
  // Store the value, so we have the type later:
  Value v;
  v.type=type;
  valuemap[var]=v;
  snmp_add_null_var(pdu, anOID, anOID_len);
}

/** Adds an integer variable to the PDU */
void WAP11Config::AddVar(struct snmp_pdu *pdu, const char *var, int i)
{
  UpdateMiniMIB(var, Value::Integer);
  
  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;

  if(get_node(var, anOID, &anOID_len)==0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(snmp_errno) );
  }

  char val[32];
  sprintf(val, "%d", i);
  snmp_add_var(pdu, anOID, anOID_len, 'i', val);
}

/** Sets the IP address or name of the AP. You must call this before calling Get(). */
void WAP11Config::SetIPAddress(const std::string &ip_address)
{
  access_point_address=ip_address;
}

/** Sets the password (community, actually). You must call this before calling Get(). */
void WAP11Config::SetPassword(const std::string &p)
{
  password=p;
}

/** Returns the firmware version from the system description variable. */
std::string WAP11Config::GetFirmwareVersion()
{
  char *s=new char[16];
  memset(s, 0, 16);
  char *s1=s;
  bool copy=false;
  std::string sysdescr;
  GetVariable("sysCtrlGRP.sysDescr.0", sysdescr);
  for(const char *p=sysdescr.c_str(); *p; ++p)
    if(copy)
      if(*p==')')
        copy=false;
      else
        *s1++=*p;
    else
      if(*p=='(')
        copy=true;
  std::string retval(s);
  delete s;
  return retval;
}

/** Returns the i'th WEP key. All 40 bits of it. */
std::string WAP11Config::GetWEPKey(int i)
{
  if(i<1 && i>4)
  {
    THROW_WAP11_EXCEPTION2( WAP11ParamException, "Illegal Key Requested" );
  }
  bytevector_t wep_key;
  std::stringstream svar;
  svar << "privacyGRP.defaultWEPKey" << i << ".0";
  std::string var=svar.str();
  GetVariable(var, wep_key);
  int wep_enable;
  GetVariable("privacyGRP.privacyWEPEnable.0", wep_enable);
  std::string key;
  key=FormatKey(wep_key, wep_enable);
  return key;
}

/** Sets the passphrase, and computes the WEP keys from it. Adapted from linksys-config.pl */
void WAP11Config::SetPassphrase(const char *pp, int wep_enable)
{
  const char *genstr=pp;
  const int WLAN_WEP_NKEYS = 4;

  int len = strlen(genstr);
  unsigned char tmp_wep_keys[4][13];
  memset(tmp_wep_keys, 0, sizeof(tmp_wep_keys));
  int keylen = 5;
  int i, j;
  unsigned char pseed[4]={0, 0, 0, 0};
  bytevector_t wep_key[4];

  if( wep_enable==3 )
  {
    keylen = 13;
  }

  for (i = 0; i < len; i++)
  {
     pseed[i % 4] ^= (unsigned char)genstr[i];
  }

  int randNumber = pseed[0] |
                   (pseed[1]) << 8 |
                   (pseed[2]) << 16 |
                   (pseed[3]) << 24;

  for (i = 0; i < WLAN_WEP_NKEYS; i++)
  {
     for (j = 0; j < keylen; j++)
     {
        randNumber *= 0x343fd;
        randNumber += 0x269ec3;
        tmp_wep_keys[i][j] = ((randNumber >> 16) & 0x7fff) & 0xff;
     }
  }

  for(i=0; i<4; ++i)
  {
    for(j=0; j<13 /*keylen*/; ++j)
    {
      wep_key[i].push_back(tmp_wep_keys[i][j]);
    }
    std::string keyid="privacyGRP.defaultWEPKey";
    std::string n="1";
    n[0]=n[0]+i;
    SetVariable(keyid+n+".0", wep_key[i]);
  }

  return;
}

/** Set the WAP11's configuration via SNMP. */
void WAP11Config::Upload()
{
  {
    vardesc_t vars[]=
    {
      {"OperationalSettingsGRP.operChannelID.0", Value::Unknown},
      {"OperationalSettingsGRP.operESSID.0", Value::Unknown},
      {"OperationalSettingsGRP.operESSIDLength.0", Value::Unknown},
      {"OperationalSettingsGRP.operRTSThreshold.0", Value::Unknown},
      {"OperationalSettingsGRP.operFragmentationThreshold.0", Value::Unknown},
      {"OperationalSettingsGRP.operPreambleType.0", Value::Unknown},
      {"OperationalSettingsGRP.operAuthenticationType.0", Value::Unknown},
      {"OperationalSettingsGRP.operBasicRates.0", Value::Unknown},
      {"OperationalSettingsGRP.operAutoRateFallBack.0", Value::Unknown},
      {"OperationalSettingsGRP.operAccessPointName.0", Value::Unknown},
      {"AuthorizedMacAddressesGRP.AuthorizationMacEnable.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }

  {
    vardesc_t vars[]=
    {
      {"operBridgingLevel.operIPAddress.0", Value::Unknown},
      {"operBridgingLevel.operIPMask.0", Value::Unknown},
      {"operBridgingLevel.operEthernetAddress.0", Value::Unknown},
      {"operBridgingLevel.operGateway.0", Value::Unknown},
      {"operBridgingLevel.operDHCP.0", Value::Unknown},
      {"operBridgingLevel.operPrimaryPort.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }

  {
    vardesc_t vars[]=
    {
      {"privacyGRP.defaultWEPKey1.0", Value::Unknown},
      {"privacyGRP.defaultWEPKey2.0", Value::Unknown},
      {"privacyGRP.defaultWEPKey3.0", Value::Unknown},
      {"privacyGRP.defaultWEPKey4.0", Value::Unknown},
      {"privacyGRP.privacyDefaultWEPKeyID.0", Value::Unknown},
      {"privacyGRP.privacyWEPEnable.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }

  {
    vardesc_t vars[]=
    {
      {"operBridgeOperationalMode.bridgeOperationalMode.0", Value::Unknown},
      {"operBridgeOperationalMode.bridgeRemoteBridgeBSSID.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }

  {
    vardesc_t vars[]=
    {
      {"TestModeSettingsGRP.TestModeRadioConfiguration.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }

  {
    vardesc_t vars[]= {
      {"OperationalSettingsGRP.operSSIDBroadcasting.0", Value::Integer},
      {"OperationalSettingsGRP.operAntennaSettings.0", Value::Integer},
      {"", Value::Unknown} };
    UploadVariables(vars);
  }

  SetVariable("sysCtrlGRP.sysUpload.0", 1);
  {
    vardesc_t vars[]=
    {
      {"sysCtrlGRP.sysUpload.0", Value::Unknown},
      {"", Value::Unknown}
    };
    UploadVariables(vars);
  }
}

/** Sets a value in the value map, with key 'key'. */
void WAP11Config::SetVariable(const std::string &key, int value)
{
  Value v;
  v.type=Value::Integer;
  v.integer=value;
  valuemap[key]=v;
}

/** Sets a string value in the value map, with key 'key'. */
void WAP11Config::SetVariable(const std::string &key, const std::string &value)
{
  Value v;
  v.type=Value::String;
  v.str=value;
  valuemap[key]=v;
}

/** Sets an IP Address value in the value map, with key 'key' */
void WAP11Config::SetVariable(const std::string &key, uint32_t value)
{
  Value v;
  v.type=Value::IPAddress;
  v.ipa=value;
  valuemap[key]=v;
}

/** Adds an octet stream value to valuemap, with key 'key' */
void WAP11Config::SetVariable(const std::string &key, const bytevector_t &value)
{
  Value v;
  v.type=Value::OctetString;
  v.octstr=value;
  valuemap[key]=v;
}

/** Extract the variables from the response structure, and store them in valuemap. */
void WAP11Config::ExtractVariables(const snmp_pdu *response)
{
  struct variable_list *vars;
  int count=1;
  char *sp;

  for(vars = response->variables; vars; vars = vars->next_variable)
  {
    std::string name;
    for(int i=0; i<(int)vars->name_length; i++)
    {
      char t[16];
      sprintf(t, ".%d", (int)vars->name_loc[i]);
      name+=t;
    }

    switch( vars->type )
    {
    case ASN_OCTET_STR:
    {
      sp = (char *)malloc(1 + vars->val_len);
      memcpy(sp, vars->val.string, vars->val_len);
      sp[vars->val_len] = '\0';

      Value v;
      if(valuemap[mini_mib[name]].type==Value::OctetString)
      {
        fflush(stdout);
        v.type=Value::OctetString;
        bytevector_t bv(vars->val_len);
        for(int i=0; i<(int)vars->val_len; ++i)
        {
          bv[i]=sp[i];
        }
        v.octstr=bv;
      }
      else
      {
        v.type=Value::String;
        v.str=sp;
      }
      valuemap[mini_mib[name]]=v;
      free(sp);
      break;
    }
    case ASN_INTEGER:
    {
      Value v;
      v.type=Value::Integer;
      v.integer=*vars->val.integer;
      valuemap[mini_mib[name]]=v;
      break;
    }
    case ASN_IPADDRESS:
    {
      Value v;
      v.type=Value::IPAddress;
      v.str=DotNotation(*vars->val.integer);
      valuemap[mini_mib[name]]=v;
      break;
    }
    default:
      break;
    }
    ++count;
  }
}

/** Sets value to the variable's value */
void WAP11Config::GetVariable(const std::string &key, int &value)
{
  value=valuemap[key].integer;
}

/** Retrieve a string value */
void WAP11Config::GetVariable(const std::string &key, std::string &value)
{
  value=valuemap[key].str;
}

/** Retrieve an IP Address value */
void WAP11Config::GetVariable(const std::string &key, uint32_t &value)
{
  value=valuemap[key].ipa;
}

/** Converts a 32 bit number to an IP Address in dot notation. */
std::string WAP11Config::DotNotation(uint32_t ipa)
{
  char s[16];
  sprintf(s, "%d.%d.%d.%d",
          (unsigned char)(ipa&0xff),
          (unsigned char)((ipa&0xff00)>>8),
          (unsigned char)((ipa&0xff0000)>>16),
          (unsigned char)((ipa&0xff000000)>>24));
  std::string ret(s);
  return ret;
}

/** Retrieves an Octet String value
 */
void WAP11Config::GetVariable(const std::string &key, bytevector_t &value)
{
  value=valuemap[key].octstr;
}

/** Adds a string value to the PDU. */
void WAP11Config::AddVar(struct snmp_pdu *pdu, const char *var, std::string s)
{
  UpdateMiniMIB(var, Value::String);
  
  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;

  if(get_node(var, anOID, &anOID_len)==0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(snmp_errno));
  }
  snmp_pdu_add_variable(pdu, anOID, anOID_len, ASN_OCTET_STR,
                        (u_char*)s.c_str(), s.length());
}

/** Adds an octet string variable to a pdu */
void WAP11Config::AddVar(struct snmp_pdu *pdu, const char *var, bytevector_t &v)
{
  UpdateMiniMIB(var, Value::OctetString);
  
  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;

  u_char *val=new u_char[v.size()];
  for(unsigned int i=0; i<v.size(); ++i)
  {
    val[i]=v[i];
  }
  if(get_node(var, anOID, &anOID_len)==0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(snmp_errno) );
  }
  snmp_pdu_add_variable(pdu, anOID, anOID_len, ASN_OCTET_STR,
                        val, v.size());
  delete [] val;
}

/** Stuffs the variable's new value in the pdu. */
void WAP11Config::UpdateVar(struct snmp_pdu *pdu, const char *var)
{
  Value &v=valuemap[var];

  switch(v.type)
  {
    case Value::String:
      AddVar(pdu, var, v.str);
      break;
    case Value::Integer:
      AddVar(pdu, var, v.integer);
      break;
    case Value::OctetString:
      AddVar(pdu, var, v.octstr);
      break;
    case Value::IPAddress:
      AddVar(pdu, var, v.ipa);
      break;
    case Value::Unknown:
      break;
  }
}

/** Returns a string with the formatted key. The length depends on wep_enable - 
1 - 5 octets
2 - empty string
3 - 13 octets */
std::string WAP11Config::FormatKey(const bytevector_t &v, int wep_enable)
{
  int lengths[]={0, 5, 0, 13};
  std::string key;

  for(int i=0; i<lengths[wep_enable]; ++i)
  {
    key=key+HexDigit((v[i]&0xf0)>>4);
    key=key+HexDigit(v[i]&0x0f);
    key=key+" ";
  }

  return key;
}

/** Convert a number between 0 & 15 to a hex digit. */
char WAP11Config::HexDigit(int n)
{
  return n<10 ? '0'+n : 'A'+n-10;
}

/** Construct a GET pdu and retreieve a synchronous response. */
void WAP11Config::SNMPGet(vardesc_t *vars)
{
  struct snmp_pdu *pdu = snmp_pdu_create(SNMP_MSG_GET);
  struct snmp_pdu *response;
  for(int i=0; strcmp(vars[i].var, "")!=0; ++i)
  {
    AddVar(pdu, vars[i].var, vars[i].type);
  }
  int status = snmp_synch_response(ss, pdu, &response);
  if(status!=0)
  {
    THROW_WAP11_EXCEPTION3( WAP11SNMPException,
                            snmp_errstring(status),
                            WAP11SNMPException::W11G_AP_DOESNT_LISTEN );
  }
  if(response->errstat!=0)
  {
    snmp_free_pdu(response);
    THROW_WAP11_EXCEPTION3( WAP11SNMPException,
                            snmp_errstring(response->errstat),
                            WAP11SNMPException::W11G_AUTHORIZATION );
  }
  ExtractVariables(response);
  snmp_free_pdu(response);  
}

/** Sets the variables with an SNMP request, and retrieves the results. */
void WAP11Config::SNMPSet(vardesc_t *vars)
{
  struct snmp_pdu *pdu=snmp_pdu_create(SNMP_MSG_SET);
  struct snmp_pdu *response;
  for(int i=0; strcmp(vars[i].var, "")!=0; ++i)
  {
    UpdateVar(pdu, vars[i].var);
  }
  int status = snmp_synch_response(ss, pdu, &response);
  if(status!=0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException,
                            snmp_errstring(status) );
  }
  if(response->errstat!=0)
  {
    snmp_free_pdu(response);
    THROW_WAP11_EXCEPTION2( WAP11SNMPException,
                            snmp_errstring(response->errstat) );
  }
  ExtractVariables(response);
  snmp_free_pdu(response);  
}


/** Issues a soft reset, by setting the sysReset node to 1 (enable). */
void WAP11Config::Reset()
{
  SetVariable("sysCtrlGRP.sysReset.0", 1);
  vardesc_t vars[]=
  {
    {"sysCtrlGRP.sysReset.0", Value::OctetString},
    {"", Value::Unknown}
  };
  UploadVariables(vars);
}

/** Retrieves the ethernet and wireless stats fields from the AP */
void WAP11Config::GetStats()
{
  {
  vardesc_t vars[]= {
    {"EthStatisticsGRP.EthRxStatistics.0", Value::OctetString},
    {"EthStatisticsGRP.EthTxStatistics.0", Value::OctetString},
    {"WLstatisticsGRP.wirelessStatistics.0", Value::OctetString},
    {"", Value::Unknown} };
    SNMPGet(vars);
  };
}

/** Returns the community string */
const std::string & WAP11Config::GetPassword()
{
  return password;
}

/** Updates the authorization password (community string), and uploads it to the AP. */
void WAP11Config::UpdatePassword(const std::string &pass)
{
  SetVariable("AuthorizedSettingsGRP.AuthorizedAdminPass.0", pass);
  struct snmp_pdu *pdu = snmp_pdu_create(SNMP_MSG_SET);
  struct snmp_pdu *response;

  UpdateVar(pdu, "AuthorizedSettingsGRP.AuthorizedAdminPass.0");
  int status = snmp_synch_response(ss, pdu, &response);
  if(status!=0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(status) );
  }
  if(response->errstat!=0)
  {
    snmp_free_pdu(response);
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(response->errstat) );
  }
  snmp_free_pdu(response);
  SetPassword(pass);
}

/** Uploads the variables in 'vars' to the AP. Throws an exception if the upload fails. */
void WAP11Config::UploadVariables(const vardesc_t *vars)
{
  struct snmp_pdu *pdu = snmp_pdu_create(SNMP_MSG_SET);
  struct snmp_pdu *response;

  for(int i=0; strcmp(vars[i].var, "")!=0; ++i)
  {
    UpdateVar(pdu, vars[i].var);
  }
  int status = snmp_synch_response(ss, pdu, &response);
  if(status!=0)
  {
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(status) );
  }
  if(response->errstat!=0)
  {
    snmp_free_pdu(response);
    THROW_WAP11_EXCEPTION2( WAP11SNMPException, snmp_errstring(response->errstat) );
  }
  snmp_free_pdu(response);
}

/** Restores the access point to factory defaults. */
void WAP11Config::RestoreDefaults()
{
  SetVariable("sysCtrlGRP.sysLoadDefaults.0", 1);
  vardesc_t vars[]=
  {
    {"sysCtrlGRP.sysLoadDefaults.0", Value::OctetString},
    {"", Value::Unknown}
  };
  UploadVariables(vars);
}
/** Extracts and returns the regulatory domain */
long WAP11Config::GetRegDomain()
{
  vardesc_t vars[]=
  {
    {"sysCtrlGRP.sysDeviceInfo.0", Value::OctetString},
    {"", Value::Unknown}
  };
  SNMPGet(vars);
  bytevector_t v;
  GetVariable("sysCtrlGRP.sysDeviceInfo.0", v);

  int i=v[12]+((long)v[13]<<8)+((long)v[14]<<16)+((long)v[15]<<24);

  return i;
}
/** Decodes the regulatory domain number passed as an integer, and return
    a string describing the domain. */
std::string WAP11Config::DecodeDomain(int domain)
{
  switch(domain)
  {
    case 0x10:
      return "FCC";
    case 0x20:
      return "DOC";
    case 0x30:
      return "ETSI";
    case 0x31:
      return "Spain";
    case 0x32:
      return "France";
    case 0x40:
      return "Japan";
    default:
      return "Unknown";
  }
}

/** Returns the list of authorized MAC addresses */
std::vector<std::string> WAP11Config::GetMACList()
{
  std::vector<std::string> vs;
  AuthorizedMacTableString get, result;

  bytevector_t val(sizeof(get));
  get.action=0x02; // Get
  get.num_of_all_table_addresses=0;
  get.num_of_current_address=0;
  std::copy((char*)&get, (char*)&get+sizeof(get), val.begin());
  SetVariable("AuthorizedMacAddressesGRP.AuthorizedMac.0", val);
  vardesc_t vars[] = {
    {"AuthorizedMacAddressesGRP.AuthorizedMac.0", Value::OctetString},
    {"", Value::Unknown}
  };
  SNMPSet(vars);
  GetVariable("AuthorizedMacAddressesGRP.AuthorizedMac.0", val);
  std::copy(val.begin(), val.end(), (char*)&result);
  short num_entries=result.num_of_all_table_addresses;

  for( int i=0; i<(int)num_entries; ++i )
  {
    get.num_of_current_address=(unsigned short)i+1;
    std::copy((char*)&get, (char*)&get+sizeof(get), val.begin());    
    SetVariable("AuthorizedMacAddressesGRP.AuthorizedMac.0", val);
    SNMPSet(vars);
    GetVariable("AuthorizedMacAddressesGRP.AuthorizedMac.0", val);
    std::copy(val.begin(), val.end(), (char*)&result);    
    bytevector_t v(6);
    std::copy((char*)&result.mac_address, (char*)&result.mac_address+6,v.begin());
    vs.insert(vs.end(), MacToHex(v));
  }

  return vs;
}

/** Sets the authorized MAC list. */
void WAP11Config::SetMACList(const std::vector<std::string> &macs)
{
  AuthorizedMacTableString set;
  bytevector_t val(sizeof(set));
  vardesc_t vars[] = {
    {"AuthorizedMacAddressesGRP.AuthorizedMac.0", Value::OctetString},
    {"", Value::Unknown}
  };

  set.action=0x01; // Set
  set.num_of_all_table_addresses=macs.size();
  
  for( int addr=0; addr<(int)macs.size(); ++addr )
  {
    set.num_of_current_address=(unsigned short)addr+1;
    bytevector_t v=HexToMac(macs[addr]);
    std::copy(v.begin(), v.end(), (char*)&set.mac_address);
    std::copy((char*)&set, (char*)&set+sizeof(set), val.begin());
    SetVariable("AuthorizedMacAddressesGRP.AuthorizedMac.0", val);
    SNMPSet(vars);
  }
}

/** Converts a vector containing 6 octets into a string with a hex representation of the MAC address. */
std::string WAP11Config::MacToHex(const bytevector_t &v)
{
  std::string mac;

  for(int i=0; i<6; ++i)
  {
    mac=mac+HexDigit((v[i]&0xf0)>>4);
    mac=mac+HexDigit(v[i]&0x0f);
    mac=mac+" ";
  }

  return mac;
}

/** Convert a string containing 6 hex octets into a byte vector. */
WAP11Config::bytevector_t WAP11Config::HexToMac(const std::string &s)
{
  bytevector_t v;
  int n=0;
  int ndigit=0;

  for(int i=0; i<(int)s.length(); ++i)
  {
    if(isxdigit(s[i]))
    {
      n<<=4;
      if(isdigit(s[i]))
      n+=s[i]-'0';
      else
      n+=toupper(s[i])-'A'+10;
      ndigit++;
    }
    else
    {
      if(ndigit!=0)
        ndigit=2; // force the current number out
    }

    if(ndigit>1)
    {
      v.push_back((char)n);
      n=0;
      ndigit=0;
    }
  }
  return v;
}
