/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <wangle/client/ssl/test/TestUtil.h>

namespace wangle {

const size_t kNumSessions = 3;
const size_t kSessionDataLen = 1060;

// Faked but real raw session data.
const unsigned char kSessionData[kNumSessions][kSessionDataLen] = {
    // Session 1
    {
        48,  130, 4,   32,  2,   1,   1,   2,   2,   3,   1,   4,   2,   0,
        56,  4,   32,  83,  116, 16,  17,  57,  108, 197, 133, 94,  141, 239,
        83,  184, 65,  171, 17,  90,  82,  17,  245, 146, 215, 220, 102, 13,
        200, 155, 72,  85,  125, 138, 188, 4,   48,  69,  44,  55,  135, 168,
        101, 191, 115, 0,   225, 79,  217, 91,  173, 24,  109, 228, 189, 39,
        192, 112, 237, 37,  92,  27,  110, 49,  224, 110, 108, 96,  55,  131,
        177, 124, 17,  55,  67,  217, 180, 36,  222, 143, 210, 15,  63,  221,
        153, 161, 6,   2,   4,   83,  116, 16,  17,  162, 4,   2,   2,   1,
        44,  163, 130, 3,   166, 48,  130, 3,   162, 48,  130, 2,   138, 2,
        5,   17,  17,  17,  17,  24,  48,  13,  6,   9,   42,  134, 72,  134,
        247, 13,  1,   1,   11,  5,   0,   48,  58,  49,  11,  48,  9,   6,
        3,   85,  4,   6,   19,  2,   85,  83,  49,  11,  48,  9,   6,   3,
        85,  4,   8,   19,  2,   67,  65,  49,  17,  48,  15,  6,   3,   85,
        4,   10,  19,  8,   70,  97,  99,  101, 98,  111, 111, 107, 49,  11,
        48,  9,   6,   3,   85,  4,   3,   19,  2,   67,  65,  48,  30,  23,
        13,  49,  52,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,  90,
        23,  13,  49,  53,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,
        90,  48,  91,  49,  11,  48,  9,   6,   3,   85,  4,   6,   19,  2,
        85,  83,  49,  11,  48,  9,   6,   3,   85,  4,   8,   19,  2,   85,
        83,  49,  17,  48,  15,  6,   3,   85,  4,   10,  19,  8,   102, 97,
        99,  101, 98,  111, 111, 107, 49,  17,  48,  15,  6,   3,   85,  4,
        11,  19,  8,   102, 97,  99,  101, 98,  111, 111, 107, 49,  25,  48,
        23,  6,   3,   85,  4,   3,   19,  16,  119, 119, 119, 46,  102, 97,
        99,  101, 98,  111, 111, 107, 46,  99,  111, 109, 48,  130, 1,   183,
        48,  130, 1,   44,  6,   7,   42,  134, 72,  206, 56,  4,   1,   48,
        130, 1,   31,  2,   129, 129, 0,   253, 127, 83,  129, 29,  117, 18,
        41,  82,  223, 74,  156, 46,  236, 228, 231, 246, 17,  183, 82,  60,
        239, 68,  0,   195, 30,  63,  128, 182, 81,  38,  105, 69,  93,  64,
        34,  81,  251, 89,  61,  141, 88,  250, 191, 197, 245, 186, 48,  246,
        203, 155, 85,  108, 215, 129, 59,  128, 29,  52,  111, 242, 102, 96,
        183, 107, 153, 80,  165, 164, 159, 159, 232, 4,   123, 16,  34,  194,
        79,  187, 169, 215, 254, 183, 198, 27,  248, 59,  87,  231, 198, 168,
        166, 21,  15,  4,   251, 131, 246, 211, 197, 30,  195, 2,   53,  84,
        19,  90,  22,  145, 50,  246, 117, 243, 174, 43,  97,  215, 42,  239,
        242, 34,  3,   25,  157, 209, 72,  1,   199, 2,   21,  0,   151, 96,
        80,  143, 21,  35,  11,  204, 178, 146, 185, 130, 162, 235, 132, 11,
        240, 88,  28,  245, 2,   129, 129, 0,   247, 225, 160, 133, 214, 155,
        61,  222, 203, 188, 171, 92,  54,  184, 87,  185, 121, 148, 175, 187,
        250, 58,  234, 130, 249, 87,  76,  11,  61,  7,   130, 103, 81,  89,
        87,  142, 186, 212, 89,  79,  230, 113, 7,   16,  129, 128, 180, 73,
        22,  113, 35,  232, 76,  40,  22,  19,  183, 207, 9,   50,  140, 200,
        166, 225, 60,  22,  122, 139, 84,  124, 141, 40,  224, 163, 174, 30,
        43,  179, 166, 117, 145, 110, 163, 127, 11,  250, 33,  53,  98,  241,
        251, 98,  122, 1,   36,  59,  204, 164, 241, 190, 168, 81,  144, 137,
        168, 131, 223, 225, 90,  229, 159, 6,   146, 139, 102, 94,  128, 123,
        85,  37,  100, 1,   76,  59,  254, 207, 73,  42,  3,   129, 132, 0,
        2,   129, 128, 90,  172, 11,  241, 173, 170, 216, 73,  212, 145, 255,
        30,  26,  9,   15,  207, 43,  78,  66,  19,  239, 76,  26,  190, 227,
        22,  242, 109, 46,  46,  175, 215, 254, 22,  160, 233, 31,  143, 255,
        186, 226, 156, 111, 37,  178, 2,   80,  99,  156, 208, 19,  30,  145,
        253, 152, 234, 26,  143, 231, 20,  150, 179, 116, 132, 203, 109, 223,
        242, 208, 57,  177, 77,  163, 224, 210, 110, 60,  227, 221, 123, 5,
        129, 238, 4,   83,  16,  48,  124, 84,  246, 98,  97,  176, 147, 187,
        173, 198, 83,  102, 31,  57,  214, 219, 220, 58,  245, 208, 85,  27,
        71,  192, 225, 91,  128, 128, 216, 116, 30,  45,  116, 67,  66,  185,
        183, 215, 145, 219, 155, 48,  13,  6,   9,   42,  134, 72,  134, 247,
        13,  1,   1,   11,  5,   0,   3,   130, 1,   1,   0,   131, 39,  108,
        45,  93,  90,  177, 99,  73,  210, 121, 4,   179, 90,  223, 73,  137,
        74,  146, 142, 26,  230, 134, 5,   79,  159, 26,  82,  139, 69,  223,
        55,  208, 181, 136, 250, 108, 223, 165, 235, 14,  116, 146, 235, 124,
        63,  38,  150, 212, 236, 181, 52,  240, 136, 59,  12,  132, 162, 204,
        202, 219, 72,  7,   254, 210, 109, 182, 249, 126, 61,  55,  37,  82,
        169, 198, 234, 134, 85,  10,  98,  51,  232, 239, 92,  216, 107, 170,
        94,  73,  171, 92,  55,  219, 8,   120, 154, 223, 88,  8,   135, 152,
        101, 112, 184, 71,  37,  6,   238, 53,  233, 113, 227, 222, 248, 102,
        161, 190, 78,  163, 170, 23,  48,  8,   222, 1,   30,  233, 251, 161,
        246, 246, 90,  51,  151, 158, 60,  70,  93,  14,  121, 5,   181, 157,
        8,   243, 211, 252, 95,  44,  201, 186, 80,  6,   55,  36,  54,  128,
        87,  76,  1,   155, 152, 99,  92,  206, 105, 69,  90,  145, 50,  121,
        14,  40,  70,  52,  217, 246, 244, 142, 176, 221, 202, 32,  78,  189,
        229, 99,  91,  141, 81,  173, 112, 138, 251, 131, 255, 141, 3,   191,
        68,  129, 235, 219, 57,  133, 241, 7,   53,  223, 50,  226, 88,  28,
        157, 23,  110, 9,   237, 64,  67,  60,  77,  74,  110, 177, 130, 204,
        41,  52,  57,  121, 103, 93,  154, 248, 114, 140, 163, 124, 227, 102,
        186, 10,  220, 29,  2,   254, 147, 8,   121, 79,  8,   148, 127, 32,
        175, 164, 2,   4,   0,   165, 3,   2,   1,   19,
    },

    // Session 2
    {
        48,  130, 4,   32,  2,   1,   1,   2,   2,   3,   1,   4,   2,   0,
        56,  4,   32,  83,  116, 20,  234, 169, 155, 79,  215, 151, 164, 227,
        219, 181, 70,  17,  183, 202, 177, 252, 12,  110, 80,  38,  213, 174,
        138, 226, 40,  228, 196, 203, 211, 4,   48,  194, 187, 205, 1,   145,
        36,  175, 197, 202, 233, 209, 23,  191, 235, 94,  55,  73,  32,  40,
        166, 23,  153, 103, 249, 237, 103, 172, 110, 20,  165, 51,  92,  228,
        186, 223, 146, 215, 4,   220, 18,  82,  244, 243, 133, 67,  232, 193,
        23,  161, 6,   2,   4,   83,  116, 20,  235, 162, 4,   2,   2,   1,
        44,  163, 130, 3,   166, 48,  130, 3,   162, 48,  130, 2,   138, 2,
        5,   17,  17,  17,  17,  24,  48,  13,  6,   9,   42,  134, 72,  134,
        247, 13,  1,   1,   11,  5,   0,   48,  58,  49,  11,  48,  9,   6,
        3,   85,  4,   6,   19,  2,   85,  83,  49,  11,  48,  9,   6,   3,
        85,  4,   8,   19,  2,   67,  65,  49,  17,  48,  15,  6,   3,   85,
        4,   10,  19,  8,   70,  97,  99,  101, 98,  111, 111, 107, 49,  11,
        48,  9,   6,   3,   85,  4,   3,   19,  2,   67,  65,  48,  30,  23,
        13,  49,  52,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,  90,
        23,  13,  49,  53,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,
        90,  48,  91,  49,  11,  48,  9,   6,   3,   85,  4,   6,   19,  2,
        85,  83,  49,  11,  48,  9,   6,   3,   85,  4,   8,   19,  2,   85,
        83,  49,  17,  48,  15,  6,   3,   85,  4,   10,  19,  8,   102, 97,
        99,  101, 98,  111, 111, 107, 49,  17,  48,  15,  6,   3,   85,  4,
        11,  19,  8,   102, 97,  99,  101, 98,  111, 111, 107, 49,  25,  48,
        23,  6,   3,   85,  4,   3,   19,  16,  119, 119, 119, 46,  102, 97,
        99,  101, 98,  111, 111, 107, 46,  99,  111, 109, 48,  130, 1,   183,
        48,  130, 1,   44,  6,   7,   42,  134, 72,  206, 56,  4,   1,   48,
        130, 1,   31,  2,   129, 129, 0,   253, 127, 83,  129, 29,  117, 18,
        41,  82,  223, 74,  156, 46,  236, 228, 231, 246, 17,  183, 82,  60,
        239, 68,  0,   195, 30,  63,  128, 182, 81,  38,  105, 69,  93,  64,
        34,  81,  251, 89,  61,  141, 88,  250, 191, 197, 245, 186, 48,  246,
        203, 155, 85,  108, 215, 129, 59,  128, 29,  52,  111, 242, 102, 96,
        183, 107, 153, 80,  165, 164, 159, 159, 232, 4,   123, 16,  34,  194,
        79,  187, 169, 215, 254, 183, 198, 27,  248, 59,  87,  231, 198, 168,
        166, 21,  15,  4,   251, 131, 246, 211, 197, 30,  195, 2,   53,  84,
        19,  90,  22,  145, 50,  246, 117, 243, 174, 43,  97,  215, 42,  239,
        242, 34,  3,   25,  157, 209, 72,  1,   199, 2,   21,  0,   151, 96,
        80,  143, 21,  35,  11,  204, 178, 146, 185, 130, 162, 235, 132, 11,
        240, 88,  28,  245, 2,   129, 129, 0,   247, 225, 160, 133, 214, 155,
        61,  222, 203, 188, 171, 92,  54,  184, 87,  185, 121, 148, 175, 187,
        250, 58,  234, 130, 249, 87,  76,  11,  61,  7,   130, 103, 81,  89,
        87,  142, 186, 212, 89,  79,  230, 113, 7,   16,  129, 128, 180, 73,
        22,  113, 35,  232, 76,  40,  22,  19,  183, 207, 9,   50,  140, 200,
        166, 225, 60,  22,  122, 139, 84,  124, 141, 40,  224, 163, 174, 30,
        43,  179, 166, 117, 145, 110, 163, 127, 11,  250, 33,  53,  98,  241,
        251, 98,  122, 1,   36,  59,  204, 164, 241, 190, 168, 81,  144, 137,
        168, 131, 223, 225, 90,  229, 159, 6,   146, 139, 102, 94,  128, 123,
        85,  37,  100, 1,   76,  59,  254, 207, 73,  42,  3,   129, 132, 0,
        2,   129, 128, 90,  172, 11,  241, 173, 170, 216, 73,  212, 145, 255,
        30,  26,  9,   15,  207, 43,  78,  66,  19,  239, 76,  26,  190, 227,
        22,  242, 109, 46,  46,  175, 215, 254, 22,  160, 233, 31,  143, 255,
        186, 226, 156, 111, 37,  178, 2,   80,  99,  156, 208, 19,  30,  145,
        253, 152, 234, 26,  143, 231, 20,  150, 179, 116, 132, 203, 109, 223,
        242, 208, 57,  177, 77,  163, 224, 210, 110, 60,  227, 221, 123, 5,
        129, 238, 4,   83,  16,  48,  124, 84,  246, 98,  97,  176, 147, 187,
        173, 198, 83,  102, 31,  57,  214, 219, 220, 58,  245, 208, 85,  27,
        71,  192, 225, 91,  128, 128, 216, 116, 30,  45,  116, 67,  66,  185,
        183, 215, 145, 219, 155, 48,  13,  6,   9,   42,  134, 72,  134, 247,
        13,  1,   1,   11,  5,   0,   3,   130, 1,   1,   0,   131, 39,  108,
        45,  93,  90,  177, 99,  73,  210, 121, 4,   179, 90,  223, 73,  137,
        74,  146, 142, 26,  230, 134, 5,   79,  159, 26,  82,  139, 69,  223,
        55,  208, 181, 136, 250, 108, 223, 165, 235, 14,  116, 146, 235, 124,
        63,  38,  150, 212, 236, 181, 52,  240, 136, 59,  12,  132, 162, 204,
        202, 219, 72,  7,   254, 210, 109, 182, 249, 126, 61,  55,  37,  82,
        169, 198, 234, 134, 85,  10,  98,  51,  232, 239, 92,  216, 107, 170,
        94,  73,  171, 92,  55,  219, 8,   120, 154, 223, 88,  8,   135, 152,
        101, 112, 184, 71,  37,  6,   238, 53,  233, 113, 227, 222, 248, 102,
        161, 190, 78,  163, 170, 23,  48,  8,   222, 1,   30,  233, 251, 161,
        246, 246, 90,  51,  151, 158, 60,  70,  93,  14,  121, 5,   181, 157,
        8,   243, 211, 252, 95,  44,  201, 186, 80,  6,   55,  36,  54,  128,
        87,  76,  1,   155, 152, 99,  92,  206, 105, 69,  90,  145, 50,  121,
        14,  40,  70,  52,  217, 246, 244, 142, 176, 221, 202, 32,  78,  189,
        229, 99,  91,  141, 81,  173, 112, 138, 251, 131, 255, 141, 3,   191,
        68,  129, 235, 219, 57,  133, 241, 7,   53,  223, 50,  226, 88,  28,
        157, 23,  110, 9,   237, 64,  67,  60,  77,  74,  110, 177, 130, 204,
        41,  52,  57,  121, 103, 93,  154, 248, 114, 140, 163, 124, 227, 102,
        186, 10,  220, 29,  2,   254, 147, 8,   121, 79,  8,   148, 127, 32,
        175, 164, 2,   4,   0,   165, 3,   2,   1,   19,
    },

    // Session 3
    {
        48,  130, 4,   32,  2,   1,   1,   2,   2,   3,   1,   4,   2,   0,
        56,  4,   32,  83,  116, 21,  50,  186, 222, 2,   46,  159, 227, 91,
        165, 32,  209, 124, 156, 57,  111, 20,  98,  150, 88,  20,  242, 184,
        169, 64,  24,  105, 185, 128, 62,  4,   48,  16,  7,   108, 47,  93,
        126, 249, 205, 90,  168, 186, 194, 74,  148, 7,   106, 242, 3,   174,
        174, 164, 134, 24,  159, 195, 191, 86,  10,  212, 217, 14,  156, 50,
        157, 112, 31,  226, 222, 41,  182, 130, 235, 245, 191, 254, 5,   206,
        236, 161, 6,   2,   4,   83,  116, 21,  50,  162, 4,   2,   2,   1,
        44,  163, 130, 3,   166, 48,  130, 3,   162, 48,  130, 2,   138, 2,
        5,   17,  17,  17,  17,  24,  48,  13,  6,   9,   42,  134, 72,  134,
        247, 13,  1,   1,   11,  5,   0,   48,  58,  49,  11,  48,  9,   6,
        3,   85,  4,   6,   19,  2,   85,  83,  49,  11,  48,  9,   6,   3,
        85,  4,   8,   19,  2,   67,  65,  49,  17,  48,  15,  6,   3,   85,
        4,   10,  19,  8,   70,  97,  99,  101, 98,  111, 111, 107, 49,  11,
        48,  9,   6,   3,   85,  4,   3,   19,  2,   67,  65,  48,  30,  23,
        13,  49,  52,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,  90,
        23,  13,  49,  53,  48,  52,  49,  54,  48,  57,  48,  55,  49,  54,
        90,  48,  91,  49,  11,  48,  9,   6,   3,   85,  4,   6,   19,  2,
        85,  83,  49,  11,  48,  9,   6,   3,   85,  4,   8,   19,  2,   85,
        83,  49,  17,  48,  15,  6,   3,   85,  4,   10,  19,  8,   102, 97,
        99,  101, 98,  111, 111, 107, 49,  17,  48,  15,  6,   3,   85,  4,
        11,  19,  8,   102, 97,  99,  101, 98,  111, 111, 107, 49,  25,  48,
        23,  6,   3,   85,  4,   3,   19,  16,  119, 119, 119, 46,  102, 97,
        99,  101, 98,  111, 111, 107, 46,  99,  111, 109, 48,  130, 1,   183,
        48,  130, 1,   44,  6,   7,   42,  134, 72,  206, 56,  4,   1,   48,
        130, 1,   31,  2,   129, 129, 0,   253, 127, 83,  129, 29,  117, 18,
        41,  82,  223, 74,  156, 46,  236, 228, 231, 246, 17,  183, 82,  60,
        239, 68,  0,   195, 30,  63,  128, 182, 81,  38,  105, 69,  93,  64,
        34,  81,  251, 89,  61,  141, 88,  250, 191, 197, 245, 186, 48,  246,
        203, 155, 85,  108, 215, 129, 59,  128, 29,  52,  111, 242, 102, 96,
        183, 107, 153, 80,  165, 164, 159, 159, 232, 4,   123, 16,  34,  194,
        79,  187, 169, 215, 254, 183, 198, 27,  248, 59,  87,  231, 198, 168,
        166, 21,  15,  4,   251, 131, 246, 211, 197, 30,  195, 2,   53,  84,
        19,  90,  22,  145, 50,  246, 117, 243, 174, 43,  97,  215, 42,  239,
        242, 34,  3,   25,  157, 209, 72,  1,   199, 2,   21,  0,   151, 96,
        80,  143, 21,  35,  11,  204, 178, 146, 185, 130, 162, 235, 132, 11,
        240, 88,  28,  245, 2,   129, 129, 0,   247, 225, 160, 133, 214, 155,
        61,  222, 203, 188, 171, 92,  54,  184, 87,  185, 121, 148, 175, 187,
        250, 58,  234, 130, 249, 87,  76,  11,  61,  7,   130, 103, 81,  89,
        87,  142, 186, 212, 89,  79,  230, 113, 7,   16,  129, 128, 180, 73,
        22,  113, 35,  232, 76,  40,  22,  19,  183, 207, 9,   50,  140, 200,
        166, 225, 60,  22,  122, 139, 84,  124, 141, 40,  224, 163, 174, 30,
        43,  179, 166, 117, 145, 110, 163, 127, 11,  250, 33,  53,  98,  241,
        251, 98,  122, 1,   36,  59,  204, 164, 241, 190, 168, 81,  144, 137,
        168, 131, 223, 225, 90,  229, 159, 6,   146, 139, 102, 94,  128, 123,
        85,  37,  100, 1,   76,  59,  254, 207, 73,  42,  3,   129, 132, 0,
        2,   129, 128, 90,  172, 11,  241, 173, 170, 216, 73,  212, 145, 255,
        30,  26,  9,   15,  207, 43,  78,  66,  19,  239, 76,  26,  190, 227,
        22,  242, 109, 46,  46,  175, 215, 254, 22,  160, 233, 31,  143, 255,
        186, 226, 156, 111, 37,  178, 2,   80,  99,  156, 208, 19,  30,  145,
        253, 152, 234, 26,  143, 231, 20,  150, 179, 116, 132, 203, 109, 223,
        242, 208, 57,  177, 77,  163, 224, 210, 110, 60,  227, 221, 123, 5,
        129, 238, 4,   83,  16,  48,  124, 84,  246, 98,  97,  176, 147, 187,
        173, 198, 83,  102, 31,  57,  214, 219, 220, 58,  245, 208, 85,  27,
        71,  192, 225, 91,  128, 128, 216, 116, 30,  45,  116, 67,  66,  185,
        183, 215, 145, 219, 155, 48,  13,  6,   9,   42,  134, 72,  134, 247,
        13,  1,   1,   11,  5,   0,   3,   130, 1,   1,   0,   131, 39,  108,
        45,  93,  90,  177, 99,  73,  210, 121, 4,   179, 90,  223, 73,  137,
        74,  146, 142, 26,  230, 134, 5,   79,  159, 26,  82,  139, 69,  223,
        55,  208, 181, 136, 250, 108, 223, 165, 235, 14,  116, 146, 235, 124,
        63,  38,  150, 212, 236, 181, 52,  240, 136, 59,  12,  132, 162, 204,
        202, 219, 72,  7,   254, 210, 109, 182, 249, 126, 61,  55,  37,  82,
        169, 198, 234, 134, 85,  10,  98,  51,  232, 239, 92,  216, 107, 170,
        94,  73,  171, 92,  55,  219, 8,   120, 154, 223, 88,  8,   135, 152,
        101, 112, 184, 71,  37,  6,   238, 53,  233, 113, 227, 222, 248, 102,
        161, 190, 78,  163, 170, 23,  48,  8,   222, 1,   30,  233, 251, 161,
        246, 246, 90,  51,  151, 158, 60,  70,  93,  14,  121, 5,   181, 157,
        8,   243, 211, 252, 95,  44,  201, 186, 80,  6,   55,  36,  54,  128,
        87,  76,  1,   155, 152, 99,  92,  206, 105, 69,  90,  145, 50,  121,
        14,  40,  70,  52,  217, 246, 244, 142, 176, 221, 202, 32,  78,  189,
        229, 99,  91,  141, 81,  173, 112, 138, 251, 131, 255, 141, 3,   191,
        68,  129, 235, 219, 57,  133, 241, 7,   53,  223, 50,  226, 88,  28,
        157, 23,  110, 9,   237, 64,  67,  60,  77,  74,  110, 177, 130, 204,
        41,  52,  57,  121, 103, 93,  154, 248, 114, 140, 163, 124, 227, 102,
        186, 10,  220, 29,  2,   254, 147, 8,   121, 79,  8,   148, 127, 32,
        175, 164, 2,   4,   0,   165, 3,   2,   1,   19,
    }};

const unsigned char kSessionDataWithTicket[] = {
    48,  130, 6,   65,  2,   1,   1,   2,   2,   3,   3,   4,   2,   192, 43,
    4,   32,  210, 43,  250, 203, 148, 216, 240, 105, 223, 166, 40,  126, 54,
    36,  175, 185, 214, 213, 114, 48,  50,  202, 82,  158, 6,   141, 149, 181,
    31,  38,  204, 204, 4,   48,  250, 59,  244, 243, 75,  147, 73,  231, 212,
    166, 183, 110, 166, 214, 93,  88,  225, 229, 155, 123, 11,  207, 190, 196,
    25,  198, 36,  252, 71,  218, 12,  67,  161, 25,  240, 28,  141, 251, 176,
    240, 170, 122, 162, 200, 157, 0,   220, 217, 161, 6,   2,   4,   85,  186,
    198, 92,  162, 4,   2,   2,   1,   44,  163, 130, 4,   235, 48,  130, 4,
    231, 48,  130, 3,   207, 160, 3,   2,   1,   2,   2,   16,  4,   200, 173,
    121, 70,  20,  4,   241, 110, 145, 123, 2,   222, 229, 117, 116, 48,  13,
    6,   9,   42,  134, 72,  134, 247, 13,  1,   1,   5,   5,   0,   48,  102,
    49,  11,  48,  9,   6,   3,   85,  4,   6,   19,  2,   85,  83,  49,  21,
    48,  19,  6,   3,   85,  4,   10,  19,  12,  68,  105, 103, 105, 67,  101,
    114, 116, 32,  73,  110, 99,  49,  25,  48,  23,  6,   3,   85,  4,   11,
    19,  16,  119, 119, 119, 46,  100, 105, 103, 105, 99,  101, 114, 116, 46,
    99,  111, 109, 49,  37,  48,  35,  6,   3,   85,  4,   3,   19,  28,  68,
    105, 103, 105, 67,  101, 114, 116, 32,  72,  105, 103, 104, 32,  65,  115,
    115, 117, 114, 97,  110, 99,  101, 32,  67,  65,  45,  51,  48,  30,  23,
    13,  49,  52,  48,  56,  50,  56,  48,  48,  48,  48,  48,  48,  90,  23,
    13,  49,  53,  49,  48,  49,  53,  49,  50,  48,  48,  48,  48,  90,  48,
    97,  49,  11,  48,  9,   6,   3,   85,  4,   6,   19,  2,   85,  83,  49,
    11,  48,  9,   6,   3,   85,  4,   8,   19,  2,   67,  65,  49,  19,  48,
    17,  6,   3,   85,  4,   7,   19,  10,  77,  101, 110, 108, 111, 32,  80,
    97,  114, 107, 49,  23,  48,  21,  6,   3,   85,  4,   10,  19,  14,  70,
    97,  99,  101, 98,  111, 111, 107, 44,  32,  73,  110, 99,  46,  49,  23,
    48,  21,  6,   3,   85,  4,   3,   12,  14,  42,  46,  102, 97,  99,  101,
    98,  111, 111, 107, 46,  99,  111, 109, 48,  89,  48,  19,  6,   7,   42,
    134, 72,  206, 61,  2,   1,   6,   8,   42,  134, 72,  206, 61,  3,   1,
    7,   3,   66,  0,   4,   216, 209, 221, 53,  189, 226, 89,  182, 251, 155,
    31,  84,  21,  140, 219, 191, 78,  88,  189, 71,  190, 184, 16,  252, 34,
    233, 210, 158, 152, 248, 73,  42,  37,  251, 148, 70,  228, 66,  153, 132,
    80,  28,  95,  1,   253, 20,  37,  49,  92,  78,  217, 100, 253, 197, 12,
    179, 70,  210, 161, 188, 112, 180, 135, 142, 163, 130, 2,   95,  48,  130,
    2,   91,  48,  31,  6,   3,   85,  29,  35,  4,   24,  48,  22,  128, 20,
    80,  234, 115, 137, 219, 41,  251, 16,  143, 158, 229, 1,   32,  212, 222,
    121, 153, 72,  131, 247, 48,  29,  6,   3,   85,  29,  14,  4,   22,  4,
    20,  67,  9,   147, 64,  250, 17,  75,  48,  51,  236, 242, 135, 110, 141,
    113, 24,  207, 138, 188, 142, 48,  129, 183, 6,   3,   85,  29,  17,  4,
    129, 175, 48,  129, 172, 130, 14,  42,  46,  102, 97,  99,  101, 98,  111,
    111, 107, 46,  99,  111, 109, 130, 12,  102, 97,  99,  101, 98,  111, 111,
    107, 46,  99,  111, 109, 130, 8,   42,  46,  102, 98,  46,  99,  111, 109,
    130, 6,   102, 98,  46,  99,  111, 109, 130, 11,  42,  46,  102, 98,  115,
    98,  120, 46,  99,  111, 109, 130, 11,  42,  46,  102, 98,  99,  100, 110,
    46,  110, 101, 116, 130, 14,  42,  46,  120, 120, 46,  102, 98,  99,  100,
    110, 46,  110, 101, 116, 130, 14,  42,  46,  120, 121, 46,  102, 98,  99,
    100, 110, 46,  110, 101, 116, 130, 14,  42,  46,  120, 122, 46,  102, 98,
    99,  100, 110, 46,  110, 101, 116, 130, 16,  42,  46,  109, 46,  102, 97,
    99,  101, 98,  111, 111, 107, 46,  99,  111, 109, 130, 15,  42,  46,  109,
    101, 115, 115, 101, 110, 103, 101, 114, 46,  99,  111, 109, 130, 13,  109,
    101, 115, 115, 101, 110, 103, 101, 114, 46,  99,  111, 109, 48,  14,  6,
    3,   85,  29,  15,  1,   1,   255, 4,   4,   3,   2,   3,   136, 48,  29,
    6,   3,   85,  29,  37,  4,   22,  48,  20,  6,   8,   43,  6,   1,   5,
    5,   7,   3,   1,   6,   8,   43,  6,   1,   5,   5,   7,   3,   2,   48,
    97,  6,   3,   85,  29,  31,  4,   90,  48,  88,  48,  42,  160, 40,  160,
    38,  134, 36,  104, 116, 116, 112, 58,  47,  47,  99,  114, 108, 51,  46,
    100, 105, 103, 105, 99,  101, 114, 116, 46,  99,  111, 109, 47,  99,  97,
    51,  45,  103, 50,  57,  46,  99,  114, 108, 48,  42,  160, 40,  160, 38,
    134, 36,  104, 116, 116, 112, 58,  47,  47,  99,  114, 108, 52,  46,  100,
    105, 103, 105, 99,  101, 114, 116, 46,  99,  111, 109, 47,  99,  97,  51,
    45,  103, 50,  57,  46,  99,  114, 108, 48,  66,  6,   3,   85,  29,  32,
    4,   59,  48,  57,  48,  55,  6,   9,   96,  134, 72,  1,   134, 253, 108,
    1,   1,   48,  42,  48,  40,  6,   8,   43,  6,   1,   5,   5,   7,   2,
    1,   22,  28,  104, 116, 116, 112, 115, 58,  47,  47,  119, 119, 119, 46,
    100, 105, 103, 105, 99,  101, 114, 116, 46,  99,  111, 109, 47,  67,  80,
    83,  48,  123, 6,   8,   43,  6,   1,   5,   5,   7,   1,   1,   4,   111,
    48,  109, 48,  36,  6,   8,   43,  6,   1,   5,   5,   7,   48,  1,   134,
    24,  104, 116, 116, 112, 58,  47,  47,  111, 99,  115, 112, 46,  100, 105,
    103, 105, 99,  101, 114, 116, 46,  99,  111, 109, 48,  69,  6,   8,   43,
    6,   1,   5,   5,   7,   48,  2,   134, 57,  104, 116, 116, 112, 58,  47,
    47,  99,  97,  99,  101, 114, 116, 115, 46,  100, 105, 103, 105, 99,  101,
    114, 116, 46,  99,  111, 109, 47,  68,  105, 103, 105, 67,  101, 114, 116,
    72,  105, 103, 104, 65,  115, 115, 117, 114, 97,  110, 99,  101, 67,  65,
    45,  51,  46,  99,  114, 116, 48,  12,  6,   3,   85,  29,  19,  1,   1,
    255, 4,   2,   48,  0,   48,  13,  6,   9,   42,  134, 72,  134, 247, 13,
    1,   1,   5,   5,   0,   3,   130, 1,   1,   0,   157, 209, 176, 90,  7,
    249, 48,  235, 219, 240, 17,  203, 96,  137, 113, 197, 247, 246, 48,  88,
    238, 215, 180, 84,  206, 55,  183, 72,  120, 84,  88,  175, 143, 167, 196,
    89,  207, 250, 14,  77,  239, 138, 81,  224, 245, 184, 241, 31,  89,  208,
    174, 115, 198, 37,  201, 209, 242, 243, 214, 218, 32,  165, 166, 51,  139,
    239, 115, 70,  87,  44,  185, 94,  194, 231, 135, 139, 181, 86,  148, 224,
    6,   2,   196, 163, 23,  82,  227, 111, 119, 202, 161, 225, 46,  132, 196,
    125, 89,  25,  143, 6,   135, 112, 3,   220, 88,  50,  81,  74,  86,  207,
    231, 56,  24,  81,  183, 4,   22,  149, 149, 110, 54,  107, 115, 81,  102,
    141, 212, 123, 12,  209, 196, 106, 76,  241, 247, 153, 161, 181, 97,  164,
    18,  241, 47,  82,  228, 213, 44,  109, 120, 89,  20,  106, 88,  24,  233,
    188, 176, 10,  26,  120, 174, 156, 60,  99,  155, 85,  75,  233, 181, 220,
    69,  236, 69,  39,  245, 235, 73,  122, 237, 230, 91,  28,  14,  109, 172,
    218, 93,  28,  229, 19,  8,   182, 172, 141, 32,  249, 157, 112, 220, 5,
    193, 14,  79,  95,  39,  133, 217, 31,  175, 73,  208, 35,  116, 246, 160,
    165, 237, 245, 64,  233, 24,  246, 41,  111, 70,  145, 149, 122, 74,  247,
    44,  124, 88,  99,  189, 38,  249, 140, 24,  85,  98,  83,  21,  180, 77,
    0,   184, 151, 155, 188, 158, 192, 61,  221, 165, 133, 164, 2,   4,   0,
    166, 18,  4,   16,  119, 119, 119, 46,  102, 97,  99,  101, 98,  111, 111,
    107, 46,  99,  111, 109, 169, 5,   2,   3,   1,   79,  240, 170, 129, 195,
    4,   129, 192, 238, 5,   167, 149, 22,  69,  66,  195, 66,  199, 37,  228,
    51,  14,  95,  148, 40,  81,  237, 132, 113, 140, 5,   50,  55,  161, 254,
    160, 244, 140, 46,  231, 248, 101, 230, 101, 175, 234, 52,  102, 99,  68,
    140, 16,  65,  116, 229, 170, 73,  57,  98,  118, 137, 190, 223, 253, 122,
    21,  140, 29,  20,  4,   175, 198, 205, 110, 205, 178, 203, 55,  14,  70,
    56,  216, 120, 167, 110, 214, 198, 182, 221, 2,   141, 36,  136, 43,  22,
    34,  67,  91,  152, 244, 30,  155, 1,   140, 23,  183, 205, 200, 98,  56,
    41,  17,  172, 213, 180, 37,  111, 49,  150, 123, 35,  199, 252, 234, 228,
    6,   202, 113, 194, 171, 32,  55,  104, 75,  158, 140, 100, 80,  57,  2,
    49,  70,  34,  85,  183, 32,  172, 6,   113, 154, 113, 48,  221, 18,  27,
    250, 53,  48,  209, 28,  187, 83,  215, 149, 231, 36,  115, 174, 116, 13,
    217, 199, 167, 38,  222, 204, 0,   175, 80,  31,  26,  253, 119, 182, 235,
    242, 67,  248, 105, 119, 45,  29,  2,   69,  80,  162, 186, 96,  175, 100};

std::vector<std::pair<SSL_SESSION*, size_t>> getSessions() {
  std::vector<std::pair<SSL_SESSION*, size_t>> sessions;

  for (size_t i = 0; i < kNumSessions; ++i) {
    const unsigned char* p = kSessionData[i];
    auto s = d2i_SSL_SESSION(nullptr, &p, kSessionDataLen);
    CHECK(s) << "Invalid session " << i;
    sessions.emplace_back(s, kSessionDataLen);

    // Make sure the same session compares same to itself.
    CHECK(isSameSession(sessions[i], sessions[i]));
  }
  // Make sure all sessions are different from each other
  CHECK(!isSameSession(sessions[0], sessions[1]));
  CHECK(!isSameSession(sessions[1], sessions[2]));
  CHECK(!isSameSession(sessions[2], sessions[0]));

  return sessions;
}

std::pair<SSL_SESSION*, size_t> getSessionWithTicket() {
  const unsigned char* p = kSessionDataWithTicket;
  size_t sessionDataWithTicketLen = sizeof(kSessionDataWithTicket);
  auto s = d2i_SSL_SESSION(nullptr, &p, sessionDataWithTicketLen);
  CHECK(isSameSession(
      std::make_pair(s, sessionDataWithTicketLen),
      std::make_pair(s, sessionDataWithTicketLen)));
  return std::make_pair(s, sessionDataWithTicketLen);
}

folly::ssl::SSLSessionUniquePtr createPersistentTestSession(
    std::pair<SSL_SESSION*, size_t> session) {
  if (session.first) {
    // This simulates what openssl does before handing control to the cache.
    SSL_SESSION_up_ref(session.first);
  }
  return folly::ssl::SSLSessionUniquePtr(session.first);
}

std::string getSessionData(SSL_SESSION* s, size_t expectedLength) {
  std::unique_ptr<unsigned char[]> sessionData(
      new unsigned char[expectedLength]);
  auto p = sessionData.get();
  auto len = (size_t)i2d_SSL_SESSION(s, &p);
  CHECK(expectedLength == len);
  return std::string(reinterpret_cast<const char*>(sessionData.get()), len);
}

bool isSameSession(
    std::pair<SSL_SESSION*, size_t> lhs,
    std::pair<SSL_SESSION*, size_t> rhs) {
  return getSessionData(lhs.first, lhs.second) ==
      getSessionData(rhs.first, rhs.second);
}

} // namespace wangle
