#ifndef rgrl_trans_reduced_quad_h_
#define rgrl_trans_reduced_quad_h_
//:
// \file
// \brief Derived class to represent a reduced quadratic transformation in 2D arbitrary dimensions.
// \author Charlene Tsai
// \date Sep 2003

#include <iostream>
#include <iosfwd>
#include "rgrl_transformation.h"
#ifdef _MSC_VER
#  include <vcl_msvc_warnings.h>
#endif

//:
// A reduced quadratic transformation, when centered, consists of a
// similarity transform + 2 2nd-order radial terms. Uncentering the transform
// makes it look like a quadratic transform
//
class rgrl_trans_reduced_quad
  : public rgrl_transformation
{
 public:
  //: Initialize to the identity transformation.
  //
  rgrl_trans_reduced_quad( unsigned int dimension = 0);

  //: Construct uncentered quadratic standard transform
  //
  //  The transform is q = \a Q * hot(p) + \a A * p + \a trans, where
  //  hot(p)= [px^2 py^2 pxpy]^t in 2D, and
  //  hot(p)= [px^2 py^2 pz^2 pxpy pypz pxpz]^t in 3D.
  //
  rgrl_trans_reduced_quad(vnl_matrix<double> Q, vnl_matrix<double> A,
                          vnl_vector<double> const &trans,
                          vnl_matrix<double> const &covar);

  //: Construct uncentered quadratic standard transform with unknown covar
  //
  //  The transform is q = \a Q * hot(p) + \a A * p + \a trans, where
  //  hot(p)= [px^2 py^2 pxpy]^t in 2D, and
  //  hot(p)= [px^2 py^2 pz^2 pxpy pypz pxpz]^t in 3D.
  //
  //  The covariance matrix is a zero matrix.
  //
  rgrl_trans_reduced_quad(vnl_matrix<double> Q, vnl_matrix<double> A,
                          vnl_vector<double> const &trans);

  //: Construct a centered quadratic transform.
  //
  //  The transform is q = \a Q * hot(p) + \a A * p + \a trans, where
  //  hot(p)= [px^2 py^2 pxpy]^t in 2D, and
  //  hot(p)= [px^2 py^2 pz^2 pxpy pypz pxpz]^t in 3D.
  //
  rgrl_trans_reduced_quad(vnl_matrix<double> Q, vnl_matrix<double> A,
                          vnl_vector<double> trans,
                          vnl_matrix<double> const &covar,
                          vnl_vector<double> const &from_centre,
                          vnl_vector<double> const &to_centre);

  void set_from_center( vnl_vector<double> const& from_center );

  vnl_matrix<double> transfer_error_covar( vnl_vector<double> const& p  ) const override;

  //:  Provide the covariance matrix of the centered estimate (scale is factored in)
  //
  //   For the centered transformation A, the covar matrix is for the parameter
  //   vector [c d a b tx ty]^t, where c and d are for the 2nd order terms,
  //   and a, b, tx and ty are for the similarity transform.
  //
  // defined in base class
  // vnl_matrix<double> covar() const;

  //: The 2nd-order component of the quadratic transform
  vnl_matrix<double> const& Q() const;

  //: The 1st-order component of the quadratic transform
  vnl_matrix<double> const& A() const;

  //: The translation component of the quadratic transform
  vnl_vector<double> const& t() const;

  //:  Inverse map with an initial guess
  void inv_map( const vnl_vector<double>& to,
                        bool initialize_next,
                        const vnl_vector<double>& to_delta,
                        vnl_vector<double>& from,
                        vnl_vector<double>& from_next_est) const override;

  //: Compute jacobian w.r.t. location
  void jacobian_wrt_loc( vnl_matrix<double>& jac, vnl_vector<double> const& from_loc ) const override;

  //:  transform the transformation for images of different resolution
  rgrl_transformation_sptr scale_by( double scale ) const override;

  // Defines type-related functions
  rgrl_type_macro( rgrl_trans_reduced_quad, rgrl_transformation );

  //: Output UNCENTERED transformation, with the origin as the center.
  void write(std::ostream& os ) const override;

  // for input
  bool read(std::istream& is ) override;

  //: make a clone copy
  rgrl_transformation_sptr clone() const override;

 protected:
  void map_loc( vnl_vector<double> const& from,
                vnl_vector<double>      & to ) const override;

  void map_dir( vnl_vector<double> const& from_loc,
                vnl_vector<double> const& from_dir,
                vnl_vector<double>      & to_dir    ) const override;
 private:
  //: Return the vector of 2nd order terms of p = [x y]^t
  vnl_vector<double> higher_order_terms(vnl_vector<double> p) const;

 private:
  vnl_matrix<double> Q_;
  vnl_matrix<double> A_;
  vnl_vector<double> trans_;
  vnl_vector<double> from_centre_;

  // TODO - pure virtual functions of rgrl_transformation
  void inv_map(vnl_vector<double> const&, vnl_vector<double>&) const override;
  rgrl_transformation_sptr inverse_transform() const override;
};


#endif
