#ifndef boxm2_vecf_cranium_scene_h_
#define boxm2_vecf_cranium_scene_h_
//:
// \file
// \brief  boxm2_vecf_cranium_scene models the cranium
//
// \author J.L. Mundy
// \date   6 Nov 2015
//
// For the time being the cranium shape will not be adjusted except for the
// global affine transformation of the skull (including the cranium)
//
#include <string>
#include <vector>
#include <iostream>
#include <set>
#include <boxm2/boxm2_block.h>
#include <boxm2/vecf/boxm2_vecf_articulated_scene.h>
#include <boxm2/vecf/boxm2_vecf_articulated_params.h>
#include <boxm2/boxm2_scene.h>
#include <boxm2/boxm2_data.h>
#include <vgl/algo/vgl_rotation_3d.h>
#include "boxm2_vecf_cranium_params.h"
#include "boxm2_vecf_cranium.h"
#include <vgl/vgl_point_3d.h>
#ifdef _MSC_VER
#  include <vcl_msvc_warnings.h>
#endif

class boxm2_vecf_cranium_scene : public boxm2_vecf_articulated_scene
{
 public:
  enum anat_type { CRANIUM, NO_TYPE};
  boxm2_vecf_cranium_scene(): cranium_data_(nullptr),boxm2_vecf_articulated_scene(){}

  //: set parameters
  bool set_params(boxm2_vecf_articulated_params const& params) override;

  //: construct from scene file specification, use exising database unless initialize == true
  // otherwise compute voxel contents from the cranium parameters
  boxm2_vecf_cranium_scene(std::string const& scene_file);

  boxm2_vecf_cranium_scene(std::string const& scene_file, std::string const& geometry_file);

  //: map cranium data to the target scene
  void map_to_target(boxm2_scene_sptr target_scene) override;

  bool inverse_vector_field(vgl_point_3d<double> const& target_pt, vgl_vector_3d<double>& inv_vf) const override;
  bool apply_vector_field(cell_info const& target_cell, vgl_vector_3d<double> const& inv_vf) override;


  //: compute an inverse vector field for rotation of cranium
  void inverse_vector_field(std::vector<vgl_vector_3d<double> >& vfield, std::vector<bool>& valid) const;

  //: test the anat_type (CRANIUM) of the voxel that contains a global point
  bool is_type_global(vgl_point_3d<double> const& global_pt, anat_type type) const;

 //: test the anat_type (CRANIUM) of a given data index
 bool is_type_data_index(unsigned data_index, anat_type type) const;

  //: set up pointers to source block databases
 void extract_block_data();

 //: interpolate the alpha and appearance data around the vector field source location
 void interpolate_vector_field(vgl_point_3d<double> const& src, unsigned sindx, unsigned dindx, unsigned tindx,
                                std::vector<vgl_point_3d<double> > & cell_centers,
                                std::map<unsigned, std::vector<unsigned> >& cell_neighbor_cell_index,
                               std::map<unsigned, std::vector<unsigned> >&cell_neighbor_data_index);


 void apply_vector_field_to_target(std::vector<vgl_vector_3d<double> > const& vf, std::vector<bool> const& valid);

 // find nearest cell and return the data index of the nearest cell (found depth is for debug, remove at some point)
 bool find_nearest_data_index(boxm2_vecf_cranium_scene::anat_type type, vgl_point_3d<double> const& probe, double cell_len, unsigned& data_indx, int& found_depth) const;


  //re-create geometry according to params_
  void rebuild();
  //check for intrinsic parameter change
  bool vfield_params_change_check(const boxm2_vecf_cranium_params& params);
  // store the neigbors of each cell for each anatomical component in a vector;
  void cache_neighbors();

  //: refine target cells to match the refinement level of the source block
  int prerefine_target_sub_block(vgl_point_3d<double> const& sub_block_pt, unsigned pt_index) override;

  //: compute inverse vector field for unrefined sub_block centers
  void inverse_vector_field_unrefined(std::vector<vgl_point_3d<double> > const& unrefined_target_pts) override;

    // ============   cranium methods ================
 //: construct manidble from parameters
 void create_cranium();
 //:read block eye data and reset indices
 void recreate_cranium();
 //: cache index of neighboring cells
 void find_cell_neigborhoods();
 //: set manible anatomy flags
 void cache_cell_centers_from_anatomy_labels();
 //: scan dense set of points on the manbible
 void build_cranium();
 //: assign appearance to parts of the cranium
 void paint_cranium();
 void reset_buffers();

  //: members
 boxm2_vecf_cranium cranium_geo_;

 std::vector<cell_info> box_cell_centers_;       // cell centers in the target block


  boxm2_vecf_cranium_params params_;               // parameter struct
  // =============  cranium ===============
  boxm2_data_base* cranium_base_;
  boxm2_data<BOXM2_PIXEL>::datatype* cranium_data_;        // is voxel a cranium point

  std::vector<vgl_point_3d<double> > cranium_cell_centers_; // centers of cranium cells
  std::vector<unsigned> cranium_cell_data_index_;           // corresponding data indices
  //      cell_index          cell_index
  std::map<unsigned, std::vector<unsigned> > cell_neighbor_cell_index_; // neighbors of each cranium voxel
  //     data_index cell_index
  std::map<unsigned, unsigned > data_index_to_cell_index_;             // data index to cell index
  //      data_index          data_index
  std::map<unsigned, std::vector<unsigned> > cell_neighbor_data_index_; // data index to neighbor data indices

private:

 //: assign target cell centers that map to the source scene bounding box
  void determine_target_box_cell_centers();
};

#endif // boxm2_vecf_cranium_scene_h_
