/*

Copyright (c) 2010--2011, Stephane Magnenat, ASL, ETHZ, Switzerland
You can contact the author at <stephane at magnenat dot net>

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL ETH-ASL BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef __NABO_H
#define __NABO_H
// remove dependency on Eigen
// replaced with vnl - JLM

//#include "Eigen/Core"
//#if EIGEN_VERSION_AT_LEAST(2,92,0)
//#define EIGEN3_API
//#endif
//#ifndef EIGEN3_API
//        #include "Eigen/Array"
//#endif

#include <vector>
#include <map>
#include <limits>
#include <vnl/vnl_vector.h>
#include <vnl/vnl_matrix.h>


 //remove dependency on boost - JLM
 //#include <boost/any.hpp>


//:
// \file bvgl_nabo.h
// \brief  Stephane Magnenat's nabo k nearest neighbor algorithm ported to use vnl
//
// \author J.L. Mundy
// \date   6 November 2015
//


/*!

\mainpage libnabo

from http://github.com/ethz-asl/libnabo by Stéphane Magnenat (http://stephane.magnenat.net),
ASL-ETHZ, Switzerland (http://www.asl.ethz.ch)

libnabo is a fast K Nearest Neighbour library for low-dimensional spaces.
It provides a clean, legacy-free, scalar-type–agnostic API thanks to C++ templates.
Its current CPU implementation is strongly inspired by \ref ANN, but with more compact data types.
On the average, libnabo is 5% to 20% faster than \ref ANN.
==============================================================================
====  These dependencies are eliminated: Eigen replaced with vnl - JLM ========
//libnabo depends on \ref Eigen, a modern C++ matrix and linear-algebra library.
//libnabo works with either version 2 or 3 of Eigen.
//libnabo also depends on \ref Boost, a C++ general library.
================================================================================

\section ConstructionParameters Construction parameters

The following additional construction parameters are available in KDTREE_ algorithms:
- \c bucketSize (\c unsigned): bucket size, defaults to 8

\section Citing Citing libnabo

If you use libnabo in the academic context, please cite this paper that evaluates its performances in the contex of ICP:
\code
@article{elsebergcomparison,
        title={Comparison of nearest-neighbor-search strategies and implementations for efficient shape registration},
        author={Elseberg, J. and Magnenat, S. and Siegwart, R. and N{\"u}chter, A.},
        journal={Journal of Software Engineering for Robotics (JOSER)},
        pages={2--12},
        volume={3},
        number={1},
        year={2012},
        issn={2035-3928}
}
\endcode

\section BugReporting Bug reporting

Please use <a href="http://github.com/ethz-asl/libnabo/issues">github's issue tracker</a> to report bugs.

\section License

libnabo is released under a permissive BSD license.

\section Faq

\subsection ANN

libnabo differs from \ref ANN on the following points:

* API
- templates for scalar type
- self-match option as execution-time (instead of compile-time) parameter
- range search instead of radius search
-  uses vnl for vectors and matrices
- reentrant

* limitations
- only euclidean distance
- only KD-tree, no BD-tree
- only ANN_KD_SL_MIDPT splitting rules

* implementation
- optional O(log(n)) tree heap instead of O(n) vector heap
- compact memory representation, one memory allocation for all nodes, 5-fold smaller memory footprint compared than ANN
- implicit reference to left child (always next node in array)
- do not store bounds in nodes (that is, I do it like in ANN's article instead of like in ANN's source code)

* performances
- about 5% to 20% faster than ANN (both -O3 -NDEBUG), probably due to the smaller memory footprint
- clearly memory-bound, neither OpenMP nor boost::thread improve performances

\section References

\li \anchor ANN ANN: http://www.cs.umd.edu/~mount/ANN
\li \anchor CMake CMake: http://www.cmake.org
\li \anchor Boost Boost: http://www.boost.org

*/
//! Namespace for Nabo
namespace Nabo
{
  //! \defgroup public public interface
  //@{

  //! version of the Nabo library as string
#define NABO_VERSION "1.0.6"
  //! version of the Nabo library as an int
#define NABO_VERSION_INT 10006

  //! Parameter vector - in original Nabo T, was defined by boost::any, now templated
  template<typename T>
  struct Parameters: public std::map<std::string, T>
    {
      //! Create an empty parameter vector
      Parameters()= default;
      //! Create a parameter vector with a single entry
      /** \param key entry key
       * \param value entry value
       */
      Parameters(const std::string& key, const T& value){(*this)[key] = value;}
      //! Get the value of a key, return defaultValue if the key does not exist
      /** \param key requested key
       * \param defaultValue value to return if the key does not exist
       * \return value of the key, or defaultValue if the key does not exist
       */
      T get(const std::string& key, const T& defaultValue) const
        {
          typename std::map<std::string, T>::const_iterator it(this->find(key));
          if (it != this->end())
            return it->second;
          else
            return defaultValue;
        }
    };

  //! Nearest neighbour search interface, templatized on scalar coordinate type
  // the vector and matrix classes changed from Eigen to vnl
  template<typename T, typename Cloud_T = vnl_matrix<T> >
    struct NearestNeighbourSearch
    {
      //! a vector of type T, to hold the coordinates of a point
      typedef vnl_vector<T> Vector;
      //! a vnl matrix in which each column is a point; this matrix has dim rows
      typedef vnl_matrix<T> Matrix;
      //! a vnl matrix in which each column is a point; this matrix has dim rows
      typedef Cloud_T CloudType;
      //! an index to a Vector or a Matrix, for refering to data points
      typedef int Index;
      //! a vector of indices to data points
      typedef vnl_vector<Index> IndexVector;
      //! a matrix of indices to data points
      typedef vnl_matrix<Index> IndexMatrix;

      //! the data-point cloud
      const CloudType cloud;
      //! the dimensionality of the data-point cloud
      const Index dim;
      //! creation options
      const unsigned creationOptionFlags;
      //! the low bound of the search space (axis-aligned bounding box)
      const Vector minBound;
      //! the high bound of the search space (axis-aligned bounding box)
      const Vector maxBound;

      //! type of search
      enum SearchType
        {
          BRUTE_FORCE = 0, //!< brute force, check distance to every point in the data
          KDTREE_LINEAR_HEAP, //!< kd-tree with linear heap, good for small k (~up to 30)
          KDTREE_TREE_HEAP, //!< kd-tree with tree heap, good for large k (~from 30)
          KDTREE_CL_PT_IN_NODES, //!< kd-tree using openCL, pt in nodes, only available if OpenCL enabled, UNSTABLE API
          KDTREE_CL_PT_IN_LEAVES, //!< kd-tree using openCL, pt in leaves, only available if OpenCL enabled, UNSTABLE API
          BRUTE_FORCE_CL, //!< brute-force using openCL, only available if OpenCL enabled, UNSTABLE API
          SEARCH_TYPE_COUNT //!< number of search types
        };

      //! creation option
      enum CreationOptionFlags
        {
          TOUCH_STATISTICS = 1 //!< perform statistics on the number of points touched
        };

      //! search option
      enum SearchOptionFlags
        {
          ALLOW_SELF_MATCH = 1, //!< allows the return of the same point as the query, if this point is in the data cloud; forbidden by default
          SORT_RESULTS = 2 //!< sort points by distances, when k > 1; do not sort by default
        };

      //! Find the k nearest neighbours of query
      /*!        If the search finds less than k points, the empty entries in dists2 will be filled with infinity and the indices with 0.
       *        \If you must query more than one point at once, use the version of the knn() function taking matrices as input, because it is much faster.
       *        \param query query point
       *        \param indices indices of nearest neighbours, must be of size k
       *        \param dists2 squared distances to nearest neighbours, must be of size k
       *        \param k number of nearest neighbour requested
       *        \param epsilon maximal ratio of error for approximate search, 0 for exact search;
       *        \has no effect if the number of neighbour found is smaller than the number requested
       *        \param optionFlags search options, a bitwise OR of elements of SearchOptionFlags
       *        \param maxRadius maximum radius in which to search, can be used to prune search, is not affected by epsilon
       *        \return if creationOptionFlags contains TOUCH_STATISTICS, return the number of points touched, otherwise return 0
       */
      unsigned long knn(const Vector& query, IndexVector& indices, Vector& dists2, const Index k = 1, const T epsilon = 0, const unsigned optionFlags = 0, const T maxRadius = std::numeric_limits<T>::infinity()) const;

      //! Find the k nearest neighbours for each point of query
      /*!        If the search finds less than k points, the empty entries in dists2 will be filled with infinity and the indices with 0.
       *        \param query query points
       *        \param indices indices of nearest neighbours, must be of size k x query.cols()
       *        \param dists2 squared distances to nearest neighbours, must be of size k x query.cols()
       *        \param k number of nearest neighbour requested
       *        \param epsilon maximal ratio of error for approximate search, 0 for exact search;
       *        \has no effect if the number of neighbour found is smaller than the number requested
       *        \param optionFlags search options, a bitwise OR of elements of SearchOptionFlags
       *        \param maxRadius maximum radius in which to search, can be used to prune search, is not affected by epsilon
       *        \return if creationOptionFlags contains TOUCH_STATISTICS, return the number of points touched, otherwise return 0
       */
      virtual unsigned long knn(const Matrix& query, IndexMatrix& indices, Matrix& dists2, const Index k = 1, const T epsilon = 0, const unsigned optionFlags = 0, const T maxRadius = std::numeric_limits<T>::infinity()) const = 0;

      //! Find the k nearest neighbours for each point of query
      /*!        If the search finds less than k points, the empty entries in dists2 will be filled with infinity and the indices with 0.
       *        \param query query points
       *        \param indices indices of nearest neighbours, must be of size k x query.cols()
       *        \param dists2 squared distances to nearest neighbours, must be of size k x query.cols()
       *        \param maxRadii vector of maximum radii in which to search, used to prune search, is not affected by epsilon
       *        \param k number of nearest neighbour requested
       *        \param epsilon maximal ratio of error for approximate search, 0 for exact search;
       *        \has no effect if the number of neighbour found is smaller than the number requested
       *        \param optionFlags search options, a bitwise OR of elements of SearchOptionFlags
       *        \return if creationOptionFlags contains TOUCH_STATISTICS, return the number of point touched, otherwise return 0
       */
      virtual unsigned long knn(const Matrix& query, IndexMatrix& indices, Matrix& dists2, const Vector& maxRadii, const Index k = 1, const T epsilon = 0, const unsigned optionFlags = 0) const = 0;

      //! Create a nearest-neighbour search
      /*!        \param cloud data-point cloud in which to search
       *        \param dim number of dimensions to consider, must be lower or equal to cloud.rows()
       *        \param preferedType type of search, one of SearchType
       *        \param creationOptionFlags creation options, a bitwise OR of elements of CreationOptionFlags
       *        \param additionalParameters additional parameters, currently only useful for KDTREE_
       *        \return an object on which to run nearest neighbour queries */
      static NearestNeighbourSearch* create(const CloudType& cloud, const Index dim = std::numeric_limits<Index>::max(), const SearchType preferedType = KDTREE_LINEAR_HEAP, const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>());

      //! Create a nearest-neighbour search, using brute-force search, useful for comparison only
      /*!        This is an helper function, you can also use create() with BRUTE_FORCE as preferedType
       *        \param cloud data-point cloud in which to search
       *        \param dim number of dimensions to consider, must be lower or equal to cloud.rows()
       *        \param creationOptionFlags creation options, a bitwise OR of elements of CreationOptionFlags
       *        \return an object on which to run nearest neighbour queries */
      static NearestNeighbourSearch* createBruteForce(const CloudType& cloud, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0);

      //! Create a nearest-neighbour search, using a kd-tree with linear heap, good for small k (~up to 30)
      /*!        This is an helper function, you can also use create() with KDTREE_LINEAR_HEAP as preferedType
       *        \param cloud data-point cloud in which to search
       *        \param dim number of dimensions to consider, must be lower or equal to cloud.rows()
       *        \param creationOptionFlags creation options, a bitwise OR of elements of CreationOptionFlags
       *        \param additionalParameters additional parameters
       *         \return an object on which to run nearest neighbour queries */
      static NearestNeighbourSearch* createKDTreeLinearHeap(const CloudType& cloud, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>());

      //! Create a nearest-neighbour search, using a kd-tree with tree heap, good for large k (~from 30)
      /*!        This is an helper function, you can also use create() with KDTREE_TREE_HEAP as preferedType
       *        \param cloud data-point cloud in which to search
       *        \param dim number of dimensions to consider, must be lower or equal to cloud.rows()
       *        \param creationOptionFlags creation options, a bitwise OR of elements of CreationOptionFlags
       *        \param additionalParameters additional parameters
       *         \return an object on which to run nearest neighbour queries */
      static NearestNeighbourSearch* createKDTreeTreeHeap(const CloudType& cloud, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>());


      //! Prevent creation of trees with the wrong matrix type. Currently only dynamic size matrices are supported.
      template <typename WrongMatrixType>
      static NearestNeighbourSearch* create(const WrongMatrixType& cloud, const Index dim = std::numeric_limits<Index>::max(), const SearchType preferedType = KDTREE_LINEAR_HEAP, const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>())
      {
        typedef int Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef[sizeof(WrongMatrixType) > 0 ? -1 : 1];
        Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef dummy;
        return NULL;
      }

      //! Prevent creation of trees with the wrong matrix type. Currently only dynamic size matrices are supported.
      template <typename WrongMatrixType>
      static NearestNeighbourSearch* createBruteForce(const WrongMatrixType& cloud, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0)
      {
        typedef int Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef[sizeof(WrongMatrixType) > 0 ? -1 : 1];
        Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef dummy;
        return NULL;
      }

      //! Prevent creation of trees with the wrong matrix type. Currently only dynamic size matrices are supported.
      template <typename WrongMatrixType>
      static NearestNeighbourSearch* createKDTreeLinearHeap(const WrongMatrixType& cloud, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>())
      {
        typedef int Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef[sizeof(WrongMatrixType) > 0 ? -1 : 1];
        Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef dummy;
        return NULL;
      }

      //! Prevent creation of trees with the wrong matrix type. Currently only dynamic size matrices are supported.
      template <typename WrongMatrixType>
      static NearestNeighbourSearch* createKDTreeTreeHeap(const WrongMatrixType&, const Index dim = std::numeric_limits<Index>::max(), const unsigned creationOptionFlags = 0, const Parameters<unsigned>& additionalParameters = Parameters<unsigned>())
      {
        typedef int Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef[sizeof(WrongMatrixType) > 0 ? -1 : 1];
        Please_make_sure_that_the_decltype_of_the_first_parameter_is_equal_to_the_Matrix_typedef dummy;
        return NULL;
      }

      //! virtual destructor
      virtual ~NearestNeighbourSearch() = default;

    protected:
      //! constructor
      NearestNeighbourSearch(const CloudType& cloud, const Index dim, const unsigned creationOptionFlags);

      //! Make sure that the output matrices have the right sizes. Throw an exception otherwise.
      /*!        \param query query points
       *        \param k number of nearest neighbour requested
       *        \param indices indices of nearest neighbours, must be of size k x query.cols()
       *        \param dists2 squared distances to nearest neighbours, must be of size k x query.cols()
       *        \param optionFlags the options passed to knn()
       \param maxRadii if non 0, maximum radii, must be of size k */
      void checkSizesKnn(const Matrix& query, const IndexMatrix& indices, const Matrix& dists2, const Index k, const unsigned optionFlags, const Vector* maxRadii = nullptr) const;
 };

  // Convenience typedefs

  //! nearest neighbour search with scalars of type float
  typedef NearestNeighbourSearch<float> NNSearchF;
  //! nearest neighbour search with scalars of type double
  typedef NearestNeighbourSearch<double> NNSearchD;

  //@}
}

#endif // __NABO_H
