//============================================================================
//  The contents of this file are covered by the Viskores license. See
//  LICENSE.txt for details.
//
//  By contributing to this file, all contributors agree to the Developer
//  Certificate of Origin Version 1.1 (DCO 1.1) as stated in DCO.txt.
//============================================================================

//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#ifndef viskores_worklet_mir_mirtables_h
#define viskores_worklet_mir_mirtables_h

#include <viskores/CellShape.h>
#include <viskores/Types.h>

namespace viskores
{
namespace worklet
{
namespace MIRCases
{

// Defines all clipping cases for MIR. Based off of clipping cases for Clip worklet, but includes color data for MIR.
// Firstly, define all points.
static constexpr viskores::IdComponent P0 = 0;
static constexpr viskores::IdComponent P1 = 1;
static constexpr viskores::IdComponent P2 = 2;
static constexpr viskores::IdComponent P3 = 3;
static constexpr viskores::IdComponent P4 = 4;
static constexpr viskores::IdComponent P5 = 5;
static constexpr viskores::IdComponent P6 = 6;
static constexpr viskores::IdComponent P7 = 7;

// EA -> EL are edges 0->11 respectively
static constexpr viskores::IdComponent EA = 8;
static constexpr viskores::IdComponent EB = 9;
static constexpr viskores::IdComponent EC = 10;
static constexpr viskores::IdComponent ED = 11;
static constexpr viskores::IdComponent EE = 12;
static constexpr viskores::IdComponent EF = 13;
static constexpr viskores::IdComponent EG = 14;
static constexpr viskores::IdComponent EH = 15;
static constexpr viskores::IdComponent EI = 16;
static constexpr viskores::IdComponent EJ = 17;
static constexpr viskores::IdComponent EK = 18;
static constexpr viskores::IdComponent EL = 19;

// Center point (N0 is the reference to the center point, SH_PNT means to create the centerpoint)
static constexpr viskores::IdComponent N0 = 127;
static constexpr viskores::IdComponent SH_PNT = 128;

// Viskores shapes, with correct IDs for each. Extracted from the clip filter in viskores.
static constexpr viskores::IdComponent SH_EMP = 0;
static constexpr viskores::IdComponent SH_VTX = 1;
static constexpr viskores::IdComponent SH_PVTX = 2;
static constexpr viskores::IdComponent SH_LIN = 3;
static constexpr viskores::IdComponent SH_PLIN = 4;
static constexpr viskores::IdComponent SH_TRI = 5;
static constexpr viskores::IdComponent SH_TRSP = 6;
static constexpr viskores::IdComponent SH_POLY = 7;
static constexpr viskores::IdComponent SH_PIX = 8;
static constexpr viskores::IdComponent SH_QUA = 9;
static constexpr viskores::IdComponent SH_TET = 10;
static constexpr viskores::IdComponent SH_VOX = 11;
static constexpr viskores::IdComponent SH_HEX = 12;
static constexpr viskores::IdComponent SH_WDG = 13;
static constexpr viskores::IdComponent SH_PYR = 14;


// Color defines, nocolor is only used with center points, meant to convey that the main color for that point is unpredictable.
static constexpr viskores::IdComponent COLOR0 = 0;
static constexpr viskores::IdComponent COLOR1 = 1;
static constexpr viskores::IdComponent NOCOLOR = 2;

// clang-format off

VISKORES_STATIC_CONSTEXPR_ARRAY viskores::UInt8 MIRShapeCount[] = {
//{
1,1, // Vertex
1,  2,  2,  1, // Line
1,  2,  2,  2,  2,  2,  2,  1, // Triangle
// Pixel
1,  3,  3,  2,  3,  2,  4,  3, // cases 0 - 7
3,  4,  2,  3,  2,  3,  3,  1,  // cases 8 - 15
// Quad
1,  3,  3,  2,  3,  4,  2,  3, // cases 0 - 7
3,  2,  4,  3,  2,  3,  3,  1,  // cases 8 - 15
// Tet
1,  2,  2,  2,  2,  2,  2,  2, // cases 0 - 7
2,  2,  2,  2,  2,  2,  2,  1,  // cases 8 - 15
// Voxel
1,  10,  10,  3,  10,  3,  18,  15, // cases 0 - 7
10,  18,  3,  15,  3,  15,  15,  2, // cases 8 - 15
10,  3,  18,  15,  18,  15,  11,  10, // cases 16 - 23
8,  18,  18,  17,  18,  17,  16,  15, // cases 24 - 31
10,  18,  3,  15,  8,  18,  18,  17, // cases 32 - 39
18,  11,  15,  10,  18,  16,  17,  15, // cases 40 - 47
3,  15,  15,  2,  18,  17,  16,  15, // cases 48 - 55
18,  16,  17,  15,  4,  13,  13,  3, // cases 56 - 63
10,  18,  8,  18,  3,  15,  18,  17, // cases 64 - 71
18,  11,  18,  16,  15,  10,  17,  15, // cases 72 - 79
3,  15,  18,  17,  15,  2,  16,  15, // cases 80 - 87
18,  16,  4,  13,  17,  15,  13,  3, // cases 88 - 95
18,  11,  18,  16,  18,  16,  4,  13, // cases 96 - 103
11,  9,  16,  8,  16,  8,  13,  7, // cases 104 - 111
15,  10,  17,  15,  17,  15,  13,  3, // cases 112 - 119
16,  8,  13,  7,  13,  7,  8,  10, // cases 120 - 127
10,  8,  18,  18,  18,  18,  11,  16, // cases 128 - 135
3,  18,  15,  17,  15,  17,  10,  15, // cases 136 - 143
18,  18,  11,  16,  11,  16,  9,  8, // cases 144 - 151
18,  4,  16,  13,  16,  13,  8,  7, // cases 152 - 159
3,  18,  15,  17,  18,  4,  16,  13, // cases 160 - 167
15,  16,  2,  15,  17,  13,  15,  3, // cases 168 - 175
15,  17,  10,  15,  16,  13,  8,  7, // cases 176 - 183
17,  13,  15,  3,  13,  8,  7,  10, // cases 184 - 191
3,  18,  18,  4,  15,  17,  16,  13, // cases 192 - 199
15,  16,  17,  13,  2,  15,  15,  3, // cases 200 - 207
15,  17,  16,  13,  10,  15,  8,  7, // cases 208 - 215
17,  13,  13,  8,  15,  3,  7,  10, // cases 216 - 223
15,  16,  17,  13,  17,  13,  13,  8, // cases 224 - 231
10,  8,  15,  7,  15,  7,  3,  10, // cases 232 - 239
2,  15,  15,  3,  15,  3,  7,  10, // cases 240 - 247
15,  7,  3,  10,  3,  10,  10,  1,  // cases 248 - 255
// Hex
1,  10,  10,  3,  10,  18,  3,  15, // cases 0 - 7
10,  3,  18,  15,  3,  15,  15,  2, // cases 8 - 15
10,  3,  18,  15,  8,  18,  18,  17, // cases 16 - 23
18,  15,  11,  10,  18,  17,  16,  15, // cases 24 - 31
10,  18,  3,  15,  18,  11,  15,  10, // cases 32 - 39
8,  18,  18,  17,  18,  16,  17,  15, // cases 40 - 47
3,  15,  15,  2,  18,  16,  17,  15, // cases 48 - 55
18,  17,  16,  15,  4,  13,  13,  3, // cases 56 - 63
10,  8,  18,  18,  3,  18,  15,  17, // cases 64 - 71
18,  18,  11,  16,  15,  17,  10,  15, // cases 72 - 79
18,  18,  11,  16,  18,  4,  16,  13, // cases 80 - 87
11,  16,  9,  8,  16,  13,  8,  7, // cases 88 - 95
3,  18,  15,  17,  15,  16,  2,  15, // cases 96 - 103
18,  4,  16,  13,  17,  13,  15,  3, // cases 104 - 111
15,  17,  10,  15,  17,  13,  15,  3, // cases 112 - 119
16,  13,  8,  7,  13,  8,  7,  10, // cases 120 - 127
10,  18,  8,  18,  18,  11,  18,  16, // cases 128 - 135
3,  15,  18,  17,  15,  10,  17,  15, // cases 136 - 143
3,  15,  18,  17,  18,  16,  4,  13, // cases 144 - 151
15,  2,  16,  15,  17,  15,  13,  3, // cases 152 - 159
18,  11,  18,  16,  11,  9,  16,  8, // cases 160 - 167
18,  16,  4,  13,  16,  8,  13,  7, // cases 168 - 175
15,  10,  17,  15,  16,  8,  13,  7, // cases 176 - 183
17,  15,  13,  3,  13,  7,  8,  10, // cases 184 - 191
3,  18,  18,  4,  15,  16,  17,  13, // cases 192 - 199
15,  17,  16,  13,  2,  15,  15,  3, // cases 200 - 207
15,  17,  16,  13,  17,  13,  13,  8, // cases 208 - 215
10,  15,  8,  7,  15,  3,  7,  10, // cases 216 - 223
15,  16,  17,  13,  10,  8,  15,  7, // cases 224 - 231
17,  13,  13,  8,  15,  7,  3,  10, // cases 232 - 239
2,  15,  15,  3,  15,  7,  3,  10, // cases 240 - 247
15,  3,  7,  10,  3,  10,  10,  1,  // cases 248 - 255
//Wedge
1,  3,  3,  9,  3,  9,  9,  2, // cases 0 - 7
3,  2,  15,  13,  15,  13,  10,  9, // cases 8 - 15
3,  15,  2,  13,  15,  10,  13,  9, // cases 16 - 23
9,  13,  13,  2,  10,  5,  5,  3, // cases 24 - 31
3,  15,  15,  10,  2,  13,  13,  9, // cases 32 - 39
9,  13,  10,  5,  13,  2,  5,  3, // cases 40 - 47
9,  10,  13,  5,  13,  5,  2,  3, // cases 48 - 55
2,  9,  9,  3,  9,  3,  3,  1,  // cases 56 - 63
// Pyramid
 1,  3,  3,  8,  3,  4,  8,  4, // cases 0 - 7
3,  8,  4,  4,  8,  4,  4,  2, // cases 8 - 15
2,  4,  4,  8,  4,  4,  8,  3, // cases 16 - 23
4,  8,  4,  3,  8,  3,  3,  1,  // cases 24 - 31
//}
// Poly 5
2,  3,  3,  3,  3,  4,  3,  3, // cases 0 - 7
3,  4,  4,  5,  3,  5,  3,  3, // cases 8 - 15
3,  3,  4,  3,  4,  5,  5,  3, // cases 16 - 23
3,  3,  5,  3,  3,  3,  3,  2,  // cases 24 - 31
// Poly 6
2,  4,  4,  3,  4,  5,  3,  4, // cases 0 - 7
4,  4,  5,  5,  3,  5,  4,  3, // cases 8 - 15
4,  5,  4,  5,  5,  7,  5,  5, // cases 16 - 23
3,  5,  5,  5,  4,  5,  3,  4, // cases 24 - 31
4,  3,  5,  4,  4,  5,  5,  3, // cases 32 - 39
5,  5,  7,  5,  5,  5,  5,  4, // cases 40 - 47
3,  4,  5,  3,  5,  5,  5,  4, // cases 48 - 55
4,  3,  5,  4,  3,  4,  4,  2,  // cases 56 - 63
// Poly 7
3,  4,  4,  4,  4,  5,  4,  4, // cases 0 - 7
4,  5,  5,  6,  4,  6,  4,  4, // cases 8 - 15
4,  5,  5,  5,  5,  7,  6,  5, // cases 16 - 23
4,  5,  6,  5,  4,  5,  4,  4, // cases 24 - 31
4,  5,  5,  6,  5,  7,  5,  5, // cases 32 - 39
5,  7,  7,  7,  6,  7,  5,  6, // cases 40 - 47
4,  6,  5,  5,  6,  7,  5,  6, // cases 48 - 55
4,  5,  5,  6,  4,  6,  4,  4, // cases 56 - 63
4,  4,  5,  4,  5,  6,  6,  4, // cases 64 - 71
5,  5,  7,  5,  5,  5,  5,  4, // cases 72 - 79
5,  6,  7,  5,  7,  7,  7,  6, // cases 80 - 87
6,  5,  7,  6,  5,  6,  6,  4, // cases 88 - 95
4,  4,  6,  4,  5,  5,  5,  4, // cases 96 - 103
6,  5,  7,  6,  5,  6,  6,  4, // cases 104 - 111
4,  4,  5,  4,  5,  6,  6,  4, // cases 112 - 119
4,  4,  6,  4,  4,  4,  4,  3,  // cases 120 - 127
// Poly 8
3,  5,  5,  4,  5,  6,  4,  5, // cases 0 - 7
5,  5,  6,  6,  4,  6,  5,  4, // cases 8 - 15
5,  6,  5,  6,  6,  8,  6,  6, // cases 16 - 23
4,  6,  6,  6,  5,  6,  4,  5, // cases 24 - 31
5,  5,  6,  6,  5,  7,  6,  5, // cases 32 - 39
6,  7,  8,  7,  6,  7,  6,  6, // cases 40 - 47
4,  6,  6,  5,  6,  7,  6,  6, // cases 48 - 55
5,  5,  6,  6,  4,  6,  5,  4, // cases 56 - 63
5,  6,  5,  6,  6,  8,  6,  6, // cases 64 - 71
5,  7,  7,  7,  6,  7,  5,  6, // cases 72 - 79
6,  8,  7,  7,  8,  9,  7,  8, // cases 80 - 87
6,  7,  7,  8,  6,  8,  6,  6, // cases 88 - 95
4,  6,  6,  6,  6,  7,  5,  6, // cases 96 - 103
6,  7,  7,  8,  6,  8,  6,  6, // cases 104 - 111
5,  6,  5,  6,  6,  8,  6,  6, // cases 112 - 119
4,  6,  6,  6,  5,  6,  4,  5, // cases 120 - 127
5,  4,  6,  5,  5,  6,  6,  4, // cases 128 - 135
6,  6,  8,  6,  6,  6,  6,  5, // cases 136 - 143
5,  6,  7,  5,  7,  7,  7,  6, // cases 144 - 151
6,  5,  7,  6,  5,  6,  6,  4, // cases 152 - 159
6,  6,  8,  6,  7,  7,  7,  6, // cases 160 - 167
8,  7,  9,  8,  7,  8,  8,  6, // cases 168 - 175
6,  6,  7,  6,  7,  8,  8,  6, // cases 176 - 183
6,  6,  8,  6,  6,  6,  6,  5, // cases 184 - 191
4,  5,  6,  4,  6,  6,  6,  5, // cases 192 - 199
6,  5,  7,  6,  5,  6,  6,  4, // cases 200 - 207
6,  6,  7,  6,  7,  8,  8,  6, // cases 208 - 215
6,  6,  8,  6,  6,  6,  6,  5, // cases 216 - 223
5,  4,  6,  5,  5,  6,  6,  4, // cases 224 - 231
6,  6,  8,  6,  6,  6,  6,  5, // cases 232 - 239
4,  5,  6,  4,  6,  6,  6,  5, // cases 240 - 247
5,  4,  6,  5,  4,  5,  5,  3  // cases 248 - 255
};

VISKORES_STATIC_CONSTEXPR_ARRAY viskores::UInt16 MIRTableIndicies[] = {
//{
0, 3, // vtx
6, 10, 18, 26,// line
30, 35, 46, 57, 68, 79, 90, 101, // Triangle
106, 112, 128, 144, 156, 172, 184, 206, 222, 238, 260, 272, 288, 300, 316, 332, // Pixel
338, 344, 360, 376, 388, 404, 426, 438, 454, 470, 482, 504, 520, 532, 548, 564, // Quad
570, 576, 590, 604, 620, 634, 650, 666, 680, 694, 710, 726, 740, 756, 770, 784, // Tet
// Voxel
790, 800, 870, 940, 966, 1036, 1062, 1177, 1278, 1348, 1463, 1489, 1590, 1616,
1717, 1818, 1838, 1908, 1934, 2049, 2150, 2265, 2366, 2442, 2510, 2562, 2681, 2800,
2914, 3033, 3147, 3255, 3356, 3426, 3541, 3567, 3668, 3720, 3839, 3958, 4072, 4187,
4263, 4364, 4432, 4551, 4659, 4773, 4874, 4900, 5001, 5102, 5122, 5241, 5355, 5463,
5564, 5683, 5791, 5905, 6006, 6042, 6133, 6224, 6250, 6320, 6435, 6487, 6606, 6632,
6733, 6852, 6966, 7081, 7157, 7276, 7384, 7485, 7553, 7667, 7768, 7794, 7895, 8014,
8128, 8229, 8249, 8357, 8458, 8577, 8685, 8721, 8812, 8926, 9027, 9118, 9144, 9259,
9335, 9454, 9562, 9681, 9789, 9825, 9916, 9992, 10054, 10162, 10216, 10324, 10378,
10469, 10516, 10617, 10685, 10799, 10900, 11014, 11115, 11206, 11232, 11340, 11394,
11485, 11532, 11623, 11670, 11722, 11792, 11862, 11914, 12029, 12148, 12263, 12382,
12458, 12566, 12592, 12711, 12812, 12926, 13027, 13141, 13209, 13310, 13425, 13544,
13620, 13728, 13804, 13912, 13974, 14028, 14147, 14183, 14291, 14382, 14490, 14581,
14635, 14682, 14708, 14827, 14928, 15042, 15161, 15197, 15305, 15396, 15497, 15605,
15625, 15726, 15840, 15931, 16032, 16058, 16159, 16273, 16341, 16442, 16550, 16641,
16695, 16742, 16856, 16947, 17048, 17074, 17165, 17217, 17264, 17334, 17360, 17479,
17598, 17634, 17735, 17849, 17957, 18048, 18149, 18257, 18371, 18462, 18482, 18583,
18684, 18710, 18811, 18925, 19033, 19124, 19192, 19293, 19347, 19394, 19508, 19599,
19690, 19742, 19843, 19869, 19916, 19986, 20087, 20195, 20309, 20400, 20514, 20605,
20696, 20748, 20816, 20870, 20971, 21018, 21119, 21166, 21192, 21262, 21282, 21383,
21484, 21510, 21611, 21637, 21684, 21754, 21855, 21902, 21928, 21998, 22024, 22094,
22164,
// Hex
22174, 22184, 22254, 22324, 22350, 22420, 22535, 22561, 22662, 22732, 22758, 22873,
22974, 23000, 23101, 23202, 23222, 23292, 23318, 23433, 23534, 23586, 23705, 23824,
23938, 24053, 24154, 24230, 24298, 24417, 24531, 24639, 24740, 24810, 24925, 24951,
25052, 25167, 25243, 25344, 25412, 25464, 25583, 25702, 25816, 25935, 26043, 26157,
26258, 26284, 26385, 26486, 26506, 26625, 26733, 26847, 26948, 27067, 27181, 27289,
27390, 27426, 27517, 27608, 27634, 27704, 27756, 27871, 27990, 28016, 28135, 28236,
28350, 28465, 28584, 28660, 28768, 28869, 28983, 29051, 29152, 29267, 29386, 29462,
29570, 29689, 29725, 29833, 29924, 30000, 30108, 30170, 30224, 30332, 30423, 30477,
30524, 30550, 30669, 30770, 30884, 30985, 31093, 31113, 31214, 31333, 31369, 31477,
31568, 31682, 31773, 31874, 31900, 32001, 32115, 32183, 32284, 32398, 32489, 32590,
32616, 32724, 32815, 32869, 32916, 33007, 33059, 33106, 33176, 33246, 33361, 33413,
33532, 33647, 33723, 33842, 33950, 33976, 34077, 34196, 34310, 34411, 34479, 34593,
34694, 34720, 34821, 34940, 35054, 35173, 35281, 35317, 35408, 35509, 35529, 35637,
35738, 35852, 35953, 36044, 36070, 36185, 36261, 36380, 36488, 36564, 36626, 36734,
36788, 36907, 37015, 37051, 37142, 37250, 37304, 37395, 37442, 37543, 37611, 37725,
37826, 37934, 37988, 38079, 38126, 38240, 38341, 38432, 38458, 38549, 38596, 38648,
38718, 38744, 38863, 38982, 39018, 39119, 39227, 39341, 39432, 39533, 39647, 39755,
39846, 39866, 39967, 40068, 40094, 40195, 40309, 40417, 40508, 40622, 40713, 40804,
40856, 40924, 41025, 41079, 41126, 41227, 41253, 41300, 41370, 41471, 41579, 41693,
41784, 41852, 41906, 42007, 42054, 42168, 42259, 42350, 42402, 42503, 42550, 42576,
42646, 42666, 42767, 42868, 42894, 42995, 43042, 43068, 43138, 43239, 43265, 43312,
43382, 43408, 43478, 43548,
//Wedge
43558, 43566, 43587, 43608, 43673, 43694, 43759, 43824, 43840, 43861, 43879, 43979,
44066, 44166, 44253, 44326, 44391, 44412, 44512, 44530, 44617, 44717, 44790, 44877,
44942, 45007, 45094, 45181, 45199, 45272, 45306, 45340, 45361, 45382, 45482, 45582,
45655, 45673, 45760, 45847, 45912, 45977, 46064, 46137, 46171, 46258, 46276, 46310,
46331, 46396, 46469, 46556, 46590, 46677, 46711, 46729, 46750, 46766, 46831, 46896,
46917, 46982, 47003, 47024,
//Pyramid
47032, 47039, 47059, 47079, 47138, 47158, 47190, 47249, 47279, 47299, 47358, 47390,
47420, 47479, 47509, 47539, 47556, 47573, 47603, 47633, 47692, 47722, 47750, 47809,
47829, 47859, 47918, 47946, 47966, 48025, 48045, 48065,
//}

// Poly 5
48072,48083,48100,48117,48134,48151,48174,48191, // cases 0 to 7
48208,48225,48248,48271,48298,48315,48342,48359, // cases 8 to 15
48376,48393,48410,48433,48450,48473,48500,48527, // cases 16 to 23
48544,48561,48578,48605,48622,48639,48656,48673, // cases 24 to 31
// Poly 6
48684,48696,48718,48740,48758,48780,48808,48826, // cases 0 to 7
48848,48870,48894,48922,48950,48968,48996,49018, // cases 8 to 15
49036,49058,49086,49110,49138,49166,49204,49232, // cases 16 to 23
49260,49278,49306,49334,49362,49384,49412,49430, // cases 24 to 31
49452,49474,49492,49520,49542,49566,49594,49622, // cases 32 to 39
49640,49668,49696,49734,49762,49790,49818,49846, // cases 40 to 47
49868,49886,49908,49936,49954,49982,50010,50038, // cases 48 to 55
50060,50082,50100,50128,50150,50168,50190,50212, // cases 56 to 63
// Poly 7
50224,50241,50264,50287,50310,50333,50362,50385, // cases 0 to 7
50408,50431,50460,50489,50522,50545,50578,50601, // cases 8 to 15
50624,50647,50676,50705,50734,50763,50802,50835, // cases 16 to 23
50864,50887,50916,50949,50978,51001,51030,51053, // cases 24 to 31
51076,51099,51128,51157,51190,51219,51258,51287, // cases 32 to 39
51316,51345,51384,51423,51462,51495,51534,51563, // cases 40 to 47
51596,51619,51652,51681,51710,51743,51782,51811, // cases 48 to 55
51844,51867,51896,51925,51958,51981,52014,52037, // cases 56 to 63
52060,52083,52106,52135,52158,52187,52220,52253, // cases 64 to 71
52276,52305,52334,52373,52402,52431,52460,52489, // cases 72 to 79
52512,52541,52574,52613,52642,52681,52720,52759, // cases 80 to 87
52792,52825,52854,52893,52926,52955,52988,53021, // cases 88 to 95
53044,53067,53090,53123,53146,53175,53204,53233, // cases 96 to 103
53256,53289,53318,53357,53390,53419,53452,53485, // cases 104 to 111
53508,53531,53554,53583,53606,53635,53668,53701, // cases 112 to 119
53724,53747,53770,53803,53826,53849,53872,53895, // cases 120 to 127
// Poly 8
53912,53930,53958,53986,54010,54038,54072,54096, // cases 0 to 7
54124,54152,54182,54216,54250,54274,54308,54336, // cases 8 to 15
54360,54388,54422,54452,54486,54520,54564,54598, // cases 16 to 23
54632,54656,54690,54724,54758,54786,54820,54844, // cases 24 to 31
54872,54900,54930,54964,54998,55028,55068,55102, // cases 32 to 39
55132,55166,55206,55250,55290,55324,55364,55398, // cases 40 to 47
55432,55456,55490,55524,55554,55588,55628,55662, // cases 48 to 55
55696,55724,55754,55788,55822,55846,55880,55908, // cases 56 to 63
55932,55960,55994,56024,56058,56092,56136,56170, // cases 64 to 71
56204,56234,56274,56314,56354,56388,56428,56458, // cases 72 to 79
56492,56526,56570,56610,56650,56694,56744,56784, // cases 80 to 87
56828,56862,56902,56942,56986,57020,57064,57098, // cases 88 to 95
57132,57156,57190,57224,57258,57292,57332,57362, // cases 96 to 103
57396,57430,57470,57510,57554,57588,57632,57666, // cases 104 to 111
57700,57728,57762,57792,57826,57860,57904,57938, // cases 112 to 119
57972,57996,58030,58064,58098,58126,58160,58184, // cases 120 to 127
58212,58240,58264,58298,58326,58356,58390,58424, // cases 128 to 135
58448,58482,58516,58560,58594,58628,58662,58696, // cases 136 to 143
58724,58754,58788,58828,58858,58898,58938,58978, // cases 144 to 151
59012,59046,59076,59116,59150,59180,59214,59248, // cases 152 to 159
59272,59306,59340,59384,59418,59458,59498,59538, // cases 160 to 167
59572,59616,59656,59706,59750,59790,59834,59878, // cases 168 to 175
59912,59946,59980,60020,60054,60094,60138,60182, // cases 176 to 183
60216,60250,60284,60328,60362,60396,60430,60464, // cases 184 to 191
60492,60516,60544,60578,60602,60636,60670,60704, // cases 192 to 199
60732,60766,60796,60836,60870,60900,60934,60968, // cases 200 to 207
60992,61026,61060,61100,61134,61174,61218,61262, // cases 208 to 215
61296,61330,61364,61408,61442,61476,61510,61544, // cases 216 to 223
61572,61600,61624,61658,61686,61716,61750,61784, // cases 224 to 231
61808,61842,61876,61920,61954,61988,62022,62056, // cases 232 to 239
62084,62108,62136,62170,62194,62228,62262,62296, // cases 240 to 247
62324,62352,62376,62410,62438,62462,62490,62518  // cases 248 to 255
};

// Note, comment blocks using //{ and //} are included below to separate all cases. Included are the case IDs
VISKORES_STATIC_CONSTEXPR_ARRAY viskores::UInt8 MIRInstructions[] = {
//{ Vertex
// Case #0: Unique case #1
SH_VTX, COLOR0, P0,
// Case #1: Unique case #2
SH_VTX, COLOR1, P0,
//}
//{Line
// Case #0: Unique case #1
SH_LIN, COLOR0, P0, P1,
// Case #1: Unique case #2
SH_LIN, COLOR0, EA, P1,
SH_LIN, COLOR1, P0, EA,
// Case #2: (cloned #1)
SH_LIN, COLOR0, EA, P0,
SH_LIN, COLOR1, P1, EA,
// Case #3: Unique case #3
SH_LIN, COLOR1, P0, P1,
//}
//{Triangle
// Case #0: Unique case #1
SH_TRI, COLOR0, P0, P1, P2,
// Case #1: Unique case #2
SH_QUA, COLOR0, P1, P2, EC, EA,
SH_TRI, COLOR1, P0, EA, EC,
// Case #2: (cloned #1)
SH_QUA, COLOR0, P2, P0, EA, EB,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_TRI, COLOR0, EC, EB, P2,
SH_QUA, COLOR1, P0, P1, EB, EC,
// Case #4: (cloned #1)
SH_QUA, COLOR0, P0, P1, EB, EC,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: (cloned #3)
SH_TRI, COLOR0, EB, EA, P1,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #6: (cloned #3)
SH_TRI, COLOR0, EA, EC, P0,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #4
SH_TRI, COLOR1, P0, P1, P2,
//}
//{ Pixel
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P3, P2,
// Case #1: Unique case #2
SH_QUA, COLOR0, ED, EA, P1, P2,
SH_TRI, COLOR0, P2, P1, P3,
SH_TRI, COLOR1, P0, EA, ED,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EA, EB, P3, P0,
SH_TRI, COLOR0, P0, P3, P2,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, ED, EB, P3, P2,
SH_QUA, COLOR1, P0, P1, EB, ED,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EC, ED, P0, P3,
SH_TRI, COLOR0, P3, P0, P1,
SH_TRI, COLOR1, P2, ED, EC,
// Case #5: (cloned #3)
SH_QUA, COLOR0, EC, EA, P1, P3,
SH_QUA, COLOR1, P2, P0, EA, EC,
// Case #6: Unique case #4
SH_TRI, COLOR0, EA, ED, P0,
SH_TRI, COLOR0, EC, EB, P3,
SH_QUA, COLOR1, P2, P1, EB, EC,
SH_QUA, COLOR1, P1, P2, ED, EA,
// Case #7: Unique case #5
SH_TRI, COLOR0, EC, EB, P3,
SH_QUA, COLOR1, P2, P1, EB, EC,
SH_TRI, COLOR1, P0, P1, P2,
// Case #8: (cloned #1)
SH_QUA, COLOR0, EB, EC, P2, P1,
SH_TRI, COLOR0, P1, P2, P0,
SH_TRI, COLOR1, P3, EC, EB,
// Case #9: (cloned #6)
SH_TRI, COLOR0, EB, EA, P1,
SH_TRI, COLOR0, ED, EC, P2,
SH_QUA, COLOR1, P0, P3, EC, ED,
SH_QUA, COLOR1, P3, P0, EA, EB,
// Case #10: (cloned #3)
SH_QUA, COLOR0, EA, EC, P2, P0,
SH_QUA, COLOR1, P1, P3, EC, EA,
// Case #11: (cloned #7)
SH_TRI, COLOR0, ED, EC, P2,
SH_QUA, COLOR1, P0, P3, EC, ED,
SH_TRI, COLOR1, P1, P3, P0,
// Case #12: (cloned #3)
SH_QUA, COLOR0, EB, ED, P0, P1,
SH_QUA, COLOR1, P3, P2, ED, EB,
// Case #13: (cloned #7)
SH_TRI, COLOR0, EB, EA, P1,
SH_QUA, COLOR1, P3, P0, EA, EB,
SH_TRI, COLOR1, P2, P0, P3,
// Case #14: (cloned #7)
SH_TRI, COLOR0, EA, ED, P0,
SH_QUA, COLOR1, P1, P2, ED, EA,
SH_TRI, COLOR1, P3, P2, P1,
// Case #15: Unique case #6
SH_QUA, COLOR1, P0, P1, P3, P2,
//}
//{ Quad
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P2, P3,
// Case #1: Unique case #2
SH_QUA, COLOR0, ED, EA, P1, P3,
SH_TRI, COLOR0, P3, P1, P2,
SH_TRI, COLOR1, P0, EA, ED,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EA, EB, P2, P0,
SH_TRI, COLOR0, P0, P2, P3,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, ED, EB, P2, P3,
SH_QUA, COLOR1, P0, P1, EB, ED,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EB, EC, P3, P1,
SH_TRI, COLOR0, P1, P3, P0,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: Unique case #4
SH_TRI, COLOR0, ED, EC, P3,
SH_TRI, COLOR0, EB, EA, P1,
SH_QUA, COLOR1, P2, P0, EA, EB,
SH_QUA, COLOR1, P0, P2, EC, ED,
// Case #6: (cloned #3)
SH_QUA, COLOR0, EA, EC, P3, P0,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #5
SH_TRI, COLOR0, ED, EC, P3,
SH_QUA, COLOR1, P0, P2, EC, ED,
SH_TRI, COLOR1, P1, P2, P0,
// Case #8: (cloned #1)
SH_QUA, COLOR0, EC, ED, P0, P2,
SH_TRI, COLOR0, P2, P0, P1,
SH_TRI, COLOR1, P3, ED, EC,
// Case #9: (cloned #3)
SH_QUA, COLOR0, EC, EA, P1, P2,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #10: (cloned #5)
SH_TRI, COLOR0, EA, ED, P0,
SH_TRI, COLOR0, EC, EB, P2,
SH_QUA, COLOR1, P3, P1, EB, EC,
SH_QUA, COLOR1, P1, P3, ED, EA,
// Case #11: (cloned #7)
SH_TRI, COLOR0, EC, EB, P2,
SH_QUA, COLOR1, P3, P1, EB, EC,
SH_TRI, COLOR1, P0, P1, P3,
// Case #12: (cloned #3)
SH_QUA, COLOR0, EB, ED, P0, P1,
SH_QUA, COLOR1, P2, P3, ED, EB,
// Case #13: (cloned #7)
SH_TRI, COLOR0, EB, EA, P1,
SH_QUA, COLOR1, P2, P0, EA, EB,
SH_TRI, COLOR1, P3, P0, P2,
// Case #14: (cloned #7)
SH_TRI, COLOR0, EA, ED, P0,
SH_QUA, COLOR1, P1, P3, ED, EA,
SH_TRI, COLOR1, P2, P3, P1,
// Case #15: Unique case #6
SH_QUA, COLOR1, P0, P1, P2, P3,
//}
//{ Tet
// Case #0: Unique case #1
SH_TET, COLOR0, P0, P1, P2, P3,
// Case #1: Unique case #2
SH_WDG, COLOR0, EA, ED, EC, P1, P3, P2,
SH_TET, COLOR1, P0, EA, EC, ED,
// Case #2: (cloned #1)
SH_WDG, COLOR0, P0, P3, P2, EA, EE, EB,
SH_TET, COLOR1, P1, EB, EA, EE,
// Case #3: Unique case #3
SH_WDG, COLOR0, P3, ED, EE, P2, EC, EB,
SH_WDG, COLOR1, P0, ED, EC, P1, EE, EB,
// Case #4: (cloned #1)
SH_WDG, COLOR0, EC, EF, EB, P0, P3, P1,
SH_TET, COLOR1, P2, EC, EB, EF,
// Case #5: (cloned #3)
SH_WDG, COLOR0, P1, EA, EB, P3, ED, EF,
SH_WDG, COLOR1, P2, EF, EB, P0, ED, EA,
// Case #6: (cloned #3)
SH_WDG, COLOR0, P3, EE, EF, P0, EA, EC,
SH_WDG, COLOR1, P1, EE, EA, P2, EF, EC,
// Case #7: Unique case #4
SH_TET, COLOR0, ED, EE, EF, P3,
SH_WDG, COLOR1, ED, EE, EF, P0, P1, P2,
// Case #8: (cloned #1)
SH_WDG, COLOR0, P0, P2, P1, ED, EF, EE,
SH_TET, COLOR1, P3, EE, ED, EF,
// Case #9: (cloned #3)
SH_WDG, COLOR0, P2, EC, EF, P1, EA, EE,
SH_WDG, COLOR1, P0, EC, EA, P3, EF, EE,
// Case #10: (cloned #3)
SH_WDG, COLOR0, P0, EA, ED, P2, EB, EF,
SH_WDG, COLOR1, P3, EF, ED, P1, EB, EA,
// Case #11: (cloned #7)
SH_TET, COLOR0, EC, EF, EB, P2,
SH_WDG, COLOR1, P0, P1, P3, EC, EB, EF,
// Case #12: (cloned #3)
SH_WDG, COLOR0, P1, EB, EE, P0, EC, ED,
SH_WDG, COLOR1, P2, EB, EC, P3, EE, ED,
// Case #13: (cloned #7)
SH_TET, COLOR0, EA, EB, EE, P1,
SH_WDG, COLOR1, EA, EB, EE, P0, P2, P3,
// Case #14: (cloned #7)
SH_TET, COLOR0, EA, ED, EC, P0,
SH_WDG, COLOR1, P1, P2, P3, EA, EC, ED,
// Case #15: Unique case #5
SH_TET, COLOR1, P0, P1, P2, P3,
//}
//{ Voxel
// Case #0: Unique case #1
SH_HEX, COLOR0, P0, P1, P3, P2, P4, P5, P7, P6,
// Case #1: Unique case #2
SH_PNT, 0, COLOR0, 7, P1, P3, P2, P4, P5, P7, P6,
SH_WDG, COLOR0, P1, P2, P4, EA, ED, EI,
SH_TET, COLOR0, P1, P2, P4, N0,
SH_TET, COLOR0, P1, P3, P2, N0,
SH_PYR, COLOR0, P7, P6, P2, P3, N0,
SH_PYR, COLOR0, P5, P7, P3, P1, N0,
SH_PYR, COLOR0, P4, P6, P7, P5, N0,
SH_TET, COLOR0, P2, P6, P4, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR1, P0, EA, ED, EI,
// Case #2: (cloned #1)
SH_PNT, 0, COLOR0, 7, P5, P4, P0, P3, P7, P6, P2,
SH_WDG, COLOR0, EJ, EA, EB, P5, P0, P3,
SH_TET, COLOR0, P5, P3, P0, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P6, P4, P0, P2, N0,
SH_PYR, COLOR0, P7, P5, P4, P6, N0,
SH_PYR, COLOR0, P3, P7, P6, P2, N0,
SH_TET, COLOR0, P0, P3, P2, N0,
SH_TET, COLOR0, P3, P5, P7, N0,
SH_TET, COLOR1, P1, EA, EJ, EB,
// Case #3: Unique case #3
SH_HEX, COLOR0, EB, P3, P2, ED, EJ, P5, P4, EI,
SH_WDG, COLOR0, P3, P7, P5, P2, P6, P4,
SH_WDG, COLOR1, P1, EB, EJ, P0, ED, EI,
// Case #4: (cloned #1)
SH_PNT, 0, COLOR0, 7, P3, P1, P0, P6, P7, P5, P4,
SH_WDG, COLOR0, EC, ED, EK, P3, P0, P6,
SH_TET, COLOR0, P3, P6, P0, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_PYR, COLOR0, P5, P1, P0, P4, N0,
SH_PYR, COLOR0, P7, P3, P1, P5, N0,
SH_PYR, COLOR0, P6, P7, P5, P4, N0,
SH_TET, COLOR0, P0, P6, P4, N0,
SH_TET, COLOR0, P6, P3, P7, N0,
SH_TET, COLOR1, P2, ED, EC, EK,
// Case #5: (cloned #3)
SH_HEX, COLOR0, EK, P6, P4, EI, EC, P3, P1, EA,
SH_WDG, COLOR0, P6, P7, P3, P4, P5, P1,
SH_WDG, COLOR1, P2, EK, EC, P0, EI, EA,
// Case #6: Unique case #4
SH_PNT, 0, NOCOLOR, 2, EK, EJ,
SH_PYR, COLOR0, P6, P7, P5, P4, N0,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_TET, COLOR0, P6, P3, P7, N0,
SH_TET, COLOR0, P0, P6, P4, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_PYR, COLOR0, P3, EB, EJ, P5, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_PYR, COLOR0, P6, EK, EC, P3, N0,
SH_PYR, COLOR0, P6, P0, ED, EK, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR1, P2, ED, EA, P1, N0,
SH_PYR, COLOR1, EC, P2, P1, EB, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EC, P2, N0, EK,
SH_TET, COLOR1, ED, P2, EK, N0,
// Case #7: Unique case #5
SH_PNT, 0, NOCOLOR, 5, EJ, EI, EB, EC, EK,
SH_PYR, COLOR0, P5, P4, P6, P7, N0,
SH_TET, COLOR0, P6, N0, P3, P7,
SH_PYR, COLOR0, P4, EI, EK, P6, N0,
SH_PYR, COLOR0, EJ, EI, P4, P5, N0,
SH_TET, COLOR0, P3, P5, P7, N0,
SH_PYR, COLOR0, P3, EB, EJ, P5, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_PYR, COLOR0, EK, EC, P3, P6, N0,
SH_PYR, COLOR1, EI, P0, P2, EK, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_PYR, COLOR1, EB, EC, P2, P1, N0,
SH_TET, COLOR1, P2, EC, EK, N0,
// Case #8: (cloned #1)
SH_PNT, 0, COLOR0, 7, P7, P5, P1, P2, P6, P4, P0,
SH_WDG, COLOR0, EL, EB, EC, P7, P1, P2,
SH_TET, COLOR0, P7, P2, P1, N0,
SH_TET, COLOR0, P7, P1, P5, N0,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_PYR, COLOR0, P6, P7, P5, P4, N0,
SH_PYR, COLOR0, P2, P6, P4, P0, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_TET, COLOR0, P2, P7, P6, N0,
SH_TET, COLOR1, P3, EB, EL, EC,
// Case #9: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EI, EL,
SH_PYR, COLOR0, P4, P6, P7, P5, N0,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR0, P2, P6, P4, N0,
SH_TET, COLOR0, P7, P6, P2, N0,
SH_PYR, COLOR0, P7, P2, EC, EL, N0,
SH_PYR, COLOR0, P1, P7, EL, EB, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_PYR, COLOR0, P4, EI, ED, P2, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_PYR, COLOR1, P0, P3, EC, ED, N0,
SH_PYR, COLOR1, EA, EB, P3, P0, N0,
SH_TET, COLOR1, EB, EL, P3, N0,
SH_TET, COLOR1, P3, EL, EC, N0,
SH_TET, COLOR1, EA, N0, P0, EI,
SH_TET, COLOR1, ED, EI, P0, N0,
// Case #10: (cloned #3)
SH_HEX, COLOR0, EC, P2, P0, EA, EL, P7, P5, EJ,
SH_WDG, COLOR0, P2, P6, P7, P0, P4, P5,
SH_WDG, COLOR1, P3, EC, EL, P1, EA, EJ,
// Case #11: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EC, ED, EI,
SH_PYR, COLOR0, P7, P5, P4, P6, N0,
SH_TET, COLOR0, P4, N0, P2, P6,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_PYR, COLOR0, EL, EJ, P5, P7, N0,
SH_TET, COLOR0, P2, P7, P6, N0,
SH_PYR, COLOR0, P2, EC, EL, P7, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_PYR, COLOR0, EI, ED, P2, P4, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_TET, COLOR1, EC, P3, EL, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_PYR, COLOR1, EC, ED, P0, P3, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
// Case #12: (cloned #3)
SH_HEX, COLOR0, EL, P7, P6, EK, EB, P1, P0, ED,
SH_WDG, COLOR0, P0, P4, P6, P1, P5, P7,
SH_WDG, COLOR1, P2, ED, EK, P3, EB, EL,
// Case #13: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EB, EA, EI,
SH_PYR, COLOR0, P7, P5, P4, P6, N0,
SH_TET, COLOR0, P4, P1, N0, P5,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_PYR, COLOR0, EL, P7, P6, EK, N0,
SH_TET, COLOR0, P1, P5, P7, N0,
SH_PYR, COLOR0, P1, P7, EL, EB, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, EI, P4, P1, EA, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_TET, COLOR1, EB, EL, P3, N0,
SH_TET, COLOR1, P3, P2, P0, N0,
SH_PYR, COLOR1, EB, P3, P0, EA, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
// Case #14: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EK, EL, ED, EA, EJ,
SH_PYR, COLOR0, P6, P7, P5, P4, N0,
SH_TET, COLOR0, P5, N0, P0, P4,
SH_PYR, COLOR0, P7, EL, EJ, P5, N0,
SH_PYR, COLOR0, EK, EL, P7, P6, N0,
SH_TET, COLOR0, P0, P6, P4, N0,
SH_PYR, COLOR0, P0, ED, EK, P6, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, EJ, EA, P0, P5, N0,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_PYR, COLOR1, EK, P2, P3, EL, N0,
SH_TET, COLOR1, ED, P2, EK, N0,
SH_TET, COLOR1, P2, P1, P3, N0,
SH_PYR, COLOR1, ED, EA, P1, P2, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
// Case #15: Unique case #6
SH_HEX, COLOR0, EI, EJ, EL, EK, P4, P5, P7, P6,
SH_HEX, COLOR1, P0, P1, P3, P2, EI, EJ, EL, EK,
// Case #16: (cloned #1)
SH_PNT, 0, COLOR0, 7, P5, P1, P0, P6, P7, P3, P2,
SH_WDG, COLOR0, P5, P0, P6, EE, EI, EH,
SH_TET, COLOR0, P5, P0, P6, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P3, P2, P0, P1, N0,
SH_PYR, COLOR0, P7, P3, P1, P5, N0,
SH_PYR, COLOR0, P6, P2, P3, P7, N0,
SH_TET, COLOR0, P0, P2, P6, N0,
SH_TET, COLOR0, P6, P7, P5, N0,
SH_TET, COLOR1, P4, EE, EI, EH,
// Case #17: (cloned #3)
SH_HEX, COLOR0, EE, P5, P1, EA, EH, P6, P2, ED,
SH_WDG, COLOR0, P2, P3, P1, P6, P7, P5,
SH_WDG, COLOR1, P0, ED, EA, P4, EH, EE,
// Case #18: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EH, EB,
SH_PYR, COLOR0, P6, P2, P3, P7, N0,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_TET, COLOR0, P6, P7, P5, N0,
SH_TET, COLOR0, P0, P2, P6, N0,
SH_TET, COLOR0, P3, P2, P0, N0,
SH_PYR, COLOR0, P3, P0, EA, EB, N0,
SH_PYR, COLOR0, P5, P3, EB, EJ, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_PYR, COLOR0, P6, P5, EE, EH, N0,
SH_PYR, COLOR0, P6, EH, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_TET, COLOR1, EE, N0, P4, EH,
SH_TET, COLOR1, EI, EH, P4, N0,
// Case #19: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EB, ED, EJ, EE, EH,
SH_PYR, COLOR0, P3, P7, P6, P2, N0,
SH_TET, COLOR0, P6, P5, N0, P7,
SH_PYR, COLOR0, P2, P6, EH, ED, N0,
SH_PYR, COLOR0, EB, P3, P2, ED, N0,
SH_TET, COLOR0, P5, P7, P3, N0,
SH_PYR, COLOR0, P5, P3, EB, EJ, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_PYR, COLOR0, EH, P6, P5, EE, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
// Case #20: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EC, EE,
SH_PYR, COLOR0, P3, P1, P5, P7, N0,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_TET, COLOR0, P3, P7, P6, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_PYR, COLOR0, P6, P5, EE, EH, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_PYR, COLOR0, P3, P6, EK, EC, N0,
SH_PYR, COLOR0, P3, EC, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR1, P2, P4, EI, ED, N0,
SH_PYR, COLOR1, EK, EH, P4, P2, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EK, N0, P2, EC,
SH_TET, COLOR1, ED, EC, P2, N0,
// Case #21: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EE, EA, EH, EK, EC,
SH_PYR, COLOR0, P5, P7, P3, P1, N0,
SH_TET, COLOR0, P3, P6, N0, P7,
SH_PYR, COLOR0, P1, P3, EC, EA, N0,
SH_PYR, COLOR0, EE, P5, P1, EA, N0,
SH_TET, COLOR0, P6, P7, P5, N0,
SH_PYR, COLOR0, P6, P5, EE, EH, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_PYR, COLOR0, EC, P3, P6, EK, N0,
SH_PYR, COLOR1, EA, EC, P2, P0, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_PYR, COLOR1, EH, P4, P2, EK, N0,
SH_TET, COLOR1, P2, EC, EK, N0,
// Case #22: Unique case #7
SH_TET, COLOR0, P0, EA, ED, EI,
SH_TET, COLOR0, P5, P6, P7, P3,
SH_PYR, COLOR0, EC, P3, P6, EK, EH,
SH_PYR, COLOR0, EB, EJ, P5, P3, EE,
SH_PYR, COLOR0, P6, P5, EE, EH, P3,
SH_PYR, COLOR0, EH, EE, EB, EC, P3,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_PYR, COLOR1, P2, EK, EH, P4, EC,
SH_PYR, COLOR1, EE, EJ, P1, P4, EB,
SH_PYR, COLOR1, EC, P2, P1, EB, P4,
SH_PYR, COLOR1, EC, EB, EE, EH, P4,
// Case #23: Unique case #8
SH_TET, COLOR0, P5, P6, P7, P3,
SH_PYR, COLOR0, EC, P3, P6, EK, EH,
SH_PYR, COLOR0, EB, EJ, P5, P3, EE,
SH_PYR, COLOR0, P6, P5, EE, EH, P3,
SH_PYR, COLOR0, EH, EE, EB, EC, P3,
SH_TET, COLOR1, P0, P1, P2, P4,
SH_PYR, COLOR1, EH, P4, P2, EK, EC,
SH_PYR, COLOR1, EE, EJ, P1, P4, EB,
SH_PYR, COLOR1, P2, P1, EB, EC, P4,
SH_PYR, COLOR1, EH, EC, EB, EE, P4,
// Case #24: Unique case #9
SH_WDG, COLOR0, EB, EC, EL, P1, P2, P7,
SH_WDG, COLOR0, P0, P6, P5, EI, EH, EE,
SH_TET, COLOR0, P2, P1, P7, P6,
SH_TET, COLOR0, P5, P6, P7, P1,
SH_TET, COLOR0, P0, P5, P1, P6,
SH_TET, COLOR0, P2, P6, P0, P1,
SH_TET, COLOR1, P4, EE, EI, EH,
SH_TET, COLOR1, P3, EC, EB, EL,
// Case #25: Unique case #10
SH_PNT, 0, NOCOLOR, 4, EE, EH, EL, EL,
SH_PYR, COLOR0, P7, P2, EC, EL, N0,
SH_TET, COLOR0, EC, P2, ED, N0,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_TET, COLOR0, P7, P6, P2, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_PYR, COLOR0, P1, P7, EL, EB, N0,
SH_TET, COLOR0, P5, P6, P7, N0,
SH_PYR, COLOR0, P5, EE, EH, P6, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_PYR, COLOR1, P3, EC, ED, P0, N0,
SH_PYR, COLOR1, EA, EB, P3, P0, N0,
SH_TET, COLOR1, P3, EL, EC, N0,
SH_TET, COLOR1, EB, EL, P3, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_PYR, COLOR1, P0, P4, EE, EA, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
// Case #26: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EL, EC, EH, EH,
SH_PYR, COLOR0, P6, EH, EI, P0, N0,
SH_TET, COLOR0, EI, EA, P0, N0,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_TET, COLOR0, P6, P0, P2, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_PYR, COLOR0, P5, EE, EH, P6, N0,
SH_TET, COLOR0, P7, P6, P2, N0,
SH_PYR, COLOR0, P7, P2, EC, EL, N0,
SH_PYR, COLOR0, P7, EL, EJ, P5, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
SH_TET, COLOR1, P4, EI, EH, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
SH_PYR, COLOR1, EA, P1, P3, EC, N0,
SH_PYR, COLOR1, P1, EJ, EL, P3, N0,
SH_TET, COLOR1, EL, EC, P3, N0,
// Case #27: Unique case #11
SH_PNT, 0, NOCOLOR, 6, ED, EC, EL, EJ, EE, EH,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_PYR, COLOR0, P7, EL, EJ, P5, N0,
SH_TET, COLOR0, P2, P7, P6, N0,
SH_PYR, COLOR0, P2, EC, EL, P7, N0,
SH_TET, COLOR0, ED, EC, P2, N0,
SH_PYR, COLOR0, P2, P6, EH, ED, N0,
SH_PYR, COLOR0, EH, P6, P5, EE, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, ED, P0, P3, EC, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_TET, COLOR1, P4, P1, P0, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_PYR, COLOR1, EJ, EL, P3, P1, N0,
SH_TET, COLOR1, EL, EC, P3, N0,
// Case #28: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EL, EB, EE, EE,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_TET, COLOR0, EI, P0, ED, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_PYR, COLOR0, P6, P5, EE, EH, N0,
SH_TET, COLOR0, P7, P1, P5, N0,
SH_PYR, COLOR0, P7, EL, EB, P1, N0,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_PYR, COLOR1, P4, EI, ED, P2, N0,
SH_PYR, COLOR1, EK, EH, P4, P2, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_PYR, COLOR1, ED, EB, P3, P2, N0,
SH_PYR, COLOR1, P2, P3, EL, EK, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
// Case #29: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EA, EB, EL, EK, EH, EE,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_TET, COLOR0, P1, P5, P7, N0,
SH_PYR, COLOR0, P1, P7, EL, EB, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_PYR, COLOR0, EE, EH, P6, P5, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR1, P0, P3, P2, N0,
SH_PYR, COLOR1, EA, EB, P3, P0, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
SH_PYR, COLOR1, P4, P2, EK, EH, N0,
SH_PYR, COLOR1, EK, P2, P3, EL, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
// Case #30: Unique case #12
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EK, EH, EE,
SH_TET, COLOR0, P0, EA, ED, EI,
SH_PYR, COLOR0, P5, P7, EL, EJ, N0,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR0, P7, P5, P6, N0,
SH_PYR, COLOR0, P6, P5, EE, EH, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_TET, COLOR1, P1, P2, P4, N0,
SH_PYR, COLOR1, P2, EK, EH, P4, N0,
SH_TET, COLOR1, P3, P2, P1, N0,
SH_PYR, COLOR1, EJ, EL, P3, P1, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
// Case #31: Unique case #13
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EK, EE, EH,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_PYR, COLOR0, P5, P7, EL, EJ, N0,
SH_TET, COLOR0, EE, P5, EJ, N0,
SH_PYR, COLOR0, EH, P6, P5, EE, N0,
SH_TET, COLOR0, P6, P7, P5, N0,
SH_PYR, COLOR1, P0, P1, P3, P2, N0,
SH_TET, COLOR1, P2, P4, P0, N0,
SH_TET, COLOR1, P4, P1, P0, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_PYR, COLOR1, EJ, EL, P3, P1, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_PYR, COLOR1, EK, EH, P4, P2, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
// Case #32: (cloned #1)
SH_PNT, 0, COLOR0, 7, P7, P3, P1, P4, P6, P2, P0,
SH_WDG, COLOR0, P7, P1, P4, EF, EJ, EE,
SH_TET, COLOR0, P7, P1, P4, N0,
SH_TET, COLOR0, P7, P3, P1, N0,
SH_PYR, COLOR0, P2, P0, P1, P3, N0,
SH_PYR, COLOR0, P6, P2, P3, P7, N0,
SH_PYR, COLOR0, P4, P0, P2, P6, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_TET, COLOR0, P4, P6, P7, N0,
SH_TET, COLOR1, P5, EF, EJ, EE,
// Case #33: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, ED, EF,
SH_PYR, COLOR0, P2, P3, P7, P6, N0,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_TET, COLOR0, P2, P1, P3, N0,
SH_TET, COLOR0, P4, P2, P6, N0,
SH_TET, COLOR0, P7, P4, P6, N0,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_PYR, COLOR0, P1, EJ, EF, P7, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, P2, ED, EA, P1, N0,
SH_PYR, COLOR0, P2, P4, EI, ED, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_TET, COLOR1, EA, P0, N0, ED,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #34: (cloned #3)
SH_HEX, COLOR0, EF, P7, P3, EB, EE, P4, P0, EA,
SH_WDG, COLOR0, P0, P2, P3, P4, P6, P7,
SH_WDG, COLOR1, P1, EA, EB, P5, EE, EF,
// Case #35: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EE, EI, ED,
SH_PYR, COLOR0, P7, P6, P2, P3, N0,
SH_TET, COLOR0, P2, P4, N0, P6,
SH_PYR, COLOR0, P3, P2, ED, EB, N0,
SH_PYR, COLOR0, EF, P7, P3, EB, N0,
SH_TET, COLOR0, P4, P6, P7, N0,
SH_PYR, COLOR0, P4, P7, EF, EE, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR0, ED, P2, P4, EI, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_TET, COLOR1, EE, EF, P5, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_PYR, COLOR1, EE, P5, P0, EI, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
// Case #36: (cloned #24)
SH_WDG, COLOR0, P1, P4, P7, EJ, EE, EF,
SH_WDG, COLOR0, ED, EK, EC, P0, P6, P3,
SH_TET, COLOR0, P4, P7, P1, P6,
SH_TET, COLOR0, P3, P7, P6, P1,
SH_TET, COLOR0, P0, P1, P3, P6,
SH_TET, COLOR0, P4, P0, P6, P1,
SH_TET, COLOR1, P2, ED, EC, EK,
SH_TET, COLOR1, P5, EJ, EE, EF,
// Case #37: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EC, EK, EF, EF,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_TET, COLOR0, EE, EI, P4, N0,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_TET, COLOR0, P7, P4, P6, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_PYR, COLOR0, P1, EJ, EF, P7, N0,
SH_TET, COLOR0, P3, P7, P6, N0,
SH_PYR, COLOR0, P3, P6, EK, EC, N0,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_PYR, COLOR1, P5, P0, EI, EE, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
SH_PYR, COLOR1, EI, P0, P2, EK, N0,
SH_PYR, COLOR1, P0, EA, EC, P2, N0,
SH_TET, COLOR1, EC, EK, P2, N0,
// Case #38: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EF, EE, EK, EK,
SH_PYR, COLOR0, P6, P0, ED, EK, N0,
SH_TET, COLOR0, ED, P0, EA, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P6, P4, P0, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_TET, COLOR0, P7, P6, P3, N0,
SH_PYR, COLOR0, P3, P6, EK, EC, N0,
SH_TET, COLOR0, P7, P4, P6, N0,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_PYR, COLOR0, P7, P3, EB, EF, N0,
SH_PYR, COLOR1, P2, ED, EA, P1, N0,
SH_PYR, COLOR1, EB, EC, P2, P1, N0,
SH_TET, COLOR1, P2, EK, ED, N0,
SH_TET, COLOR1, EC, EK, P2, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, P1, P5, EF, EB, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #39: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EI, EE, EF, EB, EC, EK,
SH_TET, COLOR0, P7, P6, P3, N0,
SH_PYR, COLOR0, P7, P3, EB, EF, N0,
SH_TET, COLOR0, P4, P6, P7, N0,
SH_PYR, COLOR0, P4, P7, EF, EE, N0,
SH_TET, COLOR0, EI, P4, EE, N0,
SH_PYR, COLOR0, P4, EI, EK, P6, N0,
SH_PYR, COLOR0, EK, EC, P3, P6, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_PYR, COLOR1, EI, P0, P2, EK, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_TET, COLOR1, P2, EC, EK, N0,
SH_PYR, COLOR1, P2, P1, EB, EC, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #40: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EC, EE,
SH_PYR, COLOR0, P2, P6, P4, P0, N0,
SH_TET, COLOR0, P6, P7, P4, N0,
SH_TET, COLOR0, P2, P7, P6, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_PYR, COLOR0, P2, EC, EL, P7, N0,
SH_PYR, COLOR0, P2, P1, EB, EC, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR1, P3, EB, EJ, P5, N0,
SH_PYR, COLOR1, EL, P3, P5, EF, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EL, P3, N0, EC,
SH_TET, COLOR1, EB, P3, EC, N0,
// Case #41: (cloned #22)
SH_TET, COLOR0, P1, EA, EJ, EB,
SH_TET, COLOR0, P7, P6, P2, P4,
SH_PYR, COLOR0, EI, ED, P2, P4, EC,
SH_PYR, COLOR0, EE, P4, P7, EF, EL,
SH_PYR, COLOR0, P2, EC, EL, P7, P4,
SH_PYR, COLOR0, EC, EI, EE, EL, P4,
SH_WDG, COLOR1, P0, P5, P3, EA, EJ, EB,
SH_PYR, COLOR1, P0, P3, EC, ED, EI,
SH_PYR, COLOR1, EL, P3, P5, EF, EE,
SH_PYR, COLOR1, EI, EE, P5, P0, P3,
SH_PYR, COLOR1, EI, EC, EL, EE, P3,
// Case #42: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EE, EA, EF, EL, EC,
SH_PYR, COLOR0, P4, P0, P2, P6, N0,
SH_TET, COLOR0, P2, N0, P7, P6,
SH_PYR, COLOR0, P0, EA, EC, P2, N0,
SH_PYR, COLOR0, EE, EA, P0, P4, N0,
SH_TET, COLOR0, P7, P4, P6, N0,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_PYR, COLOR0, EC, EL, P7, P2, N0,
SH_PYR, COLOR1, EA, P1, P3, EC, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_TET, COLOR1, P5, P3, P1, N0,
SH_PYR, COLOR1, EF, EL, P3, P5, N0,
SH_TET, COLOR1, P3, EL, EC, N0,
// Case #43: (cloned #23)
SH_TET, COLOR0, P7, P6, P2, P4,
SH_PYR, COLOR0, EI, ED, P2, P4, EC,
SH_PYR, COLOR0, EE, P4, P7, EF, EL,
SH_PYR, COLOR0, P2, EC, EL, P7, P4,
SH_PYR, COLOR0, EC, EI, EE, EL, P4,
SH_TET, COLOR1, P1, P0, P5, P3,
SH_PYR, COLOR1, EC, ED, P0, P3, EI,
SH_PYR, COLOR1, EL, P3, P5, EF, EE,
SH_PYR, COLOR1, P0, EI, EE, P5, P3,
SH_PYR, COLOR1, EC, EL, EE, EI, P3,
// Case #44: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EK, ED, EE, EE,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_TET, COLOR0, EJ, EB, P1, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_TET, COLOR0, P6, P7, P4, N0,
SH_PYR, COLOR0, P7, EF, EE, P4, N0,
SH_TET, COLOR0, P6, P4, P0, N0,
SH_PYR, COLOR0, P6, P0, ED, EK, N0,
SH_PYR, COLOR0, P6, EK, EL, P7, N0,
SH_PYR, COLOR1, P5, P3, EB, EJ, N0,
SH_PYR, COLOR1, EL, P3, P5, EF, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_PYR, COLOR1, P3, EL, EK, P2, N0,
SH_TET, COLOR1, EK, ED, P2, N0,
// Case #45: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EI, EE, EF,
SH_TET, COLOR0, P1, EB, EA, EJ,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR0, P6, P7, P4, N0,
SH_PYR, COLOR0, P4, P7, EF, EE, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_WDG, COLOR1, EA, EB, EJ, P0, P3, P5,
SH_TET, COLOR1, P3, P0, P5, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_TET, COLOR1, P2, P0, P3, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, P5, EF, EL, P3, N0,
SH_TET, COLOR1, EE, EF, P5, N0,
// Case #46: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EA, ED, EK, EL, EF, EE,
SH_TET, COLOR0, P6, P7, P4, N0,
SH_PYR, COLOR0, P6, EK, EL, P7, N0,
SH_TET, COLOR0, P0, P6, P4, N0,
SH_PYR, COLOR0, P0, ED, EK, P6, N0,
SH_TET, COLOR0, EA, ED, P0, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_PYR, COLOR0, EE, P4, P7, EF, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_TET, COLOR1, P1, P3, P2, N0,
SH_PYR, COLOR1, EA, P1, P2, ED, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_TET, COLOR1, P5, P3, P1, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_PYR, COLOR1, P5, EF, EL, P3, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_TET, COLOR1, EK, ED, P2, N0,
// Case #47: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EI, EF, EE,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR0, P7, P6, EK, EL, N0,
SH_TET, COLOR0, EF, P7, EL, N0,
SH_PYR, COLOR0, EE, P4, P7, EF, N0,
SH_TET, COLOR0, P4, P6, P7, N0,
SH_PYR, COLOR1, P1, P3, P2, P0, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_TET, COLOR1, P5, P3, P1, N0,
SH_PYR, COLOR1, P5, EF, EL, P3, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #48: (cloned #3)
SH_HEX, COLOR0, EJ, P1, P0, EI, EF, P7, P6, EH,
SH_WDG, COLOR0, P1, P3, P7, P0, P2, P6,
SH_WDG, COLOR1, P5, EJ, EF, P4, EI, EH,
// Case #49: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EF, EH, EJ, EA, ED,
SH_PYR, COLOR0, P7, P6, P2, P3, N0,
SH_TET, COLOR0, P2, N0, P1, P3,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_PYR, COLOR0, EF, EH, P6, P7, N0,
SH_TET, COLOR0, P1, P7, P3, N0,
SH_PYR, COLOR0, P1, EJ, EF, P7, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, ED, EA, P1, P2, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
SH_TET, COLOR1, P5, P0, P4, N0,
SH_PYR, COLOR1, EJ, EA, P0, P5, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
// Case #50: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EH, EF, EI, EA, EB,
SH_PYR, COLOR0, P6, P2, P3, P7, N0,
SH_TET, COLOR0, P3, P0, N0, P2,
SH_PYR, COLOR0, P7, P3, EB, EF, N0,
SH_PYR, COLOR0, EH, P6, P7, EF, N0,
SH_TET, COLOR0, P0, P2, P6, N0,
SH_PYR, COLOR0, P0, P6, EH, EI, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, EB, P3, P0, EA, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, EI, EH, P4, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
// Case #51: (cloned #15)
SH_HEX, COLOR0, P2, P3, P7, P6, ED, EB, EF, EH,
SH_HEX, COLOR1, ED, EB, EF, EH, P0, P1, P5, P4,
// Case #52: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EF, EJ, EC, EC,
SH_PYR, COLOR0, P3, EC, ED, P0, N0,
SH_TET, COLOR0, ED, EI, P0, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR0, P7, P6, P3, N0,
SH_PYR, COLOR0, P6, EK, EC, P3, N0,
SH_TET, COLOR0, P7, P3, P1, N0,
SH_PYR, COLOR0, P7, P1, EJ, EF, N0,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_PYR, COLOR1, P2, P4, EI, ED, N0,
SH_PYR, COLOR1, EH, P4, P2, EK, N0,
SH_TET, COLOR1, P2, ED, EC, N0,
SH_TET, COLOR1, EK, P2, EC, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, P4, EH, EF, P5, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #53: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EA, EJ, EF, EH, EK, EC,
SH_TET, COLOR0, P7, P6, P3, N0,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_TET, COLOR0, P1, P7, P3, N0,
SH_PYR, COLOR0, P1, EJ, EF, P7, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, P1, P3, EC, EA, N0,
SH_PYR, COLOR0, EC, P3, P6, EK, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_PYR, COLOR1, EA, EC, P2, P0, N0,
SH_TET, COLOR1, P2, P4, P0, N0,
SH_TET, COLOR1, P2, EC, EK, N0,
SH_PYR, COLOR1, P2, EK, EH, P4, N0,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #54: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EH, EK, EC,
SH_TET, COLOR0, P0, EI, EA, ED,
SH_PYR, COLOR0, P3, EB, EF, P7, N0,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_TET, COLOR0, P7, P6, P3, N0,
SH_PYR, COLOR0, P6, EK, EC, P3, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_WDG, COLOR1, P4, P1, P2, EI, EA, ED,
SH_TET, COLOR1, P1, P2, P4, N0,
SH_PYR, COLOR1, P4, P2, EK, EH, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, P2, P1, EB, EC, N0,
SH_TET, COLOR1, EK, P2, EC, N0,
// Case #55: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EB, EF, EH, EC, EK,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_TET, COLOR0, P6, EH, EK, N0,
SH_PYR, COLOR0, P3, EB, EF, P7, N0,
SH_TET, COLOR0, EC, EB, P3, N0,
SH_PYR, COLOR0, EK, EC, P3, P6, N0,
SH_TET, COLOR0, P6, P3, P7, N0,
SH_PYR, COLOR1, P0, P4, P5, P1, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_PYR, COLOR1, P2, P1, EB, EC, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P2, EK, N0,
SH_TET, COLOR1, EC, EK, P2, N0,
// Case #56: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EH, EI, EC, EC,
SH_PYR, COLOR0, P2, P1, EB, EC, N0,
SH_TET, COLOR0, EB, P1, EJ, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_TET, COLOR0, P6, P2, P7, N0,
SH_PYR, COLOR0, P7, P2, EC, EL, N0,
SH_TET, COLOR0, P6, P0, P2, N0,
SH_PYR, COLOR0, P6, EH, EI, P0, N0,
SH_PYR, COLOR0, P6, P7, EF, EH, N0,
SH_PYR, COLOR1, P3, EB, EJ, P5, N0,
SH_PYR, COLOR1, EF, EL, P3, P5, N0,
SH_TET, COLOR1, P3, EC, EB, N0,
SH_TET, COLOR1, EL, EC, P3, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, P5, P4, EH, EF, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #57: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EH, EF, ED, EC, EL,
SH_TET, COLOR0, P1, EA, EJ, EB,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_TET, COLOR0, P6, P2, P7, N0,
SH_PYR, COLOR0, P2, EC, EL, P7, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_WDG, COLOR1, P0, P5, P3, EA, EJ, EB,
SH_TET, COLOR1, P5, P3, P0, N0,
SH_PYR, COLOR1, P0, P3, EC, ED, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, P3, P5, EF, EL, N0,
SH_TET, COLOR1, EC, P3, EL, N0,
// Case #58: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EA, EI, EH, EF, EL, EC,
SH_TET, COLOR0, P6, P2, P7, N0,
SH_PYR, COLOR0, P6, P7, EF, EH, N0,
SH_TET, COLOR0, P0, P2, P6, N0,
SH_PYR, COLOR0, P0, P6, EH, EI, N0,
SH_TET, COLOR0, EA, P0, EI, N0,
SH_PYR, COLOR0, P0, EA, EC, P2, N0,
SH_PYR, COLOR0, EC, EL, P7, P2, N0,
SH_TET, COLOR0, P7, EL, EF, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_PYR, COLOR1, EA, P1, P3, EC, N0,
SH_TET, COLOR1, P3, P1, P5, N0,
SH_TET, COLOR1, P3, EL, EC, N0,
SH_PYR, COLOR1, P3, P5, EF, EL, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #59: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EH, ED, EL, EC,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_PYR, COLOR0, P7, EF, EH, P6, N0,
SH_TET, COLOR0, EL, EF, P7, N0,
SH_PYR, COLOR0, EC, EL, P7, P2, N0,
SH_TET, COLOR0, P2, P7, P6, N0,
SH_PYR, COLOR1, P1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_TET, COLOR1, P3, P1, P5, N0,
SH_PYR, COLOR1, P3, P5, EF, EL, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P3, EC, N0,
SH_TET, COLOR1, EL, EC, P3, N0,
// Case #60: Unique case #14
SH_WDG, COLOR0, P1, EB, EJ, P0, ED, EI,
SH_WDG, COLOR0, P7, EF, EL, P6, EH, EK,
SH_HEX, COLOR1, P2, P4, P5, P3, EK, EH, EF, EL,
SH_HEX, COLOR1, ED, EI, EJ, EB, P2, P4, P5, P3,
// Case #61: Unique case #15
SH_PNT, 0, COLOR1, 6, P0, P3, P2, P4, EF, EH,
SH_WDG, COLOR0, EH, P6, EK, EF, P7, EL,
SH_TET, COLOR0, EA, P1, EB, EJ,
SH_WDG, COLOR1, P0, P5, P3, EA, EJ, EB,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_PYR, COLOR1, EK, EH, P4, P2, N0,
SH_PYR, COLOR1, EL, EK, P2, P3, N0,
SH_TET, COLOR1, P2, P0, P3, N0,
SH_PYR, COLOR1, EF, EH, EK, EL, N0,
SH_TET, COLOR1, P3, P0, P5, N0,
SH_PYR, COLOR1, EF, EL, P3, P5, N0,
// Case #62: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P2, P3, P5, EH, EF,
SH_WDG, COLOR0, EH, P6, EK, EF, P7, EL,
SH_TET, COLOR0, EA, ED, P0, EI,
SH_WDG, COLOR1, EA, EI, ED, P1, P4, P2,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_TET, COLOR1, P5, P3, P1, N0,
SH_PYR, COLOR1, EL, P3, P5, EF, N0,
SH_PYR, COLOR1, EK, P2, P3, EL, N0,
SH_TET, COLOR1, P3, P2, P1, N0,
SH_PYR, COLOR1, EH, EK, EL, EF, N0,
SH_TET, COLOR1, P2, P4, P1, N0,
SH_PYR, COLOR1, EH, P4, P2, EK, N0,
// Case #63: Unique case #16
SH_WDG, COLOR0, P6, EK, EH, P7, EL, EF,
SH_HEX, COLOR1, P2, P4, P5, P3, EK, EH, EF, EL,
SH_WDG, COLOR1, P1, P3, P5, P0, P2, P4,
// Case #64: (cloned #1)
SH_PNT, 0, COLOR0, 7, P7, P5, P4, P2, P3, P1, P0,
SH_WDG, COLOR0, P7, P4, P2, EG, EH, EK,
SH_TET, COLOR0, P7, P4, P2, N0,
SH_TET, COLOR0, P7, P5, P4, N0,
SH_PYR, COLOR0, P1, P0, P4, P5, N0,
SH_PYR, COLOR0, P3, P1, P5, P7, N0,
SH_PYR, COLOR0, P2, P0, P1, P3, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_TET, COLOR0, P2, P3, P7, N0,
SH_TET, COLOR1, P6, EG, EH, EK,
// Case #65: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EA, EG,
SH_PYR, COLOR0, P1, P5, P7, P3, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_TET, COLOR0, P2, P1, P3, N0,
SH_TET, COLOR0, P7, P2, P3, N0,
SH_PYR, COLOR0, P7, EG, EK, P2, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_PYR, COLOR0, P1, P2, ED, EA, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_PYR, COLOR1, P0, ED, EK, P6, N0,
SH_PYR, COLOR1, EI, P0, P6, EH, N0,
SH_TET, COLOR1, EH, P6, EG, N0,
SH_TET, COLOR1, P6, EK, EG, N0,
SH_TET, COLOR1, EI, P0, N0, EA,
SH_TET, COLOR1, ED, P0, EA, N0,
// Case #66: (cloned #24)
SH_WDG, COLOR0, P4, P2, P7, EH, EK, EG,
SH_WDG, COLOR0, EA, EB, EJ, P0, P3, P5,
SH_TET, COLOR0, P2, P7, P4, P3,
SH_TET, COLOR0, P5, P7, P3, P4,
SH_TET, COLOR0, P0, P4, P5, P3,
SH_TET, COLOR0, P2, P0, P3, P4,
SH_TET, COLOR1, P1, EA, EJ, EB,
SH_TET, COLOR1, P6, EH, EK, EG,
// Case #67: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EJ, EB, EG, EG,
SH_PYR, COLOR0, P7, EG, EK, P2, N0,
SH_TET, COLOR0, EK, ED, P2, N0,
SH_PYR, COLOR0, P3, P2, ED, EB, N0,
SH_TET, COLOR0, P7, P2, P3, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_TET, COLOR0, P5, P7, P3, N0,
SH_PYR, COLOR0, P5, P3, EB, EJ, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_PYR, COLOR1, P6, P0, ED, EK, N0,
SH_PYR, COLOR1, EI, P0, P6, EH, N0,
SH_TET, COLOR1, P6, EK, EG, N0,
SH_TET, COLOR1, EH, P6, EG, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, P0, EI, EJ, P1, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
// Case #68: (cloned #3)
SH_HEX, COLOR0, EH, P4, P0, ED, EG, P7, P3, EC,
SH_WDG, COLOR0, P4, P5, P7, P0, P1, P3,
SH_WDG, COLOR1, P6, EH, EG, P2, ED, EC,
// Case #69: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EH, EI, EA,
SH_PYR, COLOR0, P7, P3, P1, P5, N0,
SH_TET, COLOR0, P1, N0, P4, P5,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_PYR, COLOR0, EG, EC, P3, P7, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, EA, EI, P4, P1, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_TET, COLOR1, EH, P6, EG, N0,
SH_TET, COLOR1, P6, P0, P2, N0,
SH_PYR, COLOR1, EH, EI, P0, P6, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
// Case #70: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EG, EH, EJ, EJ,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_TET, COLOR0, EA, ED, P0, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_PYR, COLOR0, P3, EB, EJ, P5, N0,
SH_TET, COLOR0, P7, P5, P4, N0,
SH_PYR, COLOR0, P7, P4, EH, EG, N0,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_PYR, COLOR1, P1, P2, ED, EA, N0,
SH_PYR, COLOR1, EC, P2, P1, EB, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_PYR, COLOR1, ED, P2, P6, EH, N0,
SH_PYR, COLOR1, P2, EC, EG, P6, N0,
SH_TET, COLOR1, EG, EH, P6, N0,
// Case #71: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EI, EH, EG, EC, EB, EJ,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_TET, COLOR0, EI, EH, P4, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_PYR, COLOR0, EJ, P5, P3, EB, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_TET, COLOR1, P0, P2, P6, N0,
SH_PYR, COLOR1, EI, P0, P6, EH, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_TET, COLOR1, P1, EJ, EB, N0,
SH_PYR, COLOR1, P1, EB, EC, P2, N0,
SH_PYR, COLOR1, EC, EG, P6, P2, N0,
SH_TET, COLOR1, EG, EH, P6, N0,
// Case #72: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EH, EB,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_TET, COLOR0, P2, P4, P0, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P2, N0,
SH_PYR, COLOR0, P7, EL, EB, P1, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_PYR, COLOR0, P4, P2, EK, EH, N0,
SH_TET, COLOR0, P2, EC, EK, N0,
SH_PYR, COLOR1, P6, EK, EC, P3, N0,
SH_PYR, COLOR1, EG, P6, P3, EL, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
SH_TET, COLOR1, P3, EC, EB, N0,
SH_TET, COLOR1, EG, P6, N0, EH,
SH_TET, COLOR1, EK, P6, EH, N0,
// Case #73: (cloned #22)
SH_TET, COLOR0, P2, ED, EC, EK,
SH_TET, COLOR0, P7, P5, P4, P1,
SH_PYR, COLOR0, EA, EI, P4, P1, EH,
SH_PYR, COLOR0, EB, P1, P7, EL, EG,
SH_PYR, COLOR0, P4, EH, EG, P7, P1,
SH_PYR, COLOR0, EH, EA, EB, EG, P1,
SH_WDG, COLOR1, P0, P3, P6, ED, EC, EK,
SH_PYR, COLOR1, P0, P6, EH, EI, EA,
SH_PYR, COLOR1, EG, P6, P3, EL, EB,
SH_PYR, COLOR1, EA, EB, P3, P0, P6,
SH_PYR, COLOR1, EA, EH, EG, EB, P6,
// Case #74: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EJ, EA, EH, EH,
SH_PYR, COLOR0, P4, P2, EK, EH, N0,
SH_TET, COLOR0, EK, P2, EC, N0,
SH_PYR, COLOR0, P0, EA, EC, P2, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P7, P4, EH, EG, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_PYR, COLOR0, P5, P7, EL, EJ, N0,
SH_PYR, COLOR1, P6, EK, EC, P3, N0,
SH_PYR, COLOR1, EL, EG, P6, P3, N0,
SH_TET, COLOR1, P6, EH, EK, N0,
SH_TET, COLOR1, EG, EH, P6, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, P3, P1, EJ, EL, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #75: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EI, EH, EG,
SH_TET, COLOR0, P2, ED, EC, EK,
SH_PYR, COLOR0, P7, EL, EJ, P5, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P4, EH, EG, P7, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_WDG, COLOR1, P0, P3, P6, ED, EC, EK,
SH_TET, COLOR1, P3, P6, P0, N0,
SH_PYR, COLOR1, P0, P6, EH, EI, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_PYR, COLOR1, P6, P3, EL, EG, N0,
SH_TET, COLOR1, EH, P6, EG, N0,
// Case #76: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EB, ED, EL, EG, EH,
SH_PYR, COLOR0, P1, P0, P4, P5, N0,
SH_TET, COLOR0, P4, N0, P7, P5,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_PYR, COLOR0, EB, ED, P0, P1, N0,
SH_TET, COLOR0, P7, P1, P5, N0,
SH_PYR, COLOR0, P7, EL, EB, P1, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_PYR, COLOR0, EH, EG, P7, P4, N0,
SH_PYR, COLOR1, ED, P2, P6, EH, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
SH_TET, COLOR1, P3, P6, P2, N0,
SH_PYR, COLOR1, EL, EG, P6, P3, N0,
SH_TET, COLOR1, P6, EG, EH, N0,
// Case #77: (cloned #23)
SH_TET, COLOR0, P7, P5, P4, P1,
SH_PYR, COLOR0, EA, EI, P4, P1, EH,
SH_PYR, COLOR0, EB, P1, P7, EL, EG,
SH_PYR, COLOR0, P4, EH, EG, P7, P1,
SH_PYR, COLOR0, EH, EA, EB, EG, P1,
SH_TET, COLOR1, P2, P0, P3, P6,
SH_PYR, COLOR1, EH, EI, P0, P6, EA,
SH_PYR, COLOR1, EG, P6, P3, EL, EB,
SH_PYR, COLOR1, P0, EA, EB, P3, P6,
SH_PYR, COLOR1, EH, EG, EB, EA, P6,
// Case #78: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, ED, EA, EJ, EL, EG, EH,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P5, P7, EL, EJ, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, ED, P0, EA, N0,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_PYR, COLOR0, EH, EG, P7, P4, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_TET, COLOR1, P2, P1, P3, N0,
SH_PYR, COLOR1, ED, EA, P1, P2, N0,
SH_PYR, COLOR1, ED, P2, P6, EH, N0,
SH_TET, COLOR1, P6, P2, P3, N0,
SH_TET, COLOR1, P6, EG, EH, N0,
SH_PYR, COLOR1, P6, P3, EL, EG, N0,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #79: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EI, EG, EH,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P7, EL, EJ, P5, N0,
SH_TET, COLOR0, EG, EL, P7, N0,
SH_PYR, COLOR0, EH, EG, P7, P4, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_PYR, COLOR1, P2, P0, P1, P3, N0,
SH_TET, COLOR1, P0, P2, P6, N0,
SH_TET, COLOR1, P6, P2, P3, N0,
SH_PYR, COLOR1, P6, P3, EL, EG, N0,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_PYR, COLOR1, EI, P0, P6, EH, N0,
SH_TET, COLOR1, EG, EH, P6, N0,
// Case #80: (cloned #3)
SH_HEX, COLOR0, EG, P7, P5, EE, EK, P2, P0, EI,
SH_WDG, COLOR0, P0, P1, P5, P2, P3, P7,
SH_WDG, COLOR1, P4, EI, EE, P6, EK, EG,
// Case #81: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EK, ED, EA,
SH_PYR, COLOR0, P7, P3, P1, P5, N0,
SH_TET, COLOR0, P1, P2, N0, P3,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_PYR, COLOR0, EG, P7, P5, EE, N0,
SH_TET, COLOR0, P2, P3, P7, N0,
SH_PYR, COLOR0, P2, P7, EG, EK, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_PYR, COLOR0, EA, P1, P2, ED, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_TET, COLOR1, EK, EG, P6, N0,
SH_TET, COLOR1, P6, P4, P0, N0,
SH_PYR, COLOR1, EK, P6, P0, ED, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
// Case #82: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EG, EK, EB, EB,
SH_PYR, COLOR0, P3, P0, EA, EB, N0,
SH_TET, COLOR0, EA, P0, EI, N0,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_TET, COLOR0, P3, P2, P0, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_PYR, COLOR0, P5, P3, EB, EJ, N0,
SH_TET, COLOR0, P7, P2, P3, N0,
SH_PYR, COLOR0, P7, EG, EK, P2, N0,
SH_PYR, COLOR0, P7, P5, EE, EG, N0,
SH_PYR, COLOR1, P1, EA, EI, P4, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_PYR, COLOR1, EI, EK, P6, P4, N0,
SH_PYR, COLOR1, P4, P6, EG, EE, N0,
SH_TET, COLOR1, EG, P6, EK, N0,
// Case #83: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, ED, EK, EG, EE, EJ, EB,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_PYR, COLOR0, P7, P5, EE, EG, N0,
SH_TET, COLOR0, P2, P3, P7, N0,
SH_PYR, COLOR0, P2, P7, EG, EK, N0,
SH_TET, COLOR0, ED, P2, EK, N0,
SH_PYR, COLOR0, P2, ED, EB, P3, N0,
SH_PYR, COLOR0, EB, EJ, P5, P3, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR1, P0, P6, P4, N0,
SH_PYR, COLOR1, ED, EK, P6, P0, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_TET, COLOR1, P1, EJ, EB, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_PYR, COLOR1, EE, P4, P6, EG, N0,
SH_TET, COLOR1, EG, P6, EK, N0,
// Case #84: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EC, EG, ED, EI, EE,
SH_PYR, COLOR0, P3, P1, P5, P7, N0,
SH_TET, COLOR0, P5, P0, N0, P1,
SH_PYR, COLOR0, P7, P5, EE, EG, N0,
SH_PYR, COLOR0, EC, P3, P7, EG, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_PYR, COLOR0, P0, P3, EC, ED, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, EE, P5, P0, EI, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_PYR, COLOR1, EC, EG, P6, P2, N0,
SH_TET, COLOR1, ED, EC, P2, N0,
SH_TET, COLOR1, P2, P6, P4, N0,
SH_PYR, COLOR1, ED, P2, P4, EI, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
// Case #85: (cloned #15)
SH_HEX, COLOR0, P1, P5, P7, P3, EA, EE, EG, EC,
SH_HEX, COLOR1, EA, EE, EG, EC, P0, P4, P6, P2,
// Case #86: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EC, EB, EJ,
SH_TET, COLOR0, P0, ED, EI, EA,
SH_PYR, COLOR0, P5, EE, EG, P7, N0,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_TET, COLOR0, P7, P3, P5, N0,
SH_PYR, COLOR0, P3, EB, EJ, P5, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_WDG, COLOR1, P2, P4, P1, ED, EI, EA,
SH_TET, COLOR1, P4, P1, P2, N0,
SH_PYR, COLOR1, P2, P1, EB, EC, N0,
SH_TET, COLOR1, P6, P4, P2, N0,
SH_PYR, COLOR1, EE, P4, P6, EG, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
// Case #87: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EC, EJ, EB,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_TET, COLOR0, P3, EC, EB, N0,
SH_PYR, COLOR0, P5, EE, EG, P7, N0,
SH_TET, COLOR0, EJ, EE, P5, N0,
SH_PYR, COLOR0, EB, EJ, P5, P3, N0,
SH_TET, COLOR0, P3, P5, P7, N0,
SH_PYR, COLOR1, P0, P2, P6, P4, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_PYR, COLOR1, EE, P4, P6, EG, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_PYR, COLOR1, EC, P2, P1, EB, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
// Case #88: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EE, EI, EB, EB,
SH_PYR, COLOR0, P1, EB, EC, P2, N0,
SH_TET, COLOR0, EC, EK, P2, N0,
SH_PYR, COLOR0, P0, P2, EK, EI, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_PYR, COLOR0, P7, EL, EB, P1, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_PYR, COLOR0, P5, EE, EG, P7, N0,
SH_PYR, COLOR1, P3, P6, EK, EC, N0,
SH_PYR, COLOR1, EG, P6, P3, EL, N0,
SH_TET, COLOR1, P3, EC, EB, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, P6, EG, EE, P4, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #89: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EA, EB, EL,
SH_TET, COLOR0, P2, EK, ED, EC,
SH_PYR, COLOR0, P7, P5, EE, EG, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_PYR, COLOR0, P1, P7, EL, EB, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_WDG, COLOR1, ED, EK, EC, P0, P6, P3,
SH_TET, COLOR1, P6, P0, P3, N0,
SH_PYR, COLOR1, P0, EA, EB, P3, N0,
SH_TET, COLOR1, P4, P0, P6, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_PYR, COLOR1, P3, EL, EG, P6, N0,
SH_TET, COLOR1, EB, EL, P3, N0,
// Case #90: (cloned #60)
SH_WDG, COLOR0, P2, EK, EC, P0, EI, EA,
SH_WDG, COLOR0, P7, EL, EG, P5, EJ, EE,
SH_HEX, COLOR1, P4, P1, P3, P6, EE, EJ, EL, EG,
SH_HEX, COLOR1, EI, EA, EC, EK, P4, P1, P3, P6,
// Case #91: (cloned #61)
SH_PNT, 0, COLOR1, 6, P0, P6, P4, P1, EL, EJ,
SH_WDG, COLOR0, EJ, P5, EE, EL, P7, EG,
SH_TET, COLOR0, ED, P2, EK, EC,
SH_WDG, COLOR1, P0, P3, P6, ED, EC, EK,
SH_PYR, COLOR1, EJ, EL, P3, P1, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_TET, COLOR1, P4, P0, P6, N0,
SH_PYR, COLOR1, EL, EJ, EE, EG, N0,
SH_TET, COLOR1, P6, P0, P3, N0,
SH_PYR, COLOR1, EL, EG, P6, P3, N0,
// Case #92: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, ED, EI, EE, EG, EL, EB,
SH_TET, COLOR0, P5, P7, P1, N0,
SH_PYR, COLOR0, P5, EE, EG, P7, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, ED, EI, P0, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_PYR, COLOR0, EB, P1, P7, EL, N0,
SH_TET, COLOR0, P7, EG, EL, N0,
SH_TET, COLOR1, P2, P6, P4, N0,
SH_PYR, COLOR1, ED, P2, P4, EI, N0,
SH_PYR, COLOR1, ED, EB, P3, P2, N0,
SH_TET, COLOR1, P3, P6, P2, N0,
SH_TET, COLOR1, P3, EB, EL, N0,
SH_PYR, COLOR1, P3, EL, EG, P6, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #93: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EA, EL, EB,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, P7, P5, EE, EG, N0,
SH_TET, COLOR0, EL, P7, EG, N0,
SH_PYR, COLOR0, EB, P1, P7, EL, N0,
SH_TET, COLOR0, P1, P5, P7, N0,
SH_PYR, COLOR1, P2, P6, P4, P0, N0,
SH_TET, COLOR1, P0, P3, P2, N0,
SH_TET, COLOR1, P3, P6, P2, N0,
SH_PYR, COLOR1, P3, EL, EG, P6, N0,
SH_PYR, COLOR1, EG, EE, P4, P6, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_PYR, COLOR1, EA, EB, P3, P0, N0,
SH_TET, COLOR1, EL, P3, EB, N0,
// Case #94: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P4, P6, P3, EJ, EL,
SH_WDG, COLOR0, EJ, P5, EE, EL, P7, EG,
SH_TET, COLOR0, ED, EI, P0, EA,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_PYR, COLOR1, EL, P3, P1, EJ, N0,
SH_TET, COLOR1, P3, P2, P1, N0,
SH_TET, COLOR1, P3, P6, P2, N0,
SH_PYR, COLOR1, EG, P6, P3, EL, N0,
SH_PYR, COLOR1, EE, P4, P6, EG, N0,
SH_TET, COLOR1, P6, P4, P2, N0,
SH_PYR, COLOR1, EJ, EE, EG, EL, N0,
SH_TET, COLOR1, P4, P1, P2, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
// Case #95: (cloned #63)
SH_WDG, COLOR0, P5, EE, EJ, P7, EG, EL,
SH_HEX, COLOR1, P4, P1, P3, P6, EE, EJ, EL, EG,
SH_WDG, COLOR1, P2, P6, P3, P0, P4, P1,
// Case #96: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EK, EJ,
SH_PYR, COLOR0, P2, P0, P1, P3, N0,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_TET, COLOR0, P2, P3, P7, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_PYR, COLOR0, P7, P1, EJ, EF, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_PYR, COLOR0, P2, P7, EG, EK, N0,
SH_PYR, COLOR0, P2, EK, EH, P4, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR1, P6, P5, EE, EH, N0,
SH_PYR, COLOR1, EG, EF, P5, P6, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EG, N0, P6, EK,
SH_TET, COLOR1, EH, EK, P6, N0,
// Case #97: (cloned #22)
SH_TET, COLOR0, P4, EE, EI, EH,
SH_TET, COLOR0, P7, P2, P3, P1,
SH_PYR, COLOR0, EA, P1, P2, ED, EK,
SH_PYR, COLOR0, EJ, EF, P7, P1, EG,
SH_PYR, COLOR0, P2, P7, EG, EK, P1,
SH_PYR, COLOR0, EK, EG, EJ, EA, P1,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P6,
SH_PYR, COLOR1, P0, ED, EK, P6, EA,
SH_PYR, COLOR1, EG, EF, P5, P6, EJ,
SH_PYR, COLOR1, EA, P0, P5, EJ, P6,
SH_PYR, COLOR1, EA, EJ, EG, EK, P6,
// Case #98: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EB, EA, EK, EK,
SH_PYR, COLOR0, P2, EK, EH, P4, N0,
SH_TET, COLOR0, EH, EE, P4, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P2, P4, P0, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_TET, COLOR0, P3, P7, P2, N0,
SH_PYR, COLOR0, P7, EG, EK, P2, N0,
SH_TET, COLOR0, P3, P2, P0, N0,
SH_PYR, COLOR0, P3, P0, EA, EB, N0,
SH_PYR, COLOR0, P3, EB, EF, P7, N0,
SH_PYR, COLOR1, P6, P5, EE, EH, N0,
SH_PYR, COLOR1, EF, P5, P6, EG, N0,
SH_TET, COLOR1, P6, EH, EK, N0,
SH_TET, COLOR1, EG, P6, EK, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, P5, EF, EB, P1, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #99: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EB, EF, ED, EK, EG,
SH_TET, COLOR0, P4, EE, EI, EH,
SH_PYR, COLOR0, P7, P3, EB, EF, N0,
SH_PYR, COLOR0, P3, P2, ED, EB, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_TET, COLOR0, P3, P7, P2, N0,
SH_PYR, COLOR0, P2, P7, EG, EK, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P6,
SH_TET, COLOR1, P5, P0, P6, N0,
SH_PYR, COLOR1, P0, ED, EK, P6, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, P6, EG, EF, P5, N0,
SH_TET, COLOR1, EK, EG, P6, N0,
// Case #100: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EC, ED, EJ, EJ,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_PYR, COLOR0, P7, P1, EJ, EF, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_PYR, COLOR0, P3, EC, ED, P0, N0,
SH_PYR, COLOR0, P3, P7, EG, EC, N0,
SH_PYR, COLOR1, P5, EE, EH, P6, N0,
SH_PYR, COLOR1, EG, EF, P5, P6, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, P6, P2, EC, EG, N0,
SH_TET, COLOR1, EC, P2, ED, N0,
// Case #101: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EA, EJ, EF,
SH_TET, COLOR0, P4, EI, EH, EE,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_PYR, COLOR0, P1, EJ, EF, P7, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_WDG, COLOR1, P0, P6, P5, EI, EH, EE,
SH_TET, COLOR1, P6, P5, P0, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_TET, COLOR1, P2, P6, P0, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_PYR, COLOR1, P5, P6, EG, EF, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
// Case #102: (cloned #60)
SH_WDG, COLOR0, P0, ED, EA, P4, EH, EE,
SH_WDG, COLOR0, P3, EB, EC, P7, EF, EG,
SH_HEX, COLOR1, EC, EB, EF, EG, P2, P1, P5, P6,
SH_HEX, COLOR1, P2, P1, P5, P6, ED, EA, EE, EH,
// Case #103: (cloned #61)
SH_PNT, 0, COLOR1, 6, P0, P6, P2, P1, EF, EB,
SH_WDG, COLOR0, EF, P7, EG, EB, P3, EC,
SH_TET, COLOR0, EI, EH, P4, EE,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P6,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_PYR, COLOR1, EC, P2, P1, EB, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_TET, COLOR1, P2, P6, P0, N0,
SH_PYR, COLOR1, EF, EG, EC, EB, N0,
SH_TET, COLOR1, P6, P5, P0, N0,
SH_PYR, COLOR1, EF, P5, P6, EG, N0,
// Case #104: (cloned #22)
SH_TET, COLOR0, P7, EF, EG, EL,
SH_TET, COLOR0, P2, P0, P1, P4,
SH_PYR, COLOR0, EE, EJ, P1, P4, EB,
SH_PYR, COLOR0, EH, P4, P2, EK, EC,
SH_PYR, COLOR0, P1, EB, EC, P2, P4,
SH_PYR, COLOR0, EB, EE, EH, EC, P4,
SH_WDG, COLOR1, P5, P6, P3, EF, EG, EL,
SH_PYR, COLOR1, P5, P3, EB, EJ, EE,
SH_PYR, COLOR1, EC, P3, P6, EK, EH,
SH_PYR, COLOR1, EE, EH, P6, P5, P3,
SH_PYR, COLOR1, EE, EB, EC, EH, P3,
// Case #105: Unique case #17
SH_TET, COLOR0, EH, EI, EE, P4,
SH_TET, COLOR0, EK, EC, ED, P2,
SH_TET, COLOR0, EG, EF, EL, P7,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, P3, P0, P5, EB, EA, EJ,
SH_TET, COLOR1, P3, P0, P5, P6,
SH_WDG, COLOR1, EG, EF, EL, P6, P5, P3,
SH_WDG, COLOR1, ED, EK, EC, P0, P6, P3,
SH_WDG, COLOR1, EE, EH, EI, P5, P6, P0,
// Case #106: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EE, EH, EK,
SH_TET, COLOR0, P7, EL, EF, EG,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_TET, COLOR0, P0, P2, P4, N0,
SH_PYR, COLOR0, P4, P2, EK, EH, N0,
SH_TET, COLOR0, P2, EC, EK, N0,
SH_WDG, COLOR1, EF, EL, EG, P5, P3, P6,
SH_TET, COLOR1, P3, P5, P6, N0,
SH_PYR, COLOR1, P5, EE, EH, P6, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, P6, EK, EC, P3, N0,
SH_TET, COLOR1, EH, EK, P6, N0,
// Case #107: Unique case #18
SH_TET, COLOR0, EL, EG, EF, P7,
SH_TET, COLOR0, EC, P2, ED, EK,
SH_TET, COLOR0, EE, EI, P4, EH,
SH_WDG, COLOR1, EH, EI, EE, P6, P0, P5,
SH_WDG, COLOR1, P3, P6, P0, EC, EK, ED,
SH_WDG, COLOR1, EL, EG, EF, P3, P6, P5,
SH_TET, COLOR1, P6, P5, P0, P3,
SH_TET, COLOR1, P3, P1, P5, P0,
// Case #108: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EB, EJ, EE,
SH_TET, COLOR0, P7, EG, EL, EF,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_TET, COLOR0, P0, P4, P1, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_WDG, COLOR1, EL, EG, EF, P3, P6, P5,
SH_TET, COLOR1, P6, P3, P5, N0,
SH_PYR, COLOR1, P3, EB, EJ, P5, N0,
SH_TET, COLOR1, P2, P3, P6, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, ED, EB, P3, P2, N0,
SH_PYR, COLOR1, P5, EE, EH, P6, N0,
SH_TET, COLOR1, EJ, EE, P5, N0,
// Case #109: (cloned #107)
SH_TET, COLOR0, EG, EF, EL, P7,
SH_TET, COLOR0, EH, P4, EI, EE,
SH_TET, COLOR0, EB, EA, P1, EJ,
SH_WDG, COLOR1, EJ, EA, EB, P5, P0, P3,
SH_WDG, COLOR1, P6, P5, P0, EH, EE, EI,
SH_WDG, COLOR1, EG, EF, EL, P6, P5, P3,
SH_TET, COLOR1, P5, P3, P0, P6,
SH_TET, COLOR1, P6, P2, P3, P0,
// Case #110: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P5, P1, P2, EH, ED,
SH_WDG, COLOR0, EH, P4, EE, ED, P0, EA,
SH_TET, COLOR0, EL, EF, P7, EG,
SH_WDG, COLOR1, EL, EG, EF, P3, P6, P5,
SH_PYR, COLOR1, ED, P2, P6, EH, N0,
SH_TET, COLOR1, P2, P3, P6, N0,
SH_TET, COLOR1, P2, P1, P3, N0,
SH_PYR, COLOR1, EA, P1, P2, ED, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_PYR, COLOR1, EH, EE, EA, ED, N0,
SH_TET, COLOR1, P5, P6, P3, N0,
SH_PYR, COLOR1, EH, P6, P5, EE, N0,
// Case #111: Unique case #19
SH_TET, COLOR0, EE, EH, EI, P4,
SH_TET, COLOR0, EF, EL, EG, P7,
SH_WDG, COLOR1, EG, EF, EL, P6, P5, P3,
SH_WDG, COLOR1, P6, P5, P0, EH, EE, EI,
SH_PYR, COLOR1, P2, P0, P1, P3, P6,
SH_TET, COLOR1, P5, P1, P0, P6,
SH_TET, COLOR1, P5, P3, P1, P6,
// Case #112: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EJ, EI, EF, EG, EK,
SH_PYR, COLOR0, P1, P3, P2, P0, N0,
SH_TET, COLOR0, P2, P7, N0, P3,
SH_PYR, COLOR0, P0, P2, EK, EI, N0,
SH_PYR, COLOR0, EJ, P1, P0, EI, N0,
SH_TET, COLOR0, P7, P3, P1, N0,
SH_PYR, COLOR0, P7, P1, EJ, EF, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_PYR, COLOR0, EK, P2, P7, EG, N0,
SH_PYR, COLOR1, EI, EK, P6, P4, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_PYR, COLOR1, EF, P5, P6, EG, N0,
SH_TET, COLOR1, P6, EK, EG, N0,
// Case #113: (cloned #23)
SH_TET, COLOR0, P7, P2, P3, P1,
SH_PYR, COLOR0, EA, P1, P2, ED, EK,
SH_PYR, COLOR0, EJ, EF, P7, P1, EG,
SH_PYR, COLOR0, P2, P7, EG, EK, P1,
SH_PYR, COLOR0, EK, EG, EJ, EA, P1,
SH_TET, COLOR1, P4, P5, P0, P6,
SH_PYR, COLOR1, EK, P6, P0, ED, EA,
SH_PYR, COLOR1, EG, EF, P5, P6, EJ,
SH_PYR, COLOR1, P0, P5, EJ, EA, P6,
SH_PYR, COLOR1, EK, EA, EJ, EG, P6,
// Case #114: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EI, EA, EB, EF, EG, EK,
SH_TET, COLOR0, P3, P7, P2, N0,
SH_PYR, COLOR0, P3, EB, EF, P7, N0,
SH_TET, COLOR0, P0, P3, P2, N0,
SH_PYR, COLOR0, P0, EA, EB, P3, N0,
SH_TET, COLOR0, EI, EA, P0, N0,
SH_PYR, COLOR0, P0, P2, EK, EI, N0,
SH_PYR, COLOR0, EK, P2, P7, EG, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_PYR, COLOR1, EI, EK, P6, P4, N0,
SH_TET, COLOR1, P6, P5, P4, N0,
SH_TET, COLOR1, P6, EK, EG, N0,
SH_PYR, COLOR1, P6, EG, EF, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #115: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EB, ED, EG, EK,
SH_PYR, COLOR0, P3, P2, ED, EB, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_PYR, COLOR0, P7, P3, EB, EF, N0,
SH_TET, COLOR0, EG, P7, EF, N0,
SH_PYR, COLOR0, EK, P2, P7, EG, N0,
SH_TET, COLOR0, P2, P3, P7, N0,
SH_PYR, COLOR1, P4, P5, P1, P0, N0,
SH_TET, COLOR1, P0, P6, P4, N0,
SH_TET, COLOR1, P6, P5, P4, N0,
SH_PYR, COLOR1, P6, EG, EF, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, ED, EK, P6, P0, N0,
SH_TET, COLOR1, EG, P6, EK, N0,
// Case #116: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EI, ED, EC, EG, EF, EJ,
SH_TET, COLOR0, P3, P1, P7, N0,
SH_PYR, COLOR0, P3, P7, EG, EC, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_PYR, COLOR0, P0, P3, EC, ED, N0,
SH_TET, COLOR0, EI, P0, ED, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_PYR, COLOR0, EJ, EF, P7, P1, N0,
SH_TET, COLOR0, P7, EF, EG, N0,
SH_TET, COLOR1, P4, P2, P6, N0,
SH_PYR, COLOR1, EI, ED, P2, P4, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_TET, COLOR1, P5, EF, EJ, N0,
SH_PYR, COLOR1, P5, P6, EG, EF, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_TET, COLOR1, EC, P2, ED, N0,
// Case #117: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EA, EF, EJ,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, P7, EG, EC, P3, N0,
SH_TET, COLOR0, EF, EG, P7, N0,
SH_PYR, COLOR0, EJ, EF, P7, P1, N0,
SH_TET, COLOR0, P1, P7, P3, N0,
SH_PYR, COLOR1, P4, P0, P2, P6, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_PYR, COLOR1, P5, P6, EG, EF, N0,
SH_PYR, COLOR1, EG, P6, P2, EC, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #118: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P2, P6, P5, EB, EF,
SH_WDG, COLOR0, EF, P7, EG, EB, P3, EC,
SH_TET, COLOR0, EI, P0, ED, EA,
SH_WDG, COLOR1, P4, P1, P2, EI, EA, ED,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_PYR, COLOR1, EG, EF, P5, P6, N0,
SH_PYR, COLOR1, EC, EG, P6, P2, N0,
SH_TET, COLOR1, P6, P4, P2, N0,
SH_PYR, COLOR1, EB, EF, EG, EC, N0,
SH_TET, COLOR1, P2, P4, P1, N0,
SH_PYR, COLOR1, EB, EC, P2, P1, N0,
// Case #119: (cloned #63)
SH_WDG, COLOR0, P7, EG, EF, P3, EC, EB,
SH_HEX, COLOR1, EC, EB, EF, EG, P2, P1, P5, P6,
SH_WDG, COLOR1, P0, P2, P1, P4, P6, P5,
// Case #120: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EJ, EB, EC,
SH_TET, COLOR0, P7, EF, EG, EL,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_PYR, COLOR0, P1, EB, EC, P2, N0,
SH_TET, COLOR0, P2, EC, EK, N0,
SH_WDG, COLOR1, P5, P6, P3, EF, EG, EL,
SH_TET, COLOR1, P6, P3, P5, N0,
SH_PYR, COLOR1, P5, P3, EB, EJ, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, P3, P6, EK, EC, N0,
SH_TET, COLOR1, EB, P3, EC, N0,
// Case #121: (cloned #107)
SH_TET, COLOR0, EG, EF, EL, P7,
SH_TET, COLOR0, EK, ED, P2, EC,
SH_TET, COLOR0, EJ, P1, EA, EB,
SH_WDG, COLOR1, P3, P0, P5, EB, EA, EJ,
SH_WDG, COLOR1, EK, EC, ED, P6, P3, P0,
SH_WDG, COLOR1, P6, P3, P5, EG, EL, EF,
SH_TET, COLOR1, P3, P0, P5, P6,
SH_TET, COLOR1, P6, P5, P4, P0,
// Case #122: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P3, P1, P4, EK, EI,
SH_WDG, COLOR0, EI, P0, EA, EK, P2, EC,
SH_TET, COLOR0, EF, P7, EL, EG,
SH_WDG, COLOR1, P5, P6, P3, EF, EG, EL,
SH_PYR, COLOR1, EI, EK, P6, P4, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_PYR, COLOR1, EK, EI, EA, EC, N0,
SH_TET, COLOR1, P3, P5, P6, N0,
SH_PYR, COLOR1, EK, EC, P3, P6, N0,
// Case #123: (cloned #111)
SH_TET, COLOR0, EC, ED, EK, P2,
SH_TET, COLOR0, EL, EG, EF, P7,
SH_WDG, COLOR1, P6, P3, P5, EG, EL, EF,
SH_WDG, COLOR1, EK, EC, ED, P6, P3, P0,
SH_PYR, COLOR1, P4, P5, P1, P0, P6,
SH_TET, COLOR1, P3, P0, P1, P6,
SH_TET, COLOR1, P3, P1, P5, P6,
// Case #124: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P5, P4, P2, EB, ED,
SH_WDG, COLOR0, ED, P0, EI, EB, P1, EJ,
SH_TET, COLOR0, EG, P7, EF, EL,
SH_WDG, COLOR1, P6, P3, P5, EG, EL, EF,
SH_PYR, COLOR1, ED, EB, P3, P2, N0,
SH_TET, COLOR1, P2, P3, P6, N0,
SH_TET, COLOR1, P2, P6, P4, N0,
SH_PYR, COLOR1, EI, ED, P2, P4, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_PYR, COLOR1, EB, ED, EI, EJ, N0,
SH_TET, COLOR1, P5, P6, P3, N0,
SH_PYR, COLOR1, EB, EJ, P5, P3, N0,
// Case #125: (cloned #111)
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_TET, COLOR0, EF, EL, EG, P7,
SH_WDG, COLOR1, P3, P5, P6, EL, EF, EG,
SH_WDG, COLOR1, EB, EJ, EA, P3, P5, P0,
SH_PYR, COLOR1, P2, P6, P4, P0, P3,
SH_TET, COLOR1, P5, P0, P4, P3,
SH_TET, COLOR1, P5, P4, P6, P3,
// Case #126: Unique case #20
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EL, EG, EF, P7,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_WDG, COLOR1, EL, EG, EF, P3, P6, P5,
SH_TET, COLOR1, P5, P4, P6, P3,
SH_TET, COLOR1, P3, P2, P1, P4,
SH_TET, COLOR1, P2, P3, P6, P4,
SH_TET, COLOR1, P3, P1, P5, P4,
// Case #127: Unique case #21
SH_PNT, 0, COLOR1, 7, P1, P0, P2, P3, P5, P4, P6,
SH_TET, COLOR0, EF, EL, EG, P7,
SH_WDG, COLOR1, P5, P6, P3, EF, EG, EL,
SH_TET, COLOR1, P5, P6, P3, N0,
SH_PYR, COLOR1, P4, P0, P2, P6, N0,
SH_TET, COLOR1, P6, P2, P3, N0,
SH_PYR, COLOR1, P1, P3, P2, P0, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_PYR, COLOR1, P1, P0, P4, P5, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
// Case #128: (cloned #1)
SH_PNT, 0, COLOR0, 7, P6, P4, P5, P3, P2, P0, P1,
SH_WDG, COLOR0, EG, EF, EL, P6, P5, P3,
SH_TET, COLOR0, P6, P3, P5, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_PYR, COLOR0, P2, P6, P4, P0, N0,
SH_PYR, COLOR0, P3, P2, P0, P1, N0,
SH_TET, COLOR0, P5, P3, P1, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_TET, COLOR1, P7, EF, EG, EL,
// Case #129: (cloned #24)
SH_WDG, COLOR0, P5, P6, P3, EF, EG, EL,
SH_WDG, COLOR0, EI, ED, EA, P4, P2, P1,
SH_TET, COLOR0, P6, P3, P5, P2,
SH_TET, COLOR0, P1, P3, P2, P5,
SH_TET, COLOR0, P4, P5, P1, P2,
SH_TET, COLOR0, P6, P4, P2, P5,
SH_TET, COLOR1, P0, EI, EA, ED,
SH_TET, COLOR1, P7, EF, EG, EL,
// Case #130: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EA, EG,
SH_PYR, COLOR0, P0, P2, P6, P4, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_TET, COLOR0, P3, P2, P0, N0,
SH_TET, COLOR0, P6, P2, P3, N0,
SH_PYR, COLOR0, P6, P3, EL, EG, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_PYR, COLOR0, P0, EA, EB, P3, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_PYR, COLOR1, P1, P7, EL, EB, N0,
SH_PYR, COLOR1, EJ, EF, P7, P1, N0,
SH_TET, COLOR1, EF, EG, P7, N0,
SH_TET, COLOR1, P7, EG, EL, N0,
SH_TET, COLOR1, EJ, N0, P1, EA,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #131: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EI, ED, EG, EG,
SH_PYR, COLOR0, P6, P3, EL, EG, N0,
SH_TET, COLOR0, EL, P3, EB, N0,
SH_PYR, COLOR0, P2, ED, EB, P3, N0,
SH_TET, COLOR0, P6, P2, P3, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, P4, P2, P6, N0,
SH_PYR, COLOR0, P4, EI, ED, P2, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_PYR, COLOR1, P7, EL, EB, P1, N0,
SH_PYR, COLOR1, EJ, EF, P7, P1, N0,
SH_TET, COLOR1, P7, EG, EL, N0,
SH_TET, COLOR1, EF, EG, P7, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, P1, P0, EI, EJ, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #132: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, ED, EF,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_TET, COLOR0, P6, P4, P0, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_PYR, COLOR0, P3, P5, EF, EL, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_PYR, COLOR0, P0, P3, EC, ED, N0,
SH_PYR, COLOR0, P0, ED, EK, P6, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_PYR, COLOR1, P2, P7, EG, EK, N0,
SH_PYR, COLOR1, EC, EL, P7, P2, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
SH_TET, COLOR1, P7, EF, EG, N0,
SH_TET, COLOR1, EC, N0, P2, ED,
SH_TET, COLOR1, EK, ED, P2, N0,
// Case #133: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EA, EI, EF, EF,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, EG, P6, EK, N0,
SH_PYR, COLOR0, P4, EI, EK, P6, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_PYR, COLOR0, P3, P5, EF, EL, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_PYR, COLOR0, P1, P3, EC, EA, N0,
SH_PYR, COLOR1, P7, EG, EK, P2, N0,
SH_PYR, COLOR1, EC, EL, P7, P2, N0,
SH_TET, COLOR1, P7, EF, EG, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, P2, P0, EA, EC, N0,
SH_TET, COLOR1, EA, P0, EI, N0,
// Case #134: (cloned #22)
SH_TET, COLOR0, P3, EB, EL, EC,
SH_TET, COLOR0, P6, P4, P0, P5,
SH_PYR, COLOR0, EJ, EA, P0, P5, ED,
SH_PYR, COLOR0, EF, P5, P6, EG, EK,
SH_PYR, COLOR0, P0, ED, EK, P6, P5,
SH_PYR, COLOR0, ED, EJ, EF, EK, P5,
SH_WDG, COLOR1, P1, P7, P2, EB, EL, EC,
SH_PYR, COLOR1, P1, P2, ED, EA, EJ,
SH_PYR, COLOR1, EK, P2, P7, EG, EF,
SH_PYR, COLOR1, EJ, EF, P7, P1, P2,
SH_PYR, COLOR1, EJ, ED, EK, EF, P2,
// Case #135: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EJ, EF, EG,
SH_TET, COLOR0, P3, EC, EB, EL,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_WDG, COLOR1, EB, EC, EL, P1, P2, P7,
SH_TET, COLOR1, P2, P1, P7, N0,
SH_PYR, COLOR1, P1, EJ, EF, P7, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_PYR, COLOR1, P7, EG, EK, P2, N0,
SH_TET, COLOR1, EF, EG, P7, N0,
// Case #136: (cloned #3)
SH_HEX, COLOR0, EG, P6, P2, EC, EF, P5, P1, EB,
SH_WDG, COLOR0, P1, P0, P2, P5, P4, P6,
SH_WDG, COLOR1, P3, EB, EC, P7, EF, EG,
// Case #137: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EG, EF, EI, EI,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_TET, COLOR0, P6, P4, P2, N0,
SH_PYR, COLOR0, P2, P4, EI, ED, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P6, EG, EF, P5, N0,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_PYR, COLOR1, P0, EA, EB, P3, N0,
SH_PYR, COLOR1, EC, ED, P0, P3, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_TET, COLOR1, ED, EI, P0, N0,
SH_PYR, COLOR1, EB, EF, P7, P3, N0,
SH_PYR, COLOR1, P3, P7, EG, EC, N0,
SH_TET, COLOR1, EG, P7, EF, N0,
// Case #138: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EF, EJ, EA,
SH_PYR, COLOR0, P6, P4, P0, P2, N0,
SH_TET, COLOR0, P0, P5, N0, P4,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_PYR, COLOR0, EG, P6, P2, EC, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, EA, P0, P5, EJ, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_TET, COLOR1, EF, EG, P7, N0,
SH_TET, COLOR1, P7, P3, P1, N0,
SH_PYR, COLOR1, EF, P7, P1, EJ, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
// Case #139: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EJ, EF, EG, EC, ED, EI,
SH_TET, COLOR0, P6, P4, P2, N0,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P5, P6, EG, EF, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_PYR, COLOR0, EI, ED, P2, P4, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_TET, COLOR1, P1, P7, P3, N0,
SH_PYR, COLOR1, EJ, EF, P7, P1, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
SH_PYR, COLOR1, P0, P3, EC, ED, N0,
SH_PYR, COLOR1, EC, P3, P7, EG, N0,
SH_TET, COLOR1, EG, P7, EF, N0,
// Case #140: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EG, EK, ED,
SH_PYR, COLOR0, P5, P1, P0, P4, N0,
SH_TET, COLOR0, P0, N0, P6, P4,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_PYR, COLOR0, EF, EB, P1, P5, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P6, EG, EF, P5, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_PYR, COLOR0, ED, EK, P6, P0, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_TET, COLOR1, EG, P7, EF, N0,
SH_TET, COLOR1, P7, P2, P3, N0,
SH_PYR, COLOR1, EG, EK, P2, P7, N0,
SH_TET, COLOR1, P2, EK, ED, N0,
// Case #141: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EK, EG, EF, EB, EA, EI,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P6, EG, EF, P5, N0,
SH_TET, COLOR0, EK, EG, P6, N0,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_PYR, COLOR0, EI, P4, P1, EA, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR1, P2, P3, P7, N0,
SH_PYR, COLOR1, EK, P2, P7, EG, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_TET, COLOR1, P0, P3, P2, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_PYR, COLOR1, P0, EA, EB, P3, N0,
SH_PYR, COLOR1, EB, EF, P7, P3, N0,
SH_TET, COLOR1, EF, EG, P7, N0,
// Case #142: (cloned #23)
SH_TET, COLOR0, P6, P4, P0, P5,
SH_PYR, COLOR0, EJ, EA, P0, P5, ED,
SH_PYR, COLOR0, EF, P5, P6, EG, EK,
SH_PYR, COLOR0, P0, ED, EK, P6, P5,
SH_PYR, COLOR0, ED, EJ, EF, EK, P5,
SH_TET, COLOR1, P3, P1, P7, P2,
SH_PYR, COLOR1, ED, EA, P1, P2, EJ,
SH_PYR, COLOR1, EK, P2, P7, EG, EF,
SH_PYR, COLOR1, P1, EJ, EF, P7, P2,
SH_PYR, COLOR1, ED, EK, EF, EJ, P2,
// Case #143: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EK, EI, EJ, EG, EF,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, P6, P4, EI, EK, N0,
SH_TET, COLOR0, EG, P6, EK, N0,
SH_PYR, COLOR0, EF, P5, P6, EG, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR1, P3, P2, P0, P1, N0,
SH_TET, COLOR1, P1, P7, P3, N0,
SH_TET, COLOR1, P7, P2, P3, N0,
SH_PYR, COLOR1, P7, EG, EK, P2, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_PYR, COLOR1, EJ, EF, P7, P1, N0,
SH_TET, COLOR1, EG, P7, EF, N0,
// Case #144: (cloned #6)
SH_PNT, 0, NOCOLOR, 2, EI, EL,
SH_PYR, COLOR0, P0, P1, P3, P2, N0,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_TET, COLOR0, P6, P0, P2, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_PYR, COLOR0, P3, EL, EG, P6, N0,
SH_PYR, COLOR0, P5, EF, EL, P3, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_PYR, COLOR0, P0, P6, EH, EI, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_PYR, COLOR1, P4, EH, EG, P7, N0,
SH_PYR, COLOR1, EE, P4, P7, EF, N0,
SH_TET, COLOR1, EF, P7, EL, N0,
SH_TET, COLOR1, P7, EG, EL, N0,
SH_TET, COLOR1, EE, P4, N0, EI,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #145: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EA, ED, EL, EL,
SH_PYR, COLOR0, P3, EL, EG, P6, N0,
SH_TET, COLOR0, EG, EH, P6, N0,
SH_PYR, COLOR0, P2, P6, EH, ED, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_PYR, COLOR0, P5, EF, EL, P3, N0,
SH_TET, COLOR0, P1, P3, P2, N0,
SH_PYR, COLOR0, P1, P2, ED, EA, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_PYR, COLOR1, P7, P4, EH, EG, N0,
SH_PYR, COLOR1, EE, P4, P7, EF, N0,
SH_TET, COLOR1, P7, EG, EL, N0,
SH_TET, COLOR1, EF, P7, EL, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, P4, EE, EA, P0, N0,
SH_TET, COLOR1, EA, ED, P0, N0,
// Case #146: (cloned #22)
SH_TET, COLOR0, P5, EF, EJ, EE,
SH_TET, COLOR0, P6, P0, P2, P3,
SH_PYR, COLOR0, EB, P3, P0, EA, EI,
SH_PYR, COLOR0, EL, EG, P6, P3, EH,
SH_PYR, COLOR0, P0, P6, EH, EI, P3,
SH_PYR, COLOR0, EI, EH, EL, EB, P3,
SH_WDG, COLOR1, EJ, EF, EE, P1, P7, P4,
SH_PYR, COLOR1, P1, EA, EI, P4, EB,
SH_PYR, COLOR1, EH, EG, P7, P4, EL,
SH_PYR, COLOR1, EB, P1, P7, EL, P4,
SH_PYR, COLOR1, EB, EL, EH, EI, P4,
// Case #147: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EB, EL, EG,
SH_TET, COLOR0, P5, EJ, EE, EF,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_PYR, COLOR0, P2, ED, EB, P3, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_TET, COLOR0, P2, P3, P6, N0,
SH_PYR, COLOR0, P3, EL, EG, P6, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_WDG, COLOR1, P1, P4, P7, EJ, EE, EF,
SH_TET, COLOR1, P4, P7, P1, N0,
SH_PYR, COLOR1, P1, P7, EL, EB, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, P7, P4, EH, EG, N0,
SH_TET, COLOR1, EL, P7, EG, N0,
// Case #148: (cloned #22)
SH_TET, COLOR0, P6, EG, EH, EK,
SH_TET, COLOR0, P3, P0, P1, P5,
SH_PYR, COLOR0, EE, P5, P0, EI, ED,
SH_PYR, COLOR0, EF, EL, P3, P5, EC,
SH_PYR, COLOR0, P0, P3, EC, ED, P5,
SH_PYR, COLOR0, ED, EC, EF, EE, P5,
SH_WDG, COLOR1, EH, EG, EK, P4, P7, P2,
SH_PYR, COLOR1, P4, EI, ED, P2, EE,
SH_PYR, COLOR1, EC, EL, P7, P2, EF,
SH_PYR, COLOR1, EE, P4, P7, EF, P2,
SH_PYR, COLOR1, EE, EF, EC, ED, P2,
// Case #149: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EE, EF, EL,
SH_TET, COLOR0, P6, EH, EK, EG,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_PYR, COLOR0, P5, EF, EL, P3, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_WDG, COLOR1, P4, P2, P7, EH, EK, EG,
SH_TET, COLOR1, P2, P7, P4, N0,
SH_PYR, COLOR1, P4, P7, EF, EE, N0,
SH_TET, COLOR1, P0, P2, P4, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_PYR, COLOR1, P7, P2, EC, EL, N0,
SH_TET, COLOR1, EF, P7, EL, N0,
// Case #150: (cloned #105)
SH_TET, COLOR0, EH, EG, EK, P6,
SH_TET, COLOR0, EI, ED, EA, P0,
SH_TET, COLOR0, EE, EJ, EF, P5,
SH_TET, COLOR0, EB, EC, EL, P3,
SH_WDG, COLOR1, EB, EC, EL, P1, P2, P7,
SH_TET, COLOR1, P1, P7, P2, P4,
SH_WDG, COLOR1, P4, P7, P1, EE, EF, EJ,
SH_WDG, COLOR1, P2, P4, P1, ED, EI, EA,
SH_WDG, COLOR1, P7, P4, P2, EG, EH, EK,
// Case #151: (cloned #107)
SH_TET, COLOR0, EK, EH, EG, P6,
SH_TET, COLOR0, EC, EB, P3, EL,
SH_TET, COLOR0, EE, P5, EJ, EF,
SH_WDG, COLOR1, P7, P1, P4, EF, EJ, EE,
SH_WDG, COLOR1, EC, EL, EB, P2, P7, P1,
SH_WDG, COLOR1, P2, P7, P4, EK, EG, EH,
SH_TET, COLOR1, P7, P1, P4, P2,
SH_TET, COLOR1, P2, P4, P0, P1,
// Case #152: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EC, EB, EI, EI,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_TET, COLOR0, P2, P6, P0, N0,
SH_PYR, COLOR0, P6, EH, EI, P0, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_PYR, COLOR0, P2, P1, EB, EC, N0,
SH_PYR, COLOR0, P2, EC, EG, P6, N0,
SH_PYR, COLOR1, P4, P7, EF, EE, N0,
SH_PYR, COLOR1, EG, P7, P4, EH, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_PYR, COLOR1, P7, EG, EC, P3, N0,
SH_TET, COLOR1, EC, EB, P3, N0,
// Case #153: (cloned #60)
SH_WDG, COLOR0, P6, EH, EG, P2, ED, EC,
SH_WDG, COLOR0, P5, EF, EE, P1, EB, EA,
SH_HEX, COLOR1, P0, P3, P7, P4, EA, EB, EF, EE,
SH_HEX, COLOR1, ED, EC, EG, EH, P0, P3, P7, P4,
// Case #154: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EA, EI, EH,
SH_TET, COLOR0, P5, EF, EJ, EE,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_TET, COLOR0, P2, P6, P0, N0,
SH_PYR, COLOR0, P0, P6, EH, EI, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_WDG, COLOR1, EJ, EF, EE, P1, P7, P4,
SH_TET, COLOR1, P7, P1, P4, N0,
SH_PYR, COLOR1, P1, EA, EI, P4, N0,
SH_TET, COLOR1, P3, P1, P7, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, P4, EH, EG, P7, N0,
SH_TET, COLOR1, EI, EH, P4, N0,
// Case #155: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P4, P0, P3, EG, EC,
SH_WDG, COLOR0, EG, P6, EH, EC, P2, ED,
SH_TET, COLOR0, EJ, EE, P5, EF,
SH_WDG, COLOR1, EJ, EF, EE, P1, P7, P4,
SH_PYR, COLOR1, EC, P3, P7, EG, N0,
SH_TET, COLOR1, P3, P1, P7, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_PYR, COLOR1, ED, P0, P3, EC, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EG, EH, ED, EC, N0,
SH_TET, COLOR1, P4, P7, P1, N0,
SH_PYR, COLOR1, EG, P7, P4, EH, N0,
// Case #156: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EB, EF, ED, EI, EE,
SH_TET, COLOR0, P6, EK, EG, EH,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_TET, COLOR0, P1, P0, P5, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_WDG, COLOR1, P2, P7, P4, EK, EG, EH,
SH_TET, COLOR1, P7, P4, P2, N0,
SH_PYR, COLOR1, P2, P4, EI, ED, N0,
SH_TET, COLOR1, P3, P7, P2, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_PYR, COLOR1, P4, P7, EF, EE, N0,
SH_TET, COLOR1, EI, P4, EE, N0,
// Case #157: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P4, P0, P3, EF, EB,
SH_WDG, COLOR0, EB, P1, EA, EF, P5, EE,
SH_TET, COLOR0, EK, P6, EH, EG,
SH_WDG, COLOR1, P2, P7, P4, EK, EG, EH,
SH_PYR, COLOR1, EB, EF, P7, P3, N0,
SH_TET, COLOR1, P3, P7, P2, N0,
SH_TET, COLOR1, P3, P2, P0, N0,
SH_PYR, COLOR1, EA, EB, P3, P0, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, P0, P2, P4, N0,
SH_PYR, COLOR1, EF, EB, EA, EE, N0,
SH_TET, COLOR1, P4, P2, P7, N0,
SH_PYR, COLOR1, EF, EE, P4, P7, N0,
// Case #158: (cloned #107)
SH_TET, COLOR0, EJ, EF, EE, P5,
SH_TET, COLOR0, EA, ED, P0, EI,
SH_TET, COLOR0, EG, P6, EK, EH,
SH_WDG, COLOR1, P4, P2, P7, EH, EK, EG,
SH_WDG, COLOR1, EA, EI, ED, P1, P4, P2,
SH_WDG, COLOR1, P1, P4, P7, EJ, EE, EF,
SH_TET, COLOR1, P4, P2, P7, P1,
SH_TET, COLOR1, P1, P7, P3, P2,
// Case #159: (cloned #111)
SH_TET, COLOR0, EG, EK, EH, P6,
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_WDG, COLOR1, P4, P7, P1, EE, EF, EJ,
SH_WDG, COLOR1, EH, EG, EK, P4, P7, P2,
SH_PYR, COLOR1, P0, P1, P3, P2, P4,
SH_TET, COLOR1, P7, P2, P3, P4,
SH_TET, COLOR1, P7, P3, P1, P4,
// Case #160: (cloned #3)
SH_HEX, COLOR0, EL, P3, P1, EJ, EG, P6, P4, EE,
SH_WDG, COLOR0, P3, P2, P6, P1, P0, P4,
SH_WDG, COLOR1, P7, EL, EG, P5, EJ, EE,
// Case #161: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EG, EL, ED, ED,
SH_PYR, COLOR0, P2, ED, EA, P1, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, P3, P1, EJ, EL, N0,
SH_TET, COLOR0, P2, P1, P3, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR0, P6, P4, P2, N0,
SH_PYR, COLOR0, P4, EI, ED, P2, N0,
SH_TET, COLOR0, P6, P2, P3, N0,
SH_PYR, COLOR0, P6, P3, EL, EG, N0,
SH_PYR, COLOR0, P6, EG, EE, P4, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EE, P5, P0, EI, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
SH_PYR, COLOR1, EJ, P5, P7, EL, N0,
SH_PYR, COLOR1, P5, EE, EG, P7, N0,
SH_TET, COLOR1, EG, EL, P7, N0,
// Case #162: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EL, EB, EA,
SH_PYR, COLOR0, P6, P4, P0, P2, N0,
SH_TET, COLOR0, P0, N0, P3, P2,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_PYR, COLOR0, EG, EE, P4, P6, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_PYR, COLOR0, P3, EL, EG, P6, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_PYR, COLOR0, EA, EB, P3, P0, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_TET, COLOR1, EL, P7, EG, N0,
SH_TET, COLOR1, P7, P1, P5, N0,
SH_PYR, COLOR1, EL, EB, P1, P7, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
// Case #163: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EB, EL, EG, EE, EI, ED,
SH_TET, COLOR0, P6, P4, P2, N0,
SH_PYR, COLOR0, P6, EG, EE, P4, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_PYR, COLOR0, P3, EL, EG, P6, N0,
SH_TET, COLOR0, EB, EL, P3, N0,
SH_PYR, COLOR0, P3, P2, ED, EB, N0,
SH_PYR, COLOR0, ED, P2, P4, EI, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR1, P1, P5, P7, N0,
SH_PYR, COLOR1, EB, P1, P7, EL, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EE, EG, P7, P5, N0,
SH_TET, COLOR1, EG, EL, P7, N0,
// Case #164: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EE, EJ, ED, ED,
SH_PYR, COLOR0, P0, P3, EC, ED, N0,
SH_TET, COLOR0, EC, P3, EL, N0,
SH_PYR, COLOR0, P1, EJ, EL, P3, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_TET, COLOR0, P4, P0, P6, N0,
SH_PYR, COLOR0, P6, P0, ED, EK, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_PYR, COLOR0, P4, P6, EG, EE, N0,
SH_PYR, COLOR1, P2, EC, EL, P7, N0,
SH_PYR, COLOR1, EG, EK, P2, P7, N0,
SH_TET, COLOR1, P2, ED, EC, N0,
SH_TET, COLOR1, EK, ED, P2, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_PYR, COLOR1, P7, P5, EE, EG, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #165: (cloned #60)
SH_WDG, COLOR0, P4, EI, EE, P6, EK, EG,
SH_WDG, COLOR0, P1, EJ, EA, P3, EL, EC,
SH_HEX, COLOR1, EA, EJ, EL, EC, P0, P5, P7, P2,
SH_HEX, COLOR1, P0, P5, P7, P2, EI, EE, EG, EK,
// Case #166: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EA, ED, EK,
SH_TET, COLOR0, P3, EB, EL, EC,
SH_PYR, COLOR0, P6, EG, EE, P4, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_TET, COLOR0, P4, P0, P6, N0,
SH_PYR, COLOR0, P0, ED, EK, P6, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_WDG, COLOR1, P1, P7, P2, EB, EL, EC,
SH_TET, COLOR1, P7, P2, P1, N0,
SH_PYR, COLOR1, P1, P2, ED, EA, N0,
SH_TET, COLOR1, P5, P7, P1, N0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, P2, P7, EG, EK, N0,
SH_TET, COLOR1, ED, P2, EK, N0,
// Case #167: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P2, P0, P5, EG, EE,
SH_WDG, COLOR0, EE, P4, EI, EG, P6, EK,
SH_TET, COLOR0, EB, P3, EC, EL,
SH_WDG, COLOR1, P1, P7, P2, EB, EL, EC,
SH_PYR, COLOR1, EE, EG, P7, P5, N0,
SH_TET, COLOR1, P5, P7, P1, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_PYR, COLOR1, EK, EI, P0, P2, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_PYR, COLOR1, EG, EE, EI, EK, N0,
SH_TET, COLOR1, P2, P1, P7, N0,
SH_PYR, COLOR1, EG, EK, P2, P7, N0,
// Case #168: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EB, EJ, EE,
SH_PYR, COLOR0, P2, P6, P4, P0, N0,
SH_TET, COLOR0, P4, N0, P1, P0,
SH_PYR, COLOR0, P6, EG, EE, P4, N0,
SH_PYR, COLOR0, EC, EG, P6, P2, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P2, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, EE, EJ, P1, P4, N0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_PYR, COLOR1, EC, P3, P7, EG, N0,
SH_TET, COLOR1, EB, P3, EC, N0,
SH_TET, COLOR1, P3, P5, P7, N0,
SH_PYR, COLOR1, EB, EJ, P5, P3, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
// Case #169: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EC, ED, EI,
SH_TET, COLOR0, P1, EJ, EB, EA,
SH_PYR, COLOR0, P4, P6, EG, EE, N0,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_TET, COLOR0, P6, P4, P2, N0,
SH_PYR, COLOR0, P2, P4, EI, ED, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_WDG, COLOR1, EB, EJ, EA, P3, P5, P0,
SH_TET, COLOR1, P5, P3, P0, N0,
SH_PYR, COLOR1, P3, EC, ED, P0, N0,
SH_TET, COLOR1, P7, P3, P5, N0,
SH_PYR, COLOR1, EE, EG, P7, P5, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_TET, COLOR1, ED, EI, P0, N0,
// Case #170: (cloned #15)
SH_HEX, COLOR0, EA, EE, EG, EC, P0, P4, P6, P2,
SH_HEX, COLOR1, P1, P5, P7, P3, EA, EE, EG, EC,
// Case #171: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EC, EI, ED,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_TET, COLOR0, P2, ED, EC, N0,
SH_PYR, COLOR0, P4, P6, EG, EE, N0,
SH_TET, COLOR0, EI, P4, EE, N0,
SH_PYR, COLOR0, ED, P2, P4, EI, N0,
SH_TET, COLOR0, P2, P6, P4, N0,
SH_PYR, COLOR1, P1, P5, P7, P3, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EE, EG, P7, P5, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_PYR, COLOR1, EC, ED, P0, P3, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #172: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EB, EJ, EE, EG, EK, ED,
SH_TET, COLOR0, P4, P0, P6, N0,
SH_PYR, COLOR0, P4, P6, EG, EE, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, EB, P1, EJ, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_PYR, COLOR0, ED, EK, P6, P0, N0,
SH_TET, COLOR0, P6, EK, EG, N0,
SH_TET, COLOR1, P3, P5, P7, N0,
SH_PYR, COLOR1, EB, EJ, P5, P3, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_TET, COLOR1, P2, P3, P7, N0,
SH_TET, COLOR1, P2, EK, ED, N0,
SH_PYR, COLOR1, P2, P7, EG, EK, N0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #173: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P0, P2, P7, EE, EG,
SH_WDG, COLOR0, EE, P4, EI, EG, P6, EK,
SH_TET, COLOR0, EB, EA, P1, EJ,
SH_WDG, COLOR1, EB, EJ, EA, P3, P5, P0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_TET, COLOR1, P7, P3, P5, N0,
SH_TET, COLOR1, P7, P2, P3, N0,
SH_PYR, COLOR1, EK, P2, P7, EG, N0,
SH_PYR, COLOR1, EI, P0, P2, EK, N0,
SH_TET, COLOR1, P2, P0, P3, N0,
SH_PYR, COLOR1, EE, EI, EK, EG, N0,
SH_TET, COLOR1, P0, P5, P3, N0,
SH_PYR, COLOR1, EE, P5, P0, EI, N0,
// Case #174: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EA, EK, ED,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, P6, EG, EE, P4, N0,
SH_TET, COLOR0, EK, EG, P6, N0,
SH_PYR, COLOR0, ED, EK, P6, P0, N0,
SH_TET, COLOR0, P0, P6, P4, N0,
SH_PYR, COLOR1, P3, P1, P5, P7, N0,
SH_TET, COLOR1, P1, P3, P2, N0,
SH_TET, COLOR1, P2, P3, P7, N0,
SH_PYR, COLOR1, P2, P7, EG, EK, N0,
SH_PYR, COLOR1, EG, P7, P5, EE, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, EA, P1, P2, ED, N0,
SH_TET, COLOR1, EK, ED, P2, N0,
// Case #175: (cloned #63)
SH_WDG, COLOR0, P4, EI, EE, P6, EK, EG,
SH_HEX, COLOR1, P0, P5, P7, P2, EI, EE, EG, EK,
SH_WDG, COLOR1, P3, P2, P7, P1, P0, P5,
// Case #176: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EG, EH, EI,
SH_PYR, COLOR0, P3, P2, P0, P1, N0,
SH_TET, COLOR0, P0, P6, N0, P2,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_PYR, COLOR0, EL, P3, P1, EJ, N0,
SH_TET, COLOR0, P6, P2, P3, N0,
SH_PYR, COLOR0, P6, P3, EL, EG, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_PYR, COLOR0, EI, P0, P6, EH, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_TET, COLOR1, EG, EL, P7, N0,
SH_TET, COLOR1, P7, P5, P4, N0,
SH_PYR, COLOR1, EG, P7, P4, EH, N0,
SH_TET, COLOR1, P4, EI, EH, N0,
// Case #177: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EH, EG, EL, EJ, EA, ED,
SH_TET, COLOR0, P3, P2, P1, N0,
SH_PYR, COLOR0, P3, P1, EJ, EL, N0,
SH_TET, COLOR0, P6, P2, P3, N0,
SH_PYR, COLOR0, P6, P3, EL, EG, N0,
SH_TET, COLOR0, EH, P6, EG, N0,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_PYR, COLOR0, ED, EA, P1, P2, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_PYR, COLOR1, EH, EG, P7, P4, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EJ, P5, P7, EL, N0,
SH_TET, COLOR1, EL, P7, EG, N0,
// Case #178: (cloned #23)
SH_TET, COLOR0, P6, P0, P2, P3,
SH_PYR, COLOR0, EB, P3, P0, EA, EI,
SH_PYR, COLOR0, EL, EG, P6, P3, EH,
SH_PYR, COLOR0, P0, P6, EH, EI, P3,
SH_PYR, COLOR0, EI, EH, EL, EB, P3,
SH_TET, COLOR1, P5, P7, P1, P4,
SH_PYR, COLOR1, EI, P4, P1, EA, EB,
SH_PYR, COLOR1, EH, EG, P7, P4, EL,
SH_PYR, COLOR1, P1, P7, EL, EB, P4,
SH_PYR, COLOR1, EI, EB, EL, EH, P4,
// Case #179: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EH, ED, EB, EG, EL,
SH_PYR, COLOR0, P2, ED, EB, P3, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_PYR, COLOR0, P6, EH, ED, P2, N0,
SH_TET, COLOR0, EG, EH, P6, N0,
SH_PYR, COLOR0, EL, EG, P6, P3, N0,
SH_TET, COLOR0, P3, P6, P2, N0,
SH_PYR, COLOR1, P5, P1, P0, P4, N0,
SH_TET, COLOR1, P1, P5, P7, N0,
SH_TET, COLOR1, P7, P5, P4, N0,
SH_PYR, COLOR1, P7, P4, EH, EG, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, EB, P1, P7, EL, N0,
SH_TET, COLOR1, EG, EL, P7, N0,
// Case #180: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EI, ED, EC,
SH_TET, COLOR0, P6, EG, EH, EK,
SH_PYR, COLOR0, P3, P1, EJ, EL, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR0, P0, P3, EC, ED, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_WDG, COLOR1, EH, EG, EK, P4, P7, P2,
SH_TET, COLOR1, P7, P4, P2, N0,
SH_PYR, COLOR1, P4, EI, ED, P2, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, P2, EC, EL, P7, N0,
SH_TET, COLOR1, ED, EC, P2, N0,
// Case #181: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P2, P0, P5, EL, EJ,
SH_WDG, COLOR0, EL, P3, EC, EJ, P1, EA,
SH_TET, COLOR0, EH, EK, P6, EG,
SH_WDG, COLOR1, EH, EG, EK, P4, P7, P2,
SH_PYR, COLOR1, EJ, P5, P7, EL, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_TET, COLOR1, P5, P0, P4, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_TET, COLOR1, P0, P2, P4, N0,
SH_PYR, COLOR1, EL, EC, EA, EJ, N0,
SH_TET, COLOR1, P2, P7, P4, N0,
SH_PYR, COLOR1, EL, P7, P2, EC, N0,
// Case #182: (cloned #107)
SH_TET, COLOR0, EB, EC, EL, P3,
SH_TET, COLOR0, EA, P0, EI, ED,
SH_TET, COLOR0, EG, EH, P6, EK,
SH_WDG, COLOR1, EK, EH, EG, P2, P4, P7,
SH_WDG, COLOR1, P1, P2, P4, EA, ED, EI,
SH_WDG, COLOR1, EB, EC, EL, P1, P2, P7,
SH_TET, COLOR1, P2, P7, P4, P1,
SH_TET, COLOR1, P1, P5, P7, P4,
// Case #183: (cloned #111)
SH_TET, COLOR0, EG, EK, EH, P6,
SH_TET, COLOR0, EL, EB, EC, P3,
SH_WDG, COLOR1, EC, EL, EB, P2, P7, P1,
SH_WDG, COLOR1, P2, P7, P4, EK, EG, EH,
SH_PYR, COLOR1, P0, P4, P5, P1, P2,
SH_TET, COLOR1, P7, P5, P4, P2,
SH_TET, COLOR1, P7, P1, P5, P2,
// Case #184: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EJ, EB, EC, EG, EH, EI,
SH_TET, COLOR0, P2, P6, P0, N0,
SH_PYR, COLOR0, P2, EC, EG, P6, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P2, N0,
SH_TET, COLOR0, EJ, EB, P1, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_PYR, COLOR0, EI, P0, P6, EH, N0,
SH_TET, COLOR0, P6, EG, EH, N0,
SH_TET, COLOR1, P5, P7, P3, N0,
SH_PYR, COLOR1, EJ, P5, P3, EB, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_TET, COLOR1, P4, EI, EH, N0,
SH_PYR, COLOR1, P4, EH, EG, P7, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_TET, COLOR1, EC, EB, P3, N0,
// Case #185: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P0, P4, P7, EC, EG,
SH_WDG, COLOR0, EG, P6, EH, EC, P2, ED,
SH_TET, COLOR0, EJ, P1, EA, EB,
SH_WDG, COLOR1, P5, P3, P0, EJ, EB, EA,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_TET, COLOR1, P7, P3, P5, N0,
SH_TET, COLOR1, P7, P5, P4, N0,
SH_PYR, COLOR1, EH, EG, P7, P4, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_PYR, COLOR1, EC, EG, EH, ED, N0,
SH_TET, COLOR1, P0, P5, P3, N0,
SH_PYR, COLOR1, EC, ED, P0, P3, N0,
// Case #186: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EA, EH, EI,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, P6, P2, EC, EG, N0,
SH_TET, COLOR0, EH, P6, EG, N0,
SH_PYR, COLOR0, EI, P0, P6, EH, N0,
SH_TET, COLOR0, P0, P2, P6, N0,
SH_PYR, COLOR1, P5, P7, P3, P1, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_PYR, COLOR1, P4, EH, EG, P7, N0,
SH_PYR, COLOR1, EG, EC, P3, P7, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #187: (cloned #63)
SH_WDG, COLOR0, P6, EH, EG, P2, ED, EC,
SH_HEX, COLOR1, ED, EC, EG, EH, P0, P3, P7, P4,
SH_WDG, COLOR1, P1, P0, P3, P5, P4, P7,
// Case #188: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P4, P5, P3, ED, EB,
SH_WDG, COLOR0, ED, P0, EI, EB, P1, EJ,
SH_TET, COLOR0, EG, EH, P6, EK,
SH_WDG, COLOR1, EG, EK, EH, P7, P2, P4,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_TET, COLOR1, P3, P7, P2, N0,
SH_TET, COLOR1, P3, P5, P7, N0,
SH_PYR, COLOR1, EJ, P5, P3, EB, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, ED, EI, EJ, EB, N0,
SH_TET, COLOR1, P4, P2, P7, N0,
SH_PYR, COLOR1, ED, P2, P4, EI, N0,
// Case #189: (cloned #126)
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_TET, COLOR0, EH, EG, EK, P6,
SH_WDG, COLOR1, P0, P5, P3, EA, EJ, EB,
SH_WDG, COLOR1, P4, P2, P7, EH, EK, EG,
SH_TET, COLOR1, P7, P2, P3, P4,
SH_TET, COLOR1, P4, P5, P0, P3,
SH_TET, COLOR1, P0, P2, P4, P3,
SH_TET, COLOR1, P4, P7, P5, P3,
// Case #190: (cloned #111)
SH_TET, COLOR0, EI, ED, EA, P0,
SH_TET, COLOR0, EH, EG, EK, P6,
SH_WDG, COLOR1, EK, EH, EG, P2, P4, P7,
SH_WDG, COLOR1, P2, P4, P1, ED, EI, EA,
SH_PYR, COLOR1, P3, P1, P5, P7, P2,
SH_TET, COLOR1, P4, P5, P1, P2,
SH_TET, COLOR1, P4, P7, P5, P2,
// Case #191: (cloned #127)
SH_PNT, 0, COLOR1, 7, P5, P1, P0, P4, P7, P3, P2,
SH_TET, COLOR0, EG, EK, EH, P6,
SH_WDG, COLOR1, EG, EK, EH, P7, P2, P4,
SH_TET, COLOR1, P7, P4, P2, N0,
SH_PYR, COLOR1, P3, P2, P0, P1, N0,
SH_TET, COLOR1, P2, P4, P0, N0,
SH_PYR, COLOR1, P5, P1, P0, P4, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, P5, P7, P3, P1, N0,
SH_TET, COLOR1, P7, P2, P3, N0,
// Case #192: (cloned #3)
SH_HEX, COLOR0, EF, P5, P4, EH, EL, P3, P2, EK,
SH_WDG, COLOR0, P5, P1, P3, P4, P0, P2,
SH_WDG, COLOR1, P7, EF, EL, P6, EH, EK,
// Case #193: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EL, EF, EA, EA,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_TET, COLOR0, EI, EH, P4, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_TET, COLOR0, P3, P2, P1, N0,
SH_PYR, COLOR0, P2, ED, EA, P1, N0,
SH_TET, COLOR0, P3, P1, P5, N0,
SH_PYR, COLOR0, P3, P5, EF, EL, N0,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_PYR, COLOR1, P0, P6, EH, EI, N0,
SH_PYR, COLOR1, EK, P6, P0, ED, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_TET, COLOR1, ED, P0, EA, N0,
SH_PYR, COLOR1, EH, P6, P7, EF, N0,
SH_PYR, COLOR1, P6, EK, EL, P7, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
// Case #194: (cloned #25)
SH_PNT, 0, NOCOLOR, 4, EK, EH, EA, EA,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, P4, EH, EF, P5, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_TET, COLOR0, P2, P0, P3, N0,
SH_PYR, COLOR0, P3, P0, EA, EB, N0,
SH_TET, COLOR0, P2, P4, P0, N0,
SH_PYR, COLOR0, P2, EK, EH, P4, N0,
SH_PYR, COLOR0, P2, P3, EL, EK, N0,
SH_PYR, COLOR1, P1, EJ, EF, P7, N0,
SH_PYR, COLOR1, EL, EB, P1, P7, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_PYR, COLOR1, P7, P6, EK, EL, N0,
SH_TET, COLOR1, EK, P6, EH, N0,
// Case #195: (cloned #60)
SH_WDG, COLOR0, P2, ED, EK, P3, EB, EL,
SH_WDG, COLOR0, P4, EH, EI, P5, EF, EJ,
SH_HEX, COLOR1, EI, EH, EF, EJ, P0, P6, P7, P1,
SH_HEX, COLOR1, P0, P6, P7, P1, ED, EK, EL, EB,
// Case #196: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EF, EH, EL, EC, ED,
SH_PYR, COLOR0, P5, P1, P0, P4, N0,
SH_TET, COLOR0, P0, P3, N0, P1,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_PYR, COLOR0, EF, P5, P4, EH, N0,
SH_TET, COLOR0, P3, P1, P5, N0,
SH_PYR, COLOR0, P3, P5, EF, EL, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_PYR, COLOR0, ED, P0, P3, EC, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
SH_TET, COLOR1, P7, P6, P2, N0,
SH_PYR, COLOR1, EL, P7, P2, EC, N0,
SH_TET, COLOR1, P2, ED, EC, N0,
// Case #197: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EC, EL, EF, EH, EI, EA,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P3, P1, P5, N0,
SH_PYR, COLOR0, P3, P5, EF, EL, N0,
SH_TET, COLOR0, EC, P3, EL, N0,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_PYR, COLOR0, EA, EI, P4, P1, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR1, P2, P7, P6, N0,
SH_PYR, COLOR1, EC, EL, P7, P2, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_TET, COLOR1, P0, P2, P6, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_PYR, COLOR1, P0, P6, EH, EI, N0,
SH_PYR, COLOR1, EH, P6, P7, EF, N0,
SH_TET, COLOR1, EF, P7, EL, N0,
// Case #198: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EH, EF, ED, EA, EJ,
SH_TET, COLOR0, P3, EL, EC, EB,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_TET, COLOR0, P4, P5, P0, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_WDG, COLOR1, EC, EL, EB, P2, P7, P1,
SH_TET, COLOR1, P7, P2, P1, N0,
SH_PYR, COLOR1, P2, ED, EA, P1, N0,
SH_TET, COLOR1, P6, P2, P7, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, P1, EJ, EF, P7, N0,
SH_TET, COLOR1, EA, EJ, P1, N0,
// Case #199: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P1, P0, P6, EF, EH,
SH_WDG, COLOR0, EF, P5, EJ, EH, P4, EI,
SH_TET, COLOR0, EC, EB, P3, EL,
SH_WDG, COLOR1, EC, EL, EB, P2, P7, P1,
SH_PYR, COLOR1, EH, P6, P7, EF, N0,
SH_TET, COLOR1, P6, P2, P7, N0,
SH_TET, COLOR1, P6, P0, P2, N0,
SH_PYR, COLOR1, EI, P0, P6, EH, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_PYR, COLOR1, EF, EJ, EI, EH, N0,
SH_TET, COLOR1, P1, P7, P2, N0,
SH_PYR, COLOR1, EF, P7, P1, EJ, N0,
// Case #200: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EH, EF, EK, EC, EB,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_TET, COLOR0, P1, N0, P2, P0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_PYR, COLOR0, EH, EF, P5, P4, N0,
SH_TET, COLOR0, P2, P4, P0, N0,
SH_PYR, COLOR0, P2, EK, EH, P4, N0,
SH_TET, COLOR0, P2, EC, EK, N0,
SH_PYR, COLOR0, EB, EC, P2, P1, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_PYR, COLOR1, EH, P6, P7, EF, N0,
SH_TET, COLOR1, EK, P6, EH, N0,
SH_TET, COLOR1, P6, P3, P7, N0,
SH_PYR, COLOR1, EK, EC, P3, P6, N0,
SH_TET, COLOR1, P3, EC, EB, N0,
// Case #201: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EH, EI, EA,
SH_TET, COLOR0, P2, EC, EK, ED,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_WDG, COLOR1, EK, EC, ED, P6, P3, P0,
SH_TET, COLOR1, P3, P6, P0, N0,
SH_PYR, COLOR1, P6, EH, EI, P0, N0,
SH_TET, COLOR1, P7, P6, P3, N0,
SH_PYR, COLOR1, EB, EF, P7, P3, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_PYR, COLOR1, P0, EA, EB, P3, N0,
SH_TET, COLOR1, EI, EA, P0, N0,
// Case #202: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EC, EK, EH, EF, EJ, EA,
SH_TET, COLOR0, P4, P5, P0, N0,
SH_PYR, COLOR0, P4, EH, EF, P5, N0,
SH_TET, COLOR0, P2, P4, P0, N0,
SH_PYR, COLOR0, P2, EK, EH, P4, N0,
SH_TET, COLOR0, EC, EK, P2, N0,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_PYR, COLOR0, EA, P0, P5, EJ, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR1, P3, P7, P6, N0,
SH_PYR, COLOR1, EC, P3, P6, EK, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_TET, COLOR1, P1, P7, P3, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_PYR, COLOR1, P1, EJ, EF, P7, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_TET, COLOR1, EH, EK, P6, N0,
// Case #203: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P0, P1, P7, EH, EF,
SH_WDG, COLOR0, EF, P5, EJ, EH, P4, EI,
SH_TET, COLOR0, EC, P2, ED, EK,
SH_WDG, COLOR1, P3, P6, P0, EC, EK, ED,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_TET, COLOR1, P7, P6, P3, N0,
SH_TET, COLOR1, P7, P3, P1, N0,
SH_PYR, COLOR1, EJ, EF, P7, P1, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_PYR, COLOR1, EH, EF, EJ, EI, N0,
SH_TET, COLOR1, P0, P3, P6, N0,
SH_PYR, COLOR1, EH, EI, P0, P6, N0,
// Case #204: (cloned #15)
SH_HEX, COLOR0, ED, EB, EF, EH, P0, P1, P5, P4,
SH_HEX, COLOR1, P2, P3, P7, P6, ED, EB, EF, EH,
// Case #205: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EB, EF, EH, EA, EI,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, EI, P4, P1, EA, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_PYR, COLOR1, P2, P3, P7, P6, N0,
SH_TET, COLOR1, P6, P0, P2, N0,
SH_TET, COLOR1, P0, P3, P2, N0,
SH_PYR, COLOR1, P0, EA, EB, P3, N0,
SH_PYR, COLOR1, EB, EF, P7, P3, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_PYR, COLOR1, EH, EI, P0, P6, N0,
SH_TET, COLOR1, EA, P0, EI, N0,
// Case #206: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EH, ED, EJ, EA,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, EA, P0, P5, EJ, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_PYR, COLOR1, P3, P7, P6, P2, N0,
SH_TET, COLOR1, P2, P1, P3, N0,
SH_TET, COLOR1, P1, P7, P3, N0,
SH_PYR, COLOR1, P1, EJ, EF, P7, N0,
SH_PYR, COLOR1, EF, EH, P6, P7, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, ED, EA, P1, P2, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #207: (cloned #63)
SH_WDG, COLOR0, P5, EJ, EF, P4, EI, EH,
SH_HEX, COLOR1, EI, EH, EF, EJ, P0, P6, P7, P1,
SH_WDG, COLOR1, P2, P0, P6, P3, P1, P7,
// Case #208: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EF, EE, EI,
SH_PYR, COLOR0, P3, P2, P0, P1, N0,
SH_TET, COLOR0, P0, N0, P5, P1,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_PYR, COLOR0, EL, EK, P2, P3, N0,
SH_TET, COLOR0, P5, P3, P1, N0,
SH_PYR, COLOR0, P5, EF, EL, P3, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, EI, EE, P5, P0, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_TET, COLOR1, EF, P7, EL, N0,
SH_TET, COLOR1, P7, P4, P6, N0,
SH_PYR, COLOR1, EF, EE, P4, P7, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
// Case #209: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EE, EF, EL, EK, ED, EA,
SH_TET, COLOR0, P3, P2, P1, N0,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_TET, COLOR0, P5, P3, P1, N0,
SH_PYR, COLOR0, P5, EF, EL, P3, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_PYR, COLOR0, EA, P1, P2, ED, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_TET, COLOR1, P4, P6, P7, N0,
SH_PYR, COLOR1, EE, P4, P7, EF, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, P0, P6, P4, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_PYR, COLOR1, P0, ED, EK, P6, N0,
SH_PYR, COLOR1, EK, EL, P7, P6, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
// Case #210: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EI, EA, EB,
SH_TET, COLOR0, P5, EE, EF, EJ,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_TET, COLOR0, P2, P0, P3, N0,
SH_PYR, COLOR0, P0, EA, EB, P3, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_WDG, COLOR1, P4, P7, P1, EE, EF, EJ,
SH_TET, COLOR1, P7, P1, P4, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_TET, COLOR1, P6, P7, P4, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, P1, P7, EL, EB, N0,
SH_TET, COLOR1, EA, P1, EB, N0,
// Case #211: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P1, P0, P6, EL, EK,
SH_WDG, COLOR0, EK, P2, ED, EL, P3, EB,
SH_TET, COLOR0, EE, P5, EJ, EF,
SH_WDG, COLOR1, P4, P7, P1, EE, EF, EJ,
SH_PYR, COLOR1, EK, EL, P7, P6, N0,
SH_TET, COLOR1, P6, P7, P4, N0,
SH_TET, COLOR1, P6, P4, P0, N0,
SH_PYR, COLOR1, ED, EK, P6, P0, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EL, EK, ED, EB, N0,
SH_TET, COLOR1, P1, P4, P7, N0,
SH_PYR, COLOR1, EL, EB, P1, P7, N0,
// Case #212: (cloned #23)
SH_TET, COLOR0, P3, P0, P1, P5,
SH_PYR, COLOR0, EE, P5, P0, EI, ED,
SH_PYR, COLOR0, EF, EL, P3, P5, EC,
SH_PYR, COLOR0, P0, P3, EC, ED, P5,
SH_PYR, COLOR0, ED, EC, EF, EE, P5,
SH_TET, COLOR1, P6, P7, P4, P2,
SH_PYR, COLOR1, ED, P2, P4, EI, EE,
SH_PYR, COLOR1, EC, EL, P7, P2, EF,
SH_PYR, COLOR1, P4, P7, EF, EE, P2,
SH_PYR, COLOR1, ED, EE, EF, EC, P2,
// Case #213: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EC, EA, EE, EL, EF,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, P3, EC, EA, P1, N0,
SH_TET, COLOR0, EL, EC, P3, N0,
SH_PYR, COLOR0, EF, EL, P3, P5, N0,
SH_TET, COLOR0, P5, P3, P1, N0,
SH_PYR, COLOR1, P6, P4, P0, P2, N0,
SH_TET, COLOR1, P4, P6, P7, N0,
SH_TET, COLOR1, P7, P6, P2, N0,
SH_PYR, COLOR1, P7, P2, EC, EL, N0,
SH_PYR, COLOR1, EC, P2, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_PYR, COLOR1, EE, P4, P7, EF, N0,
SH_TET, COLOR1, EL, EF, P7, N0,
// Case #214: (cloned #107)
SH_TET, COLOR0, EE, EJ, EF, P5,
SH_TET, COLOR0, EI, P0, ED, EA,
SH_TET, COLOR0, EL, EC, P3, EB,
SH_WDG, COLOR1, EB, EC, EL, P1, P2, P7,
SH_WDG, COLOR1, P4, P1, P2, EI, EA, ED,
SH_WDG, COLOR1, EE, EJ, EF, P4, P1, P7,
SH_TET, COLOR1, P1, P7, P2, P4,
SH_TET, COLOR1, P4, P6, P7, P2,
// Case #215: (cloned #111)
SH_TET, COLOR0, EL, EB, EC, P3,
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_WDG, COLOR1, EJ, EF, EE, P1, P7, P4,
SH_WDG, COLOR1, P1, P7, P2, EB, EL, EC,
SH_PYR, COLOR1, P0, P2, P6, P4, P1,
SH_TET, COLOR1, P7, P6, P2, P1,
SH_TET, COLOR1, P7, P4, P6, P1,
// Case #216: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EK, EC, EB, EF, EE, EI,
SH_TET, COLOR0, P1, P0, P5, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_PYR, COLOR0, P2, P1, EB, EC, N0,
SH_TET, COLOR0, EK, P2, EC, N0,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_PYR, COLOR0, EI, EE, P5, P0, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR1, P6, P3, P7, N0,
SH_PYR, COLOR1, EK, EC, P3, P6, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_TET, COLOR1, P4, P6, P7, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_PYR, COLOR1, P4, P7, EF, EE, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_TET, COLOR1, EB, P3, EC, N0,
// Case #217: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P0, P4, P7, EB, EF,
SH_WDG, COLOR0, EB, P1, EA, EF, P5, EE,
SH_TET, COLOR0, EK, ED, P2, EC,
SH_WDG, COLOR1, EK, EC, ED, P6, P3, P0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_TET, COLOR1, P7, P6, P3, N0,
SH_TET, COLOR1, P7, P4, P6, N0,
SH_PYR, COLOR1, EE, P4, P7, EF, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_TET, COLOR1, P4, P0, P6, N0,
SH_PYR, COLOR1, EB, EA, EE, EF, N0,
SH_TET, COLOR1, P0, P3, P6, N0,
SH_PYR, COLOR1, EB, P3, P0, EA, N0,
// Case #218: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P1, P3, P6, EI, EK,
SH_WDG, COLOR0, EI, P0, EA, EK, P2, EC,
SH_TET, COLOR0, EF, EJ, P5, EE,
SH_WDG, COLOR1, EF, EE, EJ, P7, P4, P1,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_TET, COLOR1, P6, P7, P4, N0,
SH_TET, COLOR1, P6, P3, P7, N0,
SH_PYR, COLOR1, EC, P3, P6, EK, N0,
SH_PYR, COLOR1, EA, P1, P3, EC, N0,
SH_TET, COLOR1, P3, P1, P7, N0,
SH_PYR, COLOR1, EI, EA, EC, EK, N0,
SH_TET, COLOR1, P1, P4, P7, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
// Case #219: (cloned #126)
SH_TET, COLOR0, EC, ED, EK, P2,
SH_TET, COLOR0, EJ, EF, EE, P5,
SH_WDG, COLOR1, P0, P3, P6, ED, EC, EK,
SH_WDG, COLOR1, P1, P4, P7, EJ, EE, EF,
SH_TET, COLOR1, P7, P4, P6, P1,
SH_TET, COLOR1, P1, P3, P0, P6,
SH_TET, COLOR1, P0, P4, P1, P6,
SH_TET, COLOR1, P1, P7, P3, P6,
// Case #220: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EB, ED, EE, EI,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, EI, EE, P5, P0, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_PYR, COLOR1, P6, P2, P3, P7, N0,
SH_TET, COLOR1, P2, P6, P4, N0,
SH_TET, COLOR1, P4, P6, P7, N0,
SH_PYR, COLOR1, P4, P7, EF, EE, N0,
SH_PYR, COLOR1, EF, P7, P3, EB, N0,
SH_PYR, COLOR1, EB, P3, P2, ED, N0,
SH_PYR, COLOR1, ED, P2, P4, EI, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #221: (cloned #63)
SH_WDG, COLOR0, P1, EA, EB, P5, EE, EF,
SH_HEX, COLOR1, P0, P3, P7, P4, EA, EB, EF, EE,
SH_WDG, COLOR1, P6, P4, P7, P2, P0, P3,
// Case #222: (cloned #111)
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EJ, EF, EE, P5,
SH_WDG, COLOR1, EE, EJ, EF, P4, P1, P7,
SH_WDG, COLOR1, P4, P1, P2, EI, EA, ED,
SH_PYR, COLOR1, P6, P2, P3, P7, P4,
SH_TET, COLOR1, P1, P3, P2, P4,
SH_TET, COLOR1, P1, P7, P3, P4,
// Case #223: (cloned #127)
SH_PNT, 0, COLOR1, 7, P3, P2, P0, P1, P7, P6, P4,
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_WDG, COLOR1, EF, EE, EJ, P7, P4, P1,
SH_TET, COLOR1, P7, P1, P4, N0,
SH_PYR, COLOR1, P6, P4, P0, P2, N0,
SH_TET, COLOR1, P4, P1, P0, N0,
SH_PYR, COLOR1, P3, P2, P0, P1, N0,
SH_TET, COLOR1, P3, P1, P7, N0,
SH_PYR, COLOR1, P3, P7, P6, P2, N0,
SH_TET, COLOR1, P7, P4, P6, N0,
// Case #224: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EH, EE, EJ,
SH_PYR, COLOR0, P2, P0, P1, P3, N0,
SH_TET, COLOR0, P1, P4, N0, P0,
SH_PYR, COLOR0, P3, P1, EJ, EL, N0,
SH_PYR, COLOR0, EK, P2, P3, EL, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_PYR, COLOR0, P4, P2, EK, EH, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR0, EJ, P1, P4, EE, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_PYR, COLOR1, EK, EL, P7, P6, N0,
SH_TET, COLOR1, EH, EK, P6, N0,
SH_TET, COLOR1, P6, P7, P5, N0,
SH_PYR, COLOR1, EH, P6, P5, EE, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
// Case #225: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EK, ED, EA,
SH_TET, COLOR0, P4, EH, EE, EI,
SH_PYR, COLOR0, P1, EJ, EL, P3, N0,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_TET, COLOR0, P3, P2, P1, N0,
SH_PYR, COLOR0, P2, ED, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_WDG, COLOR1, P6, P5, P0, EH, EE, EI,
SH_TET, COLOR1, P5, P0, P6, N0,
SH_PYR, COLOR1, P6, P0, ED, EK, N0,
SH_TET, COLOR1, P7, P5, P6, N0,
SH_PYR, COLOR1, EJ, P5, P7, EL, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_TET, COLOR1, ED, P0, EA, N0,
// Case #226: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EE, EH, EK, EL, EB, EA,
SH_TET, COLOR0, P2, P0, P3, N0,
SH_PYR, COLOR0, P2, P3, EL, EK, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_PYR, COLOR0, P4, P2, EK, EH, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_PYR, COLOR0, EA, EB, P3, P0, N0,
SH_TET, COLOR0, P3, EB, EL, N0,
SH_TET, COLOR1, P5, P6, P7, N0,
SH_PYR, COLOR1, EE, EH, P6, P5, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, P1, P5, P7, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_PYR, COLOR1, P1, P7, EL, EB, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_TET, COLOR1, EK, P6, EH, N0,
// Case #227: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P0, P1, P7, EK, EL,
SH_WDG, COLOR0, EK, P2, ED, EL, P3, EB,
SH_TET, COLOR0, EE, EI, P4, EH,
SH_WDG, COLOR1, EE, EH, EI, P5, P6, P0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_TET, COLOR1, P7, P5, P6, N0,
SH_TET, COLOR1, P7, P1, P5, N0,
SH_PYR, COLOR1, EB, P1, P7, EL, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_PYR, COLOR1, EK, ED, EB, EL, N0,
SH_TET, COLOR1, P0, P6, P5, N0,
SH_PYR, COLOR1, EK, P6, P0, ED, N0,
// Case #228: (cloned #27)
SH_PNT, 0, NOCOLOR, 6, EH, EE, EJ, EL, EC, ED,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR0, P1, EJ, EL, P3, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_TET, COLOR0, EH, EE, P4, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_PYR, COLOR0, ED, P0, P3, EC, N0,
SH_TET, COLOR0, P3, EL, EC, N0,
SH_TET, COLOR1, P6, P7, P5, N0,
SH_PYR, COLOR1, EH, P6, P5, EE, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_TET, COLOR1, P2, P7, P6, N0,
SH_TET, COLOR1, P2, ED, EC, N0,
SH_PYR, COLOR1, P2, EC, EL, P7, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_TET, COLOR1, EJ, EE, P5, N0,
// Case #229: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P0, P2, P7, EJ, EL,
SH_WDG, COLOR0, EL, P3, EC, EJ, P1, EA,
SH_TET, COLOR0, EH, P4, EI, EE,
SH_WDG, COLOR1, P6, P5, P0, EH, EE, EI,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_TET, COLOR1, P7, P5, P6, N0,
SH_TET, COLOR1, P7, P6, P2, N0,
SH_PYR, COLOR1, EC, EL, P7, P2, N0,
SH_PYR, COLOR1, EA, EC, P2, P0, N0,
SH_TET, COLOR1, P2, P6, P0, N0,
SH_PYR, COLOR1, EJ, EL, EC, EA, N0,
SH_TET, COLOR1, P0, P6, P5, N0,
SH_PYR, COLOR1, EJ, EA, P0, P5, N0,
// Case #230: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P1, P5, P6, ED, EH,
SH_WDG, COLOR0, EH, P4, EE, ED, P0, EA,
SH_TET, COLOR0, EL, P3, EB, EC,
SH_WDG, COLOR1, P7, P2, P1, EL, EC, EB,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_TET, COLOR1, P6, P2, P7, N0,
SH_TET, COLOR1, P6, P7, P5, N0,
SH_PYR, COLOR1, EE, EH, P6, P5, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_TET, COLOR1, P5, P7, P1, N0,
SH_PYR, COLOR1, ED, EH, EE, EA, N0,
SH_TET, COLOR1, P1, P7, P2, N0,
SH_PYR, COLOR1, ED, EA, P1, P2, N0,
// Case #231: (cloned #126)
SH_TET, COLOR0, EE, EH, EI, P4,
SH_TET, COLOR0, EB, EC, EL, P3,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P6,
SH_WDG, COLOR1, EB, EC, EL, P1, P2, P7,
SH_TET, COLOR1, P7, P6, P2, P1,
SH_TET, COLOR1, P1, P0, P5, P6,
SH_TET, COLOR1, P0, P1, P2, P6,
SH_TET, COLOR1, P1, P5, P7, P6,
// Case #232: (cloned #23)
SH_TET, COLOR0, P2, P0, P1, P4,
SH_PYR, COLOR0, EE, EJ, P1, P4, EB,
SH_PYR, COLOR0, EH, P4, P2, EK, EC,
SH_PYR, COLOR0, P1, EB, EC, P2, P4,
SH_PYR, COLOR0, EB, EE, EH, EC, P4,
SH_TET, COLOR1, P7, P5, P6, P3,
SH_PYR, COLOR1, EB, EJ, P5, P3, EE,
SH_PYR, COLOR1, EC, P3, P6, EK, EH,
SH_PYR, COLOR1, P5, EE, EH, P6, P3,
SH_PYR, COLOR1, EB, EC, EH, EE, P3,
// Case #233: (cloned #107)
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_TET, COLOR0, EE, P4, EH, EI,
SH_TET, COLOR0, EC, EK, P2, ED,
SH_WDG, COLOR1, ED, EK, EC, P0, P6, P3,
SH_WDG, COLOR1, P5, P0, P6, EE, EI, EH,
SH_WDG, COLOR1, EJ, EA, EB, P5, P0, P3,
SH_TET, COLOR1, P0, P3, P6, P5,
SH_TET, COLOR1, P5, P7, P3, P6,
// Case #234: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EC, EA, EE, EK, EH,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR0, P2, P0, EA, EC, N0,
SH_TET, COLOR0, EK, P2, EC, N0,
SH_PYR, COLOR0, EH, P4, P2, EK, N0,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_PYR, COLOR1, P7, P3, P1, P5, N0,
SH_TET, COLOR1, P5, P6, P7, N0,
SH_TET, COLOR1, P6, P3, P7, N0,
SH_PYR, COLOR1, P6, EK, EC, P3, N0,
SH_PYR, COLOR1, EC, EA, P1, P3, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, EE, EH, P6, P5, N0,
SH_TET, COLOR1, EK, P6, EH, N0,
// Case #235: (cloned #111)
SH_TET, COLOR0, EK, EC, ED, P2,
SH_TET, COLOR0, EH, EI, EE, P4,
SH_WDG, COLOR1, P0, P6, P5, EI, EH, EE,
SH_WDG, COLOR1, ED, EK, EC, P0, P6, P3,
SH_PYR, COLOR1, P1, P5, P7, P3, P0,
SH_TET, COLOR1, P6, P3, P7, P0,
SH_TET, COLOR1, P6, P7, P5, P0,
// Case #236: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EH, ED, EB, EE, EJ,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, EJ, P1, P4, EE, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR1, P7, P6, P2, P3, N0,
SH_TET, COLOR1, P3, P5, P7, N0,
SH_TET, COLOR1, P5, P6, P7, N0,
SH_PYR, COLOR1, P5, EE, EH, P6, N0,
SH_PYR, COLOR1, EH, ED, P2, P6, N0,
SH_PYR, COLOR1, ED, EB, P3, P2, N0,
SH_PYR, COLOR1, EB, EJ, P5, P3, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #237: (cloned #111)
SH_TET, COLOR0, EE, EH, EI, P4,
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_WDG, COLOR1, P0, P5, P3, EA, EJ, EB,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P6,
SH_PYR, COLOR1, P2, P3, P7, P6, P0,
SH_TET, COLOR1, P5, P6, P7, P0,
SH_TET, COLOR1, P5, P7, P3, P0,
// Case #238: (cloned #63)
SH_WDG, COLOR0, P4, EE, EH, P0, EA, ED,
SH_HEX, COLOR1, EA, ED, EH, EE, P1, P2, P6, P5,
SH_WDG, COLOR1, P3, P1, P2, P7, P5, P6,
// Case #239: (cloned #127)
SH_PNT, 0, COLOR1, 7, P7, P3, P1, P5, P6, P2, P0,
SH_TET, COLOR0, EH, EI, EE, P4,
SH_WDG, COLOR1, EH, EI, EE, P6, P0, P5,
SH_TET, COLOR1, P6, P5, P0, N0,
SH_PYR, COLOR1, P2, P0, P1, P3, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, P7, P3, P1, P5, N0,
SH_TET, COLOR1, P7, P5, P6, N0,
SH_PYR, COLOR1, P7, P6, P2, P3, N0,
SH_TET, COLOR1, P6, P0, P2, N0,
// Case #240: (cloned #15)
SH_HEX, COLOR0, P0, P1, P3, P2, EI, EJ, EL, EK,
SH_HEX, COLOR1, EI, EJ, EL, EK, P4, P5, P7, P6,
// Case #241: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EK, EA, ED,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_TET, COLOR0, P2, EK, ED, N0,
SH_PYR, COLOR0, P1, EJ, EL, P3, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, ED, EA, P1, P2, N0,
SH_TET, COLOR0, P2, P1, P3, N0,
SH_PYR, COLOR1, P4, P6, P7, P5, N0,
SH_TET, COLOR1, P6, P4, P0, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EJ, P5, P7, EL, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_PYR, COLOR1, EK, P6, P0, ED, N0,
SH_TET, COLOR1, EA, ED, P0, N0,
// Case #242: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EI, EB, EA,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, P3, EL, EK, P2, N0,
SH_TET, COLOR0, EB, EL, P3, N0,
SH_PYR, COLOR0, EA, EB, P3, P0, N0,
SH_TET, COLOR0, P0, P3, P2, N0,
SH_PYR, COLOR1, P5, P4, P6, P7, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_TET, COLOR1, P1, P5, P7, N0,
SH_PYR, COLOR1, P1, P7, EL, EB, N0,
SH_PYR, COLOR1, EL, P7, P6, EK, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #243: (cloned #63)
SH_WDG, COLOR0, P2, ED, EK, P3, EB, EL,
SH_HEX, COLOR1, P0, P6, P7, P1, ED, EK, EL, EB,
SH_WDG, COLOR1, P5, P1, P7, P4, P0, P6,
// Case #244: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EI, EC, ED,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, P3, P1, EJ, EL, N0,
SH_TET, COLOR0, EC, P3, EL, N0,
SH_PYR, COLOR0, ED, P0, P3, EC, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_PYR, COLOR1, P6, P7, P5, P4, N0,
SH_TET, COLOR1, P4, P2, P6, N0,
SH_TET, COLOR1, P2, P7, P6, N0,
SH_PYR, COLOR1, P2, EC, EL, P7, N0,
SH_PYR, COLOR1, EL, EJ, P5, P7, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, EI, ED, P2, P4, N0,
SH_TET, COLOR1, EC, P2, ED, N0,
// Case #245: (cloned #63)
SH_WDG, COLOR0, P3, EC, EL, P1, EA, EJ,
SH_HEX, COLOR1, EA, EJ, EL, EC, P0, P5, P7, P2,
SH_WDG, COLOR1, P4, P0, P5, P6, P2, P7,
// Case #246: (cloned #111)
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EB, EC, EL, P3,
SH_WDG, COLOR1, P2, P1, P7, EC, EB, EL,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_PYR, COLOR1, P6, P7, P5, P4, P2,
SH_TET, COLOR1, P1, P4, P5, P2,
SH_TET, COLOR1, P1, P5, P7, P2,
// Case #247: (cloned #127)
SH_PNT, 0, COLOR1, 7, P5, P4, P0, P1, P7, P6, P2,
SH_TET, COLOR0, EL, EB, EC, P3,
SH_WDG, COLOR1, P7, P2, P1, EL, EC, EB,
SH_TET, COLOR1, P7, P2, P1, N0,
SH_PYR, COLOR1, P6, P4, P0, P2, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_PYR, COLOR1, P5, P1, P0, P4, N0,
SH_TET, COLOR1, P5, P7, P1, N0,
SH_PYR, COLOR1, P5, P4, P6, P7, N0,
SH_TET, COLOR1, P7, P6, P2, N0,
// Case #248: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EK, EI, EJ, EC, EB,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, P2, EK, EI, P0, N0,
SH_TET, COLOR0, EC, EK, P2, N0,
SH_PYR, COLOR0, EB, EC, P2, P1, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR1, P7, P5, P4, P6, N0,
SH_TET, COLOR1, P5, P7, P3, N0,
SH_TET, COLOR1, P3, P7, P6, N0,
SH_PYR, COLOR1, P3, P6, EK, EC, N0,
SH_PYR, COLOR1, EK, P6, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, EJ, P5, P3, EB, N0,
SH_TET, COLOR1, EC, EB, P3, N0,
// Case #249: (cloned #111)
SH_TET, COLOR0, EC, ED, EK, P2,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, EA, EB, EJ, P0, P3, P5,
SH_WDG, COLOR1, P0, P3, P6, ED, EC, EK,
SH_PYR, COLOR1, P4, P6, P7, P5, P0,
SH_TET, COLOR1, P3, P7, P6, P0,
SH_TET, COLOR1, P3, P5, P7, P0,
// Case #250: (cloned #63)
SH_WDG, COLOR0, P0, EA, EI, P2, EC, EK,
SH_HEX, COLOR1, P1, P4, P6, P3, EA, EI, EK, EC,
SH_WDG, COLOR1, P7, P3, P6, P5, P1, P4,
// Case #251: (cloned #127)
SH_PNT, 0, COLOR1, 7, P7, P5, P1, P3, P6, P4, P0,
SH_TET, COLOR0, EK, EC, ED, P2,
SH_WDG, COLOR1, P6, P0, P3, EK, ED, EC,
SH_TET, COLOR1, P6, P0, P3, N0,
SH_PYR, COLOR1, P4, P5, P1, P0, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, P7, P3, P1, P5, N0,
SH_TET, COLOR1, P7, P6, P3, N0,
SH_PYR, COLOR1, P7, P5, P4, P6, N0,
SH_TET, COLOR1, P6, P4, P0, N0,
// Case #252: (cloned #63)
SH_WDG, COLOR0, P0, EI, ED, P1, EJ, EB,
SH_HEX, COLOR1, P4, P2, P3, P5, EI, ED, EB, EJ,
SH_WDG, COLOR1, P7, P5, P3, P6, P4, P2,
// Case #253: (cloned #127)
SH_PNT, 0, COLOR1, 7, P7, P6, P4, P5, P3, P2, P0,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, P3, P0, P5, EB, EA, EJ,
SH_TET, COLOR1, P3, P0, P5, N0,
SH_PYR, COLOR1, P2, P6, P4, P0, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, P7, P5, P4, P6, N0,
SH_TET, COLOR1, P7, P3, P5, N0,
SH_PYR, COLOR1, P7, P6, P2, P3, N0,
SH_TET, COLOR1, P3, P2, P0, N0,
// Case #254: (cloned #127)
SH_PNT, 0, COLOR1, 7, P6, P7, P5, P4, P2, P3, P1,
SH_TET, COLOR0, ED, EA, EI, P0,
SH_WDG, COLOR1, ED, EA, EI, P2, P1, P4,
SH_TET, COLOR1, P2, P4, P1, N0,
SH_PYR, COLOR1, P3, P1, P5, P7, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_PYR, COLOR1, P6, P7, P5, P4, N0,
SH_TET, COLOR1, P6, P4, P2, N0,
SH_PYR, COLOR1, P6, P2, P3, P7, N0,
SH_TET, COLOR1, P2, P1, P3, N0,
// Case #255: Unique case #22
SH_HEX, COLOR1, P0, P1, P3, P2, P4, P5, P7, P6,
//}
//{ Hex
// Case #0: Unique case #1
SH_HEX, COLOR0, P0, P1, P2, P3, P4, P5, P6, P7,
// Case #1: Unique case #2
SH_PNT, 0, COLOR0, 7, P1, P2, P3, P4, P5, P6, P7,
SH_WDG, COLOR0, P1, P3, P4, EA, ED, EI,
SH_TET, COLOR0, P1, P3, P4, N0,
SH_TET, COLOR0, P1, P2, P3, N0,
SH_PYR, COLOR0, P6, P7, P3, P2, N0,
SH_PYR, COLOR0, P5, P6, P2, P1, N0,
SH_PYR, COLOR0, P4, P7, P6, P5, N0,
SH_TET, COLOR0, P3, P7, P4, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR1, P0, EA, ED, EI,
// Case #2: (cloned #1)
SH_PNT, 0, COLOR0, 7, P5, P4, P0, P2, P6, P7, P3,
SH_WDG, COLOR0, EJ, EA, EB, P5, P0, P2,
SH_TET, COLOR0, P5, P2, P0, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P7, P4, P0, P3, N0,
SH_PYR, COLOR0, P6, P5, P4, P7, N0,
SH_PYR, COLOR0, P2, P6, P7, P3, N0,
SH_TET, COLOR0, P0, P2, P3, N0,
SH_TET, COLOR0, P2, P5, P6, N0,
SH_TET, COLOR1, P1, EA, EJ, EB,
// Case #3: Unique case #3
SH_HEX, COLOR0, EB, P2, P3, ED, EJ, P5, P4, EI,
SH_WDG, COLOR0, P2, P6, P5, P3, P7, P4,
SH_WDG, COLOR1, P1, EB, EJ, P0, ED, EI,
// Case #4: (cloned #1)
SH_PNT, 0, COLOR0, 7, P6, P5, P1, P3, P7, P4, P0,
SH_WDG, COLOR0, EL, EB, EC, P6, P1, P3,
SH_TET, COLOR0, P6, P3, P1, N0,
SH_TET, COLOR0, P6, P1, P5, N0,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_PYR, COLOR0, P7, P6, P5, P4, N0,
SH_PYR, COLOR0, P3, P7, P4, P0, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_TET, COLOR0, P3, P6, P7, N0,
SH_TET, COLOR1, P2, EB, EL, EC,
// Case #5: Unique case #4
SH_PNT, 0, NOCOLOR, 2, EI, EL,
SH_PYR, COLOR0, P4, P7, P6, P5, N0,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR0, P3, P7, P4, N0,
SH_TET, COLOR0, P6, P7, P3, N0,
SH_PYR, COLOR0, P6, P3, EC, EL, N0,
SH_PYR, COLOR0, P1, P6, EL, EB, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_PYR, COLOR0, P4, EI, ED, P3, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_PYR, COLOR1, P0, P2, EC, ED, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_TET, COLOR1, EB, EL, P2, N0,
SH_TET, COLOR1, P2, EL, EC, N0,
SH_TET, COLOR1, EA, N0, P0, EI,
SH_TET, COLOR1, ED, EI, P0, N0,
// Case #6: (cloned #3)
SH_HEX, COLOR0, EC, P3, P0, EA, EL, P6, P5, EJ,
SH_WDG, COLOR0, P3, P7, P6, P0, P4, P5,
SH_WDG, COLOR1, P2, EC, EL, P1, EA, EJ,
// Case #7: Unique case #5
SH_PNT, 0, NOCOLOR, 5, EI, EJ, ED, EC, EL,
SH_PYR, COLOR0, P4, P7, P6, P5, N0,
SH_TET, COLOR0, P6, P3, N0, P7,
SH_PYR, COLOR0, P5, P6, EL, EJ, N0,
SH_PYR, COLOR0, EI, P4, P5, EJ, N0,
SH_TET, COLOR0, P3, P7, P4, N0,
SH_PYR, COLOR0, P3, P4, EI, ED, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_PYR, COLOR0, EL, P6, P3, EC, N0,
SH_PYR, COLOR1, EJ, EL, P2, P1, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_TET, COLOR1, ED, EI, P0, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_PYR, COLOR1, ED, P0, P2, EC, N0,
SH_TET, COLOR1, P2, EL, EC, N0,
// Case #8: (cloned #1)
SH_PNT, 0, COLOR0, 7, P2, P1, P0, P7, P6, P5, P4,
SH_WDG, COLOR0, EC, ED, EK, P2, P0, P7,
SH_TET, COLOR0, P2, P7, P0, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_PYR, COLOR0, P5, P1, P0, P4, N0,
SH_PYR, COLOR0, P6, P2, P1, P5, N0,
SH_PYR, COLOR0, P7, P6, P5, P4, N0,
SH_TET, COLOR0, P0, P7, P4, N0,
SH_TET, COLOR0, P7, P2, P6, N0,
SH_TET, COLOR1, P3, ED, EC, EK,
// Case #9: (cloned #3)
SH_HEX, COLOR0, EK, P7, P4, EI, EC, P2, P1, EA,
SH_WDG, COLOR0, P7, P6, P2, P4, P5, P1,
SH_WDG, COLOR1, P3, EK, EC, P0, EI, EA,
// Case #10: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EK, EJ,
SH_PYR, COLOR0, P7, P6, P5, P4, N0,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_TET, COLOR0, P7, P2, P6, N0,
SH_TET, COLOR0, P0, P7, P4, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_PYR, COLOR0, P2, EB, EJ, P5, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_PYR, COLOR0, P7, EK, EC, P2, N0,
SH_PYR, COLOR0, P7, P0, ED, EK, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR1, P3, ED, EA, P1, N0,
SH_PYR, COLOR1, EC, P3, P1, EB, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EC, P3, N0, EK,
SH_TET, COLOR1, ED, P3, EK, N0,
// Case #11: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EJ, EI, EB, EC, EK,
SH_PYR, COLOR0, P5, P4, P7, P6, N0,
SH_TET, COLOR0, P7, N0, P2, P6,
SH_PYR, COLOR0, P4, EI, EK, P7, N0,
SH_PYR, COLOR0, EJ, EI, P4, P5, N0,
SH_TET, COLOR0, P2, P5, P6, N0,
SH_PYR, COLOR0, P2, EB, EJ, P5, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_PYR, COLOR0, EK, EC, P2, P7, N0,
SH_PYR, COLOR1, EI, P0, P3, EK, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_PYR, COLOR1, EB, EC, P3, P1, N0,
SH_TET, COLOR1, P3, EC, EK, N0,
// Case #12: (cloned #3)
SH_HEX, COLOR0, EL, P6, P7, EK, EB, P1, P0, ED,
SH_WDG, COLOR0, P0, P4, P7, P1, P5, P6,
SH_WDG, COLOR1, P3, ED, EK, P2, EB, EL,
// Case #13: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EA, EB, EL,
SH_PYR, COLOR0, P4, P7, P6, P5, N0,
SH_TET, COLOR0, P6, N0, P1, P5,
SH_PYR, COLOR0, P7, EK, EL, P6, N0,
SH_PYR, COLOR0, EI, EK, P7, P4, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, EL, EB, P1, P6, N0,
SH_PYR, COLOR1, EK, P3, P2, EL, N0,
SH_PYR, COLOR1, EI, P0, P3, EK, N0,
SH_TET, COLOR1, EA, P0, EI, N0,
SH_TET, COLOR1, P0, P2, P3, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_TET, COLOR1, P2, EB, EL, N0,
// Case #14: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EK, EL, ED, EA, EJ,
SH_PYR, COLOR0, P7, P6, P5, P4, N0,
SH_TET, COLOR0, P5, N0, P0, P4,
SH_PYR, COLOR0, P6, EL, EJ, P5, N0,
SH_PYR, COLOR0, EK, EL, P6, P7, N0,
SH_TET, COLOR0, P0, P7, P4, N0,
SH_PYR, COLOR0, P0, ED, EK, P7, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, EJ, EA, P0, P5, N0,
SH_PYR, COLOR1, EL, P2, P1, EJ, N0,
SH_PYR, COLOR1, EK, P3, P2, EL, N0,
SH_TET, COLOR1, ED, P3, EK, N0,
SH_TET, COLOR1, P3, P1, P2, N0,
SH_PYR, COLOR1, ED, EA, P1, P3, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
// Case #15: Unique case #6
SH_HEX, COLOR0, EI, EJ, EL, EK, P4, P5, P6, P7,
SH_HEX, COLOR1, P0, P1, P2, P3, EI, EJ, EL, EK,
// Case #16: (cloned #1)
SH_PNT, 0, COLOR0, 7, P5, P1, P0, P7, P6, P2, P3,
SH_WDG, COLOR0, P5, P0, P7, EE, EI, EH,
SH_TET, COLOR0, P5, P0, P7, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P2, P3, P0, P1, N0,
SH_PYR, COLOR0, P6, P2, P1, P5, N0,
SH_PYR, COLOR0, P7, P3, P2, P6, N0,
SH_TET, COLOR0, P0, P3, P7, N0,
SH_TET, COLOR0, P7, P6, P5, N0,
SH_TET, COLOR1, P4, EE, EI, EH,
// Case #17: (cloned #3)
SH_HEX, COLOR0, EE, P5, P1, EA, EH, P7, P3, ED,
SH_WDG, COLOR0, P3, P2, P1, P7, P6, P5,
SH_WDG, COLOR1, P0, ED, EA, P4, EH, EE,
// Case #18: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EH, EB,
SH_PYR, COLOR0, P7, P3, P2, P6, N0,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_TET, COLOR0, P7, P6, P5, N0,
SH_TET, COLOR0, P0, P3, P7, N0,
SH_TET, COLOR0, P2, P3, P0, N0,
SH_PYR, COLOR0, P2, P0, EA, EB, N0,
SH_PYR, COLOR0, P5, P2, EB, EJ, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_PYR, COLOR0, P7, P5, EE, EH, N0,
SH_PYR, COLOR0, P7, EH, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_TET, COLOR1, EE, N0, P4, EH,
SH_TET, COLOR1, EI, EH, P4, N0,
// Case #19: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EB, ED, EJ, EE, EH,
SH_PYR, COLOR0, P2, P6, P7, P3, N0,
SH_TET, COLOR0, P7, P5, N0, P6,
SH_PYR, COLOR0, P3, P7, EH, ED, N0,
SH_PYR, COLOR0, EB, P2, P3, ED, N0,
SH_TET, COLOR0, P5, P6, P2, N0,
SH_PYR, COLOR0, P5, P2, EB, EJ, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_PYR, COLOR0, EH, P7, P5, EE, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
// Case #20: Unique case #7
SH_WDG, COLOR0, EB, EC, EL, P1, P3, P6,
SH_WDG, COLOR0, P0, P7, P5, EI, EH, EE,
SH_TET, COLOR0, P3, P1, P6, P7,
SH_TET, COLOR0, P5, P7, P6, P1,
SH_TET, COLOR0, P0, P5, P1, P7,
SH_TET, COLOR0, P3, P7, P0, P1,
SH_TET, COLOR1, P4, EE, EI, EH,
SH_TET, COLOR1, P2, EC, EB, EL,
// Case #21: Unique case #8
SH_PNT, 0, NOCOLOR, 4, EE, EH, EL, EL,
SH_PYR, COLOR0, P6, P3, EC, EL, N0,
SH_TET, COLOR0, EC, P3, ED, N0,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_TET, COLOR0, P6, P7, P3, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_PYR, COLOR0, P1, P6, EL, EB, N0,
SH_TET, COLOR0, P5, P7, P6, N0,
SH_PYR, COLOR0, P5, EE, EH, P7, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_PYR, COLOR1, P2, EC, ED, P0, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_TET, COLOR1, P2, EL, EC, N0,
SH_TET, COLOR1, EB, EL, P2, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_PYR, COLOR1, P0, P4, EE, EA, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
// Case #22: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EL, EC, EH, EH,
SH_PYR, COLOR0, P7, EH, EI, P0, N0,
SH_TET, COLOR0, EI, EA, P0, N0,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_TET, COLOR0, P7, P0, P3, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_PYR, COLOR0, P5, EE, EH, P7, N0,
SH_TET, COLOR0, P6, P7, P3, N0,
SH_PYR, COLOR0, P6, P3, EC, EL, N0,
SH_PYR, COLOR0, P6, EL, EJ, P5, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
SH_TET, COLOR1, P4, EI, EH, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_PYR, COLOR1, P1, EJ, EL, P2, N0,
SH_TET, COLOR1, EL, EC, P2, N0,
// Case #23: Unique case #9
SH_PNT, 0, NOCOLOR, 6, ED, EC, EL, EJ, EE, EH,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_PYR, COLOR0, P6, EL, EJ, P5, N0,
SH_TET, COLOR0, P3, P6, P7, N0,
SH_PYR, COLOR0, P3, EC, EL, P6, N0,
SH_TET, COLOR0, ED, EC, P3, N0,
SH_PYR, COLOR0, P3, P7, EH, ED, N0,
SH_PYR, COLOR0, EH, P7, P5, EE, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_PYR, COLOR1, ED, P0, P2, EC, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_TET, COLOR1, P4, P1, P0, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_PYR, COLOR1, EJ, EL, P2, P1, N0,
SH_TET, COLOR1, EL, EC, P2, N0,
// Case #24: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EC, EE,
SH_PYR, COLOR0, P2, P1, P5, P6, N0,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_TET, COLOR0, P2, P6, P7, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_PYR, COLOR0, P7, P5, EE, EH, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_PYR, COLOR0, P2, P7, EK, EC, N0,
SH_PYR, COLOR0, P2, EC, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR1, P3, P4, EI, ED, N0,
SH_PYR, COLOR1, EK, EH, P4, P3, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EK, N0, P3, EC,
SH_TET, COLOR1, ED, EC, P3, N0,
// Case #25: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EE, EA, EH, EK, EC,
SH_PYR, COLOR0, P5, P6, P2, P1, N0,
SH_TET, COLOR0, P2, P7, N0, P6,
SH_PYR, COLOR0, P1, P2, EC, EA, N0,
SH_PYR, COLOR0, EE, P5, P1, EA, N0,
SH_TET, COLOR0, P7, P6, P5, N0,
SH_PYR, COLOR0, P7, P5, EE, EH, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_PYR, COLOR0, EC, P2, P7, EK, N0,
SH_PYR, COLOR1, EA, EC, P3, P0, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_TET, COLOR1, P4, P0, P3, N0,
SH_PYR, COLOR1, EH, P4, P3, EK, N0,
SH_TET, COLOR1, P3, EC, EK, N0,
// Case #26: Unique case #10
SH_TET, COLOR0, P0, EA, ED, EI,
SH_TET, COLOR0, P5, P7, P6, P2,
SH_PYR, COLOR0, EC, P2, P7, EK, EH,
SH_PYR, COLOR0, EB, EJ, P5, P2, EE,
SH_PYR, COLOR0, P7, P5, EE, EH, P2,
SH_PYR, COLOR0, EH, EE, EB, EC, P2,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_PYR, COLOR1, P3, EK, EH, P4, EC,
SH_PYR, COLOR1, EE, EJ, P1, P4, EB,
SH_PYR, COLOR1, EC, P3, P1, EB, P4,
SH_PYR, COLOR1, EC, EB, EE, EH, P4,
// Case #27: Unique case #11
SH_TET, COLOR0, P5, P7, P6, P2,
SH_PYR, COLOR0, EC, P2, P7, EK, EH,
SH_PYR, COLOR0, EB, EJ, P5, P2, EE,
SH_PYR, COLOR0, P7, P5, EE, EH, P2,
SH_PYR, COLOR0, EH, EE, EB, EC, P2,
SH_TET, COLOR1, P0, P1, P3, P4,
SH_PYR, COLOR1, EH, P4, P3, EK, EC,
SH_PYR, COLOR1, EE, EJ, P1, P4, EB,
SH_PYR, COLOR1, P3, P1, EB, EC, P4,
SH_PYR, COLOR1, EH, EC, EB, EE, P4,
// Case #28: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EL, EB, EE, EE,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_TET, COLOR0, EI, P0, ED, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_PYR, COLOR0, P7, P5, EE, EH, N0,
SH_TET, COLOR0, P6, P1, P5, N0,
SH_PYR, COLOR0, P6, EL, EB, P1, N0,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_PYR, COLOR1, P4, EI, ED, P3, N0,
SH_PYR, COLOR1, EK, EH, P4, P3, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_PYR, COLOR1, P3, P2, EL, EK, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
// Case #29: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EA, EB, EL, EK, EH, EE,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_TET, COLOR0, P1, P5, P6, N0,
SH_PYR, COLOR0, P1, P6, EL, EB, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_PYR, COLOR0, EE, EH, P7, P5, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR1, P0, P2, P3, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_TET, COLOR1, P4, P0, P3, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
SH_PYR, COLOR1, P4, P3, EK, EH, N0,
SH_PYR, COLOR1, EK, P3, P2, EL, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
// Case #30: Unique case #12
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EK, EH, EE,
SH_TET, COLOR0, P0, EA, ED, EI,
SH_PYR, COLOR0, P5, P6, EL, EJ, N0,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR0, P6, P5, P7, N0,
SH_PYR, COLOR0, P7, P5, EE, EH, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_TET, COLOR1, P1, P3, P4, N0,
SH_PYR, COLOR1, P3, EK, EH, P4, N0,
SH_TET, COLOR1, P2, P3, P1, N0,
SH_PYR, COLOR1, EJ, EL, P2, P1, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_TET, COLOR1, EH, EE, P4, N0,
// Case #31: Unique case #13
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EK, EE, EH,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_PYR, COLOR0, P5, P6, EL, EJ, N0,
SH_TET, COLOR0, EE, P5, EJ, N0,
SH_PYR, COLOR0, EH, P7, P5, EE, N0,
SH_TET, COLOR0, P7, P6, P5, N0,
SH_PYR, COLOR1, P0, P1, P2, P3, N0,
SH_TET, COLOR1, P3, P4, P0, N0,
SH_TET, COLOR1, P4, P1, P0, N0,
SH_PYR, COLOR1, P4, EE, EJ, P1, N0,
SH_PYR, COLOR1, EJ, EL, P2, P1, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_PYR, COLOR1, EK, EH, P4, P3, N0,
SH_TET, COLOR1, EE, P4, EH, N0,
// Case #32: (cloned #1)
SH_PNT, 0, COLOR0, 7, P6, P2, P1, P4, P7, P3, P0,
SH_WDG, COLOR0, P6, P1, P4, EF, EJ, EE,
SH_TET, COLOR0, P6, P1, P4, N0,
SH_TET, COLOR0, P6, P2, P1, N0,
SH_PYR, COLOR0, P3, P0, P1, P2, N0,
SH_PYR, COLOR0, P7, P3, P2, P6, N0,
SH_PYR, COLOR0, P4, P0, P3, P7, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_TET, COLOR0, P4, P7, P6, N0,
SH_TET, COLOR1, P5, EF, EJ, EE,
// Case #33: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, ED, EF,
SH_PYR, COLOR0, P3, P2, P6, P7, N0,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_TET, COLOR0, P3, P1, P2, N0,
SH_TET, COLOR0, P4, P3, P7, N0,
SH_TET, COLOR0, P6, P4, P7, N0,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_PYR, COLOR0, P1, EJ, EF, P6, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, P3, ED, EA, P1, N0,
SH_PYR, COLOR0, P3, P4, EI, ED, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_TET, COLOR1, EA, P0, N0, ED,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #34: (cloned #3)
SH_HEX, COLOR0, EF, P6, P2, EB, EE, P4, P0, EA,
SH_WDG, COLOR0, P0, P3, P2, P4, P7, P6,
SH_WDG, COLOR1, P1, EA, EB, P5, EE, EF,
// Case #35: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, ED, EB, EI, EE, EF,
SH_PYR, COLOR0, P3, P2, P6, P7, N0,
SH_TET, COLOR0, P6, N0, P4, P7,
SH_PYR, COLOR0, P2, EB, EF, P6, N0,
SH_PYR, COLOR0, ED, EB, P2, P3, N0,
SH_TET, COLOR0, P4, P3, P7, N0,
SH_PYR, COLOR0, P4, EI, ED, P3, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR0, EF, EE, P4, P6, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
// Case #36: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EC, EE,
SH_PYR, COLOR0, P3, P7, P4, P0, N0,
SH_TET, COLOR0, P7, P6, P4, N0,
SH_TET, COLOR0, P3, P6, P7, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_PYR, COLOR0, P3, EC, EL, P6, N0,
SH_PYR, COLOR0, P3, P1, EB, EC, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR1, P2, EB, EJ, P5, N0,
SH_PYR, COLOR1, EL, P2, P5, EF, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EL, P2, N0, EC,
SH_TET, COLOR1, EB, P2, EC, N0,
// Case #37: (cloned #26)
SH_TET, COLOR0, P1, EA, EJ, EB,
SH_TET, COLOR0, P6, P7, P3, P4,
SH_PYR, COLOR0, EI, ED, P3, P4, EC,
SH_PYR, COLOR0, EE, P4, P6, EF, EL,
SH_PYR, COLOR0, P3, EC, EL, P6, P4,
SH_PYR, COLOR0, EC, EI, EE, EL, P4,
SH_WDG, COLOR1, P0, P5, P2, EA, EJ, EB,
SH_PYR, COLOR1, P0, P2, EC, ED, EI,
SH_PYR, COLOR1, EL, P2, P5, EF, EE,
SH_PYR, COLOR1, EI, EE, P5, P0, P2,
SH_PYR, COLOR1, EI, EC, EL, EE, P2,
// Case #38: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EE, EA, EF, EL, EC,
SH_PYR, COLOR0, P4, P0, P3, P7, N0,
SH_TET, COLOR0, P3, N0, P6, P7,
SH_PYR, COLOR0, P0, EA, EC, P3, N0,
SH_PYR, COLOR0, EE, EA, P0, P4, N0,
SH_TET, COLOR0, P6, P4, P7, N0,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_PYR, COLOR0, EC, EL, P6, P3, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_TET, COLOR1, P5, P2, P1, N0,
SH_PYR, COLOR1, EF, EL, P2, P5, N0,
SH_TET, COLOR1, P2, EL, EC, N0,
// Case #39: (cloned #27)
SH_TET, COLOR0, P6, P7, P3, P4,
SH_PYR, COLOR0, EI, ED, P3, P4, EC,
SH_PYR, COLOR0, EE, P4, P6, EF, EL,
SH_PYR, COLOR0, P3, EC, EL, P6, P4,
SH_PYR, COLOR0, EC, EI, EE, EL, P4,
SH_TET, COLOR1, P1, P0, P5, P2,
SH_PYR, COLOR1, EC, ED, P0, P2, EI,
SH_PYR, COLOR1, EL, P2, P5, EF, EE,
SH_PYR, COLOR1, P0, EI, EE, P5, P2,
SH_PYR, COLOR1, EC, EL, EE, EI, P2,
// Case #40: (cloned #20)
SH_WDG, COLOR0, P1, P4, P6, EJ, EE, EF,
SH_WDG, COLOR0, ED, EK, EC, P0, P7, P2,
SH_TET, COLOR0, P4, P6, P1, P7,
SH_TET, COLOR0, P2, P6, P7, P1,
SH_TET, COLOR0, P0, P1, P2, P7,
SH_TET, COLOR0, P4, P0, P7, P1,
SH_TET, COLOR1, P3, ED, EC, EK,
SH_TET, COLOR1, P5, EJ, EE, EF,
// Case #41: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EC, EK, EF, EF,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_TET, COLOR0, EE, EI, P4, N0,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_TET, COLOR0, P6, P4, P7, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_PYR, COLOR0, P1, EJ, EF, P6, N0,
SH_TET, COLOR0, P2, P6, P7, N0,
SH_PYR, COLOR0, P2, P7, EK, EC, N0,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_PYR, COLOR1, P5, P0, EI, EE, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
SH_PYR, COLOR1, EI, P0, P3, EK, N0,
SH_PYR, COLOR1, P0, EA, EC, P3, N0,
SH_TET, COLOR1, EC, EK, P3, N0,
// Case #42: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EF, EE, EK, EK,
SH_PYR, COLOR0, P7, P0, ED, EK, N0,
SH_TET, COLOR0, ED, P0, EA, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P7, P4, P0, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_TET, COLOR0, P6, P7, P2, N0,
SH_PYR, COLOR0, P2, P7, EK, EC, N0,
SH_TET, COLOR0, P6, P4, P7, N0,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_PYR, COLOR0, P6, P2, EB, EF, N0,
SH_PYR, COLOR1, P3, ED, EA, P1, N0,
SH_PYR, COLOR1, EB, EC, P3, P1, N0,
SH_TET, COLOR1, P3, EK, ED, N0,
SH_TET, COLOR1, EC, EK, P3, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, P1, P5, EF, EB, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #43: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EI, EE, EF, EB, EC, EK,
SH_TET, COLOR0, P6, P7, P2, N0,
SH_PYR, COLOR0, P6, P2, EB, EF, N0,
SH_TET, COLOR0, P4, P7, P6, N0,
SH_PYR, COLOR0, P4, P6, EF, EE, N0,
SH_TET, COLOR0, EI, P4, EE, N0,
SH_PYR, COLOR0, P4, EI, EK, P7, N0,
SH_PYR, COLOR0, EK, EC, P2, P7, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_PYR, COLOR1, EI, P0, P3, EK, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_TET, COLOR1, P3, EC, EK, N0,
SH_PYR, COLOR1, P3, P1, EB, EC, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #44: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EK, ED, EE, EE,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_TET, COLOR0, EJ, EB, P1, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_TET, COLOR0, P7, P6, P4, N0,
SH_PYR, COLOR0, P6, EF, EE, P4, N0,
SH_TET, COLOR0, P7, P4, P0, N0,
SH_PYR, COLOR0, P7, P0, ED, EK, N0,
SH_PYR, COLOR0, P7, EK, EL, P6, N0,
SH_PYR, COLOR1, P5, P2, EB, EJ, N0,
SH_PYR, COLOR1, EL, P2, P5, EF, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_PYR, COLOR1, P2, EL, EK, P3, N0,
SH_TET, COLOR1, EK, ED, P3, N0,
// Case #45: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EI, EE, EF,
SH_TET, COLOR0, P1, EB, EA, EJ,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR0, P7, P6, P4, N0,
SH_PYR, COLOR0, P4, P6, EF, EE, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_WDG, COLOR1, EA, EB, EJ, P0, P2, P5,
SH_TET, COLOR1, P2, P0, P5, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_TET, COLOR1, P3, P0, P2, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_PYR, COLOR1, P5, EF, EL, P2, N0,
SH_TET, COLOR1, EE, EF, P5, N0,
// Case #46: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EA, ED, EK, EL, EF, EE,
SH_TET, COLOR0, P7, P6, P4, N0,
SH_PYR, COLOR0, P7, EK, EL, P6, N0,
SH_TET, COLOR0, P0, P7, P4, N0,
SH_PYR, COLOR0, P0, ED, EK, P7, N0,
SH_TET, COLOR0, EA, ED, P0, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_PYR, COLOR0, EE, P4, P6, EF, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_TET, COLOR1, P1, P2, P3, N0,
SH_PYR, COLOR1, EA, P1, P3, ED, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_TET, COLOR1, P5, P2, P1, N0,
SH_TET, COLOR1, P5, EE, EF, N0,
SH_PYR, COLOR1, P5, EF, EL, P2, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_TET, COLOR1, EK, ED, P3, N0,
// Case #47: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EI, EF, EE,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_PYR, COLOR0, P6, P7, EK, EL, N0,
SH_TET, COLOR0, EF, P6, EL, N0,
SH_PYR, COLOR0, EE, P4, P6, EF, N0,
SH_TET, COLOR0, P4, P7, P6, N0,
SH_PYR, COLOR1, P1, P2, P3, P0, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_TET, COLOR1, P5, P2, P1, N0,
SH_PYR, COLOR1, P5, EF, EL, P2, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_TET, COLOR1, EF, P5, EE, N0,
// Case #48: (cloned #3)
SH_HEX, COLOR0, EJ, P1, P0, EI, EF, P6, P7, EH,
SH_WDG, COLOR0, P1, P2, P6, P0, P3, P7,
SH_WDG, COLOR1, P5, EJ, EF, P4, EI, EH,
// Case #49: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EA, EJ, EF,
SH_PYR, COLOR0, P3, P2, P6, P7, N0,
SH_TET, COLOR0, P6, P1, N0, P2,
SH_PYR, COLOR0, P7, P6, EF, EH, N0,
SH_PYR, COLOR0, ED, P3, P7, EH, N0,
SH_TET, COLOR0, P1, P2, P3, N0,
SH_PYR, COLOR0, P1, P3, ED, EA, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, EF, P6, P1, EJ, N0,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_TET, COLOR1, EA, ED, P0, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, P5, EF, EJ, N0,
// Case #50: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EH, EF, EI, EA, EB,
SH_PYR, COLOR0, P7, P3, P2, P6, N0,
SH_TET, COLOR0, P2, P0, N0, P3,
SH_PYR, COLOR0, P6, P2, EB, EF, N0,
SH_PYR, COLOR0, EH, P7, P6, EF, N0,
SH_TET, COLOR0, P0, P3, P7, N0,
SH_PYR, COLOR0, P0, P7, EH, EI, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, EB, P2, P0, EA, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, EI, EH, P4, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
// Case #51: (cloned #15)
SH_HEX, COLOR0, P3, P2, P6, P7, ED, EB, EF, EH,
SH_HEX, COLOR1, ED, EB, EF, EH, P0, P1, P5, P4,
// Case #52: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EH, EI, EC, EC,
SH_PYR, COLOR0, P3, P1, EB, EC, N0,
SH_TET, COLOR0, EB, P1, EJ, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_TET, COLOR0, P7, P3, P6, N0,
SH_PYR, COLOR0, P6, P3, EC, EL, N0,
SH_TET, COLOR0, P7, P0, P3, N0,
SH_PYR, COLOR0, P7, EH, EI, P0, N0,
SH_PYR, COLOR0, P7, P6, EF, EH, N0,
SH_PYR, COLOR1, P2, EB, EJ, P5, N0,
SH_PYR, COLOR1, EF, EL, P2, P5, N0,
SH_TET, COLOR1, P2, EC, EB, N0,
SH_TET, COLOR1, EL, EC, P2, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, P5, P4, EH, EF, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #53: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EH, EF, ED, EC, EL,
SH_TET, COLOR0, P1, EA, EJ, EB,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_TET, COLOR0, P7, P3, P6, N0,
SH_PYR, COLOR0, P3, EC, EL, P6, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_WDG, COLOR1, P0, P5, P2, EA, EJ, EB,
SH_TET, COLOR1, P5, P2, P0, N0,
SH_PYR, COLOR1, P0, P2, EC, ED, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, P2, P5, EF, EL, N0,
SH_TET, COLOR1, EC, P2, EL, N0,
// Case #54: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EA, EI, EH, EF, EL, EC,
SH_TET, COLOR0, P7, P3, P6, N0,
SH_PYR, COLOR0, P7, P6, EF, EH, N0,
SH_TET, COLOR0, P0, P3, P7, N0,
SH_PYR, COLOR0, P0, P7, EH, EI, N0,
SH_TET, COLOR0, EA, P0, EI, N0,
SH_PYR, COLOR0, P0, EA, EC, P3, N0,
SH_PYR, COLOR0, EC, EL, P6, P3, N0,
SH_TET, COLOR0, P6, EL, EF, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_TET, COLOR1, P2, P1, P5, N0,
SH_TET, COLOR1, P2, EL, EC, N0,
SH_PYR, COLOR1, P2, P5, EF, EL, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #55: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EH, ED, EL, EC,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_TET, COLOR0, EL, EF, P6, N0,
SH_PYR, COLOR0, EC, EL, P6, P3, N0,
SH_TET, COLOR0, P3, P6, P7, N0,
SH_PYR, COLOR1, P1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_TET, COLOR1, P2, P1, P5, N0,
SH_PYR, COLOR1, P2, P5, EF, EL, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P2, EC, N0,
SH_TET, COLOR1, EL, EC, P2, N0,
// Case #56: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EF, EJ, EC, EC,
SH_PYR, COLOR0, P2, EC, ED, P0, N0,
SH_TET, COLOR0, ED, EI, P0, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR0, P6, P7, P2, N0,
SH_PYR, COLOR0, P7, EK, EC, P2, N0,
SH_TET, COLOR0, P6, P2, P1, N0,
SH_PYR, COLOR0, P6, P1, EJ, EF, N0,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_PYR, COLOR1, P3, P4, EI, ED, N0,
SH_PYR, COLOR1, EH, P4, P3, EK, N0,
SH_TET, COLOR1, P3, ED, EC, N0,
SH_TET, COLOR1, EK, P3, EC, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, P4, EH, EF, P5, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #57: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EA, EJ, EF, EH, EK, EC,
SH_TET, COLOR0, P6, P7, P2, N0,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_TET, COLOR0, P1, P6, P2, N0,
SH_PYR, COLOR0, P1, EJ, EF, P6, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, P1, P2, EC, EA, N0,
SH_PYR, COLOR0, EC, P2, P7, EK, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_PYR, COLOR1, EA, EC, P3, P0, N0,
SH_TET, COLOR1, P3, P4, P0, N0,
SH_TET, COLOR1, P3, EC, EK, N0,
SH_PYR, COLOR1, P3, EK, EH, P4, N0,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #58: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EH, EK, EC,
SH_TET, COLOR0, P0, EI, EA, ED,
SH_PYR, COLOR0, P2, EB, EF, P6, N0,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_TET, COLOR0, P6, P7, P2, N0,
SH_PYR, COLOR0, P7, EK, EC, P2, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_WDG, COLOR1, P4, P1, P3, EI, EA, ED,
SH_TET, COLOR1, P1, P3, P4, N0,
SH_PYR, COLOR1, P4, P3, EK, EH, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, P3, P1, EB, EC, N0,
SH_TET, COLOR1, EK, P3, EC, N0,
// Case #59: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EB, EF, EH, EC, EK,
SH_PYR, COLOR0, P6, EF, EH, P7, N0,
SH_TET, COLOR0, P7, EH, EK, N0,
SH_PYR, COLOR0, P2, EB, EF, P6, N0,
SH_TET, COLOR0, EC, EB, P2, N0,
SH_PYR, COLOR0, EK, EC, P2, P7, N0,
SH_TET, COLOR0, P7, P2, P6, N0,
SH_PYR, COLOR1, P0, P4, P5, P1, N0,
SH_TET, COLOR1, P4, P0, P3, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_PYR, COLOR1, P3, P1, EB, EC, N0,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_PYR, COLOR1, EH, P4, P3, EK, N0,
SH_TET, COLOR1, EC, EK, P3, N0,
// Case #60: Unique case #14
SH_WDG, COLOR0, P1, EB, EJ, P0, ED, EI,
SH_WDG, COLOR0, P6, EF, EL, P7, EH, EK,
SH_HEX, COLOR1, P3, P4, P5, P2, EK, EH, EF, EL,
SH_HEX, COLOR1, ED, EI, EJ, EB, P3, P4, P5, P2,
// Case #61: Unique case #15
SH_PNT, 0, COLOR1, 6, P0, P2, P3, P4, EF, EH,
SH_WDG, COLOR0, EH, P7, EK, EF, P6, EL,
SH_TET, COLOR0, EA, P1, EB, EJ,
SH_WDG, COLOR1, P0, P5, P2, EA, EJ, EB,
SH_PYR, COLOR1, EH, EF, P5, P4, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_TET, COLOR1, P4, P0, P3, N0,
SH_PYR, COLOR1, EK, EH, P4, P3, N0,
SH_PYR, COLOR1, EL, EK, P3, P2, N0,
SH_TET, COLOR1, P3, P0, P2, N0,
SH_PYR, COLOR1, EF, EH, EK, EL, N0,
SH_TET, COLOR1, P2, P0, P5, N0,
SH_PYR, COLOR1, EF, EL, P2, P5, N0,
// Case #62: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P3, P2, P5, EH, EF,
SH_WDG, COLOR0, EH, P7, EK, EF, P6, EL,
SH_TET, COLOR0, EA, ED, P0, EI,
SH_WDG, COLOR1, EA, EI, ED, P1, P4, P3,
SH_PYR, COLOR1, EF, P5, P4, EH, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_TET, COLOR1, P5, P2, P1, N0,
SH_PYR, COLOR1, EL, P2, P5, EF, N0,
SH_PYR, COLOR1, EK, P3, P2, EL, N0,
SH_TET, COLOR1, P2, P3, P1, N0,
SH_PYR, COLOR1, EH, EK, EL, EF, N0,
SH_TET, COLOR1, P3, P4, P1, N0,
SH_PYR, COLOR1, EH, P4, P3, EK, N0,
// Case #63: Unique case #16
SH_WDG, COLOR0, P7, EK, EH, P6, EL, EF,
SH_HEX, COLOR1, P3, P4, P5, P2, EK, EH, EF, EL,
SH_WDG, COLOR1, P1, P2, P5, P0, P3, P4,
// Case #64: (cloned #1)
SH_PNT, 0, COLOR0, 7, P7, P4, P5, P2, P3, P0, P1,
SH_WDG, COLOR0, EG, EF, EL, P7, P5, P2,
SH_TET, COLOR0, P7, P2, P5, N0,
SH_TET, COLOR0, P7, P5, P4, N0,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_PYR, COLOR0, P3, P7, P4, P0, N0,
SH_PYR, COLOR0, P2, P3, P0, P1, N0,
SH_TET, COLOR0, P5, P2, P1, N0,
SH_TET, COLOR0, P2, P7, P3, N0,
SH_TET, COLOR1, P6, EF, EG, EL,
// Case #65: (cloned #20)
SH_WDG, COLOR0, P5, P7, P2, EF, EG, EL,
SH_WDG, COLOR0, EI, ED, EA, P4, P3, P1,
SH_TET, COLOR0, P7, P2, P5, P3,
SH_TET, COLOR0, P1, P2, P3, P5,
SH_TET, COLOR0, P4, P5, P1, P3,
SH_TET, COLOR0, P7, P4, P3, P5,
SH_TET, COLOR1, P0, EI, EA, ED,
SH_TET, COLOR1, P6, EF, EG, EL,
// Case #66: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EA, EG,
SH_PYR, COLOR0, P0, P3, P7, P4, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_TET, COLOR0, P2, P3, P0, N0,
SH_TET, COLOR0, P7, P3, P2, N0,
SH_PYR, COLOR0, P7, P2, EL, EG, N0,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_PYR, COLOR0, P0, EA, EB, P2, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_PYR, COLOR1, P1, P6, EL, EB, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_TET, COLOR1, EF, EG, P6, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
SH_TET, COLOR1, EJ, N0, P1, EA,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #67: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EI, ED, EG, EG,
SH_PYR, COLOR0, P7, P2, EL, EG, N0,
SH_TET, COLOR0, EL, P2, EB, N0,
SH_PYR, COLOR0, P3, ED, EB, P2, N0,
SH_TET, COLOR0, P7, P3, P2, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_TET, COLOR0, P4, P3, P7, N0,
SH_PYR, COLOR0, P4, EI, ED, P3, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_PYR, COLOR1, P6, EL, EB, P1, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
SH_TET, COLOR1, EF, EG, P6, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, P1, P0, EI, EJ, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #68: (cloned #3)
SH_HEX, COLOR0, EG, P7, P3, EC, EF, P5, P1, EB,
SH_WDG, COLOR0, P1, P0, P3, P5, P4, P7,
SH_WDG, COLOR1, P2, EB, EC, P6, EF, EG,
// Case #69: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EG, EF, EI, EI,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_TET, COLOR0, P7, P4, P3, N0,
SH_PYR, COLOR0, P3, P4, EI, ED, N0,
SH_TET, COLOR0, P7, P5, P4, N0,
SH_PYR, COLOR0, P7, EG, EF, P5, N0,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_PYR, COLOR1, P0, EA, EB, P2, N0,
SH_PYR, COLOR1, EC, ED, P0, P2, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_TET, COLOR1, ED, EI, P0, N0,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_PYR, COLOR1, P2, P6, EG, EC, N0,
SH_TET, COLOR1, EG, P6, EF, N0,
// Case #70: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EJ, EF, EG,
SH_PYR, COLOR0, P0, P3, P7, P4, N0,
SH_TET, COLOR0, P7, N0, P5, P4,
SH_PYR, COLOR0, P3, EC, EG, P7, N0,
SH_PYR, COLOR0, EA, EC, P3, P0, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, EG, EF, P5, P7, N0,
SH_PYR, COLOR1, EC, P2, P6, EG, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_TET, COLOR1, P6, EF, EG, N0,
// Case #71: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EJ, EF, EG, EC, ED, EI,
SH_TET, COLOR0, P7, P4, P3, N0,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_PYR, COLOR0, EI, ED, P3, P4, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, P0, P1, P2, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
SH_PYR, COLOR1, P0, P2, EC, ED, N0,
SH_PYR, COLOR1, EC, P2, P6, EG, N0,
SH_TET, COLOR1, EG, P6, EF, N0,
// Case #72: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, ED, EF,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_TET, COLOR0, P1, P5, P2, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_TET, COLOR0, P7, P4, P0, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_PYR, COLOR0, P2, P5, EF, EL, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_PYR, COLOR0, P0, P2, EC, ED, N0,
SH_PYR, COLOR0, P0, ED, EK, P7, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_PYR, COLOR1, P3, P6, EG, EK, N0,
SH_PYR, COLOR1, EC, EL, P6, P3, N0,
SH_TET, COLOR1, EL, EF, P6, N0,
SH_TET, COLOR1, P6, EF, EG, N0,
SH_TET, COLOR1, EC, N0, P3, ED,
SH_TET, COLOR1, EK, ED, P3, N0,
// Case #73: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EA, EI, EF, EF,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_TET, COLOR0, EG, P7, EK, N0,
SH_PYR, COLOR0, P4, EI, EK, P7, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_TET, COLOR0, P1, P5, P2, N0,
SH_PYR, COLOR0, P2, P5, EF, EL, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_PYR, COLOR0, P1, P2, EC, EA, N0,
SH_PYR, COLOR1, P6, EG, EK, P3, N0,
SH_PYR, COLOR1, EC, EL, P6, P3, N0,
SH_TET, COLOR1, P6, EF, EG, N0,
SH_TET, COLOR1, EL, EF, P6, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_PYR, COLOR1, P3, P0, EA, EC, N0,
SH_TET, COLOR1, EA, P0, EI, N0,
// Case #74: (cloned #26)
SH_TET, COLOR0, P2, EB, EL, EC,
SH_TET, COLOR0, P7, P4, P0, P5,
SH_PYR, COLOR0, EJ, EA, P0, P5, ED,
SH_PYR, COLOR0, EF, P5, P7, EG, EK,
SH_PYR, COLOR0, P0, ED, EK, P7, P5,
SH_PYR, COLOR0, ED, EJ, EF, EK, P5,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_PYR, COLOR1, P1, P3, ED, EA, EJ,
SH_PYR, COLOR1, EK, P3, P6, EG, EF,
SH_PYR, COLOR1, EJ, EF, P6, P1, P3,
SH_PYR, COLOR1, EJ, ED, EK, EF, P3,
// Case #75: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EJ, EF, EG,
SH_TET, COLOR0, P2, EC, EB, EL,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR0, P4, P7, P5, N0,
SH_PYR, COLOR0, P5, P7, EG, EF, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_WDG, COLOR1, EB, EC, EL, P1, P3, P6,
SH_TET, COLOR1, P3, P1, P6, N0,
SH_PYR, COLOR1, P1, EJ, EF, P6, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_PYR, COLOR1, P6, EG, EK, P3, N0,
SH_TET, COLOR1, EF, EG, P6, N0,
// Case #76: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, ED, EB, EK, EG, EF,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_TET, COLOR0, P5, P7, N0, P4,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_PYR, COLOR0, ED, P0, P1, EB, N0,
SH_TET, COLOR0, P7, P4, P0, N0,
SH_PYR, COLOR0, P7, P0, ED, EK, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_PYR, COLOR0, EF, P5, P7, EG, N0,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_TET, COLOR1, EK, ED, P3, N0,
SH_TET, COLOR1, P3, P2, P6, N0,
SH_PYR, COLOR1, EK, P3, P6, EG, N0,
SH_TET, COLOR1, P6, EF, EG, N0,
// Case #77: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EK, EG, EF, EB, EA, EI,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, P7, P5, P4, N0,
SH_PYR, COLOR0, P7, EG, EF, P5, N0,
SH_TET, COLOR0, EK, EG, P7, N0,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_PYR, COLOR0, EI, P4, P1, EA, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR1, P3, P2, P6, N0,
SH_PYR, COLOR1, EK, P3, P6, EG, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_TET, COLOR1, P0, P2, P3, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_PYR, COLOR1, P0, EA, EB, P2, N0,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_TET, COLOR1, EF, EG, P6, N0,
// Case #78: (cloned #27)
SH_TET, COLOR0, P7, P4, P0, P5,
SH_PYR, COLOR0, EJ, EA, P0, P5, ED,
SH_PYR, COLOR0, EF, P5, P7, EG, EK,
SH_PYR, COLOR0, P0, ED, EK, P7, P5,
SH_PYR, COLOR0, ED, EJ, EF, EK, P5,
SH_TET, COLOR1, P2, P1, P6, P3,
SH_PYR, COLOR1, ED, EA, P1, P3, EJ,
SH_PYR, COLOR1, EK, P3, P6, EG, EF,
SH_PYR, COLOR1, P1, EJ, EF, P6, P3,
SH_PYR, COLOR1, ED, EK, EF, EJ, P3,
// Case #79: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EK, EI, EJ, EG, EF,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_PYR, COLOR0, P7, P4, EI, EK, N0,
SH_TET, COLOR0, EG, P7, EK, N0,
SH_PYR, COLOR0, EF, P5, P7, EG, N0,
SH_TET, COLOR0, P5, P4, P7, N0,
SH_PYR, COLOR1, P2, P3, P0, P1, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
SH_TET, COLOR1, P6, P3, P2, N0,
SH_PYR, COLOR1, P6, EG, EK, P3, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_TET, COLOR1, EG, P6, EF, N0,
// Case #80: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EI, EL,
SH_PYR, COLOR0, P0, P1, P2, P3, N0,
SH_TET, COLOR0, P1, P5, P2, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_TET, COLOR0, P7, P0, P3, N0,
SH_TET, COLOR0, P2, P7, P3, N0,
SH_PYR, COLOR0, P2, EL, EG, P7, N0,
SH_PYR, COLOR0, P5, EF, EL, P2, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_PYR, COLOR0, P0, P7, EH, EI, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_PYR, COLOR1, P4, EH, EG, P6, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_TET, COLOR1, EF, P6, EL, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
SH_TET, COLOR1, EE, P4, N0, EI,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #81: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EA, ED, EL, EL,
SH_PYR, COLOR0, P2, EL, EG, P7, N0,
SH_TET, COLOR0, EG, EH, P7, N0,
SH_PYR, COLOR0, P3, P7, EH, ED, N0,
SH_TET, COLOR0, P2, P7, P3, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR0, P1, P5, P2, N0,
SH_PYR, COLOR0, P5, EF, EL, P2, N0,
SH_TET, COLOR0, P1, P2, P3, N0,
SH_PYR, COLOR0, P1, P3, ED, EA, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_PYR, COLOR1, P6, P4, EH, EG, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
SH_TET, COLOR1, EF, P6, EL, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, P4, EE, EA, P0, N0,
SH_TET, COLOR1, EA, ED, P0, N0,
// Case #82: (cloned #26)
SH_TET, COLOR0, P5, EF, EJ, EE,
SH_TET, COLOR0, P7, P0, P3, P2,
SH_PYR, COLOR0, EB, P2, P0, EA, EI,
SH_PYR, COLOR0, EL, EG, P7, P2, EH,
SH_PYR, COLOR0, P0, P7, EH, EI, P2,
SH_PYR, COLOR0, EI, EH, EL, EB, P2,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_PYR, COLOR1, P1, EA, EI, P4, EB,
SH_PYR, COLOR1, EH, EG, P6, P4, EL,
SH_PYR, COLOR1, EB, P1, P6, EL, P4,
SH_PYR, COLOR1, EB, EL, EH, EI, P4,
// Case #83: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EB, EL, EG,
SH_TET, COLOR0, P5, EJ, EE, EF,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_PYR, COLOR0, P3, ED, EB, P2, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_TET, COLOR0, P3, P2, P7, N0,
SH_PYR, COLOR0, P2, EL, EG, P7, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_WDG, COLOR1, P1, P4, P6, EJ, EE, EF,
SH_TET, COLOR1, P4, P6, P1, N0,
SH_PYR, COLOR1, P1, P6, EL, EB, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, P6, P4, EH, EG, N0,
SH_TET, COLOR1, EL, P6, EG, N0,
// Case #84: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EC, EB, EI, EI,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_TET, COLOR0, P3, P7, P0, N0,
SH_PYR, COLOR0, P7, EH, EI, P0, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_PYR, COLOR0, P3, P1, EB, EC, N0,
SH_PYR, COLOR0, P3, EC, EG, P7, N0,
SH_PYR, COLOR1, P4, P6, EF, EE, N0,
SH_PYR, COLOR1, EG, P6, P4, EH, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_PYR, COLOR1, P6, EG, EC, P2, N0,
SH_TET, COLOR1, EC, EB, P2, N0,
// Case #85: (cloned #60)
SH_WDG, COLOR0, P7, EH, EG, P3, ED, EC,
SH_WDG, COLOR0, P5, EF, EE, P1, EB, EA,
SH_HEX, COLOR1, P0, P2, P6, P4, EA, EB, EF, EE,
SH_HEX, COLOR1, ED, EC, EG, EH, P0, P2, P6, P4,
// Case #86: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EA, EI, EH,
SH_TET, COLOR0, P5, EF, EJ, EE,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_TET, COLOR0, P3, P7, P0, N0,
SH_PYR, COLOR0, P0, P7, EH, EI, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_TET, COLOR1, P6, P1, P4, N0,
SH_PYR, COLOR1, P1, EA, EI, P4, N0,
SH_TET, COLOR1, P2, P1, P6, N0,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_PYR, COLOR1, P4, EH, EG, P6, N0,
SH_TET, COLOR1, EI, EH, P4, N0,
// Case #87: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P4, P0, P2, EG, EC,
SH_WDG, COLOR0, EG, P7, EH, EC, P3, ED,
SH_TET, COLOR0, EJ, EE, P5, EF,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_PYR, COLOR1, EC, P2, P6, EG, N0,
SH_TET, COLOR1, P2, P1, P6, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_PYR, COLOR1, ED, P0, P2, EC, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EG, EH, ED, EC, N0,
SH_TET, COLOR1, P4, P6, P1, N0,
SH_PYR, COLOR1, EG, P6, P4, EH, N0,
// Case #88: (cloned #26)
SH_TET, COLOR0, P7, EG, EH, EK,
SH_TET, COLOR0, P2, P0, P1, P5,
SH_PYR, COLOR0, EE, P5, P0, EI, ED,
SH_PYR, COLOR0, EF, EL, P2, P5, EC,
SH_PYR, COLOR0, P0, P2, EC, ED, P5,
SH_PYR, COLOR0, ED, EC, EF, EE, P5,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_PYR, COLOR1, P4, EI, ED, P3, EE,
SH_PYR, COLOR1, EC, EL, P6, P3, EF,
SH_PYR, COLOR1, EE, P4, P6, EF, P3,
SH_PYR, COLOR1, EE, EF, EC, ED, P3,
// Case #89: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EE, EF, EL,
SH_TET, COLOR0, P7, EH, EK, EG,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR0, P1, P5, P2, N0,
SH_PYR, COLOR0, P5, EF, EL, P2, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_WDG, COLOR1, P4, P3, P6, EH, EK, EG,
SH_TET, COLOR1, P3, P6, P4, N0,
SH_PYR, COLOR1, P4, P6, EF, EE, N0,
SH_TET, COLOR1, P0, P3, P4, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_PYR, COLOR1, P6, P3, EC, EL, N0,
SH_TET, COLOR1, EF, P6, EL, N0,
// Case #90: Unique case #17
SH_TET, COLOR0, EH, EG, EK, P7,
SH_TET, COLOR0, EI, ED, EA, P0,
SH_TET, COLOR0, EE, EJ, EF, P5,
SH_TET, COLOR0, EB, EC, EL, P2,
SH_WDG, COLOR1, EB, EC, EL, P1, P3, P6,
SH_TET, COLOR1, P1, P6, P3, P4,
SH_WDG, COLOR1, P4, P6, P1, EE, EF, EJ,
SH_WDG, COLOR1, P3, P4, P1, ED, EI, EA,
SH_WDG, COLOR1, P6, P4, P3, EG, EH, EK,
// Case #91: Unique case #18
SH_TET, COLOR0, EE, EJ, EF, P5,
SH_TET, COLOR0, EH, EK, P7, EG,
SH_TET, COLOR0, EB, P2, EC, EL,
SH_WDG, COLOR1, P6, P3, P1, EL, EC, EB,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_WDG, COLOR1, P4, P6, P1, EE, EF, EJ,
SH_TET, COLOR1, P6, P3, P1, P4,
SH_TET, COLOR1, P4, P1, P0, P3,
// Case #92: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EB, EF, ED, EI, EE,
SH_TET, COLOR0, P7, EK, EG, EH,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_TET, COLOR0, P1, P0, P5, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_WDG, COLOR1, P3, P6, P4, EK, EG, EH,
SH_TET, COLOR1, P6, P4, P3, N0,
SH_PYR, COLOR1, P3, P4, EI, ED, N0,
SH_TET, COLOR1, P2, P6, P3, N0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_PYR, COLOR1, P4, P6, EF, EE, N0,
SH_TET, COLOR1, EI, P4, EE, N0,
// Case #93: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P4, P0, P2, EF, EB,
SH_WDG, COLOR0, EB, P1, EA, EF, P5, EE,
SH_TET, COLOR0, EK, P7, EH, EG,
SH_WDG, COLOR1, P3, P6, P4, EK, EG, EH,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_TET, COLOR1, P2, P6, P3, N0,
SH_TET, COLOR1, P2, P3, P0, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, P0, P3, P4, N0,
SH_PYR, COLOR1, EF, EB, EA, EE, N0,
SH_TET, COLOR1, P4, P3, P6, N0,
SH_PYR, COLOR1, EF, EE, P4, P6, N0,
// Case #94: (cloned #91)
SH_TET, COLOR0, EK, EH, EG, P7,
SH_TET, COLOR0, ED, P0, EA, EI,
SH_TET, COLOR0, EF, EJ, P5, EE,
SH_WDG, COLOR1, EE, EJ, EF, P4, P1, P6,
SH_WDG, COLOR1, P3, P4, P1, ED, EI, EA,
SH_WDG, COLOR1, EK, EH, EG, P3, P4, P6,
SH_TET, COLOR1, P4, P6, P1, P3,
SH_TET, COLOR1, P3, P2, P6, P1,
// Case #95: Unique case #19
SH_TET, COLOR0, EG, EK, EH, P7,
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_WDG, COLOR1, P4, P6, P1, EE, EF, EJ,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_PYR, COLOR1, P0, P1, P2, P3, P4,
SH_TET, COLOR1, P6, P3, P2, P4,
SH_TET, COLOR1, P6, P2, P1, P4,
// Case #96: (cloned #3)
SH_HEX, COLOR0, EL, P2, P1, EJ, EG, P7, P4, EE,
SH_WDG, COLOR0, P2, P3, P7, P1, P0, P4,
SH_WDG, COLOR1, P6, EL, EG, P5, EJ, EE,
// Case #97: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EG, EL, ED, ED,
SH_PYR, COLOR0, P3, ED, EA, P1, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, P2, P1, EJ, EL, N0,
SH_TET, COLOR0, P3, P1, P2, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR0, P7, P4, P3, N0,
SH_PYR, COLOR0, P4, EI, ED, P3, N0,
SH_TET, COLOR0, P7, P3, P2, N0,
SH_PYR, COLOR0, P7, P2, EL, EG, N0,
SH_PYR, COLOR0, P7, EG, EE, P4, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EE, P5, P0, EI, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_PYR, COLOR1, P5, EE, EG, P6, N0,
SH_TET, COLOR1, EG, EL, P6, N0,
// Case #98: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EA, EE, EB, EL, EG,
SH_PYR, COLOR0, P0, P3, P7, P4, N0,
SH_TET, COLOR0, P7, P2, N0, P3,
SH_PYR, COLOR0, P4, P7, EG, EE, N0,
SH_PYR, COLOR0, EA, P0, P4, EE, N0,
SH_TET, COLOR0, P2, P3, P0, N0,
SH_PYR, COLOR0, P2, P0, EA, EB, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_PYR, COLOR0, EG, P7, P2, EL, N0,
SH_PYR, COLOR1, EE, EG, P6, P5, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
SH_PYR, COLOR1, EB, P1, P6, EL, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
// Case #99: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EB, EL, EG, EE, EI, ED,
SH_TET, COLOR0, P7, P4, P3, N0,
SH_PYR, COLOR0, P7, EG, EE, P4, N0,
SH_TET, COLOR0, P2, P7, P3, N0,
SH_PYR, COLOR0, P2, EL, EG, P7, N0,
SH_TET, COLOR0, EB, EL, P2, N0,
SH_PYR, COLOR0, P2, P3, ED, EB, N0,
SH_PYR, COLOR0, ED, P3, P4, EI, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
SH_PYR, COLOR1, EB, P1, P6, EL, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_TET, COLOR1, P0, ED, EI, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EE, EG, P6, P5, N0,
SH_TET, COLOR1, EG, EL, P6, N0,
// Case #100: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EB, EJ, EE,
SH_PYR, COLOR0, P3, P7, P4, P0, N0,
SH_TET, COLOR0, P4, N0, P1, P0,
SH_PYR, COLOR0, P7, EG, EE, P4, N0,
SH_PYR, COLOR0, EC, EG, P7, P3, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P3, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, EE, EJ, P1, P4, N0,
SH_PYR, COLOR1, EG, P6, P5, EE, N0,
SH_PYR, COLOR1, EC, P2, P6, EG, N0,
SH_TET, COLOR1, EB, P2, EC, N0,
SH_TET, COLOR1, P2, P5, P6, N0,
SH_PYR, COLOR1, EB, EJ, P5, P2, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
// Case #101: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EC, ED, EI,
SH_TET, COLOR0, P1, EJ, EB, EA,
SH_PYR, COLOR0, P4, P7, EG, EE, N0,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_TET, COLOR0, P7, P4, P3, N0,
SH_PYR, COLOR0, P3, P4, EI, ED, N0,
SH_TET, COLOR0, P4, EE, EI, N0,
SH_WDG, COLOR1, EB, EJ, EA, P2, P5, P0,
SH_TET, COLOR1, P5, P2, P0, N0,
SH_PYR, COLOR1, P2, EC, ED, P0, N0,
SH_TET, COLOR1, P6, P2, P5, N0,
SH_PYR, COLOR1, EE, EG, P6, P5, N0,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_TET, COLOR1, ED, EI, P0, N0,
// Case #102: (cloned #15)
SH_HEX, COLOR0, EA, EE, EG, EC, P0, P4, P7, P3,
SH_HEX, COLOR1, P1, P5, P6, P2, EA, EE, EG, EC,
// Case #103: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EC, EI, ED,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_TET, COLOR0, P3, ED, EC, N0,
SH_PYR, COLOR0, P4, P7, EG, EE, N0,
SH_TET, COLOR0, EI, P4, EE, N0,
SH_PYR, COLOR0, ED, P3, P4, EI, N0,
SH_TET, COLOR0, P3, P7, P4, N0,
SH_PYR, COLOR1, P1, P5, P6, P2, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_TET, COLOR1, P0, P5, P1, N0,
SH_PYR, COLOR1, P0, EI, EE, P5, N0,
SH_PYR, COLOR1, EE, EG, P6, P5, N0,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_PYR, COLOR1, EC, ED, P0, P2, N0,
SH_TET, COLOR1, EI, P0, ED, N0,
// Case #104: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EE, EJ, ED, ED,
SH_PYR, COLOR0, P0, P2, EC, ED, N0,
SH_TET, COLOR0, EC, P2, EL, N0,
SH_PYR, COLOR0, P1, EJ, EL, P2, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_TET, COLOR0, P4, P0, P7, N0,
SH_PYR, COLOR0, P7, P0, ED, EK, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_PYR, COLOR0, P4, P7, EG, EE, N0,
SH_PYR, COLOR1, P3, EC, EL, P6, N0,
SH_PYR, COLOR1, EG, EK, P3, P6, N0,
SH_TET, COLOR1, P3, ED, EC, N0,
SH_TET, COLOR1, EK, ED, P3, N0,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_PYR, COLOR1, P6, P5, EE, EG, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #105: (cloned #60)
SH_WDG, COLOR0, P4, EI, EE, P7, EK, EG,
SH_WDG, COLOR0, P1, EJ, EA, P2, EL, EC,
SH_HEX, COLOR1, EA, EJ, EL, EC, P0, P5, P6, P3,
SH_HEX, COLOR1, P0, P5, P6, P3, EI, EE, EG, EK,
// Case #106: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EA, ED, EK,
SH_TET, COLOR0, P2, EB, EL, EC,
SH_PYR, COLOR0, P7, EG, EE, P4, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_TET, COLOR0, P4, P0, P7, N0,
SH_PYR, COLOR0, P0, ED, EK, P7, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_TET, COLOR1, P6, P3, P1, N0,
SH_PYR, COLOR1, P1, P3, ED, EA, N0,
SH_TET, COLOR1, P5, P6, P1, N0,
SH_PYR, COLOR1, EG, P6, P5, EE, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, P3, P6, EG, EK, N0,
SH_TET, COLOR1, ED, P3, EK, N0,
// Case #107: (cloned #61)
SH_PNT, 0, COLOR1, 6, P1, P3, P0, P5, EG, EE,
SH_WDG, COLOR0, EE, P4, EI, EG, P7, EK,
SH_TET, COLOR0, EB, P2, EC, EL,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_PYR, COLOR1, EE, EG, P6, P5, N0,
SH_TET, COLOR1, P5, P6, P1, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_PYR, COLOR1, EI, EE, P5, P0, N0,
SH_PYR, COLOR1, EK, EI, P0, P3, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, EG, EE, EI, EK, N0,
SH_TET, COLOR1, P3, P1, P6, N0,
SH_PYR, COLOR1, EG, EK, P3, P6, N0,
// Case #108: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EB, EJ, EE, EG, EK, ED,
SH_TET, COLOR0, P4, P0, P7, N0,
SH_PYR, COLOR0, P4, P7, EG, EE, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, EB, P1, EJ, N0,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_PYR, COLOR0, ED, EK, P7, P0, N0,
SH_TET, COLOR0, P7, EK, EG, N0,
SH_TET, COLOR1, P2, P5, P6, N0,
SH_PYR, COLOR1, EB, EJ, P5, P2, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_TET, COLOR1, P3, P2, P6, N0,
SH_TET, COLOR1, P3, EK, ED, N0,
SH_PYR, COLOR1, P3, P6, EG, EK, N0,
SH_PYR, COLOR1, EG, P6, P5, EE, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #109: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P0, P3, P6, EE, EG,
SH_WDG, COLOR0, EE, P4, EI, EG, P7, EK,
SH_TET, COLOR0, EB, EA, P1, EJ,
SH_WDG, COLOR1, EB, EJ, EA, P2, P5, P0,
SH_PYR, COLOR1, EG, P6, P5, EE, N0,
SH_TET, COLOR1, P6, P2, P5, N0,
SH_TET, COLOR1, P6, P3, P2, N0,
SH_PYR, COLOR1, EK, P3, P6, EG, N0,
SH_PYR, COLOR1, EI, P0, P3, EK, N0,
SH_TET, COLOR1, P3, P0, P2, N0,
SH_PYR, COLOR1, EE, EI, EK, EG, N0,
SH_TET, COLOR1, P0, P5, P2, N0,
SH_PYR, COLOR1, EE, P5, P0, EI, N0,
// Case #110: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EA, EK, ED,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, P7, EG, EE, P4, N0,
SH_TET, COLOR0, EK, EG, P7, N0,
SH_PYR, COLOR0, ED, EK, P7, P0, N0,
SH_TET, COLOR0, P0, P7, P4, N0,
SH_PYR, COLOR1, P2, P1, P5, P6, N0,
SH_TET, COLOR1, P1, P2, P3, N0,
SH_TET, COLOR1, P3, P2, P6, N0,
SH_PYR, COLOR1, P3, P6, EG, EK, N0,
SH_PYR, COLOR1, EG, P6, P5, EE, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, EA, P1, P3, ED, N0,
SH_TET, COLOR1, EK, ED, P3, N0,
// Case #111: (cloned #63)
SH_WDG, COLOR0, P4, EI, EE, P7, EK, EG,
SH_HEX, COLOR1, P0, P5, P6, P3, EI, EE, EG, EK,
SH_WDG, COLOR1, P2, P3, P6, P1, P0, P5,
// Case #112: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EI, EJ, EH, EG, EL,
SH_PYR, COLOR0, P0, P1, P2, P3, N0,
SH_TET, COLOR0, P2, N0, P7, P3,
SH_PYR, COLOR0, P1, EJ, EL, P2, N0,
SH_PYR, COLOR0, EI, EJ, P1, P0, N0,
SH_TET, COLOR0, P7, P0, P3, N0,
SH_PYR, COLOR0, P7, EH, EI, P0, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_PYR, COLOR0, EL, EG, P7, P2, N0,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_PYR, COLOR1, EH, EG, P6, P4, N0,
SH_TET, COLOR1, P6, EG, EL, N0,
// Case #113: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EH, EG, EL, EJ, EA, ED,
SH_TET, COLOR0, P2, P3, P1, N0,
SH_PYR, COLOR0, P2, P1, EJ, EL, N0,
SH_TET, COLOR0, P7, P3, P2, N0,
SH_PYR, COLOR0, P7, P2, EL, EG, N0,
SH_TET, COLOR0, EH, P7, EG, N0,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_PYR, COLOR0, ED, EA, P1, P3, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_PYR, COLOR1, EH, EG, P6, P4, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_TET, COLOR1, EL, P6, EG, N0,
// Case #114: (cloned #27)
SH_TET, COLOR0, P7, P0, P3, P2,
SH_PYR, COLOR0, EB, P2, P0, EA, EI,
SH_PYR, COLOR0, EL, EG, P7, P2, EH,
SH_PYR, COLOR0, P0, P7, EH, EI, P2,
SH_PYR, COLOR0, EI, EH, EL, EB, P2,
SH_TET, COLOR1, P5, P6, P1, P4,
SH_PYR, COLOR1, EI, P4, P1, EA, EB,
SH_PYR, COLOR1, EH, EG, P6, P4, EL,
SH_PYR, COLOR1, P1, P6, EL, EB, P4,
SH_PYR, COLOR1, EI, EB, EL, EH, P4,
// Case #115: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EH, ED, EB, EG, EL,
SH_PYR, COLOR0, P3, ED, EB, P2, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_PYR, COLOR0, P7, EH, ED, P3, N0,
SH_TET, COLOR0, EG, EH, P7, N0,
SH_PYR, COLOR0, EL, EG, P7, P2, N0,
SH_TET, COLOR0, P2, P7, P3, N0,
SH_PYR, COLOR1, P5, P1, P0, P4, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
SH_TET, COLOR1, P6, P5, P4, N0,
SH_PYR, COLOR1, P6, P4, EH, EG, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, EB, P1, P6, EL, N0,
SH_TET, COLOR1, EG, EL, P6, N0,
// Case #116: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EJ, EB, EC, EG, EH, EI,
SH_TET, COLOR0, P3, P7, P0, N0,
SH_PYR, COLOR0, P3, EC, EG, P7, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P3, N0,
SH_TET, COLOR0, EJ, EB, P1, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_PYR, COLOR0, EI, P0, P7, EH, N0,
SH_TET, COLOR0, P7, EG, EH, N0,
SH_TET, COLOR1, P5, P6, P2, N0,
SH_PYR, COLOR1, EJ, P5, P2, EB, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_TET, COLOR1, P4, EI, EH, N0,
SH_PYR, COLOR1, P4, EH, EG, P6, N0,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_TET, COLOR1, EC, EB, P2, N0,
// Case #117: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P0, P4, P6, EC, EG,
SH_WDG, COLOR0, EG, P7, EH, EC, P3, ED,
SH_TET, COLOR0, EJ, P1, EA, EB,
SH_WDG, COLOR1, P5, P2, P0, EJ, EB, EA,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_TET, COLOR1, P6, P2, P5, N0,
SH_TET, COLOR1, P6, P5, P4, N0,
SH_PYR, COLOR1, EH, EG, P6, P4, N0,
SH_PYR, COLOR1, ED, EH, P4, P0, N0,
SH_TET, COLOR1, P4, P5, P0, N0,
SH_PYR, COLOR1, EC, EG, EH, ED, N0,
SH_TET, COLOR1, P0, P5, P2, N0,
SH_PYR, COLOR1, EC, ED, P0, P2, N0,
// Case #118: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EA, EH, EI,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, P7, P3, EC, EG, N0,
SH_TET, COLOR0, EH, P7, EG, N0,
SH_PYR, COLOR0, EI, P0, P7, EH, N0,
SH_TET, COLOR0, P0, P3, P7, N0,
SH_PYR, COLOR1, P5, P6, P2, P1, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
SH_PYR, COLOR1, P4, EH, EG, P6, N0,
SH_PYR, COLOR1, EG, EC, P2, P6, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_TET, COLOR1, EH, P4, EI, N0,
// Case #119: (cloned #63)
SH_WDG, COLOR0, P7, EH, EG, P3, ED, EC,
SH_HEX, COLOR1, ED, EC, EG, EH, P0, P2, P6, P4,
SH_WDG, COLOR1, P1, P0, P2, P5, P4, P6,
// Case #120: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EI, ED, EC,
SH_TET, COLOR0, P7, EG, EH, EK,
SH_PYR, COLOR0, P2, P1, EJ, EL, N0,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR0, P0, P2, EC, ED, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_TET, COLOR1, P6, P4, P3, N0,
SH_PYR, COLOR1, P4, EI, ED, P3, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, P3, EC, EL, P6, N0,
SH_TET, COLOR1, ED, EC, P3, N0,
// Case #121: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P3, P0, P5, EL, EJ,
SH_WDG, COLOR0, EL, P2, EC, EJ, P1, EA,
SH_TET, COLOR0, EH, EK, P7, EG,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_TET, COLOR1, P5, P0, P4, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_TET, COLOR1, P0, P3, P4, N0,
SH_PYR, COLOR1, EL, EC, EA, EJ, N0,
SH_TET, COLOR1, P3, P6, P4, N0,
SH_PYR, COLOR1, EL, P6, P3, EC, N0,
// Case #122: (cloned #91)
SH_TET, COLOR0, EH, EG, EK, P7,
SH_TET, COLOR0, EI, EA, P0, ED,
SH_TET, COLOR0, EL, P2, EB, EC,
SH_WDG, COLOR1, P3, P1, P6, EC, EB, EL,
SH_WDG, COLOR1, EI, ED, EA, P4, P3, P1,
SH_WDG, COLOR1, P4, P3, P6, EH, EK, EG,
SH_TET, COLOR1, P3, P1, P6, P4,
SH_TET, COLOR1, P4, P6, P5, P1,
// Case #123: (cloned #95)
SH_TET, COLOR0, EG, EK, EH, P7,
SH_TET, COLOR0, EL, EB, EC, P2,
SH_WDG, COLOR1, EC, EL, EB, P3, P6, P1,
SH_WDG, COLOR1, P3, P6, P4, EK, EG, EH,
SH_PYR, COLOR1, P0, P4, P5, P1, P3,
SH_TET, COLOR1, P6, P5, P4, P3,
SH_TET, COLOR1, P6, P1, P5, P3,
// Case #124: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P4, P5, P2, ED, EB,
SH_WDG, COLOR0, ED, P0, EI, EB, P1, EJ,
SH_TET, COLOR0, EG, EH, P7, EK,
SH_WDG, COLOR1, EG, EK, EH, P6, P3, P4,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_TET, COLOR1, P2, P6, P3, N0,
SH_TET, COLOR1, P2, P5, P6, N0,
SH_PYR, COLOR1, EJ, P5, P2, EB, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_TET, COLOR1, P5, P4, P6, N0,
SH_PYR, COLOR1, ED, EI, EJ, EB, N0,
SH_TET, COLOR1, P4, P3, P6, N0,
SH_PYR, COLOR1, ED, P3, P4, EI, N0,
// Case #125: Unique case #20
SH_TET, COLOR0, EG, EK, EH, P7,
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_WDG, COLOR1, EJ, EA, EB, P5, P0, P2,
SH_TET, COLOR1, P2, P3, P0, P5,
SH_TET, COLOR1, P5, P4, P6, P3,
SH_TET, COLOR1, P4, P5, P0, P3,
SH_TET, COLOR1, P5, P6, P2, P3,
// Case #126: (cloned #95)
SH_TET, COLOR0, EI, ED, EA, P0,
SH_TET, COLOR0, EH, EG, EK, P7,
SH_WDG, COLOR1, EK, EH, EG, P3, P4, P6,
SH_WDG, COLOR1, P3, P4, P1, ED, EI, EA,
SH_PYR, COLOR1, P2, P1, P5, P6, P3,
SH_TET, COLOR1, P4, P5, P1, P3,
SH_TET, COLOR1, P4, P6, P5, P3,
// Case #127: Unique case #21
SH_PNT, 0, COLOR1, 7, P0, P1, P2, P3, P4, P5, P6,
SH_TET, COLOR0, EH, EG, EK, P7,
SH_WDG, COLOR1, EH, EG, EK, P4, P6, P3,
SH_TET, COLOR1, P4, P3, P6, N0,
SH_PYR, COLOR1, P5, P6, P2, P1, N0,
SH_TET, COLOR1, P6, P3, P2, N0,
SH_PYR, COLOR1, P0, P1, P2, P3, N0,
SH_TET, COLOR1, P0, P3, P4, N0,
SH_PYR, COLOR1, P0, P4, P5, P1, N0,
SH_TET, COLOR1, P4, P6, P5, N0,
// Case #128: (cloned #1)
SH_PNT, 0, COLOR0, 7, P6, P5, P4, P3, P2, P1, P0,
SH_WDG, COLOR0, P6, P4, P3, EG, EH, EK,
SH_TET, COLOR0, P6, P4, P3, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P1, P0, P4, P5, N0,
SH_PYR, COLOR0, P2, P1, P5, P6, N0,
SH_PYR, COLOR0, P3, P0, P1, P2, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_TET, COLOR0, P3, P2, P6, N0,
SH_TET, COLOR1, P7, EG, EH, EK,
// Case #129: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EA, EG,
SH_PYR, COLOR0, P1, P5, P6, P2, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_TET, COLOR0, P3, P1, P2, N0,
SH_TET, COLOR0, P6, P3, P2, N0,
SH_PYR, COLOR0, P6, EG, EK, P3, N0,
SH_PYR, COLOR0, P4, EH, EG, P6, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_PYR, COLOR0, P1, P3, ED, EA, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_PYR, COLOR1, P0, ED, EK, P7, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_TET, COLOR1, EH, P7, EG, N0,
SH_TET, COLOR1, P7, EK, EG, N0,
SH_TET, COLOR1, EI, P0, N0, EA,
SH_TET, COLOR1, ED, P0, EA, N0,
// Case #130: (cloned #20)
SH_WDG, COLOR0, P4, P3, P6, EH, EK, EG,
SH_WDG, COLOR0, EA, EB, EJ, P0, P2, P5,
SH_TET, COLOR0, P3, P6, P4, P2,
SH_TET, COLOR0, P5, P6, P2, P4,
SH_TET, COLOR0, P0, P4, P5, P2,
SH_TET, COLOR0, P3, P0, P2, P4,
SH_TET, COLOR1, P1, EA, EJ, EB,
SH_TET, COLOR1, P7, EH, EK, EG,
// Case #131: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EJ, EB, EG, EG,
SH_PYR, COLOR0, P6, EG, EK, P3, N0,
SH_TET, COLOR0, EK, ED, P3, N0,
SH_PYR, COLOR0, P2, P3, ED, EB, N0,
SH_TET, COLOR0, P6, P3, P2, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P4, EH, EG, P6, N0,
SH_TET, COLOR0, P5, P6, P2, N0,
SH_PYR, COLOR0, P5, P2, EB, EJ, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_PYR, COLOR1, P7, P0, ED, EK, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_TET, COLOR1, P7, EK, EG, N0,
SH_TET, COLOR1, EH, P7, EG, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_PYR, COLOR1, P0, EI, EJ, P1, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
// Case #132: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EH, EB,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR0, P1, EB, EC, P3, N0,
SH_PYR, COLOR0, P6, EL, EB, P1, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_PYR, COLOR0, P4, EH, EG, P6, N0,
SH_PYR, COLOR0, P4, P3, EK, EH, N0,
SH_TET, COLOR0, P3, EC, EK, N0,
SH_PYR, COLOR1, P7, EK, EC, P2, N0,
SH_PYR, COLOR1, EG, P7, P2, EL, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
SH_TET, COLOR1, P2, EC, EB, N0,
SH_TET, COLOR1, EG, P7, N0, EH,
SH_TET, COLOR1, EK, P7, EH, N0,
// Case #133: (cloned #26)
SH_TET, COLOR0, P3, ED, EC, EK,
SH_TET, COLOR0, P6, P5, P4, P1,
SH_PYR, COLOR0, EA, EI, P4, P1, EH,
SH_PYR, COLOR0, EB, P1, P6, EL, EG,
SH_PYR, COLOR0, P4, EH, EG, P6, P1,
SH_PYR, COLOR0, EH, EA, EB, EG, P1,
SH_WDG, COLOR1, P0, P2, P7, ED, EC, EK,
SH_PYR, COLOR1, P0, P7, EH, EI, EA,
SH_PYR, COLOR1, EG, P7, P2, EL, EB,
SH_PYR, COLOR1, EA, EB, P2, P0, P7,
SH_PYR, COLOR1, EA, EH, EG, EB, P7,
// Case #134: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EJ, EA, EH, EH,
SH_PYR, COLOR0, P4, P3, EK, EH, N0,
SH_TET, COLOR0, EK, P3, EC, N0,
SH_PYR, COLOR0, P0, EA, EC, P3, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P6, P4, EH, EG, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_PYR, COLOR0, P5, P6, EL, EJ, N0,
SH_PYR, COLOR1, P7, EK, EC, P2, N0,
SH_PYR, COLOR1, EL, EG, P7, P2, N0,
SH_TET, COLOR1, P7, EH, EK, N0,
SH_TET, COLOR1, EG, EH, P7, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_PYR, COLOR1, P2, P1, EJ, EL, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #135: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EI, EH, EG,
SH_TET, COLOR0, P3, ED, EC, EK,
SH_PYR, COLOR0, P6, EL, EJ, P5, N0,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P4, EH, EG, P6, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_WDG, COLOR1, P0, P2, P7, ED, EC, EK,
SH_TET, COLOR1, P2, P7, P0, N0,
SH_PYR, COLOR1, P0, P7, EH, EI, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_PYR, COLOR1, EL, P2, P1, EJ, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_PYR, COLOR1, P7, P2, EL, EG, N0,
SH_TET, COLOR1, EH, P7, EG, N0,
// Case #136: (cloned #3)
SH_HEX, COLOR0, EH, P4, P0, ED, EG, P6, P2, EC,
SH_WDG, COLOR0, P4, P5, P6, P0, P1, P2,
SH_WDG, COLOR1, P7, EH, EG, P3, ED, EC,
// Case #137: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EI, EH, EG,
SH_PYR, COLOR0, P1, P5, P6, P2, N0,
SH_TET, COLOR0, P6, P4, N0, P5,
SH_PYR, COLOR0, P2, P6, EG, EC, N0,
SH_PYR, COLOR0, EA, P1, P2, EC, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, EG, P6, P4, EH, N0,
SH_PYR, COLOR1, EC, EG, P7, P3, N0,
SH_PYR, COLOR1, EA, EC, P3, P0, N0,
SH_TET, COLOR1, EI, EA, P0, N0,
SH_TET, COLOR1, P0, P3, P7, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_TET, COLOR1, P7, EG, EH, N0,
// Case #138: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EG, EH, EJ, EJ,
SH_PYR, COLOR0, P5, EJ, EA, P0, N0,
SH_TET, COLOR0, EA, ED, P0, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P5, P0, P4, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_PYR, COLOR0, P2, EB, EJ, P5, N0,
SH_TET, COLOR0, P6, P5, P4, N0,
SH_PYR, COLOR0, P6, P4, EH, EG, N0,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_PYR, COLOR1, P1, P3, ED, EA, N0,
SH_PYR, COLOR1, EC, P3, P1, EB, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
SH_PYR, COLOR1, ED, P3, P7, EH, N0,
SH_PYR, COLOR1, P3, EC, EG, P7, N0,
SH_TET, COLOR1, EG, EH, P7, N0,
// Case #139: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EI, EH, EG, EC, EB, EJ,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_PYR, COLOR0, P4, EH, EG, P6, N0,
SH_TET, COLOR0, EI, EH, P4, N0,
SH_PYR, COLOR0, P4, P5, EJ, EI, N0,
SH_PYR, COLOR0, EJ, P5, P2, EB, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_TET, COLOR1, P0, P3, P7, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_TET, COLOR1, P1, EJ, EB, N0,
SH_PYR, COLOR1, P1, EB, EC, P3, N0,
SH_PYR, COLOR1, EC, EG, P7, P3, N0,
SH_TET, COLOR1, EG, EH, P7, N0,
// Case #140: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EB, ED, EL, EG, EH,
SH_PYR, COLOR0, P1, P0, P4, P5, N0,
SH_TET, COLOR0, P4, N0, P6, P5,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_PYR, COLOR0, EB, ED, P0, P1, N0,
SH_TET, COLOR0, P6, P1, P5, N0,
SH_PYR, COLOR0, P6, EL, EB, P1, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_PYR, COLOR0, EH, EG, P6, P4, N0,
SH_PYR, COLOR1, ED, P3, P7, EH, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
SH_TET, COLOR1, P2, P7, P3, N0,
SH_PYR, COLOR1, EL, EG, P7, P2, N0,
SH_TET, COLOR1, P7, EG, EH, N0,
// Case #141: (cloned #27)
SH_TET, COLOR0, P6, P5, P4, P1,
SH_PYR, COLOR0, EA, EI, P4, P1, EH,
SH_PYR, COLOR0, EB, P1, P6, EL, EG,
SH_PYR, COLOR0, P4, EH, EG, P6, P1,
SH_PYR, COLOR0, EH, EA, EB, EG, P1,
SH_TET, COLOR1, P3, P0, P2, P7,
SH_PYR, COLOR1, EH, EI, P0, P7, EA,
SH_PYR, COLOR1, EG, P7, P2, EL, EB,
SH_PYR, COLOR1, P0, EA, EB, P2, P7,
SH_PYR, COLOR1, EH, EG, EB, EA, P7,
// Case #142: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, ED, EA, EJ, EL, EG, EH,
SH_TET, COLOR0, P5, P4, P6, N0,
SH_PYR, COLOR0, P5, P6, EL, EJ, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, ED, P0, EA, N0,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_PYR, COLOR0, EH, EG, P6, P4, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_TET, COLOR1, P3, P1, P2, N0,
SH_PYR, COLOR1, ED, EA, P1, P3, N0,
SH_PYR, COLOR1, ED, P3, P7, EH, N0,
SH_TET, COLOR1, P7, P3, P2, N0,
SH_TET, COLOR1, P7, EG, EH, N0,
SH_PYR, COLOR1, P7, P2, EL, EG, N0,
SH_PYR, COLOR1, EL, P2, P1, EJ, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #143: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EI, EG, EH,
SH_PYR, COLOR0, P5, EJ, EI, P4, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P6, EL, EJ, P5, N0,
SH_TET, COLOR0, EG, EL, P6, N0,
SH_PYR, COLOR0, EH, EG, P6, P4, N0,
SH_TET, COLOR0, P4, P6, P5, N0,
SH_PYR, COLOR1, P3, P0, P1, P2, N0,
SH_TET, COLOR1, P0, P3, P7, N0,
SH_TET, COLOR1, P7, P3, P2, N0,
SH_PYR, COLOR1, P7, P2, EL, EG, N0,
SH_PYR, COLOR1, EL, P2, P1, EJ, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_TET, COLOR1, EG, EH, P7, N0,
// Case #144: (cloned #3)
SH_HEX, COLOR0, EG, P6, P5, EE, EK, P3, P0, EI,
SH_WDG, COLOR0, P0, P1, P5, P3, P2, P6,
SH_WDG, COLOR1, P4, EI, EE, P7, EK, EG,
// Case #145: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EA, EE, ED, EK, EG,
SH_PYR, COLOR0, P1, P5, P6, P2, N0,
SH_TET, COLOR0, P6, N0, P3, P2,
SH_PYR, COLOR0, P5, EE, EG, P6, N0,
SH_PYR, COLOR0, EA, EE, P5, P1, N0,
SH_TET, COLOR0, P3, P1, P2, N0,
SH_PYR, COLOR0, P3, ED, EA, P1, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_PYR, COLOR0, EG, EK, P3, P6, N0,
SH_PYR, COLOR1, EE, P4, P7, EG, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_TET, COLOR1, ED, P0, EA, N0,
SH_TET, COLOR1, P0, P7, P4, N0,
SH_PYR, COLOR1, ED, EK, P7, P0, N0,
SH_TET, COLOR1, P7, EK, EG, N0,
// Case #146: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EG, EK, EB, EB,
SH_PYR, COLOR0, P2, P0, EA, EB, N0,
SH_TET, COLOR0, EA, P0, EI, N0,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_TET, COLOR0, P2, P3, P0, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_PYR, COLOR0, P5, P2, EB, EJ, N0,
SH_TET, COLOR0, P6, P3, P2, N0,
SH_PYR, COLOR0, P6, EG, EK, P3, N0,
SH_PYR, COLOR0, P6, P5, EE, EG, N0,
SH_PYR, COLOR1, P1, EA, EI, P4, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
SH_PYR, COLOR1, EI, EK, P7, P4, N0,
SH_PYR, COLOR1, P4, P7, EG, EE, N0,
SH_TET, COLOR1, EG, P7, EK, N0,
// Case #147: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, ED, EK, EG, EE, EJ, EB,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_PYR, COLOR0, P6, P5, EE, EG, N0,
SH_TET, COLOR0, P3, P2, P6, N0,
SH_PYR, COLOR0, P3, P6, EG, EK, N0,
SH_TET, COLOR0, ED, P3, EK, N0,
SH_PYR, COLOR0, P3, ED, EB, P2, N0,
SH_PYR, COLOR0, EB, EJ, P5, P2, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_TET, COLOR1, P0, P7, P4, N0,
SH_PYR, COLOR1, ED, EK, P7, P0, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_TET, COLOR1, P1, EJ, EB, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_PYR, COLOR1, EE, P4, P7, EG, N0,
SH_TET, COLOR1, EG, P7, EK, N0,
// Case #148: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EE, EI, EB, EB,
SH_PYR, COLOR0, P1, EB, EC, P3, N0,
SH_TET, COLOR0, EC, EK, P3, N0,
SH_PYR, COLOR0, P0, P3, EK, EI, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_PYR, COLOR0, P6, EL, EB, P1, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_PYR, COLOR0, P5, EE, EG, P6, N0,
SH_PYR, COLOR1, P2, P7, EK, EC, N0,
SH_PYR, COLOR1, EG, P7, P2, EL, N0,
SH_TET, COLOR1, P2, EC, EB, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_PYR, COLOR1, P7, EG, EE, P4, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #149: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EA, EB, EL,
SH_TET, COLOR0, P3, EK, ED, EC,
SH_PYR, COLOR0, P6, P5, EE, EG, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_PYR, COLOR0, P1, P6, EL, EB, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_WDG, COLOR1, ED, EK, EC, P0, P7, P2,
SH_TET, COLOR1, P7, P0, P2, N0,
SH_PYR, COLOR1, P0, EA, EB, P2, N0,
SH_TET, COLOR1, P4, P0, P7, N0,
SH_PYR, COLOR1, EG, EE, P4, P7, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_PYR, COLOR1, P2, EL, EG, P7, N0,
SH_TET, COLOR1, EB, EL, P2, N0,
// Case #150: (cloned #60)
SH_WDG, COLOR0, P3, EK, EC, P0, EI, EA,
SH_WDG, COLOR0, P6, EL, EG, P5, EJ, EE,
SH_HEX, COLOR1, P4, P1, P2, P7, EE, EJ, EL, EG,
SH_HEX, COLOR1, EI, EA, EC, EK, P4, P1, P2, P7,
// Case #151: (cloned #61)
SH_PNT, 0, COLOR1, 6, P0, P7, P4, P1, EL, EJ,
SH_WDG, COLOR0, EJ, P5, EE, EL, P6, EG,
SH_TET, COLOR0, ED, P3, EK, EC,
SH_WDG, COLOR1, P0, P2, P7, ED, EC, EK,
SH_PYR, COLOR1, EJ, EL, P2, P1, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, EE, EJ, P1, P4, N0,
SH_PYR, COLOR1, EG, EE, P4, P7, N0,
SH_TET, COLOR1, P4, P0, P7, N0,
SH_PYR, COLOR1, EL, EJ, EE, EG, N0,
SH_TET, COLOR1, P7, P0, P2, N0,
SH_PYR, COLOR1, EL, EG, P7, P2, N0,
// Case #152: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EC, EG, ED, EI, EE,
SH_PYR, COLOR0, P2, P1, P5, P6, N0,
SH_TET, COLOR0, P5, P0, N0, P1,
SH_PYR, COLOR0, P6, P5, EE, EG, N0,
SH_PYR, COLOR0, EC, P2, P6, EG, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_PYR, COLOR0, P0, P2, EC, ED, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, EE, P5, P0, EI, N0,
SH_PYR, COLOR1, EG, EE, P4, P7, N0,
SH_PYR, COLOR1, EC, EG, P7, P3, N0,
SH_TET, COLOR1, ED, EC, P3, N0,
SH_TET, COLOR1, P3, P7, P4, N0,
SH_PYR, COLOR1, ED, P3, P4, EI, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
// Case #153: (cloned #15)
SH_HEX, COLOR0, P1, P5, P6, P2, EA, EE, EG, EC,
SH_HEX, COLOR1, EA, EE, EG, EC, P0, P4, P7, P3,
// Case #154: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EC, EB, EJ,
SH_TET, COLOR0, P0, ED, EI, EA,
SH_PYR, COLOR0, P5, EE, EG, P6, N0,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_TET, COLOR0, P6, P2, P5, N0,
SH_PYR, COLOR0, P2, EB, EJ, P5, N0,
SH_TET, COLOR0, P5, EJ, EE, N0,
SH_WDG, COLOR1, P3, P4, P1, ED, EI, EA,
SH_TET, COLOR1, P4, P1, P3, N0,
SH_PYR, COLOR1, P3, P1, EB, EC, N0,
SH_TET, COLOR1, P7, P4, P3, N0,
SH_PYR, COLOR1, EE, P4, P7, EG, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_TET, COLOR1, EB, P1, EJ, N0,
// Case #155: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EE, EG, EC, EJ, EB,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_TET, COLOR0, P2, EC, EB, N0,
SH_PYR, COLOR0, P5, EE, EG, P6, N0,
SH_TET, COLOR0, EJ, EE, P5, N0,
SH_PYR, COLOR0, EB, EJ, P5, P2, N0,
SH_TET, COLOR0, P2, P5, P6, N0,
SH_PYR, COLOR1, P0, P3, P7, P4, N0,
SH_TET, COLOR1, P3, P0, P1, N0,
SH_TET, COLOR1, P1, P0, P4, N0,
SH_PYR, COLOR1, P1, P4, EE, EJ, N0,
SH_PYR, COLOR1, EE, P4, P7, EG, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_PYR, COLOR1, EC, P3, P1, EB, N0,
SH_TET, COLOR1, EJ, EB, P1, N0,
// Case #156: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, ED, EI, EE, EG, EL, EB,
SH_TET, COLOR0, P5, P6, P1, N0,
SH_PYR, COLOR0, P5, EE, EG, P6, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_PYR, COLOR0, P0, EI, EE, P5, N0,
SH_TET, COLOR0, ED, EI, P0, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_PYR, COLOR0, EB, P1, P6, EL, N0,
SH_TET, COLOR0, P6, EG, EL, N0,
SH_TET, COLOR1, P3, P7, P4, N0,
SH_PYR, COLOR1, ED, P3, P4, EI, N0,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_TET, COLOR1, P2, P7, P3, N0,
SH_TET, COLOR1, P2, EB, EL, N0,
SH_PYR, COLOR1, P2, EL, EG, P7, N0,
SH_PYR, COLOR1, EG, EE, P4, P7, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #157: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EE, EA, EL, EB,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_PYR, COLOR0, P6, P5, EE, EG, N0,
SH_TET, COLOR0, EL, P6, EG, N0,
SH_PYR, COLOR0, EB, P1, P6, EL, N0,
SH_TET, COLOR0, P1, P5, P6, N0,
SH_PYR, COLOR1, P3, P7, P4, P0, N0,
SH_TET, COLOR1, P0, P2, P3, N0,
SH_TET, COLOR1, P2, P7, P3, N0,
SH_PYR, COLOR1, P2, EL, EG, P7, N0,
SH_PYR, COLOR1, EG, EE, P4, P7, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_PYR, COLOR1, EA, EB, P2, P0, N0,
SH_TET, COLOR1, EL, P2, EB, N0,
// Case #158: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P4, P7, P2, EJ, EL,
SH_WDG, COLOR0, EJ, P5, EE, EL, P6, EG,
SH_TET, COLOR0, ED, EI, P0, EA,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_PYR, COLOR1, EL, P2, P1, EJ, N0,
SH_TET, COLOR1, P2, P3, P1, N0,
SH_TET, COLOR1, P2, P7, P3, N0,
SH_PYR, COLOR1, EG, P7, P2, EL, N0,
SH_PYR, COLOR1, EE, P4, P7, EG, N0,
SH_TET, COLOR1, P7, P4, P3, N0,
SH_PYR, COLOR1, EJ, EE, EG, EL, N0,
SH_TET, COLOR1, P4, P1, P3, N0,
SH_PYR, COLOR1, EJ, P1, P4, EE, N0,
// Case #159: (cloned #63)
SH_WDG, COLOR0, P5, EE, EJ, P6, EG, EL,
SH_HEX, COLOR1, P4, P1, P2, P7, EE, EJ, EL, EG,
SH_WDG, COLOR1, P3, P7, P2, P0, P4, P1,
// Case #160: (cloned #5)
SH_PNT, 0, NOCOLOR, 2, EK, EJ,
SH_PYR, COLOR0, P3, P0, P1, P2, N0,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_TET, COLOR0, P3, P2, P6, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_PYR, COLOR0, P6, P1, EJ, EF, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_PYR, COLOR0, P3, P6, EG, EK, N0,
SH_PYR, COLOR0, P3, EK, EH, P4, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR1, P7, P5, EE, EH, N0,
SH_PYR, COLOR1, EG, EF, P5, P7, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EG, N0, P7, EK,
SH_TET, COLOR1, EH, EK, P7, N0,
// Case #161: (cloned #26)
SH_TET, COLOR0, P4, EE, EI, EH,
SH_TET, COLOR0, P6, P3, P2, P1,
SH_PYR, COLOR0, EA, P1, P3, ED, EK,
SH_PYR, COLOR0, EJ, EF, P6, P1, EG,
SH_PYR, COLOR0, P3, P6, EG, EK, P1,
SH_PYR, COLOR0, EK, EG, EJ, EA, P1,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P7,
SH_PYR, COLOR1, P0, ED, EK, P7, EA,
SH_PYR, COLOR1, EG, EF, P5, P7, EJ,
SH_PYR, COLOR1, EA, P0, P5, EJ, P7,
SH_PYR, COLOR1, EA, EJ, EG, EK, P7,
// Case #162: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EB, EA, EK, EK,
SH_PYR, COLOR0, P3, EK, EH, P4, N0,
SH_TET, COLOR0, EH, EE, P4, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_TET, COLOR0, P2, P6, P3, N0,
SH_PYR, COLOR0, P6, EG, EK, P3, N0,
SH_TET, COLOR0, P2, P3, P0, N0,
SH_PYR, COLOR0, P2, P0, EA, EB, N0,
SH_PYR, COLOR0, P2, EB, EF, P6, N0,
SH_PYR, COLOR1, P7, P5, EE, EH, N0,
SH_PYR, COLOR1, EF, P5, P7, EG, N0,
SH_TET, COLOR1, P7, EH, EK, N0,
SH_TET, COLOR1, EG, P7, EK, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_PYR, COLOR1, P5, EF, EB, P1, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #163: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EB, EF, ED, EK, EG,
SH_TET, COLOR0, P4, EE, EI, EH,
SH_PYR, COLOR0, P6, P2, EB, EF, N0,
SH_PYR, COLOR0, P2, P3, ED, EB, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_TET, COLOR0, P2, P6, P3, N0,
SH_PYR, COLOR0, P3, P6, EG, EK, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P7,
SH_TET, COLOR1, P5, P0, P7, N0,
SH_PYR, COLOR1, P0, ED, EK, P7, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, P7, EG, EF, P5, N0,
SH_TET, COLOR1, EK, EG, P7, N0,
// Case #164: (cloned #26)
SH_TET, COLOR0, P6, EF, EG, EL,
SH_TET, COLOR0, P3, P0, P1, P4,
SH_PYR, COLOR0, EE, EJ, P1, P4, EB,
SH_PYR, COLOR0, EH, P4, P3, EK, EC,
SH_PYR, COLOR0, P1, EB, EC, P3, P4,
SH_PYR, COLOR0, EB, EE, EH, EC, P4,
SH_WDG, COLOR1, P5, P7, P2, EF, EG, EL,
SH_PYR, COLOR1, P5, P2, EB, EJ, EE,
SH_PYR, COLOR1, EC, P2, P7, EK, EH,
SH_PYR, COLOR1, EE, EH, P7, P5, P2,
SH_PYR, COLOR1, EE, EB, EC, EH, P2,
// Case #165: (cloned #90)
SH_TET, COLOR0, EH, EI, EE, P4,
SH_TET, COLOR0, EK, EC, ED, P3,
SH_TET, COLOR0, EG, EF, EL, P6,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, P2, P0, P5, EB, EA, EJ,
SH_TET, COLOR1, P2, P0, P5, P7,
SH_WDG, COLOR1, EG, EF, EL, P7, P5, P2,
SH_WDG, COLOR1, ED, EK, EC, P0, P7, P2,
SH_WDG, COLOR1, EE, EH, EI, P5, P7, P0,
// Case #166: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EE, EH, EK,
SH_TET, COLOR0, P6, EL, EF, EG,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_TET, COLOR0, P0, P3, P4, N0,
SH_PYR, COLOR0, P4, P3, EK, EH, N0,
SH_TET, COLOR0, P3, EC, EK, N0,
SH_WDG, COLOR1, EF, EL, EG, P5, P2, P7,
SH_TET, COLOR1, P2, P5, P7, N0,
SH_PYR, COLOR1, P5, EE, EH, P7, N0,
SH_TET, COLOR1, P1, P5, P2, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, P7, EK, EC, P2, N0,
SH_TET, COLOR1, EH, EK, P7, N0,
// Case #167: (cloned #91)
SH_TET, COLOR0, EL, EG, EF, P6,
SH_TET, COLOR0, EC, P3, ED, EK,
SH_TET, COLOR0, EE, EI, P4, EH,
SH_WDG, COLOR1, EH, EI, EE, P7, P0, P5,
SH_WDG, COLOR1, P2, P7, P0, EC, EK, ED,
SH_WDG, COLOR1, EL, EG, EF, P2, P7, P5,
SH_TET, COLOR1, P7, P5, P0, P2,
SH_TET, COLOR1, P2, P1, P5, P0,
// Case #168: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EC, ED, EJ, EJ,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, P0, ED, EH, P4, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_PYR, COLOR0, P6, P1, EJ, EF, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_PYR, COLOR0, P2, EC, ED, P0, N0,
SH_PYR, COLOR0, P2, P6, EG, EC, N0,
SH_PYR, COLOR1, P5, EE, EH, P7, N0,
SH_PYR, COLOR1, EG, EF, P5, P7, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_PYR, COLOR1, P7, P3, EC, EG, N0,
SH_TET, COLOR1, EC, P3, ED, N0,
// Case #169: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EC, EG, EA, EJ, EF,
SH_TET, COLOR0, P4, EI, EH, EE,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_PYR, COLOR0, P1, EJ, EF, P6, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_WDG, COLOR1, P0, P7, P5, EI, EH, EE,
SH_TET, COLOR1, P7, P5, P0, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_TET, COLOR1, P3, P7, P0, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_PYR, COLOR1, P5, P7, EG, EF, N0,
SH_TET, COLOR1, EJ, P5, EF, N0,
// Case #170: (cloned #60)
SH_WDG, COLOR0, P0, ED, EA, P4, EH, EE,
SH_WDG, COLOR0, P2, EB, EC, P6, EF, EG,
SH_HEX, COLOR1, EC, EB, EF, EG, P3, P1, P5, P7,
SH_HEX, COLOR1, P3, P1, P5, P7, ED, EA, EE, EH,
// Case #171: (cloned #61)
SH_PNT, 0, COLOR1, 6, P0, P7, P3, P1, EF, EB,
SH_WDG, COLOR0, EF, P6, EG, EB, P2, EC,
SH_TET, COLOR0, EI, EH, P4, EE,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P7,
SH_PYR, COLOR1, EB, P1, P5, EF, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_TET, COLOR1, P1, P3, P0, N0,
SH_PYR, COLOR1, EC, P3, P1, EB, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_TET, COLOR1, P3, P7, P0, N0,
SH_PYR, COLOR1, EF, EG, EC, EB, N0,
SH_TET, COLOR1, P7, P5, P0, N0,
SH_PYR, COLOR1, EF, P5, P7, EG, N0,
// Case #172: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EB, EJ, EE,
SH_TET, COLOR0, P6, EG, EL, EF,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_TET, COLOR0, P0, P4, P1, N0,
SH_PYR, COLOR0, P1, P4, EE, EJ, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_WDG, COLOR1, EL, EG, EF, P2, P7, P5,
SH_TET, COLOR1, P7, P2, P5, N0,
SH_PYR, COLOR1, P2, EB, EJ, P5, N0,
SH_TET, COLOR1, P3, P2, P7, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_PYR, COLOR1, P5, EE, EH, P7, N0,
SH_TET, COLOR1, EJ, EE, P5, N0,
// Case #173: (cloned #91)
SH_TET, COLOR0, EG, EF, EL, P6,
SH_TET, COLOR0, EH, P4, EI, EE,
SH_TET, COLOR0, EB, EA, P1, EJ,
SH_WDG, COLOR1, EJ, EA, EB, P5, P0, P2,
SH_WDG, COLOR1, P7, P5, P0, EH, EE, EI,
SH_WDG, COLOR1, EG, EF, EL, P7, P5, P2,
SH_TET, COLOR1, P5, P2, P0, P7,
SH_TET, COLOR1, P7, P3, P2, P0,
// Case #174: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P5, P1, P3, EH, ED,
SH_WDG, COLOR0, EH, P4, EE, ED, P0, EA,
SH_TET, COLOR0, EL, EF, P6, EG,
SH_WDG, COLOR1, EL, EG, EF, P2, P7, P5,
SH_PYR, COLOR1, ED, P3, P7, EH, N0,
SH_TET, COLOR1, P3, P2, P7, N0,
SH_TET, COLOR1, P3, P1, P2, N0,
SH_PYR, COLOR1, EA, P1, P3, ED, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, P1, P5, P2, N0,
SH_PYR, COLOR1, EH, EE, EA, ED, N0,
SH_TET, COLOR1, P5, P7, P2, N0,
SH_PYR, COLOR1, EH, P7, P5, EE, N0,
// Case #175: (cloned #95)
SH_TET, COLOR0, EE, EH, EI, P4,
SH_TET, COLOR0, EF, EL, EG, P6,
SH_WDG, COLOR1, EG, EF, EL, P7, P5, P2,
SH_WDG, COLOR1, P7, P5, P0, EH, EE, EI,
SH_PYR, COLOR1, P3, P0, P1, P2, P7,
SH_TET, COLOR1, P5, P1, P0, P7,
SH_TET, COLOR1, P5, P2, P1, P7,
// Case #176: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EJ, EI, EF, EG, EK,
SH_PYR, COLOR0, P1, P2, P3, P0, N0,
SH_TET, COLOR0, P3, P6, N0, P2,
SH_PYR, COLOR0, P0, P3, EK, EI, N0,
SH_PYR, COLOR0, EJ, P1, P0, EI, N0,
SH_TET, COLOR0, P6, P2, P1, N0,
SH_PYR, COLOR0, P6, P1, EJ, EF, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_PYR, COLOR0, EK, P3, P6, EG, N0,
SH_PYR, COLOR1, EI, EK, P7, P4, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, EF, P5, P7, EG, N0,
SH_TET, COLOR1, P7, EK, EG, N0,
// Case #177: (cloned #27)
SH_TET, COLOR0, P6, P3, P2, P1,
SH_PYR, COLOR0, EA, P1, P3, ED, EK,
SH_PYR, COLOR0, EJ, EF, P6, P1, EG,
SH_PYR, COLOR0, P3, P6, EG, EK, P1,
SH_PYR, COLOR0, EK, EG, EJ, EA, P1,
SH_TET, COLOR1, P4, P5, P0, P7,
SH_PYR, COLOR1, EK, P7, P0, ED, EA,
SH_PYR, COLOR1, EG, EF, P5, P7, EJ,
SH_PYR, COLOR1, P0, P5, EJ, EA, P7,
SH_PYR, COLOR1, EK, EA, EJ, EG, P7,
// Case #178: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EI, EA, EB, EF, EG, EK,
SH_TET, COLOR0, P2, P6, P3, N0,
SH_PYR, COLOR0, P2, EB, EF, P6, N0,
SH_TET, COLOR0, P0, P2, P3, N0,
SH_PYR, COLOR0, P0, EA, EB, P2, N0,
SH_TET, COLOR0, EI, EA, P0, N0,
SH_PYR, COLOR0, P0, P3, EK, EI, N0,
SH_PYR, COLOR0, EK, P3, P6, EG, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_PYR, COLOR1, EI, EK, P7, P4, N0,
SH_TET, COLOR1, P7, P5, P4, N0,
SH_TET, COLOR1, P7, EK, EG, N0,
SH_PYR, COLOR1, P7, EG, EF, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #179: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EB, ED, EG, EK,
SH_PYR, COLOR0, P2, P3, ED, EB, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_PYR, COLOR0, P6, P2, EB, EF, N0,
SH_TET, COLOR0, EG, P6, EF, N0,
SH_PYR, COLOR0, EK, P3, P6, EG, N0,
SH_TET, COLOR0, P3, P2, P6, N0,
SH_PYR, COLOR1, P4, P5, P1, P0, N0,
SH_TET, COLOR1, P0, P7, P4, N0,
SH_TET, COLOR1, P7, P5, P4, N0,
SH_PYR, COLOR1, P7, EG, EF, P5, N0,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_PYR, COLOR1, ED, EK, P7, P0, N0,
SH_TET, COLOR1, EG, P7, EK, N0,
// Case #180: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EJ, EB, EC,
SH_TET, COLOR0, P6, EF, EG, EL,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_TET, COLOR0, P0, P1, P3, N0,
SH_PYR, COLOR0, P1, EB, EC, P3, N0,
SH_TET, COLOR0, P3, EC, EK, N0,
SH_WDG, COLOR1, P5, P7, P2, EF, EG, EL,
SH_TET, COLOR1, P7, P2, P5, N0,
SH_PYR, COLOR1, P5, P2, EB, EJ, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, P2, P7, EK, EC, N0,
SH_TET, COLOR1, EB, P2, EC, N0,
// Case #181: (cloned #91)
SH_TET, COLOR0, EG, EF, EL, P6,
SH_TET, COLOR0, EK, ED, P3, EC,
SH_TET, COLOR0, EJ, P1, EA, EB,
SH_WDG, COLOR1, P2, P0, P5, EB, EA, EJ,
SH_WDG, COLOR1, EK, EC, ED, P7, P2, P0,
SH_WDG, COLOR1, P7, P2, P5, EG, EL, EF,
SH_TET, COLOR1, P2, P0, P5, P7,
SH_TET, COLOR1, P7, P5, P4, P0,
// Case #182: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P2, P1, P4, EK, EI,
SH_WDG, COLOR0, EI, P0, EA, EK, P3, EC,
SH_TET, COLOR0, EF, P6, EL, EG,
SH_WDG, COLOR1, P5, P7, P2, EF, EG, EL,
SH_PYR, COLOR1, EI, EK, P7, P4, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_PYR, COLOR1, EA, EI, P4, P1, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_TET, COLOR1, P1, P5, P2, N0,
SH_PYR, COLOR1, EK, EI, EA, EC, N0,
SH_TET, COLOR1, P2, P5, P7, N0,
SH_PYR, COLOR1, EK, EC, P2, P7, N0,
// Case #183: (cloned #95)
SH_TET, COLOR0, EC, ED, EK, P3,
SH_TET, COLOR0, EL, EG, EF, P6,
SH_WDG, COLOR1, P7, P2, P5, EG, EL, EF,
SH_WDG, COLOR1, EK, EC, ED, P7, P2, P0,
SH_PYR, COLOR1, P4, P5, P1, P0, P7,
SH_TET, COLOR1, P2, P0, P1, P7,
SH_TET, COLOR1, P2, P1, P5, P7,
// Case #184: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EI, ED, EC, EG, EF, EJ,
SH_TET, COLOR0, P2, P1, P6, N0,
SH_PYR, COLOR0, P2, P6, EG, EC, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_PYR, COLOR0, P0, P2, EC, ED, N0,
SH_TET, COLOR0, EI, P0, ED, N0,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_PYR, COLOR0, EJ, EF, P6, P1, N0,
SH_TET, COLOR0, P6, EF, EG, N0,
SH_TET, COLOR1, P4, P3, P7, N0,
SH_PYR, COLOR1, EI, ED, P3, P4, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_TET, COLOR1, P5, EF, EJ, N0,
SH_PYR, COLOR1, P5, P7, EG, EF, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_TET, COLOR1, EC, P3, ED, N0,
// Case #185: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EG, EC, EA, EF, EJ,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_PYR, COLOR0, P6, EG, EC, P2, N0,
SH_TET, COLOR0, EF, EG, P6, N0,
SH_PYR, COLOR0, EJ, EF, P6, P1, N0,
SH_TET, COLOR0, P1, P6, P2, N0,
SH_PYR, COLOR1, P4, P0, P3, P7, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, P5, P7, EG, EF, N0,
SH_PYR, COLOR1, EG, P7, P3, EC, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P5, EJ, N0,
SH_TET, COLOR1, EF, EJ, P5, N0,
// Case #186: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P3, P7, P5, EB, EF,
SH_WDG, COLOR0, EF, P6, EG, EB, P2, EC,
SH_TET, COLOR0, EI, P0, ED, EA,
SH_WDG, COLOR1, P4, P1, P3, EI, EA, ED,
SH_PYR, COLOR1, EF, EB, P1, P5, N0,
SH_TET, COLOR1, P5, P1, P4, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
SH_PYR, COLOR1, EG, EF, P5, P7, N0,
SH_PYR, COLOR1, EC, EG, P7, P3, N0,
SH_TET, COLOR1, P7, P4, P3, N0,
SH_PYR, COLOR1, EB, EF, EG, EC, N0,
SH_TET, COLOR1, P3, P4, P1, N0,
SH_PYR, COLOR1, EB, EC, P3, P1, N0,
// Case #187: (cloned #63)
SH_WDG, COLOR0, P6, EG, EF, P2, EC, EB,
SH_HEX, COLOR1, EC, EB, EF, EG, P3, P1, P5, P7,
SH_WDG, COLOR1, P0, P3, P1, P4, P7, P5,
// Case #188: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P5, P4, P3, EB, ED,
SH_WDG, COLOR0, ED, P0, EI, EB, P1, EJ,
SH_TET, COLOR0, EG, P6, EF, EL,
SH_WDG, COLOR1, P7, P2, P5, EG, EL, EF,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_TET, COLOR1, P3, P2, P7, N0,
SH_TET, COLOR1, P3, P7, P4, N0,
SH_PYR, COLOR1, EI, ED, P3, P4, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_TET, COLOR1, P4, P7, P5, N0,
SH_PYR, COLOR1, EB, ED, EI, EJ, N0,
SH_TET, COLOR1, P5, P7, P2, N0,
SH_PYR, COLOR1, EB, EJ, P5, P2, N0,
// Case #189: (cloned #95)
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_TET, COLOR0, EF, EL, EG, P6,
SH_WDG, COLOR1, P2, P5, P7, EL, EF, EG,
SH_WDG, COLOR1, EB, EJ, EA, P2, P5, P0,
SH_PYR, COLOR1, P3, P7, P4, P0, P2,
SH_TET, COLOR1, P5, P0, P4, P2,
SH_TET, COLOR1, P5, P4, P7, P2,
// Case #190: (cloned #125)
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EL, EG, EF, P6,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_WDG, COLOR1, EL, EG, EF, P2, P7, P5,
SH_TET, COLOR1, P5, P4, P7, P2,
SH_TET, COLOR1, P2, P3, P1, P4,
SH_TET, COLOR1, P3, P2, P7, P4,
SH_TET, COLOR1, P2, P1, P5, P4,
// Case #191: (cloned #127)
SH_PNT, 0, COLOR1, 7, P1, P0, P3, P2, P5, P4, P7,
SH_TET, COLOR0, EF, EL, EG, P6,
SH_WDG, COLOR1, P5, P7, P2, EF, EG, EL,
SH_TET, COLOR1, P5, P7, P2, N0,
SH_PYR, COLOR1, P4, P0, P3, P7, N0,
SH_TET, COLOR1, P7, P3, P2, N0,
SH_PYR, COLOR1, P1, P2, P3, P0, N0,
SH_TET, COLOR1, P1, P5, P2, N0,
SH_PYR, COLOR1, P1, P0, P4, P5, N0,
SH_TET, COLOR1, P5, P4, P7, N0,
// Case #192: (cloned #3)
SH_HEX, COLOR0, EF, P5, P4, EH, EL, P2, P3, EK,
SH_WDG, COLOR0, P5, P1, P2, P4, P0, P3,
SH_WDG, COLOR1, P6, EF, EL, P7, EH, EK,
// Case #193: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EL, EF, EA, EA,
SH_PYR, COLOR0, P1, EA, EI, P4, N0,
SH_TET, COLOR0, EI, EH, P4, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P1, P4, P5, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_TET, COLOR0, P2, P3, P1, N0,
SH_PYR, COLOR0, P3, ED, EA, P1, N0,
SH_TET, COLOR0, P2, P1, P5, N0,
SH_PYR, COLOR0, P2, P5, EF, EL, N0,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_PYR, COLOR1, P0, P7, EH, EI, N0,
SH_PYR, COLOR1, EK, P7, P0, ED, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_TET, COLOR1, ED, P0, EA, N0,
SH_PYR, COLOR1, EH, P7, P6, EF, N0,
SH_PYR, COLOR1, P7, EK, EL, P6, N0,
SH_TET, COLOR1, EL, EF, P6, N0,
// Case #194: (cloned #21)
SH_PNT, 0, NOCOLOR, 4, EK, EH, EA, EA,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, P4, EH, EF, P5, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_TET, COLOR0, P3, P0, P2, N0,
SH_PYR, COLOR0, P2, P0, EA, EB, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_PYR, COLOR0, P3, EK, EH, P4, N0,
SH_PYR, COLOR0, P3, P2, EL, EK, N0,
SH_PYR, COLOR1, P1, EJ, EF, P6, N0,
SH_PYR, COLOR1, EL, EB, P1, P6, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_PYR, COLOR1, P6, P7, EK, EL, N0,
SH_TET, COLOR1, EK, P7, EH, N0,
// Case #195: (cloned #60)
SH_WDG, COLOR0, P3, ED, EK, P2, EB, EL,
SH_WDG, COLOR0, P4, EH, EI, P5, EF, EJ,
SH_HEX, COLOR1, EI, EH, EF, EJ, P0, P7, P6, P1,
SH_HEX, COLOR1, P0, P7, P6, P1, ED, EK, EL, EB,
// Case #196: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EH, EF, EK, EC, EB,
SH_PYR, COLOR0, P4, P5, P1, P0, N0,
SH_TET, COLOR0, P1, N0, P3, P0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_PYR, COLOR0, EH, EF, P5, P4, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_PYR, COLOR0, P3, EK, EH, P4, N0,
SH_TET, COLOR0, P3, EC, EK, N0,
SH_PYR, COLOR0, EB, EC, P3, P1, N0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_PYR, COLOR1, EH, P7, P6, EF, N0,
SH_TET, COLOR1, EK, P7, EH, N0,
SH_TET, COLOR1, P7, P2, P6, N0,
SH_PYR, COLOR1, EK, EC, P2, P7, N0,
SH_TET, COLOR1, P2, EC, EB, N0,
// Case #197: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EF, EB, EH, EI, EA,
SH_TET, COLOR0, P3, EC, EK, ED,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P4, P1, EA, EI, N0,
SH_TET, COLOR0, P1, EB, EA, N0,
SH_WDG, COLOR1, EK, EC, ED, P7, P2, P0,
SH_TET, COLOR1, P2, P7, P0, N0,
SH_PYR, COLOR1, P7, EH, EI, P0, N0,
SH_TET, COLOR1, P6, P7, P2, N0,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_PYR, COLOR1, P0, EA, EB, P2, N0,
SH_TET, COLOR1, EI, EA, P0, N0,
// Case #198: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EC, EK, EH, EF, EJ, EA,
SH_TET, COLOR0, P4, P5, P0, N0,
SH_PYR, COLOR0, P4, EH, EF, P5, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_PYR, COLOR0, P3, EK, EH, P4, N0,
SH_TET, COLOR0, EC, EK, P3, N0,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_PYR, COLOR0, EA, P0, P5, EJ, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_TET, COLOR1, P2, P6, P7, N0,
SH_PYR, COLOR1, EC, P2, P7, EK, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
SH_TET, COLOR1, P1, EA, EJ, N0,
SH_PYR, COLOR1, P1, EJ, EF, P6, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_TET, COLOR1, EH, EK, P7, N0,
// Case #199: (cloned #61)
SH_PNT, 0, COLOR1, 6, P2, P0, P1, P6, EH, EF,
SH_WDG, COLOR0, EF, P5, EJ, EH, P4, EI,
SH_TET, COLOR0, EC, P3, ED, EK,
SH_WDG, COLOR1, P2, P7, P0, EC, EK, ED,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_TET, COLOR1, P6, P7, P2, N0,
SH_TET, COLOR1, P6, P2, P1, N0,
SH_PYR, COLOR1, EJ, EF, P6, P1, N0,
SH_PYR, COLOR1, EI, EJ, P1, P0, N0,
SH_TET, COLOR1, P1, P2, P0, N0,
SH_PYR, COLOR1, EH, EF, EJ, EI, N0,
SH_TET, COLOR1, P0, P2, P7, N0,
SH_PYR, COLOR1, EH, EI, P0, P7, N0,
// Case #200: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, ED, EH, EC, EL, EF,
SH_PYR, COLOR0, P0, P4, P5, P1, N0,
SH_TET, COLOR0, P5, N0, P2, P1,
SH_PYR, COLOR0, P4, EH, EF, P5, N0,
SH_PYR, COLOR0, ED, EH, P4, P0, N0,
SH_TET, COLOR0, P2, P0, P1, N0,
SH_PYR, COLOR0, P2, EC, ED, P0, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_PYR, COLOR0, EF, EL, P2, P5, N0,
SH_PYR, COLOR1, EH, P7, P6, EF, N0,
SH_PYR, COLOR1, ED, P3, P7, EH, N0,
SH_TET, COLOR1, EC, P3, ED, N0,
SH_TET, COLOR1, P3, P6, P7, N0,
SH_PYR, COLOR1, EC, EL, P6, P3, N0,
SH_TET, COLOR1, P6, EL, EF, N0,
// Case #201: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EC, EL, EF, EH, EI, EA,
SH_TET, COLOR0, P5, P1, P4, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P2, P1, P5, N0,
SH_PYR, COLOR0, P2, P5, EF, EL, N0,
SH_TET, COLOR0, EC, P2, EL, N0,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_PYR, COLOR0, EA, EI, P4, P1, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_TET, COLOR1, P3, P6, P7, N0,
SH_PYR, COLOR1, EC, EL, P6, P3, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_TET, COLOR1, P0, P3, P7, N0,
SH_TET, COLOR1, P0, EI, EA, N0,
SH_PYR, COLOR1, P0, P7, EH, EI, N0,
SH_PYR, COLOR1, EH, P7, P6, EF, N0,
SH_TET, COLOR1, EF, P6, EL, N0,
// Case #202: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EH, EF, ED, EA, EJ,
SH_TET, COLOR0, P2, EL, EC, EB,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_TET, COLOR0, P4, P5, P0, N0,
SH_PYR, COLOR0, P0, P5, EJ, EA, N0,
SH_TET, COLOR0, P5, EF, EJ, N0,
SH_WDG, COLOR1, EC, EL, EB, P3, P6, P1,
SH_TET, COLOR1, P6, P3, P1, N0,
SH_PYR, COLOR1, P3, ED, EA, P1, N0,
SH_TET, COLOR1, P7, P3, P6, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_PYR, COLOR1, P1, EJ, EF, P6, N0,
SH_TET, COLOR1, EA, EJ, P1, N0,
// Case #203: (cloned #61)
SH_PNT, 0, COLOR1, 6, P3, P1, P0, P7, EF, EH,
SH_WDG, COLOR0, EF, P5, EJ, EH, P4, EI,
SH_TET, COLOR0, EC, EB, P2, EL,
SH_WDG, COLOR1, EC, EL, EB, P3, P6, P1,
SH_PYR, COLOR1, EH, P7, P6, EF, N0,
SH_TET, COLOR1, P7, P3, P6, N0,
SH_TET, COLOR1, P7, P0, P3, N0,
SH_PYR, COLOR1, EI, P0, P7, EH, N0,
SH_PYR, COLOR1, EJ, P1, P0, EI, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, EF, EJ, EI, EH, N0,
SH_TET, COLOR1, P1, P6, P3, N0,
SH_PYR, COLOR1, EF, P6, P1, EJ, N0,
// Case #204: (cloned #15)
SH_HEX, COLOR0, ED, EB, EF, EH, P0, P1, P5, P4,
SH_HEX, COLOR1, P3, P2, P6, P7, ED, EB, EF, EH,
// Case #205: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EB, EF, EH, EA, EI,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, P4, EI, EH, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, EA, P1, EB, N0,
SH_PYR, COLOR0, EI, P4, P1, EA, N0,
SH_TET, COLOR0, P4, P5, P1, N0,
SH_PYR, COLOR1, P3, P2, P6, P7, N0,
SH_TET, COLOR1, P7, P0, P3, N0,
SH_TET, COLOR1, P0, P2, P3, N0,
SH_PYR, COLOR1, P0, EA, EB, P2, N0,
SH_PYR, COLOR1, EB, EF, P6, P2, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_PYR, COLOR1, EH, EI, P0, P7, N0,
SH_TET, COLOR1, EA, P0, EI, N0,
// Case #206: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EH, ED, EJ, EA,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, P0, EA, ED, N0,
SH_PYR, COLOR0, P5, P4, EH, EF, N0,
SH_TET, COLOR0, EJ, P5, EF, N0,
SH_PYR, COLOR0, EA, P0, P5, EJ, N0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_PYR, COLOR1, P2, P6, P7, P3, N0,
SH_TET, COLOR1, P3, P1, P2, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
SH_PYR, COLOR1, P1, EJ, EF, P6, N0,
SH_PYR, COLOR1, EF, EH, P7, P6, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_PYR, COLOR1, ED, EA, P1, P3, N0,
SH_TET, COLOR1, EJ, P1, EA, N0,
// Case #207: (cloned #63)
SH_WDG, COLOR0, P5, EJ, EF, P4, EI, EH,
SH_HEX, COLOR1, EI, EH, EF, EJ, P0, P7, P6, P1,
SH_WDG, COLOR1, P3, P0, P7, P2, P1, P6,
// Case #208: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EI, EK, EE, EF, EL,
SH_PYR, COLOR0, P0, P1, P2, P3, N0,
SH_TET, COLOR0, P2, P5, N0, P1,
SH_PYR, COLOR0, P3, P2, EL, EK, N0,
SH_PYR, COLOR0, EI, P0, P3, EK, N0,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, P5, P0, EI, EE, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, EL, P2, P5, EF, N0,
SH_PYR, COLOR1, EK, EL, P6, P7, N0,
SH_PYR, COLOR1, EI, EK, P7, P4, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
SH_TET, COLOR1, P4, P7, P6, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_TET, COLOR1, P6, EL, EF, N0,
// Case #209: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EE, EF, EL, EK, ED, EA,
SH_TET, COLOR0, P2, P3, P1, N0,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_TET, COLOR0, P5, P2, P1, N0,
SH_PYR, COLOR0, P5, EF, EL, P2, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, P5, P1, EA, EE, N0,
SH_PYR, COLOR0, EA, P1, P3, ED, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_TET, COLOR1, P4, P7, P6, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_PYR, COLOR1, EE, EA, P0, P4, N0,
SH_TET, COLOR1, P0, P7, P4, N0,
SH_TET, COLOR1, P0, EA, ED, N0,
SH_PYR, COLOR1, P0, ED, EK, P7, N0,
SH_PYR, COLOR1, EK, EL, P6, P7, N0,
SH_TET, COLOR1, EL, EF, P6, N0,
// Case #210: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EI, EA, EB,
SH_TET, COLOR0, P5, EE, EF, EJ,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_TET, COLOR0, P3, P0, P2, N0,
SH_PYR, COLOR0, P0, EA, EB, P2, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_WDG, COLOR1, P4, P6, P1, EE, EF, EJ,
SH_TET, COLOR1, P6, P1, P4, N0,
SH_PYR, COLOR1, P4, P1, EA, EI, N0,
SH_TET, COLOR1, P7, P6, P4, N0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_PYR, COLOR1, P1, P6, EL, EB, N0,
SH_TET, COLOR1, EA, P1, EB, N0,
// Case #211: (cloned #61)
SH_PNT, 0, COLOR1, 6, P4, P1, P0, P7, EL, EK,
SH_WDG, COLOR0, EK, P3, ED, EL, P2, EB,
SH_TET, COLOR0, EE, P5, EJ, EF,
SH_WDG, COLOR1, P4, P6, P1, EE, EF, EJ,
SH_PYR, COLOR1, EK, EL, P6, P7, N0,
SH_TET, COLOR1, P7, P6, P4, N0,
SH_TET, COLOR1, P7, P4, P0, N0,
SH_PYR, COLOR1, ED, EK, P7, P0, N0,
SH_PYR, COLOR1, EB, ED, P0, P1, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, EL, EK, ED, EB, N0,
SH_TET, COLOR1, P1, P4, P6, N0,
SH_PYR, COLOR1, EL, EB, P1, P6, N0,
// Case #212: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EK, EC, EB, EF, EE, EI,
SH_TET, COLOR0, P1, P0, P5, N0,
SH_PYR, COLOR0, P1, P5, EF, EB, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_PYR, COLOR0, P3, P1, EB, EC, N0,
SH_TET, COLOR0, EK, P3, EC, N0,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_PYR, COLOR0, EI, EE, P5, P0, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_TET, COLOR1, P7, P2, P6, N0,
SH_PYR, COLOR1, EK, EC, P2, P7, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_TET, COLOR1, P4, P7, P6, N0,
SH_TET, COLOR1, P4, EE, EI, N0,
SH_PYR, COLOR1, P4, P6, EF, EE, N0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_TET, COLOR1, EB, P2, EC, N0,
// Case #213: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P0, P4, P6, EB, EF,
SH_WDG, COLOR0, EB, P1, EA, EF, P5, EE,
SH_TET, COLOR0, EK, ED, P3, EC,
SH_WDG, COLOR1, EK, EC, ED, P7, P2, P0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_TET, COLOR1, P6, P7, P2, N0,
SH_TET, COLOR1, P6, P4, P7, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_TET, COLOR1, P4, P0, P7, N0,
SH_PYR, COLOR1, EB, EA, EE, EF, N0,
SH_TET, COLOR1, P0, P2, P7, N0,
SH_PYR, COLOR1, EB, P2, P0, EA, N0,
// Case #214: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P1, P2, P7, EI, EK,
SH_WDG, COLOR0, EI, P0, EA, EK, P3, EC,
SH_TET, COLOR0, EF, EJ, P5, EE,
SH_WDG, COLOR1, EF, EE, EJ, P6, P4, P1,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_TET, COLOR1, P7, P6, P4, N0,
SH_TET, COLOR1, P7, P2, P6, N0,
SH_PYR, COLOR1, EC, P2, P7, EK, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_TET, COLOR1, P2, P1, P6, N0,
SH_PYR, COLOR1, EI, EA, EC, EK, N0,
SH_TET, COLOR1, P1, P4, P6, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
// Case #215: (cloned #125)
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_TET, COLOR0, EC, ED, EK, P3,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_WDG, COLOR1, EC, ED, EK, P2, P0, P7,
SH_TET, COLOR1, P7, P4, P0, P2,
SH_TET, COLOR1, P2, P1, P6, P4,
SH_TET, COLOR1, P1, P2, P0, P4,
SH_TET, COLOR1, P2, P6, P7, P4,
// Case #216: (cloned #27)
SH_TET, COLOR0, P2, P0, P1, P5,
SH_PYR, COLOR0, EE, P5, P0, EI, ED,
SH_PYR, COLOR0, EF, EL, P2, P5, EC,
SH_PYR, COLOR0, P0, P2, EC, ED, P5,
SH_PYR, COLOR0, ED, EC, EF, EE, P5,
SH_TET, COLOR1, P7, P6, P4, P3,
SH_PYR, COLOR1, ED, P3, P4, EI, EE,
SH_PYR, COLOR1, EC, EL, P6, P3, EF,
SH_PYR, COLOR1, P4, P6, EF, EE, P3,
SH_PYR, COLOR1, ED, EE, EF, EC, P3,
// Case #217: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EC, EA, EE, EL, EF,
SH_PYR, COLOR0, P1, EA, EE, P5, N0,
SH_TET, COLOR0, P5, EE, EF, N0,
SH_PYR, COLOR0, P2, EC, EA, P1, N0,
SH_TET, COLOR0, EL, EC, P2, N0,
SH_PYR, COLOR0, EF, EL, P2, P5, N0,
SH_TET, COLOR0, P5, P2, P1, N0,
SH_PYR, COLOR1, P7, P4, P0, P3, N0,
SH_TET, COLOR1, P4, P7, P6, N0,
SH_TET, COLOR1, P6, P7, P3, N0,
SH_PYR, COLOR1, P6, P3, EC, EL, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_PYR, COLOR1, EA, P0, P4, EE, N0,
SH_PYR, COLOR1, EE, P4, P6, EF, N0,
SH_TET, COLOR1, EL, EF, P6, N0,
// Case #218: (cloned #91)
SH_TET, COLOR0, EC, EL, EB, P2,
SH_TET, COLOR0, ED, EI, P0, EA,
SH_TET, COLOR0, EF, P5, EE, EJ,
SH_WDG, COLOR1, P1, P4, P6, EJ, EE, EF,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_WDG, COLOR1, P3, P1, P6, EC, EB, EL,
SH_TET, COLOR1, P1, P4, P6, P3,
SH_TET, COLOR1, P3, P6, P7, P4,
// Case #219: (cloned #95)
SH_TET, COLOR0, EL, EB, EC, P2,
SH_TET, COLOR0, EF, EE, EJ, P5,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_PYR, COLOR1, P0, P3, P7, P4, P1,
SH_TET, COLOR1, P6, P7, P3, P1,
SH_TET, COLOR1, P6, P4, P7, P1,
// Case #220: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EF, EB, ED, EE, EI,
SH_PYR, COLOR0, P1, EB, ED, P0, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, P5, EF, EB, P1, N0,
SH_TET, COLOR0, EE, EF, P5, N0,
SH_PYR, COLOR0, EI, EE, P5, P0, N0,
SH_TET, COLOR0, P0, P5, P1, N0,
SH_PYR, COLOR1, P7, P3, P2, P6, N0,
SH_TET, COLOR1, P3, P7, P4, N0,
SH_TET, COLOR1, P4, P7, P6, N0,
SH_PYR, COLOR1, P4, P6, EF, EE, N0,
SH_PYR, COLOR1, EF, P6, P2, EB, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_PYR, COLOR1, ED, P3, P4, EI, N0,
SH_TET, COLOR1, EE, EI, P4, N0,
// Case #221: (cloned #63)
SH_WDG, COLOR0, P1, EA, EB, P5, EE, EF,
SH_HEX, COLOR1, P0, P2, P6, P4, EA, EB, EF, EE,
SH_WDG, COLOR1, P7, P4, P6, P3, P0, P2,
// Case #222: (cloned #95)
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EJ, EF, EE, P5,
SH_WDG, COLOR1, EE, EJ, EF, P4, P1, P6,
SH_WDG, COLOR1, P4, P1, P3, EI, EA, ED,
SH_PYR, COLOR1, P7, P3, P2, P6, P4,
SH_TET, COLOR1, P1, P2, P3, P4,
SH_TET, COLOR1, P1, P6, P2, P4,
// Case #223: (cloned #127)
SH_PNT, 0, COLOR1, 7, P0, P3, P7, P4, P1, P2, P6,
SH_TET, COLOR0, EJ, EF, EE, P5,
SH_WDG, COLOR1, EJ, EF, EE, P1, P6, P4,
SH_TET, COLOR1, P1, P4, P6, N0,
SH_PYR, COLOR1, P2, P6, P7, P3, N0,
SH_TET, COLOR1, P6, P4, P7, N0,
SH_PYR, COLOR1, P0, P3, P7, P4, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_PYR, COLOR1, P0, P1, P2, P3, N0,
SH_TET, COLOR1, P1, P6, P2, N0,
// Case #224: (cloned #7)
SH_PNT, 0, NOCOLOR, 5, EK, EL, EH, EE, EJ,
SH_PYR, COLOR0, P3, P0, P1, P2, N0,
SH_TET, COLOR0, P1, P4, N0, P0,
SH_PYR, COLOR0, P2, P1, EJ, EL, N0,
SH_PYR, COLOR0, EK, P3, P2, EL, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_PYR, COLOR0, P4, P3, EK, EH, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR0, EJ, P1, P4, EE, N0,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_PYR, COLOR1, EK, EL, P6, P7, N0,
SH_TET, COLOR1, EH, EK, P7, N0,
SH_TET, COLOR1, P7, P6, P5, N0,
SH_PYR, COLOR1, EH, P7, P5, EE, N0,
SH_TET, COLOR1, P5, EJ, EE, N0,
// Case #225: (cloned #30)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EK, ED, EA,
SH_TET, COLOR0, P4, EH, EE, EI,
SH_PYR, COLOR0, P1, EJ, EL, P2, N0,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_TET, COLOR0, P2, P3, P1, N0,
SH_PYR, COLOR0, P3, ED, EA, P1, N0,
SH_TET, COLOR0, P1, EA, EJ, N0,
SH_WDG, COLOR1, P7, P5, P0, EH, EE, EI,
SH_TET, COLOR1, P5, P0, P7, N0,
SH_PYR, COLOR1, P7, P0, ED, EK, N0,
SH_TET, COLOR1, P6, P5, P7, N0,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_TET, COLOR1, ED, P0, EA, N0,
// Case #226: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EE, EH, EK, EL, EB, EA,
SH_TET, COLOR0, P3, P0, P2, N0,
SH_PYR, COLOR0, P3, P2, EL, EK, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_PYR, COLOR0, P4, P3, EK, EH, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, P4, EE, EA, P0, N0,
SH_PYR, COLOR0, EA, EB, P2, P0, N0,
SH_TET, COLOR0, P2, EB, EL, N0,
SH_TET, COLOR1, P5, P7, P6, N0,
SH_PYR, COLOR1, EE, EH, P7, P5, N0,
SH_PYR, COLOR1, EE, P5, P1, EA, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
SH_TET, COLOR1, P1, EB, EA, N0,
SH_PYR, COLOR1, P1, P6, EL, EB, N0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_TET, COLOR1, EK, P7, EH, N0,
// Case #227: (cloned #61)
SH_PNT, 0, COLOR1, 6, P5, P0, P1, P6, EK, EL,
SH_WDG, COLOR0, EK, P3, ED, EL, P2, EB,
SH_TET, COLOR0, EE, EI, P4, EH,
SH_WDG, COLOR1, EE, EH, EI, P5, P7, P0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_TET, COLOR1, P6, P5, P7, N0,
SH_TET, COLOR1, P6, P1, P5, N0,
SH_PYR, COLOR1, EB, P1, P6, EL, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, P1, P0, P5, N0,
SH_PYR, COLOR1, EK, ED, EB, EL, N0,
SH_TET, COLOR1, P0, P7, P5, N0,
SH_PYR, COLOR1, EK, P7, P0, ED, N0,
// Case #228: (cloned #27)
SH_TET, COLOR0, P3, P0, P1, P4,
SH_PYR, COLOR0, EE, EJ, P1, P4, EB,
SH_PYR, COLOR0, EH, P4, P3, EK, EC,
SH_PYR, COLOR0, P1, EB, EC, P3, P4,
SH_PYR, COLOR0, EB, EE, EH, EC, P4,
SH_TET, COLOR1, P6, P5, P7, P2,
SH_PYR, COLOR1, EB, EJ, P5, P2, EE,
SH_PYR, COLOR1, EC, P2, P7, EK, EH,
SH_PYR, COLOR1, P5, EE, EH, P7, P2,
SH_PYR, COLOR1, EB, EC, EH, EE, P2,
// Case #229: (cloned #91)
SH_TET, COLOR0, EC, ED, EK, P3,
SH_TET, COLOR0, EB, P1, EJ, EA,
SH_TET, COLOR0, EH, EE, P4, EI,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P7,
SH_WDG, COLOR1, P2, P0, P5, EB, EA, EJ,
SH_WDG, COLOR1, EC, ED, EK, P2, P0, P7,
SH_TET, COLOR1, P0, P7, P5, P2,
SH_TET, COLOR1, P2, P6, P7, P5,
// Case #230: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EC, EA, EE, EK, EH,
SH_PYR, COLOR0, P0, P4, EE, EA, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_TET, COLOR0, EK, P3, EC, N0,
SH_PYR, COLOR0, EH, P4, P3, EK, N0,
SH_TET, COLOR0, P4, P0, P3, N0,
SH_PYR, COLOR1, P6, P2, P1, P5, N0,
SH_TET, COLOR1, P5, P7, P6, N0,
SH_TET, COLOR1, P7, P2, P6, N0,
SH_PYR, COLOR1, P7, EK, EC, P2, N0,
SH_PYR, COLOR1, EC, EA, P1, P2, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_PYR, COLOR1, EE, EH, P7, P5, N0,
SH_TET, COLOR1, EK, P7, EH, N0,
// Case #231: (cloned #95)
SH_TET, COLOR0, EK, EC, ED, P3,
SH_TET, COLOR0, EH, EI, EE, P4,
SH_WDG, COLOR1, P0, P7, P5, EI, EH, EE,
SH_WDG, COLOR1, ED, EK, EC, P0, P7, P2,
SH_PYR, COLOR1, P1, P5, P6, P2, P0,
SH_TET, COLOR1, P7, P2, P6, P0,
SH_TET, COLOR1, P7, P6, P5, P0,
// Case #232: (cloned #23)
SH_PNT, 0, NOCOLOR, 6, EH, EE, EJ, EL, EC, ED,
SH_TET, COLOR0, P1, P2, P0, N0,
SH_PYR, COLOR0, P1, EJ, EL, P2, N0,
SH_TET, COLOR0, P4, P1, P0, N0,
SH_PYR, COLOR0, P4, EE, EJ, P1, N0,
SH_TET, COLOR0, EH, EE, P4, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_PYR, COLOR0, ED, P0, P2, EC, N0,
SH_TET, COLOR0, P2, EL, EC, N0,
SH_TET, COLOR1, P7, P6, P5, N0,
SH_PYR, COLOR1, EH, P7, P5, EE, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_TET, COLOR1, P3, P6, P7, N0,
SH_TET, COLOR1, P3, ED, EC, N0,
SH_PYR, COLOR1, P3, EC, EL, P6, N0,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_TET, COLOR1, EJ, EE, P5, N0,
// Case #233: (cloned #61)
SH_PNT, 0, COLOR1, 6, P7, P0, P3, P6, EJ, EL,
SH_WDG, COLOR0, EL, P2, EC, EJ, P1, EA,
SH_TET, COLOR0, EH, P4, EI, EE,
SH_WDG, COLOR1, P7, P5, P0, EH, EE, EI,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_TET, COLOR1, P6, P5, P7, N0,
SH_TET, COLOR1, P6, P7, P3, N0,
SH_PYR, COLOR1, EC, EL, P6, P3, N0,
SH_PYR, COLOR1, EA, EC, P3, P0, N0,
SH_TET, COLOR1, P3, P7, P0, N0,
SH_PYR, COLOR1, EJ, EL, EC, EA, N0,
SH_TET, COLOR1, P0, P7, P5, N0,
SH_PYR, COLOR1, EJ, EA, P0, P5, N0,
// Case #234: (cloned #61)
SH_PNT, 0, COLOR1, 6, P6, P1, P5, P7, ED, EH,
SH_WDG, COLOR0, EH, P4, EE, ED, P0, EA,
SH_TET, COLOR0, EL, P2, EB, EC,
SH_WDG, COLOR1, P6, P3, P1, EL, EC, EB,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_TET, COLOR1, P7, P3, P6, N0,
SH_TET, COLOR1, P7, P6, P5, N0,
SH_PYR, COLOR1, EE, EH, P7, P5, N0,
SH_PYR, COLOR1, EA, EE, P5, P1, N0,
SH_TET, COLOR1, P5, P6, P1, N0,
SH_PYR, COLOR1, ED, EH, EE, EA, N0,
SH_TET, COLOR1, P1, P6, P3, N0,
SH_PYR, COLOR1, ED, EA, P1, P3, N0,
// Case #235: (cloned #125)
SH_TET, COLOR0, EL, EB, EC, P2,
SH_TET, COLOR0, EE, EH, EI, P4,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_WDG, COLOR1, P5, P0, P7, EE, EI, EH,
SH_TET, COLOR1, P7, P0, P3, P5,
SH_TET, COLOR1, P5, P6, P1, P3,
SH_TET, COLOR1, P1, P0, P5, P3,
SH_TET, COLOR1, P5, P7, P6, P3,
// Case #236: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EH, ED, EB, EE, EJ,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, P4, P0, ED, EH, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, EJ, P1, P4, EE, N0,
SH_TET, COLOR0, P1, P0, P4, N0,
SH_PYR, COLOR1, P6, P7, P3, P2, N0,
SH_TET, COLOR1, P2, P5, P6, N0,
SH_TET, COLOR1, P5, P7, P6, N0,
SH_PYR, COLOR1, P5, EE, EH, P7, N0,
SH_PYR, COLOR1, EH, ED, P3, P7, N0,
SH_PYR, COLOR1, ED, EB, P2, P3, N0,
SH_PYR, COLOR1, EB, EJ, P5, P2, N0,
SH_TET, COLOR1, EE, P5, EJ, N0,
// Case #237: (cloned #95)
SH_TET, COLOR0, EE, EH, EI, P4,
SH_TET, COLOR0, EJ, EA, EB, P1,
SH_WDG, COLOR1, P0, P5, P2, EA, EJ, EB,
SH_WDG, COLOR1, EI, EE, EH, P0, P5, P7,
SH_PYR, COLOR1, P3, P2, P6, P7, P0,
SH_TET, COLOR1, P5, P7, P6, P0,
SH_TET, COLOR1, P5, P6, P2, P0,
// Case #238: (cloned #63)
SH_WDG, COLOR0, P4, EE, EH, P0, EA, ED,
SH_HEX, COLOR1, EA, ED, EH, EE, P1, P3, P7, P5,
SH_WDG, COLOR1, P2, P1, P3, P6, P5, P7,
// Case #239: (cloned #127)
SH_PNT, 0, COLOR1, 7, P3, P2, P1, P0, P7, P6, P5,
SH_TET, COLOR0, EH, EI, EE, P4,
SH_WDG, COLOR1, P7, P5, P0, EH, EE, EI,
SH_TET, COLOR1, P7, P5, P0, N0,
SH_PYR, COLOR1, P6, P2, P1, P5, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_PYR, COLOR1, P3, P0, P1, P2, N0,
SH_TET, COLOR1, P3, P7, P0, N0,
SH_PYR, COLOR1, P3, P2, P6, P7, N0,
SH_TET, COLOR1, P7, P6, P5, N0,
// Case #240: (cloned #15)
SH_HEX, COLOR0, P0, P1, P2, P3, EI, EJ, EL, EK,
SH_HEX, COLOR1, EI, EJ, EL, EK, P4, P5, P6, P7,
// Case #241: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EJ, EL, EK, EA, ED,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_TET, COLOR0, P3, EK, ED, N0,
SH_PYR, COLOR0, P1, EJ, EL, P2, N0,
SH_TET, COLOR0, EA, EJ, P1, N0,
SH_PYR, COLOR0, ED, EA, P1, P3, N0,
SH_TET, COLOR0, P3, P1, P2, N0,
SH_PYR, COLOR1, P4, P7, P6, P5, N0,
SH_TET, COLOR1, P7, P4, P0, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_PYR, COLOR1, P0, P5, EJ, EA, N0,
SH_PYR, COLOR1, EJ, P5, P6, EL, N0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_PYR, COLOR1, EK, P7, P0, ED, N0,
SH_TET, COLOR1, EA, ED, P0, N0,
// Case #242: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EK, EI, EB, EA,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_TET, COLOR0, P0, EI, EA, N0,
SH_PYR, COLOR0, P2, EL, EK, P3, N0,
SH_TET, COLOR0, EB, EL, P2, N0,
SH_PYR, COLOR0, EA, EB, P2, P0, N0,
SH_TET, COLOR0, P0, P2, P3, N0,
SH_PYR, COLOR1, P5, P4, P7, P6, N0,
SH_TET, COLOR1, P4, P5, P1, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
SH_PYR, COLOR1, P1, P6, EL, EB, N0,
SH_PYR, COLOR1, EL, P6, P7, EK, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P1, EA, N0,
SH_TET, COLOR1, EB, EA, P1, N0,
// Case #243: (cloned #63)
SH_WDG, COLOR0, P3, ED, EK, P2, EB, EL,
SH_HEX, COLOR1, P0, P7, P6, P1, ED, EK, EL, EB,
SH_WDG, COLOR1, P5, P1, P6, P4, P0, P7,
// Case #244: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EK, EI, EJ, EC, EB,
SH_PYR, COLOR0, P0, EI, EJ, P1, N0,
SH_TET, COLOR0, P1, EJ, EB, N0,
SH_PYR, COLOR0, P3, EK, EI, P0, N0,
SH_TET, COLOR0, EC, EK, P3, N0,
SH_PYR, COLOR0, EB, EC, P3, P1, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_PYR, COLOR1, P6, P5, P4, P7, N0,
SH_TET, COLOR1, P5, P6, P2, N0,
SH_TET, COLOR1, P2, P6, P7, N0,
SH_PYR, COLOR1, P2, P7, EK, EC, N0,
SH_PYR, COLOR1, EK, P7, P4, EI, N0,
SH_PYR, COLOR1, EI, P4, P5, EJ, N0,
SH_PYR, COLOR1, EJ, P5, P2, EB, N0,
SH_TET, COLOR1, EC, EB, P2, N0,
// Case #245: (cloned #95)
SH_TET, COLOR0, EC, ED, EK, P3,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, EA, EB, EJ, P0, P2, P5,
SH_WDG, COLOR1, P0, P2, P7, ED, EC, EK,
SH_PYR, COLOR1, P4, P7, P6, P5, P0,
SH_TET, COLOR1, P2, P6, P7, P0,
SH_TET, COLOR1, P2, P5, P6, P0,
// Case #246: (cloned #63)
SH_WDG, COLOR0, P0, EA, EI, P3, EC, EK,
SH_HEX, COLOR1, P1, P4, P7, P2, EA, EI, EK, EC,
SH_WDG, COLOR1, P6, P2, P7, P5, P1, P4,
// Case #247: (cloned #127)
SH_PNT, 0, COLOR1, 7, P4, P5, P1, P0, P7, P6, P2,
SH_TET, COLOR0, EK, EC, ED, P3,
SH_WDG, COLOR1, EK, EC, ED, P7, P2, P0,
SH_TET, COLOR1, P7, P0, P2, N0,
SH_PYR, COLOR1, P6, P2, P1, P5, N0,
SH_TET, COLOR1, P2, P0, P1, N0,
SH_PYR, COLOR1, P4, P5, P1, P0, N0,
SH_TET, COLOR1, P4, P0, P7, N0,
SH_PYR, COLOR1, P4, P7, P6, P5, N0,
SH_TET, COLOR1, P7, P2, P6, N0,
// Case #248: (cloned #31)
SH_PNT, 0, NOCOLOR, 5, EL, EJ, EI, EC, ED,
SH_PYR, COLOR0, P1, P0, EI, EJ, N0,
SH_TET, COLOR0, P0, ED, EI, N0,
SH_PYR, COLOR0, P2, P1, EJ, EL, N0,
SH_TET, COLOR0, EC, P2, EL, N0,
SH_PYR, COLOR0, ED, P0, P2, EC, N0,
SH_TET, COLOR0, P0, P1, P2, N0,
SH_PYR, COLOR1, P7, P6, P5, P4, N0,
SH_TET, COLOR1, P4, P3, P7, N0,
SH_TET, COLOR1, P3, P6, P7, N0,
SH_PYR, COLOR1, P3, EC, EL, P6, N0,
SH_PYR, COLOR1, EL, EJ, P5, P6, N0,
SH_PYR, COLOR1, EJ, EI, P4, P5, N0,
SH_PYR, COLOR1, EI, ED, P3, P4, N0,
SH_TET, COLOR1, EC, P3, ED, N0,
// Case #249: (cloned #63)
SH_WDG, COLOR0, P2, EC, EL, P1, EA, EJ,
SH_HEX, COLOR1, EA, EJ, EL, EC, P0, P5, P6, P3,
SH_WDG, COLOR1, P4, P0, P5, P7, P3, P6,
// Case #250: (cloned #95)
SH_TET, COLOR0, EA, EI, ED, P0,
SH_TET, COLOR0, EB, EC, EL, P2,
SH_WDG, COLOR1, P3, P1, P6, EC, EB, EL,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_PYR, COLOR1, P7, P6, P5, P4, P3,
SH_TET, COLOR1, P1, P4, P5, P3,
SH_TET, COLOR1, P1, P5, P6, P3,
// Case #251: (cloned #127)
SH_PNT, 0, COLOR1, 7, P0, P4, P7, P3, P1, P5, P6,
SH_TET, COLOR0, EB, EC, EL, P2,
SH_WDG, COLOR1, P1, P6, P3, EB, EL, EC,
SH_TET, COLOR1, P1, P6, P3, N0,
SH_PYR, COLOR1, P5, P4, P7, P6, N0,
SH_TET, COLOR1, P6, P7, P3, N0,
SH_PYR, COLOR1, P0, P3, P7, P4, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, P0, P4, P5, P1, N0,
SH_TET, COLOR1, P1, P5, P6, N0,
// Case #252: (cloned #63)
SH_WDG, COLOR0, P0, EI, ED, P1, EJ, EB,
SH_HEX, COLOR1, P4, P3, P2, P5, EI, ED, EB, EJ,
SH_WDG, COLOR1, P6, P5, P2, P7, P4, P3,
// Case #253: (cloned #127)
SH_PNT, 0, COLOR1, 7, P3, P7, P4, P0, P2, P6, P5,
SH_TET, COLOR0, EB, EJ, EA, P1,
SH_WDG, COLOR1, EB, EJ, EA, P2, P5, P0,
SH_TET, COLOR1, P2, P0, P5, N0,
SH_PYR, COLOR1, P6, P5, P4, P7, N0,
SH_TET, COLOR1, P5, P0, P4, N0,
SH_PYR, COLOR1, P3, P7, P4, P0, N0,
SH_TET, COLOR1, P3, P0, P2, N0,
SH_PYR, COLOR1, P3, P2, P6, P7, N0,
SH_TET, COLOR1, P2, P5, P6, N0,
// Case #254: (cloned #127)
SH_PNT, 0, COLOR1, 7, P7, P6, P5, P4, P3, P2, P1,
SH_TET, COLOR0, ED, EA, EI, P0,
SH_WDG, COLOR1, ED, EA, EI, P3, P1, P4,
SH_TET, COLOR1, P3, P4, P1, N0,
SH_PYR, COLOR1, P2, P1, P5, P6, N0,
SH_TET, COLOR1, P1, P4, P5, N0,
SH_PYR, COLOR1, P7, P6, P5, P4, N0,
SH_TET, COLOR1, P7, P4, P3, N0,
SH_PYR, COLOR1, P7, P3, P2, P6, N0,
SH_TET, COLOR1, P3, P1, P2, N0,
// Case #255: Unique case #22
SH_HEX, COLOR1, P0, P1, P2, P3, P4, P5, P6, P7,
//}
//{Wedge
// Case #0: Unique case #1
SH_WDG, COLOR0, P0, P1, P2, P3, P4, P5,
// Case #1: Unique case #2
SH_WDG, COLOR0, EA, EC, EG, P1, P2, P3,
SH_PYR, COLOR0, P1, P2, P5, P4, P3,
SH_TET, COLOR1, EG, EA, EC, P0,
// Case #2: (cloned #1)
SH_WDG, COLOR0, EB, EA, EH, P2, P0, P4,
SH_PYR, COLOR0, P2, P0, P3, P5, P4,
SH_TET, COLOR1, EH, EB, EA, P1,
// Case #3: Unique case #3
SH_PNT, 0, COLOR0, 7, P2, P3, P4, EB, EC, EG, EH,
SH_TET, COLOR0, P4, P5, P3, P2,
SH_TET, COLOR0, P2, P3, P4, N0,
SH_PYR, COLOR0, EG, EH, P4, P3, N0,
SH_PYR, COLOR0, EB, EH, EG, EC, N0,
SH_TET, COLOR0, P2, EB, EC, N0,
SH_PYR, COLOR0, P2, EC, EG, P3, N0,
SH_PYR, COLOR0, EH, EB, P2, P4, N0,
SH_WDG, COLOR1, EC, EG, P0, EB, EH, P1,
// Case #4: (cloned #1)
SH_WDG, COLOR0, EC, EB, EI, P0, P1, P5,
SH_PYR, COLOR0, P0, P1, P4, P3, P5,
SH_TET, COLOR1, EI, EC, EB, P2,
// Case #5: (cloned #3)
SH_PNT, 0, COLOR0, 7, P1, P5, P3, EA, EB, EI, EG,
SH_TET, COLOR0, P3, P4, P5, P1,
SH_TET, COLOR0, P1, P5, P3, N0,
SH_PYR, COLOR0, EI, EG, P3, P5, N0,
SH_PYR, COLOR0, EA, EG, EI, EB, N0,
SH_TET, COLOR0, P1, EA, EB, N0,
SH_PYR, COLOR0, P1, EB, EI, P5, N0,
SH_PYR, COLOR0, EG, EA, P1, P3, N0,
SH_WDG, COLOR1, EB, EI, P2, EA, EG, P0,
// Case #6: (cloned #3)
SH_PNT, 0, COLOR0, 7, P0, P4, P5, EC, EA, EH, EI,
SH_TET, COLOR0, P5, P3, P4, P0,
SH_TET, COLOR0, P0, P4, P5, N0,
SH_PYR, COLOR0, EH, EI, P5, P4, N0,
SH_PYR, COLOR0, EC, EI, EH, EA, N0,
SH_TET, COLOR0, P0, EC, EA, N0,
SH_PYR, COLOR0, P0, EA, EH, P4, N0,
SH_PYR, COLOR0, EI, EC, P0, P5, N0,
SH_WDG, COLOR1, EA, EH, P1, EC, EI, P2,
// Case #7: Unique case #4
SH_WDG, COLOR0, EG, EH, EI, P3, P4, P5,
SH_WDG, COLOR1, P0, P1, P2, EG, EH, EI,
// Case #8: (cloned #1)
SH_WDG, COLOR0, P4, P5, P0, ED, EF, EG,
SH_PYR, COLOR0, P4, P1, P2, P5, P0,
SH_TET, COLOR1, EG, EF, ED, P3,
// Case #9: Unique case #5
SH_HEX, COLOR0, P1, P2, P5, P4, EA, EC, EF, ED,
SH_WDG, COLOR1, P0, EA, EC, P3, ED, EF,
// Case #10: Unique case #6
SH_PNT, 0, NOCOLOR, 6, EA, EB, EH, ED, EF, EG,
SH_PYR, COLOR0, P5, P0, EG, EF, N0,
SH_TET, COLOR0, P0, EA, EG, N0,
SH_PYR, COLOR0, P0, P2, EB, EA, N0,
SH_TET, COLOR0, P5, P2, P0, N0,
SH_PYR, COLOR0, P4, EH, EB, P2, N0,
SH_TET, COLOR0, P5, P4, P2, N0,
SH_PYR, COLOR0, EF, ED, P4, P5, N0,
SH_TET, COLOR0, ED, EH, P4, N0,
SH_PYR, COLOR1, EG, EA, P1, P3, N0,
SH_PYR, COLOR1, P3, P1, EH, ED, N0,
SH_TET, COLOR1, P3, ED, EF, N0,
SH_TET, COLOR1, EF, EG, P3, N0,
SH_TET, COLOR1, P1, EB, EH, N0,
SH_TET, COLOR1, P1, EA, EB, N0,
// Case #11: Unique case #7
SH_PNT, 0, NOCOLOR, 5, EB, EC, EF, ED, EH,
SH_PYR, COLOR0, P4, P5, EF, ED, N0,
SH_TET, COLOR0, ED, EH, P4, N0,
SH_PYR, COLOR0, EC, EF, P5, P2, N0,
SH_PYR, COLOR0, EB, P2, P4, EH, N0,
SH_TET, COLOR0, P4, P2, P5, N0,
SH_TET, COLOR0, P2, EB, EC, N0,
SH_TET, COLOR1, P0, P1, P3, N0,
SH_PYR, COLOR1, EC, EB, P1, P0, N0,
SH_PYR, COLOR1, EC, P0, P3, EF, N0,
SH_TET, COLOR1, EF, P3, ED, N0,
SH_PYR, COLOR1, P3, P1, EH, ED, N0,
SH_TET, COLOR1, P1, EB, EH, N0,
// Case #12: (cloned #10)
SH_PNT, 0, NOCOLOR, 6, EF, ED, EG, EC, EB, EI,
SH_PYR, COLOR0, P1, EB, EI, P5, N0,
SH_TET, COLOR0, P5, EI, EF, N0,
SH_PYR, COLOR0, P5, EF, ED, P4, N0,
SH_TET, COLOR0, P1, P5, P4, N0,
SH_PYR, COLOR0, P0, P4, ED, EG, N0,
SH_TET, COLOR0, P1, P4, P0, N0,
SH_PYR, COLOR0, EB, P1, P0, EC, N0,
SH_TET, COLOR0, EC, P0, EG, N0,
SH_PYR, COLOR1, EI, P2, P3, EF, N0,
SH_PYR, COLOR1, P2, EC, EG, P3, N0,
SH_TET, COLOR1, P2, EB, EC, N0,
SH_TET, COLOR1, EB, P2, EI, N0,
SH_TET, COLOR1, P3, EG, ED, N0,
SH_TET, COLOR1, P3, ED, EF, N0,
// Case #13: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EB, EA, ED, EF, EI,
SH_PYR, COLOR0, P5, EF, ED, P4, N0,
SH_TET, COLOR0, EF, P5, EI, N0,
SH_PYR, COLOR0, EA, P1, P4, ED, N0,
SH_PYR, COLOR0, EB, EI, P5, P1, N0,
SH_TET, COLOR0, P5, P4, P1, N0,
SH_TET, COLOR0, P1, EA, EB, N0,
SH_TET, COLOR1, P0, P3, P2, N0,
SH_PYR, COLOR1, EA, P0, P2, EB, N0,
SH_PYR, COLOR1, EA, ED, P3, P0, N0,
SH_TET, COLOR1, ED, EF, P3, N0,
SH_PYR, COLOR1, P3, EF, EI, P2, N0,
SH_TET, COLOR1, P2, EI, EB, N0,
// Case #14: Unique case #8
SH_PNT, 0, COLOR1, 7, ED, EF, EI, EH, P3, P2, P1,
SH_TET, COLOR0, P0, EC, EA, EG,
SH_WDG, COLOR0, EF, EI, P5, ED, EH, P4,
SH_WDG, COLOR1, P2, P1, P3, EC, EA, EG,
SH_PYR, COLOR1, EF, ED, EH, EI, N0,
SH_PYR, COLOR1, EH, P1, P2, EI, N0,
SH_TET, COLOR1, P3, P2, P1, N0,
SH_TET, COLOR1, P3, ED, EF, N0,
SH_PYR, COLOR1, ED, P3, P1, EH, N0,
SH_PYR, COLOR1, EI, P2, P3, EF, N0,
// Case #15: Unique case #9
SH_PNT, 0, COLOR1, 7, P1, P2, P3, EF, ED, EH, EI,
SH_WDG, COLOR0, ED, P4, EH, EF, P5, EI,
SH_TET, COLOR1, P0, P2, P1, P3,
SH_PYR, COLOR1, EF, ED, EH, EI, N0,
SH_PYR, COLOR1, EI, EH, P1, P2, N0,
SH_TET, COLOR1, P3, P2, P1, N0,
SH_TET, COLOR1, P3, ED, EF, N0,
SH_PYR, COLOR1, P3, P1, EH, ED, N0,
SH_PYR, COLOR1, P2, P3, EF, EI, N0,
// Case #16: (cloned #1)
SH_WDG, COLOR0, P5, P3, P1, EE, ED, EH,
SH_PYR, COLOR0, P5, P2, P0, P3, P1,
SH_TET, COLOR1, EH, ED, EE, P4,
// Case #17: (cloned #10)
SH_PNT, 0, NOCOLOR, 6, ED, EE, EH, EA, EC, EG,
SH_PYR, COLOR0, P2, EC, EG, P3, N0,
SH_TET, COLOR0, P3, EG, ED, N0,
SH_PYR, COLOR0, P3, ED, EE, P5, N0,
SH_TET, COLOR0, P2, P3, P5, N0,
SH_PYR, COLOR0, P1, P5, EE, EH, N0,
SH_TET, COLOR0, P2, P5, P1, N0,
SH_PYR, COLOR0, EC, P2, P1, EA, N0,
SH_TET, COLOR0, EA, P1, EH, N0,
SH_PYR, COLOR1, EG, P0, P4, ED, N0,
SH_PYR, COLOR1, P0, EA, EH, P4, N0,
SH_TET, COLOR1, P0, EC, EA, N0,
SH_TET, COLOR1, EC, P0, EG, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
SH_TET, COLOR1, P4, EE, ED, N0,
// Case #18: (cloned #9)
SH_HEX, COLOR0, P2, P0, P3, P5, EB, EA, ED, EE,
SH_WDG, COLOR1, P1, EB, EA, P4, EE, ED,
// Case #19: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EC, EB, EE, ED, EG,
SH_PYR, COLOR0, P3, ED, EE, P5, N0,
SH_TET, COLOR0, ED, P3, EG, N0,
SH_PYR, COLOR0, EB, P2, P5, EE, N0,
SH_PYR, COLOR0, EC, EG, P3, P2, N0,
SH_TET, COLOR0, P3, P5, P2, N0,
SH_TET, COLOR0, P2, EB, EC, N0,
SH_TET, COLOR1, P1, P4, P0, N0,
SH_PYR, COLOR1, EB, P1, P0, EC, N0,
SH_PYR, COLOR1, EB, EE, P4, P1, N0,
SH_TET, COLOR1, EE, ED, P4, N0,
SH_PYR, COLOR1, P4, ED, EG, P0, N0,
SH_TET, COLOR1, P0, EG, EC, N0,
// Case #20: (cloned #10)
SH_PNT, 0, NOCOLOR, 6, EB, EC, EI, EE, ED, EH,
SH_PYR, COLOR0, P3, P1, EH, ED, N0,
SH_TET, COLOR0, P1, EB, EH, N0,
SH_PYR, COLOR0, P1, P0, EC, EB, N0,
SH_TET, COLOR0, P3, P0, P1, N0,
SH_PYR, COLOR0, P5, EI, EC, P0, N0,
SH_TET, COLOR0, P3, P5, P0, N0,
SH_PYR, COLOR0, ED, EE, P5, P3, N0,
SH_TET, COLOR0, EE, EI, P5, N0,
SH_PYR, COLOR1, EH, EB, P2, P4, N0,
SH_PYR, COLOR1, P4, P2, EI, EE, N0,
SH_TET, COLOR1, P4, EE, ED, N0,
SH_TET, COLOR1, ED, EH, P4, N0,
SH_TET, COLOR1, P2, EC, EI, N0,
SH_TET, COLOR1, P2, EB, EC, N0,
// Case #21: (cloned #14)
SH_PNT, 0, COLOR1, 7, EE, ED, EG, EI, P4, P0, P2,
SH_TET, COLOR0, P1, EA, EB, EH,
SH_WDG, COLOR0, ED, EG, P3, EE, EI, P5,
SH_WDG, COLOR1, P0, P2, P4, EA, EB, EH,
SH_PYR, COLOR1, ED, EE, EI, EG, N0,
SH_PYR, COLOR1, EI, P2, P0, EG, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_TET, COLOR1, P4, EE, ED, N0,
SH_PYR, COLOR1, EE, P4, P2, EI, N0,
SH_PYR, COLOR1, EG, P0, P4, ED, N0,
// Case #22: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EC, EA, ED, EE, EI,
SH_PYR, COLOR0, P5, P3, ED, EE, N0,
SH_TET, COLOR0, EE, EI, P5, N0,
SH_PYR, COLOR0, EA, ED, P3, P0, N0,
SH_PYR, COLOR0, EC, P0, P5, EI, N0,
SH_TET, COLOR0, P5, P0, P3, N0,
SH_TET, COLOR0, P0, EC, EA, N0,
SH_TET, COLOR1, P1, P2, P4, N0,
SH_PYR, COLOR1, EA, EC, P2, P1, N0,
SH_PYR, COLOR1, EA, P1, P4, ED, N0,
SH_TET, COLOR1, ED, P4, EE, N0,
SH_PYR, COLOR1, P4, P2, EI, EE, N0,
SH_TET, COLOR1, P2, EC, EI, N0,
// Case #23: (cloned #15)
SH_PNT, 0, COLOR1, 7, P2, P0, P4, ED, EE, EI, EG,
SH_WDG, COLOR0, EE, P5, EI, ED, P3, EG,
SH_TET, COLOR1, P1, P0, P2, P4,
SH_PYR, COLOR1, ED, EE, EI, EG, N0,
SH_PYR, COLOR1, EG, EI, P2, P0, N0,
SH_TET, COLOR1, P4, P0, P2, N0,
SH_TET, COLOR1, P4, EE, ED, N0,
SH_PYR, COLOR1, P4, P2, EI, EE, N0,
SH_PYR, COLOR1, P0, P4, ED, EG, N0,
// Case #24: (cloned #3)
SH_PNT, 0, COLOR0, 7, P5, P0, P1, EE, EF, EG, EH,
SH_TET, COLOR0, P1, P0, P2, P5,
SH_TET, COLOR0, P5, P1, P0, N0,
SH_PYR, COLOR0, EG, P0, P1, EH, N0,
SH_PYR, COLOR0, EE, EF, EG, EH, N0,
SH_TET, COLOR0, P5, EF, EE, N0,
SH_PYR, COLOR0, P5, P0, EG, EF, N0,
SH_PYR, COLOR0, EH, P1, P5, EE, N0,
SH_WDG, COLOR1, EE, EH, P4, EF, EG, P3,
// Case #25: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EE, EF, EC, EA, EH,
SH_PYR, COLOR0, P1, EA, EC, P2, N0,
SH_TET, COLOR0, EA, P1, EH, N0,
SH_PYR, COLOR0, EF, P5, P2, EC, N0,
SH_PYR, COLOR0, EE, EH, P1, P5, N0,
SH_TET, COLOR0, P1, P2, P5, N0,
SH_TET, COLOR0, P5, EF, EE, N0,
SH_TET, COLOR1, P3, P0, P4, N0,
SH_PYR, COLOR1, EF, P3, P4, EE, N0,
SH_PYR, COLOR1, EF, EC, P0, P3, N0,
SH_TET, COLOR1, EC, EA, P0, N0,
SH_PYR, COLOR1, P0, EA, EH, P4, N0,
SH_TET, COLOR1, P4, EH, EE, N0,
// Case #26: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EF, EE, EB, EA, EG,
SH_PYR, COLOR0, P0, P2, EB, EA, N0,
SH_TET, COLOR0, EA, EG, P0, N0,
SH_PYR, COLOR0, EE, EB, P2, P5, N0,
SH_PYR, COLOR0, EF, P5, P0, EG, N0,
SH_TET, COLOR0, P0, P5, P2, N0,
SH_TET, COLOR0, P5, EF, EE, N0,
SH_TET, COLOR1, P4, P3, P1, N0,
SH_PYR, COLOR1, EE, EF, P3, P4, N0,
SH_PYR, COLOR1, EE, P4, P1, EB, N0,
SH_TET, COLOR1, EB, P1, EA, N0,
SH_PYR, COLOR1, P1, P3, EG, EA, N0,
SH_TET, COLOR1, P3, EF, EG, N0,
// Case #27: Unique case #10
SH_WDG, COLOR0, EF, P5, EE, EC, P2, EB,
SH_HEX, COLOR1, P3, P4, EE, EF, P0, P1, EB, EC,
// Case #28: (cloned #14)
SH_PNT, 0, COLOR1, 7, EC, EB, EH, EG, P2, P4, P3,
SH_TET, COLOR0, P5, EF, EE, EI,
SH_WDG, COLOR0, EC, EG, P0, EB, EH, P1,
SH_WDG, COLOR1, EE, EF, EI, P4, P3, P2,
SH_PYR, COLOR1, EB, EH, EG, EC, N0,
SH_PYR, COLOR1, EG, EH, P4, P3, N0,
SH_TET, COLOR1, P2, P3, P4, N0,
SH_TET, COLOR1, P2, EB, EC, N0,
SH_PYR, COLOR1, EC, EG, P3, P2, N0,
SH_PYR, COLOR1, EH, EB, P2, P4, N0,
// Case #29: Unique case #11
SH_TET, COLOR0, P1, EA, EB, EH,
SH_TET, COLOR0, EF, EE, P5, EI,
SH_WDG, COLOR1, P2, P3, P4, EI, EF, EE,
SH_TET, COLOR1, P2, P3, P4, P0,
SH_WDG, COLOR1, P2, P4, P0, EB, EH, EA,
// Case #30: (cloned #29)
SH_TET, COLOR0, P5, EF, EE, EI,
SH_TET, COLOR0, EA, P0, EC, EG,
SH_WDG, COLOR1, EG, EA, EC, P3, P1, P2,
SH_TET, COLOR1, P3, P2, P1, P4,
SH_WDG, COLOR1, EF, EI, EE, P3, P2, P4,
// Case #31: Unique case #12
SH_TET, COLOR0, EF, EI, EE, P5,
SH_WDG, COLOR1, EI, EE, EF, P2, P4, P3,
SH_PYR, COLOR1, P0, P1, P4, P3, P2,
// Case #32: (cloned #1)
SH_WDG, COLOR0, P3, P4, P2, EF, EE, EI,
SH_PYR, COLOR0, P3, P0, P1, P4, P2,
SH_TET, COLOR1, EI, EE, EF, P5,
// Case #33: (cloned #10)
SH_PNT, 0, NOCOLOR, 6, EC, EA, EG, EF, EE, EI,
SH_PYR, COLOR0, P4, P2, EI, EE, N0,
SH_TET, COLOR0, P2, EC, EI, N0,
SH_PYR, COLOR0, P2, P1, EA, EC, N0,
SH_TET, COLOR0, P4, P1, P2, N0,
SH_PYR, COLOR0, P3, EG, EA, P1, N0,
SH_TET, COLOR0, P4, P3, P1, N0,
SH_PYR, COLOR0, EE, EF, P3, P4, N0,
SH_TET, COLOR0, EF, EG, P3, N0,
SH_PYR, COLOR1, EI, EC, P0, P5, N0,
SH_PYR, COLOR1, P5, P0, EG, EF, N0,
SH_TET, COLOR1, P5, EF, EE, N0,
SH_TET, COLOR1, EE, EI, P5, N0,
SH_TET, COLOR1, P0, EA, EG, N0,
SH_TET, COLOR1, P0, EC, EA, N0,
// Case #34: (cloned #10)
SH_PNT, 0, NOCOLOR, 6, EE, EF, EI, EB, EA, EH,
SH_PYR, COLOR0, P0, EA, EH, P4, N0,
SH_TET, COLOR0, P4, EH, EE, N0,
SH_PYR, COLOR0, P4, EE, EF, P3, N0,
SH_TET, COLOR0, P0, P4, P3, N0,
SH_PYR, COLOR0, P2, P3, EF, EI, N0,
SH_TET, COLOR0, P0, P3, P2, N0,
SH_PYR, COLOR0, EA, P0, P2, EB, N0,
SH_TET, COLOR0, EB, P2, EI, N0,
SH_PYR, COLOR1, EH, P1, P5, EE, N0,
SH_PYR, COLOR1, P1, EB, EI, P5, N0,
SH_TET, COLOR1, P1, EA, EB, N0,
SH_TET, COLOR1, EA, P1, EH, N0,
SH_TET, COLOR1, P5, EI, EF, N0,
SH_TET, COLOR1, P5, EF, EE, N0,
// Case #35: (cloned #14)
SH_PNT, 0, COLOR1, 7, EF, EE, EH, EG, P5, P1, P0,
SH_TET, COLOR0, P2, EB, EC, EI,
SH_WDG, COLOR0, EE, EH, P4, EF, EG, P3,
SH_WDG, COLOR1, P1, P0, P5, EB, EC, EI,
SH_PYR, COLOR1, EE, EF, EG, EH, N0,
SH_PYR, COLOR1, EG, P0, P1, EH, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_TET, COLOR1, P5, EF, EE, N0,
SH_PYR, COLOR1, EF, P5, P0, EG, N0,
SH_PYR, COLOR1, EH, P1, P5, EE, N0,
// Case #36: (cloned #9)
SH_HEX, COLOR0, P0, P1, P4, P3, EC, EB, EE, EF,
SH_WDG, COLOR1, P2, EC, EB, P5, EF, EE,
// Case #37: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EA, EB, EE, EF, EG,
SH_PYR, COLOR0, P3, P4, EE, EF, N0,
SH_TET, COLOR0, EF, EG, P3, N0,
SH_PYR, COLOR0, EB, EE, P4, P1, N0,
SH_PYR, COLOR0, EA, P1, P3, EG, N0,
SH_TET, COLOR0, P3, P1, P4, N0,
SH_TET, COLOR0, P1, EA, EB, N0,
SH_TET, COLOR1, P2, P0, P5, N0,
SH_PYR, COLOR1, EB, EA, P0, P2, N0,
SH_PYR, COLOR1, EB, P2, P5, EE, N0,
SH_TET, COLOR1, EE, P5, EF, N0,
SH_PYR, COLOR1, P5, P0, EG, EF, N0,
SH_TET, COLOR1, P0, EA, EG, N0,
// Case #38: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EA, EC, EF, EE, EH,
SH_PYR, COLOR0, P4, EE, EF, P3, N0,
SH_TET, COLOR0, EE, P4, EH, N0,
SH_PYR, COLOR0, EC, P0, P3, EF, N0,
SH_PYR, COLOR0, EA, EH, P4, P0, N0,
SH_TET, COLOR0, P4, P3, P0, N0,
SH_TET, COLOR0, P0, EC, EA, N0,
SH_TET, COLOR1, P2, P5, P1, N0,
SH_PYR, COLOR1, EC, P2, P1, EA, N0,
SH_PYR, COLOR1, EC, EF, P5, P2, N0,
SH_TET, COLOR1, EF, EE, P5, N0,
SH_PYR, COLOR1, P5, EE, EH, P1, N0,
SH_TET, COLOR1, P1, EH, EA, N0,
// Case #39: (cloned #15)
SH_PNT, 0, COLOR1, 7, P0, P1, P5, EE, EF, EG, EH,
SH_WDG, COLOR0, EF, P3, EG, EE, P4, EH,
SH_TET, COLOR1, P2, P1, P0, P5,
SH_PYR, COLOR1, EE, EF, EG, EH, N0,
SH_PYR, COLOR1, EH, EG, P0, P1, N0,
SH_TET, COLOR1, P5, P1, P0, N0,
SH_TET, COLOR1, P5, EF, EE, N0,
SH_PYR, COLOR1, P5, P0, EG, EF, N0,
SH_PYR, COLOR1, P1, P5, EE, EH, N0,
// Case #40: (cloned #3)
SH_PNT, 0, COLOR0, 7, P4, P2, P0, ED, EE, EI, EG,
SH_TET, COLOR0, P0, P2, P1, P4,
SH_TET, COLOR0, P4, P0, P2, N0,
SH_PYR, COLOR0, EI, P2, P0, EG, N0,
SH_PYR, COLOR0, ED, EE, EI, EG, N0,
SH_TET, COLOR0, P4, EE, ED, N0,
SH_PYR, COLOR0, P4, P2, EI, EE, N0,
SH_PYR, COLOR0, EG, P0, P4, ED, N0,
SH_WDG, COLOR1, ED, EG, P3, EE, EI, P5,
// Case #41: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EE, ED, EA, EC, EI,
SH_PYR, COLOR0, P2, P1, EA, EC, N0,
SH_TET, COLOR0, EC, EI, P2, N0,
SH_PYR, COLOR0, ED, EA, P1, P4, N0,
SH_PYR, COLOR0, EE, P4, P2, EI, N0,
SH_TET, COLOR0, P2, P4, P1, N0,
SH_TET, COLOR0, P4, EE, ED, N0,
SH_TET, COLOR1, P3, P5, P0, N0,
SH_PYR, COLOR1, ED, EE, P5, P3, N0,
SH_PYR, COLOR1, ED, P3, P0, EA, N0,
SH_TET, COLOR1, EA, P0, EC, N0,
SH_PYR, COLOR1, P0, P5, EI, EC, N0,
SH_TET, COLOR1, P5, EE, EI, N0,
// Case #42: (cloned #14)
SH_PNT, 0, COLOR1, 7, EB, EA, EG, EI, P1, P3, P5,
SH_TET, COLOR0, P4, EE, ED, EH,
SH_WDG, COLOR0, EB, EI, P2, EA, EG, P0,
SH_WDG, COLOR1, ED, EE, EH, P3, P5, P1,
SH_PYR, COLOR1, EA, EG, EI, EB, N0,
SH_PYR, COLOR1, EI, EG, P3, P5, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_TET, COLOR1, P1, EA, EB, N0,
SH_PYR, COLOR1, EB, EI, P5, P1, N0,
SH_PYR, COLOR1, EG, EA, P1, P3, N0,
// Case #43: (cloned #29)
SH_TET, COLOR0, P4, EE, ED, EH,
SH_TET, COLOR0, EC, P2, EB, EI,
SH_WDG, COLOR1, EI, EC, EB, P5, P0, P1,
SH_TET, COLOR1, P5, P1, P0, P3,
SH_WDG, COLOR1, EE, EH, ED, P5, P1, P3,
// Case #44: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, ED, EE, EB, EC, EG,
SH_PYR, COLOR0, P0, EC, EB, P1, N0,
SH_TET, COLOR0, EC, P0, EG, N0,
SH_PYR, COLOR0, EE, P4, P1, EB, N0,
SH_PYR, COLOR0, ED, EG, P0, P4, N0,
SH_TET, COLOR0, P0, P1, P4, N0,
SH_TET, COLOR0, P4, EE, ED, N0,
SH_TET, COLOR1, P5, P2, P3, N0,
SH_PYR, COLOR1, EE, P5, P3, ED, N0,
SH_PYR, COLOR1, EE, EB, P2, P5, N0,
SH_TET, COLOR1, EB, EC, P2, N0,
SH_PYR, COLOR1, P2, EC, EG, P3, N0,
SH_TET, COLOR1, P3, EG, ED, N0,
// Case #45: (cloned #27)
SH_WDG, COLOR0, EE, P4, ED, EB, P1, EA,
SH_HEX, COLOR1, P5, P3, ED, EE, P2, P0, EA, EB,
// Case #46: (cloned #29)
SH_TET, COLOR0, P0, EC, EA, EG,
SH_TET, COLOR0, EE, ED, P4, EH,
SH_WDG, COLOR1, P1, P5, P3, EH, EE, ED,
SH_TET, COLOR1, P1, P5, P3, P2,
SH_WDG, COLOR1, P1, P3, P2, EA, EG, EC,
// Case #47: (cloned #31)
SH_TET, COLOR0, EE, EH, ED, P4,
SH_WDG, COLOR1, EH, ED, EE, P1, P3, P5,
SH_PYR, COLOR1, P2, P0, P3, P5, P1,
// Case #48: (cloned #3)
SH_PNT, 0, COLOR0, 7, P3, P1, P2, EF, ED, EH, EI,
SH_TET, COLOR0, P2, P1, P0, P3,
SH_TET, COLOR0, P3, P2, P1, N0,
SH_PYR, COLOR0, EH, P1, P2, EI, N0,
SH_PYR, COLOR0, EF, ED, EH, EI, N0,
SH_TET, COLOR0, P3, ED, EF, N0,
SH_PYR, COLOR0, P3, P1, EH, ED, N0,
SH_PYR, COLOR0, EI, P2, P3, EF, N0,
SH_WDG, COLOR1, EF, EI, P5, ED, EH, P4,
// Case #49: (cloned #14)
SH_PNT, 0, COLOR1, 7, EA, EC, EI, EH, P0, P5, P4,
SH_TET, COLOR0, P3, ED, EF, EG,
SH_WDG, COLOR0, EA, EH, P1, EC, EI, P2,
SH_WDG, COLOR1, EF, ED, EG, P5, P4, P0,
SH_PYR, COLOR1, EC, EI, EH, EA, N0,
SH_PYR, COLOR1, EH, EI, P5, P4, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, EC, EA, N0,
SH_PYR, COLOR1, EA, EH, P4, P0, N0,
SH_PYR, COLOR1, EI, EC, P0, P5, N0,
// Case #50: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, EF, ED, EA, EB, EI,
SH_PYR, COLOR0, P2, EB, EA, P0, N0,
SH_TET, COLOR0, EB, P2, EI, N0,
SH_PYR, COLOR0, ED, P3, P0, EA, N0,
SH_PYR, COLOR0, EF, EI, P2, P3, N0,
SH_TET, COLOR0, P2, P0, P3, N0,
SH_TET, COLOR0, P3, ED, EF, N0,
SH_TET, COLOR1, P4, P1, P5, N0,
SH_PYR, COLOR1, ED, P4, P5, EF, N0,
SH_PYR, COLOR1, ED, EA, P1, P4, N0,
SH_TET, COLOR1, EA, EB, P1, N0,
SH_PYR, COLOR1, P1, EB, EI, P5, N0,
SH_TET, COLOR1, P5, EI, EF, N0,
// Case #51: (cloned #29)
SH_TET, COLOR0, P2, EB, EC, EI,
SH_TET, COLOR0, ED, EF, P3, EG,
SH_WDG, COLOR1, P0, P4, P5, EG, ED, EF,
SH_TET, COLOR1, P0, P4, P5, P1,
SH_WDG, COLOR1, P0, P5, P1, EC, EI, EB,
// Case #52: (cloned #11)
SH_PNT, 0, NOCOLOR, 5, ED, EF, EC, EB, EH,
SH_PYR, COLOR0, P1, P0, EC, EB, N0,
SH_TET, COLOR0, EB, EH, P1, N0,
SH_PYR, COLOR0, EF, EC, P0, P3, N0,
SH_PYR, COLOR0, ED, P3, P1, EH, N0,
SH_TET, COLOR0, P1, P3, P0, N0,
SH_TET, COLOR0, P3, ED, EF, N0,
SH_TET, COLOR1, P5, P4, P2, N0,
SH_PYR, COLOR1, EF, ED, P4, P5, N0,
SH_PYR, COLOR1, EF, P5, P2, EC, N0,
SH_TET, COLOR1, EC, P2, EB, N0,
SH_PYR, COLOR1, P2, P4, EH, EB, N0,
SH_TET, COLOR1, P4, ED, EH, N0,
// Case #53: (cloned #29)
SH_TET, COLOR0, P3, ED, EF, EG,
SH_TET, COLOR0, EB, P1, EA, EH,
SH_WDG, COLOR1, EH, EB, EA, P4, P2, P0,
SH_TET, COLOR1, P4, P0, P2, P5,
SH_WDG, COLOR1, ED, EG, EF, P4, P0, P5,
// Case #54: (cloned #27)
SH_WDG, COLOR0, ED, P3, EF, EA, P0, EC,
SH_HEX, COLOR1, P4, P5, EF, ED, P1, P2, EC, EA,
// Case #55: (cloned #31)
SH_TET, COLOR0, ED, EG, EF, P3,
SH_WDG, COLOR1, EG, EF, ED, P0, P5, P4,
SH_PYR, COLOR1, P1, P2, P5, P4, P0,
// Case #56: (cloned #7)
SH_WDG, COLOR0, P0, P1, P2, EG, EH, EI,
SH_WDG, COLOR1, EG, EH, EI, P3, P4, P5,
// Case #57: (cloned #15)
SH_PNT, 0, COLOR1, 7, P4, P5, P0, EC, EA, EH, EI,
SH_WDG, COLOR0, EC, P2, EI, EA, P1, EH,
SH_TET, COLOR1, P3, P4, P5, P0,
SH_PYR, COLOR1, EC, EI, EH, EA, N0,
SH_PYR, COLOR1, EI, P5, P4, EH, N0,
SH_TET, COLOR1, P0, P4, P5, N0,
SH_TET, COLOR1, P0, EC, EA, N0,
SH_PYR, COLOR1, P0, EA, EH, P4, N0,
SH_PYR, COLOR1, P5, EI, EC, P0, N0,
// Case #58: (cloned #15)
SH_PNT, 0, COLOR1, 7, P5, P3, P1, EA, EB, EI, EG,
SH_WDG, COLOR0, EA, P0, EG, EB, P2, EI,
SH_TET, COLOR1, P4, P5, P3, P1,
SH_PYR, COLOR1, EA, EG, EI, EB, N0,
SH_PYR, COLOR1, EG, P3, P5, EI, N0,
SH_TET, COLOR1, P1, P5, P3, N0,
SH_TET, COLOR1, P1, EA, EB, N0,
SH_PYR, COLOR1, P1, EB, EI, P5, N0,
SH_PYR, COLOR1, P3, EG, EA, P1, N0,
// Case #59: (cloned #31)
SH_TET, COLOR0, EC, EB, EI, P2,
SH_WDG, COLOR1, P5, P1, P0, EI, EB, EC,
SH_PYR, COLOR1, P3, P0, P1, P4, P5,
// Case #60: (cloned #15)
SH_PNT, 0, COLOR1, 7, P3, P4, P2, EB, EC, EG, EH,
SH_WDG, COLOR0, EB, P1, EH, EC, P0, EG,
SH_TET, COLOR1, P5, P3, P4, P2,
SH_PYR, COLOR1, EB, EH, EG, EC, N0,
SH_PYR, COLOR1, EH, P4, P3, EG, N0,
SH_TET, COLOR1, P2, P3, P4, N0,
SH_TET, COLOR1, P2, EB, EC, N0,
SH_PYR, COLOR1, P2, EC, EG, P3, N0,
SH_PYR, COLOR1, P4, EH, EB, P2, N0,
// Case #61: (cloned #31)
SH_TET, COLOR0, EB, EA, EH, P1,
SH_WDG, COLOR1, P4, P0, P2, EH, EA, EB,
SH_PYR, COLOR1, P5, P2, P0, P3, P4,
// Case #62: (cloned #31)
SH_TET, COLOR0, EA, EC, EG, P0,
SH_WDG, COLOR1, P3, P2, P1, EG, EC, EA,
SH_PYR, COLOR1, P4, P1, P2, P5, P3,
// Case #63: Unique case #13
SH_WDG, COLOR1, P0, P1, P2, P3, P4, P5,
//}
//{Pyramid
 // Case #0: Unique case #1
SH_PYR, COLOR0, P0, P1, P2, P3, P4,
// Case #1: Unique case #2
SH_WDG, COLOR0, EA, EE, ED, P1, P4, P3,
SH_TET, COLOR0, P1, P2, P3, P4,
SH_TET, COLOR1, P0, EA, ED, EE,
// Case #2: (cloned #1)
SH_WDG, COLOR0, EB, EF, EA, P2, P4, P0,
SH_TET, COLOR0, P2, P3, P0, P4,
SH_TET, COLOR1, P1, EB, EA, EF,
// Case #3: Unique case #3
SH_PNT, 0, COLOR0, 7, P4, EF, EE, EB, ED, P2, P3,
SH_TET, COLOR0, EE, P4, EF, N0,
SH_PYR, COLOR0, EB, ED, EE, EF, N0,
SH_PYR, COLOR0, EB, EF, P4, P2, N0,
SH_TET, COLOR0, P2, P4, P3, N0,
SH_PYR, COLOR0, P3, P4, EE, ED, N0,
SH_PYR, COLOR0, P2, P3, ED, EB, N0,
SH_WDG, COLOR1, EB, EF, P1, ED, EE, P0,
// Case #4: (cloned #1)
SH_WDG, COLOR0, EC, EG, EB, P3, P4, P1,
SH_TET, COLOR0, P3, P0, P1, P4,
SH_TET, COLOR1, P2, EC, EB, EG,
// Case #5: Unique case #4
SH_WDG, COLOR0, EE, P4, EG, EA, P1, EB,
SH_WDG, COLOR0, P4, EE, EG, P3, ED, EC,
SH_WDG, COLOR1, P0, EA, EE, P2, EB, EG,
SH_WDG, COLOR1, ED, P0, EE, EC, P2, EG,
// Case #6: (cloned #3)
SH_PNT, 0, COLOR0, 7, P4, EG, EF, EC, EA, P3, P0,
SH_TET, COLOR0, EF, P4, EG, N0,
SH_PYR, COLOR0, EC, EA, EF, EG, N0,
SH_PYR, COLOR0, EC, EG, P4, P3, N0,
SH_TET, COLOR0, P3, P4, P0, N0,
SH_PYR, COLOR0, P0, P4, EF, EA, N0,
SH_PYR, COLOR0, P3, P0, EA, EC, N0,
SH_WDG, COLOR1, EC, EG, P2, EA, EF, P1,
// Case #7: Unique case #5
SH_TET, COLOR0, EE, EF, EG, P4,
SH_WDG, COLOR0, EC, ED, P3, EG, EE, P4,
SH_WDG, COLOR1, EE, EF, EG, P0, P1, P2,
SH_WDG, COLOR1, P2, EC, EG, P0, ED, EE,
// Case #8: (cloned #1)
SH_WDG, COLOR0, ED, EH, EC, P0, P4, P2,
SH_TET, COLOR0, P0, P1, P2, P4,
SH_TET, COLOR1, P3, ED, EC, EH,
// Case #9: (cloned #3)
SH_PNT, 0, COLOR0, 7, P4, EE, EH, EA, EC, P1, P2,
SH_TET, COLOR0, EH, P4, EE, N0,
SH_PYR, COLOR0, EA, EC, EH, EE, N0,
SH_PYR, COLOR0, EA, EE, P4, P1, N0,
SH_TET, COLOR0, P1, P4, P2, N0,
SH_PYR, COLOR0, P2, P4, EH, EC, N0,
SH_PYR, COLOR0, P1, P2, EC, EA, N0,
SH_WDG, COLOR1, EA, EE, P0, EC, EH, P3,
// Case #10: (cloned #5)
SH_WDG, COLOR0, EH, P4, EF, ED, P0, EA,
SH_WDG, COLOR0, P4, EH, EF, P2, EC, EB,
SH_WDG, COLOR1, P3, ED, EH, P1, EA, EF,
SH_WDG, COLOR1, EC, P3, EH, EB, P1, EF,
// Case #11: (cloned #7)
SH_TET, COLOR0, EH, EE, EF, P4,
SH_WDG, COLOR0, EB, EC, P2, EF, EH, P4,
SH_WDG, COLOR1, EH, EE, EF, P3, P0, P1,
SH_WDG, COLOR1, P1, EB, EF, P3, EC, EH,
// Case #12: (cloned #3)
SH_PNT, 0, COLOR0, 7, P4, EH, EG, ED, EB, P0, P1,
SH_TET, COLOR0, EG, P4, EH, N0,
SH_PYR, COLOR0, ED, EB, EG, EH, N0,
SH_PYR, COLOR0, ED, EH, P4, P0, N0,
SH_TET, COLOR0, P0, P4, P1, N0,
SH_PYR, COLOR0, P1, P4, EG, EB, N0,
SH_PYR, COLOR0, P0, P1, EB, ED, N0,
SH_WDG, COLOR1, ED, EH, P3, EB, EG, P2,
// Case #13: (cloned #7)
SH_TET, COLOR0, EG, EH, EE, P4,
SH_WDG, COLOR0, EA, EB, P1, EE, EG, P4,
SH_WDG, COLOR1, EG, EH, EE, P2, P3, P0,
SH_WDG, COLOR1, P0, EA, EE, P2, EB, EG,
// Case #14: (cloned #7)
SH_TET, COLOR0, EF, EG, EH, P4,
SH_WDG, COLOR0, ED, EA, P0, EH, EF, P4,
SH_WDG, COLOR1, EF, EG, EH, P1, P2, P3,
SH_WDG, COLOR1, P3, ED, EH, P1, EA, EF,
// Case #15: Unique case #6
SH_PYR, COLOR0, EE, EF, EG, EH, P4,
SH_HEX, COLOR1, P0, P1, P2, P3, EE, EF, EG, EH,
// Case #16: Unique case #7
SH_HEX, COLOR0, P0, P1, P2, P3, EE, EF, EG, EH,
SH_PYR, COLOR1, EE, EF, EG, EH, P4,
// Case #17: Unique case #8
SH_WDG, COLOR0, ED, EH, P3, EA, EF, P1,
SH_WDG, COLOR0, EF, EG, EH, P1, P2, P3,
SH_WDG, COLOR1, P4, EF, EH, P0, EA, ED,
SH_TET, COLOR1, EF, EG, EH, P4,
// Case #18: (cloned #17)
SH_WDG, COLOR0, EA, EE, P0, EB, EG, P2,
SH_WDG, COLOR0, EG, EH, EE, P2, P3, P0,
SH_WDG, COLOR1, P4, EG, EE, P1, EB, EA,
SH_TET, COLOR1, EG, EH, EE, P4,
// Case #19: Unique case #9
SH_PNT, 0, COLOR1, 7, EH, EG, ED, EB, P0, P1, P4,
SH_WDG, COLOR0, ED, EH, P3, EB, EG, P2,
SH_PYR, COLOR1, EG, EH, ED, EB, N0,
SH_PYR, COLOR1, ED, P0, P1, EB, N0,
SH_TET, COLOR1, P0, P4, P1, N0,
SH_TET, COLOR1, EH, EG, P4, N0,
SH_PYR, COLOR1, EH, P4, P0, ED, N0,
SH_PYR, COLOR1, P4, EG, EB, P1, N0,
// Case #20: (cloned #17)
SH_WDG, COLOR0, EB, EF, P1, EC, EH, P3,
SH_WDG, COLOR0, EH, EE, EF, P3, P0, P1,
SH_WDG, COLOR1, P4, EH, EF, P2, EC, EB,
SH_TET, COLOR1, EH, EE, EF, P4,
// Case #21: Unique case #10
SH_TET, COLOR0, EA, P1, EB, EF,
SH_TET, COLOR0, P3, ED, EC, EH,
SH_WDG, COLOR1, EA, EB, EF, P0, P2, P4,
SH_WDG, COLOR1, EC, ED, EH, P2, P0, P4,
// Case #22: (cloned #19)
SH_PNT, 0, COLOR1, 7, EE, EH, EA, EC, P1, P2, P4,
SH_WDG, COLOR0, EA, EE, P0, EC, EH, P3,
SH_PYR, COLOR1, EH, EE, EA, EC, N0,
SH_PYR, COLOR1, EA, P1, P2, EC, N0,
SH_TET, COLOR1, P1, P4, P2, N0,
SH_TET, COLOR1, EE, EH, P4, N0,
SH_PYR, COLOR1, EE, P4, P1, EA, N0,
SH_PYR, COLOR1, P4, EH, EC, P2, N0,
// Case #23: Unique case #11
SH_TET, COLOR0, P3, ED, EC, EH,
SH_WDG, COLOR1, P0, P2, P4, ED, EC, EH,
SH_TET, COLOR1, P0, P1, P2, P4,
// Case #24: (cloned #17)
SH_WDG, COLOR0, EC, EG, P2, ED, EE, P0,
SH_WDG, COLOR0, EE, EF, EG, P0, P1, P2,
SH_WDG, COLOR1, P4, EE, EG, P3, ED, EC,
SH_TET, COLOR1, EE, EF, EG, P4,
// Case #25: (cloned #19)
SH_PNT, 0, COLOR1, 7, EG, EF, EC, EA, P3, P0, P4,
SH_WDG, COLOR0, EC, EG, P2, EA, EF, P1,
SH_PYR, COLOR1, EF, EG, EC, EA, N0,
SH_PYR, COLOR1, EC, P3, P0, EA, N0,
SH_TET, COLOR1, P3, P4, P0, N0,
SH_TET, COLOR1, EG, EF, P4, N0,
SH_PYR, COLOR1, EG, P4, P3, EC, N0,
SH_PYR, COLOR1, P4, EF, EA, P0, N0,
// Case #26: (cloned #21)
SH_TET, COLOR0, ED, P0, EA, EE,
SH_TET, COLOR0, P2, EC, EB, EG,
SH_WDG, COLOR1, ED, EA, EE, P3, P1, P4,
SH_WDG, COLOR1, EB, EC, EG, P1, P3, P4,
// Case #27: (cloned #23)
SH_TET, COLOR0, P2, EC, EB, EG,
SH_WDG, COLOR1, P3, P1, P4, EC, EB, EG,
SH_TET, COLOR1, P3, P0, P1, P4,
// Case #28: (cloned #19)
SH_PNT, 0, COLOR1, 7, EF, EE, EB, ED, P2, P3, P4,
SH_WDG, COLOR0, EB, EF, P1, ED, EE, P0,
SH_PYR, COLOR1, EE, EF, EB, ED, N0,
SH_PYR, COLOR1, EB, P2, P3, ED, N0,
SH_TET, COLOR1, P2, P4, P3, N0,
SH_TET, COLOR1, EF, EE, P4, N0,
SH_PYR, COLOR1, EF, P4, P2, EB, N0,
SH_PYR, COLOR1, P4, EE, ED, P3, N0,
// Case #29: (cloned #23)
SH_TET, COLOR0, P1, EB, EA, EF,
SH_WDG, COLOR1, P2, P0, P4, EB, EA, EF,
SH_TET, COLOR1, P2, P3, P0, P4,
// Case #30: (cloned #23)
SH_TET, COLOR0, P0, EA, ED, EE,
SH_WDG, COLOR1, P1, P3, P4, EA, ED, EE,
SH_TET, COLOR1, P1, P2, P3, P4,
// Case #31: Unique case #12
SH_PYR, COLOR1, P0, P1, P2, P3, P4,
//}
//{Poly5
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P2, P3,
SH_TRI, COLOR0, P0, P3, P4,
// Case #1: Unique case #2
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_TRI, COLOR1, P0, EA, EE,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P0, EA,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P0, P1, EB, EE,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EC, P3, P4, P0,
SH_QUA, COLOR0, EC, P0, P1, EB,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: Unique case #4
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, EE,
// Case #6: (cloned #3)
SH_QUA, COLOR0, EC, P3, P4, P0,
SH_TRI, COLOR0, EC, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #5
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_TRI, COLOR1, EC, EE, P0,
// Case #8: (cloned #1)
SH_QUA, COLOR0, ED, P4, P0, P1,
SH_QUA, COLOR0, ED, P1, P2, EC,
SH_TRI, COLOR1, P3, ED, EC,
// Case #9: (cloned #5)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #10: (cloned #5)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EA,
// Case #11: Unique case #6
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_TRI, COLOR1, P0, ED, EE,
// Case #12: (cloned #3)
SH_QUA, COLOR0, ED, P4, P0, P1,
SH_TRI, COLOR0, ED, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EB,
// Case #13: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #14: (cloned #7)
SH_QUA, COLOR0, ED, P4, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_TRI, COLOR1, ED, EA, P1,
// Case #15: Unique case #7
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EE,
// Case #16: (cloned #1)
SH_QUA, COLOR0, EE, P0, P1, P2,
SH_QUA, COLOR0, EE, P2, P3, ED,
SH_TRI, COLOR1, P4, EE, ED,
// Case #17: (cloned #3)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #18: (cloned #5)
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EA, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #19: (cloned #7)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P0, P1, EB,
SH_TRI, COLOR1, EB, ED, P4,
// Case #20: (cloned #5)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EB,
// Case #21: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P0, EA, EB,
SH_QUA, COLOR1, P4, EB, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #22: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_TRI, COLOR1, P1, EE, EA,
// Case #23: (cloned #15)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P0, P1, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #24: (cloned #3)
SH_QUA, COLOR0, EE, P0, P1, P2,
SH_TRI, COLOR0, EE, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EC,
// Case #25: (cloned #7)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P0, EA,
SH_TRI, COLOR1, EA, EC, P3,
// Case #26: (cloned #11)
SH_TRI, COLOR0, EE, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EA,
SH_QUA, COLOR1, P3, EA, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #27: (cloned #15)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P0, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #28: (cloned #7)
SH_QUA, COLOR0, EE, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_TRI, COLOR1, EE, EB, P2,
// Case #29: (cloned #15)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #30: (cloned #15)
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #31: Unique case #8
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_TRI, COLOR1, P0, P3, P4,
//}
//{Poly6
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P2, P3,
SH_QUA, COLOR0, P0, P3, P4, P5,
// Case #1: Unique case #2
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_TRI, COLOR0, EA, P5, EF,
SH_TRI, COLOR1, P0, EA, EF,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P0,
SH_TRI, COLOR0, EB, P0, EA,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P0, P1, EB, EF,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P0, P1,
SH_TRI, COLOR0, EC, P1, EB,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: Unique case #4
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, EF,
// Case #6: (cloned #3)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #5
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_TRI, COLOR1, P0, EC, EF,
// Case #8: (cloned #1)
SH_QUA, COLOR0, ED, P4, P5, P0,
SH_QUA, COLOR0, ED, P0, P1, P2,
SH_TRI, COLOR0, ED, P2, EC,
SH_TRI, COLOR1, P3, ED, EC,
// Case #9: Unique case #6
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P0, EA, EC, P3,
SH_QUA, COLOR1, P0, P3, ED, EF,
// Case #10: (cloned #5)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P0,
SH_TRI, COLOR0, ED, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EA,
// Case #11: Unique case #7
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_TRI, COLOR1, P0, ED, EF,
// Case #12: (cloned #3)
SH_QUA, COLOR0, ED, P4, P5, P0,
SH_QUA, COLOR0, ED, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EB,
// Case #13: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, EA, EB, P2, P3,
SH_QUA, COLOR1, EF, P0, EA, P3,
SH_TRI, COLOR1, ED, EF, P3,
// Case #14: (cloned #7)
SH_QUA, COLOR0, ED, P4, P5, P0,
SH_TRI, COLOR0, ED, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_TRI, COLOR1, P1, ED, EA,
// Case #15: Unique case #8
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EF,
// Case #16: (cloned #1)
SH_QUA, COLOR0, EE, P5, P0, P1,
SH_QUA, COLOR0, EE, P1, P2, P3,
SH_TRI, COLOR0, EE, P3, ED,
SH_TRI, COLOR1, P4, EE, ED,
// Case #17: (cloned #5)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P0,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #18: (cloned #9)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P0, EA,
SH_QUA, COLOR1, P1, EB, ED, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #19: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, EE, EF, P0, P1,
SH_QUA, COLOR1, ED, P4, EE, P1,
SH_TRI, COLOR1, EB, ED, P1,
// Case #20: (cloned #5)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P0, P1,
SH_TRI, COLOR0, EE, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EB,
// Case #21: Unique case #9
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_TRI, COLOR1, P0, EE, EF,
// Case #22: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_TRI, COLOR1, P1, EE, EA,
// Case #23: Unique case #10
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, EE, EF,
// Case #24: (cloned #3)
SH_QUA, COLOR0, EE, P5, P0, P1,
SH_QUA, COLOR0, EE, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EC,
// Case #25: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #26: (cloned #11)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EE, P5, P0, EA,
SH_QUA, COLOR1, EB, EC, P3, P4,
SH_QUA, COLOR1, EA, P1, EB, P4,
SH_TRI, COLOR1, EE, EA, P4,
// Case #27: Unique case #11
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, P4,
SH_QUA, COLOR1, P0, P4, EE, EF,
// Case #28: (cloned #7)
SH_QUA, COLOR0, EE, P5, P0, P1,
SH_TRI, COLOR0, EE, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_TRI, COLOR1, P2, EE, EB,
// Case #29: (cloned #23)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #30: (cloned #15)
SH_QUA, COLOR0, EE, P5, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #31: Unique case #12
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, EE,
SH_TRI, COLOR1, EE, EF, P0,
// Case #32: (cloned #1)
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_QUA, COLOR0, EF, P2, P3, P4,
SH_TRI, COLOR0, EF, P4, EE,
SH_TRI, COLOR1, P5, EF, EE,
// Case #33: (cloned #3)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, P0, EA, EE,
// Case #34: (cloned #5)
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, EF, EA, P1,
SH_QUA, COLOR1, P5, P1, EB, EE,
// Case #35: (cloned #7)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P0, P1, EB,
SH_TRI, COLOR1, P5, EB, EE,
// Case #36: (cloned #9)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, EE, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #37: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P0, EA, EB,
SH_QUA, COLOR1, P5, EB, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #38: (cloned #11)
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, EF, EA, P1, P2,
SH_QUA, COLOR1, EE, P5, EF, P2,
SH_TRI, COLOR1, EC, EE, P2,
// Case #39: (cloned #15)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P0, P1, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #40: (cloned #5)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_TRI, COLOR0, EF, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EC,
// Case #41: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, ED, EE, P5, P0,
SH_QUA, COLOR1, EC, P3, ED, P0,
SH_TRI, COLOR1, EA, EC, P0,
// Case #42: (cloned #21)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #43: (cloned #23)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P0, P1, EB,
SH_QUA, COLOR1, P5, EB, EC, P3,
SH_QUA, COLOR1, P5, P3, ED, EE,
// Case #44: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, EF,
SH_TRI, COLOR1, P2, EF, EB,
// Case #45: (cloned #27)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #46: (cloned #23)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #47: (cloned #31)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P0, P1, P2,
SH_QUA, COLOR1, P5, P2, P3, ED,
SH_TRI, COLOR1, ED, EE, P5,
// Case #48: (cloned #3)
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_QUA, COLOR0, EF, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, ED,
// Case #49: (cloned #7)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, P0, EA,
SH_TRI, COLOR1, P4, EA, ED,
// Case #50: (cloned #11)
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EA,
SH_QUA, COLOR1, P4, EA, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #51: (cloned #15)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P0, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #52: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, EC, ED, P4, P5,
SH_QUA, COLOR1, EB, P2, EC, P5,
SH_TRI, COLOR1, EF, EB, P5,
// Case #53: (cloned #23)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P0, EA,
SH_QUA, COLOR1, P4, EA, EB, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #54: (cloned #27)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #55: (cloned #31)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P0, P1,
SH_QUA, COLOR1, P4, P1, P2, EC,
SH_TRI, COLOR1, EC, ED, P4,
// Case #56: (cloned #7)
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_TRI, COLOR0, EF, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_TRI, COLOR1, P3, EF, EC,
// Case #57: (cloned #15)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #58: (cloned #23)
SH_TRI, COLOR0, EF, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EA, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #59: (cloned #31)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P0,
SH_QUA, COLOR1, P3, P0, P1, EB,
SH_TRI, COLOR1, EB, EC, P3,
// Case #60: (cloned #15)
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #61: (cloned #31)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P0, EA,
SH_TRI, COLOR1, EA, EB, P2,
// Case #62: (cloned #31)
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, EF,
SH_TRI, COLOR1, EF, EA, P1,
// Case #63: Unique case #13
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
//}
//{Poly7
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P2, P3,
SH_QUA, COLOR0, P0, P3, P4, P5,
SH_TRI, COLOR0, P0, P5, P6,
// Case #1: Unique case #2
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_QUA, COLOR0, EA, P5, P6, EG,
SH_TRI, COLOR1, P0, EA, EG,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_QUA, COLOR0, EB, P6, P0, EA,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_TRI, COLOR0, EB, P6, EG,
SH_QUA, COLOR1, P0, P1, EB, EG,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P0,
SH_QUA, COLOR0, EC, P0, P1, EB,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: Unique case #4
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, EG,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, EG,
// Case #6: (cloned #3)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P0,
SH_TRI, COLOR0, EC, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #5
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_TRI, COLOR1, P0, EC, EG,
// Case #8: (cloned #1)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P0, P1,
SH_QUA, COLOR0, ED, P1, P2, EC,
SH_TRI, COLOR1, P3, ED, EC,
// Case #9: Unique case #6
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P0, EA, EC, P3,
SH_QUA, COLOR1, P0, P3, ED, EG,
// Case #10: (cloned #5)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EA,
// Case #11: Unique case #7
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_TRI, COLOR1, P0, ED, EG,
// Case #12: (cloned #3)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P0, P1,
SH_TRI, COLOR0, ED, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EB,
// Case #13: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, P4, P5, P6, EG,
SH_TRI, COLOR0, ED, P4, EG,
SH_QUA, COLOR1, EA, EB, P2, P3,
SH_QUA, COLOR1, EG, P0, EA, P3,
SH_TRI, COLOR1, ED, EG, P3,
// Case #14: (cloned #7)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_TRI, COLOR1, P1, ED, EA,
// Case #15: Unique case #8
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EG,
// Case #16: (cloned #1)
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_QUA, COLOR0, EE, P0, P1, P2,
SH_QUA, COLOR0, EE, P2, P3, ED,
SH_TRI, COLOR1, P4, EE, ED,
// Case #17: (cloned #9)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, EE, EG, P0,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #18: (cloned #9)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR1, P1, EB, ED, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #19: Unique case #9
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P0, P1, EB, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_TRI, COLOR1, P0, EE, EG,
// Case #20: (cloned #5)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_QUA, COLOR0, EE, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EB,
// Case #21: Unique case #10
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_TRI, COLOR1, P0, EE, EG,
// Case #22: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_TRI, COLOR1, P1, EE, EA,
// Case #23: Unique case #11
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, EE, EG,
// Case #24: (cloned #3)
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_QUA, COLOR0, EE, P0, P1, P2,
SH_TRI, COLOR0, EE, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EC,
// Case #25: (cloned #19)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EG,
SH_QUA, COLOR1, P3, EG, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #26: (cloned #11)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, P5, P6, P0, EA,
SH_TRI, COLOR0, EE, P5, EA,
SH_QUA, COLOR1, EB, EC, P3, P4,
SH_QUA, COLOR1, EA, P1, EB, P4,
SH_TRI, COLOR1, EE, EA, P4,
// Case #27: Unique case #12
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, P4,
SH_QUA, COLOR1, P0, P4, EE, EG,
// Case #28: (cloned #7)
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_QUA, COLOR0, EE, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_TRI, COLOR1, P2, EE, EB,
// Case #29: (cloned #23)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, EB, P2, P3, P4,
SH_QUA, COLOR1, P0, EA, EB, P4,
SH_QUA, COLOR1, EE, EG, P0, P4,
// Case #30: (cloned #15)
SH_QUA, COLOR0, EE, P5, P6, P0,
SH_TRI, COLOR0, EE, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #31: Unique case #13
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, EE,
SH_TRI, COLOR1, P0, EE, EG,
// Case #32: (cloned #1)
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_QUA, COLOR0, EF, P1, P2, P3,
SH_QUA, COLOR0, EF, P3, P4, EE,
SH_TRI, COLOR1, P5, EF, EE,
// Case #33: (cloned #5)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P0,
SH_QUA, COLOR1, P5, P0, EA, EE,
// Case #34: (cloned #9)
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, EF, EA, P1,
SH_QUA, COLOR1, P5, P1, EB, EE,
// Case #35: (cloned #11)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, P2, P3, P4, EE,
SH_TRI, COLOR0, EB, P2, EE,
SH_QUA, COLOR1, EF, EG, P0, P1,
SH_QUA, COLOR1, EE, P5, EF, P1,
SH_TRI, COLOR1, EB, EE, P1,
// Case #36: (cloned #9)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_TRI, COLOR0, EF, P1, EB,
SH_QUA, COLOR1, P2, EC, EE, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #37: (cloned #21)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P0,
SH_QUA, COLOR1, P5, P0, EA, EB,
SH_QUA, COLOR1, P5, EB, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #38: (cloned #19)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #39: (cloned #23)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, EG, P0, P1, P2,
SH_QUA, COLOR1, P5, EF, EG, P2,
SH_QUA, COLOR1, EC, EE, P5, P2,
// Case #40: (cloned #5)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_QUA, COLOR0, EF, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EC,
// Case #41: (cloned #21)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EG,
SH_QUA, COLOR1, P3, EG, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #42: (cloned #21)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #43: Unique case #14
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_QUA, COLOR1, P0, ED, EE, P5,
SH_QUA, COLOR1, P0, P5, EF, EG,
// Case #44: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_TRI, COLOR0, EF, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, EF,
SH_TRI, COLOR1, P2, EF, EB,
// Case #45: (cloned #43)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, EF,
SH_QUA, COLOR1, P2, EF, EG, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #46: (cloned #23)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #47: Unique case #15
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EE,
SH_QUA, COLOR1, P0, EE, P5, EF,
SH_TRI, COLOR1, P0, EF, EG,
// Case #48: (cloned #3)
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_QUA, COLOR0, EF, P1, P2, P3,
SH_TRI, COLOR0, EF, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, ED,
// Case #49: (cloned #11)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P0, EA,
SH_TRI, COLOR1, P4, EA, ED,
// Case #50: (cloned #19)
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EA,
SH_QUA, COLOR1, P4, EA, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #51: (cloned #27)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P0, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #52: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, P6, P0, P1, EB,
SH_TRI, COLOR0, EF, P6, EB,
SH_QUA, COLOR1, EC, ED, P4, P5,
SH_QUA, COLOR1, EB, P2, EC, P5,
SH_TRI, COLOR1, EF, EB, P5,
// Case #53: (cloned #43)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P0, EA,
SH_QUA, COLOR1, P4, EA, EB, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #54: (cloned #27)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #55: Unique case #16
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, P5, EF,
SH_TRI, COLOR1, P0, EF, EG,
// Case #56: (cloned #7)
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_QUA, COLOR0, EF, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_TRI, COLOR1, P3, EF, EC,
// Case #57: (cloned #23)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #58: (cloned #23)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, EC, P3, P4, P5,
SH_QUA, COLOR1, P1, EB, EC, P5,
SH_QUA, COLOR1, EF, EA, P1, P5,
// Case #59: (cloned #55)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P0,
SH_QUA, COLOR1, P3, P0, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #60: (cloned #15)
SH_QUA, COLOR0, EF, P6, P0, P1,
SH_TRI, COLOR0, EF, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #61: (cloned #47)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EG,
SH_QUA, COLOR1, P2, EG, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #62: (cloned #31)
SH_QUA, COLOR0, EF, P6, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #63: Unique case #17
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
SH_QUA, COLOR1, P0, P5, EF, EG,
// Case #64: (cloned #1)
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_QUA, COLOR0, EG, P2, P3, P4,
SH_QUA, COLOR0, EG, P4, P5, EF,
SH_TRI, COLOR1, P6, EG, EF,
// Case #65: (cloned #3)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_TRI, COLOR0, EA, P5, EF,
SH_QUA, COLOR1, P6, P0, EA, EF,
// Case #66: (cloned #5)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P6, EG, EA, P1,
SH_QUA, COLOR1, P6, P1, EB, EF,
// Case #67: (cloned #7)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, EB,
SH_TRI, COLOR1, P6, EB, EF,
// Case #68: (cloned #9)
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, EG, EB, P2,
SH_QUA, COLOR1, P6, P2, EC, EF,
// Case #69: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P0, EA, EB,
SH_QUA, COLOR1, P6, EB, P2, EC,
SH_TRI, COLOR1, P6, EC, EF,
// Case #70: (cloned #11)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, P3, P4, P5, EF,
SH_TRI, COLOR0, EC, P3, EF,
SH_QUA, COLOR1, EG, EA, P1, P2,
SH_QUA, COLOR1, EF, P6, EG, P2,
SH_TRI, COLOR1, EC, EF, P2,
// Case #71: (cloned #15)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, P2,
SH_QUA, COLOR1, P6, P2, EC, EF,
// Case #72: (cloned #9)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_TRI, COLOR0, EG, P2, EC,
SH_QUA, COLOR1, P3, ED, EF, P6,
SH_QUA, COLOR1, P3, P6, EG, EC,
// Case #73: (cloned #19)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P0, EA, EC,
SH_QUA, COLOR1, P6, EC, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #74: (cloned #21)
SH_TRI, COLOR0, EG, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, EG, EA, P1,
SH_QUA, COLOR1, P6, P1, EB, EC,
SH_QUA, COLOR1, P6, EC, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #75: (cloned #23)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, EB,
SH_QUA, COLOR1, P6, EB, EC, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #76: (cloned #19)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EF,
SH_QUA, COLOR1, P2, EF, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #77: (cloned #27)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P0, EA, EB,
SH_QUA, COLOR1, P6, EB, P2, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #78: (cloned #23)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, EA, P1, P2, P3,
SH_QUA, COLOR1, P6, EG, EA, P3,
SH_QUA, COLOR1, ED, EF, P6, P3,
// Case #79: (cloned #31)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, P2,
SH_QUA, COLOR1, P6, P2, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #80: (cloned #5)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_QUA, COLOR0, EG, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, EG, ED,
// Case #81: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, P1, P2, P3, ED,
SH_TRI, COLOR0, EA, P1, ED,
SH_QUA, COLOR1, EE, EF, P6, P0,
SH_QUA, COLOR1, ED, P4, EE, P0,
SH_TRI, COLOR1, EA, ED, P0,
// Case #82: (cloned #21)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, EG, EA,
SH_QUA, COLOR1, P4, EA, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #83: (cloned #23)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, EF, P6, P0, P1,
SH_QUA, COLOR1, P4, EE, EF, P1,
SH_QUA, COLOR1, EB, ED, P4, P1,
// Case #84: (cloned #21)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EF,
SH_QUA, COLOR1, P2, EF, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #85: (cloned #43)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P0, EA, EB,
SH_QUA, COLOR1, P6, EB, P2, EC,
SH_QUA, COLOR1, P6, EC, ED, P4,
SH_QUA, COLOR1, P6, P4, EE, EF,
// Case #86: (cloned #43)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_QUA, COLOR1, P1, EE, EF, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #87: (cloned #47)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, P2,
SH_QUA, COLOR1, P6, P2, EC, ED,
SH_QUA, COLOR1, P6, ED, P4, EE,
SH_TRI, COLOR1, P6, EE, EF,
// Case #88: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_TRI, COLOR0, EG, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, EG,
SH_TRI, COLOR1, P3, EG, EC,
// Case #89: (cloned #27)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #90: (cloned #43)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, EG,
SH_QUA, COLOR1, P3, EG, EA, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #91: (cloned #55)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, EB,
SH_QUA, COLOR1, P6, EB, EC, P3,
SH_QUA, COLOR1, P6, P3, P4, EE,
SH_TRI, COLOR1, P6, EE, EF,
// Case #92: (cloned #23)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, EG, EB,
// Case #93: (cloned #55)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #94: (cloned #47)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EF,
SH_QUA, COLOR1, P1, EF, P6, EG,
SH_TRI, COLOR1, P1, EG, EA,
// Case #95: (cloned #63)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P0, P1, P2,
SH_QUA, COLOR1, P6, P2, P3, P4,
SH_QUA, COLOR1, P6, P4, EE, EF,
// Case #96: (cloned #3)
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_QUA, COLOR0, EG, P2, P3, P4,
SH_TRI, COLOR0, EG, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EE,
// Case #97: (cloned #7)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, EA,
SH_TRI, COLOR1, P5, EA, EE,
// Case #98: (cloned #11)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EA,
SH_QUA, COLOR1, P5, EA, P1, EB,
SH_TRI, COLOR1, P5, EB, EE,
// Case #99: (cloned #15)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, P1,
SH_QUA, COLOR1, P5, P1, EB, EE,
// Case #100: (cloned #19)
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EB,
SH_QUA, COLOR1, P5, EB, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #101: (cloned #23)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, EA,
SH_QUA, COLOR1, P5, EA, EB, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #102: (cloned #27)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EA,
SH_QUA, COLOR1, P5, EA, P1, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #103: (cloned #31)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, P1,
SH_QUA, COLOR1, P5, P1, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #104: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, P0, P1, P2, EC,
SH_TRI, COLOR0, EG, P0, EC,
SH_QUA, COLOR1, ED, EE, P5, P6,
SH_QUA, COLOR1, EC, P3, ED, P6,
SH_TRI, COLOR1, EG, EC, P6,
// Case #105: (cloned #23)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, EE, P5, P6, P0,
SH_QUA, COLOR1, P3, ED, EE, P0,
SH_QUA, COLOR1, EA, EC, P3, P0,
// Case #106: (cloned #43)
SH_TRI, COLOR0, EG, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EA,
SH_QUA, COLOR1, P5, EA, P1, EB,
SH_QUA, COLOR1, P5, EB, EC, P3,
SH_QUA, COLOR1, P5, P3, ED, EE,
// Case #107: (cloned #47)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, P1,
SH_QUA, COLOR1, P5, P1, EB, EC,
SH_QUA, COLOR1, P5, EC, P3, ED,
SH_TRI, COLOR1, P5, ED, EE,
// Case #108: (cloned #27)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, P6,
SH_QUA, COLOR1, P2, P6, EG, EB,
// Case #109: (cloned #55)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, EA,
SH_QUA, COLOR1, P5, EA, EB, P2,
SH_QUA, COLOR1, P5, P2, P3, ED,
SH_TRI, COLOR1, P5, ED, EE,
// Case #110: (cloned #55)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, P6, EG,
SH_TRI, COLOR1, P1, EG, EA,
// Case #111: (cloned #63)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P0, P1,
SH_QUA, COLOR1, P5, P1, P2, P3,
SH_QUA, COLOR1, P5, P3, ED, EE,
// Case #112: (cloned #7)
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_QUA, COLOR0, EG, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_TRI, COLOR1, P4, EG, ED,
// Case #113: (cloned #15)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P0,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #114: (cloned #23)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EA, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #115: (cloned #31)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P0,
SH_QUA, COLOR1, P4, P0, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #116: (cloned #23)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, ED, P4, P5, P6,
SH_QUA, COLOR1, P2, EC, ED, P6,
SH_QUA, COLOR1, EG, EB, P2, P6,
// Case #117: (cloned #47)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P0,
SH_QUA, COLOR1, P4, P0, EA, EB,
SH_QUA, COLOR1, P4, EB, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #118: (cloned #55)
SH_TRI, COLOR0, EG, P0, EA,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EA, P1,
SH_QUA, COLOR1, P4, P1, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #119: (cloned #63)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P0,
SH_QUA, COLOR1, P4, P0, P1, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #120: (cloned #15)
SH_QUA, COLOR0, EG, P0, P1, P2,
SH_TRI, COLOR0, EG, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, EG, EC,
// Case #121: (cloned #31)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #122: (cloned #47)
SH_TRI, COLOR0, EG, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, EG, EA,
SH_QUA, COLOR1, P3, EA, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #123: (cloned #63)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P0, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #124: (cloned #31)
SH_QUA, COLOR0, EG, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #125: (cloned #63)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #126: (cloned #63)
SH_TRI, COLOR0, EG, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #127: Unique case #18
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
SH_TRI, COLOR1, P0, P5, P6,
//}
//{Poly8
// Case #0: Unique case #1
SH_QUA, COLOR0, P0, P1, P2, P3,
SH_QUA, COLOR0, P0, P3, P4, P5,
SH_QUA, COLOR0, P0, P5, P6, P7,
// Case #1: Unique case #2
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_QUA, COLOR0, EA, P5, P6, P7,
SH_TRI, COLOR0, EA, P7, EH,
SH_TRI, COLOR1, P0, EA, EH,
// Case #2: (cloned #1)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_QUA, COLOR0, EB, P6, P7, P0,
SH_TRI, COLOR0, EB, P0, EA,
SH_TRI, COLOR1, P1, EB, EA,
// Case #3: Unique case #3
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_QUA, COLOR0, EB, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, EH,
// Case #4: (cloned #1)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P7,
SH_QUA, COLOR0, EC, P7, P0, P1,
SH_TRI, COLOR0, EC, P1, EB,
SH_TRI, COLOR1, P2, EC, EB,
// Case #5: Unique case #4
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P7,
SH_TRI, COLOR0, EC, P7, EH,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, EH,
// Case #6: (cloned #3)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P7,
SH_QUA, COLOR0, EC, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EA,
// Case #7: Unique case #5
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, P7,
SH_TRI, COLOR0, EC, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_TRI, COLOR1, P0, EC, EH,
// Case #8: (cloned #1)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, P0,
SH_QUA, COLOR0, ED, P0, P1, P2,
SH_TRI, COLOR0, ED, P2, EC,
SH_TRI, COLOR1, P3, ED, EC,
// Case #9: Unique case #6
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, EH,
SH_QUA, COLOR1, P0, EA, EC, P3,
SH_QUA, COLOR1, P0, P3, ED, EH,
// Case #10: (cloned #5)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, P0,
SH_TRI, COLOR0, ED, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EA,
// Case #11: Unique case #7
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_TRI, COLOR1, P0, ED, EH,
// Case #12: (cloned #3)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, P0,
SH_QUA, COLOR0, ED, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EB,
// Case #13: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, P5, P6, P7, EH,
SH_QUA, COLOR0, ED, P4, P5, EH,
SH_QUA, COLOR1, EA, EB, P2, P3,
SH_QUA, COLOR1, EH, P0, EA, P3,
SH_TRI, COLOR1, ED, EH, P3,
// Case #14: (cloned #7)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, P0,
SH_TRI, COLOR0, ED, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_TRI, COLOR1, P1, ED, EA,
// Case #15: Unique case #8
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_QUA, COLOR0, ED, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EH,
// Case #16: (cloned #1)
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, P1,
SH_QUA, COLOR0, EE, P1, P2, P3,
SH_TRI, COLOR0, EE, P3, ED,
SH_TRI, COLOR1, P4, EE, ED,
// Case #17: Unique case #9
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, EA, ED, P4,
SH_QUA, COLOR1, P0, P4, EE, EH,
// Case #18: (cloned #9)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, EA,
SH_QUA, COLOR1, P1, EB, ED, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #19: Unique case #10
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_TRI, COLOR1, P0, EE, EH,
// Case #20: (cloned #5)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, P1,
SH_TRI, COLOR0, EE, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EB,
// Case #21: Unique case #11
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_TRI, COLOR1, P0, EE, EH,
// Case #22: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_TRI, COLOR1, P1, EE, EA,
// Case #23: Unique case #12
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, EE, EH,
// Case #24: (cloned #3)
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, P1,
SH_QUA, COLOR0, EE, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EC,
// Case #25: (cloned #19)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, P5, P6, P7, EH,
SH_TRI, COLOR0, EE, P5, EH,
SH_QUA, COLOR1, EA, EC, P3, P4,
SH_QUA, COLOR1, EH, P0, EA, P4,
SH_TRI, COLOR1, EE, EH, P4,
// Case #26: (cloned #11)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, P6, P7, P0, EA,
SH_QUA, COLOR0, EE, P5, P6, EA,
SH_QUA, COLOR1, EB, EC, P3, P4,
SH_QUA, COLOR1, EA, P1, EB, P4,
SH_TRI, COLOR1, EE, EA, P4,
// Case #27: Unique case #13
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, P4,
SH_QUA, COLOR1, P0, P4, EE, EH,
// Case #28: (cloned #7)
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, P1,
SH_TRI, COLOR0, EE, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_TRI, COLOR1, P2, EE, EB,
// Case #29: (cloned #23)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, P5, P6, P7, EH,
SH_TRI, COLOR0, EE, P5, EH,
SH_QUA, COLOR1, EB, P2, P3, P4,
SH_QUA, COLOR1, P0, EA, EB, P4,
SH_QUA, COLOR1, EE, EH, P0, P4,
// Case #30: (cloned #15)
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_QUA, COLOR0, EE, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EA,
// Case #31: Unique case #14
SH_QUA, COLOR0, EE, P5, P6, P7,
SH_TRI, COLOR0, EE, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, EE,
SH_TRI, COLOR1, P0, EE, EH,
// Case #32: (cloned #1)
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_QUA, COLOR0, EF, P2, P3, P4,
SH_TRI, COLOR0, EF, P4, EE,
SH_TRI, COLOR1, P5, EF, EE,
// Case #33: (cloned #9)
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, EF, EH, P0,
SH_QUA, COLOR1, P5, P0, EA, EE,
// Case #34: (cloned #17)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, EB, EE, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #35: (cloned #19)
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR0, P2, P3, P4, EE,
SH_TRI, COLOR0, EB, P2, EE,
SH_QUA, COLOR1, EF, EH, P0, P1,
SH_QUA, COLOR1, EE, P5, EF, P1,
SH_TRI, COLOR1, EB, EE, P1,
// Case #36: (cloned #9)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, EE, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #37: Unique case #15
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, EE,
SH_QUA, COLOR1, P0, EE, P5, EF,
SH_TRI, COLOR1, P0, EF, EH,
// Case #38: (cloned #19)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #39: Unique case #16
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, EE, P5,
SH_QUA, COLOR1, P0, P5, EF, EH,
// Case #40: (cloned #5)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_TRI, COLOR0, EF, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EC,
// Case #41: (cloned #37)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EH,
SH_QUA, COLOR1, P3, EH, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #42: (cloned #21)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #43: Unique case #17
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, ED,
SH_QUA, COLOR1, P0, ED, EE, P5,
SH_QUA, COLOR1, P0, P5, EF, EH,
// Case #44: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, EF,
SH_TRI, COLOR1, P2, EF, EB,
// Case #45: Unique case #18
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, P3, ED,
SH_QUA, COLOR1, P0, ED, EE, P5,
SH_QUA, COLOR1, P0, P5, EF, EH,
// Case #46: (cloned #23)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #47: Unique case #19
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EE,
SH_QUA, COLOR1, P0, EE, P5, EF,
SH_TRI, COLOR1, P0, EF, EH,
// Case #48: (cloned #3)
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_QUA, COLOR0, EF, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, ED,
// Case #49: (cloned #19)
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EH,
SH_QUA, COLOR1, P4, EH, P0, EA,
SH_TRI, COLOR1, P4, EA, ED,
// Case #50: (cloned #19)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, P6, P7, P0, EA,
SH_TRI, COLOR0, EF, P6, EA,
SH_QUA, COLOR1, EB, ED, P4, P5,
SH_QUA, COLOR1, EA, P1, EB, P5,
SH_TRI, COLOR1, EF, EA, P5,
// Case #51: Unique case #20
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, ED,
SH_QUA, COLOR1, P0, ED, P4, P5,
SH_QUA, COLOR1, P0, P5, EF, EH,
// Case #52: (cloned #11)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, P7, P0, P1, EB,
SH_QUA, COLOR0, EF, P6, P7, EB,
SH_QUA, COLOR1, EC, ED, P4, P5,
SH_QUA, COLOR1, EB, P2, EC, P5,
SH_TRI, COLOR1, EF, EB, P5,
// Case #53: (cloned #43)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, EC, ED, P4, P5,
SH_QUA, COLOR1, EB, P2, EC, P5,
SH_QUA, COLOR1, P0, EA, EB, P5,
SH_QUA, COLOR1, EF, EH, P0, P5,
// Case #54: (cloned #27)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, P5,
SH_QUA, COLOR1, P1, P5, EF, EA,
// Case #55: Unique case #21
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, P5, EF,
SH_TRI, COLOR1, P0, EF, EH,
// Case #56: (cloned #7)
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, P2,
SH_TRI, COLOR0, EF, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_TRI, COLOR1, P3, EF, EC,
// Case #57: (cloned #39)
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EH, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #58: (cloned #23)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, P6, P7, P0, EA,
SH_TRI, COLOR0, EF, P6, EA,
SH_QUA, COLOR1, EC, P3, P4, P5,
SH_QUA, COLOR1, P1, EB, EC, P5,
SH_QUA, COLOR1, EF, EA, P1, P5,
// Case #59: (cloned #55)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, EC, P3, P4, P5,
SH_QUA, COLOR1, P1, EB, EC, P5,
SH_QUA, COLOR1, EH, P0, P1, P5,
SH_TRI, COLOR1, EF, EH, P5,
// Case #60: (cloned #15)
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_QUA, COLOR0, EF, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EB,
// Case #61: (cloned #47)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, EA, EB, P2, P5,
SH_QUA, COLOR1, EH, P0, EA, P5,
SH_TRI, COLOR1, EF, EH, P5,
// Case #62: (cloned #31)
SH_QUA, COLOR0, EF, P6, P7, P0,
SH_TRI, COLOR0, EF, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, EF,
SH_TRI, COLOR1, P1, EF, EA,
// Case #63: Unique case #22
SH_QUA, COLOR0, EF, P6, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
SH_QUA, COLOR1, P0, P5, EF, EH,
// Case #64: (cloned #1)
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, P3,
SH_QUA, COLOR0, EG, P3, P4, P5,
SH_TRI, COLOR0, EG, P5, EF,
SH_TRI, COLOR1, P6, EG, EF,
// Case #65: (cloned #5)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_TRI, COLOR0, EA, P5, EF,
SH_QUA, COLOR1, P6, EG, EH, P0,
SH_QUA, COLOR1, P6, P0, EA, EF,
// Case #66: (cloned #9)
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P6, EG, EA, P1,
SH_QUA, COLOR1, P6, P1, EB, EF,
// Case #67: (cloned #11)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, P3, P4, P5, EF,
SH_QUA, COLOR0, EB, P2, P3, EF,
SH_QUA, COLOR1, EG, EH, P0, P1,
SH_QUA, COLOR1, EF, P6, EG, P1,
SH_TRI, COLOR1, EB, EF, P1,
// Case #68: (cloned #17)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, EC, EF, P6,
SH_QUA, COLOR1, P2, P6, EG, EB,
// Case #69: (cloned #21)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, EG, EH, P0,
SH_QUA, COLOR1, P6, P0, EA, EB,
SH_QUA, COLOR1, P6, EB, P2, EC,
SH_TRI, COLOR1, P6, EC, EF,
// Case #70: (cloned #19)
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR0, P3, P4, P5, EF,
SH_TRI, COLOR0, EC, P3, EF,
SH_QUA, COLOR1, EG, EA, P1, P2,
SH_QUA, COLOR1, EF, P6, EG, P2,
SH_TRI, COLOR1, EC, EF, P2,
// Case #71: (cloned #23)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, P3, P4, P5, EF,
SH_TRI, COLOR0, EC, P3, EF,
SH_QUA, COLOR1, EH, P0, P1, P2,
SH_QUA, COLOR1, P6, EG, EH, P2,
SH_QUA, COLOR1, EC, EF, P6, P2,
// Case #72: (cloned #9)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EF, P6,
SH_QUA, COLOR1, P3, P6, EG, EC,
// Case #73: (cloned #37)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, EG, EH, P0,
SH_QUA, COLOR1, P6, P0, EA, EC,
SH_QUA, COLOR1, P6, EC, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #74: (cloned #37)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EF,
SH_QUA, COLOR1, P1, EF, P6, EG,
SH_TRI, COLOR1, P1, EG, EA,
// Case #75: (cloned #45)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, EG, EH, P0,
SH_QUA, COLOR1, P6, P0, P1, EB,
SH_QUA, COLOR1, P6, EB, EC, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #76: (cloned #19)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EF,
SH_QUA, COLOR1, P2, EF, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #77: (cloned #43)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, EA, EB, P2, P3,
SH_QUA, COLOR1, EH, P0, EA, P3,
SH_QUA, COLOR1, P6, EG, EH, P3,
SH_QUA, COLOR1, ED, EF, P6, P3,
// Case #78: (cloned #39)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EF, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #79: (cloned #47)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, EG, EH, P0, P3,
SH_QUA, COLOR1, EF, P6, EG, P3,
SH_TRI, COLOR1, ED, EF, P3,
// Case #80: (cloned #5)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, P3,
SH_TRI, COLOR0, EG, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, EG, ED,
// Case #81: (cloned #21)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, EG, EH,
SH_QUA, COLOR1, P4, EH, P0, EA,
SH_TRI, COLOR1, P4, EA, ED,
// Case #82: (cloned #37)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, EG, EA,
SH_QUA, COLOR1, P4, EA, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #83: (cloned #43)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, EG, EH, P0, P1,
SH_QUA, COLOR1, EF, P6, EG, P1,
SH_QUA, COLOR1, P4, EE, EF, P1,
SH_QUA, COLOR1, EB, ED, P4, P1,
// Case #84: (cloned #21)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EF,
SH_QUA, COLOR1, P2, EF, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #85: Unique case #23
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, EA, EB, P2,
SH_QUA, COLOR1, P0, P2, EC, ED,
SH_QUA, COLOR1, P0, ED, P4, EE,
SH_QUA, COLOR1, P0, EE, EF, P6,
SH_QUA, COLOR1, P0, P6, EG, EH,
// Case #86: (cloned #43)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, EE,
SH_QUA, COLOR1, P1, EE, EF, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #87: Unique case #24
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, EE, EF,
SH_QUA, COLOR1, P0, EF, P6, EG,
SH_TRI, COLOR1, P0, EG, EH,
// Case #88: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, EG,
SH_TRI, COLOR1, P3, EG, EC,
// Case #89: (cloned #43)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, EG,
SH_QUA, COLOR1, P3, EG, EH, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #90: (cloned #45)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, P4, EE,
SH_QUA, COLOR1, P1, EE, EF, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #91: Unique case #25
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, EB, EC,
SH_QUA, COLOR1, P0, EC, P3, P4,
SH_QUA, COLOR1, P0, P4, EE, EF,
SH_QUA, COLOR1, P0, EF, P6, EG,
SH_TRI, COLOR1, P0, EG, EH,
// Case #92: (cloned #23)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, EG, EB,
// Case #93: (cloned #87)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, EG, EH,
SH_QUA, COLOR1, P2, EH, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #94: (cloned #47)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EF,
SH_QUA, COLOR1, P1, EF, P6, EG,
SH_TRI, COLOR1, P1, EG, EA,
// Case #95: Unique case #26
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, EE,
SH_QUA, COLOR1, P0, EE, EF, P6,
SH_QUA, COLOR1, P0, P6, EG, EH,
// Case #96: (cloned #3)
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, P3,
SH_QUA, COLOR0, EG, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EE,
// Case #97: (cloned #11)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EH,
SH_QUA, COLOR1, P5, EH, P0, EA,
SH_TRI, COLOR1, P5, EA, EE,
// Case #98: (cloned #19)
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EA,
SH_QUA, COLOR1, P5, EA, P1, EB,
SH_TRI, COLOR1, P5, EB, EE,
// Case #99: (cloned #27)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EH,
SH_QUA, COLOR1, P5, EH, P0, P1,
SH_QUA, COLOR1, P5, P1, EB, EE,
// Case #100: (cloned #19)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, P7, P0, P1, EB,
SH_TRI, COLOR0, EG, P7, EB,
SH_QUA, COLOR1, EC, EE, P5, P6,
SH_QUA, COLOR1, EB, P2, EC, P6,
SH_TRI, COLOR1, EG, EB, P6,
// Case #101: (cloned #43)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EH,
SH_QUA, COLOR1, P5, EH, P0, EA,
SH_QUA, COLOR1, P5, EA, EB, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #102: (cloned #51)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, EE,
SH_QUA, COLOR1, P1, EE, P5, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #103: (cloned #55)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, EH, P0, P1, P2,
SH_QUA, COLOR1, P6, EG, EH, P2,
SH_QUA, COLOR1, EE, P5, P6, P2,
SH_TRI, COLOR1, EC, EE, P2,
// Case #104: (cloned #11)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, P0, P1, P2, EC,
SH_QUA, COLOR0, EG, P7, P0, EC,
SH_QUA, COLOR1, ED, EE, P5, P6,
SH_QUA, COLOR1, EC, P3, ED, P6,
SH_TRI, COLOR1, EG, EC, P6,
// Case #105: (cloned #45)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, P6, EG,
SH_QUA, COLOR1, P3, EG, EH, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #106: (cloned #43)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, ED, EE, P5, P6,
SH_QUA, COLOR1, EC, P3, ED, P6,
SH_QUA, COLOR1, P1, EB, EC, P6,
SH_QUA, COLOR1, EG, EA, P1, P6,
// Case #107: (cloned #91)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, EG, EH,
SH_QUA, COLOR1, P5, EH, P0, P1,
SH_QUA, COLOR1, P5, P1, EB, EC,
SH_QUA, COLOR1, P5, EC, P3, ED,
SH_TRI, COLOR1, P5, ED, EE,
// Case #108: (cloned #27)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, P6,
SH_QUA, COLOR1, P2, P6, EG, EB,
// Case #109: (cloned #91)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, P6,
SH_QUA, COLOR1, P2, P6, EG, EH,
SH_QUA, COLOR1, P2, EH, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #110: (cloned #55)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, P6, EG,
SH_TRI, COLOR1, P1, EG, EA,
// Case #111: Unique case #27
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, ED, EE,
SH_QUA, COLOR1, P0, EE, P5, P6,
SH_QUA, COLOR1, P0, P6, EG, EH,
// Case #112: (cloned #7)
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, P3,
SH_TRI, COLOR0, EG, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_TRI, COLOR1, P4, EG, ED,
// Case #113: (cloned #23)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EH, P0,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #114: (cloned #39)
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EA, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #115: (cloned #55)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EH, P0,
SH_QUA, COLOR1, P4, P0, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #116: (cloned #23)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, P7, P0, P1, EB,
SH_TRI, COLOR0, EG, P7, EB,
SH_QUA, COLOR1, ED, P4, P5, P6,
SH_QUA, COLOR1, P2, EC, ED, P6,
SH_QUA, COLOR1, EG, EB, P2, P6,
// Case #117: (cloned #87)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, EG,
SH_QUA, COLOR1, P4, EG, EH, P0,
SH_QUA, COLOR1, P4, P0, EA, EB,
SH_QUA, COLOR1, P4, EB, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #118: (cloned #55)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, ED, P4, P5, P6,
SH_QUA, COLOR1, P2, EC, ED, P6,
SH_QUA, COLOR1, EA, P1, P2, P6,
SH_TRI, COLOR1, EG, EA, P6,
// Case #119: Unique case #28
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, EC,
SH_QUA, COLOR1, P0, EC, ED, P4,
SH_QUA, COLOR1, P0, P4, P5, P6,
SH_QUA, COLOR1, P0, P6, EG, EH,
// Case #120: (cloned #15)
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_QUA, COLOR0, EG, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, EG, EC,
// Case #121: (cloned #47)
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, EG, EH,
SH_QUA, COLOR1, P3, EH, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #122: (cloned #47)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, EB, EC, P3, P6,
SH_QUA, COLOR1, EA, P1, EB, P6,
SH_TRI, COLOR1, EG, EA, P6,
// Case #123: (cloned #111)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, EG, EH,
SH_QUA, COLOR1, P3, EH, P0, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #124: (cloned #31)
SH_QUA, COLOR0, EG, P7, P0, P1,
SH_TRI, COLOR0, EG, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, EG,
SH_TRI, COLOR1, P2, EG, EB,
// Case #125: (cloned #95)
SH_TRI, COLOR0, EG, P7, EH,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, EG,
SH_QUA, COLOR1, P2, EG, EH, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #126: (cloned #63)
SH_QUA, COLOR0, EG, P7, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, P6,
SH_QUA, COLOR1, P1, P6, EG, EA,
// Case #127: Unique case #29
SH_TRI, COLOR0, EG, P7, EH,
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
SH_QUA, COLOR1, P0, P5, P6, EG,
SH_TRI, COLOR1, P0, EG, EH,
// Case #128: (cloned #1)
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, P4,
SH_QUA, COLOR0, EH, P4, P5, P6,
SH_TRI, COLOR0, EH, P6, EG,
SH_TRI, COLOR1, P7, EH, EG,
// Case #129: (cloned #3)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_QUA, COLOR0, EA, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EG,
// Case #130: (cloned #5)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_TRI, COLOR0, EB, P6, EG,
SH_QUA, COLOR1, P7, EH, EA, P1,
SH_QUA, COLOR1, P7, P1, EB, EG,
// Case #131: (cloned #7)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, P6,
SH_TRI, COLOR0, EB, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, EB,
SH_TRI, COLOR1, P7, EB, EG,
// Case #132: (cloned #9)
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, EG,
SH_QUA, COLOR1, P7, EH, EB, P2,
SH_QUA, COLOR1, P7, P2, EC, EG,
// Case #133: (cloned #11)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EB,
SH_QUA, COLOR1, P7, EB, P2, EC,
SH_TRI, COLOR1, P7, EC, EG,
// Case #134: (cloned #11)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, P4, P5, P6, EG,
SH_QUA, COLOR0, EC, P3, P4, EG,
SH_QUA, COLOR1, EH, EA, P1, P2,
SH_QUA, COLOR1, EG, P7, EH, P2,
SH_TRI, COLOR1, EC, EG, P2,
// Case #135: (cloned #15)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_QUA, COLOR0, EC, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, EC, EG,
// Case #136: (cloned #17)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, ED, EG, P7,
SH_QUA, COLOR1, P3, P7, EH, EC,
// Case #137: (cloned #19)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EC,
SH_QUA, COLOR1, P7, EC, P3, ED,
SH_TRI, COLOR1, P7, ED, EG,
// Case #138: (cloned #21)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P7, EH, EA, P1,
SH_QUA, COLOR1, P7, P1, EB, EC,
SH_QUA, COLOR1, P7, EC, P3, ED,
SH_TRI, COLOR1, P7, ED, EG,
// Case #139: (cloned #23)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, EB,
SH_QUA, COLOR1, P7, EB, EC, P3,
SH_QUA, COLOR1, P7, P3, ED, EG,
// Case #140: (cloned #19)
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR0, P4, P5, P6, EG,
SH_TRI, COLOR0, ED, P4, EG,
SH_QUA, COLOR1, EH, EB, P2, P3,
SH_QUA, COLOR1, EG, P7, EH, P3,
SH_TRI, COLOR1, ED, EG, P3,
// Case #141: (cloned #27)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EB,
SH_QUA, COLOR1, P7, EB, P2, P3,
SH_QUA, COLOR1, P7, P3, ED, EG,
// Case #142: (cloned #23)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, P4, P5, P6, EG,
SH_TRI, COLOR0, ED, P4, EG,
SH_QUA, COLOR1, EA, P1, P2, P3,
SH_QUA, COLOR1, P7, EH, EA, P3,
SH_QUA, COLOR1, ED, EG, P7, P3,
// Case #143: (cloned #31)
SH_QUA, COLOR0, ED, P4, P5, P6,
SH_TRI, COLOR0, ED, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, P3, ED,
SH_TRI, COLOR1, P7, ED, EG,
// Case #144: (cloned #9)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EG, P7,
SH_QUA, COLOR1, P4, P7, EH, ED,
// Case #145: (cloned #19)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, P1, P2, P3, ED,
SH_TRI, COLOR0, EA, P1, ED,
SH_QUA, COLOR1, EE, EG, P7, P0,
SH_QUA, COLOR1, ED, P4, EE, P0,
SH_TRI, COLOR1, EA, ED, P0,
// Case #146: (cloned #37)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, EH, EA, P1,
SH_QUA, COLOR1, P7, P1, EB, ED,
SH_QUA, COLOR1, P7, ED, P4, EE,
SH_TRI, COLOR1, P7, EE, EG,
// Case #147: (cloned #39)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, EB,
SH_QUA, COLOR1, P7, EB, ED, P4,
SH_QUA, COLOR1, P7, P4, EE, EG,
// Case #148: (cloned #37)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, EE, EG,
SH_QUA, COLOR1, P2, EG, P7, EH,
SH_TRI, COLOR1, P2, EH, EB,
// Case #149: (cloned #43)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EB,
SH_QUA, COLOR1, P7, EB, P2, EC,
SH_QUA, COLOR1, P7, EC, ED, P4,
SH_QUA, COLOR1, P7, P4, EE, EG,
// Case #150: (cloned #45)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, EH, EA, P1,
SH_QUA, COLOR1, P7, P1, P2, EC,
SH_QUA, COLOR1, P7, EC, ED, P4,
SH_QUA, COLOR1, P7, P4, EE, EG,
// Case #151: (cloned #47)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, EC, ED,
SH_QUA, COLOR1, P7, ED, P4, EE,
SH_TRI, COLOR1, P7, EE, EG,
// Case #152: (cloned #19)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EG,
SH_QUA, COLOR1, P3, EG, P7, EH,
SH_TRI, COLOR1, P3, EH, EC,
// Case #153: (cloned #51)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EG,
SH_QUA, COLOR1, P3, EG, P7, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #154: (cloned #43)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, EB, EC, P3, P4,
SH_QUA, COLOR1, EA, P1, EB, P4,
SH_QUA, COLOR1, P7, EH, EA, P4,
SH_QUA, COLOR1, EE, EG, P7, P4,
// Case #155: (cloned #55)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, EB,
SH_QUA, COLOR1, P7, EB, EC, P3,
SH_QUA, COLOR1, P7, P3, P4, EE,
SH_TRI, COLOR1, P7, EE, EG,
// Case #156: (cloned #39)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EG, P7,
SH_QUA, COLOR1, P2, P7, EH, EB,
// Case #157: (cloned #55)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, EB, P2, P3, P4,
SH_QUA, COLOR1, P0, EA, EB, P4,
SH_QUA, COLOR1, EG, P7, P0, P4,
SH_TRI, COLOR1, EE, EG, P4,
// Case #158: (cloned #47)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, EH, EA, P1, P4,
SH_QUA, COLOR1, EG, P7, EH, P4,
SH_TRI, COLOR1, EE, EG, P4,
// Case #159: (cloned #63)
SH_QUA, COLOR0, EE, P5, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, P3, P4,
SH_QUA, COLOR1, P7, P4, EE, EG,
// Case #160: (cloned #5)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, P4,
SH_TRI, COLOR0, EH, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P7,
SH_QUA, COLOR1, P5, P7, EH, EE,
// Case #161: (cloned #11)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, P2, P3, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EE,
SH_QUA, COLOR1, EF, EG, P7, P0,
SH_QUA, COLOR1, EE, P5, EF, P0,
SH_TRI, COLOR1, EA, EE, P0,
// Case #162: (cloned #21)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P7,
SH_QUA, COLOR1, P5, P7, EH, EA,
SH_QUA, COLOR1, P5, EA, P1, EB,
SH_TRI, COLOR1, P5, EB, EE,
// Case #163: (cloned #23)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, P2, P3, P4, EE,
SH_TRI, COLOR0, EB, P2, EE,
SH_QUA, COLOR1, EG, P7, P0, P1,
SH_QUA, COLOR1, P5, EF, EG, P1,
SH_QUA, COLOR1, EB, EE, P5, P1,
// Case #164: (cloned #37)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P7,
SH_QUA, COLOR1, P5, P7, EH, EB,
SH_QUA, COLOR1, P5, EB, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #165: (cloned #45)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, EF, EG, P7,
SH_QUA, COLOR1, P5, P7, P0, EA,
SH_QUA, COLOR1, P5, EA, EB, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #166: (cloned #43)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, EH, EA, P1, P2,
SH_QUA, COLOR1, EG, P7, EH, P2,
SH_QUA, COLOR1, P5, EF, EG, P2,
SH_QUA, COLOR1, EC, EE, P5, P2,
// Case #167: (cloned #47)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, EF, EG, P7, P2,
SH_QUA, COLOR1, EE, P5, EF, P2,
SH_TRI, COLOR1, EC, EE, P2,
// Case #168: (cloned #21)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, ED, EE, P5,
SH_QUA, COLOR1, P3, P5, EF, EG,
SH_QUA, COLOR1, P3, EG, P7, EH,
SH_TRI, COLOR1, P3, EH, EC,
// Case #169: (cloned #43)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, EF, EG, P7, P0,
SH_QUA, COLOR1, EE, P5, EF, P0,
SH_QUA, COLOR1, P3, ED, EE, P0,
SH_QUA, COLOR1, EA, EC, P3, P0,
// Case #170: (cloned #85)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, EB, EC, P3,
SH_QUA, COLOR1, P1, P3, ED, EE,
SH_QUA, COLOR1, P1, EE, P5, EF,
SH_QUA, COLOR1, P1, EF, EG, P7,
SH_QUA, COLOR1, P1, P7, EH, EA,
// Case #171: (cloned #87)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, EB,
SH_QUA, COLOR1, P7, EB, EC, P3,
SH_QUA, COLOR1, P7, P3, ED, EE,
SH_QUA, COLOR1, P7, EE, P5, EF,
SH_TRI, COLOR1, P7, EF, EG,
// Case #172: (cloned #43)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EE,
SH_QUA, COLOR1, P2, EE, P5, EF,
SH_QUA, COLOR1, P2, EF, EG, P7,
SH_QUA, COLOR1, P2, P7, EH, EB,
// Case #173: (cloned #91)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P7, P0, EA, EB,
SH_QUA, COLOR1, P7, EB, P2, P3,
SH_QUA, COLOR1, P7, P3, ED, EE,
SH_QUA, COLOR1, P7, EE, P5, EF,
SH_TRI, COLOR1, P7, EF, EG,
// Case #174: (cloned #87)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, EF, EG,
SH_QUA, COLOR1, P1, EG, P7, EH,
SH_TRI, COLOR1, P1, EH, EA,
// Case #175: (cloned #95)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, P3, ED,
SH_QUA, COLOR1, P7, ED, EE, P5,
SH_QUA, COLOR1, P7, P5, EF, EG,
// Case #176: (cloned #11)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P7, EH,
SH_TRI, COLOR1, P4, EH, ED,
// Case #177: (cloned #27)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P7, P0,
SH_QUA, COLOR1, P4, P0, EA, ED,
// Case #178: (cloned #43)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P7, EH,
SH_QUA, COLOR1, P4, EH, EA, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #179: (cloned #55)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, EG, P7, P0, P1,
SH_QUA, COLOR1, P5, EF, EG, P1,
SH_QUA, COLOR1, ED, P4, P5, P1,
SH_TRI, COLOR1, EB, ED, P1,
// Case #180: (cloned #45)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, EC, ED, P4,
SH_QUA, COLOR1, P2, P4, P5, EF,
SH_QUA, COLOR1, P2, EF, EG, P7,
SH_QUA, COLOR1, P2, P7, EH, EB,
// Case #181: (cloned #91)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, EF, EG,
SH_QUA, COLOR1, P4, EG, P7, P0,
SH_QUA, COLOR1, P4, P0, EA, EB,
SH_QUA, COLOR1, P4, EB, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #182: (cloned #91)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, EC, ED,
SH_QUA, COLOR1, P1, ED, P4, P5,
SH_QUA, COLOR1, P1, P5, EF, EG,
SH_QUA, COLOR1, P1, EG, P7, EH,
SH_TRI, COLOR1, P1, EH, EA,
// Case #183: (cloned #111)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, EC, ED,
SH_QUA, COLOR1, P7, ED, P4, P5,
SH_QUA, COLOR1, P7, P5, EF, EG,
// Case #184: (cloned #23)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P7,
SH_QUA, COLOR1, P3, P7, EH, EC,
// Case #185: (cloned #55)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P7,
SH_QUA, COLOR1, P3, P7, P0, EA,
SH_TRI, COLOR1, P3, EA, EC,
// Case #186: (cloned #87)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P7,
SH_QUA, COLOR1, P3, P7, EH, EA,
SH_QUA, COLOR1, P3, EA, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #187: (cloned #119)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, EF,
SH_QUA, COLOR1, P3, EF, EG, P7,
SH_QUA, COLOR1, P3, P7, P0, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #188: (cloned #47)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EG,
SH_QUA, COLOR1, P2, EG, P7, EH,
SH_TRI, COLOR1, P2, EH, EB,
// Case #189: (cloned #111)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, EF, EG,
SH_QUA, COLOR1, P2, EG, P7, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #190: (cloned #95)
SH_TRI, COLOR0, EF, P6, EG,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, EF,
SH_QUA, COLOR1, P1, EF, EG, P7,
SH_QUA, COLOR1, P1, P7, EH, EA,
// Case #191: (cloned #127)
SH_TRI, COLOR0, EF, P6, EG,
SH_QUA, COLOR1, P7, P0, P1, P2,
SH_QUA, COLOR1, P7, P2, P3, P4,
SH_QUA, COLOR1, P7, P4, P5, EF,
SH_TRI, COLOR1, P7, EF, EG,
// Case #192: (cloned #3)
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, P4,
SH_QUA, COLOR0, EH, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EF,
// Case #193: (cloned #7)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, P5,
SH_TRI, COLOR0, EA, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, EA,
SH_TRI, COLOR1, P6, EA, EF,
// Case #194: (cloned #11)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EA,
SH_QUA, COLOR1, P6, EA, P1, EB,
SH_TRI, COLOR1, P6, EB, EF,
// Case #195: (cloned #15)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_QUA, COLOR0, EB, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, EB, EF,
// Case #196: (cloned #19)
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EB,
SH_QUA, COLOR1, P6, EB, P2, EC,
SH_TRI, COLOR1, P6, EC, EF,
// Case #197: (cloned #23)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, EA,
SH_QUA, COLOR1, P6, EA, EB, P2,
SH_QUA, COLOR1, P6, P2, EC, EF,
// Case #198: (cloned #27)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EA,
SH_QUA, COLOR1, P6, EA, P1, P2,
SH_QUA, COLOR1, P6, P2, EC, EF,
// Case #199: (cloned #31)
SH_QUA, COLOR0, EC, P3, P4, P5,
SH_TRI, COLOR0, EC, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, P2, EC,
SH_TRI, COLOR1, P6, EC, EF,
// Case #200: (cloned #19)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, P0, P1, P2, EC,
SH_TRI, COLOR0, EH, P0, EC,
SH_QUA, COLOR1, ED, EF, P6, P7,
SH_QUA, COLOR1, EC, P3, ED, P7,
SH_TRI, COLOR1, EH, EC, P7,
// Case #201: (cloned #39)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, EA,
SH_QUA, COLOR1, P6, EA, EC, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #202: (cloned #43)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EA,
SH_QUA, COLOR1, P6, EA, P1, EB,
SH_QUA, COLOR1, P6, EB, EC, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #203: (cloned #47)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, EB, EC,
SH_QUA, COLOR1, P6, EC, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #204: (cloned #51)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, ED, EF,
SH_QUA, COLOR1, P2, EF, P6, P7,
SH_QUA, COLOR1, P2, P7, EH, EB,
// Case #205: (cloned #55)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, EA,
SH_QUA, COLOR1, P6, EA, EB, P2,
SH_QUA, COLOR1, P6, P2, P3, ED,
SH_TRI, COLOR1, P6, ED, EF,
// Case #206: (cloned #55)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, EA, P1, P2, P3,
SH_QUA, COLOR1, P7, EH, EA, P3,
SH_QUA, COLOR1, EF, P6, P7, P3,
SH_TRI, COLOR1, ED, EF, P3,
// Case #207: (cloned #63)
SH_QUA, COLOR0, ED, P4, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, P2, P3,
SH_QUA, COLOR1, P6, P3, ED, EF,
// Case #208: (cloned #11)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, P1, P2, P3, ED,
SH_QUA, COLOR0, EH, P0, P1, ED,
SH_QUA, COLOR1, EE, EF, P6, P7,
SH_QUA, COLOR1, ED, P4, EE, P7,
SH_TRI, COLOR1, EH, ED, P7,
// Case #209: (cloned #23)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, P1, P2, P3, ED,
SH_TRI, COLOR0, EA, P1, ED,
SH_QUA, COLOR1, EF, P6, P7, P0,
SH_QUA, COLOR1, P4, EE, EF, P0,
SH_QUA, COLOR1, EA, ED, P4, P0,
// Case #210: (cloned #45)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, EE, EF, P6,
SH_QUA, COLOR1, P4, P6, P7, EH,
SH_QUA, COLOR1, P4, EH, EA, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #211: (cloned #47)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, EE, EF, P6, P1,
SH_QUA, COLOR1, ED, P4, EE, P1,
SH_TRI, COLOR1, EB, ED, P1,
// Case #212: (cloned #43)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, EE, EF, P6, P7,
SH_QUA, COLOR1, ED, P4, EE, P7,
SH_QUA, COLOR1, P2, EC, ED, P7,
SH_QUA, COLOR1, EH, EB, P2, P7,
// Case #213: (cloned #87)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, EA,
SH_QUA, COLOR1, P6, EA, EB, P2,
SH_QUA, COLOR1, P6, P2, EC, ED,
SH_QUA, COLOR1, P6, ED, P4, EE,
SH_TRI, COLOR1, P6, EE, EF,
// Case #214: (cloned #91)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P7, EH, EA,
SH_QUA, COLOR1, P6, EA, P1, P2,
SH_QUA, COLOR1, P6, P2, EC, ED,
SH_QUA, COLOR1, P6, ED, P4, EE,
SH_TRI, COLOR1, P6, EE, EF,
// Case #215: (cloned #95)
SH_TRI, COLOR0, EC, P3, ED,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, P2, EC,
SH_QUA, COLOR1, P6, EC, ED, P4,
SH_QUA, COLOR1, P6, P4, EE, EF,
// Case #216: (cloned #27)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, P7,
SH_QUA, COLOR1, P3, P7, EH, EC,
// Case #217: (cloned #55)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, EF, P6, P7, P0,
SH_QUA, COLOR1, P4, EE, EF, P0,
SH_QUA, COLOR1, EC, P3, P4, P0,
SH_TRI, COLOR1, EA, EC, P0,
// Case #218: (cloned #91)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, EE, EF,
SH_QUA, COLOR1, P3, EF, P6, P7,
SH_QUA, COLOR1, P3, P7, EH, EA,
SH_QUA, COLOR1, P3, EA, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #219: (cloned #111)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, EB, EC,
SH_QUA, COLOR1, P6, EC, P3, P4,
SH_QUA, COLOR1, P6, P4, EE, EF,
// Case #220: (cloned #55)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, P7, EH,
SH_TRI, COLOR1, P2, EH, EB,
// Case #221: (cloned #119)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, EE,
SH_QUA, COLOR1, P2, EE, EF, P6,
SH_QUA, COLOR1, P2, P6, P7, P0,
SH_QUA, COLOR1, P2, P0, EA, EB,
// Case #222: (cloned #111)
SH_TRI, COLOR0, EE, P5, EF,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, EE, EF,
SH_QUA, COLOR1, P1, EF, P6, P7,
SH_QUA, COLOR1, P1, P7, EH, EA,
// Case #223: (cloned #127)
SH_TRI, COLOR0, EE, P5, EF,
SH_QUA, COLOR1, P6, P7, P0, P1,
SH_QUA, COLOR1, P6, P1, P2, P3,
SH_QUA, COLOR1, P6, P3, P4, EE,
SH_TRI, COLOR1, P6, EE, EF,
// Case #224: (cloned #7)
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, P4,
SH_TRI, COLOR0, EH, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, EH,
SH_TRI, COLOR1, P5, EH, EE,
// Case #225: (cloned #15)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_QUA, COLOR0, EA, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, EA, EE,
// Case #226: (cloned #23)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, EH,
SH_QUA, COLOR1, P5, EH, EA, P1,
SH_QUA, COLOR1, P5, P1, EB, EE,
// Case #227: (cloned #31)
SH_QUA, COLOR0, EB, P2, P3, P4,
SH_TRI, COLOR0, EB, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, P1, EB,
SH_TRI, COLOR1, P5, EB, EE,
// Case #228: (cloned #39)
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, EH,
SH_QUA, COLOR1, P5, EH, EB, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #229: (cloned #47)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, EA, EB,
SH_QUA, COLOR1, P5, EB, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #230: (cloned #55)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, EH,
SH_QUA, COLOR1, P5, EH, EA, P1,
SH_QUA, COLOR1, P5, P1, P2, EC,
SH_TRI, COLOR1, P5, EC, EE,
// Case #231: (cloned #63)
SH_QUA, COLOR0, EC, P3, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, P1, P2,
SH_QUA, COLOR1, P5, P2, EC, EE,
// Case #232: (cloned #23)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, P0, P1, P2, EC,
SH_TRI, COLOR0, EH, P0, EC,
SH_QUA, COLOR1, EE, P5, P6, P7,
SH_QUA, COLOR1, P3, ED, EE, P7,
SH_QUA, COLOR1, EH, EC, P3, P7,
// Case #233: (cloned #47)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, ED, EE, P5, P0,
SH_QUA, COLOR1, EC, P3, ED, P0,
SH_TRI, COLOR1, EA, EC, P0,
// Case #234: (cloned #87)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, EH,
SH_QUA, COLOR1, P5, EH, EA, P1,
SH_QUA, COLOR1, P5, P1, EB, EC,
SH_QUA, COLOR1, P5, EC, P3, ED,
SH_TRI, COLOR1, P5, ED, EE,
// Case #235: (cloned #95)
SH_TRI, COLOR0, EB, P2, EC,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, P1, EB,
SH_QUA, COLOR1, P5, EB, EC, P3,
SH_QUA, COLOR1, P5, P3, ED, EE,
// Case #236: (cloned #55)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, EE, P5, P6, P7,
SH_QUA, COLOR1, P3, ED, EE, P7,
SH_QUA, COLOR1, EB, P2, P3, P7,
SH_TRI, COLOR1, EH, EB, P7,
// Case #237: (cloned #111)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, EA, EB,
SH_QUA, COLOR1, P5, EB, P2, P3,
SH_QUA, COLOR1, P5, P3, ED, EE,
// Case #238: (cloned #119)
SH_TRI, COLOR0, ED, P4, EE,
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, ED,
SH_QUA, COLOR1, P1, ED, EE, P5,
SH_QUA, COLOR1, P1, P5, P6, P7,
SH_QUA, COLOR1, P1, P7, EH, EA,
// Case #239: (cloned #127)
SH_TRI, COLOR0, ED, P4, EE,
SH_QUA, COLOR1, P5, P6, P7, P0,
SH_QUA, COLOR1, P5, P0, P1, P2,
SH_QUA, COLOR1, P5, P2, P3, ED,
SH_TRI, COLOR1, P5, ED, EE,
// Case #240: (cloned #15)
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_QUA, COLOR0, EH, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, EH, ED,
// Case #241: (cloned #31)
SH_QUA, COLOR0, EA, P1, P2, P3,
SH_TRI, COLOR0, EA, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, P0, EA,
SH_TRI, COLOR1, P4, EA, ED,
// Case #242: (cloned #47)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, EH, EA,
SH_QUA, COLOR1, P4, EA, P1, EB,
SH_TRI, COLOR1, P4, EB, ED,
// Case #243: (cloned #63)
SH_QUA, COLOR0, EB, P2, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, P0, P1,
SH_QUA, COLOR1, P4, P1, EB, ED,
// Case #244: (cloned #47)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, EC, ED, P4, P7,
SH_QUA, COLOR1, EB, P2, EC, P7,
SH_TRI, COLOR1, EH, EB, P7,
// Case #245: (cloned #95)
SH_TRI, COLOR0, EA, P1, EB,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, P0, EA,
SH_QUA, COLOR1, P4, EA, EB, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #246: (cloned #111)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, EH, EA,
SH_QUA, COLOR1, P4, EA, P1, P2,
SH_QUA, COLOR1, P4, P2, EC, ED,
// Case #247: (cloned #127)
SH_TRI, COLOR0, EC, P3, ED,
SH_QUA, COLOR1, P4, P5, P6, P7,
SH_QUA, COLOR1, P4, P7, P0, P1,
SH_QUA, COLOR1, P4, P1, P2, EC,
SH_TRI, COLOR1, P4, EC, ED,
// Case #248: (cloned #31)
SH_QUA, COLOR0, EH, P0, P1, P2,
SH_TRI, COLOR0, EH, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P7, EH,
SH_TRI, COLOR1, P3, EH, EC,
// Case #249: (cloned #63)
SH_QUA, COLOR0, EA, P1, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P7, P0,
SH_QUA, COLOR1, P3, P0, EA, EC,
// Case #250: (cloned #95)
SH_TRI, COLOR0, EH, P0, EA,
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P7, EH,
SH_QUA, COLOR1, P3, EH, EA, P1,
SH_QUA, COLOR1, P3, P1, EB, EC,
// Case #251: (cloned #127)
SH_TRI, COLOR0, EB, P2, EC,
SH_QUA, COLOR1, P3, P4, P5, P6,
SH_QUA, COLOR1, P3, P6, P7, P0,
SH_QUA, COLOR1, P3, P0, P1, EB,
SH_TRI, COLOR1, P3, EB, EC,
// Case #252: (cloned #63)
SH_QUA, COLOR0, EH, P0, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, P7,
SH_QUA, COLOR1, P2, P7, EH, EB,
// Case #253: (cloned #127)
SH_TRI, COLOR0, EA, P1, EB,
SH_QUA, COLOR1, P2, P3, P4, P5,
SH_QUA, COLOR1, P2, P5, P6, P7,
SH_QUA, COLOR1, P2, P7, P0, EA,
SH_TRI, COLOR1, P2, EA, EB,
// Case #254: (cloned #127)
SH_TRI, COLOR0, EH, P0, EA,
SH_QUA, COLOR1, P1, P2, P3, P4,
SH_QUA, COLOR1, P1, P4, P5, P6,
SH_QUA, COLOR1, P1, P6, P7, EH,
SH_TRI, COLOR1, P1, EH, EA,
// Case #255: Unique case #30
SH_QUA, COLOR1, P0, P1, P2, P3,
SH_QUA, COLOR1, P0, P3, P4, P5,
SH_QUA, COLOR1, P0, P5, P6, P7
//}
};
enum {
MIR_TABLES_DATA_SIZE = sizeof(MIRInstructions) / sizeof(MIRInstructions[0]),
MIR_TABLES_IND_SIZE = sizeof(MIRTableIndicies) / sizeof(MIRTableIndicies[0]),
MIR_TABLES_SHAPE_SIZE = sizeof(MIRShapeCount) / sizeof(MIRShapeCount[0])
};

VISKORES_STATIC_CONSTEXPR_ARRAY viskores::UInt8 MIRCellEdges[456] = {
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_EMPTY_CELL
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_VERTEX
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLY_VERTEX
0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_LINE
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLY_LINE
0, 1, 1, 2, 2, 0, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_TRIANGLE
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_TRIANGLE_STRIP
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLYGON
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_PIXEL
0, 1, 1, 2, 3, 2, 0, 3, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_QUAD
0, 1, 1, 2, 2, 0, 0, 3, 1, 3, 2, 3,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_TETRA
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_VOXEL
0, 1, 1, 2, 3, 2, 0, 3, 4, 5, 5, 6,
7, 6, 4, 7, 0, 4, 1, 5, 3, 7, 2, 6, // viskores::CELL_SHAPE_HEXAHEDRON
0, 1, 1, 2, 2, 0, 3, 4, 4, 5, 5, 3,
0, 3, 1, 4, 2, 5, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_WEDGE
0, 1, 1, 2, 2, 3, 3, 0, 0, 4, 1, 4,
2, 4, 3, 4, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_PYRAMID
0, 1, 1, 2, 2, 3, 3, 4, 4, 0, 255, 255,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLYGON (5 points)
0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 0,
255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLYGON (6 points)
0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6,
6, 0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // viskores::CELL_SHAPE_POLYGON (7 points)
0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6,
6, 7, 7, 0, 255, 255, 255, 255, 255, 255, 255, 255 // viskores::CELL_SHAPE_POLYGON (8 points)
};

// clang-format on

class MIRTables : public viskores::cont::ExecutionObjectBase
{
public:
  using EdgeVec = viskores::IdComponent2;
  template <typename DeviceAdapter>
  class MIRDevicePortal
  {
  public:
    VISKORES_EXEC viskores::Id GetNumberOfIndices(viskores::Id shape) const
    {
      int IndLookup[viskores::NUMBER_OF_CELL_SHAPES] = {
        0,
        0, //All unimplemented besides quad and tris
        0, // Poly vert
        0, // line
        0, // poly line
        3, // tri
        0, // tri strip
        0, // poly
        0, // pixe
        4, // quad
        4, // tet
        0, // voxel
        8, // hex
        6, // wedge
        5  // pyr
      };
      // Note the unimplemented cases are because the MIR algorithm will never call for a shape of that style
      // meaning, no case will call for a poly or a tri-strip. Hence getting a 0 is an error!
      return IndLookup[shape];
    }

    VISKORES_EXEC viskores::Id GetCaseIndex(viskores::Id shape,
                                            viskores::Id caseId,
                                            viskores::IdComponent numPoints) const
    {
      int CellIndexLookup[viskores::NUMBER_OF_CELL_SHAPES] = {
        // This converts the shape id to clipTableIndicies (above) case
        -1,  // Empty
        0,   // Vertex
        -1,  // Poly vertex (omitted?)
        2,   // Line
        -1,  // Poly line
        6,   // Triangle
        -1,  // Tri strip
        -1,  // Polygon? (Maybe override it?)
        14,  // Pixel...actually we have that don't we?
        30,  // Quad
        46,  // Tet
        62,  // Voxel
        318, // Hex
        574, // Wedge
        638  // Pyramid
      };
      viskores::Id index = CellIndexLookup[shape];
      if (shape == 7)
      {
        if (numPoints == 5)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 670);
        }
        else if (numPoints == 6)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 702);
        }
        else if (numPoints == 7)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 766);
        }
        else if (numPoints == 8)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 894);
        }
        else if (numPoints == 3)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 46);
        }
        else if (numPoints == 4)
        {
          return this->MIRTablesIndicesPortal.Get(caseId + 30);
        }
        else
        {
          return static_cast<viskores::Id>(-1);
        }
      }
      return this->MIRTablesIndicesPortal.Get(index + caseId);
    }

    VISKORES_EXEC viskores::UInt8 ValueAt(viskores::Id idx) const
    {
      return this->MIRTablesDataPortal.Get(idx);
    }

    VISKORES_EXEC viskores::Id GetNumberOfShapes(viskores::Id shape,
                                                 viskores::Id caseId,
                                                 viskores::IdComponent numPoints) const
    {
      int CellIndexLookup[viskores::NUMBER_OF_CELL_SHAPES] = {
        // This converts the shape id to clipTableIndicies (above) case
        -1,  // Empty
        0,   // Vertex
        -1,  // Poly vertex (omitted?)
        2,   // Line
        -1,  // Poly line
        6,   // Triangle
        -1,  // Tri strip
        -1,  // Polygon? (Maybe override it?)
        14,  // Pixel...actually we have that don't we?
        30,  // Quad
        46,  // Tet
        62,  // Voxel
        318, // Hex
        574, // Wedge
        638  // Pyramid
      };
      if (shape == 7)
      {
        if (numPoints == 5)
        {
          return this->MIRTableSizesPortal.Get(caseId + 670);
        }
        else if (numPoints == 6)
        {
          return this->MIRTableSizesPortal.Get(caseId + 702);
        }
        else if (numPoints == 7)
        {
          return this->MIRTableSizesPortal.Get(caseId + 766);
        }
        else if (numPoints == 8)
        {
          return this->MIRTableSizesPortal.Get(caseId + 894);
        }
        else if (numPoints == 3)
        {
          return this->MIRTableSizesPortal.Get(caseId + 46);
        }
        else if (numPoints == 4)
        {
          return this->MIRTableSizesPortal.Get(caseId + 30);
        }
        else
        {
          return -1;
        }
      }
      viskores::Id index = CellIndexLookup[shape];
      return this->MIRTableSizesPortal.Get(index + caseId);
    }

    VISKORES_EXEC EdgeVec GetEdge(viskores::Id shape,
                                  viskores::Id edgeId,
                                  viskores::IdComponent numPoints) const
    {
      viskores::Id index = ((shape * 12) + edgeId) * 2;
      if (shape == 7)
      {
        if (numPoints == 4)
        {
          index = (12 * 9 + edgeId) * 2;
        }
        else if (numPoints == 3)
        {
          index = (12 * 5 + edgeId) * 2;
        }
        else
        {
          index = 372 + ((numPoints - 5) * 12 + edgeId) * 2;
        }
      }
      else
      {
        (void)index;
      }

      viskores::IdComponent v1 = this->CellEdgesPortal.Get(index);
      viskores::IdComponent v2 = this->CellEdgesPortal.Get(index + 1);
      return EdgeVec(v1, v2);
    }

    VISKORES_EXEC viskores::IdComponent GetNumberOfFaces(viskores::Id shape) const
    {
      viskores::IdComponent FacesLookup[viskores::NUMBER_OF_CELL_SHAPES] = {
        0,  // Empty
        0,  // Vertex
        0,  // Poly vertex (omitted?)
        0,  // Line
        0,  // Poly line
        0,  // Triangle
        0,  // Tri strip
        0,  // Polygon? (Maybe override it?)
        0,  // Pixel...actually we have that don't we?
        0,  // Quad
        4,  // Tet
        0,  // Voxel
        12, // Hex
        8,  // Wedge
        6   // Pyramid
      };
      return FacesLookup[shape];
    }

  private:
    typename viskores::cont::ArrayHandle<viskores::UInt8>::ReadPortalType MIRTablesDataPortal;
    typename viskores::cont::ArrayHandle<viskores::UInt16>::ReadPortalType MIRTablesIndicesPortal;
    typename viskores::cont::ArrayHandle<viskores::UInt8>::ReadPortalType CellEdgesPortal;
    typename viskores::cont::ArrayHandle<viskores::UInt8>::ReadPortalType MIRTableSizesPortal;
    friend class MIRTables;
  };
  MIRTables()
    : MIRTablesDataArray(viskores::cont::make_ArrayHandle(MIRInstructions,
                                                          MIR_TABLES_DATA_SIZE,
                                                          viskores::CopyFlag::Off))
    , MIRTablesIndicesArray(viskores::cont::make_ArrayHandle(MIRTableIndicies,
                                                             MIR_TABLES_IND_SIZE,
                                                             viskores::CopyFlag::Off))
    , MIREdgesArray(viskores::cont::make_ArrayHandle(MIRCellEdges, 456, viskores::CopyFlag::Off))
    , MIRSizeArray(viskores::cont::make_ArrayHandle(MIRShapeCount,
                                                    MIR_TABLES_SHAPE_SIZE,
                                                    viskores::CopyFlag::Off))
  {
  }

  template <typename DeviceAdapter>
  MIRDevicePortal<DeviceAdapter> PrepareForExecution(DeviceAdapter, viskores::cont::Token& token)
  {
    MIRDevicePortal<DeviceAdapter> portal;
    portal.MIRTablesDataPortal = this->MIRTablesDataArray.PrepareForInput(DeviceAdapter(), token);
    portal.MIRTablesIndicesPortal =
      this->MIRTablesIndicesArray.PrepareForInput(DeviceAdapter(), token);
    portal.CellEdgesPortal = this->MIREdgesArray.PrepareForInput(DeviceAdapter(), token);
    portal.MIRTableSizesPortal = this->MIRSizeArray.PrepareForInput(DeviceAdapter(), token);
    return portal;
  }

private:
  viskores::cont::ArrayHandle<viskores::UInt8> MIRTablesDataArray;
  viskores::cont::ArrayHandle<viskores::UInt16> MIRTablesIndicesArray;
  viskores::cont::ArrayHandle<viskores::UInt8> MIREdgesArray;
  viskores::cont::ArrayHandle<viskores::UInt8> MIRSizeArray;
};

} // namespace MIRCases
} // namespace worklet
} // namespace viskores

#endif
