# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vpype',
 'vpype_cli',
 'vpype_viewer',
 'vpype_viewer.image',
 'vpype_viewer.qtviewer']

package_data = \
{'': ['*'],
 'vpype': ['fonts/*'],
 'vpype_viewer': ['resources/*', 'shaders/*'],
 'vpype_viewer.qtviewer': ['resources/*']}

install_requires = \
['Shapely[vectorized]==1.8.0',
 'asteval>=0.9.26,<0.10.0',
 'cachetools>=4.2.2',
 'click>=8.0.1,<8.1.0',
 'multiprocess>=0.70.11,<0.71.0',
 'numpy>=1.20',
 'pnoise>=0.1.0,<0.2.0',
 'scipy>=1.6,<2.0',
 'setuptools>=51.0.0,<52.0.0',
 'svgelements==1.6.10',
 'svgwrite>=1.4,<1.5',
 'tomli>=2.0.0,<3.0.0']

extras_require = \
{'all': ['matplotlib>=3.3.2,<3.6.0',
         'glcontext>=2.3.2',
         'moderngl>=5.6.2,<6.0.0',
         'Pillow>=9.0.0,!=9.0.1',
         'PySide2>=5.15.2,<6.0.0'],
 'docs': ['Sphinx>=4.0.1,<5.0.0',
          'sphinx-click>=2.5.0',
          'sphinx-autodoc-typehints>=1.12.0',
          'sphinx-rtd-theme>=1.0.0',
          'recommonmark>=0.6,<0.8']}

entry_points = \
{'console_scripts': ['vpype = vpype_cli.cli:cli']}

setup_kwargs = {
    'name': 'vpype',
    'version': '1.9.0',
    'description': 'The Swiss Army knife of vector graphics for pen plotters',
    'long_description': '![banner](https://github.com/abey79/vpype/raw/master/docs/images/banner_ua.png)\n\n\n# _vpype_\n\n[![PyPI](https://img.shields.io/pypi/v/vpype?label=PyPI&logo=pypi)](https://pypi.org/project/vpype/)\n[![python](https://img.shields.io/github/languages/top/abey79/vpype)](https://www.python.org)\n[![Downloads](https://pepy.tech/badge/vpype)](https://pepy.tech/project/vpype)\n[![license](https://img.shields.io/pypi/l/vpype)](https://vpype.readthedocs.io/en/latest/license.html)\n![Test](https://img.shields.io/github/workflow/status/abey79/vpype/Lint%20and%20Tests?label=Tests&logo=github)\n[![codecov](https://codecov.io/gh/abey79/vpype/branch/master/graph/badge.svg?token=CE7FD9D6XO)](https://codecov.io/gh/abey79/vpype)\n[![Sonarcloud Status](https://sonarcloud.io/api/project_badges/measure?project=abey79_vpype&metric=alert_status)](https://sonarcloud.io/dashboard?id=abey79_vpype)\n[![Documentation Status](https://img.shields.io/readthedocs/vpype?label=Read%20the%20Docs&logo=read-the-docs)](https://vpype.readthedocs.io/en/latest/?badge=latest)\n[![Code style: Black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n_vpype_ is the Swiss-Army-knife command-line tool for plotter vector graphics.\n\n#### Contents\n\n* [What _vpype_ is?](#what-vpype-is)\n* [How does it work?](#how-does-it-work)\n* [Examples](#examples)\n* [What _vpype_ isn\'t?](#what-vpype-isnt)\n* [Installation](#installation)\n* [Documentation](#documentation)\n* [Feature overview](#feature-overview)\n    * [General](#general)\n    * [Input/Output](#inputoutput)\n    * [Layout and transforms](#layout-and-transforms)\n    * [Plotting optimization](#plotting-optimization)\n    * [Generation](#generation)\n    * [Extensibility and API](#extensibility-and-api)\n* [Plug-ins](#plug-ins)\n* [Contributing](#contributing)\n* [License](#license)\n\n\n## What _vpype_ is?\n\n_vpype_ is the Swiss-Army-knife command-line tool for plotter vector graphics. Here is what it can do:\n - **layout** existing vector files with precise control on position, scale and page format;\n - **optimize** existing SVG files for faster and cleaner plots;\n - create **HPGL output** for vintage plotters;\n - create **generative artwork** from scratch through built-in commands or [plug-ins](#plug-ins);\n - create, modify and process **multi-layer vector files** for multi-colour plots;\n - and much more...\n \n_vpype_ is highly **extensible** through plug-ins that can greatly extend its capabilities. For example, plug-ins\nalready exists for plotting [pixel art](https://github.com/abey79/vpype-pixelart),\n[half-toning with hatches](https://github.com/abey79/hatched), and much more. See below for a\n[list of existing plug-ins](#plug-ins).\n\n_vpype_ is also a [well documented](https://vpype.readthedocs.io/en/latest/api.html) **Python library**\nuseful to create generative art and tools for plotters. It includes data structures, utility and I/O functions, as well as\na hardware-accelerated flexible viewer for vector graphics. For example, the plotter generative art environment\n[vsketch](https://github.com/abey79/vsketch) is built upon _vpype_.\n\nCheck the [documentation](https://vpype.readthedocs.io/en/latest/) for a more thorough introduction to _vpype_.\n\n## How does it work?\n\n_vpype_ works by building so-called _pipelines_ of _commands_, where each command\'s output is fed to the next command\'s input.\nSome commands load geometries into the pipeline (e.g. the [`read`](https://vpype.readthedocs.io/en/latest/reference.html#read)\ncommand which loads geometries from a SVG file). Other commands modify these geometries, e.g. by cropping\nthem ([`crop`](https://vpype.readthedocs.io/en/latest/reference.html#crop)) or reordering them to minimize pen-up\ntravels ([`linesort`](https://vpype.readthedocs.io/en/latest/reference.html#linesort)). Finally, some other commands\njust read the geometries in the pipeline for display purposes ([`show`](https://vpype.readthedocs.io/en/latest/reference.html#show))\nor output to file ([`write`](https://vpype.readthedocs.io/en/latest/reference.html#write)).\n\nPipeline are defined using the _vpype_\'s CLI (command-line interface) in a terminal by typing `vpype` followed by the\nlist of commands, each with their optional parameters and their arguments:\n\n![command line](https://github.com/abey79/vpype/raw/master/docs/images/command_line.svg)\n\nThis pipeline uses five commands (in bold):\n- [`read`](https://vpype.readthedocs.io/en/latest/reference.html#read) loads geometries from a SVG file.\n- [`linemerge`](https://vpype.readthedocs.io/en/latest/reference.html#linemerge) merges paths whose extremities are close to each other (within the provided tolerance).\n- [`linesort`](https://vpype.readthedocs.io/en/latest/reference.html#linesort) reorder paths such as to minimise the pen-up travel.\n- [`crop`](https://vpype.readthedocs.io/en/latest/reference.html#crop), well, crops.\n- [`write`](https://vpype.readthedocs.io/en/latest/reference.html#write) export the resulting geometries to a SVG file.\n\nThere are many more commands available in *vpype*, see the [overview](#feature-overview) below.\n\nSome commands have arguments, which are always required (in italic). For example, a file path must be provided to the\n[`read`](https://vpype.readthedocs.io/en/latest/reference.html#read) command and dimensions must be provided to the [`crop`](https://vpype.readthedocs.io/en/latest/reference.html#crop) commands. A command may also have options which are, well,\noptional. In this example, `--page-size a4` means that the [`write`](https://vpype.readthedocs.io/en/latest/reference.html#write) command will generate a A4-sized SVG (otherwise it\nwould have the same size as _in.svg_). Likewise, because `--center` is used, the [`write`](https://vpype.readthedocs.io/en/latest/reference.html#write) command will center geometries\non the page before saving the SVG (otherwise the geometries would have been left at their original location).\n\n\n## Examples\n\n**Note**: The following examples are laid out over multiple lines using end-of-line escaping (`\\`). This is done to highlight the various commands of which the pipeline is made and would typically not be done in real-world use. \n\nLoad an SVG file, scale it to a specific size, and export it centered on an A4-sized, ready-to-plot SVG file:\n```bash\n$ vpype \\\n  read input.svg \\\n  layout --fit-to-margins 2cm a4 \\\n  write output.svg\n```\n\nOptimize paths to reduce plotting time (merge connected lines, sort them to minimize pen-up distance, randomize closed paths\' seam, and reduce the number of nodes):\n```bash\n$ vpype \\\n  read input.svg \\\n  linemerge --tolerance 0.1mm \\\n  linesort \\\n  reloop \\\n  linesimplify \\\n  write output.svg\n```\n\nLoad a SVG and display it in *vpype*\'s viewer, which enable close inspection of the layer and path structure):\n```bash\n$ vpype \\\n  read input.svg \\\n  show\n```\n\nLoad several SVG files and save them as a single, multi-layer SVG file (e.g. for multicolored drawings):\n```bash\n$ vpype \\\n  forfile "*.svg" \\\n    read --layer %_i% %_path% \\\n  end \\\n  write output.svg\n```\n\nExport a SVG to HPGL for vintage plotters:\n```bash\n$ vpype \\\n  read input.svg \\\n  layout --fit-to-margins 2cm --landscape a4 \\\n  write --device hp7475a output.hpgl\n```\n\nDraw the layer name on a SVG (this example uses [property substitution](https://vpype.readthedocs.io/en/latest/fundamentals.html#property-substitution)):\n```bash\n$ vpype \\\n    read input.svg \\\n    text --layer 1 "{vp_name}" \\\n    write output.svg\n```\n\nMerge multiple SVG files in a grid layout (this example uses [expression substitution](https://vpype.readthedocs.io/en/latest/fundamentals.html#expression-substitution)):\n```bash\n$ vpype \\\n    eval "files=glob(\'*.svg\')" \\\n    eval "cols=3; rows=ceil(len(files)/cols)" \\\n    grid -o 10cm 10cm "%cols%" "%rows%" \\\n        read --no-fail "%files[_i] if _i < len(files) else \'\'%" \\\n        layout -m 0.5cm 10x10cm \\\n    end \\\n    write combined_on_a_grid.svg\n```\n\nAn interactive version of the previous example is available in `examples/grid.vpy`. It makes use of `input()` expressions to ask parameters from the user:\n```bash\n$ vpype -I examples/grid.vpy\nFiles [*.svg]?\nNumber of columns [3]? 4\nColumn width [10cm]?\nRow height [10cm]? 15cm\nMargin [0.5cm]?\nOutput path [output.svg]?\n```\n\nSplit a SVG into one file per layer:\n```bash\n$ vpype \\\n    read input.svg \\\n    forlayer \\\n      write "output_%_name or _lid%.svg" \\\n    end\n```\n\nMore examples and recipes are available in the [cookbook](https://vpype.readthedocs.io/en/latest/cookbook.html). \n  \n## What _vpype_ isn\'t?\n\n_vpype_ caters to plotter generative art and does not aim to be a general purpose (think\nIllustrator/InkScape) vector graphic tools. One of the main reason for this is the fact _vpype_ converts everything \ncurvy (circles, bezier curves, etc.) to lines made of small segments. _vpype_ does import metadata such stroke and fill color, stroke width, etc., it only makes partial use of them and does not aim to maintain a full consistency with the SVG specification. These design choices make _vpype_\'s rich feature set possible, but limits its use for, e.g., printed media. \n \n \n## Installation\n\nDetailed installation instructions are available in the [latest documentation](https://vpype.readthedocs.io/en/latest/install.html).\n\nTL;DR:\n- Python 3.9 is recommended, but *vpype* is also compatible with Python 3.7 and 3.8. *vpype* is **not** compatible with Python 3.10 yet. \n- *vpype* is published on the [Python Package Index](https://pypi.org) and can be installed with the following command (preferably in a virtual environment):\n  ```bash\n  pip install "vpype[all]"\n  ```\n- A Windows installer is available [here](https://github.com/abey79/vpype/releases) (plug-ins cannot be installed\nwhen using this installation method).\n- A CLI-only version of *vpype* can be installed using the following command:\n  ```bash\n  pip install vpype\n  ```\n  This version does not include the [`show`](https://vpype.readthedocs.io/en/latest/reference.html#show) command but does not require some of the dependencies which are more difficult or impossible to install on some platforms (such as matplotlib, PySide2, and ModernGL).\n\n\n## Documentation\n\nThe _vpype_ CLI includes its own, detailed documentation:\n\n```bash\nvpype --help          # general help and command list\nvpype COMMAND --help  # help for a specific command\n``` \n\nIn addition, the [online documentation](https://vpype.readthedocs.io/en/latest/) provides extensive background\ninformation on the fundamentals behind _vpype_, a cookbook covering most common tasks, the _vpype_ API documentation,\nand much more.\n\n\n## Feature overview\n\n#### General\n\n- Easy to use **CLI** interface with integrated help (`vpype --help`and `vpype COMMAND --help`) and support for arbitrary units (e.g. `vpype read input.svg translate 3cm 2in`). \n- First-class **multi-layer support** with global or per-layer processing (e.g. `vpype COMMANDNAME --layer 1,3`) and optionally-probabilistic layer edition commands ([`lmove`](https://vpype.readthedocs.io/en/latest/reference.html#lmove), [`lcopy`](https://vpype.readthedocs.io/en/latest/reference.html#lcopy), [`ldelete`](https://vpype.readthedocs.io/en/latest/reference.html#ldelete), [`lswap`](https://vpype.readthedocs.io/en/latest/reference.html#lswap), [`lreverse`](https://vpype.readthedocs.io/en/latest/reference.html#lreverse)).\n- Support for **per-layer and global properties**, which acts as metadata and is used by multiple commands and plug-ins.\n- Support for [**property**](https://vpype.readthedocs.io/en/latest/fundamentals.html#property-substitution) and [**expression substitution**](https://vpype.readthedocs.io/en/latest/fundamentals.html#expression-substitution) in CLI user input.\n- Support for complex, **per-layer** processing ([`perlayer`](https://vpype.readthedocs.io/en/latest/reference.html#perlayer)).\n- Powerful hardware-accelerated **display** command with adjustable units, optional per-line coloring, optional pen-up trajectories display and per-layer visibility control ([`show`](https://vpype.readthedocs.io/en/latest/reference.html#show)).\n- Geometry **statistics** extraction ([`stat`](https://vpype.readthedocs.io/en/latest/reference.html#stat)).\n- Support for  **command history** recording (`vpype -H [...]`)\n- Support for **RNG seed** configuration for generative plug-ins (`vpype -s 37 [...]`).\n\n\n#### Input/Output\n\n- Single- and multi-layer **SVG input** with adjustable precision, parallel processing for large SVGs, and supports percent or missing width/height ([`read`](https://vpype.readthedocs.io/en/latest/reference.html#read)).\n- Support for **SVG output** with fine layout control (page size and orientation, centering), layer support with custom layer names, optional display of pen-up trajectories, various option for coloring ([`write`](https://vpype.readthedocs.io/en/latest/reference.html#write)).\n- Support for **HPGL output** config-based generation of HPGL code with fine layout control (page size and orientation, centering).\n- Support for pattern-based **file collection** processing ([`forfile`](https://vpype.readthedocs.io/en/latest/reference.html#forfile)). \n\n\n#### Layout and transforms\n\n- Easy and flexible **layout** command for centring and fitting to margin with selectable le horizontal and vertical alignment\n  ([`layout`](https://vpype.readthedocs.io/en/latest/reference.html#layout)).\n- **Page rotation** command ([`pagerotate`](https://vpype.readthedocs.io/en/latest/reference.html#pagerotate)).\n- Powerful **transform** commands for scaling, translating, skewing and rotating geometries ([`scale`](https://vpype.readthedocs.io/en/latest/reference.html#scale), [`translate`](https://vpype.readthedocs.io/en/latest/reference.html#translate), [`skew`](https://vpype.readthedocs.io/en/latest/reference.html#skew), [`rotate`](https://vpype.readthedocs.io/en/latest/reference.html#rotate)).\n- Support for **scaling** and **cropping** to arbitrary dimensions ([`scaleto`](https://vpype.readthedocs.io/en/latest/reference.html#scaleto), [`crop`](https://vpype.readthedocs.io/en/latest/reference.html#crop)).\n- Support for **trimming** geometries by an arbitrary amount ([`trim`](https://vpype.readthedocs.io/en/latest/reference.html#trim)).\n- Arbitrary **page size** definition ([`pagesize`](https://vpype.readthedocs.io/en/latest/reference.html#pagesize)). \n\n\n#### Metadata\n\n- Adjust layer **color**, **pen width** and **name** ([`color`](https://vpype.readthedocs.io/en/latest/reference.html#color), [`penwidth`](https://vpype.readthedocs.io/en/latest/reference.html#penwidth), [`name`](https://vpype.readthedocs.io/en/latest/reference.html#name)).\n- Apply provided or fully customisable **pen configurations** ([`pens`](https://vpype.readthedocs.io/en/latest/reference.html#pens)).\n- Manipulate global and per-layer **properties** ([`propset`](https://vpype.readthedocs.io/en/latest/reference.html#propset), [`propget`](https://vpype.readthedocs.io/en/latest/reference.html#propget), [`proplist`](https://vpype.readthedocs.io/en/latest/reference.html#proplist), [`propdel`](https://vpype.readthedocs.io/en/latest/reference.html#propdel), [`propclear`](https://vpype.readthedocs.io/en/latest/reference.html#propclear)).\n\n\n#### Plotting optimization\n\n- **Line merging** with optional path reversal and configurable merging threshold ([`linemerge`](https://vpype.readthedocs.io/en/latest/reference.html#linemerge)).\n- **Line sorting** with optional path reversal ([`linesort`](https://vpype.readthedocs.io/en/latest/reference.html#linesort)).\n- **Line simplification** with adjustable accuracy ([`linesimplify`](https://vpype.readthedocs.io/en/latest/reference.html#linesimplify)).\n- Closed paths\' **seam location randomization**, to reduce the visibility of pen-up/pen-down artifacts ([`reloop`](https://vpype.readthedocs.io/en/latest/reference.html#reloop)).\n- Support for generating **multiple passes** on each line ([`multipass`](https://vpype.readthedocs.io/en/latest/reference.html#multipass)).\n\n#### Filters\n\n- Support for **filtering** by line lengths or closed-ness ([`filter`](https://vpype.readthedocs.io/en/latest/reference.html#filter)).\n- **Squiggle** filter for shaky-hand or liquid-like styling ([`squiggles`](https://vpype.readthedocs.io/en/latest/reference.html#squiggles))\n- Support for **splitting** all lines to their constituent segments ([`splitall`](https://vpype.readthedocs.io/en/latest/reference.html#splitall)).\n- Support for **reversing** order of paths within their layers ([`reverse`](https://vpype.readthedocs.io/en/latest/reference.html#reverse)).\n\n#### Generation\n \n - Generation of arbitrary **primitives** including lines, rectangles, circles, ellipses and arcs ([`line`](https://vpype.readthedocs.io/en/latest/reference.html#line), [`rect`](https://vpype.readthedocs.io/en/latest/reference.html#rect), [`circle`](https://vpype.readthedocs.io/en/latest/reference.html#circle), [`ellipse`](https://vpype.readthedocs.io/en/latest/reference.html#ellipse), [`arc`](https://vpype.readthedocs.io/en/latest/reference.html#arc)).\n - Generation of **text** using bundled Hershey fonts ([`text`](https://vpype.readthedocs.io/en/latest/reference.html#text))\n - Generation of grid-like layouts ([`grid`](https://vpype.readthedocs.io/en/latest/reference.html#grid)).\n - Generation of a **frame** around the geometries ([`frame`](https://vpype.readthedocs.io/en/latest/reference.html#frame)).\n - Generation of random lines for debug/learning purposes ([`random`](https://vpype.readthedocs.io/en/latest/reference.html#random))\n\n#### Extensibility and API\n\n - First-class support for **plug-in** extensions (e.g [vpype-text](https://github.com/abey79/vpype-text), [hatched](https://github.com/abey79/hatched), [occult](https://github.com/LoicGoulefert/occult)).\n - Support for **script-based** generation ([`script`](https://vpype.readthedocs.io/en/latest/reference.html#script)).\n - Powerful and [well-documented](https://vpype.readthedocs.io/en/latest/api.html) **API** for plug-ins and other plotter generative art projects.\n \n \n ## Plug-ins\n \n Here is a list of known vpype plug-ins (please make a pull request if yours is missing):\n \n - [vsketch](https://github.com/abey79/vsketch): *vsketch* is complete framework for plotter generative artists implemented using *vpype*\'s API\n - [vpype-pixelart](https://github.com/abey79/vpype-pixelart): plot pixel art\n - [hatched](https://github.com/abey79/hatched): half-toning with hatches\n - [vpype-flow-imager](https://github.com/serycjon/vpype-flow-imager): convert images to flow-line-based designs\n - [occult](https://github.com/LoicGoulefert/occult): perform hidden line removal\n - [vpype-explorations](https://github.com/abey79/vpype-explorations): my personal grab bag of experiments and utilities\n - [vpype-gcode](https://github.com/tatarize/vpype-gcode/): flexible export command for gcode and other text-based format\n - [vpype-dxf](https://github.com/tatarize/vpype-dxf/): read dxf files\n - [vpype-embroidery](https://github.com/EmbroidePy/vpype-embroidery): various embroidery-related utilities, including read from/write to most embroidery formats \n - [vpype-vectrace](https://github.com/tatarize/vpype-vectrace): create outlines from images with vector tracing\n \n \n ## Contributing\n\nContributions to this project are welcome and do not necessarily require software development skills! Check the\n[Contributing section](https://vpype.readthedocs.io/en/latest/contributing.html) of the documentation for more\ninformation.  \n\n\n## License\n\nThis project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.\n',
    'author': 'Antoine Beyeler',
    'author_email': 'abeyeler@ab-ware.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/abey79/vpype',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7, !=2.7.*, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*, !=3.5.*, !=3.6.*',
}


setup(**setup_kwargs)
