#include <stdio.h>
#include <stdlib.h>

#include "../uxn.h"

#include "system.h"

/*
Copyright (c) 2022-2024 Devine Lu Linvega, Andrew Alderwick

Permission to use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE.
*/

Uxn bios;
char *boot_rom;

static unsigned char bios_rom[] = {
	0xa0,
	0x01,
	0x14,
	0x80,
	0x20,
	0x37,
	0x80,
	0x24,
	0x36,
	0xa0,
	0x00,
	0x30,
	0x39,
	0x80,
	0x17,
	0x33,
	0x60,
	0x00,
	0x80,
	0x00,
	0x80,
	0x00,
	0x81,
	0x80,
	0xfb,
	0x13,
	0x80,
	0x07,
	0x1c,
	0x20,
	0x00,
	0x72,
	0xa0,
	0x00,
	0x08,
	0x80,
	0x28,
	0x37,
	0xa0,
	0x00,
	0x00,
	0x80,
	0x2a,
	0x37,
	0xa0,
	0x80,
	0x00,
	0x80,
	0x04,
	0x16,
	0xa0,
	0x03,
	0x35,
	0x60,
	0x00,
	0xed,
	0xa0,
	0x00,
	0x08,
	0x80,
	0x28,
	0x37,
	0x80,
	0x2a,
	0xb6,
	0xa0,
	0x00,
	0x19,
	0x38,
	0x05,
	0x37,
	0xa0,
	0x81,
	0x00,
	0x80,
	0x05,
	0x16,
	0xa0,
	0x03,
	0x39,
	0x60,
	0x00,
	0xd2,
	0x80,
	0x01,
	0x20,
	0x00,
	0x3a,
	0xa0,
	0x00,
	0x28,
	0x80,
	0x28,
	0x37,
	0xa0,
	0x00,
	0x08,
	0x80,
	0x2a,
	0x37,
	0x80,
	0x01,
	0xa0,
	0x03,
	0x0e,
	0x15,
	0xa0,
	0x0a,
	0x0e,
	0x60,
	0x01,
	0x49,
	0xa0,
	0x00,
	0x00,
	0x1d,
	0x20,
	0x00,
	0x01,
	0x00,
	0xa0,
	0x26,
	0x26,
	0x17,
	0xa0,
	0x00,
	0x08,
	0x26,
	0x80,
	0x28,
	0x37,
	0x80,
	0x2a,
	0x37,
	0xa0,
	0x09,
	0x7d,
	0x80,
	0x2c,
	0x37,
	0xa0,
	0xc5,
	0x2f,
	0x97,
	0x97,
	0x17,
	0x00,
	0xa0,
	0x02,
	0x04,
	0x80,
	0x02,
	0x37,
	0x80,
	0x65,
	0x32,
	0xa0,
	0x80,
	0x06,
	0x28,
	0x80,
	0x60,
	0x12,
	0x80,
	0x37,
	0x08,
	0x1c,
	0x20,
	0x00,
	0x01,
	0x6c,
	0x80,
	0x51,
	0x32,
	0xa0,
	0x02,
	0x14,
	0x35,
	0xa0,
	0x02,
	0x0f,
	0x80,
	0x02,
	0x37,
	0x80,
	0x00,
	0xa0,
	0x01,
	0x54,
	0x15,
	0xa0,
	0x0a,
	0x0e,
	0x60,
	0x01,
	0x4e,
	0x21,
	0x94,
	0x20,
	0x00,
	0x02,
	0x22,
	0x6c,
	0x94,
	0x0f,
	0x21,
	0x26,
	0x80,
	0x00,
	0x4f,
	0x80,
	0x03,
	0x1a,
	0x38,
	0x24,
	0x94,
	0x80,
	0x83,
	0x08,
	0x20,
	0x00,
	0x0b,
	0xa0,
	0x00,
	0x03,
	0x38,
	0xaa,
	0x20,
	0xff,
	0xf1,
	0x22,
	0x22,
	0x6c,
	0x21,
	0x34,
	0xa0,
	0x02,
	0x1f,
	0x35,
	0xa0,
	0x02,
	0x1a,
	0x80,
	0x02,
	0x37,
	0xa0,
	0x09,
	0x7d,
	0xa0,
	0x01,
	0x71,
	0x35,
	0x22,
	0x6c,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x01,
	0x00,
	0x06,
	0x00,
	0x01,
	0x01,
	0x00,
	0x00,
	0x00,
	0x01,
	0xfe,
	0x01,
	0x00,
	0x80,
	0x00,
	0x01,
	0x00,
	0x00,
	0x00,
	0x00,
	0x0a,
	0x0d,
	0x01,
	0x00,
	0x90,
	0x00,
	0x01,
	0x00,
	0x00,
	0x00,
	0x00,
	0x09,
	0x7d,
	0x60,
	0x00,
	0x91,
	0x80,
	0x28,
	0xb6,
	0xa0,
	0x00,
	0x08,
	0x38,
	0x05,
	0x37,
	0x80,
	0x03,
	0xa0,
	0x03,
	0x0e,
	0x15,
	0x06,
	0x60,
	0x00,
	0xac,
	0x80,
	0x28,
	0xb6,
	0xa0,
	0x00,
	0x08,
	0x38,
	0x05,
	0x37,
	0x80,
	0x01,
	0xa0,
	0x03,
	0x0e,
	0x15,
	0x80,
	0x08,
	0x19,
	0x80,
	0x00,
	0x04,
	0x38,
	0x26,
	0xa0,
	0x00,
	0x08,
	0x38,
	0x24,
	0x8f,
	0xcf,
	0x80,
	0x80,
	0x0b,
	0x80,
	0x02,
	0x04,
	0x19,
	0xa0,
	0x03,
	0x0e,
	0x15,
	0x94,
	0x60,
	0x00,
	0x7f,
	0xcf,
	0x80,
	0xff,
	0x09,
	0x20,
	0x00,
	0x29,
	0xa0,
	0x25,
	0x26,
	0x17,
	0xa0,
	0x03,
	0x65,
	0x80,
	0x2c,
	0x37,
	0x80,
	0x2a,
	0xb6,
	0xa0,
	0x00,
	0x04,
	0x39,
	0x05,
	0x37,
	0xa0,
	0x03,
	0x2f,
	0x17,
	0x80,
	0x28,
	0xb6,
	0xa0,
	0x00,
	0x08,
	0x39,
	0x05,
	0x37,
	0x80,
	0x2a,
	0xb6,
	0xa0,
	0x00,
	0x04,
	0x38,
	0x05,
	0x37,
	0x80,
	0x28,
	0xb6,
	0xa0,
	0x00,
	0x08,
	0x38,
	0x05,
	0x37,
	0x41,
	0x21,
	0xaa,
	0x20,
	0xff,
	0xb1,
	0x22,
	0x22,
	0x42,
	0xa0,
	0x15,
	0x26,
	0x17,
	0xa0,
	0x03,
	0x7d,
	0x80,
	0x2c,
	0x37,
	0xa0,
	0x01,
	0x2f,
	0x17,
	0x6c,
	0x80,
	0x28,
	0x36,
	0x80,
	0x0c,
	0x33,
	0xa0,
	0x15,
	0x26,
	0x17,
	0x94,
	0x80,
	0x0a,
	0x09,
	0x20,
	0x00,
	0x12,
	0xa0,
	0x00,
	0x00,
	0x80,
	0x28,
	0x37,
	0x80,
	0x2a,
	0xb6,
	0xa0,
	0x00,
	0x10,
	0x38,
	0x05,
	0x37,
	0x40,
	0x00,
	0x04,
	0x94,
	0x60,
	0x00,
	0x1d,
	0x21,
	0x94,
	0x20,
	0xff,
	0xde,
	0x22,
	0x6c,
	0xa0,
	0x15,
	0x26,
	0x17,
	0x06,
	0x80,
	0x04,
	0x1f,
	0x60,
	0x00,
	0x00,
	0x80,
	0x0f,
	0x1c,
	0x80,
	0x00,
	0x04,
	0xa0,
	0x03,
	0x3d,
	0x38,
	0x14,
	0x80,
	0x20,
	0x19,
	0x80,
	0x00,
	0x04,
	0x80,
	0x40,
	0x3f,
	0xa0,
	0x03,
	0x8d,
	0x38,
	0x80,
	0x2c,
	0x37,
	0xa0,
	0x01,
	0x2f,
	0x17,
	0x6c,
	0x21,
	0x94,
	0x20,
	0xff,
	0xfb,
	0x6c,
	0x04,
	0x60,
	0x00,
	0x00,
	0x06,
	0x80,
	0x04,
	0x1f,
	0x60,
	0x00,
	0x00,
	0x80,
	0x0f,
	0x1c,
	0x06,
	0x80,
	0x09,
	0x0a,
	0x80,
	0x27,
	0x1a,
	0x18,
	0x80,
	0x30,
	0x18,
	0x80,
	0x19,
	0x17,
	0x6c,
	0x57,
	0x53,
	0x54,
	0x00,
	0x52,
	0x53,
	0x54,
	0x00,
	0x30,
	0x31,
	0x32,
	0x33,
	0x34,
	0x35,
	0x36,
	0x37,
	0x38,
	0x39,
	0x61,
	0x62,
	0x63,
	0x64,
	0x65,
	0x66,
	0xff,
	0xff,
	0xff,
	0xff,
	0xff,
	0xff,
	0xff,
	0xff,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x10,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x20,
	0x40,
	0x80,
	0x40,
	0x20,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x38,
	0x38,
	0x38,
	0x38,
	0x38,
	0x10,
	0x10,
	0x10,
	0x10,
	0x00,
	0x00,
	0x10,
	0x38,
	0x10,
	0x6c,
	0x6c,
	0x24,
	0x24,
	0x48,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x12,
	0x12,
	0x12,
	0x7f,
	0x24,
	0x24,
	0x24,
	0x24,
	0x24,
	0xfe,
	0x48,
	0x48,
	0x48,
	0x48,
	0x00,
	0x10,
	0x38,
	0x54,
	0x92,
	0x96,
	0x90,
	0x50,
	0x38,
	0x14,
	0x12,
	0xd2,
	0x92,
	0x94,
	0x78,
	0x10,
	0x10,
	0x02,
	0x62,
	0x94,
	0x94,
	0x94,
	0x98,
	0x68,
	0x10,
	0x10,
	0x2c,
	0x32,
	0x52,
	0x52,
	0x52,
	0x8c,
	0x80,
	0x00,
	0x30,
	0x48,
	0x48,
	0x48,
	0x50,
	0x20,
	0x2e,
	0x54,
	0x54,
	0x94,
	0x88,
	0x8c,
	0x72,
	0x00,
	0x00,
	0xe0,
	0xe0,
	0x20,
	0x20,
	0xc0,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x02,
	0x04,
	0x08,
	0x08,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x08,
	0x08,
	0x04,
	0x02,
	0x80,
	0x40,
	0x20,
	0x20,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x20,
	0x20,
	0x40,
	0x80,
	0x00,
	0x00,
	0x00,
	0x10,
	0x38,
	0x92,
	0xd6,
	0x38,
	0xd6,
	0x92,
	0x38,
	0x10,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x10,
	0x10,
	0x10,
	0x10,
	0xfe,
	0x10,
	0x10,
	0x10,
	0x10,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xe0,
	0xe0,
	0x20,
	0x20,
	0xc0,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x40,
	0xe0,
	0xe0,
	0x40,
	0x00,
	0x02,
	0x02,
	0x04,
	0x04,
	0x08,
	0x08,
	0x08,
	0x10,
	0x10,
	0x10,
	0x20,
	0x20,
	0x40,
	0x40,
	0x80,
	0x80,
	0x00,
	0x18,
	0x24,
	0x24,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x24,
	0x24,
	0x18,
	0x00,
	0x00,
	0x10,
	0x70,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x7c,
	0x00,
	0x00,
	0x00,
	0x18,
	0x24,
	0x42,
	0x62,
	0x02,
	0x04,
	0x08,
	0x08,
	0x10,
	0x20,
	0x22,
	0x42,
	0x7e,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0x82,
	0x02,
	0x04,
	0x38,
	0x04,
	0x02,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0x08,
	0x18,
	0x28,
	0x28,
	0x48,
	0x48,
	0x88,
	0x88,
	0xfe,
	0x08,
	0x08,
	0x08,
	0x3c,
	0x00,
	0x00,
	0x00,
	0xfc,
	0x80,
	0x80,
	0x80,
	0xb8,
	0xc4,
	0x82,
	0x02,
	0x02,
	0xc2,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0x3c,
	0x42,
	0x46,
	0x80,
	0x80,
	0xb8,
	0xc4,
	0x82,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x82,
	0x82,
	0x04,
	0x04,
	0x04,
	0x08,
	0x08,
	0x08,
	0x08,
	0x10,
	0x10,
	0x10,
	0x10,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x44,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0x82,
	0x82,
	0x82,
	0x46,
	0x3a,
	0x02,
	0x02,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x38,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x38,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x38,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x38,
	0x18,
	0x10,
	0x30,
	0x02,
	0x04,
	0x04,
	0x08,
	0x08,
	0x10,
	0x10,
	0x20,
	0x20,
	0x10,
	0x10,
	0x08,
	0x08,
	0x04,
	0x04,
	0x02,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x80,
	0x40,
	0x40,
	0x20,
	0x20,
	0x10,
	0x10,
	0x08,
	0x08,
	0x10,
	0x10,
	0x20,
	0x20,
	0x40,
	0x40,
	0x80,
	0x00,
	0x38,
	0x44,
	0x82,
	0xc2,
	0x02,
	0x04,
	0x04,
	0x08,
	0x10,
	0x10,
	0x00,
	0x00,
	0x10,
	0x38,
	0x10,
	0x00,
	0x3c,
	0x42,
	0x82,
	0x9a,
	0xa6,
	0xa2,
	0xa2,
	0xa2,
	0xa6,
	0x9a,
	0x80,
	0x42,
	0x3c,
	0x00,
	0x00,
	0x00,
	0x10,
	0x28,
	0x28,
	0x28,
	0x44,
	0x44,
	0x44,
	0x44,
	0x7c,
	0x82,
	0x82,
	0x82,
	0xc6,
	0x00,
	0x00,
	0x00,
	0xf8,
	0x44,
	0x42,
	0x42,
	0x42,
	0x44,
	0x78,
	0x44,
	0x42,
	0x42,
	0x42,
	0x42,
	0xfc,
	0x00,
	0x00,
	0x00,
	0x3a,
	0x46,
	0x42,
	0x80,
	0x80,
	0x80,
	0x80,
	0x80,
	0x80,
	0x82,
	0x42,
	0x42,
	0x3c,
	0x00,
	0x00,
	0x00,
	0xf8,
	0x44,
	0x44,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x44,
	0x44,
	0xf8,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x42,
	0x42,
	0x40,
	0x48,
	0x48,
	0x78,
	0x48,
	0x48,
	0x42,
	0x42,
	0x42,
	0xfe,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x42,
	0x42,
	0x40,
	0x48,
	0x48,
	0x78,
	0x48,
	0x48,
	0x40,
	0x40,
	0x40,
	0xf0,
	0x00,
	0x00,
	0x00,
	0x1a,
	0x26,
	0x42,
	0x40,
	0x80,
	0x80,
	0x8f,
	0x82,
	0x82,
	0x82,
	0x42,
	0x66,
	0x1a,
	0x00,
	0x00,
	0x00,
	0xe7,
	0x42,
	0x42,
	0x42,
	0x42,
	0x7e,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0xe7,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0xfe,
	0x00,
	0x00,
	0x00,
	0x1f,
	0x02,
	0x02,
	0x02,
	0x02,
	0x02,
	0x02,
	0x02,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0xe6,
	0x44,
	0x44,
	0x48,
	0x48,
	0x70,
	0x50,
	0x48,
	0x48,
	0x44,
	0x44,
	0x42,
	0xe3,
	0x00,
	0x00,
	0x00,
	0xf0,
	0x40,
	0x40,
	0x40,
	0x40,
	0x40,
	0x40,
	0x40,
	0x40,
	0x42,
	0x42,
	0x42,
	0xfe,
	0x00,
	0x00,
	0x00,
	0x82,
	0xc6,
	0xaa,
	0xaa,
	0xaa,
	0x92,
	0x92,
	0x92,
	0x82,
	0x82,
	0x82,
	0x82,
	0xc6,
	0x00,
	0x00,
	0x00,
	0x87,
	0xc2,
	0xa2,
	0xa2,
	0xa2,
	0x92,
	0x92,
	0x92,
	0x8a,
	0x8a,
	0x8a,
	0x86,
	0xc2,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0xf8,
	0x44,
	0x42,
	0x42,
	0x42,
	0x42,
	0x44,
	0x78,
	0x40,
	0x40,
	0x40,
	0x40,
	0xf0,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x44,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0xba,
	0x44,
	0x44,
	0x38,
	0x08,
	0x06,
	0x00,
	0xf8,
	0x44,
	0x42,
	0x42,
	0x42,
	0x44,
	0x78,
	0x48,
	0x44,
	0x44,
	0x44,
	0x42,
	0xe3,
	0x00,
	0x00,
	0x00,
	0x34,
	0x4c,
	0x84,
	0x80,
	0x80,
	0x60,
	0x18,
	0x04,
	0x82,
	0x82,
	0x82,
	0xc4,
	0xb8,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x92,
	0x92,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x7c,
	0x00,
	0x00,
	0x00,
	0xe7,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x3c,
	0x00,
	0x00,
	0x00,
	0xc6,
	0x82,
	0x82,
	0x82,
	0x82,
	0x44,
	0x44,
	0x44,
	0x44,
	0x28,
	0x28,
	0x10,
	0x10,
	0x00,
	0x00,
	0x00,
	0xc6,
	0x82,
	0x82,
	0x82,
	0x92,
	0x92,
	0x92,
	0xaa,
	0xaa,
	0xaa,
	0x44,
	0x44,
	0x44,
	0x00,
	0x00,
	0x00,
	0xee,
	0x44,
	0x44,
	0x28,
	0x28,
	0x10,
	0x28,
	0x28,
	0x28,
	0x44,
	0x44,
	0x82,
	0xc6,
	0x00,
	0x00,
	0x00,
	0xc6,
	0x82,
	0x44,
	0x44,
	0x44,
	0x28,
	0x28,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x7c,
	0x00,
	0x00,
	0x00,
	0xfe,
	0x84,
	0x88,
	0x08,
	0x10,
	0x10,
	0x10,
	0x20,
	0x20,
	0x42,
	0x42,
	0x82,
	0xfe,
	0x00,
	0x00,
	0x1e,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x1e,
	0x00,
	0x80,
	0x80,
	0x40,
	0x40,
	0x20,
	0x20,
	0x10,
	0x10,
	0x08,
	0x08,
	0x04,
	0x04,
	0x02,
	0x02,
	0x00,
	0xf0,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0xf0,
	0x10,
	0x28,
	0x44,
	0x82,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xff,
	0x30,
	0x30,
	0x20,
	0x20,
	0x10,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x3c,
	0x42,
	0x02,
	0x3e,
	0x42,
	0x82,
	0x82,
	0x86,
	0x7b,
	0x00,
	0x00,
	0x00,
	0xc0,
	0x40,
	0x40,
	0x40,
	0x78,
	0x44,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x44,
	0x78,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x3a,
	0x46,
	0x82,
	0x80,
	0x80,
	0x80,
	0x82,
	0x42,
	0x3c,
	0x00,
	0x00,
	0x00,
	0x06,
	0x04,
	0x04,
	0x04,
	0x3c,
	0x44,
	0x84,
	0x84,
	0x84,
	0x84,
	0x84,
	0x44,
	0x3e,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0xfe,
	0x80,
	0x80,
	0x82,
	0x42,
	0x3c,
	0x00,
	0x00,
	0x00,
	0x0e,
	0x11,
	0x10,
	0x10,
	0xfe,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x7c,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x3b,
	0x44,
	0x44,
	0x44,
	0x38,
	0x40,
	0x78,
	0x84,
	0x82,
	0x82,
	0x7c,
	0x00,
	0xc0,
	0x40,
	0x40,
	0x40,
	0x5c,
	0x62,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0xe7,
	0x00,
	0x00,
	0x18,
	0x18,
	0x00,
	0x00,
	0x00,
	0x78,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0xff,
	0x00,
	0x00,
	0x06,
	0x06,
	0x00,
	0x00,
	0x00,
	0x3e,
	0x02,
	0x02,
	0x02,
	0x02,
	0x02,
	0x02,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0xc0,
	0x40,
	0x40,
	0x40,
	0x42,
	0x44,
	0x48,
	0x58,
	0x64,
	0x44,
	0x42,
	0x42,
	0xe3,
	0x00,
	0x00,
	0x00,
	0x78,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0xff,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x6c,
	0x92,
	0x92,
	0x92,
	0x92,
	0x92,
	0x92,
	0x92,
	0xdb,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xdc,
	0x62,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0xe7,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x38,
	0x44,
	0x82,
	0x82,
	0x82,
	0x82,
	0x82,
	0x44,
	0x38,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xf8,
	0x44,
	0x42,
	0x42,
	0x42,
	0x42,
	0x44,
	0x78,
	0x40,
	0x40,
	0xf0,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x3e,
	0x44,
	0x84,
	0x84,
	0x84,
	0x84,
	0x44,
	0x3c,
	0x04,
	0x04,
	0x1e,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xec,
	0x32,
	0x22,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0xfc,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x3a,
	0x46,
	0x42,
	0x40,
	0x3c,
	0x02,
	0x82,
	0xc2,
	0xbc,
	0x00,
	0x00,
	0x00,
	0x00,
	0x20,
	0x20,
	0x20,
	0xfc,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0x22,
	0x22,
	0x1c,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xc6,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x42,
	0x46,
	0x39,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xc6,
	0x82,
	0x82,
	0x44,
	0x44,
	0x44,
	0x28,
	0x28,
	0x10,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x92,
	0x92,
	0x92,
	0x92,
	0xaa,
	0xaa,
	0x44,
	0x44,
	0x44,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xee,
	0x44,
	0x28,
	0x28,
	0x10,
	0x28,
	0x28,
	0x44,
	0xee,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xe7,
	0x42,
	0x22,
	0x24,
	0x14,
	0x08,
	0x08,
	0x10,
	0x90,
	0xa0,
	0x40,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x7e,
	0x44,
	0x08,
	0x08,
	0x10,
	0x10,
	0x22,
	0x42,
	0xfe,
	0x00,
	0x00,
	0x06,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x10,
	0x20,
	0x10,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x08,
	0x06,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0x10,
	0xc0,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0x10,
	0x08,
	0x10,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0x20,
	0xc0,
	0x60,
	0x92,
	0x0c,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
	0xaa,
	0x55,
};

static void
system_zero(Uxn *u, int soft)
{
	int i;
	for(i = soft ? 0x100 : 0; i < 0x10000; i++)
		u->ram[i] = 0;
	for(i = 0x0; i < 0x100; i++)
		u->dev[i] = 0;
	u->wst.ptr = u->rst.ptr = 0;
}

static int
system_load(Uint8 *ram, char *filename)
{
	FILE *f = fopen(filename, "rb");
	if(f) {
		int i = 0, l = fread(ram, 0x10000 - PAGE_PROGRAM, 1, f);
		while(l && ++i < RAM_PAGES)
			l = fread(ram + 0x10000 * i - PAGE_PROGRAM, 0x10000, 1, f);
		fclose(f);
	}
	return !!f;
}

static void
system_print(Stack *s)
{
	Uint8 i;
	for(i = s->ptr - 8; i != (Uint8)(s->ptr); i++)
		fprintf(stderr, "%02x%c", s->dat[i], i == 0xff ? '|' : ' ');
	fprintf(stderr, "< \n");
}

void
system_inspect(void)
{
	fprintf(stderr, "WST "), system_print(&uxn.wst);
	fprintf(stderr, "RST "), system_print(&uxn.rst);
}

int
system_error(char *msg, const char *err)
{
	fprintf(stderr, "%s: %s\n", msg, err), fflush(stderr);
	return 0;
}

void
system_reboot(int soft)
{
	system_zero(&uxn, soft);
	if(system_load(&uxn.ram[PAGE_PROGRAM], boot_rom))
		uxn_eval(&uxn, PAGE_PROGRAM);
}

int
system_boot(Uint8 *ram, char *boot)
{
	/* Load rom(bios) */
	bios.ram = ram;
	bios.dev = bios.ram + 0x8300;
	bios.wst.dat = bios.ram + 0x8400;
	bios.rst.dat = bios.ram + 0x8500;
	bios.dev[0xe] = 0xff;
	if(!system_load(bios.ram + PAGE_PROGRAM, "bios.rom")) {
		size_t i, bios_len = sizeof(bios_rom) / sizeof(bios_rom[0]);
		for(i = 0; i < bios_len; i++)
			bios.ram[PAGE_PROGRAM + i] = bios_rom[i];
	}
	/* Load rom(main) */
	uxn.ram = ram + 0x10000;
	uxn.dev = bios.ram + 0x8000;
	uxn.wst.dat = bios.ram + 0x8100;
	uxn.rst.dat = bios.ram + 0x8200;
	boot_rom = boot;
	return system_load(uxn.ram + PAGE_PROGRAM, boot);
}

/* IO */

Uint8
system_dei(Uxn *u, Uint8 addr)
{
	switch(addr) {
	case 0x4: return uxn.wst.ptr;
	case 0x5: return uxn.rst.ptr;
	default: return u->dev[addr];
	}
}

void
system_deo(Uxn *u, Uint8 port)
{
	switch(port) {
	case 0x3: {
		Uint16 addr = PEEK2(u->dev + 2);
		if(u->ram[addr] == 0x0) {
			Uint16 i, value = u->ram[addr + 7], length = PEEK2(u->ram + addr + 1);
			unsigned int dst_page = PEEK2(u->ram + addr + 3), dst_addr = PEEK2(u->ram + addr + 5);
			unsigned int dst = dst_page * 0x10000;
			if(dst_page < RAM_PAGES)
				for(i = 0; i < length; i++)
					u->ram[dst + (Uint16)(dst_addr + i)] = value;
		} else if(u->ram[addr] == 0x1) {
			Uint16 i, length = PEEK2(u->ram + addr + 1);
			unsigned int a_page = PEEK2(u->ram + addr + 3), a_addr = PEEK2(u->ram + addr + 5);
			unsigned int b_page = PEEK2(u->ram + addr + 7), b_addr = PEEK2(u->ram + addr + 9);
			unsigned int src = a_page * 0x10000, dst = b_page * 0x10000;
			if(a_page < RAM_PAGES && b_page < RAM_PAGES)
				for(i = 0; i < length; i++)
					u->ram[dst + (Uint16)(b_addr + i)] = u->ram[src + (Uint16)(a_addr + i)];
		} else if(u->ram[addr] == 0x2) {
			Uint16 i, length = PEEK2(u->ram + addr + 1);
			unsigned int a_page = PEEK2(u->ram + addr + 3), a_addr = PEEK2(u->ram + addr + 5);
			unsigned int b_page = PEEK2(u->ram + addr + 7), b_addr = PEEK2(u->ram + addr + 9);
			unsigned int src = a_page * 0x10000, dst = b_page * 0x10000;
			if(a_page < RAM_PAGES && b_page < RAM_PAGES)
				for(i = length - 1; i != 0xffff; i--)
					u->ram[dst + (Uint16)(b_addr + i)] = u->ram[src + (Uint16)(a_addr + i)];
		} else
			fprintf(stderr, "Unknown Expansion Command 0x%02x\n", u->ram[addr]);
		break;
	}
	case 0x4:
		uxn.wst.ptr = uxn.dev[4];
		break;
	case 0x5:
		uxn.rst.ptr = uxn.dev[5];
		break;
	case 0xe:
		system_inspect();
		break;
	}
}
