//
// Copyright 2016 Pixar
//
// Licensed under the Apache License, Version 2.0 (the "Apache License")
// with the following modification; you may not use this file except in
// compliance with the Apache License and the following modification to it:
// Section 6. Trademarks. is deleted and replaced with:
//
// 6. Trademarks. This License does not grant permission to use the trade
//    names, trademarks, service marks, or product names of the Licensor
//    and its affiliates, except as required to comply with Section 4(c) of
//    the License and to reproduce the content of the NOTICE file.
//
// You may obtain a copy of the Apache License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the Apache License with the above modification is
// distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. See the Apache License for the specific
// language governing permissions and limitations under the Apache License.
//
#ifndef PXR_IMAGING_HIO_GLSLFX_H
#define PXR_IMAGING_HIO_GLSLFX_H

/// \file hio/glslfx.h

#include "pxr/pxr.h"
#include "pxr/imaging/hio/api.h"
#include "pxr/imaging/hio/glslfxConfig.h"

#include "pxr/base/tf/token.h"
#include "pxr/base/tf/staticTokens.h"

#include <boost/scoped_ptr.hpp>

#include <string>
#include <vector>
#include <set>
#include <map>

PXR_NAMESPACE_OPEN_SCOPE

#define HIO_GLSLFX_TOKENS       \
    (glslfx)                    \
                                \
    (fragmentShader)            \
    (geometryShader)            \
    (geometryShaderInjection)   \
    (preamble)                  \
    (tessControlShader)         \
    (tessEvalShader)            \
    (vertexShader)              \
    (vertexShaderInjection)     \
                                \
    (surfaceShader)             \
    (displacementShader)        \
    (volumeShader)              \


TF_DECLARE_PUBLIC_TOKENS(HioGlslfxTokens, HIO_API, HIO_GLSLFX_TOKENS);

/// \class HioGlslfx
///
/// A class representing the config and shader source of a glslfx file.
///
/// a HioGlslfx object is constructed by providing the path of a file whose
/// contents look something like this:
///
/// \code
/// -- glslfx version 0.1
/// 
/// -- configuration
/// 
/// {
///
///     'textures' : {
///         'texture_1':{
///             'documentation' : 'a useful texture.',
///         },
///         'texture_2':{
///             'documentation' : 'another useful texture.',
///         },
///     },
///     'parameters': {
///         'param_1' : {
///             'default' : 1.0,
///             'documentation' : 'the first parameter'
///         },
///         'param_2' : {
///             'default' : [1.0, 1.0, 1.0],
///             'documentation' : 'a vec3f parameter'
///         },
///         'param_3' : {
///             'default' : 2.0
///         },
///         'param_4' : {
///             'default' : True
///         },
///         'param_5' : {
///             'default' : [1.0, 1.0, 1.0],
///             'role' : 'color'
///             'documentation' : 'specifies a color for use in the shader'
///         },
///     },
///     'parameterOrder': ['param_1',
///                        'param_2',
///                        'param_3',
///                        'param_4',
///                        'param_5'],
/// 
///     'techniques': {
///         'default': {
///             'fragmentShader': {
///                 'source': [ 'MyFragment' ]
///             }
///         }
///     }
/// }
/// 
/// -- glsl MyFragment
/// 
/// uniform float param_1;
/// uniform float param_2;
/// uniform float param_3;
/// uniform float param_4;
/// uniform float param_5;
/// 
/// void main()
/// {
///     // ...
///     // glsl code which consumes the various uniforms, and perhaps sets
///     // gl_FragColor = someOutputColor;
///     // ...
/// }
/// \endcode
///
class HioGlslfx
{
public:
    /// Create an invalid glslfx object
    HIO_API
    HioGlslfx();

    /// Create a glslfx object from a file
    HIO_API
    HioGlslfx(std::string const & filePath);

    /// Create a glslfx object from a stream
    HIO_API
    HioGlslfx(std::istream &is);

    /// Return the parameters specified in the configuration
    HIO_API
    HioGlslfxConfig::Parameters GetParameters() const;

    /// Return the textures specified in the configuration
    HIO_API
    HioGlslfxConfig::Textures GetTextures() const;

    /// Return the attributes specified in the configuration
    HIO_API
    HioGlslfxConfig::Attributes GetAttributes() const;

    /// Return the metadata specified in the configuration
    HIO_API
    HioGlslfxConfig::MetadataDictionary GetMetadata() const;

    /// Returns true if this is a valid glslfx file
    HIO_API
    bool IsValid(std::string *reason=NULL) const;

    /// \name Access to commonly used shader sources.
    /// @{

    /// Get the surface source string
    HIO_API
    std::string GetSurfaceSource() const;

    /// Get the displacement source string
    HIO_API
    std::string GetDisplacementSource() const;

    /// Get the volume source string
    HIO_API
    std::string GetVolumeSource() const;

    /// @}

    /// Get the shader source associated with given key
    HIO_API
    std::string GetSource(const TfToken &shaderStageKey) const;

    /// Get the original file name passed to the constructor
    const std::string &GetFilePath() const { return _globalContext.filename; }

    /// Return set of all files processed for this glslfx object.
    /// This includes the original file given to the constructor
    /// as well as any other files that were imported. This set
    /// will only contain files that exist.
    const std::set<std::string>& GetFiles() const { return _seenFiles; }

    /// Return the computed hash value based on the string
    size_t GetHash() const { return _hash; }

private:
    class _ParseContext {
    public:
        _ParseContext() { }

        _ParseContext(std::string const & filePath) :
            filename(filePath), lineNo(0), version(-1.0) { }

        std::string filename;
        int lineNo;
        double version;
        std::string currentLine;
        std::string currentSectionType;
        std::string currentSectionId;
        std::vector<std::string> imports;
    };

private:
    bool _ProcessFile(std::string const & filePath,
                      _ParseContext & context);
    bool _ProcessInput(std::istream * input,
                       _ParseContext & context);
    bool _ProcessImport(_ParseContext & context);
    bool _ParseSectionLine(_ParseContext & context);
    bool _ParseGLSLSectionLine(std::vector<std::string> const &tokens,
                               _ParseContext & context);
    bool _ParseVersionLine(std::vector<std::string> const &tokens,
                           _ParseContext & context);
    bool _ParseConfigurationLine(_ParseContext & context);
    bool _ComposeConfiguration(std::string *reason);
    std::string _GetSource(const TfToken &shaderStageKey) const;

private:
    _ParseContext _globalContext;

    std::set<std::string> _importedFiles;

    typedef std::map<std::string, std::string> _SourceMap;

    _SourceMap _sourceMap;
    _SourceMap _configMap;
    std::vector<std::string> _configOrder;
    std::set<std::string> _seenFiles;

    boost::scoped_ptr<HioGlslfxConfig> _config;

    bool _valid;
    std::string _invalidReason; // if _valid is false, reason why
    size_t _hash;
};


PXR_NAMESPACE_CLOSE_SCOPE

#endif

