/*
 Uptime Client v5.0 beta

 $Id: stats-bsd.c,v 1.39 2003/05/26 20:37:56 carstenklapp Exp $

 Logs system uptime and statistics with Uptimes Project servers

 Copyright (C) 1999-2002 Martijn Broenland, Alex C. de Haas, Carsten Klapp

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 Carsten Klapp <carstenklapp@users.sourceforge.net>
 Alex C. de Haas <alex@uptimes.net>
 Martijn Broenland <tgm@uptimes.net>
 */

/**
 * @filename    stats-bsd.c
 *
 * @desc        Retrieve stats for the *BSD platforms.
 *              Darwin / Mac OS X will use idle functions from stats-mach.c
 */

#if defined PLATFORM_BSD

/*@unused@*/ static const char rcsid[] =
    "@(#)$Id: stats-bsd.c,v 1.39 2003/05/26 20:37:56 carstenklapp Exp $";

/* My includes */
#include "upclient.h"
#include "options.h"
#include "stats.h"
#include "uplog.h"

/* System includes */
#include <err.h>        /* FreeBSD */
#include <errno.h>
#include <fcntl.h>
#include <kvm.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/dkstat.h>
#include <sys/param.h>
#if defined __FreeBSD__ && __FreeBSD_Version
#   if __FreeBSD_version >= 500000
#      include <sys/resource.h>
#   endif /* __FreeBSD_version >= 500000 */
#endif /* __FreeBSD__ && __FreeBSD_Version__ */
#if defined __NetBSD__ && __NetBSD_Version__ >= 104000000
/* sys/sched.h is available only on recent NetBSD. */
#   include <sys/sched.h>       /* CPUSTATES, CP_IDLE */
#endif /* __NetBSD__ && __NetBSD_Version__ >= 104000000 */
#include <sys/sysctl.h>
#include <sys/time.h>
#include <sys/types.h>  /* size_t */
#include <sys/utsname.h>

#include "locale.h"     /* gettext */

/**
 * @desc    Verbose level 3 logging of calulations
 */
void
logcalc(char *whatwascalculateddesc, char *value)
{
#if defined DEBUG
    uplog(LOG_DEBUG, _("%s calculated: %s"), whatwascalculateddesc, value);
#endif /* DEBUG */
}

/**
 * @desc    Get cpu once
 */
void
initCPU(char *cpu)
{
    char   buf[CPU_SIZE] = "";
    static int initialized = 0;
    size_t size;
    int    mib[2];

    if (!initialized) {
        initialized = 1;

        mib[0] = CTL_HW;
        if (cfg_sendcpudetail)
            mib[1] = HW_MODEL;  /* Send architecture's specific model */
        else
            mib[1] = HW_MACHINE;        /* Send architecture */
        size = CPU_SIZE;

        if (sysctl(mib, 2, buf, &size, NULL, 0) == -1) {
            uplog(LOG_WARNING, "sysctl: hw.model");
        }
        strlcpy(cpu, buf, CPU_SIZE);
        logcalc(_("CPU"), cpu);
    }
}

/**
 * @desc    Get os & osversion once
 */
void
initOS(char *osname, char *osversion)
{
    static int initialized = 0;

    if (!initialized) {
        struct utsname uts;

        initialized = 1;

        uname(&uts);

        if (cfg_sendosname) {
            strlcpy(osname, uts.sysname, OS_SIZE);
            logcalc(_("OS"), osname);
        }

        if (cfg_sendosversion) {
            strlcpy(osversion, uts.release, OSVERSION_SIZE);
            logcalc(_("OS version"), osversion);
        }
    }
}

time_t
initBoottime(void)
{
    static time_t boottimesecs;
    static int initialized = 0;

    if (!initialized) {
        struct timeval boottime;
        int    mib[2];
        size_t size;

        initialized = 1;

        mib[0] = CTL_KERN;
        mib[1] = KERN_BOOTTIME;
        size = sizeof(boottime);

        if (sysctl(mib, 2, &boottime, &size, NULL, 0) == -1) {
            uplog(LOG_ERR, "sysctl: kern.boottime");
            err(1, "sysctl: kern.boottime");
        }
        boottimesecs = boottime.tv_sec;
#if defined DEBUG
        uplog(LOG_DEBUG, "initBoottime() initialized %d", boottimesecs);
#endif /* DEBUG */
    }
    return boottimesecs;
}

void
getLoadavg(double *loadavg)
{
    double loadavgs[3] = { 0. };
    int    loads = getloadavg(loadavgs, 3);

    if (loads == -2) {
        uplog(LOG_WARNING,
              _("%s %s not implemented for this operating system %s"),
              _("WARNING:"), _("Load-average"), strerror(errno));
        cfg_SendUsage = 0;
        return;
    }
    else if (loads < 0) {
        uplog(LOG_WARNING, _("%s Could not get load-average %s"), _("WARNING:"),
              strerror(errno));
        return;
    }

   /* Use the 3rd element (15 minute Load-average) */
    *loadavg = loadavgs[2];
#if defined DEBUG
    uplog(LOG_DEBUG, "getLoadavg() %.2f", *loadavg);
#endif /* DEBUG */
}

#if !defined __MACH__   /* Mach kernel stuff is in stats-mach.c */
void
getLoadIdle(double *UsagePercent, double *IdlePercent)
{
    static kvm_t *kp;
    static int initialized = 0;
    static struct nlist namelist[] = {
#define X_CP_TIME 0
        {"_cp_time"},
        {NULL},
    };

    if (!initialized) {
        char   errbuf[_POSIX2_LINE_MAX];

        initialized = 1;

        setgid(getgid());

        kp = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, errbuf);

        if (!kp) {
            uplog(LOG_WARNING, "kvm_openfiles: %s. %s", errbuf,
                  _("(Did you forget to start upclient as root?)"));
        }
        else if (kvm_nlist(kp, namelist) == -1) {
            uplog(LOG_WARNING, "kvm_nlist: %s", kvm_geterr(kp));
        }
    }

    if (kp) {
        long   ctime[CPUSTATES];
        static long stime[CPUSTATES];
        static int first_time = 1;

        if (first_time) {
            first_time = 0;

            if (kvm_read(kp, namelist[X_CP_TIME].n_value, stime, sizeof(stime))
                == -1) {
                uplog(LOG_WARNING, "kvm_read: %s", kvm_geterr(kp));
            }
            else
                sleep(1);       /* to avoid the initial 0 value */
        }

        if (kvm_read(kp, namelist[X_CP_TIME].n_value, ctime, sizeof(ctime)) ==
            -1) {
            uplog(LOG_WARNING, "kvm_read: %s", kvm_geterr(kp));
        }
        else {
            int    state;
            double time = 0.;
            float  nonidlecpu = 0.;

            for (state = 0; state < CPUSTATES; state++) {
                time += ctime[state] - stime[state];
                if (!(state == CP_IDLE)) {
                    nonidlecpu += (float)(ctime[state] - stime[state]);
                }
            }

            if (time == 0.)
                time = 1.;

            *UsagePercent = (double)(nonidlecpu * 100. / time);
            *IdlePercent = (ctime[CP_IDLE] - stime[CP_IDLE]) * 100. / time;

           /* store values for next calculation interval */
            for (state = 0; state < CPUSTATES; state++)
                stime[state] = ctime[state];
        }
    }
    else {
#if defined DEBUG
        uplog(LOG_DEBUG,
              "getLoadIdle() setting cfg_SendUsage and cfg_SendIdle to 0");
#endif /* DEBUG */
        cfg_SendUsage = 0;
        cfg_SendIdle = 0;
    }
#if defined DEBUG
    uplog(LOG_DEBUG, "getLoadIdle() load=%.1f idle=%.1f", *UsagePercent,
          *IdlePercent);
#endif /* DEBUG */
}
#endif /* !__MACH__ */

void
getUptime(unsigned long *uptimeminutes)
{
    static time_t boottimesecs;

    boottimesecs = initBoottime();

    if (boottimesecs) {
        time_t now;

        time(&now);

        *uptimeminutes = (now - boottimesecs) / 60;
#if defined DEBUG
        uplog(LOG_DEBUG, "getUptime() %d", *uptimeminutes);
#endif /* DEBUG */
    }
    else {
        uplog(LOG_INFO, _("%s: %s failed"), _("ERROR:"),
              "getUptime() boottime.tv_sec");
    }
}

/**
 * @desc    Get statistics
 */
void
getstats(unsigned long *uptimeminutes, double *UsagePercent,
         double *IdlePercent, char *osname, char *osversion, char *cpu,
         double *loadavg)
{
    getUptime(&*uptimeminutes);

    if (cfg_SendUsage || cfg_SendIdle)
        getLoadIdle(&*UsagePercent, &*IdlePercent);

    if (cfg_sendosname || cfg_sendosversion)
        initOS(&*osname, &*osversion);

    if (cfg_sendcpu)
        initCPU(&*cpu);

   /* WARNING: loadavg is not implemented on Wonko server! */
   /* cfg_SendUsage is implemented instead. (i.e. only CPU usage % a la `top`) */
   /* if (cfg_sendloadavg) */
    getLoadavg(&*loadavg);

}
#endif /* PLATFORM_BSD */
