#if canImport(TuistCloud)
    import Foundation
    import TSCBasic
    import TuistCloud
    import TuistCore
    import TuistCoreTesting
    import TuistGraph
    import TuistGraphTesting
    import TuistLoader
    import TuistLoaderTesting
    import TuistSupport
    import XCTest

    @testable import TuistCloudTesting
    @testable import TuistCore
    @testable import TuistKit
    @testable import TuistSupportTesting

    final class CacheControllerTests: TuistUnitTestCase {
        var generator: MockGenerator!
        var focusedGenerator: MockGenerator!
        var generatorFactory: MockGeneratorFactory!
        var cacheGraphContentHasher: MockCacheGraphContentHasher!
        var artifactBuilder: MockCacheArtifactBuilder!
        var bundleArtifactBuilder: MockCacheArtifactBuilder!
        var manifestLoader: MockManifestLoader!
        var cache: MockCacheStorage!
        var subject: CacheController!
        var config: Config!
        var cacheGraphLinter: MockCacheGraphLinter!

        override func setUp() {
            generatorFactory = MockGeneratorFactory()
            generator = MockGenerator()
            focusedGenerator = MockGenerator()
            generatorFactory.stubbedCacheResult = generator
            generatorFactory.stubbedFocusResult = focusedGenerator
            artifactBuilder = MockCacheArtifactBuilder()
            bundleArtifactBuilder = MockCacheArtifactBuilder()
            cache = MockCacheStorage()
            manifestLoader = MockManifestLoader()
            cacheGraphContentHasher = MockCacheGraphContentHasher()
            config = .test()
            cacheGraphLinter = MockCacheGraphLinter()
            subject = CacheController(
                cache: cache,
                artifactBuilder: artifactBuilder,
                bundleArtifactBuilder: bundleArtifactBuilder,
                generatorFactory: generatorFactory,
                cacheGraphContentHasher: cacheGraphContentHasher,
                cacheGraphLinter: cacheGraphLinter
            )

            super.setUp()
        }

        override func tearDown() {
            generator = nil
            focusedGenerator = nil
            generatorFactory = nil
            artifactBuilder = nil
            bundleArtifactBuilder = nil
            cacheGraphContentHasher = nil
            manifestLoader = nil
            cache = nil
            subject = nil
            config = nil
            super.tearDown()
        }

        func test_cache_builds_and_caches_the_frameworks() async throws {
            // Given
            let path = try temporaryPath()
            let xcworkspacePath = path.appending(component: "Project.xcworkspace")
            let project = Project.test(path: path, name: "Cache")
            let targetNames = ["foo", "bar", "baz"].shuffled()
            let aTarget = Target.test(name: targetNames[0])
            let bTarget = Target.test(name: targetNames[1])
            let cTarget = Target.test(name: targetNames[2])
            let targetReferences = [aTarget, bTarget, cTarget]
                .map { TargetReference(projectPath: xcworkspacePath, name: $0.name) }
            let scheme = Scheme(
                name: "\(Constants.AutogeneratedScheme.binariesSchemeNamePrefix)-iOS",
                buildAction: .test(targets: targetReferences)
            )

            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let cGraphTarget = GraphTarget.test(path: project.path, target: cTarget, project: project)
            let nodeWithHashes = [
                aGraphTarget: "\(aTarget.name)_HASH",
                bGraphTarget: "\(bTarget.name)_HASH",
                cGraphTarget: "\(cTarget.name)_HASH",
            ]
            let graph = Graph.test(
                workspace: .test(schemes: [scheme]),
                projects: [project.path: project],
                targets: nodeWithHashes.keys
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    .target(name: bGraphTarget.target.name, path: bGraphTarget.path): [
                        .target(name: aGraphTarget.target.name, path: aGraphTarget.path),
                    ],
                    .target(name: cGraphTarget.target.name, path: cGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                ]
            )

            manifestLoader.manifestsAtStub = { (loadPath: AbsolutePath) -> Set<Manifest> in
                XCTAssertEqual(loadPath, path)
                return Set(arrayLiteral: .project)
            }
            generator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            focusedGenerator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            cacheGraphContentHasher.contentHashesStub = { _, _, _, _ in
                nodeWithHashes
            }
            artifactBuilder.stubbedCacheOutputType = .xcframework([.device, .simulator])

            // When
            try await subject.cache(
                config: .test(),
                path: path,
                cacheProfile: .test(configuration: "Debug"),
                includedTargets: [],
                dependenciesOnly: false
            )

            // Then
            let targetsToBeCached = "bar, baz, foo"
            XCTAssertPrinterOutputContains("""
            Hashing cacheable targets
            Targets to be cached: \(targetsToBeCached)
            Building cacheable targets
            Storing 3 cacheable targets: \(targetsToBeCached)
            All cacheable targets have been cached successfully as xcframeworks
            """)
            XCTAssertEqual(cacheGraphLinter.invokedLintCount, 1)
            XCTAssertEqual(artifactBuilder.invokedBuildSchemeProjectCount, 1)
            XCTAssertEqual(artifactBuilder.invokedBuildSchemeProjectParameters?.scheme, scheme)
        }

        func test_cache_when_cache_fails_throws() async throws {
            // Given
            let path = try temporaryPath()
            let xcworkspacePath = path.appending(component: "Project.xcworkspace")
            let project = Project.test(path: path, name: "Cache")
            let targetNames = ["foo", "bar", "baz"].shuffled()
            let aTarget = Target.test(name: targetNames[0])
            let bTarget = Target.test(name: targetNames[1])
            let cTarget = Target.test(name: targetNames[2])
            let targetReferences = [aTarget, bTarget, cTarget]
                .map { TargetReference(projectPath: xcworkspacePath, name: $0.name) }
            let scheme = Scheme(
                name: "\(Constants.AutogeneratedScheme.binariesSchemeNamePrefix)-iOS",
                buildAction: .test(targets: targetReferences)
            )

            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let cGraphTarget = GraphTarget.test(path: project.path, target: cTarget, project: project)
            let nodeWithHashes = [
                aGraphTarget: "\(aTarget.name)_HASH",
                bGraphTarget: "\(bTarget.name)_HASH",
                cGraphTarget: "\(cTarget.name)_HASH",
            ]
            let graph = Graph.test(
                workspace: .test(schemes: [scheme]),
                projects: [project.path: project],
                targets: nodeWithHashes.keys
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    .target(name: bGraphTarget.target.name, path: bGraphTarget.path): [
                        .target(name: aGraphTarget.target.name, path: aGraphTarget.path),
                    ],
                    .target(name: cGraphTarget.target.name, path: cGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                ]
            )

            manifestLoader.manifestsAtStub = { (loadPath: AbsolutePath) -> Set<Manifest> in
                XCTAssertEqual(loadPath, path)
                return Set(arrayLiteral: .project)
            }
            generator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            focusedGenerator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            cacheGraphContentHasher.contentHashesStub = { _, _, _, _ in
                nodeWithHashes
            }
            artifactBuilder.stubbedCacheOutputType = .xcframework([.device, .simulator])

            let remoteCacheError = TestError("remote cache error")
            cache.existsStub = { _, _ in throw remoteCacheError }
            // When / Then
            await XCTAssertThrowsSpecific(
                try await subject.cache(
                    config: .test(),
                    path: path,
                    cacheProfile: .test(configuration: "Debug"),
                    includedTargets: [],
                    dependenciesOnly: false
                ),
                remoteCacheError
            )
        }

        func test_cache_early_exit_if_nothing_to_cache() async throws {
            // Given
            let path = try temporaryPath()
            let xcworkspacePath = path.appending(component: "Project.xcworkspace")
            let project = Project.test(path: path, name: "Cache")
            let targetNames = ["foo", "bar", "baz"].shuffled()
            let aTarget = Target.test(name: targetNames[0])
            let bTarget = Target.test(name: targetNames[1])
            let cTarget = Target.test(name: targetNames[2])
            let targetReferences = [aTarget, bTarget, cTarget]
                .map { TargetReference(projectPath: xcworkspacePath, name: $0.name) }
            let scheme = Scheme(
                name: "\(Constants.AutogeneratedScheme.binariesSchemeNamePrefix)-iOS",
                buildAction: .test(targets: targetReferences)
            )

            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let cGraphTarget = GraphTarget.test(path: project.path, target: cTarget, project: project)
            let nodeWithHashes = [
                aGraphTarget: "\(aTarget.name)_HASH",
                bGraphTarget: "\(bTarget.name)_HASH",
                cGraphTarget: "\(cTarget.name)_HASH",
            ]
            let graph = Graph.test(
                workspace: .test(schemes: [scheme]),
                projects: [project.path: project],
                targets: nodeWithHashes.keys
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    .target(name: bGraphTarget.target.name, path: bGraphTarget.path): [
                        .target(name: aGraphTarget.target.name, path: aGraphTarget.path),
                    ],
                    .target(name: cGraphTarget.target.name, path: cGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                ]
            )

            cache.existsStub = { _, _ in
                true
            }
            manifestLoader.manifestsAtStub = { _ in
                XCTFail("ManifestLoader should not be invoked")
                return Set(arrayLiteral: .project)
            }
            generator.generateWithGraphStub = { loadPath in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            focusedGenerator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            cacheGraphContentHasher.contentHashesStub = { _, _, _, _ in
                nodeWithHashes
            }
            artifactBuilder.stubbedCacheOutputType = .xcframework([.device, .simulator])

            // When
            try await subject.cache(
                config: .test(),
                path: path,
                cacheProfile: .test(configuration: "Debug"),
                includedTargets: [],
                dependenciesOnly: false
            )

            // Then
            XCTAssertPrinterOutputContains("All cacheable targets are already cached")
            XCTAssertEqual(cacheGraphLinter.invokedLintCount, 1)
        }

        func test_filtered_cache_builds_and_caches_the_frameworks() async throws {
            // Given
            let path = try temporaryPath()
            let xcworkspacePath = path.appending(component: "Project.xcworkspace")
            let project = Project.test(path: path, name: "Cache")
            let targetNames = ["foo", "bar", "baz"].shuffled()
            let aTarget = Target.test(name: targetNames[0])
            let bTarget = Target.test(name: targetNames[1])
            let cTarget = Target.test(name: targetNames[2])
            let targetReferences = [aTarget, bTarget, cTarget]
                .map { TargetReference(projectPath: xcworkspacePath, name: $0.name) }
            let scheme = Scheme(
                name: "\(Constants.AutogeneratedScheme.binariesSchemeNamePrefix)-iOS",
                buildAction: .test(targets: targetReferences)
            )

            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let cGraphTarget = GraphTarget.test(path: project.path, target: cTarget, project: project)
            let nodeWithHashes = [
                aGraphTarget: "\(aTarget.name)_HASH",
                bGraphTarget: "\(bTarget.name)_HASH",
                cGraphTarget: "\(cTarget.name)_HASH",
            ]
            let filteredNodeWithHashes = [
                aGraphTarget: "\(aTarget.name)_HASH",
                bGraphTarget: "\(bTarget.name)_HASH",
            ]
            let graph = Graph.test(
                workspace: .test(schemes: [scheme]),
                projects: [project.path: project],
                targets: nodeWithHashes.keys
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    .target(name: bGraphTarget.target.name, path: bGraphTarget.path): [
                        .target(name: aGraphTarget.target.name, path: aGraphTarget.path),
                    ],
                    .target(name: cGraphTarget.target.name, path: cGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                ]
            )

            manifestLoader.manifestsAtStub = { (loadPath: AbsolutePath) -> Set<Manifest> in
                XCTAssertEqual(loadPath, path)
                return Set(arrayLiteral: .project)
            }
            generator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            focusedGenerator.generateWithGraphStub = { loadPath -> (AbsolutePath, Graph) in
                XCTAssertEqual(loadPath, path)
                return (xcworkspacePath, graph)
            }
            cacheGraphContentHasher.contentHashesStub = { _, _, _, _ in
                filteredNodeWithHashes
            }
            artifactBuilder.stubbedCacheOutputType = .xcframework([.device, .simulator])

            // When
            try await subject.cache(
                config: .test(),
                path: path,
                cacheProfile: .test(configuration: "Debug"),
                includedTargets: [bTarget.name],
                dependenciesOnly: false
            )

            // Then
            let targetsToBeCached = [aTarget.name, bTarget.name].sorted().joined(separator: ", ")
            XCTAssertPrinterOutputContains("""
            Hashing cacheable targets
            Targets to be cached: \(targetsToBeCached)
            Building cacheable targets
            Storing 2 cacheable targets: \(targetsToBeCached)
            All cacheable targets have been cached successfully as xcframeworks
            """)
            XCTAssertEqual(cacheGraphLinter.invokedLintCount, 1)
            XCTAssertEqual(artifactBuilder.invokedBuildSchemeProjectCount, 1)
            XCTAssertEqual(artifactBuilder.invokedBuildSchemeProjectParameters?.scheme, scheme)
        }

        func test_filtered_cache_builds_with_dependencies_only_and_caches_the_frameworks() async throws {
            // Given
            let project = Project.test()
            let aTarget = Target.test(name: "a")
            let bTarget = Target.test(name: "b")
            let cTarget = Target.test(name: "c")
            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let cGraphTarget = GraphTarget.test(path: project.path, target: cTarget, project: project)
            let graphTargets = [aGraphTarget, bGraphTarget, cGraphTarget]
            let graph = Graph.test(
                projects: [project.path: project],
                targets: graphTargets
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    // `bTarget` is a dependency of `aTarget`.
                    .target(name: aGraphTarget.target.name, path: aGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                    // `cTarget` is a dependency of `bTarget`.
                    .target(name: bGraphTarget.target.name, path: bGraphTarget.path): [
                        .target(name: cGraphTarget.target.name, path: cGraphTarget.path),
                    ],
                ]
            )

            let nodeWithHashes = [
                cGraphTarget: "\(cTarget.name)_HASH",
            ]
            cacheGraphContentHasher.contentHashesStub = { _, _, _, excludedTargets in
                // a and b should be excluded due to dependenciesOnly flag
                XCTAssertEqual(excludedTargets, [aTarget.name, bTarget.name])
                return nodeWithHashes
            }

            artifactBuilder.stubbedCacheOutputType = .xcframework([.device, .simulator])

            // When
            let results = try await subject.makeHashesByTargetToBeCached(
                for: graph,
                cacheProfile: .test(),
                cacheOutputType: .framework,
                includedTargets: [aTarget.name, bTarget.name],
                dependenciesOnly: true
            )

            // Then
            XCTAssertEqual(results.count, 1)
            let first = try XCTUnwrap(results.first)
            XCTAssertEqual(first.0, cGraphTarget)
            XCTAssertEqual(first.1, nodeWithHashes[cGraphTarget])
        }

        func test_given_target_to_filter_is_not_cacheable_should_cache_its_depedendencies() async throws {
            // Given
            let project = Project.test()
            let aTarget = Target.test(name: "a")
            let bTarget = Target.test(name: "b")
            let aGraphTarget = GraphTarget.test(path: project.path, target: aTarget, project: project)
            let bGraphTarget = GraphTarget.test(path: project.path, target: bTarget, project: project)
            let graph = Graph.test(
                projects: [project.path: project],
                targets: [aGraphTarget, bGraphTarget]
                    .reduce(into: [project.path: [String: Target]()]) { $0[project.path]?[$1.target.name] = $1.target },
                dependencies: [
                    // `bTarget` is a dependency of `aTarget`.
                    .target(name: aGraphTarget.target.name, path: aGraphTarget.path): [
                        .target(name: bGraphTarget.target.name, path: bGraphTarget.path),
                    ],
                ]
            )

            // `aTarget` is not cacheable, but `bTarget` (its dependency) it is.
            let nodeWithHashes = [
                bGraphTarget: "\(bTarget.name)_HASH",
            ]
            cacheGraphContentHasher.contentHashesStub = { _, _, _, _ in
                nodeWithHashes
            }

            // When
            let results = try await subject.makeHashesByTargetToBeCached(
                for: graph,
                cacheProfile: .test(),
                cacheOutputType: .xcframework([.device, .simulator]),
                includedTargets: [aTarget.name],
                dependenciesOnly: false
            )

            // Then
            XCTAssertEqual(results.count, 1)
            let first = try XCTUnwrap(results.first)
            XCTAssertEqual(first.0, bGraphTarget)
            XCTAssertEqual(first.1, nodeWithHashes[bGraphTarget])
        }
    }
#endif
