import TSCUtility

extension Workspace {
    /// Contains options related to the workspace generation.
    public struct GenerationOptions: Codable, Equatable {
        /// Contains options for autogenerated workspace schemes
        public enum AutogeneratedWorkspaceSchemes: Codable, Equatable {
            /// Contains options for code coverage
            public enum CodeCoverageMode: Codable, Equatable {
                /// Gather code coverage for all targets
                case all
                /// Gather code coverage only for targets, that have code coverage enabled at least in one scheme
                case relevant
                /// Gather code coverage only for given targets
                case targets([TargetReference])
                /// Do not gather code coverage
                case disabled
            }

            /// Tuist will not automatically generate the workspace schemes
            case disabled
            /// Tuist will generate the workspace schemes
            case enabled(
                codeCoverageMode: CodeCoverageMode = .disabled,
                testingOptions: TestingOptions = [],
                testLanguage: String? = nil,
                testRegion: String? = nil,
                testScreenCaptureFormat: ScreenCaptureFormat? = nil
            )

            public var codeCoverageMode: CodeCoverageMode {
                switch self {
                case let .enabled(codeCoverageMode, _, _, _, _):
                    return codeCoverageMode
                case .disabled:
                    return .disabled
                }
            }

            public var testingOptions: TestingOptions {
                switch self {
                case let .enabled(_, testingOptions, _, _, _):
                    return testingOptions
                case .disabled:
                    return []
                }
            }

            public var testLanguage: String? {
                switch self {
                case let .enabled(_, _, language, _, _):
                    return language
                case .disabled:
                    return nil
                }
            }

            public var testRegion: String? {
                switch self {
                case let .enabled(_, _, _, region, _):
                    return region
                case .disabled:
                    return nil
                }
            }

            public var testScreenCaptureFormat: ScreenCaptureFormat? {
                switch self {
                case let .enabled(_, _, _, _, testScreenCaptureFormat):
                    return testScreenCaptureFormat
                case .disabled:
                    return nil
                }
            }
        }

        /// Tuist generates a WorkspaceSettings.xcsettings file, setting the related key to the associated value.
        public let enableAutomaticXcodeSchemes: Bool?

        /// Options for autogenerated workspace schemes
        public let autogeneratedWorkspaceSchemes: AutogeneratedWorkspaceSchemes

        /// Allows to suppress warnings in Xcode about updates to recommended settings added in or below the specified Xcode
        /// version. The warnings appear when Xcode version has been upgraded.
        /// It is recommended to set the version option to Xcode's version that is used for development of a project, for example
        /// `.lastXcodeUpgradeCheck(Version(13, 0, 0))` for Xcode 13.0.0.
        public let lastXcodeUpgradeCheck: Version?

        /// Allows to render markdown files inside the workspace including an .xcodesamples.plist inside it.
        public let renderMarkdownReadme: Bool

        public init(
            enableAutomaticXcodeSchemes: Bool?,
            autogeneratedWorkspaceSchemes: AutogeneratedWorkspaceSchemes,
            lastXcodeUpgradeCheck: Version?,
            renderMarkdownReadme: Bool
        ) {
            self.enableAutomaticXcodeSchemes = enableAutomaticXcodeSchemes
            self.autogeneratedWorkspaceSchemes = autogeneratedWorkspaceSchemes
            self.lastXcodeUpgradeCheck = lastXcodeUpgradeCheck
            self.renderMarkdownReadme = renderMarkdownReadme
        }
    }
}
