//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2005-2020, Thierry Lelegard
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------

#include "tsSelectionInformationTable.h"
#include "tsRST.h"
#include "tsNames.h"
#include "tsBinaryTable.h"
#include "tsStreamIdentifierDescriptor.h"
#include "tsTablesDisplay.h"
#include "tsPSIRepository.h"
#include "tsPSIBuffer.h"
#include "tsDuckContext.h"
#include "tsxmlElement.h"
TSDUCK_SOURCE;

#define MY_XML_NAME u"selection_information_table"
#define MY_CLASS ts::SelectionInformationTable
#define MY_TID ts::TID_SIT
#define MY_PID ts::PID_SIT
#define MY_STD ts::Standards::DVB

TS_REGISTER_TABLE(MY_CLASS, {MY_TID}, MY_STD, MY_XML_NAME, MY_CLASS::DisplaySection, nullptr, {MY_PID});


//----------------------------------------------------------------------------
// Constructors
//----------------------------------------------------------------------------

ts::SelectionInformationTable::SelectionInformationTable(uint8_t version_, bool is_current_) :
    AbstractLongTable(MY_TID, MY_XML_NAME, MY_STD, version_, is_current_),
    descs(this),
    services(this)
{
}

ts::SelectionInformationTable::SelectionInformationTable(const SelectionInformationTable& other) :
    AbstractLongTable(other),
    descs(this, other.descs),
    services(this, other.services)
{
}

ts::SelectionInformationTable::SelectionInformationTable(DuckContext& duck, const BinaryTable& table) :
    SelectionInformationTable()
{
    deserialize(duck, table);
}

ts::SelectionInformationTable::Service::Service(const AbstractTable* table, uint8_t status) :
    EntryWithDescriptors(table),
    running_status(status)
{
}


//----------------------------------------------------------------------------
// Get the table id extension.
//----------------------------------------------------------------------------

uint16_t ts::SelectionInformationTable::tableIdExtension() const
{
    return 0xFFFF;
}


//----------------------------------------------------------------------------
// Clear the content of the table.
//----------------------------------------------------------------------------

void ts::SelectionInformationTable::clearContent()
{
    descs.clear();
    services.clear();
}


//----------------------------------------------------------------------------
// Deserialization
//----------------------------------------------------------------------------

void ts::SelectionInformationTable::deserializePayload(PSIBuffer& buf, const Section& section)
{
    buf.getDescriptorListWithLength(descs);
    while (!buf.error() && !buf.endOfRead()) {
        Service& srv(services[buf.getUInt16()]);
        buf.skipBits(1);
        srv.running_status = buf.getBits<uint8_t>(3);
        buf.getDescriptorListWithLength(srv.descs);
    }
}


//----------------------------------------------------------------------------
// Serialization
//----------------------------------------------------------------------------

void ts::SelectionInformationTable::serializePayload(BinaryTable& table, PSIBuffer& buf) const
{
    // A Selection Information Table is not allowed to use more than one section, see ETSI EN 300 468, 7.1.2.

    buf.putPartialDescriptorListWithLength(descs);
    for (auto it = services.begin(); !buf.error() && it != services.end(); ++it) {
        buf.putUInt16(it->first); // service id
        buf.putBit(1);
        buf.putBits(it->second.running_status, 3);
        buf.putPartialDescriptorListWithLength(it->second.descs);
    }
}


//----------------------------------------------------------------------------
// A static method to display a SelectionInformationTable section.
//----------------------------------------------------------------------------

void ts::SelectionInformationTable::DisplaySection(TablesDisplay& disp, const ts::Section& section, PSIBuffer& buf, const UString& margin)
{
    disp.displayDescriptorListWithLength(section, buf, margin, u"Global information:");
    while (!buf.error() && buf.remainingReadBytes() >= 4) {
        disp << margin << UString::Format(u"Service id: %d (0x%<X)", {buf.getUInt16()});
        buf.skipBits(1);
        disp << ", Status: " << RST::RunningStatusNames.name(buf.getBits<uint8_t>(3)) << std::endl;
        disp.displayDescriptorListWithLength(section, buf, margin);
    }
    disp.displayExtraData(buf, margin);
}


//----------------------------------------------------------------------------
// XML serialization
//----------------------------------------------------------------------------

void ts::SelectionInformationTable::buildXML(DuckContext& duck, xml::Element* root) const
{
    root->setIntAttribute(u"version", version);
    root->setBoolAttribute(u"current", is_current);
    descs.toXML(duck, root);

    for (auto it = services.begin(); it != services.end(); ++it) {
        xml::Element* e = root->addElement(u"service");
        e->setIntAttribute(u"service_id", it->first, true);
        e->setEnumAttribute(RST::RunningStatusNames, u"running_status", it->second.running_status);
        it->second.descs.toXML(duck, e);
    }
}


//----------------------------------------------------------------------------
// XML deserialization
//----------------------------------------------------------------------------

bool ts::SelectionInformationTable::analyzeXML(DuckContext& duck, const xml::Element* element)
{
    xml::ElementVector children;
    bool ok =
        element->getIntAttribute<uint8_t>(version, u"version", false, 0, 0, 31) &&
        element->getBoolAttribute(is_current, u"current", false, true) &&
        descs.fromXML(duck, children, element, u"service");

    for (size_t index = 0; ok && index < children.size(); ++index) {
        uint16_t id = 0;
        ok = children[index]->getIntAttribute<uint16_t>(id, u"service_id", true) &&
             children[index]->getIntEnumAttribute<uint8_t>(services[id].running_status, RST::RunningStatusNames, u"running_status", true);
             services[id].descs.fromXML(duck, children[index]);
    }
    return ok;
}
