//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2005-2020, Thierry Lelegard
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------

#include "tsERT.h"
#include "tsBinaryTable.h"
#include "tsTablesDisplay.h"
#include "tsPSIRepository.h"
#include "tsPSIBuffer.h"
#include "tsDuckContext.h"
#include "tsxmlElement.h"
#include "tsNames.h"
TSDUCK_SOURCE;

#define MY_XML_NAME u"ERT"
#define MY_CLASS ts::ERT
#define MY_TID ts::TID_ERT
#define MY_PID ts::PID_ERT
#define MY_STD ts::Standards::ISDB

TS_REGISTER_TABLE(MY_CLASS, {MY_TID}, MY_STD, MY_XML_NAME, MY_CLASS::DisplaySection, nullptr, {MY_PID});


//----------------------------------------------------------------------------
// Constructors.
//----------------------------------------------------------------------------

ts::ERT::ERT(uint8_t vers, bool cur) :
    AbstractLongTable(MY_TID, MY_XML_NAME, MY_STD, vers, cur),
    event_relation_id(0),
    information_provider_id(0),
    relation_type(0),
    relations(this)
{
}

ts::ERT::ERT(const ERT& other) :
    AbstractLongTable(other),
    event_relation_id(other.event_relation_id),
    information_provider_id(other.information_provider_id),
    relation_type(other.relation_type),
    relations(this, other.relations)
{
}

ts::ERT::ERT(DuckContext& duck, const BinaryTable& table) :
    ERT()
{
    deserialize(duck, table);
}

ts::ERT::Relation::Relation(const AbstractTable* table) :
    EntryWithDescriptors(table),
    node_id(0),
    collection_mode(0),
    parent_node_id(0),
    reference_number(0)
{
}


//----------------------------------------------------------------------------
// Get the table id extension.
//----------------------------------------------------------------------------

uint16_t ts::ERT::tableIdExtension() const
{
    return event_relation_id;
}


//----------------------------------------------------------------------------
// Clear the content of the table.
//----------------------------------------------------------------------------

void ts::ERT::clearContent()
{
    event_relation_id = 0;
    information_provider_id = 0;
    relation_type = 0;
    relations.clear();
}


//----------------------------------------------------------------------------
// Deserialization
//----------------------------------------------------------------------------

void ts::ERT::deserializePayload(PSIBuffer& buf, const Section& section)
{
    // Get common properties (should be identical in all sections)
    event_relation_id = section.tableIdExtension();
    information_provider_id = buf.getUInt16();
    relation_type = buf.getBits<uint8_t>(4);
    buf.skipBits(4);

    // Loop across all relations.
    while (!buf.error() && !buf.endOfRead()) {
        Relation& rel(relations.newEntry());
        rel.node_id = buf.getUInt16();
        rel.collection_mode = buf.getBits<uint8_t>(4);
        buf.skipBits(4);
        rel.parent_node_id = buf.getUInt16();
        rel.reference_number = buf.getUInt8();
        buf.getDescriptorListWithLength(rel.descs);
    }
}


//----------------------------------------------------------------------------
// Serialization
//----------------------------------------------------------------------------

void ts::ERT::serializePayload(BinaryTable& table, PSIBuffer& buf) const
{
    // Fixed part, to be repeated on all sections.
    buf.putUInt16(information_provider_id);
    buf.putBits(relation_type, 4);
    buf.putBits(0xFF, 4);
    buf.pushState();

    // Minimum payload size, before loop of local events.
    const size_t payload_min_size = buf.currentWriteByteOffset();

    // Add all relations.
    for (auto it = relations.begin(); it != relations.end(); ++it) {
        const Relation& rel(it->second);

        // Binary size of this entry.
        const size_t entry_size = 8 + rel.descs.binarySize();

        // If we are not at the beginning of the relations loop, make sure that the entire
        // description fits in the section. If it does not fit, start a new section.
        if (entry_size > buf.remainingWriteBytes() && buf.currentWriteByteOffset() > payload_min_size) {
            // Create a new section.
            addOneSection(table, buf);
        }

        // Serialize the relation entry. If the descriptor loop is too long, it is truncated.
        buf.putUInt16(rel.node_id);
        buf.putBits(rel.collection_mode, 4);
        buf.putBits(0xFF, 4);
        buf.putUInt16(rel.parent_node_id);
        buf.putUInt8(rel.reference_number);
        buf.putPartialDescriptorListWithLength(rel.descs);
    }
}


//----------------------------------------------------------------------------
// A static method to display a ERT section.
//----------------------------------------------------------------------------

void ts::ERT::DisplaySection(TablesDisplay& disp, const ts::Section& section, PSIBuffer& buf, const UString& margin)
{
    disp << margin << UString::Format(u"Event relation id: 0x%X (%<d)", {section.tableIdExtension()}) << std::endl;

    if (buf.remainingReadBytes() < 3) {
        buf.setUserError();
    }
    else {
        disp << margin << UString::Format(u"Information provider id: 0x%X (%<d)", {buf.getUInt16()}) << std::endl;
        disp << margin << "Relation type: " << NameFromSection(u"ISDBRelationType", buf.getBits<uint8_t>(4), names::DECIMAL_FIRST) << std::endl;
        buf.skipBits(4);
    }

    // Loop across all relations.
    while (!buf.error() && buf.remainingReadBytes() >= 8) {
        disp << margin << UString::Format(u"- Node id: 0x%X (%<d)", {buf.getUInt16()}) << std::endl;
        disp << margin << "  Collection mode: " << NameFromSection(u"ISDBCollectionMode", buf.getBits<uint8_t>(4), names::DECIMAL_FIRST) << std::endl;
        buf.skipBits(4);
        disp << margin << UString::Format(u"  Parent node id: 0x%X (%<d)", {buf.getUInt16()}) << std::endl;
        disp << margin << UString::Format(u"  Reference number: 0x%X (%<d)", {buf.getUInt8()}) << std::endl;
        disp.displayDescriptorListWithLength(section, buf, margin + u"  ");
    }

    disp.displayExtraData(buf, margin);
}


//----------------------------------------------------------------------------
// XML serialization
//----------------------------------------------------------------------------

void ts::ERT::buildXML(DuckContext& duck, xml::Element* root) const
{
    root->setIntAttribute(u"version", version);
    root->setBoolAttribute(u"current", is_current);
    root->setIntAttribute(u"event_relation_id", event_relation_id, true);
    root->setIntAttribute(u"information_provider_id", information_provider_id, true);
    root->setIntAttribute(u"relation_type", relation_type);

    for (auto it = relations.begin(); it != relations.end(); ++it) {
        xml::Element* e = root->addElement(u"relation");
        e->setIntAttribute(u"node_id", it->second.node_id, true);
        e->setIntAttribute(u"collection_mode", it->second.collection_mode);
        e->setIntAttribute(u"parent_node_id", it->second.parent_node_id, true);
        e->setIntAttribute(u"reference_number", it->second.reference_number, true);
        it->second.descs.toXML(duck, e);
    }
}


//----------------------------------------------------------------------------
// XML deserialization
//----------------------------------------------------------------------------

bool ts::ERT::analyzeXML(DuckContext& duck, const xml::Element* element)
{
    xml::ElementVector xrel;
    bool ok =
        element->getIntAttribute<uint8_t>(version, u"version", false, 0, 0, 31) &&
        element->getBoolAttribute(is_current, u"current", false, true) &&
        element->getIntAttribute<uint16_t>(event_relation_id, u"event_relation_id", true) &&
        element->getIntAttribute<uint16_t>(information_provider_id, u"information_provider_id", true) &&
        element->getIntAttribute<uint8_t>(relation_type, u"relation_type", true, 0, 0, 15) &&
        element->getChildren(xrel, u"relation");

    for (auto it = xrel.begin(); ok && it != xrel.end(); ++it) {
        Relation& rel(relations.newEntry());
        ok = (*it)->getIntAttribute<uint16_t>(rel.node_id, u"node_id", true) &&
             (*it)->getIntAttribute<uint8_t>(rel.collection_mode, u"collection_mode", true, 0, 0, 15) &&
             (*it)->getIntAttribute<uint16_t>(rel.parent_node_id, u"parent_node_id", true) &&
             (*it)->getIntAttribute<uint8_t>(rel.reference_number, u"reference_number", true) &&
             rel.descs.fromXML(duck, *it);
    }
    return ok;
}
