/*
 *  This file is part of timespan.
 *
 *  Timespan is the legal property of its authors.  Refer to the file
 *  AUTHORS in the top-level directory of the source distribution for
 *  details.
 *  
 *  Timespan is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Timespan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with timespan; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "timespan.hpp"

void args_parse_driver(const int argc, char *argv[])
{ //call functions to parse different parts of arguments
	//private helper functions
	void args_parse_for_flags(const int, char * [], Flag &);
	void args_parse_for_ops(const int, char * [], const Flag);
	
	if(argc < 2) //2 or more args needed to do something useful
		usage(); //explain lack of arguments
	else //parse args
	{
		Flag flags = {false,false,false,false,false,false,false,true,true};
		args_parse_for_flags(argc, argv, flags);
		args_parse_for_ops(argc, argv, flags);
	}
	return; //return control to calling function
}

void args_parse_for_ops(const int argc, char *argv[], const Flag flags)
{ //parses arguments for specific operations and calls helper functions
	//private helper functions
	void args_parse_epoch(const int, char * [], int &, const Flag);
	void args_parse_now(const int, char * [], int &, const Flag);
	void args_parse_between(const int, char * [], int &, const Flag);
	
	for(int i = 1; i < argc; i++)
	{
		//check for version option
		if(!strcmp(argv[i], "-v") || !strcmp(argv[i], "--version"))
			version();
		//checks for help/usage option
		else if(!strcmp(argv[i], "-H") || !strcmp(argv[i], "--help"))
			usage();
		//check for between
		else if(!strcmp(argv[i], "between"))
			args_parse_between(argc, argv, i, flags);
		//check for epoch
		else if(!strcmp(argv[i], "epoch"))
			args_parse_epoch(argc, argv, i, flags);
		//check for now
		else if(!strcmp(argv[i], "now"))
			args_parse_now(argc, argv, i, flags);
	}

	return; //return control to calling function
}

void args_parse_for_flags(const int argc, char *argv[], Flag &flags)
{
	unsigned int flag_counter = 0; //counter for tracking number of answer flags
	
	for(int i = 1; i < argc; i++)
	{
		//check for non-US I/O option
		if(!strcmp(argv[i], "-ee") || !strcmp(argv[i], "--nous"))
		{
			flags.usin = false;
			flags.usout = false;
		}
		//checks for non-US date input format option
		else if(!strcmp(argv[i], "-ei") || !strcmp(argv[i], "--nousin"))
			flags.usin = false;
		//checks for non-US date output format option
		else if(!strcmp(argv[i], "-eo") || !strcmp(argv[i], "--nousout"))
			flags.usout = false;
		//checks for extra line option--a blank line is left after all ops finish
		else if(!strcmp(argv[i], "-e") || !strcmp(argv[i], "--extraline"))
			flags.extraline = true;
		//check for all option--this allows answers in days, hours, minutes, and
		//seconds to be given simultaneously
		else if(!strcmp(argv[i], "-a") || !strcmp(argv[i], "--all"))
		{
			flags.wantdate = true;
			flags.wantdays = true;
			flags.wanthours = true;
			flags.wantminutes = true;
			flags.wantseconds = true;
			flag_counter++;
		}
		//check for long option--when a date is printed, use a word for the month
		else if(!strcmp(argv[i], "-l") || !strcmp(argv[i], "--long"))
			flags.wantlongdate = true;
		//check for date option--specify one answer wanted is a date
		else if(!strcmp(argv[i], "-t") || !strcmp(argv[i], "--date"))
		{
			flags.wantdate = true;
			flag_counter++;
		}
		//check for days option--specify one answer wanted is a number of days
		else if(!strcmp(argv[i], "-d") || !strcmp(argv[i], "--days"))
		{
			flags.wantdays = true;
			flag_counter++;
		}
		//check for hours option--specify one answer wanted is a number of hours
		else if(!strcmp(argv[i], "-h") || !strcmp(argv[i], "--hours"))
		{
			flags.wanthours = true;
			flag_counter++;
		}
		//check for minutes option--specify one answer wanted is a number
		//of minutes
		else if(!strcmp(argv[i], "-m") || !strcmp(argv[i], "--minutes"))
		{
			flags.wantminutes = true;
			flag_counter++;
		}
		//check for seconds option--specify one answer wanted is a number
		//of seconds
		else if(!strcmp(argv[i], "-s") || !strcmp(argv[i], "--seconds"))
		{
			flags.wantseconds = true;
			flag_counter++;
		}
	}

	//if no answer flags specified, force a default of wanting only a number
	//of days
	if(flag_counter == 0)
		flags.wantdays = true;

	return; //return cpntrol to calling function
}

void args_parse_epoch(const int argc, char *argv[], int &i,
							 const Flag flags)
{ //parse the args after "epoch" and do some work
	/*
	 * this first compound test covers the cases where "epoch" is:
	 *	 1.  the only option on the command line
	 *	 2.  the last option on the command line
	 *	 3.  in the middle of a command line, but has no arguments following
	 *		  that belong to it
	 */
	if((i + 1 == argc) || ((i + 1 < argc) && strcmp(argv[i + 1], "+")))
	{
		Date epoch = { 1, 1, 1970 };
		print_date(epoch, flags);
		
		if(flags.extraline)
			cout << endl;
	}
	else if(i + 3 < argc) //3 more args needed to complete calculations
	{
		if(!strcmp(argv[i + 1], "+")) //make sure user wants to add
		{
			uint_least64_t temp = static_cast<uint_least64_t>(atol(argv[i + 2])),
								temp_days = 0, temp_hours = 0, temp_minutes = 0,
								temp_seconds = 0;
			Date temp_date = { 0, 0, 0 };
			
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in days and must be converted to other
			 * units.
			 */
			if(!strcmp(argv[i + 3], "days"))
			{
				if(temp >= 1)
				{
					temp_date = calcdate(temp);
					temp_days = temp;
					temp_hours = temp * hours_in_day;
					temp_minutes = temp * minutes_in_day;
					temp_seconds = temp * seconds_in_day;
				}
				else
				{
					cout << "Please try adding 1 or more days." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in hours and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "hours"))
			{
				if(temp >= hours_in_day)
				{
					temp_date = calcdate(temp / hours_in_day);
					temp_days = temp / hours_in_day;
					temp_hours = temp;
					temp_minutes = temp * minutes_in_hour;
					temp_seconds = temp * seconds_in_hour;

				}
				else
				{
					cout << "Please try adding " << hours_in_day
					<< " or more hours." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in minutes and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "minutes"))
			{
				if(temp >= minutes_in_day)
				{
					temp_date = calcdate(temp / minutes_in_day);
					temp_days = temp / minutes_in_day;
					temp_hours = temp / minutes_in_hour;
					temp_minutes = temp;
					temp_seconds = temp * seconds_in_minute;
				}
				else
				{
					cout << "Please try adding " << minutes_in_day
					<< " or more minutes." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in seconds and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "seconds"))
			{
				if(temp >= seconds_in_day)
				{
					temp_date = calcdate(temp / seconds_in_day);
					temp_days = temp / seconds_in_day;
					temp_hours = temp / seconds_in_hour;
					temp_minutes = temp / seconds_in_minute;
					temp_seconds = temp;
				}
				else
				{
					cout << "Please try adding " << seconds_in_day
					<< " or more seconds." << endl;
					usage();
				}
			}

			if(flags.wantdate)
				print_date(temp_date, flags);
			if(flags.wantdays)
				cout << temp_days << endl;
			if(flags.wanthours)
				cout << temp_hours << endl;
			if(flags.wantminutes)
				cout << temp_minutes << endl;
			if(flags.wantseconds)
				cout << temp_seconds << endl;
			if(flags.extraline)
				cout << endl;
			
			i += 3; //advance i to avoid reparsing args
		}
		else
			usage();
	}
	else //show proper usage
		usage();
	
	return;
}

void args_parse_now(const int argc, char *argv[], int &i, const Flag flags)
{ //parse the arguments after "now" and do some work
	Date date = get_todays_date(), //today's date needed to continue
		  temp_date = { 0, 0, 0 };  //temporary variable used for output purposes
	
	//bool used in test below for readability
	bool no_more_now_args = i + 1 < argc && strcmp(argv[i + 1], "+") &&
									strcmp(argv[i + 1], "-");

	//temporary variables for output purposes
	uint_least64_t temp_days = 0, temp_hours = 0, temp_minutes = 0,
						temp_seconds = 0;
	
	/*
	 * this compound test covers the cases where "now" is:
	 *	 1.  the only option on the command line
	 *	 2.  the last option on the command line
	 *	 3.  in the middle of a command line, but has no arguments following
	 *		  that belong to it
	 */
	if(i + 1 == argc || no_more_now_args)
	{
		print_date(date, flags);

		if(flags.extraline)
			cout << endl;
		
		return;
	}
	else if(i + 3 < argc) //3 more args needed to complete calculations
	{
		if(!strcmp(argv[i + 1], "+")) //make sure user wants to add
		{
			uint_least64_t temp = static_cast<uint_least64_t>(atol(argv[i + 2]));
			
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in days and must be converted to other
			 * units.
			 */
			if(!strcmp(argv[i + 3], "days"))
			{
				if(temp >= 1)
				{
					temp_date = calcdate(calcdays(date) + temp);
					temp_days = calcdays(date) + temp;
					temp_hours = calchours(date) + (temp * hours_in_day);
					temp_minutes = calcminutes(date) + (temp * minutes_in_day);
					temp_seconds = calcseconds(date) + (temp * seconds_in_day);
				}
				else
				{
					cout << "Please try adding 1 or more days." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in hours and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "hours"))
			{
				if(temp >= hours_in_day)
				{
					temp_date = calcdate(calcdays(date) + (temp / hours_in_day));
					temp_days = calcdays(date) + temp / hours_in_day;
					temp_hours = calchours(date) + temp;
					temp_minutes = calcminutes(date) + (temp * minutes_in_hour);
					temp_seconds = calcseconds(date) + (temp * seconds_in_hour);
				}
				else
				{
					cout << "Please try adding " << hours_in_day
						  << "or more hours." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in minutes and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "minutes"))
			{
				if(temp >= minutes_in_day)
				{
					temp_date = calcdate(calcdays(date) + (temp / minutes_in_day));
					temp_days = calcdays(date) + temp / minutes_in_day;
					temp_hours = calchours(date) + (temp / minutes_in_hour);
					temp_minutes = calcminutes(date) + temp;
					temp_seconds = calcseconds(date) + (temp * seconds_in_minute);
				}
				else
				{
					cout << "Please try adding " << minutes_in_day
						  << " or more minutes." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in seconds and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "seconds"))
			{
				if(temp >= seconds_in_day)
				{
					temp_date = calcdate(calcdays(date) + (temp / seconds_in_day));
					temp_days = calcdays(date) + (temp / seconds_in_day);
					temp_hours = calchours(date) + (temp / seconds_in_hour);
					temp_minutes = calcminutes(date) + (temp * seconds_in_minute);
					temp_seconds = calcseconds(date) + temp;
				}
				else
				{
					cout << "Please try adding " << seconds_in_day
						  << " or more seconds." << endl;
					usage();
				}
			}
		}
		else if(!strcmp(argv[i + 1], "-")) //user wants subtraction
		{
			uint_least64_t temp = static_cast<uint_least64_t>(atol(argv[i + 2]));
			
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in days and must be converted to other
			 * units.
			 */
			if(!strcmp(argv[i + 3], "days"))
			{
				if(calcdays(date) < temp)
				{
					cout << "The value " << temp << " is too large.\n" << endl;

					usage();
				}

				if(temp >= 1)
				{
					temp_date = calcdate(calcdays(date) - temp);
					temp_days = calcdays(date) - temp;
					temp_hours = calchours(date) - (temp * hours_in_day);
					temp_minutes = calcminutes(date) - (temp * minutes_in_day);
					temp_seconds = calcseconds(date) - (temp * seconds_in_day);
				}
				else
				{
					cout << "Please try subtracting 1 or more days." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in hours and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "hours"))
			{
				if(calchours(date) < temp)
				{
					cout << "The value " << temp << " is too large.\n" << endl;

					usage();
				}
				
				if(temp >= hours_in_day)
				{
					temp_date = calcdate(calcdays(date) - (temp / hours_in_day));
					temp_days = calcdays(date) - (temp / hours_in_day);
					temp_hours = calchours(date) - temp;
					temp_minutes = calcminutes(date) - (temp * minutes_in_hour);
					temp_seconds = calcseconds(date) - (temp * seconds_in_hour);
				}
				else
				{
					cout << "Please try subtracting " << hours_in_day
						  << "or more hours." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in minutes and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "minutes"))
			{
				if(calcminutes(date) < temp)
				{
					cout << "The value " << temp << " is too large.\n" << endl;

					usage();
				}
				
				if(temp >= minutes_in_day)
				{
					temp_date = calcdate(calcdays(date) - (temp / minutes_in_day));
					temp_days = calcdays(date) - (temp / minutes_in_day);
					temp_hours = calchours(date) - (temp / minutes_in_hour);
					temp_minutes = calcminutes(date) - temp;
					temp_seconds = calcseconds(date) - (temp * seconds_in_minute);
				}
				else
				{
					cout << "Please try subtracting " << minutes_in_day
						  << " or more minutes." << endl;
					usage();
				}
			}
			/*
			 * below, temp is multiplied or divided in various places
			 * because it is in seconds and must be converted to other
			 * units.
			 */
			else if(!strcmp(argv[i + 3], "seconds"))
			{
				if(calcseconds(date) < temp)
				{
					cout << "The value " << temp << " is too large.\n" << endl;

					usage();
				}
				
				if(temp >= seconds_in_day)
				{
					temp_date = calcdate(calcdays(date) - (temp / seconds_in_day));
					temp_days = calcdays(date) - (temp / seconds_in_day);
					temp_hours = calchours(date) - (temp / seconds_in_hour);
					temp_minutes = calcminutes(date) - (temp * seconds_in_minute);
					temp_seconds = calcseconds(date) - temp;
				}
				else
				{
					cout << "Please try subtracting " << seconds_in_day
						  << " or more seconds." << endl;
					usage();
				}
			}
			
			i += 3; //advance i to avoid reparsing args
		}
	} 
	else //show proper usage
		usage();

	//if execution reaches here then it's safe to print this output
	if(flags.wantdate)
	  print_date(temp_date, flags);
	if(flags.wantdays)
		cout << temp_days << endl;
	if(flags.wanthours)
		cout << temp_hours << endl;
	if(flags.wantminutes)
		cout << temp_minutes << endl;
	if(flags.wantseconds)
		cout << temp_seconds << endl;
	if(flags.extraline)
		cout << endl;

	return; //return control to calling function
}

void args_parse_between(const int argc, char *argv[], int &i, const Flag flags)
{
	Date first_date = { 0, 0, 0 }, second_date = { 0, 0, 0 };
	
	//these are compound tests because the syntax must be exact
	if(i + 7 < argc && !strcmp(argv[i + 4], "and"))
	{ //two dates case--7 more args needed to complete calculations
		first_date = get_date(argv, i + 1, flags);
		second_date = get_date(argv, i + 5, flags);
		i += 7; //advance i to avoid reparsing args
	} //epoch as first date case--5 more args needed to complete calculations
	else if(i + 5 < argc && !strcmp(argv[i + 1], "epoch") &&
			  !strcmp(argv[1 + 2], "and"))
	{
		first_date.imonth = 1; first_date.day = 1; first_date.year = 1970;
		second_date = get_date(argv, i + 3, flags);
		i += 5; //advance i to avoid reparsing args
	} //epoch as second date case--5 more args needed to complete calculations
	else if(i + 5 < argc && !strcmp(argv[i + 5], "epoch") &&
			  !strcmp(argv[i + 4], "and"))
	{
		first_date = get_date(argv, i + 1, flags);
		second_date.imonth = 1; second_date.day = 1; second_date.year = 1970;
		i += 5; //advance i to avoid reparsing args
	} //now as first date case--5 more args needed to complete calculations
	else if(i + 5 < argc && !strcmp(argv[i + 1], "now") &&
			  !strcmp(argv[1 + 2], "and"))
	{
		first_date = get_todays_date();
		second_date = get_date(argv, i + 3, flags);
		i += 5; //advance i to avoid reparsing args
	} //now as second date case--5 more args needed to complete calculations
	else if(i + 5 < argc && !strcmp(argv[i + 5], "now") &&
			  !strcmp(argv[i + 4], "and"))
	{
		first_date = get_date(argv, i + 1, flags);
		second_date = get_todays_date();
		i += 5; //advance i to avoid reparsing args
	} //now and epoch or epoch and now cases--3 more args needed
	else if(i + 3 < argc && !strcmp(argv[i + 2], "and") &&
			  ((!strcmp(argv[i + 1], "epoch") && !strcmp(argv[i + 3], "now")) ||
			  (!strcmp(argv[i + 3], "epoch") && !strcmp(argv[i + 1], "now"))))
	{
		first_date = get_todays_date();
		second_date.imonth = 1; second_date.day = 1; second_date.year = 1970;
		i += 3; //advance i to avoid reparsing args
	} //improper usage case
	else
		usage();
	
	//if execution reaches here it's safe to print output
	if(flags.wantdays)
		cout << calc_days_between(first_date, second_date) << endl;
	if(flags.wanthours)
		cout << calc_hours_between(first_date, second_date) << endl;
	if(flags.wantminutes)
		cout << calc_minutes_between(first_date, second_date) << endl;
	if(flags.wantseconds)
		cout << calc_seconds_between(first_date, second_date) << endl;
	if(flags.extraline)
		cout << endl;
	
	return;
}
