#!/usr/bin/env python
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup
# Requires Duplicity Version 0.6.04 released August 01, 2009.
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

import os.path

from PyQt4 import QtCore, QtGui, uic

from timedrive import BackInTime_Icons_rc
from timedrive.utils import gui_utils
from timedrive.backupsettings.settings import Settings
from timedrive.treesortfilter import TreeSortFilter
import timedrive.utils as utils
from timedrive.background import loadarchive

class RestoreDialog(QtGui.QDialog, object):
	"""Local and Remote Duplicity Archive Browser"""
	
	_changed = False
	
	def __init__(self, parent):
		QtGui.QDialog.__init__(self, parent)
		uic.properties.Properties._string = gui_utils.translate
		gui_utils.loadUi(self, "uiRestoreFiles.ui")
		self.ui = self
		self.modelList = None
		self.ViewType = None
		self.RemoteUrl = None
		self.AmazonS3_LoginInfo = None
		self.BackupFolders = None
		self.ArchiveUrls = None
		self.gnuPassPhrase = None
		self.RestoreTime = None
		self.timedata = None
		
		# Create the Search Model
		self.searchModel = TreeSortFilter(self)
		self.searchModel.setDynamicSortFilter(True)
		
		# Create Background Threads and Connect Events
		self.LoadArchive = loadarchive.LoadArchive(self)
		QtCore.QObject.connect(self.LoadArchive, QtCore.SIGNAL("ReturnFileList(QStandardItemModel*)"),
			self.LoadArchive_SetModelList)
		QtCore.QObject.connect(self.LoadArchive, QtCore.SIGNAL("toggleControls(bool)"), 
			self.enableTimeline)
		
		QtCore.QObject.connect(self.parent().LoadSnapshots,
			QtCore.SIGNAL("toggleControls(bool)"), self.enableTimeline)
		
		# Connect Signals and Slots
		QtCore.QObject.connect(self.ui.treeBackupFolders, QtCore.SIGNAL("clicked(QModelIndex)"), self.ChangeBackupDirectory)
		QtCore.QObject.connect(self.ui.treeBackupFolders, QtCore.SIGNAL("activated(QModelIndex)"), self.ChangeBackupDirectory)
		QtCore.QObject.connect(self.ui.treeBackupFolders, QtCore.SIGNAL("pressed(QModelIndex)"), self.ChangeBackupDirectory)
		QtCore.QObject.connect(self.ui.buttonAddFiles, QtCore.SIGNAL("clicked()"), self.buttonAddFiles_Pressed)
		QtCore.QObject.connect(self.ui.buttonSearch, QtCore.SIGNAL("clicked()"), self.UpdateSearchResults)
		QtCore.QObject.connect(self.ui.buttonClearSearch, QtCore.SIGNAL("clicked()"), self.buttonClearSearch_Pressed)
		QtCore.QObject.connect(self.ui.txtSearch, QtCore.SIGNAL("returnPressed()"), self.UpdateSearchResults)
		QtCore.QObject.connect(self.ui.buttonSnapshotRight, QtCore.SIGNAL("clicked()"), self.buttonSnapshotRight_Pressed)
		QtCore.QObject.connect(self.ui.buttonSnapshotLeft, QtCore.SIGNAL("clicked()"), self.buttonSnapshotLeft_Pressed)
		QtCore.QObject.connect(self.ui.buttonReloadSnapshot, QtCore.SIGNAL("clicked()"), self.buttonReloadSnapshot_Pressed)
		QtCore.QObject.connect(self.ui.buttonCloseWindow, QtCore.SIGNAL("clicked()"),
		self.close)
	
	
	def LoadArchive_FromSettings(self, restore_time = None):
		"""Load the archive file list from the values in the settings"""
		
		self.showTimeline(True)
		
		if (self.modelList == None)|(self.ViewType == 'Remote')|(self.RestoreTime != restore_time)|self.isChanged():
			self.changed = False
			self.ViewType = 'Local'
			self.BackupFolders = None
			self.ArchiveUrls = None
			self.AmazonS3_LoginInfo = None
			settings = Settings()
			self.ui.treeBackupFolders.clear()
			
			if self.modelList != None:
				self.modelList.clear()
				self.SetModelColumns()
			else:
				self.modelList = QtGui.QStandardItemModel()
				self.SetModelColumns()
			
			if restore_time != None:
				restore_time = str(restore_time)
				self.LoadSliderData(self.timedata, int(utils.time_String2Int(restore_time)))
			else:
				restore_time = None
				self.LoadSliderData(self.timedata)
			
			self.LoadArchive.FromSettings(settings, restore_time)
			self.RestoreTime = restore_time
		
			for folder in settings.IncludeList:
			# Add Folder Name to the Backup Folders List
				folderIcon = QtGui.QIcon(":/SVG/Resources/Folder.svg")
				includeItem = QtGui.QTreeWidgetItem([QtCore.QString(utils.determine_folderName(folder))])
				includeItem.setIcon(0, folderIcon)
				self.ui.treeBackupFolders.addTopLevelItem(includeItem)
		else:
			self.show()
	
	
	def LoadArchive_Remote(self, archiveUrl, gnu_passphrase, AmazonS3 = False):
		"""Load the archive file list from the values in the Advanced Restore Pane
		of the MainWindow."""
		
		self.showTimeline(False)
		
		if (self.modelList == None)|(self.ViewType == 'Local')|(self.RemoteUrl!=archiveUrl):
			
			self.BackupFolders = None
			self.ArchiveUrls = None
			self.ui.treeBackupFolders.clear()
			
			if self.modelList != None:
				self.modelList.clear()
				self.SetModelColumns()
			else:	
				self.modelList = QtGui.QStandardItemModel()
				self.SetModelColumns()
			
			# Start the Backgroud Process and Load Archive Data
			if AmazonS3 == False:
				self.ViewType = "Remote"
				self.AmazonS3_LoginInfo = None
				self.LoadArchive.FromUrl(archiveUrl, gnu_passphrase)
			elif AmazonS3 == True:
				self.ViewType = "AmazonS3"
				archiveUrl, S3_AccessId, S3_SecretKey = str(archiveUrl).split()
				self.LoadArchive.FromUrl(archiveUrl, gnu_passphrase,
					S3_AccessId, S3_SecretKey)
				self.AmazonS3_LoginInfo = S3_AccessId + " " + S3_SecretKey
			
			displayName = os.path.basename(archiveUrl)
			folderIcon = QtGui.QIcon(":/PNG/Resources/Network-Archive.png")
			includeItem = QtGui.QTreeWidgetItem([QtCore.QString(displayName)])
			includeItem.setIcon(0, folderIcon)
			self.ui.treeBackupFolders.addTopLevelItem(includeItem)
		
		elif self.RemoteUrl == archiveUrl:
			self.show()
	
	
	def LoadArchive_FromTimeline(self, timeInt):
		self.SetTimeline(timeInt)
		restore_time, prettyTime = self.timedata[timeInt]
		
		if (self.modelList == None)|(self.ViewType == 'Remote')|(self.RestoreTime != restore_time):
			self.ViewType = 'Local'
			self.BackupFolders = None
			self.ArchiveUrls = None
			settings = Settings()
			
			self.modelList.clear()
			self.SetModelColumns()
			self.LoadArchive.FromTimeline(settings, str(restore_time))
			self.RestoreTime = restore_time
			titleText = _("Restore Files - ") + prettyTime
			self.ui.groupRestoreFiles.setTitle(titleText)
	
	
	def SetTimeline(self, timeInt):
		self.ui.sliderTimeline.setValue(timeInt)
		
		if self.timedata[timeInt]:
			timeString, timePretty = self.timedata[timeInt]
			self.ui.labelSnapshot.setText(timePretty)
		else:
			timePretty = utils.time_String2Pretty(utils.time_Int2String(timeInt))
			self.ui.labelSnapshot.setText(timePretty)
	
	
	def LoadArchive_SetModelList(self, transferModel):
		"""Sets the file list model, search model, and sets up the search options."""
		receiveModel = self.modelList
		i = 0
		
		while i < transferModel.rowCount():
			moveRow = transferModel.takeRow(i)
			receiveModel.appendRow(moveRow)

		self.searchModel.setSourceModel(receiveModel)	
		self.searchModel.setFilterKeyColumn(0)
		self.ui.treeViewFiles.setModel(self.searchModel)
		self.ChangeBackupDirectory()
	
	
	def buttonClearSearch_Pressed(self):
		self.ClearSearch()
		self.ChangeBackupDirectory(self.ui.treeBackupFolders.currentIndex())		
	
	
	def ChangeBackupDirectory(self, modelIndex=None):
		self.ClearSearch()
		
		if modelIndex != None:
			self.ui.treeViewFiles.setRootIndex(self.searchModel.index(modelIndex.row()+1,0))
		else:
			self.ui.treeViewFiles.setRootIndex(self.searchModel.index(1,0))
			
		self.ui.treeViewFiles.setColumnHidden(2, True)
		self.ui.treeViewFiles.setColumnHidden(3, True)
		self.ui.treeViewFiles.setColumnWidth(0, 300)
		self.ui.treeViewFiles.setColumnWidth(1, 200)
	
	
	def ClearSearch(self):
		self.ui.txtSearch.setText('')
		self.searchModel.setFilterRegExp('')
	
	
	def UpdateSearchResults(self):
		searchText = self.ui.txtSearch.text()
		
		if searchText != "":
			caseSensitivity = QtCore.Qt.CaseInsensitive
			searchFilter = QtCore.QRegExp(searchText, caseSensitivity)
			self.searchModel.setFilterRegExp(searchFilter)
		else:
			self.ChangeBackupDirectory(self.ui.treeBackupFolders.currentIndex())
		
	
	def buttonAddFiles_Pressed(self):
		"""Adds files/folders to the restore list."""
		
		selectedIndex = self.searchModel.mapToSource(self.ui.treeViewFiles.currentIndex())
		referenceModel = self.modelList
		
		originalItem = referenceModel.itemFromIndex(selectedIndex.sibling(selectedIndex.row(), 0))
		
		# Find a reference to the directory root item
		rootItem = originalItem.parent()
		while rootItem.parent() != None:
			rootItem = rootItem.parent()
		
		# Create an icon that responds to whether the item is a file or folder
		if originalItem.hasChildren():
			itemIcon = QtGui.QIcon(':/SVG/Resources/Folder.svg')
		else:
			itemIcon = QtGui.QIcon(':/SVG/Resources/File.svg')
			
		# Create Copies of the Information to be Passed to Restore List
		itemFileName = originalItem.clone()
		itemFileName.setIcon(itemIcon)
		itemDateModified = referenceModel.itemFromIndex(selectedIndex.sibling(selectedIndex.row(), 1)).clone()
		itemRelPath = referenceModel.itemFromIndex(selectedIndex.sibling(selectedIndex.row(), 2)).clone()
		itemArchiveUrl = referenceModel.itemFromIndex(rootItem.index().sibling(rootItem.row(), 1)).clone()
		itemPassphrase = referenceModel.itemFromIndex(rootItem.index().sibling(rootItem.row(), 2)).clone()
		itemRestoreTime = referenceModel.itemFromIndex(rootItem.index().sibling(rootItem.row(), 0)).clone()
		
		if self.AmazonS3_LoginInfo != None:
			itemAmazonS3 = QtGui.QStandardItem(self.AmazonS3_LoginInfo)
		else:
			itemAmazonS3 = QtGui.QStandardItem()
		
		# Add to the Restore List
		self.parent().modelRestoreList.appendRow([itemFileName, itemDateModified, 
			itemRelPath, itemArchiveUrl, itemPassphrase, itemRestoreTime, itemAmazonS3])
	
	
	def SetModelColumns(self):
		"""Sets the headers and widths of the file list tree view."""
		self.modelList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_("File Name")))
		self.modelList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_("Date Modified")))
		self.modelList.setHorizontalHeaderItem(2, QtGui.QStandardItem(_("Relative Path")))
		self.ui.treeViewFiles.setColumnHidden(2, True)
		self.ui.treeViewFiles.setColumnWidth(0, 300)
		self.ui.treeViewFiles.setColumnWidth(1, 200)
	
	
	def showTimeline(self, option):
		"""Toggles the timeline between visible and invisible."""
		self.ui.sliderTimeline.setVisible(option)
		self.ui.labelSnapshot.setVisible(option)
		self.ui.buttonSnapshotLeft.setVisible(option)
		self.ui.buttonSnapshotRight.setVisible(option)
		self.ui.buttonReloadSnapshot.setVisible(option)
		self.ui.labelOldestSnapshot.setVisible(option)
		self.ui.labelNewestSnapshot.setVisible(option)
	
	
	def enableTimeline(self, option):
		"""Enables/disables the timieline controls"""
		self.ui.sliderTimeline.setEnabled(option)
		self.ui.labelSnapshot.setEnabled(option)
		self.ui.buttonSnapshotLeft.setEnabled(option)
		self.ui.buttonSnapshotRight.setEnabled(option)
		self.ui.buttonReloadSnapshot.setEnabled(option)
		self.ui.labelOldestSnapshot.setEnabled(option)
		self.ui.labelNewestSnapshot.setEnabled(option)
	
	
	def buttonReloadSnapshot_Pressed(self):
		newSnapshot = self.ui.sliderTimeline.value()
		self.LoadArchive_FromTimeline(newSnapshot)
	
	
	def buttonSnapshotRight_Pressed(self):
		valueList = self.timedata.keys()
		valueList.sort()
		
		currentValue = self.ui.sliderTimeline.value()
		numItems = len(valueList)
		i = 1
		
		if numItems > 1:
		
			if currentValue == min(self.timedata.keys()):
				nextValue = valueList[1]
			
			if currentValue == max(self.timedata.keys()):
				nextValue = valueList[numItems-1]
			
			while i < numItems - 1:
				value = valueList[i]
				if ((currentValue > value) & (currentValue < valueList[i+1]))|(currentValue == value):
					nextValue = valueList[i+1]
				elif (currentValue < value) & (currentValue > valueList[i-1]):
					nextValue = value
				i = i+1
		else:
			nextValue = valueList[0]
		
		self.SetTimeline(nextValue)
	
	
	def buttonSnapshotLeft_Pressed(self):
		valueList = self.timedata.keys()
		valueList.sort()
		
		currentValue = self.ui.sliderTimeline.value()
		numItems = len(valueList)
		i = 1
		
		if numItems > 1:
		
			if currentValue == min(self.timedata.keys()):
				nextValue = valueList[0]
			
			if currentValue == max(self.timedata.keys()):
				nextValue = valueList[numItems-2]
			
			while i < numItems - 1:
				value = valueList[i]
				if (currentValue > value) & (currentValue < valueList[i+1]):
					nextValue = value
				elif ((currentValue < value) & (currentValue > valueList[i-1]))|(currentValue == value):
					nextValue = valueList[i-1]
				i = i+1
		
		else:
			nextValue = valueList[0]
		
		self.SetTimeline(nextValue)
	
	
	def LoadSliderData(self, sliderData, currentSnapshot = None):
		"""Sets default values for the snapshot slider and associated labels."""
		
		sliderValues = sliderData.keys()
		newestSnapshot = max(sliderValues)
		oldestSnapshot = min(sliderValues)
		newestTimeString, newestPrettyTime = self.timedata[newestSnapshot]
		oldestTimeString, oldestPrettyTime = self.timedata[oldestSnapshot]
		self.ui.sliderTimeline.setMaximum(newestSnapshot)
		self.ui.labelNewestSnapshot.setText(newestPrettyTime)
		self.ui.sliderTimeline.setMinimum(oldestSnapshot)
		self.ui.labelOldestSnapshot.setText(oldestPrettyTime)
			
		
		if currentSnapshot == None:
			snapshotTitle = newestPrettyTime
			self.SetTimeline(newestSnapshot)
		else:
			timeString, prettyTime = self.timedata[currentSnapshot]
			snapshotTitle = prettyTime
			self.SetTimeline(currentSnapshot)
		
		titleText = _("Restore Files - ") + snapshotTitle
		self.ui.groupRestoreFiles.setTitle(titleText)
		
	def isChanged(self):
		return self._changed
		
	
	def setChanged(self, value):
		self._changed = value
		
	changed = property(isChanged, setChanged)
