#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
import timedrive.utils as utils
from timedrive.background import backgroundtask
from timedrive.backupsettings.settings import Settings
from timedrive.utils import duplicity_interface
from timedrive.utils import log
from timedrive.utils import archiveurl

class ManageBackups(backgroundtask.BackgroundTask):
	"""Background task that deletes and manages backup sets,
	snapshots, and associated metadata"""
	
	opTypeDELETEBACKUP = 1
	opTypeCLEANUPBACKUP = 2

	def __init__(self, parent = None):
		"""
		Initialization function for Manage Backups
		
		@type parent: BackgroundTask Object
		@param parent:
		"""
		super(ManageBackups, self).__init__(_("Manage Backups"), parent)
		self.operationType = None
		self.backupTime = None
		self.folder = None
	
	def DeleteBackupSet(self, backupTime, folder):
		self.operationType = self.opTypeDELETEBACKUP
		self.backupTime = backupTime
		self.folder = folder
		self.start()
	
	def CleanupBackup(self, folder):
		self.operationType = self.opTypeCLEANUPBACKUP
		self.folder = folder
		self.start()
		
	def RunOperation(self, operationType, folder, stringTime = None):
		"""
		Deletes the snapshots as specified by time.
		
		@param operationType:
		@param folder:
		@param stringTime:
		"""
		
		if stringTime != None:
			backupTime = utils.time_String2Int(str(stringTime))
		
		settings = Settings()
		
		if settings.AmazonS3Backup == True:
			S3_AccessId = settings.AmazonS3_AccessId
			S3_SecretKey = settings.AmazonS3_SecretKey
		else:
			S3_AccessId = None
			S3_SecretKey = None
		
		if settings.AmazonS3_Location == "European":
			S3_BucketType = settings.AmazonS3_Location
		else:
			S3_BucketType = None
		
		au = archiveurl.ArchiveUrl_FromSettings(folder, settings)
		if au.available:
			
			# Delete Specified Backup Set
			if operationType == self.opTypeDELETEBACKUP:
				operationSuccess = duplicity_interface.delete_backup(str(au.archiveUrl),
					str(au.gnuPassphrase), backupTime, True, S3_AccessId, S3_SecretKey, 
					S3_BucketType)
				return operationSuccess, ""
					
			# Run Duplicity Cleanup on Specified Set
			elif operationType == self.opTypeCLEANUPBACKUP:
				operationSuccess, operationMessage = duplicity_interface.run_cleanup_backupset(str(au.archiveUrl),
					str(au.gnuPassphrase), True, S3_AccessId, S3_SecretKey, S3_BucketType)
				return operationSuccess, operationMessage
		else:
			self._error_available(self.task_name, folder)
			return False, _("location not available")
	
	
	def run(self):
		"""Primary method of the class."""
		
		self.toggleControls(False)
		folderName = utils.determine_folderName(self.folder)
		# Delete the Backup Set Specified by self.backupTime and self.folder
		if self.operationType == self.opTypeDELETEBACKUP:
			self.setupTaskProgress(0, _("Deleting Backup Set ... "))
			operationResult = self.RunOperation(self.operationType, self.folder, self.backupTime)
			if operationResult == True:
				userMessage = _("Backup Deleted Successfully \n") + folderName + "\n -" + utils.time_String2Pretty(self.backupTime)

			else:
				userMessage = _("Unable to Delete Backup Set \n") + folderName + "\n -" + utils.time_String2Pretty(self.backupTime)

			log.Info(str(userMessage))
			self.taskComplete(_("Delete Backup"), userMessage)
		
		elif self.operationType == self.opTypeCLEANUPBACKUP:
			self.setupTaskProgress(0, _("Cleaning Up Backup Set ... "))
			operationResult, operationMessage = self.RunOperation(self.operationType, self.folder)
			if operationResult == True:
				userMessage = operationMessage + "\n" + folderName
			else:
				userMessage = _("Unable to Complete Cleanup \n") + folderName
			
			log.Info(str(userMessage))
			self.taskComplete(_("Cleanup Backup Set"), userMessage)
			
		self.toggleControls(True)
		
