#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
from PyQt4 import QtCore, QtGui
import timedrive.utils as utils
from timedrive.utils import duplicity_interface
from timedrive.background import backgroundtask
from timedrive.utils import archiveurl

class LoadSnapshots(backgroundtask.BackgroundTask):
	"""Background class that loads a list of the archive snapshots"""
	
	def __init__(self, parent = None):
		"""
		Initialization function for Load Snapshots
		
		@type parent: BackgroundTask Object
		@param parent:
		"""
		super(LoadSnapshots, self).__init__(_("Load Snapshots"), parent)
		self.snapshotList = QtGui.QStandardItemModel()
		self.folder = None
		self.list_type = None
		self.sync = None
		self.settings = None
	
	def MainList_FromSettings(self, settings, sync = False):
		"""
		Produceses a list of snapshots, returns the results with the following signal:
			ReturnSnapshotList_Main(QStandardItemModel*)
		
		@type settings: Settings Object
		@param settings:
		@type sync: Boolean
		@param sync: default = False
		"""
		self.settings = settings
		self.folder = None
		self.sync = sync
		self.list_type = "main"
		self.start()
		
	def GetSnapshotList_Main(self, settings, sync = False):
		"""
		Get a full list of all snapshots from all archives.  Returned as a dictionary.
		See also: GetSnapshotList_Folder.
		
		@type settings: Settings Object
		@param settings:
		@type sync: Boolean
		@param sync: default = False
		"""
	
		archiveList = self.SnapshotList_FetchUrls(settings)
		allSnapshots = {}
		
		# only general defined
		if self.settings.AmazonS3Backup == True:
			S3_AccessId = self.settings.AmazonS3_AccessId
			S3_SecretKey = self.settings.AmazonS3_SecretKey
		else:
			S3_AccessId = None
			S3_SecretKey = None
		
		if settings.AmazonS3_Location == "European":
			S3_BucketType = settings.AmazonS3_Location
		else:
			S3_BucketType = None

		for folder in archiveList.keys():
			archiveUrl, gnu_passphrase = archiveList[folder]
			snapshotList = duplicity_interface.get_snapshot_list(str(archiveUrl),
				str(gnu_passphrase), sync, S3_AccessId, S3_SecretKey, S3_BucketType)
		
			if snapshotList != None:
				for snapshot in snapshotList:
					snapshotType, dateTime, numFiles = snapshot.split()
					prettyTime = utils.time_String2Pretty(dateTime)
					allSnapshots[dateTime] = [prettyTime, snapshotType, numFiles]
			else:
				pass
		
		return allSnapshots
	
	def GetSnapshotList_Folder(self, folder, settings, sync = False,
		S3_AccessId = None, S3_SecretKey = None):
		"""
		Get a list of the snapshots for a given folder.  Returned as a dictionary.
		See also: GetSnapshotList_Main.
		
		@type settings: Settings Object
		@param settings:
		@type sync: Boolean
		@param sync: default = False
		"""
	
		folderSnapshots = {}
	
		au = archiveurl.ArchiveUrl_FromSettings(folder, settings)

		if au.available:
			snapshotList = duplicity_interface.get_snapshot_list(str(au.archiveUrl), 
				str(au.gnuPassphrase), sync)
		
			if snapshotList != None:
				for snapshot in snapshotList:
					snapshotType, dateTime, numFiles = snapshot.split()
					prettyTime = utils.time_String2Pretty(dateTime)
					folderSnapshots[dateTime] = [prettyTime, snapshotType, numFiles]
				else:
					pass
		else:
			self._error_available(folder)
	
		return folderSnapshots

	def SnapshotList_FetchUrls(self, settings):
		"""Function used to create a list of archive Urls from the
		backupsettings.IncludeList"""
		archiveList = {}
		for folder in settings.IncludeList:
			au = archiveurl.ArchiveUrl_FromSettings(folder, settings)
			if au.available:
				archiveList[folder] = [au.archiveUrl, au.gnuPassphrase]
			else:
				self._error_available(self.task_name, folder)

		return archiveList

	def FolderList(self, settings, folder, sync = False):
		"""
		Produceses a list of snapshots, returns the results with the following signal:
			ReturnSnapshotList_Folder(QStandardItemModel*)
			
		@type settings: Settings Object
		@param settings:
		@type folder: IncludeItem Object
		@param folder:
		@type sync: Boolean
		@param sync: default = False
		"""
		self.settings = settings
		self.folder = folder
		self.sync = sync
		self.list_type = "folder"
		self.start()
	
	def returnSnapshotList_Main(self, slist):
		"""
		returns the results list for everything using a emit qtcore signal
		
		@type slist: QStandardItemModel*
		@param slist:
		"""
		self.emit(QtCore.SIGNAL("ReturnSnapshotList_Main(QStandardItemModel*)"), slist)
	
	def returnSnapshotList_Folder(self, slist):
		"""
		returns the results list for one folder using a emit qtcore signal
		
		@type slist: QStandardItemModel*
		@param slist:
		"""
		self.emit(QtCore.SIGNAL("ReturnSnapshotList_Folder(QStandardItemModel*)"), slist)

	def run(self):
		"""Primary method of the class.  First, fetches a list of the snapshots
		 available in all archives.  Then, translates the list to a QStandardItemModel."""

		self.setupTaskProgress(0, _("Loading Snapshots ... "))
		self.toggleControls(False)
		
		# Main Snapshot List
		if self.list_type == "main":
			snapshotsList = self.GetSnapshotList_Main(self.settings, self.sync)
			sortedSnapshots = snapshotsList.keys()
			sortedSnapshots.sort()
			sortedSnapshots.reverse()
		
			# Check to see that there are valid snapshots to display
			if len(sortedSnapshots) > 0:
			
				# Filter snapshots included in the same backup
				filterRelated = utils.time_String2Int(sortedSnapshots[0])
				
				for dateTime in sortedSnapshots:
					filterDateTime = utils.time_String2Int(dateTime)
					diffTime = filterRelated - filterDateTime
				
					if (diffTime > 1000)|(filterDateTime==filterRelated):
						prettyTime, snapshotType, numFiles = snapshotsList[dateTime]
						itemDate = QtGui.QStandardItem(dateTime)
						itemPrettyTime = QtGui.QStandardItem(prettyTime)
						itemBackupType = QtGui.QStandardItem(snapshotType)
						itemTimeInt = QtGui.QStandardItem(str(filterDateTime))
						self.snapshotList.appendRow([itemPrettyTime, itemDate, itemTimeInt, itemBackupType])
						filterRelated = filterDateTime
		
			self.returnSnapshotList_Main(self.snapshotList)
		
		# Folder Specific List
		elif self.list_type == "folder":
			snapshotsList = self.GetSnapshotList_Folder(self.folder, self.settings, self.sync)
			
			# Sort Snapshots by Date
			sortedSnapshots = snapshotsList.keys()
			sortedSnapshots.sort()
			
			# Check to ensure that there are valid snapshots to display
			if len(sortedSnapshots) > 0:
				for dateTime in sortedSnapshots:
					intTime = utils.time_String2Int(dateTime)
					prettyTime, snapshotType, numFiles = snapshotsList[dateTime]
					itemDate = QtGui.QStandardItem(dateTime)
					itemPrettyTime = QtGui.QStandardItem(prettyTime)
					itemBackupType = QtGui.QStandardItem(snapshotType)
					itemTimeInt = QtGui.QStandardItem(str(intTime))
					self.snapshotList.appendRow([itemPrettyTime, itemDate, itemTimeInt, itemBackupType])
			
			self.returnSnapshotList_Folder(self.snapshotList)
		
		# Clean Up Settings
		self.toggleControls(True)
		self.settings = None
		self.folder = None
		self.sync = None
		self.list_type = None


