#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
from PyQt4 import QtCore
import threading
import timedrive.utils as utils
from timedrive.utils import log

running_threads = list()

class BackgroundTask(QtCore.QThread, object):
	"""
	BackgroundTask class: base class for threads used in time-drive
	"""
	
	_restart = None
	_default_message = "Default Message"
	_task_name = None

	def __init__(self, taskName, parent = None):
		"""
		Initialization function for BackgroundTask
		
		@type parent: QThread Object
		@param parent:
		"""
		super(BackgroundTask, self).__init__()
		self.task_name = taskName
		self.parent = parent
		self.settings = None

	def taskComplete(self, taskName, userMessage):
		"""
		emits a message when a run is complete need to call manually
		
		@type taskName: String
		@param taskName: the task who sents the message
		@type userMessage: String
		@param userMessage: the message you want to show the user 
		"""
		self.emit(QtCore.SIGNAL("taskComplete(QString, QString)"), 
			QtCore.QString(taskName), QtCore.QString(userMessage))
	
	def taskError(self, taskName, error, userMessage):
		"""
		emits a message when a task has an error need to call manually
		
		@type taskName: String
		@param taskName: the task who sents the message
		@type error: String
		@param error: type of error as string
		@type userMessage : String
		@param userMessage: the message you want to show the user 
		"""
		self.emit(QtCore.SIGNAL("taskError(QString, QString, QString)"), 
			QtCore.QString(taskName), QtCore.QString(error), QtCore.QString(userMessage))
	
	def setupTaskProgress(self, maximum, message = None):
		"""
		emits a message with task progress need to call manually
		
		@type maximum: int
		@param maximum: the maximum of elements to process
		@type message: String
		@param message: default=None(will be the default message defined),
				informative message or task name 
		"""
		if message is None:
			message = self.default_message
			
		self.emit(QtCore.SIGNAL("setupTaskProgress(int, QString)"),
			maximum, QtCore.QString(message))
	
	def updateTaskProgress(self, progress, message = None):
		"""
		emits a int to update the progress bar
		
		@type progress: int
		@param progress:
		@type message: String
		@param message: default=None(will be the default message defined),
				informative message or task name
		"""
		if message is None:
			message = self.default_message
			
		self.emit(QtCore.SIGNAL("updateTaskProgress(int, QString)"),
			progress, QtCore.QString(message))
	
	def toggleControls(self, option):
		"""
		emits a boolean to indicate if the controls need to be toggled in the ui
		
		@type option: Boolean
		@param option:
		"""
		self.emit(QtCore.SIGNAL("toggleControls(bool)"), option)

	def questionPrompt(self, title, question):
		"""
		emits a question and passes on the thread that fire this method
		
		@type title: String
		@param title:
		@type question: String
		@param question:
		"""
		self.cond = threading.Condition()
		self.emit(QtCore.SIGNAL("questionPrompt(QString,QString,PyQt_PyObject)"), title, question, self.cond)
		self.cond.acquire()		
		self.cond.wait()
		self.cond.release()
		
	def start(self):
		"""
		The start method, keeps tracks af all the threads calling the start method
		Invokes the super start method to start the Qthread run 
		"""
		running_threads.append(self)
		self.stop = False
		super(BackgroundTask, self).start()
		
	def _error_available(self, task, include):
		"""
		display and write to log about an location isn't available
		
		@type task: String
		@param task: the name of the task calling this method
		@type include: IncludeItem Object
		@param inlcude:
		"""
		folderName = utils.determine_folderName(include)
		message = _("Failed: location not available %s with backup type %s") % (folderName, utils.backupTypeToText(include))
		log.Error(_("%s, include: %s") % (task,  message))
		self.taskError(task, "host down", message)	

	def _set_default_message(self, value):
		"""
		default message when emiting a signal and no message is defined
		
		@type value: String
		@param value:  
		"""
		self._default_message = value
		
	def _get_default_message(self):
		"""
		default message when emiting a signal and no message is defined
		
		@rtype: String
		@return:  
		"""
		return self._default_message
	
	default_message = property(_get_default_message, _set_default_message)

	def _set_task_name(self, value):
		"""
		the name of the task, for logging and output purpose
		
		@type value: String
		@param value: 
		"""
		self._task_name = value

	def _get_task_name(self):
		"""
		the name of the task, for logging and output purpose
		
		@rtype: String
		"""
		return self._task_name

	task_name = property(_get_task_name, _set_task_name)
	
	def _set_restart(self, value):
		"""
		restart status
		
		@type value: Boolean
		@param value: Indication if restart is needed 
		"""
		self._restart = value

	def _get_restart(self):
		"""
		restart status
		
		@rtype: Boolean
		"""
		return self._restart

	restart = property(_get_restart, _set_restart)
	
	def _set_stop(self, value):
		"""
		stop status of current task
		
		@type value: Boolean
		@param value: Indication if stop is needed 
		"""
		self._stop = value

	def _get_stop(self):
		"""
		stop status of current task
		
		@rtype: Boolean
		"""
		return self._stop

	stop = property(_get_stop, _set_stop)
