#!/usr/bin/env python
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

import os

from PyQt4 import QtCore, QtGui, uic
from timedrive.generalsettingsdialog import GeneralSettingsDialog
from timedrive.restoredialog import RestoreDialog
from timedrive.loggingdialog import LoggingDialog

import timedrive.utils as utils
from timedrive.utils.usernotifications import UserNotifications

from timedrive.utils import gui_utils
from timedrive.utils import archiveurl
from timedrive import BackInTime_Icons_rc
from timedrive.background import testconnectionurl, backupjob, restorefiles
from timedrive.background import loadsnapshots, managebackups, backgroundtask

from timedrive.backupsettings.settings import Settings
from timedrive.backupsettings import globals

class MainWindow(QtGui.QWidget, testconnectionurl.TestConnectionControl):
	"""Main Window class"""

	def __init__(self):
		"""
		Initialization function for MainWindow
		"""
		QtGui.QWidget.__init__(self)
		uic.properties.Properties._string = gui_utils.translate
		gui_utils.loadUi(self, "uiMainWindow.ui")
		self.ui = self
		settings = Settings()
		settings.Mode = globals.GuiMode

		if settings.ErrorOccurred:
			QtGui.QMessageBox.warning(self, _("Cron Backup Error"), _("Error Occured during run of background backup"), QtGui.QMessageBox.Ok)
			settings.ErrorOccurred = False

		self.NotifyUser = UserNotifications()
		
		# Create Background Threads
		self.BackupJob = backupjob.BackupJob(self)
		self.RestoreFiles = restorefiles.RestoreFiles(self)
		self.LoadSnapshots = loadsnapshots.LoadSnapshots(self)
		self.ManageBackups = managebackups.ManageBackups(self)
		
		# Setup SettingsDialog, RestoreDialog
		self.ui.windowRestore = RestoreDialog(self)
		self.ui.windowSettings = GeneralSettingsDialog(self, settings)
		self.ui.windowLogging = LoggingDialog(self)
		
		# Set Up the Restore Files List
		self.modelRestoreList = self.setupRestoreList()
		self.ui.listRestoreFiles.setModel(self.modelRestoreList)
		self.ui.listRestoreFiles.setColumnHidden(2, True)
		self.ui.listRestoreFiles.setColumnHidden(3, True)
		self.ui.listRestoreFiles.setColumnHidden(4, True)
		self.ui.listRestoreFiles.setColumnHidden(5, True)
		self.ui.listRestoreFiles.setColumnHidden(6, True)
		self.ui.listRestoreFiles.setColumnWidth(0, 190)
		self.ui.listRestoreFiles.setColumnWidth(1, 150)
		
		# Set Up the Snapshots List
		self.modelSnapshotsList = self.setupSnapshotsList()
		self.ui.treeSnapshots.setModel(self.modelSnapshotsList)
		self.ui.treeSnapshots.setColumnHidden(1, True)
		self.ui.treeSnapshots.setColumnHidden(2, True)
		self.ui.treeSnapshots.setColumnHidden(3, True)
		self.ui.treeSnapshots.setCurrentIndex(self.modelSnapshotsList.index(0, 0))
		self.ui.buttonRefreshSnapshot.setEnabled(False)
		
		# Hide Advanced Restore and Progress Indicators
		self.AdvancedRestore_SetVisible(False)
		self.ProgressIndicators_SetVisible(False)
		self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(False)
		
		# Connect Events and Methods - Toolbar Buttons
		QtCore.QObject.connect(self.ui.buttonSettings, QtCore.SIGNAL("pressed()"), 
			self.buttonSettings_Pressed)
		QtCore.QObject.connect(self.ui.buttonAddToQueue, QtCore.SIGNAL("pressed()"), 
			self.buttonAddToQueue_Pressed)
		QtCore.QObject.connect(self.ui.buttonAdvancedRestore, QtCore.SIGNAL("pressed()"), 
			self.buttonAdvancedRestore_Pressed)
		QtCore.QObject.connect(self.ui.buttonAdvancedRestore_Test, 
			QtCore.SIGNAL("pressed()"), self.TestAdvancedRestoreOptions)
		QtCore.QObject.connect(self.ui.buttonBackup, QtCore.SIGNAL("pressed()"),
			self.buttonBackup_Pressed)
		QtCore.QObject.connect(self.ui.buttonRestoreFiles, QtCore.SIGNAL("clicked()"),
			self.RestoreFiles_Start)
		QtCore.QObject.connect(self.ui.buttonRemoveFromQueue, 
			QtCore.SIGNAL("clicked()"), self.buttonRemoveFromQueue_Pressed)
		QtCore.QObject.connect(self.ui.comboRemoteOptions, 
			QtCore.SIGNAL("activated(const QString &)"),
			self.AdvancedRestore_comboRemoteOptions_currentIndexChanged)
		QtCore.QObject.connect(self.ui.buttonAdvancedRestore_LoadDirectory, 
			QtCore.SIGNAL("pressed()"), self.AdvancedRestore_PickFolder)
		QtCore.QObject.connect(self.ui.buttonRefreshSnapshot, 
			QtCore.SIGNAL("clicked()"), self.buttonRefreshSnapshotList_Pressed)
		QtCore.QObject.connect(self.ui.buttonExit, QtCore.SIGNAL("clicked()"),
			self.buttonExit_Pressed)
		QtCore.QObject.connect(self.ui.buttonOpenLog, QtCore.SIGNAL("clicked()"),
			self.buttonOpenLog_Pressed)
		QtCore.QObject.connect(self.ui.buttonStopTask, QtCore.SIGNAL("clicked()"),
			self.buttonStopTask_Pressed)
			
		# Signals/Slots Related to Background Threads		
		QtCore.QObject.connect(self.BackupJob, QtCore.SIGNAL("started()"),
			self.backupStarted)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("taskComplete(QString, QString)"), self.backupFinished)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("taskError(QString, QString, QString)"), self.taskError)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("questionPrompt(QString,QString,PyQt_PyObject)"), self.questionPrompt)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("setupTaskProgress(int, QString)"), self.ProgressIndicators_Setup)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("updateTaskProgress(int, QString)"), self.ProgressIndicators_Update)
		QtCore.QObject.connect(self.BackupJob, 
			QtCore.SIGNAL("taskComplete(QString, QString)"), self.ProgressIndicators_Shutdown)
		
		QtCore.QObject.connect(self.ui.windowRestore.LoadArchive, 
			QtCore.SIGNAL("toggleControls(bool)"), self.RestoreDialog_Show)
		QtCore.QObject.connect(self.ui.windowRestore.LoadArchive,
			QtCore.SIGNAL("toggleControls(bool)"), self.ui.buttonRefreshSnapshot.setEnabled)
		QtCore.QObject.connect(self.ui.windowRestore.LoadArchive, 
			QtCore.SIGNAL("setupTaskProgress(int, QString)"), self.ProgressIndicators_Setup)
		QtCore.QObject.connect(self.ui.windowRestore.LoadArchive, 
			QtCore.SIGNAL("updateTaskProgress(int, QString)"), self.ProgressIndicators_Update)
		QtCore.QObject.connect(self.ui.windowRestore.LoadArchive, 
			QtCore.SIGNAL("taskComplete(QString, QString)"), self.ProgressIndicators_Shutdown)
		
		QtCore.QObject.connect(self.RestoreFiles, QtCore.SIGNAL("started()"),
			self.restoreFilesStarted)
		QtCore.QObject.connect(self.RestoreFiles, 
			QtCore.SIGNAL("setupTaskProgress(int, QString)"),
			self.ProgressIndicators_Setup)
		QtCore.QObject.connect(self.RestoreFiles, 
			QtCore.SIGNAL("updateTaskProgress(int, QString)"),
			self.ProgressIndicators_Update)
		QtCore.QObject.connect(self.RestoreFiles,
			QtCore.SIGNAL("taskComplete(QString, QString)"), self.restoreFilesFinished)
		
		QtCore.QObject.connect(self.LoadSnapshots, 
			QtCore.SIGNAL("ReturnSnapshotList_Main(QStandardItemModel*)"), 
			self.LoadSnapshots_SetModelList)
		QtCore.QObject.connect(self.LoadSnapshots, QtCore.SIGNAL("started()"), 
			self.loadSnapshotsStarted)
		QtCore.QObject.connect(self.LoadSnapshots, 
			QtCore.SIGNAL("setupTaskProgress(int, QString)"),
			self.ProgressIndicators_Setup)
		QtCore.QObject.connect(self.LoadSnapshots, QtCore.SIGNAL("finished()"), 
			self.loadSnapshotsFinished)
		QtCore.QObject.connect(self.LoadSnapshots, 
			QtCore.SIGNAL("taskError(QString, QString, QString)"), self.taskError)
			
		QtCore.QObject.connect(self.ManageBackups, 
			QtCore.SIGNAL("setupTaskProgress(int, QString)"), 
			self.ProgressIndicators_Setup)
		QtCore.QObject.connect(self.ManageBackups, 
			QtCore.SIGNAL("taskComplete(QString, QString)"),
			self.manageBackups_Finished)
		QtCore.QObject.connect(self.ManageBackups, QtCore.SIGNAL("toggleControls(bool)"),
			self.ui.buttonRefreshSnapshot.setEnabled)
		QtCore.QObject.connect(self.ManageBackups, QtCore.SIGNAL("toggleControls(bool)"),
			self.BackupControls_Toggle)
		QtCore.QObject.connect(self.ManageBackups, QtCore.SIGNAL("toggleControls(bool)"),
			self.RestoreOptions_Toggle)
		
		# Run Startup Events
		self.LoadSnapshots.MainList_FromSettings(settings, False)
	
	
	def setupRestoreList(self, currentModel = None):
		if currentModel == None:
			restoreList = QtGui.QStandardItemModel()
		else:
			restoreList = currentModel
			
		restoreList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_("File Name")))
		restoreList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_("Date Modified")))
		restoreList.setHorizontalHeaderItem(2, QtGui.QStandardItem(_("Relative Path")))
		restoreList.setHorizontalHeaderItem(3, QtGui.QStandardItem(_("Archive Url")))
		restoreList.setHorizontalHeaderItem(4, QtGui.QStandardItem(_("Archive Passphrase")))
		restoreList.setHorizontalHeaderItem(5, QtGui.QStandardItem(_("Restore Time")))
		restoreList.setHorizontalHeaderItem(6, QtGui.QStandardItem(_("S3 Login")))
		
		return restoreList
	
	
	def setupSnapshotsList(self, currentModel = None):
		if currentModel == None:
			snapshotsList = QtGui.QStandardItemModel()
		else:
			snapshotsList = currentModel
			
		snapshotsList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_("Snapshot History")))
		snapshotsList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_("Timestamp")))
		snapshotsList.setHorizontalHeaderItem(2, QtGui.QStandardItem(_("Int")))
		snapshotsList.setHorizontalHeaderItem(3, QtGui.QStandardItem(_("Backup Type")))
		itemMostRecent = QtGui.QStandardItem(_("Most Recent"))
		itemTime = QtGui.QStandardItem("")
		
		snapshotsList.appendRow([itemMostRecent, itemTime])
		
		return snapshotsList
	
	
	def buttonSettings_Pressed(self):
		self.ui.windowSettings.LoadBackupSettings(Settings())
		self.ui.windowSettings.show()
	
	
	def buttonAddToQueue_Pressed(self):
		if self.ui.buttonAdvancedRestore.isChecked():
			archiveUrl = str(self.AdvancedRestore_FetchRemoteUrl())
			gnu_passphrase = self.ui.txtAdvancedRestore_EncryptionPassphrase.text()
			
			try:
				if self.ui.comboRemoteOptions.currentText() == "AmazonS3":
					self.ui.windowRestore.LoadArchive_Remote(archiveUrl,
						gnu_passphrase, True)
				else:
					self.ui.windowRestore.LoadArchive_Remote(archiveUrl, 
						gnu_passphrase, False)
			except:
				msgText = _("The archive url that you entered is invalid: \n") + archiveUrl
				QtGui.QMessageBox.warning(self, 
					_("Invalid Archive Url"), msgText, QtGui.QMessageBox.Ok)
		else:
			curSelection = self.ui.treeSnapshots.currentIndex()
			restoreTime = self.modelSnapshotsList.data(curSelection.sibling(curSelection.row(), 1)).toString()
			self.ui.windowRestore.timedata = self.CreateTimeData()
			
			if restoreTime != '':
				self.ui.windowRestore.LoadArchive_FromSettings(restoreTime)
			else:
				self.ui.windowRestore.LoadArchive_FromSettings()
	
	
	def buttonRemoveFromQueue_Pressed(self):
		toRemove = self.ui.listRestoreFiles.currentIndex()
		if toRemove.isValid():
			self.modelRestoreList.removeRow(toRemove.row(), toRemove.parent())
	
	def loadSnapshotsStarted(self):
		self.RestoreOptions_Toggle(False)

	def loadSnapshotsFinished(self):
		self.RestoreOptions_Toggle(True)
		self.ui.buttonRefreshSnapshot.setEnabled(True)
		self.ProgressIndicators_Shutdown()

	def restoreFilesStarted(self):
		# Disable the Restore Options
		self.RestoreOptions_Toggle(False)
		
		self.doNotify(_("Restoring Files from Archive"))
	
	def restoreFilesFinished(self, taskName, message):
		self.RestoreOptions_Toggle(True)
		self.RestoreFiles_RemoveFromList(True)
		self.ProgressIndicators_Shutdown()
		self.doNotify(message)


	def RestoreOptions_Toggle(self, option):
		self.ui.buttonAddToQueue.setEnabled(option)
		self.ui.buttonRemoveFromQueue.setEnabled(option)
		self.ui.buttonFileInformation.setEnabled(option)
		self.ui.buttonAdvancedRestore.setEnabled(option)
		self.ui.buttonRestoreFiles.setEnabled(option)
	
	
	def RestoreDialog_Show(self, option):
		"""Show the restore files dialog after a file list has been loaded."""
		self.RestoreOptions_Toggle(option)
		if option == True:
			self.ui.windowRestore.show()
		
	
	def buttonAdvancedRestore_Pressed(self):
		self.AdvancedRestore_SetVisible(not self.ui.buttonAdvancedRestore.isChecked())
		if not self.ui.buttonAdvancedRestore.isChecked():
			self.ui.txtAdvancedRestore_Port.setText(str(utils.getPortProtocol(self.ui.comboRemoteOptions.currentText())))
	
	
	def AdvancedRestore_FetchRemoteUrl(self):
		"""Constructs a valid remote url from the Advanced Restore Dialog"""
		connectionType = self.ui.comboRemoteOptions.currentText()
		connectionPath = self.ui.txtAdvancedRestore_Path.text()
		connectionPort = self.ui.txtAdvancedRestore_Port.text()
		connectionHost = self.ui.txtAdvancedRestore_Host.text()
		connectionUsername = self.ui.txtAdvancedRestore_Username.text()
		connectionPassword = self.ui.txtAdvancedRestore_Password.text()
		if connectionType == "file":
			completeUrl = archiveurl.buildLocalUrl(connectionPath)
		elif connectionType == "AmazonS3":
			completeUrl = archiveurl.buildAmazonS3Url(os.path.normpath(str(connectionHost))) + " " + connectionUsername + " " + connectionPassword
		else:
			completeUrl = archiveurl.buildRemoteUrl(connectionType, connectionUsername, connectionPassword, connectionHost, connectionPort, connectionPath)
		return completeUrl
	
	
	def AdvancedRestore_SetVisible(self, status):
		"""
		Toggles the Advanced Restore Pane Visbile/Invisible
		
		@type status: Boolean
		@param status:
		"""
		self.ui.buttonAdvancedRestore_Test.setVisible(status)
		self.ui.buttonHelp_2.setVisible(status)
		self.ui.comboRemoteOptions.setVisible(status)
		self.ui.lblAdvancedRestore_Path.setVisible(status)
		self.ui.txtAdvancedRestore_Path.setVisible(status)
		self.ui.lblAdvancedRestore_Host.setVisible(status)
		self.ui.txtAdvancedRestore_Host.setVisible(status)
		self.ui.lblAdvancedRestore_Port.setVisible(status)
		self.ui.txtAdvancedRestore_Port.setVisible(status)
		self.ui.lblAdvancedRestore_Password.setVisible(status)
		self.ui.txtAdvancedRestore_Password.setVisible(status)
		self.ui.txtAdvancedRestore_Username.setVisible(status)
		self.ui.lblAdvancedRestore_Username.setVisible(status)
		self.ui.lblAdvancedRestore_EncryptionPassphrase.setVisible(status)
		self.ui.txtAdvancedRestore_EncryptionPassphrase.setVisible(status)
		self.ui.lblTstResult.setVisible(False)
		self.ui.treeSnapshots.setCurrentIndex(self.modelSnapshotsList.index(0, 0))
		self.ui.treeSnapshots.setEnabled(not status)
		
		if (status == True) & (self.ui.comboRemoteOptions.currentIndex() == 1):
			self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(True)
		else:
			self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(False)
	
	
	def AdvancedRestore_comboRemoteOptions_currentIndexChanged(self, index):
		"""
		Toggles the file selection button on/off depending on the
		remote option selected.
		
		@type index: int
		@param index:		
		"""
		
		def setDefaultLabels():
			self.ui.lblAdvancedRestore_Username.setText(_("Username"))
			self.ui.lblAdvancedRestore_Password.setText(_("Password"))
			self.ui.lblAdvancedRestore_Host.setText(_("Server"))
		
		def setLoginEnabled(option):
			self.ui.lblAdvancedRestore_Username.setEnabled(option)
			self.ui.lblAdvancedRestore_Password.setEnabled(option)
			self.ui.txtAdvancedRestore_Username.setEnabled(option)
			self.ui.txtAdvancedRestore_Password.setEnabled(option)
			self.ui.txtAdvancedRestore_Port.setVisible(option)
			self.ui.lblAdvancedRestore_Port.setVisible(option)
			self.ui.txtAdvancedRestore_Host.setVisible(option)
			self.ui.lblAdvancedRestore_Host.setVisible(option)
			self.ui.txtAdvancedRestore_Path.setVisible(option)
			self.ui.lblAdvancedRestore_Path.setVisible(option)
		
		if index == "file":
			self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(True)
			self.ui.txtAdvancedRestore_Port.setText("")
			setDefaultLabels()
			setLoginEnabled(False)
			self.ui.txtAdvancedRestore_Path.setVisible(True)
			self.ui.lblAdvancedRestore_Path.setVisible(True)
		elif index == "AmazonS3":
			self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(False)
			self.ui.txtAdvancedRestore_Port.setText("")
			setLoginEnabled(True)
			self.ui.lblAdvancedRestore_Username.setText(_("Access ID"))
			self.ui.lblAdvancedRestore_Password.setText(_("Secret Key"))
			self.ui.lblAdvancedRestore_Host.setText(_("Bucket"))
			self.ui.txtAdvancedRestore_Port.setVisible(False)
			self.ui.lblAdvancedRestore_Port.setVisible(False)
			self.ui.txtAdvancedRestore_Path.setVisible(False)
			self.ui.lblAdvancedRestore_Path.setVisible(False)
		else:
			self.ui.buttonAdvancedRestore_LoadDirectory.setVisible(False)
			self.ui.txtAdvancedRestore_Port.setText(str(utils.getPortProtocol(index)))
			setDefaultLabels()
			setLoginEnabled(True)
	
	
	def AdvancedRestore_PickFolder(self):
		folderName = QtGui.QFileDialog.getExistingDirectory()
		
		if folderName != '':
			self.ui.txtAdvancedRestore_Path.setText(folderName)
	
	def ProgressIndicators_SetVisible(self, status):
		self.ui.statusLabel.setVisible(status)
		self.ui.backupStatus.setVisible(status)
		self.ui.buttonStopTask.setVisible(status)
		
	
	def BackupControls_Toggle(self, option):
		self.ui.buttonBackup.setEnabled(option)
		self.ui.buttonRefreshSnapshot.setEnabled(option)
		self.ui.buttonNameSnapshot.setEnabled(option)

	
	def TestAdvancedRestoreOptions(self):
		"""Test the connection to see if it is valid"""
		connectionType = self.ui.comboRemoteOptions.currentText()
		connectionPath = self.ui.txtAdvancedRestore_Path.text()
		connectionPort = self.ui.txtAdvancedRestore_Port.text()
		connectionHost = self.ui.txtAdvancedRestore_Host.text()
		connectionUsername = self.ui.txtAdvancedRestore_Username.text()
		connectionPassword = self.ui.txtAdvancedRestore_Password.text()
		self.ui.buttonAdvancedRestore_Test.setEnabled(False)
		self.doTestConnection(connectionType, connectionUsername, connectionPassword, connectionHost, connectionPort, connectionPath)
	
	
	def buttonBackup_Pressed(self):
		settings = Settings()
		self.BackupJob.StartBackup(settings)

	
	def backupStarted(self):
		"""Disable Backup Related Ui Controls and notify user that the backup
		has started (Ubuntu 9.04, Mac OS X)"""
		
		self.BackupControls_Toggle(False)
		self.doNotify(_("Starting File Backup"))
		self.ui.windowRestore.changed = True
	
	
	def backupFinished(self, taskName, message):
		"""Notify User that the Backup has ended
		(Ubuntu 9.04, Mac OS X)
		
		@type taskName: String
		@param taskName:
		@type message: String
		@param message:
		"""
		
		self.BackupControls_Toggle(True)
		self.RefreshSnapshotList(True)
		self.doNotify(message)
	
	
	def taskError(self, taskName, error, message):
		"""
		Notify user that there was an error.
		
		@type taskName: String
		@param taskName:
		@type error: String (constant)
		@param error: Mostly static defined in Backgroundtask Object
		@type message: String
		@param message:
		"""
		if error == "host down":
			QtGui.QMessageBox.warning(self, taskName + " : " + _("location not available"), message, QtGui.QMessageBox.Ok)
		else:
			self.doNotify(message)
	
	
	def manageBackups_Finished(self, taskName, message):
		"""Notify user that the operation has completed successfully.
		Refresh the snapshot list.
		
		@type taskName: String
		@param taskName:
		@type message: String
		@param message:
		"""
		self.RefreshSnapshotList(True)
		self.doNotify(message)
	
	
	def ShowRemoteResults(self, tstResult):
		"""
		Updates the settings view based on the results of the connection test.
		
		@type tstResult: Boolean
		@param tstResult:
		"""
		self.ui.lblTstResult.setVisible(True)
		
		if tstResult == True:
			imageSuccess = QtGui.QPixmap(':/SVG/Resources/Dialog-Ok.svg')
			self.ui.lblTstResult.setPixmap(imageSuccess)
		if tstResult == False:
			imageFailure = QtGui.QPixmap(':/SVG/Resources/Dialog-Cancel.svg')
			self.ui.lblTstResult.setPixmap(imageFailure)
		
		self.ui.buttonAdvancedRestore_Test.setEnabled(True)
	
	
	def RestoreFiles_Start(self):
		settings = Settings()
		RestoreFiles = { }
		i = 0
		
		if self.modelRestoreList.rowCount() == 0:
			QtGui.QMessageBox.warning(self, _("Warning"),
						  _("No files or folders on the queue to restore."),
						  QtGui.QMessageBox.Ok)
			return
		
		while i < self.modelRestoreList.rowCount():
			restoreKey = QtCore.QPersistentModelIndex(self.modelRestoreList.index(i, 0))
			RelPath = self.modelRestoreList.data(restoreKey.sibling(restoreKey.row(), 2)).toString()
			ArchiveUrl = self.modelRestoreList.data(restoreKey.sibling(restoreKey.row(), 3)).toString()
			PassPhrase = str(self.modelRestoreList.data(restoreKey.sibling(restoreKey.row(), 4)).toString())
			RestoreTime = self.modelRestoreList.data(restoreKey.sibling(restoreKey.row(), 5)).toString()
			S3_LoginInfo = str(self.modelRestoreList.data(restoreKey.sibling(restoreKey.row(), 6)).toString())
			
			RestoreFiles[restoreKey] = [RelPath, ArchiveUrl, PassPhrase, 
				RestoreTime, S3_LoginInfo]
			i = i+1

		if settings.Advanced_DefaultRestoreEnabled == True:
			self.RestoreFiles.StartRestore(RestoreFiles, settings)
		else:
			self.RestoreFiles.StartRestore(RestoreFiles, settings, QtGui.QFileDialog.getExistingDirectory())
	
	
	def RestoreFiles_RemoveFromList(self, operationComplete):
		"""
		Clears the restore queue of all files.
		
		@type operationComplete: Boolean
		@param operationComplete:		
		"""
		if operationComplete:
			self.modelRestoreList.clear()
			self.setupRestoreList(self.modelRestoreList)
			self.ui.listRestoreFiles.setColumnHidden(2, True)
			self.ui.listRestoreFiles.setColumnHidden(3, True)
			self.ui.listRestoreFiles.setColumnHidden(4, True)
			self.ui.listRestoreFiles.setColumnHidden(5, True)
			self.ui.listRestoreFiles.setColumnHidden(6, True)
			self.ui.listRestoreFiles.setColumnWidth(0, 190)
			self.ui.listRestoreFiles.setColumnWidth(1, 150)
		
	def ProgressIndicators_Setup(self, maximum, message = None):
		"""
		Initialize the progress bar, set name and current task
		
		@type maximum: Int
		@param maximum: the maximum of elements to process
		@type message: String
		@param message: default=None, informative message or task name 
		"""
		self.ProgressIndicators_SetVisible(True)
		self.ui.backupStatus.setMinimum(0)
		self.ui.backupStatus.setMaximum(maximum)

		if message != None:
			self.ui.statusLabel.setText(QtCore.QString(message))
	
	
	def ProgressIndicators_Update(self, progress, message = None):
		"""
		Updates the progress bar 
		
		@type progress: Int
		@param progress:
		@type message: String
		@param message: default=None, informative message or task name 
		"""
		self.ui.backupStatus.setValue(progress)
		if message != None:
			self.ui.statusLabel.setText(message)
		
	
	def ProgressIndicators_Shutdown(self):
		self.ProgressIndicators_SetVisible(False)
		self.ui.statusLabel.setText("")
		self.ui.backupStatus.setMinimum(0)
		self.ui.backupStatus.setMaximum(0)
	
	
	def buttonRefreshSnapshotList_Pressed(self):
		self.RefreshSnapshotList(True)
	
	
	def RefreshSnapshotList(self, sync = False):
		"""
		Refresh the snapshot list
		
		@type sync: Boolean
		@param sync: default = False
		"""
		settings = Settings()
		self.ui.buttonRefreshSnapshot.setEnabled(False)
		
		# Clear the Existing Snapshots, Reset Column Headers
		self.modelSnapshotsList.clear()
		self.modelSnapshotsList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_("Snapshot History")))
		self.modelSnapshotsList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_("Timestamp")))
		self.ui.treeSnapshots.setColumnHidden(1, True)
		
		# Set First Item to "Most Recent"
		itemMostRecent = QtGui.QStandardItem(_("Most Recent"))
		itemTime = QtGui.QStandardItem("")
		
		self.modelSnapshotsList.appendRow([itemMostRecent, itemTime])
		self.LoadSnapshots.MainList_FromSettings(settings, sync)
		self.ui.treeSnapshots.setCurrentIndex(itemMostRecent.index())
	
	
	def LoadSnapshots_SetModelList(self, transferModel):
		"""
		Loads the available snapshots to treeSnapshots
		
		@type transferModel: List Model
		@param transferModel: 
		"""
		receiveModel = self.modelSnapshotsList
		i = 0
		
		iconFullBackup = QtGui.QIcon(QtGui.QPixmap(":/PNG/Resources/Full-Backup-Emblem.png"))
		iconIncBackup = QtGui.QIcon(QtGui.QPixmap(":/PNG/Resources/Inc-Backup-Emblem.png"))
		numRows = transferModel.rowCount()
		
		while i < numRows:
			moveRow = transferModel.takeRow(0)
			receiveModel.appendRow(moveRow)
			backupType = receiveModel.data(receiveModel.index(i+1, 3)).toString()
			itemTimePretty = receiveModel.itemFromIndex(receiveModel.index(i+1, 0))
			
			if backupType == "full":
				itemTimePretty.setIcon(iconFullBackup)
			else:
				itemTimePretty.setIcon(iconIncBackup)
			
			i = i+1
		
		self.ui.treeSnapshots.setColumnHidden(1, True)
		self.ui.treeSnapshots.setColumnHidden(2, True)
		self.ui.treeSnapshots.setColumnHidden(3, True)
	
	
	def buttonExit_Pressed(self):
		self.close()
	
	
	def CreateTimeData(self):
		timeData = { }
		i = 1
		
		while i < self.modelSnapshotsList.rowCount():
			restoreKey = self.modelSnapshotsList.index(i, 0)
			timeInt = self.modelSnapshotsList.data(restoreKey.sibling(restoreKey.row(), 2)).toInt()
			timeString = self.modelSnapshotsList.data(restoreKey.sibling(restoreKey.row(), 1)).toString()
			timePretty = self.modelSnapshotsList.data(restoreKey.sibling(restoreKey.row(), 0)).toString()
			timeData[timeInt[0]] = [timeString, timePretty]
			i = i+1
		
		if self.modelSnapshotsList.rowCount() == 1:
			timeString = utils.time_GetCurrentTime()
			timeInt = utils.time_String2Int(timeString)
			timePretty = utils.time_String2Pretty(timeString)
			timeData[timeInt] = [timeString, timePretty]
			
		return timeData
	
	
	def doNotify(self, userMessage):
		"""
		Notifies the user, only if the notify is enabled.
		"""
		settings = Settings()
		if settings.Options_EnableNotifications == True:
			self.NotifyUser.Notify(globals.APPLICATIONNAME, str(userMessage))

		
	def buttonOpenLog_Pressed(self):
		self.ui.windowLogging.show()

	def questionPrompt(self, title, message, th):
		"""
		Prompts a question to the user from an background thread
		
		@type title: String
		@param title:
		@type message: String
		@param message:
		@type th: QThread Object
		@param th: 
		"""
		th.acquire()
		if(QtGui.QMessageBox.question(self, title, message, QtGui.QMessageBox.Yes|QtGui.QMessageBox.No) == QtGui.QMessageBox.Yes):
			self.BackupJob.restart = True
		else:
			self.BackupJob.restart = False
		th.notify()
		th.release()
		
	def buttonStopTask_Pressed(self):
		"""
		stop the task, if it's possible, since everything uses a loop it's easy
		"""
		#get first in the list
		currentTask = backgroundtask.running_threads.pop()
		#first check if it's still needed might just be finished
		if currentTask.isRunning():
			currentTask.stop = True