/**
 * @file sieve.cpp
 *
 * @section LICENSE
 *
 * The MIT License
 *
 * @copyright Copyright (c) 2022 TileDB, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Demo program sieve of Eratosthenes, function components for block
 * (and parallelizable) implementation.
 *
 * The block sieve algorithm begins by sequentially finding all the primes in
 * [2, sqrt(n)).  Using that initial set of primes, the algorithm finds primes
 * in each block of numbers delimited by
 *
 *       [sqrt(n) + p*block_size, sqrt(n) + (p+1)*block_size)
 *
 *  for p in [0, n/blocksize).
 *
 * This file provides a decomposition of that computation into the following
 * five tasks:
 *   input_body() generates p, a sequence of integers, starting at 0
 *   gen_range() creates a bitmap for indicating primality (or not)
 *   range_sieve() applies sieve, to block p, using initial set of
 *     sqrt(n) primes and records results in bitmap obtained from
 *     gen_range()
 *   sieve_to_primes_part() generates a list of prime numbers from the
 *     bitmap generated by range_sieve()
 *   output_body() saves the list of primes in a vector at location p+1.
 *     The original set of sqrt(n) primes is stored at loccation 0.
 *   A set of n/block_size parallel task chains is launched to carry
 *     out the computation.
 *
 * These functions take regular values as input parameters and return regular
 * values. They can be composed together to produce the sieve algorithm
 * described above.
 */

#include <cassert>
#include <chrono>
#include <cmath>
#include <functional>
#include <future>
#include <iostream>
#include <string>
#include <tuple>
#include <vector>

/**
 * Nullify the CHECK macro that might be spread throughout the code for
 * debugging/testing purposes.
 */
#define CHECK(str)

#include "experimental/tiledb/common/dag/ports/ports.h"
#include "experimental/tiledb/common/dag/ports/test/pseudo_nodes.h"
#include "experimental/tiledb/common/dag/state_machine/fsm.h"
#include "experimental/tiledb/common/dag/state_machine/item_mover.h"
#include "experimental/tiledb/common/dag/state_machine/policies.h"

using namespace tiledb::common;
using namespace std::placeholders;

bool debug = false;
bool chart = false;

/**
 * Some convenience type aliases
 */
template <class bool_t>
using part_info =
    std::tuple<size_t, size_t, size_t, std::shared_ptr<std::vector<bool_t>>>;
using prime_info = std::tuple<size_t, std::shared_ptr<std::vector<size_t>>>;

/**
 * Takes a vector of "bool" (which may be actual bool, or may be char) and
 * extracts the indicated prime numbers.
 */
template <class bool_t>
auto sieve_to_primes(std::vector<bool_t>& sieve) {
  std::vector<size_t> primes;

  for (size_t i = 2; i < sieve.size(); ++i) {
    if (sieve[i]) {
      primes.push_back(i);
    }
  }
  return primes;
}

template <class bool_t>
auto sieve_to_primes(
    std::vector<bool_t>& sieve,
    std::vector<size_t>& base_primes,
    size_t sqrt_n) {
  std::vector<size_t> primes(base_primes);

  for (size_t i = sqrt_n; i < sieve.size(); ++i) {
    assert(i < sieve.size());
    if (sieve[i]) {
      primes.push_back(i);
    }
  }
  return primes;
}

template <class bool_t>
auto sieve_seq(size_t n) {
  std::vector<bool_t> sieve(n, true);

  sieve[0] = sieve[1] = true;

  size_t sqrt_n = static_cast<size_t>(std::ceil(std::sqrt(n)));

  for (size_t i = 2; i < sqrt_n; ++i) {
    assert(i < sieve.size());
    if (sieve[i]) {
      for (size_t j = i * i; j < n; j += i) {
        assert(j < sieve.size());
        sieve[j] = false;
      }
    }
  }

  return sieve;
}

/**
 * Generate a (thread safe) sequence of integers, starting at 0
 * @return integer, value one greater than previously returned
 */
class input_body {
  std::atomic<size_t> p{0};

 public:
  input_body()
      : p{0} {
  }
  input_body(const input_body& rhs)
      : p{rhs.p.load()} {
  }
  size_t operator()() {
    if (debug)
      std::cout << "input_body " << p << std::endl;
    return p++;
  }
};

/**
 * Create a bitmap for storing sieve results
 * @tparam type for bitmap
 * @param the block number to create bitmap for
 * @return tuple with block number, bitmap range, and bitmap
 */
template <class bool_t>
auto gen_range(size_t p, size_t block_size, size_t sqrt_n, size_t n) {
  if (debug)
    std::cout << "gen_range " << p << std::endl;

  size_t sieve_start = std::min(sqrt_n + p * block_size, n);
  size_t sieve_end = std::min(sieve_start + block_size, n);
  return std::make_tuple(
      p + 1,
      sieve_start,
      sieve_end,
      std::make_shared<std::vector<bool_t>>(sieve_end - sieve_start, true));
};

/**
 * Find primes in indicated range and record in bitmap
 * @param tuple with block number, bitmap range, and bitmap
 * @return tuple with block number, bitmap range, and bitmap
 */
template <class bool_t>
auto range_sieve(
    const part_info<bool_t>& in, const std::vector<size_t>& base_primes) {
  auto [p, sieve_start, sieve_end, local_sieve] = in;
  if (debug)
    std::cout << "range_sieve " << p << std::endl;

  for (size_t i = 0; i < base_primes.size(); ++i) {
    assert(i < base_primes.size());
    size_t pr = base_primes[i];

    size_t q = (pr + sieve_start - 1) / pr;
    q *= pr;

    for (size_t j = q - sieve_start; j < sieve_end - sieve_start; j += pr) {
      assert(j < local_sieve->size());
      (*local_sieve)[j] = false;
    }
  }

  return in;
};

/**
 * Create list of primes from bitmap
 * @param tuple with block number, bitmap range, and bitmap
 * @return tuple with block number and shared_ptr to vector of primes
 */
template <class bool_t>
auto sieve_to_primes_part(const part_info<bool_t>& in) {
  auto [p, sieve_start, sieve_end, local_sieve] = in;
  if (debug)
    std::cout << "sieve_to_primes_part " << p << std::endl;

  std::vector<size_t> primes;
  for (size_t i = 0; i < local_sieve->size(); ++i) {
    assert(i < local_sieve->size());
    if ((*local_sieve)[i]) {
      primes.push_back(i + sieve_start);
    }
  }
  return std::make_tuple(p, std::make_shared<std::vector<size_t>>(primes));
};

/**
 * Store list of primes in vector
 * @param tuple with block number and shared_ptr to vector of primes
 */
auto output_body(
    const prime_info& in,
    std::vector<std::shared_ptr<std::vector<size_t>>>& prime_list) {
  auto [p, primes] = in;
  if (debug)
    std::cout << "output_body " << p << " / " << prime_list.size() << std::endl;
  assert(p < prime_list.size());
  prime_list[p] = primes;
};

/**
 * Some structures to simulate a task graph.  Here, a "task graph" is a tuple of
 * pseudo nodes.  We later construct a vector of these graphs so that we can run
 * multiple graphs in parallel.
 *
 * @todo (IMPORTANT) Only run a subset of graphs at a time, rather than all of
 * them.
 *
 * @todo Run with TileDB ThreadPool.
 */
template <class... Ts>
using the = std::vector<std::tuple<Ts...>>;

template <class Block>
using AsyncMover2 = ItemMover<AsyncPolicy, two_stage, Block>;

template <template <class> class AsyncMover, class bool_t>
using GraphType = the<
    ProducerNode<AsyncMover, size_t>,

    FunctionNode<AsyncMover, size_t, AsyncMover, part_info<bool_t>>,

    FunctionNode<AsyncMover, part_info<bool_t>, AsyncMover, part_info<bool_t>>,

    FunctionNode<AsyncMover, part_info<bool_t>, AsyncMover, prime_info>,

    ConsumerNode<AsyncMover, prime_info>>;  //    thingo;

template <class bool_t>
GraphType<AsyncMover2, bool_t> graph;

/**
 * Main sieve function
 *
 * @brief Generate primes from 2 to n using sieve of Eratosthenes.
 * @tparam bool_t the type to use for the "bitmap"
 * @param n upper bound of sieve
 * @param block_size how many primes to search for given a base set of primes
 */
template <class bool_t>
auto sieve_async_block(size_t n, size_t block_size, size_t width) {
  input_body gen;

  std::vector<std::thread> threads;

  threads.clear();
  graph<bool_t>.clear();

  size_t sqrt_n = static_cast<size_t>(std::ceil(std::sqrt(n)));

  /* Generate base set of sqrt(n) primes to be used for subsequent sieving */
  auto first_sieve = sieve_seq<bool_t>(sqrt_n);
  std::vector<size_t> base_primes = sieve_to_primes(first_sieve);

  /* Store vector of list of primes (each list generated by separate task chain)
   */
  std::vector<std::shared_ptr<std::vector<size_t>>> prime_list(
      n / block_size + 2 + n % block_size);

  if (debug)
    std::cout << prime_list.size() << std::endl;

  prime_list[0] = std::make_shared<std::vector<size_t>>(base_primes);

  size_t rounds = (n / block_size + 2) / width + 1;

  using time_t = std::chrono::time_point<std::chrono::high_resolution_clock>;

  std::vector<std::tuple<size_t, size_t, std::string, double>> timestamps(
      width * rounds * 20);
  std::atomic<size_t> time_index{0};
  time_t start_time = std::chrono::high_resolution_clock::now();

  //  print_types(std::bind(gen_range<bool_t>, _1, block_size, sqrt_n, n));

  graph<bool_t>.reserve(5 * width);

  /**
   * Create the graphs by emplacing the nodes for each one into a vector.
   */
  for (size_t w = 0; w < width; ++w) {
    if (debug)
      std::cout << "w: " << w << std::endl;

    graph<bool_t>.emplace_back(

        std::ref(gen),

        std::bind(gen_range<bool_t>, _1, block_size, sqrt_n, n),

        std::bind(range_sieve<bool_t>, _1, std::cref(base_primes)),

        sieve_to_primes_part<bool_t>,

        std::bind(output_body, _1, std::ref(prime_list))

    );

    /**
     * Connect the nodes in the graph;
     */
    attach(
        std::get<0>(graph<bool_t>.back()), std::get<1>(graph<bool_t>.back()));
    attach(
        std::get<1>(graph<bool_t>.back()), std::get<2>(graph<bool_t>.back()));
    attach(
        std::get<2>(graph<bool_t>.back()), std::get<3>(graph<bool_t>.back()));
    attach(
        std::get<3>(graph<bool_t>.back()), std::get<4>(graph<bool_t>.back()));

    /**
     * Launch a thread to execute the graph.  We use the "abundant thread"
     * scheduling policy, which launches a thread to run each node in each
     * graph.
     *
     * @todo Only launch a subset of the graphs and launch new ones as running
     * ones complete.  It might be easier to do this with `std::async` rather
     * than `std::thread`.
     */
    threads.emplace_back(

        [&, w]() {
          if (debug)
            std::cout << "thread 0 " << w << std::endl;
          size_t N = rounds;
          while (N--) {
            if (chart)
              timestamps[time_index++] = std::make_tuple(
                  time_index.load(),
                  0,
                  "start",
                  std::chrono::duration_cast<std::chrono::microseconds>(
                      (time_t)std::chrono::high_resolution_clock::now() -
                      start_time)
                      .count());
            std::get<0>(graph<bool_t>[w]).get();
            if (chart)
              timestamps[time_index++] = std::make_tuple(
                  time_index.load(),
                  0,
                  "stop",
                  std::chrono::duration_cast<std::chrono::microseconds>(
                      (time_t)std::chrono::high_resolution_clock::now() -
                      start_time)
                      .count());
          }
        });

    threads.emplace_back([&, w]() {
      if (debug)
        std::cout << "thread 1 " << w << std::endl;

      size_t N = rounds;

      while (N--) {
        if (chart)
          timestamps[time_index++] = std::make_tuple(
              time_index.load(),
              1,
              "start",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count());
        std::get<1>(graph<bool_t>[w]).run();
        if (chart)
          timestamps[time_index++] = std::make_tuple(
              time_index.load(),
              1,
              "stop",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count());
      }
    });

    threads.emplace_back([&, w]() {
      if (debug)
        std::cout << "thread 2 " << w << std::endl;
      size_t N = rounds;
      while (N--) {
        if (chart)
          timestamps[time_index++] = std::make_tuple(
              time_index.load(),
              2,
              "start",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count());

        std::get<2>(graph<bool_t>[w]).run();
        if (chart)
          timestamps[time_index++] = {
              time_index.load(),
              2,
              "stop",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count()};
      }
    });

    threads.emplace_back([&, w]() {
      if (debug)

        std::cout << "thread 3 " << w << std::endl;
      size_t N = rounds;
      while (N--) {
        if (chart)
          timestamps[time_index++] = std::make_tuple(
              time_index.load(),
              3,
              "start",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count());

        std::get<3>(graph<bool_t>[w]).run();
        if (chart)
          timestamps[time_index++] = {
              time_index.load(),
              3,
              "stop",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count()};
      }
    });

    threads.emplace_back([&, w]() {
      if (debug)

        std::cout << "thread 4 " << w << std::endl;
      size_t N = rounds;
      while (N--) {
        if (chart)
          timestamps[time_index++] = std::make_tuple(
              time_index.load(),
              4,
              "start",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count());
        std::get<4>(graph<bool_t>[w]).put();
        if (chart)
          timestamps[time_index++] = {
              time_index.load(),
              4,
              "stop",
              std::chrono::duration_cast<std::chrono::microseconds>(
                  (time_t)std::chrono::high_resolution_clock::now() -
                  start_time)
                  .count()};
      }
    });
  }

  if (debug)
    std::cout << "threads size: " << threads.size() << std::endl;

  for (size_t i = 0; i < threads.size(); ++i) {
    threads[i].join();
  }

  if (chart) {
    std::for_each(
        timestamps.begin(), timestamps.begin() + time_index, [](auto&& a) {
          auto&& [idx, id, str, tm] = a;

          size_t col = 0;
          if (id < 5) {
            col = 2 * id;
          } else {
            col = 2 * (id % 5) + 1;
          }
          col = id;
          std::cout << idx << "\t" << id << "\t" << tm << "\t";
          for (size_t k = 0; k < col; ++k) {
            std::cout << "\t";
          }
          std::cout << str << std::endl;
        });
  }

  return prime_list;
}

template <class F>
auto timer_2(F f, size_t max, size_t blocksize, size_t width) {
  auto start = std::chrono::high_resolution_clock::now();
  auto s = f(max, blocksize, width);
  auto stop = std::chrono::high_resolution_clock::now();

  size_t num = 0;
  for (auto& j : s) {
    if (j) {
      num += j->size();
    }
  }
  std::cout << num << ": " << std::endl;

  return stop - start;
}

int main(int argc, char* argv[]) {
  size_t number = 100'000'000;
  size_t block_size = 100;

  if (argc >= 2) {
    number = std::stol(argv[1]);
  }
  if (argc >= 3) {
    block_size = std::stol(argv[2]);
  }

#if 0
#ifdef __clang__
  std::cout << "Found __clang__" << std::endl;
#endif
#ifdef __GNUG__
  std::cout << "Found __GNUG__" << std::endl;
#endif

#ifdef _GLIBCXX_USE_TBB_PAR_BACKEND
#if _GLIBCXX_USE_TBB_PAR_BACKEND
  std::cout << "Found _GLIBCXX_USE_TBB_PAR_BACKEND is true" << std::endl;
#else
  std::cout << "Found _GLIBCXX_USE_TBB_PAR_BACKEND is false" << std::endl;
#endif
#else
  std::cout << "Did not find _GLIBCXX_USE_TBB_PAR_BACKEND" << std::endl;
#endif

  tbb::global_control(tbb::global_control::max_allowed_parallelism, 2);
#endif
  size_t width = 24;

  for (size_t i = 0; i < 3; ++i) {
    //  auto using_bool_async_block =
    //      timer_2(sieve_async_block<bool>, number, block_size * 1024,
    //      width);
    auto using_char_async_block =
        timer_2(sieve_async_block<char>, number, block_size * 1024, width);

    //  std::cout << "Time using bool async block: "
    //            << std::chrono::duration_cast<std::chrono::milliseconds>(
    //                   using_bool_async_block)
    //                   .count()
    //            << "\n";
    std::cout << "Time using char async block: "
              << std::chrono::duration_cast<std::chrono::milliseconds>(
                     using_char_async_block)
                     .count()
              << "\n";
  }

  return 0;
}
