/* Copyright (C) 2012 Kevin W. Hamlen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * The latest version of this program can be obtained from
 * http://songs.sourceforge.net.
 *
 *
 * This program generates a hyphenation report from an .sbd (song book data)
 * file. A hyphenation report consists of a series of warnings, one for each
 * instance where chord change or measure bar in the .sbd file divides a word
 * at a point that LaTeX does not recognize as a valid hyphenation point for
 * that word.
 *
 * The report generation process occurs in three stages:
 *   (1) Use this program to generate a hyphenation-checking .tex file from
 *   the .sbd file by executing:
 *         sbdchk -x filename.sbd filename.tex
 *   (2) Run the resulting hyphenation-checking .tex file through LaTeX:
 *         pdflatex filename.tex
 *   (3) Generate the final hyphenation report (a .txt file) by using this
 *   program to analyze the .log file generated by LaTeX in step 2:
 *         sbdchk -r filename.log filename.hyp
 * The resulting report is contained in filename.hyp.
 *
 * This program is written in straight ansi C, since at present that is
 * probably the most widely known and compatible language around. It should
 * compile on unix with
 *
 *   cc sbdchk.c -o sbdchk
 *
 * On Windows, you'll need to find yourself a C compiler in order to compile
 * it, but the file distribution should have come with a pre-compiled binary
 * for Windows already.
 */

#if HAVE_CONFIG_H
#  include "config.h"
#else
#  include "vsconfig.h"
#endif

#include "sbdchk.h"
#include "chars.h"

#if HAVE_STDIO_H
#  include <stdio.h>
#endif

#if HAVE_STRING_H
#  include <string.h>
#elif HAVE_STRINGS_H
#  include <strings.h>
#endif

#if HAVE_SETLOCALE
#if HAVE_LOCALE_H
#  include <locale.h>
#endif
#endif

extern int extracthyphens(FILE *fin, FILE *fout);
extern int reporthyphens(FILE *fin, FILE *fout);

#if HAVE_STRRCHR
#  define STRRCHR(x,y) strrchr((x),(y))
#else
#  define STRRCHR(x,y) local_strrchr((x),(y))
char *
local_strrchr(s,c)
  const char *s;
  int c;
{
  const char *t;
  if (!s) return NULL;
  for (t=s; *t; ++s) ;
  for (; t>=s; --s)
    if (*t == c) return t;
  return NULL;
}
#endif

int
main(argc,argv)
  int argc;
  char *argv[];
{
  FILE *fin, *fout;
  int result;
  int i;
  int xrflag;
  const char *inname, *outname, *locale;

  /* Process the command line options. */
  xrflag = 0;
  inname = outname = locale = NULL;
  for (i=1; i<argc; ++i)
  {
    if (!strcmp(argv[i],"-v") || !strcmp(argv[i],"--version"))
    {
      printf("sbdchk " VERSION "\n"
        "Copyright (C) 2012 Kevin W. Hamlen\n"
        "License GPLv2: GNU GPL version 2 or later"
        " <http://gnu.org/licenses/gpl.html>\n"
        "This is free software: you are free to change and redistribute it.\n"
        "There is NO WARRANTY, to the extent permitted by law.\n");
      return 0;
    }
    else if (!strcmp(argv[i],"-h") || !strcmp(argv[i],"--help"))
    {
      printf(
        "Syntax 1: %s -x [options] infile.sbd [tempfile.tex]\n"
        "  Extract words from song data file infile.sbd to be checked.  The\n"
        "  resulting outfile.tex should be passed through LaTeX to yield a\n"
        "  tempfile.log file used in Syntax 2 (below).\n"
        "\n"
        "Syntax 2: %s -r [options] tempfile.log [outfile.hyp]\n"
        "  Analyze tempfile.log (produced by LaTeX; see Syntax 1), yielding\n"
        "  a text file named outfile.hyp that reports any hyphenation errors\n"
        "  or other errors discovered.\n"
        "\n"
        "Available options:\n"
        "  -l LOCALE            Overrides the default system locale (affects\n"
        "  --locale LOCALE       (how non-English characters are parsed).\n"
        "                        See local system help for valid LOCALE's.\n"
        "\n"
        "  -h                   Displays this help file and stops.\n"
        "  --help\n"
        "\n"
        "  -o FILE              Sets the output filename.\n"
        "  --output FILE\n"
        "\n"
        "  -v                   Print version information and stop.\n"
        "  --version\n"
        "\n"
        "If omitted, output files default to the input filename but with\n"
        "the file extension renamed to .tex (for Syntax 1) or .hyp (for\n"
        "Syntax 2). To read or write to stdin or stdout, use '-' in place of\n"
        "the input or output filename.\n"
        "\n"
        "For support go to: http://songs.sourceforge.net\n", argv[0], argv[0]);
      return 0;
    }
    else if (!strcmp(argv[i],"-l") || !strcmp(argv[i],"--locale"))
    {
      if (locale)
      {
        fprintf(stderr, "songidx: multiple locales specified\n");
        return 2;
      }
      else if (++i < argc)
        locale = argv[i];
      else
      {
        fprintf(stderr, "songidx: %s option requires an argument\n", argv[i-1]);
        return 2;
      }
    }
    else if (!strcmp(argv[i],"-x") || !strcmp(argv[i],"--extract"))
    {
      if (xrflag == 2)
      {
        fprintf(stderr, "sbdchk: only one of -x and -r may be specified\n");
        return 2;
      }
      xrflag = 1;
    }
    else if (!strcmp(argv[i],"-r") || !strcmp(argv[i],"--report"))
    {
      if (xrflag == 1)
      {
        fprintf(stderr, "sbdchk: only one of -x and -r may be specified\n");
        return 2;
      }
      xrflag = 2;
    }
    else if (!strcmp(argv[i],"-o") || !strcmp(argv[i],"--output"))
    {
      if (outname)
      {
        fprintf(stderr, "sbdchk: multiple output files specified\n");
        return 2;
      }
      else if (++i < argc)
        outname = argv[i];
      else
      {
        fprintf(stderr, "sbdchk: %s option requires an argument\n", argv[i-1]);
        return 2;
      }
    }
    else if ((argv[i][0]=='-') && (argv[i][1] != '\0'))
    {
      fprintf(stderr, "sbdchk: unknown option %s\n", argv[i]);
      return 2;
    }
    else if (!inname) inname=argv[i];
    else if (!outname) outname=argv[i];
    else
    {
      fprintf(stderr, "sbdchk: too many command line arguments\n");
      return 2;
    }
  }

#if HAVE_SETLOCALE
  if (!locale)
    (void) setlocale(LC_ALL, "");
  else if (!setlocale(LC_ALL, locale))
  {
    fprintf(stderr, "songidx: invalid locale: %s\n", locale);
    return 2;
  }
#endif

  if (!xrflag)
  {
    fprintf(stderr, "sbdchk: one of -x or -r must be specified\n");
    return 2;
  }
  if (!inname)
  {
    fprintf(stderr, "sbdchk: no input file specified\n");
    return 2;
  }
  if (!outname)
  {
    if (!strcmp(inname,"-"))
      outname = "-";
    else
    {
      char *b;
      const char *p1 = STRRCHR(inname,'.');
      if ((p1 < STRRCHR(inname,'/')) || (p1 < STRRCHR(inname,'\\')))
        p1 = inname+strlen(inname);
      b = (char *) malloc(p1-inname+5);
      strncpy(b,inname,p1-inname);
      strcpy(b+(p1-inname), (xrflag==1) ? ".tex" : ".hyp");
      outname = b;
    }
  }

  /* Open files for reading and writing. */
  if (!strcmp(inname,"-"))
    fin = stdin;
  else if ((fin = fopen(inname, "r")) == NULL)
  {
    fprintf(stderr, "sbdchk: Cannot open \"%s\" for reading.\n", inname);
    return 2;
  }
  if (!strcmp(outname,"-"))
    fout = stdout;
  else if ((fout = fopen(outname, "w")) == NULL)
  {
    fprintf(stderr, "sbdchk: Cannot open \"%s\" for writing.\n", outname);
    return 2;
  }

  if (xrflag==1)
  {
    /* Stage 1:
     * We've been asked to parse a song book .sbd file and generate
     * a .tex file that, when run through LaTeX, will result in a log
     * that shows the valid hyphenation points for every word found
     * in the original .sbd file. */
    result = extracthyphens(fin,fout);
  }
  else
  {
    /* Stage 2:
     * We've been asked to read the .log file generated by LaTeX
     * when it processed the .tex file generated in stage 1. From that
     * .log file, we can print out a message for any chord that
     * breaks a word in a place that LaTeX doesn't recognize as a valid
     * hyphenation point. */
    result = reporthyphens(fin,fout);
  }
  if (fin != stdin) fclose(fin);
  if (fout == stdout)
    fflush(fout);
  else
    fclose(fout);
  return result;
}
