/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <stdlib.h>
#include "gettext.h"
#include "input.h"
#include "interpolation.h"
#include "scene.h"
#include "time_controls.h"

#define _(String) gettext(String)

/*
 * Handler for forward button press.  First argument is main application
 * widget, second argument is unused.
 */
static void     on_forward_clicked(GtkWidget * widget, gpointer);

/*
 * Handler for backward button press.  First argument is main application
 * widget, second argument is unused.
 */
static void     on_backward_clicked(GtkWidget * widget, gpointer data);

/*
 * Handler for stop button press.  First argument is main application widget,
 * second argument is unused.
 */
static void     on_stop_clicked(GtkWidget * widget, gpointer data);

/*
 * Handler for seconds spin button change.  First argument is main
 * application widget, second argument is unused.
 */
static void     on_seconds_value_changed(GtkWidget * widget, gpointer data);

/*
 * Handler for step spin button change.  First argument is main application
 * widget, second argument is unused.
 */
static void     on_step_value_changed(GtkWidget * widget, gpointer data);

/*
 * Handler for interval scale widget change.  First argument is main
 * application widget, second argument is unused.
 */
static void     on_interval_value_changed(GtkWidget * widget, gpointer data);

/*
 * Handler for timeout event.  Handler argument is main application widget
 */
static gint     on_timeout(gpointer data);

GtkWidget      *
time_controls_create(GtkWidget * window)
{
	Input          *input;	/* Input object. */
	GtkTooltips    *tooltips;	/* Tooltips object. */
	GtkWidget      *frame;	/* Frame containing time controls. */
	GtkWidget      *eventbox;	/* Event box widget. */
	GtkWidget      *vbox;	/* Vertical box for widgets packing. */
	GtkWidget      *hbox;	/* Horizontal box for widgets packing. */
	GtkWidget      *button;	/* Button widget. */
	GtkWidget      *image;	/* Image widget. */
	GtkWidget      *label;	/* Label widget. */
	GtkWidget      *spin;	/* Spin widget. */
	GtkWidget      *scale;	/* Scale widget. */
	gchar          *string;	/* String containing sampling frequency
				 * label. */
	int             interval;	/* Interval between successive
					 * animation frames. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(window), "input");
	assert(input != NULL);

	/* Get tooltips object. */
	tooltips = GTK_TOOLTIPS(g_object_get_data(G_OBJECT(window), "tooltips"));

	/* Create frame widget for time controls layout. */
	frame = gtk_frame_new(_("Time"));

	/* Create and setup vertical box for widgets packing. */
	vbox = gtk_vbox_new(FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 3);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* Create and setup horizontal box for first row of widgets. */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* Create and setup sampling frequency label widget. */
	eventbox = gtk_event_box_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), eventbox, _("Input signal sampling rate"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), eventbox, FALSE, FALSE, 3);
	string = g_strdup_printf(_("Sampling frequency: %gHz"), input->frequency);
	label = gtk_label_new(string);
	g_free(string);
	gtk_container_add(GTK_CONTAINER(eventbox), label);

	/* Create and setup horizontal box for second row of widgets. */
	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* Create and setup current time label widget. */
	label = gtk_label_new(_("Current time:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	/* Create and setup current time spin widget. */
	spin = gtk_spin_button_new_with_range(0, input->last / input->frequency, 1 / input->frequency);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), spin, _("Current animation time"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 3);
	g_object_set_data(G_OBJECT(window), "seconds_spin", spin);
	g_signal_connect_swapped(G_OBJECT(spin), "value_changed", G_CALLBACK(on_seconds_value_changed), G_OBJECT(window));

	/* Create and setup time units label widget. */
	label = gtk_label_new(_("sec"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	/* Create and setup horizontal box for third row of widgets. */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* Create and setup forward button widget. */
	button = gtk_toggle_button_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Play animation forward"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	image = gtk_image_new_from_stock(GTK_STOCK_GO_FORWARD, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(button), image);
	g_object_set_data(G_OBJECT(window), "forward_button", button);
	g_signal_connect_swapped(G_OBJECT(button), "clicked", G_CALLBACK(on_forward_clicked), G_OBJECT(window));

	/* Create and setup backward button widget. */
	button = gtk_toggle_button_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Play animation backward"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	image = gtk_image_new_from_stock(GTK_STOCK_GO_BACK, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(button), image);
	g_object_set_data(G_OBJECT(window), "backward_button", button);
	g_signal_connect_swapped(G_OBJECT(button), "clicked", G_CALLBACK(on_backward_clicked), G_OBJECT(window));

	/* Create and setup stop button widget. */
	button = gtk_toggle_button_new();
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Stop animation"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	image = gtk_image_new_from_stock(GTK_STOCK_STOP, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(button), image);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	g_object_set_data(G_OBJECT(window), "stop_button", button);
	g_signal_connect_swapped(G_OBJECT(button), "clicked", G_CALLBACK(on_stop_clicked), G_OBJECT(window));

	/* Create and setup step units label widget. */
	label = gtk_label_new(_("samples"));
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	/* Create and setup time step spin widget. */
	spin = gtk_spin_button_new_with_range(1, input->last, 1);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), spin, _("Number of samples to advance during each animation step"), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, FALSE, 3);
	g_object_set_data(G_OBJECT(window), "step_spin", spin);
	g_signal_connect_swapped(G_OBJECT(spin), "value_changed", G_CALLBACK(on_step_value_changed), G_OBJECT(window));

	/* Create and setup time step label widget. */
	label = gtk_label_new(_("Step:"));
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	/* Create and setup horizontal box for fourth row of widgets. */
	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* Create and setup first animation speed label widget. */
	label = gtk_label_new(_("Slower"));
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

	/*
	 * Initialize and remember interval between successive animation
	 * frames (in ms).
	 */
	interval = 500;
	g_object_set_data(G_OBJECT(window), "interval", GINT_TO_POINTER(interval));

	/*
	 * Create and setup animation speed scale widget.  Allowed values for
	 * interval between successive animation frames are from 10 to
	 * 1000ms.  Corresponding scale widget values are from 1000 (faster)
	 * down to 10 (slower).
	 */
	scale = gtk_hscale_new_with_range(10, 1000, 1);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), scale, _("Animation speed"), NULL);
	gtk_range_set_value(GTK_RANGE(scale), 1010 - interval);
	gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), scale, TRUE, TRUE, 0);
	g_object_set_data(G_OBJECT(window), "interval_scale", scale);
	g_signal_connect_swapped(G_OBJECT(scale), "value_changed", G_CALLBACK(on_interval_value_changed), G_OBJECT(window));

	/* Create and setup second animation speed label widget. */
	label = gtk_label_new(_("Faster"));
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

	return frame;
}

static void
on_forward_clicked(GtkWidget * widget, gpointer data)
{
	GtkWidget      *button;	/* Button widget. */
	GtkWidget      *spin;	/* Spin widget. */
	int             step;	/* Step units. */
	int             interval;	/* Interval between successive
					 * animation frames. */

	/*
	 * Get forward button widget and proceed only if button not already
	 * pressed.
	 */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "forward_button"));
	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
		return;

	/* Turn off backward and stop buttons. */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "backward_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "stop_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);

	/* Disable current time and time step spin widgets. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, FALSE);
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "step_spin"));
	gtk_widget_set_sensitive(spin, FALSE);

	/* Make time step positive and remember new value. */
	step = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "step"));
	step = abs(step);
	g_object_set_data(G_OBJECT(widget), "step", GINT_TO_POINTER(step));

	/*
	 * If timeout not already scheduled, schedule one and set animation
	 * active flag.
	 */
	if (!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "animation"))) {
		interval = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "interval"));
		gtk_timeout_add(interval, on_timeout, widget);
		g_object_set_data(G_OBJECT(widget), "animation", GINT_TO_POINTER(1));
	}
}

static void
on_backward_clicked(GtkWidget * widget, gpointer data)
{
	GtkWidget      *button;	/* Button widget. */
	GtkWidget      *spin;	/* Spin widget. */
	int             step;	/* Step units. */
	int             interval;	/* Interval between successive
					 * animation frames. */

	/*
	 * Get backward button widget and proceed only if button not already
	 * pressed.
	 */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "backward_button"));
	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
		return;

	/* Turn off forward and stop buttons. */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "forward_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "stop_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);

	/* Disable current time and time step spin widgets. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, FALSE);
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "step_spin"));
	gtk_widget_set_sensitive(spin, FALSE);

	/* Make time step negative and remember new value. */
	step = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "step"));
	step = -abs(step);
	g_object_set_data(G_OBJECT(widget), "step", GINT_TO_POINTER(step));

	/*
	 * If timeout not already scheduled, schedule one and set animation
	 * active flag.
	 */
	if (!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "animation"))) {
		interval = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "interval"));
		gtk_timeout_add(interval, on_timeout, widget);
		g_object_set_data(G_OBJECT(widget), "animation", GINT_TO_POINTER(1));
	}
}

static void
on_stop_clicked(GtkWidget * widget, gpointer data)
{
	GtkWidget      *button;	/* Button widget. */
	GtkWidget      *spin;	/* Spin widget. */

	/*
	 * Get stop button widget and proceed only if button not already
	 * pressed.
	 */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "stop_button"));
	if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
		return;

	/* Turn off forward and backward buttons. */
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "forward_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "backward_button"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);

	/* Enable current time and time step spin widgets. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, TRUE);
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "step_spin"));
	gtk_widget_set_sensitive(spin, TRUE);

	/* Reset animation active flag. */
	g_object_set_data(G_OBJECT(widget), "animation", GINT_TO_POINTER(0));
}

static void
on_seconds_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *spin;	/* Spin widget. */
	Input          *input;	/* Input object. */
	int             curr;	/* current record in input file. */
	int             lo;	/* Lowest allowed position in input file. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Get spin widget and proceed only if widget enabled. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	assert(spin != NULL);
	if (!GTK_WIDGET_IS_SENSITIVE(spin))
		return;

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Calculate current record in input file. */
	curr = (int)(gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) * input->frequency);

	/*
	 * Calculate lowest allowed position in input file according to
	 * current score.
	 */
	switch (GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "score"))) {
	case RAW_POTENTIAL:
		lo = 0;
		break;

	case DFT_AMPLITUDE:
	case DFT_PHASE:
		lo = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "window")) - 1;
		break;
	}

	/*
	 * Adjust current record in input file and current time spin widget
	 * if necessary.
	 */
	if (curr < lo) {
		curr = lo;
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), curr / input->frequency);

	} else if (curr > input->last) {
		curr = input->last;
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), curr / input->frequency);
	}
	/* Remember current record in input file. */
	g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_step_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *spin;	/* Spin widget. */
	Input          *input;	/* Input object. */
	int             step;	/* Step units. */

	/* Get spin widget and proceed only if widget enabled. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "step_spin"));
	assert(spin != NULL);
	if (!GTK_WIDGET_IS_SENSITIVE(spin))
		return;

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Remember step units. */
	step = ((GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "step")) > 0) ? 1 : -1) * gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
	g_object_set_data(G_OBJECT(widget), "step", GINT_TO_POINTER(step));
}

static void
on_interval_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *scale;	/* Scale widget. */
	int             interval;	/* Interval between successive
					 * animation frames. */

	/* Get scale widget. */
	scale = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "interval_scale"));
	assert(scale != NULL);

	/* Remember interval between successive animation frames. */
	interval = 1010 - gtk_range_get_value(GTK_RANGE(scale));
	g_object_set_data(G_OBJECT(widget), "interval", GINT_TO_POINTER(interval));
}

static          gint
on_timeout(gpointer data)
{
	GtkWidget      *widget;	/* Main window widget. */
	GtkWidget      *spin;	/* Spind widget. */
	GtkWidget      *button;	/* Button widget. */
	int             interval;	/* Interval between successive
					 * animation frames. */
	Input          *input;	/* Input object. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */
	int             curr;	/* current record in input file. */
	int             lo;	/* Lowest allowed position in input file. */

	/*
	 * Get main window widget and proceed only if animation flag is
	 * active.
	 */
	widget = GTK_WIDGET(data);
	if (GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "animation")) == 0)
		return FALSE;

	/* Schedule next timeout. */
	interval = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "interval"));
	gtk_timeout_add(interval, on_timeout, widget);

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Get current record in input file. */
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));

	/* Adjust seconds spin. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	assert(spin != NULL);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), curr / input->frequency);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);

	/* Increment current record in input file. */
	curr += GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "step"));

	/*
	 * Calculate lowest allowed position in input file according to
	 * current score.
	 */
	switch (GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "score"))) {
	case RAW_POTENTIAL:
		lo = 0;
		break;

	case DFT_AMPLITUDE:
	case DFT_PHASE:
		lo = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "window")) - 1;
		break;
	}

	/*
	 * Adjust current record in input file and stop animation if
	 * necessary.
	 */
	if (curr < lo) {
		curr = lo;
		button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "stop_button"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	} else if (curr > input->last) {
		curr = input->last;
		button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "stop_button"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	}
	/* Remember current record in input file. */
	g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));

	return FALSE;
}
