/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <float.h>
#include <locale.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gtk/gtkgl.h>
#include <libgnome/libgnome.h>
#include "gettext.h"
#include "input.h"
#include "interpolation.h"
#include "scene.h"
#include "visualization.h"
#include "info_controls.h"
#include "legend_controls.h"
#include "score_controls.h"
#include "time_controls.h"

#define _(String) gettext(String)

/* Filename of file containing 3D face model. */
#define FACE "face.mo"

/* Filename of file containing 3D scalp model. */
#define SCALP "scalp.mo"

/* Filename of file containing 3D electrodes positions. */
#define ELECTRODES "10_10.ms"

/*
 * Handler for help button pressing.  First argument is main window widget,
 * second argument is unused.
 */
static void     on_help(GtkWidget * widget, gpointer);

/*
 * Handler for destroying main window widget.  First argument is main window
 * widget, second argument is unused.
 */
static void     on_destroy(GtkWidget * widget, gpointer data);

/*
 * Update main window with next map.  First argument is main window widget.
 */
static void     update(GtkWidget * widget);

int
main(int argc, char **argv)
{
        char           *locale; /* Current locale. */
	Scene          *scene;	/* Scene object. */
	Sensors        *sensors;/* Sensors object. */
	Input          *input;	/* Input object. */
	Interpolation  *interpolation;	/* Interpolation object. */
	int             count;	/* Number of different scores. */
	float          *lo, *hi, *min, *max;	/* Arrays with current and
						 * widest boundaries for
						 * interpolation of score
						 * values. */
	GtkTooltips    *tooltips;	/* Tooltips object. */
	GtkWidget      *window;	/* Main window widget. */
	GtkWidget      *visualization;	/* Visualization widget. */
	GtkWidget      *hbox;	/* Horizontal box for widgets packing. */
	GtkWidget      *vbox;	/* Vertical box for widgets packing. */
	GtkWidget      *frame;	/* Frame widget. */
	GtkWidget      *button;	/* Button widget. */
	int             i;	/* Loop index. */


        /* Set locale. */
        setlocale(LC_ALL, "");

#if ENABLE_NLS
        /* Enable internationalization. */
        bindtextdomain(PACKAGE, LOCALEDIR);
        bind_textdomain_codeset(PACKAGE, "UTF-8");
        textdomain(PACKAGE);
#endif
	/* Initialize GTK and GNOME libraries. */
	gtk_init(&argc, &argv);
	gtk_gl_init(&argc, &argv);
	gnome_program_init("tempo", "1.1.4", LIBGNOME_MODULE, argc, argv, GNOME_PARAM_APP_DATADIR, DATADIR, NULL);

	/*
	 * Check number of arguments and print usage message if appropriate.
	 * Filename with EEG recording must be supplied as command line
	 * argument.
	 */
	if (argc != 2) {
		fprintf(stderr, _("Usage: tempo edf-file\n"));
		exit(EXIT_FAILURE);
	}

        /* 
         * Remember current locale and switch to C locale in order to
         * read input files. 
         */
	locale = g_strdup(setlocale (LC_NUMERIC, NULL));
        setlocale(LC_NUMERIC, "C");

	/* Create scene, sensors, input and interpolation objects. */
	scene = scene_create(DATADIR "/" SCALP, DATADIR "/" FACE);
	sensors = sensors_create(DATADIR "/" ELECTRODES);
	input = input_create(argv[1], sensors);
	interpolation = interpolation_create(scene->mapped, input, sensors);
	sensors_destroy(sensors);

        /* Restore locale. */
        setlocale(LC_NUMERIC, locale);
	g_free(locale);

	/* Create tooltips object. */
	tooltips = gtk_tooltips_new();

	/* Create and setup main window widget. */
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "TEMPO");
	gtk_container_set_border_width(GTK_CONTAINER(window), 3);
	g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(on_destroy), NULL);

	/*
	 * Initialize values that will be remembered with main window widget.
	 * First remember tooltips object and input, scene and interpolation
	 * objects, as well as main window update function.
	 */
	g_object_set_data(G_OBJECT(window), "tooltips", tooltips);
	g_object_set_data(G_OBJECT(window), "scene", scene);
	g_object_set_data(G_OBJECT(window), "input", input);
	g_object_set_data(G_OBJECT(window), "interpolation", interpolation);
	g_object_set_data(G_OBJECT(window), "update", update);

	/*
	 * Then, initialize and remember current score, DFT window length,
	 * current DFT frequency, current record index, time step and
	 * animation active flag.
	 */
	g_object_set_data(G_OBJECT(window), "score", GINT_TO_POINTER(RAW_POTENTIAL));
	g_object_set_data(G_OBJECT(window), "window", GINT_TO_POINTER(128));
	g_object_set_data(G_OBJECT(window), "frequency", GINT_TO_POINTER(10));
	g_object_set_data(G_OBJECT(window), "curr", GINT_TO_POINTER(0));
	g_object_set_data(G_OBJECT(window), "step", GINT_TO_POINTER(1));
	g_object_set_data(G_OBJECT(window), "animation", GINT_TO_POINTER(0));

	/*
	 * Finally, allocate and remember arrays with current and widest
	 * boundaries for interpolation of score values.  Also initialize
	 * boundary values for current score.
	 */
	count = 1 + 2 * input->frequency;
	lo = (float *)malloc(count * sizeof(float));
	assert(lo != NULL);
	hi = (float *)malloc(count * sizeof(float));
	assert(hi != NULL);
	g_object_set_data(G_OBJECT(window), "lo", lo);
	g_object_set_data(G_OBJECT(window), "hi", hi);
	min = (float *)malloc(count * sizeof(float));
	assert(min != NULL);
	max = (float *)malloc(count * sizeof(float));
	assert(max != NULL);
	g_object_set_data(G_OBJECT(window), "min", min);
	g_object_set_data(G_OBJECT(window), "max", max);
	for (i = 0; i < count; i++)
		lo[i] = FLT_MAX;
	g_object_set_data(G_OBJECT(window), "lo_curr", lo);
	g_object_set_data(G_OBJECT(window), "hi_curr", hi);

	/*
	 * Initialize random number generator for selecting random samples
	 * from input file while estimating above boundaries.
	 */
	srand(0);

	/* Create and setup horizontal box for widgets packing. */
	hbox = gtk_hbox_new(FALSE, 3);
	gtk_container_add(GTK_CONTAINER(window), hbox);

	/*
	 * Create and setup and visualization widget.  Remember visualization
	 * widget with main window and scene object with visualization
	 * widget.
	 */
	visualization = visualization_create();
	gtk_box_pack_start(GTK_BOX(hbox), visualization, TRUE, TRUE, 0);
	g_object_set_data(G_OBJECT(window), "visualization", visualization);
	g_object_set_data(G_OBJECT(visualization), "scene", scene);

	/* Create and setup vertical box for GUI controls packing. */
	vbox = gtk_vbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	/* Create info controls. */
	frame = info_controls_create(window);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* Create score controls. */
	frame = score_controls_create(window);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* Create legend controls. */
	frame = legend_controls_create(window);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* Create time controls. */
	frame = time_controls_create(window);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* Create and setup horizontal box for last row of buttons. */
	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* Create and setup save button. */
	button = gtk_button_new_from_stock(GTK_STOCK_SAVE);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Save current map in PNG format"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
	g_signal_connect_swapped(G_OBJECT(button), "clicked", G_CALLBACK(on_visualization_save), visualization);

	/* Create and setup help button. */
	button = gtk_button_new_from_stock(GTK_STOCK_HELP);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Show on-line help"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(on_help), NULL);

	/* Create and setup quit button. */
	button = gtk_button_new_from_stock(GTK_STOCK_QUIT);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), button, _("Quit program"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
	g_signal_connect_swapped(G_OBJECT(button), "clicked", G_CALLBACK(gtk_widget_destroy), G_OBJECT(window));

	/* Show all widgets, update main window and enter main loop. */
	gtk_widget_show_all(window);
	update(window);
	gtk_main();

	/* Destory input, scene and interpolation objects. */
	scene_destroy(scene);
	input_destroy(input);
	interpolation_destroy(interpolation);

	/* Finish program. */
	exit(EXIT_SUCCESS);
}

static void
on_help(GtkWidget * widget, gpointer data)
{
	/* Display help topics. */
	gnome_help_display("tempo", NULL, NULL);
}

static void
on_destroy(GtkWidget * widget, gpointer data)
{
	/* Reset animation active flag. */
	g_object_set_data(G_OBJECT(widget), "animation", GINT_TO_POINTER(0));

	/*
	 * Free memory used for arrays with current and widest boundaries for
	 * interpolation of score values.
	 */
	free(g_object_get_data(G_OBJECT(widget), "lo"));
	free(g_object_get_data(G_OBJECT(widget), "hi"));
	free(g_object_get_data(G_OBJECT(widget), "min"));
	free(g_object_get_data(G_OBJECT(widget), "max"));

	/* Finish main loop. */
	gtk_main_quit();
}

static void
update(GtkWidget * widget)
{
	GtkWidget      *visualization;	/* Visualization widget. */
	Input          *input;	/* Input object. */
	Scene          *scene;	/* Scene object. */
	Interpolation  *interpolation;	/* Interpolatin object. */
	int             curr;	/* Current record index. */
	float          *values;	/* Vector with current score values. */
	int             score;	/* Current score. */
	int             window;	/* DFT window lenght. */
	int             frequency;	/* Current DFT frequency. */
	float          *lo_curr, *hi_curr;	/* Boundary values for
						 * current score. */

	/* Get visualization widget. */
	visualization = (GtkWidget *) g_object_get_data(G_OBJECT(widget), "visualization");
	assert(visualization != NULL);

	/* Get input, scene and interpolation objects. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);
	scene = (Scene *) g_object_get_data(G_OBJECT(widget), "scene");
	assert(scene != NULL);
	interpolation = (Interpolation *) g_object_get_data(G_OBJECT(widget), "interpolation");
	assert(interpolation != NULL);

	/* Get current record index. */
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));

	/* Allocate memory for vector of current score values. */
	values = (float *)malloc(input->count * sizeof(values));
	assert(values != NULL);

	/* Get current score and input vector of current score values. */
	score = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "score"));
	switch (score) {
	case RAW_POTENTIAL:
		input_values(input, curr, score, input->count, values);
		break;

	case DFT_AMPLITUDE:
	case DFT_PHASE:
		window = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "window"));
		frequency = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "frequency"));
		input_values(input, curr, score, input->count, values, window, frequency);
		break;

	}

	/* Create copy of scalp object. */
	object_destroy(scene->mapped);
	scene->mapped = object_copy(scene->copy);

	/* Get boundary values for current score. */
	lo_curr = (float *)g_object_get_data(G_OBJECT(widget), "lo_curr");
	assert(lo_curr != NULL);
	hi_curr = (float *)g_object_get_data(G_OBJECT(widget), "hi_curr");
	assert(hi_curr != NULL);

	/* Interpolate vector of current score values over scalp region. */
	interpolation_process(interpolation, scene->mapped, input->count, values, *lo_curr, *hi_curr);

	/* Free memory used for vector of current score values. */
	free(values);

	/* Force repaint of visuzalization window. */
	gdk_window_invalidate_rect(visualization->window, NULL, FALSE);
}
