/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <stdlib.h>
#include "gettext.h"
#include "input.h"
#include "score_controls.h"

#define _(String) gettext(String)

/*
 * Count of random score values used to estimate score minimal and maximal
 * value for interpolation.
 */
#define COUNT 64

/*
 * Scaling factor to multiply score maximal and eventually minimal value to
 * calculate allowed interval for user changes of these values.
 */
#define SCALE 2

/*
 * Handler for realize event.  First argument is main application widget,
 * second argument is unused.
 */
static void     on_controls_realize(GtkWidget * widget, gpointer);

/*
 * Handler for activating raw potential score.  First argument is main
 * application widget, second argument is unused.
 */
static void     on_raw_potential_activate(GtkWidget * widget, gpointer);

/*
 * Handler for activating DFT amplitude score.  First argument is main
 * application widget, second argument is unused.
 */
static void     on_dft_amplitude_activate(GtkWidget * widget, gpointer);

/*
 * Handler for activating DFT phase score.  First argument is main
 * application widget, second argument is unused.
 */
static void     on_dft_phase_activate(GtkWidget * widget, gpointer);

/*
 * Handler for activating 128 samples long DFT window.  First argument is
 * main application widget, second argument is unused.
 */
static void     on_128_samples_activate(GtkWidget * widget, gpointer);

/*
 * Handler for activating 256 samples long DFT window.  First argument is
 * main application widget, second argument is unused.
 */
static void     on_256_samples_activate(GtkWidget * widget, gpointer);

/*
 * Handler for activating 512 samples long DFT window.  First argument is
 * main application widget, second argument is unused.
 */
static void     on_512_samples_activate(GtkWidget * widget, gpointer);

/*
 * Handler for changing frequency to return data for DFT scores.  First
 * argument is main application widget, second argument is unused.
 */
static void     on_frequency_value_changed(GtkWidget * widget, gpointer);

/*
 * Update boundaries for current score interpolation. First argument is main
 * application widget, second argument is flag determining if corresponding
 * spin buttons should be updated.
 */
static void     ranges_update(GtkWidget * window, int flag);

GtkWidget      *
score_controls_create(GtkWidget * window)
{
	Input          *input;	/* Input object. */
	GtkTooltips    *tooltips;	/* Tooltips object. */
	GtkWidget      *frame;	/* Frame containing score controls. */
	GtkWidget      *table;	/* Table widget. */
	GtkWidget      *hbox;	/* Horizontal box for widgets packing. */
	GtkWidget      *label;	/* Label widget. */
	GtkWidget      *menu;	/* Menu widget. */
	GtkWidget      *item;	/* Menu item widget. */
	GtkWidget      *option;	/* Option widget. */
	GtkWidget      *spin;	/* Spin widget. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(window), "input");
	assert(input != NULL);

	/* Get tooltips object. */
	tooltips = GTK_TOOLTIPS(g_object_get_data(G_OBJECT(window), "tooltips"));

	/* Create and setup frame widget for score controls layout. */
	frame = gtk_frame_new(_("Score"));
	g_signal_connect_swapped(G_OBJECT(frame), "realize", G_CALLBACK(on_controls_realize), G_OBJECT(window));

	/* Create and setup table widget. */
	table = gtk_table_new(3, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(table), 3);
	gtk_table_set_col_spacings(GTK_TABLE(table), 3);
	gtk_container_add(GTK_CONTAINER(frame), table);

	/*
	 * Create and setup horizontal box and label for first row, first
	 * column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 0, 1);
	label = gtk_label_new(_("Score: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/*
	 * Create and setup horizontal box and menu widget for first row,
	 * second column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 0, 1);
	menu = gtk_menu_new();
	item = gtk_menu_item_new_with_label(_("Raw potential (uV)"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_raw_potential_activate), G_OBJECT(window));
	item = gtk_menu_item_new_with_label(_("DFT amplitude (uV)"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_dft_amplitude_activate), G_OBJECT(window));
	item = gtk_menu_item_new_with_label(_("DFT phase (radians)"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_dft_phase_activate), G_OBJECT(window));
	option = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(option), 0);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), option, _("Score mapped"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), option, TRUE, TRUE, 0);

	/*
	 * Create and setup horizontal box and label for second row, first
	 * column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 1, 2);
	label = gtk_label_new(_("Window: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	g_object_set_data(G_OBJECT(window), "window_label", label);

	/*
	 * Create and setup horizontal box and menu widget for second row,
	 * second column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 1, 2);
	menu = gtk_menu_new();
	item = gtk_menu_item_new_with_label(_("128 samples"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_128_samples_activate), G_OBJECT(window));
	item = gtk_menu_item_new_with_label(_("256 samples"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_256_samples_activate), G_OBJECT(window));
	item = gtk_menu_item_new_with_label(_("512 samples"));
	gtk_menu_append(GTK_MENU(menu), item);
	g_signal_connect_swapped(G_OBJECT(item), "activate", G_CALLBACK(on_512_samples_activate), G_OBJECT(window));
	option = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(option), 0);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), option, _("Number of samples for Fourier transform"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), option, TRUE, TRUE, 0);
	g_object_set_data(G_OBJECT(window), "window_option", option);

	/*
	 * Create and setup horizontal box and label for third row, first
	 * column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 2, 3);
	label = gtk_label_new(_("Frequency: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	g_object_set_data(G_OBJECT(window), "frequency_label", label);

	/*
	 * Create and setup horizontal box and spin widget for second row,
	 * second column table cell .
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 2, 3);
	spin = gtk_spin_button_new_with_range(0, (input->frequency / 2 - 1 <= 127) ? input->frequency / 2 - 1 : 127, 1);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), 10);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), spin, _("Frequency mapped for DFT scores"), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), spin, TRUE, TRUE, 0);
	g_object_set_data(G_OBJECT(window), "frequency_spin", spin);
	g_signal_connect_swapped(G_OBJECT(spin), "value_changed", G_CALLBACK(on_frequency_value_changed), G_OBJECT(window));

	/*
	 * Perform initial update of boundaries for current score
	 * interpolation.
	 */
	ranges_update(window, 0);

	return frame;
}

static void
on_controls_realize(GtkWidget * widget, gpointer data)
{
	/* Disable widgets related to DFT scores. */
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_label")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_option")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_label")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin")), FALSE);
}

static void
on_raw_potential_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	GtkWidget      *spin;	/* Spin widget. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Disable widgets related to DFT scores. */
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_label")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_option")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_label")), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin")), FALSE);

	/* Remember current score. */
	g_object_set_data(G_OBJECT(widget), "score", GINT_TO_POINTER(RAW_POTENTIAL));

	/* Get input object and adjust range of current time spin widget. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	assert(spin != NULL);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), 0, input->last / input->frequency);

	/* Update boundaries for score interpolation. */
	ranges_update(widget, 1);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_dft_amplitude_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	GtkWidget      *button;	/* Button widget. */
	int             window;	/* DFT window length. */
	int             curr;	/* Current data record. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Enable widgets related to DFT scores. */
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_label")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_option")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_label")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin")), TRUE);

	/* Remember current score. */
	g_object_set_data(G_OBJECT(widget), "score", GINT_TO_POINTER(DFT_AMPLITUDE));

	/* Get input object and adjust range of current time spin widget. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	assert(button != NULL);
	window = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "window"));
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(button), (window - 1) / input->frequency, input->last / input->frequency);
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));
	if (curr < window - 1) {
		curr = window - 1;
		assert(curr <= input->last);
		g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(button), curr / input->frequency);
	}
	/* Update boundaries for score interpolation. */
	ranges_update(widget, 1);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_dft_phase_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	GtkWidget      *button;	/* Button widget. */
	int             window;	/* DFT window length. */
	int             curr;	/* Current data record. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Enable widgets related to DFT scores. */
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_label")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "window_option")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_label")), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin")), TRUE);

	/* Remember current score. */
	g_object_set_data(G_OBJECT(widget), "score", GINT_TO_POINTER(DFT_PHASE));

	/* Get input object and adjust range of current time spin widget. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);
	button = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	assert(button != NULL);
	window = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "window"));
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(button), (window - 1) / input->frequency, input->last / input->frequency);
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));
	if (curr < window - 1) {
		curr = window - 1;
		assert(curr <= input->last);
		g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(button), curr / input->frequency);
	}
	/* Update boundaries for score interpolation. */
	ranges_update(widget, 1);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_128_samples_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	int             window;	/* DFT window length. */
	GtkWidget      *spin;	/* Spin widget. */
	int             curr;	/* Current data record. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Remember new DFT window lenght. */
	window = 128;
	assert(window - 1 <= input->last);
	g_object_set_data(G_OBJECT(widget), "window", GINT_TO_POINTER(window));

	/*
	 * Set allowed DFT frequency to return data for according to Nyquist
	 * theorem.
	 */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin"));
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), 0, (input->frequency / 2 - 1 <= window - 1) ? input->frequency / 2 - 1 : window - 1);

	/* Adjust range of current time spin widget. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, FALSE);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), (window - 1) / input->frequency, input->last / input->frequency);
	gtk_widget_set_sensitive(spin, TRUE);

	/* Eventually adjust number of current data record. */
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));
	if (curr < window - 1) {
		curr = window - 1;
		g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));
	}
	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_256_samples_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	int             window;	/* DFT window length. */
	GtkWidget      *spin;	/* Spin widget. */
	int             curr;	/* Current data record. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Remember new DFT window lenght. */
	window = 256;
	assert(window - 1 <= input->last);
	g_object_set_data(G_OBJECT(widget), "window", GINT_TO_POINTER(window));

	/*
	 * Set allowed DFT frequency to return data for according to Nyquist
	 * theorem.
	 */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin"));
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), 0, (input->frequency / 2 - 1 <= window - 1) ? input->frequency / 2 - 1 : window - 1);

	/* Adjust range of current time spin widget. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, FALSE);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), (window - 1) / input->frequency, input->last / input->frequency);
	gtk_widget_set_sensitive(spin, TRUE);

	/* Eventually adjust number of current data record. */
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));
	if (curr < window - 1) {
		curr = window - 1;
		g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));
	}
	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_512_samples_activate(GtkWidget * widget, gpointer data)
{
	Input          *input;	/* Input object. */
	int             window;	/* DFT window length. */
	GtkWidget      *spin;	/* Spin widget. */
	int             curr;	/* Current data record. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");
	assert(input != NULL);

	/* Remember new DFT window lenght. */
	window = 512;
	assert(window - 1 <= input->last);
	g_object_set_data(G_OBJECT(widget), "window", GINT_TO_POINTER(window));

	/*
	 * Set allowed DFT frequency to return data for according to Nyquist
	 * theorem.
	 */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin"));
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), 0, (input->frequency / 2 - 1 <= window - 1) ? input->frequency / 2 - 1 : window - 1);

	/* Adjust range of current time spin widget. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "seconds_spin"));
	gtk_widget_set_sensitive(spin, FALSE);
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), (window - 1) / input->frequency, input->last / input->frequency);
	gtk_widget_set_sensitive(spin, TRUE);

	/* Eventually adjust number of current data record. */
	curr = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "curr"));
	if (curr < window - 1) {
		curr = window - 1;
		g_object_set_data(G_OBJECT(widget), "curr", GINT_TO_POINTER(curr));
	}
	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
on_frequency_value_changed(GtkWidget * widget, gpointer data)
{
	GtkWidget      *spin;	/* Spin widget. */
	void            (*update) (GtkWidget *);	/* Main window update
							 * function. */

	/* Remember new frequency to return DFT data for. */
	spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "frequency_spin"));
	g_object_set_data(G_OBJECT(widget), "frequency", GINT_TO_POINTER(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin))));

	/* Update boundaries for score interpolation. */
	ranges_update(widget, 1);

	/* Update main window. */
	update = g_object_get_data(G_OBJECT(widget), "update");
	assert(update != NULL);
	(*update) (widget);
}

static void
ranges_update(GtkWidget * widget, int flag)
{
	Input          *input;	/* Input object. */
	float          *lo, *hi, *min, *max;	/* Arrays with current and
						 * widest boundaries for
						 * interpolation of score
						 * values. */

	int             score;	/* Current score. */
	int             frequency;	/* Frequency to return DFT values
					 * for. */
	int             index;	/* Index of elements in boundary arrays
				 * corresponding to current score. */
	float          *values;	/* Score values. */
	GtkWidget      *spin;	/* Spin widget. */
	int             i, j;	/* Loop indices. */

	/* Get input object. */
	input = (Input *) g_object_get_data(G_OBJECT(widget), "input");

	/* Get boundaries arrays. */
	lo = (float *)g_object_get_data(G_OBJECT(widget), "lo");
	assert(lo != NULL);
	hi = (float *)g_object_get_data(G_OBJECT(widget), "hi");
	assert(hi != NULL);
	min = (float *)g_object_get_data(G_OBJECT(widget), "min");
	assert(lo != NULL);
	max = (float *)g_object_get_data(G_OBJECT(widget), "max");
	assert(hi != NULL);

	/*
	 * Get current score and calculate index of its values in boundaries
	 * arays.
	 */
	score = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "score"));
	switch (score) {
	case RAW_POTENTIAL:
		index = 0;
		break;

	case DFT_AMPLITUDE:
		frequency = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "frequency"));
		index = 1 + frequency;
		break;

	case DFT_PHASE:
		frequency = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "frequency"));
		index = 1 + input->frequency + frequency;
		break;
	}

	/*
	 * Check if boundaries for current score already calculated and
	 * calculate them if not.
	 */
	if (lo[index] == FLT_MAX) {
		/* Alocate memory for score values. */
		values = (float *)malloc(input->count * sizeof(values));
		assert(values != NULL);

		/* Calculate boundaries depending on score. */
		switch (score) {
		case RAW_POTENTIAL:
			/*
			 * Get number of values from input file and find
			 * minimum and maximum raw potential value among
			 * them.  Make boundaries symmetric.
			 */
			lo[index] = FLT_MAX, hi[index] = -FLT_MAX;
			for (i = 0; i < COUNT; i++) {
				input_values(input, (int)(rand() / (RAND_MAX + 1.0) * input->last), RAW_POTENTIAL, input->count, values);
				for (j = 0; j < input->count; j++) {
					if (lo[index] > values[j])
						lo[index] = values[j];
					if (hi[index] < values[j])
						hi[index] = values[j];
				}
			}
			if (fabs(lo[index]) < fabs(hi[index]))
				lo[index] = -hi[index];
			else
				hi[index] = -lo[index];
			min[index] = lo[index] * SCALE, max[index] = hi[index] * SCALE;
			break;

		case DFT_AMPLITUDE:
			/*
			 * Calculate DFT for number of values from input file
			 * and find maximum amplitude at given frequency
			 * value among them. Use 0 as mimimum.
			 */
			lo[index] = hi[index] = 0;
			for (i = 0; i < COUNT; i++) {
				input_values(input, 127 + (int)(rand() / (RAND_MAX + 1.0) * (input->last - 128)), DFT_AMPLITUDE, input->count, values, 128, frequency);

				for (j = 0; j < input->count; j++)
					if (hi[index] < values[j])
						hi[index] = values[j];
			}
			min[index] = lo[index] * SCALE, max[index] = hi[index] * SCALE;
			break;

		case DFT_PHASE:
			/* Use -pi as low and pi as high boundary. */
			lo[index] = min[index] = -M_PI;
			hi[index] = max[index] = M_PI;
			break;
		}


		free(values);
	}
	/* Remember boundaries. */
	g_object_set_data(G_OBJECT(widget), "lo_curr", lo + index);
	g_object_set_data(G_OBJECT(widget), "hi_curr", hi + index);

	/*
	 * If update of spin buttons representing boundaries requested,
	 * perform update.  Be careful regarding fact that update of spin
	 * widget range and value is not an atomic operation.
	 */
	if (flag) {
		spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "lo_spin"));
		g_object_set_data(G_OBJECT(widget), "change_flag", GINT_TO_POINTER(0));
		gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), min[index], hi[index]);
		g_object_set_data(G_OBJECT(widget), "change_flag", GINT_TO_POINTER(1));
		gtk_spin_button_set_increments(GTK_SPIN_BUTTON(spin), (max[index] - min[index]) / 100, (max[index] - min[index]) / 10);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), lo[index]);

		spin = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "hi_spin"));
		g_object_set_data(G_OBJECT(widget), "change_flag", GINT_TO_POINTER(0));
		gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin), lo[index], max[index]);
		g_object_set_data(G_OBJECT(widget), "change_flag", GINT_TO_POINTER(1));
		gtk_spin_button_set_increments(GTK_SPIN_BUTTON(spin), (max[index] - min[index]) / 100, (max[index] - min[index]) / 10);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), hi[index]);
	}
}
